<?php
require_once 'includes/db.php';

$event_id = 3;
$heat_number = 1;

echo "<h2>Testing Figures Data for Event $event_id, Heat $heat_number</h2>";

// Check if figures_json column exists
$hasColumnsQuery = "SHOW COLUMNS FROM scores LIKE 'figures_json'";
$hasColumnsStmt = $pdo->query($hasColumnsQuery);
$hasFiguresColumn = $hasColumnsStmt->rowCount() > 0;

echo "<p><strong>figures_json column exists:</strong> " . ($hasFiguresColumn ? "YES" : "NO") . "</p>";

if (!$hasFiguresColumn) {
    echo "<p style='color:red;'>The figures_json column doesn't exist in the scores table!</p>";
    exit;
}

echo "<p>Preparing to query scores...</p>";
flush();

// Get scores with figures
$sql = "
    SELECT 
        s.id as score_id,
        s.score_value,
        s.figures_json,
        ep.heat_number,
        r.run_number,
        u.username as judge_name,
        p.first_name,
        p.last_name
    FROM scores s
    JOIN runs r ON s.run_id = r.id
    JOIN event_participants ep ON r.event_participant_id = ep.id
    JOIN participants p ON ep.participant_id = p.id
    LEFT JOIN users u ON s.judge_id = u.id
    WHERE ep.event_id = ?
    AND ep.heat_number = ?
    AND s.figures_json IS NOT NULL
    AND s.figures_json != ''
    ORDER BY p.last_name, r.run_number, u.username
";

try {
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$event_id, $heat_number]);
    $scores = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<p>Query executed successfully!</p>";
    flush();
} catch (Exception $e) {
    echo "<p style='color:red;'>SQL Error: " . $e->getMessage() . "</p>";
    exit;
}

echo "<p><strong>Found " . count($scores) . " scores with figures_json data</strong></p>";

if (empty($scores)) {
    echo "<p style='color:orange;'>No scores found with figures_json data. This means judges haven't recorded any figures yet.</p>";
    
    // Check if there are any scores at all
    $check_sql = "
        SELECT COUNT(*) as count
        FROM scores s
        JOIN runs r ON s.run_id = r.id
        JOIN event_participants ep ON r.event_participant_id = ep.id
        WHERE ep.event_id = ?
        AND ep.heat_number = ?
    ";
    $check_stmt = $pdo->prepare($check_sql);
    $check_stmt->execute([$event_id, $heat_number]);
    $result = $check_stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "<p>Total scores (with or without figures): " . $result['count'] . "</p>";
} else {
    echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
    echo "<tr><th>Participant</th><th>Heat</th><th>Run</th><th>Judge</th><th>Score</th><th>Figures JSON</th><th>Parsed Figures</th></tr>";
    
    foreach ($scores as $score) {
        $participant = $score['first_name'] . ' ' . $score['last_name'];
        $figures_json = $score['figures_json'];
        
        // Try to parse the JSON
        $figures_data = json_decode($figures_json, true);
        $parsed = "";
        
        if (is_array($figures_data)) {
            $parsed = "<pre>" . print_r($figures_data, true) . "</pre>";
        } else {
            $parsed = "<span style='color:red;'>Invalid JSON</span>";
        }
        
        echo "<tr>";
        echo "<td>" . htmlspecialchars($participant) . "</td>";
        echo "<td>H" . $score['heat_number'] . "</td>";
        echo "<td>R" . $score['run_number'] . "</td>";
        echo "<td>" . htmlspecialchars($score['judge_name']) . "</td>";
        echo "<td>" . $score['score_value'] . "</td>";
        echo "<td><small>" . htmlspecialchars(substr($figures_json, 0, 100)) . "...</small></td>";
        echo "<td>" . $parsed . "</td>";
        echo "</tr>";
    }
    
    echo "</table>";
}

echo "<hr>";
echo "<h3>Check what head_judge_override.php shows</h3>";
echo "<p>You mentioned figures appear in <a href='admin/head_judge_override.php?event_id=3' target='_blank'>head_judge_override.php</a></p>";
echo "<p>That page might be querying the data differently.</p>";
