<?php
/**
 * Database Migration: Add participant_status column to event_participants table
 * 
 * This migration adds a status tracking system for participants in each heat/run
 * Statuses: initial, next_up, start_gate, performing, completed, dns (did not start), dnf (did not finish)
 */

include '../includes/db.php';

try {
    echo "<h2>Adding participant_status column to event_participants table</h2>";
    
    // Check if column already exists
    $check_column = $pdo->query("SHOW COLUMNS FROM event_participants LIKE 'participant_status'");
    
    if ($check_column->rowCount() > 0) {
        echo "<p style='color: orange;'>⚠️ Column 'participant_status' already exists. Skipping creation.</p>";
    } else {
        // Add participant_status column
        $pdo->exec("
            ALTER TABLE event_participants 
            ADD COLUMN participant_status VARCHAR(20) DEFAULT 'initial' 
            AFTER sort_order
        ");
        echo "<p style='color: green;'>✅ Column 'participant_status' added successfully</p>";
    }
    
    // Check if run_number column exists
    $check_run = $pdo->query("SHOW COLUMNS FROM event_participants LIKE 'run_number'");
    
    if ($check_run->rowCount() > 0) {
        echo "<p style='color: orange;'>⚠️ Column 'run_number' already exists. Skipping creation.</p>";
    } else {
        // Add run_number column for tracking which run the status applies to
        $pdo->exec("
            ALTER TABLE event_participants 
            ADD COLUMN run_number INT DEFAULT 1 
            AFTER heat_number
        ");
        echo "<p style='color: green;'>✅ Column 'run_number' added successfully</p>";
    }
    
    // Set all existing records to 'initial' status
    $update_result = $pdo->exec("
        UPDATE event_participants 
        SET participant_status = 'initial' 
        WHERE participant_status IS NULL OR participant_status = ''
    ");
    echo "<p style='color: green;'>✅ Updated {$update_result} existing records to 'initial' status</p>";
    
    // Create index for better query performance
    try {
        $pdo->exec("
            CREATE INDEX idx_participant_status 
            ON event_participants(event_id, heat_number, run_number, participant_status)
        ");
        echo "<p style='color: green;'>✅ Index created for participant_status queries</p>";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') !== false) {
            echo "<p style='color: orange;'>⚠️ Index already exists. Skipping.</p>";
        } else {
            throw $e;
        }
    }
    
    echo "<h3 style='color: green;'>Migration completed successfully!</h3>";
    echo "<p><strong>Valid statuses:</strong></p>";
    echo "<ul>";
    echo "<li><code>initial</code> - Default state, participant hasn't started yet</li>";
    echo "<li><code>next_up</code> - Next participant to perform</li>";
    echo "<li><code>start_gate</code> - Participant is at the start gate, ready to begin</li>";
    echo "<li><code>performing</code> - Currently performing</li>";
    echo "<li><code>completed</code> - Finished performance</li>";
    echo "<li><code>dns</code> - Did Not Start (no-show)</li>";
    echo "<li><code>dnf</code> - Did Not Finish (started but didn't complete)</li>";
    echo "</ul>";
    
    echo "<p><a href='../admin/'>← Back to Admin</a></p>";
    
} catch (PDOException $e) {
    echo "<h3 style='color: red;'>❌ Migration failed!</h3>";
    echo "<p style='color: red;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p><a href='../admin/'>← Back to Admin</a></p>";
}
?>
