<?php
/**
 * Active Heat Panel - Standalone Component
 * Displays real-time heat status, judge panel status, and results
 * 
 * Usage:
 * 1. Include directly: <?php include 'active_heat_panel.php'; ?>
 * 2. Standalone: active_heat_panel.php?event_id=3&heat=1
 * 3. AJAX load: Load into a container via fetch/AJAX
 */

include_once '../includes/auth.php';
include_once '../includes/db.php';

$event_id = $_GET['event_id'] ?? $_POST['event_id'] ?? null;
$heat_number = $_GET['heat'] ?? $_POST['heat'] ?? null;
$standalone = isset($_GET['standalone']) || !headers_sent();

if ($standalone): ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Active Heat Panel</title>
    <?php include_once '../includes/stylesheets.php'; ?>
</head>
<body class="body-bg-aurora-bright">
    <?php if (!isset($_GET['embed'])): ?>
        <?php include '../menu.php'; ?>
    <?php endif; ?>
    <div class="container-StyleScore p-4">
<?php endif; ?>

<!-- Active Heat Panel Container -->
<div id="activeHeatPanelContainer" class="border-0 shadow-none">
    
    <!-- Panel Header -->
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="mb-0 fw-bold text-primary">
            <i class="fas fa-fire me-2"></i>Active Heat Monitor
        </h5>
        <div class="d-flex gap-2 align-items-center">
            <!-- Refresh Interval Controls -->
            <div class="btn-group btn-group-sm" role="group" aria-label="Refresh interval">
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh2s" value="2" autocomplete="off">
                <label class="btn btn-outline-primary" for="refresh2s" title="Refresh every 2 seconds">2s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh5s" value="5" autocomplete="off">
                <label class="btn btn-outline-primary" for="refresh5s" title="Refresh every 5 seconds">5s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh10s" value="10" autocomplete="off" checked>
                <label class="btn btn-outline-primary" for="refresh10s" title="Refresh every 10 seconds">10s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh20s" value="20" autocomplete="off">
                <label class="btn btn-outline-primary" for="refresh20s" title="Refresh every 20 seconds">20s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refreshPause" value="0" autocomplete="off">
                <label class="btn btn-outline-warning" for="refreshPause" title="Pause refresh">
                    <i class="fas fa-pause"></i>
                </label>
            </div>
            
            <!-- Last Update Time -->
            <span class="badge bg-info" id="activeHeatUpdateTime">
                <i class="fas fa-clock me-1"></i>Updated: --
            </span>
            
            <!-- Manual Refresh Button -->
            <button class="btn btn-sm btn-outline-primary" onclick="loadActiveHeatCard()" title="Refresh now">
                <i class="fas fa-sync-alt"></i>
            </button>
        </div>
    </div>
    <div class="row">
        <div class="col-4">
            <!-- Heat Card Display Area -->
            <div id="activeHeatCardContainer" class="row">
                <!-- Active heat card will be loaded here -->
                <div class="col-12 text-center text-muted py-5">
                    <i class="fas fa-fire fa-3x mb-3 opacity-25"></i>
                    <p class="mb-0">No active heat selected</p>
                    <p class="small text-muted">Heat status and judge panels will appear here</p>
                </div>
            </div>
        </div>
        <div class="col-8">
            <!-- Participant List Section -->
            <div id="participantListSection" class="mt-4" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="mb-0 fw-bold text-primary">
                        <i class="fas fa-users me-2"></i>Participants & Scores
                    </h6>
                    <div class="d-flex gap-2">
                        <!-- Filter by category -->
                        <select id="categoryFilter" class="form-select form-select-sm" style="width: auto;">
                            <option value="">All Categories</option>
                        </select>
                        <!-- Filter scored participants -->
                        <select id="scoredFilter" class="form-select form-select-sm" style="width: auto;">
                            <option value="all">All Participants</option>
                            <option value="unscored">Unscored Only</option>
                            <option value="scored_bottom">Scored at Bottom</option>
                        </select>
                        <!-- Sort options -->
                        <select id="sortOrder" class="form-select form-select-sm" style="width: auto;">
                            <option value="sort_order" selected>Start Order</option>
                            <option value="bib">BIB Order</option>
                            <option value="name">Name (A-Z)</option>
                            <option value="score_desc">Score (High-Low)</option>
                        </select>
                        <button class="btn btn-sm btn-outline-primary" onclick="loadParticipantList()" title="Refresh list">
                            <i class="fas fa-sync-alt"></i>
                        </button>
                    </div>
                </div>
                
                <!-- Participant Cards Grid -->
                <div id="participantCardsGrid" class="row g-3">
                    <!-- Participant cards will be loaded here -->
                    <div class="col-12 text-center text-muted py-3">
                        <i class="fas fa-spinner fa-spin fa-2x mb-2"></i>
                        <p class="mb-0">Loading participants...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>    
    <!-- Results Modal Integration -->
    <div class="modal fade" id="heatResultsModal" tabindex="-1" aria-labelledby="heatResultsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-fullscreen p-5">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="heatResultsModalLabel">
                        <i class="fas fa-trophy me-2"></i>
                        <span id="resultsHeatName">Heat Results</span>
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <div class="row g-0">
                        <!-- Sidebar: Saved Configurations -->
                        <div class="col-md-3 bg-light border-end">
                            <div class="p-3">
                                <h6 class="fw-bold mb-3">
                                    <i class="fas fa-list me-2"></i>Available Reports
                                </h6>
                                <div id="savedConfigsList" class="d-flex flex-column gap-2">
                                    <!-- Configurations will be loaded here -->
                                    <div class="text-center text-muted py-3">
                                        <i class="fas fa-spinner fa-spin"></i>
                                        <p class="small mb-0 mt-2">Loading configurations...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Main Content: Results Display -->
                        <div class="col-md-9">
                            <div class="p-4">
                                <div id="resultsContentDisplay">
                                    <!-- Results content will be loaded here -->
                                    <div class="text-center text-muted py-5">
                                        <i class="fas fa-chart-bar fa-3x mb-3 opacity-25"></i>
                                        <p class="mb-0">Select a report from the sidebar</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
</div>

<style>
/* Active Heat Panel Styles */
#activeHeatCardContainer {
    min-height: 200px;
}

#activeHeatCardContainer .heat-card {
    transition: all 0.3s ease;
}

#activeHeatCardContainer .heat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.judge-panel-status {
    animation: pulse-status 2s infinite;
}

@keyframes pulse-status {
    0% { opacity: 1; }
    50% { opacity: 0.7; }
    100% { opacity: 1; }
}

/* Results Modal Styles */
.config-button {
    transition: all 0.2s ease;
    cursor: pointer;
    border-radius: 8px;
    padding: 12px;
    border: 2px solid transparent;
}

.config-button:hover {
    background-color: #e9ecef;
    border-color: #0d6efd;
}

.config-button.active {
    background-color: #0d6efd;
    color: white;
    border-color: #0d6efd;
}

#resultsContentDisplay {
    min-height: 400px;
}

/* Participant List Styles */
.participant-score-card {
    background: white;
    border: 2px solid #dee2e6;
    border-radius: 12px;
    padding: 16px;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.participant-score-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    border-color: #0d6efd;
}

.participant-score-card.has-score {
    border-color: #198754;
}

.participant-score-card.performing {
    border-color: #ffc107;
    background: linear-gradient(135deg, rgba(255, 193, 7, 0.1), white);
}

.participant-bib-badge {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, #0d6efd, #0a58ca);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.2rem;
    box-shadow: 0 2px 8px rgba(0,0,0,0.2);
}

.participant-info {
    flex-grow: 1;
    min-width: 0;
}

.participant-name {
    font-weight: 600;
    font-size: 1.1rem;
    color: #212529;
    margin-bottom: 4px;
}

.participant-details {
    font-size: 0.85rem;
    color: #6c757d;
}

.score-display {
    text-align: right;
}

.score-value {
    font-size: 1.5rem;
    font-weight: bold;
    color: #198754;
}

.score-label {
    font-size: 0.75rem;
    color: #6c757d;
    text-transform: uppercase;
}

.status-badge {
    position: absolute;
    top: 8px;
    right: 8px;
    font-size: 0.7rem;
    padding: 4px 8px;
}

.criteria-scores {
    display: flex;
    gap: 8px;
    margin-top: 8px;
    flex-wrap: wrap;
}

.criteria-score-item {
    background: #f8f9fa;
    padding: 4px 8px;
    border-radius: 6px;
    font-size: 0.75rem;
    border: 1px solid #dee2e6;
}

.criteria-score-item strong {
    color: #495057;
    margin-right: 4px;
}

/* Position Controls */
.position-controls {
    min-width: 50px;
}

.position-controls .btn {
    padding: 4px 8px;
    font-size: 0.75rem;
    line-height: 1;
}

.position-controls .btn:hover {
    transform: scale(1.1);
}

/* Toast notifications */
.toast-container {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 9999;
}

.notification-toast {
    min-width: 300px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
}
#activeHeatCardContainer .col-md-4.col-lg-3.mb-3 {
    width: 100%;
}
</style>

<script>
// Active Heat Panel JavaScript
let activeHeatCardInterval = null;
let activeHeatCardRefreshRate = 10; // Default 10 seconds
const currentEventId = <?= json_encode($event_id) ?>;
let currentHeatNumber = <?= json_encode($heat_number) ?>;

document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Active Heat Panel v2.1 - Position Swap Feature');
    console.log('Event ID:', currentEventId);
    console.log('Heat Number:', currentHeatNumber);
    
    // Setup event listeners
    setupRefreshControls();
    
    // Initial load
    loadActiveHeatCard();
    
    // Start auto-refresh
    restartActiveHeatCardInterval();
    
    // Setup filter/sort listeners
    setupParticipantListControls();
});

function setupParticipantListControls() {
    const categoryFilter = document.getElementById('categoryFilter');
    const scoredFilter = document.getElementById('scoredFilter');
    const sortOrder = document.getElementById('sortOrder');
    
    if (categoryFilter) {
        categoryFilter.addEventListener('change', () => {
            renderParticipantList();
        });
    }
    
    if (scoredFilter) {
        scoredFilter.addEventListener('change', () => {
            renderParticipantList();
        });
    }
    
    if (sortOrder) {
        sortOrder.addEventListener('change', () => {
            renderParticipantList();
        });
    }
}

function setupRefreshControls() {
    // Heat card refresh interval listeners
    document.querySelectorAll('input[name="heatCardRefresh"]').forEach(radio => {
        radio.addEventListener('change', (e) => {
            activeHeatCardRefreshRate = parseInt(e.target.value);
            console.log('Refresh interval changed to:', activeHeatCardRefreshRate);
            
            if (activeHeatCardRefreshRate === 0) {
                showInfoToast('⏸️ Auto-refresh Paused', 'Heat card will not auto-update');
            } else {
                showSuccessToast(`⏱️ Refresh: ${activeHeatCardRefreshRate}s`, 'Heat card will update automatically');
            }
            
            restartActiveHeatCardInterval();
        });
    });
}

async function loadActiveHeatCard(heatNumber = null) {
    const container = document.getElementById('activeHeatCardContainer');
    const updateTime = document.getElementById('activeHeatUpdateTime');
    
    // Use provided heat number or fall back to current
    const targetHeat = heatNumber || currentHeatNumber;
    
    if (!currentEventId) {
        container.innerHTML = `
            <div class="col-12 text-center text-muted py-5">
                <i class="fas fa-exclamation-triangle fa-3x mb-3 text-warning"></i>
                <p class="mb-0">No event ID provided</p>
            </div>
        `;
        updateTime.innerHTML = '<i class="fas fa-clock me-1"></i>Updated: --';
        return;
    }
    
    try {
        // Load heat cards from API
        const response = await fetch(`heat_cards_api.php?event_id=${currentEventId}`);
        const data = await response.json();
        
        if (!response.ok || !data.success) {
            throw new Error(data.error || 'Failed to load heat cards');
        }
        
        // Parse the HTML and extract heat cards
        const parser = new DOMParser();
        const doc = parser.parseFromString(data.data, 'text/html');
        const allCards = doc.querySelectorAll('.col-md-4');
        
        let activeCard = null;
        let foundHeat = null;
        
        // If specific heat number provided, find that heat
        if (targetHeat) {
            allCards.forEach(card => {
                const heatTitle = card.querySelector('.card-title');
                if (heatTitle && heatTitle.textContent.includes(`Heat ${targetHeat}`)) {
                    activeCard = card;
                    foundHeat = targetHeat;
                }
            });
        }
        
        // If no specific heat or not found, look for active heat
        if (!activeCard) {
            allCards.forEach(card => {
                // Look for the ACTIVE badge (bg-warning text-dark)
                const activeStatus = card.querySelector('.badge.bg-warning.text-dark');
                if (activeStatus && activeStatus.textContent.includes('ACTIVE')) {
                    activeCard = card;
                    const heatTitle = card.querySelector('.card-title, h5');
                    if (heatTitle) {
                        const match = heatTitle.textContent.match(/Heat (\d+)/);
                        if (match) foundHeat = match[1];
                    }
                }
                
                // Alternative: Check if card has active border class
                if (!activeCard && card.querySelector('.card.border-success')) {
                    activeCard = card;
                    const heatTitle = card.querySelector('.card-title, h5');
                    if (heatTitle) {
                        const match = heatTitle.textContent.match(/Heat (\d+)/);
                        if (match) foundHeat = match[1];
                    }
                }
            });
        }
        
        if (activeCard) {
            // Update current heat number for future reference
            if (foundHeat) currentHeatNumber = foundHeat;
            
            // Display the heat card
            container.innerHTML = activeCard.outerHTML;
            
            // Update timestamp
            const now = new Date();
            updateTime.innerHTML = '<i class="fas fa-clock me-1"></i>Updated: ' + now.toLocaleTimeString();
            
            console.log('Loaded heat card for Heat:', foundHeat);
            
            // Load participant list for this heat
            if (foundHeat) {
                loadParticipantList(foundHeat);
            }
        } else {
            container.innerHTML = `
                <div class="col-12 text-center text-muted py-5">
                    <i class="fas fa-fire fa-3x mb-3 opacity-25"></i>
                    <p class="mb-0">No active heat found</p>
                    <p class="small text-muted">Activate a heat to see details here</p>
                </div>
            `;
            updateTime.innerHTML = '<i class="fas fa-clock me-1"></i>No active heat';
        }
        
    } catch (error) {
        console.error('Error loading active heat card:', error);
        container.innerHTML = `
            <div class="col-12">
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error loading heat data:</strong> ${error.message}
                </div>
            </div>
        `;
        updateTime.innerHTML = '<i class="fas fa-exclamation-triangle me-1"></i>Error';
    }
}

function restartActiveHeatCardInterval() {
    // Clear existing interval
    if (activeHeatCardInterval) {
        clearInterval(activeHeatCardInterval);
        activeHeatCardInterval = null;
    }
    
    // Only start if not paused (rate > 0)
    if (activeHeatCardRefreshRate > 0) {
        activeHeatCardInterval = setInterval(() => {
            loadActiveHeatCard();
        }, activeHeatCardRefreshRate * 1000);
        console.log(`Active heat card refresh interval set to ${activeHeatCardRefreshRate}s`);
    } else {
        console.log('Active heat card auto-refresh paused');
    }
}

// ============================================================================
// PARTICIPANT LIST FUNCTIONS
// ============================================================================

let participantListData = [];

async function loadParticipantList(heatNumber = null) {
    const targetHeat = heatNumber || currentHeatNumber;
    
    if (!currentEventId || !targetHeat) {
        console.log('Cannot load participants: missing event_id or heat_number');
        return;
    }
    
    const section = document.getElementById('participantListSection');
    const grid = document.getElementById('participantCardsGrid');
    
    // Show loading
    grid.innerHTML = `
        <div class="col-12 text-center text-muted py-3">
            <i class="fas fa-spinner fa-spin fa-2x mb-2"></i>
            <p class="mb-0">Loading participants...</p>
        </div>
    `;
    
    try {
        // Load participants from heat management API
        const response = await fetch(`heat_management_api.php?action=get_queue&event_id=${currentEventId}&heat_number=${targetHeat}&mode=sort_order`);
        const data = await response.json();
        
        if (!response.ok) {
            throw new Error('API request failed');
        }
        
        // The API returns { queue: [], heat_settings: {}, mode: '' }
        participantListData = data.queue || [];
        
        // Debug: Check participant data structure
        if (participantListData.length > 0) {
            console.log('🔍 First participant object keys:', Object.keys(participantListData[0]));
            console.log('🔍 First participant sample:', participantListData[0]);
        }
        
        // Load scores for participants
        await loadParticipantScores(targetHeat);
        
        // Populate category filter
        populateCategoryFilter();
        
        // Render the list
        renderParticipantList();
        
        // Show the section
        section.style.display = 'block';
        
        console.log(`Loaded ${participantListData.length} participants for Heat ${targetHeat}`);
        
    } catch (error) {
        console.error('Error loading participant list:', error);
        grid.innerHTML = `
            <div class="col-12">
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error loading participants:</strong> ${error.message}
                </div>
            </div>
        `;
    }
}

async function loadParticipantScores(heatNumber) {
    // Load scores from scoring_results table for this specific heat/run
    try {
        // Get the active run number from the heat card or default to 1
        let runNumber = 1;
        const runBadge = document.querySelector('#activeHeatCardContainer .badge.bg-success');
        if (runBadge && runBadge.textContent.includes('/')) {
            const match = runBadge.textContent.match(/(\d+)\/\d+/);
            if (match) runNumber = parseInt(match[1]);
        }
        
        console.log('🔍 Loading scores for Heat ' + heatNumber + ', Run ' + runNumber);
        console.log('📊 Participant count:', participantListData.length);
        
        const url = 'heat_management_api.php?action=get_heat_run_scores&event_id=' + currentEventId + '&heat_number=' + heatNumber + '&run_number=' + runNumber;
        console.log('🌐 API URL:', url);
        
        const response = await fetch(url);
        const data = await response.json();
        
        console.log('📥 API Response:', data);
        
        if (data.success && Array.isArray(data.scores)) {
            console.log('✅ Loaded ' + data.scores.length + ' score records');
            
            // Merge scores into participant data
            participantListData.forEach(participant => {
                const epId = participant.id || participant.event_participant_id;
                const participantScores = data.scores.filter(s => s.event_participant_id == epId);
                
                console.log('👤 Participant BIB ' + participant.bib_number + ' (ID: ' + epId + '): ' + participantScores.length + ' scores');
                
                if (participantScores.length > 0) {
                    participant.has_score = true;
                    participant.judge_count = participantScores.length;
                    
                    // Calculate average score if multiple judges
                    const totalScore = participantScores.reduce((sum, s) => sum + parseFloat(s.score_value || 0), 0);
                    participant.total_score = (totalScore / participantScores.length).toFixed(2);
                    
                    // Store first judge's data as representative
                    const firstScore = participantScores[0];
                    participant.figures_json = firstScore.figures_json;
                    participant.status = firstScore.status;
                    participant.run_number = runNumber;
                } else {
                    participant.has_score = false;
                    participant.judge_count = 0;
                    participant.total_score = null;
                }
            });
            
            renderParticipantList();
        } else {
            console.warn('⚠️ No scores found for this heat/run');
            console.log('Response data:', data);
            // Mark all as no score
            participantListData.forEach(participant => {
                participant.has_score = false;
                participant.total_score = null;
            });
            renderParticipantList();
        }
    } catch (error) {
        console.error('❌ Error loading scores:', error);
    }
}

function populateCategoryFilter() {
    const categoryFilter = document.getElementById('categoryFilter');
    if (!categoryFilter) return;
    
    // Get unique categories
    const categories = [...new Set(participantListData
        .map(p => p.display_category || p.event_category || p.participant_category)
        .filter(c => c))];
    
    categoryFilter.innerHTML = '<option value="">All Categories</option>';
    categories.forEach(cat => {
        const option = document.createElement('option');
        option.value = cat;
        option.textContent = cat;
        categoryFilter.appendChild(option);
    });
}

function renderParticipantList() {
    const grid = document.getElementById('participantCardsGrid');
    const categoryFilter = document.getElementById('categoryFilter');
    const scoredFilter = document.getElementById('scoredFilter');
    const sortOrder = document.getElementById('sortOrder');
    
    if (!participantListData || participantListData.length === 0) {
        grid.innerHTML = `
            <div class="col-12 text-center text-muted py-3">
                <i class="fas fa-users fa-2x mb-2 opacity-25"></i>
                <p class="mb-0">No participants in this heat</p>
            </div>
        `;
        return;
    }
    
    // Filter by category
    let filtered = participantListData;
    const selectedCategory = categoryFilter?.value;
    if (selectedCategory) {
        filtered = filtered.filter(p => 
            (p.display_category === selectedCategory) ||
            (p.event_category === selectedCategory) ||
            (p.participant_category === selectedCategory)
        );
    }
    
    // Filter by scored status
    const scoredFilterValue = scoredFilter?.value || 'all';
    let unscored = [];
    let scored = [];
    
    if (scoredFilterValue === 'unscored') {
        // Only show unscored participants
        filtered = filtered.filter(p => !p.has_score);
    } else if (scoredFilterValue === 'scored_bottom') {
        // Separate scored and unscored
        unscored = filtered.filter(p => !p.has_score);
        scored = filtered.filter(p => p.has_score);
    }
    
    // Sort
    const sortBy = sortOrder?.value || 'bib';
    const sortFunction = (a, b) => {
        switch (sortBy) {
            case 'bib':
                return parseInt(a.bib_number) - parseInt(b.bib_number);
            case 'sort_order':
                const result = (a.sort_order || 0) - (b.sort_order || 0);
                return result;
            case 'name':
                const nameA = `${a.display_first_name} ${a.display_last_name}`.toLowerCase();
                const nameB = `${b.display_first_name} ${b.display_last_name}`.toLowerCase();
                return nameA.localeCompare(nameB);
            case 'score_desc':
                return (b.total_score || 0) - (a.total_score || 0);
            default:
                return 0;
        }
    };
    
    if (scoredFilterValue === 'scored_bottom') {
        // Sort both groups independently
        unscored.sort(sortFunction);
        scored.sort(sortFunction);
        // Combine: unscored first, then scored
        filtered = [...unscored, ...scored];
    } else {
        filtered.sort(sortFunction);
    }
    
    // Debug: Log the sorted order
    if (sortBy === 'sort_order') {
        console.log('📋 Rendering order (sort_order):');
        filtered.slice(0, 10).forEach((p, idx) => {
            console.log(`  ${idx + 1}. BIB ${p.bib_number} - sort_order: ${p.sort_order}`);
        });
    }
    
    // Render cards
    if (filtered.length === 0) {
        grid.innerHTML = `
            <div class="col-12 text-center text-muted py-3">
                <i class="fas fa-filter fa-2x mb-2 opacity-25"></i>
                <p class="mb-0">No participants match the selected filter</p>
            </div>
        `;
        return;
    }
    
    // Build HTML with optional separator
    let html = '';
    if (scoredFilterValue === 'scored_bottom' && unscored.length > 0 && scored.length > 0) {
        // Render unscored participants
        html += unscored.map(participant => renderParticipantCard(participant)).join('');
        
        // Add separator
        html += `
            <div class="col-12">
                <div class="border-top border-2 border-success my-3 position-relative">
                    <span class="badge bg-success position-absolute top-50 start-50 translate-middle px-3">
                        <i class="fas fa-check-circle me-1"></i>
                        Already Scored (${scored.length})
                    </span>
                </div>
            </div>
        `;
        
        // Render scored participants
        html += scored.map(participant => renderParticipantCard(participant)).join('');
    } else {
        html = filtered.map(participant => renderParticipantCard(participant)).join('');
    }
    
    grid.innerHTML = html;
}

function renderParticipantCard(participant) {
    const name = `${participant.display_first_name || ''} ${participant.display_last_name || ''}`.trim() || 'Unnamed';
    const category = participant.display_category || participant.event_category || participant.participant_category || 'N/A';
    const country = participant.country || '';
    const club = participant.club || '';
    
    const hasScore = participant.has_score && participant.total_score !== null;
    const score = hasScore ? parseFloat(participant.total_score).toFixed(2) : '--';
    const judgeCount = participant.judge_count || 0;
    
    const statusClass = participant.status === 'current' ? 'performing' : (hasScore ? 'has-score' : '');
    
    let statusBadge = '';
    if (participant.status === 'current') {
        statusBadge = '<span class="status-badge badge bg-warning text-dark">Performing</span>';
    } else if (hasScore) {
        statusBadge = '<span class="status-badge badge bg-success">Scored (' + judgeCount + ' judges)</span>';
    }
    
    let figuresHTML = '';
    if (participant.figures_json) {
        try {
            const figures = typeof participant.figures_json === 'string' 
                ? JSON.parse(participant.figures_json) 
                : participant.figures_json;
            
            if (figures && typeof figures === 'object') {
                figuresHTML = '<div class="criteria-scores">';
                
                // Check if figures has a 'figures' property (nested structure)
                const figureData = figures.figures || figures;
                
                for (const [key, value] of Object.entries(figureData)) {
                    if (key !== 'validation' && value !== null && value !== '') {
                        const displayValue = Array.isArray(value) ? value.join(', ') : value;
                        figuresHTML += '<div class="criteria-score-item"><strong>' + key + ':</strong> ' + displayValue + '</div>';
                    }
                }
                figuresHTML += '</div>';
            }
        } catch (e) {
            console.error('Error parsing figures:', e);
        }
    }
    
    return `
        <div class="col-md-12">
            <div class="participant-score-card ${statusClass}">
                ${statusBadge}
                
                <div class="d-flex align-items-center gap-3">
                    <div class="participant-bib-badge">
                        ${participant.bib_number}
                    </div>
                    
                    <div class="participant-info flex-grow-1">
                        <div class="participant-name">${name}</div>
                        <div class="participant-details">
                            <div><i class="fas fa-tag me-1"></i>${category}</div>
                            ${country ? `<div><i class="fas fa-flag me-1"></i>${country}</div>` : ''}
                            ${club ? `<div><i class="fas fa-building me-1"></i>${club}</div>` : ''}
                        </div>
                    </div>
                    
                    <div class="score-display">
                        <div class="score-value">${score}</div>
                        <div class="score-label">${hasScore && judgeCount > 1 ? 'Avg (' + judgeCount + ' judges)' : 'Score'}</div>
                    </div>
                    
                    <div class="position-controls d-flex flex-column gap-1">
                        <button class="btn btn-sm btn-outline-primary" onclick="moveParticipantUp(${participant.event_participant_id})" title="Move up in start order">
                            <i class="fas fa-arrow-up"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-primary" onclick="moveParticipantDown(${participant.event_participant_id})" title="Move down in start order">
                            <i class="fas fa-arrow-down"></i>
                        </button>
                    </div>
                </div>
                
                ${figuresHTML}
            </div>
        </div>
    `;
}

// ============================================================================
// PARTICIPANT POSITION MANAGEMENT
// ============================================================================

async function moveParticipantUp(participantId) {
    // participantId is actually event_participant_id
    const participant = participantListData.find(p => p.event_participant_id === participantId);
    if (!participant) {
        showErrorToast('Error', 'Participant not found');
        return;
    }
    
    console.log('⬆️ Moving up - Selected participant ID:', participantId, 'BIB:', participant.bib_number);
    
    // Find the participant above (with next lower sort_order)
    const sortedList = [...participantListData].sort((a, b) => (a.sort_order || 0) - (b.sort_order || 0));
    const currentIndex = sortedList.findIndex(p => p.event_participant_id === participantId);
    
    console.log('  Current index in sorted list:', currentIndex);
    console.log('  Total participants:', sortedList.length);
    
    if (currentIndex <= 0) {
        showInfoToast('Info', 'Participant is already at the top');
        return;
    }
    
    const prevParticipant = sortedList[currentIndex - 1];
    console.log('  Previous participant ID:', prevParticipant.event_participant_id, 'BIB:', prevParticipant.bib_number);
    
    if (participant.event_participant_id === prevParticipant.event_participant_id) {
        console.error('❌ ERROR: Trying to swap participant with itself!');
        showErrorToast('Error', 'Cannot swap participant with itself');
        return;
    }
    
    await swapParticipantPositions(participant, prevParticipant);
}

async function moveParticipantDown(participantId) {
    // participantId is actually event_participant_id
    const participant = participantListData.find(p => p.event_participant_id === participantId);
    if (!participant) {
        showErrorToast('Error', 'Participant not found');
        return;
    }
    
    console.log('⬇️ Moving down - Selected participant ID:', participantId, 'BIB:', participant.bib_number);
    
    // Find the participant below (with next higher sort_order)
    const sortedList = [...participantListData].sort((a, b) => (a.sort_order || 0) - (b.sort_order || 0));
    const currentIndex = sortedList.findIndex(p => p.event_participant_id === participantId);
    
    console.log('  Current index in sorted list:', currentIndex);
    console.log('  Total participants:', sortedList.length);
    
    if (currentIndex >= sortedList.length - 1) {
        showInfoToast('Info', 'Participant is already at the bottom');
        return;
    }
    
    const nextParticipant = sortedList[currentIndex + 1];
    console.log('  Next participant ID:', nextParticipant.event_participant_id, 'BIB:', nextParticipant.bib_number);
    
    if (participant.event_participant_id === nextParticipant.event_participant_id) {
        console.error('❌ ERROR: Trying to swap participant with itself!');
        showErrorToast('Error', 'Cannot swap participant with itself');
        return;
    }
    
    await swapParticipantPositions(participant, nextParticipant);
}

async function swapParticipantPositions(participant1, participant2) {
    try {
        console.log('🔄 Swapping positions:');
        console.log('  Participant 1 ID:', participant1.event_participant_id, 'BIB:', participant1.bib_number, 'current sort_order:', participant1.sort_order);
        console.log('  Participant 2 ID:', participant2.event_participant_id, 'BIB:', participant2.bib_number, 'current sort_order:', participant2.sort_order);
        
        // Store original values
        const originalOrder1 = participant1.sort_order;
        const originalOrder2 = participant2.sort_order;
        
        // Update in database - send the NEW order for each participant (swapped)
        const formData = new FormData();
        formData.append('action', 'swap_positions');
        formData.append('event_id', currentEventId);
        formData.append('participant1_id', participant1.event_participant_id);
        formData.append('participant2_id', participant2.event_participant_id);
        formData.append('participant1_order', originalOrder2); // participant1 gets participant2's order
        formData.append('participant2_order', originalOrder1); // participant2 gets participant1's order
        
        const response = await fetch('heat_management_api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        console.log('📥 Swap result:', result);
        
        if (result.success) {
            // Update local data with new values
            participant1.sort_order = originalOrder2;
            participant2.sort_order = originalOrder1;
            
            console.log('✅ Updated local data:');
            console.log('  Participant 1:', participant1.bib_number, 'new sort_order:', participant1.sort_order);
            console.log('  Participant 2:', participant2.bib_number, 'new sort_order:', participant2.sort_order);
            
            // Re-render the list with updated order
            renderParticipantList();
            showSuccessToast('Success', 'Position updated');
        } else {
            showErrorToast('Error', result.message || 'Failed to update position');
        }
    } catch (error) {
        console.error('❌ Error swapping positions:', error);
        showErrorToast('Error', 'Failed to update position');
    }
}

// ============================================================================
// RESULTS MODAL FUNCTIONS
// ============================================================================

function showHeatResults(heatNumber) {
    if (!currentEventId) {
        showErrorToast('❌ Error', 'No event ID available');
        return;
    }
    
    // Update modal title with heat number
    document.getElementById('resultsHeatName').textContent = `Heat ${heatNumber} Results`;
    
    // Show the modal
    const modal = new bootstrap.Modal(document.getElementById('heatResultsModal'));
    modal.show();
    
    // Load configurations for this heat
    loadHeatConfigurations(currentEventId, heatNumber);
}

function loadHeatConfigurations(eventId, heatNumber) {
    const configsList = document.getElementById('savedConfigsList');
    
    // Show loading state
    configsList.innerHTML = `
        <div class="text-center text-muted py-3">
            <i class="fas fa-spinner fa-spin"></i>
            <p class="small mb-0 mt-2">Loading reports...</p>
        </div>
    `;
    
    // Fetch all configurations for the event
    fetch(`../api/public_dashboard_api.php?action=get_configurations&event_id=${eventId}`)
        .then(response => response.json())
        .then(data => {
            if (!data.success || !data.configurations || data.configurations.length === 0) {
                configsList.innerHTML = `
                    <div class="text-center text-muted py-3">
                        <i class="fas fa-info-circle fa-2x mb-2"></i>
                        <p class="small mb-0">No reports configured</p>
                    </div>
                `;
                return;
            }
            
            // Filter configurations for this heat or general (null heat_number)
            const relevantConfigs = data.configurations.filter(config => 
                config.heat_number === null || config.heat_number == heatNumber
            );
            
            if (relevantConfigs.length === 0) {
                configsList.innerHTML = `
                    <div class="text-center text-muted py-3">
                        <i class="fas fa-info-circle fa-2x mb-2"></i>
                        <p class="small mb-0">No reports for this heat</p>
                    </div>
                `;
                return;
            }
            
            // Render configuration buttons
            configsList.innerHTML = relevantConfigs.map(config => {
                const icon = config.view_type === 'start_list' 
                    ? 'fa-list-ol' 
                    : 'fa-trophy';
                
                const label = config.view_type === 'start_list' 
                    ? 'Start List' 
                    : 'Results';
                
                const heatLabel = config.heat_number 
                    ? `Heat ${config.heat_number}` 
                    : 'All Heats';
                
                return `
                    <button class="config-button btn btn-outline-primary text-start" 
                            data-config-id="${config.config_id}"
                            data-view-type="${config.view_type}"
                            onclick="loadHeatConfiguration(${config.config_id}, '${config.view_type}')">
                        <i class="fas ${icon} me-2"></i>
                        <div>
                            <div class="fw-bold">${config.config_name || label}</div>
                            <small class="text-muted">${heatLabel}</small>
                        </div>
                    </button>
                `;
            }).join('');
            
            // Auto-load first configuration
            if (relevantConfigs.length > 0) {
                loadHeatConfiguration(relevantConfigs[0].config_id, relevantConfigs[0].view_type);
            }
        })
        .catch(error => {
            console.error('Error loading configurations:', error);
            configsList.innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Error loading reports
                </div>
            `;
        });
}

function loadHeatConfiguration(configId, viewType) {
    const contentDisplay = document.getElementById('resultsContentDisplay');
    
    // Update active state
    document.querySelectorAll('#savedConfigsList .config-button').forEach(btn => {
        btn.classList.remove('active');
    });
    const activeBtn = document.querySelector(`[data-config-id="${configId}"]`);
    if (activeBtn) activeBtn.classList.add('active');
    
    // Show loading
    contentDisplay.innerHTML = `
        <div class="text-center text-muted py-5">
            <i class="fas fa-spinner fa-spin fa-3x mb-3"></i>
            <p class="mb-0">Loading report...</p>
        </div>
    `;
    
    // Determine API endpoint based on view type
    const apiEndpoint = viewType === 'start_list' 
        ? '../api/public_dashboard_api.php?action=get_start_list'
        : '../api/summary_table_api.php';
    
    // Load configuration
    fetch(`${apiEndpoint}&config_id=${configId}&format=html`)
        .then(response => response.text())
        .then(html => {
            contentDisplay.innerHTML = html;
        })
        .catch(error => {
            console.error('Error loading configuration:', error);
            contentDisplay.innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error loading report:</strong> ${error.message}
                </div>
            `;
        });
}

// ============================================================================
// HEAT CARD ACTION FUNCTIONS (called from heat card buttons)
// ============================================================================

async function changeActiveRun(heatNumber, direction) {
    showInfoToast('🔄 Changing Run', `Switching run for Heat ${heatNumber}...`);
    // This would need integration with heat management API
    console.log('Change run:', heatNumber, direction);
}

async function activateHeat(heatNumber) {
    showInfoToast('🔥 Activating Heat', `Setting Heat ${heatNumber} as active...`);
    currentHeatNumber = heatNumber;
    loadActiveHeatCard(heatNumber);
}

async function closeHeat(heatNumber) {
    if (!confirm('Mark this heat as FINISHED?')) return;
    showInfoToast('✅ Closing Heat', `Marking Heat ${heatNumber} as finished...`);
}

async function setPendingHeat(heatNumber) {
    showInfoToast('⏳ Setting Pending', `Marking Heat ${heatNumber} as pending...`);
}

async function cancelHeat(heatNumber) {
    if (!confirm('CANCEL this heat?')) return;
    showWarningToast('❌ Cancelling Heat', `Heat ${heatNumber} cancelled`);
}

async function rescheduleHeat(heatNumber) {
    showInfoToast('📅 Rescheduling', `Heat ${heatNumber} marked for rescheduling`);
}

function openHeatModal(heatNumber) {
    window.location.href = `heats_configure.php?event_id=${currentEventId}&heat=${heatNumber}`;
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

function showSuccessToast(title, message = '') {
    createToast('success', title, message);
}

function showErrorToast(title, message = '') {
    createToast('danger', title, message);
}

function showInfoToast(title, message = '') {
    createToast('info', title, message);
}

function showWarningToast(title, message = '') {
    createToast('warning', title, message);
}

function createToast(type, title, message) {
    // Create toast container if it doesn't exist
    let container = document.getElementById('toastContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toastContainer';
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        document.body.appendChild(container);
    }
    
    // Create toast element
    const toastId = 'toast_' + Date.now();
    const toast = document.createElement('div');
    toast.id = toastId;
    toast.className = `toast notification-toast align-items-center text-white bg-${type} border-0`;
    toast.setAttribute('role', 'alert');
    
    const iconMap = {
        success: 'fa-check-circle',
        danger: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">
                <i class="fas ${iconMap[type]} me-2"></i>
                <strong>${title}</strong>
                ${message ? `<div class="mt-1 small">${message}</div>` : ''}
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;
    
    container.appendChild(toast);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        toast.remove();
    }, 5000);
    
    // Animate in
    toast.style.opacity = '0';
    toast.style.transform = 'translateX(100%)';
    toast.style.transition = 'all 0.3s ease';
    
    setTimeout(() => {
        toast.style.opacity = '1';
        toast.style.transform = 'translateX(0)';
    }, 10);
}
</script>

<!-- Bootstrap JS Bundle (required for modals) -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<?php if ($standalone): ?>
    </div>
    <?php include '../admin/footer.php'; ?>
</body>
</html>
<?php endif; ?>
