# Judge Status Display in Heat Cards

## Feature Overview
Added real-time judge status monitoring to the **active heat card only** in the admin panel. The active heat card displays:
- Assigned judges working on that heat
- Current status of each judge (Ready, Pending, Danger, Warning, Info, Off)
- Judge's current location (Heat/Run number)
- Time since last status update
- Head judge indicator
- Status summary with badge counts

**Note:** Judge status is only displayed for the currently active heat to reduce clutter and focus attention on the heat that's actively being scored.

## Implementation Details

### Modified Files

#### 1. `admin/heat_cards_api.php`
**Changes:**
- Added query to fetch judge assignments with status information
- Changed JOIN from `judges` table to `users` table (correct schema)
- Filters judges by assigned heat number or shows all if no assignments
- Generates judge status section for each heat card

**Key SQL:**
```php
SELECT 
    ja.id,
    ja.event_id,
    ja.judge_id,
    ja.status,
    ja.status_updated_at,
    ja.heat_number as assigned_heat_number,
    ja.run_number as assigned_run_number,
    u.username as judge_name,
    CASE WHEN u.role = 'head_judge' THEN 1 ELSE 0 END as is_head
FROM judge_assignments ja
JOIN users u ON ja.judge_id = u.id
WHERE ja.event_id = ?
```

**Display Logic:**
- Judge status section **only appears for the active heat** (status = 'active')
- Filters judges per heat: `assigned_heat_number == $h OR assigned_heat_number === null`
- Falls back to showing all event judges if no heat-specific assignments exist
- Counts judges by status for summary badges
- Highlights judges assigned to current heat with `bg-light` background

### Why Only Active Heat?
- **Reduced Visual Clutter**: Prevents information overload across all heat cards
- **Focused Monitoring**: Admins concentrate on judges scoring the current heat
- **Better Performance**: Less HTML to render and update
- **Clear Status**: Easy to see which heat is being monitored

### Visual Features

#### Status Colors & Icons
| Status  | Color     | Icon | Badge       |
|---------|-----------|------|-------------|
| ready   | Green     | ✓    | bg-success  |
| pending | Yellow    | ⏸    | bg-warning  |
| danger  | Red       | ⚠    | bg-danger   |
| warning | Yellow    | ⚠    | bg-warning  |
| info    | Blue      | ℹ    | bg-info     |
| off     | Gray      | ●    | bg-secondary|

#### Judge Card Layout
```
┌─────────────────────────────────┐
│ 👔 Judges (2)      ✓1 ℹ1       │
├─────────────────────────────────┤
│ ℹ admin              1h ago    │
│ ✓ judge1    H2R1     1h ago    │ ← Highlighted (assigned to this heat)
└─────────────────────────────────┘
```

#### Information Display
- **Judge Name**: Truncated with tooltip for full name
- **Head Judge**: ⭐ star icon before name
- **Location Badge**: Shows Heat/Run (e.g., "H2R1" or "Heat 2")
- **Time Ago**: Relative time since last status update
  - "now" - Less than 60 seconds
  - "Xm ago" - Minutes
  - "Xh ago" - Hours

#### Status Summary Badges
At top-right of judge section:
- Only shows statuses with count > 0
- Compact display with emoji + count
- Helps identify issues at a glance (e.g., "⚠ 2" shows 2 judges in danger state)

### Auto-Refresh Integration
Heat cards auto-refresh based on interval selector:
- Manual (no refresh)
- 5 seconds
- 10 seconds
- 30 seconds (default)

Judge status updates appear automatically on each refresh cycle without page reload.

### User Experience

#### For Event Coordinators
- **Quick Status Overview**: See all judges at a glance across all heats
- **Problem Detection**: Red/yellow badges immediately show issues
- **Resource Allocation**: See which judges are working on which heats/runs
- **Head Judge Identification**: Star icons highlight head judges
- **Real-time Monitoring**: Auto-refresh keeps status current

#### For Head Judges
- **Team Status**: Monitor judge readiness across event
- **Location Tracking**: Know exactly which heat/run each judge is scoring
- **Response Time**: Time stamps show recent activity
- **Capacity Planning**: See judge distribution across heats

### Performance Considerations
- Single SQL query fetches all judge data per event
- Client-side filtering by heat number (no additional queries)
- Efficient status counting in PHP (no extra database hits)
- HTML generation cached until next refresh cycle
- Max height: 120px with scroll for many judges

### Database Schema Requirements
**judge_assignments table:**
- `status` - Current judge status
- `status_updated_at` - Timestamp of last status change
- `heat_number` - Heat judge is working on
- `run_number` - Run judge is working on

**users table:**
- `username` - Display name for judge
- `role` - Determines if head_judge ('head_judge' or 'judge')

### Testing Results
**Event 6 (SNOWFEST Festival):**
- 8 heats configured
- 2 judges assigned
- Heat 1: 1 judge (admin, info status)
- Heat 2: 2 judges (admin + judge1)
  - admin: Info status, no specific location
  - judge1: Ready status, assigned to H2R1
- All other heats: Show both judges (fallback)

**API Response:**
- Success: true
- Heat count: 8
- Judge section: Present in all cards
- Judge counts: 1, 2, 1, 1, 1, 1, 1, 1 per heat

## Usage

### For Administrators
1. Navigate to **Admin → Configure Heats**
2. Select event from dropdown
3. Heat cards load with judge status sections
4. Monitor judge activity in real-time
5. Use auto-refresh to keep data current

### Interpreting Status
- **✓ Ready**: Judge is available and working normally
- **⏸ Pending**: Judge is paused or waiting
- **⚠ Danger**: Critical issue, judge needs immediate attention
- **⚠ Warning**: Non-critical issue, judge flagged something
- **ℹ Info**: Judge provided informational update
- **● Off**: Judge is offline/inactive

### Location Badges
- **H2R1**: Heat 2, Run 1 (judge actively scoring)
- **Heat 2**: Heat 2 assigned, no specific run
- *(no badge)*: Judge not assigned to specific heat yet

## Future Enhancements
- [ ] Click judge name to view detailed status history
- [ ] Filter heats by judge availability
- [ ] Send notifications when judge sets danger/warning status
- [ ] Judge performance metrics (avg response time, scores per hour)
- [ ] Batch status updates for all judges
- [ ] Judge communication panel (send messages from admin)

## Related Files
- `api/judge_status_api.php` - Backend API for status management
- `judge/score.php` - Judge scoring panel with status selector
- `admin/admin_heat_config.php` - Main heat configuration page
- `admin/heat_cards_api.php` - AJAX endpoint for card generation
- Database migrations:
  - `add_judge_status_migration.php`
  - `add_run_number_column.php`
