<?php
session_start();
include 'includes/db.php';
require_once __DIR__ . '/includes/dompdf/autoload.inc.php';
use Dompdf\Dompdf;

$selected_event  = $_GET['event_id']  ?? 0;
$filter_category = $_GET['category']  ?? 'all';
$filter_gender   = $_GET['gender']    ?? 'all';
$sort_by         = $_GET['sort']      ?? 'none';
$export          = $_GET['export']   ?? 'none';


$events = $pdo
    ->query("SELECT id, name FROM events ORDER BY date DESC")
    ->fetchAll(PDO::FETCH_ASSOC);

$categories = $pdo
    ->query("SELECT DISTINCT category FROM participants ORDER BY category")
    ->fetchAll(PDO::FETCH_COLUMN);

// Holders for the summary data
$summary_data = [];
$heat_numbers = [];

if ($selected_event) {
    $categories_to_use = $filter_category === 'all'
        ? $categories
        : [ $filter_category ];

    foreach ($categories_to_use as $cat) {
        $sql = "
            SELECT
                p.category,
                p.bip,
                p.first_name,
                p.last_name,
                p.gender,
                p.club,
                ep.heat_number,
                r.run_number,
                AVG(s.score_value) AS avg_score
            FROM runs r
            JOIN event_participants ep
              ON r.event_participant_id = ep.id
            JOIN participants p
              ON ep.participant_id = p.id
            LEFT JOIN scores s
              ON r.id = s.run_id
            WHERE ep.event_id = ?
              AND p.category = ?
            GROUP BY p.id, ep.heat_number, r.run_number
            ORDER BY ep.heat_number, r.run_number
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([ $selected_event, $cat ]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($rows as $row) {
            if ($filter_gender !== 'all'
             && $row['gender'] !== $filter_gender) {
                continue;
            }

            $key   = "{$row['bip']} - {$row['first_name']} {$row['last_name']}";
            $score = floatval($row['avg_score']);
            $heat  = $row['heat_number'];
            $run   = $row['run_number'];
            $code  = "H{$heat}R{$run}";

            $summary_data[$cat][$key]['club']       = $row['club']   ?? '-';
            $summary_data[$cat][$key]['gender']     = $row['gender'];
            $summary_data[$cat][$key][$code]        = number_format($score, 2);
            $summary_data[$cat][$key]['raw'][$code] = $score;
            $summary_data[$cat][$key]['all_scores'][] = $score;

            $heat_numbers[$code] = true;
        }
    }

    // Compute best and average
    foreach ($summary_data as $cat => &$participants) {
        foreach ($participants as &$p) {
            for ($h = 1; $h <= 2; $h++) {
                $r1 = $p['raw']["H{$h}R1"] ?? null;
                $r2 = $p['raw']["H{$h}R2"] ?? null;
                $scores = array_filter([$r1, $r2], fn($v) => $v !== null);
                if (!empty($scores)) {
                    $p["H{$h}Best"]    = number_format(max($scores), 2);
                    $p["H{$h}Average"] = number_format(array_sum($scores)/count($scores), 2);
                }
            }
            $p["OverallBest"] = number_format(max($p['all_scores'] ?? [0]), 2);
            $heat_avgs = array_filter([
                floatval($p["H1Average"] ?? 0),
                floatval($p["H2Average"] ?? 0)
            ]);
            $p["HighestAverage"] = $heat_avgs
                ? number_format(max($heat_avgs), 2)
                : '-';
        }
    }
}

// Build header list and insert derived columns
$headers = array_keys($heat_numbers);
sort($headers);
$final_headers = [];
foreach ($headers as $h) {
    $final_headers[] = $h;
    if ($h === 'H1R2') {
        $final_headers[] = 'H1Best';
        $final_headers[] = 'H1Average';
    }
    if ($h === 'H2R2') {
        $final_headers[] = 'H2Best';
        $final_headers[] = 'H2Average';
    }
}
$final_headers[] = 'OverallBest';
$final_headers[] = 'HighestAverage';
?>
<?php
// 3) Handle PDF export via Dompdf
if ($export === 'pdf' && $selected_event && $filter_category !== 'all') {
    // Build the same HTML table fragment
    ob_start(); ?>
    <h2>Category: <?= htmlspecialchars($filter_category) ?></h2>
    <table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse;">
      <thead>
        <tr>
          <th>Rank</th><th>Participant</th><th>Club</th><th>Gender</th>
          <?php foreach ($final_headers as $h): ?>
            <th><?= htmlspecialchars($h) ?></th>
          <?php endforeach; ?>
        </tr>
      </thead>
      <tbody>
        <?php
        $parts = $summary_data[$filter_category];
        if ($sort_by !== 'none') {
            uasort($parts, fn($a,$b) => floatval($b[$sort_by]??0) <=> floatval($a[$sort_by]??0));
        }
        $rnk = 1;
        foreach ($parts as $nm => $vals): ?>
          <tr>
            <td><?= $rnk++ ?></td>
            <td><?= htmlspecialchars($nm) ?></td>
            <td><?= htmlspecialchars($vals['club']   ?? '-') ?></td>
            <td><?= htmlspecialchars($vals['gender'] ?? '-') ?></td>
            <?php foreach ($final_headers as $h): ?>
              <td><?= htmlspecialchars($vals[$h] ?? '-') ?></td>
            <?php endforeach; ?>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
    <?php
    $html = ob_get_clean();
    $dompdf = new Dompdf();
    $dompdf->loadHtml($html);
    $dompdf->setPaper('A4','landscape');
    $dompdf->render();
    $dompdf->stream("summary.pdf", ["Attachment" => true]);
    exit;
}
?>


<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>View 3 - Summary Per Participant</title>
    <link
      href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"
      rel="stylesheet"
    >
    <style>
        .table th, .table td { white-space: nowrap; }
    </style>
</head>
<body class="p-4 bg-light">
<div class="container">
    <h2>View 3: Summary Table</h2>
<!-- Export Buttons -->
      <?php $base = $_SERVER['PHP_SELF'].'?'.http_build_query([
        'event_id'=>$selected_event,
        'category'=>$filter_category,
        'gender'=>$filter_gender,
        'sort'=>$sort_by
      ]); ?>
      <div class="col-md-6 text-end">
        <a href="<?= $base.'&export=csv' ?>"
           class="btn btn-outline-primary me-2">Export CSV</a>
        <a href="<?= $base.'&export=pdf' ?>"
           class="btn btn-outline-primary">Export PDF</a>
      </div>

    <!-- Filters & Sorting -->
    <form method="get" class="row g-3 mb-4">
        <div class="col-md-3">
            <label>Select Event</label>
            <select name="event_id" class="form-select" onchange="this.form.submit()">
                <option value="">-- Choose Event --</option>
                <?php foreach ($events as $ev): ?>
                <option
                  value="<?= $ev['id'] ?>"
                  <?= $selected_event == $ev['id'] ? 'selected' : '' ?>>
                  <?= htmlspecialchars($ev['name']) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-3">
            <label>Category</label>
            <select name="category" class="form-select" onchange="this.form.submit()">
                <option value="all">All Categories</option>
                <?php foreach ($categories as $cat): ?>
                <option
                  value="<?= $cat ?>"
                  <?= $filter_category == $cat ? 'selected' : '' ?>>
                  <?= htmlspecialchars($cat) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-3">
            <label>Gender</label>
            <select name="gender" class="form-select" onchange="this.form.submit()">
                <option value="all" <?= $filter_gender=='all'?'selected':'' ?>>All</option>
                <option value="M"   <?= $filter_gender=='M'  ?'selected':'' ?>>Male</option>
                <option value="F"   <?= $filter_gender=='F'  ?'selected':'' ?>>Female</option>
                <option value="OPEN"<?= $filter_gender=='OPEN'?'selected':'' ?>>Open</option>
            </select>
        </div>
        <div class="col-md-3">
            <label>Sort by</label>
            <select name="sort" class="form-select" onchange="this.form.submit()">
                <option value="none"           <?= $sort_by==='none'           ?'selected':'' ?>>None</option>
                <option value="OverallBest"    <?= $sort_by==='OverallBest'    ?'selected':'' ?>>Overall Best</option>
                <option value="HighestAverage" <?= $sort_by==='HighestAverage' ?'selected':'' ?>>Highest Average</option>
            </select>
        </div>
    </form>

    <?php if (
        $selected_event
        && $filter_category !== 'all'
        && isset($summary_data[$filter_category])
    ): ?>
    <?php
        $participants = $summary_data[$filter_category];
        if ($sort_by !== 'none') {
            uasort($participants, function($a, $b) use ($sort_by) {
                return floatval($b[$sort_by] ?? 0)
                     <=> floatval($a[$sort_by] ?? 0);
            });
        }
    ?>

    <!-- Column Toggles -->
    <div class="mb-3">
        <label class="form-label">Toggle Columns:</label><br>
        <div class="form-check form-check-inline">
            <input
              type="checkbox"
              class="form-check-input col-toggle"
              data-col="Club"
              checked
            >
            <label class="form-check-label">Club</label>
        </div>
        <?php foreach ($final_headers as $h): ?>
        <div class="form-check form-check-inline">
            <input
              type="checkbox"
              class="form-check-input col-toggle"
              data-col="<?= $h ?>"
              checked
            >
            <label class="form-check-label"><?= $h ?></label>
        </div>
        <?php endforeach; ?>
    </div>

    <!-- Summary Table -->
    <h4 class="mt-4"><?= htmlspecialchars($filter_category) ?></h4>
    <table class="table table-bordered bg-white">
        <thead>
            <tr>
                <th>Rank</th>
                <th>Participant</th>
                <th>Club</th>
                <th>Gender</th>
                <?php foreach ($final_headers as $h): ?>
                <th><?= $h ?></th>
                <?php endforeach; ?>
            </tr>
        </thead>
        <tbody>
            <?php $rank = 1; ?>
            <?php foreach ($participants as $name => $runs): ?>
            <tr>
                <td><?= $rank++ ?></td>
                <td><?= htmlspecialchars($name) ?></td>
                <td><?= htmlspecialchars($runs['club']   ?? '-') ?></td>
                <td><?= htmlspecialchars($runs['gender'] ?? '-') ?></td>
                <?php foreach ($final_headers as $h): ?>
                <td><?= $runs[$h] ?? '-' ?></td>
                <?php endforeach; ?>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php endif; ?>
</div>

<script>
(function() {
    const KEY = 'columnVisibility';
    // Read stored prefs or default all true
    let prefs = JSON.parse(localStorage.getItem(KEY) || '{}');

    function applyVisibility(colName, visible) {
        const tbl = document.querySelector('table');
        if (!tbl) return;

        // find header index
        const headers = tbl.querySelectorAll('thead th');
        let idx = -1;
        headers.forEach((th,i) => {
            if (th.textContent.trim() === colName) idx = i;
        });
        if (idx < 0) return;

        // toggle header
        headers[idx].style.display = visible ? '' : 'none';

        // toggle body cells
        tbl.querySelectorAll('tbody tr').forEach(row => {
            const cell = row.querySelectorAll('td')[idx];
            if (cell) cell.style.display = visible ? '' : 'none';
        });
    }

    // initialize checkboxes & table
    document.querySelectorAll('.col-toggle').forEach(cb => {
        const col = cb.dataset.col;
        const vis = prefs[col] !== undefined ? prefs[col] : true;
        cb.checked = vis;
        applyVisibility(col, vis);
        cb.addEventListener('change', () => {
            prefs[col] = cb.checked;
            localStorage.setItem(KEY, JSON.stringify(prefs));
            applyVisibility(col, cb.checked);
        });
    });
})();
</script>
</body>
</html>