<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TV Overlay Display - StyleScore</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        html, body {
            width: 100vw;
            height: 100vh;
            background: transparent;
            font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
            overflow: hidden;
        }
        
        :root {
            --lime: #39FF14;
            --blue: #4B7BFF;
            --violet: #AD26FF;
            --white: #FFFFFF;
            --dark: #0B1220;
            --grad: linear-gradient(135deg, var(--lime), var(--blue) 45%, var(--violet));
        }
        
        .overlay-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 1000;
        }
        
        /* Lower Third Graphics */
        .lower-third {
            position: absolute;
            bottom: 60px;
            left: 40px;
            background: linear-gradient(90deg, rgba(11,18,32,0.95), rgba(11,18,32,0.85));
            backdrop-filter: blur(10px);
            border-radius: 16px;
            border: 2px solid rgba(255,255,255,0.1);
            padding: 20px 30px;
            min-width: 400px;
            max-width: 600px;
            color: white;
            box-shadow: 0 20px 40px rgba(0,0,0,0.5);
            transform: translateX(-100%);
            transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .lower-third.show {
            transform: translateX(0);
        }
        
        .lower-third.position-top-left { top: 40px; left: 40px; bottom: auto; }
        .lower-third.position-top-center { top: 40px; left: 50%; bottom: auto; transform: translateX(-50%); }
        .lower-third.position-top-right { top: 40px; right: 40px; bottom: auto; left: auto; }
        .lower-third.position-center-left { top: 50%; left: 40px; bottom: auto; transform: translateY(-50%); }
        .lower-third.position-center { top: 50%; left: 50%; bottom: auto; transform: translate(-50%, -50%); }
        .lower-third.position-center-right { top: 50%; right: 40px; bottom: auto; left: auto; transform: translateY(-50%); }
        .lower-third.position-bottom-center { bottom: 60px; left: 50%; transform: translateX(-50%); }
        .lower-third.position-bottom-right { bottom: 60px; right: 40px; left: auto; }
        
        .participant-name {
            font-size: 24px;
            font-weight: 700;
            background: var(--grad);
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
            margin-bottom: 8px;
        }
        
        .participant-info {
            display: flex;
            align-items: center;
            gap: 20px;
            margin-bottom: 12px;
        }
        
        .country-flag {
            width: 32px;
            height: 24px;
            border-radius: 4px;
            background: #ccc;
        }
        
        .bib-number {
            background: var(--grad);
            color: var(--dark);
            padding: 4px 12px;
            border-radius: 8px;
            font-weight: 700;
            font-size: 18px;
        }
        
        .scores-container {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .score-item {
            text-align: center;
        }
        
        .score-label {
            font-size: 12px;
            color: rgba(255,255,255,0.7);
            margin-bottom: 2px;
        }
        
        .score-value {
            font-size: 20px;
            font-weight: 700;
            color: var(--lime);
        }
        
        .current-score {
            font-size: 28px;
            color: white;
        }
        
        /* Rankings Table */
        .rankings-table {
            position: absolute;
            top: 40px;
            right: 40px;
            background: linear-gradient(180deg, rgba(11,18,32,0.95), rgba(11,18,32,0.85));
            backdrop-filter: blur(10px);
            border-radius: 16px;
            border: 2px solid rgba(255,255,255,0.1);
            padding: 20px;
            min-width: 350px;
            max-width: 450px;
            color: white;
            box-shadow: 0 20px 40px rgba(0,0,0,0.5);
            transform: translateX(100%);
            transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .rankings-table.show {
            transform: translateX(0);
        }
        
        .table-header {
            background: var(--grad);
            color: var(--dark);
            padding: 12px 16px;
            border-radius: 8px;
            font-weight: 700;
            text-align: center;
            margin-bottom: 16px;
        }
        
        .ranking-row {
            display: flex;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            opacity: 0;
            transform: translateY(20px);
            animation: slideInRow 0.5s ease forwards;
        }
        
        .ranking-row:nth-child(even) {
            animation-delay: 0.1s;
        }
        
        .ranking-row:nth-child(odd) {
            animation-delay: 0.05s;
        }
        
        .rank-position {
            font-size: 18px;
            font-weight: 700;
            width: 40px;
            text-align: center;
        }
        
        .rank-position.first { color: #FFD700; }
        .rank-position.second { color: #C0C0C0; }
        .rank-position.third { color: #CD7F32; }
        
        .rank-info {
            flex: 1;
            margin-left: 12px;
        }
        
        .rank-name {
            font-weight: 600;
            font-size: 16px;
        }
        
        .rank-country {
            font-size: 12px;
            color: rgba(255,255,255,0.7);
        }
        
        .rank-score {
            font-size: 18px;
            font-weight: 700;
            color: var(--lime);
        }
        
        /* Animations */
        @keyframes slideInRow {
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes flipIn {
            0% {
                transform: perspective(400px) rotateY(-90deg);
                opacity: 0;
            }
            100% {
                transform: perspective(400px) rotateY(0deg);
                opacity: 1;
            }
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        @keyframes zoomIn {
            0% {
                transform: scale(0.3);
                opacity: 0;
            }
            100% {
                transform: scale(1);
                opacity: 1;
            }
        }
        
        /* Animation Classes */
        .anim-slide { transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1); }
        .anim-fade { animation: fadeIn 0.5s ease forwards; }
        .anim-flip { animation: flipIn 0.6s ease forwards; }
        .anim-zoom { animation: zoomIn 0.4s ease forwards; }
        
        /* Responsive Scaling */
        @media (max-width: 1920px) {
            .lower-third { font-size: 0.9em; }
            .rankings-table { font-size: 0.9em; }
        }
        
        @media (max-width: 1366px) {
            .lower-third { font-size: 0.8em; }
            .rankings-table { font-size: 0.8em; }
        }
        
        @media (max-width: 1024px) {
            .lower-third { font-size: 0.7em; }
            .rankings-table { font-size: 0.7em; }
        }
        
        /* Custom scaling */
        .scale-50 { transform: scale(0.5); transform-origin: top left; }
        .scale-75 { transform: scale(0.75); transform-origin: top left; }
        .scale-125 { transform: scale(1.25); transform-origin: top left; }
        .scale-150 { transform: scale(1.5); transform-origin: top left; }
        
        /* Top 3 Podium */
        .podium-container {
            position: absolute;
            bottom: 40px;
            left: 50%;
            transform: translateX(-50%);
            display: flex;
            gap: 20px;
            align-items: end;
        }
        
        .podium-step {
            background: linear-gradient(180deg, rgba(11,18,32,0.95), rgba(11,18,32,0.85));
            backdrop-filter: blur(10px);
            border-radius: 16px;
            border: 2px solid rgba(255,255,255,0.1);
            padding: 20px;
            text-align: center;
            color: white;
            min-width: 180px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.5);
        }
        
        .podium-step.first { height: 200px; order: 2; }
        .podium-step.second { height: 160px; order: 1; }
        .podium-step.third { height: 120px; order: 3; }
        
        .podium-number {
            font-size: 36px;
            font-weight: 700;
            margin-bottom: 8px;
        }
        
        .podium-number.first { color: #FFD700; }
        .podium-number.second { color: #C0C0C0; }
        .podium-number.third { color: #CD7F32; }
        
        .podium-name {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 4px;
        }
        
        .podium-score {
            font-size: 20px;
            font-weight: 700;
            color: var(--lime);
        }
    </style>
</head>
<body>
    <div class="overlay-container" id="overlayContainer">
        <!-- Lower Third -->
        <div class="lower-third anim-slide" id="lowerThird">
            <div class="participant-name" id="participantName">John Smith</div>
            <div class="participant-info">
                <div class="country-flag" id="countryFlag"></div>
                <div class="bib-number" id="bibNumber">#42</div>
                <div style="flex: 1;">
                    <div style="font-size: 14px; color: rgba(255,255,255,0.7);">Snowboard Slopestyle - Finals</div>
                    <div style="font-size: 12px; color: rgba(255,255,255,0.5);">Run 2 of 3</div>
                </div>
            </div>
            <div class="scores-container">
                <div class="score-item">
                    <div class="score-label">CURRENT</div>
                    <div class="score-value current-score" id="currentScore">87.50</div>
                </div>
                <div class="score-item">
                    <div class="score-label">BEST</div>
                    <div class="score-value" id="bestScore">89.25</div>
                </div>
                <div class="score-item">
                    <div class="score-label">RANK</div>
                    <div class="score-value" id="currentRank">2nd</div>
                </div>
            </div>
        </div>
        
        <!-- Rankings Table -->
        <div class="rankings-table anim-slide" id="rankingsTable" style="display: none;">
            <div class="table-header">LIVE RANKINGS</div>
            <div id="rankingRows">
                <!-- Dynamic ranking rows will be inserted here -->
            </div>
        </div>
        
        <!-- Podium (Top 3) -->
        <div class="podium-container" id="podiumContainer" style="display: none;">
            <div class="podium-step second">
                <div class="podium-number second">2</div>
                <div class="podium-name">Jane Doe</div>
                <div class="podium-score">85.75</div>
            </div>
            <div class="podium-step first">
                <div class="podium-number first">1</div>
                <div class="podium-name">John Smith</div>
                <div class="podium-score">89.25</div>
            </div>
            <div class="podium-step third">
                <div class="podium-number third">3</div>
                <div class="podium-name">Bob Wilson</div>
                <div class="podium-score">82.50</div>
            </div>
        </div>
    </div>

    <script>
        // Global overlay state
        let overlayState = {
            type: 'lower-third',
            position: 'bottom-left',
            animation: 'slide',
            scale: 100,
            visible: true,
            eventId: <?= isset($_GET['event_id']) ? (int)$_GET['event_id'] : 'null' ?>,
            currentParticipant: null
        };
        
        // Listen for messages from parent control panel
        window.addEventListener('message', function(event) {
            if (event.origin !== window.location.origin) return;
            
            const command = event.data;
            handleCommand(command);
        });
        
        function handleCommand(command) {
            switch(command.action) {
                case 'setType':
                    setOverlayType(command.type);
                    break;
                case 'setPosition':
                    setPosition(command.position);
                    break;
                case 'setAnimation':
                    setAnimation(command.animation);
                    break;
                case 'setScale':
                    setScale(command.scale);
                    break;
                case 'show':
                    showOverlay();
                    break;
                case 'hide':
                    hideOverlay();
                    break;
                case 'updateData':
                    updateData(command.data);
                    break;
                case 'setParticipant':
                    setCurrentParticipant(command.participantId);
                    break;
            }
        }
        
        function setOverlayType(type) {
            overlayState.type = type;
            
            // Hide all overlays
            document.getElementById('lowerThird').style.display = 'none';
            document.getElementById('rankingsTable').style.display = 'none';
            document.getElementById('podiumContainer').style.display = 'none';
            
            // Show selected overlay
            switch(type) {
                case 'lower-third':
                    document.getElementById('lowerThird').style.display = 'block';
                    break;
                case 'rankings':
                case 'top5':
                    document.getElementById('rankingsTable').style.display = 'block';
                    updateRankingsTable(type === 'top5' ? 5 : 10);
                    break;
                case 'top3':
                    document.getElementById('podiumContainer').style.display = 'flex';
                    break;
                case 'none':
                    // All hidden
                    break;
            }
            
            if (type !== 'none') {
                showOverlay();
            }
        }
        
        function setPosition(position) {
            overlayState.position = position;
            const lowerThird = document.getElementById('lowerThird');
            
            // Remove all position classes
            lowerThird.className = lowerThird.className.replace(/position-\S+/g, '');
            
            // Add new position class
            lowerThird.classList.add('position-' + position);
        }
        
        function setAnimation(animation) {
            overlayState.animation = animation;
            const elements = document.querySelectorAll('.lower-third, .rankings-table');
            
            elements.forEach(el => {
                el.className = el.className.replace(/anim-\S+/g, '');
                el.classList.add('anim-' + animation);
            });
        }
        
        function setScale(scale) {
            overlayState.scale = scale;
            const container = document.getElementById('overlayContainer');
            
            if (scale === 100) {
                container.style.transform = '';
            } else {
                container.style.transform = `scale(${scale / 100})`;
                container.style.transformOrigin = 'top left';
            }
        }
        
        function showOverlay() {
            overlayState.visible = true;
            const activeElements = getActiveElements();
            
            activeElements.forEach(el => {
                el.classList.add('show');
            });
        }
        
        function hideOverlay() {
            overlayState.visible = false;
            const activeElements = getActiveElements();
            
            activeElements.forEach(el => {
                el.classList.remove('show');
            });
        }
        
        function getActiveElements() {
            const elements = [];
            switch(overlayState.type) {
                case 'lower-third':
                    elements.push(document.getElementById('lowerThird'));
                    break;
                case 'rankings':
                case 'top5':
                    elements.push(document.getElementById('rankingsTable'));
                    break;
                case 'top3':
                    elements.push(document.getElementById('podiumContainer'));
                    break;
            }
            return elements;
        }
        
        // Helper function to get URL parameters
        function getUrlParameter(name) {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(name);
        }
        
        function parseTableBodyToParticipants(tableBodyHtml) {
            // Create a temporary element to parse the HTML
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = tableBodyHtml;
            
            const participants = [];
            const rows = tempDiv.querySelectorAll('tr.group-participant-row');
            
            rows.forEach((row, index) => {
                const cells = row.querySelectorAll('td');
                if (cells.length >= 5) {
                    // Extract data from table cells
                    const rankCell = cells[0].querySelector('.group-rank');
                    const bibCell = cells[1].querySelector('.badge');
                    const nameCell = cells[2].querySelector('strong');
                    const clubCell = cells[3].querySelector('.badge');
                    const scoreCell = cells[cells.length - 1]; // Last cell is usually the best score
                    
                    const rank = rankCell ? parseInt(rankCell.textContent.trim()) : index + 1;
                    const bib = bibCell ? bibCell.textContent.trim() : '';
                    const name = nameCell ? nameCell.textContent.trim() : 'Unknown';
                    const club = clubCell ? clubCell.textContent.trim() : '';
                    const scoreText = scoreCell ? scoreCell.textContent.trim() : '0.00';
                    const score = parseFloat(scoreText) || 0.00;
                    
                    participants.push({
                        rank: rank,
                        bib: bib,
                        name: name,
                        club: club,
                        score: score.toFixed(2)
                    });
                }
            });
            
            return participants;
        }
        
        function updateRankingsTable(limit = 10, configId = null) {
            // Get event ID from URL parameter
            const eventId = getUrlParameter('event_id') || '3'; // Default for testing
            
            // Use the provided configId or try to get from parent window
            let selectedConfigId = configId;
            if (!selectedConfigId) {
                try {
                    selectedConfigId = parent.window.selectedConfigId;
                } catch (e) {
                    // Can't access parent window
                }
            }
            
            // Build URL to call YOUR StyleScore API directly
            let apiUrl;
            if (selectedConfigId) {
                apiUrl = `../api/summary_table_api.php?config_id=${selectedConfigId}&format=json&limit=${limit}`;
            } else {
                apiUrl = `../api/summary_table_api.php?event_id=${eventId}&format=json&limit=${limit}&sort=OverallAverage&sort_direction=desc`;
            }
            
            console.log('Calling YOUR API directly:', apiUrl);
            
            fetch(apiUrl)
                .then(response => response.json())
                .then(data => {
                    console.log('YOUR API Response:', data); // Debug log
                    const container = document.getElementById('rankingRows');
                    container.innerHTML = '';
                    
                    if (data.success && data.table_body) {
                        // Parse HTML table data to extract participant information
                        const participants = parseTableBodyToParticipants(data.table_body);
                        
                        participants.slice(0, limit).forEach((participant, index) => {
                            setTimeout(() => {
                                const row = document.createElement('div');
                                row.className = 'ranking-row';
                                
                                const rank = participant.rank || index + 1;
                                let rankClass = '';
                                if (rank === 1) rankClass = 'first';
                                else if (rank === 2) rankClass = 'second';
                                else if (rank === 3) rankClass = 'third';
                                
                                row.innerHTML = `
                                    <div class="rank ${rankClass}">${rank}</div>
                                    <div class="participant-info">
                                        <div class="name">${participant.name}</div>
                                        <div class="country">${participant.club || 'Club'}</div>
                                    </div>
                                    <div class="score">${participant.score}</div>
                                `;
                                
                                container.appendChild(row);
                                
                                // Add animation
                                setTimeout(() => {
                                    row.style.opacity = '1';
                                    row.style.transform = 'translateX(0)';
                                }, 50);
                            }, index * 100);
                        });
                    } else {
                        const message = data.message || 'No ranking data available';
                        container.innerHTML = `
                            <div class="ranking-row">
                                <div class="rank">-</div>
                                <div class="participant-info">
                                    <div class="name">${message}</div>
                                    <div class="country">Check configuration</div>
                                </div>
                                <div class="score">0.00</div>
                            </div>
                        `;
                    }
                        });
                    } else {
                        // Show error message instead of demo data
                        console.error('API Error or no data:', data);
                        const container = document.getElementById('rankingRows');
                        container.innerHTML = `
                            <div class="ranking-row">
                                <div class="rank">-</div>
                                <div class="participant-info">
                                    <div class="name">No data available</div>
                                    <div class="country">Check API connection</div>
                                </div>
                                <div class="score">0.00</div>
                            </div>
                        `;
                    }
                })
                .catch(error => {
                    console.error('Error loading rankings:', error);
                    const container = document.getElementById('rankingRows');
                    container.innerHTML = `
                        <div class="ranking-row">
                            <div class="rank">!</div>
                            <div class="participant-info">
                                <div class="name">API Connection Error</div>
                                <div class="country">${error.message}</div>
                            </div>
                            <div class="score">--</div>
                        </div>
                    `;
                });
        }
        
        function setCurrentParticipant(participantId) {
            overlayState.currentParticipant = participantId;
            // Fetch participant data and update lower third
            updateParticipantData(participantId);
        }
        
        function updateParticipantData(participantId) {
            // In real implementation, fetch from API
            // For demo, use sample data
            const sampleData = {
                name: 'Sarah Johnson',
                country: 'USA',
                bib: '#15',
                currentScore: 92.75,
                bestScore: 92.75,
                rank: '1st'
            };
            
            document.getElementById('participantName').textContent = sampleData.name;
            document.getElementById('bibNumber').textContent = sampleData.bib;
            document.getElementById('currentScore').textContent = sampleData.currentScore;
            document.getElementById('bestScore').textContent = sampleData.bestScore;
            document.getElementById('currentRank').textContent = sampleData.rank;
        }
        
        // Auto-refresh data every 5 seconds
        setInterval(() => {
            if (overlayState.visible && overlayState.eventId) {
                // Refresh data based on current overlay type
                switch(overlayState.type) {
                    case 'rankings':
                    case 'top5':
                        updateRankingsTable(overlayState.type === 'top5' ? 5 : 10);
                        break;
                    case 'lower-third':
                        if (overlayState.currentParticipant) {
                            updateParticipantData(overlayState.currentParticipant);
                        }
                        break;
                }
            }
        }, 5000);
        
        // Initialize
        setTimeout(() => {
            setOverlayType('lower-third');
            setPosition('bottom-left');
            showOverlay();
        }, 500);
        
        // Listen for messages from parent window
        window.addEventListener('message', function(event) {
            if (event.origin !== window.location.origin) return;
            
            const command = event.data;
            if (!command.action) return;
            
            switch(command.action) {
                case 'setType':
                    setOverlayType(command.type);
                    break;
                case 'setPosition':
                    setPosition(command.position);
                    break;
                case 'setAnimation':
                    setAnimation(command.animation);
                    break;
                case 'setScale':
                    setScale(command.scale);
                    break;
                case 'show':
                    showOverlay();
                    break;
                case 'hide':
                    hideOverlay();
                    break;
                case 'updateParticipant':
                    updateParticipantData(command.participant);
                    break;
                case 'loadConfiguration':
                    updateRankingsTable(10, command.configId);
                    break;
                case 'updateRankings':
                    updateRankingsTable(10, command.configId);
                    break;
            }
        });
    </script>
</body>
</html>
