<?php
// TV Overlay Control Panel
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

try {
    require_once '../includes/db.php';
    
    // Test database connection
    if (!isset($pdo)) {
        die('Database connection not established');
    }
    
    // Get events for dropdown using correct StyleScore column names
    try {
        $events_query = "SELECT id as event_id, name as event_name, date as event_date FROM events ORDER BY date DESC LIMIT 20";
        $events_stmt = $pdo->query($events_query);
        $events = $events_stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        echo "Database Error: " . $e->getMessage() . "<br>";
        // Fallback: create a minimal event structure for demo
        $events = [['event_id' => 1, 'event_name' => 'Default Event', 'event_date' => date('Y-m-d')]];
    }
    
    // If no events exist, show message
    if (empty($events)) {
        echo "<div class='alert alert-warning'>No events found in database. Please create an event first.</div>";
        $events = [['event_id' => 1, 'event_name' => 'Demo Event', 'event_date' => date('Y-m-d')]];
    }
    
    $selected_event = isset($_GET['event_id']) ? (int)$_GET['event_id'] : ($events[0]['event_id'] ?? 1);
    
} catch (Exception $e) {
    echo "Database Error: " . $e->getMessage() . "<br>";
    echo "Please check your database connection in includes/db.php<br>";
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TV Overlay Control Panel - StyleScore</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --lime: #39FF14;
            --blue: #4B7BFF;
            --violet: #AD26FF;
            --ink: #EAF2FF;
            --muted: #9FB0C9;
            --dark: #0B1220;
            --card: #0f1730;
            --grad: linear-gradient(135deg, var(--lime), var(--blue) 45%, var(--violet));
            --shadow: 0 25px 60px rgba(0,0,0,.45);
        }
        
        body {
            background: 
                radial-gradient(1200px 600px at 15% 0%, rgba(72,160,255,.08), transparent),
                radial-gradient(1200px 600px at 85% 20%, rgba(173,38,255,.08), transparent),
                var(--dark);
            color: var(--ink);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            min-height: 100vh;
        }
        
        .control-panel {
            background: var(--card);
            border: 1px solid rgba(255,255,255,.06);
            border-radius: 20px;
            padding: 24px;
            margin-bottom: 20px;
            box-shadow: var(--shadow);
            backdrop-filter: blur(4px);
        }
        
        .preview-window {
            background: #000;
            border: 2px solid rgba(255,255,255,.1);
            border-radius: 16px;
            aspect-ratio: 16/9;
            position: relative;
            overflow: hidden;
            margin-bottom: 20px;
        }
        
        .overlay-preview {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            pointer-events: none;
        }
        
        .control-btn {
            background: var(--grad);
            color: var(--dark);
            border: none;
            padding: 12px 24px;
            border-radius: 12px;
            font-weight: 600;
            margin: 5px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .control-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(57,255,20,0.3);
        }
        
        .control-btn.active {
            background: var(--blue);
            color: white;
        }
        
        .position-controls {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin: 15px 0;
        }
        
        .position-btn {
            aspect-ratio: 1;
            border: 2px solid rgba(255,255,255,.1);
            background: rgba(255,255,255,.05);
            border-radius: 8px;
            color: var(--muted);
            font-size: 12px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .position-btn.active {
            border-color: var(--lime);
            background: rgba(57,255,20,0.1);
            color: var(--lime);
        }
        
        .overlay-controls {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }
        
        .form-control, .form-select {
            background: rgba(255,255,255,.05);
            border: 1px solid rgba(255,255,255,.1);
            color: var(--ink);
            border-radius: 8px;
        }
        
        .form-control:focus, .form-select:focus {
            background: rgba(255,255,255,.08);
            border-color: var(--blue);
            color: var(--ink);
            box-shadow: 0 0 0 0.2rem rgba(75,123,255,0.25);
        }
        
        .preview-scale {
            position: absolute;
            top: 10px;
            right: 10px;
            background: rgba(0,0,0,0.7);
            color: white;
            padding: 5px 10px;
            border-radius: 6px;
            font-size: 12px;
        }
        
        .status-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }
        
        .status-live { background: #ff4757; }
        .status-ready { background: #2ed573; }
        .status-loading { background: #ffa502; }
    </style>
</head>
<body>
    <div class="container-fluid p-4">
        <div class="row">
            <!-- Control Panel -->
            <div class="col-lg-4">
                <div class="control-panel">
                    <h3 class="mb-4">
                        <i class="fas fa-tv me-2"></i>
                        TV Overlay Control
                        <span class="status-indicator status-ready"></span>
                    </h3>
                    
                    <!-- Event Selection -->
                    <div class="mb-4">
                        <label class="form-label">Event</label>
                        <select class="form-select" id="eventSelect" onchange="loadEventConfigurations()">
                            <?php foreach ($events as $event): ?>
                                <option value="<?= $event['event_id'] ?>" <?= $event['event_id'] == $selected_event ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($event['event_name']) ?> - <?= date('M j, Y', strtotime($event['event_date'])) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Ranking Configurations -->
                    <div class="mb-4" id="configurationSection">
                        <label class="form-label">
                            <i class="fas fa-list-ol me-2"></i>Ranking Configurations
                        </label>
                        <div id="configurationList" class="border rounded p-3 bg-dark">
                            <div class="text-muted text-center">
                                <i class="fas fa-info-circle me-1"></i>
                                Select an event to load configurations
                            </div>
                        </div>
                    </div>
                    
                    <!-- Overlay Type Controls -->
                    <div class="mb-4">
                        <label class="form-label">Overlay Type</label>
                        <div class="d-grid gap-2">
                            <button class="control-btn active" onclick="setOverlayType('lower-third')" id="btn-lower-third">
                                <i class="fas fa-user me-2"></i>Lower Third - Current Participant
                            </button>
                            <button class="control-btn" onclick="setOverlayType('rankings')" id="btn-rankings">
                                <i class="fas fa-list-ol me-2"></i>Live Rankings
                            </button>
                            <button class="control-btn" onclick="setOverlayType('top5')" id="btn-top5">
                                <i class="fas fa-trophy me-2"></i>Top 5 Results
                            </button>
                            <button class="control-btn" onclick="setOverlayType('top3')" id="btn-top3">
                                <i class="fas fa-medal me-2"></i>Top 3 Podium
                            </button>
                            <button class="control-btn" onclick="setOverlayType('startlist')" id="btn-startlist">
                                <i class="fas fa-clipboard-list me-2"></i>Start List
                            </button>
                            <button class="control-btn" onclick="setOverlayType('judges')" id="btn-judges">
                                <i class="fas fa-gavel me-2"></i>Judge Scores
                            </button>
                            <button class="control-btn" onclick="setOverlayType('none')" id="btn-none">
                                <i class="fas fa-eye-slash me-2"></i>Hide All
                            </button>
                        </div>
                    </div>
                    
                    <!-- Position Controls -->
                    <div class="mb-4">
                        <label class="form-label">Position</label>
                        <div class="position-controls">
                            <button class="position-btn" onclick="setPosition('top-left')" data-pos="top-left">
                                <i class="fas fa-arrow-up"></i><i class="fas fa-arrow-left"></i><br>TL
                            </button>
                            <button class="position-btn" onclick="setPosition('top-center')" data-pos="top-center">
                                <i class="fas fa-arrow-up"></i><br>TC
                            </button>
                            <button class="position-btn" onclick="setPosition('top-right')" data-pos="top-right">
                                <i class="fas fa-arrow-up"></i><i class="fas fa-arrow-right"></i><br>TR
                            </button>
                            <button class="position-btn" onclick="setPosition('center-left')" data-pos="center-left">
                                <i class="fas fa-arrow-left"></i><br>CL
                            </button>
                            <button class="position-btn" onclick="setPosition('center')" data-pos="center">
                                <i class="fas fa-crosshairs"></i><br>C
                            </button>
                            <button class="position-btn" onclick="setPosition('center-right')" data-pos="center-right">
                                <i class="fas fa-arrow-right"></i><br>CR
                            </button>
                            <button class="position-btn active" onclick="setPosition('bottom-left')" data-pos="bottom-left">
                                <i class="fas fa-arrow-down"></i><i class="fas fa-arrow-left"></i><br>BL
                            </button>
                            <button class="position-btn" onclick="setPosition('bottom-center')" data-pos="bottom-center">
                                <i class="fas fa-arrow-down"></i><br>BC
                            </button>
                            <button class="position-btn" onclick="setPosition('bottom-right')" data-pos="bottom-right">
                                <i class="fas fa-arrow-down"></i><i class="fas fa-arrow-right"></i><br>BR
                            </button>
                        </div>
                    </div>
                    
                    <!-- Advanced Controls -->
                    <div class="overlay-controls">
                        <div>
                            <label class="form-label">Animation</label>
                            <select class="form-select" id="animationType">
                                <option value="slide">Slide In</option>
                                <option value="fade">Fade In</option>
                                <option value="flip">Flip In</option>
                                <option value="zoom">Zoom In</option>
                            </select>
                        </div>
                        <div>
                            <label class="form-label">Duration (sec)</label>
                            <input type="number" class="form-control" id="displayDuration" value="10" min="1" max="60">
                        </div>
                        <div>
                            <label class="form-label">Auto-Update</label>
                            <select class="form-select" id="autoUpdate">
                                <option value="off">Manual</option>
                                <option value="5">Every 5s</option>
                                <option value="10">Every 10s</option>
                                <option value="30">Every 30s</option>
                            </select>
                        </div>
                        <div>
                            <label class="form-label">Scale %</label>
                            <input type="range" class="form-range" id="scaleSlider" min="50" max="150" value="100" onchange="updateScale()">
                            <small class="text-muted" id="scaleValue">100%</small>
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="mt-4 d-grid gap-2 d-md-flex">
                        <button class="control-btn flex-fill" onclick="showOverlay()">
                            <i class="fas fa-play me-2"></i>Show
                        </button>
                        <button class="control-btn flex-fill" onclick="hideOverlay()">
                            <i class="fas fa-stop me-2"></i>Hide
                        </button>
                        <button class="control-btn flex-fill" onclick="openFullscreen()">
                            <i class="fas fa-expand me-2"></i>Fullscreen
                        </button>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="control-panel">
                    <h5 class="mb-3">Quick Actions</h5>
                    <div class="d-grid gap-2">
                        <button class="btn btn-outline-light" onclick="nextParticipant()">
                            <i class="fas fa-step-forward me-2"></i>Next Participant
                        </button>
                        <button class="btn btn-outline-light" onclick="previousParticipant()">
                            <i class="fas fa-step-backward me-2"></i>Previous Participant
                        </button>
                        <button class="btn btn-outline-light" onclick="refreshData()">
                            <i class="fas fa-sync me-2"></i>Refresh Data
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Preview Window -->
            <div class="col-lg-8">
                <div class="preview-window" id="previewWindow">
                    <div class="preview-scale" id="previewScale">100%</div>
                    <iframe src="overlay-display.php?event_id=<?= $selected_event ?>" 
                            id="overlayFrame" 
                            frameborder="0" 
                            style="width: 100%; height: 100%; transform-origin: top left;">
                    </iframe>
                </div>
                
                <!-- Control Buttons Row -->
                <div class="row mt-3">
                    <div class="col-md-6">
                        <div class="control-panel">
                            <h6 class="mb-3">Participant Control</h6>
                            <div class="input-group mb-2">
                                <span class="input-group-text">Current</span>
                                <select class="form-select" id="participantSelect">
                                    <option>Loading participants...</option>
                                </select>
                            </div>
                            <div class="row">
                                <div class="col-6">
                                    <button class="btn btn-outline-light w-100" onclick="selectParticipant('prev')">
                                        <i class="fas fa-chevron-left"></i> Prev
                                    </button>
                                </div>
                                <div class="col-6">
                                    <button class="btn btn-outline-light w-100" onclick="selectParticipant('next')">
                                        Next <i class="fas fa-chevron-right"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="control-panel">
                            <h6 class="mb-3">Display Status</h6>
                            <div class="mb-2">
                                <small class="text-muted">Current View:</small>
                                <div id="currentView" class="fw-bold text-success">Lower Third</div>
                            </div>
                            <div class="mb-2">
                                <small class="text-muted">Position:</small>
                                <div id="currentPosition" class="fw-bold">Bottom Left</div>
                            </div>
                            <div>
                                <small class="text-muted">Last Update:</small>
                                <div id="lastUpdate" class="fw-bold">--:--:--</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="overlay-control.js"></script>
</body>
</html>
