// TV Overlay Control System JavaScript
class OverlayController {
    constructor() {
        this.state = {
            overlayType: 'lower-third',
            position: 'bottom-left',
            animation: 'slide',
            scale: 100,
            visible: false,
            eventId: null,
            currentParticipant: null,
            autoUpdate: 'off',
            displayDuration: 10
        };
        
        this.overlayFrame = document.getElementById('overlayFrame');
        this.participants = [];
        this.currentParticipantIndex = 0;
        
        this.init();
    }
    
    init() {
        // Load initial data
        this.loadParticipants();
        this.updateStatus();
        
        // Setup auto-update interval
        this.setupAutoUpdate();
        
        // Initialize scale slider
        this.updateScaleDisplay();
    }
    
    sendCommand(action, data = {}) {
        const command = { action, ...data };
        this.overlayFrame.contentWindow.postMessage(command, window.location.origin);
    }
    
    loadParticipants() {
        const eventId = document.getElementById('eventSelect').value;
        if (!eventId) return;
        
        // In real implementation, fetch from API
        // For demo, use sample data
        this.participants = [
            { id: 1, name: 'John Smith', country: 'USA', bib: '#42' },
            { id: 2, name: 'Jane Doe', country: 'CAN', bib: '#15' },
            { id: 3, name: 'Bob Wilson', country: 'NOR', bib: '#23' },
            { id: 4, name: 'Anna Lee', country: 'KOR', bib: '#8' },
            { id: 5, name: 'Max Weber', country: 'GER', bib: '#31' }
        ];
        
        this.updateParticipantDropdown();
    }
    
    updateParticipantDropdown() {
        const select = document.getElementById('participantSelect');
        select.innerHTML = '';
        
        this.participants.forEach((participant, index) => {
            const option = document.createElement('option');
            option.value = participant.id;
            option.textContent = `${participant.bib} ${participant.name} (${participant.country})`;
            if (index === this.currentParticipantIndex) {
                option.selected = true;
            }
            select.appendChild(option);
        });
    }
    
    updateStatus() {
        const typeNames = {
            'lower-third': 'Lower Third',
            'rankings': 'Live Rankings',
            'top5': 'Top 5 Results',
            'top3': 'Top 3 Podium',
            'startlist': 'Start List',
            'judges': 'Judge Scores',
            'none': 'Hidden'
        };
        
        const positionNames = {
            'top-left': 'Top Left',
            'top-center': 'Top Center',
            'top-right': 'Top Right',
            'center-left': 'Center Left',
            'center': 'Center',
            'center-right': 'Center Right',
            'bottom-left': 'Bottom Left',
            'bottom-center': 'Bottom Center',
            'bottom-right': 'Bottom Right'
        };
        
        document.getElementById('currentView').textContent = typeNames[this.state.overlayType] || 'Unknown';
        document.getElementById('currentPosition').textContent = positionNames[this.state.position] || 'Unknown';
        document.getElementById('lastUpdate').textContent = new Date().toLocaleTimeString();
    }
    
    setupAutoUpdate() {
        const autoUpdateSelect = document.getElementById('autoUpdate');
        
        if (this.autoUpdateInterval) {
            clearInterval(this.autoUpdateInterval);
        }
        
        const interval = parseInt(autoUpdateSelect.value);
        if (interval > 0) {
            this.autoUpdateInterval = setInterval(() => {
                this.refreshData();
            }, interval * 1000);
        }
    }
    
    updateScaleDisplay() {
        const slider = document.getElementById('scaleSlider');
        const display = document.getElementById('scaleValue');
        const previewScale = document.getElementById('previewScale');
        
        display.textContent = slider.value + '%';
        previewScale.textContent = slider.value + '%';
        
        this.state.scale = parseInt(slider.value);
        this.sendCommand('setScale', { scale: this.state.scale });
    }
}

// Global controller instance
let overlayController;

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    overlayController = new OverlayController();
    
    // Load configurations for the initially selected event
    loadEventConfigurations();
});

// Control Functions (called from HTML)
function setOverlayType(type) {
    // Update button states
    document.querySelectorAll('[id^="btn-"]').forEach(btn => {
        btn.classList.remove('active');
    });
    document.getElementById('btn-' + type).classList.add('active');
    
    // Show/hide ranking configuration section
    const rankingConfigSection = document.getElementById('rankingConfigSection');
    if (type === 'rankings' || type === 'top5' || type === 'top3') {
        rankingConfigSection.style.display = 'block';
        loadRankingConfigs();
    } else {
        rankingConfigSection.style.display = 'none';
    }
    
    overlayController.state.overlayType = type;
    overlayController.sendCommand('setType', { type });
    overlayController.updateStatus();
}

// Load ranking configurations for the current event
function loadEventConfigurations() {
    const eventId = document.getElementById('eventSelect').value;
    const configList = document.getElementById('configurationList');
    
    if (!eventId) {
        configList.innerHTML = '<div class="text-muted text-center"><i class="fas fa-info-circle me-1"></i>Select an event to load configurations</div>';
        return;
    }
    
    configList.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin me-2"></i>Loading configurations...</div>';
    
    // Use YOUR StyleScore API directly
    fetch(`../api/result_configurations_api.php?action=list&event_id=${eventId}&format=json`)
        .then(response => response.json())
        .then(data => {
            console.log('Configuration API Response:', data); // Debug log
            if (data.success && data.data && data.data.length > 0) {
                let html = '';
                data.data.forEach(config => {
                    html += `
                        <div class="configuration-item border rounded p-3 mb-2 bg-secondary bg-opacity-10">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="mb-1">${config.name || `Configuration ${config.id}`}</h6>
                                    <small class="text-muted">Config ID: ${config.id} | Type: ${config.view_type || 'summary'}</small>
                                </div>
                                <div class="d-grid gap-1">
                                    <button type="button" class="btn btn-sm btn-primary" onclick="loadConfiguration(${config.id})">
                                        <i class="fas fa-download me-1"></i>Load
                                    </button>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <button type="button" onclick="window.open('../api/summary_table_api.php?config_id=${config.id}&format=pdf', '_blank')" class="btn btn-outline-danger" title="Export PDF">
                                            <i class="fas fa-file-pdf"></i>
                                        </button>
                                        <button type="button" onclick="window.open('../api/summary_table_api.php?config_id=${config.id}&format=html', '_blank')" class="btn btn-outline-secondary" title="Export HTML">
                                            <i class="fas fa-table"></i>
                                        </button>
                                        <button type="button" onclick="window.location.href='../api/summary_table_api.php?config_id=${config.id}&format=csv'" class="btn btn-outline-success" title="Export CSV">
                                            <i class="fas fa-file-csv"></i>
                                        </button>
                                        <button type="button" onclick="window.open('../api/summary_table_api.php?config_id=${config.id}&format=json', '_blank')" class="btn btn-outline-info" title="View JSON">
                                            <i class="fas fa-code"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `;
                });
                configList.innerHTML = html;
            } else {
                const message = data.message || 'No configurations found for this event';
                configList.innerHTML = `<div class="text-muted text-center"><i class="fas fa-exclamation-circle me-1"></i>${message}</div>`;
            }
        })
        .catch(error => {
            console.error('Error loading configurations:', error);
            configList.innerHTML = `<div class="text-danger text-center"><i class="fas fa-exclamation-triangle me-1"></i>Error loading configurations: ${error.message}</div>`;
        });
}

// Load a specific configuration for TV overlay
function loadConfiguration(configId) {
    console.log('Loading configuration:', configId);
    
    // Set the overlay type to rankings
    setOverlayType('rankings');
    
    // Store the selected config ID
    window.selectedConfigId = configId;
    
    // Send message to overlay iframe to update with this configuration
    const overlayFrame = document.getElementById('overlayFrame');
    if (overlayFrame && overlayFrame.contentWindow) {
        overlayFrame.contentWindow.postMessage({
            action: 'loadConfiguration',
            configId: configId
        }, '*');
    }
    
    // Update overlay with this configuration
    overlayController.sendMessage({
        action: 'updateRankings',
        configId: configId
    });
}

function setPosition(position) {
    // Update position button states
    document.querySelectorAll('.position-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-pos="${position}"]`).classList.add('active');
    
    overlayController.state.position = position;
    overlayController.sendCommand('setPosition', { position });
    overlayController.updateStatus();
}

function showOverlay() {
    overlayController.state.visible = true;
    overlayController.sendCommand('show');
    overlayController.updateStatus();
    
    // Auto-hide after duration if set
    const duration = parseInt(document.getElementById('displayDuration').value);
    if (duration > 0) {
        setTimeout(() => {
            hideOverlay();
        }, duration * 1000);
    }
}

function hideOverlay() {
    overlayController.state.visible = false;
    overlayController.sendCommand('hide');
    overlayController.updateStatus();
}

function updateScale() {
    overlayController.updateScaleDisplay();
}

function changeEvent() {
    const eventId = document.getElementById('eventSelect').value;
    overlayController.state.eventId = eventId;
    overlayController.loadParticipants();
    
    // Reload overlay frame with new event
    const currentSrc = overlayController.overlayFrame.src;
    const url = new URL(currentSrc);
    url.searchParams.set('event_id', eventId);
    overlayController.overlayFrame.src = url.toString();
}

function nextParticipant() {
    if (overlayController.participants.length === 0) return;
    
    overlayController.currentParticipantIndex = 
        (overlayController.currentParticipantIndex + 1) % overlayController.participants.length;
    
    selectCurrentParticipant();
}

function previousParticipant() {
    if (overlayController.participants.length === 0) return;
    
    overlayController.currentParticipantIndex = 
        overlayController.currentParticipantIndex === 0 
        ? overlayController.participants.length - 1 
        : overlayController.currentParticipantIndex - 1;
    
    selectCurrentParticipant();
}

function selectCurrentParticipant() {
    const participant = overlayController.participants[overlayController.currentParticipantIndex];
    if (!participant) return;
    
    overlayController.state.currentParticipant = participant.id;
    overlayController.sendCommand('setParticipant', { participantId: participant.id });
    overlayController.updateParticipantDropdown();
}

function selectParticipant(direction) {
    if (direction === 'next') {
        nextParticipant();
    } else if (direction === 'prev') {
        previousParticipant();
    } else {
        // Direct selection from dropdown
        const participantId = parseInt(document.getElementById('participantSelect').value);
        const index = overlayController.participants.findIndex(p => p.id === participantId);
        if (index !== -1) {
            overlayController.currentParticipantIndex = index;
            selectCurrentParticipant();
        }
    }
}

function refreshData() {
    overlayController.sendCommand('updateData');
    overlayController.updateStatus();
}

function openFullscreen() {
    const overlayWindow = window.open(
        overlayController.overlayFrame.src,
        'overlay_fullscreen',
        'fullscreen=yes,toolbar=no,location=no,status=no,menubar=no,scrollbars=no,resizable=yes'
    );
    
    if (overlayWindow) {
        overlayWindow.focus();
    }
}

// Auto-update setup change
document.getElementById('autoUpdate').addEventListener('change', function() {
    overlayController.state.autoUpdate = this.value;
    overlayController.setupAutoUpdate();
});

// Animation type change
document.getElementById('animationType').addEventListener('change', function() {
    overlayController.state.animation = this.value;
    overlayController.sendCommand('setAnimation', { animation: this.value });
});

// Participant dropdown change
document.getElementById('participantSelect').addEventListener('change', function() {
    selectParticipant();
});

// Keyboard shortcuts
document.addEventListener('keydown', function(event) {
    if (event.ctrlKey || event.metaKey) {
        switch(event.key) {
            case '1':
                setOverlayType('lower-third');
                event.preventDefault();
                break;
            case '2':
                setOverlayType('rankings');
                event.preventDefault();
                break;
            case '3':
                setOverlayType('top3');
                event.preventDefault();
                break;
            case 'h':
                hideOverlay();
                event.preventDefault();
                break;
            case 's':
                showOverlay();
                event.preventDefault();
                break;
            case 'ArrowRight':
                nextParticipant();
                event.preventDefault();
                break;
            case 'ArrowLeft':
                previousParticipant();
                event.preventDefault();
                break;
        }
    }
});

// Status indicators
function updateStatusIndicator(status) {
    const indicator = document.querySelector('.status-indicator');
    indicator.className = 'status-indicator';
    
    switch(status) {
        case 'live':
            indicator.classList.add('status-live');
            break;
        case 'ready':
            indicator.classList.add('status-ready');
            break;
        case 'loading':
            indicator.classList.add('status-loading');
            break;
    }
}

// Initialize status as ready
updateStatusIndicator('ready');
