<?php
// TV Overlay Control Panel
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

try {
    require_once '../includes/db.php';
    
    // Test database connection
    if (!isset($pdo)) {
        die('Database connection not established');
    }
    
    // Get events for dropdown using correct StyleScore column names
    try {
        $events_query = "SELECT id as event_id, name as event_name, date as event_date FROM events ORDER BY date DESC LIMIT 20";
        $events_stmt = $pdo->query($events_query);
        $events = $events_stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        echo "Database Error: " . $e->getMessage() . "<br>";
        // Fallback: create a minimal event structure for demo
        $events = [['event_id' => 1, 'event_name' => 'Default Event', 'event_date' => date('Y-m-d')]];
    }
    
    // If no events exist, show message
    if (empty($events)) {
        echo "<div class='alert alert-warning'>No events found in database. Please create an event first.</div>";
        $events = [['event_id' => 1, 'event_name' => 'Demo Event', 'event_date' => date('Y-m-d')]];
    }
    
    $selected_event = isset($_GET['event_id']) ? (int)$_GET['event_id'] : ($events[0]['event_id'] ?? 1);
    
} catch (Exception $e) {
    echo "Database Error: " . $e->getMessage() . "<br>";
    echo "Please check your database connection in includes/db.php<br>";
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TV Overlay Control Panel - StyleScore</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --lime: #39FF14;
            --blue: #4B7BFF;
            --violet: #AD26FF;
            --ink: #EAF2FF;
            --muted: #9FB0C9;
            --dark: #0B1220;
            --card: #0f1730;
            --grad: linear-gradient(135deg, var(--lime), var(--blue) 45%, var(--violet));
            --shadow: 0 25px 60px rgba(0,0,0,.45);
        }
        
        body {
            background: 
                radial-gradient(1200px 600px at 15% 0%, rgba(72,160,255,.08), transparent),
                radial-gradient(1200px 600px at 85% 20%, rgba(173,38,255,.08), transparent),
                var(--dark);
            color: var(--ink);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            min-height: 100vh;
        }
        div#controlPanel {
    max-height: 100vh;
    overflow: auto;
}
        .control-panel {
            background: var(--card);
            border: 1px solid rgba(255,255,255,.06);
            border-radius: 20px;
            padding: 24px;
            margin-bottom: 20px;
            box-shadow: var(--shadow);
            backdrop-filter: blur(4px);
        }
        
        .draggable-panel {
            position: relative !important;
            background: var(--card);
            border: 1px solid rgba(255,255,255,.06);
            border-radius: 20px;
            margin-bottom: 20px;
            box-shadow: var(--shadow);
            backdrop-filter: blur(4px);
            transition: all 0.3s ease;
        }
        
        .draggable-panel.floating {
            position: fixed !important;
            z-index: 1000;
            min-width: 300px;
            max-width: 500px;
            cursor: move;
        }
        
        .draggable-panel.minimized {
            height: 50px !important;
            overflow: hidden;
        }
        
        .panel-header {
            display: flex;
            justify-content: between;
            align-items: center;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(255,255,255,.06);
            cursor: move;
            user-select: none;
            border-radius: 20px 20px 0 0;
        }
        
        .panel-header h5 {
            margin: 0;
            flex: 1;
        }
        
        .panel-controls {
            display: flex;
            gap: 8px;
        }
        
        .panel-btn {
            width: 28px;
            height: 28px;
            border: none;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            font-size: 12px;
        }
        
        .panel-btn.minimize {
            background: rgba(255,193,7,0.2);
            color: #ffc107;
        }
        
        .panel-btn.minimize:hover {
            background: rgba(255,193,7,0.3);
        }
        
        .panel-btn.move-up {
            background: rgba(40,167,69,0.2);
            color: #28a745;
        }
        
        .panel-btn.move-up:hover {
            background: rgba(40,167,69,0.3);
        }
        
        .panel-btn.move-down {
            background: rgba(40,167,69,0.2);
            color: #28a745;
        }
        
        .panel-btn.move-down:hover {
            background: rgba(40,167,69,0.3);
        }
        
        .panel-btn.float {
            background: rgba(0,123,255,0.2);
            color: #007bff;
        }
        
        .panel-btn.float:hover {
            background: rgba(0,123,255,0.3);
        }
        
        .panel-btn.close {
            background: rgba(220,53,69,0.2);
            color: #dc3545;
        }
        
        .panel-btn.close:hover {
            background: rgba(220,53,69,0.3);
        }
        
        .panel-content {
            padding: 24px;
            transition: all 0.3s ease;
        }
        
        .floating .panel-header {
            background: linear-gradient(135deg, rgba(57,255,20,0.1), rgba(75,123,255,0.1));
        }
        
        .draggable-panel.floating:hover {
            box-shadow: 0 15px 35px rgba(0,0,0,0.6);
        }
        
        .panel-manager {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 1100;
            background: var(--card);
            border: 1px solid rgba(255,255,255,.06);
            border-radius: 12px;
            padding: 10px;
            box-shadow: var(--shadow);
            backdrop-filter: blur(4px);
            display: flex;
            gap: 8px;
        }
        
        .panel-manager-btn {
            width: 36px;
            height: 36px;
            border: none;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            font-size: 14px;
            background: rgba(255,255,255,0.05);
            color: var(--ink);
        }
        
        .panel-manager-btn:hover {
            background: rgba(255,255,255,0.1);
            transform: translateY(-1px);
        }
        
        .panel-manager-btn.active {
            background: var(--lime);
            color: var(--dark);
        }
        
        .panel-manager-separator {
            width: 1px;
            height: 30px;
            background: rgba(255,255,255,0.2);
            margin: 0 8px;
        }
        
        .preview-window {
            background: #000;
            border: 2px solid rgba(255,255,255,.1);
            border-radius: 16px;
            aspect-ratio: 16/9;
            position: relative;
            overflow: hidden;
            margin-bottom: 20px;
        }
        
        .overlay-preview {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            pointer-events: none;
        }
        
        .control-btn {
            background: var(--grad);
            color: var(--dark);
            border: none;
            padding: 12px 24px;
            border-radius: 12px;
            font-weight: 600;
            margin: 5px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .control-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(57,255,20,0.3);
        }
        
        .control-btn.active {
            background: var(--blue);
            color: white;
        }
        
        .position-controls {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin: 15px 0;
        }
        
        .position-btn {
            aspect-ratio: 1;
            border: 2px solid rgba(255,255,255,.1);
            background: rgba(255,255,255,.05);
            border-radius: 8px;
            color: var(--muted);
            font-size: 12px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .position-btn.active {
            border-color: var(--lime);
            background: rgba(57,255,20,0.1);
            color: var(--lime);
        }
        
        .overlay-controls {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }
        
        .form-control, .form-select {
            background: rgba(255,255,255,.05);
            border: 1px solid rgba(255,255,255,.1);
            color: var(--ink);
            border-radius: 8px;
        }
        
        .form-control:focus, .form-select:focus {
            background: rgba(255,255,255,.08);
            border-color: var(--blue);
            color: var(--ink);
            box-shadow: 0 0 0 0.2rem rgba(75,123,255,0.25);
        }
        
        .preview-scale {
            position: absolute;
            top: 10px;
            right: 10px;
            background: rgba(0,0,0,0.7);
            color: white;
            padding: 5px 10px;
            border-radius: 6px;
            font-size: 12px;
        }
        
        .status-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }
        
        .status-live { background: #ff4757; }
        .status-ready { background: #2ed573; }
        .status-loading { background: #ffa502; }
        
        /* Panel Settings Management */
        .panel-settings {
            background: rgba(11, 18, 32, 0.4);
            border: 1px solid rgba(57, 255, 20, 0.2);
            border-radius: 8px;
        }
        
        .panel-settings .btn {
            font-size: 0.75rem;
            padding: 0.375rem 0.5rem;
            border-radius: 4px;
            transition: all 0.2s ease;
        }
        
        .panel-settings .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        }
        
        .panel-settings .btn-outline-success:hover {
            box-shadow: 0 4px 12px rgba(46, 213, 115, 0.4);
        }
        
        .panel-settings .btn-outline-info:hover {
            box-shadow: 0 4px 12px rgba(75, 123, 255, 0.4);
        }
        
        .panel-settings .btn-outline-warning:hover {
            box-shadow: 0 4px 12px rgba(255, 165, 2, 0.4);
        }
        
        .panel-settings .btn-outline-danger:hover {
            box-shadow: 0 4px 12px rgba(255, 71, 87, 0.4);
        }
    </style>
</head>
<body>
    <div class="container-fluid container-StyleScore p-4">
        <div class="row">
            <!-- Control Panel -->
            <div class="col-lg-4" id="controlPanel">
                <div class="control-panel">
                    <h3 class="mb-4">
                        <i class="fas fa-tv me-2"></i>
                        TV Overlay Control
                        <span class="status-indicator status-ready"></span>
                    </h3>
                    
                    <!-- Panel Settings Management -->
                    <div class="mb-3 p-2 panel-settings">
                        <h6 class="small mb-2 text-primary">
                            <i class="fas fa-cog me-1"></i>Panel Settings
                        </h6>
                        <div class="d-flex gap-1 flex-wrap">
                            <button class="btn btn-outline-success btn-sm flex-fill" onclick="savePanelOrder(); savePanelSettings();" title="Save Current Layout">
                                <i class="fas fa-save me-1"></i>Save
                            </button>
                            <button class="btn btn-outline-info btn-sm flex-fill" onclick="loadPanelOrder(); loadPanelSettings();" title="Reload Saved Layout">
                                <i class="fas fa-undo me-1"></i>Reload
                            </button>
                            <button class="btn btn-outline-warning btn-sm flex-fill" onclick="resetPanelLayout();" title="Reset to Default">
                                <i class="fas fa-home me-1"></i>Reset
                            </button>
                            <button class="btn btn-outline-danger btn-sm flex-fill" onclick="clearPanelSettings();" title="Clear Saved Settings">
                                <i class="fas fa-trash me-1"></i>Clear
                            </button>
                        </div>
                    </div>
                    
                    <!-- Event Selection -->
                    <div class="mb-4">
                        <label class="form-label">Event</label>
                        <select class="form-select" id="eventSelect" onchange="loadEventConfigurations()">
                            <?php foreach ($events as $event): ?>
                                <option value="<?= $event['event_id'] ?>" <?= $event['event_id'] == $selected_event ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($event['event_name']) ?> - <?= date('M j, Y', strtotime($event['event_date'])) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                                <!-- Ranking Configurations -->
                <div class="minimized" id="configPanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-list-ol me-2"></i>Ranking Configurations
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('configPanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('configPanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('configPanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('configPanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('configPanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <div id="configurationList" class="border rounded p-3 bg-dark">
                            <div class="text-muted text-center">
                                <i class="fas fa-info-circle me-1"></i>
                                Select an event to load configurations
                            </div>
                        </div>
                    </div>
                </div>
                <!-- Display Controls - Compact -->
                <div class="minimized" id="displayPanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-display me-2"></i>Display Controls
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('displayPanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('displayPanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('displayPanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('displayPanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('displayPanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <!-- Action Buttons -->
                        <div class="d-grid gap-2 d-md-flex mb-3">
                            <button class="control-btn flex-fill" onclick="showOverlay()">
                                <i class="fas fa-play me-2"></i>Show
                            </button>
                            <button class="control-btn flex-fill" onclick="hideOverlay()">
                                <i class="fas fa-stop me-2"></i>Hide
                            </button>
                            <button class="control-btn flex-fill" onclick="openFullscreen()">
                                <i class="fas fa-expand me-2"></i>Fullscreen
                            </button>
                        </div>
                        
                        <!-- Advanced Controls -->
                        <div class="row g-2">
                            <div class="col-6">
                                <label class="form-label small">Animation</label>
                                <select class="form-select form-select-sm" id="animationType">
                                    <option value="slide">Slide In</option>
                                    <option value="fade">Fade In</option>
                                    <option value="flip">Flip In</option>
                                    <option value="zoom">Zoom In</option>
                                </select>
                            </div>
                            <div class="col-6">
                                <label class="form-label small">Duration (sec)</label>
                                <input type="number" class="form-control form-control-sm" id="displayDuration" value="10" min="1" max="60">
                            </div>
                            <div class="col-6">
                                <label class="form-label small">Auto-Update</label>
                                <select class="form-select form-select-sm" id="autoUpdate">
                                    <option value="off">Manual</option>
                                    <option value="5">Every 5s</option>
                                    <option value="10">Every 10s</option>
                                    <option value="30">Every 30s</option>
                                </select>
                            </div>
                            <div class="col-6">
                                <label class="form-label small">Scale %</label>
                                <input type="range" class="form-range" id="scaleSlider" min="50" max="150" value="100" onchange="updateScale()">
                                <small class="text-muted" id="scaleValue">100%</small>
                            </div>
                        </div>
                    </div>
                </div>
                
                
                    

                
                
                <!-- Overlay Type Controls -->
                <div class="minimized" id="overlayTypePanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-layer-group me-2"></i>Overlay Type
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('overlayTypePanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('overlayTypePanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('overlayTypePanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('overlayTypePanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('overlayTypePanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <div class="d-grid gap-2">
                            <button class="control-btn active" onclick="setOverlayType('lower-third')" id="btn-lower-third">
                                <i class="fas fa-user me-2"></i>Lower Third - Current Participant
                            </button>
                            <button class="control-btn" onclick="setOverlayType('rankings')" id="btn-rankings">
                                <i class="fas fa-list-ol me-2"></i>Live Rankings
                            </button>
                            <button class="control-btn" onclick="setOverlayType('top5')" id="btn-top5">
                                <i class="fas fa-trophy me-2"></i>Top 5 Results
                            </button>
                            <button class="control-btn" onclick="setOverlayType('top3')" id="btn-top3">
                                <i class="fas fa-medal me-2"></i>Top 3 Podium
                            </button>
                            <button class="control-btn" onclick="setOverlayType('startlist')" id="btn-startlist">
                                <i class="fas fa-clipboard-list me-2"></i>Start List
                            </button>
                            <button class="control-btn" onclick="setOverlayType('judges')" id="btn-judges">
                                <i class="fas fa-gavel me-2"></i>Judge Scores
                            </button>
                            <button class="control-btn" onclick="setOverlayType('none')" id="btn-none">
                                <i class="fas fa-eye-slash me-2"></i>Hide All
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Position Controls -->
                <div class="minimized" id="positionPanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-crosshairs me-2"></i>Position
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('positionPanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('positionPanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('positionPanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('positionPanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('positionPanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <div class="position-controls">
                            <button class="position-btn" onclick="setPosition('top-left')" data-pos="top-left">
                                <i class="fas fa-arrow-up"></i><i class="fas fa-arrow-left"></i><br>TL
                            </button>
                            <button class="position-btn" onclick="setPosition('top-center')" data-pos="top-center">
                                <i class="fas fa-arrow-up"></i><br>TC
                            </button>
                            <button class="position-btn" onclick="setPosition('top-right')" data-pos="top-right">
                                <i class="fas fa-arrow-up"></i><i class="fas fa-arrow-right"></i><br>TR
                            </button>
                            <button class="position-btn" onclick="setPosition('center-left')" data-pos="center-left">
                                <i class="fas fa-arrow-left"></i><br>CL
                            </button>
                            <button class="position-btn" onclick="setPosition('center')" data-pos="center">
                                <i class="fas fa-crosshairs"></i><br>C
                            </button>
                            <button class="position-btn" onclick="setPosition('center-right')" data-pos="center-right">
                                <i class="fas fa-arrow-right"></i><br>CR
                            </button>
                            <button class="position-btn active" onclick="setPosition('bottom-left')" data-pos="bottom-left">
                                <i class="fas fa-arrow-down"></i><i class="fas fa-arrow-left"></i><br>BL
                            </button>
                            <button class="position-btn" onclick="setPosition('bottom-center')" data-pos="bottom-center">
                                <i class="fas fa-arrow-down"></i><br>BC
                            </button>
                            <button class="position-btn" onclick="setPosition('bottom-right')" data-pos="bottom-right">
                                <i class="fas fa-arrow-down"></i><i class="fas fa-arrow-right"></i><br>BR
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="minimized" style="padding: 0;">
                    
                    <!-- Overlay Type Controls -->
                    <div class="mb-4" style="display: none;">
                        <label class="form-label">Overlay Type</label>
                        <div class="d-grid gap-2">
                            <button class="control-btn active" onclick="setOverlayType('lower-third')" id="btn-lower-third">
                                <i class="fas fa-user me-2"></i>Lower Third - Current Participant
                            </button>
                            <button class="control-btn" onclick="setOverlayType('rankings')" id="btn-rankings">
                                <i class="fas fa-list-ol me-2"></i>Live Rankings
                            </button>
                            <button class="control-btn" onclick="setOverlayType('top5')" id="btn-top5">
                                <i class="fas fa-trophy me-2"></i>Top 5 Results
                            </button>
                            <button class="control-btn" onclick="setOverlayType('top3')" id="btn-top3">
                                <i class="fas fa-medal me-2"></i>Top 3 Podium
                            </button>
                            <button class="control-btn" onclick="setOverlayType('startlist')" id="btn-startlist">
                                <i class="fas fa-clipboard-list me-2"></i>Start List
                            </button>
                            <button class="control-btn" onclick="setOverlayType('judges')" id="btn-judges">
                                <i class="fas fa-gavel me-2"></i>Judge Scores
                            </button>
                            <button class="control-btn" onclick="setOverlayType('none')" id="btn-none">
                                <i class="fas fa-eye-slash me-2"></i>Hide All
                            </button>
                        </div>
                    </div>
                    
                    <!-- Position Controls -->
                    <div class="mb-4" style="display: none;">
                        <label class="form-label">Position</label>
                        <div class="position-controls">
                            <button class="position-btn" onclick="setPosition('top-left')" data-pos="top-left">
                                <i class="fas fa-arrow-up"></i><i class="fas fa-arrow-left"></i><br>TL
                            </button>
                            <button class="position-btn" onclick="setPosition('top-center')" data-pos="top-center">
                                <i class="fas fa-arrow-up"></i><br>TC
                            </button>
                            <button class="position-btn" onclick="setPosition('top-right')" data-pos="top-right">
                                <i class="fas fa-arrow-up"></i><i class="fas fa-arrow-right"></i><br>TR
                            </button>
                            <button class="position-btn" onclick="setPosition('center-left')" data-pos="center-left">
                                <i class="fas fa-arrow-left"></i><br>CL
                            </button>
                            <button class="position-btn" onclick="setPosition('center')" data-pos="center">
                                <i class="fas fa-crosshairs"></i><br>C
                            </button>
                            <button class="position-btn" onclick="setPosition('center-right')" data-pos="center-right">
                                <i class="fas fa-arrow-right"></i><br>CR
                            </button>
                            <button class="position-btn active" onclick="setPosition('bottom-left')" data-pos="bottom-left">
                                <i class="fas fa-arrow-down"></i><i class="fas fa-arrow-left"></i><br>BL
                            </button>
                            <button class="position-btn" onclick="setPosition('bottom-center')" data-pos="bottom-center">
                                <i class="fas fa-arrow-down"></i><br>BC
                            </button>
                            <button class="position-btn" onclick="setPosition('bottom-right')" data-pos="bottom-right">
                                <i class="fas fa-arrow-down"></i><i class="fas fa-arrow-right"></i><br>BR
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="minimized" id="actionsPanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-bolt me-2"></i>Quick Actions
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('actionsPanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('actionsPanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('actionsPanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('actionsPanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('actionsPanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <div class="d-grid gap-2">
                            <button class="btn btn-outline-light" onclick="nextParticipant()">
                                <i class="fas fa-step-forward me-2"></i>Next Participant
                            </button>
                            <button class="btn btn-outline-light" onclick="previousParticipant()">
                                <i class="fas fa-step-backward me-2"></i>Previous Participant
                            </button>
                            <button class="btn btn-outline-light" onclick="refreshData()">
                                <i class="fas fa-sync me-2"></i>Refresh Data
                            </button>
                        </div>
                    </div>
                </div>
                   <!-- Column Controls Panel -->
                <div class="minimized" id="columnControlsPanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-columns me-2"></i>Column Controls
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('columnControlsPanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('columnControlsPanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('columnControlsPanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('columnControlsPanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('columnControlsPanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <!-- Width Controls -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-expand-arrows-alt me-1"></i>Width Controls</h6>
                            <div class="row g-2">
                                <div class="col-6">
                                    <label class="form-label small">Lower Third Width</label>
                                    <input type="range" class="form-range" id="lowerThirdWidth" min="400" max="800" value="600" onchange="updateStyling()">
                                    <small class="text-muted" id="lowerThirdWidthValue">600px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Rankings Width</label>
                                    <input type="range" class="form-range" id="rankingsWidth" min="300" max="600" value="480" onchange="updateStyling()">
                                    <small class="text-muted" id="rankingsWidthValue">480px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Result Column Width</label>
                                    <input type="range" class="form-range" id="resultColumnWidth" min="60" max="150" value="100" onchange="updateStyling()">
                                    <small class="text-muted" id="resultColumnWidthValue">100px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Rank Column Width</label>
                                    <input type="range" class="form-range" id="rankColumnWidth" min="40" max="80" value="60" onchange="updateStyling()">
                                    <small class="text-muted" id="rankColumnWidthValue">60px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Bib Column Width</label>
                                    <input type="range" class="form-range" id="bibColumnWidth" min="40" max="80" value="60" onchange="updateStyling()">
                                    <small class="text-muted" id="bibColumnWidthValue">60px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Judges Column Width</label>
                                    <input type="range" class="form-range" id="judgesColumnWidth" min="80" max="160" value="120" onchange="updateStyling()">
                                    <small class="text-muted" id="judgesColumnWidthValue">120px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Start List Column Width</label>
                                    <input type="range" class="form-range" id="startlistColumnWidth" min="60" max="120" value="80" onchange="updateStyling()">
                                    <small class="text-muted" id="startlistColumnWidthValue">80px</small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Panel Position Controls -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-arrows-alt me-1"></i>Panel Position Controls</h6>
                            <div class="row g-2">
                                <div class="col-6">
                                    <label class="form-label small">Offscreen X (%)</label>
                                    <input type="range" class="form-range" id="panelOffsetXOffscreen" min="-200" max="200" value="-100" onchange="updateStyling()">
                                    <small class="text-muted" id="panelOffsetXOffscreenValue">-100%</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Offscreen Y (px)</label>
                                    <input type="range" class="form-range" id="panelOffsetYOffscreen" min="-500" max="500" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="panelOffsetYOffscreenValue">0px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Onscreen X (px)</label>
                                    <input type="range" class="form-range" id="panelOffsetXOnscreen" min="-200" max="200" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="panelOffsetXOnscreenValue">0px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Onscreen Y (px)</label>
                                    <input type="range" class="form-range" id="panelOffsetYOnscreen" min="-200" max="200" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="panelOffsetYOnscreenValue">0px</small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Name Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-user me-1"></i>Name Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="nameAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center">Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="nameOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="nameOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="nameOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="nameOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showName" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showName">Show Name</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Club Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-building me-1"></i>Club Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="clubAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center">Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="clubOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="clubOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="clubOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="clubOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showClub" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showClub">Show Club</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Country Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-flag me-1"></i>Country Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="countryAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center">Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="countryOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="countryOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="countryOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="countryOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showCountry" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showCountry">Show Country</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Rank Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-trophy me-1"></i>Rank Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="rankAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center" selected>Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="rankOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="rankOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="rankOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="rankOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showRank" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showRank">Show Rank</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Score Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-star me-1"></i>Score Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="scoreAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center" selected>Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="scoreOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="scoreOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="scoreOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="scoreOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showScore" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showScore">Show Score</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Bib Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-hashtag me-1"></i>Bib Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="bibAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center" selected>Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="bibOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="bibOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="bibOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="bibOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showBib" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showBib">Show Bib</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Judges Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-gavel me-1"></i>Judges Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="judgesAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center" selected>Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="judgesOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="judgesOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="judgesOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="judgesOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showJudges" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showJudges">Show Judges</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Start List Column -->
                        <div class="mb-3 p-2 border rounded">
                            <h6 class="small mb-2"><i class="fas fa-list-ol me-1"></i>Start List Column</h6>
                            <div class="row g-2">
                                <div class="col-4">
                                    <label class="form-label small">Align</label>
                                    <select class="form-select form-select-sm" id="startlistAlign" onchange="updateStyling()">
                                        <option value="left">Left</option>
                                        <option value="center" selected>Center</option>
                                        <option value="right">Right</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset X</label>
                                    <input type="range" class="form-range" id="startlistOffsetX" min="-50" max="50" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="startlistOffsetXValue">0px</small>
                                </div>
                                <div class="col-4">
                                    <label class="form-label small">Offset Y</label>
                                    <input type="range" class="form-range" id="startlistOffsetY" min="-20" max="20" value="0" onchange="updateStyling()">
                                    <small class="text-muted" id="startlistOffsetYValue">0px</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="showStartlist" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="showStartlist">Show Start List</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div> 
                <!-- Styling Controls -->
                <div class="minimized" id="stylingPanel">
                    <div class="panel-header">
                        <h5>
                            <i class="fas fa-palette me-2"></i>Styling Controls
                        </h5>
                        <div class="panel-controls">
                            <button class="panel-btn move-up" onclick="movePanelUp('stylingPanel')" title="Move Up">
                                <i class="fas fa-chevron-up"></i>
                            </button>
                            <button class="panel-btn move-down" onclick="movePanelDown('stylingPanel')" title="Move Down">
                                <i class="fas fa-chevron-down"></i>
                            </button>
                            <button class="panel-btn minimize" onclick="toggleMinimize('stylingPanel')" title="Minimize">
                                <i class="fas fa-minus"></i>
                            </button>
                            <button class="panel-btn float" onclick="toggleFloat('stylingPanel')" title="Float/Dock">
                                <i class="fas fa-external-link-alt"></i>
                            </button>
                            <button class="panel-btn close" onclick="closePanel('stylingPanel')" title="Close">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                    <div class="panel-content">
                        <!-- Typography Controls -->
                        <div class="mb-3">
                            <h6 class="text-primary mb-2">Typography</h6>
                            <div class="row g-2">
                                <div class="col-6">
                                    <label class="form-label small">Rank Font Size</label>
                                    <input type="range" class="form-range" id="rankFontSize" min="12" max="48" value="32" onchange="updateStyling()">
                                    <small class="text-muted" id="rankFontSizeValue">32px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Name Font Size</label>
                                    <input type="range" class="form-range" id="nameFontSize" min="14" max="36" value="24" onchange="updateStyling()">
                                    <small class="text-muted" id="nameFontSizeValue">24px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Score Font Size</label>
                                    <input type="range" class="form-range" id="scoreFontSize" min="12" max="32" value="20" onchange="updateStyling()">
                                    <small class="text-muted" id="scoreFontSizeValue">20px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Country Font Size</label>
                                    <input type="range" class="form-range" id="countryFontSize" min="10" max="18" value="14" onchange="updateStyling()">
                                    <small class="text-muted" id="countryFontSizeValue">14px</small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Spacing Controls -->
                        <div class="mb-3">
                            <h6 class="text-primary mb-2">Spacing</h6>
                            <div class="row g-2">
                                <div class="col-6">
                                    <label class="form-label small">Row Height</label>
                                    <input type="range" class="form-range" id="rowHeight" min="40" max="100" value="70" onchange="updateStyling()">
                                    <small class="text-muted" id="rowHeightValue">70px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Row Spacing</label>
                                    <input type="range" class="form-range" id="rowSpacing" min="0" max="20" value="8" onchange="updateStyling()">
                                    <small class="text-muted" id="rowSpacingValue">8px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Padding</label>
                                    <input type="range" class="form-range" id="rankingPadding" min="10" max="40" value="20" onchange="updateStyling()">
                                    <small class="text-muted" id="rankingPaddingValue">20px</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Border Radius</label>
                                    <input type="range" class="form-range" id="borderRadius" min="0" max="20" value="10" onchange="updateStyling()">
                                    <small class="text-muted" id="borderRadiusValue">10px</small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Visual Effects Controls -->
                        <div class="mb-3">
                            <h6 class="text-primary mb-2">Visual Effects</h6>
                            <div class="row g-2">
                                <div class="col-6">
                                    <label class="form-label small">Background Opacity</label>
                                    <input type="range" class="form-range" id="bgOpacity" min="0" max="100" value="90" onchange="updateStyling()">
                                    <small class="text-muted" id="bgOpacityValue">90%</small>
                                </div>
                                <div class="col-6">
                                    <label class="form-label small">Shadow Intensity</label>
                                    <input type="range" class="form-range" id="shadowIntensity" min="0" max="100" value="50" onchange="updateStyling()">
                                    <small class="text-muted" id="shadowIntensityValue">50%</small>
                                </div>
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="enableGlow" checked onchange="updateStyling()">
                                        <label class="form-check-label" for="enableGlow">
                                            Enable Podium Glow Effect
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Reset and Presets -->
                        <div class="d-grid gap-2 d-md-flex">
                            <button class="btn btn-outline-warning btn-sm flex-fill" onclick="resetStyling()">
                                <i class="fas fa-undo me-1"></i>Reset
                            </button>
                            <div class="dropdown flex-fill">
                                <button class="btn btn-outline-info btn-sm dropdown-toggle w-100" type="button" data-bs-toggle="dropdown">
                                    <i class="fas fa-swatchbook me-1"></i>Presets
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="#" onclick="applyStylingPreset('compact')">Compact</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="applyStylingPreset('large')">Large Display</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="applyStylingPreset('minimal')">Minimal</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="applyStylingPreset('broadcast')">Broadcast</a></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Preview Window -->
            <div class="col-lg-8" id="previewContainer">
                <div class="preview-window" id="previewWindow">
                    <div class="preview-scale" id="previewScale">100%</div>
                    <iframe src="overlay-display.php?event_id=<?= $selected_event ?>" 
                            id="overlayFrame" 
                            frameborder="0" 
                            style="width: 100%; height: 100%; transform-origin: top left;">
                    </iframe>
                </div>
                
                <!-- Control Buttons Row -->
                <div class="row mt-3">
                    <div class="col-md-6">
                        <div class="control-panel">
                            <h6 class="mb-3">Participant Control</h6>
                            <div class="input-group mb-2">
                                <span class="input-group-text">Current</span>
                                <select class="form-select" id="participantSelect">
                                    <option>Loading participants...</option>
                                </select>
                            </div>
                            <div class="row">
                                <div class="col-6">
                                    <button class="btn btn-outline-light w-100" onclick="selectParticipant('prev')">
                                        <i class="fas fa-chevron-left"></i> Prev
                                    </button>
                                </div>
                                <div class="col-6">
                                    <button class="btn btn-outline-light w-100" onclick="selectParticipant('next')">
                                        Next <i class="fas fa-chevron-right"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="control-panel">
                            <h6 class="mb-3">Display Status</h6>
                        <div class="row">
                            <div class="col-4 mb-2">
                                <small class="">Current View:</small>
                                <div id="currentView" class="fw-bold text-success">Lower Third</div>
                            </div>
                            <div class="col-4 mb-2">
                                <small class="">Position:</small>
                                <div id="currentPosition" class="fw-bold">Bottom Left</div>
                            </div>
                            <div class="col-4 mb-2">
                                <small class="">Last Update:</small>
                                <div id="lastUpdate" class="fw-bold">--:--:--</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>



    <!-- Panel Manager Toolbar -->
    <div class="panel-manager">
        <button class="panel-manager-btn" onclick="toggleAllPanels()" title="Show/Hide All Panels">
            <i class="fas fa-th"></i>
        </button>
        <button class="panel-manager-btn" onclick="floatAllPanels()" title="Float All Panels">
            <i class="fas fa-external-link-alt"></i>
        </button>
        <button class="panel-manager-btn" onclick="dockAllPanels()" title="Dock All Panels">
            <i class="fas fa-compress"></i>
        </button>
        <button class="panel-manager-btn" onclick="minimizeAllPanels()" title="Minimize All Panels">
            <i class="fas fa-minus"></i>
        </button>
        <button class="panel-manager-btn" onclick="resetPanelLayout()" title="Reset Panel Layout">
            <i class="fas fa-undo"></i>
        </button>
        <div class="panel-manager-separator"></div>
        <button class="panel-manager-btn" onclick="reopenPanel('columnControlsPanel')" title="Column Controls">
            <i class="fas fa-columns"></i>
        </button>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="overlay-control.js"></script>
</body>
</html>
