# Category Colors Usage Guide

## 🎨 How to Use Dynamic Category Colors Across StyleScore

### 1. **Basic Setup - Include the Utility**

```php
// Include the category colors utility
include_once '../includes/category_colors.php';
// or from admin folder:
include_once 'includes/category_colors.php';
```

### 2. **Dashboard Integration**

```php
<?php
// In public_event_dashboard.php or similar files

// Get all category colors for the event
$category_colors = getAllCategoryColors($pdo, $event_id);

// Generate CSS for the page
echo '<style>' . generateCategoryColorsCSS($category_colors) . '</style>';

// Display participants with category colors
foreach ($participants as $participant) {
    $colors = getCategoryColors($pdo, $participant['category'], $event_id);
    
    echo '<div class="participant-card" style="' . getCategoryColorStyle($colors, 'light') . '">';
    echo createCategoryColorDot($colors, 20);
    echo '<strong>' . htmlspecialchars($participant['name']) . '</strong>';
    echo createCategoryBadge($participant['category'], $colors);
    echo '</div>';
}
?>
```

### 3. **Results Tables Integration**

```php
<?php
// In summary_table_api.php or similar files

// Add to your existing generateParticipantCSS function:
function generateParticipantCSS($participant_colors, $event_id = null, $pdo = null) {
    $css = "/* Participant badge colors */\n";
    
    // Existing participant badge CSS...
    foreach ($participant_colors as $bib => $color_data) {
        // ... existing code ...
    }
    
    // Add category colors
    if ($event_id && $pdo) {
        $category_colors = getAllCategoryColors($pdo, $event_id);
        $css .= "\n" . generateCategoryColorsCSS($category_colors);
    }
    
    return $css;
}

// In your HTML table generation:
foreach ($participants as $participant) {
    $category_colors = getCategoryColors($pdo, $participant['category'], $event_id);
    $css_class = $category_colors['css_class'] ?? '';
    
    echo '<tr>';
    echo '<td class="' . $css_class . '-light">' . htmlspecialchars($participant['category']) . '</td>';
    echo '<td>' . htmlspecialchars($participant['name']) . '</td>';
    echo '</tr>';
}
?>
```

### 4. **Start List Integration**

```php
<?php
// In start_list_api.php

// Include category colors
include_once '../includes/category_colors.php';

// In your generateHTMLTable function:
case 'category':
    $colors = getCategoryColors($pdo, $p['category'], $event_id);
    $html .= '<td class="css-category">';
    $html .= createCategoryColorDot($colors, 16);
    $html .= createCategoryBadge($p['category'], $colors, null, null, 'sm');
    $html .= '</td>';
    break;
?>
```

### 5. **Live Scoreboard Integration**

```php
<?php
// In live_scoreboard.php

// Get category colors for dynamic styling
$category_colors = getAllCategoryColors($pdo, $event_id);

// Add CSS to page head
echo '<style>';
echo generateCategoryColorsCSS($category_colors);
echo '
.scoreboard-row[data-category] {
    border-left: 4px solid var(--category-color);
}
.category-header {
    background: var(--category-bg-light);
    color: var(--category-border);
}
';
echo '</style>';

// In JavaScript for dynamic updates
?>
<script>
const categoryColors = <?= json_encode($category_colors) ?>;

function updateParticipantRow(participant) {
    const row = document.querySelector(`[data-participant="${participant.id}"]`);
    const colors = categoryColors[participant.category];
    
    if (colors && row) {
        row.style.setProperty('--category-color', colors['background-color']);
        row.style.setProperty('--category-bg-light', colors['background-light']);
        row.style.setProperty('--category-border', colors['border-color']);
        row.dataset.category = participant.category;
    }
}
</script>
```

### 6. **Event Dashboard Integration**

```php
<?php
// In admin/event_dashboard.php

// Create category overview with colors
$category_colors = getAllCategoryColors($pdo, $event_id);

echo '<div class="row mb-3">';
echo '<div class="col-12">';
echo '<h6><i class="fas fa-tags me-2"></i>Categories</h6>';
echo '<div class="d-flex flex-wrap gap-2">';

foreach ($category_colors as $category_name => $colors) {
    // Get participant count for this category
    $count_stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM event_participants 
        WHERE event_id = ? AND category = ?
    ");
    $count_stmt->execute([$event_id, $category_name]);
    $count = $count_stmt->fetch()['count'];
    
    echo '<div class="category-summary" style="' . getCategoryColorStyle($colors, 'light') . '">';
    echo createCategoryColorDot($colors, 20);
    echo '<strong>' . htmlspecialchars($category_name) . '</strong>';
    echo '<span class="badge ms-2" style="' . getCategoryColorStyle($colors) . '">' . $count . '</span>';
    echo '</div>';
}

echo '</div>';
echo '</div>';
echo '</div>';
?>
```

### 7. **TV Overlay Integration**

```php
<?php
// In tv-overlay files

// Get category colors for broadcast graphics
$category_colors = getAllCategoryColors($pdo, $event_id);

// Generate CSS for broadcast styling
echo '<style>';
echo generateCategoryColorsCSS($category_colors);
echo '
.broadcast-participant {
    border-left: 8px solid var(--category-color);
    background: linear-gradient(90deg, var(--category-bg-light), transparent);
}
.broadcast-category-label {
    background: var(--category-color);
    color: var(--category-text);
}
';
echo '</style>';

// In participant display
foreach ($participants as $participant) {
    $colors = getCategoryColors($pdo, $participant['category'], $event_id);
    
    echo '<div class="broadcast-participant" style="
        --category-color: ' . ($colors['background-color'] ?? '#6c757d') . ';
        --category-bg-light: ' . ($colors['background-light'] ?? '#e9ecef') . ';
        --category-text: ' . ($colors['color'] ?? '#ffffff') . ';
    ">';
    
    echo '<div class="broadcast-category-label">' . htmlspecialchars($participant['category']) . '</div>';
    echo '<div class="broadcast-name">' . htmlspecialchars($participant['name']) . '</div>';
    echo '</div>';
}
?>
```

### 8. **Simple Badge Usage**

```php
<?php
// Quick category badge anywhere in the system
include_once 'includes/category_colors.php';

// Simple usage:
echo createCategoryBadge('Men Pro', null, $pdo, $event_id);
echo createCategoryBadge('Women Amateur', null, $pdo, $event_id, 'lg');

// With pre-fetched colors (more efficient):
$colors = getCategoryColors($pdo, 'Men Pro', $event_id);
echo createCategoryBadge('Men Pro', $colors);
?>
```

### 9. **API Response Integration**

```php
<?php
// In API responses (JSON format)

// Add category colors to API responses
$response = [
    'participants' => $participants,
    'category_colors' => getAllCategoryColors($pdo, $event_id),
    'category_css' => generateCategoryColorsCSS(getAllCategoryColors($pdo, $event_id))
];

// Frontend can then use:
// - category_colors for JavaScript styling
// - category_css for dynamic CSS injection
?>
```

### 10. **Custom CSS Classes**

```css
/* Use generated category classes in your CSS */

/* Example for Men's Pro category (auto-generated class: category-mens-pro) */
.participant-row.category-mens-pro {
    border-left: 4px solid var(--category-border-color);
}

.category-mens-pro-highlight {
    box-shadow: 0 0 10px var(--category-background-color);
}

/* Apply to any element */
.my-custom-element.category-mens-pro {
    /* Automatically gets the category colors */
}
```

### 11. **Performance Tips**

```php
<?php
// For pages with many category references, fetch all colors once:
$all_category_colors = getAllCategoryColors($pdo, $event_id);

// Then use without database calls:
foreach ($participants as $participant) {
    $colors = $all_category_colors[$participant['category']] ?? null;
    echo createCategoryBadge($participant['category'], $colors);
}

// Generate CSS once per page:
echo '<style>' . generateCategoryColorsCSS($all_category_colors) . '</style>';
?>
```

## 🎯 **Integration Checklist**

- [ ] Run the SQL migration: `add_category_colors.sql`
- [ ] Include `category_colors.php` utility in your files
- [ ] Use `getAllCategoryColors()` for page-wide color schemes
- [ ] Use `getCategoryColors()` for individual category styling
- [ ] Use `createCategoryBadge()` for consistent badge display
- [ ] Use `generateCategoryColorsCSS()` for CSS generation
- [ ] Test color regeneration functionality in event_categories.php

## 🚀 **Quick Start Example**

```php
<?php
// Complete example for any page
include_once 'includes/category_colors.php';

$event_id = $_GET['event_id'];
$category_colors = getAllCategoryColors($pdo, $event_id);
?>

<style>
<?= generateCategoryColorsCSS($category_colors) ?>
</style>

<div class="participants">
    <?php foreach ($participants as $participant): ?>
        <?php $colors = $category_colors[$participant['category']] ?? null; ?>
        <div class="participant" style="<?= getCategoryColorStyle($colors, 'light') ?>">
            <?= createCategoryColorDot($colors, 20) ?>
            <strong><?= htmlspecialchars($participant['name']) ?></strong>
            <?= createCategoryBadge($participant['category'], $colors) ?>
        </div>
    <?php endforeach; ?>
</div>
```

This system is now ready to use across your entire StyleScore platform! 🎨✨
