<?php
// Debug publish functionality

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../includes/db.php';
require_once 'publish_generators.php';

echo "<h1>Debug Publish System</h1>";

// Test FTP extension
echo "<h2>PHP FTP Extension Check</h2>";
if (extension_loaded('ftp')) {
    echo "<p style='color: green;'>✅ FTP extension is loaded</p>";
} else {
    echo "<p style='color: red;'>❌ FTP extension is NOT loaded</p>";
}

// List available extensions
echo "<h3>Available Extensions:</h3>";
$extensions = get_loaded_extensions();
echo "<pre>" . implode(", ", $extensions) . "</pre>";

// Test database connection
echo "<h2>Database Connection Test</h2>";
try {
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM events");
    $result = $stmt->fetch();
    echo "<p style='color: green;'>✅ Database connected successfully. Found {$result['count']} events.</p>";
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Database error: " . $e->getMessage() . "</p>";
}

// Test event ID
$event_id = $_GET['event_id'] ?? 1;
echo "<h2>Event Test (ID: $event_id)</h2>";

try {
    $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($event) {
        echo "<p style='color: green;'>✅ Event found: " . htmlspecialchars($event['name']) . "</p>";
    } else {
        echo "<p style='color: red;'>❌ Event not found with ID: $event_id</p>";
    }
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error fetching event: " . $e->getMessage() . "</p>";
}

// Test template file
echo "<h2>Template File Test</h2>";
$template_path = dirname(__FILE__) . '/templates/remote_dashboard.html';
if (file_exists($template_path)) {
    $size = filesize($template_path);
    echo "<p style='color: green;'>✅ Template file exists: $template_path (Size: $size bytes)</p>";
} else {
    echo "<p style='color: red;'>❌ Template file missing: $template_path</p>";
}

// Test temp directory creation
echo "<h2>Temp Directory Test</h2>";
$temp_dir = sys_get_temp_dir() . '/stylescore_debug_' . time();
if (mkdir($temp_dir, 0755, true)) {
    echo "<p style='color: green;'>✅ Temp directory created: $temp_dir</p>";
    
    // Test file generation
    echo "<h3>File Generation Test</h3>";
    try {
        if ($event) {
            $files = generateHTMLDashboard($event_id, $temp_dir, $pdo);
            echo "<p style='color: green;'>✅ Generated " . count($files) . " files:</p>";
            echo "<ul>";
            foreach ($files as $file) {
                $size = file_exists($file['local']) ? filesize($file['local']) : 0;
                echo "<li>{$file['remote']} → {$file['local']} ($size bytes)</li>";
            }
            echo "</ul>";
        }
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ File generation error: " . $e->getMessage() . "</p>";
    }
    
    // Clean up
    if (is_dir($temp_dir)) {
        $files = glob($temp_dir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) unlink($file);
        }
        rmdir($temp_dir);
        echo "<p>Cleaned up temp directory</p>";
    }
} else {
    echo "<p style='color: red;'>❌ Failed to create temp directory: $temp_dir</p>";
}

// Test publish servers table
echo "<h2>Publish Servers Table Test</h2>";
try {
    $stmt = $pdo->query("SHOW TABLES LIKE 'publish_servers'");
    if ($stmt->rowCount() > 0) {
        echo "<p style='color: green;'>✅ publish_servers table exists</p>";
        
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM publish_servers");
        $result = $stmt->fetch();
        echo "<p>Found {$result['count']} publish servers configured</p>";
        
        if ($result['count'] > 0) {
            $stmt = $pdo->query("SELECT id, name, host, data_types FROM publish_servers LIMIT 5");
            echo "<h3>Configured Servers:</h3>";
            echo "<table border='1' cellpadding='5'>";
            echo "<tr><th>ID</th><th>Name</th><th>Host</th><th>Data Types</th></tr>";
            while ($row = $stmt->fetch()) {
                echo "<tr>";
                echo "<td>{$row['id']}</td>";
                echo "<td>" . htmlspecialchars($row['name']) . "</td>";
                echo "<td>" . htmlspecialchars($row['host']) . "</td>";
                echo "<td>" . htmlspecialchars($row['data_types']) . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
    } else {
        echo "<p style='color: red;'>❌ publish_servers table does not exist</p>";
    }
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error checking publish_servers table: " . $e->getMessage() . "</p>";
}

echo "<h2>Debug Complete</h2>";
echo "<p><a href='publish_servers.php?event_id=$event_id'>← Back to Publish Servers</a></p>";
?>
