# Base URL Configuration - StyleScore

## Overview
StyleScore now uses **dynamic base URL detection** so the application works seamlessly whether accessed via:
- `http://localhost/v2/`
- `http://127.0.0.1/v2/`
- `http://192.168.1.100/v2/` (local network)
- `http://yourdomain.com/v2/` (production domain)

## Files Modified

### 1. **includes/config.php** (NEW)
Central configuration file with dynamic URL generation:
- `BASE_URL` - Main application URL
- `API_URL` - API endpoints URL
- `ASSETS_URL` - Static assets (CSS, JS, images)
- `PHOTOS_URL` - Participant photos

### 2. **includes/db.php** (UPDATED)
- Now includes `config.php` for unified configuration
- Removed duplicate `get_base_url()` function

### 3. **public_event_dashboard.php** (UPDATED)
- Changed hardcoded `http://localhost/v2/assets/...` to `ASSETS_URL`
- Logo image now uses dynamic URL

### 4. **tv-overlay/api.php** (UPDATED)
- All hardcoded localhost URLs replaced with `BASE_URL` constant
- API calls now work from any network location

## How It Works

The system detects the current request and builds URLs dynamically:

```php
function get_base_url() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https://" : "http://";
    $host = $_SERVER['HTTP_HOST'];  // Uses the actual domain/IP being accessed
    $path = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
    
    // Ensures we're at the /v2/ level
    $pathParts = explode('/', $path);
    $basePath = '/v2';
    
    if (in_array('v2', $pathParts)) {
        $v2Index = array_search('v2', $pathParts);
        $basePath = '/' . implode('/', array_slice($pathParts, 0, $v2Index + 1));
    }
    
    return $protocol . $host . $basePath . '/';
}
```

## Usage in Code

### PHP Files
```php
<?php
// Include at the top of any file
require_once 'includes/config.php';

// Use constants for URLs
echo '<img src="' . ASSETS_URL . 'img/logo.svg">';
echo '<a href="' . BASE_URL . 'dashboard.php">Dashboard</a>';

// API calls
$api_endpoint = API_URL . 'summary_table_api.php?event_id=1';
```

### Already Updated Files
Files using `$_SERVER['SERVER_NAME']` already work dynamically:
- `menu.php` - All navigation links
- Other files using relative paths

## Testing

Visit: `http://YOUR_IP/v2/test_base_url.php`

This diagnostic page shows:
- Current server configuration
- Generated URLs
- Test links to verify navigation works

## Network Access Setup

### Find Your Local IP

**Windows (PowerShell):**
```powershell
ipconfig | findstr IPv4
```

**Output example:**
```
IPv4 Address. . . . . . . . . . . : 192.168.1.100
```

**Linux/Mac:**
```bash
ifconfig | grep "inet "
```

### Access from Another Device

1. **On your server computer:** Note the IP address (e.g., `192.168.1.100`)
2. **On another device (phone, tablet, laptop):** 
   - Connect to the same WiFi network
   - Open browser and go to: `http://192.168.1.100/v2/`
3. **All navigation and API calls will automatically work**

## Firewall Configuration

If you can't access from other devices, check Windows Firewall:

```powershell
# Allow Apache HTTP traffic
netsh advfirewall firewall add rule name="Apache HTTP" dir=in action=allow protocol=TCP localport=80

# Check if firewall is blocking
netsh advfirewall show allprofiles state
```

## Remaining Hardcoded URLs

Some test/debug files still use localhost (not critical for production):
- `test_static_dashboard.php`
- `debug_api_heat_filter.php`
- `test_event6_debug.php`
- `test_api_direct.php`

These are development/testing files and don't affect normal operation.

## Migration from Old Code

### Before:
```php
<img src="http://localhost/v2/assets/img/logo.svg">
<script src="http://localhost/v2/api/summary_table_api.php"></script>
```

### After:
```php
<?php require_once 'includes/config.php'; ?>
<img src="<?php echo ASSETS_URL; ?>img/logo.svg">
<script src="<?php echo API_URL; ?>summary_table_api.php"></script>
```

## Benefits

✅ **Network Access** - Works from any device on local network  
✅ **Domain Flexibility** - Automatically adapts to production domain  
✅ **HTTPS Support** - Detects and uses HTTPS when available  
✅ **No Configuration** - Zero manual setup required  
✅ **Portable** - Move installation to any server without code changes

## Troubleshooting

### Issue: URLs still pointing to localhost
**Solution:** Clear browser cache, ensure `includes/config.php` is included

### Issue: Cannot access from network
**Solution:** 
1. Check firewall settings
2. Verify both devices on same network
3. Test with: `ping 192.168.1.100` from other device

### Issue: Assets not loading
**Solution:** Check file permissions on photos/ and assets/ directories

## Next Steps

To migrate remaining files, search for hardcoded URLs:
```bash
grep -r "http://localhost/v2" --include="*.php"
```

Replace with appropriate constant:
- Links: `BASE_URL`
- API calls: `API_URL`
- Images/CSS/JS: `ASSETS_URL`
- Photos: `PHOTOS_URL`
