
document.addEventListener('DOMContentLoaded', () => {
    const tableBody = document.querySelector("tbody");
    if (!tableBody) return;

    let draggingEle;
    let placeholder;
    let isDraggingStarted = false;

    const swap = function(nodeA, nodeB) {
        const parentA = nodeA.parentNode;
        const siblingA = nodeA.nextSibling === nodeB ? nodeA : nodeA.nextSibling;
        nodeB.parentNode.insertBefore(nodeA, nodeB);
        parentA.insertBefore(nodeB, siblingA);
    };

    const isAbove = function(nodeA, nodeB) {
        const rectA = nodeA.getBoundingClientRect();
        const rectB = nodeB.getBoundingClientRect();
        return (rectA.top + rectA.height / 2) < (rectB.top + rectB.height / 2);
    };

    const mouseDownHandler = function(e) {
        const targetRow = e.target.closest("tr");
        if (!targetRow || targetRow.classList.contains("locked")) return;

        draggingEle = targetRow;
        const rect = draggingEle.getBoundingClientRect();

        placeholder = document.createElement("tr");
        placeholder.classList.add("placeholder");
        placeholder.innerHTML = draggingEle.innerHTML;
        placeholder.style.background = "#f0f0f0";
        draggingEle.parentNode.insertBefore(placeholder, draggingEle.nextSibling);

        draggingEle.classList.add("dragging");

        x = e.clientX;
        y = e.clientY;

        document.addEventListener("mousemove", mouseMoveHandler);
        document.addEventListener("mouseup", mouseUpHandler);
    };

    const mouseMoveHandler = function(e) {
        const draggingRect = draggingEle.getBoundingClientRect();

        if (!isDraggingStarted) {
            isDraggingStarted = true;
            draggingEle.style.position = 'absolute';
            draggingEle.style.zIndex = 1000;
            draggingEle.style.left = `${draggingRect.left}px`;
            draggingEle.style.top = `${draggingRect.top}px`;
            draggingEle.style.width = `${draggingRect.width}px`;
        }

        draggingEle.style.top = `${e.clientY}px`;
        draggingEle.style.left = `${e.clientX}px`;

        const prevEle = placeholder.previousElementSibling;
        const nextEle = placeholder.nextElementSibling;

        if (prevEle && !prevEle.classList.contains('dragging') && isAbove(draggingEle, prevEle)) {
            swap(placeholder, prevEle);
            return;
        }

        if (nextEle && !nextEle.classList.contains('dragging') && isAbove(nextEle, draggingEle)) {
            swap(nextEle, placeholder);
        }
    };

    const mouseUpHandler = function() {
        placeholder.parentNode.insertBefore(draggingEle, placeholder);
        draggingEle.classList.remove("dragging");
        draggingEle.style.removeProperty("top");
        draggingEle.style.removeProperty("left");
        draggingEle.style.removeProperty("position");
        draggingEle.style.removeProperty("z-index");
        placeholder.remove();
        isDraggingStarted = false;

        document.removeEventListener("mousemove", mouseMoveHandler);
        document.removeEventListener("mouseup", mouseUpHandler);

        // Send new order to backend
        const data = [];
        tableBody.querySelectorAll("tr").forEach((tr, index) => {
            const id = tr.dataset.id;
            if (id) {
                tr.querySelector(".order-number").textContent = index + 1;
                data.push({ id: id, sort: index + 1 });
            }
        });

        fetch("save_sort_order.php", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(data)
        }).then(res => {
            if (!res.ok) alert("Saving order failed.");
        });
    };

    tableBody.querySelectorAll("tr").forEach(row => {
        if (!row.classList.contains("locked")) {
            row.addEventListener("mousedown", mouseDownHandler);
        }
    });
});
