<?php
/**
 * Install Scoring Formats Database Structure
 */

require_once '../includes/db.php';

$success = true;
$messages = [];

try {
    // Read and execute the migration SQL
    $migrationFile = __DIR__ . '/scoring_formats_migration.sql';
    
    if (!file_exists($migrationFile)) {
        throw new Exception("Migration file not found: $migrationFile");
    }
    
    $sql = file_get_contents($migrationFile);
    
    // Split into individual statements
    $statements = array_filter(array_map('trim', explode(';', $sql)));
    
    $pdo->beginTransaction();
    
    foreach ($statements as $statement) {
        if (empty($statement) || strpos($statement, '--') === 0) {
            continue; // Skip empty lines and comments
        }
        
        try {
            $pdo->exec($statement);
            $messages[] = "✓ Executed: " . substr($statement, 0, 50) . "...";
        } catch (Exception $e) {
            $messages[] = "✗ Error: " . $e->getMessage();
            $success = false;
        }
    }
    
    if ($success) {
        $pdo->commit();
        $messages[] = "✅ All tables created successfully!";
    } else {
        $pdo->rollBack();
        $messages[] = "❌ Installation failed. Changes have been rolled back.";
    }
    
} catch (Exception $e) {
    $success = false;
    $messages[] = "❌ Fatal error: " . $e->getMessage();
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Install Scoring Formats</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container-fluid mt-4">
        <div class="card">
            <div class="card-header bg-<?= $success ? 'success' : 'danger' ?> text-white">
                <h4 class="mb-0">
                    <i class="fas fa-<?= $success ? 'check-circle' : 'exclamation-triangle' ?> me-2"></i>
                    Scoring Formats Installation
                </h4>
            </div>
            <div class="card-body">
                <div class="alert alert-<?= $success ? 'success' : 'danger' ?>">
                    <?php if ($success): ?>
                        <h5>Installation Completed Successfully!</h5>
                        <p>The scoring formats database structure has been created.</p>
                    <?php else: ?>
                        <h5>Installation Failed</h5>
                        <p>There were errors during the installation process.</p>
                    <?php endif; ?>
                </div>
                
                <h6>Installation Log:</h6>
                <div class="bg-dark text-light p-3 rounded" style="max-height: 400px; overflow-y: auto;">
                    <?php foreach ($messages as $message): ?>
                        <div><?= htmlspecialchars($message) ?></div>
                    <?php endforeach; ?>
                </div>
                
                <div class="mt-4">
                    <?php if ($success): ?>
                        <a href="../admin/scoring_formats_seeder.php" class="btn btn-primary">
                            <i class="fas fa-database me-2"></i>Seed Default Data
                        </a>
                        <a href="../admin/scoring_formats.php" class="btn btn-success">
                            <i class="fas fa-cog me-2"></i>Manage Formats
                        </a>
                    <?php else: ?>
                        <button class="btn btn-danger" onclick="location.reload()">
                            <i class="fas fa-redo me-2"></i>Retry Installation
                        </button>
                    <?php endif; ?>
                    <a href="../dashboard.php" class="btn btn-secondary">
                        <i class="fas fa-home me-2"></i>Back to Dashboard
                    </a>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
