<?php
session_start();
include 'includes/db.php';

// Get system statistics for help context
$stats = [];
try {
    $stats['total_events'] = $pdo->query("SELECT COUNT(*) FROM events")->fetchColumn();
    $stats['active_events'] = $pdo->query("SELECT COUNT(*) FROM events WHERE status = 'open'")->fetchColumn();
    $stats['total_participants'] = $pdo->query("SELECT COUNT(*) FROM participants")->fetchColumn();
    $stats['active_judges'] = $pdo->query("SELECT COUNT(*) FROM judges")->fetchColumn();
} catch (Exception $e) {
    $stats = ['total_events' => 0, 'active_events' => 0, 'total_participants' => 0, 'active_judges' => 0];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>StyleScore Help & User Guide</title>
    <?php include_once 'includes/stylesheets.php'; ?>
    <style>
        .help-section {
            scroll-margin-top: 80px;
        }
        .workflow-step {
            position: relative;
            padding-left: 40px;
            margin-bottom: 20px;
        }
        .workflow-step::before {
            content: attr(data-step);
            position: absolute;
            left: 0;
            top: 0;
            width: 30px;
            height: 30px;
            background: #0d6efd;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 14px;
        }
        .panel-card:hover {
            transform: translateY(-5px);
            transition: transform 0.3s ease;
        }
        .search-highlight {
            background-color: yellow;
            padding: 2px 4px;
            border-radius: 3px;
        }
    </style>
</head>
<body class="bg-light">
    <?php include 'menu.php'; ?>

    <div class="container py-4">
        
        <!-- Header Section -->
        <div class="bg-gradient" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
            <div class="rounded-3 p-5 mb-4 text-center">
                <h1 class="display-4 fw-bold mb-3">
                    <i class="fas fa-question-circle me-3"></i>StyleScore Help Center
                </h1>
                <p class="fs-5 mb-3">Complete guide to managing events, participants, judges, and live scoring</p>
                <div class="row text-center mt-4">
                    <div class="col-6 col-md-3">
                        <div class="bg-white bg-opacity-20 rounded p-3">
                            <i class="fas fa-calendar-alt fa-2x mb-2"></i>
                            <div class="fw-bold"><?= $stats['total_events'] ?></div>
                            <small>Events Created</small>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="bg-white bg-opacity-20 rounded p-3">
                            <i class="fas fa-play-circle fa-2x mb-2"></i>
                            <div class="fw-bold"><?= $stats['active_events'] ?></div>
                            <small>Active Events</small>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="bg-white bg-opacity-20 rounded p-3">
                            <i class="fas fa-users fa-2x mb-2"></i>
                            <div class="fw-bold"><?= $stats['total_participants'] ?></div>
                            <small>Participants</small>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="bg-white bg-opacity-20 rounded p-3">
                            <i class="fas fa-gavel fa-2x mb-2"></i>
                            <div class="fw-bold"><?= $stats['active_judges'] ?></div>
                            <small>Judges</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search Bar -->
        <div class="mb-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="input-group input-group-lg">
                        <span class="input-group-text bg-primary text-white border-0">
                            <i class="fas fa-search"></i>
                        </span>
                        <input type="text" class="form-control border-0" id="helpSearch" 
                               placeholder="Search for help topics, features, or functions...">
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Table of Contents -->
            <div class="col-lg-3">
                <div class="card border-0 shadow-sm sticky-top" style="top: 20px;">
                    <div class="card-header bg-dark text-white">
                        <h5 class="mb-0"><i class="fas fa-list me-2"></i>Table of Contents</h5>
                    </div>
                    <div class="card-body p-0">
                        <nav class="nav flex-column">
                            <a class="nav-link" href="#getting-started"><i class="fas fa-play me-2"></i>Getting Started</a>
                            <a class="nav-link" href="#event-management"><i class="fas fa-calendar-alt me-2"></i>Event Management</a>
                            <a class="nav-link" href="#participant-management"><i class="fas fa-users me-2"></i>Participant Management</a>
                            <a class="nav-link" href="#judge-management"><i class="fas fa-gavel me-2"></i>Judge Management</a>
                            <a class="nav-link" href="#live-scoring"><i class="fas fa-star me-2"></i>Live Scoring</a>
                            <a class="nav-link" href="#results-analytics"><i class="fas fa-chart-bar me-2"></i>Results & Analytics</a>
                            <a class="nav-link" href="#admin-tools"><i class="fas fa-cogs me-2"></i>Admin Tools</a>
                            <a class="nav-link" href="#workflows"><i class="fas fa-sitemap me-2"></i>Complete Workflows</a>
                            <a class="nav-link" href="#troubleshooting"><i class="fas fa-tools me-2"></i>Troubleshooting</a>
                            <a class="nav-link" href="#faq"><i class="fas fa-question me-2"></i>FAQ</a>
                        </nav>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-lg-9">
                
                <!-- Getting Started Section -->
                <section id="getting-started" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-success text-white">
                            <h2 class="mb-0"><i class="fas fa-play me-2"></i>Getting Started</h2>
                        </div>
                        <div class="card-body">
                            <p class="lead">Welcome to StyleScore! This comprehensive guide will help you navigate through all features and functions of the system.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-6">
                                    <div class="card h-100 border-success">
                                        <div class="card-body text-center">
                                            <i class="fas fa-tachometer-alt fa-3x text-success mb-3"></i>
                                            <h5>Quick Start</h5>
                                            <p class="small">Access the main dashboard to get an overview of your system.</p>
                                            <a href="dashboard.php" class="btn btn-success btn-sm">Go to Dashboard</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="card h-100 border-info">
                                        <div class="card-body text-center">
                                            <i class="fas fa-video fa-3x text-info mb-3"></i>
                                            <h5>Video Tutorials</h5>
                                            <p class="small">Watch step-by-step video guides for common tasks.</p>
                                            <a href="#workflows" class="btn btn-info btn-sm">View Workflows</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-route me-2"></i>System Overview</h4>
                            <p>StyleScore is divided into several main areas:</p>
                            <ul class="list-group list-group-flush">
                                <li class="list-group-item d-flex align-items-center">
                                    <i class="fas fa-calendar-alt text-primary me-3"></i>
                                    <strong>Event Management:</strong> Create and manage sporting events, configure heats and runs
                                </li>
                                <li class="list-group-item d-flex align-items-center">
                                    <i class="fas fa-users text-info me-3"></i>
                                    <strong>Participant Management:</strong> Register participants and assign them to events and heats
                                </li>
                                <li class="list-group-item d-flex align-items-center">
                                    <i class="fas fa-gavel text-warning me-3"></i>
                                    <strong>Judge Management:</strong> Manage judges and their event assignments
                                </li>
                                <li class="list-group-item d-flex align-items-center">
                                    <i class="fas fa-star text-success me-3"></i>
                                    <strong>Live Scoring:</strong> Real-time scoring interface for judges
                                </li>
                                <li class="list-group-item d-flex align-items-center">
                                    <i class="fas fa-chart-bar text-secondary me-3"></i>
                                    <strong>Results & Analytics:</strong> View results, generate reports, and analyze performance
                                </li>
                            </ul>
                        </div>
                    </div>
                </section>

                <!-- Event Management Section -->
                <section id="event-management" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <h2 class="mb-0"><i class="fas fa-calendar-alt me-2"></i>Event Management</h2>
                        </div>
                        <div class="card-body">
                            <p>Create, configure, and manage all aspects of your sporting events.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-primary">
                                        <div class="card-body text-center">
                                            <i class="fas fa-plus-circle fa-2x text-primary mb-3"></i>
                                            <h6>Create New Event</h6>
                                            <p class="small">Set up a new sporting event with all configurations.</p>
                                            <a href="admin/event_create.php" class="btn btn-primary btn-sm">Create Event</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-primary">
                                        <div class="card-body text-center">
                                            <i class="fas fa-list fa-2x text-primary mb-3"></i>
                                            <h6>Manage Events</h6>
                                            <p class="small">View, edit, and manage existing events.</p>
                                            <a href="admin/events.php" class="btn btn-primary btn-sm">View Events</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-primary">
                                        <div class="card-body text-center">
                                            <i class="fas fa-fire fa-2x text-primary mb-3"></i>
                                            <h6>Heat Management</h6>
                                            <p class="small">Configure heats and assign participants.</p>
                                            <a href="admin/heat_management.php" class="btn btn-primary btn-sm">Manage Heats</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-info-circle me-2"></i>Event Creation Process</h4>
                            <div class="workflow-step" data-step="1">
                                <h6>Basic Event Information</h6>
                                <p>Enter event name, date, location, and description. Choose event type and status.</p>
                            </div>
                            <div class="workflow-step" data-step="2">
                                <h6>Configure Heats and Runs</h6>
                                <p>Set the number of heats and runs per heat. This determines the competition structure.</p>
                            </div>
                            <div class="workflow-step" data-step="3">
                                <h6>Scoring Configuration</h6>
                                <p>Define scoring ranges, decimal places, and any special scoring rules.</p>
                            </div>
                            <div class="workflow-step" data-step="4">
                                <h6>Activate Event</h6>
                                <p>Change status to "Open" to allow participant registration and judge assignments.</p>
                            </div>

                            <div class="alert alert-info">
                                <h6><i class="fas fa-lightbulb me-2"></i>Pro Tip</h6>
                                <p class="mb-0">Always configure your event completely before activating it. Once participants are registered and scoring begins, major changes become more difficult.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Participant Management Section -->
                <section id="participant-management" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-info text-white">
                            <h2 class="mb-0"><i class="fas fa-users me-2"></i>Participant Management</h2>
                        </div>
                        <div class="card-body">
                            <p>Register participants, manage their information, and assign them to events and heats.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-info">
                                        <div class="card-body text-center">
                                            <i class="fas fa-user-plus fa-2x text-info mb-3"></i>
                                            <h6>Add Participant</h6>
                                            <p class="small">Register new participants in the system.</p>
                                            <a href="admin/participant_add.php" class="btn btn-info btn-sm">Add Participant</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-info">
                                        <div class="card-body text-center">
                                            <i class="fas fa-users-cog fa-2x text-info mb-3"></i>
                                            <h6>Manage Participants</h6>
                                            <p class="small">View, edit, and manage participant information.</p>
                                            <a href="admin/participants.php" class="btn btn-info btn-sm">View Participants</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-info">
                                        <div class="card-body text-center">
                                            <i class="fas fa-clipboard-list fa-2x text-info mb-3"></i>
                                            <h6>Event Registration</h6>
                                            <p class="small">Register participants for specific events.</p>
                                            <a href="admin/participant_registration.php" class="btn btn-info btn-sm">Register</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-user-check me-2"></i>Participant Registration Process</h4>
                            <div class="workflow-step" data-step="1">
                                <h6>Add Participant Information</h6>
                                <p>Enter personal details: name, gender, date of birth, club, FIS code, license number.</p>
                            </div>
                            <div class="workflow-step" data-step="2">
                                <h6>Assign Category</h6>
                                <p>Select appropriate category based on age, skill level, or competition rules.</p>
                            </div>
                            <div class="workflow-step" data-step="3">
                                <h6>Event Registration</h6>
                                <p>Register participant for specific events and assign BIP numbers.</p>
                            </div>
                            <div class="workflow-step" data-step="4">
                                <h6>Heat Assignment</h6>
                                <p>Assign participant to appropriate heats based on category and event structure.</p>
                            </div>

                            <div class="alert alert-warning">
                                <h6><i class="fas fa-exclamation-triangle me-2"></i>Important</h6>
                                <p class="mb-0">BIP numbers must be unique within each event. The system will automatically suggest the next available number.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Judge Management Section -->
                <section id="judge-management" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-warning text-white">
                            <h2 class="mb-0"><i class="fas fa-gavel me-2"></i>Judge Management</h2>
                        </div>
                        <div class="card-body">
                            <p>Manage judges, their credentials, and assign them to events for scoring.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-warning">
                                        <div class="card-body text-center">
                                            <i class="fas fa-user-tie fa-2x text-warning mb-3"></i>
                                            <h6>Judge List</h6>
                                            <p class="small">View and manage all judges in the system.</p>
                                            <a href="admin/judges.php" class="btn btn-warning btn-sm">View Judges</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-warning">
                                        <div class="card-body text-center">
                                            <i class="fas fa-tasks fa-2x text-warning mb-3"></i>
                                            <h6>Judge Assignments</h6>
                                            <p class="small">Assign judges to events and manage their roles.</p>
                                            <a href="admin/judge_assign.php" class="btn btn-warning btn-sm">Assignments</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-warning">
                                        <div class="card-body text-center">
                                            <i class="fas fa-id-card fa-2x text-warning mb-3"></i>
                                            <h6>Judge Credentials</h6>
                                            <p class="small">Manage judge access codes and permissions.</p>
                                            <a href="admin/judge_credentials.php" class="btn btn-warning btn-sm">Credentials</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-user-cog me-2"></i>Judge Setup Process</h4>
                            <div class="workflow-step" data-step="1">
                                <h6>Add Judge Information</h6>
                                <p>Enter judge name, certification level, and contact information.</p>
                            </div>
                            <div class="workflow-step" data-step="2">
                                <h6>Generate Access Code</h6>
                                <p>Create unique access code for judge to login to scoring interface.</p>
                            </div>
                            <div class="workflow-step" data-step="3">
                                <h6>Event Assignment</h6>
                                <p>Assign judge to specific events and define their role (Judge, Head Judge, etc.).</p>
                            </div>
                            <div class="workflow-step" data-step="4">
                                <h6>Test Access</h6>
                                <p>Verify judge can access scoring interface with their credentials.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Live Scoring Section -->
                <section id="live-scoring" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-success text-white">
                            <h2 class="mb-0"><i class="fas fa-star me-2"></i>Live Scoring</h2>
                        </div>
                        <div class="card-body">
                            <p>Real-time scoring interface for judges and live scoreboard display for spectators.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-success">
                                        <div class="card-body text-center">
                                            <i class="fas fa-star fa-2x text-success mb-3"></i>
                                            <h6>Judge Scoring</h6>
                                            <p class="small">Main scoring interface for judges.</p>
                                            <a href="judge/score.php" class="btn btn-success btn-sm">Start Scoring</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-success">
                                        <div class="card-body text-center">
                                            <i class="fas fa-tv fa-2x text-success mb-3"></i>
                                            <h6>Live Scoreboard</h6>
                                            <p class="small">Public display for spectators and participants.</p>
                                            <a href="live_scoreboard_3.php" class="btn btn-success btn-sm">View Scoreboard</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-success">
                                        <div class="card-body text-center">
                                            <i class="fas fa-chart-line fa-2x text-success mb-3"></i>
                                            <h6>Summary Table</h6>
                                            <p class="small">Comprehensive scoring overview and results.</p>
                                            <a href="ajax_summary_table.php" class="btn btn-success btn-sm">View Summary</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-stopwatch me-2"></i>Scoring Process</h4>
                            <div class="workflow-step" data-step="1">
                                <h6>Judge Login</h6>
                                <p>Judge enters their access code to login to the scoring interface.</p>
                            </div>
                            <div class="workflow-step" data-step="2">
                                <h6>Select Event & Heat</h6>
                                <p>Choose the event and heat currently being judged.</p>
                            </div>
                            <div class="workflow-step" data-step="3">
                                <h6>Score Participants</h6>
                                <p>Enter scores for each participant's run using the numeric keypad or slider.</p>
                            </div>
                            <div class="workflow-step" data-step="4">
                                <h6>Submit Scores</h6>
                                <p>Confirm and submit scores. They appear immediately on the live scoreboard.</p>
                            </div>

                            <div class="alert alert-success">
                                <h6><i class="fas fa-rocket me-2"></i>Live Features</h6>
                                <ul class="mb-0">
                                    <li>Scores update in real-time on all displays</li>
                                    <li>Automatic calculation of averages and rankings</li>
                                    <li>Mobile-responsive design works on tablets and phones</li>
                                    <li>Offline capability - scores sync when connection returns</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Results & Analytics Section -->
                <section id="results-analytics" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-secondary text-white">
                            <h2 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Results & Analytics</h2>
                        </div>
                        <div class="card-body">
                            <p>Generate reports, analyze performance, and export results in various formats.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-secondary">
                                        <div class="card-body text-center">
                                            <i class="fas fa-chart-pie fa-2x text-secondary mb-3"></i>
                                            <h6>Scoring Analytics</h6>
                                            <p class="small">Detailed scoring analysis and judge performance.</p>
                                            <a href="admin/scoring_analytics.php" class="btn btn-secondary btn-sm">View Analytics</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-secondary">
                                        <div class="card-body text-center">
                                            <i class="fas fa-trophy fa-2x text-secondary mb-3"></i>
                                            <h6>Final Results</h6>
                                            <p class="small">Generate comprehensive final results and rankings.</p>
                                            <a href="admin/ajax_generate_results.php" class="btn btn-secondary btn-sm">Generate Results</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-secondary">
                                        <div class="card-body text-center">
                                            <i class="fas fa-download fa-2x text-secondary mb-3"></i>
                                            <h6>Export Data</h6>
                                            <p class="small">Export results to Excel, CSV, or PDF formats.</p>
                                            <a href="admin/export.php" class="btn btn-secondary btn-sm">Export Data</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-file-alt me-2"></i>Available Reports</h4>
                            <div class="row">
                                <div class="col-md-6">
                                    <ul class="list-group list-group-flush">
                                        <li class="list-group-item"><i class="fas fa-medal text-warning me-2"></i>Final Rankings</li>
                                        <li class="list-group-item"><i class="fas fa-fire text-danger me-2"></i>Heat Results</li>
                                        <li class="list-group-item"><i class="fas fa-chart-line text-info me-2"></i>Score Progressions</li>
                                        <li class="list-group-item"><i class="fas fa-balance-scale text-secondary me-2"></i>Judge Consistency</li>
                                    </ul>
                                </div>
                                <div class="col-md-6">
                                    <ul class="list-group list-group-flush">
                                        <li class="list-group-item"><i class="fas fa-users text-primary me-2"></i>Participant Lists</li>
                                        <li class="list-group-item"><i class="fas fa-certificate text-success me-2"></i>Certificates</li>
                                        <li class="list-group-item"><i class="fas fa-table text-dark me-2"></i>Summary Tables</li>
                                        <li class="list-group-item"><i class="fas fa-file-excel text-success me-2"></i>Excel Exports</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Admin Tools Section -->
                <section id="admin-tools" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-dark text-white">
                            <h2 class="mb-0"><i class="fas fa-cogs me-2"></i>Administrative Tools</h2>
                        </div>
                        <div class="card-body">
                            <p>Advanced administrative functions for system management and maintenance.</p>
                            
                            <div class="row g-3 mb-4">
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-dark">
                                        <div class="card-body text-center">
                                            <i class="fas fa-tachometer-alt fa-2x text-dark mb-3"></i>
                                            <h6>Admin Dashboard</h6>
                                            <p class="small">Central administrative control panel.</p>
                                            <a href="admin/dashboard.php" class="btn btn-dark btn-sm">Admin Panel</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-dark">
                                        <div class="card-body text-center">
                                            <i class="fas fa-edit fa-2x text-dark mb-3"></i>
                                            <h6>Score Override</h6>
                                            <p class="small">Head judge score correction and override.</p>
                                            <a href="admin/head_judge_override.php" class="btn btn-dark btn-sm">Override Scores</a>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card panel-card h-100 border-dark">
                                        <div class="card-body text-center">
                                            <i class="fas fa-users-cog fa-2x text-dark mb-3"></i>
                                            <h6>User Management</h6>
                                            <p class="small">Manage system users and permissions.</p>
                                            <a href="admin/users.php" class="btn btn-dark btn-sm">Manage Users</a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <h4><i class="fas fa-shield-alt me-2"></i>Administrative Functions</h4>
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead class="table-dark">
                                        <tr>
                                            <th>Function</th>
                                            <th>Description</th>
                                            <th>Access Level</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td><i class="fas fa-edit me-2"></i>Score Override</td>
                                            <td>Correct or modify submitted scores</td>
                                            <td><span class="badge bg-warning">Head Judge</span></td>
                                        </tr>
                                        <tr>
                                            <td><i class="fas fa-database me-2"></i>Data Backup</td>
                                            <td>Create system backups and restore data</td>
                                            <td><span class="badge bg-danger">Administrator</span></td>
                                        </tr>
                                        <tr>
                                            <td><i class="fas fa-chart-bar me-2"></i>System Analytics</td>
                                            <td>View system usage and performance metrics</td>
                                            <td><span class="badge bg-info">Administrator</span></td>
                                        </tr>
                                        <tr>
                                            <td><i class="fas fa-users-cog me-2"></i>User Management</td>
                                            <td>Create, modify, and delete user accounts</td>
                                            <td><span class="badge bg-danger">Administrator</span></td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Complete Workflows Section -->
                <section id="workflows" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-info text-white">
                            <h2 class="mb-0"><i class="fas fa-sitemap me-2"></i>Complete Workflows</h2>
                        </div>
                        <div class="card-body">
                            <p>Step-by-step workflows for common tasks and complete event management processes.</p>

                            <!-- Workflow Tabs -->
                            <ul class="nav nav-tabs mb-4" id="workflowTabs" role="tablist">
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link active" id="new-event-tab" data-bs-toggle="tab" data-bs-target="#new-event" type="button" role="tab">
                                        <i class="fas fa-plus me-2"></i>New Event Setup
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="live-scoring-tab" data-bs-toggle="tab" data-bs-target="#live-scoring-workflow" type="button" role="tab">
                                        <i class="fas fa-star me-2"></i>Live Scoring
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="final-results-tab" data-bs-toggle="tab" data-bs-target="#final-results" type="button" role="tab">
                                        <i class="fas fa-trophy me-2"></i>Final Results
                                    </button>
                                </li>
                            </ul>

                            <div class="tab-content" id="workflowTabContent">
                                <!-- New Event Setup Workflow -->
                                <div class="tab-pane fade show active" id="new-event" role="tabpanel">
                                    <h4><i class="fas fa-calendar-plus me-2"></i>Complete New Event Setup</h4>
                                    <div class="workflow-step" data-step="1">
                                        <h6>Create Event</h6>
                                        <p>Go to <a href="admin/event_create.php">Create New Event</a> and fill in basic information:</p>
                                        <ul>
                                            <li>Event name and date</li>
                                            <li>Location and description</li>
                                            <li>Number of heats and runs per heat</li>
                                            <li>Scoring configuration</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="2">
                                        <h6>Add Participants</h6>
                                        <p>Use <a href="admin/participant_add.php">Add Participant</a> to register competitors:</p>
                                        <ul>
                                            <li>Enter participant details</li>
                                            <li>Assign categories</li>
                                            <li>Register for the event</li>
                                            <li>Assign BIP numbers</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="3">
                                        <h6>Assign Judges</h6>
                                        <p>Go to <a href="admin/judge_assign.php">Judge Assignments</a>:</p>
                                        <ul>
                                            <li>Assign judges to the event</li>
                                            <li>Generate access codes</li>
                                            <li>Test judge login access</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="4">
                                        <h6>Configure Heat Assignments</h6>
                                        <p>Use <a href="admin/heat_management.php">Heat Management</a>:</p>
                                        <ul>
                                            <li>Assign participants to heats</li>
                                            <li>Balance heat sizes</li>
                                            <li>Verify assignments</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="5">
                                        <h6>Activate Event</h6>
                                        <p>Change event status to "Open" or "Active" to begin scoring.</p>
                                    </div>
                                </div>

                                <!-- Live Scoring Workflow -->
                                <div class="tab-pane fade" id="live-scoring-workflow" role="tabpanel">
                                    <h4><i class="fas fa-stopwatch me-2"></i>Live Scoring Process</h4>
                                    <div class="workflow-step" data-step="1">
                                        <h6>Prepare Scoring Station</h6>
                                        <ul>
                                            <li>Set up tablets/devices for judges</li>
                                            <li>Ensure internet connectivity</li>
                                            <li>Test <a href="judge/score.php">Judge Scoring</a> interface</li>
                                            <li>Display <a href="live_scoreboard_3.php">Live Scoreboard</a> for spectators</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="2">
                                        <h6>Judge Login</h6>
                                        <p>Each judge logs in using their assigned access code.</p>
                                    </div>
                                    <div class="workflow-step" data-step="3">
                                        <h6>Start Scoring</h6>
                                        <ul>
                                            <li>Select event and current heat</li>
                                            <li>Begin scoring participant runs</li>
                                            <li>Submit scores after each run</li>
                                            <li>Monitor live scoreboard for real-time updates</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="4">
                                        <h6>Heat Completion</h6>
                                        <ul>
                                            <li>Verify all scores submitted</li>
                                            <li>Check for any missing or incorrect scores</li>
                                            <li>Use <a href="admin/head_judge_override.php">Score Override</a> if corrections needed</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="5">
                                        <h6>Progress to Next Heat</h6>
                                        <p>Advance to next heat and repeat the process until all heats are completed.</p>
                                    </div>
                                </div>

                                <!-- Final Results Workflow -->
                                <div class="tab-pane fade" id="final-results" role="tabpanel">
                                    <h4><i class="fas fa-medal me-2"></i>Final Results Generation</h4>
                                    <div class="workflow-step" data-step="1">
                                        <h6>Verify All Scores</h6>
                                        <p>Use <a href="admin/scoring_analytics.php">Scoring Analytics</a> to verify completion:</p>
                                        <ul>
                                            <li>Check all heats have scores</li>
                                            <li>Verify judge consistency</li>
                                            <li>Correct any issues found</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="2">
                                        <h6>Generate Results</h6>
                                        <p>Go to <a href="admin/ajax_generate_results.php">Generate Results</a>:</p>
                                        <ul>
                                            <li>Select event and categories</li>
                                            <li>Choose scoring method</li>
                                            <li>Configure display options</li>
                                            <li>Generate final rankings</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="3">
                                        <h6>Review and Approve</h6>
                                        <ul>
                                            <li>Review calculated results</li>
                                            <li>Verify rankings are correct</li>
                                            <li>Make any necessary adjustments</li>
                                        </ul>
                                    </div>
                                    <div class="workflow-step" data-step="4">
                                        <h6>Export and Distribute</h6>
                                        <ul>
                                            <li>Export results to PDF/Excel</li>
                                            <li>Generate certificates</li>
                                            <li>Publish on live scoreboard</li>
                                            <li>Distribute to participants and officials</li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Pages & Features Section -->
                <section id="pages-features" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <h2 class="mb-0"><i class="fas fa-sitemap me-2"></i>Pages & Features</h2>
                        </div>
                        <div class="card-body">
                            <p>A practical index of the main pages across the system and what you can do with each. Use this to jump directly to the right tool.</p>

                            <div class="row g-3 mb-4">
                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-primary">
                                        <div class="card-body">
                                            <h5><i class="fas fa-home me-2 text-primary"></i>Core</h5>
                                            <ul class="mb-0">
                                                <li><a href="dashboard.php">dashboard.php</a> — Overview, quick links, status.</li>
                                                <li><a href="login.php">login.php</a> / <a href="logout.php">logout.php</a> — Access control.</li>
                                                <li><a href="live_score.php">live_score.php</a> — Judge scoring panel.</li>
                                                <li><a href="live_scoreboard.php">live_scoreboard.php</a> · <a href="live_scoreboard_1.php">_1</a> · <a href="live_scoreboard_2.php">_2</a> · <a href="live_scoreboard_3.php">_3</a> · <a href="live_scoreboard_tabs.php">tabs</a> — Audience views.</li>
                                                <li><a href="public_event_dashboard.php">public_event_dashboard.php</a> · <a href="public_event_dashboard2.php">_2</a> — Public status pages.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-primary">
                                        <div class="card-body">
                                            <h5><i class="fas fa-calendar-alt me-2 text-primary"></i>Admin — Events</h5>
                                            <ul class="mb-0">
                                                <li><a href="admin/event_create.php">event_create.php</a> — Create event, rounds, runs, formats.</li>
                                                <li><a href="admin/events.php">events.php</a> — List/edit events.</li>
                                                <li><a href="admin/event_edit.php">event_edit.php</a> — Update details, presets, branding.</li>
                                                <li><a href="admin/event_start_list.php">event_start_list.php</a> — Generate/print start orders.</li>
                                                <li><a href="admin/event_dashboard.php">event_dashboard.php</a> — Event control & monitoring.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-success">
                                        <div class="card-body">
                                            <h5><i class="fas fa-users me-2 text-success"></i>Admin — Participants</h5>
                                            <ul class="mb-0">
                                                <li><a href="admin/participants.php">participants.php</a> — Bulk import, manage athletes.</li>
                                                <li><a href="admin/participant_add.php">participant_add.php</a> — Manual add.</li>
                                                <li><a href="admin/participant_edit.php">participant_edit.php</a> — Edit records.</li>
                                                <li><a href="admin/bib_assign.php">bib_assign.php</a> / <a href="admin/bib_distribute.php">bib_distribute.php</a> — Bib tools.</li>
                                                <li><a href="admin/export_assignments.php">export_assignments.php</a> · <a href="admin/export_assignments_pdf.php">_pdf</a> — Export rosters.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-warning">
                                        <div class="card-body">
                                            <h5><i class="fas fa-gavel me-2 text-warning"></i>Admin — Judges</h5>
                                            <ul class="mb-0">
                                                <li><a href="admin/judge_add.php">judge_add.php</a> — Add judges, mark Head Judge.</li>
                                                <li><a href="admin/judges.php">judges.php</a> — Manage judge accounts.</li>
                                                <li><a href="admin/judge_assign.php">judge_assign.php</a> — Assign judges to heats.</li>
                                                <li><a href="admin/head_judge_override.php">head_judge_override.php</a> — Approve/override scores.</li>
                                                <li><a href="admin/judge_export.php">judge_export.php</a> — Export judges.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-info">
                                        <div class="card-body">
                                            <h5><i class="fas fa-fire me-2 text-info"></i>Admin — Heats & Flow</h5>
                                            <ul class="mb-0">
                                                <li><a href="admin/heats_configure.php">heats_configure.php</a> — Configure heat sizes.</li>
                                                <li><a href="admin/heat_management_helper.php">heat_management_helper.php</a> — Move/reseed tools.</li>
                                                <li><a href="admin/heat_flow_preview.php">heat_flow_preview.php</a> — Preview progression.</li>
                                                <li><a href="admin/results_panel.php">results_panel.php</a> — Finalize/publish results.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-secondary">
                                        <div class="card-body">
                                            <h5><i class="fas fa-sliders-h me-2 text-secondary"></i>Admin — Scoring & Styling</h5>
                                            <ul class="mb-0">
                                                <li><a href="admin/scoring_formats.php">scoring_formats.php</a> — Define formats & weights.</li>
                                                <li><a href="admin/scoring_analytics.php">scoring_analytics.php</a> — Judge variance, distributions.</li>
                                                <li><a href="admin/figures_management.php">figures_management.php</a> — Figures dictionary.</li>
                                                <li><a href="admin/results_styling.php">results_styling.php</a> — Branding & layouts.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-dark">
                                        <div class="card-body">
                                            <h5><i class="fas fa-coins me-2"></i>Admin — Tokens & Users</h5>
                                            <ul class="mb-0">
                                                <li><a href="admin/token_management.php">token_management.php</a> — Actions, packages, budgets.</li>
                                                <li><a href="admin/user_management.php">user_management.php</a> — Roles, permissions.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-dark">
                                        <div class="card-body">
                                            <h5><i class="fas fa-broadcast-tower me-2"></i>Public & Live</h5>
                                            <ul class="mb-0">
                                                <li><a href="live_scoreboard.php">live_scoreboard.php</a> — Standard live board.</li>
                                                <li><a href="live_scoreboard_1.php">live_scoreboard_1.php</a> · <a href="live_scoreboard_2.php">_2</a> · <a href="live_scoreboard_3.php">_3</a> · <a href="live_scoreboard_tabs.php">tabs</a> — Alternative views.</li>
                                                <li><a href="public_event_dashboard.php">public_event_dashboard.php</a> · <a href="public_event_dashboard2.php">_2</a> — Public dashboards.</li>
                                                <li><a href="print-template.html">print-template.html</a> — Print layouts.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card panel-card h-100 border-dark">
                                        <div class="card-body">
                                            <h5><i class="fas fa-code me-2"></i>APIs & Feeds</h5>
                                            <ul class="mb-0">
                                                <li><a href="get_results.php">get_results.php</a> — JSON results feed.</li>
                                                <li><a href="api/public_dashboard_api.php">api/public_dashboard_api.php</a> — Public event API.</li>
                                                <li><a href="api/start_list_api.php">api/start_list_api.php</a> — Start list.
                                                </li>
                                                <li><a href="api/tables_new_api.php">api/tables_new_api.php</a> — Tables composite.</li>
                                                <li><a href="api/result_configurations_api.php">api/result_configurations_api.php</a> — Result config.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Troubleshooting Section -->
                <section id="troubleshooting" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-danger text-white">
                            <h2 class="mb-0"><i class="fas fa-tools me-2"></i>Troubleshooting</h2>
                        </div>
                        <div class="card-body">
                            <h4><i class="fas fa-exclamation-triangle me-2"></i>Common Issues</h4>
                            
                            <div class="accordion" id="troubleshootingAccordion">
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="scoring-issues">
                                        <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#collapse-scoring">
                                            <i class="fas fa-star text-danger me-2"></i>Scoring Issues
                                        </button>
                                    </h2>
                                    <div id="collapse-scoring" class="accordion-collapse collapse show" data-bs-parent="#troubleshootingAccordion">
                                        <div class="accordion-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <h6>Judge Can't Access Scoring</h6>
                                                    <ul>
                                                        <li>Verify access code is correct</li>
                                                        <li>Check judge is assigned to event</li>
                                                        <li>Ensure event status is "Active"</li>
                                                        <li>Check internet connection</li>
                                                    </ul>
                                                </div>
                                                <div class="col-md-6">
                                                    <h6>Scores Not Appearing</h6>
                                                    <ul>
                                                        <li>Refresh the scoreboard page</li>
                                                        <li>Check if scores were submitted</li>
                                                        <li>Verify correct event/heat selected</li>
                                                        <li>Clear browser cache</li>
                                                    </ul>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="participant-issues">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapse-participants">
                                            <i class="fas fa-users text-info me-2"></i>Participant Issues
                                        </button>
                                    </h2>
                                    <div id="collapse-participants" class="accordion-collapse collapse" data-bs-parent="#troubleshootingAccordion">
                                        <div class="accordion-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <h6>Participant Not Listed</h6>
                                                    <ul>
                                                        <li>Check participant is registered for event</li>
                                                        <li>Verify correct category selection</li>
                                                        <li>Ensure participant is assigned to a heat</li>
                                                        <li>Check BIP number assignment</li>
                                                    </ul>
                                                </div>
                                                <div class="col-md-6">
                                                    <h6>Wrong Heat Assignment</h6>
                                                    <ul>
                                                        <li>Go to Heat Management</li>
                                                        <li>Reassign participant to correct heat</li>
                                                        <li>Update live displays</li>
                                                        <li>Notify judges of changes</li>
                                                    </ul>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="technical-issues">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapse-technical">
                                            <i class="fas fa-laptop text-secondary me-2"></i>Technical Issues
                                        </button>
                                    </h2>
                                    <div id="collapse-technical" class="accordion-collapse collapse" data-bs-parent="#troubleshootingAccordion">
                                        <div class="accordion-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <h6>Slow Performance</h6>
                                                    <ul>
                                                        <li>Check internet connection speed</li>
                                                        <li>Close unnecessary browser tabs</li>
                                                        <li>Clear browser cache and cookies</li>
                                                        <li>Use updated browser version</li>
                                                    </ul>
                                                </div>
                                                <div class="col-md-6">
                                                    <h6>Display Issues</h6>
                                                    <ul>
                                                        <li>Refresh page (F5 or Ctrl+R)</li>
                                                        <li>Check device screen resolution</li>
                                                        <li>Try different browser</li>
                                                        <li>Disable browser extensions</li>
                                                    </ul>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="alert alert-info mt-4">
                                <h6><i class="fas fa-phone me-2"></i>Need More Help?</h6>
                                <p class="mb-0">If you're experiencing issues not covered here, contact technical support or refer to the detailed documentation for your specific setup.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- FAQ Section -->
                <section id="faq" class="help-section mb-5">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <h2 class="mb-0"><i class="fas fa-question me-2"></i>Frequently Asked Questions</h2>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5>General Questions</h5>
                                    <div class="mb-3">
                                        <h6><i class="fas fa-question-circle text-primary me-2"></i>How many judges can score simultaneously?</h6>
                                        <p>The system supports unlimited judges scoring simultaneously. Each judge gets their own interface and scores are combined in real-time.</p>
                                    </div>
                                    <div class="mb-3">
                                        <h6><i class="fas fa-question-circle text-primary me-2"></i>Can I modify scores after submission?</h6>
                                        <p>Yes, head judges and administrators can use the Score Override function to correct submitted scores.</p>
                                    </div>
                                    <div class="mb-3">
                                        <h6><i class="fas fa-question-circle text-primary me-2"></i>How are ties handled in rankings?</h6>
                                        <p>Tied participants receive the same rank, and the next rank is skipped accordingly (e.g., if two participants tie for 2nd, the next is ranked 4th).</p>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <h5>Technical Questions</h5>
                                    <div class="mb-3">
                                        <h6><i class="fas fa-question-circle text-success me-2"></i>What devices are supported?</h6>
                                        <p>StyleScore works on any device with a modern web browser - tablets, smartphones, laptops, and desktop computers.</p>
                                    </div>
                                    <div class="mb-3">
                                        <h6><i class="fas fa-question-circle text-success me-2"></i>Does it work offline?</h6>
                                        <p>The scoring interface has limited offline capability. Scores are cached locally and sync when connection is restored.</p>
                                    </div>
                                    <div class="mb-3">
                                        <h6><i class="fas fa-question-circle text-success me-2"></i>How is data backed up?</h6>
                                        <p>All data is automatically saved to the database. Administrators can create manual backups and exports at any time.</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

            </div>
        </div>

        <!-- Help Footer -->
        <div class="text-center mt-5 p-4 bg-dark text-white rounded">
            <h5><i class="fas fa-headset me-2"></i>Still Need Help?</h5>
            <p class="mb-3">Our support team is here to help you get the most out of StyleScore.</p>
            <div class="row">
                <div class="col-md-4">
                    <i class="fas fa-envelope fa-2x mb-2"></i>
                    <h6>Email Support</h6>
                    <p class="small">support@stylescore.com</p>
                </div>
                <div class="col-md-4">
                    <i class="fas fa-phone fa-2x mb-2"></i>
                    <h6>Phone Support</h6>
                    <p class="small">+1 (555) 123-4567</p>
                </div>
                <div class="col-md-4">
                    <i class="fas fa-clock fa-2x mb-2"></i>
                    <h6>Support Hours</h6>
                    <p class="small">Mon-Fri 9AM-6PM EST</p>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Search functionality
        document.getElementById('helpSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const sections = document.querySelectorAll('.help-section');
            
            sections.forEach(section => {
                const content = section.textContent.toLowerCase();
                if (content.includes(searchTerm) || searchTerm === '') {
                    section.style.display = 'block';
                    
                    // Highlight search terms
                    if (searchTerm !== '') {
                        highlightText(section, searchTerm);
                    } else {
                        removeHighlights(section);
                    }
                } else {
                    section.style.display = 'none';
                }
            });
        });

        // Smooth scrolling for navigation links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Highlight search terms
        function highlightText(element, searchTerm) {
            removeHighlights(element);
            const walker = document.createTreeWalker(
                element,
                NodeFilter.SHOW_TEXT,
                null,
                false
            );
            
            const textNodes = [];
            let node;
            while (node = walker.nextNode()) {
                textNodes.push(node);
            }
            
            textNodes.forEach(textNode => {
                const text = textNode.textContent;
                const regex = new RegExp(`(${searchTerm})`, 'gi');
                if (regex.test(text)) {
                    const highlightedText = text.replace(regex, '<span class="search-highlight">$1</span>');
                    const wrapper = document.createElement('div');
                    wrapper.innerHTML = highlightedText;
                    textNode.parentNode.replaceChild(wrapper, textNode);
                }
            });
        }

        // Remove highlights
        function removeHighlights(element) {
            const highlights = element.querySelectorAll('.search-highlight');
            highlights.forEach(highlight => {
                highlight.outerHTML = highlight.innerHTML;
            });
        }

        // Auto-update TOC active states
        window.addEventListener('scroll', function() {
            const sections = document.querySelectorAll('.help-section');
            const navLinks = document.querySelectorAll('.nav-link[href^="#"]');
            
            let current = '';
            sections.forEach(section => {
                const sectionTop = section.offsetTop;
                const sectionHeight = section.clientHeight;
                if (window.pageYOffset >= sectionTop - 200) {
                    current = section.getAttribute('id');
                }
            });
            
            navLinks.forEach(link => {
                link.classList.remove('active');
                if (link.getAttribute('href') === '#' + current) {
                    link.classList.add('active');
                }
            });
        });

        // Panel card hover effects
        document.querySelectorAll('.panel-card').forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-5px)';
                this.style.transition = 'transform 0.3s ease';
            });
            
            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });
        });
    </script>
</body>
</html>