<?php
/*
API: GET /v2/api/get_categories.php
Description:
- Returns distinct participant categories for a given event.

Query parameters:
- event_id (required, int): The event ID.

Responses (application/json):
- Success:
  {
    "success": true,
    "categories": ["Pro", "Amateur"],
    "count": 2
  }
- Error:
  {
    "success": false,
    "message": "No event ID provided"
  }

Usage examples:

1) curl (Windows)
- Using curl.exe:
  curl.exe "http://localhost/v2/api/get_categories.php?event_id=123"

2) PowerShell
  Invoke-RestMethod -Uri "http://localhost/v2/api/get_categories.php?event_id=123"

3) Browser JavaScript (fetch)
  fetch('/v2/api/get_categories.php?event_id=123')
    .then(r => r.json())
    .then(({ success, categories, count, message }) => {
      if (!success) { console.error(message); return; }
      console.log('Count:', count, 'Categories:', categories);
    })
    .catch(console.error);

4) PHP
  <?php
  $url = 'http://localhost/v2/api/get_categories.php?event_id=123';
  $json = file_get_contents($url);
  $data = json_decode($json, true);
  if (!empty($data['success'])) {
      foreach ($data['categories'] as $cat) {
          echo $cat, PHP_EOL;
      }
  } else {
      error_log($data['message'] ?? 'Unknown error');
  }
  ?>

Notes:
- Ensure the event_id exists; otherwise you'll get success=false.
*/
header('Content-Type: application/json');
include '../includes/db.php';

$event_id = $_GET['event_id'] ?? null;

try {
    if (!$event_id) {
        echo json_encode([
            'success' => false,
            'message' => 'No event ID provided'
        ]);
        exit;
    }

    $categories_stmt = $pdo->prepare("SELECT DISTINCT category FROM event_participants WHERE event_id = ? AND category IS NOT NULL AND category != '' ORDER BY category");
    $categories_stmt->execute([$event_id]);
    $categories = $categories_stmt->fetchAll(PDO::FETCH_COLUMN);

    echo json_encode([
        'success' => true,
        'categories' => $categories,
        'count' => count($categories)
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ]);
}