#!/usr/bin/env php
<?php
/**
 * StyleScore Stylesheet Updater
 * This script updates all files to use the centralized stylesheet system
 */

// Define the old patterns to replace
$oldPatterns = [
    // Pattern 1: Full Bootstrap + Font Awesome 6.0.0
    '/<!-- style skin holder -->\s*<link[^>]*bootswatch[^>]*>\s*<link[^>]*backgrounds\.css[^>]*>\s*<link[^>]*font-awesome\/6\.0\.0[^>]*>/s',
    
    // Pattern 2: Just Font Awesome 6.0.0
    '/<link[^>]*font-awesome\/6\.0\.0[^>]*>/s',
    
    // Pattern 3: Bootstrap theme line
    '/<link[^>]*bootswatch[^>]*bootstrap\.min\.css[^>]*>/s'
];

// The replacement for each pattern
$newReplacement = '<?php include_once \'../includes/stylesheets.php\'; ?>';
$relativeReplacement = '<?php include_once \'includes/stylesheets.php\'; ?>';

// Files to process (admin directory)
$adminFiles = [
    'admin/admin_heat_config.php',
    'admin/bib_assign.php', 
    'admin/bib_edit.php',
    'admin/bib_distribute.php',
    'admin/event_categories.php',
    'admin/event_start_list.php',
    'admin/event_dashboard.php',
    'admin/head_judge_override.php',
    'admin/head_judge_override_test.php',
    'admin/heat_flow_preview.php',
    'admin/heat_management_modal.php',
    'admin/heats_configure.php',
    'admin/judge_export.php',
    'admin/judge_assign.php',
    'admin/login.php',
    'admin/scoring_format_view.php',
    'admin/scoring_formats.php',
    'admin/scoring_analytics.php',
    'admin/participants_updated.php',
    'admin/results_panel.php'
];

// Root level files  
$rootFiles = [
    'login.php',
    'live_score.php',
    'live_scoreboard.php'
];

function updateFile($filepath, $useRelative = false) {
    global $oldPatterns, $newReplacement, $relativeReplacement;
    
    if (!file_exists($filepath)) {
        echo "⚠️  File not found: $filepath\n";
        return false;
    }
    
    $content = file_get_contents($filepath);
    $originalContent = $content;
    $replacement = $useRelative ? $relativeReplacement : $newReplacement;
    
    // Check for old Font Awesome 6.0.0 references
    if (preg_match('/font-awesome\/6\.0\.0/', $content)) {
        // Replace old Font Awesome with new stylesheet include
        $content = preg_replace(
            '/<link[^>]*font-awesome\/6\.0\.0[^>]*>/',
            '',  // Remove old font-awesome
            $content
        );
        
        // If no stylesheet include exists yet, add it after title
        if (!strpos($content, 'stylesheets.php')) {
            $content = preg_replace(
                '/(<title>[^<]*<\/title>)/i',
                '$1' . "\n    " . $replacement,
                $content
            );
        }
    }
    
    // Remove duplicate Bootstrap if any
    if (preg_match_all('/<link[^>]*bootswatch[^>]*bootstrap\.min\.css[^>]*>/', $content, $matches)) {
        if (count($matches[0]) > 0) {
            // Remove all bootstrap includes as stylesheets.php handles it
            $content = preg_replace('/<link[^>]*bootswatch[^>]*bootstrap\.min\.css[^>]*>/', '', $content);
            
            if (!strpos($content, 'stylesheets.php')) {
                $content = preg_replace(
                    '/(<title>[^<]*<\/title>)/i', 
                    '$1' . "\n    " . $replacement,
                    $content
                );
            }
        }
    }
    
    // Clean up multiple newlines
    $content = preg_replace('/\n\s*\n\s*\n/', "\n\n", $content);
    
    if ($content !== $originalContent) {
        file_put_contents($filepath, $content);
        echo "✅ Updated: $filepath\n";
        return true;
    } else {
        echo "ℹ️  No changes needed: $filepath\n";
        return false;
    }
}

echo "🔄 Starting StyleScore stylesheet standardization...\n\n";

$updatedCount = 0;

// Process admin files
echo "📁 Updating admin files...\n";
foreach ($adminFiles as $file) {
    if (updateFile($file, false)) {
        $updatedCount++;
    }
}

echo "\n📁 Updating root files...\n";
foreach ($rootFiles as $file) {
    if (updateFile($file, true)) {
        $updatedCount++;
    }
}

echo "\n🎉 Stylesheet standardization complete!\n";
echo "📊 Updated $updatedCount files\n";
echo "✨ All files now use Font Awesome 6.4.0 and consistent Bootstrap themes\n";
?>
