<?php
session_start();
require_once 'includes/db.php';
require_once 'includes/permissions.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$message = '';
$msg_type = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_profile':
                $message = handleProfileUpdate($pdo, $user_id, $_POST);
                $msg_type = strpos($message, 'success') === 0 ? 'success' : 'danger';
                break;
            
            case 'change_password':
                $message = handlePasswordChange($pdo, $user_id, $_POST);
                $msg_type = strpos($message, 'success') === 0 ? 'success' : 'danger';
                break;
            
            case 'upload_avatar':
                $message = handleAvatarUpload($pdo, $user_id, $_FILES);
                $msg_type = strpos($message, 'success') === 0 ? 'success' : 'danger';
                break;
        }
    }
}

// Get user data
try {
    $stmt = $pdo->prepare("
        SELECT u.id, u.username, u.role, u.created_at, u.last_login,
               p.full_name, p.email, p.phone, p.address, 
               p.emergency_contact, p.emergency_phone, p.notes, p.avatar, p.is_active
        FROM users u
        LEFT JOIN user_profiles p ON u.id = p.user_id
        WHERE u.id = ?
    ");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        die('User not found');
    }
    
    // Get assigned events
    $events_stmt = $pdo->prepare("
        SELECT e.id, e.name, e.date, e.location, e.status,
               uae.assigned_at
        FROM user_assigned_events uae
        JOIN events e ON uae.event_id = e.id
        WHERE uae.user_id = ?
        ORDER BY e.date DESC
    ");
    $events_stmt->execute([$user_id]);
    $assigned_events = $events_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get user permissions
    $user_permissions = getRolePermissions($user['role']);
    
    // Group permissions by category
    $permissions_grouped = [];
    foreach ($user_permissions as $perm) {
        $category = $perm['category'] ?? 'general';
        if (!isset($permissions_grouped[$category])) {
            $permissions_grouped[$category] = [];
        }
        $permissions_grouped[$category][] = $perm;
    }
    
    // Get token balance and statistics if token system exists
    $token_balance = 0;
    $total_spent = 0;
    $total_earned = 0;
    $recent_transactions = [];
    try {
        // Get budget through group membership
        $budget_stmt = $pdo->prepare("
            SELECT b.id, b.balance, b.total_spent 
            FROM budgets b
            JOIN user_group_memberships ugm ON b.group_id = ugm.group_id
            WHERE ugm.user_id = ?
            LIMIT 1
        ");
        $budget_stmt->execute([$user_id]);
        $budget = $budget_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($budget) {
            $token_balance = $budget['balance'] ?? 0;
            $total_spent = $budget['total_spent'] ?? 0;
            
            // Get recent transactions
            $trans_stmt = $pdo->prepare("
                SELECT tt.*, ta.action_name, ta.action_description
                FROM token_transactions tt
                LEFT JOIN token_actions ta ON tt.action_id = ta.id
                WHERE tt.budget_id = ?
                ORDER BY tt.created_at DESC
                LIMIT 20
            ");
            $trans_stmt->execute([$budget['id']]);
            $recent_transactions = $trans_stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Calculate total earned (credits and purchases)
            $earned_stmt = $pdo->prepare("
                SELECT SUM(amount) as total_earned
                FROM token_transactions
                WHERE budget_id = ? AND transaction_type IN ('credit', 'purchase')
            ");
            $earned_stmt->execute([$budget['id']]);
            $earned_result = $earned_stmt->fetch(PDO::FETCH_ASSOC);
            $total_earned = $earned_result['total_earned'] ?? 0;
        }
    } catch (Exception $e) {
        // Token system not available
    }
    
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

// Helper functions
function handleProfileUpdate($pdo, $user_id, $data) {
    try {
        $full_name = trim($data['full_name'] ?? '');
        $email = trim($data['email'] ?? '');
        $phone = trim($data['phone'] ?? '');
        $address = trim($data['address'] ?? '');
        $emergency_contact = trim($data['emergency_contact'] ?? '');
        $emergency_phone = trim($data['emergency_phone'] ?? '');
        
        // Check if profile exists
        $check = $pdo->prepare("SELECT user_id FROM user_profiles WHERE user_id = ?");
        $check->execute([$user_id]);
        
        if ($check->fetch()) {
            // Update existing profile
            $stmt = $pdo->prepare("
                UPDATE user_profiles 
                SET full_name = ?, email = ?, phone = ?, address = ?, 
                    emergency_contact = ?, emergency_phone = ?, updated_at = NOW()
                WHERE user_id = ?
            ");
            $stmt->execute([$full_name, $email, $phone, $address, $emergency_contact, $emergency_phone, $user_id]);
        } else {
            // Create new profile
            $stmt = $pdo->prepare("
                INSERT INTO user_profiles (user_id, full_name, email, phone, address, emergency_contact, emergency_phone, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([$user_id, $full_name, $email, $phone, $address, $emergency_contact, $emergency_phone]);
        }
        
        return 'success: Profile updated successfully';
        
    } catch (Exception $e) {
        return 'error: ' . $e->getMessage();
    }
}

function handlePasswordChange($pdo, $user_id, $data) {
    try {
        $current_password = $data['current_password'] ?? '';
        $new_password = $data['new_password'] ?? '';
        $confirm_password = $data['confirm_password'] ?? '';
        
        if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
            return 'error: All password fields are required';
        }
        
        if ($new_password !== $confirm_password) {
            return 'error: New passwords do not match';
        }
        
        if (strlen($new_password) < 4) {
            return 'error: Password must be at least 4 characters long';
        }
        
        // Verify current password
        $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $stored_password = $stmt->fetchColumn();
        
        if (!password_verify($current_password, $stored_password)) {
            return 'error: Current password is incorrect';
        }
        
        // Update password
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        $update_stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $update_stmt->execute([$hashed_password, $user_id]);
        
        return 'success: Password changed successfully';
        
    } catch (Exception $e) {
        return 'error: ' . $e->getMessage();
    }
}

function handleAvatarUpload($pdo, $user_id, $files) {
    try {
        if (!isset($files['avatar']) || $files['avatar']['error'] === UPLOAD_ERR_NO_FILE) {
            return 'error: No file selected';
        }
        
        if ($files['avatar']['error'] !== UPLOAD_ERR_OK) {
            return 'error: Upload failed';
        }
        
        $upload_dir = 'uploads/avatars/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        // Validate file type
        $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($finfo, $files['avatar']['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mime_type, $allowed_types)) {
            return 'error: Invalid file type. Only JPG, PNG, GIF, and WebP allowed';
        }
        
        // Validate file size (max 5MB)
        if ($files['avatar']['size'] > 5 * 1024 * 1024) {
            return 'error: File too large. Maximum size is 5MB';
        }
        
        // Get old avatar to delete
        $old_avatar_stmt = $pdo->prepare("SELECT avatar FROM user_profiles WHERE user_id = ?");
        $old_avatar_stmt->execute([$user_id]);
        $old_avatar = $old_avatar_stmt->fetchColumn();
        
        // Generate unique filename
        $extension = pathinfo($files['avatar']['name'], PATHINFO_EXTENSION);
        $filename = 'avatar_' . $user_id . '_' . time() . '.' . $extension;
        $filepath = $upload_dir . $filename;
        
        // Move uploaded file
        if (move_uploaded_file($files['avatar']['tmp_name'], $filepath)) {
            // Update database
            $update_stmt = $pdo->prepare("UPDATE user_profiles SET avatar = ?, updated_at = NOW() WHERE user_id = ?");
            $update_stmt->execute([$filepath, $user_id]);
            
            // Delete old avatar file
            if ($old_avatar && file_exists($old_avatar)) {
                unlink($old_avatar);
            }
            
            return 'success: Avatar updated successfully';
        }
        
        return 'error: Failed to upload file';
        
    } catch (Exception $e) {
        return 'error: ' . $e->getMessage();
    }
}

function getRoleColor($role) {
    $colors = [
        'super_admin' => 'danger',
        'admin' => 'warning',
        'event_organizer' => 'info',
        'office' => 'success',
        'head_judge' => 'purple',
        'judge' => 'primary',
        'media' => 'secondary',
        'participant' => 'dark'
    ];
    return $colors[$role] ?? 'secondary';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Profile - StyleScore</title>
    <?php include_once 'includes/stylesheets.php'; ?>
</head>
<body class="body-bg-aurora-bright">
<?php include 'menu.php'; ?>
     <!-- style skin holder -->
    <?php include_once 'includes/stylesheets.php'; ?>

<div class="container-fluid container-StyleScore p-4">
        <!-- Event Actions Menu -->
    <div class="row mb-3">
        <div class="col-12">
            <?php $use_localStorage = true; $show_header = false; include 'admin/event_actions_menu.php'; ?>
        </div>
    </div>
    <div class="container-fluid p-0">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h1 class="h3 mb-1">
                            <i class="fas fa-user-circle text-primary me-2"></i>My Profile
                        </h1>
                        <p class="text-muted mb-0">Manage your account settings and information</p>
                    </div>
                </div>

                <!-- Messages -->
                <?php if ($message): ?>
                    <div class="alert alert-<?= $msg_type ?> alert-dismissible fade show">
                        <i class="fas fa-<?= $msg_type === 'success' ? 'check-circle' : 'exclamation-triangle' ?> me-2"></i>
                        <?= htmlspecialchars(substr($message, strpos($message, ':') + 1)) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <div class="row g-4">
                    
                    <!-- Left Column: Profile Info & Avatar -->
                    <div class="col-lg-4">
                        
                        <!-- Profile Card -->
                        <div class="card shadow-sm mb-4">
                            <div class="card-body text-center p-4">
                                <!-- Avatar -->
                                <div class="mb-3">
                                    <?php if (!empty($user['avatar']) && file_exists($user['avatar'])): ?>
                                        <img src="<?= htmlspecialchars($user['avatar']) ?>" 
                                             alt="Avatar" 
                                             class="rounded-circle mb-3" 
                                             style="width: 150px; height: 150px; object-fit: cover;">
                                    <?php else: ?>
                                        <div class="bg-primary text-white rounded-circle d-inline-flex align-items-center justify-content-center mb-3" 
                                             style="width: 150px; height: 150px; font-size: 3rem;">
                                            <?= strtoupper(substr($user['full_name'] ?: $user['username'], 0, 2)) ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- Upload Avatar Button -->
                                    <div>
                                        <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#avatarModal">
                                            <i class="fas fa-camera me-1"></i>Change Avatar
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Basic Info -->
                                <h4 class="mb-1"><?= htmlspecialchars($user['full_name'] ?: $user['username']) ?></h4>
                                <p class="text-muted mb-2">@<?= htmlspecialchars($user['username']) ?></p>
                                <span class="badge bg-<?= getRoleColor($user['role']) ?> mb-3">
                                    <?= ucfirst(str_replace('_', ' ', $user['role'])) ?>
                                </span>
                                
                                <!-- Status -->
                                <div class="mb-3">
                                    <i class="fas fa-circle <?= $user['is_active'] ? 'text-success' : 'text-danger' ?>"></i>
                                    <small><?= $user['is_active'] ? 'Active' : 'Inactive' ?></small>
                                </div>
                                
                                <!-- Stats -->
                                <div class="row text-center border-top pt-3 mt-3">
                                    <div class="col-6">
                                        <h5 class="mb-0"><?= count($assigned_events) ?></h5>
                                        <small class="text-muted">Events</small>
                                    </div>
                                    <div class="col-6">
                                        <h5 class="mb-0"><?= count($user_permissions) ?></h5>
                                        <small class="text-muted">Permissions</small>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Account Info Card -->
                        <div class="card shadow-sm">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">
                                    <i class="fas fa-info-circle me-2"></i>Account Information
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <small class="text-muted d-block">Member Since</small>
                                    <strong><?= date('F j, Y', strtotime($user['created_at'])) ?></strong>
                                </div>
                                <?php if ($user['last_login']): ?>
                                <div class="mb-3">
                                    <small class="text-muted d-block">Last Login</small>
                                    <strong><?= date('M j, Y g:i A', strtotime($user['last_login'])) ?></strong>
                                </div>
                                <?php endif; ?>
                                <div class="mb-3">
                                    <small class="text-muted d-block">Token Balance</small>
                                    <strong class="text-<?= $token_balance > 10 ? 'success' : 'danger' ?>">
                                        <i class="fas fa-coins me-1"></i><?= number_format($token_balance, 2) ?> tokens
                                    </strong>
                                </div>
                                <div class="mb-3">
                                    <small class="text-muted d-block">Total Spent</small>
                                    <strong class="text-danger"><?= number_format($total_spent, 2) ?> tokens</strong>
                                </div>
                                <div>
                                    <small class="text-muted d-block">Total Earned</small>
                                    <strong class="text-success"><?= number_format($total_earned, 2) ?> tokens</strong>
                                </div>
                            </div>
                        </div>

                    </div>

                    <!-- Right Column: Tabs -->
                    <div class="col-lg-8">
                        
                        <div class="card shadow-sm bg-transparent">
                            <div class="card-header p-3 pt-0 bg-transparent">
                                <ul class="nav nav-tabs card-header-tabs" role="tablist">
                                    <li class="nav-item">
                                        <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#profile-tab" type="button">
                                            <i class="fas fa-user me-2"></i>Profile
                                        </button>
                                    </li>
                                    <li class="nav-item">
                                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#events-tab" type="button">
                                            <i class="fas fa-calendar me-2"></i>Events
                                        </button>
                                    </li>
                                    <li class="nav-item">
                                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#permissions-tab" type="button">
                                            <i class="fas fa-shield-alt me-2"></i>Permissions
                                        </button>
                                    </li>
                                    <li class="nav-item">
                                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tokens-tab" type="button">
                                            <i class="fas fa-coins me-2"></i>Tokens
                                        </button>
                                    </li>
                                    <li class="nav-item">
                                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#security-tab" type="button">
                                            <i class="fas fa-lock me-2"></i>Security
                                        </button>
                                    </li>
                                </ul>
                            </div>
                            
                            <div class="card-body bg-white">
                                <div class="tab-content">
                                    
                                    <!-- Profile Tab -->
                                    <div class="tab-pane fade show active" id="profile-tab">
                                        <h5 class="mb-4">Edit Profile Information</h5>
                                        
                                        <form method="post">
                                            <input type="hidden" name="action" value="update_profile">
                                            
                                            <div class="row g-3">
                                                <div class="col-md-6">
                                                    <label class="form-label">Full Name</label>
                                                    <input type="text" class="form-control" name="full_name" 
                                                           value="<?= htmlspecialchars($user['full_name'] ?? '') ?>">
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">Email</label>
                                                    <input type="email" class="form-control" name="email" 
                                                           value="<?= htmlspecialchars($user['email'] ?? '') ?>">
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">Phone</label>
                                                    <input type="tel" class="form-control" name="phone" 
                                                           value="<?= htmlspecialchars($user['phone'] ?? '') ?>">
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">Username (Read-only)</label>
                                                    <input type="text" class="form-control" 
                                                           value="<?= htmlspecialchars($user['username']) ?>" disabled>
                                                </div>
                                                <div class="col-12">
                                                    <label class="form-label">Address</label>
                                                    <textarea class="form-control" name="address" rows="2"><?= htmlspecialchars($user['address'] ?? '') ?></textarea>
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">Emergency Contact</label>
                                                    <input type="text" class="form-control" name="emergency_contact" 
                                                           value="<?= htmlspecialchars($user['emergency_contact'] ?? '') ?>">
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">Emergency Phone</label>
                                                    <input type="tel" class="form-control" name="emergency_phone" 
                                                           value="<?= htmlspecialchars($user['emergency_phone'] ?? '') ?>">
                                                </div>
                                            </div>
                                            
                                            <div class="mt-4">
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="fas fa-save me-2"></i>Save Changes
                                                </button>
                                            </div>
                                        </form>
                                    </div>

                                    <!-- Events Tab -->
                                    <div class="tab-pane fade" id="events-tab">
                                        <h5 class="mb-4">Assigned Events</h5>
                                        
                                        <?php if (empty($assigned_events)): ?>
                                            <div class="text-center text-muted py-5">
                                                <i class="fas fa-calendar-times fa-3x mb-3"></i>
                                                <p>You are not assigned to any events yet.</p>
                                            </div>
                                        <?php else: ?>
                                            <div class="table-responsive">
                                                <table class="table table-hover">
                                                    <thead>
                                                        <tr>
                                                            <th>Event Name</th>
                                                            <th>Date</th>
                                                            <th>Location</th>
                                                            <th>Status</th>
                                                            <th>Assigned</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($assigned_events as $event): ?>
                                                        <tr>
                                                            <td>
                                                                <strong><?= htmlspecialchars($event['name']) ?></strong>
                                                            </td>
                                                            <td><?= date('M j, Y', strtotime($event['date'])) ?></td>
                                                            <td><?= htmlspecialchars($event['location'] ?? 'TBD') ?></td>
                                                            <td>
                                                                <span class="badge bg-<?= $event['status'] === 'active' ? 'success' : 'secondary' ?>">
                                                                    <?= ucfirst($event['status'] ?? 'pending') ?>
                                                                </span>
                                                            </td>
                                                            <td>
                                                                <small class="text-muted">
                                                                    <?= date('M j, Y', strtotime($event['assigned_at'])) ?>
                                                                </small>
                                                            </td>
                                                        </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                        <?php endif; ?>
                                    </div>

                                    <!-- Permissions Tab -->
                                    <div class="tab-pane fade" id="permissions-tab">
                                        <h5 class="mb-4">Your Permissions</h5>
                                        
                                        <div class="alert alert-info">
                                            <i class="fas fa-info-circle me-2"></i>
                                            You have <strong><?= count($user_permissions) ?></strong> permissions based on your role: 
                                            <span class="badge bg-<?= getRoleColor($user['role']) ?>">
                                                <?= ucfirst(str_replace('_', ' ', $user['role'])) ?>
                                            </span>
                                        </div>
                                        
                                        <?php foreach ($permissions_grouped as $category => $perms): ?>
                                            <div class="mb-4">
                                                <h6 class="text-uppercase text-muted mb-3">
                                                    <i class="fas fa-folder me-2"></i>
                                                    <?= ucwords(str_replace('_', ' ', $category)) ?>
                                                </h6>
                                                <div class="row g-2">
                                                    <?php foreach ($perms as $perm): ?>
                                                        <div class="col-md-6">
                                                            <div class="card bg-light">
                                                                <div class="card-body py-2">
                                                                    <div class="d-flex align-items-start">
                                                                        <i class="fas fa-check-circle text-success me-2 mt-1"></i>
                                                                        <div>
                                                                            <strong class="d-block"><?= htmlspecialchars($perm['display_name']) ?></strong>
                                                                            <small class="text-muted"><?= htmlspecialchars($perm['description']) ?></small>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>

                                    <!-- Tokens Tab -->
                                    <div class="tab-pane fade" id="tokens-tab">
                                        <h5 class="mb-4">Token Management</h5>
                                        
                                        <!-- Token Statistics -->
                                        <div class="row g-3 mb-4">
                                            <div class="col-md-4">
                                                <div class="card bg-light border-start border-primary border-4">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <small class="text-muted d-block">Current Balance</small>
                                                                <h4 class="mb-0 text-<?= $token_balance > 10 ? 'success' : 'danger' ?>">
                                                                    <?= number_format($token_balance, 2) ?>
                                                                </h4>
                                                            </div>
                                                            <i class="fas fa-wallet fa-2x text-primary opacity-25"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="card bg-light border-start border-danger border-4">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <small class="text-muted d-block">Total Spent</small>
                                                                <h4 class="mb-0 text-danger">
                                                                    <?= number_format($total_spent, 2) ?>
                                                                </h4>
                                                            </div>
                                                            <i class="fas fa-arrow-down fa-2x text-danger opacity-25"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="card bg-light border-start border-success border-4">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <small class="text-muted d-block">Total Earned</small>
                                                                <h4 class="mb-0 text-success">
                                                                    <?= number_format($total_earned, 2) ?>
                                                                </h4>
                                                            </div>
                                                            <i class="fas fa-arrow-up fa-2x text-success opacity-25"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <?php if ($token_balance <= 10): ?>
                                        <div class="alert alert-warning">
                                            <i class="fas fa-exclamation-triangle me-2"></i>
                                            <strong>Low Balance!</strong> You have <?= number_format($token_balance, 2) ?> tokens remaining. Consider purchasing more tokens to continue using premium features.
                                        </div>
                                        <?php endif; ?>

                                        <!-- Recent Transactions -->
                                        <h6 class="mb-3">Recent Transactions</h6>
                                        
                                        <?php if (empty($recent_transactions)): ?>
                                            <div class="text-center text-muted py-4">
                                                <i class="fas fa-receipt fa-3x mb-3 opacity-25"></i>
                                                <p class="mb-0">No transactions yet</p>
                                            </div>
                                        <?php else: ?>
                                            <div class="table-responsive">
                                                <table class="table table-hover table-sm">
                                                    <thead class="table-light">
                                                        <tr>
                                                            <th>Date</th>
                                                            <th>Action</th>
                                                            <th>Type</th>
                                                            <th class="text-end">Amount</th>
                                                            <th class="text-end">Balance After</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($recent_transactions as $trans): ?>
                                                        <tr>
                                                            <td>
                                                                <small class="text-muted">
                                                                    <?= date('M j, Y g:i A', strtotime($trans['created_at'])) ?>
                                                                </small>
                                                            </td>
                                                            <td>
                                                                <?php if ($trans['action_name']): ?>
                                                                    <strong><?= htmlspecialchars($trans['action_name']) ?></strong>
                                                                    <?php if ($trans['action_description']): ?>
                                                                        <br><small class="text-muted"><?= htmlspecialchars($trans['action_description']) ?></small>
                                                                    <?php endif; ?>
                                                                <?php else: ?>
                                                                    <span class="text-muted"><?= htmlspecialchars($trans['description'] ?? 'N/A') ?></span>
                                                                <?php endif; ?>
                                                            </td>
                                                            <td>
                                                                <?php
                                                                $type_colors = [
                                                                    'debit' => 'danger',
                                                                    'credit' => 'success',
                                                                    'purchase' => 'primary',
                                                                    'refund' => 'info'
                                                                ];
                                                                $type_color = $type_colors[$trans['transaction_type']] ?? 'secondary';
                                                                ?>
                                                                <span class="badge bg-<?= $type_color ?>">
                                                                    <?= ucfirst($trans['transaction_type']) ?>
                                                                </span>
                                                            </td>
                                                            <td class="text-end">
                                                                <span class="text-<?= $trans['transaction_type'] === 'debit' ? 'danger' : 'success' ?>">
                                                                    <?= $trans['transaction_type'] === 'debit' ? '-' : '+' ?><?= number_format($trans['amount'], 2) ?>
                                                                </span>
                                                            </td>
                                                            <td class="text-end">
                                                                <strong><?= number_format($trans['balance_after'], 2) ?></strong>
                                                            </td>
                                                        </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                            
                                            <?php if (count($recent_transactions) >= 20): ?>
                                            <div class="text-center mt-3">
                                                <small class="text-muted">
                                                    <i class="fas fa-info-circle me-1"></i>
                                                    Showing last 20 transactions
                                                </small>
                                            </div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </div>

                                    <!-- Security Tab -->
                                    <div class="tab-pane fade" id="security-tab">
                                        <h5 class="mb-4">Change Password</h5>
                                        
                                        <form method="post" id="passwordForm">
                                            <input type="hidden" name="action" value="change_password">
                                            
                                            <div class="row g-3">
                                                <div class="col-12">
                                                    <label class="form-label">Current Password *</label>
                                                    <input type="password" class="form-control" name="current_password" required>
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">New Password *</label>
                                                    <input type="password" class="form-control" name="new_password" 
                                                           id="new_password" required minlength="4">
                                                    <small class="text-muted">Minimum 4 characters</small>
                                                </div>
                                                <div class="col-md-6">
                                                    <label class="form-label">Confirm New Password *</label>
                                                    <input type="password" class="form-control" name="confirm_password" 
                                                           id="confirm_password" required minlength="4">
                                                </div>
                                                <div class="col-12">
                                                    <div id="password_match_error" class="text-danger small" style="display:none;">
                                                        <i class="fas fa-exclamation-triangle me-1"></i>Passwords do not match
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="mt-4">
                                                <button type="submit" class="btn btn-warning">
                                                    <i class="fas fa-key me-2"></i>Change Password
                                                </button>
                                            </div>
                                        </form>
                                    </div>

                                </div>
                            </div>
                        </div>

                    </div>
                </div>

            </div>
        </div>
    </div>

    <!-- Avatar Upload Modal -->
    <div class="modal fade" id="avatarModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-camera me-2"></i>Change Avatar
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="upload_avatar">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Select Image</label>
                            <input type="file" class="form-control" name="avatar" accept="image/*" required>
                            <small class="text-muted">Allowed: JPG, PNG, GIF, WebP (max 5MB)</small>
                        </div>
                        <div class="alert alert-info mb-0">
                            <i class="fas fa-info-circle me-1"></i>
                            For best results, use a square image (e.g., 500x500px)
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-upload me-1"></i>Upload
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Password match validation
        const passwordForm = document.getElementById('passwordForm');
        const newPassword = document.getElementById('new_password');
        const confirmPassword = document.getElementById('confirm_password');
        const errorDiv = document.getElementById('password_match_error');
        
        function checkPasswordMatch() {
            if (confirmPassword.value && newPassword.value !== confirmPassword.value) {
                errorDiv.style.display = 'block';
                confirmPassword.setCustomValidity('Passwords do not match');
            } else {
                errorDiv.style.display = 'none';
                confirmPassword.setCustomValidity('');
            }
        }
        
        newPassword.addEventListener('input', checkPasswordMatch);
        confirmPassword.addEventListener('input', checkPasswordMatch);
        
        passwordForm.addEventListener('submit', function(e) {
            if (newPassword.value !== confirmPassword.value) {
                e.preventDefault();
                errorDiv.style.display = 'block';
            }
        });
    </script>
</body>
</html>
