<?php
/**
 * Migration: Hash all plain text passwords in users table
 * This script will convert all plain text passwords to bcrypt hashed passwords
 * while preserving the original password value for users to login with
 */

require_once 'includes/db.php';

echo "=== Password Migration to Bcrypt ===\n\n";

// Find all users with plain text passwords
$stmt = $pdo->query("SELECT id, username, password, role FROM users");
$users = $stmt->fetchAll(PDO::FETCH_ASSOC);

$plain_text_users = [];
$already_hashed = 0;

foreach ($users as $user) {
    $is_hashed = (strlen($user['password']) === 60 && strpos($user['password'], '$2y$') === 0);
    
    if (!$is_hashed) {
        $plain_text_users[] = $user;
    } else {
        $already_hashed++;
    }
}

echo "Found:\n";
echo "- Already hashed: $already_hashed users\n";
echo "- Plain text: " . count($plain_text_users) . " users\n\n";

if (empty($plain_text_users)) {
    echo "✓ All passwords are already hashed. No migration needed.\n";
    exit;
}

echo "Users to migrate:\n";
foreach ($plain_text_users as $user) {
    echo "- {$user['username']} (Role: {$user['role']}, Current password: {$user['password']})\n";
}

echo "\n";
echo "WARNING: This will hash all plain text passwords.\n";
echo "Make sure to note down current passwords before proceeding!\n";
echo "\nType 'YES' to proceed with migration: ";

$handle = fopen("php://stdin", "r");
$confirmation = trim(fgets($handle));
fclose($handle);

if ($confirmation !== 'YES') {
    echo "\nMigration cancelled.\n";
    exit;
}

echo "\n=== Starting Migration ===\n\n";

$pdo->beginTransaction();

try {
    $update_stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
    
    foreach ($plain_text_users as $user) {
        $hashed_password = password_hash($user['password'], PASSWORD_DEFAULT);
        $update_stmt->execute([$hashed_password, $user['id']]);
        
        echo "✓ Migrated: {$user['username']} (password: {$user['password']} → hashed)\n";
    }
    
    $pdo->commit();
    
    echo "\n=== Migration Complete ===\n";
    echo "Successfully hashed " . count($plain_text_users) . " passwords.\n";
    echo "Users can still login with their original passwords.\n";
    echo "The system will automatically detect and verify hashed passwords.\n";
    
} catch (Exception $e) {
    $pdo->rollBack();
    echo "\n✗ ERROR: Migration failed - " . $e->getMessage() . "\n";
    echo "No changes were made to the database.\n";
}
?>
