<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/event_access.php';

// Debug: Log session info for score.php page load
error_log("score.php PAGE LOAD - Session ID: " . session_id());
error_log("score.php PAGE LOAD - user_id: " . ($_SESSION['user_id'] ?? 'NOT SET'));

// Add this new block to handle AJAX overwrite score submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['overwrite_score_id']) && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    $score_id = $_POST['overwrite_score_id'];
    $new_score_value = $_POST['overwrite_score_value'] !== '' ? floatval($_POST['overwrite_score_value']) : null;
    $new_status = $_POST['overwrite_status'];
    $new_figures_json = $_POST['overwrite_figures_json'] ?? '';
    
    // Get judge_id using users table
    $user_type = $_SESSION['user_type'] ?? $_SESSION['role'] ?? 'judge';
    $user_id = $_SESSION['user_id'] ?? 0;
    
    if (!$user_id) {
        echo json_encode([
            'success' => false,
            'message' => 'No user ID found in session. Please log in again.'
        ]);
        exit;
    }
    
    if ($user_type == 'admin') {
        // For admin, use their user_id directly or a specific judge_id
        $current_judge_id = $user_id; // Or use a specific judge_id if needed
    } else {
        // For judge, use their user_id as judge_id
        $current_judge_id = $user_id;
    }
    
    // Validate that this score belongs to the current judge
    $validate_stmt = $pdo->prepare("SELECT id FROM scores WHERE id = ? AND judge_id = ?");
    $validate_stmt->execute([$score_id, $current_judge_id]);
    if (!$validate_stmt->fetch()) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid score ID or you don\'t have permission to edit this score.'
        ]);
        exit;
    }
    
    try {
        // Resolve event + heat for this score to compute current control_point_id
        $score_ctx = $pdo->prepare("SELECT ep.event_id, ep.heat_number
                                    FROM scores s
                                    JOIN runs r ON s.run_id = r.id
                                    JOIN event_participants ep ON r.event_participant_id = ep.id
                                    WHERE s.id = ? AND s.judge_id = ?");
        $score_ctx->execute([$score_id, $current_judge_id]);
        $ctx = $score_ctx->fetch(PDO::FETCH_ASSOC) ?: null;

        $control_point_id = null;
        if ($ctx && !empty($ctx['event_id'])) {
            try {
                $cp_stmt = $pdo->prepare("
                    SELECT ecp.id
                    FROM judge_assignments ja
                    JOIN event_control_points ecp ON ecp.id = ja.event_control_point_id
                    WHERE ja.event_id = ?
                      AND ja.judge_id = ?
                      AND (ja.heat_number = ? OR ja.heat_number IS NULL)
                    ORDER BY (ja.heat_number = ?) DESC, ja.heat_number DESC
                    LIMIT 1
                ");
                $cp_stmt->execute([$ctx['event_id'], $current_judge_id, $ctx['heat_number'], $ctx['heat_number']]);
                $control_point_id = $cp_stmt->fetchColumn() ?: null;
            } catch (Exception $e) {
                $control_point_id = null;
            }
        }

        // Check if columns exist
        $hasCPCol = false;
        $hasFiguresCol = false;
        try { $pdo->query("SELECT control_point_id FROM scores WHERE 1=0"); $hasCPCol = true; } catch (Exception $e) { $hasCPCol = false; }
        try { $pdo->query("SELECT figures_json FROM scores WHERE 1=0"); $hasFiguresCol = true; } catch (Exception $e) { $hasFiguresCol = false; }

        // Update the score with conditional column support
        if ($hasCPCol && $hasFiguresCol) {
            $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, control_point_id = ?, figures_json = ?, updated_at = NOW() WHERE id = ? AND judge_id = ?");
            $update_stmt->execute([$new_score_value, $new_status, $control_point_id, $new_figures_json ?: null, $score_id, $current_judge_id]);
        } elseif ($hasCPCol && !$hasFiguresCol) {
            $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, control_point_id = ?, updated_at = NOW() WHERE id = ? AND judge_id = ?");
            $update_stmt->execute([$new_score_value, $new_status, $control_point_id, $score_id, $current_judge_id]);
        } elseif (!$hasCPCol && $hasFiguresCol) {
            $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, figures_json = ?, updated_at = NOW() WHERE id = ? AND judge_id = ?");
            $update_stmt->execute([$new_score_value, $new_status, $new_figures_json ?: null, $score_id, $current_judge_id]);
        } else {
            $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, updated_at = NOW() WHERE id = ? AND judge_id = ?");
            $update_stmt->execute([$new_score_value, $new_status, $score_id, $current_judge_id]);
        }
        
        // Fetch updated score details for response
        $fetch_stmt = $pdo->prepare("
            SELECT s.id AS score_id, s.score_value, s.status, s.figures_json, r.run_number, ep.heat_number, ep.bib_number as bib, 
                   p.first_name, p.last_name, s.created_at
            FROM scores s
            JOIN runs r ON s.run_id = r.id
            JOIN event_participants ep ON r.event_participant_id = ep.id
            JOIN participants p ON ep.participant_id = p.id
            WHERE s.id = ?
        ");
        $fetch_stmt->execute([$score_id]);
        $updated_score = $fetch_stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Score updated successfully!',
            'score' => $updated_score
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
    
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_score'])) {
    $stmt = $pdo->prepare("
        DELETE s FROM scores s
        JOIN runs r ON s.run_id = r.id
        JOIN event_participants ep ON r.event_participant_id = ep.id
        WHERE s.judge_id = ? AND ep.event_id = ?
    ");
    $stmt->execute([$_POST['judge_id'], $_POST['event_id']]);
    header("Location: score.php?event_id=" . intval($_POST['event_id']) . "&reset=1");
    exit;
}

// Handle AJAX score submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ep_id']) && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    try {
        $run_number = $_POST['run_number'];
        $ep_id = $_POST['ep_id'];
        $p_bib = $_POST['p_bib'];
        $heat_number = $_POST['heat_number'];
    $score_value = $_POST['score_value'] !== '' ? floatval($_POST['score_value']) : null;
    $status = $_POST['status'];
    $sel_figure_cat = $_POST['figure_category'] ?? '';
    $sel_figure_item = $_POST['figure_item'] ?? '';
    $figures_json = $_POST['figures_json'] ?? '';

        // Process criteria scores if provided
        $criteria_scores_json = $_POST['criteria_scores_json'] ?? '';
        $criteria_scores_data = [];
        if (!empty($criteria_scores_json)) {
            try {
                $criteria_scores_data = json_decode($criteria_scores_json, true) ?: [];
            } catch (Exception $e) {
                $criteria_scores_data = [];
            }
        }

        // Judge identity
        $user_type = $_SESSION['user_type'] ?? $_SESSION['role'] ?? 'judge';
        $user_id = $_SESSION['user_id'] ?? 0;
        
        if (!$user_id) {
            echo json_encode(['success' => false, 'message' => 'No user ID found in session.']);
            exit;
        }
        
        $judge_id = $user_id;
        
        // Debug: Log score submission
        error_log("SCORE SUBMISSION - Judge ID: $judge_id, EP ID: $ep_id, Run: $run_number, Score: $score_value");

        // Resolve event_id for this participant
        $evStmt = $pdo->prepare("SELECT event_id FROM event_participants WHERE id = ?");
        $evStmt->execute([$ep_id]);
        $event_id = (int)$evStmt->fetchColumn();

        // Validate participant
        $validate_stmt = $pdo->prepare("SELECT 1 FROM event_participants WHERE id = ?");
        $validate_stmt->execute([$ep_id]);
        if (!$validate_stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'Participant not assigned to event/heat.']);
            exit;
        }

        // Create or get run
        $run_stmt = $pdo->prepare("SELECT id FROM runs WHERE event_participant_id = ? AND run_number = ?");
        $run_stmt->execute([$ep_id, $run_number]);
        $run = $run_stmt->fetch(PDO::FETCH_ASSOC);

        if (!$run) {
            $insert_run = $pdo->prepare("INSERT INTO runs (event_participant_id, run_number) VALUES (?, ?)");
            $insert_run->execute([$ep_id, $run_number]);
            $run_id = $pdo->lastInsertId();
        } else {
            $run_id = $run['id'];
        }

        // Resolve current control point for judge (match by event and prefer exact heat)
        $control_point_id = null;
        if ($event_id) {
            $cp_stmt = $pdo->prepare("
                SELECT ecp.id
                FROM judge_assignments ja
                JOIN event_control_points ecp ON ecp.id = ja.event_control_point_id
                WHERE ja.event_id = ?
                  AND ja.judge_id = ?
                  AND (ja.heat_number = ? OR ja.heat_number IS NULL)
                ORDER BY (ja.heat_number = ?) DESC, ja.heat_number DESC
                LIMIT 1
            ");
            $cp_stmt->execute([$event_id, $judge_id, $heat_number, $heat_number]);
            $control_point_id = $cp_stmt->fetchColumn() ?: null;
        }

        // Insert or update score with control_point_id (if column exists) and figures_json (if column exists)
        $hasFiguresCol = false; $hasCPCol = false; $hasCriteriaCol = false;
        try { $pdo->query("SELECT figures_json FROM scores WHERE 1=0"); $hasFiguresCol = true; } catch (Exception $e) { $hasFiguresCol = false; }
        try { $pdo->query("SELECT control_point_id FROM scores WHERE 1=0"); $hasCPCol = true; } catch (Exception $e) { $hasCPCol = false; }
        try { $pdo->query("SELECT criteria_scores FROM scores WHERE 1=0"); $hasCriteriaCol = true; } catch (Exception $e) { $hasCriteriaCol = false; }

        // Prepare criteria scores JSON for storage
        $criteria_scores_for_db = null;
        if ($hasCriteriaCol && !empty($criteria_scores_data)) {
            $criteria_scores_for_db = json_encode($criteria_scores_data);
        }

        // Handle different column combinations
        if ($hasCPCol && $hasFiguresCol && $hasCriteriaCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, control_point_id, score_value, status, figures_json, criteria_scores) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $control_point_id, $score_value, $status, $figures_json ?: null, $criteria_scores_for_db]);
        } elseif ($hasCPCol && $hasFiguresCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, control_point_id, score_value, status, figures_json) VALUES (?, ?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $control_point_id, $score_value, $status, $figures_json ?: null]);
        } elseif ($hasCPCol && $hasCriteriaCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, control_point_id, score_value, status, criteria_scores) VALUES (?, ?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $control_point_id, $score_value, $status, $criteria_scores_for_db]);
        } elseif ($hasFiguresCol && $hasCriteriaCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, score_value, status, figures_json, criteria_scores) VALUES (?, ?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $score_value, $status, $figures_json ?: null, $criteria_scores_for_db]);
        } elseif ($hasCPCol && !$hasFiguresCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, control_point_id, score_value, status) VALUES (?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $control_point_id, $score_value, $status]);
        } elseif (!$hasCPCol && $hasFiguresCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, score_value, status, figures_json) VALUES (?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $score_value, $status, $figures_json ?: null]);
        } elseif (!$hasCPCol && $hasCriteriaCol) {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, score_value, status, criteria_scores) VALUES (?, ?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $score_value, $status, $criteria_scores_for_db]);
        } else {
            $stmtIns = $pdo->prepare("REPLACE INTO scores (run_id, judge_id, score_value, status) VALUES (?, ?, ?, ?)");
            $stmtIns->execute([$run_id, $judge_id, $score_value, $status]);
        }

        // Update participant status to 'scoring' when a score is added
        $update_status_stmt = $pdo->prepare("
            UPDATE event_participants 
            SET participant_status = 'scoring' 
            WHERE id = ?
        ");
        $update_status_stmt->execute([$ep_id]);

        // Fetch the newly created/updated score with participant details
        $fetch_stmt = $pdo->prepare("
            SELECT s.id AS score_id, s.score_value, s.status, r.run_number, ep.heat_number, ep.bib_number as bib, 
                   p.first_name, p.last_name, s.created_at, s.figures_json
            FROM scores s
            JOIN runs r ON s.run_id = r.id
            JOIN event_participants ep ON r.event_participant_id = ep.id
            JOIN participants p ON ep.participant_id = p.id
            WHERE s.run_id = ? AND s.judge_id = ?
        ");
        $fetch_stmt->execute([$run_id, $judge_id]);
        $new_score = $fetch_stmt->fetch(PDO::FETCH_ASSOC);

        $figureNote = '';
        if (!empty($sel_figure_cat) && !empty($sel_figure_item)) {
            $catEsc = htmlspecialchars($sel_figure_cat, ENT_QUOTES, 'UTF-8');
            $itemEsc = htmlspecialchars($sel_figure_item, ENT_QUOTES, 'UTF-8');
            $figureNote = " <span class=\"badge bg-success ms-2\"><i class=\"fas fa-shapes me-1\"></i>{$catEsc}: {$itemEsc}</span>";
        }

        echo json_encode([
            'success' => true,
            'message' => "Score submitted! BIB <strong>{$p_bib}</strong>. Score: <strong>" . ($score_value === null ? '-' : $score_value) . "</strong>" . $figureNote,
            'score' => $new_score,
            'debug' => [
                'judge_id' => $judge_id,
                'ep_id' => $ep_id,
                'run_id' => $run_id,
                'run_number' => $run_number
            ]
        ]);

    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    }
    exit;
}

// Main logic - UPDATED to use users table
$user_type = $_SESSION['user_type'] ?? $_SESSION['role'] ?? 'judge';
$user_id = $_SESSION['user_id'] ?? 0;

if (!$user_id) {
    header('Location: ../login.php');
    exit;
}

if ($user_type == 'admin') {
    // For admin, use their user_id directly
    $judge_id = $user_id;
    } else {
    // For judge, use their user_id as judge_id
    $judge_id = $user_id;
}

$message = '';
$manual_mode = isset($_GET['manual']) && $_GET['manual'] == 1;

// Get assigned events with access control
$accessible_events = getUserAccessibleEvents();

if (empty($accessible_events)) {
    // Admin - get all events assigned to this judge
    $assigned_event_stmt = $pdo->prepare("SELECT e.id, e.name FROM events e
        JOIN judge_assignments ja ON e.id = ja.event_id
        WHERE ja.judge_id = ?");
    $assigned_event_stmt->execute([$judge_id]);
} else {
    // Non-admin - filter by assigned events
    $placeholders = str_repeat('?,', count($accessible_events) - 1) . '?';
    $assigned_event_stmt = $pdo->prepare("SELECT e.id, e.name FROM events e
        JOIN judge_assignments ja ON e.id = ja.event_id
        WHERE ja.judge_id = ? AND e.id IN ($placeholders)");
    $params = array_merge([$judge_id], $accessible_events);
    $assigned_event_stmt->execute($params);
}

$assigned_events = $assigned_event_stmt->fetchAll(PDO::FETCH_ASSOC);

// Redirect to dashboard if judge has no events assigned
if (empty($assigned_events) && !$manual_mode) {
    header("Location: dashboard.php?message=no_events_assigned");
    exit;
}

// Get selected event
$event_id = $manual_mode ? ($_POST['event_id'] ?? $_GET['event_id'] ?? '') : ($assigned_events[0]['id'] ?? null);

// Verify access to selected event
if ($event_id && !hasEventAccess($event_id)) {
    header("Location: dashboard.php?error=access_denied");
    exit;
}

// Get judge name - UPDATED to use users table
$judge_name = '';
$judge_stmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
$judge_stmt->execute([$judge_id]);
$judge_name = $judge_stmt->fetchColumn() ?? 'Unknown';

// Get event date
$event_date = '';
$event_name = '';
if ($event_id) {
    $event_info_stmt = $pdo->prepare("SELECT date, name FROM events WHERE id = ?");
    $event_info_stmt->execute([$event_id]);
    $event_info = $event_info_stmt->fetch(PDO::FETCH_ASSOC);
    $event_date = $event_info['date'] ?? '';
    $event_name = $event_info['name'] ?? '';
}

// Fetch event scoring format + related metadata (for Format Info card)
$format_info = null;
$uses_figures = false;
$figure_categories = [];
$figure_items_by_category = [];
$format_icon_url = null;
// Diversity rules + mode configs (attempts/tricks) for validation UI
$diversity_rules = [];
$mode_attempts = ['max' => null, 'bestN' => null];
$mode_tricks = ['attempts' => null, 'bestN' => null];
// Layout configuration
$layout_config = null;
// Initialize heat_number (can be overridden later from GET/POST)
$heat_number = $_GET['heat'] ?? $_POST['heat'] ?? null;
if ($event_id) {
    $fmtStmt = $pdo->prepare("SELECT e.scoring_format, e.scoring_layout_config,
                                     sf.name AS format_name,
                                     sf.sport,
                                     sf.mode,
                                     jf.min_judges, jf.max_judges, jf.scale_min, jf.scale_max, jf.scale_type, jf.precision_decimal, jf.drop_rule,
                                     (SELECT COUNT(*) FROM scoring_format_criteria sc WHERE sc.format_id = sf.format_id) AS criteria_count,
                                     (SELECT COUNT(*) FROM scoring_format_results sr WHERE sr.format_id = sf.format_id) AS results_count
                              FROM events e
                              LEFT JOIN scoring_formats sf ON e.scoring_format = sf.format_id
                              LEFT JOIN scoring_format_judges jf ON sf.format_id = jf.format_id
                              WHERE e.id = ?");
    $fmtStmt->execute([$event_id]);
    $format_info = $fmtStmt->fetch(PDO::FETCH_ASSOC) ?: null;
    
    // Get heat-specific layout config if heat number provided
    $heat_layout_config = null;
    if ($heat_number) {
        $heat_stmt = $pdo->prepare("
            SELECT scoring_layout_config 
            FROM heat_settings 
            WHERE event_id = ? AND heat_number = ?
        ");
        $heat_stmt->execute([$event_id, $heat_number]);
        $heat_settings = $heat_stmt->fetch(PDO::FETCH_ASSOC);
        if ($heat_settings && !empty($heat_settings['scoring_layout_config'])) {
            $heat_layout_config = json_decode($heat_settings['scoring_layout_config'], true);
        }
    }

    // Parse layout configuration (event-level)
    if ($format_info && !empty($format_info['scoring_layout_config'])) {
        $layout_config = json_decode($format_info['scoring_layout_config'], true);
    }
    
    // Determine which categories to show (heat overrides event)
    $visible_categories = null;
    if ($heat_layout_config && isset($heat_layout_config['cards']['figuresCard']['visibleCategories'])) {
        $visible_categories = $heat_layout_config['cards']['figuresCard']['visibleCategories'];
    } elseif ($layout_config && isset($layout_config['cards']['figuresCard']['visibleCategories'])) {
        $visible_categories = $layout_config['cards']['figuresCard']['visibleCategories'];
    }
    
    // Determine if Drawing Board Button should be shown (heat overrides event, default true)
    $show_drawing_board_button = true; // default
    if ($heat_layout_config && isset($heat_layout_config['cards']['drawingBoardButton']['enabled'])) {
        $show_drawing_board_button = (bool)$heat_layout_config['cards']['drawingBoardButton']['enabled'];
    } elseif ($layout_config && isset($layout_config['cards']['drawingBoardButton']['enabled'])) {
        $show_drawing_board_button = (bool)$layout_config['cards']['drawingBoardButton']['enabled'];
    }

    if ($format_info && !empty($format_info['scoring_format'])) {
        // Resolve SVG icon from ../assets/img/SVG/[format_id].svg or .SVG
        $fid = $format_info['scoring_format'];
        $slug = preg_replace('/[^a-zA-Z0-9_-]+/', '_', strtolower($fid));
        $svgRel = "../assets/img/SVG/{$slug}.svg";
        $SVGRel = "../assets/img/SVG/{$slug}.SVG";
        $svgFs = __DIR__ . "/../assets/img/SVG/{$slug}.svg";
        $SVGFs = __DIR__ . "/../assets/img/SVG/{$slug}.SVG";
        if (file_exists($svgFs)) {
            $format_icon_url = $svgRel;
        } elseif (file_exists($SVGFs)) {
            $format_icon_url = $SVGRel;
        }

        // Load diversity rules for this format (for UI summary/validation)
        // Check if diversity rules are enabled for at least one heat in this event
        try {
            // Check if there's at least one heat with diversity rules enabled
            $diversity_enabled_stmt = $pdo->prepare("
                SELECT COUNT(*) 
                FROM event_heat_settings 
                WHERE event_id = ? AND diversity_rules_enabled = 1
            ");
            $diversity_enabled_stmt->execute([$event_id]);
            $has_diversity_enabled = $diversity_enabled_stmt->fetchColumn() > 0;
            
            // Only load diversity rules if at least one heat has them enabled
            if ($has_diversity_enabled) {
                $divQ = $pdo->prepare("SELECT rule_type, scope, group_name, min_distinct, is_optional FROM scoring_format_diversity_rules WHERE format_id = ? ORDER BY rule_type ASC, group_name ASC");
                $divQ->execute([$format_info['scoring_format']]);
                $diversity_rules = $divQ->fetchAll(PDO::FETCH_ASSOC) ?: [];
            } else {
                $diversity_rules = [];
            }
        } catch (Exception $e) {
            // If event_heat_settings doesn't exist or query fails, load diversity rules by default
            $divQ = $pdo->prepare("SELECT rule_type, scope, group_name, min_distinct, is_optional FROM scoring_format_diversity_rules WHERE format_id = ? ORDER BY rule_type ASC, group_name ASC");
            $divQ->execute([$format_info['scoring_format']]);
            $diversity_rules = $divQ->fetchAll(PDO::FETCH_ASSOC) ?: [];
        }

        // Load attempts/tricks configs used by some diversity rules (bestN, attempts count)
        try {
            $cfgStmt = $pdo->prepare("SELECT config_type, config_key, config_value FROM scoring_format_mode_config WHERE format_id = ? AND (config_type IN ('attempts','tricks'))");
            $cfgStmt->execute([$format_info['scoring_format']]);
            $rowsCfg = $cfgStmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
            foreach ($rowsCfg as $cfg) {
                if ($cfg['config_type'] === 'attempts') {
                    if ($cfg['config_key'] === 'max') $mode_attempts['max'] = (int)$cfg['config_value'];
                    if ($cfg['config_key'] === 'bestN') $mode_attempts['bestN'] = (int)$cfg['config_value'];
                }
                if ($cfg['config_type'] === 'tricks') {
                    if ($cfg['config_key'] === 'attempts') $mode_tricks['attempts'] = (int)$cfg['config_value'];
                    if ($cfg['config_key'] === 'bestN') $mode_tricks['bestN'] = (int)$cfg['config_value'];
                }
            }
        } catch (Exception $e) {
            // ignore
        }

        // Determine if sport has figures available
        if (!empty($format_info['sport'])) {
            $figCheck = $pdo->prepare("SELECT COUNT(*)
                                       FROM figure_categories fc
                                       JOIN figure_items fi ON fi.sport_name = fc.sport_name AND fi.category_name = fc.category_name
                                       WHERE fc.sport_name = ? AND fc.is_active = 1 AND fi.is_active = 1");
            $figCheck->execute([$format_info['sport']]);
            $uses_figures = (int)$figCheck->fetchColumn() > 0;

            if ($uses_figures) {
                // Load categories - filter by visible_categories if configured
                if (!empty($visible_categories) && is_array($visible_categories)) {
                    $placeholders = implode(',', array_fill(0, count($visible_categories), '?'));
                    $catStmt = $pdo->prepare("SELECT category_name FROM figure_categories WHERE sport_name = ? AND is_active = 1 AND category_name IN ($placeholders) ORDER BY sort_order, category_name");
                    $catStmt->execute(array_merge([$format_info['sport']], $visible_categories));
                } else {
                    $catStmt = $pdo->prepare("SELECT category_name FROM figure_categories WHERE sport_name = ? AND is_active = 1 ORDER BY sort_order, category_name");
                    $catStmt->execute([$format_info['sport']]);
                }
                $figure_categories = $catStmt->fetchAll(PDO::FETCH_COLUMN) ?: [];

                // Load items grouped by category - filter by visible categories
                if (!empty($visible_categories) && is_array($visible_categories)) {
                    $placeholders = implode(',', array_fill(0, count($visible_categories), '?'));
                    $itemsStmt = $pdo->prepare("SELECT category_name, item_name FROM figure_items WHERE sport_name = ? AND is_active = 1 AND category_name IN ($placeholders) ORDER BY category_name, sort_order, item_name");
                    $itemsStmt->execute(array_merge([$format_info['sport']], $visible_categories));
                } else {
                    $itemsStmt = $pdo->prepare("SELECT category_name, item_name FROM figure_items WHERE sport_name = ? AND is_active = 1 ORDER BY category_name, sort_order, item_name");
                    $itemsStmt->execute([$format_info['sport']]);
                }
                $rows = $itemsStmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
                foreach ($rows as $r) {
                    $c = $r['category_name'];
                    if (!isset($figure_items_by_category[$c])) $figure_items_by_category[$c] = [];
                    $figure_items_by_category[$c][] = $r['item_name'];
                }
            }
        }
    }
}

// Load scoring criteria for this event
$scoring_criteria = [];
if ($event_id && !empty($format_info['scoring_format'])) {
    try {
        $criteriaStmt = $pdo->prepare("
            SELECT criteria_type, criteria_name, sort_order 
            FROM scoring_format_criteria 
            WHERE format_id = ? 
            ORDER BY sort_order ASC, criteria_type ASC, criteria_name ASC
        ");
        $criteriaStmt->execute([$format_info['scoring_format']]);
        $scoring_criteria = $criteriaStmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Exception $e) {
        $scoring_criteria = [];
    }
}

// Fetch event settings for min/max
$heats_total = 10;
$runs_per_heat = 3;
if ($event_id) {
    $event_settings_stmt = $pdo->prepare("SELECT heats_total, runs_per_heat FROM events WHERE id = ?");
    $event_settings_stmt->execute([$event_id]);
    $event_settings = $event_settings_stmt->fetch(PDO::FETCH_ASSOC);
    if ($event_settings) {
        $heats_total = intval($event_settings['heats_total']);
        $runs_per_heat = intval($event_settings['runs_per_heat']);
    }
}

// Fetch participants and runs
$participants = [];
if ($event_id) {
    $stmt = $pdo->prepare("SELECT ep.id as ep_id, p.first_name, p.last_name, ep.bib_number as bib, ep.heat_number
                           FROM event_participants ep 
                           JOIN participants p ON ep.participant_id = p.id
                           WHERE ep.event_id = ? ORDER BY ep.sort_order ASC, ep.bib_number ASC");
    $stmt->execute([$event_id]);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $scored_runs = $pdo->prepare("
        SELECT DISTINCT r.event_participant_id, r.run_number
        FROM scores s
        JOIN runs r ON s.run_id = r.id
        WHERE s.judge_id = ?
    ");
    $scored_runs->execute([$judge_id]);
    $locked_scores = [];
    foreach ($scored_runs as $row) {
        $locked_scores[$row['event_participant_id']][$row['run_number']] = true;
    }
}

$run_number = $_GET['run'] ?? $_POST['run'] ?? 1;
// $heat_number is already defined earlier in the file

// Fetch 4 latest scores for this judge and event
$latest_scores = [];
if ($event_id && $judge_id) {
    $latest_stmt = $pdo->prepare("
        SELECT s.id AS score_id, s.score_value, s.status, s.figures_json, r.run_number, ep.heat_number, ep.bib_number as bib, p.first_name, p.last_name, s.created_at
        FROM scores s
        JOIN runs r ON s.run_id = r.id
        JOIN event_participants ep ON r.event_participant_id = ep.id
        JOIN participants p ON ep.participant_id = p.id
        WHERE s.judge_id = ? AND ep.event_id = ?
        ORDER BY s.created_at DESC
        LIMIT 4
    ");
    $latest_stmt->execute([$judge_id, $event_id]);
    $latest_scores = $latest_stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Fetch heats for this event
$heats = [];
if ($event_id) {
    $heat_stmt = $pdo->prepare("SELECT heat_number, heat_name, scoring_type, runs_count FROM event_heat_settings WHERE event_id = ? ORDER BY heat_number ASC");
    $heat_stmt->execute([$event_id]);
    $heats = $heat_stmt->fetchAll(PDO::FETCH_ASSOC);
     //$runs_per_heat = intval($heats['runs_count']);
}

// Load scoring layout configuration - TEMPORARILY DISABLED TO DEBUG RENDERING ISSUE
$scoring_layout_config = null;
/*
$scoring_layout_config_path = __DIR__ . '/../includes/ScoringLayoutConfig.php';
if (file_exists($scoring_layout_config_path)) {
    try {
        include_once $scoring_layout_config_path;
        if (!class_exists('ScoringLayoutConfig')) {
            error_log("ScoringLayoutConfig class not found after include");
        } else {
            $current_heat = $_GET['heat'] ?? ($_POST['heat_number'] ?? 1);
            $scoring_layout_config = ScoringLayoutConfig::getEffectiveConfig($pdo, $event_id, $current_heat);
        }
    } catch (Exception $e) {
        error_log("Scoring layout config error: " . $e->getMessage());
        error_log("Stack trace: " . $e->getTraceAsString());
        $scoring_layout_config = null;
    }
} else {
    error_log("ScoringLayoutConfig file not found at: " . $scoring_layout_config_path);
}
*/

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Judge Scoring</title>
    <?php include_once '../judge/stylesheets_judge.php'; ?>
    
    <style>
        html {
            scroll-behavior: smooth;
            scroll-padding-top: 130px;
        }
       
        .bg-gradient {
            background-image: linear-gradient(359deg, rgb(255 255 255 / 29%), rgba(255, 255, 255, 0)) !important;
        }

        .container-StyleScore:before {
        
            background: gainsboro;
        
        }
                .text-primary { --bs-text-opacity: 1; color: var(--bs-teal) #2ed3be !important;
        }
        .text-ui-bigger { font-size: 1rem !important; }
                .score-status-buttons label {
            display: flex
        ;
            align-content: center;
            justify-content: center;
            align-items: center;
        }


        /* Layout Configuration Styles */
        .layout-hidden {
            display: none !important;
        }
        
        /* Compact Mode Styles */
        body.compact-mode .card {
            margin-bottom: 0.5rem !important;
        }
        
        body.compact-mode .card-body {
            padding: 0.75rem !important;
        }
        
        body.compact-mode .card-header {
            padding: 0.5rem 0.75rem !important;
        }
        
        body.compact-mode .score-container {
            padding: 0.5rem !important;
        }
        
        body.compact-mode .mb-3 {
            margin-bottom: 0.75rem !important;
        }
        
        body.compact-mode .mb-4 {
            margin-bottom: 1rem !important;
        }
        
        body.compact-mode .form-control,
        body.compact-mode .form-select {
            padding: 0.375rem 0.75rem !important;
        }
        
        /* Hide Empty Cards */
        body.hide-empty-cards .empty-card-hidden {
            display: none !important;
        }
        
        /* Mobile-first responsive design */
        body {
           
            font-size: 16px; /* Better for mobile touch */
        }
        
        .score-container {
           
            margin: 0 auto;
            padding: 1rem;
        }
        
        /* Large touch-friendly buttons */
      
        
        /* Large form inputs for mobile */
        .form-control, .form-select {
            
            font-size: 1rem;           
        }
        
        .score-input {
            
           
            font-weight: bold !important;
            text-align: center;
            border: 3px solid #0d6efd;
        }
        
        .bib-input {
            font-size: 1.5rem;
            font-weight: bold;
            text-align: center;
        }
        
        /* Status buttons */
        .status-btn {
            
            font-size: 14px;
            flex: 1;
        }
        
     
        
        /* Compact cards */
        
        
       
        
        /* Latest scores table */
        .scores-table {
            font-size: 0.9rem;
        }
        
        .scores-table td, .scores-table th {
            padding: 0.5rem;
            vertical-align: middle;
        }
        
        /* Quick action badges */
        .quick-badge {
            font-size: 0.8rem;
            padding: 0.25rem 0.5rem;
        }
        .criteria-input {
            font-size: 1.8em;
            font-weight: bold !important;
            text-align: center;
            border: 2px solid #0d6efd;
        }
        /* Mobile keyboard optimization */
        
        
        input[type="number"]::-webkit-outer-spin-button,
        input[type="number"]::-webkit-inner-spin-button {
            -webkit-appearance: none;
            margin: 0;
        }
        
        /* Responsive adjustments */
        @media (max-width: 576px) {
            .container-fluid {
                padding-left: 0.5rem;
                padding-right: 0.5rem;
            }
            
            .score-input {
              
                min-height: 70px;
            }
            
            .submit-btn {
                min-height: 60px;
                font-size: 1.3rem;
            }
            
            .heat-btn, .run-btn {
                min-height: 45px;
                font-size: 1rem;
            }
            
            /* Stack cards vertically on mobile with less spacing */
            .score-container .row {
                margin: 0 -0.5rem;
            }
            
            .score-container .row > [class*="col-"] {
                padding: 0 0.5rem;
                margin-bottom: 1rem;
            }
            
            /* Adjust card spacing on mobile */
            .score-container .card {
                margin-bottom: 0.75rem;
            }
        }
        
        @media (min-width: 577px) and (max-width: 991px) {
            /* Tablet adjustments */
            .score-container .card {
                margin-bottom: 1rem;
            }
            
            .form-control, .form-select {
                min-height: 45px;
                font-size: 1rem;
                   
            }
            
            /* Better spacing for tablet */
            .score-container .row {
                margin: 0 -0.75rem;
            }
            
            .score-container .row > [class*="col-"] {
                padding: 0 0.75rem;
            }
        }
        
        @media (min-width: 992px) {
            /* Desktop adjustments */
            .score-container .row {
                align-items: start;
            }
            
            /* Optimize column heights on desktop */
            .score-container .col-12 {
                display: flex;
                flex-direction: column;
            }
        }
        
        /* Loading states */
        .loading {
            opacity: 0.7;
            pointer-events: none;
        }
        
        /* Success animation */
        .score-submitted {
            animation: successPulse 0.6s ease-in-out;
        }
        
        @keyframes successPulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); background-color: #d1e7dd; }
            100% { transform: scale(1); }
        }
        .pulse-animation {
        animation: pulse 0.5s ease-in-out;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }

        .form-check-input:checked {
            background-color: #198754;
            border-color: #198754;
        }

        /* Figure toggle chips - Enhanced for better UX */
        .figure-row { 
            border: 1px dashed rgba(0,0,0,.1); 
            
            background: rgba(255,255,255,.8); 
            transition: all 0.2s ease;
        }
        .figure-row:hover {
            background: rgba(255,255,255,.95);
            border-color: rgba(0,0,0,.2);
        }

        /* Category-specific colors */
        .figure-row[data-category="Rotation"] { border-left: 4px solid #007bff; }
        .figure-row[data-category="Direction"] { border-left: 4px solid #28a745; }
        .figure-row[data-category="Axis"] { border-left: 4px solid #dc3545; }
        .figure-row[data-category="Grab"] { border-left: 4px solid #fd7e14; }
        .figure-row[data-category="Feature"] { border-left: 4px solid #6610f2; }
        .figure-row[data-category="Rail"] { border-left: 4px solid #20c997; }

        .figure-chip {
            border-radius: 8px !important;
            font-weight: 500;
            min-width: 60px;
            transition: all 0.15s 
        ease;
            position: relative;
            font-size: 2em;
            padding-left: .8em;
            padding-right: .8em;
        }
        .figure-chip:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(0,0,0,0.15);
        }
        .figure-chip.active { 
            color: #fff !important; 
            background-color: #198754 !important; 
            border-color: #198754 !important;
            box-shadow: 0 2px 8px rgba(25,135,84,0.3);
        }
        .figure-chip.has-history { 
            position: relative; 
            font-weight: 500; 
            color: #664d03; 
            background-color: #fff3cd; 
            border-color: #ffecb5; 
        }
        .figure-chip.has-history.active { 
            color: #fff !important; 
            background-color: #198754 !important; 
            border-color: #198754 !important; 
        }
        .figure-chip.has-history .text-muted { font-size: 0.75em; font-weight: normal; }
        .figure-chip.historical { color: #664d03; background-color: #fff3cd; border-color: #ffecb5; opacity: 0.8; position: relative; }
        .figure-chip.historical::after { content: " (" attr(data-run) ")"; font-size: 0.75em; opacity: 0.7; white-space: nowrap; }

        .figure-category-header {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 8px;
            font-weight: 600;
            color: #495057;
        }

        .figure-category-icon {
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            color: white;
            font-size: 12px;
        }

        .clear-cat { 
            border-radius: 8px !important; 
            min-width: 40px;
        }
        .clear-cat:hover {
            background-color: #dc3545 !important;
            border-color: #dc3545 !important;
            color: white !important;
        }

        /* Figures card toggle styling */
        .figures-card-header {
            cursor: pointer;
            transition: background-color 0.2s ease;
        }

        .figures-card-header:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }

        .toggle-icon {
            transition: transform 0.3s ease;
        }

        .toggle-icon.collapsed {
            transform: rotate(180deg);
        }

        #figuresCardBody {
            transition: all 0.3s ease;
            overflow: hidden;
        }

        #figuresCardBody.collapsed {
            max-height: 0;
            padding-top: 0;
            padding-bottom: 0;
            opacity: 0;
        }

        /* Diversity validation styling */
        #diversityValidation {
            transition: all 0.3s ease;
        }

        #diversityValidation .alert {
            margin-bottom: 0.5rem;
        }

        #diversityValidation .alert:last-child {
            margin-bottom: 0;
        }

        .diversity-requirement-item {
            background: rgba(255,255,255,0.8);
            border-left: 4px solid #dee2e6;
            transition: border-color 0.3s ease;
        }

        .diversity-requirement-item.error {
            border-left-color: #dc3545;
        }

        .diversity-requirement-item.warning {
            border-left-color: #ffc107;
        }

        .diversity-requirement-item.success {
            border-left-color: #198754;
        }

        .submit-btn.btn-warning {
            animation: pulse-warning 2s infinite;
        }

        @keyframes pulse-warning {
            0% { box-shadow: 0 0 0 0 rgba(255, 193, 7, 0.4); }
            70% { box-shadow: 0 0 0 10px rgba(255, 193, 7, 0); }
            100% { box-shadow: 0 0 0 0 rgba(255, 193, 7, 0); }
        }

        /* Edit Modal Figure Selection Styling */
        #editFigureGrid .figure-row {
            border: 1px dashed rgba(0,0,0,.1);
            
            background: rgba(255,255,255,.7);
        }

        #editFigureGrid .figure-chip {
            border-radius: 999px;
            transition: all 0.2s ease;
        }

        #editFigureGrid .figure-chip.active {
            color: #0b5137;
            background-color: #d1e7dd;
            border-color: #badbcc;
            font-weight: 600;
        }

        #editFigureGrid .figure-chip:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        #editFigureGrid .clear-cat {
            border-radius: 999px;
        }

        #editFigureSummary {
            border-left: 4px solid #0dcaf0;
        }

        .criteria-input-grid {
            display: grid;
            grid-template-columns: auto auto;
            transition: opacity 0.3s ease, filter 0.3s ease;
        }
        .criteria-input-grid.disabled {
            opacity: 0.5;
            pointer-events: none;
            filter: grayscale(50%);
        }

        .criteria_input_field {
            box-shadow: none !important;
            border-width: 1px !important;
            
        }
        input.score-input, input.criteria-input {
            color: #000cff !important;
            background-color: transparent;
            
            margin-left: auto;
            margin-right: auto;
        }
        input#score_value, input#bibInput {
        
            font-size: clamp(2rem, 0.1818rem + 3.6364cqi, 3rem) !important;
        }
        div#current-participant-name {
            font-size: clamp(.85rem, 0.1818rem + 3.6364cqi, 1.5rem) !important;
        }
        .score-input-wrapper {
        
            margin-left: auto;
            margin-right: auto;
        
        }

        .key {
            min-height: 9vh;
            border: 2px solid;
            border-radius: var(--bs-border-radius-lg) !important;
        }

        #nextButton.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .score-status-buttons.disabled {
            opacity: 0.5;
            pointer-events: none;
            filter: grayscale(50%);
        }

        #submitBtn.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        /* Loading state for participant dropdown */
        select[name="ep_id"].loading {
            background: linear-gradient(90deg, #f8f9fa 0%, #e9ecef 50%, #f8f9fa 100%);
            background-size: 200% 100%;
            animation: shimmer 1.5s infinite;
            pointer-events: none;
        }

        @keyframes shimmer {
            0% { background-position: -200% 0; }
            100% { background-position: 200% 0; }
        }

        select[name="ep_id"].loading option {
            color: #6c757d;
            font-style: italic;
        }

        /* Manual Override Warning Banner - Persistent bottom notification */
        .manual-override-banner {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: linear-gradient(135deg, #ff6b6b 0%, #ee5a24 100%);
            color: white;
            padding: 0.6rem 1rem;
            box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.3);
            z-index: 9999;
            display: none;
            animation: slideUpBanner 0.3s ease-out;
            border-top: 3px solid rgba(255, 255, 255, 0.3);
        }

        @keyframes slideUpBanner {
            from {
                transform: translateY(100%);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }

        .manual-override-content {
            display: flex;
            align-items: center;
            gap: 1rem;
            max-width: 1400px;
            margin: 0 auto;
        }

        .manual-override-icon {
            font-size: 1.3rem;
            flex-shrink: 0;
            animation: pulseWarning 2s infinite;
            line-height: 1;
        }

        @keyframes pulseWarning {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.7; transform: scale(1.1); }
        }

        .manual-override-message {
            flex: 1;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .manual-override-message strong {
            font-size: 0.95rem;
            white-space: nowrap;
        }

        .manual-override-details {
            font-size: 0.85rem;
            opacity: 0.95;
            line-height: 1.3;
        }

        .manual-override-actions {
            display: flex;
            gap: 0.5rem;
            flex-shrink: 0;
        }

        .manual-override-banner .btn {
            white-space: nowrap;
            font-weight: 500;
            padding: 0.35rem 0.75rem;
            font-size: 0.85rem;
        }

        .manual-override-banner .btn-light {
            background: white;
            color: #ee5a24;
            border: none;
        }

        .manual-override-banner .btn-light:hover {
            background: rgba(255, 255, 255, 0.9);
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
        }

        .manual-override-banner .btn-outline-light {
            border-color: rgba(255, 255, 255, 0.5);
            color: white;
            padding: 0.35rem 0.6rem;
        }

        .manual-override-banner .btn-outline-light:hover {
            background: rgba(255, 255, 255, 0.2);
            border-color: white;
        }

        /* Responsive adjustments for mobile */
        @media (max-width: 768px) {
            .manual-override-content {
                flex-direction: column;
                gap: 1rem;
                text-align: center;
            }
            
            .manual-override-icon {
                font-size: 1.5rem;
            }
            
            .manual-override-actions {
                width: 100%;
                justify-content: center;
            }
            
            .manual-override-message strong {
                font-size: 1rem;
            }
            
            .manual-override-details {
                font-size: 0.85rem;
            }
        }

        .key span {
            position: absolute;
            transform: translateY(-50%) translateX(-50%);
        }
        div#otherHeatScores tbody{
            display: block;    
            overflow: auto;
        }

        div#otherHeatScores thead, div#otherHeatScores tbody tr {
            display: table;
            width: 100%;
            table-layout: fixed;/* even columns width , fix width of table too*/
        }
        div#otherHeatScores thead {
            width: calc( 100% - 1em )/* scrollbar is average 1em/16px width, remove it from thead width */
        }

        /* Dark Mode Styles */
        body.dark-mode {
            background: #1a1a1a !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .body-bg-aurora-bright {
            background: #1a1a1a !important;
        }

     

        /* Theme option card hover effect */
        .theme-option-card:hover {
            transform: translateY(-5px);
            transition: transform 0.2s ease;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }

        body.dark-mode .card {
            background-color: #2d2d2d !important;
            border-color: #404040 !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .card-header {
            background-color: #333 !important;
            border-color: #404040 !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .form-control,
        body.dark-mode .form-select {
            background-color: #3a3a3a !important;
            border-color: #555 !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .form-control:focus,
        body.dark-mode .form-select:focus {
            background-color: #404040 !important;
            border-color: #0d6efd !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .btn-outline-secondary {
            color: #e0e0e0 !important;
            border-color: #555 !important;
        }

        body.dark-mode .btn-outline-secondary:hover {
            background-color: #404040 !important;
            border-color: #666 !important;
        }

        body.dark-mode .table {
            color: #000000 !important;
                filter: invert(1);
        }

        body.dark-mode .table-light {
            background-color: #333 !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .text-muted {
            color: #aaa !important;
        }

        body.dark-mode .badge.bg-light {
            background-color: #444 !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .alert {
            background-color: #2d2d2d !important;
            border-color: #404040 !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .alert-info {
            background-color: #1a3a52 !important;
            border-color: #0d6efd !important;
        }

        body.dark-mode .alert-success {
            background-color: #1a3a2a !important;
            border-color: #198754 !important;
        }

        body.dark-mode .alert-warning {
            background-color: #4a3a1a !important;
            border-color: #ffc107 !important;
        }

        body.dark-mode .modal-content {
            background-color: #2d2d2d !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .modal-header {
            border-color: #404040 !important;
        }

        body.dark-mode .modal-footer {
            border-color: #404040 !important;
        }

        body.dark-mode .figure-row {
            background: rgba(45,45,45,.8) !important;
        }

        body.dark-mode .figure-row:hover {
            background: rgba(60,60,60,.95) !important;
        }

        body.dark-mode input[type="number"],
        body.dark-mode input[type="text"] {
            background-color: #3a3a3a !important;
            color: #e0e0e0 !important;
        }

        body.dark-mode .criteria-input,
        body.dark-mode .score-input {
            background-color: #2d2d2d !important;
            color: #0d6efd !important;
            border-color: #0d6efd !important;
        }

        body.dark-mode .container-StyleScore:before {
            background: #1a1a1a !important;
        }
        .dark-mode .score-input-wrapper button{
                background-color: rgb(44, 62, 80) !important;
                    color: var(--bs-info-bg-subtle) !important;
            border: 2px solid rgba(24, 57, 188, 1) !important;

        }
        .dark-mode .score-input-wrapper button.submit{
                background-color: rgba(var(--bs-info-rgb)) !important;
                    color: var(--bs-info-bg-subtle) !important;
            border: 2px solid rgba(6, 111, 231, 1) !important;

        }

        .dark-mode #bibKeypadCollapse button{
                background-color: rgb(24 188 156 / 29%) !important;
                    color: var(--bs-info-bg-subtle) !important;
            border: 2px solid var(--bs-info-border-subtle) !important;

        }
        .dark-mode #bibKeypadCollapse button.submit{
                background-color: rgb(24, 188, 156) !important;
                    color: rgb(214, 234, 248) !important;
            border: 2px solid var(--bs-info-border-subtle) !important;

        }
        .score-input-wrapper .btn, div#bibKeypadCollapse  .btn {
            transition: all 0.2s ease;
        }

        .score-input-wrapper .btn:hover {background-color: #0099ff !important;box-shadow: var(--bs-box-shadow) !important;scale: .95;}
        div#bibKeypadCollapse  .btn:hover {background-color: #02eca7 !important;box-shadow: var(--bs-box-shadow) !important;scale: .95;}


        /* Floating Dark Mode Toggle */
        .floating-dark-mode-toggle {
            position: fixed;
            bottom: 0px;
            
            z-index: 1000;
            transition: all 0.3s ease;
        }
        input#score_value, #bibInput, select {border: 2px solid;}


        select#ep_id {
            border: 2px solid;
            font-size: 1.5em !important;
        }
        .dark-mode-btn {
                width: 35px;
                height: 35px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            transition: all 0.3s ease;
        }

        .dark-mode-btn:hover {
            transform: scale(1.1);
            box-shadow: 0 6px 16px rgba(0,0,0,0.4);
        }

        body.dark-mode .dark-mode-btn {
            background-color: #ffc107 !important;
            border-color: #ffc107 !important;
            color: #000 !important;
        }

        /* Responsive positioning for mobile */
        @media (max-width: 991px) {
            .floating-dark-mode-toggle {
                bottom: 0px;
            
            }
            
            .dark-mode-btn {
                width: 35px;
                height: 35px;
                font-size: 20px;
            }
        }

        /* No Active Heat Overlay */
        .no-active-heat-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.85);
            backdrop-filter: blur(8px);
            z-index: 1050;
            display: none;
            align-items: center;
            justify-content: center;
            animation: fadeIn 0.3s ease-out;
        }

        .no-active-heat-overlay.show {
            display: flex;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .no-active-heat-content {
            background: white;
            border-radius: 16px;
            padding: 3rem 2rem;
            
            width: 90%;
            margin: 1rem;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            animation: slideUp 0.4s ease-out;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .no-active-heat-icon {
            font-size: 4rem;
            color: #ffc107;
            margin-bottom: 1.5rem;
            animation: pulseIcon 2s infinite;
        }

        @keyframes pulseIcon {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.7; transform: scale(1.1); }
        }

        .no-active-heat-content h3 {
            color: #333;
            margin-bottom: 1rem;
            font-size: 1.75rem;
            font-weight: 600;
        }

        .no-active-heat-content p {
            color: #666;
            margin-bottom: 1.5rem;
            font-size: 1.1rem;
            line-height: 1.6;
        }

        .no-active-heat-info {
            background: #f8f9fa;
            border-left: 4px solid #0d6efd;
            padding: 1rem;
            border-radius: 8px;
            margin-top: 1.5rem;
            text-align: left;
        }

        .no-active-heat-info small {
            color: #495057;
            display: block;
            margin-bottom: 0.5rem;
        }

        .no-active-heat-info strong {
            color: #0d6efd;
            font-size: 1.1rem;
        }

        .checking-status {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: #6c757d;
            font-size: 0.9rem;
            margin-top: 1rem;
        }

        .checking-status .spinner {
            width: 16px;
            height: 16px;
            border: 2px solid #e9ecef;
            border-top-color: #0d6efd;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Dark mode styles for overlay */
        body.dark-mode .no-active-heat-content {
            background: #2d2d2d;
            color: #e0e0e0;
        }

        body.dark-mode .no-active-heat-content h3 {
            color: #e0e0e0;
        }

        body.dark-mode .no-active-heat-content p {
            color: #b0b0b0;
        }

        body.dark-mode .no-active-heat-info {
            background: #1a1a1a;
            border-color: #0d6efd;
        }

        body.dark-mode .no-active-heat-info small {
            color: #b0b0b0;
        }

        /* Custom Tooltip Styling for Heat Bracket Cards */
        .tooltip {
            z-index: 8500 !important;
        }

        .tooltip .tooltip-inner {
            background-color: #ffffff !important;
            color: #212529 !important;
            border: 1px solid #dee2e6 !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15) !important;
            text-align: left !important;
            max-width: 280px !important;
            padding: 12px 14px !important;
            font-size: 13px !important;
            line-height: 1.5 !important;
        }

        .tooltip .tooltip-arrow::before {
            border-top-color: #ffffff !important;
            border-bottom-color: #ffffff !important;
        }

        /* Dark mode tooltip styles */
        body.dark-mode .tooltip .tooltip-inner {
            background-color: #2d2d2d !important;
            color: #e0e0e0 !important;
            border-color: #495057 !important;
        }

        body.dark-mode .tooltip .tooltip-arrow::before {
            border-top-color: #2d2d2d !important;
            border-bottom-color: #2d2d2d !important;
        }

        /* ===== DRAWING BOARD MODAL STYLES ===== */
        .drawing-toolbar {
            padding: 8px 8px 10px;
            background: #ffffff;
            border-top: 1px solid #dddddd;
            display: flex;
            flex-wrap: nowrap;
            gap: 6px;
            align-items: center;
            justify-content: flex-start;
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }

        .toolbar-group {
            display: inline-flex;
            gap: 4px;
            align-items: center;
            padding-right: 6px;
            border-right: 1px solid #eee;
            margin-right: 4px;
        }

        .toolbar-group:last-child {
            border-right: none;
            margin-right: 0;
            padding-right: 0;
        }

        .toolbar-group-label {
            font-size: 11px;
            color: #555;
            white-space: nowrap;
            margin-right: 4px;
        }

        .drawing-toolbar button {
            border: 1px solid #ccc;
            background: #fafafa;
            padding: 6px 10px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 12px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 4px;
            min-width: 44px;
            min-height: 38px;
            white-space: nowrap;
        }

        .drawing-toolbar button span.icon {
            font-size: 16px;
            line-height: 1;
        }

        .drawing-toolbar button.active {
            border-color: #007bff;
            background: #e7f0ff;
        }

        .drawing-toolbar button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .color-dot {
            width: 24px;
            height: 24px;
            border-radius: 50%;
            border: 1px solid #aaa;
            padding: 0;
            min-width: 32px !important;
            min-height: 32px !important;
        }

        .color-dot.active {
            border: 3px solid #007bff;
            box-shadow: 0 0 0 2px rgba(0, 123, 255, 0.2);
        }

        .sizes button {
            width: 32px;
            height: 32px;
            padding: 0 !important;
        }

        .layer-lock-label,
        .layer-opacity-label {
            font-size: 11px;
            white-space: nowrap;
            display: inline-flex;
            align-items: center;
            gap: 4px;
        }

        #bgOpacityRange {
            width: 70px;
        }

        .drawing-trigger-btn {
            position: relative;
        }

        .drawing-trigger-btn.has-drawing::after {
            content: '';
            position: absolute;
            top: -4px;
            right: -4px;
            width: 12px;
            height: 12px;
            background: #28a745;
            border: 2px solid white;
            border-radius: 50%;
        }

        body.dark-mode .drawing-toolbar {
            background: #2b3035;
            border-top-color: #495057;
        }

        body.dark-mode .drawing-toolbar button {
            background: #495057;
            border-color: #6c757d;
            color: #f8f9fa;
        }

        body.dark-mode .drawing-toolbar button.active {
            border-color: #0d6efd;
            background: #0a3a6e;
        }

        body.dark-mode .toolbar-group {
            border-right-color: #495057;
        }

        body.dark-mode .toolbar-group-label {
            color: #adb5bd;
        }

        body.dark-mode #drawingCanvas {
            background: #ffffff !important;
        }

        @media (min-width: 900px) {
            .drawing-toolbar {
                justify-content: center;
            }
        }

        /* Heat Flow Preview in No Active Heat Overlay */
        .heat-flow-preview {
            background: #f8f9fa;
            border-radius: 12px;
            padding: 1.5rem;
                max-height: 65vh;
            overflow-x: auto;
            overflow-y: auto;
        }

        .heat-flow-preview h6 {
            color: #495057;
            font-size: 0.9rem;
            font-weight: 600;
            margin-bottom: 1rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Bracket Flow Layout */
        .bracket-flow-container {
            position: relative;
            display: flex;
            align-items: center;
            justify-content: flex-start;
            flex-wrap: nowrap;
            min-width: 800px;
            padding: 20px;
        }

        .bracket-column {
            display: flex;
            flex-direction: column;
            justify-content: center;
            gap: 20px;
            min-width: 160px;
            margin-right: 60px;
            flex: 1;
        }

        .bracket-column:last-child {
            margin-right: 0;
        }

        .bracket-column-header {
            text-align: center;
            font-size: 0.75rem;
            font-weight: 600;
            color: #6c757d;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 1rem;
            padding: 0.5rem;
            background: white;
            border-radius: 6px;
        }

        .bracket-heat-compact {
            position: relative;
            width: 180px;
            margin-bottom: 40px;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .bracket-heat-compact:hover {
            transform: translateY(-2px);
        }

        .bracket-heat-card-mini {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 0.75rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            transition: all 0.2s ease;
        }

        .bracket-heat-compact:hover .bracket-heat-card-mini {
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            border-color: #0d6efd;
        }

        .bracket-heat-card-mini.completed {
            background: #d1e7dd;
            border-color: #198754;
        }

        .bracket-heat-card-mini.active {
            background: #cfe2ff;
            border-color: #0d6efd;
            border-width: 3px;
            animation: activeGlow 2s infinite;
        }

        @keyframes activeGlow {
            0%, 100% { box-shadow: 0 0 0 0 rgba(13, 110, 253, 0.4); }
            50% { box-shadow: 0 0 0 6px rgba(13, 110, 253, 0); }
        }

        .bracket-heat-card-mini.next-up {
            background: #fff3cd;
            border-color: #ffc107;
            border-width: 3px;
        }

        .bracket-heat-header-mini {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 0.5rem;
        }

        .bracket-heat-number-mini {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 28px;
            height: 28px;
            background: #e9ecef;
            color: #495057;
            border-radius: 50%;
            font-weight: 700;
            font-size: 0.85rem;
        }

        .bracket-heat-card-mini.completed .bracket-heat-number-mini {
            background: #198754;
            color: white;
        }

        .bracket-heat-card-mini.active .bracket-heat-number-mini {
            background: #0d6efd;
            color: white;
        }

        .bracket-heat-card-mini.next-up .bracket-heat-number-mini {
            background: #ffc107;
            color: #000;
        }

        .bracket-heat-status-icon {
            font-size: 0.75rem;
        }

        .bracket-heat-name-mini {
            font-size: 0.8rem;
            font-weight: 600;
            color: #212529;
            margin-bottom: 0.25rem;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .bracket-heat-info-mini {
            display: flex;
            gap: 0.5rem;
            font-size: 0.7rem;
            color: #6c757d;
        }

        .bracket-heat-info-mini i {
            font-size: 0.65rem;
        }

        /* SVG Connections */
        .bracket-connections {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 0;
        }

        .bracket-heat-compact {
            position: relative;
            z-index: 1;
        }

        .connection-path {
            stroke: #adb5bd;
            stroke-width: 2;
            fill: none;
        }

        .connection-path.completed {
            stroke: #198754;
            stroke-width: 3;
        }

        .connection-path.active {
            stroke: #0d6efd;
            stroke-width: 3;
        }

        /* Heat Detail Popover */
        .heat-detail-popover {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            z-index: 10001;
            max-width: 400px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            animation: popIn 0.2s ease-out;
        }

        @keyframes popIn {
            from {
                opacity: 0;
                transform: translate(-50%, -50%) scale(0.9);
            }
            to {
                opacity: 1;
                transform: translate(-50%, -50%) scale(1);
            }
        }

        .heat-detail-backdrop {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            z-index: 10000;
        }

        .heat-detail-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 1rem;
            padding-bottom: 0.75rem;
            border-bottom: 2px solid #e9ecef;
        }

        .heat-detail-title {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .heat-detail-number {
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #0d6efd;
            color: white;
            border-radius: 50%;
            font-weight: 700;
            font-size: 1.1rem;
        }

        .heat-detail-name {
            font-size: 1.1rem;
            font-weight: 600;
            color: #212529;
        }

        .heat-detail-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: all 0.2s ease;
        }

        .heat-detail-close:hover {
            background: #f8f9fa;
            color: #212529;
        }

        .heat-detail-section {
            margin-bottom: 1rem;
        }

        .heat-detail-section-title {
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            color: #6c757d;
            margin-bottom: 0.5rem;
            letter-spacing: 0.5px;
        }

        .heat-detail-participants {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .heat-detail-participant {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.5rem;
            background: #f8f9fa;
            border-radius: 6px;
            margin-bottom: 0.5rem;
        }

        .heat-detail-participant:last-child {
            margin-bottom: 0;
        }

        .heat-detail-bib {
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: white;
            border: 2px solid #0d6efd;
            border-radius: 4px;
            font-weight: 700;
            font-size: 0.85rem;
            color: #0d6efd;
        }

        .heat-detail-participant-name {
            flex: 1;
            font-size: 0.9rem;
            color: #212529;
        }

        .heat-detail-meta {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 0.75rem;
        }

        .heat-detail-meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem;
            background: #f8f9fa;
            border-radius: 6px;
        }

        .heat-detail-meta-item i {
            color: #0d6efd;
        }

        .heat-detail-meta-label {
            font-size: 0.75rem;
            color: #6c757d;
        }

        .heat-detail-meta-value {
            font-size: 1rem;
            font-weight: 600;
            color: #212529;
        }

        /* Dark mode heat flow preview */
        body.dark-mode .heat-flow-preview {
            background: #1a1a1a;
        }

        body.dark-mode .heat-flow-preview h6 {
            color: #b0b0b0;
        }

        body.dark-mode .bracket-column-header {
            background: #2d2d2d;
            color: #b0b0b0;
        }

        body.dark-mode .bracket-heat-card-mini {
            background: #2d2d2d;
            border-color: #404040;
        }

        body.dark-mode .bracket-heat-card-mini.completed {
            background: #1a3a2a;
            border-color: #198754;
        }

        body.dark-mode .bracket-heat-card-mini.active {
            background: #1a2a4a;
            border-color: #0d6efd;
        }

        body.dark-mode .bracket-heat-card-mini.next-up {
            background: #4a3a1a;
            border-color: #ffc107;
        }

        body.dark-mode .bracket-heat-number-mini {
            background: #404040;
            color: #e0e0e0;
        }

        body.dark-mode .bracket-heat-name-mini {
            color: #e0e0e0;
        }

        body.dark-mode .bracket-heat-info-mini {
            color: #aaa;
        }

        body.dark-mode .heat-detail-popover {
            background: #2d2d2d;
            color: #e0e0e0;
        }

        body.dark-mode .heat-detail-header {
            border-color: #404040;
        }

        body.dark-mode .heat-detail-name {
            color: #e0e0e0;
        }

        body.dark-mode .heat-detail-close {
            color: #b0b0b0;
        }

        body.dark-mode .heat-detail-close:hover {
            background: #404040;
            color: #e0e0e0;
        }

        body.dark-mode .heat-detail-participant {
            background: #1a1a1a;
        }

        body.dark-mode .heat-detail-bib {
            background: #2d2d2d;
        }

        body.dark-mode .heat-detail-participant-name {
            color: #e0e0e0;
        }

        body.dark-mode .heat-detail-meta-item {
            background: #1a1a1a;
        }

        body.dark-mode .heat-detail-meta-value {
            color: #e0e0e0;
        }

        /* Responsive adjustments for heat flow */
        @media (max-width: 576px) {
            .no-active-heat-content {
                padding: 2rem 1rem;
                width: 95%;
            }
            
            .heat-flow-preview {
                padding: 1rem;
                max-height: 300px;
            }
            
            .bracket-flow-container {
                min-width: 600px;
            }
            
            .bracket-column {
                margin-right: 40px;
            }
            
            .bracket-heat-compact {
                width: 140px;
                margin-bottom: 30px;
            }
            
            .heat-detail-popover {
                width: 95%;
                padding: 1rem;
            }
        }

        /* Judge Status Modal Overlay */
        .judge-status-modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(4px);
            z-index: 10000;
            display: none;
            align-items: center;
            justify-content: center;
            animation: fadeIn 0.3s ease-out;
        }

        .judge-status-modal-overlay.show {
            display: flex;
        }

        .judge-status-modal-content {
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
            max-width: 600px;
            width: 90%;
            animation: slideUpModal 0.3s ease-out;
        }
        #judgeStatusHeader {
            display: flex;
            flex-direction: row !important;
        }

        .position-sticky #judgeStatusHeader {
            display: flex;
            flex-direction: column !important;
        }

        @keyframes slideUpModal {
            from {
                transform: translateY(50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
.figure-badge-A {
    background-color: #ffeaea !important;
}
.figure-badge-B {
    background-color: #d5ebff !important;
}
        body.dark-mode .judge-status-modal-content {
            background: #2d3748;
            color: #e2e8f0;
        }

        body.dark-mode .judge-status-modal-overlay {
            background: rgba(0, 0, 0, 0.7);
        }

        /* Prevent scrolling when modal is active */
        body.modal-overlay-active {
            overflow: hidden;
            position: fixed;
            width: 100%;
        }
    </style>
</head>

<body class="body-bg-aurora-bright">
    
    <?php
    // Apply scoring layout configuration - TEMPORARILY DISABLED
    if (false && $scoring_layout_config) {
        echo "<!-- Scoring Layout Configuration: Temporarily Disabled -->\n";
    }
    ?>
    
    <div class="container-fluid container-StyleScore p-2 p-md-4">
        
        <!-- Debug Information Boxes -->
        <div class="row mb-3" id="debugSection" style="display: none;">
            <div class="col-md-6">
                <!-- Incoming Data Debug Box -->
                <div class="card border-info">
                    <div class="card-header bg-info text-white py-2">
                        <h6 class="mb-0">
                            <i class="fas fa-download me-2"></i>Incoming Data
                            <button type="button" class="btn btn-sm btn-outline-light float-end" onclick="refreshIncomingData()">
                                <i class="fas fa-refresh"></i>
                            </button>
                        </h6>
                    </div>
                    <div class="card-body p-2">
                        <div class="small" style="font-family: monospace; background: #f8f9fa; padding: 10px; border-radius: 4px; max-height: 300px; overflow-y: auto;">
                            <div id="incomingDataContent">Loading...</div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <!-- Outgoing API Calls Debug Box -->
                <div class="card border-warning">
                    <div class="card-header bg-warning text-dark py-2">
                        <h6 class="mb-0">
                            <i class="fas fa-upload me-2"></i>Outgoing API Calls
                            <button type="button" class="btn btn-sm btn-outline-dark float-end" onclick="clearApiLog()">
                                <i class="fas fa-trash"></i>
                            </button>
                        </h6>
                    </div>
                    <div class="card-body p-2">
                        <div class="small" style="font-family: monospace; background: #f8f9fa; padding: 10px; border-radius: 4px; max-height: 300px; overflow-y: auto;">
                            <div id="apiCallsContent">No API calls yet...</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        
        
        <div class="score-container p-0">
            <!-- Main Scoring Form -->
            <form method="post" id="scoreForm" class="mb-0">
                <!-- Judge Info Header -->
                        <div class="card judge-info mb-3 bg-info position-sticky score-card sticky-top p-2 text-white bg-black">                
                            <div class="align-items-center d-flex flex-row justify-content-between">
                                
                                                    <span>                          
                                        <strong> <?= htmlspecialchars($judge_name) ?></strong>
                                        <span class="badge bg-primary ms-1"><?= $judge_id ?></span>
                                    
                                        </span>
                                        <span>  
                                        <strong><?= htmlspecialchars($assigned_events[0]['name'] ?? 'No Event') ?></strong>
                                        <?php if ($event_date): ?>
                                            <small class=""><?= date('M d, Y', strtotime($event_date)) ?></small>
                                        <?php endif; ?>
                                     </span>
                           
                                
                                
                                
                            </div>
                        </div>
                <div class="row g-3">
                    
                    <div class="col-12 col-lg-4 m-0">
                        

                        <!-- Format Info Card - Button to Open Modal -->
                        <?php if ($format_info): ?>
                        <div class="card score-card mb-3" id="formatInfoCard">
                            <div class="card-body p-2">
                                <button type="button" class="btn btn-outline-info w-100" data-bs-toggle="modal" data-bs-target="#formatInfoModal">
                                    <div class="d-flex align-items-center justify-content-center gap-2">
                                        <?php if ($format_icon_url): ?>
                                            <img src="<?= htmlspecialchars($format_icon_url) ?>" alt="format icon" style="width:24px;height:24px;object-fit:contain;" onerror="this.style.display='none'">
                                        <?php else: ?>
                                            <i class="fas fa-puzzle-piece"></i>
                                        <?php endif; ?>
                                        <strong><?= htmlspecialchars($format_info['format_name'] ?: $format_info['scoring_format']) ?></strong>
                                        <span class="badge bg-light text-dark"><?= htmlspecialchars($format_info['sport'] ?: '—') ?></span>
                                        <?php if (!empty($format_info['mode'])): ?>
                                            <span class="badge bg-primary"><?= htmlspecialchars(str_replace('_',' ', $format_info['mode'])) ?></span>
                                        <?php endif; ?>
                                        <i class="fas fa-info-circle ms-2"></i>
                                    </div>
                                </button>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Manual Event Selection -->
                        <?php if ($manual_mode): ?>
                        <div class="card score-card mb-3">
                            <div class="card-header p-2">
                                <i class="fas fa-calendar me-2"></i>Select Event
                            </div>
                            <div class="card-body p-2">
                                <select class="form-select" name="event_id" onchange="this.form.submit()">
                                    <option value="">-- Select Event --</option>
                                    <?php foreach ($assigned_events as $ev): ?>
                                        <option value="<?= $ev['id'] ?>" <?= $event_id == $ev['id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($ev['name']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <?php endif; ?>


                        <!-- Message Display -->
                        <?php if ($message): ?>
                            <div class="alert alert-success alert alert-dismissible fade show" role="alert">
                                <?= $message ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="col-12 col-lg-4 m-0">
                         <!-- Heat & Run Selection -->
                            <div class="card score-card sticky-top" id="heat-run-navigation" style="top: 40px; z-index: 1030;">
                                <div class="card-header p-2 bg-dark-subtle" id="heatRunCollapseHeader">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div class="d-flex align-items-center gap-2 w-100 justify-content-between">
                                            
                                            <span id="heatRunSummary" class="badge bg-primary">H-- / R-- / --%</span>
                                             <span class="d-flex align-items-center gap-1">
                                                <span id="heatRunBadge" class="badge bg-warning" style="display:none;">
                                                    <i class="fas fa-history me-1"></i>Restored
                                                </span>
                                            
                                                
                                                <div class="form-check form-switch" onclick="event.stopPropagation();">
                                                    <input class="form-check-input" type="checkbox" id="autoModeToggle" checked>
                                                    <label class="form-check-label small" for="autoModeToggle">Auto</label>
                                                </div>
                                                <i class="fas fa-chevron-down ps-3 pe-1" id="heatRunToggleIcon" style="cursor: pointer;"></i>
                                            </span>
                                        </div>
                                    </div>
                                </div>
                                <div class="card-body collapse bg-dark-subtle" id="heatRunCollapseBody">
                                    <div class="card-body p-2">
                                    <!-- Mode Toggle -->
                            
                                    <div class="btn-group btn-group-sm" onclick="event.stopPropagation();">
                                        <a href="score.php" class="btn btn-<?= !$manual_mode ? 'outline-primary' : 'primary' ?> btn-sm">
                                            <i class="fas fa-magic me-1"></i>Auto
                                        </a>
                                        <a href="?manual=1" class="btn btn-<?= $manual_mode ? 'outline-primary' : 'primary' ?> btn-sm">
                                            <i class="fas fa-hand-paper me-1"></i>Manual
                                        </a>
                                    </div>
                                    <!-- Auto Mode Status Display -->
                                    <div id="autoModeStatus" class="alert alert-info mb-3 mt-3" style="display:none;">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <i class="fas fa-robot me-2"></i>
                                                <strong>Auto Mode Active</strong>
                                                <div class="small text-muted mt-1">
                                                    Automatically selecting active heat and run from event flow
                                                </div>
                                            </div>
                                            <div class="text-end">
                                                <div class="small text-muted">Last update:</div>
                                                <div id="lastUpdateTime" class="small">--:--</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Manual Override Notice -->
                                    <div id="manualOverrideNotice" class="alert alert-warning mb-3 mt-3" style="display:none;">
                                        <i class="fas fa-hand-paper me-2"></i>
                                        <strong>Manual Override Active</strong>
                                        <div class="small">You have manually selected heat/run. Auto mode is disabled.</div>
                                    </div>
                                    
                                    <!-- Active Heat Info Display -->
                                    <div id="activeHeatInfo" class="alert alert-success mb-3" style="display:none;">
                                        <div class="row align-items-center">
                                            <div class="col">
                                                <div class="d-flex align-items-center">
                                                    <i class="fa fa-fire text-white me-2"></i>
                                                    <div>
                                                        <strong id="activeHeatName">Heat --</strong>
                                                        <div class="small text-muted" id="activeHeatDisplay">--</div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-auto">
                                                <div class="text-center">
                                                    <div class="badge bg-success-subtle" id="activeRunDisplay">Run --</div>
                                                    <div class="small text-muted">Active</div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="row mt-2" id="activeHeatProgress" style="display:none;">
                                            <div class="col">
                                                <div class="small text-muted">
                                                    Progress: <span id="progressText">-- / --</span>
                                                    (<span id="progressPercentage">--%</span>)
                                                </div>
                                                <div class="progress" style="height: 4px;">
                                                    <div class="progress-bar bg-success" id="progressBar" style="width: 0%"></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    </div>
                                    <!-- Heat Selection -->
                                    <div class="mb-3" onclick="event.stopPropagation();">
                                        <label class="form-label fw-bold">Heat Number</label>
                                        <select class="form-select" name="heat_number" id="heatSelect" onchange="handleHeatChange()" required>
                                            <option value="">-- Select Heat --</option>
                                            <?php foreach ($heats as $heat): ?>
                                                <option value="<?= $heat['heat_number'] ?>" <?= $heat_number == $heat['heat_number'] ? 'selected' : '' ?>>
                                                    Heat <?= $heat['heat_number'] ?><?= $heat['heat_name'] ? ' – ' . htmlspecialchars($heat['heat_name']) : '' ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <!-- Run Selection -->
                                    <div class="" onclick="event.stopPropagation();">
                                        <label class="form-label fw-bold">Run Number</label>
                                        <div class="d-grid gap-2">
                                            <div class="btn-group" role="group" id="runButtonGroup">
                                                <?php for ($r = 1; $r <= $runs_per_heat; $r++): ?>
                                                    <input type="radio" class="btn-check" name="run_number" id="run<?= $r ?>" 
                                                        value="<?= $r ?>" <?= $run_number == $r ? 'checked' : '' ?> 
                                                        onchange="handleRunChange()" required>
                                                    <label class="btn btn-outline-success run-btn" for="run<?= $r ?>">
                                                        <i class="fas fa-play me-2"></i>Run <?= $r ?>
                                                    </label>
                                                <?php endfor; ?>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    
                                </div>
                           
                        </div>
                        
                        


                        
                            
                        <div class="m-0 mt-3">
                        <!-- Judge Status Card - Always visible, sticky when pending -->
                        <div class="card mb-3 m-2" id="judge-status-card">
                            <div class="card-header p-2 d-flex justify-content-between align-items-center d-flex flex-column gap-3" id="judgeStatusHeader" >
                                <div class="d-flex justify-content-between align-items-center w-100">
                                    <div class="text-center flex-grow-1">
                                        <i class="fas fa-circle me-2" id="judgeStatusIcon"></i>
                                        <span class="fw-bold fs-5" id="judgeStatusText">Status: Ready</span>
                                    </div>
                                    
                                </div>
                                
                                <div id="judgeStatusActions">
                                    <!-- Action buttons will be inserted here -->
                                </div>
                            </div>
                        </div>
                        
                        <!-- Judge Status Options Collapse -->
                        <div class="collapse mb-3" id="judgeStatusCollapse">
                            <div class="card shadow-lg m-2">
                                <div class="card-header p-2 d-flex justify-content-between align-items-center bg-secondary-subtle">
                                    <span><i class="fas fa-exchange-alt me-2"></i>Change Status</span>
                                    <button type="button" class="btn btn-sm btn-outline-secondary" data-bs-toggle="collapse" data-bs-target="#judgeStatusCollapse">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                <div class="card-body p-3 bg-dark-subtle">
                                    <div class="d-grid gap-2">
                                        <div class="form-check form-switch ms-2 fs-5">
                                        <input class="form-check-input" type="checkbox" id="autoReadyToggle" onchange="handleAutoReadyToggle()">
                                        <label class="form-check-label small" for="autoReadyToggle" title="Automatically approve to Ready status">Auto Ready</label>
                                    </div>
                                        <button type="button" class="btn btn-success btn-lg" onclick="setJudgeStatus('ready')">
                                            <i class="fas fa-check-circle me-2"></i>Ready
                                        </button>
                                        <button type="button" class="btn btn-outline-warning bg-warning-subtle btn-lg" onclick="setJudgeStatus('pending')">
                                            <i class="fas fa-clock me-2"></i>Pending
                                        </button>
                                        <button type="button" class="btn btn-info btn-lg" onclick="setJudgeStatus('info')">
                                            <i class="fas fa-info-circle me-2"></i>Info
                                        </button>
                                        <button type="button" class="btn btn-warning btn-lg" onclick="setJudgeStatus('warning')">
                                            <i class="fas fa-exclamation-circle me-2"></i>Warning
                                        </button>
                                        <button type="button" class="btn btn-danger btn-lg" onclick="setJudgeStatus('danger')">
                                            <i class="fas fa-exclamation-triangle me-2"></i>Danger
                                        </button>
                                        
                                        
                                        <button type="button" class="btn btn-secondary btn-lg" onclick="setJudgeStatus('off')">
                                            <i class="fas fa-power-off me-2"></i>Off
                                        </button>
                                    </div>
                                    <div class="small text-muted mt-2 text-center">
                                        Status is visible to head judge and event coordinators
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Scoring Cards Container - Hidden when status is Pending -->
                        <div id="scoring-cards-container">
                        
                            <!-- Participant Selection Card -->
                            <div class="card mb-1 m-2 " id="participant-selection">
                                <div class="card-header p-2 fw-bold bg-success-subtle">
                                    <i class="fas fa-user me-2"></i>Participant Selection
                                </div>
                                <div class="card-body p-3 bg-dark-subtle">
                                    <!-- Participant Dropdown -->
                                    <div class=" mb-2" style="top: 50px; z-index: 10; border-bottom: 1px solid #dee2e6;">
                                        <label class="form-label fw-bold">Select Participant</label>
                                        <select class="text-ui-bigger form-select bg-gradient" name="ep_id" id="ep_id" required>
                                            <option value="" disabled>-- Select Participant --</option>
                                        </select>
                                    </div>
                                    <!-- BIB Quick Entry -->
                                    <div class="mb-3">
                                        <label class="form-label fw-bold">BIB Number</label>
                                        <input type="text" class="form-control bib-input ss-f-number bg-gradient" name="p_bib" id="bibInput" 
                                            inputmode="none" pattern="[0-9]*" placeholder="Click to open keypad" 
                                            autocomplete="off" readonly onclick="toggleBibKeypad()" style="cursor: pointer;">
                                        <div class="text-muted">Click above to open numeric keypad</div>
                                        
                                        <!-- BIB Numeric Keypad (Initially Collapsed) -->
                                        <div class="collapse mt-3" id="bibKeypadCollapse">
                                            <div class="btn-group-vertical rounded-2 w-100 p-3 mb-0" role="group" aria-label="BIB numeric keypad" id="bibKeypad">
                                                <div class="btn-group gap-2 mb-2">
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('1');"><span>1</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('2');"><span>2</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('3');"><span>3</span></button>
                                                </div>
                                                <div class="btn-group gap-2 mb-2">
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('4');"><span>4</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('5');"><span>5</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('6');"><span>6</span></button>
                                                </div>
                                                <div class="btn-group gap-2 mb-2">
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('7');"><span>7</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('8');"><span>8</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('9');"><span>9</span></button>
                                                </div>
                                                <div class="btn-group gap-2 mb-0">
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="backspaceBibInput();"><span>&lt;</span></button>
                                                    <button type="button" class="bg-gradient bg-success-subtle btn btn-lg btn-outline-success fs-1 key py-3 " onclick="addToBibInput('0');"><span>0</span></button>
                                                    <button type="button" class="bg-gradient bg-success btn btn-lg btn-primary py-3 key fs-1 submit" onclick="applyBibInput();"><span>OK</span></button>
                                                </div>
                                            </div>
                                        </div>
                                        <button id="nextButton" type="button" class="bg-success btn btn-lg btn-primary py-3 key fs-1 w-100 mt-2" onclick="scrollToCriteriaInput();">
                                            <span>Confirm</span>
                                        </button>
                                    </div>
                                    
                                    
                                </div>
                            </div>
                             
                          <!-- current participant strip info -->
                         <div class="card score-card sticky-top" id="current-participant-info" style="top: 80px; display: none;">
                                <?php
                                // Check for panel_theme cookie
                                $panel_theme = $_COOKIE['panel_theme'] ?? 'brite'; // Default to 'brite'
                                $text_color_class = ($panel_theme === 'brite') ? 'text-dark' : 'text-white';
                                ?>
                                <div class="card-body p-2 bg-primary bg-gradient <?= $text_color_class ?>">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div class="d-flex align-items-center gap-3 w-100">
                                            <span class="badge bg-white text-dark fs-5 px-2 py-1" id="current-participant-bib">--</span>
                                            <div class="d-flex flex-row w-100 align-items-center">
                                                <div class="fw-bold fs-3" id="current-participant-name">--</div>
                                                <div class="small ps-2 ms-auto" id="current-participant-country">
                                                    <span class="country-flag" id="current-participant-flag"></span>
                                                    <span id="current-participant-nationality">--</span>
                                                </div>
                                                <div class="badge bg-white fs-3 fw-bold me-3 ms-auto text-dark" id="current-participant-score">--</div>
                                            </div>
                                        </div>
                                        <button type="button" class="btn btn-sm btn-light" onclick="clearCurrentParticipant()" title="Clear selection">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>

                            <!-- Status Selection -->
                                <div class="p-3">                                    
                                    <div class="d-grid gap-1 score-status-buttons">
                                        <div class="btn-group bg-white" role="group">
                                            <input type="radio" class="btn-check" name="status" id="statusOK" value="OK" checked>
                                            <label class="btn btn-outline-success status-btn fs-6" for="statusOK">
                                                <i class="fas fa-check me-2"></i>OK
                                            </label>

                                            <input type="radio" class="btn-check" name="status" id="statusDNS" value="DNS">
                                            <label class="btn btn-outline-warning status-btn fs-6" for="statusDNS">
                                                <i class="fas fa-ban me-2"></i>DNS
                                            </label>
                                        
                                            <input type="radio" class="btn-check" name="status" id="statusDNF" value="DNF">
                                            <label class="btn btn-outline-danger status-btn fs-6" for="statusDNF">
                                                <i class="fas fa-flag me-2"></i>DNF
                                            </label>
                                            
                                            <input type="radio" class="btn-check" name="status" id="statusDSQ" value="DSQ">
                                            <label class="btn btn-outline-dark status-btn fs-6" for="statusDSQ">
                                                <i class="fas fa-times-circle me-2"></i>DSQ
                                            </label>
                                        </div>
                                        
                                        <?php if ($show_drawing_board_button): ?>
                                        <!-- Drawing Board Button -->
                                        <button type="button" class="btn btn-outline-primary btn-lg mt-2 drawing-trigger-btn" id="openDrawingBoardBtn" title="Add situation drawing">
                                            <i class="fas fa-palette me-2"></i>Add Drawing
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                              <!-- Additional Cards Wrapper - Hidden when status is not OK -->
                              <div id="additionalCardsWrapper">
                                <!-- Other Heat Scores Card -->
                        <div class="card mb-3" id="otherHeatScoresCard" style="display: none;">
                            <div class="card-header p-2 d-flex justify-content-between align-items-center">
                                <div>
                                    <i class="fas fa-users me-2"></i>Other Judges Scores
                                </div>
                                <button class="btn btn-sm btn-outline-secondary" type="button" 
                                        data-bs-toggle="collapse" data-bs-target="#otherHeatScoresBody" 
                                        aria-expanded="true" aria-controls="otherHeatScoresBody">
                                    <i class="fas fa-chevron-up"></i>
                                </button>
                            </div>
                            <div class="collapse show" id="otherHeatScoresBody">
                                <div class="card-body p-0">
                                    <div id="otherHeatScores"></div>
                                </div>
                            </div>
                        </div>
                              <!-- Figure History Display Card -->
                        <div class="card mb-2" id="figureHistoryCard" style="display: none;">
                            <div class="card-header p-2">
                                <i class="fas fa-history me-2"></i>Figure History
                            </div>
                            <div class="card-body p-2">
                                <div id="figureHistoryDisplay"></div>
                            </div>
                        </div>

                              <!-- Score Criteria -->
                        <?php if (!empty($scoring_criteria) && (!$scoring_layout_config || ($scoring_layout_config['criteria-input-grid'] ?? true))): ?>
                        <div class="card score-criteria mb-1 m-2 " id="criteria-input-grid">
                            <div class="card-header p-2 fw-bold bg-info-subtle">
                                <i class="fas fa-star me-2"></i>Score input 
                                <span class="badge bg-light text-dark"><?= count($scoring_criteria) ?></span>
                            </div>
                            <div class="card-body p-2 criteria-input-grid"> 
                                <!-- Dynamic Score Criteria Inputs -->
                                <?php foreach ($scoring_criteria as $index => $criterion): ?>
                                <div class="p-2 criteria_input_field scale-<?= $index + 1 ?>">
                                    <label class="form-label fw-bold">
                                        <?= htmlspecialchars($criterion['criteria_name']) ?>
                                        <span class="badge bg-secondary ms-1"><?= htmlspecialchars(strtoupper($criterion['criteria_type'])) ?></span>
                                    </label>
                                    <input type="number" 
                                        class="form-control criteria-input fs-2 ss-f-number" 
                                        name="criteria_scores[<?= $index ?>][value]" 
                                        data-criteria-type="<?= htmlspecialchars($criterion['criteria_type']) ?>"
                                        data-criteria-name="<?= htmlspecialchars($criterion['criteria_name']) ?>"
                                        min="0" max="<?= htmlspecialchars($format_info['scale_max'] ?? 99) ?>" 
                                        step="<?= ($format_info['precision_decimal'] ?? 0) ? '0.1' : '1' ?>" 
                                        inputmode="numeric" 
                                        placeholder="0" 
                                        autocomplete="off" 
                                        oninput="updateCriteriaRange(this, <?= $index ?>)">
                                    <input type="range" 
                                        class="form-range mt-2" 
                                        name="criteria_scores[<?= $index ?>][range]" 
                                        min="0" 
                                        max="<?= htmlspecialchars($format_info['scale_max'] ?? 99) ?>" 
                                        step="<?= ($format_info['precision_decimal'] ?? 0) ? '0.1' : '1' ?>"
                                        oninput="updateCriteriaInput(this, <?= $index ?>)">
                                    <!-- Hidden fields for criteria metadata -->
                                    <input type="hidden" name="criteria_scores[<?= $index ?>][type]" value="<?= htmlspecialchars($criterion['criteria_type']) ?>">
                                    <input type="hidden" name="criteria_scores[<?= $index ?>][name]" value="<?= htmlspecialchars($criterion['criteria_name']) ?>">
                                    <input type="hidden" name="criteria_scores[<?= $index ?>][sort_order]" value="<?= htmlspecialchars($criterion['sort_order']) ?>">
                                </div>
                                <?php endforeach; ?>
                                </div>
                                <!-- Final Calculated Score Display -->
                                <div class="p-0 m-0" id="final-calculated-score">
                                    <div class="d-flex justify-content-between align-items-center  score-input-wrapper">                                      
                                       
                                        <div class="card-body p-4 pt-3 pb-0 bg-dark-subtle">
                                <!-- Score Input -->
                                    <div class="mb-3">
                                        
                                        <input type="text" class=" bg-gradient form-control score-input ss-f-number bg-light" name="score_value" id="score_value"
                                            placeholder="Use keypad below" 
                                            autocomplete="off" required onkeydown="return false;" onfocus="this.blur()" oninput="updateMainScoreAndCriteria(this, 'range')">
                                    </div>
                                    <div class="mb-1 mt-4">
                                        <input type="range" class="form-range" name="score_range" min="0" max="99" id="customRange2" oninput="updateMainScoreAndCriteria(this, 'input')">
                                    </div>

                                    <!-- Numeric Keypad -->
                                    <div class="btn-group-vertical mt-0 rounded-2 w-100  p-3 mb-0 " role="group" aria-label="Numeric keypad">
                                        <div class="btn-group gap-2 mb-2">
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('1');"><span>1</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('2');"><span>2</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('3');"><span>3</span></button>
                                        </div>
                                        <div class="btn-group gap-2 mb-2">
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('4');"><span>4</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('5');"><span>5</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('6');"><span>6</span></button>
                                        </div>
                                        <div class="btn-group gap-2 mb-2">
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('7');"><span>7</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('8');"><span>8</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('9');"><span>9</span></button>
                                        </div>
                                        <div class="btn-group gap-2 mb-1">
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="backspaceScoreValue();"><span>&lt;</span></button>
                                            <button type="button" class="bg-gradient bg-info-subtle  btn btn-lg btn-outline-info fs-1 key py-3 " onclick="addToScoreValue('0');"><span>0</span></button>
                                            <button type="button" class="bg-gradient bg-info btn btn-lg btn-primary py-3 key fs-1 border-0 submit" onclick="applyScoreValue();"><span>OK</span></button>
                                        </div>
                                    </div>
                                </div>
                                    </div>
                                    <div class="d-flex justify-content-between align-items-center mt-1 p-3 pt-0">

                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="clearScoreInputsOnly()" title="Clear only score inputs (criteria and main score)" style="display: none;">
                                            <i class="fas fa-eraser me-1"></i>Clear Scores
                                        </button>
                                    </div>
                                </div>
                            </div>
                       
                        <?php else: ?>
                        <!-- Fallback: Single Score Criteria -->
                        <div class="card score-criteria mb-3">
                            <div class="card-header p-2">
                                <i class="fas fa-star me-2"></i>Score Criteria 
                            </div>
                            <div class="card-body p-2">
                                <!-- Score Criteria Input -->
                                <div class="mb-3">
                                    <label class="form-label fw-bold">Criteria (0-<?= htmlspecialchars($format_info['scale_max'] ?? 99) ?>)</label>
                                    <input type="number" class="form-control criteria-input" name="criteria_value" 
                                        min="0" max="<?= htmlspecialchars($format_info['scale_max'] ?? 99) ?>" 
                                        step="<?= ($format_info['precision_decimal'] ?? 0) ? '0.1' : '1' ?>" 
                                        inputmode="numeric" placeholder="0" 
                                        autocomplete="off" required  oninput="this.form.criteria_range.value=this.value">
                                </div>
                                <div class="mb-0">
                                    <input type="range" class="form-range" name="criteria_range" min="0" max="<?= htmlspecialchars($format_info['scale_max'] ?? 99) ?>" id="customRange2" oninput="this.form.criteria_value.value=this.value">
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>  

                     

                        <!-- Figures Card -->
                        <?php if ($uses_figures && !empty($figure_categories)): ?>
                        <div class="card mt-3 mb-3" id="figures-card">
                            <div class="card-header p-2" style="cursor: pointer;" onclick="toggleFiguresCard()" id="figuresCardHeader">
                                <div class="d-flex justify-content-between align-items-center">
                                    <span>
                                        <i class="fas fa-shapes me-2"></i>Figures
                                    </span>
                                    <i class="fas fa-chevron-up toggle-icon" id="figuresToggleIcon"></i>
                                </div>
                            </div>
                            <div class="card-body bg-dark-subtle" id="figuresCardBody">
                                <!-- Toggle chips per category -->
                                <div id="figureToggleGrid" class="d-flex flex-column gap-2">
                                    <?php 
                                    // Category icons mapping
                                    $category_icons = [
                                        'Rotation' => 'fas fa-sync-alt',
                                        'Direction' => 'fas fa-arrows-alt-h', 
                                        'Axis' => 'fas fa-globe',
                                        'Grab' => 'fas fa-hand-rock',
                                        'Feature' => 'fas fa-mountain',
                                        'Rail' => 'fas fa-minus'
                                    ];
                                    $category_colors = [
                                        'Rotation' => '#007bff',
                                        'Direction' => '#28a745',
                                        'Axis' => '#dc3545', 
                                        'Grab' => '#fd7e14',
                                        'Feature' => '#6610f2',
                                        'Rail' => '#20c997'
                                    ];
                                    ?>
                                    <?php foreach ($figure_categories as $cat): $items = $figure_items_by_category[$cat] ?? []; ?>
                                        <div class="figure-row p-2 mb-3 bg-dark-subtle" data-category="<?= htmlspecialchars($cat) ?>">
                                            <div class="figure-category-header">
                                                <div class="figure-category-icon" style="background-color: <?= $category_colors[$cat] ?? '#495055ff' ?>;">
                                                    <i class="<?= $category_icons[$cat] ?? 'fas fa-cube' ?>"></i>
                                                </div>
                                                <span><?= htmlspecialchars($cat) ?></span>
                                                <span class="badge bg-light text-dark ms-auto"><?= count($items) ?></span>
                                            </div>
                                            <div class="d-flex flex-wrap gap-4 justify-content-center">
                                                <?php foreach ($items as $it): ?>
                                                    <button type="button" class="btn btn-sm btn-outline-secondary figure-chip figure-badge-<?= htmlspecialchars($it) ?>" 
                                                            data-cat="<?= htmlspecialchars($cat) ?>" 
                                                            data-item="<?= htmlspecialchars($it) ?>"
                                                            title="Select <?= htmlspecialchars($it) ?> in <?= htmlspecialchars($cat) ?>">
                                                        <?= htmlspecialchars($it) ?>
                                                    </button>
                                                <?php endforeach; ?>
                                                <button type="button" class="btn btn-sm btn-outline-danger ms-5 clear-cat" 
                                                        data-cat="<?= htmlspecialchars($cat) ?>"
                                                        title="Clear all <?= htmlspecialchars($cat) ?> selections">
                                                    <i class="fas fa-times"></i>
                                                </button>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                    
                                    <!-- Quick Actions Row -->
                                    <div class="d-flex justify-content-between align-items-center m-1 pt-2 border-top">
                                        <button type="button" class="btn btn-outline-info btn-sm" onclick="showFiguresSummary()">
                                            <i class="fas fa-list me-1"></i>Summary
                                        </button>
                                        <button type="button" class="btn btn-outline-warning btn-sm" onclick="clearAllFigures()">
                                            <i class="fas fa-eraser me-1"></i>Clear All
                                        </button>
                                        <button type="button" class="btn btn-outline-success btn-sm" onclick="copyFromLastRun()">
                                            <i class="fas fa-copy me-1"></i>Copy Last
                                        </button>
                                    </div>
                                </div>

                                <!-- Hidden fields to include with score submission -->
                                <input type="hidden" name="figure_category" id="figure_category_hidden">
                                <input type="hidden" name="figure_item" id="figure_item_hidden">
                                <input type="hidden" name="figures_json" id="figures_json_hidden">
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        
                        </div>
                        <!-- End Additional Cards Wrapper -->
                        
                        <!-- Score Status & Submit Card -->
                        <div class="card score-card m-0 p-3 border-0 bg-transparent" id="submit-button-section">                           
                            <div class="card-body p-0">
                               
                                
                                <!-- Submit Button -->
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-success submit-btn fs-1" id="submitBtn">
                                        <i class="fas fa-save me-2"></i>Submit Score
                                    </button>
                                </div>
                                
                                <!-- Validation Summary -->
                                <div id="validationSummary" class="mt-2" style="display: none;">
                                    <div class="alert alert-sm mb-0" role="alert" style="display: none;">
                                        <div class="small">
                                            <i class="fas fa-info-circle me-1"></i>
                                            <span id="validationSummaryText">Checking diversity rules...</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        </div>
                        <!-- End Scoring Cards Container -->
                                </div>
                    
                        
                        


                    </div>
                    <div class="col-12 col-lg-4">    
                        <!-- Diversity Validation Card -->
                        <div class="card mb-3" id="diversity-validation-panel" style="display: none;">
                            <div class="card-header p-2">
                                <i class="fas fa-check-circle me-2"></i>Validation
                            </div>
                            <div class="card-body p-2">
                                <div id="diversityValidation" class="mt-0 small"></div>
                            </div>
                        </div>           
                        
                        <!-- Latest Scores -->
                        <div class="card score-card mb-3" id="score-history-table">
                            <div class="card-header p-2 d-flex justify-content-between align-items-center bg-dark text-white">
                                <span><i class="fas fa-history me-2"></i>Latest Scores</span>
                                <span class="badge bg-light text-dark"><?= count($latest_scores) ?>/4</span>
                            </div>
                            <div class="card-body p-0">
                                <?php if (count($latest_scores)): ?>
                                    <div class="table-responsive">
                                        <table  id="latestScoresTable" class="table table-sm scores-table mb-0">
                                            <thead class="small">
                                                <tr>
                                                    <th class="small bg-dark-subtle">BIB</th>
                                                    <th class="small bg-dark-subtle">Name</th>
                                                    <th class="small bg-dark-subtle">H/R</th>
                                                    <th class="small bg-dark-subtle">Score</th>
                                                    <th class="small bg-dark-subtle"></th>
                                                    <th class="small bg-dark-subtle"></th>
                                                    <th class="small bg-dark-subtle"></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($latest_scores as $i => $row): ?>
                                                <tr>
                                                    <td clsass="css-bib">
                                                        <span class="badge bg-warning-subtle"><?= htmlspecialchars($row['bib']) ?></span>
                                                    </td>
                                                    <td>
                                                        <small><?= htmlspecialchars($row['first_name'] . ' ' . $row['last_name']) ?></small>
                                                    </td>
                                                    <td>
                                                        <small><?= $row['heat_number'] ?>/<?= $row['run_number'] ?></small>
                                                    </td>
                                                    <td>
                                                        <strong class="text-primary"><?php 
                                                            // Format score to only show decimals when needed
                                                            $scoreValue = $row['score_value'];
                                                            if ($scoreValue !== null && $scoreValue !== '') {
                                                                $numValue = floatval($scoreValue);
                                                                // Check if it's a whole number
                                                                echo htmlspecialchars($numValue == floor($numValue) ? intval($numValue) : $numValue);
                                                            } else {
                                                                echo '-';
                                                            }
                                                        ?></strong>
                                                    </td>
                                                    <td>
                                                        <div class="d-flex flex-wrap gap-1"><?php 
                                                            // Display figures if available
                                                            if (!empty($row['figures_json'])) {
                                                                $figures = json_decode($row['figures_json'], true);
                                                                if ($figures && isset($figures['figures'])) {
                                                                    foreach ($figures['figures'] as $cat => $items) {
                                                                        if (is_array($items) && count($items) > 0) {
                                                                            foreach ($items as $item) {
                                                                                echo '<span class="badge bg-secondary quick-badge figure-badge-' . htmlspecialchars($item) . '" title="' . htmlspecialchars($cat) . '"><strong>';
                                                                                echo htmlspecialchars($item);
                                                                                echo '</strong><small class="ms-1 opacity-75">(' . htmlspecialchars($cat) . ')</small>';
                                                                                echo '</span> ';
                                                                            }
                                                                        }
                                                                    }
                                                                } else {
                                                                    echo '<small class="text-muted">—</small>';
                                                                }
                                                            } else {
                                                                echo '<small class="text-muted">—</small>';
                                                            }
                                                        ?></div>
                                                    </td>
                                                    <td>
                                                        <span class="badge bg-<?= $row['status'] === 'OK' ? 'success' : 'warning' ?> quick-badge">
                                                            <?= htmlspecialchars($row['status']) ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <button type="button" class="btn btn-warning btn-sm" onclick="showOverwriteModal(<?= $i ?>)">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="text-center text-muted p-4">
                                        <i class="fas fa-clipboard-list fa-2x mb-2 opacity-50"></i>
                                        <p class="mb-0">No scores entered yet</p>
                                        <small>Scores will appear here after submission</small>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Reset Scores -->
                        <div class=" text-center mb-3">
                            <div class="">
                                <div class="btn-group bg-light" role="group">
                                    <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#judgeOverviewModal">
                                        <i class="fas fa-chart-line me-1"></i>My Scores Overview
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-info" onclick="openThemeSelector()">
                                        <i class="fas fa-palette me-1"></i>Change Theme
                                    </button>
                                    <!--<button type="button" class="btn btn-info d-none" data-bs-toggle="modal" data-bs-target="#eventSummaryModal">
                                        <i class="fas fa-trophy me-1"></i>Event Results
                                    </button>-->
                                </div>                                
                            </div>
                            <div class="btn-group bg-light mt-2" role="group">                                   
                                    <button type="button" class="btn" onclick="window.location.href='dashboard.php'">
                                        <i class="fas fa-arrow-left me-1"></i>Go to dashboard
                                    </button>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
            
        </div>
    </div>

    <!-- Drawing Board Modal -->
    <div class="modal fade" id="drawingBoardModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-fullscreen">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-palette me-2"></i>
                        <span id="drawingModalTitle">Situation Drawing</span>
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0" style="display: flex; flex-direction: column; height: calc(100vh - 120px);">
                    
                    <!-- Drawing Context Info Bar -->
                    <div class="bg-light border-bottom p-2">
                        <div class="d-flex justify-content-between align-items-center flex-wrap gap-2">
                            <div class="d-flex gap-3 align-items-center">
                                <span class="badge bg-primary" id="drawingEventInfo">Event</span>
                                <span class="badge bg-secondary" id="drawingHeatRunInfo">Heat/Run</span>
                                <span class="badge bg-info" id="drawingParticipantInfo">Participant</span>
                            </div>
                            <div class="d-flex gap-2">
                                <button type="button" class="btn btn-sm btn-outline-secondary" id="loadTemplateBtn" title="Load background template">
                                    <i class="fas fa-folder-open me-1"></i>Template
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-primary" id="saveAsTemplateBtn" title="Save current as template">
                                    <i class="fas fa-save me-1"></i>Save as Template
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Canvas Wrapper -->
                    <div id="canvasWrapper" style="flex: 1; display: flex; justify-content: center; align-items: center; padding: 6px; overflow: hidden;">
                        <canvas id="drawingCanvas" style="background: #ffffff; border: 1px solid #ccc; touch-action: none; max-width: 100%; max-height: 100%;"></canvas>
                    </div>

                    <!-- Drawing Toolbar -->
                    <div class="drawing-toolbar">
                        <!-- Template Management -->
                        <div class="toolbar-group">
                            <span class="toolbar-group-label">Background:</span>
                            <button id="loadLastBgBtn" title="Load last used background">
                                <span class="icon">⏮️</span><span>Last BG</span>
                            </button>
                            <button id="loadTemplateBtn" title="Load saved template">
                                <span class="icon">📁</span><span>Templates</span>
                            </button>
                            <button id="saveAsTemplateBtn" title="Save current background as template">
                                <span class="icon">💾</span><span>Save BG</span>
                            </button>
                        </div>
                        
                        <div class="toolbar-group">
                            <span class="toolbar-group-label">Color:</span>
                            <button class="color-dot active" data-color="#000000" style="background:#000000;" title="Black"></button>
                            <button class="color-dot" data-color="#ffffff" style="background:#ffffff; border: 2px solid #666;" title="White"></button>
                            <button class="color-dot" data-color="#ff0000" style="background:#ff0000;" title="Red"></button>
                            <button class="color-dot" data-color="#00aa00" style="background:#00aa00;" title="Green"></button>
                            <button class="color-dot" data-color="#0000ff" style="background:#0000ff;" title="Blue"></button>
                            <button class="color-dot" data-color="#ffa500" style="background:#ffa500;" title="Orange"></button>
                        </div>

                        <div class="toolbar-group sizes">
                            <span class="toolbar-group-label">Size:</span>
                            <button class="size-btn active" data-size="2" title="Thin">
                                <span class="dot" style="width:4px;height:4px; background:#000; display:inline-block; border-radius:50%;"></span>
                            </button>
                            <button class="size-btn" data-size="5" title="Medium">
                                <span class="dot" style="width:7px;height:7px; background:#000; display:inline-block; border-radius:50%;"></span>
                            </button>
                            <button class="size-btn" data-size="10" title="Thick">
                                <span class="dot" style="width:12px;height:12px; background:#000; display:inline-block; border-radius:50%;"></span>
                            </button>
                        </div>

                        <div class="toolbar-group">
                            <button id="drawModeBtn" class="active" title="Draw mode">
                                <span class="icon">✏️</span><span>Draw</span>
                            </button>
                            <button id="eraseModeBtn" title="Eraser (removes entire lines)">
                                <span class="icon">🧽</span><span>Erase</span>
                            </button>
                            <button id="selectModeBtn" title="Select/edit lines">
                                <span class="icon">🔲</span><span>Select</span>
                            </button>
                        </div>

                        <div class="toolbar-group">
                            <button id="clearBtn" title="Clear all">
                                <span class="icon">🗑️</span><span>Clear</span>
                            </button>
                            <button id="undoBtn" title="Undo last stroke" disabled>
                                <span class="icon">↶</span><span>Undo</span>
                            </button>
                        </div>

                        <div class="toolbar-group">
                            <span class="toolbar-group-label">Layers:</span>
                            <button id="layerFgBtn" class="active" title="Edit layer: drawing">
                                <span class="icon">🎨</span><span>Drawing</span>
                            </button>
                            <button id="layerBgBtn" title="Edit layer: background">
                                <span class="icon">📐</span><span>Background</span>
                            </button>
                            <label class="layer-lock-label">
                                <input type="checkbox" id="bgLockCheckbox" checked>
                                <span>Lock BG</span>
                            </label>
                            <label class="layer-opacity-label">
                                <span>BG Opacity</span>
                                <input type="range" id="bgOpacityRange" min="10" max="100" value="30">
                            </label>
                        </div>

                        <div class="toolbar-group">
                            <span class="toolbar-group-label">Selection:</span>
                            <button id="scaleUpBtn" title="Scale up selection">
                                <span class="icon">＋</span>
                            </button>
                            <button id="scaleDownBtn" title="Scale down selection">
                                <span class="icon">－</span>
                            </button>
                            <button id="rotateLeftBtn" title="Rotate left">
                                <span class="icon">⟲</span>
                            </button>
                            <button id="rotateRightBtn" title="Rotate right">
                                <span class="icon">⟳</span>
                            </button>
                        </div>
                    </div>

                    <!-- Optional Notes Section -->
                    <div class="border-top p-2 bg-light">
                        <label class="form-label small mb-1">Notes (optional):</label>
                        <textarea id="drawingNotes" class="form-control form-control-sm" rows="2" placeholder="Add notes about this situation..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-success" id="saveDrawingBtn">
                        <i class="fas fa-save me-1"></i>Save Drawing
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Overwrite Score Modal -->
    <div class="modal fade" id="overwriteScoreModal" tabindex="-1" style="z-index: 1060;">
        <div class="modal-dialog modal-lg">
            <form method="post" id="overwriteScoreForm">
                <div class="modal-content">
                    <div class="modal-header bg-dark text-white">
                        <h5 class="modal-title">
                            <i class="fas fa-edit me-2"></i>Edit Score
                            <span id="editScoreParticipant" class=""></span>
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body bg-dark-subtle">
                        <input type="hidden" name="overwrite_score_id" id="overwrite_score_id">
                        <input type="hidden" name="ajax" value="1">
                        <?php if ($manual_mode): ?>
                            <input type="hidden" name="event_id" value="<?= $event_id ?>">
                        <?php endif; ?>
                        
                        <!-- Score Value -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="overwrite_score_value" class="form-label fw-bold">Score (0-99)</label>
                                <input type="number" class="form-control score-input" 
                                    id="overwrite_score_value" name="overwrite_score_value" 
                                    min="0" max="99" step="1" required>
                            </div>
                            <div class="col-md-6">
                                <label for="overwrite_status" class="form-label fw-bold">Status</label>
                                <select class="form-select" id="overwrite_status" name="overwrite_status" required>
                                    <option value="OK">OK - Valid Score</option>
                                    <option value="DNS">DNS - Did Not Start</option>
                                    <option value="DNF">DNF - Did Not Finish</option>
                                    <option value="DSQ">DSQ - Disqualified</option>
                                </select>
                            </div>
                        </div>
                        
                        <!-- Figure Selection (shown only if figures are available) -->
                        <div id="editFigureSelection" style="display: none;">
                            <hr>
                            <h6><i class="fas fa-shapes me-2"></i>Figure Selection</h6>
                            <input type="hidden" name="overwrite_figures_json" id="overwrite_figures_json">
                            
                            <!-- Figure Categories Grid -->
                            <div id="editFigureGrid" class="mb-3">
                                <!-- Dynamically populated -->
                            </div>
                            
                            <!-- Current Figure Summary -->
                            <div id="editFigureSummary" class="alert alert-info" style="display: none;">
                                <strong>Selected Figures:</strong>
                                <div id="editFigureSummaryContent"></div>
                            </div>
                            
                            <!-- Figure Validation -->
                            <div id="editDiversityValidation" class="mt-2" style="display: none;">
                                <!-- Validation messages will be inserted here -->
                            </div>
                        </div>
                        
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Warning:</strong> This will permanently update the score. Make sure the values are correct.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="submit" class="btn btn-warning">
                            <i class="fas fa-save me-2"></i>Update Score
                        </button>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Generic Confirmation Modal -->
    <div class="modal fade" id="confirmModal" tabindex="-1" aria-labelledby="confirmModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="confirmModalLabel">
                        <i class="fas fa-question-circle me-2"></i>Confirm Action
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="confirmModalMessage">
                    Are you sure you want to proceed?
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="confirmModalBtn">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Judge Overview Modal -->
    <div class="modal fade" id="judgeOverviewModal" tabindex="-1" aria-labelledby="judgeOverviewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header bg-dark text-white">
                    <h5 class="modal-title" id="judgeOverviewModalLabel">
                        <i class="fas fa-chart-line me-2"></i>My Scores Overview - <?= htmlspecialchars($judge_name) ?>
                    </h5>
                    <button type="button" class="btn-close text-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-1">
                    <div id="judgeOverviewContent">
                        <div class="text-center">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <div class="mt-2">Loading your scores...</div>
                        </div>
                    </div>

                </div>
                <div class="modal-footer bg-dark-subtle">
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="refreshJudgeOverview()">
                        <i class="fas fa-sync-alt me-1"></i>Refresh
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Format Information Modal -->
    <?php if ($format_info): ?>
    <div class="modal fade" id="formatInfoModal" tabindex="-1" aria-labelledby="formatInfoModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-info">
                    <h5 class="modal-title" id="formatInfoModalLabel">
                        <?php if ($format_icon_url): ?>
                            <img src="<?= htmlspecialchars($format_icon_url) ?>" alt="format icon" style="width:28px;height:28px;object-fit:contain;margin-right:10px;" onerror="this.style.display='none'">
                        <?php else: ?>
                            <i class="fas fa-puzzle-piece me-2"></i>
                        <?php endif; ?>
                        Scoring Format Information
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <!-- Format Header -->
                    <div class="mb-4">
                        <h4><?= htmlspecialchars($format_info['format_name'] ?: $format_info['scoring_format']) ?></h4>
                        <div class="d-flex flex-wrap gap-2 mt-2">
                            <span class="badge bg-secondary"><?= htmlspecialchars($format_info['sport'] ?: '—') ?></span>
                            <?php if (!empty($format_info['mode'])): ?>
                                <span class="badge bg-primary"><?= htmlspecialchars(str_replace('_',' ', $format_info['mode'])) ?></span>
                            <?php endif; ?>
                            <?php if ($uses_figures): ?>
                                <span class="badge bg-success"><i class="fas fa-shapes me-1"></i>Figures Required</span>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Quick Stats -->
                    <div class="row g-2 mb-4">
                        <?php if (isset($format_info['min_judges'])): ?>
                        <div class="col-md-4">
                            <div class="border rounded p-3 text-center">
                                <div class="text-muted small">Judges</div>
                                <div class="fs-5 fw-bold"><?= htmlspecialchars($format_info['min_judges']) ?><?= $format_info['max_judges'] ? '–'.htmlspecialchars($format_info['max_judges']) : '' ?></div>
                            </div>
                        </div>
                        <?php endif; ?>
                        <?php if ($format_info['scale_min'] !== null): ?>
                        <div class="col-md-4">
                            <div class="border rounded p-3 text-center">
                                <div class="text-muted small">Score Scale</div>
                                <div class="fs-5 fw-bold"><?= htmlspecialchars($format_info['scale_min']) ?>–<?= htmlspecialchars($format_info['scale_max']) ?></div>
                                <?php if ($format_info['precision_decimal'] ?? 0): ?>
                                    <div class="small text-muted">Decimals allowed</div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                        <div class="col-md-4">
                            <div class="border rounded p-3 text-center">
                                <div class="text-muted small">Criteria / Results</div>
                                <div class="fs-5 fw-bold"><?= (int)($format_info['criteria_count'] ?? 0) ?> / <?= (int)($format_info['results_count'] ?? 0) ?></div>
                            </div>
                        </div>
                        <?php if (!empty($format_info['drop_rule']) && strtolower($format_info['drop_rule']) !== 'none'): ?>
                        <div class="col-12">
                            <div class="alert alert-warning mb-0">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                <strong>Drop Rule:</strong> <?= htmlspecialchars($format_info['drop_rule']) ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Detailed Sections -->
                    <div class="row g-3">
                        <!-- Tiebreak Rules -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-primary">
                                    <span class="me-2">⚖️</span>
                                    <strong>Tiebreak Rules</strong>
                                </div>
                                <div class="card-body">
                                    <?php 
                                    try {
                                        $tiebreakStmt = $pdo->prepare("SELECT tiebreak_rule FROM scoring_format_tiebreak WHERE format_id = ? ORDER BY sort_order");
                                        $tiebreakStmt->execute([$format_info['scoring_format']]);
                                        $tiebreakRules = $tiebreakStmt->fetchAll(PDO::FETCH_COLUMN);
                                        if (!empty($tiebreakRules)): ?>
                                            <ol class="mb-0">
                                                <?php foreach ($tiebreakRules as $rule): ?>
                                                    <li><?= htmlspecialchars($rule) ?></li>
                                                <?php endforeach; ?>
                                            </ol>
                                        <?php else: ?>
                                            <em class="text-muted">Standard highest score wins</em>
                                        <?php endif;
                                    } catch (Exception $e) { 
                                        echo '<em class="text-muted">Standard highest score wins</em>';
                                    } ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Diversity Rules -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-success text-white">
                                    <span class="me-2">🎯</span>
                                    <strong>Diversity Rules</strong>
                                </div>
                                <div class="card-body">
                                    <?php if (!empty($diversity_rules)): ?>
                                        <ul class="mb-0">
                                            <?php foreach ($diversity_rules as $rule): ?>
                                                <li>
                                                    <strong><?= htmlspecialchars($rule['rule_type']) ?></strong>
                                                    <?php if ($rule['group_name']): ?>
                                                        in <em><?= htmlspecialchars($rule['group_name']) ?></em>
                                                    <?php endif; ?>
                                                    <?php if ($rule['min_distinct']): ?>
                                                        (min: <?= (int)$rule['min_distinct'] ?>)
                                                    <?php endif; ?>
                                                    <?php if ($rule['scope']): ?>
                                                        <span class="badge bg-light text-dark ms-1"><?= htmlspecialchars($rule['scope']) ?></span>
                                                    <?php endif; ?>
                                                    <?php if ($rule['is_optional']): ?>
                                                        <span class="badge bg-warning text-dark ms-1">optional</span>
                                                    <?php endif; ?>
                                                </li>
                                            <?php endforeach; ?>
                                        </ul>
                                    <?php else: ?>
                                        <em class="text-muted">No diversity restrictions</em>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Rounds Configuration -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-info text-white">
                                    <span class="me-2">🔄</span>
                                    <strong>Rounds Configuration</strong>
                                </div>
                                <div class="card-body">
                                    <?php 
                                    try {
                                        $roundsStmt = $pdo->prepare("SELECT round_name, runs, aggregation FROM scoring_format_rounds WHERE format_id = ? ORDER BY sort_order");
                                        $roundsStmt->execute([$format_info['scoring_format']]);
                                        $rounds = $roundsStmt->fetchAll(PDO::FETCH_ASSOC);
                                        if (!empty($rounds)): ?>
                                            <ul class="mb-0">
                                                <?php foreach ($rounds as $round): ?>
                                                    <li>
                                                        <strong><?= htmlspecialchars($round['round_name']) ?></strong>
                                                        <?php if ($round['runs']): ?>
                                                            (<?= (int)$round['runs'] ?> runs)
                                                        <?php endif; ?>
                                                        <?php if ($round['aggregation']): ?>
                                                            <span class="badge bg-light text-dark ms-1"><?= htmlspecialchars($round['aggregation']) ?></span>
                                                        <?php endif; ?>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php else: ?>
                                            <em class="text-muted">Single round format</em>
                                        <?php endif;
                                    } catch (Exception $e) { 
                                        echo '<em class="text-muted">Single round format</em>';
                                    } ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Mode Configuration -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-warning">
                                    <span class="me-2">⚙️</span>
                                    <strong>Mode Configuration</strong>
                                </div>
                                <div class="card-body">
                                    <?php 
                                    try {
                                        $modeStmt = $pdo->prepare("SELECT config_type, config_key, config_value FROM scoring_format_mode_config WHERE format_id = ? ORDER BY config_type, config_key");
                                        $modeStmt->execute([$format_info['scoring_format']]);
                                        $modeConfigs = $modeStmt->fetchAll(PDO::FETCH_ASSOC);
                                        if (!empty($modeConfigs)): ?>
                                            <ul class="mb-0">
                                                <?php 
                                                $groupedConfigs = [];
                                                foreach ($modeConfigs as $config) {
                                                    $groupedConfigs[$config['config_type']][$config['config_key']] = $config['config_value'];
                                                }
                                                foreach ($groupedConfigs as $type => $configs): ?>
                                                    <li>
                                                        <strong><?= htmlspecialchars(ucfirst($type)) ?></strong>: 
                                                        <?php foreach ($configs as $key => $value): ?>
                                                            <?= htmlspecialchars($key) ?>=<?= htmlspecialchars($value) ?>
                                                            <?php if ($key !== array_key_last($configs)): ?>, <?php endif; ?>
                                                        <?php endforeach; ?>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php else: ?>
                                            <em class="text-muted">Default mode settings</em>
                                        <?php endif;
                                    } catch (Exception $e) { 
                                        echo '<em class="text-muted">Default mode settings</em>';
                                    } ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Event Summary Modal -->
    <div class="modal fade" id="eventSummaryModal" tabindex="-1" aria-labelledby="eventSummaryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="eventSummaryModalLabel">
                        <i class="fas fa-trophy me-2"></i>Event Results Summary
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <div class="btn-group w-100" role="group" aria-label="Summary type">
                                <input type="radio" class="btn-check" name="summaryType" id="summaryFinal" value="final" checked>
                                <label class="btn btn-outline-primary" for="summaryFinal">Final Results</label>
                                
                                <input type="radio" class="btn-check" name="summaryType" id="summaryByHeat" value="by_heat">
                                <label class="btn btn-outline-primary" for="summaryByHeat">By Heat</label>
                                
                                <input type="radio" class="btn-check" name="summaryType" id="summaryByRun" value="by_run">
                                <label class="btn btn-outline-primary" for="summaryByRun">By Run</label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="btn-group w-100" role="group" aria-label="Display options">
                                <input type="radio" class="btn-check" name="displayFormat" id="displayTable" value="table" checked>
                                <label class="btn btn-outline-secondary" for="displayTable">
                                    <i class="fas fa-table me-1"></i>Table
                                </label>
                                
                                <input type="radio" class="btn-check" name="displayFormat" id="displayGrid" value="grid">
                                <label class="btn btn-outline-secondary" for="displayGrid">
                                    <i class="fas fa-th me-1"></i>Grid
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <div id="eventSummaryContent">
                        <div class="text-center">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <div class="mt-2">Loading event results...</div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="refreshEventSummary()">
                        <i class="fas fa-sync-alt me-1"></i>Refresh
                    </button>
                    <button type="button" class="btn btn-outline-success btn-sm" onclick="exportSummary()">
                        <i class="fas fa-download me-1"></i>Export
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Change panel theme -->
    <div class="modal fade" id="themeSelectorModal" tabindex="-1" aria-labelledby="themeSelectorModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="themeSelectorModalLabel">
                        <i class="fas fa-palette me-2"></i>Change Panel Theme
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body bg-dark-subtle">
                    <p class="text-muted mb-3">Select a visual theme for the scoring panel:</p>
                    
                    <div class="row g-3">
                        <!-- Brite Theme -->
                        <div class="col-6">
                            <div class="card theme-option-card" onclick="selectTheme('brite')" style="cursor: pointer;">
                                <div class="card-body text-center p-4" style="background: linear-gradient(135deg, #66ea71ff 0%, #15ae27ff 100%); min-height: 120px;">
                                    <i class="fas fa-sun text-white" style="font-size: 2rem;"></i>
                                    <h6 class="text-white mt-2 mb-0">Brite</h6>
                                    
                                </div>
                                <div class="card-footer text-center">
                                    <div class="form-check form-check-inline">
                                        <input class="form-check-input" type="radio" name="theme_selector" id="theme_brite" value="brite">
                                        <label class="form-check-label" for="theme_brite">Select</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Zephyr Theme -->
                        <div class="col-6">
                            <div class="card theme-option-card" onclick="selectTheme('zephyr')" style="cursor: pointer;">
                                <div class="card-body text-center p-4" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); min-height: 120px;">
                                    <i class="fas fa-wind text-white" style="font-size: 2rem;"></i>
                                    <h6 class="text-white mt-2 mb-0">Zephyr</h6>                                   
                                </div>
                                <div class="card-footer text-center">
                                    <div class="form-check form-check-inline">
                                        <input class="form-check-input" type="radio" name="theme_selector" id="theme_zephyr" value="zephyr">
                                        <label class="form-check-label" for="theme_zephyr">Select</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Zoom Control -->
                    <div class="mt-4">
                        <h6 class="mb-3">
                            <i class="fas fa-search-plus me-2"></i>Interface Zoom
                        </h6>
                        <div class="row align-items-center">
                            <div class="col-12">
                                <label for="zoomRange" class="form-label">
                                    Zoom Level: <span id="zoomValue" class="badge bg-primary">100%</span>
                                </label>
                                <input type="range" class="form-range" id="zoomRange" min="70" max="130" value="100" step="5" oninput="updateZoomDisplay(this.value)">
                                <div class="d-flex justify-content-between">
                                    <small class="text-muted">70%</small>
                                    <small class="text-muted">100%</small>
                                    <small class="text-muted">130%</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-info mt-3 mb-0">
                        <i class="fas fa-info-circle me-1"></i>
                        Your theme and zoom preferences will be saved in your browser.
                    </div>
                </div>
                <div class="modal-footer d-flex justify-content-between">
                    <button type="button" class="btn btn-outline-secondary" onclick="resetZoom()">
                        <i class="fas fa-undo me-1"></i>Reset Zoom
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="applyThemeAndZoom()">
                        <i class="fas fa-check me-1"></i>Apply
                    </button>
                </div>
            </div>
        </div>
    </div>

        <!-- Debug Toggle Button -->
        <div class="text-center mb-2">
            
            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="toggleDebugSection()">
                <i class="fas fa-bug me-1"></i>Toggle Debug Info
            </button>
        </div>
    <!-- Your existing JavaScript functions with mobile enhancements -->
    <script>
    // Current judge context for client-side logic
    const CURRENT_JUDGE_ID = <?= json_encode($judge_id) ?>;
    const CURRENT_JUDGE_NAME = <?= json_encode($judge_name) ?>;
    // Heat -> runs_count mapping (fallback to event runs_per_heat if missing)
    const HEAT_RUNS_MAP = <?= json_encode(array_column($heats, 'runs_count', 'heat_number')) ?>;
    const DEFAULT_RUNS_PER_HEAT = <?= json_encode($runs_per_heat) ?>;

    // Keep track of latest scores - declare globally
    let latestScores = <?= json_encode($latest_scores) ?>;

    // Embed figure data (if any) for client-side selection
    const FIGURE_DATA = <?= json_encode([
        'available' => (bool)$uses_figures,
        'categories' => $figure_categories,
        'itemsByCategory' => $figure_items_by_category,
    ]) ?>;
    // Embed diversity and attempts/tricks config for validation
    const FIGURE_UI = <?= json_encode([
        'diversityRules' => $diversity_rules,
        'attempts' => $mode_attempts,
        'tricks' => $mode_tricks,
        'mode' => $format_info['mode'] ?? null,
    ]) ?>;
    
    // Debug figure data
    console.log('FIGURE_DATA:', FIGURE_DATA);
    console.log('FIGURE_UI:', FIGURE_UI);

    // Auto refresh for "Scores for this participant"
    let otherScoresInterval = null;
    
    // Store figure selections across runs for diversity validation
    let figureHistoryByRun = {}; // { run_number: { category: Set(items) } }
    let currentParticipantId = null;
    
    function startOtherScoresInterval(periodMs = 10000) {
        stopOtherScoresInterval();
        otherScoresInterval = setInterval(() => {
            if (document.hidden) return; // pause when tab is hidden
            const ep = document.getElementById('ep_id')?.value;
            const heat = document.querySelector('[name=heat_number]')?.value;
            if (ep && heat) loadOtherScoresForParticipant();
        }, periodMs);
    }
    function stopOtherScoresInterval() {
        if (otherScoresInterval) {
            clearInterval(otherScoresInterval);
            otherScoresInterval = null;
        }
    }
    window.addEventListener('beforeunload', stopOtherScoresInterval);
    document.addEventListener('visibilitychange', () => {
        // Optional: restart on focus to refresh immediately
        if (!document.hidden && otherScoresInterval) {
            loadOtherScoresForParticipant();
        }
    });

    // NOTE: loadParticipants() function is defined later in the code (enhanced version with BIB auto-population)

    function saveHeatRunSelection() {
        const eventId = <?= json_encode($event_id) ?>;
        const heat = document.querySelector('[name=heat_number]').value;
        const run = document.querySelector('[name=run_number]:checked')?.value;
        if (eventId && heat && run) {
            localStorage.setItem('event_' + eventId + '_heat_run', JSON.stringify({heat, run}));
            const heatRunBadge = document.getElementById('heatRunBadge');
            if (heatRunBadge) {
                heatRunBadge.style.display = 'inline-block';
            }
        }
    }

    function updateScoreInTable(scoreIndex, updatedScore) {
        // Enhanced validation
        if (typeof scoreIndex !== 'number' || isNaN(scoreIndex) || scoreIndex < 0 || !updatedScore) {
            console.error('Invalid parameters for updateScoreInTable:', 
                         'scoreIndex:', scoreIndex, 
                         'type:', typeof scoreIndex, 
                         'isNaN:', isNaN(scoreIndex),
                         'updatedScore:', updatedScore);
            return false;
        }
        
        // Check if latestScores array exists and has the index
        if (!latestScores || !Array.isArray(latestScores) || scoreIndex >= latestScores.length) {
            console.error('Score index out of bounds:', scoreIndex, 'Array length:', latestScores?.length);
            return false;
        }
        
        // Update the latestScores array
        latestScores[scoreIndex] = updatedScore;
        
        // Get the table rows
        const rows = document.querySelectorAll('#latestScoresTable tbody tr');
        if (!rows || scoreIndex >= rows.length) {
            console.error('Table row not found for index:', scoreIndex, 'Available rows:', rows.length);
            return false;
        }
        
        const row = rows[scoreIndex];
        
        try {
            // Update BIB - with null safety
            row.cells[0].innerHTML = `<span class="badge bg-secondary">${updatedScore.bib || 'N/A'}</span>`;
            
            // Update Name - with null safety
            const firstName = updatedScore.first_name || '';
            const lastName = updatedScore.last_name || '';
            row.cells[1].innerHTML = `<small>${firstName} ${lastName}</small>`;
            
            // Update H/R - with null safety
            const heatNumber = updatedScore.heat_number || '?';
            const runNumber = updatedScore.run_number || '?';
            row.cells[2].innerHTML = `<small>${heatNumber}/${runNumber}</small>`;
            
            // Update Score - handle null/empty values and format decimals
            let scoreValue = '-';
            if (updatedScore.score_value !== null && updatedScore.score_value !== undefined) {
                // Show decimals only if the value has them
                const numValue = parseFloat(updatedScore.score_value);
                scoreValue = numValue % 1 === 0 ? Math.floor(numValue) : numValue;
            }
            row.cells[3].innerHTML = `<strong class="text-primary">${scoreValue}</strong>`;
            
            // Update Figures - format figures display
            let figuresDisplay = '—';
            if (updatedScore.figures_json) {
                try {
                    const figures = typeof updatedScore.figures_json === 'string' ? JSON.parse(updatedScore.figures_json) : updatedScore.figures_json;
                    if (figures && figures.figures) {
                        const figureList = Object.entries(figures.figures)
                            .filter(([cat, items]) => Array.isArray(items) && items.length > 0)
                            .map(([cat, items]) => cat + ': ' + items.join(', '))
                            .join(' | ');
                        figuresDisplay = figureList || '—';
                    }
                } catch (e) {
                    figuresDisplay = '—';
                }
            }
            row.cells[4].innerHTML = `<small class="text-muted">${figuresDisplay}</small>`;
            
            // Update Status - with better badge logic
            let statusBadge = 'secondary';
            switch(updatedScore.status) {
                case 'OK':
                    statusBadge = 'success';
                    break;
                case 'DNS':
                case 'DNF':
                    statusBadge = 'warning';
                    break;
                case 'DSQ':
                    statusBadge = 'danger';
                    break;
                default:
                    statusBadge = 'secondary';
            }
            row.cells[5].innerHTML = `<span class="badge bg-${statusBadge} quick-badge">${updatedScore.status || 'Unknown'}</span>`;
            
            // Update the edit button to maintain the correct index
            row.cells[6].innerHTML = `<button type="button" class="btn btn-warning btn-sm" onclick="showOverwriteModal(${scoreIndex})"><i class="fas fa-edit"></i></button>`;
            
            // Initialize Lucide icons for the updated row
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons(row);
            }
            
            // Add enhanced flash effect with animation
            row.classList.add('table-success');
            row.style.backgroundColor = '#d1e7dd';
            row.style.transition = 'background-color 0.3s ease';
            
            // Remove flash effect after animation
            setTimeout(() => {
                row.style.backgroundColor = '';
                setTimeout(() => {
                    row.classList.remove('table-success');
                    row.style.transition = '';
                }, 300);
            }, 1500);
            
            console.log('Successfully updated table row at index:', scoreIndex);
            return true;
            
        } catch (error) {
            console.error('Error updating table row:', error);
            return false;
        }
    }

    function addNewScoreToTable(newScore) {
        // Add to the beginning of latestScores array
        latestScores.unshift(newScore);
        
        // Keep only 4 latest scores
        if (latestScores.length > 4) {
            latestScores = latestScores.slice(0, 4);
        }
        
        // Check if table exists
        const tableElement = document.querySelector('#latestScoresTable');
        if (!tableElement) {
            console.warn('Latest scores table not found in DOM');
            return;
        }
        
        // Get the card body container
        const cardBody = tableElement.closest('.card-body');
        if (!cardBody) {
            console.warn('Card body not found');
            return;
        }
        
        // Show table and hide empty state
        const tableContainer = cardBody.querySelector('.table-responsive');
        const emptyState = cardBody.querySelector('.text-center.text-muted');
        
        if (tableContainer && emptyState) {
            tableContainer.style.display = 'block';
            emptyState.style.display = 'none';
        }
        
        // Rebuild the table
        const tbody = document.querySelector('#latestScoresTable tbody');
        if (!tbody) {
            console.warn('Latest scores table tbody not found');
            return;
        }
        
        tbody.innerHTML = '';
        
        latestScores.forEach((score, index) => {
            const row = tbody.insertRow();
            // Format score value to show decimals only if needed
            const scoreDisplay = score.score_value !== null && score.score_value !== undefined ? 
                (parseFloat(score.score_value) % 1 === 0 ? Math.floor(parseFloat(score.score_value)) : parseFloat(score.score_value)) : '-';
            
            // Format figures display
            let figuresDisplay = '—';
            if (score.figures_json) {
                try {
                    const figures = typeof score.figures_json === 'string' ? JSON.parse(score.figures_json) : score.figures_json;
                    if (figures && figures.figures) {
                        const figureList = Object.entries(figures.figures)
                            .filter(([cat, items]) => Array.isArray(items) && items.length > 0)
                            .map(([cat, items]) => cat + ': ' + items.join(', '))
                            .join(' | ');
                        figuresDisplay = figureList || '—';
                    }
                } catch (e) {
                    figuresDisplay = '—';
                }
            }
            
            row.innerHTML = `
                <td><span class="badge bg-secondary">${score.bib}</span></td>
                <td><small>${score.first_name} ${score.last_name}</small></td>
                <td><small>${score.heat_number}/${score.run_number}</small></td>
                <td><strong class="text-success-emphasis">${scoreDisplay}</strong></td>
                <td><small class="text-muted">${figuresDisplay}</small></td>
                <td><span class="badge bg-${score.status === 'OK' ? 'success' : 'warning'} quick-badge">${score.status}</span></td>
                <td><button type="button" class="btn btn-warning btn-sm" onclick="showOverwriteModal(${index})"><i class="fas fa-edit"></i></button></td>
            `;
            
            // Add flash effect to new row
            if (index === 0) {
                row.style.backgroundColor = '#d1e7dd';
                setTimeout(() => {
                    row.style.transition = 'background-color 1s ease';
                    row.style.backgroundColor = '';
                }, 100);
                setTimeout(() => {
                    row.style.transition = '';
                }, 1100);
            }
        });
        
        // Initialize Lucide icons for the rebuilt table
        if (window.initializeLucideIcons) {
            window.initializeLucideIcons(tbody);
        }
        
        // Update the count badge in the header
        const scoreCard = document.querySelector('#score-history-table');
        if (scoreCard) {
            const countBadge = scoreCard.querySelector('.card-header .badge');
            if (countBadge) {
                countBadge.textContent = latestScores.length + '/4';
            }
        }
        
        console.log('Latest scores table updated. Total scores:', latestScores.length);
    }

    // Criteria scoring functions
    function updateCriteriaRange(inputElement, index) {
        const rangeElement = document.querySelector(`[name="criteria_scores[${index}][range]"]`);
        if (rangeElement) {
            rangeElement.value = Math.round(inputElement.value);
            inputElement.value = Math.round(inputElement.value);
        }
        calculateFinalScore(true); // Update main score when manually adjusting criteria
    }

    function updateCriteriaInput(rangeElement, index) {
        const inputElement = document.querySelector(`[name="criteria_scores[${index}][value]"]`);
        if (inputElement) {
            const roundedValue = Math.round(rangeElement.value);
            inputElement.value = roundedValue;
            rangeElement.value = roundedValue;
        }
        calculateFinalScore(true); // Update main score when manually adjusting criteria
    }

    function calculateFinalScore(updateMainScore = true) {
        const criteriaInputs = document.querySelectorAll('[name^="criteria_scores"][name$="[value]"]');
        let total = 0;
        let count = 0;
        
        criteriaInputs.forEach(input => {
            const value = parseFloat(input.value) || 0;
            if (value > 0) {
                total += value;
                count++;
            }
        });
        
        const average = count > 0 ? Math.round(total / count) : 0;
        const calculatedScoreElement = document.getElementById('calculatedScore');
        if (calculatedScoreElement) {
            calculatedScoreElement.textContent = average;
        }
        
        // Only update the main score field if explicitly requested (to avoid loops)
        if (updateMainScore) {
            const mainScoreInput = document.querySelector('[name="score_value"]');
            if (mainScoreInput && count > 0) {
                mainScoreInput.value = average;
                // Update the range slider too
                const mainScoreRange = document.querySelector('[name="score_range"]');
                if (mainScoreRange) {
                    mainScoreRange.value = average;
                }
                
                // Update current participant score display
                updateCurrentParticipantScore();
                
                // Add subtle animation to show the change
                mainScoreInput.style.transition = 'background-color 0.1s ease';
                mainScoreInput.style.backgroundColor = '#dcf900a3';
                setTimeout(() => {
                    mainScoreInput.style.backgroundColor = '';
                    setTimeout(() => {
                        mainScoreInput.style.transition = '';
                    }, 300);
                }, 300);
            }
        }
        
        return average;
    }

    // Handle main score changes and adjust criteria proportionally
    function updateMainScoreAndCriteria(element, updateType) {
        const mainScoreInput = document.querySelector('[name="score_value"]');
        const mainScoreRange = document.querySelector('[name="score_range"]');
        
        // Update the corresponding field
        if (updateType === 'range') {
            // Input changed, update range
            mainScoreRange.value = element.value;
        } else {
            // Range changed, update input
            mainScoreInput.value = element.value;
        }
        
        const newMainScore = parseFloat(element.value) || 0;
        
        // Update current participant score display
        updateCurrentParticipantScore();
        
        // Enable status and submit buttons when score is entered
        if (newMainScore > 0) {
            enableStatusAndSubmit();
        }
        
        // Check if we have criteria inputs to adjust
        const criteriaInputs = document.querySelectorAll('[name^="criteria_scores"][name$="[value]"]');
        if (criteriaInputs.length === 0) return;
        
        // Calculate current criteria average to determine ratio
        let currentCriteriaTotal = 0;
        let criteriaCount = 0;
        let hasExistingValues = false;
        
        criteriaInputs.forEach(input => {
            const value = parseFloat(input.value) || 0;
            if (value > 0) {
                currentCriteriaTotal += value;
                criteriaCount++;
                hasExistingValues = true;
            }
        });
        
        // Only adjust if there are existing criteria values and a new main score
        if (hasExistingValues && newMainScore > 0 && criteriaCount > 0) {
            const currentCriteriaAverage = currentCriteriaTotal / criteriaCount;
            
            // Avoid division by zero
            if (currentCriteriaAverage > 0) {
                const ratio = newMainScore / currentCriteriaAverage;
                
                // Apply ratio to all criteria inputs
                criteriaInputs.forEach((input, index) => {
                    const currentValue = parseFloat(input.value) || 0;
                    if (currentValue > 0) {
                        const newValue = Math.min(Math.round(currentValue * ratio), parseFloat(input.max) || 99);
                        input.value = newValue;
                        
                        // Update corresponding range slider
                        const rangeElement = document.querySelector(`[name="criteria_scores[${index}][range]"]`);
                        if (rangeElement) {
                            rangeElement.value = newValue;
                        }
                        
                        // Add subtle animation to show the change
                        input.style.transition = 'background-color 0.1s ease';
                        input.style.backgroundColor = '#dcf900a3';
                        setTimeout(() => {
                            input.style.backgroundColor = '';
                            setTimeout(() => {
                                input.style.transition = '';
                            }, 300);
                        }, 300);
                    }
                });
                
                // Recalculate final score to show updated average (without updating main score)
                calculateFinalScore(false);
            }
        }
    }

    // Numeric keypad functions for score_value input
    function addToScoreValue(digit) {
        const scoreInput = document.getElementById('score_value');
        if (!scoreInput) return;
        
        const currentValue = scoreInput.value || '';
        const newValue = currentValue + digit;
        const numValue = parseInt(newValue);
        
        // Enforce max limit of 99
        if (numValue <= 99) {
            scoreInput.value = newValue;
            // Update current participant score display
            updateCurrentParticipantScore();
            // Trigger the oninput event to update range and criteria
            scoreInput.dispatchEvent(new Event('input', { bubbles: true }));
        }
    }

    function backspaceScoreValue() {
        const scoreInput = document.getElementById('score_value');
        if (!scoreInput) return;
        
        const currentValue = scoreInput.value || '';
        scoreInput.value = currentValue.slice(0, -1);
        // Update current participant score display
        updateCurrentParticipantScore();
        // Trigger the oninput event to update range and criteria
        scoreInput.dispatchEvent(new Event('input', { bubbles: true }));
    }

    function applyScoreValue() {
        const scoreInput = document.getElementById('score_value');
        if (!scoreInput) return;
        
        // Validate the score is within range
        const value = parseInt(scoreInput.value) || 0;
        if (value < 0) {
            scoreInput.value = 0;
            console.warn('Score cannot be negative. Resetting to 0.');
        } else if (value > 99) {
            scoreInput.value = 99;
            console.warn('Score cannot be greater than 99. Resetting to 99.');
        }
        
        // Update current participant score display
        updateCurrentParticipantScore();
        
        // Trigger the oninput event to finalize
        scoreInput.dispatchEvent(new Event('input', { bubbles: true }));
        
        // Enable status buttons and submit button when score is entered
        enableStatusAndSubmit();
        
        // Visual feedback
        scoreInput.style.transition = 'all 0.2s ease';
        scoreInput.style.backgroundColor = '#198754';
        scoreInput.style.color = 'white';
        setTimeout(() => {
            scoreInput.style.backgroundColor = '';
            scoreInput.style.color = '';
        }, 300);
        
        // Scroll to submit button
        const submitBtn = document.getElementById('submitBtn');
        if (submitBtn) {
            submitBtn.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'center' 
            });
            
            // Add pulse animation to draw attention
            submitBtn.style.transition = 'transform 0.3s ease';
            submitBtn.style.transform = 'scale(1.05)';
            setTimeout(() => {
                submitBtn.style.transform = 'scale(1)';
            }, 300);
        }
    }

    function enableStatusAndSubmit() {
        const statusButtons = document.querySelector('.score-status-buttons');
        const submitBtn = document.getElementById('submitBtn');
        
        if (statusButtons) {
            statusButtons.classList.remove('disabled');
        }
        if (submitBtn) {
            submitBtn.classList.remove('disabled');
            submitBtn.disabled = false;
        }
    }    // BIB numeric keypad functions
    function addToBibInput(digit) {
        const bibInput = document.getElementById('bibInput');
        if (!bibInput) return;
        
        const currentValue = bibInput.value || '';
        const newValue = currentValue + digit;
        
        bibInput.value = newValue;
        // Trigger the input event to update participant selection
        bibInput.dispatchEvent(new Event('input', { bubbles: true }));
    }

    function backspaceBibInput() {
        const bibInput = document.getElementById('bibInput');
        if (!bibInput) return;
        
        const currentValue = bibInput.value || '';
        bibInput.value = currentValue.slice(0, -1);
        // Trigger the input event to update participant selection
        bibInput.dispatchEvent(new Event('input', { bubbles: true }));
    }

    function applyBibInput() {
        const bibInput = document.getElementById('bibInput');
        if (!bibInput) return;
        
        // Trigger the input event to finalize BIB selection
        bibInput.dispatchEvent(new Event('input', { bubbles: true }));
        
        // Enable next button and criteria grid when BIB is entered
        enableNextButtonAndCriteria();
        
        // Visual feedback
        bibInput.style.transition = 'all 0.2s ease';
        bibInput.style.backgroundColor = '#198754';
        bibInput.style.color = 'white';
        setTimeout(() => {
            bibInput.style.backgroundColor = '';
            bibInput.style.color = '';
        }, 300);
        
        // Focus on the participant dropdown after BIB entry
        const participantSelect = document.getElementById('ep_id');
        if (participantSelect) {
            setTimeout(() => {
                participantSelect.focus();
            }, 350);
        }
        
        // Collapse the keypad after applying
        const bibKeypadCollapse = document.getElementById('bibKeypadCollapse');
        if (bibKeypadCollapse && bibKeypadCollapse.classList.contains('show')) {
            const bsCollapse = bootstrap.Collapse.getInstance(bibKeypadCollapse);
            if (bsCollapse) {
                bsCollapse.hide();
            }
        }
    }

    function enableNextButtonAndCriteria() {
        const nextButton = document.getElementById('nextButton');
        const criteriaGrid = document.getElementById('criteria-input-grid');
        
        if (nextButton) {
            nextButton.classList.remove('disabled');
            nextButton.disabled = false;
        }
        if (criteriaGrid) {
            criteriaGrid.classList.remove('disabled');
        }
    }

    function scrollToCriteriaInput() {
        const nextButton = document.getElementById('nextButton');
        const criteriaGrid = document.getElementById('criteria-input-grid');
        
        // Change status to Info when navigating to score input
        setStatusToInfo();
        
        // Enable button and grid if disabled
        if (nextButton) {
            nextButton.classList.remove('disabled');
            nextButton.disabled = false;
        }
        if (criteriaGrid) {
            criteriaGrid.classList.remove('disabled');
        }
        
        // Scroll to criteria grid
        if (criteriaGrid) {
            criteriaGrid.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'start' 
            });
            
            // Visual feedback
            criteriaGrid.style.transition = 'all 0.3s ease';
            criteriaGrid.style.boxShadow = '0 0 20px rgba(13, 110, 253, 0.5)';
            setTimeout(() => {
                criteriaGrid.style.boxShadow = '';
            }, 1000);
        }
    }

    // Toggle BIB keypad visibility
    function toggleBibKeypad() {
        const bibKeypadCollapse = document.getElementById('bibKeypadCollapse');
        if (!bibKeypadCollapse) return;
        
        const bsCollapse = bootstrap.Collapse.getOrCreateInstance(bibKeypadCollapse);
        bsCollapse.toggle();
    }

    function clearScoreForm() {
        // Clear the score input
        const scoreInput = document.querySelector('[name=score_value]');
        const scoreRange = document.querySelector('[name=score_range]');
        if (scoreInput) scoreInput.value = '';
        if (scoreRange) scoreRange.value = '0';
        
        // Clear all criteria inputs
        const criteriaInputs = document.querySelectorAll('[name^="criteria_scores"][name$="[value]"]');
        const criteriaRanges = document.querySelectorAll('[name^="criteria_scores"][name$="[range]"]');
        
        criteriaInputs.forEach(input => input.value = '');
        criteriaRanges.forEach(range => range.value = '0');
        
        // Reset calculated score display
        const calculatedScoreElement = document.getElementById('calculatedScore');
        if (calculatedScoreElement) {
            calculatedScoreElement.textContent = '0.0';
        }
        
        // Clear current participant score display
        const currentScoreDisplay = document.getElementById('current-participant-score');
        if (currentScoreDisplay) {
            currentScoreDisplay.textContent = '--';
        }
        
        // Reset status to OK
        document.getElementById('statusOK').checked = true;
        
        // Clear BIB input
        document.getElementById('bibInput').value = '';
        
        // DON'T reset participant selection here - let loadParticipants() handle it
        // The dropdown will be refreshed via AJAX after score submission
        // document.getElementById('ep_id').selectedIndex = 0;
        
        // Don't focus on BIB input - keypad only mode
        // document.getElementById('bibInput').focus();
    }

    function clearScoreInputsOnly() {
        // Clear the main score input
        const scoreInput = document.querySelector('[name=score_value]');
        const scoreRange = document.querySelector('[name=score_range]');
        if (scoreInput) scoreInput.value = '';
        if (scoreRange) scoreRange.value = '0';
        
        // Clear all criteria inputs
        const criteriaInputs = document.querySelectorAll('[name^="criteria_scores"][name$="[value]"]');
        const criteriaRanges = document.querySelectorAll('[name^="criteria_scores"][name$="[range]"]');
        
        criteriaInputs.forEach(input => input.value = '');
        criteriaRanges.forEach(range => range.value = '0');
        
        // Reset calculated score display
        const calculatedScoreElement = document.getElementById('calculatedScore');
        if (calculatedScoreElement) {
            calculatedScoreElement.textContent = '0';
        }
        
        // Don't focus on score input - keypad only mode
        // if (scoreInput) scoreInput.focus();
    }

    // Enhanced figure selection helper functions
    function showFiguresSummary() {
        if (!window.__FIGURE_SELECTIONS__ || Object.keys(window.__FIGURE_SELECTIONS__).length === 0) {
            showToast('No Selections', 'No figures selected yet', 'info');
            return;
        }
        
        let summary = 'Selected Figures:\n';
        Object.entries(window.__FIGURE_SELECTIONS__).forEach(([category, items]) => {
            if (items.size > 0) {
                summary += `• ${category}: ${Array.from(items).join(', ')}\n`;
            }
        });
        
        alert(summary);
    }

    async function clearAllFigures() {
        const confirmed = await showConfirm(
            'Clear all figure selections?', 
            'Clear Figures', 
            'Clear All', 
            'btn-warning'
        );
        
        if (confirmed) {
            // Clear all selections
            window.__FIGURE_SELECTIONS__ = {};
            
            // Clear all active chips
            document.querySelectorAll('.figure-chip.active').forEach(chip => {
                chip.classList.remove('active');
            });
            
            // Update hidden fields
            if (typeof updateFiguresJson === 'function') updateFiguresJson();
            
            showToast('Cleared', 'All figure selections cleared', 'success');
        }
    }

    function copyFromLastRun() {
        const currentRun = document.querySelector('[name="run_number"]:checked')?.value;
        if (!currentRun || currentRun <= 1) {
            showToast('No Previous Run', 'No previous run to copy from', 'warning');
            return;
        }
        
        const lastRun = parseInt(currentRun) - 1;
        if (figureHistoryByRun[lastRun]) {
            // Copy selections from last run
            Object.entries(figureHistoryByRun[lastRun]).forEach(([category, items]) => {
                if (!window.__FIGURE_SELECTIONS__[category]) {
                    window.__FIGURE_SELECTIONS__[category] = new Set();
                }
                items.forEach(item => {
                    window.__FIGURE_SELECTIONS__[category].add(item);
                    // Update UI
                    const chip = document.querySelector(`.figure-chip[data-cat="${category}"][data-item="${item}"]`);
                    if (chip) chip.classList.add('active');
                });
            });
            
            // Update hidden fields
            if (typeof updateFiguresJson === 'function') updateFiguresJson();
            
            showToast('Copied', `Figures copied from Run ${lastRun}`, 'success');
        } else {
            showToast('No Data', `No figures found for Run ${lastRun}`, 'warning');
        }
    }

    function showOverwriteModal(idx) {
        const score = latestScores[idx];
        if (!score) return;
        
        // Populate basic score data
        document.getElementById('overwrite_score_id').value = score.score_id || '';
        // Format score value to show decimals only if needed
        const scoreValue = score.score_value !== null && score.score_value !== undefined ? 
            (parseFloat(score.score_value) % 1 === 0 ? Math.floor(parseFloat(score.score_value)) : parseFloat(score.score_value)) : '';
        document.getElementById('overwrite_score_value').value = scoreValue;
        document.getElementById('overwrite_status').value = score.status;
        
        // Set participant info in header
        const participantInfo = `${score.first_name} ${score.last_name} (Bib ${score.bib}) - Run ${score.run_number}`;
        const editScoreParticipant = document.getElementById('editScoreParticipant');
        if (editScoreParticipant) {
            editScoreParticipant.textContent = participantInfo;
        }
        
        // Store the index for later use
        document.getElementById('overwriteScoreForm').dataset.scoreIndex = idx.toString();
        
        // Handle figures if available
        setupEditFigureSelection(score);
        
        var modal = new bootstrap.Modal(document.getElementById('overwriteScoreModal'));
        modal.show();
        
        // Focus on score input when modal opens
        modal._element.addEventListener('shown.bs.modal', function() {
            document.getElementById('overwrite_score_value').focus();
            document.getElementById('overwrite_score_value').select();
        });
    }

    // Setup figure selection for edit modal
    function setupEditFigureSelection(score) {
        const figureSection = document.getElementById('editFigureSelection');
        const figureGrid = document.getElementById('editFigureGrid');
        const figuresJsonField = document.getElementById('overwrite_figures_json');
        
        // Clear previous state
        window.__EDIT_FIGURE_SELECTIONS__ = {};
        figuresJsonField.value = '';
        
        if (!FIGURE_DATA.available) {
            figureSection.style.display = 'none';
            return;
        }
        
        // Show figure section
        figureSection.style.display = 'block';
        
        // Parse existing figures from score if available
        let existingFigures = {};
        if (score.figures_json) {
            try {
                const parsedData = JSON.parse(score.figures_json);
                console.log('Parsed existing figures:', parsedData);
                console.log('Type of parsedData:', typeof parsedData);
                console.log('Keys of parsedData:', Object.keys(parsedData));
                
                // Handle different possible formats
                if (parsedData.figures && typeof parsedData.figures === 'object') {
                    // Format: {"figures": {"category": ["item1", "item2"]}, "validation": {...}}
                    existingFigures = parsedData.figures;
                    console.log('Using figures property:', existingFigures);
                } else if (typeof parsedData === 'object' && !Array.isArray(parsedData)) {
                    // Direct format: {"category": ["item1", "item2"]}
                    // But exclude non-figure properties like "validation"
                    Object.entries(parsedData).forEach(([key, value]) => {
                        if (key !== 'validation' && key !== 'timestamp' && 
                            (Array.isArray(value) || typeof value === 'string')) {
                            existingFigures[key] = value;
                        }
                    });
                    console.log('Using filtered direct format:', existingFigures);
                } else {
                    console.warn('Unexpected figures_json format:', parsedData);
                }
                
                // Debug each category's data
                Object.entries(existingFigures).forEach(([cat, items]) => {
                    console.log(`Edit Modal Category "${cat}":`, items, 'Type:', typeof items, 'IsArray:', Array.isArray(items));
                });
            } catch (e) {
                console.warn('Could not parse existing figures_json:', e);
                console.log('Raw figures_json:', score.figures_json);
            }
        }
        
        // Build figure grid
        let gridHTML = '';
        FIGURE_DATA.categories.forEach(category => {
            const items = FIGURE_DATA.itemsByCategory[category] || [];
            if (items.length === 0) return;
            
            gridHTML += `
                <div class="figure-row mb-3 p-3">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h6 class="mb-0"><i class="fas fa-tag me-2"></i>${category}</h6>
                        <button type="button" class="btn btn-sm btn-outline-secondary clear-cat" 
                                onclick="clearEditFigureCategory('${category}')">
                            <i class="fas fa-times"></i> Clear
                        </button>
                    </div>
                    <div class="figure-chips d-flex flex-wrap gap-2">
            `;
            
            items.forEach(item => {
                // Check if item is selected - handle different data formats
                let isSelected = false;
                if (existingFigures[category]) {
                    if (Array.isArray(existingFigures[category])) {
                        isSelected = existingFigures[category].includes(item);
                    } else if (typeof existingFigures[category] === 'string') {
                        isSelected = existingFigures[category] === item;
                    } else if (existingFigures[category] && typeof existingFigures[category] === 'object') {
                        // Handle object case - check if item exists as key or value
                        isSelected = Object.values(existingFigures[category]).includes(item) || 
                                   Object.keys(existingFigures[category]).includes(item);
                    }
                }
                
                const activeClass = isSelected ? 'active' : '';
                gridHTML += `
                    <button type="button" 
                            class="btn btn-sm btn-outline-primary figure-chip ${activeClass}" 
                            data-cat="${category}" 
                            data-item="${item}"
                            onclick="toggleEditFigureChip(this, '${category}', '${item}')">
                        ${item}
                    </button>
                `;
            });
            
            gridHTML += `
                    </div>
                </div>
            `;
        });
        
        figureGrid.innerHTML = gridHTML;
        
        // Initialize selections from existing data
        window.__EDIT_FIGURE_SELECTIONS__ = {};
        Object.entries(existingFigures).forEach(([category, items]) => {
            // Ensure items is an array before creating Set
            if (Array.isArray(items)) {
                window.__EDIT_FIGURE_SELECTIONS__[category] = new Set(items);
            } else if (typeof items === 'string') {
                // Handle case where items might be a single string
                window.__EDIT_FIGURE_SELECTIONS__[category] = new Set([items]);
            } else if (items && typeof items === 'object') {
                // Handle case where items might be an object - try to get its values or keys
                const itemArray = Array.isArray(items) ? items : 
                                 items.hasOwnProperty('length') ? Array.from(items) :
                                 Object.values(items).length > 0 ? Object.values(items) : 
                                 Object.keys(items);
                window.__EDIT_FIGURE_SELECTIONS__[category] = new Set(itemArray);
            } else {
                // Fallback: create empty set
                console.warn('Unexpected items format for category', category, ':', items);
                window.__EDIT_FIGURE_SELECTIONS__[category] = new Set();
            }
        });
        
        // Update hidden field and summary
        updateEditFiguresJson();
        updateEditFigureSummary();
        
        console.log('Initialized edit figure selections:', window.__EDIT_FIGURE_SELECTIONS__);
    }

    // Toggle figure chip in edit modal
    function toggleEditFigureChip(button, category, item) {
        if (!window.__EDIT_FIGURE_SELECTIONS__[category]) {
            window.__EDIT_FIGURE_SELECTIONS__[category] = new Set();
        }
        
        const isActive = button.classList.contains('active');
        
        if (isActive) {
            // Remove selection
            button.classList.remove('active');
            window.__EDIT_FIGURE_SELECTIONS__[category].delete(item);
            
            // Remove category if empty
            if (window.__EDIT_FIGURE_SELECTIONS__[category].size === 0) {
                delete window.__EDIT_FIGURE_SELECTIONS__[category];
            }
        } else {
            // Add selection
            button.classList.add('active');
            window.__EDIT_FIGURE_SELECTIONS__[category].add(item);
        }
        
        updateEditFiguresJson();
        updateEditFigureSummary();
        
        console.log('Updated edit figure selections:', window.__EDIT_FIGURE_SELECTIONS__);
    }

    // Clear entire category in edit modal
    function clearEditFigureCategory(category) {
        // Remove from selections
        delete window.__EDIT_FIGURE_SELECTIONS__[category];
        
        // Update UI
        document.querySelectorAll(`#editFigureGrid .figure-chip[data-cat="${category}"]`).forEach(chip => {
            chip.classList.remove('active');
        });
        
        updateEditFiguresJson();
        updateEditFigureSummary();
        
        console.log('Cleared edit figure category:', category);
    }

    // Update hidden figures_json field for edit modal
    function updateEditFiguresJson() {
        const figuresJsonField = document.getElementById('overwrite_figures_json');
        
        // Convert Sets to arrays for JSON serialization
        const figuresData = {};
        Object.entries(window.__EDIT_FIGURE_SELECTIONS__ || {}).forEach(([category, itemSet]) => {
            if (itemSet.size > 0) {
                figuresData[category] = Array.from(itemSet);
            }
        });
        
        figuresJsonField.value = Object.keys(figuresData).length > 0 ? JSON.stringify(figuresData) : '';
        console.log('Updated edit figures JSON:', figuresJsonField.value);
    }

    // Update figure summary display in edit modal
    function updateEditFigureSummary() {
        const summaryDiv = document.getElementById('editFigureSummary');
        const summaryContent = document.getElementById('editFigureSummaryContent');
        
        const selections = window.__EDIT_FIGURE_SELECTIONS__ || {};
        const categories = Object.keys(selections);
        
        if (categories.length === 0) {
            summaryDiv.style.display = 'none';
            return;
        }
        
        let summaryHTML = '';
        categories.forEach(category => {
            const items = Array.from(selections[category]);
            summaryHTML += `<div><strong>${category}:</strong> ${items.join(', ')}</div>`;
        });
        
        summaryContent.innerHTML = summaryHTML;
        summaryDiv.style.display = 'block';
    }

    // In the overwrite form submission handler, fix the parsing:
    // AJAX overwrite form submission
    const overwriteForm = document.getElementById('overwriteScoreForm');
    if (overwriteForm) {
        overwriteForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('ajax', '1'); // Add AJAX flag
            
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            
            // FIX: Better parsing of scoreIndex with validation
            const scoreIndexStr = this.dataset.scoreIndex;
            const scoreIndex = parseInt(scoreIndexStr, 10);
            
            // Validate scoreIndex
            if (isNaN(scoreIndex) || scoreIndex < 0 || scoreIndex >= latestScores.length) {
                console.error('Invalid score index:', scoreIndexStr, 'parsed as:', scoreIndex);
                showToast('Error', 'Invalid score reference. Please refresh the page.', 'error');
                return;
            }
            
            // Show loading state
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Updating...';
            
            // Initialize Lucide icons for the loading button
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons(submitBtn);
            }
            
            // Submit via AJAX
            const overwriteRequestData = Object.fromEntries(formData.entries());
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                // Log the API call for debugging
                logApiCall('POST', window.location.href + ' (overwrite)', overwriteRequestData, data);
                
                if (data.success) {
                    // Update the table without page reload
                    const updateSuccess = updateScoreInTable(scoreIndex, data.score);
                    
                    if (updateSuccess) {
                        // Close modal
                        const modal = bootstrap.Modal.getInstance(document.getElementById('overwriteScoreModal'));
                        modal.hide();
                        
                        // Show success message
                        showToast('Success!', data.message, 'success');
                    } else {
                        showToast('Warning', 'Score updated but table refresh failed. Please reload page.', 'error');
                    }
                    
                } else {
                    throw new Error(data.message || 'Update failed');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                
                // Log the API call error for debugging
                logApiCall('POST', window.location.href + ' (overwrite)', overwriteRequestData, { 
                    error: error.message || 'Unknown error',
                    status: 'error',
                    timestamp: new Date().toISOString()
                });
                
                showToast('Error', error.message || 'Error updating score. Please try again.', 'error');
            })
            .finally(() => {
                // Restore button state
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
                
                // Initialize Lucide icons for the restored button
                if (window.initializeLucideIcons) {
                    window.initializeLucideIcons(submitBtn);
                }
            });
        });
    }

        // Toast notification function
        function showToast(title, message, type = 'info', delay = 5000) {
            // Create toast container if it doesn't exist
            let toastContainer = document.getElementById('toastContainer');
            if (!toastContainer) {
                toastContainer = document.createElement('div');
                toastContainer.id = 'toastContainer';
                toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
                toastContainer.style.zIndex = '9999';
                document.body.appendChild(toastContainer);
            }
            
            // Create toast
            const toastId = 'toast_' + Date.now();
            const bgClass = type === 'success' ? 'bg-success' : type === 'error' ? 'bg-danger' : 'bg-info';
            
            const toastHTML = `
                <div id="${toastId}" class="toast ${bgClass} text-white" role="alert">
                    <div class="toast-header ${bgClass} text-white border-0">
                        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'} me-2"></i>
                        <strong class="me-auto">${title}</strong>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
                    </div>
                    <div class="toast-body">
                        ${message}
                    </div>
                </div>
            `;
            
            toastContainer.insertAdjacentHTML('beforeend', toastHTML);
            
            // Show toast
            const toastElement = document.getElementById(toastId);
            const toast = new bootstrap.Toast(toastElement, { delay: delay });
            toast.show();
            
            // Remove from DOM after hiding
            toastElement.addEventListener('hidden.bs.toast', function() {
                this.remove();
            });
        }

        // Bootstrap modal confirmation dialog (replaces confirm())
        function showConfirm(message, title = 'Confirm Action', confirmBtnText = 'Confirm', confirmBtnClass = 'btn-primary') {
            return new Promise((resolve) => {
                const modal = document.getElementById('confirmModal');
                const messageEl = document.getElementById('confirmModalMessage');
                const titleEl = document.getElementById('confirmModalLabel');
                const confirmBtn = document.getElementById('confirmModalBtn');
                
                // Set content
                messageEl.textContent = message;
                titleEl.innerHTML = '<i class="fas fa-question-circle me-2"></i>' + title;
                confirmBtn.textContent = confirmBtnText;
                
                // Reset button classes
                confirmBtn.className = 'btn ' + confirmBtnClass;
                
                // Show modal
                const bsModal = new bootstrap.Modal(modal);
                bsModal.show();
                
                // Handle confirm
                const handleConfirm = () => {
                    bsModal.hide();
                    cleanup();
                    resolve(true);
                };
                
                // Handle cancel
                const handleCancel = () => {
                    cleanup();
                    resolve(false);
                };
                
                // Cleanup listeners
                const cleanup = () => {
                    confirmBtn.removeEventListener('click', handleConfirm);
                    modal.removeEventListener('hidden.bs.modal', handleCancel);
                };
                
                // Add listeners
                confirmBtn.addEventListener('click', handleConfirm);
                modal.addEventListener('hidden.bs.modal', handleCancel, { once: true });
            });
        }

        function showNoParticipantsNotification() {
            // Create or show notification banner
            let notification = document.getElementById('noParticipantsNotification');
            
            if (!notification) {
                notification = document.createElement('div');
                notification.id = 'noParticipantsNotification';
                notification.className = 'alert alert-info alert-dismissible fade show fs-4';
                notification.style.cssText = 'position: sticky; top: 100px; z-index: 1040; margin: 10px;';
                notification.innerHTML = `
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>No participants available</strong>
                    <p class="mb-0 mt-1">There are no more participants in the queue to perform right now.</p>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                `;
                
                // Insert after the participant selection card
                const participantCard = document.getElementById('participant-selection');
                if (participantCard) {
                    participantCard.before(notification);
                }
            } else {
                notification.style.display = 'block';
            }
            
            // Hide current participant info when no participants available
            const currentParticipantInfo = document.getElementById('current-participant-info');
            if (currentParticipantInfo) {
                currentParticipantInfo.style.display = 'none';
            }
        }

        function hideNoParticipantsNotification() {
            const notification = document.getElementById('noParticipantsNotification');
            if (notification) {
                notification.style.display = 'none';
            }
        }

        function loadOtherScoresForParticipant() {
            const ep_id = document.getElementById('ep_id').value;
            const heat = document.querySelector('[name=heat_number]').value;
            if (!ep_id || !heat) {
                // Hide the card when no participant selected
                const otherHeatScoresCard = document.getElementById('otherHeatScoresCard');
                if (otherHeatScoresCard) {
                    otherHeatScoresCard.style.display = 'none';
                }
                return;
            }

            // Store current participant for figure history
            currentParticipantId = ep_id;

            fetch(`ajax_get_other_scores.php?ep_id=${encodeURIComponent(ep_id)}&heat_number=${encodeURIComponent(heat)}&include_judges=1&include_cp=1&current_judge_id=${encodeURIComponent(CURRENT_JUDGE_ID)}`)
                .then(res => {
                    if (!res.ok) {
                        throw new Error(`HTTP error! status: ${res.status}`);
                    }
                    return res.json();
                })
                .then(scores => {
                    console.log('Other scores loaded:', scores);
                    const container = document.getElementById('otherHeatScores');
                    const card = document.getElementById('otherHeatScoresCard');
                    
                    if (!Array.isArray(scores) || scores.length === 0) {
                        container.innerHTML = '<div class="text-muted"><em>No other scores available for this participant</em></div>';
                        card.style.display = 'block';
                        return;
                    }

                    // Show the card since we have data
                    card.style.display = 'block';

                    // Load figure history for this participant after loading scores
                    loadFigureHistoryForParticipant(ep_id);

                    // Normalize
                    const norm = scores.map(s => ({
                        judge_id: String(s.judge_id ?? s.judgeId ?? s.judge?.id ?? ''),
                        judge_name: s.judge_name ?? s.judgeName ?? s.judge?.name ?? '',
                        control_point: s.control_point_name ?? s.cp_name ?? s.checkpoint ?? s.control_point ?? '—',
                        run_number: Number(s.run_number ?? s.run ?? 0),
                        score_value: s.score_value !== undefined && s.score_value !== null ? Number(s.score_value) : null,
                        status: s.status ?? 'OK',
                        updated_at: s.updated_at ?? s.created_at ?? null
                    }));

                    // Dedupe by judge_id + run_number (keep latest)
                    const mapLatest = new Map();
                    for (const it of norm) {
                        const key = `${it.judge_id}-${it.run_number}`;
                        const prev = mapLatest.get(key);
                        if (!prev) {
                            mapLatest.set(key, it);
                        } else {
                            const tPrev = prev.updated_at ? new Date(prev.updated_at).getTime() : 0;
                            const tNew = it.updated_at ? new Date(it.updated_at).getTime() : 0;
                            if (tNew >= tPrev) mapLatest.set(key, it);
                        }
                    }
                    const deduped = Array.from(mapLatest.values());

                    // Build rows: one per judge, columns per run
                    const heatNum = Number(heat);
                    const runsCount = Number(HEAT_RUNS_MAP?.[heatNum] ?? DEFAULT_RUNS_PER_HEAT ?? 3);

                    // judge -> { judge_name, control_point, runs: {run_number: {score,status}} }
                    const rowsByJudge = new Map();
                    for (const s of deduped) {
                        if (!rowsByJudge.has(s.judge_id)) {
                            rowsByJudge.set(s.judge_id, {
                                judge_id: s.judge_id,
                                judge_name: s.judge_name || (`Judge ${s.judge_id || '?'}`),
                                control_point: s.control_point,
                                runs: {}
                            });
                        }
                        rowsByJudge.get(s.judge_id).runs[s.run_number] = {
                            value: s.score_value,
                            status: s.status
                        };
                    }

                    // Order: current judge first, then others by judge_name
                    const myId = String(CURRENT_JUDGE_ID ?? '');
                    const allRows = Array.from(rowsByJudge.values());
                    const mine = allRows.filter(r => r.judge_id === myId);
                    const others = allRows.filter(r => r.judge_id !== myId)
                                        .sort((a, b) => a.judge_name.localeCompare(b.judge_name));
                    const ordered = [...mine, ...others];

                    // Build table
                    let thead = `
                        <thead class="table-light">
                            <tr>
                                <th style="white-space:nowrap;">Judge</th>
                                
                                ${Array.from({length: runsCount}, (_, i) => `<th class="text-center" style="white-space:nowrap;">Run ${i+1}</th>`).join('')}
                            </tr>
                        </thead>`;

                    const renderCell = (cell) => {
                        if (!cell) return '—';
                        const val = (cell.value !== null && cell.value !== undefined) ? Number(cell.value).toFixed(2) : '—';
                        const status = cell.status || 'OK';
                        const badge = status === 'OK' ? 'success' : (status === 'DSQ' ? 'danger' : 'warning');
                        return `<div><strong>${val}</strong></div><div><span class="badge bg-${badge}">${status}</span></div>`;
                    };

                    const tbodyRows = ordered.map(row => {
                        const isYou = row.judge_id === myId;
                        const judgeCell = isYou
                            ? `<span class="badge bg-success me-2">You</span><span>${row.judge_name}</span>`
                            : `<span class="badge bg-secondary me-2">Judge</span><span>${row.judge_name}</span>`;
                        const runCells = Array.from({length: runsCount}, (_, i) => {
                            const rno = i + 1;
                            return `<td class="text-center align-middle">${renderCell(row.runs[rno])}</td>`;
                        }).join('');
                        return `
                            <tr>
                                <td class="align-middle">${judgeCell} <small>${row.control_point || '—'}</small></td>
                                ${runCells}
                            </tr>`;
                    }).join('');

                    const tableHTML = `
                        <div class="table-responsive">
                            <table class="table table-sm scores-table mb-0">
                                ${thead}
                                <tbody>
                                    ${tbodyRows}
                                </tbody>
                            </table>
                        </div>`;

                    container.innerHTML = tableHTML;
                })
                .catch(err => {
                    console.error('Error loading other scores:', err);
                    const container = document.getElementById('otherHeatScores');
                    const card = document.getElementById('otherHeatScoresCard');
                    if (container && card) {
                        container.innerHTML = '<div class="alert alert-warning"><i class="fas fa-exclamation-triangle me-2"></i>Could not load other scores. Error: ' + err.message + '</div>';
                        card.style.display = 'block';
                    }
                });
        }

        /**
         * Load existing figure selections for current judge's score
         */
        function loadExistingFigureSelections() {
            if (!FIGURE_DATA.available) return;
            
            const ep_id = document.getElementById('ep_id').value;
            const runNumber = document.querySelector('[name="run_number"]:checked')?.value;
            const heat = document.querySelector('[name=heat_number]').value;
            
            if (!ep_id || !runNumber || !heat) return;
            
            console.log('Loading existing figures for participant:', ep_id, 'run:', runNumber);
            
            // Fetch current judge's existing score for this participant and run
            fetch(`ajax_get_other_scores.php?ep_id=${encodeURIComponent(ep_id)}&heat_number=${encodeURIComponent(heat)}&current_judge_only=1&run_number=${encodeURIComponent(runNumber)}&current_judge_id=${encodeURIComponent(CURRENT_JUDGE_ID)}`)
                .then(res => res.json())
                .then(scores => {
                    // Clear current selections first
                    if (window.__FIGURE_SELECTIONS__) {
                        window.__FIGURE_SELECTIONS__ = {};
                        document.querySelectorAll('.figure-chip.active').forEach(chip => {
                            chip.classList.remove('active');
                        });
                    }
                    
                    if (!Array.isArray(scores) || scores.length === 0) {
                        console.log('No existing score found for current judge');
                        if (typeof updateFiguresJson === 'function') {
                            updateFiguresJson();
                        }
                        return;
                    }
                    
                    // Find the score for the current run
                    const currentScore = scores.find(s => Number(s.run_number || s.run) === Number(runNumber));
                    if (!currentScore || !currentScore.figures_json) {
                        console.log('No figures found in existing score');
                        if (typeof updateFiguresJson === 'function') {
                            updateFiguresJson();
                        }
                        return;
                    }
                    
                    try {
                        const parsedData = JSON.parse(currentScore.figures_json);
                        console.log('Parsed existing figures for current score:', parsedData);
                        
                        let existingFigures = {};
                        
                        // Handle different possible formats
                        if (parsedData.figures && typeof parsedData.figures === 'object') {
                            // Format: {"figures": {"category": ["item1", "item2"]}, "validation": {...}}
                            existingFigures = parsedData.figures;
                            console.log('Using figures property:', existingFigures);
                        } else if (typeof parsedData === 'object' && !Array.isArray(parsedData)) {
                            // Direct format: {"category": ["item1", "item2"]}
                            // But exclude non-figure properties like "validation"
                            Object.entries(parsedData).forEach(([key, value]) => {
                                if (key !== 'validation' && key !== 'timestamp' && 
                                    (Array.isArray(value) || typeof value === 'string')) {
                                    existingFigures[key] = value;
                                }
                            });
                            console.log('Using filtered direct format:', existingFigures);
                        } else {
                            console.warn('Unexpected figures_json format for main selection:', parsedData);
                        }
                        
                        // Initialize figure selections
                        window.__FIGURE_SELECTIONS__ = {};
                        
                        // Load existing selections and mark chips as active
                        Object.entries(existingFigures).forEach(([category, items]) => {
                            if (Array.isArray(items)) {
                                window.__FIGURE_SELECTIONS__[category] = new Set(items);
                                
                                // Mark chips as active
                                items.forEach(item => {
                                    const chip = document.querySelector(`.figure-chip[data-cat="${CSS.escape(category)}"][data-item="${CSS.escape(item)}"]`);
                                    if (chip) {
                                        chip.classList.add('active');
                                        console.log('Marked chip as active:', category, item);
                                    }
                                });
                            } else if (typeof items === 'string') {
                                window.__FIGURE_SELECTIONS__[category] = new Set([items]);
                                
                                // Mark chip as active
                                const chip = document.querySelector(`.figure-chip[data-cat="${CSS.escape(category)}"][data-item="${CSS.escape(items)}"]`);
                                if (chip) {
                                    chip.classList.add('active');
                                    console.log('Marked chip as active:', category, items);
                                }
                            }
                        });
                        
                        // Update hidden fields and display
                        if (typeof updateFiguresJson === 'function') {
                            updateFiguresJson();
                        }
                        updateDiversityValidationWithHistory();
                        
                        console.log('Loaded existing figure selections:', window.__FIGURE_SELECTIONS__);
                        
                    } catch (e) {
                        console.warn('Could not parse existing figures_json:', e);
                        console.log('Raw figures_json:', currentScore.figures_json);
                        if (typeof updateFiguresJson === 'function') {
                            updateFiguresJson();
                        }
                    }
                })
                .catch(err => {
                    console.error('Error loading existing figure selections:', err);
                });
        }

        // Load figure history for participant across all runs
        function loadFigureHistoryForParticipant(ep_id) {
            console.log('loadFigureHistoryForParticipant called with ep_id:', ep_id);
            
            if (!FIGURE_DATA.available || !ep_id) {
                console.log('Figure data not available or no ep_id provided');
                figureHistoryByRun = {};
                return;
            }

            const eventId = <?= json_encode($event_id) ?>;
            console.log('Event ID:', eventId);
            if (!eventId) return;

            const url = `ajax_get_figure_history.php?ep_id=${encodeURIComponent(ep_id)}&judge_id=${encodeURIComponent(CURRENT_JUDGE_ID)}`;
            console.log('Fetching figure history from:', url);

            fetch(url)
                .then(res => {
                    console.log('Response status:', res.status);
                    return res.json();
                })
                .then(data => {
                    console.log('Raw figure history data received:', data);
                    figureHistoryByRun = {};
                    
                    if (Array.isArray(data)) {
                        data.forEach(score => {
                            const runNum = score.run_number;
                            const figuresJson = score.figures_json;
                            
                            console.log(`Processing run ${runNum}, figures_json:`, figuresJson);
                            
                            if (figuresJson) {
                                try {
                                    const parsedData = JSON.parse(figuresJson);
                                    console.log(`Parsed figures for run ${runNum}:`, parsedData);
                                    
                                    if (!figureHistoryByRun[runNum]) {
                                        figureHistoryByRun[runNum] = {};
                                    }
                                    
                                    let figures = {};
                                    
                                    // Handle different possible formats (same logic as other functions)
                                    if (parsedData.figures && typeof parsedData.figures === 'object') {
                                        // Format: {"figures": {"category": ["item1", "item2"]}, "validation": {...}}
                                        figures = parsedData.figures;
                                        console.log(`Using figures property for run ${runNum}:`, figures);
                                    } else if (typeof parsedData === 'object' && !Array.isArray(parsedData)) {
                                        // Direct format: {"category": ["item1", "item2"]}
                                        // But exclude non-figure properties like "validation"
                                        Object.entries(parsedData).forEach(([key, value]) => {
                                            if (key !== 'validation' && key !== 'timestamp' && 
                                                (Array.isArray(value) || typeof value === 'string')) {
                                                figures[key] = value;
                                            }
                                        });
                                        console.log(`Using filtered direct format for run ${runNum}:`, figures);
                                    } else {
                                        console.warn(`Unexpected figures_json format for run ${runNum}:`, parsedData);
                                    }
                                    
                                    // Parse figures into category->Set(items) structure
                                    Object.entries(figures).forEach(([category, items]) => {
                                        if (Array.isArray(items)) {
                                            figureHistoryByRun[runNum][category] = new Set(items);
                                        } else if (typeof items === 'string') {
                                            figureHistoryByRun[runNum][category] = new Set([items]);
                                        }
                                        console.log(`Added to figureHistoryByRun[${runNum}][${category}]:`, figureHistoryByRun[runNum][category]);
                                    });
                                } catch (e) {
                                    console.log('Failed to parse figures JSON for run', runNum, ':', e);
                                }
                            }
                        });
                    }
                    
                    console.log('Final figureHistoryByRun:', figureHistoryByRun);
                    
                    // Update diversity validation with historical data
                    console.log('Updating diversity validation with loaded figure history...');
                    updateDiversityValidationWithHistory();
                    
                    // Show figure history in UI
                    displayFigureHistory();
                })
                .catch(err => {
                    console.log('Could not load figure history:', err);
                    figureHistoryByRun = {};
                });
        }

        // Display figure history in the UI
        function displayFigureHistory() {
            const historyContainer = document.getElementById('figureHistoryDisplay');
            const historyCard = document.getElementById('figureHistoryCard');
            if (!historyContainer || !FIGURE_DATA.available) return;
            
            const currentRun = document.querySelector('[name="run_number"]:checked')?.value || 1;
            const runNumbers = Object.keys(figureHistoryByRun).map(Number).sort();
            
            // Clear existing historical markings
            document.querySelectorAll('.figure-chip.has-history').forEach(chip => {
                chip.classList.remove('has-history');
                // Restore original text
                const originalText = chip.getAttribute('data-original-text');
                if (originalText) {
                    chip.textContent = originalText;
                }
                chip.removeAttribute('data-run');
                chip.removeAttribute('data-runs');
            });
            
            // Track which figures appear in which runs for multi-run marking
            const figureRunMap = {}; // { "category:item": [run1, run2, ...] }
            
            // Build figure->runs mapping
            runNumbers.forEach(runNum => {
                if (runNum >= currentRun) return; // Only previous runs
                
                const runFigures = figureHistoryByRun[runNum];
                Object.entries(runFigures).forEach(([category, itemSet]) => {
                    itemSet.forEach(item => {
                        const figureKey = `${category}:${item}`;
                        if (!figureRunMap[figureKey]) {
                            figureRunMap[figureKey] = [];
                        }
                        figureRunMap[figureKey].push(runNum);
                    });
                });
            });
            
            // Mark chips with historical data, showing all runs where they were used
            Object.entries(figureRunMap).forEach(([figureKey, runs]) => {
                const [category, item] = figureKey.split(':', 2);
                const chip = document.querySelector(`.figure-chip[data-cat="${CSS.escape(category)}"][data-item="${CSS.escape(item)}"]`);
                if (chip) {
                    // Store original text if not already stored
                    if (!chip.hasAttribute('data-original-text')) {
                        chip.setAttribute('data-original-text', chip.textContent.trim());
                    }
                    
                    // Add run indicators to the chip text (but don't block selection)
                    const originalText = chip.getAttribute('data-original-text');
                    const runIndicator = runs.length > 1 ? `R${runs.join(',R')}` : `R${runs[0]}`;
                    
                    // Update chip text to include run indicator
                    chip.innerHTML = `${originalText} <small class="text-muted">(${runIndicator})</small>`;
                    
                    // Store run data for reference
                    chip.setAttribute('data-runs', runs.join(','));
                    chip.setAttribute('data-run', runIndicator);
                    
                    // Add a subtle visual indicator class (but keep chips selectable)
                    chip.classList.add('has-history');
                    
                    console.log(`Marked chip '${originalText}' with runs: ${runIndicator}`);
                }
            });
            
            if (runNumbers.length === 0) {
                historyContainer.innerHTML = '<div class="alert alert-info"><i class="fas fa-info-circle me-2"></i>No previous figure selections for this participant.</div>';
                historyCard.style.display = 'none';
                return;
            }
            
            // Show the history card since we have data
            historyCard.style.display = 'block';
            
            let html = '<div class="mb-3"><h6><i class="fas fa-history me-2"></i>Previous Figure Selections</h6>';
            
            // Show ALL previous runs, not just those with selections
            for (let runNum = 1; runNum < currentRun; runNum++) {
                const runFigures = figureHistoryByRun[runNum];
                
                html += `<div class="mb-2">
                    <strong class="text-primary">Run ${runNum}:</strong>`;
                
                if (runFigures && Object.keys(runFigures).length > 0) {
                    const categories = Object.keys(runFigures);
                    categories.forEach(category => {
                        const items = Array.from(runFigures[category]);
                        if (items.length > 0) {
                            html += ` <span class="badge bg-secondary me-1">${category}: ${items.join(', ')}</span>`;
                        }
                    });
                } else {
                    html += ' <span class="text-muted">No figures selected</span>';
                }
                
                html += '</div>';
            }
            
            // Show duplicate figure summary if any figures were used multiple times
            const duplicateFigures = Object.entries(figureRunMap).filter(([figureKey, runs]) => runs.length > 1);
            if (duplicateFigures.length > 0) {
                html += '<div class="small text-info mt-2 mb-2">';
                html += '<i class="fas fa-repeat me-1"></i><strong>Figures used multiple times:</strong><br>';
                duplicateFigures.forEach(([figureKey, runs]) => {
                    const [category, item] = figureKey.split(':', 2);
                    html += `• ${category}: ${item} <span class="badge bg-info">R${runs.join(', R')}</span><br>`;
                });
                html += '</div>';
            }
            
            html += '<div class="small text-muted mt-2"><i class="fas fa-info-circle me-1"></i>Previous selections are marked with yellow background and run numbers. Multiple runs shown as R1,R2.</div>';
            html += '</div>';
            historyContainer.innerHTML = html;
        }

        // Enhanced diversity validation with historical data
        function updateDiversityValidationWithHistory() {
            console.log('updateDiversityValidationWithHistory called');
            console.log('Current figureHistoryByRun:', figureHistoryByRun);
            console.log('Current window.__FIGURE_SELECTIONS__:', window.__FIGURE_SELECTIONS__);
            
            const box = document.getElementById('diversityValidation');
            const validationCard = document.getElementById('diversity-validation-panel');
            
            if (!box || !FIGURE_UI || !Array.isArray(FIGURE_UI.diversityRules) || FIGURE_UI.diversityRules.length === 0) {
                console.log('No diversity validation box or rules found');
                if (validationCard) validationCard.style.display = 'none';
                // Reset submit button if no diversity rules - ensure validation details are cleared
                window.__DIVERSITY_VALIDATION_DETAILS__ = {
                    overallCompliant: true,
                    validationErrors: [],
                    validationWarnings: [],
                    validationPassed: [],
                    rulesChecked: 0,
                    currentRun: document.querySelector('[name="run_number"]:checked')?.value || 1,
                    totalRuns: 0,
                    timestamp: Date.now()
                };
                updateSubmitButtonState(true);
                return;
            }
            
            console.log('Running diversity validation with history...');
            const isValid = runDiversityValidationWithHistory(box);
            updateSubmitButtonState(isValid);
            
            // Show validation card if there are rules and content
            if (validationCard && box.innerHTML.trim()) {
                validationCard.style.display = 'block';
            }
        }

        // Update submit button appearance based on validation state
        function updateSubmitButtonState(isValid) {
            const submitBtn = document.getElementById('submitBtn');
            const validationSummary = document.getElementById('validationSummary');
            const validationSummaryText = document.getElementById('validationSummaryText');
            
            if (!submitBtn) return;
            
            // Store validation state globally for use during submission
            window.__DIVERSITY_VALIDATION_STATE__ = {
                isValid: isValid,
                timestamp: Date.now(),
                currentRun: document.querySelector('[name="run_number"]:checked')?.value || 1
            };
            
            // Get validation details for summary
            const validationDetails = window.__DIVERSITY_VALIDATION_DETAILS__ || {};
            const rulesChecked = validationDetails.rulesChecked || 0;
            const errorCount = validationDetails.validationErrors?.length || 0;
            const warningCount = validationDetails.validationWarnings?.length || 0;
            const passedCount = validationDetails.validationPassed?.length || 0;
            const currentRun = validationDetails.currentRun || 1;
            const totalRuns = validationDetails.totalRuns || 3;
            
            // Build detailed validation summary from the actual rule results
            function buildDetailedSummary() {
                if (rulesChecked === 0) return '';
                
                let summaryHtml = '';
                const allRules = [
                    ...(validationDetails.validationErrors || []),
                    ...(validationDetails.validationWarnings || []),
                    ...(validationDetails.validationPassed || [])
                ];
                
                allRules.forEach((rule, index) => {
                    const isLast = index === allRules.length - 1;
                    const group = rule.group || '';
                    const contextDesc = rule.contextDesc || '';
                    const currentCount = rule.currentCount || 0;
                    const minDistinct = rule.minDistinct || 0;
                    const selectedItems = rule.selectedItems || [];
                    const availableItems = rule.availableItems || [];
                    const runInfo = rule.runInfo || `Run ${currentRun} of ${totalRuns}`;
                    const isLastRun = rule.isLastRun || false;
                    const validationLevel = rule.validationLevel || 'passed';
                    
                    // Build status indicator
                    let statusIcon = '';
                    let statusText = '';
                    if (validationLevel === 'error') {
                        statusIcon = '❌';
                        statusText = isLastRun ? 'FINAL RUN - Must meet requirement!' : 'Error';
                    } else if (validationLevel === 'warning') {
                        statusIcon = '⚠️';
                        statusText = 'Warning';
                    } else {
                        statusIcon = '✅';
                        statusText = `${currentCount}/${minDistinct} ✓`;
                    }
                    
                    // Build main line
                    summaryHtml += `<div class="mb-1">`;
                    summaryHtml += `<strong>${group}</strong> (${contextDesc})`;
                    if (runInfo) summaryHtml += ` • ${runInfo}`;
                    if (isLastRun && validationLevel === 'error') summaryHtml += ` • <span class="text-danger fw-bold">FINAL RUN - Must meet requirement!</span>`;
                    summaryHtml += `</div>`;
                    
                    // Build details line
                    summaryHtml += `<div class="small text-muted mb-2">`;
                    if (selectedItems.length > 0) {
                        summaryHtml += `Selected: ${selectedItems.join(', ')} `;
                    } else {
                        summaryHtml += `Selected: None `;
                    }
                    summaryHtml += `${statusIcon} ${statusText}`;
                    
                    // Add suggestions for failed rules
                    if ((validationLevel === 'error' || validationLevel === 'warning') && availableItems.length > 0) {
                        const suggestions = availableItems.filter(item => !selectedItems.includes(item)).slice(0, 5);
                        if (suggestions.length > 0) {
                            summaryHtml += `<br>Suggestions: ${suggestions.join(', ')}`;
                        }
                    }
                    summaryHtml += `</div>`;
                    
                    if (!isLast) summaryHtml += `<hr class="my-2">`;
                });
                
                return summaryHtml;
            }
            
            if (isValid) {
                submitBtn.classList.remove('btn-warning', 'btn-danger');
                submitBtn.classList.add('btn-success');
                submitBtn.innerHTML = '<i class="fas fa-save me-2"></i>Submit Score';
                submitBtn.title = 'All requirements met - ready to submit';
                
                // Update validation summary
                if (rulesChecked > 0) {
                    validationSummary.style.display = 'block';
                    const summaryAlert = validationSummary.querySelector('.alert');
                    summaryAlert.className = 'alert alert-success alert-sm mb-0';
                    
                    const detailedSummary = buildDetailedSummary();
                    if (detailedSummary) {
                        validationSummaryText.innerHTML = detailedSummary;
                    } else {
                        validationSummaryText.innerHTML = `<i class="fas fa-check-circle me-1"></i>All ${rulesChecked} diversity rules satisfied`;
                    }
                } else {
                    validationSummary.style.display = 'none';
                }
            } else {
                submitBtn.classList.remove('btn-success', 'btn-danger');
                submitBtn.classList.add('btn-warning');
                submitBtn.innerHTML = '<i class="fas fa-exclamation-triangle me-2"></i>Submit Score (Rules Warning)';
                submitBtn.title = 'Diversity requirements not fully met - score will be submitted with validation status';
                
                // Update validation summary
                if (rulesChecked > 0) {
                    validationSummary.style.display = 'block';
                    const summaryAlert = validationSummary.querySelector('.alert');
                    
                    // Use danger for errors, warning for warnings only
                    if (errorCount > 0) {
                        summaryAlert.className = 'alert alert-danger alert-sm mb-0';
                    } else {
                        summaryAlert.className = 'alert alert-warning alert-sm mb-0';
                    }
                    
                    const detailedSummary = buildDetailedSummary();
                    if (detailedSummary) {
                        validationSummaryText.innerHTML = detailedSummary;
                    } else {
                        let summaryParts = [];
                        if (errorCount > 0) summaryParts.push(`${errorCount} error${errorCount > 1 ? 's' : ''}`);
                        if (warningCount > 0) summaryParts.push(`${warningCount} warning${warningCount > 1 ? 's' : ''}`);
                        if (passedCount > 0) summaryParts.push(`${passedCount} passed`);
                        
                        const summaryText = summaryParts.length > 0 ? summaryParts.join(', ') : 'diversity issues';
                        validationSummaryText.innerHTML = `<i class="fas fa-exclamation-triangle me-1"></i>Diversity rules: ${summaryText} (${rulesChecked} checked)`;
                    }
                } else {
                    validationSummary.style.display = 'none';
                }
            }
            
            // Never disable the submit button - always allow submission
            submitBtn.disabled = false;
        }

        function runDiversityValidationWithHistory(outBox) {
            if (!outBox || !FIGURE_UI || !Array.isArray(FIGURE_UI.diversityRules) || FIGURE_UI.diversityRules.length === 0) {
                outBox.style.display = 'none';
                const validationCard = document.getElementById('diversity-validation-panel');
                if (validationCard) validationCard.style.display = 'none';
                return true;
            }

            // Get current run and heat info
            const runRadio = document.querySelector('[name="run_number"]:checked');
            const currentRun = runRadio ? parseInt(runRadio.value, 10) : 1;
            const heatSel = document.getElementById('heatSelect');
            const heatNum = heatSel ? parseInt(heatSel.value || '0', 10) : 0;
            const runsCount = Number(HEAT_RUNS_MAP?.[heatNum] ?? DEFAULT_RUNS_PER_HEAT ?? 3);
            const bestN = Number(FIGURE_UI?.attempts?.bestN || FIGURE_UI?.tricks?.bestN || 2);

            // Combine current selections with historical data
            const allRunsData = {}; // { run_number: { category: Set(items) } }
            
            // Add historical data
            Object.entries(figureHistoryByRun).forEach(([runNum, runData]) => {
                allRunsData[runNum] = runData;
            });
            
            console.log('Raw figureHistoryByRun:', figureHistoryByRun);
            console.log('All runs data from history:', allRunsData);
            
            // Add current run selections
            if (window.__FIGURE_SELECTIONS__ && Object.keys(window.__FIGURE_SELECTIONS__).length > 0) {
                allRunsData[currentRun] = window.__FIGURE_SELECTIONS__;
                console.log('Added current run selections for run', currentRun, ':', window.__FIGURE_SELECTIONS__);
            } else {
                console.log('No current selections found or empty object');
                console.log('window.__FIGURE_SELECTIONS__ exists:', !!window.__FIGURE_SELECTIONS__);
                console.log('Keys length:', Object.keys(window.__FIGURE_SELECTIONS__ || {}).length);
                // Initialize empty selections for current run
                allRunsData[currentRun] = {};
            }
            
            console.log('All runs data combined (for diversity validation):', allRunsData);
            console.log('Current run being processed:', currentRun);

            let msgs = [];
            let validationErrors = [];
            let validationWarnings = [];
            let validationPassed = [];
            let ok = true;
            const mode = FIGURE_UI.mode || '';

            // Build aggregate counts across all runs for diversity validation
            const aggregateCounts = {}; // { category: total_unique_items }
            const aggregateItems = {}; // { category: Set(all_items) }
            const runCounts = {}; // { run_number: { category: count } }
            
            Object.entries(allRunsData).forEach(([runNum, runData]) => {
                runCounts[runNum] = {};
                Object.entries(runData).forEach(([category, itemSet]) => {
                    const count = itemSet ? itemSet.size : 0;
                    runCounts[runNum][category] = count;
                    
                    // For aggregate, collect all unique items across runs
                    if (!aggregateCounts[category]) {
                        aggregateCounts[category] = new Set();
                        aggregateItems[category] = new Set();
                    }
                    if (itemSet) {
                        itemSet.forEach(item => {
                            aggregateCounts[category].add(item);
                            aggregateItems[category].add(item);
                        });
                    }
                });
            });
            
            // Convert aggregate sets to counts
            Object.keys(aggregateCounts).forEach(category => {
                aggregateCounts[category] = aggregateCounts[category].size;
            });

            // Evaluate diversity rules with historical context
            for (const r of FIGURE_UI.diversityRules) {
                const ruleType = r.rule_type || r.ruleType || '';
                const scope = r.scope || 'run';
                const group = r.group_name || r.group || '';
                const minDistinct = Number(r.min_distinct || 0);
                const isOptional = Boolean(r.is_optional);
                
                console.log('Processing diversity rule:', { ruleType, scope, group, minDistinct, isOptional });
                console.log('Rule object:', r);
                
                if (!group || minDistinct <= 0) continue;
                
                let currentCount = 0;
                let contextDesc = '';
                let selectedItems = [];
                let availableInCategory = [];
                
                // Get available items for this category to show suggestions
                if (FIGURE_DATA.itemsByCategory && FIGURE_DATA.itemsByCategory[group]) {
                    availableInCategory = FIGURE_DATA.itemsByCategory[group];
                }
                
                // Handle different scope types with progressive validation logic
                if (scope === 'run' || scope === 'attempts' || scope === 'attempt') {
                    // For attempts scope, show both current + historical for user context
                    const currentSelections = window.__FIGURE_SELECTIONS__?.[group];
                    let currentOnlyCount = currentSelections ? currentSelections.size : 0;
                    const currentItems = currentSelections ? Array.from(currentSelections) : [];
                    
                    // Also check historical data for this category to show complete picture
                    const historicalItems = new Set();
                    Object.entries(figureHistoryByRun).forEach(([runNum, runData]) => {
                        if (runNum < currentRun && runData[group]) {
                            console.log(`Historical data for group '${group}' in run ${runNum}:`, runData[group]);
                            runData[group].forEach(item => historicalItems.add(item));
                        }
                    });
                    
                    console.log(`Historical items for group '${group}':`, Array.from(historicalItems));
                    
                    // For display purposes, show total unique across all runs
                    const allItems = new Set([...currentItems, ...Array.from(historicalItems)]);
                    
                    if (scope === 'attempts' && historicalItems.size > 0) {
                        // Show combined count for attempts to give full context
                        currentCount = allItems.size;
                        selectedItems = Array.from(allItems);
                        contextDesc = `Total across runs 1-${currentRun} (${historicalItems.size} historical + ${currentOnlyCount} current)`;
                    } else {
                        // Pure current run only
                        currentCount = currentOnlyCount;
                        selectedItems = currentItems;
                        contextDesc = scope === 'attempts' || scope === 'attempt' ? `Current attempt (Run ${currentRun})` : `Run ${currentRun}`;
                    }
                    
                    console.log(`Scope '${scope}' - group: ${group}, currentCount: ${currentCount}, selectedItems:`, selectedItems);
                } else if (scope === 'total' || scope === 'aggregate') {
                    // Check across all runs INCLUDING current run
                    const allUniqueItems = new Set();
                    
                    console.log(`Processing '${scope}' scope for group '${group}'`);
                    
                    // Add historical items from all previous runs
                    Object.entries(figureHistoryByRun).forEach(([runNum, runData]) => {
                        if (runData[group]) {
                            console.log(`Adding historical items from run ${runNum} for group '${group}':`, Array.from(runData[group]));
                            runData[group].forEach(item => allUniqueItems.add(item));
                        }
                    });
                    
                    // Add current run selections
                    if (window.__FIGURE_SELECTIONS__?.[group]) {
                        console.log(`Adding current run items for group '${group}':`, Array.from(window.__FIGURE_SELECTIONS__[group]));
                        window.__FIGURE_SELECTIONS__[group].forEach(item => allUniqueItems.add(item));
                    }
                    
                    currentCount = allUniqueItems.size;
                    selectedItems = Array.from(allUniqueItems);
                    contextDesc = `Total across all runs`;
                    
                    console.log(`Final '${scope}' scope results for group '${group}': count=${currentCount}, items:`, selectedItems);
                } else if (scope === 'bestN') {
                    // Check best N runs (complex logic for best scoring runs)
                    const relevantRuns = Object.keys(runCounts)
                        .map(Number)
                        .filter(runNum => runNum <= currentRun)
                        .slice(-bestN); // Take last N runs as approximation
                    
                    const bestNSet = new Set();
                    relevantRuns.forEach(runNum => {
                        const runData = allRunsData[runNum];
                        if (runData && runData[group]) {
                            runData[group].forEach(item => bestNSet.add(item));
                        }
                    });
                    currentCount = bestNSet.size;
                    selectedItems = Array.from(bestNSet);
                    contextDesc = `Best ${bestN} runs`;
                } else {
                    // Unknown scope - treat as 'total' to include all historical + current data
                    console.warn(`Unknown scope '${scope}', treating as 'total'`);
                    
                    const allUniqueItems = new Set();
                    
                    // Add historical items from all previous runs
                    Object.entries(figureHistoryByRun).forEach(([runNum, runData]) => {
                        if (runData[group]) {
                            runData[group].forEach(item => allUniqueItems.add(item));
                        }
                    });
                    
                    // Add current run selections
                    if (window.__FIGURE_SELECTIONS__?.[group]) {
                        window.__FIGURE_SELECTIONS__[group].forEach(item => allUniqueItems.add(item));
                    }
                    
                    currentCount = allUniqueItems.size;
                    selectedItems = Array.from(allUniqueItems);
                    contextDesc = `All runs (unknown scope: ${scope})`;
                }
                
                const status = currentCount >= minDistinct;
                const shortfall = Math.max(0, minDistinct - currentCount);
                const remainingRuns = runsCount - currentRun;
                const isLastRun = currentRun >= runsCount;
                
                // Progressive validation logic - modified to never block submission
                let validationLevel = 'passed';
                let allowSubmission = true;
                
                if (!status) {
                    if (isLastRun || remainingRuns === 0) {
                        // Final run - mark as error but still allow submission
                        validationLevel = 'error';
                        if (!isOptional) {
                            ok = false; // Track compliance for display/logging but don't block
                            allowSubmission = true; // Always allow submission
                        }
                    } else {
                        // Earlier runs - warning but allow submission
                        validationLevel = 'warning';
                        allowSubmission = true;
                        // Don't set ok = false for earlier runs
                    }
                } else {
                    validationLevel = 'success';
                }
                
                // Build detailed validation message
                let validationInfo = {
                    group: group,
                    ruleType: ruleType,
                    scope: scope,
                    contextDesc: contextDesc,
                    status: status,
                    validationLevel: validationLevel,
                    isOptional: isOptional,
                    currentCount: currentCount,
                    minDistinct: minDistinct,
                    shortfall: shortfall,
                    selectedItems: selectedItems,
                    availableItems: availableInCategory,
                    remainingRuns: remainingRuns,
                    isLastRun: isLastRun,
                    runInfo: `Run ${currentRun} of ${runsCount}`,
                    allowSubmission: allowSubmission
                };
                
                if (validationLevel === 'error') {
                    validationErrors.push(validationInfo);
                } else if (validationLevel === 'warning') {
                    validationWarnings.push(validationInfo);
                } else {
                    validationPassed.push(validationInfo);
                }
            }

            // Build comprehensive validation display
            if (validationErrors.length > 0) {
                msgs.push(`
                    <div class="alert alert-danger">
                        <h6 class="alert-heading mb-2">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            Diversity Requirements Not Met
                        </h6>
                        <div class="small mb-2">
                            ${validationErrors.some(info => info.isLastRun) ? 
                                'This is the final run. All diversity requirements must be satisfied before submission.' : 
                                'The following requirements must be satisfied before submission:'}
                        </div>
                `);
                
                validationErrors.forEach(info => {
                    const suggestions = info.availableItems.filter(item => !info.selectedItems.includes(item)).slice(0, 5);
                    msgs.push(`
                        <div class="border rounded p-2 mb-2 bg-light">
                            <div class="d-flex justify-content-between align-items-start mb-1">
                                <strong class="text-danger">${info.group}</strong>
                                <span class="badge bg-danger">${info.currentCount}/${info.minDistinct}</span>
                            </div>
                            <div class="small text-muted mb-1">
                                ${info.contextDesc} • ${info.runInfo || `Run ${currentRun}`} • 
                                ${info.isLastRun ? 'FINAL RUN - Must meet requirement!' : 
                                info.remainingRuns ? `${info.remainingRuns} run(s) remaining` : 'Need ' + info.shortfall + ' more distinct items'}
                            </div>
                            ${info.selectedItems.length > 0 ? `
                                <div class="small mb-1">
                                    <strong>Selected:</strong> ${info.selectedItems.join(', ')}
                                </div>
                            ` : '<div class="small mb-1 text-muted">No items selected yet</div>'}
                            ${suggestions.length > 0 ? `
                                <div class="small">
                                    <strong>Suggestions:</strong> ${suggestions.join(', ')}${info.availableItems.length > 5 ? ` (+${info.availableItems.length - 5} more)` : ''}
                                </div>
                            ` : ''}
                        </div>
                    `);
                });
                
                msgs.push('</div>');
            }

            if (validationWarnings.length > 0) {
                msgs.push(`
                    <div class="alert alert-warning">
                        <h6 class="alert-heading mb-2">
                            <i class="fas fa-info-circle me-2"></i>
                            Optional Diversity Progress
                        </h6>
                        <div class="small mb-2">You can still submit, but consider meeting these requirements in remaining runs.</div>
                `);
                
                validationWarnings.forEach(info => {
                    const suggestions = info.availableItems.filter(item => !info.selectedItems.includes(item)).slice(0, 3);
                    msgs.push(`
                        <div class="d-flex justify-content-between align-items-center mb-1">
                            <div>
                                <strong>${info.group}</strong> <small class="text-muted">(${info.contextDesc})</small>
                                ${suggestions.length > 0 ? `<br><small class="text-muted">Try: ${suggestions.join(', ')}</small>` : ''}
                            </div>
                            <span class="badge bg-warning ">${info.currentCount}/${info.minDistinct}</span>
                        </div>
                    `);
                });
                
                msgs.push('</div>');
            }

            if (validationPassed.length > 0) {
                msgs.push(`
                    <div class="alert alert-success">
                        <h6 class="alert-heading mb-2">
                            <i class="fas fa-check-circle me-2"></i>
                            Diversity Requirements Satisfied
                        </h6>
                `);
                
                validationPassed.forEach(info => {
                    msgs.push(`
                        <div class="d-flex justify-content-between align-items-center mb-1">
                            <div>
                                <strong>${info.group}</strong> <small class="text-muted">(${info.contextDesc})</small>
                                ${info.selectedItems.length > 0 ? `<br><small class="text-success">Selected: ${info.selectedItems.slice(0, 5).join(', ')}${info.selectedItems.length > 5 ? ` (+${info.selectedItems.length - 5} more)` : ''}</small>` : ''}
                            </div>
                            <span class="badge bg-success">${info.currentCount}/${info.minDistinct} ✓</span>
                        </div>
                    `);
                });
                
                msgs.push('</div>');
            }

            // Show historical context
            if (Object.keys(figureHistoryByRun).length > 0) {
                const pastRuns = Object.keys(figureHistoryByRun).map(Number).sort();
                msgs.push(`
                    <div class="alert alert-info">
                        <div class="d-flex align-items-center mb-2">
                            <i class="fas fa-history me-2"></i>
                            <strong>Including Previous Runs: ${pastRuns.join(', ')}</strong>
                        </div>
                        <div class="small">
                            Validation includes figures selected in previous runs for this participant.
                            ${currentRun <= runsCount ? `Current: Run ${currentRun} of ${runsCount}` : ''}
                        </div>
                    </div>
                `);
            } else if (currentRun === 1) {
                msgs.push(`
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>First Run:</strong> No previous figure history available. 
                        Select figures according to diversity requirements for this run.
                    </div>
                `);
            }

            // Overall validation status with progressive logic
            const isLastRun = currentRun >= runsCount;
            const remainingRuns = runsCount - currentRun;
            
            if (ok) {
                if (validationWarnings.length > 0) {
                    msgs.unshift(`
                        <div class="alert alert-success border-success">
                            <i class="fas fa-check-circle me-2"></i>
                            <strong>Ready to submit!</strong> All requirements met or can be completed in remaining runs.
                            ${!isLastRun ? `<div class="small mt-1">Run ${currentRun} of ${runsCount} • ${remainingRuns} run(s) remaining</div>` : ''}
                        </div>
                    `);
                } else {
                    msgs.unshift(`
                        <div class="alert alert-success border-success">
                            <i class="fas fa-shield-check me-2"></i>
                            <strong>All diversity requirements met!</strong> You can proceed with submission.
                            ${!isLastRun ? `<div class="small mt-1">Run ${currentRun} of ${runsCount} • Excellent progress!</div>` : ''}
                        </div>
                    `);
                }
            } else {
                msgs.unshift(`
                    <div class="alert alert-danger border-danger">
                        <i class="fas fa-shield-alt me-2"></i>
                        <strong>${isLastRun ? 'Final run - Requirements must be met!' : 'Diversity validation failed!'}</strong> 
                        ${isLastRun ? 'This is your last chance to meet all requirements.' : 'Please address the requirements below before submission.'}
                        <div class="small mt-1">Run ${currentRun} of ${runsCount}</div>
                    </div>
                `);
            }

            outBox.style.display = 'block';
            outBox.style.visibility = 'visible';
            outBox.style.opacity = '1';
            outBox.innerHTML = msgs.join('');
            
            // Force a DOM update/reflow
            outBox.offsetHeight;
            
            // Store detailed validation results for API usage
            window.__DIVERSITY_VALIDATION_DETAILS__ = {
                overallCompliant: ok,
                validationErrors: validationErrors,
                validationWarnings: validationWarnings,
                validationPassed: validationPassed,
                rulesChecked: FIGURE_UI.diversityRules.length,
                currentRun: currentRun,
                totalRuns: runsCount,
                timestamp: Date.now()
            };
            
            // Always return true to never block form submission
            // The validation state is stored in window.__DIVERSITY_VALIDATION_STATE__ for submission
            return true;
        }

        // Heat Flow API Integration
        let heatFlowData = null;
        let autoModeEnabled = true;
        let manualOverride = false;
        let heatFlowInterval = null;
        let heatFlowPreviewInterval = null;
        let lastActiveHeat = null;
        let lastActiveRun = null;
        let isInitialPageLoad = true; // Track if this is the first load to auto-sync

        // Overlay state tracking to prevent unnecessary refreshes
        let overlayState = {
            isVisible: false,
            lastBracketDataHash: null, // Hash of bracket data to detect changes
            hasLoadedBracket: false
        };

        // Simple hash function for comparing bracket data
        function hashBracketData(data) {
            if (!data || !data.flow_chains) return null;
            
            // Create a simple hash from heat numbers, active states, and participant counts
            const heats = [];
            data.flow_chains.forEach(chain => {
                chain.forEach(heat => {
                    heats.push(`${heat.heat_number}:${heat.is_active ? 1 : 0}:${heat.participants?.length || 0}`);
                });
            });
            return heats.sort().join('|');
        }

        // Initialize heat flow monitoring
        function initializeHeatFlow() {
            const eventId = <?= json_encode($event_id) ?>;
            if (!eventId) {
                console.log('No event ID available for heat flow monitoring');
                return;
            }
            
            // Load initial data
            loadHeatFlowData();
            
            // Set up auto-refresh every 5 seconds
            if (heatFlowInterval) clearInterval(heatFlowInterval);
            heatFlowInterval = setInterval(loadHeatFlowData, 5000);
            
            // Set up auto mode toggle
            const autoToggle = document.getElementById('autoModeToggle');
            if (autoToggle) {
                autoToggle.addEventListener('change', function() {
                    autoModeEnabled = this.checked;
                    updateAutoModeDisplay();
                    
                    if (autoModeEnabled && !manualOverride) {
                        applyActiveHeatRun();
                    }
                });
            }
        }

        async function loadHeatFlowData() {
            const eventId = <?= json_encode($event_id) ?>;
            if (!eventId) return;
            
            try {
                updateConnectionStatus('connecting');
                
                const response = await fetch(`../admin/heat_flow_api.php?event_id=${eventId}`);
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}`);
                }
                
                const data = await response.json();
                heatFlowData = data;
                
                updateConnectionStatus('connected');
                updateLastUpdateTime();
                
                // Process active heat data
                const activeHeat = findActiveHeat(data);
                
                if (activeHeat) {
                    updateActiveHeatDisplay(activeHeat);
                    hideNoActiveHeatOverlay();
                    
                    // Auto-apply if enabled and no manual override
                    if (autoModeEnabled && !manualOverride) {
                        applyActiveHeatRun(activeHeat);
                    } else {
                        // Check if current selection differs from active (manual override or initial load mismatch)
                        checkIfDifferentFromActiveHeat();
                    }
                } else {
                    // No active heat - show overlay
                    clearActiveHeatDisplay();
                    
                    // Calculate hash of current bracket data
                    const currentHash = hashBracketData(data);
                    
                    // Only refresh bracket if:
                    // 1. Overlay is not yet visible (first time showing), OR
                    // 2. Bracket data has actually changed
                    const shouldRefreshBracket = !overlayState.isVisible || 
                                                !overlayState.hasLoadedBracket ||
                                                currentHash !== overlayState.lastBracketDataHash;
                    
                    if (shouldRefreshBracket) {
                        console.log('Bracket data changed or initial load, refreshing overlay...');
                        overlayState.lastBracketDataHash = currentHash;
                        showNoActiveHeatOverlay(true); // Pass true to force bracket reload
                    } else {
                        console.log('Bracket data unchanged, keeping existing overlay...');
                        showNoActiveHeatOverlay(false); // Pass false to skip bracket reload
                    }
                }
                
            } catch (error) {
                console.error('Error loading heat flow data:', error);
                updateConnectionStatus('error');
            }
        }

        function findActiveHeat(data) {
            if (!data.flow_chains) return null;
            
            for (const chain of data.flow_chains) {
                for (const heat of chain) {
                    if (heat.is_active) {
                        return heat;
                    }
                }
            }
            return null;
        }

        function updateActiveHeatDisplay(activeHeat) {
            const infoDiv = document.getElementById('activeHeatInfo');
            const heatDisplay = document.getElementById('activeHeatDisplay');
            const heatName = document.getElementById('activeHeatName');
            const runDisplay = document.getElementById('activeRunDisplay');
            const progressDiv = document.getElementById('activeHeatProgress');
            const progressText = document.getElementById('progressText');
            const progressPercentage = document.getElementById('progressPercentage');
            const progressBar = document.getElementById('progressBar');
            
            // Check if required elements exist
            if (!infoDiv || !heatDisplay || !heatName || !runDisplay) {
                console.warn('Active heat display elements not found in DOM');
                return;
            }
            
            if (!activeHeat) {
                infoDiv.style.display = 'none';
                return;
            }
            
            // Update heat info
            heatDisplay.textContent = `Heat ${activeHeat.heat_number}`;
            heatName.textContent = activeHeat.heat_name || 'No name';
            runDisplay.textContent = `Run ${activeHeat.active_run || 1}`;
            
            // Update progress if available and progress elements exist
            if (activeHeat.progress && progressDiv && progressText && progressPercentage && progressBar) {
                const progress = activeHeat.progress;
                progressText.textContent = `${progress.completed_participants} / ${progress.total_participants}`;
                progressPercentage.textContent = `${progress.completion_percentage}%`;
                progressBar.style.width = `${progress.completion_percentage}%`;
                progressDiv.style.display = 'block';
            } else if (progressDiv) {
                progressDiv.style.display = 'none';
            }
            
            infoDiv.style.display = 'block';
            
            // Update summary badge
            updateHeatRunSummary();
            
            // Store for comparison
            lastActiveHeat = activeHeat.heat_number;
            lastActiveRun = activeHeat.active_run || 1;
        }

        function clearActiveHeatDisplay() {
            const infoDiv = document.getElementById('activeHeatInfo');
            if (infoDiv) {
                infoDiv.style.display = 'none';
            }
            lastActiveHeat = null;
            lastActiveRun = null;
        }

        // Show/hide no active heat overlay
        function showNoActiveHeatOverlay(forceRefresh = false) {
            const overlay = document.getElementById('noActiveHeatOverlay');
            if (overlay) {
                const wasVisible = overlayState.isVisible;
                
                overlay.classList.add('show');
                overlayState.isVisible = true;
                
                // Only load bracket preview if:
                // 1. This is the first time showing the overlay, OR
                // 2. forceRefresh is explicitly true (data changed)
                if (!wasVisible || forceRefresh) {
                    console.log('Loading bracket preview...');
                    loadHeatFlowPreview();
                } else {
                    console.log('Overlay already visible with current data, skipping refresh');
                }
            }
        }

        function hideNoActiveHeatOverlay() {
            const overlay = document.getElementById('noActiveHeatOverlay');
            if (overlay) {
                overlay.classList.remove('show');
                overlayState.isVisible = false;
                overlayState.hasLoadedBracket = false; // Reset for next time
                
                // Stop refreshing heat flow preview
                if (heatFlowPreviewInterval) {
                    clearInterval(heatFlowPreviewInterval);
                    heatFlowPreviewInterval = null;
                }
            }
        }

        // Load simplified heat flow bracket for overlay
        async function loadHeatFlowPreview() {
            console.log('🎯 loadHeatFlowPreview() called');
            const eventId = <?= json_encode($event_id) ?>;
            if (!eventId) return;
            
            const container = document.getElementById('heatFlowPreview');
            if (!container) {
                console.error('❌ heatFlowPreview container not found!');
                return;
            }
            
            console.log('📦 Container found, loading bracket preview for event:', eventId);
            
            try {
                container.innerHTML = '<div class="text-center text-muted py-3"><div class="spinner-border spinner-border-sm me-2" role="status"></div><span>Loading heat flow...</span></div>';
                
                const response = await fetch(`../api/heat_bracket_view_api.php?event_id=${eventId}&compact=1`);
                if (!response.ok) throw new Error('Failed to load heat flow');
                
                const data = await response.json();
                
                if (!data.success) {
                    throw new Error(data.error || 'Unknown error');
                }
                
                if (!data.html) {
                    container.innerHTML = `
                        <div class="text-center text-muted py-3">
                            <i class="fas fa-exclamation-circle me-2"></i>
                            <span>No heats configured for this event</span>
                        </div>
                    `;
                    overlayState.hasLoadedBracket = true;
                    return;
                }
                
                // Insert the generated HTML
                container.innerHTML = '<h6><i class="fas fa-stream me-2"></i>Event Heat Flow</h6>' + data.html;
                
                // Store connections for drawing
                window.heatFlowConnections = data.connections;
                
                // Draw SVG connections after DOM is updated
                setTimeout(() => {
                    console.log('🔧 Post-load processing started');
                    
                    if (window.heatFlowConnections) {
                        drawBracketConnections(window.heatFlowConnections);
                    }
                    
                    // Check if Bootstrap is available
                    if (typeof bootstrap === 'undefined') {
                        console.error('❌ Bootstrap is not loaded! Tooltips will not work.');
                        return;
                    }
                    
                    // Dispose any existing tooltips first
                    const existingTooltips = document.querySelectorAll('#heatFlowPreview [data-bs-toggle="tooltip"]');
                    console.log('🗑️ Found', existingTooltips.length, 'existing tooltip elements to check');
                    existingTooltips.forEach(el => {
                        const tooltip = bootstrap.Tooltip.getInstance(el);
                        if (tooltip) {
                            tooltip.dispose();
                            console.log('🗑️ Disposed tooltip');
                        }
                    });
                    
                    // Initialize Bootstrap tooltips for bracket cards
                    const tooltipTriggerList = document.querySelectorAll('#heatFlowPreview [data-bs-toggle="tooltip"]');
                    console.log('💡 Initializing tooltips for', tooltipTriggerList.length, 'elements');
                    
                    if (tooltipTriggerList.length === 0) {
                        console.warn('⚠️ No elements with data-bs-toggle="tooltip" found in #heatFlowPreview');
                    }
                    
                    tooltipTriggerList.forEach((tooltipTriggerEl, index) => {
                        try {
                            const tooltip = new bootstrap.Tooltip(tooltipTriggerEl, {
                                html: true,
                                trigger: 'hover',
                                delay: { show: 300, hide: 100 },
                                customClass: 'heat-bracket-tooltip'
                            });
                            console.log(`✅ Tooltip ${index + 1} initialized for heat card:`, tooltipTriggerEl.dataset.heatId);
                        } catch (error) {
                            console.error('❌ Error initializing tooltip:', error);
                        }
                    });
                    
                    console.log('🔧 Post-load processing completed');
                }, 100);
                
                // Mark bracket as loaded
                overlayState.hasLoadedBracket = true;
                console.log('Bracket preview loaded successfully');
                
            } catch (error) {
                console.error('Error loading heat flow preview:', error);
                container.innerHTML = `
                    <div class="text-center text-danger py-3">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <span>Failed to load heat flow</span>
                    </div>
                `;
                overlayState.hasLoadedBracket = true; // Mark as loaded even on error to prevent retries
            }
        }

        // Draw SVG connections between heats
        function drawBracketConnections(connections) {
            const svg = document.querySelector('#heatFlowPreview .bracket-connections');
            if (!svg || !connections || connections.length === 0) return;
            
            const container = svg.closest('.bracket-flow-container');
            if (!container) return;
            
            const containerRect = container.getBoundingClientRect();
            
            // Clear existing paths
            const existingPaths = svg.querySelectorAll('path, line');
            existingPaths.forEach(p => p.remove());
            
            // Group connections by destination (for bracket-style)
            const connsByDestination = {};
            connections.forEach(conn => {
                if (!connsByDestination[conn.to]) {
                    connsByDestination[conn.to] = [];
                }
                connsByDestination[conn.to].push(conn.from);
            });
            
            // Draw bracket-style connections
            Object.keys(connsByDestination).forEach(toHeat => {
                const fromHeats = connsByDestination[toHeat];
                const toCard = container.querySelector(`[data-heat-id="${toHeat}"]`);
                if (!toCard) return;
                
                const toRect = toCard.getBoundingClientRect();
                const x2 = toRect.left - containerRect.left;
                const y2 = toRect.top + toRect.height / 2 - containerRect.top;
                
                if (fromHeats.length === 1) {
                    // Single connection - straight line
                    const fromCard = container.querySelector(`[data-heat-id="${fromHeats[0]}"]`);
                    if (!fromCard) return;
                    
                    const fromRect = fromCard.getBoundingClientRect();
                    const x1 = fromRect.right - containerRect.left;
                    const y1 = fromRect.top + fromRect.height / 2 - containerRect.top;
                    
                    const line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
                    line.setAttribute('class', 'connection-path');
                    line.setAttribute('x1', x1);
                    line.setAttribute('y1', y1);
                    line.setAttribute('x2', x2);
                    line.setAttribute('y2', y2);
                    line.setAttribute('stroke', '#adb5bd');
                    line.setAttribute('stroke-width', '2');
                    line.setAttribute('marker-end', 'url(#arrowhead)');
                    svg.appendChild(line);
                } else {
                    // Multiple connections - bracket style
                    let minY = Infinity, maxY = -Infinity;
                    const fromPositions = [];
                    
                    fromHeats.forEach(fromHeat => {
                        const fromCard = container.querySelector(`[data-heat-id="${fromHeat}"]`);
                        if (!fromCard) return;
                        
                        const fromRect = fromCard.getBoundingClientRect();
                        const x1 = fromRect.right - containerRect.left;
                        const y1 = fromRect.top + fromRect.height / 2 - containerRect.top;
                        
                        fromPositions.push({x: x1, y: y1});
                        minY = Math.min(minY, y1);
                        maxY = Math.max(maxY, y1);
                    });
                    
                    // Calculate the midpoint for the vertical connector
                    const midY = (minY + maxY) / 2;
                    const junctionX = x2 - 30; // Junction point 30px before destination
                    
                    // Draw horizontal lines from each source to junction
                    fromPositions.forEach(pos => {
                        const line1 = document.createElementNS('http://www.w3.org/2000/svg', 'line');
                        line1.setAttribute('class', 'connection-path');
                        line1.setAttribute('x1', pos.x);
                        line1.setAttribute('y1', pos.y);
                        line1.setAttribute('x2', junctionX);
                        line1.setAttribute('y2', pos.y);
                        line1.setAttribute('stroke', '#adb5bd');
                        line1.setAttribute('stroke-width', '2');
                        svg.appendChild(line1);
                    });
                    
                    // Draw vertical connector line
                    const verticalLine = document.createElementNS('http://www.w3.org/2000/svg', 'line');
                    verticalLine.setAttribute('class', 'connection-path');
                    verticalLine.setAttribute('x1', junctionX);
                    verticalLine.setAttribute('y1', minY);
                    verticalLine.setAttribute('x2', junctionX);
                    verticalLine.setAttribute('y2', maxY);
                    verticalLine.setAttribute('stroke', '#adb5bd');
                    verticalLine.setAttribute('stroke-width', '2');
                    svg.appendChild(verticalLine);
                    
                    // Draw final horizontal line to destination with arrow
                    const finalLine = document.createElementNS('http://www.w3.org/2000/svg', 'line');
                    finalLine.setAttribute('class', 'connection-path');
                    finalLine.setAttribute('x1', junctionX);
                    finalLine.setAttribute('y1', midY);
                    finalLine.setAttribute('x2', x2);
                    finalLine.setAttribute('y2', y2);
                    finalLine.setAttribute('stroke', '#adb5bd');
                    finalLine.setAttribute('stroke-width', '2');
                    finalLine.setAttribute('marker-end', 'url(#arrowhead)');
                    svg.appendChild(finalLine);
                }
            });
        }

        // Show heat detail popover
        function showHeatDetail(heatNumber) {
            // Use data from window.heatBracketData (populated by API template)
            if (!window.heatBracketData || !window.heatBracketData.heats) return;
            
            const heatData = window.heatBracketData.heats[heatNumber];
            if (!heatData) return;
            
            // Create backdrop
            const backdrop = document.createElement('div');
            backdrop.className = 'heat-detail-backdrop';
            backdrop.onclick = closeHeatDetail;
            
            // Create popover
            const popover = document.createElement('div');
            popover.className = 'heat-detail-popover';
            popover.id = 'heatDetailPopover';
            
            const isCompleted = heatData.progress && heatData.progress.completion_percentage >= 100;
            const isActive = heatData.is_active;
            
            let statusBadge = '';
            if (isActive) {
                statusBadge = '<span class="badge bg-primary"><i class="fas fa-play-circle me-1"></i>Active Now</span>';
            } else if (isCompleted) {
                statusBadge = '<span class="badge bg-success"><i class="fas fa-check-circle me-1"></i>Completed</span>';
            } else {
                statusBadge = '<span class="badge bg-secondary">Pending</span>';
            }
            
            let participantsHtml = '<p class="text-muted small mb-0">No participants assigned</p>';
            if (heatData.participant_ids && heatData.participant_ids.length > 0) {
                participantsHtml = '<ul class="heat-detail-participants">';
                heatData.participant_ids.forEach(pid => {
                    const p = window.heatBracketData.participants[pid];
                    if (!p) return;
                    
                    const flag = p.country ? getFlagEmoji(p.country) : '';
                    participantsHtml += '<li class="heat-detail-participant">';
                    participantsHtml += '<div class="heat-detail-bib">' + (p.bib_number || '?') + '</div>';
                    participantsHtml += '<div class="heat-detail-participant-name">';
                    participantsHtml += flag + ' ' + p.first_name + ' ' + p.last_name;
                    participantsHtml += '</div>';
                    participantsHtml += '</li>';
                });
                participantsHtml += '</ul>';
            }
            
            const participantCount = heatData.participant_ids ? heatData.participant_ids.length : 0;
            
            const progressText = 'N/A'; // Progress not available in compact view
            
            let popoverHtml = '<div class="heat-detail-header">';
            popoverHtml += '<div class="heat-detail-title">';
            popoverHtml += '<div class="heat-detail-number">' + heatData.heat_number + '</div>';
            popoverHtml += '<div>';
            popoverHtml += '<div class="heat-detail-name">' + (heatData.heat_name || 'Heat ' + heatData.heat_number) + '</div>';
            popoverHtml += statusBadge;
            popoverHtml += '</div>';
            popoverHtml += '</div>';
            popoverHtml += '<button class="heat-detail-close" onclick="closeHeatDetail()">';
            popoverHtml += '<i class="fas fa-times"></i>';
            popoverHtml += '</button>';
            popoverHtml += '</div>';
            
            popoverHtml += '<div class="heat-detail-section">';
            popoverHtml += '<div class="heat-detail-section-title">';
            popoverHtml += '<i class="fas fa-info-circle me-2"></i>Heat Information';
            popoverHtml += '</div>';
            popoverHtml += '<div class="heat-detail-meta">';
            popoverHtml += '<div class="heat-detail-meta-item">';
            popoverHtml += '<i class="fas fa-users"></i>';
            popoverHtml += '<div>';
            popoverHtml += '<div class="heat-detail-meta-label">Athletes</div>';
            popoverHtml += '<div class="heat-detail-meta-value">' + participantCount + '</div>';
            popoverHtml += '</div>';
            popoverHtml += '</div>';
            popoverHtml += '<div class="heat-detail-meta-item">';
            popoverHtml += '<i class="fas fa-redo"></i>';
            popoverHtml += '<div>';
            popoverHtml += '<div class="heat-detail-meta-label">Runs</div>';
            popoverHtml += '<div class="heat-detail-meta-value">' + (heatData.runs_count || 1) + '</div>';
            popoverHtml += '</div>';
            popoverHtml += '</div>';
            popoverHtml += '</div>';
            popoverHtml += '</div>';
            
            popoverHtml += '<div class="heat-detail-section">';
            popoverHtml += '<div class="heat-detail-section-title">';
            popoverHtml += '<i class="fas fa-list me-2"></i>Participants';
            popoverHtml += '</div>';
            popoverHtml += participantsHtml;
            popoverHtml += '</div>';
            
            popover.innerHTML = popoverHtml;
            
            document.body.appendChild(backdrop);
            document.body.appendChild(popover);
        }

        // Close heat detail popover
        function closeHeatDetail() {
            const backdrop = document.querySelector('.heat-detail-backdrop');
            const popover = document.getElementById('heatDetailPopover');
            
            if (backdrop) backdrop.remove();
            if (popover) popover.remove();
        }

        function applyActiveHeatRun(activeHeat = null) {
            if (!autoModeEnabled || manualOverride) return;
            
            const heat = activeHeat || (heatFlowData ? findActiveHeat(heatFlowData) : null);
            if (!heat) return;
            
            const heatSelect = document.getElementById('heatSelect');
            const currentHeat = heatSelect.value;
            const currentRun = document.querySelector('[name=run_number]:checked')?.value;
            
            // Only update if different from current selection
            if (currentHeat != heat.heat_number || currentRun != (heat.active_run || 1)) {
                // Set judge status to pending when auto-syncing to new heat/run
                setStatusToPending();
                
                // Show fullscreen notification about the change (heat.heat_name comes from API)
                showHeatRunChangeOverlay(heat.heat_number, heat.active_run || 1, heat.heat_name || null);
                
                // Update heat selection
                heatSelect.value = heat.heat_number;
                
                // Update run selection
                const runInput = document.getElementById(`run${heat.active_run || 1}`);
                if (runInput) {
                    runInput.checked = true;
                }
                
                // Clear current participant selection
                clearCurrentParticipant();
                
                // Load participants for the new selection (with delay for notification, and force reload)
                setTimeout(() => {
                    loadParticipants(true);
                }, 100);
                
                // Save the new selection
                saveHeatRunSelection();
                updateHeatRunSummary();
                
                // Show auto indicator briefly
                showAutoModeIndicator();
            }
        }

        function handleHeatChange() {
            manualOverride = true;
            updateAutoModeDisplay();
            checkIfDifferentFromActiveHeat(); // Check if manual selection differs from active
            
            // Set judge status to pending on heat change
            setStatusToPending();
            
            // Get current heat and run values
            const heatSelect = document.getElementById('heatSelect');
            const heat = heatSelect.value;
            const run = document.querySelector('[name=run_number]:checked')?.value || 1;
            
            // Get heat name from selected option
            const selectedOption = heatSelect.options[heatSelect.selectedIndex];
            const heatName = selectedOption ? selectedOption.text.replace(/^Heat \d+ – /, '').replace(/^Heat \d+$/, '') : null;
            
            // Collapse the heat/run panel
            const heatRunCollapseBody = document.getElementById('heatRunCollapseBody');
            if (heatRunCollapseBody && heatRunCollapseBody.classList.contains('show')) {
                const bsCollapse = bootstrap.Collapse.getInstance(heatRunCollapseBody);
                if (bsCollapse) {
                    bsCollapse.hide();
                } else {
                    new bootstrap.Collapse(heatRunCollapseBody, { toggle: true });
                }
            }
            
            // Show fullscreen notification
            showHeatRunChangeOverlay(heat, run, heatName);
            
            // Clear current participant selection
            clearCurrentParticipant();
            
            // Load participants with force reload to trigger auto-selection (with delay for notification)
            setTimeout(() => {
                loadParticipants(true);
            }, 100);
            
            saveHeatRunSelection();
            updateHeatRunSummary();
        }

        function handleRunChange() {
            manualOverride = true;
            updateAutoModeDisplay();
            checkIfDifferentFromActiveHeat(); // Check if manual selection differs from active
            
            // Set judge status to pending on run change
            setStatusToPending();
            
            // Get current heat and run values
            const heatSelect = document.getElementById('heatSelect');
            const heat = heatSelect.value;
            const run = document.querySelector('[name=run_number]:checked')?.value || 1;
            
            // Get heat name from selected option
            const selectedOption = heatSelect.options[heatSelect.selectedIndex];
            const heatName = selectedOption ? selectedOption.text.replace(/^Heat \d+ – /, '').replace(/^Heat \d+$/, '') : null;
            
            // Collapse the heat/run panel
            const heatRunCollapseBody = document.getElementById('heatRunCollapseBody');
            if (heatRunCollapseBody && heatRunCollapseBody.classList.contains('show')) {
                const bsCollapse = bootstrap.Collapse.getInstance(heatRunCollapseBody);
                if (bsCollapse) {
                    bsCollapse.hide();
                } else {
                    new bootstrap.Collapse(heatRunCollapseBody, { toggle: true });
                }
            }
            
            // Show fullscreen notification
            showHeatRunChangeOverlay(heat, run, heatName);
            
            // Clear current participant selection
            clearCurrentParticipant();
            
            // Load participants with force reload to trigger auto-selection (with delay for notification)
            setTimeout(() => {
                loadParticipants(true);
            }, 100);
            
            saveHeatRunSelection();
            updateHeatRunSummary();
            
            // Refresh figure history display when run changes
            if (currentParticipantId && FIGURE_DATA.available) {
                // Clear historical markings first
                document.querySelectorAll('.figure-chip.has-history').forEach(chip => {
                    chip.classList.remove('has-history');
                    // Restore original text
                    const originalText = chip.getAttribute('data-original-text');
                    if (originalText) {
                        chip.textContent = originalText;
                    }
                    chip.removeAttribute('data-runs');
                    chip.removeAttribute('data-run');
                });
                
                // Load existing figure selections for current score
                loadExistingFigureSelections();
                
                // Then update historical markings (with delay to ensure selections are loaded)
                setTimeout(() => {
                    displayFigureHistory();
                    updateDiversityValidationWithHistory();
                }, 100);
            }
        }

        // Update Heat/Run summary badge in collapsed header
        function updateHeatRunSummary() {
            const heatSelect = document.getElementById('heatSelect');
            const runButtons = document.querySelectorAll('input[name="run_number"]:checked');
            const progressBar = document.getElementById('progressBar');
            const progressPercentage = document.getElementById('progressPercentage');
            
            const heat = heatSelect ? heatSelect.value : '--';
            const run = runButtons.length > 0 ? runButtons[0].value : '--';
            const progress = progressPercentage ? progressPercentage.textContent : '--';
            
            const summaryBadge = document.getElementById('heatRunSummary');
            if (summaryBadge) {
                summaryBadge.textContent = 'Heat ' + heat + ' / Run ' + run + ' / ' + progress;
            }
        }

        function updateAutoModeDisplay() {
            const autoStatus = document.getElementById('autoModeStatus');
            const manualNotice = document.getElementById('manualOverrideNotice');
            const autoIndicator = document.getElementById('autoModeIndicator');
            
            // Only update elements that exist
            if (autoModeEnabled && !manualOverride) {
                if (autoStatus) autoStatus.style.display = 'block';
                if (manualNotice) manualNotice.style.display = 'none';
                if (autoIndicator) autoIndicator.style.display = 'inline-block';
            } else if (manualOverride) {
                if (autoStatus) autoStatus.style.display = 'none';
                if (manualNotice) manualNotice.style.display = 'block';
                if (autoIndicator) autoIndicator.style.display = 'none';
            } else {
                if (autoStatus) autoStatus.style.display = 'none';
                if (manualNotice) manualNotice.style.display = 'none';
                if (autoIndicator) autoIndicator.style.display = 'none';
            }
        }

        function showAutoModeIndicator() {
            const indicator = document.getElementById('autoModeIndicator');
            if (indicator) {
                indicator.style.display = 'inline-block';
                indicator.classList.add('pulse-animation');
                
                setTimeout(() => {
                    if (indicator) {
                        indicator.classList.remove('pulse-animation');
                    }
                }, 2000);
            }
        }

        function updateConnectionStatus(status) {
            const statusText = document.getElementById('connectionStatus');
            const icon = document.getElementById('connectionIcon');
            
            // Check if elements exist before trying to update them
            if (!statusText || !icon) {
                console.warn('Connection status elements not found in DOM');
                return;
            }
            
            switch (status) {
                case 'connected':
                    statusText.textContent = 'Connected';
                    statusText.className = 'text-success';
                    icon.className = 'fas fa-wifi text-success me-1';
                    break;
                case 'connecting':
                    statusText.textContent = 'Updating...';
                    statusText.className = 'text-info';
                    icon.className = 'fas fa-sync-alt fa-spin text-info me-1';
                    break;
                case 'error':
                    statusText.textContent = 'Connection Error';
                    statusText.className = 'text-danger';
                    icon.className = 'fas fa-wifi text-danger me-1';
                    break;
                default:
                    statusText.textContent = 'Unknown';
                    statusText.className = 'text-muted';
                    icon.className = 'fas fa-question text-muted me-1';
            }
        }

        function updateLastUpdateTime() {
            const timeElement = document.getElementById('lastUpdateTime');
            if (timeElement) {
                const now = new Date();
                timeElement.textContent = now.toLocaleTimeString();
            }
        }

        function forceRefreshHeatData() {
            // Reset manual override to allow auto selection
            manualOverride = false;
            updateAutoModeDisplay();
            
            // Hide the manual override banner
            hideManualOverrideBanner();
            
            // Force immediate refresh
            loadHeatFlowData();
            
            // Show feedback
            showToast('Refreshed', 'Heat data refreshed and auto mode restored', 'info');
        }

        // Participant info strip helper functions
        function updateCurrentParticipantDisplay() {
            console.log('updateCurrentParticipantDisplay called');
            const participantSelect = document.getElementById('ep_id');
            const selectedOption = participantSelect?.options[participantSelect.selectedIndex];
            
            console.log('Selected option:', selectedOption);
            console.log('Participant value:', participantSelect?.value);
            
            if (!selectedOption || !participantSelect.value) {
                clearCurrentParticipant();
                return;
            }
            
            // Get data from option attributes
            const bib = selectedOption.getAttribute('data-bib');
            const firstName = selectedOption.getAttribute('data-firstname') || '';
            const lastName = selectedOption.getAttribute('data-lastname') || '';
            const country = selectedOption.getAttribute('data-country') || '';
            
            console.log('Data attributes:', { bib, firstName, lastName, country });
            
            // Update display elements
            document.getElementById('current-participant-bib').textContent = bib || '--';
            document.getElementById('current-participant-name').textContent = (firstName + ' ' + lastName).trim() || '--';
            
            // Update country and flag
            const countrySpan = document.getElementById('current-participant-nationality');
            const flagSpan = document.getElementById('current-participant-flag');
            
            if (country && country !== '--') {
                countrySpan.textContent = country;
                // Add country flag emoji if available
                const flagEmoji = getFlagEmoji(country);
                flagSpan.textContent = flagEmoji ? flagEmoji + ' ' : '';
            } else {
                countrySpan.textContent = '--';
                flagSpan.textContent = '';
            }
            
            // Show the info card
            document.getElementById('current-participant-info').style.display = 'block';
            console.log('Participant info card should now be visible');
        }

        // Clear current participant display
        function clearCurrentParticipant() {
            const participantSelect = document.getElementById('ep_id');
            if (participantSelect) {
                participantSelect.value = '';
            }
            
            const bibInput = document.getElementById('bibInput');
            if (bibInput) {
                bibInput.value = '';
            }
            
            document.getElementById('current-participant-info').style.display = 'none';
            document.getElementById('current-participant-bib').textContent = '--';
            document.getElementById('current-participant-name').textContent = '--';
            document.getElementById('current-participant-nationality').textContent = '--';
            document.getElementById('current-participant-flag').textContent = '';
            document.getElementById('current-participant-score').textContent = '--';

        scrollTo(0, 0);
            clearScoreInputsOnly();
        }

        // Update current participant score display
        function updateCurrentParticipantScore() {
            const scoreInput = document.getElementById('score_value');
            const scoreDisplay = document.getElementById('current-participant-score');
            
            if (scoreInput && scoreDisplay) {
                const scoreValue = scoreInput.value || '--';
                scoreDisplay.textContent = scoreValue;
            }
        }

        // Convert country code to flag emoji
        function getFlagEmoji(countryCode) {
            if (!countryCode || countryCode.length < 2) return '';
            
            // Handle 2-letter codes directly
            const code = countryCode.substring(0, 2).toUpperCase();
            
            // Convert to flag emoji
            const codePoints = [...code].map(char => 127397 + char.charCodeAt(0));
            return String.fromCodePoint(...codePoints);
        }

        // Judge Status Functions
        let currentJudgeStatus = 'ready';
        let judgeStatusCollapse = null;

        function setJudgeStatus(status) {
            // Update current status
            currentJudgeStatus = status;
            
            // Update UI
            updateJudgeStatusUI(status);
            
            // Save to localStorage
            saveJudgeStatus(status);
            
            // Send to server
            sendJudgeStatusToServer(status);
            
            // Collapse the status options
            if (judgeStatusCollapse) {
                judgeStatusCollapse.hide();
            }
            
            // Show confirmation
            const statusLabels = {
                'ready': '✓ Ready',
                'pending': '⏱ Pending',
                'danger': '⚠ Danger',
                'warning': '⚡ Warning',
                'info': 'ℹ Info',
                'off': '⏻ Off'
            };
            showToast('Status Updated', 'Your status has been changed to: ' + statusLabels[status], 'success', 3000);
        }

        function setStatusToPending() {
            // Silently set status to pending without showing toast notification
            // This is called automatically on heat/run/participant changes
            currentJudgeStatus = 'pending';
            updateJudgeStatusUI('pending');
            saveJudgeStatus('pending');
            sendJudgeStatusToServer('pending');
            
            // Collapse the status options if open
            if (judgeStatusCollapse) {
                judgeStatusCollapse.hide();
            }
            
            // Check if Auto Ready is enabled and automatically switch to ready
            checkAutoReady();
        }

        function setStatusToInfo() {
            // Silently set status to info without showing toast notification
            // This is called when judge starts entering score data
            currentJudgeStatus = 'info';
            updateJudgeStatusUI('info');
            saveJudgeStatus('info');
            sendJudgeStatusToServer('info');
            
            // Collapse the status options if open
            if (judgeStatusCollapse) {
                judgeStatusCollapse.hide();
            }
        }

        // Auto Ready Functions
        function handleAutoReadyToggle() {
            const autoReadyToggle = document.getElementById('autoReadyToggle');
            const isEnabled = autoReadyToggle.checked;
            
            // Save to localStorage
            localStorage.setItem('judgeAutoReady', isEnabled ? 'true' : 'false');
            
            // If enabled and currently pending, automatically switch to ready
            if (isEnabled && currentJudgeStatus === 'pending') {
                setJudgeStatus('ready');
            }
            
            console.log('Auto Ready:', isEnabled ? 'Enabled' : 'Disabled');
        }

        function loadAutoReadyPreference() {
            const autoReadyToggle = document.getElementById('autoReadyToggle');
            const savedPreference = localStorage.getItem('judgeAutoReady');
            
            if (savedPreference === 'true') {
                autoReadyToggle.checked = true;
            } else {
                autoReadyToggle.checked = false;
            }
        }

        function checkAutoReady() {
            // Check if Auto Ready is enabled
            const autoReadyToggle = document.getElementById('autoReadyToggle');
            
            if (autoReadyToggle && autoReadyToggle.checked && currentJudgeStatus === 'pending') {
                // Automatically switch to ready
                setTimeout(() => {
                    setJudgeStatus('ready');
                }, 500); // Small delay to allow pending state to be visible briefly
            }
        }

        function updateJudgeStatusUI(status) {
            const statusCard = document.getElementById('judge-status-card');
            const statusHeader = document.getElementById('judgeStatusHeader');
            const statusIcon = document.getElementById('judgeStatusIcon');
            const statusText = document.getElementById('judgeStatusText');
            const statusActions = document.getElementById('judgeStatusActions');
            const scoringCardsContainer = document.getElementById('scoring-cards-container');
            
            // Status configurations
            const configs = {
                'ready': { 
                    bg: 'bg-success-subtle', 
                    icon: 'fa-check-circle text-success', 
                    text: 'Status: Ready',
                    action: 'expand'
                },
                'pending': { 
                    bg: 'bg-warning-subtle', 
                    icon: 'fa-clock text-warning', 
                    text: 'Status: Pending',
                    action: 'ready'
                },
                'danger': { 
                    bg: 'bg-danger-subtle', 
                    icon: 'fa-exclamation-triangle text-danger', 
                    text: 'Status: Danger',
                    action: 'expand'
                },
                'warning': { 
                    bg: 'bg-warning-subtle', 
                    icon: 'fa-exclamation-circle text-warning', 
                    text: 'Status: Warning',
                    action: 'expand'
                },
                'info': { 
                    bg: 'bg-info-subtle', 
                    icon: 'fa-info-circle text-info', 
                    text: 'Status: Info',
                    action: 'expand'
                },
                'off': { 
                    bg: 'bg-secondary-subtle', 
                    icon: 'fa-power-off text-secondary', 
                    text: 'Status: Off',
                    action: 'expand'
                }
            };
            
            const config = configs[status];
            
            // Remove all bg classes
            statusHeader.classList.remove('bg-success-subtle', 'bg-warning-subtle', 'bg-danger-subtle', 'bg-info-subtle', 'bg-secondary-subtle');
            statusHeader.classList.add(config.bg);
            
            // Special handling for pending status - bigger, more prominent, sticky, hide scoring cards
            if (status === 'pending') {
                // Make card sticky
                statusCard.classList.add('sticky-top', 'position-sticky');
                statusCard.style.top = '80px';
                statusCard.style.zIndex = '1000';
                
                // Enhanced pending state with bigger card
                statusCard.classList.add('border-warning', 'shadow');
                statusHeader.classList.remove('p-2');
                statusHeader.classList.add('p-4');
                
                // Update icon and text with larger styling
                statusIcon.className = 'fas fa-clock text-warning me-3';
                statusIcon.style.fontSize = '2rem';
                statusText.innerHTML = '<div style="font-size: 1.5rem; font-weight: bold;">Ready for next performance?</div>';
                statusText.style.fontSize = '';
                
                // Show prominent Ready button and expand option
                statusActions.innerHTML = `
                    <div class="d-flex gap-2">
                        <button type="button" class="btn btn-success btn-lg px-4" onclick="setJudgeStatus('ready')">
                            <i class="fas fa-check me-2"></i>READY
                        </button>
                        <button type="button" class="btn btn-outline-secondary btn-lg" data-bs-toggle="collapse" data-bs-target="#judgeStatusCollapse" title="Other status options">
                            <i class="fas fa-ellipsis-h"></i>
                        </button>
                    </div>
                `;
                
                // Hide scoring cards
                if (scoringCardsContainer) {
                    scoringCardsContainer.style.display = 'none';
                }
            } else {
                // Remove sticky positioning
                statusCard.classList.remove('sticky-top', 'position-sticky');
                statusCard.style.top = '';
                statusCard.style.zIndex = '';
                
                // Normal styling for other statuses
                statusCard.classList.remove('border-warning', 'shadow');
                statusHeader.classList.remove('p-4');
                statusHeader.classList.add('p-2');
                
                // Update icon with normal size
                statusIcon.className = 'fas ' + config.icon + ' me-2';
                statusIcon.style.fontSize = '';
                
                // Update text with normal styling
                statusText.textContent = config.text;
                statusText.style.fontSize = '';
                
                if (config.action === 'ready') {
                    // This shouldn't happen since pending is handled above, but keep for safety
                    statusActions.innerHTML = '<button type="button" class="btn btn-success btn-sm" onclick="setJudgeStatus(\'ready\')"><i class="fas fa-check me-1"></i>Ready</button>';
                } else {
                    // Show expand button for other statuses
                    statusActions.innerHTML = '<button type="button" class="btn btn-outline-secondary btn-sm" data-bs-toggle="collapse" data-bs-target="#judgeStatusCollapse"><i class="fas fa-ellipsis-v"></i></button>';
                }
                
                // Show scoring cards
                if (scoringCardsContainer) {
                    scoringCardsContainer.style.display = 'block';
                }
            }
        }

        function saveJudgeStatus(status) {
            const eventId = <?= json_encode($event_id) ?>;
            const judgeId = <?= json_encode($judge_id) ?>;
            const key = 'judge_status_' + eventId + '_' + judgeId;
            localStorage.setItem(key, status);
        }

        function loadJudgeStatus() {
            const eventId = <?= json_encode($event_id) ?>;
            const judgeId = <?= json_encode($judge_id) ?>;
            const key = 'judge_status_' + eventId + '_' + judgeId;
            const savedStatus = localStorage.getItem(key) || 'ready';
            
            currentJudgeStatus = savedStatus;
            updateJudgeStatusUI(savedStatus);
            
            // Initialize Bootstrap collapse
            const collapseElement = document.getElementById('judgeStatusCollapse');
            if (collapseElement) {
                judgeStatusCollapse = new bootstrap.Collapse(collapseElement, { toggle: false });
            }
            
            // Load Auto Ready preference
            loadAutoReadyPreference();
        }

        function sendJudgeStatusToServer(status) {
            const eventId = <?= json_encode($event_id) ?>;
            const judgeId = <?= json_encode($judge_id) ?>;
            
            // Get current heat and run selection
            const heatSelect = document.getElementById('heatSelect');
            const heatNumber = heatSelect ? parseInt(heatSelect.value) : null;
            const runNumber = document.querySelector('[name=run_number]:checked')?.value ? 
                            parseInt(document.querySelector('[name=run_number]:checked').value) : null;
            
            // Send status update to server - use relative path from judge/ folder
            fetch('../api/judge_status_api.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    event_id: eventId,
                    judge_id: judgeId,
                    status: status,
                    heat_number: heatNumber,
                    run_number: runNumber,
                    timestamp: new Date().toISOString()
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Judge status updated on server:', status);
                } else {
                    console.error('Failed to update judge status:', data.error);
                }
            })
            .catch(error => {
                console.error('Error sending judge status:', error);
            });
        }

        // Poll for judge status updates
        let judgeStatusPollInterval = null;

        function startJudgeStatusPolling() {
            // Stop any existing polling
            stopJudgeStatusPolling();
            
            // Poll every 3 seconds
            judgeStatusPollInterval = setInterval(() => {
                checkJudgeStatusUpdate();
            }, 3000);
            
            // Also check immediately
            checkJudgeStatusUpdate();
        }

        function stopJudgeStatusPolling() {
            if (judgeStatusPollInterval) {
                clearInterval(judgeStatusPollInterval);
                judgeStatusPollInterval = null;
            }
        }

        function checkJudgeStatusUpdate() {
            const judgeId = <?= json_encode($judge_id) ?>;
            
            // Get current event ID from the form/session storage
            const eventSelect = document.getElementById('eventSelect');
            const eventId = eventSelect ? eventSelect.value : null;
            
            if (!eventId || !judgeId) {
                console.log('No event selected for status polling');
                return;
            }
            
            // Fetch current status from server
            fetch(`../api/judge_status_api.php?event_id=${eventId}&judge_id=${judgeId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.data) {
                        const serverStatus = data.data.status || 'ready';
                        
                        // Only update if status has changed
                        if (serverStatus !== currentJudgeStatus) {
                            console.log('Status changed from', currentJudgeStatus, 'to', serverStatus);
                            currentJudgeStatus = serverStatus;
                            updateJudgeStatusUI(serverStatus);
                            saveJudgeStatus(serverStatus);
                            
                            // Show notification about status change
                            const statusLabels = {
                                'ready': 'Ready',
                                'pending': 'Pending',
                                'info': 'Info',
                                'warning': 'Warning',
                                'danger': 'Danger',
                                'off': 'Off'
                            };
                            showToast('Status Updated', 'Your status was changed to: ' + statusLabels[serverStatus], 'info', 4000);
                        }
                    }
                })
                .catch(error => {
                    console.error('Error polling judge status:', error);
                });
        }

        // Toggle judge status collapse panel (used when clicking modal backdrop)
        function toggleJudgeStatusCollapse() {
            if (judgeStatusCollapse) {
                judgeStatusCollapse.toggle();
            }
        }

        // Show fullscreen heat/run change notification
        function showHeatRunChangeOverlay(heat, run, heatName = null) {
            // Set judge status to pending when showing heat/run change notification
            setStatusToPending();
            
            // Create overlay container (transparent, positioned at top)
            const overlay = document.createElement('div');
            overlay.style.cssText = 'position: fixed; top: 20px; left: 20px; right: 20px; display: flex; align-items: flex-start; justify-content: center; z-index: 90000; pointer-events: none; animation: fadeIn 0.3s ease-out;';
            
            // Create content box with background
            const contentBox = document.createElement('div');
            contentBox.style.cssText = 'background: linear-gradient(135deg, #6f42c1 0%, #5a32a3 100%); color: white; padding: 1.5rem 3rem; border-radius: 12px; box-shadow: 0 8px 32px rgba(0,0,0,0.3); text-align: center;';
            
            // Build heat display with name if available
            let heatDisplay = 'Heat ' + heat;
            if (heatName) {
                heatDisplay += ' <br><u>' + heatName + '</u>';
            }

            contentBox.innerHTML = '<div style="font-size: 1.2rem; opacity: 0.9;">Event update</div><div style="font-size: 2rem; font-weight: 600; margin-bottom: 0.5rem;">' + heatDisplay + ' <br> Run ' + run + '</div><div style="font-size: 1rem; opacity: 0.9;">Loading participants...</div>';

            overlay.appendChild(contentBox);
            document.body.appendChild(overlay);
            
            // Add fade animations if not already present
            if (!document.getElementById('overlayAnimationStyles')) {
                const style = document.createElement('style');
                style.id = 'overlayAnimationStyles';
                style.textContent = '@keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } } @keyframes fadeOut { from { opacity: 1; } to { opacity: 0; } }';
                document.head.appendChild(style);
            }
            
            // Auto-remove overlay after 1.5 seconds
            setTimeout(() => {
                overlay.style.animation = 'fadeOut 0.3s ease-out';
                setTimeout(() => overlay.remove(), 300);
            }, 4000);
        }

        // Show fullscreen participant selection notification
        function showParticipantSelectedOverlay(bib, participantName, delay = 0) {
            // Set judge status to pending when showing participant change notification
            setStatusToPending();
            
            setTimeout(() => {
                // Create fullscreen overlay
                const overlay = document.createElement('div');
                overlay.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: linear-gradient(135deg, #007bff 0%, #0056b3 100%); color: white; display: flex; align-items: center; justify-content: center; z-index: 10000; animation: fadeIn 0.3s ease-out;';
                overlay.innerHTML = '<div style="text-align: center;"><div class="badge bg-light" style="font-size: 5rem; font-weight: 700; margin-bottom: 1rem;">' + bib + '</div><div style="font-size: 2.5rem; font-weight: 600;">' + participantName + '</div><div style="font-size: 1.5rem; margin-top: 0.5rem; opacity: 0.9;">Auto-Selected</div></div>';
                document.body.appendChild(overlay);
                
                // Add fade animations if not already present
                if (!document.getElementById('overlayAnimationStyles')) {
                    const style = document.createElement('style');
                    style.id = 'overlayAnimationStyles';
                    style.textContent = '@keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } } @keyframes fadeOut { from { opacity: 1; } to { opacity: 0; } }';
                    document.head.appendChild(style);
                }
                
                // Auto-remove overlay after 2 seconds
                setTimeout(() => {
                    overlay.style.animation = 'fadeOut 0.3s ease-out';
                    setTimeout(() => overlay.remove(), 300);
                }, 2000);
            }, delay);
        }

        // Enhanced loadParticipants function with heat flow integration
        let lastParticipantState = null; // Store the last loaded state for comparison
        let justSubmittedScore = false; // Track if we just submitted a score to delay participant notification

        function loadParticipants(forceReload = false) {
            const heat = document.querySelector('[name=heat_number]').value;
            const run = document.querySelector('[name=run_number]:checked')?.value;
            const target = document.querySelector('[name=ep_id]');
            const eventId = <?= json_encode($event_id) ?>;
            
            console.log('loadParticipants called - Heat:', heat, 'Run:', run, 'Event:', eventId, 'Force:', forceReload);
            
            if (!heat || !eventId || !run) {
                console.warn('loadParticipants: Missing required data - heat:', heat, 'run:', run, 'eventId:', eventId);
                return;
            }
            
            // Store current selection to restore after update
            const currentSelection = target.value;
            const currentSelectedOption = target.querySelector('option:checked');
            const currentParticipantInfo = currentSelectedOption ? {
                ep_id: currentSelectedOption.value,
                text: currentSelectedOption.textContent,
                bib: currentSelectedOption.getAttribute('data-bib')
            } : null;
            
            // Only show loading state when forcing reload (user action)
            // Don't show loading for background refresh - we'll compare first
            if (forceReload) {
                target.classList.add('loading');
                target.innerHTML = '<option disabled>Loading participants...</option>';
            }
            
            // Add cache buster to ensure fresh data
            const cacheBuster = '&_t=' + Date.now();
            const url = 'get_participants.php?event_id=' + eventId + '&heat_number=' + heat + '&run_number=' + run + cacheBuster;
            console.log('Fetching participants from:', url);
            
            fetch(url, {
                method: 'GET',
                cache: 'no-store', // Force fresh data, no caching
                headers: {
                    'Cache-Control': 'no-cache, no-store, must-revalidate',
                    'Pragma': 'no-cache'
                }
            })
                .then(res => {
                    if (!res.ok) {
                        throw new Error(`HTTP error! status: ${res.status}`);
                    }
                    return res.json();
                })
                .then(data => {
                    console.log('=== PARTICIPANT LOAD RESPONSE ===');
                    console.log('Full response:', data);
                    
                    // Handle new response format with debug info
                    const participants = data.participants || data;
                    const debugInfo = data.debug || null;
                    
                    if (debugInfo) {
                        console.log('Debug Info from Server:');
                        console.log('  Judge ID:', debugInfo.judge_id);
                        console.log('  Event:', debugInfo.event_id, 'Heat:', debugInfo.heat_number, 'Run:', debugInfo.run_number);
                        console.log('  Scored EP IDs:', debugInfo.scored_ep_ids);
                        console.log('  Total Locked:', debugInfo.total_locked);
                        
                        // Show alert if there are locked participants
                        if (debugInfo.total_locked > 0) {
                            console.warn(`⚠️ ${debugInfo.total_locked} participant(s) should be LOCKED for judge ${debugInfo.judge_id}`);
                        }
                    }
                    
                    console.log('Participants loaded successfully:', participants.length, 'participants');
                    
                    // Check if there's an error in the response
                    if (data.error) {
                        throw new Error(data.error);
                    }
                    
                    // Create state snapshot for comparison
                    const currentState = {
                        heat: heat,
                        run: run,
                        event: eventId,
                        participants: participants.map(p => ({
                            ep_id: p.ep_id,
                            bib: p.bib,
                            locked: p.locked
                        }))
                    };
                    
                    // Compare with last state
                    const stateChanged = hasParticipantStateChanged(lastParticipantState, currentState);
                    const isFirstLoad = !lastParticipantState; // Track if this is the very first load
                    
                    if (!stateChanged && !forceReload) {
                        console.log('✓ Participant state unchanged - skipping UI update');
                        // No DOM changes at all - data is identical
                        return;
                    }
                    
                    if (stateChanged) {
                        console.log('✓ Participant state changed - updating UI');
                        console.log('Changes detected:', getStateChanges(lastParticipantState, currentState));
                    }
                    
                    // Update last state
                    lastParticipantState = currentState;
                    
                    // Update UI
                    target.classList.remove('loading');
                    // Keep "Select Participant" option always enabled (no disabled attribute)
                    target.innerHTML = '<option value="">-- Select Participant --</option>';
                    
                    // Clear current participant score display when reloading participants
                    const currentScoreDisplay = document.getElementById('current-participant-score');
                    if (currentScoreDisplay) {
                        currentScoreDisplay.textContent = '--';
                    }
                    
                    // Store participant data for BIB lookup
                    const participantDataMap = new Map();
                    
                    let lockedCount = 0;
                    participants.forEach(p => {
                        const lock = p.locked ? '🔒' : '';
                        if (p.locked) lockedCount++;
                        
                        console.log('Participant data:', { 
                            bib: p.bib, 
                            first_name: p.first_name, 
                            last_name: p.last_name, 
                            country: p.country 
                        });
                        
                        const option = document.createElement('option');
                        option.value = p.ep_id;
                        option.textContent = `${p.bib} ${p.first_name} ${p.last_name} ${lock}`;
                        option.setAttribute('data-bib', p.bib);
                        option.setAttribute('data-firstname', p.first_name);
                        option.setAttribute('data-lastname', p.last_name);
                        option.setAttribute('data-country', p.country || '');
                        if (p.locked) {
                            option.disabled = true;
                            console.log(`Participant BIB ${p.bib} is LOCKED for this judge`);
                        } else {
                            console.log(`Participant BIB ${p.bib} is AVAILABLE`);
                        }
                        target.appendChild(option);
                        participantDataMap.set(p.ep_id, p);
                    });
                    
                    console.log(`Total: ${participants.length} participants, ${lockedCount} locked, ${participants.length - lockedCount} available`);
                    console.log('=== END PARTICIPANT LOAD ===');
                    
                    // Check if all participants are disabled (locked)
                    if (participants.length > 0 && lockedCount === participants.length) {
                        showNoParticipantsNotification();
                    } else {
                        hideNoParticipantsNotification();
                    }
                    
                    // Store the data map globally for BIB lookup
                    window.participantDataMap = participantDataMap;
                    
                    // Re-attach event listener (since innerHTML replacement removes it)
                    // Clone and replace to remove old listeners, then add new one
                    const newTarget = target.cloneNode(true);
                    target.parentNode.replaceChild(newTarget, target);
                    
                    // Get the new element from DOM and attach listener
                    const selectElement = document.getElementById('ep_id');
                    selectElement.addEventListener('change', function() {
                        console.log('PARTICIPANT SELECTION CHANGED! Value:', this.value);
                        if (this.value) {
                            // Set judge status to pending when participant changes
                            setStatusToPending();
                            
                            // Update the current participant info display
                            updateCurrentParticipantDisplay();
                            
                            // Populate BIB field from selected participant
                            const selectedOption = this.options[this.selectedIndex];
                            const bib = selectedOption.getAttribute('data-bib');
                            const bibInput = document.getElementById('bibInput');
                            if (bibInput && bib) {
                                bibInput.value = bib;
                                console.log(`Populated BIB from selection: ${bib}`);
                            }
                            
                            loadOtherScoresForParticipant();
                            
                            // Load figure history when participant changes
                            if (FIGURE_DATA && FIGURE_DATA.available) {
                                console.log('Loading figure history for participant:', this.value);
                                clearAllFigures();
                                currentParticipantId = this.value;
                                loadFigureHistoryForParticipant(this.value);
                            }
                        }
                    });
                    
                    // Try to restore previous selection if it still exists and is available
                    let restored = false;
                    if (currentSelection && !forceReload && !isFirstLoad) {
                        const previousOption = selectElement.querySelector(`option[value="${currentSelection}"]`);
                        if (previousOption && !previousOption.disabled) {
                            previousOption.selected = true;
                            restored = true;
                            console.log(`✓ Restored previous selection: ${previousOption.textContent}`);
                            
                            // Restore BIB field
                            const bibInput = document.getElementById('bibInput');
                            const bib = previousOption.getAttribute('data-bib');
                            if (bibInput && bib) {
                                bibInput.value = bib;
                            }
                            
                            // Update participant info strip
                            updateCurrentParticipantDisplay();
                            
                            loadOtherScoresForParticipant();
                        } else if (previousOption && previousOption.disabled) {
                            console.log(`⚠️ Previous selection (${currentParticipantInfo?.text}) is now LOCKED - will auto-select next available`);
                        } else {
                            console.log(`⚠️ Previous selection no longer exists - will auto-select first available`);
                        }
                    }
                    
                    // Auto-select first available participant if not successfully restored
                    // This covers: first load, force reload, previous locked, previous doesn't exist
                    if (!restored) {
                        const options = selectElement.querySelectorAll('option:not([disabled])');
                        let firstAvailable = null;
                        for (const option of options) {
                            if (option.value && option.value !== '') {
                                firstAvailable = option;
                                break;
                            }
                        }
                        
                        if (firstAvailable && firstAvailable.value) {
                            firstAvailable.selected = true;
                            
                            // Populate BIB field automatically
                            const bibInput = document.getElementById('bibInput');
                            const bib = firstAvailable.getAttribute('data-bib');
                            if (bibInput && bib) {
                                bibInput.value = bib;
                                console.log(`Auto-populated BIB: ${bib}`);
                            }
                            
                            // Update participant info strip
                            updateCurrentParticipantDisplay();
                            
                            // Show fullscreen participant selection notification with delay if just submitted score
                            const firstName = firstAvailable.getAttribute('data-firstname') || '';
                            const lastName = firstAvailable.getAttribute('data-lastname') || '';
                            const participantName = (firstName + ' ' + lastName).trim() || 'Participant';
                            const notificationDelay = justSubmittedScore ? 2500 : 0; // 2.5s delay after score submission
                            showParticipantSelectedOverlay(bib, participantName, notificationDelay);
                            
                            // Reset the flag after using it
                            if (justSubmittedScore) {
                                justSubmittedScore = false;
                            }
                            
                            loadOtherScoresForParticipant();
                        }
                    }
                    
                    // Try to auto-select based on BIB on start from heat flow data
                    if (heatFlowData && autoModeEnabled) {
                        const activeHeat = findActiveHeat(heatFlowData);
                        if (activeHeat && activeHeat.current_participant) {
                            const targetBip = activeHeat.current_participant.bib_number;
                            const options = target.querySelectorAll('option');
                            for (const option of options) {
                                if (option.textContent.startsWith(targetBip + ' ')) {
                                    option.selected = true;
                                    
                                    // Populate BIB field
                                    const bibInput = document.getElementById('bibInput');
                                    const bib = option.getAttribute('data-bib');
                                    if (bibInput && bib) {
                                        bibInput.value = bib;
                                        console.log(`Auto-populated BIB from heat flow: ${bib}`);
                                    }
                                    
                                    // Update participant info strip
                                    updateCurrentParticipantDisplay();
                                    
                                    // Show fullscreen participant selection notification with delay if just submitted score
                                    const firstName = option.getAttribute('data-firstname') || '';
                                    const lastName = option.getAttribute('data-lastname') || '';
                                    const participantName = (firstName + ' ' + lastName).trim() || 'Participant';
                                    const notificationDelay = justSubmittedScore ? 2500 : 0; // 2.5s delay after score submission
                                    showParticipantSelectedOverlay(bib, participantName, notificationDelay);
                                    
                                    // Reset the flag after using it
                                    if (justSubmittedScore) {
                                        justSubmittedScore = false;
                                    }
                                    
                                    loadOtherScoresForParticipant();
                                    break;
                                }
                            }
                        }
                    }
                })
                .catch(err => {
                    console.error('Error loading participants:', err);
                    target.classList.remove('loading');
                    target.innerHTML = '<option disabled>Error loading participants</option>';
                    showToast('Error', 'Failed to load participants: ' + err.message, 'danger');
                });
                
            saveHeatRunSelection();
        }

        // Compare two participant states to detect changes
        function hasParticipantStateChanged(stateA, stateB) {
            // If no previous state, consider it changed
            if (!stateA || !stateB) {
                return true;
            }
            
            // Check if heat/run/event changed
            if (stateA.heat !== stateB.heat || 
                stateA.run !== stateB.run || 
                stateA.event !== stateB.event) {
                return true;
            }
            
            // Check if number of participants changed
            if (stateA.participants.length !== stateB.participants.length) {
                return true;
            }
            
            // Check if any participant data changed (ep_id, bib, locked status)
            for (let i = 0; i < stateA.participants.length; i++) {
                const pA = stateA.participants[i];
                const pB = stateB.participants[i];
                
                if (pA.ep_id !== pB.ep_id || 
                    pA.bib !== pB.bib || 
                    pA.locked !== pB.locked) {
                    return true;
                }
            }
            
            // No changes detected
            return false;
        }

        // Get detailed information about what changed between states
        function getStateChanges(stateA, stateB) {
            if (!stateA || !stateB) {
                return 'Initial load or missing state';
            }
            
            const changes = [];
            
            if (stateA.heat !== stateB.heat) {
                changes.push(`Heat: ${stateA.heat} → ${stateB.heat}`);
            }
            if (stateA.run !== stateB.run) {
                changes.push(`Run: ${stateA.run} → ${stateB.run}`);
            }
            if (stateA.event !== stateB.event) {
                changes.push(`Event: ${stateA.event} → ${stateB.event}`);
            }
            if (stateA.participants.length !== stateB.participants.length) {
                changes.push(`Participant count: ${stateA.participants.length} → ${stateB.participants.length}`);
            }
            
            // Check for locked status changes
            const lockedChanges = [];
            for (let i = 0; i < Math.min(stateA.participants.length, stateB.participants.length); i++) {
                const pA = stateA.participants[i];
                const pB = stateB.participants[i];
                
                if (pA.locked !== pB.locked) {
                    lockedChanges.push(`BIB ${pB.bib}: ${pA.locked ? 'locked' : 'unlocked'} → ${pB.locked ? 'locked' : 'unlocked'}`);
                }
            }
            
            if (lockedChanges.length > 0) {
                changes.push(`Lock status changes: ${lockedChanges.join(', ')}`);
            }
            
            return changes.length > 0 ? changes.join('; ') : 'No changes';
        }

        // Periodic participant list refresh to detect database changes (deleted scores, head judge actions, etc.)
        let participantRefreshInterval = null;

        function startParticipantAutoRefresh() {
            // Clear any existing interval
            if (participantRefreshInterval) {
                clearInterval(participantRefreshInterval);
            }
            
            // Refresh every 10 seconds
            participantRefreshInterval = setInterval(() => {
                const heat = document.querySelector('[name=heat_number]')?.value;
                const run = document.querySelector('[name=run_number]:checked')?.value;
                const eventId = <?= json_encode($event_id) ?>;
                
                if (heat && run && eventId) {
                    console.log('Auto-refreshing participant list...');
                    loadParticipants();
                }
            }, 10000); // 10 seconds
        }

        function stopParticipantAutoRefresh() {
            if (participantRefreshInterval) {
                clearInterval(participantRefreshInterval);
                participantRefreshInterval = null;
            }
        }

        // Start auto-refresh when page loads and stop when page unloads
        window.addEventListener('load', startParticipantAutoRefresh);
        window.addEventListener('beforeunload', stopParticipantAutoRefresh);

        // Also stop when page is hidden (tab switched)
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                stopParticipantAutoRefresh();
            } else {
                startParticipantAutoRefresh();
            }
        });

        // Check if manually selected heat/run differs from active heat
        function checkIfDifferentFromActiveHeat() {
            if (!heatFlowData || !autoModeEnabled) return;
            
            const selectedHeat = document.querySelector('[name=heat_number]')?.value;
            const selectedRun = document.querySelector('[name=run_number]:checked')?.value;
            
            const activeHeat = findActiveHeat(heatFlowData);
            
            if (activeHeat) {
                const activeHeatNumber = activeHeat.heat_number;
                const activeRunNumber = activeHeat.active_run || 1; // Use active_run, not current_run
                
                if (selectedHeat != activeHeatNumber || selectedRun != activeRunNumber) {
                    // On initial page load, automatically sync silently without showing banner
                    if (isInitialPageLoad) {
                        console.log(`Initial load: Auto-syncing to active heat H${activeHeatNumber}/R${activeRunNumber}`);
                        
                        // Temporarily allow sync even with manual override flag
                        const wasManualOverride = manualOverride;
                        manualOverride = false;
                        
                        applyActiveHeatRun(activeHeat);
                        
                        // Restore manual override state if it was set
                        manualOverride = wasManualOverride;
                        
                        isInitialPageLoad = false; // Mark that we've handled the initial load
                        return; // Exit without showing banner
                    }
                    
                    // After initial load, show persistent warning banner that manual selection differs from active
                    const heatName = activeHeat.heat_name || `Heat ${activeHeatNumber}`;
                    showManualOverrideBanner(selectedHeat, selectedRun, heatName, activeRunNumber);
                    
                    console.warn(`Manual override: Selected H${selectedHeat}/R${selectedRun} but active is H${activeHeatNumber}/R${activeRunNumber}`);
                } else {
                    // Hide banner if selections now match
                    hideManualOverrideBanner();
                    // Also mark initial load as complete if selections already match
                    if (isInitialPageLoad) {
                        isInitialPageLoad = false;
                    }
                }
            } else {
                // No active heat found, just mark initial load as complete
                if (isInitialPageLoad) {
                    isInitialPageLoad = false;
                }
            }
        }

        // Helper function to get heat name from select dropdown
        function getSelectedHeatName(heatNumber) {
            const heatSelect = document.getElementById('heatSelect');
            if (!heatSelect) return '';
            
            // Find the option with matching value
            const option = Array.from(heatSelect.options).find(opt => opt.value == heatNumber);
            if (!option) return '';
            
            // Extract heat name from option text (format: "Heat X – Name" or "Heat X")
            const text = option.text;
            const match = text.match(/Heat \d+ – (.+)/);
            return match ? ' – ' + match[1] : '';
        }

        // Show persistent manual override warning banner
        function showManualOverrideBanner(selectedHeat, selectedRun, activeHeatName, activeRunNumber) {
            let banner = document.getElementById('manualOverrideBanner');
            
            // Create banner if it doesn't exist
            if (!banner) {
                banner = document.createElement('div');
                banner.id = 'manualOverrideBanner';
                banner.className = 'manual-override-banner';
                document.body.appendChild(banner);
            }
            
            banner.innerHTML = `
                <div class="manual-override-content gap-2">            
                    <div class="manual-override-message">
                        
                        <span class="manual-override-details">
                        <strong class="manual-override-title fs-3"><i class="fas fa-exclamation-triangle"></i> Manual Override:</strong> 
                        <p class="manual-override-selected fs-5">Selected <strong class="fs-4">Heat ${selectedHeat}${getSelectedHeatName(selectedHeat)} / Run ${selectedRun}</strong></p>
                        <p class="manual-override-active fs-5">Active: <strong class="fs-4">${activeHeatName} / Run ${activeRunNumber}</strong></p>
                            <p class="manual-override-scores fs-5">Scores save to manual selection</p>
                        </span>
                    </div>
                    <div class="manual-override-actions">
                        <button type="button" class="btn btn-light fs-4" onclick="forceRefreshHeatData()" title="Refresh and sync with active heat">
                            <i class="fas fa-sync-alt me-1"></i>Sync
                        </button>
                        <button type="button" class="btn btn-outline-light fs-4" onclick="hideManualOverrideBanner()" title="Close this warning">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            `;
            
            banner.style.display = 'flex';
        }

        // Hide manual override warning banner
        function hideManualOverrideBanner() {
            const banner = document.getElementById('manualOverrideBanner');
            if (banner) {
                banner.style.display = 'none';
            }
        }

        // Apply layout configuration from event settings
        function applyLayoutConfiguration() {
            const layoutConfig = <?= json_encode($layout_config) ?>;
            
            if (!layoutConfig || !layoutConfig.cards) {
                console.log('No layout configuration found, showing all components');
                return;
            }
            
            console.log('Applying layout configuration:', layoutConfig);
            
            // Component ID mappings - map config keys to actual HTML element IDs/classes
            const componentMappings = {
                'criteriaInputGrid': {
                    selectors: ['#criteriaInputCard', '#score-criteria', '.criteria-input-grid'],
                    description: 'Criteria Scoring Grid'
                },
                'finalCalculatedScore': {
                    selectors: ['#finalScoreCard', '#calculatedScoreCard', '.final-score-display'],
                    description: 'Final Score Display'
                },
                'figuresCard': {
                    selectors: ['#figuresCard'],
                    description: 'Figures Selection Card'
                },
                'latestScores': {
                    selectors: ['#latestScoresCard', '#latestScoresTable'],
                    parentSelector: '.latest-scores-container',
                    description: 'Latest Scores Table'
                },
                'otherHeatScores': {
                    selectors: ['#otherScoresCard', '#otherHeatScores'],
                    parentSelector: '.other-scores-container',
                    description: 'Other Judges Scores'
                },
                'diversityValidation': {
                    selectors: ['#diversity-validation-panel', '#diversityValidation'],
                    description: 'Diversity Validation'
                },
                'figureHistory': {
                    selectors: ['#figureHistoryCard', '#figureHistoryDisplay'],
                    description: 'Figure History'
                },
                'formatInfo': {
                    selectors: ['#formatInfoCard', '.format-info-card'],
                    description: 'Format Information Card'
                }
            };
            
            // Apply visibility settings for each component
            Object.keys(componentMappings).forEach(configKey => {
                const config = componentMappings[configKey];
                const isEnabled = layoutConfig.cards[configKey]?.enabled !== false;
                
                // Try to find elements by any of the selectors
                let elementsFound = false;
                config.selectors.forEach(selector => {
                    const elements = document.querySelectorAll(selector);
                    elements.forEach(element => {
                        elementsFound = true;
                        if (isEnabled) {
                            element.style.display = '';
                            element.classList.remove('d-none', 'layout-hidden');
                        } else {
                            element.style.display = 'none';
                            element.classList.add('d-none', 'layout-hidden');
                        }
                    });
                });
                
                // Also try parent selector if specified
                if (config.parentSelector) {
                    const parents = document.querySelectorAll(config.parentSelector);
                    parents.forEach(parent => {
                        if (parent) {
                            elementsFound = true;
                            if (isEnabled) {
                                parent.style.display = '';
                                parent.classList.remove('d-none', 'layout-hidden');
                            } else {
                                parent.style.display = 'none';
                                parent.classList.add('d-none', 'layout-hidden');
                            }
                        }
                    });
                }
                
                console.log(`${config.description} (${configKey}): ${isEnabled ? 'SHOWN' : 'HIDDEN'}${elementsFound ? '' : ' (elements not found)'}`);
            });
            
            // Apply compact mode if configured
            if (layoutConfig.layout?.compactMode === true || layoutConfig.layout?.compactMode === '1' || layoutConfig.layout?.compactMode === 1) {
                document.body.classList.add('compact-mode');
                document.querySelector('.score-container')?.classList.add('compact-mode');
                console.log('Compact mode: ENABLED');
            } else {
                console.log('Compact mode: DISABLED');
            }
            
            // Apply hide empty cards setting
            if (layoutConfig.layout?.hideEmptyCards === true || layoutConfig.layout?.hideEmptyCards === '1' || layoutConfig.layout?.hideEmptyCards === 1) {
                document.body.classList.add('hide-empty-cards');
                console.log('Hide empty cards: ENABLED');
                
                // Monitor for empty cards and hide them
                setTimeout(() => {
                    document.querySelectorAll('.card').forEach(card => {
                        const cardBody = card.querySelector('.card-body');
                        if (cardBody && cardBody.textContent.trim().length === 0) {
                            card.style.display = 'none';
                            card.classList.add('empty-card-hidden');
                        }
                    });
                }, 500);
            } else {
                console.log('Hide empty cards: DISABLED');
            }
        }

        // Update the main DOMContentLoaded event listener
        document.addEventListener('DOMContentLoaded', function() {
            // Apply layout configuration from event settings
            applyLayoutConfiguration();
            
            // Initialize Font Awesome to Lucide conversion on page load
            if (window.convertFontAwesomeToLucide) {
                window.convertFontAwesomeToLucide();
            }
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons();
            }
            
            // Don't focus on BIB input on page load - keypad only mode
            // document.getElementById('bibInput').focus();
            
            // Initialize figures card toggle state
            initializeFiguresCardState();
            
            // Initialize heat flow monitoring
            initializeHeatFlow();
            
            // Initialize judge status
            loadJudgeStatus();
            
            // Start polling for judge status updates
            startJudgeStatusPolling();
            
            // Initialize Heat/Run collapse toggle
            const heatRunCollapseBody = document.getElementById('heatRunCollapseBody');
            const heatRunToggleIcon = document.getElementById('heatRunToggleIcon');
            if (heatRunCollapseBody && heatRunToggleIcon) {
                // Add click handler to toggle icon only
                heatRunToggleIcon.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const bsCollapse = new bootstrap.Collapse(heatRunCollapseBody, {
                        toggle: true
                    });
                });
                
                // Update icon on collapse state changes
                heatRunCollapseBody.addEventListener('show.bs.collapse', function () {
                    heatRunToggleIcon.classList.remove('fa-chevron-down');
                    heatRunToggleIcon.classList.add('fa-chevron-up');
                });
                heatRunCollapseBody.addEventListener('hide.bs.collapse', function () {
                    heatRunToggleIcon.classList.remove('fa-chevron-up');
                    heatRunToggleIcon.classList.add('fa-chevron-down');
                });
                
                // Prevent collapse when clicking inside the collapse body
                heatRunCollapseBody.addEventListener('click', function(e) {
                    e.stopPropagation();
                });
            }
            
            // Initialize summary badge
            updateHeatRunSummary();
            
            // Initialize Dark Mode Toggle
            const darkModeToggle = document.getElementById('darkModeToggle');
            const darkModeIcon = document.getElementById('darkModeIcon');
            
            // Check for saved dark mode preference
            const savedDarkMode = localStorage.getItem('judgeDarkMode');
            if (savedDarkMode === 'enabled') {
                document.body.classList.add('dark-mode');
                if (darkModeIcon) {
                    darkModeIcon.classList.remove('fa-moon');
                    darkModeIcon.classList.add('fa-sun');
                }
            }
            
            // Dark mode toggle functionality
            if (darkModeToggle) {
                darkModeToggle.addEventListener('click', function() {
                    document.body.classList.toggle('dark-mode');
                    
                    if (document.body.classList.contains('dark-mode')) {
                        localStorage.setItem('judgeDarkMode', 'enabled');
                        if (darkModeIcon) {
                            darkModeIcon.classList.remove('fa-moon');
                            darkModeIcon.classList.add('fa-sun');
                        }
                    } else {
                        localStorage.setItem('judgeDarkMode', 'disabled');
                        if (darkModeIcon) {
                            darkModeIcon.classList.remove('fa-sun');
                            darkModeIcon.classList.add('fa-moon');
                        }
                    }
                });
            }
            
            // Initialize Bootstrap tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
            
            // Existing localStorage restoration code...
            const eventId = <?= json_encode($event_id) ?>;
            const stored = localStorage.getItem('event_' + eventId + '_heat_run');
            if (stored) {
                try {
                    const obj = JSON.parse(stored);
                    if (obj.heat) {
                        const heatSelect = document.querySelector('[name=heat_number]');
                        if (heatSelect) {
                            heatSelect.value = obj.heat;
                            manualOverride = true; // Mark as manual override
                        }
                    }
                    if (obj.run) {
                        const runRadio = document.getElementById('run' + obj.run);
                        if (runRadio) {
                            runRadio.checked = true;
                            manualOverride = true; // Mark as manual override
                        }
                    }
                    const heatRunBadge = document.getElementById('heatRunBadge');
                    if (heatRunBadge) {
                        heatRunBadge.style.display = 'inline-block';
                        setTimeout(() => {
                            if (heatRunBadge) {
                                heatRunBadge.style.display = 'none';
                            }
                        }, 3000);
                    }
                    
                    updateAutoModeDisplay();
                } catch(e) {}
            }
            
            // Enhanced BIB input
            const bibInput = document.getElementById('bibInput');
            bibInput.addEventListener('input', function() {
                const bib = this.value;
                const options = document.querySelectorAll('[name=ep_id] option');
                options.forEach(opt => {
                    if (opt.textContent.startsWith(bib + ' ')) {
                        opt.selected = true;
                        loadOtherScoresForParticipant();
                    }
                });
            });
            
            // Enable next button and criteria grid when participant is selected
            const participantSelect = document.getElementById('ep_id');
            if (participantSelect) {
                participantSelect.addEventListener('change', function() {
                    if (this.value) {
                        // Set judge status to pending on participant change
                        setStatusToPending();
                        enableNextButtonAndCriteria();
                    }
                });
            }
            
            // Auto-change status to "info" when judge starts entering score data
            const scoreValueInput = document.getElementById('score_value');
            if (scoreValueInput) {
                scoreValueInput.addEventListener('input', function() {
                    if (this.value && currentJudgeStatus !== 'info') {
                        setStatusToInfo();
                    }
                });
            }
            
            // Monitor all inputs within participant-selection card
            const participantSelectionCard = document.getElementById('participant-selection');
            if (participantSelectionCard) {
                const allInputs = participantSelectionCard.querySelectorAll('input, select');
                allInputs.forEach(input => {
                    input.addEventListener('change', function() {
                        if (this.value && currentJudgeStatus !== 'info') {
                            setStatusToInfo();
                        }
                    });
                });
            }
            
            // Initialize disabled states on page load
            const statusButtons = document.querySelector('.score-status-buttons');
            const submitBtn = document.getElementById('submitBtn');
            // Status buttons are now enabled by default
            // if (statusButtons) {
            //     statusButtons.classList.add('disabled');
            // }
            if (submitBtn) {
                submitBtn.classList.add('disabled');
                submitBtn.disabled = true;
            }
            
            // AJAX Score Form submission
            const scoreForm = document.getElementById('scoreForm');
            scoreForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Check diversity validation before submission if figures are available
                if (FIGURE_DATA?.available && FIGURE_UI?.diversityRules?.length > 0) {
                    const validationBox = document.getElementById('diversityValidation');
                    const isValid = runDiversityValidationWithHistory(validationBox);
                    
                    if (!isValid) {
                        // Scroll to validation box to show errors
                        validationBox.scrollIntoView({ behavior: 'smooth', block: 'center' });
                        
                        // Show error toast
                        showToast('Validation Failed', 'Please satisfy all diversity requirements before submitting.', 'error');
                        
                        // Highlight validation box with animation
                        validationBox.style.border = '2px solid #dc3545';
                        validationBox.style.boxShadow = '0 0 10px rgba(220, 53, 69, 0.3)';
                        setTimeout(() => {
                            validationBox.style.border = '';
                            validationBox.style.boxShadow = '';
                        }, 3000);
                        
                        return; // Stop submission
                    }
                }
                
                const formData = new FormData(this);
                formData.append('ajax', '1'); // Add AJAX flag

                // Include figure selection if available
                try {
                    if (FIGURE_DATA?.available) {
                        // Legacy single selection (first active chip if any)
                        const firstActive = document.querySelector('.figure-chip.active');
                        const cat = firstActive?.dataset.cat || '';
                        const item = firstActive?.dataset.item || '';
                        formData.set('figure_category', cat);
                        formData.set('figure_item', item);
                        const hidCat = document.getElementById('figure_category_hidden');
                        const hidItem = document.getElementById('figure_item_hidden');
                        if (hidCat) hidCat.value = cat;
                        if (hidItem) hidItem.value = item;
                        
                        // Full selection JSON (convert Set -> Array)
                        const selections = window.__FIGURE_SELECTIONS__ || {};
                        const serialized = {};
                        for (const [k, v] of Object.entries(selections)) {
                            const arr = Array.isArray(v) ? v : Array.from(v || []);
                            if (arr.length > 0) serialized[k] = arr;
                        }
                        
                        // Add validation status to figures_json for server-side checking
                        const validationState = window.__DIVERSITY_VALIDATION_STATE__ || { isValid: true };
                        const enrichedData = {
                            figures: serialized,
                            validation: {
                                diversity_valid: validationState.isValid,
                                validation_timestamp: validationState.timestamp || Date.now(),
                                run_number: validationState.currentRun || 1,
                                client_validated: true,
                                diversity_rules_checked: Array.isArray(FIGURE_UI?.diversityRules) ? FIGURE_UI.diversityRules.length : 0
                            }
                        };
                        
                        const jsonSel = JSON.stringify(enrichedData);
                        const hidJson = document.getElementById('figures_json_hidden');
                        if (hidJson) hidJson.value = jsonSel;
                        formData.set('figures_json', jsonSel);
                    } else {
                        // Even without figures, include validation metadata if diversity rules exist
                        if (FIGURE_UI?.diversityRules && Array.isArray(FIGURE_UI.diversityRules) && FIGURE_UI.diversityRules.length > 0) {
                            const validationState = window.__DIVERSITY_VALIDATION_STATE__ || { isValid: true };
                            const validationOnlyData = {
                                figures: {},
                                validation: {
                                    diversity_valid: validationState.isValid,
                                    validation_timestamp: validationState.timestamp || Date.now(),
                                    run_number: validationState.currentRun || 1,
                                    client_validated: true,
                                    diversity_rules_checked: FIGURE_UI.diversityRules.length,
                                    no_figures_mode: true
                                }
                            };
                            const jsonSel = JSON.stringify(validationOnlyData);
                            formData.set('figures_json', jsonSel);
                        }
                    }
                } catch (_) { /* no-op */ }

                // Collect criteria scores if they exist
                try {
                    const criteriaInputs = document.querySelectorAll('[name^="criteria_scores"][name$="[value]"]');
                    if (criteriaInputs.length > 0) {
                        const criteriaScores = {};
                        let hasCriteriaValues = false;
                        
                        criteriaInputs.forEach((input, index) => {
                            const value = parseFloat(input.value) || 0;
                            const typeInput = document.querySelector(`[name="criteria_scores[${index}][type]"]`);
                            const nameInput = document.querySelector(`[name="criteria_scores[${index}][name]"]`);
                            const sortInput = document.querySelector(`[name="criteria_scores[${index}][sort_order]"]`);
                            
                            if (typeInput && nameInput) {
                                criteriaScores[index] = {
                                    value: value,
                                    type: typeInput.value,
                                    name: nameInput.value,
                                    sort_order: parseInt(sortInput?.value || 0)
                                };
                                
                                // Check if any criteria has a value
                                if (value > 0) {
                                    hasCriteriaValues = true;
                                }
                            }
                        });
                        
                        // Add criteria scores as JSON to form data
                        formData.set('criteria_scores_json', JSON.stringify(criteriaScores));
                        
                        // Only override score_value if criteria scores are actually being used
                        if (hasCriteriaValues) {
                            const finalScore = calculateFinalScore();
                            formData.set('score_value', finalScore.toFixed(2));
                        }
                        // Otherwise, keep the original score_value from the form input
                    }
                } catch (_) { /* no-op */ }
                
                const submitBtn = document.getElementById('submitBtn');
                const originalText = submitBtn.innerHTML;
                
                // Show loading state
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Submitting...';
                
                // Add success animation
                scoreForm.classList.add('score-submitted');
                
                // Submit via AJAX
                const requestData = Object.fromEntries(formData.entries());
                
                fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    // Log the API call for debugging
                    logApiCall('POST', window.location.href, requestData, data);
                    
                    console.log('=== SCORE SUBMISSION RESPONSE ===');
                    console.log('Success:', data.success);
                    console.log('Message:', data.message);
                    if (data.debug) {
                        console.log('Debug Info:');
                        console.log('  Judge ID used:', data.debug.judge_id);
                        console.log('  EP ID:', data.debug.ep_id);
                        console.log('  Run ID:', data.debug.run_id);
                        console.log('  Run Number:', data.debug.run_number);
                    }
                    console.log('=== END SCORE SUBMISSION ===');
                    
                    if (data.success) {
                        console.log('Score submitted successfully, reloading participants...');
                        
                        // Scroll to top smoothly
                        window.scrollTo({
                            top: 0,
                            behavior: 'smooth'
                        });
                        
                        // Show fullscreen overlay notification
                        const overlay = document.createElement('div');
                        overlay.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: linear-gradient(135deg, #28a745 0%, #20c997 100%); color: white; display: flex; align-items: center; justify-content: center; z-index: 10000; animation: fadeIn 0.3s ease-out;';
                        overlay.innerHTML = '<div style="text-align: center;"><i class="fas fa-check-circle" style="font-size: 5rem; margin-bottom: 1rem; display: block;"></i><div style="font-size: 2.5rem; font-weight: 600;">Score Saved!</div></div>';
                        document.body.appendChild(overlay);
                        
                        // Add fade in/out animations if not already present
                        if (!document.getElementById('overlayAnimationStyles')) {
                            const style = document.createElement('style');
                            style.id = 'overlayAnimationStyles';
                            style.textContent = '@keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } } @keyframes fadeOut { from { opacity: 1; } to { opacity: 0; } }';
                            document.head.appendChild(style);
                        }
                        
                        // Auto-remove overlay after 2 seconds
                        setTimeout(() => {
                            overlay.style.animation = 'fadeOut 0.3s ease-out';
                            setTimeout(() => overlay.remove(), 300);
                        }, 2000);
                        
                        // Add new score to table
                        addNewScoreToTable(data.score);
                        
                        // Show success message first
                        showToast('Success!', data.message, 'success');
                        
                        // Clear the form first (but NOT the dropdown - loadParticipants will handle that)
                        clearScoreForm();

                        // Reset figure selections grid after successful submit
                        if (window.__FIGURE_SELECTIONS__) {
                            window.__FIGURE_SELECTIONS__ = {};
                        }
                        const figureGridAfterSubmit = document.getElementById('figureToggleGrid');
                        if (figureGridAfterSubmit) {
                            figureGridAfterSubmit.querySelectorAll('.figure-chip.active').forEach(chip => {
                                chip.classList.remove('active');
                            });
                        }
                        
                        // Also clear the hidden figure fields
                        const figuresJsonHidden = document.getElementById('figures_json_hidden');
                        if (figuresJsonHidden) {
                            figuresJsonHidden.value = '';
                        }
                        
                        // Close figures toggle/card after successful submission
                        const figuresCardBody = document.getElementById('figuresCardBody');
                        const figuresToggleIcon = document.getElementById('figuresToggleIcon');
                        if (figuresCardBody && figuresToggleIcon) {                   
                            figuresToggleIcon.classList.add('collapsed');
                            figuresToggleIcon.style.transform = 'rotate(180deg)';
                        }
                        
                        // Set flag to delay participant notification
                        justSubmittedScore = true;
                        
                        // Reload participants via AJAX to get updated locked status from server
                        // Use setTimeout to ensure the database has processed the score
                        setTimeout(() => {
                            console.log('Calling loadParticipants to refresh dropdown...');
                            loadParticipants();
                        }, 100);
                    } else {
                        throw new Error(data.message || 'Submission failed');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    
                    // Log the API call error for debugging
                    logApiCall('POST', window.location.href, requestData, { 
                        error: error.message || 'Unknown error',
                        status: 'error',
                        timestamp: new Date().toISOString()
                    });
                    
                    showToast('Error', error.message || 'Error submitting score. Please try again.', 'error');
                })
                .finally(() => {
                    // Restore button state
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                    scoreForm.classList.remove('score-submitted');
                });
            });
            
            // NOTE: Participant info strip helper functions (updateCurrentParticipantDisplay, clearCurrentParticipant, getFlagEmoji)
            // are now defined BEFORE loadParticipants() function to avoid "not defined" errors
            // Participant select change event listener is attached inside loadParticipants()
            
            loadParticipants();
            startOtherScoresInterval();
            
            // Initialize figure data if participant is already selected on page load
            setTimeout(() => {
                const ep_id = document.getElementById('ep_id')?.value;
                const heat = document.querySelector('[name=heat_number]')?.value;
                
                if (ep_id && heat && FIGURE_DATA?.available) {
                    console.log('Page loaded with existing participant selection, loading figure data...');
                    currentParticipantId = ep_id;
                    
                    // Load figure history and existing selections
                    loadFigureHistoryForParticipant(ep_id);
                    
                    setTimeout(() => {
                        loadExistingFigureSelections();
                        
                        setTimeout(() => {
                            displayFigureHistory();
                            updateDiversityValidationWithHistory();
                        }, 100);
                    }, 300);
                } else if (FIGURE_DATA?.available) {
                    // Still run initial diversity validation even without participant
                    console.log('Running initial diversity validation...');
                    updateDiversityValidationWithHistory();
                }
            }, 500);
        });

            
        // Handle status radio button changes
        document.querySelectorAll('input[name="status"]').forEach(radio => {
            radio.addEventListener('change', function() {
                const additionalCardsWrapper = document.getElementById('additionalCardsWrapper');
                
                if (this.checked && this.value !== 'OK') {
                    const scoreInput = document.querySelector('.score-input');
                    const rangeInput = document.querySelector('[name="score_range"]');
                    
                    scoreInput.value = 0;
                    rangeInput.value = 0;
                    
                    // Hide additional cards wrapper when status is not OK
                    if (additionalCardsWrapper) {
                        additionalCardsWrapper.style.display = 'none';
                    }
                } else if (this.checked && this.value === 'OK') {
                    const scoreInput = document.querySelector('.score-input');
                    const rangeInput = document.querySelector('[name="score_range"]');
                    
                    scoreInput.value = '';
                    rangeInput.value = '';
                    
                    // Show additional cards wrapper when status is OK
                    if (additionalCardsWrapper) {
                        additionalCardsWrapper.style.display = 'block';
                    }
                }
            });
        });

        // Figures UI wiring: toggle chips per category + diversity validation
        document.addEventListener('DOMContentLoaded', () => {
            console.log('DOM loaded, checking figure UI...');
            console.log('FIGURE_DATA.available:', FIGURE_DATA?.available);
            
            if (!FIGURE_DATA.available) {
                console.log('Figures not available, skipping UI setup');
                return;
            }
            
            window.__FIGURE_SELECTIONS__ = {}; // { category: Set(items) }
            console.log('Initialized figure selections:', window.__FIGURE_SELECTIONS__);

            const grid = document.getElementById('figureToggleGrid');
            const box = document.getElementById('diversityValidation');
            
            console.log('Grid element found:', grid);
            console.log('Validation box found:', box);
            
            if (!grid) {
                console.error('figureToggleGrid element not found!');
                return;
            }

            if (grid) {
                // Check if any chips exist
                const chips = grid.querySelectorAll('.figure-chip');
                console.log('Found', chips.length, 'figure chips');
                chips.forEach((chip, index) => {
                    if (index < 3) { // Log first 3 for debugging
                        console.log(`Chip ${index}:`, chip, 'data-cat:', chip.dataset.cat, 'data-item:', chip.dataset.item);
                    }
                });
                
                // Add event listener to the grid with proper event delegation
                grid.addEventListener('click', (e) => {
                    // Try direct click and closest() to catch the chip
                    let chip = null;
                    if (e.target.classList.contains('figure-chip')) {
                        chip = e.target;
                    } else {
                        chip = e.target.closest('.figure-chip');
                    }
                    
                    if (chip) {
                        const c = chip.dataset.cat;
                        const it = chip.dataset.item;
                        
                        if (!c || !it) {
                            console.warn('Missing category or item data:', { cat: c, item: it });
                            return;
                        }
                        
                        if (!window.__FIGURE_SELECTIONS__[c]) {
                            window.__FIGURE_SELECTIONS__[c] = new Set();
                        }
                        
                        if (chip.classList.contains('active')) {
                            chip.classList.remove('active');
                            window.__FIGURE_SELECTIONS__[c].delete(it);
                            
                            // Keep run indicator text visible after deselection
                            if (chip.classList.contains('has-history')) {
                                const originalText = chip.getAttribute('data-original-text');
                                const runIndicator = chip.getAttribute('data-run');
                                if (originalText && runIndicator) {
                                    chip.innerHTML = `${originalText} <small class="text-muted">(${runIndicator})</small>`;
                                }
                            }
                        } else {
                            // Only allow one selection per category - deselect all others in this category first
                            grid.querySelectorAll(`.figure-chip[data-cat="${CSS.escape(c)}"].active`).forEach(otherChip => {
                                otherChip.classList.remove('active');
                                
                                // Keep run indicator for deselected chips
                                if (otherChip.classList.contains('has-history')) {
                                    const originalText = otherChip.getAttribute('data-original-text');
                                    const runIndicator = otherChip.getAttribute('data-run');
                                    if (originalText && runIndicator) {
                                        otherChip.innerHTML = `${originalText} <small class="text-muted">(${runIndicator})</small>`;
                                    }
                                }
                            });
                            
                            // Clear all selections for this category and add only the new one
                            window.__FIGURE_SELECTIONS__[c].clear();
                            
                            chip.classList.add('active');
                            window.__FIGURE_SELECTIONS__[c].add(it);
                            
                            // Keep run indicator visible even when active
                            if (chip.classList.contains('has-history')) {
                                const originalText = chip.getAttribute('data-original-text');
                                const runIndicator = chip.getAttribute('data-run');
                                if (originalText && runIndicator) {
                                    chip.innerHTML = `${originalText} <small class="text-muted">(${runIndicator})</small>`;
                                }
                            }
                        }
                        
                        // Update diversity validation with historical context
                        updateDiversityValidationWithHistory();
                        return; // Prevent further processing
                    }
                    
                    // Handle clear button
                    let clearBtn = null;
                    if (e.target.classList.contains('clear-cat')) {
                        clearBtn = e.target;
                    } else {
                        clearBtn = e.target.closest('.clear-cat');
                    }
                    
                    if (clearBtn) {
                        const cat = clearBtn.dataset.cat;
                        
                        if (!cat) {
                            console.warn('Missing category data for clear button');
                            return;
                        }
                        
                        // Only clear active selections, not historical ones
                        grid.querySelectorAll(`.figure-chip[data-cat="${CSS.escape(cat)}"].active`).forEach(b => b.classList.remove('active'));
                        if (window.__FIGURE_SELECTIONS__[cat]) {
                            window.__FIGURE_SELECTIONS__[cat].clear();
                        }
                        
                        updateDiversityValidationWithHistory();
                    }
                });
                
                // Also add direct event listeners to chips as backup
                chips.forEach(chip => {
                    chip.addEventListener('click', function(e) {
                        console.log('Direct chip click listener triggered for:', this);
                        // The main handler above should catch this, but if not, this is a backup
                    });
                });
            }

            function runDiversityValidation(outBox) {
                // Use the enhanced version with history
                return runDiversityValidationWithHistory(outBox);
            }

            // Initial validation render
            console.log('Running initial diversity validation...');
            runDiversityValidation(box);
            console.log('Initial diversity validation completed');
        });

        // API function to check diversity rules server-side
        async function checkDiversityRulesAPI(ep_id, judge_id, event_id, heat_number = null) {
            try {
                const params = new URLSearchParams({
                    ep_id: ep_id,
                    judge_id: judge_id,
                    event_id: event_id
                });
                
                if (heat_number) {
                    params.append('heat_number', heat_number);
                }
                
                const response = await fetch(`../api/check_diversity_rules.php?${params}`);
                
                if (!response.ok) {
                    throw new Error(`API request failed: ${response.status}`);
                }
                
                const result = await response.json();
                
                console.log('Diversity rules API result:', result);
                return result;
                
            } catch (error) {
                console.error('Error checking diversity rules via API:', error);
                return {
                    error: error.message,
                    compliance: null
                };
            }
        }

        // Helper function to run API check for current participant
        async function checkCurrentParticipantDiversity() {
            const ep_id = document.getElementById('ep_id')?.value;
            const heat_number = document.querySelector('[name="heat_number"]')?.value;
            const event_id = <?= json_encode($event_id) ?>;
            const judge_id = <?= json_encode($judge_id) ?>;
            
            if (!ep_id || !event_id || !judge_id) {
                console.log('Missing required data for diversity check');
                return null;
            }
            
            console.log('Checking diversity rules for participant:', ep_id);
            const result = await checkDiversityRulesAPI(ep_id, judge_id, event_id, heat_number);
            
            if (result.error) {
                console.error('API error:', result.error);
            } else {
                console.log('Diversity compliance result:', result.compliance);
                console.log('Rules checked:', result.rules_checked);
                if (result.failed_rules) {
                    console.log('Failed rules:', result.failed_rules);
                }
            }
            
            return result;
        }

        // Add button to manually trigger API check (for testing)
        document.addEventListener('DOMContentLoaded', function() {
            const testButton = document.createElement('button');
            testButton.type = 'button';
            testButton.className = 'btn btn-outline-info btn-sm';
            testButton.innerHTML = '<i class="fas fa-check-circle me-1"></i>Test API Check';
            testButton.onclick = checkCurrentParticipantDiversity;
            testButton.title = 'Test server-side diversity rules checking (check console for results)';
            
            // Add to the reset scores area
            const resetArea = document.querySelector('.text-center.mb-3.mt-4');
            if (resetArea) {
                resetArea.appendChild(document.createTextNode(' '));
                resetArea.appendChild(testButton);
            }
        });

        // Judge Overview Modal Functions
        async function loadJudgeOverview() {
            const eventId = <?= json_encode($event_id) ?>;
            const judgeId = <?= json_encode($judge_id) ?>;
            const content = document.getElementById('judgeOverviewContent');
            
            if (!eventId || !judgeId) {
                content.innerHTML = '<div class="alert alert-warning">No event or judge selected</div>';
                return;
            }
            
            try {
                const response = await fetch(`ajax_judge_overview.php?event_id=${eventId}&judge_id=${judgeId}`);
                if (!response.ok) throw new Error('Failed to load overview');
                
                const data = await response.json();
                
                if (data.error) {
                    content.innerHTML = `<div class="alert alert-danger">${data.error}</div>`;
                    return;
                }
                
                // Group scores by participant
                const participantGroups = {};
                data.scores.forEach(score => {
                    const key = `${score.participant_name}_${score.bib_number}`;
                    if (!participantGroups[key]) {
                        participantGroups[key] = {
                            participant_name: score.participant_name,
                            bib_number: score.bib_number,
                            heat_number: score.heat_number,
                            scores: []
                        };
                    }
                    participantGroups[key].scores.push(score);
                });
                
                // Build HTML
                let html = `
                    <div class="mb-3 d-flex flex-row">
                        <div class="col-md-6 p-2">
                            <div class="card bg-warning p-2">
                                <div class="card-body text-center">
                                    <h4 class="card-title text-primary  m-0">${data.summary.total_scores} <span class="card-text small">Scores</span></h4>
                                    
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6 p-2">
                            <div class="card bg-success p-2">
                                <div class="card-body text-center">
                                    <h4 class="card-title text-primary  m-0">${data.summary.average_score} <span class="card-text small">Average</span></h4>
                                    
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                
                // Add participant groups
                Object.values(participantGroups).forEach(group => {
                    const sortedScores = group.scores.sort((a, b) => a.run_number - b.run_number);
                    
                    html += `
                        <div class="card mb-3">
                            <div class="card-header bg-black text-white d-flex justify-content-between align-items-center p-1">
                                <div>
                                    <span class="badge bg-warning-subtle ms-1">${group.bib_number}</span>
                                    <strong>${group.participant_name}</strong>                                    
                                    
                                </div>
                                <div>
                                    <span class="badge border-info ms-1 bg-info">Heat ${group.heat_number}</span>
                                    <span class="badge border-info ms-1 bg-info-subtle">${sortedScores.length} run${sortedScores.length !== 1 ? 's' : ''}</span>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <div class="table-responsive">
                                    <table class="table table-sm mb-0">
                                        <thead class="small">
                                            <tr>
                                                <th class="small bg-dark-subtle">Run</th>
                                                <th class="small bg-dark-subtle">Score</th>
                                                <th class="small bg-dark-subtle"></th>
                                                <th class="small bg-dark-subtle">Status</th>
                                                <th class="small bg-dark-subtle">Time</th>                                                
                                                <th class="small bg-dark-subtle"></th>
                                            </tr>
                                        </thead>
                                        <tbody>
                    `;
                    
                    sortedScores.forEach(score => {
                        const figures = score.figures_json ? 
                            (typeof score.figures_json === 'string' ? 
                                JSON.parse(score.figures_json) : score.figures_json) : null;
                        
                        let figuresDisplay = '—';
                        if (figures && figures.figures) {
                            const badgesList = [];
                            Object.entries(figures.figures).forEach(([cat, items]) => {
                                if (Array.isArray(items)) {
                                    items.forEach(item => {
                                        badgesList.push(`<span class="badge bg-secondary quick-badge figure-badge-${item}" title="${cat}"><strong>${item}</strong><small class="ms-1 opacity-75">(${cat})</small></span>`);
                                    });
                                } else if (items) {
                                    badgesList.push(`<span class="badge bg-secondary quick-badge figure-badge-${item}" title="${cat}"><strong>${items}</strong><small class="ms-1 opacity-75">(${cat})</small></span>`);
                                }
                            });
                            figuresDisplay = badgesList.length > 0 ? badgesList.join(' ') : '—';
                        }
                        
                        const statusBadge = score.status === 'OK' ? 'success' : 
                            (score.status === 'DSQ' ? 'danger' : 'warning');
                        
                        // Format score to only show decimals when needed
                        const scoreDisplay = score.score_value !== null && score.score_value !== undefined ?
                            (parseFloat(score.score_value) % 1 === 0 ? Math.floor(parseFloat(score.score_value)) : parseFloat(score.score_value)) : '-';
                        
                        html += `
                            <tr>
                                <td><span class="badge border-primary text-primary bg-success-subtle">Run ${score.run_number}</span></td>
                                <td><strong class="text-primary">${scoreDisplay}</strong></td>
                                <td><small>${figuresDisplay}</small></td>
                                <td><span class="badge bg-${statusBadge}">${score.status}</span></td>
                                <td><small class="text-muted">${new Date(score.created_at).toLocaleTimeString()}</small></td>
                                
                                <td>
                                    <button type="button" class="btn btn-warning btn-sm" 
                                        onclick="showOverwriteModalFromOverview(${score.score_id}, '${score.first_name.replace(/'/g, "\\'")}', '${score.last_name.replace(/'/g, "\\'")}', ${score.bib_number}, ${score.run_number}, ${score.score_value}, '${score.status}', '${escape(score.figures_json || '')}')">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                </td>
                            </tr>
                        `;
                    });
                    
                    html += `
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    `;
                });
                
                if (Object.keys(participantGroups).length === 0) {
                    html += '<div class="alert alert-info">No scores recorded yet.</div>';
                }
                
                content.innerHTML = html;
                
            } catch (error) {
                console.error('Error loading judge overview:', error);
                content.innerHTML = `<div class="alert alert-danger">Error loading overview: ${error.message}</div>`;
            }
        }

        async function loadEventSummary() {
            const eventId = <?= json_encode($event_id) ?>;
            const content = document.getElementById('eventSummaryContent');
            
            if (!eventId) {
                content.innerHTML = '<div class="alert alert-warning">No event selected</div>';
                return;
            }
            
            const summaryType = document.querySelector('input[name="summaryType"]:checked')?.value || 'final';
            const displayFormat = document.querySelector('input[name="displayFormat"]:checked')?.value || 'table';
            
            try {
                const response = await fetch(`ajax_event_summary.php?event_id=${eventId}&type=${summaryType}&format=${displayFormat}`);
                if (!response.ok) throw new Error('Failed to load summary');
                
                const data = await response.json();
                
                if (data.error) {
                    content.innerHTML = `<div class="alert alert-danger">${data.error}</div>`;
                    return;
                }
                
                let html = '';
                
                if (summaryType === 'final') {
                    // Final results table
                    html = `
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Rank</th>
                                        <th>BIB</th>
                                        <th>Participant</th>
                                        <th>Heat</th>
                                        <th>Best Score</th>
                                        <th>Average</th>
                                        <th>Runs</th>
                                    </tr>
                                </thead>
                                <tbody>
                    `;
                    
                    data.results.forEach((result, index) => {
                        const rankBadge = index < 3 ? 
                            ['bg-warning', 'bg-secondary', 'bg-warning'][index] : 'bg-light text-dark';
                        
                        html += `
                            <tr>
                                <td><span class="badge ${rankBadge}">${index + 1}</span></td>
                                <td><span class="badge bg-secondary">${result.bib_number}</span></td>
                                <td><strong>${result.participant_name}</strong></td>
                                <td>Heat ${result.heat_number}</td>
                                <td><strong class="text-success">${result.best_score}</strong></td>
                                <td>${result.average_score}</td>
                                <td>${result.completed_runs}/${result.total_runs}</td>
                            </tr>
                        `;
                    });
                    
                    html += '</tbody></table></div>';
                } else {
                    // Other summary types
                    html = `<div class="alert alert-info">Summary type "${summaryType}" coming soon...</div>`;
                }
                
                content.innerHTML = html;
                
            } catch (error) {
                console.error('Error loading event summary:', error);
                content.innerHTML = `<div class="alert alert-danger">Error loading summary: ${error.message}</div>`;
            }
        }

        function refreshJudgeOverview() {
            const judgeOverviewContent = document.getElementById('judgeOverviewContent');
            if (!judgeOverviewContent) return;
            
            judgeOverviewContent.innerHTML = `
                <div class="text-center">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <div class="mt-2">Refreshing your scores...</div>
                </div>
            `;
            loadJudgeOverview();
        }

        // Helper function to show overwrite modal from judge overview
        function showOverwriteModalFromOverview(scoreId, firstName, lastName, bibNumber, runNumber, scoreValue, status, figuresJsonEncoded) {
            // Decode the figures JSON
            let figuresJson = '';
            if (figuresJsonEncoded) {
                try {
                    figuresJson = unescape(figuresJsonEncoded);
                } catch (e) {
                    console.error('Error decoding figures JSON:', e);
                }
            }
            
            // Create a score object in the format that showOverwriteModal expects
            const score = {
                score_id: scoreId,
                first_name: firstName,
                last_name: lastName,
                bib: bibNumber,
                run_number: runNumber,
                score_value: scoreValue,
                status: status,
                figures_json: figuresJson
            };
            
            // Add to latestScores array temporarily (or use existing if it matches)
            const tempIndex = latestScores.findIndex(s => s.score_id === scoreId);
            if (tempIndex >= 0) {
                // Update existing entry
                latestScores[tempIndex] = score;
                showOverwriteModal(tempIndex);
            } else {
                // Add new entry and use that index
                latestScores.push(score);
                showOverwriteModal(latestScores.length - 1);
            }
        }

        function refreshEventSummary() {
            const eventSummaryContent = document.getElementById('eventSummaryContent');
            if (!eventSummaryContent) return;
            
            eventSummaryContent.innerHTML = `
                <div class="text-center">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <div class="mt-2">Refreshing event results...</div>
                </div>
            `;
            loadEventSummary();
        }

        function exportSummary() {
            const eventId = <?= json_encode($event_id) ?>;
            const summaryType = document.querySelector('input[name="summaryType"]:checked')?.value || 'final';
            
            if (!eventId) {
                alert('No event selected for export');
                return;
            }
            
            // Open export in new window
            window.open(`export_summary.php?event_id=${eventId}&type=${summaryType}`, '_blank');
        }

        // Figures card toggle functionality
        function toggleFiguresCard() {
            const cardBody = document.getElementById('figuresCardBody');
            const toggleIcon = document.getElementById('figuresToggleIcon');
            
            if (cardBody && toggleIcon) {
                const isCollapsed = cardBody.classList.contains('collapsed');
                
                if (isCollapsed) {
                    // Expand the card
                    cardBody.classList.remove('collapsed');
                    toggleIcon.classList.remove('collapsed');
                    toggleIcon.className = 'fas fa-chevron-up toggle-icon';
                    
                    // Store state in localStorage
                    localStorage.setItem('figuresCardCollapsed', 'false');
                } else {
                    // Collapse the card
                    cardBody.classList.add('collapsed');
                    toggleIcon.classList.add('collapsed');
                    toggleIcon.className = 'fas fa-chevron-down toggle-icon collapsed';
                    
                    // Store state in localStorage
                    localStorage.setItem('figuresCardCollapsed', 'true');
                }
            }
        }

        // Initialize figures card state on page load
        function initializeFiguresCardState() {
            const cardBody = document.getElementById('figuresCardBody');
            const toggleIcon = document.getElementById('figuresToggleIcon');
            
            if (cardBody && toggleIcon) {
                // Check localStorage for saved state
                const isCollapsed = localStorage.getItem('figuresCardCollapsed') === 'true';
                
                if (isCollapsed) {
                    cardBody.classList.add('collapsed');
                    toggleIcon.classList.add('collapsed');
                    toggleIcon.className = 'fas fa-chevron-down toggle-icon collapsed';
                } else {
                    cardBody.classList.remove('collapsed');
                    toggleIcon.classList.remove('collapsed');
                    toggleIcon.className = 'fas fa-chevron-up toggle-icon';
                }
            }
        }

        // Global functions for edit modal figure selection (referenced in onclick handlers)
        function toggleEditFigureChip(button, category, item) {
            if (!window.__EDIT_FIGURE_SELECTIONS__[category]) {
                window.__EDIT_FIGURE_SELECTIONS__[category] = new Set();
            }
            
            const isActive = button.classList.contains('active');
            
            if (isActive) {
                // Remove selection
                button.classList.remove('active');
                window.__EDIT_FIGURE_SELECTIONS__[category].delete(item);
                
                // Remove category if empty
                if (window.__EDIT_FIGURE_SELECTIONS__[category].size === 0) {
                    delete window.__EDIT_FIGURE_SELECTIONS__[category];
                }
            } else {
                // Add selection
                button.classList.add('active');
                window.__EDIT_FIGURE_SELECTIONS__[category].add(item);
            }
            
            updateEditFiguresJson();
            updateEditFigureSummary();
            
            console.log('Updated edit figure selections:', window.__EDIT_FIGURE_SELECTIONS__);
        }

        function clearEditFigureCategory(category) {
            // Remove from selections
            delete window.__EDIT_FIGURE_SELECTIONS__[category];
            
            // Update UI
            document.querySelectorAll(`#editFigureGrid .figure-chip[data-cat="${category}"]`).forEach(chip => {
                chip.classList.remove('active');
            });
            
            updateEditFiguresJson();
            updateEditFigureSummary();
            
            console.log('Cleared edit figure category:', category);
        }

        // Modal event listeners
        document.addEventListener('DOMContentLoaded', function() {
            // Load content when modals are shown
            document.getElementById('judgeOverviewModal').addEventListener('shown.bs.modal', loadJudgeOverview);
            document.getElementById('eventSummaryModal').addEventListener('shown.bs.modal', loadEventSummary);
            
            // Refresh summary when options change
            document.querySelectorAll('input[name="summaryType"], input[name="displayFormat"]').forEach(input => {
                input.addEventListener('change', loadEventSummary);
            });
        });

        // Debug functionality - Global scope for onclick handlers
        let apiCallLog = [];

        function toggleDebugSection() {
            const section = document.getElementById('debugSection');
            section.style.display = section.style.display === 'none' ? 'block' : 'none';
        }

        // ===== DRAWING BOARD FUNCTIONALITY =====
        let drawingBoardState = {
            canvas: null,
            ctx: null,
            backgroundStrokes: [],
            foregroundStrokes: [],
            currentLayer: 'foreground',
            backgroundLocked: true,
            backgroundOpacity: 0.3,
            currentStroke: null,
            currentColor: '#000000',
            currentSize: 2,
            mode: 'draw',
            isDrawing: false,
            isMovingSelection: false,
            lastMovePos: null,
            selectedStrokeIds: new Set(),
            currentDrawingId: null,
            undoStack: []
        };

        function initializeDrawingBoard() {
            const canvas = document.getElementById('drawingCanvas');
            if (!canvas) {
                console.error('Drawing canvas not found!');
                return;
            }
            
            const ctx = canvas.getContext('2d');
            
            drawingBoardState.canvas = canvas;
            drawingBoardState.ctx = ctx;
            
            // Fit canvas to wrapper
            fitDrawingCanvasToWrapper();
            
            // Wire up toolbar buttons
            wireDrawingToolbar();
            
            // Set up canvas event listeners
            setupDrawingCanvasEvents();
            
            // Window resize handler
            window.addEventListener('resize', fitDrawingCanvasToWrapper);
            
            console.log('Drawing board initialized successfully');
        }

        function fitDrawingCanvasToWrapper() {
            const wrapper = document.getElementById('canvasWrapper');
            const toolbar = document.querySelector('.drawing-toolbar');
            const infoBar = document.querySelector('.bg-light.border-bottom');
            const notesSection = wrapper.nextElementSibling;
            
            const wrapperRect = wrapper.getBoundingClientRect();
            const toolbarH = toolbar ? toolbar.offsetHeight : 0;
            const infoH = infoBar ? infoBar.offsetHeight : 0;
            const notesH = notesSection ? notesSection.offsetHeight : 0;
            
            const verticalPadding = 20;
            const availableHeight = window.innerHeight - toolbarH - infoH - notesH - verticalPadding - 120;
            
            const width = Math.max(400, wrapperRect.width - 12);
            const height = Math.max(400, availableHeight);
            
            drawingBoardState.canvas.width = width;
            drawingBoardState.canvas.height = height;
            redrawCanvas();
        }

        function wireDrawingToolbar() {
            // Color buttons
            document.querySelectorAll('.color-dot').forEach(btn => {
                btn.addEventListener('click', () => {
                    document.querySelectorAll('.color-dot').forEach(b => b.classList.remove('active'));
                    btn.classList.add('active');
                    drawingBoardState.currentColor = btn.getAttribute('data-color');
                    setDrawingMode('draw');
                });
            });
            
            // Size buttons
            document.querySelectorAll('.size-btn').forEach(btn => {
                btn.addEventListener('click', () => {
                    document.querySelectorAll('.size-btn').forEach(b => b.classList.remove('active'));
                    btn.classList.add('active');
                    drawingBoardState.currentSize = parseInt(btn.getAttribute('data-size'), 10);
                    setDrawingMode('draw');
                });
            });
            
            // Mode buttons
            document.getElementById('drawModeBtn').addEventListener('click', () => setDrawingMode('draw'));
            document.getElementById('eraseModeBtn').addEventListener('click', () => setDrawingMode('erase'));
            document.getElementById('selectModeBtn').addEventListener('click', () => setDrawingMode('select'));
            
            // Clear button
            document.getElementById('clearBtn').addEventListener('click', clearDrawing);
            
            // Undo button
            document.getElementById('undoBtn').addEventListener('click', undoLastStroke);
            
            // Layer buttons
            document.getElementById('layerFgBtn').addEventListener('click', () => setDrawingLayer('foreground'));
            document.getElementById('layerBgBtn').addEventListener('click', () => setDrawingLayer('background'));
            
            // Background lock
            document.getElementById('bgLockCheckbox').addEventListener('change', (e) => {
                drawingBoardState.backgroundLocked = e.target.checked;
                if (e.target.checked && drawingBoardState.currentLayer === 'background') {
                    setDrawingLayer('foreground');
                }
            });
            
            // Background opacity
            document.getElementById('bgOpacityRange').addEventListener('input', (e) => {
                const val = parseInt(e.target.value, 10);
                drawingBoardState.backgroundOpacity = Math.max(0.1, Math.min(1, val / 100));
                redrawCanvas();
            });
            
            // Transform buttons
            document.getElementById('scaleUpBtn').addEventListener('click', () => transformSelection(1.1, 0));
            document.getElementById('scaleDownBtn').addEventListener('click', () => transformSelection(0.9, 0));
            document.getElementById('rotateLeftBtn').addEventListener('click', () => transformSelection(1, -5));
            document.getElementById('rotateRightBtn').addEventListener('click', () => transformSelection(1, 5));
            
            // Save button
            const saveBtn = document.getElementById('saveDrawingBtn');
            if (saveBtn) {
                saveBtn.addEventListener('click', saveDrawing);
                console.log('Save drawing button event listener attached');
            } else {
                console.error('saveDrawingBtn not found!');
            }
            
            // Template buttons (placeholders for now)
            document.getElementById('loadTemplateBtn').addEventListener('click', loadTemplate);
            document.getElementById('saveAsTemplateBtn').addEventListener('click', saveAsTemplate);
            
            // Open drawing board button
            const openBtn = document.getElementById('openDrawingBoardBtn');
            if (openBtn) {
                openBtn.addEventListener('click', openDrawingBoard);
                console.log('Drawing board button event listener attached');
            } else {
                console.error('openDrawingBoardBtn not found!');
            }
        }

        function setDrawingMode(mode) {
            drawingBoardState.mode = mode;
            document.getElementById('drawModeBtn').classList.remove('active');
            document.getElementById('eraseModeBtn').classList.remove('active');
            document.getElementById('selectModeBtn').classList.remove('active');
            
            if (mode === 'draw') document.getElementById('drawModeBtn').classList.add('active');
            if (mode === 'erase') document.getElementById('eraseModeBtn').classList.add('active');
            if (mode === 'select') document.getElementById('selectModeBtn').classList.add('active');
        }

        function setDrawingLayer(layer) {
            if (layer === 'background' && drawingBoardState.backgroundLocked) return;
            
            drawingBoardState.currentLayer = layer;
            drawingBoardState.selectedStrokeIds.clear();
            
            document.getElementById('layerFgBtn').classList.remove('active');
            document.getElementById('layerBgBtn').classList.remove('active');
            
            if (layer === 'foreground') {
                document.getElementById('layerFgBtn').classList.add('active');
            } else {
                document.getElementById('layerBgBtn').classList.add('active');
            }
            
            redrawCanvas();
        }

        function getActiveStrokesArray() {
            return drawingBoardState.currentLayer === 'background' 
                ? drawingBoardState.backgroundStrokes 
                : drawingBoardState.foregroundStrokes;
        }

        function setupDrawingCanvasEvents() {
            const canvas = drawingBoardState.canvas;
            
            canvas.addEventListener('mousedown', handleCanvasPointerDown);
            canvas.addEventListener('mousemove', handleCanvasPointerMove);
            canvas.addEventListener('mouseup', handleCanvasPointerUp);
            canvas.addEventListener('mouseleave', handleCanvasPointerUp);
            
            canvas.addEventListener('touchstart', handleCanvasPointerDown, { passive: false });
            canvas.addEventListener('touchmove', handleCanvasPointerMove, { passive: false });
            canvas.addEventListener('touchend', handleCanvasPointerUp, { passive: false });
            canvas.addEventListener('touchcancel', handleCanvasPointerUp, { passive: false });
        }

        function getCanvasPos(evt) {
            const canvas = drawingBoardState.canvas;
            const rect = canvas.getBoundingClientRect();
            let clientX, clientY;
            
            if (evt.touches && evt.touches[0]) {
                clientX = evt.touches[0].clientX;
                clientY = evt.touches[0].clientY;
            } else {
                clientX = evt.clientX;
                clientY = evt.clientY;
            }
            
            // Calculate the scale factor between canvas size and display size
            const scaleX = canvas.width / rect.width;
            const scaleY = canvas.height / rect.height;
            
            // Convert mouse position to canvas coordinates
            return {
                x: (clientX - rect.left) * scaleX,
                y: (clientY - rect.top) * scaleY
            };
        }

        function handleCanvasPointerDown(evt) {
            evt.preventDefault();
            
            if (drawingBoardState.currentLayer === 'background' && drawingBoardState.backgroundLocked) return;
            
            const pos = getCanvasPos(evt);
            
            if (drawingBoardState.mode === 'draw') {
                drawingBoardState.isDrawing = true;
                drawingBoardState.currentStroke = {
                    id: Date.now() + '_' + Math.random().toString(16).slice(2),
                    color: drawingBoardState.currentColor,
                    size: drawingBoardState.currentSize,
                    points: [pos]
                };
                getActiveStrokesArray().push(drawingBoardState.currentStroke);
                redrawCanvas();
            }
        }

        function handleCanvasPointerMove(evt) {
            if (drawingBoardState.currentLayer === 'background' && drawingBoardState.backgroundLocked) return;
            
            if (drawingBoardState.mode === 'draw' && drawingBoardState.isDrawing) {
                evt.preventDefault();
                const pos = getCanvasPos(evt);
                drawingBoardState.currentStroke.points.push(pos);
                redrawCanvas();
            }
        }

        function handleCanvasPointerUp(evt) {
            if (drawingBoardState.mode === 'draw' && drawingBoardState.isDrawing) {
                drawingBoardState.isDrawing = false;
                // Add to undo stack
                drawingBoardState.undoStack.push({
                    layer: drawingBoardState.currentLayer,
                    stroke: drawingBoardState.currentStroke
                });
                document.getElementById('undoBtn').disabled = false;
                drawingBoardState.currentStroke = null;
            }
        }

        function redrawCanvas() {
            const ctx = drawingBoardState.ctx;
            const canvas = drawingBoardState.canvas;
            
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            // Draw background with opacity
            ctx.save();
            ctx.globalAlpha = drawingBoardState.backgroundOpacity;
            drawStrokesArray(drawingBoardState.backgroundStrokes);
            ctx.restore();
            
            // Draw foreground
            drawStrokesArray(drawingBoardState.foregroundStrokes);
        }

        function drawStrokesArray(strokes) {
            const ctx = drawingBoardState.ctx;
            
            strokes.forEach(stroke => {
                if (!stroke.points || stroke.points.length === 0) return;
                
                ctx.strokeStyle = stroke.color;
                ctx.lineWidth = stroke.size;
                ctx.lineCap = 'round';
                ctx.lineJoin = 'round';
                
                ctx.beginPath();
                ctx.moveTo(stroke.points[0].x, stroke.points[0].y);
                
                for (let i = 1; i < stroke.points.length; i++) {
                    ctx.lineTo(stroke.points[i].x, stroke.points[i].y);
                }
                
                ctx.stroke();
            });
        }

        async function clearDrawing() {
            const confirmed = await showConfirm(
                'Clear all strokes on current layer?',
                'Clear Drawing',
                'Clear Layer',
                'btn-danger'
            );
            
            if (confirmed) {
                if (drawingBoardState.currentLayer === 'foreground') {
                    drawingBoardState.foregroundStrokes = [];
                } else {
                    drawingBoardState.backgroundStrokes = [];
                }
                drawingBoardState.selectedStrokeIds.clear();
                drawingBoardState.undoStack = [];
                document.getElementById('undoBtn').disabled = true;
                redrawCanvas();
            }
        }

        function undoLastStroke() {
            if (drawingBoardState.undoStack.length === 0) return;
            
            const lastAction = drawingBoardState.undoStack.pop();
            const strokes = lastAction.layer === 'background' 
                ? drawingBoardState.backgroundStrokes 
                : drawingBoardState.foregroundStrokes;
            
            const index = strokes.findIndex(s => s.id === lastAction.stroke.id);
            if (index !== -1) {
                strokes.splice(index, 1);
            }
            
            if (drawingBoardState.undoStack.length === 0) {
                document.getElementById('undoBtn').disabled = true;
            }
            
            redrawCanvas();
        }

        function transformSelection(scaleFactor, angleDeg) {
            // Placeholder for transform functionality
            console.log('Transform selection:', scaleFactor, angleDeg);
        }

        async function openDrawingBoard() {
            console.log('openDrawingBoard() called');
            
            // Get event_id from PHP variable or hidden input
            const eventId = '<?= $event_id ?? "" ?>';
            
            // Get current context from form elements
            const heatSelect = document.querySelector('select[name="heat_number"]');
            const runRadio = document.querySelector('input[name="run_number"]:checked');
            const participantSelect = document.querySelector('select[name="ep_id"]');
            
            console.log('Context values:', {
                eventId: eventId,
                heatNumber: heatSelect ? heatSelect.value : null,
                runNumber: runRadio ? runRadio.value : null,
                participantId: participantSelect ? participantSelect.value : null
            });
            
            if (!participantSelect || !participantSelect.value) {
                showToast('No Participant Selected', 'Please select a participant first', 'warning');
                return;
            }
            
            // Set context info
            const eventName = '<?= htmlspecialchars($event_name ?? "Unknown Event") ?>';
            const heatNum = heatSelect ? heatSelect.value : '?';
            const runNum = runRadio ? runRadio.value : '?';
            const participantText = participantSelect.options[participantSelect.selectedIndex]?.text || 'Unknown';
            
            document.getElementById('drawingEventInfo').textContent = eventName;
            document.getElementById('drawingHeatRunInfo').textContent = 'Heat ' + heatNum + ', Run ' + runNum;
            document.getElementById('drawingParticipantInfo').textContent = participantText;
            
            console.log('Opening drawing modal...');
            
            // Open modal
            const modal = new bootstrap.Modal(document.getElementById('drawingBoardModal'));
            modal.show();
            
            // Fit canvas after modal is shown
            setTimeout(fitDrawingCanvasToWrapper, 300);
            
            // Try to load existing drawing for this participant/heat/run
            if (runNum && runNum !== '?') {
                await loadExistingDrawing(eventId, heatNum, runNum, participantSelect.value);
            }
        }

        async function loadExistingDrawing(eventId, heatNumber, runNumber, participantId) {
            try {
                const url = '/v2/api/judge_drawings_api.php?action=load_drawing&event_id=' + eventId + 
                            '&heat_number=' + heatNumber + '&run_number=' + runNumber + '&participant_id=' + participantId;
                
                const response = await fetch(url);
                const result = await response.json();
                
                if (result.success && result.drawing) {
                    // Load both layers
                    drawingBoardState.backgroundStrokes = result.drawing.background || [];
                    drawingBoardState.foregroundStrokes = result.drawing.foreground || [];
                    
                    // Load notes
                    const notesField = document.getElementById('drawingNotes');
                    if (notesField && result.drawing.notes) {
                        notesField.value = result.drawing.notes;
                    }
                    
                    // Redraw canvas
                    redrawCanvas();
                    
                    // Mark button as having drawing (if button exists)
                    const drawingBtn = document.getElementById('openDrawingBoardBtn');
                    if (drawingBtn) {
                        drawingBtn.classList.add('has-drawing');
                    }
                    
                    console.log('Loaded existing drawing:', result.drawing.id);
                    showToast('Drawing Loaded', 'Existing drawing loaded for editing', 'info');
                } else {
                    // No existing drawing, try to load last used background
                    console.log('No existing drawing, loading last background...');
                    await loadLastBackgroundQuietly(eventId);
                }
            } catch (error) {
                console.error('Load drawing error:', error);
                // Silently fail, just use empty canvas
            }
        }

        async function loadLastBackgroundQuietly(eventId) {
            try {
                const response = await fetch('/v2/api/judge_drawings_api.php?action=get_last_background&event_id=' + eventId);
                const result = await response.json();
                
                if (result.success && result.background) {
                    drawingBoardState.backgroundStrokes = result.background;
                    redrawCanvas();
                    console.log('Auto-loaded last background:', result.background.length, 'strokes');
                }
            } catch (error) {
                // Silently fail
                console.log('No previous background available');
            }
        }

        async function saveDrawing() {
            console.log('saveDrawing() called');
            
            // Get event_id from PHP variable
            const eventId = '<?= $event_id ?? "" ?>';
            
            const heatSelect = document.querySelector('select[name="heat_number"]');
            const runRadio = document.querySelector('input[name="run_number"]:checked');
            const participantSelect = document.querySelector('select[name="ep_id"]');
            const notesField = document.getElementById('drawingNotes');
            
            const heatNumber = heatSelect ? heatSelect.value : null;
            const runNumber = runRadio ? runRadio.value : null;
            const participantId = participantSelect ? participantSelect.value : null;
            const notes = notesField ? notesField.value : '';
            
            console.log('Form values:', {
                eventId: eventId,
                heatNumber: heatNumber,
                runNumber: runNumber,
                participantId: participantId,
                notes: notes
            });
            
            if (!participantId) {
                console.error('No participant selected');
                showToast('Error', 'No participant selected', 'error');
                return;
            }
            
            if (!runNumber) {
                console.error('No run selected');
                showToast('Error', 'Please select a run number', 'warning');
                return;
            }
            
            const drawingData = {
                event_id: eventId,
                heat_number: heatNumber,
                run_number: runNumber,
                participant_id: participantId,
                canvas_width: drawingBoardState.canvas.width,
                canvas_height: drawingBoardState.canvas.height,
                background: drawingBoardState.backgroundStrokes,
                foreground: drawingBoardState.foregroundStrokes,
                notes: notes
            };
            
            console.log('Drawing data prepared:', drawingData);
            console.log('Background strokes:', drawingBoardState.backgroundStrokes.length);
            console.log('Foreground strokes:', drawingBoardState.foregroundStrokes.length);
            
            // Send to API
            try {
                const response = await fetch('/v2/api/judge_drawings_api.php?action=save_drawing', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(drawingData)
                });
                
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers.get('content-type'));
                
                // Get raw text first to see what we're getting
                const rawText = await response.text();
                console.log('Raw response:', rawText.substring(0, 500));
                
                // Try to parse as JSON
                let result;
                try {
                    result = JSON.parse(rawText);
                } catch (parseError) {
                    console.error('JSON parse error:', parseError);
                    console.error('Response was:', rawText);
                    showToast('Error', 'Server returned invalid response', 'error');
                    return;
                }
                
                if (result.success) {
                    console.log('Drawing saved successfully:', result);
                    showToast('Drawing Saved', result.message || 'Drawing saved successfully', 'success');
                    
                    // Mark button as having drawing (if button exists)
                    const drawingBtn = document.getElementById('openDrawingBoardBtn');
                    if (drawingBtn) {
                        drawingBtn.classList.add('has-drawing');
                    }
                    
                    // Close modal
                    const modal = bootstrap.Modal.getInstance(document.getElementById('drawingBoardModal'));
                    if (modal) modal.hide();
                } else {
                    console.error('Save failed:', result.message);
                    showToast('Error', result.message || 'Failed to save drawing', 'error');
                }
            } catch (error) {
                console.error('Save error:', error);
                showToast('Error', 'Failed to save drawing: ' + error.message, 'error');
            }
        }

        async function loadLastBackground() {
            const eventId = '<?= $event_id ?? "" ?>';
            
            if (!eventId) {
                showToast('Error', 'Event ID not found', 'error');
                return;
            }
            
            try {
                const response = await fetch('/v2/api/judge_drawings_api.php?action=get_last_background&event_id=' + eventId);
                const result = await response.json();
                
                if (result.success) {
                    drawingBoardState.backgroundStrokes = result.background || [];
                    redrawCanvas();
                    showToast('Background Loaded', 'Last used background loaded', 'success');
                    console.log('Loaded background strokes:', result.background.length);
                } else {
                    showToast('Info', result.message || 'No previous background found', 'info');
                }
            } catch (error) {
                console.error('Load background error:', error);
                showToast('Error', 'Failed to load background', 'error');
            }
        }

        async function showTemplateSelector() {
            const eventId = '<?= $event_id ?? "" ?>';
            
            if (!eventId) {
                showToast('Error', 'Event ID not found', 'error');
                return;
            }
            
            try {
                const response = await fetch('/v2/api/judge_drawings_api.php?action=list_templates&event_id=' + eventId);
                const result = await response.json();
                
                if (result.success && result.templates && result.templates.length > 0) {
                    // Build template selection dialog
                    let html = '<div class="list-group">';
                    result.templates.forEach(template => {
                        html += '<button type="button" class="list-group-item list-group-item-action" onclick="loadTemplateById(' + template.id + ')">';
                        html += '<div class="d-flex justify-content-between align-items-center">';
                        html += '<div><strong>' + template.template_name + '</strong>';
                        html += '<br><small class="text-muted">' + template.created_at + '</small></div>';
                        html += '<span class="badge bg-primary">' + template.canvas_width + 'x' + template.canvas_height + '</span>';
                        html += '</div></button>';
                    });
                    html += '</div>';
                    
                    // Show in a Bootstrap modal (you can use a custom modal or sweet alert)
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({
                            title: 'Select Template',
                            html: html,
                            showConfirmButton: false,
                            width: '600px'
                        });
                    } else {
                        showToast('Templates', result.templates.length + ' template(s) available', 'info');
                        console.log('Templates:', result.templates);
                    }
                } else {
                    showToast('Info', 'No templates found. Save your first template!', 'info');
                }
            } catch (error) {
                console.error('Load templates error:', error);
                showToast('Error', 'Failed to load templates', 'error');
            }
        }

        async function loadTemplateById(templateId) {
            try {
                const response = await fetch('/v2/api/judge_drawings_api.php?action=load_template&template_id=' + templateId);
                const result = await response.json();
                
                if (result.success && result.template) {
                    drawingBoardState.backgroundStrokes = result.template.background || [];
                    redrawCanvas();
                    showToast('Template Loaded', 'Template "' + result.template.name + '" loaded', 'success');
                    console.log('Loaded template:', result.template.name);
                    
                    // Close SweetAlert if it's open
                    if (typeof Swal !== 'undefined') {
                        Swal.close();
                    }
                } else {
                    showToast('Error', result.message || 'Failed to load template', 'error');
                }
            } catch (error) {
                console.error('Load template error:', error);
                showToast('Error', 'Failed to load template', 'error');
            }
        }

        async function saveAsTemplate() {
            const eventId = '<?= $event_id ?? "" ?>';
            
            if (!eventId) {
                showToast('Error', 'Event ID not found', 'error');
                return;
            }
            
            if (drawingBoardState.backgroundStrokes.length === 0) {
                showToast('Error', 'Background layer is empty. Draw something first!', 'warning');
                return;
            }
            
            // Prompt for template name
            if (typeof Swal !== 'undefined') {
                const { value: templateName } = await Swal.fire({
                    title: 'Save Background as Template',
                    input: 'text',
                    inputLabel: 'Template Name',
                    inputPlaceholder: 'e.g., "Slopestyle Course A"',
                    showCancelButton: true,
                    inputValidator: (value) => {
                        if (!value) {
                            return 'Please enter a template name';
                        }
                    }
                });
                
                if (!templateName) return;
                
                await saveTemplateToServer(templateName);
            } else {
                const templateName = prompt('Enter template name:');
                if (!templateName) return;
                await saveTemplateToServer(templateName);
            }
        }

        async function saveTemplateToServer(templateName) {
            const eventId = '<?= $event_id ?? "" ?>';
            
            const templateData = {
                event_id: eventId,
                template_name: templateName,
                background: drawingBoardState.backgroundStrokes,
                canvas_width: drawingBoardState.canvas.width,
                canvas_height: drawingBoardState.canvas.height
            };
            
            try {
                const response = await fetch('/v2/api/judge_drawings_api.php?action=save_template', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(templateData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showToast('Template Saved', result.message || 'Template saved successfully', 'success');
                    console.log('Template saved:', result);
                } else {
                    showToast('Error', result.message || 'Failed to save template', 'error');
                }
            } catch (error) {
                console.error('Save template error:', error);
                showToast('Error', 'Failed to save template: ' + error.message, 'error');
            }
        }

        function loadTemplate() {
            // This is now replaced by showTemplateSelector
            showTemplateSelector();
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            initializeDrawingBoard();
        });

        // Debug functionality continues below
        function toggleDebugSection() {
            const debugSection = document.getElementById('debugSection');
            if (debugSection.style.display === 'none') {
                debugSection.style.display = 'block';
                refreshIncomingData();
            } else {
                debugSection.style.display = 'none';
            }
        }

        // Theme selector functions
        function selectTheme(themeName) {
            // Update radio selection
            const radio = document.getElementById('theme_' + themeName);
            if (radio) {
                radio.checked = true;
            }
        }

        function updateZoomDisplay(value) {
            const zoomValueDisplay = document.getElementById('zoomValue');
            if (zoomValueDisplay) {
                zoomValueDisplay.textContent = value + '%';
            }
        }

        function applyZoom(zoomPercent) {
            // Apply zoom to body element
            document.body.style.zoom = zoomPercent + '%';
            
            // Save to localStorage
            localStorage.setItem('panel_zoom', zoomPercent);
        }

        function resetZoom() {
            const zoomRange = document.getElementById('zoomRange');
            if (zoomRange) {
                zoomRange.value = 100;
                updateZoomDisplay(100);
            }
            applyZoom(100);
            showToast('Reset', 'Zoom reset to 100%', 'success');
        }

        function applyThemeAndZoom() {
            const selectedTheme = document.querySelector('input[name="theme_selector"]:checked');
            if (!selectedTheme) {
                showToast('Error', 'Please select a theme', 'error');
                return;
            }

            const theme = selectedTheme.value;
            const zoomValue = document.getElementById('zoomRange').value;
            
            // Set cookie for theme (365 days)
            const expiryDate = new Date();
            expiryDate.setTime(expiryDate.getTime() + (365 * 24 * 60 * 60 * 1000));
            document.cookie = 'panel_theme=' + theme + '; expires=' + expiryDate.toUTCString() + '; path=/';
            
            // Apply and save zoom
            applyZoom(zoomValue);
            
            // Show success message
            showToast('Applied', 'Theme and zoom settings applied successfully', 'success');
            
            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('themeSelectorModal'));
            if (modal) {
                modal.hide();
            }
            
            // Reload only if theme changed
            const cookies = document.cookie.split(';');
            let currentTheme = 'brite';
            for (let cookie of cookies) {
                const [name, value] = cookie.trim().split('=');
                if (name === 'panel_theme') {
                    currentTheme = value;
                    break;
                }
            }
            
            if (currentTheme !== theme) {
                setTimeout(() => {
                    window.location.reload();
                }, 500);
            }
        }

        function applySelectedTheme() {
            // Kept for backward compatibility - redirects to new function
            applyThemeAndZoom();
        }

        function openThemeSelector() {
            // Get current theme from cookie
            const cookies = document.cookie.split(';');
            let currentTheme = 'brite'; // default
            
            for (let cookie of cookies) {
                const [name, value] = cookie.trim().split('=');
                if (name === 'panel_theme') {
                    currentTheme = value;
                    break;
                }
            }
            
            // Pre-select current theme
            const radio = document.getElementById('theme_' + currentTheme);
            if (radio) {
                radio.checked = true;
            }
            
            // Load and set current zoom
            const savedZoom = localStorage.getItem('panel_zoom') || '100';
            const zoomRange = document.getElementById('zoomRange');
            if (zoomRange) {
                zoomRange.value = savedZoom;
                updateZoomDisplay(savedZoom);
            }
            
            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('themeSelectorModal'));
            modal.show();
        }
        
        // Apply saved zoom on page load
        (function() {
            const savedZoom = localStorage.getItem('panel_zoom');
            if (savedZoom) {
                document.body.style.zoom = savedZoom + '%';
            }
        })();

        function refreshIncomingData() {
            const content = document.getElementById('incomingDataContent');
            
            // Gather all incoming data
            const incomingData = {
                event_info: {
                    event_id: '<?= $event_id ?? "Not set" ?>',
                    event_name: '<?= htmlspecialchars($assigned_events[0]['name'] ?? 'No Event') ?>',
                    event_date: '<?= $event_date ?? "Not set" ?>'
                },
                judge_info: {
                    judge_id: '<?= $judge_id ?? "Not set" ?>',
                    judge_name: '<?= htmlspecialchars($judge_name ?? 'Not set') ?>',
                    user_type: '<?= $_SESSION['user_type'] ?? "Not set" ?>'
                },
                participant_info: {
                    participant_id: document.getElementById('participant_id')?.value || 'Not selected',
                    run_number: document.getElementById('run_number')?.value || 'Not set',
                    heat_number: document.getElementById('heat_number')?.value || 'Not set',
                    bib_number: document.querySelector('input[name="bib_number"]')?.value || 'Not set'
                },
                form_data: {
                    total_score: document.getElementById('totalScore')?.textContent || 'Not calculated',
                            execution_score: document.getElementById('executionScore')?.textContent || '0',
                        penalty_total: document.getElementById('penaltyTotal')?.textContent || '0',
                        selected_figures: Array.from(document.querySelectorAll('input[name="figures[]"]:checked')).map(cb => ({
                            figure_id: cb.value,
                            figure_name: cb.closest('.figure-card')?.querySelector('.figure-name')?.textContent || 'Unknown'
                        }))
                    },
                    current_status: {
                        manual_mode: <?= $manual_mode ? 'true' : 'false' ?>,
                        is_mobile: window.innerWidth <= 768,
                        form_valid: document.getElementById('scoreForm')?.checkValidity() || false,
                        timestamp: new Date().toISOString()
                    }
                };
                
                content.innerHTML = '<pre>' + JSON.stringify(incomingData, null, 2) + '</pre>';
            }
            
            function logApiCall(method, url, data, response) {
                const logEntry = {
                    timestamp: new Date().toISOString(),
                    method: method,
                    url: url,
                    request_data: data,
                    response: response,
                    status: response?.status || 'Unknown'
                };
                
                apiCallLog.unshift(logEntry); // Add to beginning
                
                // Keep only last 10 entries
                if (apiCallLog.length > 10) {
                    apiCallLog = apiCallLog.slice(0, 10);
                }
                
                updateApiCallsDisplay();
            }
            
            function updateApiCallsDisplay() {
                const content = document.getElementById('apiCallsContent');
                if (apiCallLog.length === 0) {
                    content.innerHTML = 'No API calls yet...';
                    return;
                }
                
                let html = '';
                apiCallLog.forEach((call, index) => {
                    const statusColor = call.status >= 200 && call.status < 300 ? 'text-success' : 'text-danger';
                    html += `
                        <div class="border-bottom pb-2 mb-2">
                            <div class="d-flex justify-content-between">
                                <strong class="${statusColor}">${call.method} ${call.url}</strong>
                                <small class="text-muted">${new Date(call.timestamp).toLocaleTimeString()}</small>
                            </div>
                            <div class="mt-1">
                                <small><strong>Request:</strong></small>
                                <pre class="small mt-1 p-1 bg-light">${JSON.stringify(call.request_data, null, 1)}</pre>
                                <small><strong>Response:</strong></small>
                                <pre class="small mt-1 p-1 bg-light">${JSON.stringify(call.response, null, 1)}</pre>
                            </div>
                        </div>
                    `;
                });
                content.innerHTML = html;
            }
            
            function clearApiLog() {
                apiCallLog = [];
                updateApiCallsDisplay();
            }
            
            // Initialize debug data when DOM is ready
            document.addEventListener('DOMContentLoaded', function() {
                refreshIncomingData();
                
                // Refresh incoming data periodically
                setInterval(refreshIncomingData, 5000);
            });
    </script>

        <?php // include '../admin/footer.php'; ?>
            <!-- StyleScore JavaScript includes -->
    <?php include_once __DIR__ . '/../includes/scripts.php'; ?>

    <!-- Dashboard specific JavaScript -->
    <script>
        // Dashboard initialization
        document.addEventListener('DOMContentLoaded', function() {
            // Any dashboard-specific initialization can go here
            console.log('StyleScore Dashboard loaded successfully');
            
            // Toggle chevron icon for Other Heat Scores Card collapse
            const otherHeatScoresBody = document.getElementById('otherHeatScoresBody');
            if (otherHeatScoresBody) {
                otherHeatScoresBody.addEventListener('show.bs.collapse', function () {
                    const btn = document.querySelector('[data-bs-target="#otherHeatScoresBody"]');
                    if (btn) {
                        btn.querySelector('i').className = 'fas fa-chevron-up';
                    }
                });
                
                otherHeatScoresBody.addEventListener('hide.bs.collapse', function () {
                    const btn = document.querySelector('[data-bs-target="#otherHeatScoresBody"]');
                    if (btn) {
                        btn.querySelector('i').className = 'fas fa-chevron-down';
                    }
                });
            }
        });
    </script>

    <!-- Floating Dark Mode Toggle -->
    <footer class="floating-dark-mode-toggle pe-2 ps-2 w-100">
        <!-- Connection Status -->
         <div class="row">
            <div class="col-12 d-flex flex-row justify-content-between mt-0 w-1 w-100 bg-gradient bg-body-tertiary ps-4 pe-4 pt-1 pb-1">
                <div class="d-flex justify-content-between align-items-center gap-3 w-100">
                    <div>
                        <img src="/v2/assets/img/SVG/logo-ss-_1.svg" alt="StyleScore" style="height: 24px;" class="me-2">
                        <img src="/v2/assets/img/logo-dark.png" alt="StyleScore" style="height: 35px;" class="me-2">
                    </div>
                    <div>
                        <small class="badge bg-dark-subtle" style="width: auto;">
                            <i class="fas fa-wifi me-1" id="connectionIcon"></i>
                            <span id="connectionStatus">Connecting...</span>
                        </small>
                        <button type="button" class="badge bg-body text-dark btn btn-sm" onclick="forceRefreshHeatData()">
                            <i class="fas fa-sync-alt me-1"></i>Refresh
                        </button>
                    </div>
                    <button type="button" class="btn btn-secondary btn-sm" id="darkModeToggle" title="Toggle Dark Mode">
                    <i class="fas fa-moon me-2 ms-2" id="darkModeIcon"></i>
                    </button>
                </div>
            
                
            </div> 
        </div>
        </footer>

    <!-- No Active Heat Overlay -->
    <div class="no-active-heat-overlay" id="noActiveHeatOverlay">
        <div class="no-active-heat-content p-2 bg-dark-subtle">
            <div class="mb-1">
                <img src="/v2/assets/img/SVG/pause.svg" alt="No Active Heat" style="width: 90px; height: 90px;">
            </div>
            
            <h5><strong id="noActiveHeatEventName"><?= htmlspecialchars($event_name ?: 'Unknown Event') ?></strong> - No Active Heat</h5>
            <small>
                The event is currently paused. Please wait for the head judge or admin to activate a heat before scoring.
            </small>
                   <div class="btn-group bg-light" role="group">
                                    <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#judgeOverviewModal">
                                        <i class="fas fa-chart-line me-1"></i>My Scores Overview
                                    </button>
                                    
                                    <button type="button" class="btn btn-outline-info" data-bs-toggle="modal" data-bs-target="#eventSummaryModal">
                                        <i class="fas fa-trophy me-1"></i>Event Results
                                    </button>
                                </div>    
            <!-- Heat Flow Bracket Preview -->
            <div class="heat-flow-preview mt-3  bg-dark-subtle" id="heatFlowPreview">
                <div class="text-center text-muted py-3">
                    <div class="spinner-border spinner-border-sm me-2" role="status"></div>
                    <span>Loading heat flow...</span>
                </div>
            </div>
            
            <div class="checking-status">
                <button type="button" class="btn btn-info" onclick="window.location.href='dashboard.php'">
                                       <i class="fas fa-tv  fa-md-2x d-block mb-2"></i>
                                        <strong class="d-block">Dashboard</strong>
                                        <small class="d-none d-md-block">View dashboard</small>
                                    </button>
                <div class="spinner"></div>
                <span>Checking for active heats...</span>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap Bundle (includes Popper.js) -->
<script src="/v2/assets/js/bootstrap.bundle.min.js" crossorigin="anonymous"></script>

</body>

</html>