<?php
include '../includes/auth.php';
include '../includes/db.php';

// Get judge_id from session - use user_id as fallback
$judge_id = $_SESSION['judge_id'] ?? $_SESSION['user_id'] ?? 0;
$user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';

// Check if user is a head judge in users table OR is admin
$is_head_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ? AND judge_id = 1");
$is_head_stmt->execute([$judge_id]);
$is_head = $is_head_stmt->fetchColumn();

if (!$is_head && $user_role !== 'admin') {
    header('Location: ../login.php');
    exit;
}
// Handle approve all participants action
if (isset($_POST['action']) && $_POST['action'] === 'approve_all_participant') {
        $event_id = $_POST['event_id'];
        $heat_number = $_POST['heat_number'];
        $run_number = $_POST['run_number'];
        $participant_id = $_POST['participant_id'];
        
        try {
            // Get the judge_id from users table for the approved_by field
            $judge_lookup_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ?");
            $judge_lookup_stmt->execute([$judge_id]);
            $approved_by_judge_id = $judge_lookup_stmt->fetchColumn();
            
            // Get participant details for notification
            $participant_stmt = $pdo->prepare("
                SELECT p.first_name, p.last_name, ep.bib_number
                FROM event_participants ep
                JOIN participants p ON ep.participant_id = p.id
                WHERE ep.id = ?
            ");
            $participant_stmt->execute([$participant_id]);
            $participant_details = $participant_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Get all unapproved scores for this participant in this heat/run
            $scores_stmt = $pdo->prepare("
                SELECT s.id
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                WHERE ep.event_id = ? AND ep.heat_number = ? AND r.run_number = ? 
                AND ep.id = ? AND s.is_approved = 0
            ");
            $scores_stmt->execute([$event_id, $heat_number, $run_number, $participant_id]);
            $score_ids = $scores_stmt->fetchAll(PDO::FETCH_COLUMN);
            
            // Prepare notification data for frontend
            $notification_data = null;
            if ($participant_details) {
                $participant_name = $participant_details['first_name'] . ' ' . $participant_details['last_name'];
                $notification_data = [
                    'bib' => $participant_details['bib_number'],
                    'participant_name' => $participant_name,
                    'count' => count($score_ids),
                    'heat' => $heat_number,
                    'run' => $run_number
                ];
            }
            
            if (count($score_ids) > 0) {
                // Approve all scores for this participant - using judge_id instead of user id
                $placeholders = str_repeat('?,', count($score_ids) - 1) . '?';
                $approve_stmt = $pdo->prepare("
                    UPDATE scores 
                    SET is_approved = 1, approved_by = ?, approved_at = NOW() 
                    WHERE id IN ($placeholders)
                ");
                $params = array_merge([$approved_by_judge_id], $score_ids);
                $approve_stmt->execute($params);
                
                // Update participant status to 'complete' after all scores are approved
                $participant_update = $pdo->prepare("
                    UPDATE event_participants 
                    SET participant_status = 'complete' 
                    WHERE id = ?
                ");
                $participant_update->execute([$participant_id]);
                
                // Clear bib_performing from event_heat_settings if this participant is currently performing
                if ($participant_details && isset($participant_details['bib_number'])) {
                    $clear_performing_stmt = $pdo->prepare("
                        UPDATE event_heat_settings 
                        SET bib_performing = NULL 
                        WHERE event_id = ? AND heat_number = ? AND bib_performing = ?
                    ");
                    $clear_performing_stmt->execute([$event_id, $heat_number, $participant_details['bib_number']]);
                }
                
                // Send to notification API if we have participant details
                if ($participant_details) {
                    // Get all judge scores for this participant after approval
                    $all_scores_stmt = $pdo->prepare("
                        SELECT s.score_value, s.status, u.username as judge_name, s.is_approved
                        FROM scores s
                        JOIN runs r ON s.run_id = r.id
                        JOIN event_participants ep ON r.event_participant_id = ep.id
                        JOIN users u ON s.judge_id = u.id
                        WHERE ep.id = ? AND ep.heat_number = ? AND r.run_number = ?
                        ORDER BY u.username
                    ");
                    $all_scores_stmt->execute([$participant_id, $heat_number, $run_number]);
                    $all_judge_scores = $all_scores_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    // Calculate final result and rank
                    $approved_scores = [];
                    $all_scores_info = [];
                    foreach ($all_judge_scores as $judge_score) {
                        $judge_display = $judge_score['status'] === 'OK' ? $judge_score['score_value'] : $judge_score['status'];
                        $all_scores_info[] = [
                            'judge' => $judge_score['judge_name'],
                            'score' => $judge_display,
                            'approved' => (bool)$judge_score['is_approved']
                        ];
                        
                        if ($judge_score['is_approved'] && $judge_score['status'] === 'OK' && $judge_score['score_value'] !== null) {
                            $approved_scores[] = floatval($judge_score['score_value']);
                        }
                    }
                    
                    $final_result = null;
                    $approved_count = count($approved_scores);
                    if ($approved_count > 0) {
                        // Apply drop rule if enough scores
                        if ($approved_count >= 3) {
                            sort($approved_scores);
                            array_shift($approved_scores); // Remove lowest
                            array_pop($approved_scores);   // Remove highest
                        }
                        $final_result = $approved_count > 0 ? round(array_sum($approved_scores) / count($approved_scores), 3) : null;
                    }
                    
                    // Calculate final rank position
                    $rank_position = null;
                    if ($final_result !== null) {
                        $ranking_stmt = $pdo->prepare("
                            SELECT ep.bib_number, ep.id as participant_id,
                                   AVG(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN s.score_value END) as avg_score,
                                   COUNT(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN 1 END) as approved_count
                            FROM event_participants ep
                            JOIN runs r ON ep.id = r.event_participant_id
                            JOIN scores s ON r.id = s.run_id
                            WHERE ep.event_id = ? AND ep.heat_number = ? AND r.run_number = ?
                            GROUP BY ep.id, ep.bib_number
                            HAVING approved_count > 0
                            ORDER BY avg_score DESC
                        ");
                        $ranking_stmt->execute([$event_id, $heat_number, $run_number]);
                        $rankings = $ranking_stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        foreach ($rankings as $index => $ranking) {
                            if ($ranking['participant_id'] == $participant_id) {
                                $rank_position = $index + 1;
                                break;
                            }
                        }
                    }
                    
                    // Update notification data with detailed information
                    $notification_data['all_scores'] = $all_scores_info;
                    $notification_data['final_result'] = $final_result;
                    $notification_data['rank_position'] = $rank_position;
                    $notification_data['total_judges'] = count($all_judge_scores);
                    
                    // Enhanced notification message
                    $result_text = $final_result !== null ? " | Final Result: {$final_result}" : "";
                    $rank_text = $rank_position !== null ? " | Final Rank: #{$rank_position}" : "";
                    
                    $api_data = [
                        'type' => 'success',
                        'title' => 'All Scores Approved',
                        'message' => sprintf(
                            "All %d scores approved for Bib #%s (%s) - Heat %s, Run %s%s%s",
                            count($score_ids),
                            $participant_details['bib_number'],
                            $participant_name,
                            $heat_number,
                            $run_number,
                            $result_text,
                            $rank_text
                        ),
                        'category' => 'judge',
                        'event_id' => $event_id,
                        'user_id' => $notification_user_id ?? 'head_judge',
                        'data' => json_encode([
                            'participant_id' => $participant_id,
                            'participant_bib' => $participant_details['bib_number'],
                            'participant_name' => $participant_name,
                            'scores_count' => count($score_ids),
                            'heat' => $heat_number,
                            'run' => $run_number,
                            'score_ids' => $score_ids,
                            'all_judge_scores' => $all_scores_info,
                            'final_result' => $final_result,
                            'rank_position' => $rank_position,
                            'total_judges' => count($all_judge_scores)
                        ])
                    ];
                    
                    // Send to notification API
                    $api_url = '../api/notification_api.php';
                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, $api_url);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($api_data + ['action' => 'send_notification']));
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                    curl_exec($ch);
                    curl_close($ch);
                }
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'All scores approved for participant',
                    'count' => count($score_ids),
                    'notification_data' => $notification_data
                ]);
            } else {
                echo json_encode([
                    'success' => true, 
                    'message' => 'No pending scores to approve',
                    'count' => 0,
                    'notification_data' => $notification_data
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'success' => false, 
                'message' => 'Error approving scores: ' . $e->getMessage()
            ]);
        }
        exit;
    }

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    if (isset($_POST['action']) && $_POST['action'] === 'get_scores') {
        $event_id = $_POST['event_id'] ?? null;
        $heat_number = $_POST['heat_number'] ?? null;
        $run_number = $_POST['run_number'] ?? null;
        
        if ($event_id) {
            try {
                // UPDATED: Get all judges for this event from users table
                $judges_stmt = $pdo->prepare("
                    SELECT DISTINCT u.id, u.username as name, u.judge_id as is_head, cp.control_point_name as control_point_name, cp.id as control_point_id
                    FROM users u
                    JOIN judge_assignments ja ON u.id = ja.judge_id
                    LEFT JOIN event_control_points cp ON ja.control_point_id = cp.id
                    WHERE ja.event_id = ? 
                    ORDER BY cp.control_point_name ASC, u.username ASC
                ");
                $judges_stmt->execute([$event_id]);
                $judges = $judges_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Build WHERE clause based on what filters are provided
                $where_conditions = ["ep.event_id = ?"];
                $params = [$event_id];
                
                if ($heat_number) {
                    $where_conditions[] = "ep.heat_number = ?";
                    $params[] = $heat_number;
                }
                
                $where_clause = implode(" AND ", $where_conditions);
                
                // Get all participants based on filters - UPDATED to get category from event_categories
                $participants_stmt = $pdo->prepare("
                    SELECT ep.id, ep.bib_number, ep.sort_order, ep.category as event_category, ep.heat_number,
                           p.first_name, p.last_name, p.club, p.category as base_category, p.gender,
                           COALESCE(ec.category_name, ep.category, p.category) as category
                    FROM event_participants ep
                    JOIN participants p ON ep.participant_id = p.id
                    LEFT JOIN event_categories ec ON ep.category_id = ec.id
                    WHERE {$where_clause}
                    ORDER BY ep.heat_number ASC, ep.sort_order ASC, ep.bib_number ASC
                ");
                $participants_stmt->execute($params);
                $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Get scores based on filters with approval status AND created_at for sorting
                // Check if figures_json column exists
                $figures_column = '';
                try {
                    $check_column = $pdo->query("SHOW COLUMNS FROM scores LIKE 'figures_json'");
                    if ($check_column->rowCount() > 0) {
                        $figures_column = ', s.figures_json';
                    }
                } catch (Exception $e) {
                    // Column doesn't exist, continue without it
                }
                
                // Build scores WHERE clause
                $scores_where_conditions = ["ep.event_id = ?"];
                $scores_params = [$event_id];
                
                if ($heat_number) {
                    $scores_where_conditions[] = "ep.heat_number = ?";
                    $scores_params[] = $heat_number;
                }
                
                if ($run_number) {
                    $scores_where_conditions[] = "r.run_number = ?";
                    $scores_params[] = $run_number;
                }
                
                $scores_where_clause = implode(" AND ", $scores_where_conditions);
                
                $scores_stmt = $pdo->prepare("
                    SELECT s.judge_id, ep.id as ep_id, s.score_value, s.status, s.id as score_id, 
                           COALESCE(s.is_approved, 0) as is_approved, s.created_at, r.run_number, ep.heat_number{$figures_column}
                    FROM scores s
                    JOIN runs r ON s.run_id = r.id
                    JOIN event_participants ep ON r.event_participant_id = ep.id
                    WHERE {$scores_where_clause}
                ");
                $scores_stmt->execute($scores_params);
                $scores_raw = $scores_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Organize scores by participant, run, and judge
                $scores = [];
                foreach ($scores_raw as $score) {
                    $run_key = $score['run_number'];
                    if (!isset($scores[$score['ep_id']])) {
                        $scores[$score['ep_id']] = [];
                    }
                    if (!isset($scores[$score['ep_id']][$run_key])) {
                        $scores[$score['ep_id']][$run_key] = [];
                    }
                    $scores[$score['ep_id']][$run_key][$score['judge_id']] = $score;
                }
                
                // Get drop_rule configuration with heat override support
                // Priority: 1) Heat-specific drop_rule, 2) Event format drop_rule, 3) Default 'none'
                // Note: Both events.scoring_format and scoring_format_judges.format_id contain the full unique format name
                $format_stmt = $pdo->prepare("
                    SELECT 
                        ehs.drop_rule as heat_drop_rule,
                        sfj.drop_rule as format_drop_rule,
                        COALESCE(ehs.drop_rule, sfj.drop_rule, 'none') as effective_drop_rule
                    FROM events e
                    LEFT JOIN event_heat_settings ehs ON e.id = ehs.event_id AND ehs.heat_number = ?
                    LEFT JOIN scoring_format_judges sfj ON e.scoring_format = sfj.format_id
                    WHERE e.id = ?
                    LIMIT 1
                ");
                $format_stmt->execute([$heat_number, $event_id]);
                $format_config = $format_stmt->fetch(PDO::FETCH_ASSOC);
                $drop_rule = $format_config ? $format_config['effective_drop_rule'] : 'none';
                $heat_drop_rule = $format_config ? $format_config['heat_drop_rule'] : null;
                $format_drop_rule = $format_config ? $format_config['format_drop_rule'] : null;
                
                // Load diversity rules and heat settings
                $diversity_rules = [];
                $diversity_enabled_for_heat = false;
                $scoring_format_id = null;
                
                try {
                    // Get scoring format ID
                    $format_id_stmt = $pdo->prepare("SELECT scoring_format FROM events WHERE id = ?");
                    $format_id_stmt->execute([$event_id]);
                    $scoring_format_id = $format_id_stmt->fetchColumn();
                    
                    if ($scoring_format_id && $heat_number) {
                        // Check if diversity rules are enabled for this specific heat
                        $diversity_check_stmt = $pdo->prepare("
                            SELECT diversity_rules_enabled 
                            FROM event_heat_settings 
                            WHERE event_id = ? AND heat_number = ?
                        ");
                        $diversity_check_stmt->execute([$event_id, $heat_number]);
                        $diversity_enabled_for_heat = (bool)$diversity_check_stmt->fetchColumn();
                        
                        // Load diversity rules if enabled for this heat
                        if ($diversity_enabled_for_heat) {
                            $div_stmt = $pdo->prepare("
                                SELECT rule_type, scope, group_name, min_distinct, is_optional 
                                FROM scoring_format_diversity_rules 
                                WHERE format_id = ? 
                                ORDER BY rule_type ASC, group_name ASC
                            ");
                            $div_stmt->execute([$scoring_format_id]);
                            $diversity_rules = $div_stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
                        }
                    }
                } catch (Exception $e) {
                    // If diversity check fails, continue without diversity rules
                    error_log("Diversity rules loading failed: " . $e->getMessage());
                }
                
                echo json_encode([
                    'success' => true,
                    'judges' => $judges,
                    'participants' => $participants,
                    'scores' => $scores,
                    'heat_number' => $heat_number,
                    'run_number' => $run_number,
                    'drop_rule' => $drop_rule,
                    'drop_rule_source' => $heat_drop_rule ? 'heat' : 'format',
                    'heat_drop_rule' => $heat_drop_rule,
                    'format_drop_rule' => $format_drop_rule,
                    'diversity_rules' => $diversity_rules,
                    'diversity_enabled' => $diversity_enabled_for_heat,
                    'scoring_format_id' => $scoring_format_id
                ]);
                
            } catch (Exception $e) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Database error: ' . $e->getMessage()
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        }
        exit;
    }
    
       if (isset($_POST['action']) && $_POST['action'] === 'approve_score') {
        $score_id = $_POST['score_id'];
        
        try {
            // Get the judge_id from users table for the approved_by field
            $judge_lookup_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ?");
            $judge_lookup_stmt->execute([$judge_id]);
            $approved_by_judge_id = $judge_lookup_stmt->fetchColumn();
            
            // Get score details for notification
            $score_details_stmt = $pdo->prepare("
                SELECT s.score_value, s.status, u.username as judge_name, 
                       p.first_name, p.last_name, ep.bib_number, ep.event_id,
                       ep.heat_number, r.run_number, ep.id as participant_id
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                JOIN participants p ON ep.participant_id = p.id
                JOIN users u ON s.judge_id = u.id
                WHERE s.id = ?
            ");
            $score_details_stmt->execute([$score_id]);
            $score_details = $score_details_stmt->fetch(PDO::FETCH_ASSOC);
            
            $approve_stmt = $pdo->prepare("UPDATE scores SET is_approved = 1, approved_by = ?, approved_at = NOW() WHERE id = ?");
            $approve_stmt->execute([$approved_by_judge_id, $score_id]);
            
            // Prepare notification data for frontend
            $notification_data = null;
            if ($score_details) {
                $participant_name = $score_details['first_name'] . ' ' . $score_details['last_name'];
                $score_display = $score_details['status'] === 'OK' ? $score_details['score_value'] : $score_details['status'];
                
                // Get all judge scores for this participant/heat/run
                $all_scores_stmt = $pdo->prepare("
                    SELECT s.score_value, s.status, u.username as judge_name, s.is_approved,
                           u.id as judge_id
                    FROM scores s
                    JOIN runs r ON s.run_id = r.id
                    JOIN event_participants ep ON r.event_participant_id = ep.id
                    JOIN users u ON s.judge_id = u.id
                    WHERE ep.id = ? AND ep.heat_number = ? AND r.run_number = ?
                    ORDER BY u.username
                ");
                $all_scores_stmt->execute([
                    $score_details['participant_id'], 
                    $score_details['heat_number'], 
                    $score_details['run_number']
                ]);
                $all_judge_scores = $all_scores_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Calculate current result (average of approved scores)
                $approved_scores = [];
                $all_scores_info = [];
                foreach ($all_judge_scores as $judge_score) {
                    $judge_display = $judge_score['status'] === 'OK' ? $judge_score['score_value'] : $judge_score['status'];
                    $all_scores_info[] = [
                        'judge' => $judge_score['judge_name'],
                        'score' => $judge_display,
                        'approved' => (bool)$judge_score['is_approved']
                    ];
                    
                    if ($judge_score['is_approved'] && $judge_score['status'] === 'OK' && $judge_score['score_value'] !== null) {
                        $approved_scores[] = floatval($judge_score['score_value']);
                    }
                }
                
                $current_result = null;
                $approved_count = count($approved_scores);
                if ($approved_count > 0) {
                    // Apply drop rule if enough scores (simplified version)
                    if ($approved_count >= 3) {
                        // Drop highest and lowest
                        sort($approved_scores);
                        array_shift($approved_scores); // Remove lowest
                        array_pop($approved_scores);   // Remove highest
                    }
                    $current_result = $approved_count > 0 ? round(array_sum($approved_scores) / count($approved_scores), 3) : null;
                }
                
                // Calculate current rank position
                $rank_position = null;
                if ($current_result !== null) {
                    // Get all participants' current results for ranking
                    $ranking_stmt = $pdo->prepare("
                        SELECT ep.bib_number, ep.id as participant_id,
                               AVG(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN s.score_value END) as avg_score,
                               COUNT(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN 1 END) as approved_count
                        FROM event_participants ep
                        JOIN runs r ON ep.id = r.event_participant_id
                        JOIN scores s ON r.id = s.run_id
                        WHERE ep.event_id = ? AND ep.heat_number = ? AND r.run_number = ?
                        GROUP BY ep.id, ep.bib_number
                        HAVING approved_count > 0
                        ORDER BY avg_score DESC
                    ");
                    $ranking_stmt->execute([
                        $score_details['event_id'],
                        $score_details['heat_number'],
                        $score_details['run_number']
                    ]);
                    $rankings = $ranking_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    foreach ($rankings as $index => $ranking) {
                        if ($ranking['participant_id'] == $score_details['participant_id']) {
                            $rank_position = $index + 1;
                            break;
                        }
                    }
                }
                
                $notification_data = [
                    'bib' => $score_details['bib_number'],
                    'participant_name' => $participant_name,
                    'score' => $score_display,
                    'judge' => $score_details['judge_name'],
                    'heat' => $score_details['heat_number'],
                    'run' => $score_details['run_number'],
                    'all_scores' => $all_scores_info,
                    'current_result' => $current_result,
                    'rank_position' => $rank_position,
                    'approved_count' => $approved_count,
                    'total_judges' => count($all_judge_scores)
                ];
                
                // Enhanced notification message
                $result_text = $current_result !== null ? " | Current Result: {$current_result}" : "";
                $rank_text = $rank_position !== null ? " | Rank: #{$rank_position}" : "";
                $progress_text = " | Scores: {$approved_count}/" . count($all_judge_scores);
                
                // Send to notification API
                $api_data = [
                    'type' => 'success',
                    'title' => 'Score Approved',
                    'message' => sprintf(
                        "Bib #%s (%s) - %s approved: %s (Heat %s, Run %s)%s%s%s",
                        $score_details['bib_number'],
                        $participant_name,
                        $score_details['judge_name'],
                        $score_display,
                        $score_details['heat_number'],
                        $score_details['run_number'],
                        $result_text,
                        $rank_text,
                        $progress_text
                    ),
                    'category' => 'judge',
                    'event_id' => $score_details['event_id'],
                    'user_id' => $notification_user_id ?? 'head_judge',
                    'data' => json_encode([
                        'score_id' => $score_id,
                        'participant_bib' => $score_details['bib_number'],
                        'participant_name' => $participant_name,
                        'judge' => $score_details['judge_name'],
                        'approved_score' => $score_display,
                        'heat' => $score_details['heat_number'],
                        'run' => $score_details['run_number'],
                        'all_judge_scores' => $all_scores_info,
                        'current_result' => $current_result,
                        'rank_position' => $rank_position,
                        'approved_count' => $approved_count,
                        'total_judges' => count($all_judge_scores)
                    ])
                ];
                
                // Send to notification API
                $api_url = '../api/notification_api.php';
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $api_url);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($api_data + ['action' => 'send_notification']));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                curl_exec($ch);
                curl_close($ch);
            }
            
            echo json_encode([
                'success' => true, 
                'message' => 'Score approved successfully',
                'notification_data' => $notification_data
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error approving score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    // Handle unapprove score action
    if (isset($_POST['action']) && $_POST['action'] === 'unapprove_score') {
        $score_id = $_POST['score_id'];
        
        try {
            // Get score details for notification before unapproving
            $score_details_stmt = $pdo->prepare("
                SELECT s.score_value, s.status, u.username as judge_name, 
                       p.first_name, p.last_name, ep.bib_number, ep.event_id,
                       ep.heat_number, r.run_number
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                JOIN participants p ON ep.participant_id = p.id
                JOIN users u ON s.judge_id = u.id
                WHERE s.id = ?
            ");
            $score_details_stmt->execute([$score_id]);
            $score_details = $score_details_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Unapprove the score
            $unapprove_stmt = $pdo->prepare("UPDATE scores SET is_approved = 0, approved_by = NULL, approved_at = NULL WHERE id = ?");
            $unapprove_stmt->execute([$score_id]);
            
            // Prepare notification data
            $notification_data = null;
            if ($score_details) {
                $participant_name = $score_details['first_name'] . ' ' . $score_details['last_name'];
                $score_display = $score_details['status'] === 'OK' ? $score_details['score_value'] : $score_details['status'];
                
                $notification_data = [
                    'bib' => $score_details['bib_number'],
                    'participant_name' => $participant_name,
                    'score' => $score_display,
                    'judge' => $score_details['judge_name'],
                    'heat' => $score_details['heat_number'],
                    'run' => $score_details['run_number']
                ];
            }
            
            echo json_encode([
                'success' => true, 
                'message' => 'Score unapproved successfully',
                'notification_data' => $notification_data
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error unapproving score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    if (isset($_POST['action']) && $_POST['action'] === 'delete_score') {
        $score_id = $_POST['score_id'];
        
        try {
            // Get score details for notification before deleting
            $score_details_stmt = $pdo->prepare("
                SELECT s.score_value, s.status, u.username as judge_name, 
                       p.first_name, p.last_name, ep.bib_number, ep.event_id,
                       ep.heat_number, r.run_number
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                JOIN participants p ON ep.participant_id = p.id
                JOIN users u ON s.judge_id = u.id
                WHERE s.id = ?
            ");
            $score_details_stmt->execute([$score_id]);
            $score_details = $score_details_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Delete the score
            $delete_stmt = $pdo->prepare("DELETE FROM scores WHERE id = ?");
            $delete_stmt->execute([$score_id]);
            
            // Prepare notification data
            $notification_data = null;
            if ($score_details) {
                $participant_name = $score_details['first_name'] . ' ' . $score_details['last_name'];
                $score_display = $score_details['status'] === 'OK' ? $score_details['score_value'] : $score_details['status'];
                
                $notification_data = [
                    'bib' => $score_details['bib_number'],
                    'participant_name' => $participant_name,
                    'score' => $score_display,
                    'judge' => $score_details['judge_name'],
                    'heat' => $score_details['heat_number'],
                    'run' => $score_details['run_number']
                ];
            }
            
            echo json_encode([
                'success' => true, 
                'message' => 'Score deleted successfully',
                'notification_data' => $notification_data
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error deleting score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    if (isset($_POST['action']) && $_POST['action'] === 'update_score') {
        $score_id = $_POST['score_id'];
        $new_score = $_POST['score_value'] !== '' ? floatval($_POST['score_value']) : null;
        $status = $_POST['status'];
        $figures_json = $_POST['figures_json'] ?? null;
        
        try {
            // Check if figures_json column exists
            $figures_column_exists = false;
            try {
                $check_column = $pdo->query("SHOW COLUMNS FROM scores LIKE 'figures_json'");
                $figures_column_exists = $check_column->rowCount() > 0;
            } catch (Exception $e) {
                // Column doesn't exist
            }
            
            if ($figures_column_exists && isset($_POST['figures_json'])) {
                // Handle figures_json data
                $figures_value = null;
                if ($figures_json === 'null' || $figures_json === '') {
                    $figures_value = null;
                } else {
                    // Validate it's proper JSON and not empty object with empty arrays
                    try {
                        $decoded = json_decode($figures_json, true);
                        if ($decoded && is_array($decoded)) {
                            // Check if any category has non-empty data
                            $hasData = false;
                            foreach ($decoded as $category => $items) {
                                if (is_array($items) && count($items) > 0) {
                                    $hasData = true;
                                    break;
                                }
                            }
                            $figures_value = $hasData ? $figures_json : null;
                        } else {
                            $figures_value = null;
                        }
                    } catch (Exception $e) {
                        $figures_value = null;
                    }
                }
                
                // Update with figures_json
                $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, figures_json = ?, updated_at = NOW(), is_approved = 0 WHERE id = ?");
                $update_stmt->execute([$new_score, $status, $figures_value, $score_id]);
            } else {
                // Update without figures_json
                $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, updated_at = NOW(), is_approved = 0 WHERE id = ?");
                $update_stmt->execute([$new_score, $status, $score_id]);
            }
            
            echo json_encode(['success' => true, 'message' => 'Score updated successfully (requires re-approval)']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error updating score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    // NEW: Handle create empty score action
    if (isset($_POST['action']) && $_POST['action'] === 'create_score') {
        $event_id = $_POST['event_id'];
        $heat_number = $_POST['heat_number'];
        $run_number = $_POST['run_number'];
        $participant_id = $_POST['participant_id'];
        $judge_id_target = $_POST['judge_id'];
        $score_value = $_POST['score_value'] !== '' ? floatval($_POST['score_value']) : null;
        $status = $_POST['status'];
        $figures_json = isset($_POST['figures_json']) ? $_POST['figures_json'] : null;
        
        try {
            // First, find or create the run for this participant/heat/run combination
            $run_stmt = $pdo->prepare("
                SELECT r.id 
                FROM runs r 
                JOIN event_participants ep ON r.event_participant_id = ep.id 
                WHERE ep.id = ? AND r.run_number = ?
            ");
            $run_stmt->execute([$participant_id, $run_number]);
            $run_id = $run_stmt->fetchColumn();
            
            if (!$run_id) {
                // Create the run if it doesn't exist
                $create_run_stmt = $pdo->prepare("INSERT INTO runs (event_participant_id, run_number) VALUES (?, ?)");
                $create_run_stmt->execute([$participant_id, $run_number]);
                $run_id = $pdo->lastInsertId();
            }
            
            // Check if scores table has figures_json column
            $hasColumnsQuery = "SHOW COLUMNS FROM scores LIKE 'figures_json'";
            $hasColumnsStmt = $pdo->query($hasColumnsQuery);
            $hasFiguresColumn = $hasColumnsStmt->rowCount() > 0;
            
            // Process figures_json data
            $figures_value = null;
            if ($hasFiguresColumn && $figures_json) {
                try {
                    $decoded = json_decode($figures_json, true);
                    if ($decoded && is_array($decoded)) {
                        // Check if any category has non-empty data
                        $hasData = false;
                        foreach ($decoded as $category => $items) {
                            if (is_array($items) && count($items) > 0) {
                                $hasData = true;
                                break;
                            }
                        }
                        $figures_value = $hasData ? $figures_json : null;
                    } else {
                        $figures_value = null;
                    }
                } catch (Exception $e) {
                    $figures_value = null;
                }
            }
            
            // Create the score with or without figures_json based on column existence
            if ($hasFiguresColumn) {
                $create_score_stmt = $pdo->prepare("
                    INSERT INTO scores (run_id, judge_id, score_value, status, figures_json, created_at, is_approved) 
                    VALUES (?, ?, ?, ?, ?, NOW(), 0)
                ");
                $create_score_stmt->execute([$run_id, $judge_id_target, $score_value, $status, $figures_value]);
            } else {
                $create_score_stmt = $pdo->prepare("
                    INSERT INTO scores (run_id, judge_id, score_value, status, created_at, is_approved) 
                    VALUES (?, ?, ?, ?, NOW(), 0)
                ");
                $create_score_stmt->execute([$run_id, $judge_id_target, $score_value, $status]);
            }
            
            // Update participant status to 'scoring' when a new score is added
            $update_status_stmt = $pdo->prepare("
                UPDATE event_participants 
                SET participant_status = 'scoring' 
                WHERE id = ?
            ");
            $update_status_stmt->execute([$participant_id]);
            
            echo json_encode(['success' => true, 'message' => 'Score created successfully (requires approval)']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error creating score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    // If no action matched, return error
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    exit;
}

// UPDATED: Get events - admins see all events, head judges see only assigned events
if ($user_role === 'admin') {
    // Admin sees all events
    $events_stmt = $pdo->prepare("
        SELECT DISTINCT e.id, e.name, e.date
        FROM events e
        ORDER BY e.date DESC, e.name
    ");
    $events_stmt->execute();
} else {
    // Head judges see only assigned events
    $events_stmt = $pdo->prepare("
        SELECT DISTINCT e.id, e.name, e.date
        FROM events e
        JOIN judge_assignments ja ON e.id = ja.event_id
        WHERE ja.judge_id = ?
        ORDER BY e.date DESC, e.name
    ");
    $events_stmt->execute([$judge_id]);
}
$events = $events_stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Head Judge Panel</title>
    <!-- style skin holder -->
 <?php include_once '../includes/stylesheets.php'; ?>
    <!-- Lucide Icons and translation utility are now included globally via stylesheets.php and footer.php -->
    
    <style>
        .card-hover:hover {
            transform: translateY(-2px);
            transition: transform 0.2s ease-in-out;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
        }
        .cursor-pointer {
            cursor: pointer;
        }
        
        /* Auto switches styling */
        .form-check-input {
            cursor: pointer;
        }
        .form-check-label {
            cursor: pointer;
        }
        
        /* Results Modal Styles */
        .config-button {
            width: 100%;
            text-align: left;
            padding: 0.75rem;
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
            background-color: white;
            cursor: pointer;
            transition: all 0.2s ease;
            margin-bottom: 0.5rem;
        }
        .config-button:hover {
            background-color: #f8f9fa;
            border-color: #0d6efd;
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        }
        .config-button.active {
            background-color: #0d6efd;
            color: white;
            border-color: #0d6efd;
        }
        .config-button.active .badge {
            background-color: rgba(255, 255, 255, 0.2) !important;
            color: white !important;
        }
        .config-button.active .text-muted {
            color: rgba(255, 255, 255, 0.8) !important;
        }
        #resultsContentDisplay {
            min-height: 500px;
        }
        #savedConfigsList {
            max-height: 70vh;
            overflow-y: auto;
        }
     body {
            padding-top: 0px !important;
     }         
        nav{
            display: none !important;
        }
.heat-loading-overlay {
    position: absolute;
    inset: 0;
    background: rgba(255, 255, 255, 0.9);
    backdrop-filter: blur(2px);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 5;
    opacity: 0;
    pointer-events: none;
    transition: opacity 0.2s ease;
}
.heat-loading-overlay.is-visible {
    opacity: 1;
}
.heat-loading-overlay .spinner-border {
    width: 2.5rem;
    height: 2.5rem;
}
.bg-approved {
    /* background-color: gray;*/
    background: linear-gradient(180deg, #00aa0740 0%, #00ff0a00 10%, #00ff0a00 90%, #00aa0740 100%); 
}
.bg-pending {
    /* background-color: gray; */
    background: linear-gradient(180deg, #67676740 0%, rgba(0, 51, 170, 0) 5%, rgba(0, 51, 170, 0) 95%, #67676740 100%);
}
.bg-empty {
    background: linear-gradient(180deg, #6c757d87 0%, rgba(108, 117, 125, 0) 25%, rgba(108, 117, 125, 0) 75%, #6c757d87 100%)
}
tr.bg-approved .resultCom,  tr.bg-approved .scoreCol{

    }
    
/* Sticky thead - requires max-height on table-responsive */
.table-responsive {
    max-height: 90vh;
    overflow-y: auto;
    position: relative;
}
#scoresTable thead {
    position: sticky;
    top: 0;
    z-index: 10;
    background-color: white;
}
#scoresTable thead th {
    background-color: #f8f9fa;
    box-shadow: 0 2px 2px -1px rgba(0, 0, 0, 0.4);
}

#scoresTable th {
    padding: 5px !important;
}
#scoresTable td {
    
}
/* Drop score styling */
.score_drop {
    opacity: 0.7;
}

.score_drop .score_value {
        color: #1f2225 !important;
   
}

.score_drop:hover .score_value {
    color: #495057 !important;
}

.figures-display {
    
    border-radius: 0.375rem;
    padding: 0.5rem;
    grid-template-columns: auto auto auto;
    align-self: center !important;
}
.row.score-figure {
    display: flex !important;
    grid-template-columns: auto auto;
    justify-items: center !important;
    min-height: auto;
    flex-wrap: wrap;
    flex-direction: row;
}

.table-responsive td .figures-display, .figure-s {
    transition: transform 0.4s ease-in-out;
    cursor: pointer;
}

.table-responsive td:hover .figures-display, .figure-s:hover{
    transform: scale(1.2);
    z-index: 10;
    position: relative;
}


.bg-empty .row.score-figure {
    display: grid !important;
    grid-template-columns: auto;
    align-items: center !important;
    justify-items: center !important;
}
.scores-col2 {
    align-self: center !important;
    justify-self: center !important;
}
.scores-col1 {
    align-self: center !important;
    justify-self: center !important;
}

/* Center align all table cell content */
.table td {
    vertical-align: middle !important;
    text-align: center !important;
}

/* Ensure the first column (participant info) is left-aligned for readability */
.table td:first-child {
    text-align: left !important;
}

/* Center all content within score cells */
.table td .row.score-figure {
    justify-content: center !important;
    align-items: center !important;
    width: 100% !important;
    margin: 0 !important;
}

/* Center the grid columns */
.scores-col1, .scores-col2 {
    display: flex !important;
    flex-direction: column !important;
    align-items: center !important;
    justify-content: center !important;
}

/* Center figures display content */
.figures-display {
    text-align: center !important;
    display: flex !important;
    flex-direction: row;
    flex-wrap: wrap;
    justify-content: space-evenly;
}

/* Center empty score cells */
.bg-empty .row.score-figure > div {
    
}
.pendingActions {
    backdrop-filter: blur(11px);
    width: 100px !important;
}
.OK .approval-clickable{
    display: none;
}
.editScoreButton {
    display: none !important;
}
.bg-pending .bg-danger.approval-clickable {
    display: block;
}
div#scoresContainer td {
    padding: 5px !important;
}
#scoresWrapper {
    
    overflow: auto;
    scroll-behavior: smooth;
}
td.participantCol {
    width: 280px;
}

/* Mobile participant row - hidden by default */
.participantColMobileRow {
    display: none !important;
}

/* Hide mobile-specific classes on desktop */
.participantColMobileHide {
    display: table-cell;
}

/* Responsive: Show mobile row and hide regular column on smaller screens */
@media (max-width: 991px) {
    /* Hide the regular participant column */
    td.participantCol {
        display: none !important;
    }
    
    /* Hide desktop average and actions columns (mobile uses rowspan) */
    .participantColMobileHide {
        display: none !important;
    }
    
    /* Show the mobile full-width participant rows */
    .participantColMobileRow {
        display: table-row !important;
        background-color: rgba(13, 110, 253, 0.05);
        border-bottom: 2px solid #dee2e6;
    }
    
    .participantColMobileRow td {
        font-size: 0.9rem;
    }
    
    .participantColMobileRow .badge {
        font-size: 0.75rem;
    }
}

td.resultCom.text-center {
    width: 100px;
}

td.actionsCol.text-center {
    width: 100px;
}

/* Figure chip styles */
.figure-chip {
    transition: all 0.2s ease;
    border: 1px solid #6c757d !important;
    color: #6c757d !important;
}

.figure-chip:hover {
    border-color: #0d6efd !important;
    color: #0d6efd !important;
    background-color: rgba(13, 110, 253, 0.1) !important;
}

.figure-chip.active {
    background-color: #0d6efd !important;
    border-color: #0d6efd !important;
    color: white !important;
}

.figure-chip.active:hover {
    background-color: #0b5ed7 !important;
    border-color: #0b5ed7 !important;
}

/* Active Heat Card Styles */
#activeHeatCardContainer {
    min-height: 150px;
}

#activeHeatCardContainer .heat-card {
    transition: all 0.3s ease;
}

#activeHeatCardContainer .heat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

#activeHeatCardContainer .col-md-4.col-lg-3.mb-3 {
    width: 100%;
}

/* Pulse highlight animation for buttons */
@keyframes pulseHighlight {
    0%, 100% {
        box-shadow: 0 0 0 0 rgba(13, 110, 253, 0.7);
        transform: scale(1);
    }
    50% {
        box-shadow: 0 0 0 10px rgba(13, 110, 253, 0);
        transform: scale(1.05);
    }
}

.pulse-highlight {
    animation: pulseHighlight 0.6s ease-in-out 3;
}

/* Validation styling for score inputs */
#editScoreValue.is-invalid,
#createScoreValue.is-invalid {
    border-color: #dc3545 !important;
    background-color: #f8d7da !important;
}

.figure-category h6 {
    font-weight: 600;
}

.figure-chips {
    min-height: 2.5rem;
}

/* Control Point Summary Row Styles */
.control-point-summary {
    background-color: rgba(248, 249, 250, 0.8) !important;
    border-top: 2px solid #dee2e6 !important;
    font-size: 0.85rem;
}

.control-point-summary td {
    padding: 5px 5px !important;
    vertical-align: top !important;
}
.figures-summary{    display: flex
;}
.mb-1.figuresByCategory {
    display: flex;
    flex-wrap: wrap;
    justify-content: center;
}
.control-point-summary .scores-summary .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.75rem);
    padding: 0.3em 0.5em;
}
.figures-summary label {
    display: none !important;
}
.control-point-summary .figures-summary {
    font-size: 0.75rem;
}

.control-point-summary .figures-summary .badge {
   font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.75rem);
    padding: 0.2em 0.4em;
}
.control-point-summary .figures-summary.fig_5 .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.6rem);
    padding: 0.2em 0.4em;
}
.control-point-summary .figures-summary.fig-3_5 .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.65rem);
    padding: 0.2em 0.4em;
}
.control-point-summary .figures-summary.fig-2 .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 1rem);
    padding: 0.2em 0.4em;
}
.ss-bg-A {
    background-color: #ede2ff !important;
}
.ss-bg-B {
    background-color: #fcffe2 !important;
}
/* Enhanced figures display under scoreCol */
.scores-col2 .figures-display {
    background-color: rgba(255, 255, 255, 0.9);
    border-radius: 4px;
    margin-top: 4px;
}

/* Edit Score Modal Styles */
#editScoreInfo {
    background: linear-gradient(135deg, #e8f4fd 0%, #f0f8ff 100%);
    border: 1px solid #b3d9f2;
    border-radius: 8px;
}

#editScoreInfo h6 {
    color: #1e3a8a;
    font-weight: 600;
    border-bottom: 1px solid rgba(30, 58, 138, 0.2);
    padding-bottom: 0.5rem;
}

#editScoreInfo .mb-2 {
    margin-bottom: 0.75rem !important;
    padding: 0.25rem 0;
    border-radius: 4px;
    transition: background-color 0.2s ease;
}

#editScoreInfo .mb-2:hover {
    background-color: rgba(59, 130, 246, 0.05);
    padding-left: 0.5rem;
    margin-left: -0.5rem;
    margin-right: -0.5rem;
}

#editScoreInfo .badge {
    font-size: 0.75rem;
    padding: 0.35em 0.6em;
}

#editScoreInfo .text-muted {
    opacity: 0.7;
}

/* Notification System Styles */
.css-bib {
    font-weight: bold !important;
    font-size: 0.9em !important;
    padding: 0.3em 0.6em !important;
    margin: 0 0.2em !important;
}
span.badge.bg-danger.approval-clickable {
    max-width: 90px;
}
.judge_score {
    font-weight: bold !important;
    font-size: 0.85em !important;
    padding: 0.25em 0.5em !important;
    margin: 0 0.1em !important;
}

.judge_score.approved {
    background-color: #28a745 !important;
    border-color: #28a745 !important;
}

.notification-toast .toast-body {
   
}

.notification-toast .toast-body small.text-light {
   
}

        .scores-hidden .score_value,
        .scores-hidden .judge_score,
        .scores-hidden .participant-score-total {
            filter: blur(6px);
            opacity: 0.25;
        }

        /* Head Judge Layout Enhancements */
        .head-judge-toolbar {
            position: sticky;
            top: 0;
            z-index: 1020;
            background: rgba(255, 255, 255, 0.97);
            backdrop-filter: blur(6px);
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            padding: 0.85rem 1rem;
            margin-bottom: 1.5rem;
        }

        .head-judge-toolbar .toolbar-field {
            min-width: 130px;
        }

        .head-judge-toolbar label {
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.04em;
            color: #6c757d;
            margin-bottom: 0.25rem;
        }

        .panel-layout .summary-column {
            position: sticky;
            top: 6.75rem;
            align-self: flex-start;
        }

        @media (max-width: 1199.98px) {
            .panel-layout .summary-column {
                position: static;
            }
        }

        .summary-card .metric-value {
            font-size: 2rem;
            font-weight: 700;
            color: #0d6efd;
        }

        .summary-card .metric-label {
            text-transform: uppercase;
            font-size: 0.7rem;
            letter-spacing: 0.08em;
            color: #6c757d;
        }

        .performing-chip {
            border: 1px solid rgba(13, 110, 253, 0.2);
            padding: 0.4rem 0.6rem;
            border-radius: 999px;
            font-size: 0.8rem;
            display: inline-flex;
            align-items: center;
            gap: 0.4rem;
            background: rgba(13, 110, 253, 0.08);
            color: #0d47a1;
        }

        .performing-chip .badge {
            font-size: 0.65rem;
        }

        .queue-status-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 12px;
        }

        .queue-stat {
            background: #f8f9fa;
            border-radius: 14px;
            padding: 12px;
            border: 1px solid #eef1f4;
        }

        .queue-stat-label {
            font-size: 0.7rem;
            text-transform: uppercase;
            letter-spacing: 0.08em;
            color: #77838f;
            display: block;
        }

        .queue-stat-value {
            font-size: 1.35rem;
            font-weight: 700;
            color: #182026;
        }

        .performing-slots {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }

        .performing-slot {
            border-radius: 14px;
            border: 1px dashed rgba(13, 110, 253, 0.25);
            padding: 12px;
            background: linear-gradient(135deg, rgba(13, 110, 253, 0.05), rgba(255, 255, 255, 0.9));
            will-change: transform, opacity;
            transition: background 0.2s ease, border-color 0.2s ease;
        }

        .performing-slot.next {
            border-color: rgba(40, 167, 69, 0.35);
            background: linear-gradient(135deg, rgba(40, 167, 69, 0.05), rgba(255, 255, 255, 0.9));
        }

        .slot-label {
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.08em;
            color: #6c757d;
            display: block;
        }

        .slot-content {
            margin-top: 6px;
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
        }

        .slot-chip {
            background: #fff;
            border-radius: 10px;
            padding: 6px 10px;
            border: 1px solid rgba(0, 0, 0, 0.05);
            display: inline-flex;
            gap: 6px;
            align-items: center;
            font-size: 0.85rem;
        }

        .slot-bib {
            font-weight: 700;
            color: #0d6efd;
        }

        .queue-chips {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }

        .queue-chip {
            border-radius: 14px;
            border: 1px solid #e9ecef;
            padding: 12px;
            display: flex;
            gap: 12px;
            align-items: center;
            background: #fff;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            will-change: transform, opacity;
            transition: box-shadow 0.2s ease, background 0.2s ease, border-color 0.2s ease;
        }

        .queue-chip-rank {
            width: 32px;
            height: 32px;
            border-radius: 10px;
            background: #f1f3f5;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            color: #495057;
        }

        .queue-chip-current {
            border-color: #fdc453;
            background: #fff6e6;
        }

        .queue-chip-next {
            border-color: #8be28b;
            background: #f1fff1;
        }

        .queue-chip-completed {
            opacity: 0.7;
        }

        #queuePerformingSlots {
            min-height: 140px;
        }

        #queueChipsList {
            min-height: 180px;
        }

        .queue-chip-name {
            font-weight: 600;
            color: #1d2533;
        }

        .queue-chip-meta {
            font-size: 0.75rem;
            color: #6c757d;
        }

        .heat-alerts-card .alerts-list {
            max-height: 420px;
            overflow-y: auto;
        }

        .alert-item {
            display: flex;
            gap: 12px;
            padding: 12px;
            border-radius: 14px;
            border: 1px solid #eef1f4;
            background: #fff;
            margin-bottom: 12px;
            box-shadow: 0 6px 18px rgba(0, 0, 0, 0.06);
        }

        .alert-icon {
            width: 44px;
            height: 44px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
            color: #fff;
        }

        .alert-type-info {
            background: #0dcaf0;
        }

        .alert-type-warning {
            background: #ffc107;
            color: #212529;
        }

        .alert-type-success {
            background: #198754;
        }

        .alert-type-danger {
            background: #dc3545;
        }

        .alert-body {
            flex: 1;
        }

        .alert-title {
            font-weight: 600;
            margin-bottom: 4px;
        }

        .alert-message {
            font-size: 0.9rem;
            color: #495057;
        }

        .alert-time {
            font-size: 0.75rem;
        }

        .alert-meta {
            font-size: 0.75rem;
            color: #868e96;
        }

        .badge-soft-danger,
        .badge-soft-warning,
        .badge-soft-info,
        .badge-soft-success {
            border-radius: 999px;
            font-size: 0.7rem;
            font-weight: 600;
            padding: 4px 10px;
        }

        .badge-soft-danger {
            background: rgba(220, 53, 69, 0.15);
            color: #c92a2a;
        }

        .badge-soft-warning {
            background: rgba(255, 193, 7, 0.18);
            color: #ad7d00;
        }

        .badge-soft-info {
            background: rgba(13, 202, 240, 0.18);
            color: #0b7285;
        }

        .badge-soft-success {
            background: rgba(25, 135, 84, 0.18);
            color: #0f5132;
        }

        .alerts-list::-webkit-scrollbar {
            width: 6px;
        }

        .alerts-list::-webkit-scrollbar-thumb {
            background: rgba(0, 0, 0, 0.15);
            border-radius: 999px;
        }

        .alert-placeholder {
            border-radius: 10px;
            padding: 12px;
            background: rgba(220, 53, 69, 0.08);
        }

        @media (max-width: 1199px) {
            .queue-status-grid {
                grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            }
        }

        .filter-card .card-header {
            background: rgba(13, 110, 253, 0.05);
        }

        .sticky-action-footer {
            position: sticky;
            bottom: 0;
            background: rgba(255, 255, 255, 0.95);
            border-top: 1px solid rgba(0, 0, 0, 0.05);
            padding: 0.75rem 1rem;
            backdrop-filter: blur(4px);
            margin-top: 1.5rem;
        }

    </style>
</head>

<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>

    <div class="container-fluid container-StyleScore">
            <!-- event action menu. -->
<div class="row mb-3">
    <div class="col-12">
        <?php       
        $use_localStorage = false;
        $show_header = false;
        //$show_public_view = false;
        include '../admin/event_actions_menu.php'; 
        ?>
       
    </div>
</div>
        <div class="head-judge-toolbar shadow-sm rounded-3">
            <div class="d-flex flex-wrap align-items-end gap-3">
                <div class="toolbar-field">
                    <label for="eventSelect">Event</label>
                    <select class="form-select form-select-sm" id="eventSelect">
                        <option value="">Select Event</option>
                        <?php foreach ($events as $event): ?>
                            <option value="<?= $event['id'] ?>"><?= htmlspecialchars($event['name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="toolbar-field">
                    <label for="heatSelect">Heat</label>
                    <select class="form-select form-select-sm" id="heatSelect" disabled>
                        <option value="">Heat</option>
                    </select>
                </div>
                <div class="toolbar-field">
                    <label for="runSelect">Run</label>
                    <select class="form-select form-select-sm" id="runSelect" disabled>
                        <option value="">Run</option>
                    </select>
                </div>
                <div class="d-flex align-items-end gap-2 flex-grow-1">
                    <div class="btn-group">
                        <button class="btn btn-light btn-sm" id="refreshBtn">
                            <i class="fas fa-sync-alt"></i> Refresh
                        </button>
                        <button class="btn btn-outline-secondary btn-sm" id="clearSelectionsBtn" title="Clear selections">
                            <i class="fas fa-eraser"></i>
                        </button>
                    </div>
                    <div class="d-flex align-items-center gap-3">
                        <div class="form-check form-switch mb-0">
                            <input class="form-check-input" type="checkbox" id="autoHeatRun">
                            <label class="form-check-label small" for="autoHeatRun">Auto Heat/Run</label>
                        </div>
                        <div class="form-check form-switch mb-0">
                            <input class="form-check-input" type="checkbox" id="autoRefresh" checked>
                            <label class="form-check-label small" for="autoRefresh">Auto-refresh</label>
                        </div>
                    </div>
                </div>
                <div class="ms-auto d-flex flex-wrap gap-2">
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="showEventResults()" id="showResultsBtn" style="display: none;">
                        <i class="fas fa-trophy me-1"></i> View Results
                    </button>
                    <button type="button" class="btn btn-outline-info btn-sm" onclick="openHeadJudgeHelp()">
                        <i class="fas fa-circle-question me-1"></i> Help
                    </button>
                </div>
            </div>
        </div>

        <div class="row panel-layout g-4">
            <div class="col-12 col-xl-4 summary-column">
                <div class="d-flex align-items-center justify-content-between mb-3">
                    <h5 class="mb-0 fw-semibold">
                        <i class="fas fa-crown text-warning me-2"></i>
                        Head Judge Panel
                    </h5>
                    <span class="badge bg-success">Live</span>
                </div>
                <div class="row g-3">
                    <div class="col-md-6 col-xl-12">
                        <div class="card shadow-sm summary-card">
                            <div class="card-body">
                                <div class="metric-value" id="metricToApprove">0</div>
                                <div class="metric-label">Awaiting Approval</div>
                                <div class="text-muted small">
                                    <i class="fas fa-clock me-1"></i>Last update <span id="metricLastUpdate">--:--</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 col-xl-12">
                        <div class="card shadow-sm summary-card">
                            <div class="card-body d-flex align-items-center justify-content-between">
                                <div>
                                    <div class="metric-value text-warning" id="metricPendingJudges">0</div>
                                    <div class="metric-label">Judges Pending</div>
                                </div>
                                <button class="btn btn-sm btn-outline-primary" onclick="loadActiveHeatCardInHeadJudge()">
                                    Sync Now
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="card queue-summary-card shadow-sm border-0 mb-3">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div>
                                        <h6 class="mb-0 fw-bold text-primary">
                                            <i class="fas fa-people-arrows me-2"></i>Performing Queue
                                        </h6>
                                        <small class="text-muted" id="queueSummarySubtitle">Awaiting heat data…</small>
                                    </div>
                                    <div class="d-flex align-items-center gap-2">
                                        <span class="badge bg-light text-muted" id="queueUpdatedAt">--</span>
                                        <button class="btn btn-sm btn-outline-primary" onclick="refreshHeadJudgeQueuePanel()" title="Refresh queue">
                                            <i class="fas fa-sync-alt"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="queue-status-grid mt-3" id="queueStatusGrid">
                                    <div class="queue-stat">
                                        <span class="queue-stat-label">Participants</span>
                                        <span class="queue-stat-value" id="queueTotalCount">--</span>
                                    </div>
                                    <div class="queue-stat">
                                        <span class="queue-stat-label">Performing</span>
                                        <span class="queue-stat-value text-warning" id="queuePerformingCount">0</span>
                                    </div>
                                    <div class="queue-stat">
                                        <span class="queue-stat-label">Waiting</span>
                                        <span class="queue-stat-value text-primary" id="queueWaitingCount">0</span>
                                    </div>
                                    <div class="queue-stat">
                                        <span class="queue-stat-label">Completed</span>
                                        <span class="queue-stat-value text-success" id="queueCompletedCount">0</span>
                                    </div>
                                    <div class="queue-stat">
                                        <span class="queue-stat-label">Run</span>
                                        <span class="queue-stat-value" id="queueRunLabel">--</span>
                                    </div>
                                </div>
                                <div class="performing-slots mt-3" id="queuePerformingSlots">
                                    <div class="performing-slot current">
                                        <span class="slot-label">Now Performing</span>
                                        <div class="slot-content text-muted small">Waiting for heat data…</div>
                                    </div>
                                    <div class="performing-slot next">
                                        <span class="slot-label">Up Next</span>
                                        <div class="slot-content text-muted small">Queue will appear here.</div>
                                    </div>
                                </div>
                                <div class="queue-chips mt-3" id="queueChipsList">
                                    <div class="text-muted small">Queue will appear once participants load.</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="card heat-alerts-card shadow-sm border-0">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div>
                                        <h6 class="mb-0 fw-bold text-danger">
                                            <i class="fas fa-bell me-2"></i>Heat Alerts
                                        </h6>
                                        <small class="text-muted" id="alertsStatusLabel">Not connected</small>
                                    </div>
                                    <div class="d-flex align-items-center gap-2">
                                        <span class="badge badge-soft-danger" id="alertsCountBadge">0</span>
                                        <button class="btn btn-sm btn-outline-secondary" onclick="refreshHeadJudgeAlerts(true)" title="Refresh alerts">
                                            <i class="fas fa-sync-alt"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="alerts-list mt-3" id="alertsList">
                                    <div class="text-muted small text-center py-3">
                                        Alerts will stream in automatically once available.
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-12 col-xl-8">
                <div class="card border-0 shadow-sm filter-card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0 fw-semibold">
                            <i class="fas fa-sliders-h me-2"></i>Filters & Sorting
                        </h6>
                        <div class="d-flex gap-2">
                            <button class="btn btn-sm btn-outline-secondary" id="clearFilters">
                                <i class="fas fa-times me-1"></i>Clear
                            </button>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="row g-3 align-items-end">
                            <div class="col-md-4">
                                <label class="form-label">Gender</label>
                                <select class="form-select" id="genderFilter">
                                    <option value="">All Genders</option>
                                    <option value="M">Male</option>
                                    <option value="F">Female</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Category</label>
                                <select class="form-select" id="categoryFilter">
                                    <option value="">All Categories</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Search</label>
                                <input type="text" class="form-control" id="bibNameFilter" placeholder="BIB or Name">
                            </div>
                        </div>
                        <div class="row g-3 align-items-center mt-3">
                            <div class="col-md-6">
                                <label class="form-label">Status visibility</label>
                                <div class="d-flex gap-3 flex-wrap">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="showToApprove" checked>
                                        <label class="form-check-label" for="showToApprove">To Approve</label>
                                    </div>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="showAllApproved" checked>
                                        <label class="form-check-label" for="showAllApproved">Approved</label>
                                    </div>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="showWaiting" checked>
                                        <label class="form-check-label" for="showWaiting">Waiting</label>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Sort order</label>
                                <div class="btn-group" role="group" aria-label="Sort controls">
                                    <button type="button" class="btn btn-sm btn-outline-primary active" data-sort="default">
                                        <i class="fas fa-list-ol me-1"></i>Default
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-primary" data-sort="average">
                                        <i class="fas fa-trophy me-1"></i>Average
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-primary" data-sort="latest">
                                        <i class="fas fa-clock me-1"></i>Latest
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card border-0 shadow-sm">
                    <div class="position-absolute spinner-border spinner-border-sm text-primary d-none" id="loadingSpinner">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <div class="card-body">
                        <!-- Sort Controls -->
                        <!-- Sort Controls -->
                        <div class="sort-controls d-none" id="sortControls">
                            <div class="d-flex align-items-center justify-content-between flex-wrap gap-3 mb-3">
                                <div class="d-flex align-items-center gap-2">
                                    <span class="fw-bold text-muted">
                                        <i class="fas fa-sort me-1"></i>Sort Order:
                                    </span>
                                    <div class="btn-group" role="group">
                                        <button type="button" class="btn btn-sm btn-outline-primary active" data-sort="default">
                                            <i class="fas fa-list-ol me-1"></i>Default Order
                                        </button>
                                        <button type="button" class="btn btn-sm btn-outline-primary" data-sort="average">
                                            <i class="fas fa-trophy me-1"></i>By Average (High-Low)
                                        </button>
                                        <button type="button" class="btn btn-sm btn-outline-primary" data-sort="latest">
                                            <i class="fas fa-clock me-1"></i>By Latest Score
                                        </button>
                                    </div>
                                </div>
                                <div class="d-flex align-items-center gap-2">
                                    <span class="badge bg-info" id="sortableCount">0 complete participants</span>
                                     </div>
                            </div>

                            <!-- Filter Controls -->
                            <div class="border-top pt-3">
                                <div class="row g-2 align-items-end">
                                    <div class="col-4 col-md-2">
                                        <label class="form-label fw-bold text-muted">
                                            <i class="fas fa-filter me-1"></i>Filters:
                                        </label>
                                        <select class="form-select form-select-sm" id="genderFilter">
                                            <option value="">All Genders</option>
                                            <option value="M">Male Only</option>
                                            <option value="F">Female Only</option>
                                        </select>
                                    </div>
                                    <div class="col-4 col-md-2">
                                        <label class="form-label">&nbsp;</label>
                                        <select class="form-select form-select-sm" id="categoryFilter">
                                            <option value="">All Categories</option>
                                        </select>
                                    </div>
                                    <div class="col-4 col-md-2">
                                        <label class="form-label"> <div class="">
                                                <button class="btn btn-outline-secondary btn-sm" id="clearFilters">
                                                    <i class="fas fa-times"></i>
                                                </button>
                                            </div></label>
                                        <input type="text" class="form-control form-control-sm" id="bibNameFilter" placeholder="Search by BIB or Name...">
                                    </div>
                                    <div class="col-auto ms-auto">
                                        <label class="form-label">Status:</label>
                                        <div class="d-flex gap-2 flex-wrap">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="showToApprove" checked>
                                                <label class="form-check-label" for="showToApprove">
                                                    <span class="badge bg-warning">Approve</span>
                                                </label>
                                            </div>
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="showAllApproved" checked>
                                                <label class="form-check-label" for="showAllApproved">
                                                    <span class="badge bg-success">Approved</span>
                                                </label>
                                            </div>
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="showWaiting" checked>
                                                <label class="form-check-label" for="showWaiting">
                                                    <span class="badge bg-secondary">Waiting</span>
                                                </label>
                                            </div>
                                           
                                        </div>
                                    </div>
                                </div>
                                <div class="mt-2">
                                    <div class="d-flex align-items-center justify-content-between">
                                        <small class="text-muted">
                                            <span id="filteredCount">0</span> of <span id="totalCount">0</span> participants shown
                                        </small>
                                       
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="text-center text-muted py-5" id="noDataMessage">
                            <i class="fas fa-clipboard-list fa-3x mb-3 opacity-50"></i>
                            <h5>Select Event, Heat, and Run to view scores</h5>
                            <p>Use the controls above to navigate to specific scoring sessions</p>
                        </div>
                        <div id="scoresWrapper">
                            <div id="scoresContainer" style="display: none;">
                                <!-- Dynamic content will be loaded here -->
                            </div>
                        </div>
                        <div class="sticky-action-footer d-flex flex-wrap justify-content-between align-items-center gap-2">
                            <div>
                                <strong>Session:</strong>
                                <span id="toolbarSessionInfo" class="text-muted">No selection</span>
                            </div>
                            <div class="btn-group">
                                <button class="btn btn-outline-primary btn-sm" onclick="loadScores()">
                                    <i class="fas fa-sync-alt me-1"></i>Refresh Data
                                </button>
                                <button class="btn btn-outline-success btn-sm" onclick="approveAllVisibleParticipants()">
                                    <i class="fas fa-check-double me-1"></i>Approve Visible
                                </button>
                                <div class="btn btn-outline-secondary btn-sm" id="showScoresToggle">
                                    <div class="form-check form-switch m-0">
                                        <input class="form-check-input" type="checkbox" id="scoreVisibilityToggle">
                                        <label class="form-check-label" for="scoreVisibilityToggle">Show Scores</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    
    <!-- Active Heat Card Section -->
    <div class="container-fluid mt-4" id="activeHeatSection" style="display: none;">
        <div class="row">
            <div class="col-12">
                <div class="card border-0 shadow-sm">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="mb-0 fw-bold text-primary">
                                <i class="fas fa-fire me-2"></i>Active Heat Monitor
                            </h6>
                            <div class="d-flex gap-2 align-items-center">
                                <!-- Heat Card Layout Selector -->
                                <div class="d-flex flex-column gap-1">
                                    <small class="text-muted fw-bold" style="font-size: 0.7rem;">Layout:</small>
                                    <div class="btn-group btn-group-sm" role="group" aria-label="Heat card layout">
                                        <button type="button" class="btn btn-outline-secondary active" onclick="setHeadJudgeHeatCardLayout('judges', this)" title="Judges status only">
                                            <i class="fas fa-user-tie"></i> Judges
                                        </button>
                                        <button type="button" class="btn btn-outline-secondary" onclick="setHeadJudgeHeatCardLayout('monitor-row', this)" title="Monitor row layout">
                                            <i class="fas fa-align-justify"></i> Row
                                        </button>
                                        <button type="button" class="btn btn-outline-secondary" onclick="setHeadJudgeHeatCardLayout('monitor', this)" title="Monitor layout">
                                            <i class="fas fa-desktop"></i> Monitor
                                        </button>
                                        <button type="button" class="btn btn-outline-secondary" onclick="setHeadJudgeHeatCardLayout('full', this)" title="Full details">
                                            <i class="fas fa-expand"></i> Full
                                        </button>
                                    </div>
                                </div>
                                <button class="btn btn-sm btn-outline-primary" onclick="loadActiveHeatCardInHeadJudge()" title="Refresh heat card">
                                    <i class="fas fa-sync-alt"></i> Refresh
                                </button>
                            </div>
                        </div>
                        
                        <!-- Heat Card Display Area -->
                        <div id="activeHeatCardContainer" class="row">
                            <!-- Active heat card will be loaded here -->
                            <div class="col-12 text-center text-muted py-3">
                                <i class="fas fa-fire fa-2x mb-2 opacity-25"></i>
                                <p class="mb-0 small">Loading active heat...</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Success Toast -->
    <div class="toast-container position-fixed top-0 end-0 p-3">
        <div id="successToast" class="toast" role="alert">
            <div class="toast-header bg-success text-white">
                <i class="fas fa-check-circle me-2"></i>
                <strong class="me-auto">Success</strong>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                Updated successfully
            </div>
        </div>
    </div>
    
    <!-- Edit Score Modal -->
    <div class="modal fade" id="editScoreModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i>
                        <?php echo $user_role === 'admin' ? 'Admin Override - Edit Score' : 'Head Judge Override - Edit Score'; ?>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <!-- Participant Summary (context moved below figures) -->
                    <div class="alert alert-info mb-3" id="editScoreInfo">
                        <div class="text-center text-muted">
                            <i class="fas fa-spinner fa-spin me-2"></i>Loading participant information...
                        </div>
                    </div>
                    
                    <form id="editScoreForm">
                        <input type="hidden" id="editScoreId">
                         <div class="mb-3">
                            <label class="form-label fw-bold">Status</label>
                            <div class="btn-group w-100 bg-white" role="group">
                                <input type="radio" class="btn-check" name="editStatus" id="editStatusOK" value="OK" checked>
                                <label class="btn btn-outline-success status-btn fs-5" for="editStatusOK">
                                    <i class="fas fa-check me-2"></i>OK
                                </label>

                                <input type="radio" class="btn-check" name="editStatus" id="editStatusDNS" value="DNS">
                                <label class="btn btn-outline-warning status-btn fs-5" for="editStatusDNS">
                                    <i class="fas fa-ban me-2"></i>DNS
                                </label>
                            
                                <input type="radio" class="btn-check" name="editStatus" id="editStatusDNF" value="DNF">
                                <label class="btn btn-outline-danger status-btn fs-5" for="editStatusDNF">
                                    <i class="fas fa-flag me-2"></i>DNF
                                </label>
                                
                                <input type="radio" class="btn-check" name="editStatus" id="editStatusDSQ" value="DSQ">
                                <label class="btn btn-outline-dark status-btn fs-5" for="editStatusDSQ">
                                    <i class="fas fa-times-circle me-2"></i>DSQ
                                </label>
                            </div>
                        </div>
                        <!-- Score Input with Keypad (same as score.php) -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Score (0-99)</label>
                            <input type="text" class="bg-gradient form-control fs-1 score-input ss-f-number bg-light" 
                                id="editScoreValue" 
                                placeholder="Use keypad below" 
                                autocomplete="off" required readonly>
                        </div>
                        <div class="mb-1">
                            <input type="range" class="form-range" id="editScoreRange" min="0" max="99" oninput="document.getElementById('editScoreValue').value=this.value">
                        </div>

                        <!-- Numeric Keypad -->
                        <div class="btn-group-vertical mt-0 rounded-2 w-100 p-3 mb-3" role="group" aria-label="Numeric keypad">
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('1')"><span>1</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('2')"><span>2</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('3')"><span>3</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('4')"><span>4</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('5')"><span>5</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('6')"><span>6</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('7')"><span>7</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('8')"><span>8</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('9')"><span>9</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-1">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="backspaceEditScore()"><span>&lt;</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('0')"><span>0</span></button>
                                <button type="button" class="bg-gradient bg-info btn btn-lg btn-primary py-3 key fs-1 border-0" onclick="applyEditScoreValue()"><span>Go</span></button>
                            </div>
                        </div>
                        
                       
                        
                        <div class="mb-3" id="editScoreFigures" style="display:block;">
                            <label class="form-label">Figures</label>
                            <div class="small text-muted mb-2" id="editScoreFiguresContent"></div>
                            <div id="editFiguresEditor" style="display:none;">
                                <div class="border rounded p-3 bg-light">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <small class="fw-bold text-muted">Edit Figures:</small>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="cancelEditFigures()">
                                            <i class="fas fa-times"></i> Cancel
                                        </button>
                                    </div>
                                    <div id="editFigureCategories"></div>
                                    <button type="button" class="btn btn-sm btn-success mt-2" onclick="saveFiguresEdit()">
                                        <i class="fas fa-save"></i> Save Figures
                                    </button>
                                </div>
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" id="editFiguresBtn" onclick="startEditFigures()" style="display:none;">
                                <i class="fas fa-edit"></i> Edit Figures
                            </button>
                            <!-- Score Context inserted below participant summary per request -->
                            <div id="scoreContextInfo" class="mt-3 p-3 border rounded bg-light small"></div>
                        </div>
                        
                        <div class="alert alert-<?php echo $user_role === 'admin' ? 'danger' : 'warning'; ?>">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong><?php echo $user_role === 'admin' ? 'Admin Override:' : 'Head Judge Override:'; ?></strong> 
                            This will permanently modify the judge's score and require re-approval.
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <!-- Approval buttons (hold to trigger) -->
                    <div class="me-auto d-flex gap-2" id="editModalApprovalButtons" style="display: none;">
                        <span class="approval-clickable btn btn-outline-success border-0 w-auto unapprove-btn" data-score-id="" style="cursor: pointer; position: relative; overflow: hidden;" title="Hold to unapprove">
                            <i class="fas fa-check-circle me-1"></i>Approved
                            <span class="hold-progress" style="position: absolute; bottom: 0; left: 0; height: 3px; width: 0%; background: #dc3545; transition: none;"></span>
                        </span>
                        <span class="approval-clickable btn btn-outline-danger w-auto approve-btn" data-score-id="" style="cursor: pointer; position: relative; overflow: hidden; display: none;" title="Hold to approve">
                            <i class="fas fa-times-circle me-1"></i>Not Approved
                            <span class="hold-progress" style="position: absolute; bottom: 0; left: 0; height: 3px; width: 0%; background: #28a745; transition: none;"></span>
                        </span>
                        
                        <!-- Delete Score Button (hold to trigger) -->
                        <span class="btn btn-outline-danger w-auto delete-score-btn" data-score-id="" style="cursor: pointer; position: relative; overflow: hidden;" title="Hold to delete this score permanently">
                            <i class="fas fa-trash me-1"></i>Delete Score
                            <span class="hold-progress" style="position: absolute; bottom: 0; left: 0; height: 3px; width: 0%; background: #dc3545; transition: none;"></span>
                        </span>
                    </div>
                    
                    <button type="button" class="btn btn-primary" id="saveScoreBtn">
                        <i class="fas fa-save me-2"></i>Update Score
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Score Modal -->
    <div class="modal fade" id="createScoreModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-plus me-2"></i>
                        <?php echo $user_role === 'admin' ? 'Admin - Create Score' : 'Head Judge - Create Score'; ?>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <!-- Participant Summary (context moved below figures) -->
                    <div class="alert alert-info mb-3" id="createScoreInfo">
                        <div class="text-center text-muted">
                            <i class="fas fa-spinner fa-spin me-2"></i>Loading participant information...
                        </div>
                    </div>
                    
                    <form id="createScoreForm">
                        <input type="hidden" id="createParticipantId">
                        <input type="hidden" id="createJudgeId">
                         <div class="mb-3">
                            <label class="form-label fw-bold">Status</label>
                            <div class="btn-group w-100 bg-white" role="group">
                                <input type="radio" class="btn-check" name="createStatus" id="createStatusOK" value="OK" checked>
                                <label class="btn btn-outline-success status-btn fs-5" for="createStatusOK">
                                    <i class="fas fa-check me-2"></i>OK
                                </label>

                                <input type="radio" class="btn-check" name="createStatus" id="createStatusDNS" value="DNS">
                                <label class="btn btn-outline-warning status-btn fs-5" for="createStatusDNS">
                                    <i class="fas fa-ban me-2"></i>DNS
                                </label>
                            
                                <input type="radio" class="btn-check" name="createStatus" id="createStatusDNF" value="DNF">
                                <label class="btn btn-outline-danger status-btn fs-5" for="createStatusDNF">
                                    <i class="fas fa-flag me-2"></i>DNF
                                </label>
                                
                                <input type="radio" class="btn-check" name="createStatus" id="createStatusDSQ" value="DSQ">
                                <label class="btn btn-outline-dark status-btn fs-5" for="createStatusDSQ">
                                    <i class="fas fa-times-circle me-2"></i>DSQ
                                </label>
                            </div>
                        </div>
                        <!-- Score Input with Keypad (same as score.php) -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Score (0-99)</label>
                            <input type="text" class="bg-gradient form-control fs-1 score-input ss-f-number bg-light" 
                                id="createScoreValue" 
                                placeholder="Use keypad below" 
                                autocomplete="off" required readonly>
                        </div>
                        <div class="mb-1">
                            <input type="range" class="form-range" id="createScoreRange" min="0" max="99" oninput="document.getElementById('createScoreValue').value=this.value">
                        </div>

                        <!-- Numeric Keypad -->
                        <div class="btn-group-vertical mt-0 rounded-2 w-100 p-3 mb-3" role="group" aria-label="Numeric keypad">
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('1')"><span>1</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('2')"><span>2</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('3')"><span>3</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('4')"><span>4</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('5')"><span>5</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('6')"><span>6</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('7')"><span>7</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('8')"><span>8</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('9')"><span>9</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-1">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="backspaceCreateScore()"><span>&lt;</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('0')"><span>0</span></button>
                                <button type="button" class="bg-gradient bg-info btn btn-lg btn-primary py-3 key fs-1 border-0" onclick="applyCreateScoreValue()"><span>Go</span></button>
                            </div>
                        </div>
                        
                       
                        
                        <!-- Figures Section (mirrors Edit Score Modal structure) -->
                        <div class="mb-3" id="createFiguresSection" style="display: block;">
                            <label class="form-label">Figures</label>
                            <div class="small text-muted mb-2" id="createScoreFiguresContent"></div>
                            <div id="createFiguresEditor" style="display:none;">
                                <div class="border rounded p-3 bg-light">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <small class="fw-bold text-muted">Edit Figures:</small>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="cancelCreateFiguresEdit()">
                                            <i class="fas fa-times"></i> Cancel
                                        </button>
                                    </div>
                                    <div id="createFiguresContainer">
                                        <div class="text-muted">Loading figure options...</div>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-success mt-2" onclick="saveCreateFiguresEdit()">
                                        <i class="fas fa-save"></i> Save Figures
                                    </button>
                                </div>
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" id="createFiguresBtn" onclick="startCreateFiguresEdit()" style="display:none;">
                                <i class="fas fa-edit"></i> Edit Figures
                            </button>
                            
                        </div>
                        
                        <div class="alert alert-<?php echo $user_role === 'admin' ? 'danger' : 'warning'; ?>">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong><?php echo $user_role === 'admin' ? 'Admin:' : 'Head Judge:'; ?></strong> 
                            This will create a new score entry that requires approval.
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-success" id="createScoreBtn">
                        <i class="fas fa-plus me-2"></i>Create Score
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Results Modal Integration -->
    <div class="modal fade" id="heatResultsModal" tabindex="-1" aria-labelledby="heatResultsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-fullscreen p-5">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="heatResultsModalLabel">
                        <i class="fas fa-trophy me-2"></i>
                        <span id="resultsEventName">Event Results</span>
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <div class="row g-0">
                        <!-- Sidebar: Saved Configurations -->
                        <div class="col-12 col-lg-3 bg-light border-end">
                            <div class="p-3">
                                <h6 class="fw-bold mb-3">
                                    <i class="fas fa-list me-2"></i>Available Reports
                                </h6>
                                <div id="savedConfigsList" class="d-flex flex-column gap-2">
                                    <!-- Configurations will be loaded here -->
                                    <div class="text-center text-muted py-3">
                                        <i class="fas fa-spinner fa-spin"></i>
                                        <p class="small mb-0 mt-2">Loading configurations...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Main Content: Results Display -->
                        <div class="col-12 col-lg-9">
                            <div class="p-4">
                                <div id="resultsContentDisplay">
                                    <!-- Results content will be loaded here -->
                                    <div class="text-center text-muted py-5">
                                        <i class="fas fa-chart-bar fa-3x mb-3 opacity-25"></i>
                                        <p class="mb-0">Select a report from the sidebar</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Help Modal -->
    <div class="modal fade" id="headJudgeHelpModal" tabindex="-1" aria-labelledby="headJudgeHelpTitle" aria-hidden="true">
        <div class="modal-dialog modal-xl modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="headJudgeHelpTitle">
                        <i class="fas fa-circle-question me-2"></i>Head Judge Panel – Help
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <iframe id="headJudgeHelpIframe" style="width:100%; height:80vh; border:none;" title="Head Judge Help"></iframe>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteScoreConfirmModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-danger">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle me-2"></i>Confirm Delete Score
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-danger mb-3">
                        <i class="fas fa-trash me-2"></i>
                        <strong>Warning:</strong> This action CANNOT be undone!
                    </div>
                    <p class="mb-3">Are you sure you want to permanently delete this score?</p>
                    <div id="deleteScoreDetails" class="bg-light p-3 rounded">
                        <!-- Details will be populated by JavaScript -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-danger" id="confirmDeleteScoreBtn">
                        <i class="fas fa-trash me-1"></i>Delete Score
                    </button>
                </div>
            </div>
        </div>
    </div>

    
    <script>
        function openHeadJudgeHelp() {
            const modalEl = document.getElementById('headJudgeHelpModal');
            const iframe = document.getElementById('headJudgeHelpIframe');
            if (iframe && !iframe.src) {
                iframe.src = '../help/head_judge_help.html';
            }
            const modal = new bootstrap.Modal(modalEl);
            modal.show();
        }
         let autoRefreshInterval;
        let currentData = null;
        let currentSortMode = 'default';
        let currentFilters = {
            gender: '',
            category: '',
            bibName: '',
            showToApprove: true,
            showAllApproved: true,
            showWaiting: true
        };
        let heatCardLayout = 'judges'; // Heat card layout: 'full', 'compact', 'monitor', 'minimal', 'monitor-row', 'judges'
        let currentHeatSettings = {};
        let headJudgeQueueLoading = false;
        const HEAD_JUDGE_ALERT_POLL_INTERVAL = 15000;
        let heatNotifications = [];
        let latestNotificationId = 0;
        let heatNotificationsTimer = null;
        let heatNotificationsInitialized = false;
        let notificationsLoading = false;
        
        document.addEventListener('DOMContentLoaded', function() {
            const eventSelect = document.getElementById('eventSelect');
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            const refreshBtn = document.getElementById('refreshBtn');
            const autoRefreshCheck = document.getElementById('autoRefresh');
            const scoreVisibilityToggle = document.getElementById('scoreVisibilityToggle');
            if (scoreVisibilityToggle) {
                scoreVisibilityToggle.checked = true;
                scoreVisibilityToggle.addEventListener('change', function() {
                    const wrapper = document.getElementById('scoresWrapper');
                    if (wrapper) {
                        wrapper.classList.toggle('scores-hidden', !this.checked);
                    }
                });
            }
            
            // Initialize Event Selection Manager for this page
            if (window.EventSelectionManager) {
                window.EventSelectionManager.init();
            }
            
            // Sort button handlers - using event delegation to persist through DOM updates
            document.addEventListener('click', function(e) {
                const sortBtn = e.target.closest('.btn-outline-primary[data-sort]');
                if (sortBtn && !sortBtn.classList.contains('disabled')) {
                    // Update active state
                    document.querySelectorAll('.btn-outline-primary[data-sort]').forEach(b => b.classList.remove('active'));
                    sortBtn.classList.add('active');
                    
                    // Update sort mode and re-render
                    currentSortMode = sortBtn.getAttribute('data-sort');
                    if (currentData) {
                        renderScoreGrid(currentData);
                    }
                }
            });

            // Filter handlers
            document.getElementById('genderFilter').addEventListener('change', function() {
                currentFilters.gender = this.value;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('categoryFilter').addEventListener('change', function() {
                currentFilters.category = this.value;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('bibNameFilter').addEventListener('input', function() {
                currentFilters.bibName = this.value;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('showToApprove').addEventListener('change', function() {
                currentFilters.showToApprove = this.checked;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('showAllApproved').addEventListener('change', function() {
                currentFilters.showAllApproved = this.checked;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('showWaiting').addEventListener('change', function() {
                currentFilters.showWaiting = this.checked;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('clearFilters').addEventListener('click', function() {
                // Reset filters
                currentFilters = {
                    gender: '',
                    category: '',
                    bibName: '',
                    showToApprove: true,
                    showAllApproved: true,
                    showWaiting: true
                };
                
                // Reset form controls
                document.getElementById('genderFilter').value = '';
                document.getElementById('categoryFilter').value = '';
                document.getElementById('bibNameFilter').value = '';
                document.getElementById('showToApprove').checked = true;
                document.getElementById('showAllApproved').checked = true;
                document.getElementById('showWaiting').checked = true;
                
                // Re-render
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });
            
            // Event selection handler - modified to work with localStorage
            eventSelect.addEventListener('change', function() {
                const eventId = this.value;
                heatSelect.disabled = !eventId;
                runSelect.disabled = true;
                
                if (eventId) {
                    loadHeats(eventId);
                    loadEventCategories(eventId); // Load event-specific categories
                    initializeHeadJudgeNotifications(true);
                    refreshHeadJudgeQueuePanel();
                    
                    // Check if Auto Heat/Run is enabled
                    const autoHeatRunCheck = document.getElementById('autoHeatRun');
                    if (autoHeatRunCheck.checked) {
                        // Auto-select active heat/run after heats load
                        setTimeout(() => {
                            autoSelectActiveHeatRun();
                        }, 500);
                    } else {
                        loadScores(); // Load scores even without heat/run selection
                    }
                } else {
                    heatSelect.innerHTML = '<option value="">Select Heat</option>';
                    runSelect.innerHTML = '<option value="">Select Run</option>';
                    document.getElementById('categoryFilter').innerHTML = '<option value="">All Categories</option>';
                    clearScoreGrid();
                    resetHeadJudgeNotificationsUI();
                    updateHeatQueuePanel([], {});
                }
            });
            
            // Heat selection handler - modified to work with localStorage
            heatSelect.addEventListener('change', function() {
                const heatNumber = this.value;
                runSelect.disabled = !heatNumber;
                
                if (heatNumber) {
                    loadRuns();
                    loadScores(); // Load scores for selected heat (all runs)
                    refreshHeadJudgeQueuePanel();
                } else {
                    runSelect.innerHTML = '<option value="">Select Run</option>';
                    loadScores(); // Load all heats/runs when heat is deselected
                    refreshHeadJudgeQueuePanel();
                }
            });
            
            // Run selection handler - modified to work with localStorage
            runSelect.addEventListener('change', function() {
                loadScores(); // Always load scores, even if run is deselected
            });
            
            // Refresh button
            refreshBtn.addEventListener('click', function() {
                const eventId = document.getElementById('eventSelect').value;
                if (eventId) {
                    loadScores();
                }
            });
            
            // Clear selections button
            document.getElementById('clearSelectionsBtn').addEventListener('click', function() {
                if (window.EventSelectionManager) {
                    window.EventSelectionManager.clearAllSelections();
                    
                    // Reset form elements
                    eventSelect.value = '';
                    heatSelect.innerHTML = '<option value="">Select Heat</option>';
                    heatSelect.disabled = true;
                    runSelect.innerHTML = '<option value="">Select Run</option>';
                    runSelect.disabled = true;
                    document.getElementById('categoryFilter').innerHTML = '<option value="">All Categories</option>';
                    clearScoreGrid();
                    
                    // Show visual feedback
                    const btn = this;
                    const originalHTML = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-check text-success"></i>';
                    btn.classList.add('btn-success');
                    btn.classList.remove('btn-outline-secondary');
                    
                    // Initialize Lucide icons for the button feedback
                    if (window.initializeLucideIcons) {
                        window.initializeLucideIcons();
                    }
                    
                    setTimeout(() => {
                        btn.innerHTML = originalHTML;
                        btn.classList.remove('btn-success');
                        btn.classList.add('btn-outline-secondary');
                        
                        // Initialize Lucide icons when restoring original content
                        if (window.initializeLucideIcons) {
                            window.initializeLucideIcons();
                        }
                    }, 1500);
                }
            });
            
           
            // Auto-refresh toggle
            autoRefreshCheck.addEventListener('change', function() {
                if (this.checked) {
                    startAutoRefresh();
                } else {
                    stopAutoRefresh();
                }
            });
            
            // Auto Heat/Run toggle
            const autoHeatRunCheck = document.getElementById('autoHeatRun');
            
            // Load saved state from localStorage
            const savedAutoHeatRun = localStorage.getItem('headJudge_autoHeatRun');
            if (savedAutoHeatRun === 'true') {
                autoHeatRunCheck.checked = true;
            }
            
            // Save state on change
            autoHeatRunCheck.addEventListener('change', function() {
                localStorage.setItem('headJudge_autoHeatRun', this.checked);
                
                // If enabled and event is selected, try to auto-select
                if (this.checked && eventSelect.value) {
                    autoSelectActiveHeatRun();
                }
            });
            
            // Start auto-refresh by default
            startAutoRefresh();
            
            // If Auto Heat/Run is enabled on page load, auto-select after a short delay
            if (autoHeatRunCheck.checked) {
                setTimeout(() => {
                    if (eventSelect.value) {
                        autoSelectActiveHeatRun();
                    }
                }, 500);
            }

            if (eventSelect.value) {
                initializeHeadJudgeNotifications(true);
                refreshHeadJudgeQueuePanel();
            }
        });

        // Function to auto-select active heat and run
        function autoSelectActiveHeatRun() {
            const eventId = document.getElementById('eventSelect').value;
            if (!eventId) return;
            
            // Fetch active heat/run from the server
            fetch('get_active_heat_run.php?event_id=' + eventId)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const heatSelect = document.getElementById('heatSelect');
                        const runSelect = document.getElementById('runSelect');
                        
                        // Select heat if available
                        if (data.heat_number) {
                            heatSelect.value = data.heat_number;
                            heatSelect.dispatchEvent(new Event('change'));
                            
                            // Wait for heat to load, then select run
                            if (data.run_number) {
                                setTimeout(() => {
                                    runSelect.value = data.run_number;
                                    runSelect.dispatchEvent(new Event('change'));
                                }, 300);
                            }
                            
                            // Show notification
                            showAutoSelectNotification(data.heat_number, data.run_number);
                        }
                    }
                })
                .catch(error => {
                    console.error('Error fetching active heat/run:', error);
                });
        }
        
        // Function to show auto-select notification
        function showAutoSelectNotification(heatNumber, runNumber) {
            const toast = document.createElement('div');
            toast.className = 'position-fixed top-0 start-50 translate-middle-x mt-3 alert alert-info alert-dismissible fade show';
            toast.style.zIndex = '9999';
            toast.innerHTML = `
                <i class="fas fa-bolt me-2"></i>
                Auto-selected: Heat ${heatNumber}${runNumber ? ', Run ' + runNumber : ''}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(toast);
            
            // Initialize Lucide icons
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons();
            }
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.remove();
                }
            }, 3000);
        }

         function updateCategoryFilter(participants) {
            const categoryFilter = document.getElementById('categoryFilter');
            const currentValue = categoryFilter.value;
            
            // Get unique categories from participants
            const categories = [...new Set(participants.map(p => p.category))].filter(c => c).sort();
            
            // Rebuild options
            categoryFilter.innerHTML = '<option value="">All Categories</option>';
            categories.forEach(category => {
                const option = document.createElement('option');
                option.value = category;
                option.textContent = category;
                if (category === currentValue) {
                    option.selected = true;
                }
                categoryFilter.appendChild(option);
            });
        }

        // Add function to load event categories
        function loadEventCategories(eventId) {
            if (!eventId) {
                document.getElementById('categoryFilter').innerHTML = '<option value="">All Categories</option>';
                return;
            }
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_event_categories');
            formData.append('event_id', eventId);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const categoryFilter = document.getElementById('categoryFilter');
                    const currentValue = categoryFilter.value;
                    
                    categoryFilter.innerHTML = '<option value="">All Categories</option>';
                    data.categories.forEach(category => {
                        const option = document.createElement('option');
                        option.value = category;
                        option.textContent = category;
                        if (category === currentValue) {
                            option.selected = true;
                        }
                        categoryFilter.appendChild(option);
                    });
                }
            })
            .catch(error => {
                console.error('Error loading categories:', error);
            });
        }

        function getApprovalStatus(participant, judges, scores, targetRunNumber = null) {
            let hasAnyScores = false;
            let hasPendingScores = false;
            let hasAllScores = true;
            
            const participantScores = scores[participant.id] || {};
            
            if (targetRunNumber) {
                // Specific run - check only that run
                const runScores = participantScores[targetRunNumber] || {};
                judges.forEach(judge => {
                    const score = runScores[judge.id];
                    if (score) {
                        hasAnyScores = true;
                        if (score.is_approved == 0) {
                            hasPendingScores = true;
                        }
                    } else {
                        hasAllScores = false;
                    }
                });
            } else {
                // No specific run - check all runs
                Object.keys(participantScores).forEach(runNum => {
                    const runScores = participantScores[runNum] || {};
                    judges.forEach(judge => {
                        const score = runScores[judge.id];
                        if (score) {
                            hasAnyScores = true;
                            if (score.is_approved == 0) {
                                hasPendingScores = true;
                            }
                        }
                    });
                });
                // For multi-run, we don't require all scores
                hasAllScores = hasAnyScores;
            }
            
            if (!hasAnyScores) {
                return 'waiting';
            } else if (hasPendingScores) {
                return 'to_approve';
            } else if (hasAnyScores && !hasPendingScores) {
                return 'all_approved';
            }
            return 'waiting';
        }

        function passesFilters(participant, approvalStatus) {
            // Gender filter
            if (currentFilters.gender && participant.gender !== currentFilters.gender) {
                return false;
            }
            
            // Category filter
            if (currentFilters.category && participant.category !== currentFilters.category) {
                return false;
            }
            
            // BIB/Name filter
            if (currentFilters.bibName) {
                const searchTerm = currentFilters.bibName.toLowerCase();
                const bibMatch = participant.bib_number.toString().includes(searchTerm);
                const nameMatch = (participant.first_name + ' ' + participant.last_name).toLowerCase().includes(searchTerm);
                if (!bibMatch && !nameMatch) {
                    return false;
                }
            }
            
            // Approval status filters
            if (approvalStatus === 'to_approve' && !currentFilters.showToApprove) {
                return false;
            }
            if (approvalStatus === 'all_approved' && !currentFilters.showAllApproved) {
                return false;
            }
            if (approvalStatus === 'waiting' && !currentFilters.showWaiting) {
                return false;
            }
            
            return true;
        }
        
        function loadHeats(eventId) {
            fetch('get_event_heats.php?event_id=' + eventId)
                .then(response => response.json())
                .then(data => {
                    const heatSelect = document.getElementById('heatSelect');
                    heatSelect.innerHTML = '<option value="">Select Heat</option>';
                    
                    data.forEach(heat => {
                        const option = document.createElement('option');
                        option.value = heat.heat_number;
                        option.textContent = `Heat ${heat.heat_number}${heat.heat_name ? ' - ' + heat.heat_name : ''}`;
                        heatSelect.appendChild(option);
                    });
                    
                    // Restore heat selection if available
                    if (window.EventSelectionManager) {
                        setTimeout(() => {
                            window.EventSelectionManager.restoreHeatSelection();
                        }, 100);
                    }
                })
                .catch(error => console.error('Error loading heats:', error));
        }
        
        function loadRuns() {
            const runSelect = document.getElementById('runSelect');
            runSelect.innerHTML = '<option value="">Select Run</option>';
            
            for (let i = 1; i <= 3; i++) {
                const option = document.createElement('option');
                option.value = i;
                option.textContent = `Run ${i}`;
                runSelect.appendChild(option);
            }
            
            // Restore run selection if available
            if (window.EventSelectionManager) {
                setTimeout(() => {
                    window.EventSelectionManager.restoreRunSelection();
                }, 100);
            }
        }
        
        function loadScores() {
            const eventId = document.getElementById('eventSelect').value;
            const heatNumber = document.getElementById('heatSelect').value;
            const runNumber = document.getElementById('runSelect').value;
            const showResultsBtn = document.getElementById('showResultsBtn');
            updateSessionInfo();
            
            // If no event selected, clear the grid and hide Results button
            if (!eventId) {
                clearScoreGrid();
                hideActiveHeatCard();
                if (showResultsBtn) {
                    showResultsBtn.style.display = 'none';
                }
                return;
            }
            
            // Show Results button when event is selected
            if (showResultsBtn) {
                showResultsBtn.style.display = 'inline-block';
            }
            
            showLoading(true);
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_scores');
            formData.append('event_id', eventId);
            
            // Only append heat/run if they are selected
            if (heatNumber) {
                formData.append('heat_number', heatNumber);
            }
            if (runNumber) {
                formData.append('run_number', runNumber);
            }
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    currentHeatSettings = {
                        ...currentHeatSettings,
                        heat_number: data.heat_number || heatNumber || currentHeatSettings.heat_number || null,
                        active_run: data.run_number || runNumber || currentHeatSettings.active_run || null
                    };
                    window.currentHeatSettings = currentHeatSettings;
                    currentData = data;
                    renderScoreGrid(data);
                    refreshHeadJudgeQueuePanel();
                    // Load active heat card if event is selected
                    loadActiveHeatCardInHeadJudge();
                } else {
                    alert('Error loading scores: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
            })
            .finally(() => {
                showLoading(false);
            });
        }
        
        // Active Heat Card Functions
        function getActiveHeatPlaceholderHTML() {
            return `
                <div class="col-12 text-center text-muted py-3">
                    <i class="fas fa-fire fa-2x mb-2 opacity-25"></i>
                    <p class="mb-0 small">Loading active heat...</p>
                </div>
            `;
        }

        function toggleHeadJudgeHeatLoadingOverlay(show) {
            const container = document.getElementById('activeHeatCardContainer');
            if (!container || !container.parentElement) return;
            const parent = container.parentElement;
            parent.classList.add('position-relative');
            let overlay = document.getElementById('activeHeatLoadingOverlay');
            if (!overlay) {
                overlay = document.createElement('div');
                overlay.id = 'activeHeatLoadingOverlay';
                overlay.className = 'heat-loading-overlay';
                overlay.innerHTML = `
                    <div class="text-center">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p class="small text-muted mt-2 mb-0">Updating heat...</p>
                    </div>
                `;
                parent.appendChild(overlay);
            }

            requestAnimationFrame(() => {
                if (show) {
                    overlay.classList.add('is-visible');
                } else {
                    overlay.classList.remove('is-visible');
                }
            });
        }

        /**
         * Switch heat card layout mode for head judge panel
         */
        function setHeadJudgeHeatCardLayout(layout, btn) {
            console.log('🎨 Switching heat card layout to:', layout);
            heatCardLayout = layout;
            
            // Update active button state
            document.querySelectorAll('[onclick^="setHeadJudgeHeatCardLayout"]').forEach(b => {
                b.classList.remove('active');
            });
            if (btn) {
                btn.classList.add('active');
            }
            
            // Reload heat card with new layout
            loadActiveHeatCardInHeadJudge();
        }
        
        async function loadActiveHeatCardInHeadJudge() {
            const eventId = document.getElementById('eventSelect').value;
            const heatNumber = document.getElementById('heatSelect').value;
            
            if (!eventId) {
                hideActiveHeatCard();
                return;
            }
            
            const container = document.getElementById('activeHeatCardContainer');
            const section = document.getElementById('activeHeatSection');
            
            // Show the section
            section.style.display = 'block';

            const placeholderHTML = getActiveHeatPlaceholderHTML();
            const hasExistingContent = container.dataset.hasContent === 'true' && container.children.length > 0;

            if (!hasExistingContent) {
                container.innerHTML = placeholderHTML;
            } else {
                toggleHeadJudgeHeatLoadingOverlay(true);
            }
            
            try {
                // Load heat cards from API with specified layout
                const response = await fetch(`../admin/heat_cards_api.php?event_id=${eventId}&layout=${heatCardLayout}`);
                const data = await response.json();
                
                if (!response.ok || !data.success) {
                    throw new Error(data.error || 'Failed to load heat cards');
                }
                
                // Parse the HTML and extract heat cards
                const parser = new DOMParser();
                const doc = parser.parseFromString(data.data, 'text/html');
                // Select all heat card containers (works for all layouts: col-md-4, col-12, etc.)
                const allCards = doc.querySelectorAll('[class*="col-"]');
                
                let activeCard = null;
                let foundHeat = null;
                
                // If specific heat number provided, find that heat
                if (heatNumber) {
                    allCards.forEach(card => {
                        const heatTitle = card.querySelector('.card-title');
                        if (heatTitle && heatTitle.textContent.includes(`Heat ${heatNumber}`)) {
                            activeCard = card;
                            foundHeat = heatNumber;
                        }
                    });
                }
                
                // If no specific heat or not found, look for active heat
                if (!activeCard) {
                    allCards.forEach(card => {
                        // Look for the ACTIVE badge (bg-warning text-dark)
                        const activeStatus = card.querySelector('.badge.bg-warning.text-dark');
                        if (activeStatus && activeStatus.textContent.includes('ACTIVE')) {
                            activeCard = card;
                            const heatTitle = card.querySelector('.card-title, h5');
                            if (heatTitle) {
                                const match = heatTitle.textContent.match(/Heat (\d+)/);
                                if (match) foundHeat = match[1];
                            }
                        }
                        
                        // Alternative: Check if card has active border class
                        if (!activeCard && card.querySelector('.card.border-success')) {
                            activeCard = card;
                            const heatTitle = card.querySelector('.card-title, h5');
                            if (heatTitle) {
                                const match = heatTitle.textContent.match(/Heat (\d+)/);
                                if (match) foundHeat = match[1];
                            }
                        }
                    });
                }
                
                if (activeCard) {
                    // Display the heat card
                    container.innerHTML = activeCard.outerHTML;
                    container.dataset.hasContent = 'true';
                    console.log('Loaded heat card for Heat:', foundHeat);
                } else {
                    container.innerHTML = `
                        <div class="col-12 text-center text-muted py-3">
                            <i class="fas fa-fire fa-2x mb-2 opacity-25"></i>
                            <p class="mb-0">No active heat found</p>
                            <p class="small text-muted">Activate a heat to see details here</p>
                        </div>
                    `;
                    container.dataset.hasContent = 'false';
                }
                
            } catch (error) {
                console.error('Error loading active heat card:', error);
                container.innerHTML = `
                    <div class="col-12">
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Error loading heat data:</strong> ${error.message}
                        </div>
                    </div>
                `;
                container.dataset.hasContent = 'false';
            }
            toggleHeadJudgeHeatLoadingOverlay(false);
        }
        
        function hideActiveHeatCard() {
            const section = document.getElementById('activeHeatSection');
            if (section) {
                section.style.display = 'none';
            }
            const container = document.getElementById('activeHeatCardContainer');
            if (container) {
                container.innerHTML = getActiveHeatPlaceholderHTML();
                container.dataset.hasContent = 'false';
            }
            toggleHeadJudgeHeatLoadingOverlay(false);
        }
        
        // Function to validate participant diversity rules
        function validateParticipantDiversityRules(participant, diversityRules, data) {
            if (!diversityRules || diversityRules.length === 0) {
                return { valid: true, errors: [] };
            }
            
            const errors = [];
            const figuresByCategory = {};
            
            // Collect all figures from all runs for this participant
            const participantScores = data.scores[participant.id] || {};
            Object.keys(participantScores).forEach(runNum => {
                Object.keys(participantScores[runNum]).forEach(judgeId => {
                    const score = participantScores[runNum][judgeId];
                    if (score && score.figures_json) {
                        try {
                            const figuresData = JSON.parse(score.figures_json);
                            const actualFigures = figuresData.figures || figuresData;
                            
                            // Collect figures by category
                            Object.keys(actualFigures).forEach(category => {
                                if (!figuresByCategory[category]) {
                                    figuresByCategory[category] = [];
                                }
                                const items = actualFigures[category];
                                if (Array.isArray(items)) {
                                    figuresByCategory[category].push(...items);
                                }
                            });
                        } catch (e) {
                            // Skip invalid JSON
                        }
                    }
                });
            });
            
            // Validate against each rule
            diversityRules.forEach(rule => {
                const ruleType = rule.rule_type || '';
                const groupName = rule.group_name || '';
                const minDistinct = parseInt(rule.min_distinct || 0);
                const isOptional = rule.is_optional == 1;
                
                // Skip optional rules
                if (isOptional) {
                    return;
                }
                
                // Check for requireDistinct rule type
                if ((ruleType === 'requireDistinct' || ruleType === 'min_distinct') && minDistinct > 0) {
                    const items = figuresByCategory[groupName] || [];
                    const uniqueCount = new Set(items).size;
                    
                    if (uniqueCount < minDistinct) {
                        errors.push('Requires at least ' + minDistinct + ' different ' + groupName + ' figures (has ' + uniqueCount + ')');
                    }
                }
            });
            
            return {
                valid: errors.length === 0,
                errors: errors
            };
        }
        
        // Function to calculate run score with drop rules
        function calculateRunScoreWithDrops(judgeScores, dropRule) {
            if (judgeScores.length === 0) {
                return { score: 0, dropInfo: null };
            }
            
            // Initialize drop info
            let dropInfo = {
                droppedHighest: null,
                droppedLowest: null,
                method: 'none'
            };
            
            // Only apply drop rules if drop_rule is not 'none' and we have enough judges
            if (dropRule !== 'none' && judgeScores.length >= 3) {
                // Apply drop rule based on the format configuration
                const sortedScores = [...judgeScores].sort((a, b) => a.score - b.score);
                
                if (dropRule === 'highest_and_lowest') {
                    const droppedLowest = sortedScores[0];
                    const droppedHighest = sortedScores[sortedScores.length - 1];
                    const usedScores = sortedScores.slice(1, -1);
                    
                    dropInfo = {
                        droppedHighest: droppedHighest,
                        droppedLowest: droppedLowest,
                        method: 'highest_and_lowest'
                    };
                    
                    // Calculate average of remaining scores
                    const runScore = usedScores.reduce((sum, item) => sum + item.score, 0) / usedScores.length;
                    return { score: runScore, dropInfo: dropInfo };
                } else if (dropRule === 'lowest') {
                    const droppedLowest = sortedScores[0];
                    const usedScores = sortedScores.slice(1);
                    
                    dropInfo = {
                        droppedHighest: null,
                        droppedLowest: droppedLowest,
                        method: 'lowest'
                    };
                    
                    const runScore = usedScores.reduce((sum, item) => sum + item.score, 0) / usedScores.length;
                    return { score: runScore, dropInfo: dropInfo };
                } else if (dropRule === 'highest') {
                    const droppedHighest = sortedScores[sortedScores.length - 1];
                    const usedScores = sortedScores.slice(0, -1);
                    
                    dropInfo = {
                        droppedHighest: droppedHighest,
                        droppedLowest: null,
                        method: 'highest'
                    };
                    
                    const runScore = usedScores.reduce((sum, item) => sum + item.score, 0) / usedScores.length;
                    return { score: runScore, dropInfo: dropInfo };
                }
            }
            
            // If drop_rule is 'none' or not enough judges, use all scores
            const runScore = judgeScores.reduce((sum, item) => sum + item.score, 0) / judgeScores.length;
            return { score: runScore, dropInfo: dropInfo };
        }
        
        // Function to check if a judge score should be marked as dropped
        function isJudgeScoreDropped(judgeId, dropInfo) {
            if (!dropInfo || dropInfo.method === 'none') {
                return false;
            }
            
            return (dropInfo.droppedHighest && dropInfo.droppedHighest.judge_id == judgeId) ||
                   (dropInfo.droppedLowest && dropInfo.droppedLowest.judge_id == judgeId);
        }
        
        function renderScoreGrid(data) {
    // Preserve scroll position before updating
    preserveScrollPosition();
    
    const container = document.getElementById('scoresContainer');
    const noDataMessage = document.getElementById('noDataMessage');
    const sortControls = document.getElementById('sortControls');
    
    if (!data.participants.length) {
        container.style.display = 'none';
        sortControls.classList.add('d-none');
        noDataMessage.style.display = 'block';
        
        const heatValue = document.getElementById('heatSelect').value;
        const runValue = document.getElementById('runSelect').value;
        
        let message = '';
        if (!heatValue) {
            message = `
                <i class="fas fa-info-circle fa-3x mb-3 opacity-50"></i>
                <h5>Select a Heat to View Scores</h5>
                <p>Choose a heat from the dropdown above to display participants and their scores</p>
            `;
        } else if (!runValue) {
            message = `
                <i class="fas fa-info-circle fa-3x mb-3 opacity-50"></i>
                <h5>Select a Run to View Scores</h5>
                <p>Choose a run from the dropdown above to display scores for Heat ${heatValue}</p>
            `;
        } else {
            message = `
                <i class="fas fa-users-slash fa-3x mb-3 opacity-50"></i>
                <h5>No participants found</h5>
                <p>No participants are registered for Heat ${heatValue}</p>
            `;
        }
        
        noDataMessage.innerHTML = message;
        return;
    }

    // Update category filter options
    updateCategoryFilter(data.participants);
    
    // Group judges by control points
    const controlPointGroups = {};
    const controlPointColors = ["scale-1",
  "scale-2",
  "scale-3",
  "scale-4",
  "scale-5",
  "scale-6",
  "scale-7",
  "scale-8",
  "scale-9",
  "scale-10",
  "scale-11",
  "scale-12"];
    
    // Assign unique color to each judge
    let judgeColorIndex = 0;
    data.judges.forEach(judge => {
        const cpName = judge.control_point_name || '';
        if (!controlPointGroups[cpName]) {
            controlPointGroups[cpName] = {
                judges: [],
                color: '' // Control point won't have a single color anymore
            };
        }
        // Assign unique color to each judge
        judge.uniqueColor = controlPointColors[judgeColorIndex % controlPointColors.length];
        judgeColorIndex++;
        controlPointGroups[cpName].judges.push(judge);
    });
    
    // Load scoring format configuration for drop rule calculations
    let scoringFormat = null;
    const eventId = document.getElementById('eventSelect').value;
    
    // Prepare participants with calculated data and filter them
    let processedParticipants = data.participants.map(participant => {
        let hasAllScores = true;
        let scoreValues = [];
        let latestScoreTime = null;
        
        // Determine which run to display based on data.run_number
        const targetRunNumber = data.run_number;
        
        // Collect all judge scores for this participant (for the specific run if specified)
        let judgeScores = [];
        
        if (targetRunNumber) {
            // Specific run selected - use old logic
            data.judges.forEach(judge => {
                const score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                    ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                if (!score) {
                    hasAllScores = false;
                } else {
                    // Add to judge scores array with judge info for drop rule processing
                    if (score.score_value !== null && score.status === 'OK') {
                        judgeScores.push({
                            judge_id: judge.id,
                            judge_name: judge.name,
                            score: parseFloat(score.score_value)
                        });
                        scoreValues.push(parseFloat(score.score_value));
                    }
                    
                    // Track latest score time
                    if (score.created_at) {
                        const scoreTime = new Date(score.created_at);
                        if (!latestScoreTime || scoreTime > latestScoreTime) {
                            latestScoreTime = scoreTime;
                        }
                    }
                }
            });
            // Update hasAllScores based on actual judge count
            hasAllScores = judgeScores.length === data.judges.length;
        } else {
            // No specific run - collect scores from all runs for this participant
            const participantScores = data.scores[participant.id] || {};
            Object.keys(participantScores).forEach(runNum => {
                data.judges.forEach(judge => {
                    const score = participantScores[runNum][judge.id];
                    if (score) {
                        if (score.score_value !== null && score.status === 'OK') {
                            judgeScores.push({
                                judge_id: judge.id,
                                judge_name: judge.name,
                                score: parseFloat(score.score_value),
                                run_number: runNum
                            });
                            scoreValues.push(parseFloat(score.score_value));
                        }
                        
                        // Track latest score time
                        if (score.created_at) {
                            const scoreTime = new Date(score.created_at);
                            if (!latestScoreTime || scoreTime > latestScoreTime) {
                                latestScoreTime = scoreTime;
                            }
                        }
                    }
                });
            });
            // For multi-run view, we don't require all scores
            hasAllScores = judgeScores.length > 0;
        }
        
        // Calculate run score with drop rules
        let runScore = 0;
        let dropInfo = null;
        if (judgeScores.length > 0) {
            const result = calculateRunScoreWithDrops(judgeScores, data.drop_rule || 'none');
            runScore = result.score;
            dropInfo = result.dropInfo;
        }
        
        const average = runScore;
        const approvalStatus = getApprovalStatus(participant, data.judges, data.scores, targetRunNumber);
        
        return {
            ...participant,
            hasAllScores,
            average,
            runScore,
            dropInfo,
            judgeScores,
            latestScoreTime,
            canSort: judgeScores.length > 0, // Allow sorting if participant has at least one valid score
            approvalStatus,
            passesFilter: passesFilters(participant, approvalStatus)
        };
    });

    // Filter participants
    const filteredParticipants = processedParticipants.filter(p => p.passesFilter);
    window.latestProcessedParticipants = processedParticipants;
    window.latestFilteredParticipants = filteredParticipants;
    updateSummaryMetrics(processedParticipants, filteredParticipants);
    updateHeatQueuePanel(
        data.queue ||
        data.all_performing ||
        data.performing_participants ||
        data.performing ||
        data.current_performing ||
        data.performing_now || [],
        currentHeatSettings
    );
    
    // Apply sorting
    const sortableParticipants = filteredParticipants.filter(p => p.canSort);
    const unsortableParticipants = filteredParticipants.filter(p => !p.canSort);
    
    // Sort the sortable participants based on current mode
    switch (currentSortMode) {
        case 'average':
            sortableParticipants.sort((a, b) => b.average - a.average);
            break;
        case 'latest':
            sortableParticipants.sort((a, b) => {
                if (!a.latestScoreTime && !b.latestScoreTime) return 0;
                if (!a.latestScoreTime) return 1;
                if (!b.latestScoreTime) return -1;
                return b.latestScoreTime - a.latestScoreTime;
            });
            break;
        case 'default':
        default:
            sortableParticipants.sort((a, b) => {
                if (a.sort_order !== b.sort_order) {
                    return a.sort_order - b.sort_order;
                }
                return a.bib_number - b.bib_number;
            });
            break;
    }
    
    // Combine sorted and unsorted participants
    const finalParticipants = [...sortableParticipants, ...unsortableParticipants];
    
    // Update sort controls and counters
    sortControls.classList.remove('d-none');
    document.getElementById('sortableCount').textContent = `${sortableParticipants.length} complete participants`;
    document.getElementById('filteredCount').textContent = filteredParticipants.length;
    document.getElementById('totalCount').textContent = data.participants.length;
    
    // Debug logging
    console.log('Sort Controls Debug:', {
        sortableCount: sortableParticipants.length,
        filteredCount: filteredParticipants.length,
        totalCount: data.participants.length,
        totalJudges: data.judges.length,
        hasAnyComplete: sortableParticipants.length > 0,
        currentSortMode: currentSortMode,
        sampleParticipant: processedParticipants[0] ? {
            name: processedParticipants[0].first_name,
            canSort: processedParticipants[0].canSort,
            hasAllScores: processedParticipants[0].hasAllScores,
            judgeScoresCount: processedParticipants[0].judgeScores.length,
            requiredScores: data.judges.length
        } : null
    });
    
    // Enable/disable sort buttons and restore active state
    const hasAnyComplete = sortableParticipants.length > 0;
    document.querySelectorAll('.btn-outline-primary[data-sort]').forEach(btn => {
        const sortType = btn.getAttribute('data-sort');
        
        // Remove disabled class from all first
        btn.classList.remove('disabled');
        
        // Add disabled only to average/latest if no complete participants
        if ((sortType === 'average' || sortType === 'latest') && !hasAnyComplete) {
            btn.classList.add('disabled');
        }
        
        // Restore active state based on current sort mode
        if (sortType === currentSortMode) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
    
    noDataMessage.style.display = 'none';
    container.style.display = 'block';

    // Show "no results" message if all participants are filtered out
    if (filteredParticipants.length === 0) {
        container.innerHTML = `
            <div class="text-center text-muted py-5">
                <i class="fas fa-filter fa-3x mb-3 opacity-50"></i>
                <h5>No participants match current filters</h5>
                <p>Try adjusting your filter settings to see more participants</p>
                <button class="btn btn-outline-primary" onclick="document.getElementById('clearFilters').click()">
                    <i class="fas fa-times me-1"></i>Clear All Filters
                </button>
            </div>
        `;
        return;
    }
    
    // Build the grid HTML using Bootstrap table
    let html = '';
    
    // Add drop rule debug info at the top with source indicator
    const dropRuleSource = data.drop_rule_source || 'format';
    const isHeatOverride = dropRuleSource === 'heat';
        
        html += '<div class="table-responsive"><table class="table table-bordered table-hover table-striped" id="scoresTable">';
    
    // Header row with control point groups and judge names
    html += '<thead style="border-bottom: 2px solid gray;">';
    
    // Control point header row
    html += '<th class="" rowspan="2" class="text-center align-middle">#</th>';
    Object.keys(controlPointGroups).forEach(cpName => {
        const group = controlPointGroups[cpName];
        html += `<th colspan="${group.judges.length}" class="text-center">
            <i class="fas fa-map-marker-alt me-2"></i>${cpName}
            <span class="badge bg-light text-dark ms-2">${group.judges.length} judges</span>
        </th>`;
    });
    html += '<th rowspan="2" class="text-center align-middle" style="width: 160px;">Average</th>';
    html += '<th rowspan="2" class="text-center align-middle participantColMobileHide">Actions</th></tr>';
    
    // Judge names row
    html += '';
    Object.keys(controlPointGroups).forEach(cpName => {
        const group = controlPointGroups[cpName];
        group.judges.forEach(judge => {
            const headStar = judge.is_head == 1 ? '<i data-lucide="star" class="text-warning ms-1" title="Head Judge" style="width: 14px; height: 14px;"></i>' : '';
            html += `<th class="text-center ${judge.uniqueColor} bg-opacity-25">
                <div class="small fw-bold">${judge.name}${headStar}</div>
            </th>`;
        });
    });
    html += '</th></thead><tbody>';
    
    // Participant rows
    finalParticipants.forEach((participant, index) => {
        // Validate diversity rules for this participant
        let diversityValidation = { valid: true, errors: [] };
        if (data.diversity_rules && data.diversity_rules.length > 0 && data.diversity_enabled) {
            diversityValidation = validateParticipantDiversityRules(participant, data.diversity_rules, data);
        }
        
        // Apply row class based on approval status and diversity validation
        let rowClass = participant.canSort ? 'bg-approved bg-opacity-25 border' : '';
        if (!diversityValidation.valid) {
            rowClass += ' diversity_rules_not_meet bg-warning bg-opacity-10';
        }
        
        // Position indicator for sorted participants
        const positionBadge = participant.canSort && currentSortMode !== 'default' ? 
            `<span class="badge bg-warning me-1">#${sortableParticipants.indexOf(participant) + 1}</span>` : '';
        
        // Calculate total columns for mobile row
        const totalJudges = data.judges.length;
        const totalColumns = totalJudges; // Only span across judge columns (not Average/Actions)
        
        // Calculate average for mobile rowspan display
        let mobileAverageHTML = '';
        let mobileAverageAndActionsHTML = '';
        if (!diversityValidation.valid && !data.run_number) {
            // Show DNQ if diversity rules not met (only in multi-run view)
            mobileAverageHTML = `
                <span class="badge bg-warning text-white mb-2 fs-2 fw-bold w-auto">DNQ</span>
                <small class="text-danger">${diversityValidation.errors[0]}</small>
            `;
        } else if (participant.judgeScores && participant.judgeScores.length > 0) {
            const displayScore = participant.runScore;
            const formattedScore = displayScore % 1 === 0 ? displayScore.toString() : displayScore.toFixed(2);
            mobileAverageHTML = `
                <span class="mb-2 text-dark fs-2 fw-bold">${formattedScore}</span>
                <span class="badge text-dark">${participant.judgeScores.length} scores</span>
            `;
        } else {
            mobileAverageHTML = `
                <h4 class="mb-2 text-muted">—</h4>
                <span class="badge bg-secondary">No valid scores</span>
            `;
        }
        
        // Calculate pending scores for actions column by checking participant scores
        let hasPendingScores = false;
        let hasAnyScores = false;
        
        const targetRunNumber = data.run_number;
        if (targetRunNumber) {
            // Check scores for specific run
            data.judges.forEach(judge => {
                const score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                    ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                if (score) {
                    hasAnyScores = true;
                    if (score.is_approved != 1) {
                        hasPendingScores = true;
                    }
                }
            });
        } else {
            // Check all runs
            const participantScores = data.scores[participant.id] || {};
            Object.keys(participantScores).forEach(runNum => {
                data.judges.forEach(judge => {
                    const score = participantScores[runNum][judge.id];
                    if (score) {
                        hasAnyScores = true;
                        if (score.is_approved != 1) {
                            hasPendingScores = true;
                        }
                    }
                });
            });
        }
        
        // Build actions HTML for mobile rowspan
        let mobileActionsHTML = '';
        if (hasAnyScores && hasPendingScores) {
            mobileActionsHTML = `
                <button class="btn btn-primary btn-sm mb-1" onclick="approveAllParticipant(${participant.id})" title="Approve all scores for this participant">
                    <i class="fas fa-check-double me-1"></i>Approve All
                </button>
            `;
        } else if (hasAnyScores && !hasPendingScores) {
            mobileActionsHTML = `
                <span class="badge bg-success">
                    <i class="fas fa-check-circle me-1"></i>All Approved
                </span>
            `;
        } else {
            mobileActionsHTML = `
                <span class="text-muted">
                    <i class="fas fa-clock me-1"></i>Waiting
                </span>
            `;
        }
        
        // Combine average and actions for mobile view
        mobileAverageAndActionsHTML = mobileAverageHTML + '<hr class="my-2">' + mobileActionsHTML;
        
        // Mobile-only participant info row (full width)
        html += `<tr class="participantColMobileRow d-none ${rowClass}">
            <td class="text-start p-2" rowspan="4">
                ${positionBadge}
                <span class="badge bg-white fs-6 me-2 participant-bib-badge ss-f-number">${participant.bib_number}</span>
            </td>
            <td colspan="${totalColumns}" class="text-start p-2">
                <strong class="me-2">${participant.first_name} ${participant.last_name}</strong>
                <span class="badge bg-${participant.gender === 'M' ? 'info' : 'pink'} bg-opacity-20 me-1">${participant.gender}</span>
                <span class="badge text-muted me-1">${participant.club}</span>
                <span class="badge text-dark">${participant.category}</span>
                ${!diversityValidation.valid && !data.run_number ? `
                    <div class="mt-2">
                        <span class="badge bg-warning text-white w-auto">
                            <i class="fas fa-exclamation-triangle me-1"></i>DNQ
                        </span>
                        <small class="text-danger d-block mt-1">${diversityValidation.errors.join(', ')}</small>
                    </div>
                ` : ''}
            </td>
            <td class="text-center p-2" rowspan="4">
                <div class="row score-figure">
                    ${mobileAverageAndActionsHTML}
                </div>
            </td>
        </tr>`;
        
        // Second mobile row for scores
        html += `<tr class="participantColMobileRow d-none ${rowClass}">
            <!-- Judge score TDs will be added in the loop below -->
        `;
        
        // Regular desktop row with all columns
        html += `</tr><tr class="${rowClass}">`;
        
        html += `<td rowspan="2" class="participantCol">
            ${positionBadge}
            <span class="badge bg-white fs-6 mb-1 participant-bib-badge ss-f-number">${participant.bib_number}</span>
            <strong>${participant.first_name} ${participant.last_name}</strong> 
            <span class="badge bg-${participant.gender === 'M' ? 'info' : 'pink'} bg-opacity-20 ">
                ${participant.gender}
            </span>
            <br><small><span class="badge text-muted">${participant.club}</span><br><span class="badge text-dark">${participant.category}</span></small>
            ${!diversityValidation.valid && !data.run_number ? `
                <div class="mt-2">
                    <span class="badge bg-warning text-white w-auto">
                        <i class="fas fa-exclamation-triangle me-1"></i>DNQ
                    </span>
                    <small class="text-danger d-block mt-1">${diversityValidation.errors.join(', ')}</small>
                </div>
            ` : ''}
        </td>`;
        
        hasPendingScores = false;
        hasAnyScores = false;
        let scoreValues = [];
        
        // Render judge scores grouped by control points
        Object.keys(controlPointGroups).forEach(cpName => {
            const group = controlPointGroups[cpName];
            group.judges.forEach(judge => {
                // Determine which run's score to display
                const targetRunNumber = data.run_number;
                
                // If no specific run, show all runs for this judge
                if (!targetRunNumber) {
                    // Multi-run view - show all runs
                    const participantScores = data.scores[participant.id] || {};
                    const allRunScores = [];
                    
                    // Collect all run scores for this judge
                    Object.keys(participantScores).forEach(runNum => {
                        const runScore = participantScores[runNum][judge.id];
                        if (runScore) {
                            allRunScores.push({
                                run: runNum,
                                ...runScore
                            });
                        }
                    });
                    
                    html += `<td class="scoreCol text-center ${judge.uniqueColor} bg-opacity-10" data-score-value="" data-status=""><div class="row score-figure flex-column">`;
                    
                    if (allRunScores.length > 0) {
                        // Display all run scores vertically
                        allRunScores.forEach(runScore => {
                            hasAnyScores = true;
                            const isApproved = runScore.is_approved == 1;
                            if (!isApproved) hasPendingScores = true;
                            
                            if (runScore.score_value !== null && runScore.status === 'OK') {
                                scoreValues.push(parseFloat(runScore.score_value));
                            }
                            
                            const bgClass = isApproved ? 'bg-success-subtle' : 'bg-warning-subtle';
                            let numericValue = runScore.score_value;
                            let formattedValue = '';
                            if (numericValue !== null && runScore.status === 'OK') {
                                const numValue = parseFloat(numericValue);
                                formattedValue = numValue % 1 === 0 ? numValue.toString() : numValue.toFixed(2);
                            }
                            const badgeClass = getStatusBadgeClass(runScore.status);
                            const displayHtml = (runScore.status !== 'OK')
                                ? `<span class="badge bg-${badgeClass}">${runScore.status}</span>`
                                : `<span class="fw-bold text-${badgeClass === 'success' ? 'primary' : badgeClass === 'danger' ? 'danger' : 'warning'}${isApproved ? '' : ' text-warning-emphasis'}">${formattedValue}</span>`;
                            
                            // Parse figures for this run
                            let figuresHtml = '';
                            if (runScore.figures_json) {
                                try {
                                    const figuresData = JSON.parse(runScore.figures_json);
                                    const actualFigures = figuresData.figures || figuresData;
                                    const figureEntries = Object.entries(actualFigures || {});
                                    
                                    if (figureEntries.length > 0) {
                                        // Count total figures
                                        let totalFigures = 0;
                                        figureEntries.forEach(([category, items]) => {
                                            const itemsArray = Array.isArray(items) ? items : [];
                                            totalFigures += itemsArray.length;
                                        });
                                        
                                        // Determine class based on count
                                        let figClass = '';
                                        if (totalFigures <= 2) {
                                            figClass = 'fig-2';
                                        } else if (totalFigures <= 5) {
                                            figClass = 'fig-3_5';
                                        } else {
                                            figClass = 'fig_5';
                                        }
                                        
                                        figuresHtml = `<div class="figures-summary ${figClass} mt-1">`;
                                        figureEntries.forEach(([category, items]) => {
                                            const itemsArray = Array.isArray(items) ? items : [];
                                            if (itemsArray.length > 0) {
                                                figuresHtml += `<small class="text-muted">${category}:</small> `;
                                                figuresHtml += itemsArray.map(item => `<span class="badge bg-light text-dark border me-1">${item}</span>`).join('');
                                            }
                                        });
                                        figuresHtml += `</div>`;
                                    }
                                } catch (e) {
                                    // Silently fail if figures JSON is invalid
                                }
                            }
                            
                            html += `<div class="mb-1 p-1 rounded ${bgClass} w-auto" onclick="editScore(${runScore.score_id}, ${runScore.score_value || 0}, '${runScore.status}', '${encodeURIComponent(runScore.figures_json || '')}', '${participant.first_name.replace(/'/g, "\\'")} ${participant.last_name.replace(/'/g, "\\'")}', '${participant.bib_number}', '${participant.category.replace(/'/g, "\\'")}', '${judge.name.replace(/'/g, "\\'")}', '${cpName.replace(/'/g, "\\'")}')" style="cursor: pointer;">
                                <small class="text-muted">R${runScore.run}</small>
                                ${displayHtml}
                                ${isApproved ? 
                                    '<small class="badge bg-success bg-opacity-50">✓</small>' : 
                                    '<small class="badge bg-danger bg-opacity-50">!</small>'
                                }
                                ${figuresHtml}
                            </div>`;
                        });
                    } else {
                        html += `<small class="text-muted">No scores</small>`;
                    }
                    
                    html += `</div></td>`;
                    return;
                }
                
                // Single run view - original logic
                const score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                    ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                 
                if (score) {
                    hasAnyScores = true;
                    const isApproved = score.is_approved == 1;
                    if (!isApproved) hasPendingScores = true;
                    
                    // Add to score values for average calculation if it's a valid numeric score
                    if (score.score_value !== null && score.status === 'OK') {
                        scoreValues.push(parseFloat(score.score_value));
                    }
                    
                    // Determine background class based on approval status
                    let bgClass = '';
                    if (isApproved) {
                        bgClass = 'bg-success-subtle';
                    } else {
                        bgClass = 'bg-warning-subtle';
                    }
                    
                    html += `<td class="scoreCol text-center  ${bgClass}" data-score-value="${score.score_value || ''}" data-status="${score.status}"><div class="row score-figure">`;
                    
                    // Check if this score is dropped based on drop rules
                    const isDropped = isJudgeScoreDropped(judge.id, participant.dropInfo);
                    // Format score value to remove unnecessary decimals
                    let numericValue = score.score_value;
                    let formattedValue = '';
                    if (numericValue !== null && score.status === 'OK') {
                        const numValue = parseFloat(numericValue);
                        formattedValue = numValue % 1 === 0 ? numValue.toString() : numValue.toFixed(2);
                    }
                    const badgeClass = getStatusBadgeClass(score.status);
                    const dropClass = isDropped ? 'score_drop fs-4' : 'fs-2';
                    const dropIndicator = isDropped ? '(' : '';
                    const dropIndicatorEnd = isDropped ? ')' : '';
                    const displayContent = (score.status !== 'OK')
                        ? `<span class="badge bg-${badgeClass}">${score.status}</span>`
                        : `<span class="badge bg-white fw-bold score_value m-0 text-${badgeClass === 'success' ? 'primary' : badgeClass === 'danger' ? 'danger' : 'warning'}${isApproved ? '' : ' text-warning-emphasis'}">${dropIndicator}${formattedValue}${dropIndicatorEnd}</span>`;
                    
                    html += `<div class="scores-col1 ${score.status} ${dropClass} w-auto" onclick="editScore(${score.score_id}, ${score.score_value || 0}, '${score.status}', '${encodeURIComponent(score.figures_json || '')}', '${participant.first_name.replace(/'/g, "\\'")} ${participant.last_name.replace(/'/g, "\\'")}', '${participant.bib_number}', '${participant.category.replace(/'/g, "\\'")}', '${judge.name.replace(/'/g, "\\'")}', '${cpName.replace(/'/g, "\\'")}')">
                                ${displayContent}

                       
                                <div class="btn-group btn-group-sm" role="group">
                                    <span class="badge bg-warning editScoreButton"  title="Click to edit score" style="cursor: pointer; pointer-events: auto;" onclick="editScore(${score.score_id}, ${score.score_value || 0}, '${score.status}', '${encodeURIComponent(score.figures_json || '')}', '${participant.first_name.replace(/'/g, "\\'")} ${participant.last_name.replace(/'/g, "\\'")}', '${participant.bib_number}', '${participant.category.replace(/'/g, "\\'")}', '${judge.name.replace(/'/g, "\\'")}', '${cpName.replace(/'/g, "\\'")}')">
                                        <i class="fas fa-edit"></i> Edit score
                                    </span>
                                </div>
                            </div>
                             ${isApproved ? 
                            '<button class="unapprove-btn approval-clickable border-0 btn btn-outline-success w-auto position-relative" data-score-id="' + score.score_id + '" style="cursor: pointer; pointer-events: auto; overflow: hidden;" title="Hold to unapprove">' +
                                '<div class="hold-progress position-absolute bottom-0 start-0 bg-danger" style="height: 3px; width: 0%; transition: none;"></div>' +
                                'Approved' +
                            '</button>' : 
                            '<button class="approve-btn approval-clickable btn btn-outline-danger w-auto position-relative" data-score-id="' + score.score_id + '" style="cursor: pointer; overflow: hidden;" title="Hold to approve">' +
                                '<div class="hold-progress position-absolute bottom-0 start-0 bg-success" style="height: 3px; width: 0%; transition: none;"></div>' +
                                'Not Approved' +
                            '</button>'
                        }
                        `;
                    
                    // Display figures information if available
                    if (score.figures_json) {
                        try {
                            const figuresData = JSON.parse(score.figures_json);
                            // Handle nested structure - figures might be in figuresData.figures
                            const actualFigures = figuresData.figures || figuresData;
                            const figuresParts = [];
                            Object.keys(actualFigures || {}).forEach(category => {
                                const items = Array.isArray(actualFigures[category]) ? actualFigures[category] : [];
                                if (items.length > 0) {
                                    figuresParts.push(`${category}: ${items.join(', ')}`);
                                }
                            });
                            if (figuresParts.length > 0) {
                                html += `<div class="scores-col2 w-100" style="display: none !important;">
                                    <div class="figures-display mt-1 border-top p-1 small w-100 hidden">
                                        ${Object.keys(actualFigures).map(cat => {
                                            const items = Array.isArray(actualFigures[cat]) ? actualFigures[cat] : [];
                                            if (!items.length) return '';
                                            return `
                                                <div class="m-0 figure-s w-100 text-center">
                                                    ${items.map(item => `<span class=\"fw-bold badge bg-light text-primary mx-1\"><small class="small m-0 p-0 text-dark-emphasis fig-cat fig-cat-${cat}">${cat.substring(0, 3)}:</small><strong>${item}</strong></span>`).join(' ')}
                                                </div>
                                            `;
                                        }).join('')}
                                    </div></div>
                                `;
                            }
                        } catch (e) {
                            // Ignore JSON parse errors
                        }
                    }
                } else {
                    // Empty score cell - no score exists
                    html += `<td class="scoreCol text-center ${judge.uniqueColor} bg-emptyx" data-score-value="" data-status=""><div class="row score-figure">`;
                    // Empty score cell - clickable for head judges/admins
                    html += `
                        <div  onclick="createScore(${participant.id}, ${judge.id}, '${participant.first_name} ${participant.last_name}', '${judge.name}')" title="Click to add score for this judge">
                                <div class="btn btn-outline-info"><i class="fas fa-plus-circle me-1"></i> Pending</div>
                        </div>`;
                }
                
                html += `</div></td>`;
            });
        });
        
        // Average column - only for desktop (mobile uses rowspan above)
        html += `<td rowspan="2" class="resultCom text-center participantColMobileHide"><div class="row score-figure">`;
        if (!diversityValidation.valid && !data.run_number) {
            // Show DNQ if diversity rules not met (only in multi-run view)
            html += `
                <span class="badge bg-warning text-white mb-2 fs-2 fw-bold w-auto">DNQ</span>
                <small class="text-danger">${diversityValidation.errors[0]}</small>
            `;
        } else if (participant.judgeScores && participant.judgeScores.length > 0) {
            // Use the calculated run score with drop rules and format appropriately
            const displayScore = participant.runScore;
            // Format to remove unnecessary decimals
            const formattedScore = displayScore % 1 === 0 ? displayScore.toString() : displayScore.toFixed(2);
            html += `
                <span class="badge bg-white text-dark mb-2 fs-2 fw-bold score_value">${formattedScore}</span>
                <span class="badge bg-white text-dark">${participant.judgeScores.length} scores</span>
            `;
            
            // Show drop rule info if applicable
            if (participant.dropInfo && participant.dropInfo.method !== 'none') {
                //html += `<br><small class="text-muted">Drop: ${participant.dropInfo.method.replace('_', ' ')}</small>`;
            }
        } else {
            html += `
                <h4 class="mb-2 text-muted">—</h4>
                <span class="badge bg-secondary">No valid scores</span>
            `;
        }
        html += `</div></td>`;
        
        // Actions column - only for desktop (mobile uses rowspan above)
        html += `<td rowspan="2" class="actionsCol text-center participantColMobileHide"><div class="row score-figure">`;
        if (hasAnyScores && hasPendingScores) {
            html += `
                <button class="btn btn-primary btn-sm mb-1" onclick="approveAllParticipant(${participant.id})" title="Approve all scores for this participant">
                    <i class="fas fa-check-double me-1"></i>Approve All
                </button>
            `;
        } else if (hasAnyScores && !hasPendingScores) {
            html += `
                <span class="badge bg-success">
                    <i class="fas fa-check-circle me-1"></i>All Approved
                </span>
            `;
        } else {
            html += `
                <span class="text-muted">
                    <i class="fas fa-clock me-1"></i>Waiting
                </span>
            `;
        }
        html += `</div></td>`;
        
        html += '</tr>';
        
        // Add merged control point summary row
        html += `<tr class="control-point-summary bg-light border-top-2" style="border-bottom: 2px solid gray;">`;
        html += ``;
        
        // Render individual cells for each judge showing their figures
        Object.keys(controlPointGroups).forEach(cpName => {
            const group = controlPointGroups[cpName];
            
            // Determine which run to display based on data.run_number
            const targetRunNumber = data.run_number;
            
            // For each judge in this control point, create a separate cell
            group.judges.forEach(judge => {
                let score = null;
                
                if (targetRunNumber) {
                    // Specific run selected
                    score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                        ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                } else {
                    // No specific run - get scores from all runs for this judge
                    const participantScores = data.scores[participant.id] || {};
                    Object.keys(participantScores).forEach(runNum => {
                        const runScore = participantScores[runNum][judge.id];
                        if (runScore) {
                            score = runScore;
                        }
                    });
                }
                
                // Render cell for this specific judge
                html += `<td class="text-center ${judge.uniqueColor} bg-opacity-5 p-0 small">`;
                
                if (score && score.figures_json && targetRunNumber) {
                    // Only show figures in single-run view
                    try {
                        const figuresData = JSON.parse(score.figures_json);
                        const actualFigures = figuresData.figures || figuresData;
                        
                        const figureEntries = Object.entries(actualFigures || {});
                        if (figureEntries.length > 0) {
                            // Count total figures
                            let totalFigures = 0;
                            figureEntries.forEach(([category, items]) => {
                                const itemsArray = Array.isArray(items) ? items : [];
                                totalFigures += itemsArray.length;
                            });
                            
                            // Determine class based on count
                            let figClass = '';
                            if (totalFigures <= 2) {
                                figClass = 'fig-2';
                            } else if (totalFigures <= 5) {
                                figClass = 'fig-3_5';
                            } else {
                                figClass = 'fig_5';
                            }
                            
                            html += `<div class="figures-summary ${figClass} flex-wrap gap-1 d-flex justify-content-center">`;
                            figureEntries.forEach(([category, items]) => {
                                const itemsArray = Array.isArray(items) ? items : [];
                                if (itemsArray.length > 0) {
                                    html += `<div class="mb-1">
                                        <label class="text-muted d-block">${category}:</label>
                                        ${itemsArray.map(item => `<span class="badge bg-light text-dark border ss-bg-${item}">${item}</span>`).join('')}
                                    </div>`;
                                }
                            });
                            html += `</div>`;
                        } else {
                            html += `<span class="text-muted"></span>`;
                        }
                    } catch (e) {
                        html += `<span class="text-muted"></span>`;
                    }
                } else {
                    html += `<span class="text-muted"></span>`;
                }
                
                html += `</td>`;
            });
        });
        
        // Empty cells for average and actions columns
        //html += `<td class="text-muted"></td>`;
        //html += `<td class="text-muted"></td>`;
        html += `</tr>`;
    });
    
    html += '</tbody></table></div>';
    html += `
        <div class="alert alert-info mb-3">
            <strong><i class="fas fa-info-circle me-1"></i>Drop Rule Configuration:</strong>
            <span class="badge bg-primary ms-2">${data.drop_rule || 'none'}</span>
            ${isHeatOverride ? 
                '<span class="badge bg-warning text-dark ms-1"><i class="fas fa-fire me-1"></i>Heat Override</span>' : 
                '<span class="badge bg-secondary ms-1"><i class="fas fa-cog me-1"></i>Format Default</span>'}
            ${data.drop_rule && data.drop_rule !== 'none' ? 
                `<small class="ms-2">→ Removing <strong>${data.drop_rule === 'highest' ? 'highest' : 
                    (data.drop_rule === 'lowest' ? 'lowest' : 
                    (data.drop_rule === 'highest_and_lowest' ? 'highest & lowest' : data.drop_rule))}</strong> scores from average calculation</small>` : 
                '<small class="ms-2">→ All judge scores are used in average calculation</small>'}
        </div>
    `;
    
    // Display diversity rules information if enabled for this heat
    if (data.diversity_enabled && data.diversity_rules && data.diversity_rules.length > 0) {
        html += `
            <div class="alert alert-warning mb-3">
                <strong><i class="fas fa-check-double me-1"></i>Diversity Rules Enabled for This Heat:</strong>
                <div class="mt-2">
        `;
        
        data.diversity_rules.forEach(rule => {
            const isOptional = rule.is_optional == 1;
            if (!isOptional) {
                html += `
                    <div class="mb-1">
                        <span class="badge bg-warning text-dark me-2">${rule.group_name}</span>
                        <small>Requires at least <strong>${rule.min_distinct}</strong> different ${rule.group_name.toLowerCase()} figures</small>
                    </div>
                `;
            }
        });
        
        html += `
                </div>
                <small class="text-muted d-block mt-2">
                    <i class="fas fa-info-circle me-1"></i>
                    Participants not meeting diversity rules are highlighted with a yellow background
                </small>
            </div>
        `;
    }
    
    container.innerHTML = html;
    
    // Initialize Lucide icons for the new dynamic content
    if (window.initializeLucideIcons) {
        window.initializeLucideIcons();
    }
    
    // Restore scroll position after updating
    restoreScrollPosition();
}

function createScore(participantId, judgeId, participantName, judgeName) {
    // Find the judge's control point from current data
    let controlPointName = 'Unknown Control Point';
    if (currentData && currentData.judges) {
        const judge = currentData.judges.find(j => j.id == judgeId);
        if (judge && judge.control_point_name) {
            controlPointName = judge.control_point_name;
        }
    }

    // Enrich participant data from currentData
    let bibNumber = '';
    let category = '';
    let gender = '';
    let club = '';
    if (currentData && Array.isArray(currentData.participants)) {
        const p = currentData.participants.find(pp => pp.id == participantId);
        if (p) {
            bibNumber = p.bib_number || '';
            category = p.category || '';
            gender = p.gender || '';
            club = p.club || '';
            // Prefer canonical name if available
            if (p.first_name && p.last_name) {
                participantName = `${p.first_name} ${p.last_name}`;
            }
        }
    }

    document.getElementById('createParticipantId').value = participantId;
    document.getElementById('createJudgeId').value = judgeId;
    document.getElementById('createScoreValue').value = '';
    
    // Reset radio buttons to OK (default)
    document.getElementById('createStatusOK').checked = true;

    // Participant summary (match Edit modal visual style)
    const genderBadge = gender ? `<span class="badge bg-${gender==='M'?'info':'pink'} bg-opacity-20  ms-1">${gender}</span>` : '';
    const clubLine = club ? `<span class="badge text-muted">${club}</span><br>` : '';
    document.getElementById('createScoreInfo').innerHTML = `
        <div class="participantCol">
            <span class="badge bg-white fs-6 mb-1 participant-bib-badge ss-f-number">${bibNumber}</span>
            <strong>${participantName}</strong>
            ${genderBadge}
            <br><small>${clubLine}<span class="badge text-dark">${category}</span></small>
        </div>
    `;

    // Fill score context under figures section (mirror Edit modal)
    const eventSelect = document.getElementById('eventSelect');
    const heatSelect = document.getElementById('heatSelect');
    const runSelect = document.getElementById('runSelect');
    const eventName = eventSelect.options[eventSelect.selectedIndex]?.text || 'Unknown Event';
    const heatNumber = heatSelect.value || 'Unknown';
    const runNumber = runSelect.value || 'Unknown';
    const status = 'OK';
    const scoreContextHtml = `
        <div class="row g-2">
            <div class="col-md-6 small"><i class="fas fa-calendar-alt me-1 text-muted"></i><strong>Event:</strong> ${eventName}</div>
            <div class="col-md-6 small"><i class="fas fa-layer-group me-1 text-muted"></i><strong>Heat:</strong> ${heatNumber} <span class="text-muted">|</span> <strong>Run:</strong> ${runNumber}</div>
            <div class="col-md-6 small"><i class="fas fa-user-tie me-1 text-muted"></i><strong>Judge:</strong> ${judgeName}</div>
            <div class="col-md-6 small"><i class="fas fa-map-marker-alt me-1 text-muted"></i><strong>Control Point:</strong> <span class="badge bg-success ms-1">${controlPointName}</span></div>
            <div class="col-md-6 small"><i class="fas fa-info-circle me-1 text-muted"></i><strong>Status:</strong> <span class="badge bg-${getStatusBadgeClass(status)}">${status}</span></div>
        </div>`;
    const createScoreContextEl = document.getElementById('createScoreContextInfo');
    if (createScoreContextEl) createScoreContextEl.innerHTML = scoreContextHtml;

    // Reset create figures data
    window.createFiguresData = {};

    // Load figure options for create score
    loadCreateFigures();

    const modal = new bootstrap.Modal(document.getElementById('createScoreModal'));
    modal.show();
}

// Keypad functions for create score modal
function addToCreateScore(digit) {
    const input = document.getElementById('createScoreValue');
    const rangeInput = document.getElementById('createScoreRange');
    let currentValue = input.value || '';
    
    // Limit to 2 digits for scores 0-99
    if (currentValue.length < 2) {
        currentValue += digit;
        input.value = currentValue;
        rangeInput.value = currentValue;
    }
}

function backspaceCreateScore() {
    const input = document.getElementById('createScoreValue');
    const rangeInput = document.getElementById('createScoreRange');
    input.value = input.value.slice(0, -1);
    rangeInput.value = input.value || 0;
}

function applyCreateScoreValue() {
    const input = document.getElementById('createScoreValue');
    const createBtn = document.getElementById('createScoreBtn');
    
    // Validate that score value is present
    if (!input.value || input.value.trim() === '') {
        input.classList.add('is-invalid');
        return;
    }
    
    input.classList.remove('is-invalid');
    
    // Scroll modal to bottom to show footer with create button
    const modal = document.getElementById('createScoreModal');
    const modalDialog = modal.querySelector('.modal-dialog');
    
    // Wait a brief moment for any animations to complete
    setTimeout(() => {
        // Scroll the modal footer into view
        const modalFooter = modal.querySelector('.modal-footer');
        if (modalFooter) {
            modalFooter.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    }, 100);
    
    // Highlight create button
    createBtn.classList.add('pulse-highlight');
    setTimeout(() => {
        createBtn.classList.remove('pulse-highlight');
    }, 2000);
    
    // Focus out of input to trigger any validation
    input.blur();
}

function approveScore(scoreId) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'approve_score');
    formData.append('score_id', scoreId);
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessToast('Score approved successfully');
            
            // Send specific notification about score approval
            if (typeof notifyScoreApproval === 'function' && data.notification_data) {
                const nd = data.notification_data;
                // Pass enhanced data including all judge scores, current result, and rank
                const enhancedData = {
                    all_scores: nd.all_scores,
                    current_result: nd.current_result,
                    rank_position: nd.rank_position,
                    approved_count: nd.approved_count,
                    total_judges: nd.total_judges
                };
                notifyScoreApproval(nd.bib, nd.participant_name, nd.score, nd.judge, nd.heat, nd.run, enhancedData);
            } else if (typeof sendNotification === 'function') {
                sendNotification('success', 'Score Approved', 'Head judge approved a score', 'judge');
            }
            
            // Refresh grid to show updated approval status
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
            
            // Update modal buttons if edit modal is open
            updateEditModalApprovalButtons(scoreId, true);
        } else {
            alert('Error approving score: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error approving score');
    });
}

function unapproveScore(scoreId) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'unapprove_score');
    formData.append('score_id', scoreId);
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessToast('Score unapproved successfully');
            
            // Send notification about score unapproval
            if (typeof sendNotification === 'function' && data.notification_data) {
                const nd = data.notification_data;
                sendNotification('warning', 'Score Unapproved', 
                    `BIB #${nd.bib} (${nd.participant_name}) - ${nd.judge}: ${nd.score} (Heat ${nd.heat}, Run ${nd.run})`, 
                    'judge');
            }
            
            // Refresh grid to show updated approval status
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
            
            // Update modal buttons if edit modal is open
            updateEditModalApprovalButtons(scoreId, false);
        } else {
            alert('Error unapproving score: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error unapproving score');
    });
}

function deleteScore(scoreId) {
    // Get participant info from the edit modal
    const participantInfo = document.getElementById('editScoreInfo');
    const scoreValue = document.getElementById('editScoreValue').value;
    const statusRadios = document.getElementsByName('editStatus');
    let status = 'OK';
    statusRadios.forEach(radio => {
        if (radio.checked) status = radio.value;
    });
    
    // Get participant name from the info div
    const participantText = participantInfo ? participantInfo.textContent.trim() : 'Unknown';
    
    // Get score context info
    const eventSelect = document.getElementById('eventSelect');
    const heatSelect = document.getElementById('heatSelect');
    const runSelect = document.getElementById('runSelect');
    
    const eventName = eventSelect.options[eventSelect.selectedIndex]?.text || 'Unknown Event';
    const heatNumber = heatSelect.value || 'Unknown';
    const runNumber = runSelect.value || 'Unknown';
    
    // Populate the delete confirmation modal
    const deleteDetails = document.getElementById('deleteScoreDetails');
    deleteDetails.innerHTML = `
        <div class="mb-2">
            <strong>Participant:</strong> ${participantText}
        </div>
        <div class="mb-2">
            <strong>Event:</strong> ${eventName}
        </div>
        <div class="mb-2">
            <strong>Heat / Run:</strong> ${heatNumber} / ${runNumber}
        </div>
        <div class="mb-2">
            <strong>Score:</strong> <span class="badge bg-${status === 'OK' ? 'success' : 'warning'}">${status === 'OK' ? scoreValue : status}</span>
        </div>
    `;
    
    // Store scoreId for the confirm button
    document.getElementById('confirmDeleteScoreBtn').setAttribute('data-score-id', scoreId);
    
    // Show the confirmation modal
    const confirmModal = new bootstrap.Modal(document.getElementById('deleteScoreConfirmModal'));
    confirmModal.show();
}

// Handle the actual delete when confirmed
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('confirmDeleteScoreBtn').addEventListener('click', function() {
        const scoreId = this.getAttribute('data-score-id');
        
        // Close confirmation modal
        const confirmModal = bootstrap.Modal.getInstance(document.getElementById('deleteScoreConfirmModal'));
        if (confirmModal) {
            confirmModal.hide();
        }
        
        // Perform the delete
        const formData = new FormData();
        formData.append('ajax', '1');
        formData.append('action', 'delete_score');
        formData.append('score_id', scoreId);
        
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccessToast('Score deleted successfully');
                
                // Send notification about score deletion
                if (typeof sendNotification === 'function' && data.notification_data) {
                    const nd = data.notification_data;
                    sendNotification('danger', 'Score Deleted', 
                        `BIB #${nd.bib} (${nd.participant_name}) - ${nd.judge}: ${nd.score} (Heat ${nd.heat}, Run ${nd.run}) - DELETED by head judge`, 
                        'judge');
                }
                
                // Close the edit modal
                const editModal = bootstrap.Modal.getInstance(document.getElementById('editScoreModal'));
                if (editModal) {
                    editModal.hide();
                }
                
                // Refresh grid to show updated data
                if (document.getElementById('runSelect').value) {
                    loadScores();
                }
            } else {
                alert('Error deleting score: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error deleting score');
        });
    });
});

// Add the new approve all function
function approveAllParticipant(participantId) {
    const eventId = document.getElementById('eventSelect').value;
    const heatNumber = document.getElementById('heatSelect').value;
    const runNumber = document.getElementById('runSelect').value;
    
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'approve_all_participant');
    formData.append('event_id', eventId);
    formData.append('heat_number', heatNumber);
    formData.append('run_number', runNumber);
    formData.append('participant_id', participantId);
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessToast(`${data.count} scores approved`);
            
            // Send specific notification about bulk approval
            if (data.notification_data) {
                const nd = data.notification_data;
                // Try to use notification manager if available
                if (window.globalNotificationManager && typeof window.globalNotificationManager.notifyBulkScoreApproval === 'function') {
                    try {
                        window.globalNotificationManager.notifyBulkScoreApproval(nd.bib, nd.participant_name, nd.count, nd.heat, nd.run);
                    } catch (error) {
                        console.warn('Notification manager error:', error);
                        // Fall back to generic notification
                        if (typeof sendNotification === 'function') {
                            const countText = nd.count > 0 ? ` (${nd.count} scores)` : '';
                            sendNotification('success', 'All Scores Approved', `Head judge approved all scores for #${nd.bib} ${nd.participant_name}${countText}`, 'judge');
                        }
                    }
                } else if (typeof sendNotification === 'function') {
                    const countText = nd.count > 0 ? ` (${nd.count} scores)` : '';
                    sendNotification('success', 'All Scores Approved', `Head judge approved all scores for #${nd.bib} ${nd.participant_name}${countText}`, 'judge');
                }
            } else if (typeof sendNotification === 'function') {
                const countText = data.count > 0 ? ` (${data.count} scores)` : '';
                sendNotification('success', 'All Scores Approved', `Head judge approved all scores for participant${countText}`, 'judge');
            }
            
            // Refresh grid
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
        } else {
            alert('Error approving scores: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error approving scores');
    });
}

// Update the showSuccessToast function to accept custom messages
function showSuccessToast(message = 'Updated successfully') {
    const toastEl = document.getElementById('successToast');
    const toastBody = toastEl.querySelector('.toast-body');
    toastBody.textContent = message;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
}

// Hold-to-approve/unapprove/delete mechanism
let approvalHoldTimer = null;
let approvalHoldStartTime = null;
const APPROVAL_HOLD_DURATION = 1500; // 1.5 seconds
const DELETE_HOLD_DURATION = 2000; // 2 seconds - longer for destructive action

// Helper function to update Edit Modal approval buttons
function updateEditModalApprovalButtons(scoreId, isApproved) {
    const modal = document.getElementById('editScoreModal');
    const modalInstance = bootstrap.Modal.getInstance(modal);
    
    // Only update if modal is currently shown
    if (!modalInstance || !modal.classList.contains('show')) {
        return;
    }
    
    const approvalButtonsContainer = document.getElementById('editModalApprovalButtons');
    const approveBtn = approvalButtonsContainer.querySelector('.approve-btn');
    const unapproveBtn = approvalButtonsContainer.querySelector('.unapprove-btn');
    
    // Verify this is the same score being edited
    const currentScoreId = approveBtn.getAttribute('data-score-id');
    if (currentScoreId != scoreId) {
        return;
    }
    
    // Show/hide appropriate button based on new approval status
    if (isApproved) {
        unapproveBtn.style.display = 'inline-block';
        approveBtn.style.display = 'none';
    } else {
        unapproveBtn.style.display = 'none';
        approveBtn.style.display = 'inline-block';
    }
}

// Event delegation for approve/unapprove/delete buttons (supporting both mouse and touch)
// Unified handler for starting hold action
function handleHoldStart(e) {
    const unapproveBtn = e.target.closest('.unapprove-btn');
    const approveBtn = e.target.closest('.approve-btn');
    const deleteBtn = e.target.closest('.delete-score-btn');
    const btn = unapproveBtn || approveBtn || deleteBtn;
    
    // Only handle and prevent default if we're actually on an approval or delete button
    if (btn) {
        // Prevent default behavior ONLY for these buttons to avoid conflicts
        e.preventDefault();
        
        const scoreId = btn.getAttribute('data-score-id') || document.getElementById('editScoreId').value;
        const progressBar = btn.querySelector('.hold-progress');
        const isApprove = btn.classList.contains('approve-btn');
        const isDelete = btn.classList.contains('delete-score-btn');
        const holdDuration = isDelete ? DELETE_HOLD_DURATION : APPROVAL_HOLD_DURATION;
        
        if (!progressBar) return;
        
        // Reset progress bar
        progressBar.style.transition = 'none';
        progressBar.style.width = '0%';
        
        // Start hold timer
        approvalHoldStartTime = Date.now();
        
        // Animate progress bar
        requestAnimationFrame(() => {
            progressBar.style.transition = `width ${holdDuration}ms linear`;
            progressBar.style.width = '100%';
        });
        
        // Set timer to trigger approve/unapprove/delete
        approvalHoldTimer = setTimeout(() => {
            // Trigger the appropriate action
            if (isDelete) {
                deleteScore(scoreId);
            } else if (isApprove) {
                approveScore(scoreId);
            } else {
                unapproveScore(scoreId);
            }
            
            // Visual feedback
            progressBar.style.width = '0%';
            progressBar.style.transition = 'width 200ms';
        }, holdDuration);
    }
}

// Unified handler for ending hold action
function handleHoldEnd(e) {
    const unapproveBtn = e.target.closest('.unapprove-btn');
    const approveBtn = e.target.closest('.approve-btn');
    const deleteBtn = e.target.closest('.delete-score-btn');
    const btn = unapproveBtn || approveBtn || deleteBtn;
    
    if (approvalHoldTimer) {
        clearTimeout(approvalHoldTimer);
        approvalHoldTimer = null;
        
        // Reset progress bar if released early
        if (btn) {
            const progressBar = btn.querySelector('.hold-progress');
            if (progressBar) {
                const elapsed = Date.now() - approvalHoldStartTime;
                const isDelete = btn.classList.contains('delete-score-btn');
                const holdDuration = isDelete ? DELETE_HOLD_DURATION : APPROVAL_HOLD_DURATION;
                if (elapsed < holdDuration) {
                    // Released too early - reset with animation
                    progressBar.style.transition = 'width 200ms';
                    progressBar.style.width = '0%';
                }
            }
        }
    }
}

// Unified handler for canceling hold action
function handleHoldCancel(e) {
    // For touchcancel events, e.target might not be available, so check first
    if (!e.target || typeof e.target.closest !== 'function') {
        // Clear timer anyway for safety
        if (approvalHoldTimer) {
            clearTimeout(approvalHoldTimer);
            approvalHoldTimer = null;
        }
        return;
    }
    
    const unapproveBtn = e.target.closest('.unapprove-btn');
    const approveBtn = e.target.closest('.approve-btn');
    const deleteBtn = e.target.closest('.delete-score-btn');
    const btn = unapproveBtn || approveBtn || deleteBtn;
    
    if (btn && approvalHoldTimer) {
        clearTimeout(approvalHoldTimer);
        approvalHoldTimer = null;
        
        // Reset progress bar
        const progressBar = btn.querySelector('.hold-progress');
        if (progressBar) {
            progressBar.style.transition = 'width 200ms';
            progressBar.style.width = '0%';
        }
    }
}

// Mouse events
document.addEventListener('mousedown', handleHoldStart);
document.addEventListener('mouseup', handleHoldEnd);
document.addEventListener('mouseleave', handleHoldCancel);

// Touch events for mobile/tablet support
// Using passive: false only on touchstart to allow preventDefault when needed
document.addEventListener('touchstart', handleHoldStart, { passive: false });
document.addEventListener('touchend', handleHoldEnd, { passive: true });
document.addEventListener('touchcancel', handleHoldCancel, { passive: true });
        
        function getStatusBadgeClass(status) {
            switch(status) {
                case 'OK': return 'success';
                case 'DNS': return 'dark-subtle';
                case 'DNF': return 'warning';
                case 'DSQ': return 'danger';
                default: return 'secondary';
            }
        }
        
        function editScore(scoreId, scoreValue, status, figuresEncoded = '', participantName = '', bibNumber = '', category = '', judgeName = '', controlPoint = '', gender = '', club = '') {
            console.log('editScore called with:', {
                scoreId,
                scoreValue,
                status,
                figuresEncoded,
                participantName,
                bibNumber
            });
            
            document.getElementById('editScoreId').value = scoreId;
            document.getElementById('editScoreValue').value = scoreValue;
            
            // Set the radio button state based on status
            const statusRadios = document.getElementsByName('editStatus');
            statusRadios.forEach(radio => {
                radio.checked = (radio.value === status);
            });
            
            // Update the participant and context information in the modal
            const eventSelect = document.getElementById('eventSelect');
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            
            const eventName = eventSelect.options[eventSelect.selectedIndex]?.text || 'Unknown Event';
            const heatNumber = heatSelect.value || 'Unknown';
            const runNumber = runSelect.value || 'Unknown';
            
            // Attempt to enrich gender/club if not passed
            if ((!gender || !club) && window.currentData && Array.isArray(window.currentData.participants)) {
                const found = window.currentData.participants.find(p => {
                    return (p.bib_number == bibNumber) ||
                           ((p.first_name + ' ' + p.last_name).toLowerCase() === participantName.toLowerCase());
                });
                if (found) {
                    if (!gender) gender = found.gender || '';
                    if (!club) club = found.club || '';
                    if (!category) category = found.category || category;
                }
            }

            const genderBadge = gender ? `<span class="badge bg-${gender==='M'?'info':'pink'} bg-opacity-20  ms-1">${gender}</span>` : '';
            const clubLine = club ? `<span class="badge text-muted">${club}</span><br>` : '';
            document.getElementById('editScoreInfo').innerHTML = `
                <div class="participantCol">
                    <span class="badge bg-white fs-6 mb-1 participant-bib-badge ss-f-number">${bibNumber}</span>
                    <strong>${participantName}</strong>
                    ${genderBadge}
                    <br><small>${clubLine}<span class="badge text-dark">${category}</span></small>
                </div>
            `;
            // Fill score context under figures section
            const scoreContextHtml = `
                <div class="row g-2">
                    <div class="col-md-6 small"><i class="fas fa-calendar-alt me-1 text-muted"></i><strong>Event:</strong> ${eventName}</div>
                    <div class="col-md-6 small"><i class="fas fa-layer-group me-1 text-muted"></i><strong>Heat:</strong> ${heatNumber} <span class="text-muted">|</span> <strong>Run:</strong> ${runNumber}</div>
                    <div class="col-md-6 small"><i class="fas fa-user-tie me-1 text-muted"></i><strong>Judge:</strong> ${judgeName}</div>
                    <div class="col-md-6 small"><i class="fas fa-map-marker-alt me-1 text-muted"></i><strong>Control Point:</strong> <span class="badge bg-success ms-1">${controlPoint}</span></div>
                    <div class="col-md-6 small"><i class="fas fa-info-circle me-1 text-muted"></i><strong>Status:</strong> <span class="badge bg-${getStatusBadgeClass(status)}">${status}</span></div>
                </div>`;
            const scoreContextEl = document.getElementById('scoreContextInfo');
            if (scoreContextEl) scoreContextEl.innerHTML = scoreContextHtml;
            
            // Store original figures data globally for editing
            window.originalFiguresData = null;
            window.currentFiguresData = null;
            
            // Populate figures if provided
            const figWrap = document.getElementById('editScoreFigures');
            const figContent = document.getElementById('editScoreFiguresContent');
            const editBtn = document.getElementById('editFiguresBtn');
            const editor = document.getElementById('editFiguresEditor');
            
            figContent.innerHTML = '';
            editor.style.display = 'block'; // Show editor immediately
            
            // Always show the figures section, even if no figures exist
            figWrap.style.display = 'block';
            editBtn.style.display = 'none'; // Hide edit button since editor is already open
            
            if (figuresEncoded) {
                try {
                    console.log('Encoded figures:', figuresEncoded);
                    const raw = decodeURIComponent(figuresEncoded);
                    console.log('Decoded figures:', raw);
                    if (raw) {
                        const figuresData = JSON.parse(raw);
                        console.log('Parsed figures data:', figuresData);
                        
                        // Handle nested structure - figures might be in figuresData.figures
                        const actualFigures = figuresData.figures || figuresData;
                        
                        window.originalFiguresData = actualFigures;
                        window.currentFiguresData = JSON.parse(JSON.stringify(actualFigures)); // Deep clone
                        
                        const parts = [];
                        Object.keys(actualFigures || {}).forEach(cat => {
                            const items = Array.isArray(actualFigures[cat]) ? actualFigures[cat] : [];
                            if (items.length) {
                                parts.push(`<span class="badge bg-success-subtle me-1 text-dark fs-6">${cat}: ${items.join(', ')}</span>`);
                            }
                        });
                        if (parts.length) {
                            figContent.innerHTML = parts.join(' ');
                        } else {
                            figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                        }
                    }
                } catch (e) { 
                    console.error('Error parsing figures in editScore:', e);
                    // If JSON parse fails, show empty state
                figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
            }
        } else {
            console.log('No figuresEncoded provided to editScore');
            // No figures data provided, show empty state
            figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
        }
        
        // Update approval buttons in modal footer
        const approvalButtonsContainer = document.getElementById('editModalApprovalButtons');
        const approveBtn = approvalButtonsContainer.querySelector('.approve-btn');
        const unapproveBtn = approvalButtonsContainer.querySelector('.unapprove-btn');
        const deleteBtn = document.querySelector('.delete-score-btn');
        
        // Set score ID on all buttons
        approveBtn.setAttribute('data-score-id', scoreId);
        unapproveBtn.setAttribute('data-score-id', scoreId);
        if (deleteBtn) {
            deleteBtn.setAttribute('data-score-id', scoreId);
        }
        
        // Check current approval status from currentData
        let isApproved = false;
        if (window.currentData && window.currentData.scores) {
            const targetRunNumber = window.currentData.run_number;
            // Find the score in currentData to check approval status
            for (const participantId in window.currentData.scores) {
                const participantScores = window.currentData.scores[participantId];
                if (targetRunNumber && participantScores[targetRunNumber]) {
                    for (const judgeId in participantScores[targetRunNumber]) {
                        const score = participantScores[targetRunNumber][judgeId];
                        if (score.score_id == scoreId) {
                            isApproved = score.is_approved == 1;
                            break;
                        }
                    }
                } else {
                    // Check all runs
                    for (const runNum in participantScores) {
                        for (const judgeId in participantScores[runNum]) {
                            const score = participantScores[runNum][judgeId];
                            if (score.score_id == scoreId) {
                                isApproved = score.is_approved == 1;
                                break;
                            }
                        }
                        if (isApproved !== false) break;
                    }
                }
                if (isApproved !== false) break;
            }
        }
        
        // Show/hide appropriate button based on approval status
        if (isApproved) {
            unapproveBtn.style.display = 'inline-block';
            approveBtn.style.display = 'none';
        } else {
            unapproveBtn.style.display = 'none';
            approveBtn.style.display = 'inline-block';
        }
        
        // Show the approval buttons container
        approvalButtonsContainer.style.display = 'block';
        
        const modal = new bootstrap.Modal(document.getElementById('editScoreModal'));
        modal.show();
        
        // Load figures editor immediately
        startEditFigures();
    }
    
        // Keypad functions for edit score modal
        function addToEditScore(digit) {
            const input = document.getElementById('editScoreValue');
            const rangeInput = document.getElementById('editScoreRange');
            let currentValue = input.value || '';
            
            // Limit to 2 digits for scores 0-99
            if (currentValue.length < 2) {
                currentValue += digit;
                input.value = currentValue;
                rangeInput.value = currentValue;
            }
        }
        
        function backspaceEditScore() {
            const input = document.getElementById('editScoreValue');
            const rangeInput = document.getElementById('editScoreRange');
            input.value = input.value.slice(0, -1);
            rangeInput.value = input.value || 0;
        }
        
        function applyEditScoreValue() {
            const input = document.getElementById('editScoreValue');
            const saveBtn = document.getElementById('saveScoreBtn');
            
            // Validate that score value is present
            if (!input.value || input.value.trim() === '') {
                input.classList.add('is-invalid');
                return;
            }
            
            input.classList.remove('is-invalid');
            
            // Scroll modal to bottom to show footer with save button
            const modal = document.getElementById('editScoreModal');
            const modalDialog = modal.querySelector('.modal-dialog');
            
            // Wait a brief moment for any animations to complete
            setTimeout(() => {
                // Scroll the modal footer into view
                const modalFooter = modal.querySelector('.modal-footer');
                if (modalFooter) {
                    modalFooter.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                }
            }, 100);
            
            // Highlight save button
            saveBtn.classList.add('pulse-highlight');
            setTimeout(() => {
                saveBtn.classList.remove('pulse-highlight');
            }, 2000);
            
            // Focus out of input to trigger any validation
            input.blur();
        }
        
        function startEditFigures() {
            const editor = document.getElementById('editFiguresEditor');
            const editBtn = document.getElementById('editFiguresBtn');
            const categoriesContainer = document.getElementById('editFigureCategories');
            
            editBtn.style.display = 'none';
            categoriesContainer.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading format options...</div>';
            editor.style.display = 'block';
            
            // Get current event ID
            const eventId = document.getElementById('eventSelect').value;
            
            if (!eventId) {
                categoriesContainer.innerHTML = '<div class="alert alert-warning">Please select an event first</div>';
                return;
            }
            
            // Fetch format figures from server
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_format_figures');
            formData.append('event_id', eventId);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                console.log('Format figures response:', data); // Debug logging
                
                if (data.success && data.figures && Object.keys(data.figures).length > 0) {
                    buildFiguresEditor(data.figures);
                } else {
                    let message = 'No figure options available for this scoring format';
                    if (data.message) {
                        message = data.message;
                    }
                    if (data.debug) {
                        console.log('Debug info:', data.debug);
                        if (data.debug.format_found === false) {
                            message = 'Event not found or has no scoring format assigned';
                        } else if (data.debug.figures_raw_count === 0) {
                            message = `No figure categories/items configured for format "${data.debug.format_details.name}"`;
                        } else if (!data.has_figures) {
                            message = `Format "${data.debug.format_details.name}" has categories but no figure items configured`;
                        }
                    }
                    categoriesContainer.innerHTML = `<div class="alert alert-info">${message}</div>`;
                }
            })
            .catch(error => {
                console.error('Error loading figures:', error);
                categoriesContainer.innerHTML = '<div class="alert alert-danger">Error loading figure options</div>';
            });
        }
        
        function buildFiguresEditor(availableFigures) {
            const categoriesContainer = document.getElementById('editFigureCategories');
            categoriesContainer.innerHTML = '';
            
            Object.keys(availableFigures).forEach(category => {
                const items = availableFigures[category] || [];
                const currentSelected = window.currentFiguresData ? (window.currentFiguresData[category] || []) : [];
                
                const categoryDiv = document.createElement('div');
                categoryDiv.className = 'figure-category mb-3';
                
                // Build chips for this category
                let chipsHtml = '';
                items.forEach(item => {
                    const isActive = currentSelected.includes(item.name) ? 'active' : '';
                    chipsHtml += `
                        <button type="button" class="btn btn-sm btn-outline-primary figure-chip ${isActive}" 
                                data-category="${category}" data-item="${item.name}" onclick="toggleFigureChip(this)">
                            ${item.name}
                        </button>
                    `;
                });
                
                categoryDiv.innerHTML = `
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h6 class="mb-0 text-primary">
                            <i class="fas fa-tag me-2"></i>${category}
                        </h6>
                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="clearFigureCategory('${category}')">
                            <i class="fas fa-times"></i> Clear
                        </button>
                    </div>
                    <div class="figure-chips d-flex flex-wrap gap-2">
                        ${chipsHtml}
                    </div>
                `;
                categoriesContainer.appendChild(categoryDiv);
            });
        }
        
        // Figure chip interaction functions
        function toggleFigureChip(button) {
            const category = button.getAttribute('data-category');
            const item = button.getAttribute('data-item');
            
            // Toggle active state
            button.classList.toggle('active');
            
            // Initialize currentFiguresData if needed
            if (!window.currentFiguresData) {
                window.currentFiguresData = {};
            }
            
            // Initialize category array if needed
            if (!window.currentFiguresData[category]) {
                window.currentFiguresData[category] = [];
            }
            
            // Add or remove item from the category
            const itemIndex = window.currentFiguresData[category].indexOf(item);
            if (button.classList.contains('active')) {
                // Add item if not already present
                if (itemIndex === -1) {
                    window.currentFiguresData[category].push(item);
                }
            } else {
                // Remove item if present
                if (itemIndex > -1) {
                    window.currentFiguresData[category].splice(itemIndex, 1);
                }
            }
            
            // Clean up empty categories
            if (window.currentFiguresData[category].length === 0) {
                delete window.currentFiguresData[category];
            }
        }
        
        function clearFigureCategory(category) {
            // Remove all active states from chips in this category
            const categoryChips = document.querySelectorAll(`.figure-chip[data-category="${category}"]`);
            categoryChips.forEach(chip => {
                chip.classList.remove('active');
            });
            
            // Remove category from data
            if (window.currentFiguresData && window.currentFiguresData[category]) {
                delete window.currentFiguresData[category];
            }
        }
        
        function cancelEditFigures() {
            const editor = document.getElementById('editFiguresEditor');
            const editBtn = document.getElementById('editFiguresBtn');
            
            editor.style.display = 'none';
            editBtn.style.display = 'inline-block';
            
            // Reset current data to original
            if (window.originalFiguresData) {
                window.currentFiguresData = JSON.parse(JSON.stringify(window.originalFiguresData));
            }
        }
        
        function saveFiguresEdit() {
            // The data is already collected in window.currentFiguresData by the chip interactions
            // No need to collect from select elements anymore since we're using chips
            
            // Check if we have any meaningful data
            const hasAnyData = window.currentFiguresData && Object.keys(window.currentFiguresData).some(category => 
                Array.isArray(window.currentFiguresData[category]) && window.currentFiguresData[category].length > 0
            );
            
            // Update display
            const figContent = document.getElementById('editScoreFiguresContent');
            if (hasAnyData) {
                const parts = [];
                Object.keys(window.currentFiguresData).forEach(cat => {
                    const items = window.currentFiguresData[cat];
                    if (items.length) {
                        parts.push(`<span class="badge bg-light text-dark me-1">${cat}: ${items.join(', ')}</span>`);
                    }
                });
                figContent.innerHTML = parts.join(' ');
            } else {
                figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                // Set to null if no meaningful data
                window.currentFiguresData = null;
            }
            
            // Hide editor
            const editor = document.getElementById('editFiguresEditor');
            const editBtn = document.getElementById('editFiguresBtn');
            editor.style.display = 'none';
            editBtn.style.display = 'inline-block';
        }
        
        function loadCreateFigures() {
            const eventId = document.getElementById('eventSelect').value;
            const figuresSection = document.getElementById('createFiguresSection');
            const figuresContainer = document.getElementById('createFiguresContainer');
            const figuresContent = document.getElementById('createScoreFiguresContent');
            const editBtn = document.getElementById('createFiguresBtn');
            const editor = document.getElementById('createFiguresEditor');
            
            if (!eventId) {
                figuresSection.style.display = 'none';
                return;
            }
            
            // Show loading
            figuresContainer.innerHTML = '<div class="text-muted"><i class="fas fa-spinner fa-spin me-2"></i>Loading figure options...</div>';
            figuresSection.style.display = 'block';
            // Initialize read-only content and controls similar to Edit modal
            if (figuresContent) figuresContent.innerHTML = '<span class="text-muted">No figures selected</span>';
            if (editor) editor.style.display = 'block'; // Show editor immediately
            if (editBtn) editBtn.style.display = 'none'; // Hide edit button since editor is already open
            
            const formData = new FormData();
            formData.append('action', 'get_format_figures');
            formData.append('event_id', eventId);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                console.log('Create figures response:', data);
                
                if (data.success && data.figures && Object.keys(data.figures).length > 0) {
                    buildCreateFiguresEditor(data.figures);
                    // Editor is already visible from initialization
                    if (figuresContent && !figuresContent.innerHTML.trim()) {
                        figuresContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                    }
                } else {
                    let message = 'No figure options available for this scoring format';
                    if (data.message) message = data.message;
                    figuresContainer.innerHTML = `<div class="alert alert-info mb-0">${message}</div>`;
                    // Hide edit button if nothing to edit
                    if (editBtn) editBtn.style.display = 'none';
                }
            })
            .catch(error => {
                console.error('Error loading create figures:', error);
                figuresContainer.innerHTML = '<div class="alert alert-danger mb-0">Error loading figure options</div>';
                if (editBtn) editBtn.style.display = 'none';
            });
        }
        
        function buildCreateFiguresEditor(availableFigures) {
            const container = document.getElementById('createFiguresContainer');
            let html = '';
            
            Object.keys(availableFigures).forEach(category => {
                const items = availableFigures[category];
                if (items && items.length > 0) {
                    // Build chips for this category
                    let chipsHtml = '';
                    items.forEach(item => {
                        chipsHtml += `
                            <button type="button" class="btn btn-sm btn-outline-primary figure-chip" 
                                    data-category="${category}" data-item="${item.name}" onclick="toggleCreateFigureChip(this)">
                                ${item.name}
                            </button>
                        `;
                    });
                    
                    html += `
                        <div class="figure-category mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <h6 class="mb-0 text-primary">
                                    <i class="fas fa-tag me-2"></i>${category}
                                </h6>
                                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="clearCreateFigureCategory('${category}')">
                                    <i class="fas fa-times"></i> Clear
                                </button>
                            </div>
                            <div class="figure-chips d-flex flex-wrap gap-2">
                                ${chipsHtml}
                            </div>
                        </div>
                    `;
                }
            });
            
            if (html) {
                container.innerHTML = html;
            } else {
                container.innerHTML = '<div class="alert alert-info mb-0">No figure categories configured for this format</div>';
            }
            
            // Initialize Lucide icons for the figure display content
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons();
            }
        }

        // Create modal figures editor controls (mirror Edit modal behavior)
        function startCreateFiguresEdit() {
            const editor = document.getElementById('createFiguresEditor');
            const editBtn = document.getElementById('createFiguresBtn');
            if (editBtn) editBtn.style.display = 'none';
            if (editor) editor.style.display = 'block';
        }
        function cancelCreateFiguresEdit() {
            const editor = document.getElementById('createFiguresEditor');
            const editBtn = document.getElementById('createFiguresBtn');
            if (editor) editor.style.display = 'none';
            if (editBtn) editBtn.style.display = 'inline-block';
        }
        function saveCreateFiguresEdit() {
            const figuresContent = document.getElementById('createScoreFiguresContent');
            const hasAnyData = window.createFiguresData && Object.keys(window.createFiguresData).some(category => 
                Array.isArray(window.createFiguresData[category]) && window.createFiguresData[category].length > 0
            );
            if (figuresContent) {
                if (hasAnyData) {
                    const parts = [];
                    Object.keys(window.createFiguresData).forEach(cat => {
                        const items = window.createFiguresData[cat];
                        if (items.length) {
                            parts.push(`<span class="badge bg-light text-dark me-1">${cat}: ${items.join(', ')}</span>`);
                        }
                    });
                    figuresContent.innerHTML = parts.join(' ');
                } else {
                    figuresContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                }
            }
            const editor = document.getElementById('createFiguresEditor');
            const editBtn = document.getElementById('createFiguresBtn');
            if (editor) editor.style.display = 'none';
            if (editBtn) editBtn.style.display = 'inline-block';
        }
        
        // Create figure chip interaction functions
        window.createFiguresData = {};
        
        function toggleCreateFigureChip(button) {
            const category = button.getAttribute('data-category');
            const item = button.getAttribute('data-item');
            
            // Toggle active state
            button.classList.toggle('active');
            
            // Initialize category array if needed
            if (!window.createFiguresData[category]) {
                window.createFiguresData[category] = [];
            }
            
            // Add or remove item from the category
            const itemIndex = window.createFiguresData[category].indexOf(item);
            if (button.classList.contains('active')) {
                // Add item if not already present
                if (itemIndex === -1) {
                    window.createFiguresData[category].push(item);
                }
            } else {
                // Remove item if present
                if (itemIndex > -1) {
                    window.createFiguresData[category].splice(itemIndex, 1);
                }
            }
            
            // Clean up empty categories
            if (window.createFiguresData[category].length === 0) {
                delete window.createFiguresData[category];
            }
        }
        
        function clearCreateFigureCategory(category) {
            // Remove all active states from chips in this category
            const categoryChips = document.querySelectorAll(`#createFiguresContainer .figure-chip[data-category="${category}"]`);
            categoryChips.forEach(chip => {
                chip.classList.remove('active');
            });
            
            // Remove category from data
            if (window.createFiguresData && window.createFiguresData[category]) {
                delete window.createFiguresData[category];
            }
        }
        
        function getCreateFiguresData() {
            // Return the data collected by the chip interactions
            // Check if we have any meaningful data
            const hasAnyData = window.createFiguresData && Object.keys(window.createFiguresData).some(category => 
                Array.isArray(window.createFiguresData[category]) && window.createFiguresData[category].length > 0
            );
            
            // Return null if no meaningful data, otherwise return the data
            return hasAnyData ? window.createFiguresData : null;
        }
        
        function clearScoreGrid() {
            const container = document.getElementById('scoresContainer');
            const sortControls = document.getElementById('sortControls');
            const noDataMessage = document.getElementById('noDataMessage');
            if (container) container.style.display = 'none';
            if (sortControls) sortControls.classList.add('d-none');
            if (noDataMessage) {
                noDataMessage.style.display = 'block';
                noDataMessage.innerHTML = `
                    <i class="fas fa-clipboard-list fa-3x mb-3 opacity-50"></i>
                    <h5>Select Event, Heat, and Run to view scores</h5>
                    <p>Use the controls above to navigate to specific scoring sessions</p>
                `;
            }
            currentData = null;
            updateSummaryMetrics([], []);
            updateHeatQueuePanel([], {});
            const sessionLabel = document.getElementById('toolbarSessionInfo');
            if (sessionLabel) sessionLabel.textContent = 'No selection';
        }
        
        function showLoading(show) {
            const spinner = document.getElementById('loadingSpinner');
            spinner.classList.toggle('d-none', !show);
        }
        
        // Scroll position preservation functions
        function preserveScrollPosition() {
            const tableContainer = document.querySelector('#scoresContainer .table-responsive');
            if (tableContainer) {
                window.lastScrollTop = tableContainer.scrollTop;
                window.lastScrollLeft = tableContainer.scrollLeft;
            }
        }
        
        function restoreScrollPosition() {
            const tableContainer = document.querySelector('#scoresContainer .table-responsive');
            if (tableContainer && window.lastScrollTop !== undefined) {
                // Use requestAnimationFrame for smoother restoration
                requestAnimationFrame(() => {
                    tableContainer.scrollTop = window.lastScrollTop;
                    tableContainer.scrollLeft = window.lastScrollLeft || 0;
                    
                    // Double-check restoration after a short delay
                    setTimeout(() => {
                        if (tableContainer.scrollTop !== window.lastScrollTop) {
                            tableContainer.scrollTop = window.lastScrollTop;
                            tableContainer.scrollLeft = window.lastScrollLeft || 0;
                        }
                    }, 50);
                });
            }
        }
        
        function updateSessionInfo() {
            const eventSelect = document.getElementById('eventSelect');
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            const label = document.getElementById('toolbarSessionInfo');
            if (!label) return;
            const eventName = eventSelect && eventSelect.selectedIndex > -1 ? eventSelect.options[eventSelect.selectedIndex].text : '';
            const heatValue = heatSelect ? heatSelect.value : '';
            const runValue = runSelect ? runSelect.value : '';
            if (!eventName) {
                label.textContent = 'No selection';
                return;
            }
            const sessionParts = [eventName];
            sessionParts.push(`Heat ${heatValue || '-'}`);
            sessionParts.push(`Run ${runValue || '-'}`);
            label.textContent = sessionParts.join(' • ');
        }

        function updateSummaryMetrics(processed = [], filtered = []) {
            const toApprove = processed.filter(p => p.approvalStatus === 'to_approve').length;
            const waiting = processed.filter(p => p.approvalStatus === 'waiting').length;
            const metricToApprove = document.getElementById('metricToApprove');
            const metricPendingJudges = document.getElementById('metricPendingJudges');
            const metricLastUpdate = document.getElementById('metricLastUpdate');
            if (metricToApprove) metricToApprove.textContent = toApprove;
            if (metricPendingJudges) metricPendingJudges.textContent = waiting;
            if (metricLastUpdate) {
                const now = new Date();
                metricLastUpdate.textContent = processed.length ? now.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }) : '--:--';
            }
        }

        function updateHeatQueuePanel(queue = [], heatSettings = {}) {
            const subtitleEl = document.getElementById('queueSummarySubtitle');
            if (!subtitleEl) {
                return;
            }

            const totalEl = document.getElementById('queueTotalCount');
            const performingEl = document.getElementById('queuePerformingCount');
            const waitingEl = document.getElementById('queueWaitingCount');
            const completedEl = document.getElementById('queueCompletedCount');
            const runLabelEl = document.getElementById('queueRunLabel');
            const updatedAtEl = document.getElementById('queueUpdatedAt');

            const safeQueue = Array.isArray(queue) ? queue.slice() : [];
            const resolvedHeatSettings = Object.keys(heatSettings || {}).length ? heatSettings : (currentHeatSettings || {});
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            const fallbackHeat = heatSelect && heatSelect.value ? heatSelect.value : '--';
            const fallbackRun = runSelect && runSelect.value ? runSelect.value : '--';
            const heatNumber = resolvedHeatSettings.heat_number || fallbackHeat;
            const heatStatus = (resolvedHeatSettings.status || (safeQueue.length ? 'active' : 'pending')).replace(/_/g, ' ');

            subtitleEl.textContent = safeQueue.length
                ? `Heat ${heatNumber} · ${heatStatus.toUpperCase()}`
                : (heatNumber && heatNumber !== '--'
                    ? `Heat ${heatNumber} · waiting to start`
                    : 'Waiting for participants…');

            if (totalEl) totalEl.textContent = safeQueue.length ? safeQueue.length : '--';

            const performingParticipants = safeQueue.filter(isParticipantPerforming);
            const completedParticipants = safeQueue.filter(isParticipantCompleted);
            const nextParticipants = getNextParticipants(safeQueue);
            const waitingParticipants = safeQueue.filter(participant =>
                !performingParticipants.includes(participant) &&
                !completedParticipants.includes(participant)
            );

            if (performingEl) performingEl.textContent = performingParticipants.length;
            if (waitingEl) waitingEl.textContent = waitingParticipants.length;
            if (completedEl) completedEl.textContent = completedParticipants.length;

            const activeRun = Number(resolvedHeatSettings?.active_run) || Number(safeQueue?.[0]?.current_run) || Number(fallbackRun) || 1;
            const totalRuns = Number(resolvedHeatSettings?.runs_count) || Number(safeQueue?.[0]?.total_runs) || activeRun;
            if (runLabelEl) runLabelEl.textContent = `${activeRun}/${Math.max(activeRun, totalRuns)}`;
            if (updatedAtEl) updatedAtEl.textContent = safeQueue.length
                ? new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })
                : '--';

            renderQueueSlots(performingParticipants, nextParticipants);
            renderQueueChips(waitingParticipants);
        }

        function isParticipantPerforming(participant) {
            const queueStatus = (participant?.status || '').toLowerCase();
            const participantStatus = (participant?.participant_status || '').toLowerCase();
            return queueStatus === 'current' || participantStatus === 'performing';
        }

        function isParticipantCompleted(participant) {
            const queueStatus = (participant?.status || '').toLowerCase();
            const participantStatus = (participant?.participant_status || '').toLowerCase();
            return queueStatus === 'completed' || queueStatus === 'done' || participantStatus === 'complete' || participant.queue_position === -1;
        }

        function getNextParticipants(queue = []) {
            return queue
                .filter(participant => {
                    const queueStatus = (participant?.status || '').toLowerCase();
                    return queueStatus === 'next' || participant.queue_position === 1;
                })
                .slice(0, 2);
        }

        function getQueueParticipantKey(participant = {}, fallback = '') {
            const key =
                (participant?.participant_id ??
                participant?.event_participant_id ??
                participant?.id ??
                (participant?.bib_number ? `bib-${participant.bib_number}` : '') ??
                (typeof participant?.queue_position !== 'undefined' ? `pos-${participant.queue_position}` : '') ??
                fallback);
            if (key === null || typeof key === 'undefined') {
                return String(fallback || 'unknown');
            }
            return String(key);
        }

        function buildQueueStateKey(list = []) {
            if (!Array.isArray(list) || !list.length) {
                return 'empty';
            }
            return list.map(participant => {
                const key = getQueueParticipantKey(participant, 'unknown');
                const status = (participant?.status || participant?.participant_status || '').toLowerCase();
                const run = participant?.current_run || participant?.active_run || '';
                return `${key}:${status}:${run}`;
            }).join('|');
        }

        function captureQueueFlipSnapshot(container, selector) {
            if (!container) {
                return null;
            }
            const snapshot = { height: container.offsetHeight };
            const elements = selector ? container.querySelectorAll(selector) : container.children;
            elements.forEach((el, index) => {
                const key = el.dataset?.flipKey || el.dataset?.bib || el.dataset?.queueIndex || `${selector || 'child'}-${index}`;
                if (!key) {
                    return;
                }
                snapshot[key] = el.getBoundingClientRect();
            });
            return snapshot;
        }

        function playQueueFlipAnimation(container, prevSnapshot, selector) {
            const canAnimate = typeof Element !== 'undefined' && typeof Element.prototype.animate === 'function';
            if (!container || !prevSnapshot || !canAnimate) {
                return;
            }
            if (prevSnapshot.height) {
                container.style.minHeight = `${prevSnapshot.height}px`;
            }
            const elements = selector ? container.querySelectorAll(selector) : container.children;
            elements.forEach((el, index) => {
                const key = el.dataset?.flipKey || el.dataset?.bib || el.dataset?.queueIndex || `${selector || 'child'}-${index}`;
                const prevRect = key ? prevSnapshot[key] : null;
                if (!prevRect) {
                    return;
                }
                const newRect = el.getBoundingClientRect();
                const deltaX = prevRect.left - newRect.left;
                const deltaY = prevRect.top - newRect.top;
                const scaleX = newRect.width ? prevRect.width / newRect.width : 1;
                const scaleY = newRect.height ? prevRect.height / newRect.height : 1;
                el.animate([
                    {
                        transform: `translate(${deltaX}px, ${deltaY}px) scale(${scaleX}, ${scaleY})`,
                        opacity: 0.75
                    },
                    {
                        transform: 'translate(0, 0) scale(1, 1)',
                        opacity: 1
                    }
                ], {
                    duration: 280,
                    easing: 'cubic-bezier(0.22, 1, 0.36, 1)'
                });
            });
            requestAnimationFrame(() => {
                container.style.minHeight = '';
            });
        }

        function renderQueueSlots(performing = [], next = []) {
            const container = document.getElementById('queuePerformingSlots');
            if (!container) return;

            ensureQueueSlotStructure(container);
            const newStateKey = `${buildQueueStateKey(performing)}|${buildQueueStateKey(next)}`;
            if (container.dataset.stateKey === newStateKey) {
                return;
            }

            const prevSnapshot = captureQueueFlipSnapshot(container, '.slot-chip');
            const performingContent = container.querySelector('.performing-slot.current .slot-content');
            const nextContent = container.querySelector('.performing-slot.next .slot-content');

            syncSlotParticipants(performingContent, performing, 'No athlete on course');
            syncSlotParticipants(nextContent, next, 'Queue forming soon');

            container.dataset.stateKey = newStateKey;
            requestAnimationFrame(() => {
                playQueueFlipAnimation(container, prevSnapshot, '.slot-chip');
            });
        }

        function ensureQueueSlotStructure(container) {
            const hasCurrent = container.querySelector('.performing-slot.current');
            const hasNext = container.querySelector('.performing-slot.next');
            if (hasCurrent && hasNext) {
                return;
            }
            container.innerHTML = `
                <div class="performing-slot current">
                    <span class="slot-label">Now Performing</span>
                    <div class="slot-content text-muted small" data-slot-placeholder="true">Waiting for heat data…</div>
                </div>
                <div class="performing-slot next">
                    <span class="slot-label">Up Next</span>
                    <div class="slot-content text-muted small" data-slot-placeholder="true">Queue will appear here.</div>
                </div>
            `;
        }

        function syncSlotParticipants(slotContentEl, list = [], emptyText = '') {
            if (!slotContentEl) return;

            const trimmed = Array.isArray(list) ? list.slice(0, 2) : [];
            if (!trimmed.length) {
                slotContentEl.querySelectorAll('.slot-chip').forEach(chip => chip.remove());
                let placeholder = slotContentEl.querySelector('[data-slot-placeholder]');
                if (!placeholder) {
                    placeholder = document.createElement('span');
                    placeholder.className = 'text-muted small';
                    placeholder.dataset.slotPlaceholder = 'true';
                    slotContentEl.appendChild(placeholder);
                }
                placeholder.textContent = emptyText;
                return;
            }

            const placeholder = slotContentEl.querySelector('[data-slot-placeholder]');
            if (placeholder) placeholder.remove();

            const existingMap = new Map();
            slotContentEl.querySelectorAll('.slot-chip').forEach(chip => {
                const key = chip.dataset.flipKey || chip.dataset.bib || chip.textContent.trim();
                if (key) {
                    existingMap.set(key, chip);
                }
            });

            trimmed.forEach((participant, index) => {
                const key = getQueueParticipantKey(participant, `slot-${index}`);
                let chip = existingMap.get(key);
                if (!chip) {
                    chip = document.createElement('span');
                    chip.className = 'slot-chip';
                } else {
                    existingMap.delete(key);
                }
                chip.dataset.flipKey = key;
                chip.dataset.bib = participant?.bib_number || '';
                const bib = participant?.bib_number ? `#${escapeHtml(participant.bib_number)}` : '';
                chip.innerHTML = `
                    <span class="slot-bib">${bib}</span>
                    <span>${formatQueueParticipantName(participant)}</span>
                `;
                const referenceNode = slotContentEl.children[index];
                if (referenceNode !== chip) {
                    slotContentEl.insertBefore(chip, referenceNode || null);
                }
            });

            existingMap.forEach(chip => chip.remove());
        }

        function renderQueueChips(participants = []) {
            const container = document.getElementById('queueChipsList');
            if (!container) return;

            const trimmed = Array.isArray(participants) ? participants.slice(0, 8) : [];

            if (!trimmed.length) {
                if (container.dataset.stateKey !== 'empty') {
                    clearQueueChipElements(container);
                    container.dataset.stateKey = 'empty';
                }
                ensureQueuePlaceholder(container, 'All caught up – no one waiting.');
                return;
            }

            removeQueuePlaceholder(container);

            const newStateKey = buildQueueStateKey(trimmed);
            if (container.dataset.stateKey === newStateKey) {
                return;
            }

            const prevSnapshot = captureQueueFlipSnapshot(container, '.queue-chip');
            const existingChips = new Map();
            container.querySelectorAll('.queue-chip').forEach(chip => {
                const key = chip.dataset.queueKey || chip.dataset.flipKey || chip.dataset.bib || chip.dataset.queueIndex;
                if (key) {
                    existingChips.set(key, chip);
                }
            });

            trimmed.forEach((participant, index) => {
                const key = getQueueParticipantKey(participant, `queue-${index}`);
                let chip = existingChips.get(key);
                if (!chip) {
                    chip = createQueueChipElement();
                    chip.dataset.queueKey = key;
                    chip.style.opacity = '0';
                    chip.style.transform = 'scale(0.95)';
                } else {
                    existingChips.delete(key);
                }

                updateQueueChipElement(chip, participant, index, key);
                insertQueueChipAtIndex(container, chip, index);

                requestAnimationFrame(() => {
                    chip.style.transition = 'opacity 0.25s ease, transform 0.25s ease';
                    chip.style.opacity = '1';
                    chip.style.transform = 'scale(1)';
                });
            });

            existingChips.forEach(chip => fadeAndRemoveQueueChip(chip));

            container.dataset.stateKey = newStateKey;
            requestAnimationFrame(() => {
                playQueueFlipAnimation(container, prevSnapshot, '.queue-chip');
            });
        }

        function createQueueChipElement() {
            const chip = document.createElement('div');
            chip.className = 'queue-chip';

            const rank = document.createElement('div');
            rank.className = 'queue-chip-rank';
            chip.appendChild(rank);

            const body = document.createElement('div');
            body.className = 'queue-chip-body';
            chip.appendChild(body);

            const name = document.createElement('div');
            name.className = 'queue-chip-name';
            body.appendChild(name);

            const meta = document.createElement('div');
            meta.className = 'queue-chip-meta';
            body.appendChild(meta);

            return chip;
        }

        function updateQueueChipElement(chip, participant, index, flipKey) {
            if (!chip) return;

            const queueStatus = (participant?.status || 'waiting').toLowerCase();
            let statusClass = '';
            if (queueStatus === 'current' || queueStatus === 'performing') statusClass = 'queue-chip-current';
            else if (queueStatus === 'next') statusClass = 'queue-chip-next';
            else if (queueStatus === 'completed' || queueStatus === 'done') statusClass = 'queue-chip-completed';

            chip.dataset.status = queueStatus;
            chip.dataset.bib = participant?.bib_number || '';
            chip.dataset.queueIndex = String(index);
            chip.dataset.positionIndex = String(participant?.queue_position ?? index);
            chip.dataset.participantName = participant?.participant_name || participant?.name || '';
            chip.dataset.flipKey = flipKey || chip.dataset.queueKey || chip.dataset.flipKey || '';
            chip.className = `queue-chip ${statusClass}`.trim();

            const name = formatQueueParticipantName(participant);
            const category = escapeHtml(participant?.display_category || participant?.event_category || '');
            const bib = participant?.bib_number ? `BIB ${escapeHtml(participant.bib_number)}` : '';

            const rankEl = chip.querySelector('.queue-chip-rank');
            if (rankEl) {
                rankEl.textContent = index + 1;
            }

            const nameEl = chip.querySelector('.queue-chip-name');
            if (nameEl && nameEl.textContent !== name) {
                nameEl.textContent = name;
            }

            const metaEl = chip.querySelector('.queue-chip-meta');
            if (metaEl) {
                const metaContent = `${bib ? `<span class="badge bg-light text-dark me-1">${bib}</span>` : ''}${category}`;
                if (metaEl.dataset.metaValue !== metaContent) {
                    metaEl.innerHTML = metaContent;
                    metaEl.dataset.metaValue = metaContent;
                }
            }
        }

        function insertQueueChipAtIndex(container, chip, index) {
            const chips = Array.from(container.querySelectorAll('.queue-chip')).filter(existing => existing !== chip);
            const referenceNode = chips[index] || null;
            if (referenceNode) {
                container.insertBefore(chip, referenceNode);
            } else {
                container.appendChild(chip);
            }
        }

        function fadeAndRemoveQueueChip(chip) {
            if (!chip) return;
            chip.style.transition = 'opacity 0.2s ease, transform 0.2s ease';
            chip.style.opacity = '0';
            chip.style.transform = 'scale(0.95)';
            setTimeout(() => {
                if (chip.parentNode) {
                    chip.parentNode.removeChild(chip);
                }
            }, 200);
        }

        function ensureQueuePlaceholder(container, message) {
            if (!container) return;
            let placeholder = container.querySelector('[data-queue-placeholder]');
            if (!placeholder) {
                placeholder = document.createElement('div');
                placeholder.className = 'text-muted small';
                placeholder.dataset.queuePlaceholder = 'true';
                container.appendChild(placeholder);
            }
            placeholder.textContent = message;
        }

        function removeQueuePlaceholder(container) {
            if (!container) return;
            const placeholder = container.querySelector('[data-queue-placeholder]');
            if (placeholder) {
                placeholder.remove();
            }
        }

        function clearQueueChipElements(container) {
            if (!container) return;
            container.querySelectorAll('.queue-chip').forEach(chip => chip.remove());
        }

        function formatQueueParticipantName(participant) {
            const first = participant?.display_first_name || participant?.first_name || '';
            const last = participant?.display_last_name || participant?.last_name || '';
            const combined = `${first} ${last}`.trim();
            if (combined) return escapeHtml(combined);
            if (participant?.participant_name) return escapeHtml(participant.participant_name);
            if (participant?.name) return escapeHtml(participant.name);
            if (participant?.bib_number) return `BIB ${escapeHtml(participant.bib_number)}`;
            return 'Unknown Athlete';
        }

        function resetHeadJudgeQueuePanel(message = 'Waiting for participants…') {
            const subtitleEl = document.getElementById('queueSummarySubtitle');
            if (subtitleEl) subtitleEl.textContent = message;
            const totalEl = document.getElementById('queueTotalCount');
            if (totalEl) totalEl.textContent = '--';
            ['queuePerformingCount', 'queueWaitingCount', 'queueCompletedCount'].forEach(id => {
                const el = document.getElementById(id);
                if (el) el.textContent = '0';
            });
            const runLabelEl = document.getElementById('queueRunLabel');
            if (runLabelEl) runLabelEl.textContent = '--';
            const updatedAtEl = document.getElementById('queueUpdatedAt');
            if (updatedAtEl) updatedAtEl.textContent = '--';
            const slots = document.getElementById('queuePerformingSlots');
            if (slots) {
                slots.innerHTML = `
                    <div class="performing-slot current">
                        <span class="slot-label">Now Performing</span>
                        <div class="slot-content text-muted small" data-slot-placeholder="true">Waiting for heat data…</div>
                    </div>
                    <div class="performing-slot next">
                        <span class="slot-label">Up Next</span>
                        <div class="slot-content text-muted small" data-slot-placeholder="true">Queue will appear here.</div>
                    </div>
                `;
            }
            const chips = document.getElementById('queueChipsList');
            if (chips) {
                chips.innerHTML = '<div class="text-muted small" data-queue-placeholder="true">Queue will appear once participants load.</div>';
            }
        }

        function getSelectedEventId() {
            const select = document.getElementById('eventSelect');
            return select ? select.value : '';
        }

        async function refreshHeadJudgeQueuePanel() {
            const eventId = getSelectedEventId();
            const heatSelect = document.getElementById('heatSelect');
            const heatNumber = heatSelect ? heatSelect.value : '';

            if (!eventId) {
                resetHeadJudgeQueuePanel('Select an event to monitor queue');
                return;
            }
            if (!heatNumber) {
                resetHeadJudgeQueuePanel('Select a heat to load queue');
                return;
            }
            if (headJudgeQueueLoading) {
                return;
            }

            headJudgeQueueLoading = true;
            try {
                const response = await fetch(`../admin/heat_management_api.php?action=get_queue&event_id=${eventId}&heat_number=${heatNumber}&mode=sort_order`, {
                    credentials: 'same-origin'
                });
                if (!response.ok) {
                    throw new Error('Failed to query queue API');
                }
                const data = await response.json();
                const queue = data.queue || [];
                currentHeatSettings = data.heat_settings || {
                    heat_number: heatNumber,
                    active_run: data.active_run || currentHeatSettings.active_run || document.getElementById('runSelect')?.value || 1,
                    runs_count: data.heat_settings?.runs_count || currentHeatSettings.runs_count
                };
                window.currentHeatSettings = currentHeatSettings;
                updateHeatQueuePanel(queue, currentHeatSettings);
            } catch (error) {
                console.error('Queue load failed:', error);
                const subtitleEl = document.getElementById('queueSummarySubtitle');
                if (subtitleEl) subtitleEl.textContent = 'Queue unavailable';
                const chips = document.getElementById('queueChipsList');
                if (chips) {
                    chips.innerHTML = `<div class="alert-placeholder text-danger small"><i class="fas fa-exclamation-triangle me-1"></i>${escapeHtml(error.message)}</div>`;
                }
                const updatedAtEl = document.getElementById('queueUpdatedAt');
                if (updatedAtEl) updatedAtEl.textContent = '--';
            } finally {
                headJudgeQueueLoading = false;
            }
        }

        function initializeHeadJudgeNotifications(forceRestart = false) {
            const alertsList = document.getElementById('alertsList');
            if (!alertsList) {
                return;
            }
            const eventId = getSelectedEventId();
            if (!eventId) {
                resetHeadJudgeNotificationsUI();
                return;
            }
            if (heatNotificationsInitialized && !forceRestart) {
                return;
            }
            heatNotificationsInitialized = true;
            latestNotificationId = 0;
            heatNotifications = [];
            refreshHeadJudgeAlerts(true);
            if (heatNotificationsTimer) {
                clearInterval(heatNotificationsTimer);
            }
            heatNotificationsTimer = setInterval(() => {
                if (!document.hidden) {
                    refreshHeadJudgeAlerts();
                }
            }, HEAD_JUDGE_ALERT_POLL_INTERVAL);
        }

        function resetHeadJudgeNotificationsUI(message = 'Select an event to monitor alerts') {
            if (heatNotificationsTimer) {
                clearInterval(heatNotificationsTimer);
                heatNotificationsTimer = null;
            }
            heatNotificationsInitialized = false;
            heatNotifications = [];
            latestNotificationId = 0;
            const alertsList = document.getElementById('alertsList');
            if (alertsList) {
                alertsList.innerHTML = `<div class="text-muted small text-center py-3">${escapeHtml(message)}</div>`;
            }
            const statusLabel = document.getElementById('alertsStatusLabel');
            if (statusLabel) {
                statusLabel.textContent = message;
                statusLabel.classList.remove('text-danger');
            }
            const countBadge = document.getElementById('alertsCountBadge');
            if (countBadge) {
                countBadge.textContent = '0';
            }
        }

        async function refreshHeadJudgeAlerts(forceFullReload = false) {
            const alertsList = document.getElementById('alertsList');
            const eventId = getSelectedEventId();
            if (!alertsList) {
                return;
            }
            if (!eventId) {
                resetHeadJudgeNotificationsUI();
                return;
            }
            if (notificationsLoading && !forceFullReload) {
                return;
            }

            const statusLabel = document.getElementById('alertsStatusLabel');
            if (statusLabel) {
                statusLabel.textContent = 'Updating…';
                statusLabel.classList.remove('text-danger');
            }

            notificationsLoading = true;
            try {
                const params = new URLSearchParams({
                    action: 'get_notifications',
                    event_id: eventId,
                    category: 'heat',
                    limit: '8'
                });
                if (forceFullReload || !latestNotificationId) {
                    params.append('since', '0');
                } else {
                    params.append('since', String(latestNotificationId));
                }

                const response = await fetch(`../api/notification_api.php?${params.toString()}`, {
                    credentials: 'same-origin'
                });
                const data = await response.json();
                if (!data.success) {
                    throw new Error(data.error || 'Failed to load alerts');
                }

                latestNotificationId = Math.max(latestNotificationId, data.latest_id || 0);
                mergeHeatNotifications(data.notifications || [], forceFullReload);
                renderHeatNotificationsList();

                if (statusLabel) {
                    statusLabel.textContent = `Updated ${new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}`;
                }
            } catch (error) {
                console.error('Heat alerts error:', error);
                if (statusLabel) {
                    statusLabel.textContent = 'Alerts unavailable';
                    statusLabel.classList.add('text-danger');
                }
                if (!heatNotifications.length) {
                    alertsList.innerHTML = `<div class="alert-placeholder text-danger small"><i class="fas fa-exclamation-triangle me-1"></i>${escapeHtml(error.message)}</div>`;
                }
            } finally {
                notificationsLoading = false;
            }
        }

        function mergeHeatNotifications(newNotifications = [], reset = false) {
            if (reset) {
                heatNotifications = (newNotifications || []).slice().sort((a, b) => (Number(b?.id) || 0) - (Number(a?.id) || 0));
                return;
            }

            const existingIds = new Set(heatNotifications.map(notification => Number(notification?.id)));
            (newNotifications || []).forEach(notification => {
                const id = Number(notification?.id);
                if (!existingIds.has(id)) {
                    heatNotifications.unshift(notification);
                    existingIds.add(id);
                }
            });

            heatNotifications.sort((a, b) => (Number(b?.id) || 0) - (Number(a?.id) || 0));
            heatNotifications = heatNotifications.slice(0, 8);
        }

        function renderHeatNotificationsList() {
            const alertsList = document.getElementById('alertsList');
            const countBadge = document.getElementById('alertsCountBadge');
            if (!alertsList) {
                return;
            }

            if (!heatNotifications.length) {
                alertsList.innerHTML = '<div class="text-muted small text-center py-3">No alerts yet for this heat.</div>';
                if (countBadge) {
                    countBadge.textContent = '0';
                }
                return;
            }

            alertsList.innerHTML = heatNotifications.map(buildNotificationHtml).join('');
            if (countBadge) {
                countBadge.textContent = heatNotifications.length;
            }
        }

        function buildNotificationHtml(notification) {
            const type = (notification?.type || 'info').toLowerCase();
            const iconClass = notificationTypeToClass(type);
            const icon = notificationTypeToIcon(type);
            const title = escapeHtml(notification?.title || 'System Alert');
            const message = escapeHtml(notification?.message || '');
            const category = escapeHtml(notification?.category || 'general');
            const timeAgo = formatRelativeTime(notification?.created_at || notification?.timestamp);
            const notificationId = escapeHtml(notification?.id || '');
            const badgeVariant = type === 'danger' ? 'danger' : type === 'warning' ? 'warning' : type === 'success' ? 'success' : 'info';

            return `
                <div class="alert-item" data-notification-id="${notificationId}">
                    <div class="alert-icon ${iconClass}">
                        <i class="${icon}"></i>
                    </div>
                    <div class="alert-body">
                        <div class="d-flex justify-content-between gap-3 align-items-start">
                            <div>
                                <div class="alert-title">${title}</div>
                                ${message ? `<p class="alert-message mb-1">${message}</p>` : ''}
                            </div>
                            <div class="text-end d-flex flex-column align-items-end gap-1">
                                <span class="alert-time">${timeAgo}</span>
                                <button type="button" class="btn btn-link btn-sm p-0 text-decoration-none text-muted" title="Dismiss" onclick="dismissHeatNotification('${notificationId}', this)">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                        </div>
                        <div class="alert-meta d-flex gap-2 align-items-center flex-wrap mt-2">
                            <span class="badge badge-soft-${badgeVariant}">${category}</span>
                            ${notification?.user_id ? `<span class="text-muted small">${escapeHtml(notification.user_id)}</span>` : ''}
                        </div>
                    </div>
                </div>
            `;
        }

        function notificationTypeToClass(type) {
            switch (type) {
                case 'danger':
                case 'error':
                    return 'alert-type-danger';
                case 'warning':
                    return 'alert-type-warning';
                case 'success':
                    return 'alert-type-success';
                default:
                    return 'alert-type-info';
            }
        }

        function notificationTypeToIcon(type) {
            switch (type) {
                case 'danger':
                case 'error':
                    return 'fas fa-exclamation-circle';
                case 'warning':
                    return 'fas fa-exclamation-triangle';
                case 'success':
                    return 'fas fa-check-circle';
                default:
                    return 'fas fa-info-circle';
            }
        }

        function formatRelativeTime(timestamp) {
            if (!timestamp) return '--';
            const parsed = new Date(timestamp).getTime();
            if (Number.isNaN(parsed)) {
                return '--';
            }
            const diffMs = Date.now() - parsed;
            const diffSeconds = Math.floor(diffMs / 1000);
            if (diffSeconds < 60) return `${diffSeconds}s ago`;
            const diffMinutes = Math.floor(diffSeconds / 60);
            if (diffMinutes < 60) return `${diffMinutes}m ago`;
            const diffHours = Math.floor(diffMinutes / 60);
            if (diffHours < 24) return `${diffHours}h ago`;
            const diffDays = Math.floor(diffHours / 24);
            return `${diffDays}d ago`;
        }

        async function dismissHeatNotification(notificationId, triggerButton) {
            if (!notificationId) {
                return;
            }
            try {
                if (triggerButton) {
                    triggerButton.disabled = true;
                }
                const response = await fetch('../api/notification_api.php?action=dismiss_notification', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    body: new URLSearchParams({ notification_id: notificationId }),
                    credentials: 'same-origin'
                });
                const data = await response.json();
                if (!data.success) {
                    throw new Error(data.error || 'Failed to dismiss notification');
                }
                heatNotifications = heatNotifications.filter(notification => String(notification?.id) !== String(notificationId));
                renderHeatNotificationsList();
            } catch (error) {
                console.error('Dismiss notification failed:', error);
            } finally {
                if (triggerButton) {
                    triggerButton.disabled = false;
                }
            }
        }

        document.addEventListener('visibilitychange', () => {
            if (!document.hidden && heatNotificationsInitialized) {
                refreshHeadJudgeAlerts();
            }
        });

        function approveAllVisibleParticipants() {
            if (!window.latestFilteredParticipants || !window.latestFilteredParticipants.length) {
                alert('No participants loaded');
                return;
            }
            const actionable = window.latestFilteredParticipants.filter(p => p.approvalStatus === 'to_approve');
            if (!actionable.length) {
                showSuccessToast('No pending approvals in view');
                return;
            }
            if (!confirm(`Approve scores for ${actionable.length} participant(s)?`)) {
                return;
            }
            actionable.forEach(participant => {
                approveAllParticipant(participant.id);
            });
        }
        
        function startAutoRefresh() {
            stopAutoRefresh();
            autoRefreshInterval = setInterval(() => {
                const eventId = document.getElementById('eventSelect').value;
                if (eventId) {
                    loadScores();
                }
            }, 5000);
        }
        
        function stopAutoRefresh() {
            if (autoRefreshInterval) {
                clearInterval(autoRefreshInterval);
                autoRefreshInterval = null;
            }
        }
        
        // Handle score update
        document.getElementById('saveScoreBtn').addEventListener('click', function() {
            const scoreId = document.getElementById('editScoreId').value;
            const scoreValue = document.getElementById('editScoreValue').value;
            
            // Validate score value is present
            if (!scoreValue || scoreValue.trim() === '') {
                const input = document.getElementById('editScoreValue');
                input.classList.add('is-invalid');
                input.focus();
                alert('Please enter a score value');
                return;
            }
            
            // Get status from radio buttons
            const statusRadio = document.querySelector('input[name="editStatus"]:checked');
            const status = statusRadio ? statusRadio.value : 'OK';
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'update_score');
            formData.append('score_id', scoreId);
            formData.append('score_value', scoreValue);
            formData.append('status', status);
            
            // Include figures data - send null for empty data
            if (window.currentFiguresData !== undefined) {
                // Send 'null' string for null values, JSON string for actual data
                const figuresValue = window.currentFiguresData === null ? 'null' : JSON.stringify(window.currentFiguresData);
                formData.append('figures_json', figuresValue);
            }
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const modal = bootstrap.Modal.getInstance(document.getElementById('editScoreModal'));
                    modal.hide();
                    
                    showSuccessToast();
                    
                    // Refresh grid
                    if (document.getElementById('runSelect').value) {
                        loadScores();
                    }
                } else {
                    alert('Error updating score: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error updating score');
            });
        });

        // Handle score creation
        // Handle score creation
document.getElementById('createScoreBtn').addEventListener('click', function() {
    const participantId = document.getElementById('createParticipantId').value;
    const judgeId = document.getElementById('createJudgeId').value;
    const scoreValue = document.getElementById('createScoreValue').value;
    
    // Validate score value is present
    if (!scoreValue || scoreValue.trim() === '') {
        const input = document.getElementById('createScoreValue');
        input.classList.add('is-invalid');
        input.focus();
        alert('Please enter a score value');
        return;
    }
    
    // Get status from radio buttons
    const statusRadio = document.querySelector('input[name="createStatus"]:checked');
    const status = statusRadio ? statusRadio.value : 'OK';
    
    const eventId = document.getElementById('eventSelect').value;
    const heatNumber = document.getElementById('heatSelect').value;
    const runNumber = document.getElementById('runSelect').value;
    
    // Get figures data
    const figuresData = getCreateFiguresData();
    
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'create_score');
    formData.append('event_id', eventId);
    formData.append('heat_number', heatNumber);
    formData.append('run_number', runNumber);
    formData.append('participant_id', participantId);
    formData.append('judge_id', judgeId);
    formData.append('score_value', scoreValue);
    formData.append('status', status);
    
    // Add figures data if available
    if (figuresData) {
        formData.append('figures_json', JSON.stringify(figuresData));
    }
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const modal = bootstrap.Modal.getInstance(document.getElementById('createScoreModal'));
            modal.hide();
            
            showSuccessToast('Score created successfully');
            
            // Refresh grid
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
        } else {
            alert('Error creating score: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error creating score');
    });
});

// NEW: Handle Enter key press in create score form
document.getElementById('createScoreForm').addEventListener('submit', function(e) {
    e.preventDefault(); // Prevent default form submission
    document.getElementById('createScoreBtn').click(); // Trigger the create button
});

// NEW: Handle Enter key press in score value input
document.getElementById('createScoreValue').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        e.preventDefault(); // Prevent default form submission
        document.getElementById('createScoreBtn').click(); // Trigger the create button
    }
});

// OPTIONAL: Also add the same for edit score form for consistency
document.getElementById('editScoreForm').addEventListener('submit', function(e) {
    e.preventDefault(); // Prevent default form submission
    document.getElementById('saveScoreBtn').click(); // Trigger the save button
});

document.getElementById('editScoreValue').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        e.preventDefault(); // Prevent default form submission
        document.getElementById('saveScoreBtn').click(); // Trigger the save button
    }
});

        // Utility functions for debugging localStorage
        function debugSelections() {
            if (window.EventSelectionManager) {
                console.log('Current Selections:');
                console.log('Event:', window.EventSelectionManager.getSavedSelection('event'));
                console.log('Heat:', window.EventSelectionManager.getSavedSelection('heat'));  
                console.log('Run:', window.EventSelectionManager.getSavedSelection('run'));
            }
        }
        
        // Add debug function to window for console access
        window.debugEventSelections = debugSelections;
        
        // Show visual indicator when selections are restored
        function showRestoredIndicator(type, value) {
            const indicator = document.createElement('div');
            indicator.className = 'position-fixed top-0 start-50 translate-middle-x mt-3 alert alert-info alert-dismissible fade show';
            indicator.style.zIndex = '9999';
            indicator.innerHTML = `
                <i class="fas fa-history me-2"></i>
                Restored ${type}: <strong>${value}</strong>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(indicator);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (indicator.parentNode) {
                    indicator.remove();
                }
            }, 3000);
        }
        
        // Results Modal Functions
        function showEventResults() {
            const eventId = document.getElementById('eventSelect').value;
            if (!eventId) {
                alert('Please select an event first');
                return;
            }
            
            const modal = new bootstrap.Modal(document.getElementById('heatResultsModal'));
            modal.show();
            
            // Load saved configurations
            loadHeatConfigurations(eventId);
        }
        
        function loadHeatConfigurations(eventId) {
            const configsList = document.getElementById('savedConfigsList');
            
            // Show loading state
            configsList.innerHTML = `
                <div class="text-center py-3 text-muted">
                    <div class="spinner-border spinner-border-sm" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 small">Loading configurations...</p>
                </div>
            `;
            
            // Fetch all configurations for the event using public_dashboard_api.php
            fetch('../api/public_dashboard_api.php?action=get_configurations&event_id=' + eventId)
                .then(response => response.json())
                .then(data => {
                    if (!data.success || !data.configurations || data.configurations.length === 0) {
                        configsList.innerHTML = `
                            <div class="alert alert-warning mb-0 small">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                No saved reports found for this event.
                            </div>
                        `;
                        document.getElementById('resultsContentDisplay').innerHTML = `
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                No saved reports available for this event.
                            </div>
                        `;
                        return;
                    }
                    
                    // Use all configurations
                    const allConfigs = data.configurations;
                    
                    // Build configuration list
                    let html = '';
                    allConfigs.forEach((config, index) => {
                        const isActive = index === 0 ? 'active' : '';
                        const viewTypeIcon = config.view_type === 'start_list' ? 'fa-list-ol' : 'fa-table';
                        const viewTypeLabel = config.view_type === 'start_list' ? 'Start List' : 'Summary';
                        const heatLabel = config.heat_number ? 'Heat ' + config.heat_number : 'All Heats';
                        
                        html += '<button type="button" class="config-button ' + isActive + '" ';
                        html += 'data-config-id="' + config.id + '" ';
                        html += 'onclick="loadHeatConfiguration(' + config.id + ', \'' + config.view_type + '\')" ';
                        html += 'title="Click to load">';
                        html += '<div class="d-flex flex-column">';
                        html += '<div class="d-flex justify-content-between align-items-start mb-1">';
                        html += '<span class="me-2 text-start flex-grow-1">';
                        html += '<i class="fas ' + viewTypeIcon + ' me-2"></i>' + escapeHtml(config.name);
                        html += '</span>';
                        html += '<span class="badge bg-light text-dark">' + viewTypeLabel + '</span>';
                        html += '</div>';
                        html += '<small class="text-muted">';
                        html += '<i class="fas fa-fire me-1"></i>' + heatLabel;
                        html += '</small>';
                        html += '</div>';
                        html += '</button>';
                    });
                    
                    configsList.innerHTML = html;
                    
                    // Auto-load first configuration
                    if (allConfigs.length > 0) {
                        loadHeatConfiguration(allConfigs[0].id, allConfigs[0].view_type);
                    }
                })
                .catch(error => {
                    console.error('Error loading configurations:', error);
                    configsList.innerHTML = `
                        <div class="alert alert-danger mb-0 small">
                            <i class="fas fa-times-circle me-2"></i>
                            Error loading configurations: ` + escapeHtml(error.message) + `
                        </div>
                    `;
                });
        }
        
        function loadHeatConfiguration(configId, viewType) {
            const contentDisplay = document.getElementById('resultsContentDisplay');
            
            // Update active state
            document.querySelectorAll('#savedConfigsList .config-button').forEach(btn => {
                btn.classList.remove('active');
            });
            const targetBtn = document.querySelector('[data-config-id="' + configId + '"]');
            if (targetBtn) {
                targetBtn.classList.add('active');
            }
            
            // Show loading
            contentDisplay.innerHTML = `
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 text-muted">Loading results...</p>
                </div>
            `;
            
            // Determine API endpoint based on view type
            const apiEndpoint = viewType === 'start_list' 
                ? '../api/start_list_api.php' 
                : '../api/summary_table_api.php';
            
            // Load configuration
            fetch(apiEndpoint + '?config_id=' + configId + '&format=html')
                .then(response => {
                    if (!response.ok) {
                        throw new Error('HTTP ' + response.status);
                    }
                    return response.text();
                })
                .then(html => {
                    contentDisplay.innerHTML = html;
                })
                .catch(error => {
                    console.error('Error loading configuration:', error);
                    contentDisplay.innerHTML = `
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            Error loading results: ` + error.message + `
                        </div>
                    `;
                });
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    </script>
    
    <?php
    // Render the notification system
    render_notification_system();
    ?>

    <?php include '../admin/footer.php'; ?>
</body>
</html>