<?php
// Handle AJAX requests FIRST before any output
if (isset($_POST['action']) && $_POST['action'] === 'generate_random_scores') {
    // Start output buffering to catch any unwanted output from includes
    ob_start();
    require_once '../includes/auth.php';
    require_once '../includes/db.php';
    ob_end_clean(); // Discard any output from includes
    
    header('Content-Type: application/json');
    
    $user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';
    $judge_id = $_SESSION['judge_id'] ?? $_SESSION['user_id'] ?? 0;
    
    if ($user_role !== 'admin') {
        echo json_encode(['success' => false, 'message' => 'Admin access required']);
        exit;
    }
    
    $event_id = $_POST['event_id'] ?? null;
    $heat_number = $_POST['heat_number'] ?? null;
    $run_number = $_POST['run_number'] ?? null;
    
    if (!$event_id || !$heat_number || !$run_number) {
        echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
        exit;
    }
    
    try {
        // Get participants for this heat/run
        $participants_stmt = $pdo->prepare("
            SELECT ep.id as assignment_id, ep.participant_id,
                   p.first_name, p.last_name, p.bib
            FROM event_participants ep
            JOIN participants p ON ep.participant_id = p.id
            WHERE ep.event_id = ? AND ep.heat_number = ?
            ORDER BY p.bib
        ");
        $participants_stmt->execute([$event_id, $heat_number]);
        $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($participants)) {
            echo json_encode(['success' => false, 'message' => 'No participants found in this heat']);
            exit;
        }
        
        // Get scoring format
        $format_stmt = $pdo->prepare("
            SELECT sf.* FROM events e
            JOIN scoring_formats sf ON e.scoring_format = sf.format_id
            WHERE e.id = ?
        ");
        $format_stmt->execute([$event_id]);
        $format = $format_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$format) {
            echo json_encode(['success' => false, 'message' => 'Scoring format not found']);
            exit;
        }
        
        $criteria = json_decode($format['criteria_json'] ?? '[]', true) ?? [];
        $figures = json_decode($format['figures_json'] ?? '[]', true) ?? [];
        
        // Get judges for this event
        $judges_stmt = $pdo->prepare("
            SELECT u.id as judge_id, 
                   COALESCE(CONCAT(u.first_name, ' ', u.last_name), u.username, 'Judge ' || u.id) as judge_name
            FROM judge_assignments ja
            JOIN users u ON ja.judge_id = u.id
            WHERE ja.event_id = ?
        ");
        $judges_stmt->execute([$event_id]);
        $judges = $judges_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($judges)) {
            echo json_encode(['success' => false, 'message' => 'No judges assigned to this event']);
            exit;
        }
        
        $generated_count = 0;
        
        // Generate scores for each participant and each judge
        foreach ($participants as $participant) {
            // Get or create run record for this participant and run number
            $run_stmt = $pdo->prepare("
                SELECT id FROM runs 
                WHERE event_participant_id = ? AND run_number = ?
            ");
            $run_stmt->execute([$participant['assignment_id'], $run_number]);
            $run_id = $run_stmt->fetchColumn();
            
            if (!$run_id) {
                // Create run record if it doesn't exist
                $create_run_stmt = $pdo->prepare("
                    INSERT INTO runs (event_participant_id, run_number) VALUES (?, ?)
                ");
                $create_run_stmt->execute([$participant['assignment_id'], $run_number]);
                $run_id = $pdo->lastInsertId();
            }
            
            foreach ($judges as $judge) {
                // Check if score already exists
                $check_stmt = $pdo->prepare("
                    SELECT id FROM scores
                    WHERE run_id = ? AND judge_id = ?
                ");
                $check_stmt->execute([$run_id, $judge['judge_id']]);
                
                if ($check_stmt->fetch()) {
                    continue; // Skip if score already exists
                }
                
                // Generate random score between 30 and 95
                $score_value = round((30 + rand(0, 65)) + (rand(0, 100) / 100), 2);
                
                // Generate random figures (pick 2-4 random figures)
                $selected_figures = [];
                if (!empty($figures)) {
                    $num_figures = rand(2, min(4, count($figures)));
                    $random_keys = array_rand($figures, $num_figures);
                    if (!is_array($random_keys)) $random_keys = [$random_keys];
                    
                    foreach ($random_keys as $key) {
                        $category = $figures[$key]['category'] ?? 'Rotation';
                        if (!isset($selected_figures[$category])) {
                            $selected_figures[$category] = [];
                        }
                        $selected_figures[$category][] = $figures[$key]['name'];
                    }
                }
                
                // Insert the score
                $insert_stmt = $pdo->prepare("
                    INSERT INTO scores 
                    (run_id, judge_id, score_value, status, figures_json, created_at)
                    VALUES (?, ?, ?, 'OK', ?, NOW())
                ");
                
                $insert_stmt->execute([
                    $run_id,
                    $judge['judge_id'],
                    $score_value,
                    json_encode(['figures' => $selected_figures])
                ]);
                
                $generated_count++;
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => "Generated {$generated_count} random scores",
            'count' => $generated_count,
            'participants' => count($participants),
            'judges' => count($judges)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error generating scores: ' . $e->getMessage()
        ]);
    }
    exit;
}

// Regular page load - include files and check authorization
include '../includes/auth.php';
include '../includes/db.php';

// Get judge_id from session - use user_id as fallback
$judge_id = $_SESSION['judge_id'] ?? $_SESSION['user_id'] ?? 0;
$user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';

// Check if user is a head judge in users table OR is admin
$is_head_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ? AND judge_id = 1");
$is_head_stmt->execute([$judge_id]);
$is_head = $is_head_stmt->fetchColumn();
$approve_scores = hasPermission('approve_scores');
if (!$is_head && $user_role !== 'admin' && !$approve_scores) {
    header('Location: ../login.php');
    exit;
}

// Add this new AJAX action for getting event categories
    if (isset($_POST['action']) && $_POST['action'] === 'get_event_categories') {
        $event_id = $_POST['event_id'] ?? null;
        
        if ($event_id) {
            try {
                // Get categories from event_categories table for this event
                $categories_stmt = $pdo->prepare("
                    SELECT DISTINCT ec.category_name as category
                    FROM event_categories ec
                    WHERE ec.event_id = ?
                    ORDER BY ec.category_name
                ");
                $categories_stmt->execute([$event_id]);
                $categories = $categories_stmt->fetchAll(PDO::FETCH_COLUMN);
                
                echo json_encode([
                    'success' => true,
                    'categories' => $categories
                ]);
            } catch (Exception $e) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Error fetching categories: ' . $e->getMessage()
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        }
        exit;
    }

// Add AJAX action for getting format figures
    if (isset($_POST['action']) && $_POST['action'] === 'get_format_figures') {
        $event_id = $_POST['event_id'] ?? null;
        $heat_number = $_POST['heat_number'] ?? null;
        
        if ($event_id) {
            try {
                // Get the scoring format and layout config for this event
                $format_stmt = $pdo->prepare("
                    SELECT sf.format_id, sf.name, e.scoring_layout_config
                    FROM events e
                    LEFT JOIN scoring_formats sf ON e.scoring_format = sf.format_id
                    WHERE e.id = ?
                ");
                $format_stmt->execute([$event_id]);
                $format = $format_stmt->fetch(PDO::FETCH_ASSOC);
                
                // Get heat-specific layout config if heat number provided
                $heat_layout_config = null;
                if ($heat_number) {
                    $heat_stmt = $pdo->prepare("
                        SELECT scoring_layout_config 
                        FROM heat_settings 
                        WHERE event_id = ? AND heat_number = ?
                    ");
                    $heat_stmt->execute([$event_id, $heat_number]);
                    $heat_settings = $heat_stmt->fetch(PDO::FETCH_ASSOC);
                    if ($heat_settings && !empty($heat_settings['scoring_layout_config'])) {
                        $heat_layout_config = json_decode($heat_settings['scoring_layout_config'], true);
                    }
                }
                
                // Parse event-level layout config
                $event_layout_config = [];
                if (!empty($format['scoring_layout_config'])) {
                    $event_layout_config = json_decode($format['scoring_layout_config'], true) ?? [];
                }
                
                // Determine which categories to show (heat overrides event)
                $visible_categories = null;
                if ($heat_layout_config && isset($heat_layout_config['cards']['figuresCard']['visibleCategories'])) {
                    $visible_categories = $heat_layout_config['cards']['figuresCard']['visibleCategories'];
                } elseif ($event_layout_config && isset($event_layout_config['cards']['figuresCard']['visibleCategories'])) {
                    $visible_categories = $event_layout_config['cards']['figuresCard']['visibleCategories'];
                }
                
                // Debug: Add format info to response
                $debug_info = [
                    'event_id' => $event_id,
                    'heat_number' => $heat_number,
                    'format_found' => $format && !empty($format['format_id']),
                    'format_details' => $format,
                    'visible_categories' => $visible_categories,
                    'event_config' => $event_layout_config,
                    'heat_config' => $heat_layout_config
                ];
                
                if ($format && !empty($format['format_id'])) {
                    // Get figure categories and items for this format's sport
                    // First get the sport for this format
                    $sport_stmt = $pdo->prepare("SELECT sport FROM scoring_formats WHERE format_id = ?");
                    $sport_stmt->execute([$format['format_id']]);
                    $sport_info = $sport_stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($sport_info && !empty($sport_info['sport'])) {
                        // Build WHERE clause to filter by visible categories if configured
                        $categories_where = "fc.sport_name = ? AND fc.is_active = 1";
                        $categories_params = [$sport_info['sport']];
                        
                        if (!empty($visible_categories) && is_array($visible_categories)) {
                            $placeholders = implode(',', array_fill(0, count($visible_categories), '?'));
                            $categories_where .= " AND fc.category_name IN ($placeholders)";
                            $categories_params = array_merge($categories_params, $visible_categories);
                        }
                        
                        $categories_stmt = $pdo->prepare("
                            SELECT fc.id, fc.category_name, fi.id as item_id, fi.item_name
                            FROM figure_categories fc
                            LEFT JOIN figure_items fi ON fc.sport_name = fi.sport_name AND fc.category_name = fi.category_name
                            WHERE $categories_where
                            ORDER BY fc.sort_order ASC, fi.sort_order ASC
                        ");
                        $categories_stmt->execute($categories_params);
                        $figures_raw = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        $debug_info['sport'] = $sport_info['sport'];
                        $debug_info['figures_raw_count'] = count($figures_raw);
                        $debug_info['categories_filtered'] = !empty($visible_categories);
                    
                        // Organize by categories
                        $figures = [];
                        foreach ($figures_raw as $row) {
                            $cat_name = $row['category_name'];
                            if (!isset($figures[$cat_name])) {
                                $figures[$cat_name] = [];
                            }
                            if ($row['item_id']) {
                                $figures[$cat_name][] = [
                                    'id' => $row['item_id'],
                                    'name' => $row['item_name']
                                ];
                            }
                        }
                        
                        // Check if we have any figures configured
                        $has_figures = false;
                        foreach ($figures as $items) {
                            if (count($items) > 0) {
                                $has_figures = true;
                                break;
                            }
                        }
                        
                        echo json_encode([
                            'success' => true,
                            'format' => $format,
                            'figures' => $figures,
                            'has_figures' => $has_figures,
                            'debug' => $debug_info
                        ]);
                    } else {
                        echo json_encode([
                            'success' => false,
                            'message' => 'No sport information found for this scoring format',
                            'debug' => $debug_info
                        ]);
                    }
                } else {
                    echo json_encode([
                        'success' => true,
                        'format' => null,
                        'figures' => [],
                        'debug' => $debug_info,
                        'message' => 'No format found for event'
                    ]);
                }
            } catch (Exception $e) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Error fetching format figures: ' . $e->getMessage()
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        }
        exit;
    }
// Handle approve all participants action
if (isset($_POST['action']) && $_POST['action'] === 'approve_all_participant') {
        $event_id = $_POST['event_id'];
        $heat_number = $_POST['heat_number'];
        $run_number = $_POST['run_number'];
        $participant_id = $_POST['participant_id'];
        
        try {
            // Get the judge_id from users table for the approved_by field
            $judge_lookup_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ?");
            $judge_lookup_stmt->execute([$judge_id]);
            $approved_by_judge_id = $judge_lookup_stmt->fetchColumn();
            
            // Get participant details for notification
            $participant_stmt = $pdo->prepare("
                SELECT p.first_name, p.last_name, ep.bib_number
                FROM event_participants ep
                JOIN participants p ON ep.participant_id = p.id
                WHERE ep.id = ?
            ");
            $participant_stmt->execute([$participant_id]);
            $participant_details = $participant_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Get all unapproved scores for this participant in this heat/run
            $scores_stmt = $pdo->prepare("
                SELECT s.id
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                WHERE ep.event_id = ? AND ep.heat_number = ? AND r.run_number = ? 
                AND ep.id = ? AND s.is_approved = 0
            ");
            $scores_stmt->execute([$event_id, $heat_number, $run_number, $participant_id]);
            $score_ids = $scores_stmt->fetchAll(PDO::FETCH_COLUMN);
            
            // Prepare notification data for frontend
            $notification_data = null;
            if ($participant_details) {
                $participant_name = $participant_details['first_name'] . ' ' . $participant_details['last_name'];
                $notification_data = [
                    'bib' => $participant_details['bib_number'],
                    'participant_name' => $participant_name,
                    'count' => count($score_ids),
                    'heat' => $heat_number,
                    'run' => $run_number
                ];
            }
            
            if (count($score_ids) > 0) {
                // Approve all scores for this participant - using judge_id instead of user id
                $placeholders = str_repeat('?,', count($score_ids) - 1) . '?';
                $approve_stmt = $pdo->prepare("
                    UPDATE scores 
                    SET is_approved = 1, approved_by = ?, approved_at = NOW() 
                    WHERE id IN ($placeholders)
                ");
                $params = array_merge([$approved_by_judge_id], $score_ids);
                $approve_stmt->execute($params);
                
                // Update participant status to 'complete' after all scores are approved
                $participant_update = $pdo->prepare("
                    UPDATE event_participants 
                    SET participant_status = 'complete' 
                    WHERE id = ?
                ");
                $participant_update->execute([$participant_id]);
                
                // Clear bib_performing from event_heat_settings if this participant is currently performing
                if ($participant_details && isset($participant_details['bib_number'])) {
                    $clear_performing_stmt = $pdo->prepare("
                        UPDATE event_heat_settings 
                        SET bib_performing = NULL 
                        WHERE event_id = ? AND heat_number = ? AND bib_performing = ?
                    ");
                    $clear_performing_stmt->execute([$event_id, $heat_number, $participant_details['bib_number']]);
                }
                
                // Send to notification API if we have participant details
                if ($participant_details) {
                    // Get all judge scores for this participant after approval
                    $all_scores_stmt = $pdo->prepare("
                        SELECT s.score_value, s.status, u.username as judge_name, s.is_approved
                        FROM scores s
                        JOIN runs r ON s.run_id = r.id
                        JOIN event_participants ep ON r.event_participant_id = ep.id
                        JOIN users u ON s.judge_id = u.id
                        WHERE ep.id = ? AND ep.heat_number = ? AND r.run_number = ?
                        ORDER BY u.username
                    ");
                    $all_scores_stmt->execute([$participant_id, $heat_number, $run_number]);
                    $all_judge_scores = $all_scores_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    // Calculate final result and rank
                    $approved_scores = [];
                    $all_scores_info = [];
                    foreach ($all_judge_scores as $judge_score) {
                        $judge_display = $judge_score['status'] === 'OK' ? $judge_score['score_value'] : $judge_score['status'];
                        $all_scores_info[] = [
                            'judge' => $judge_score['judge_name'],
                            'score' => $judge_display,
                            'approved' => (bool)$judge_score['is_approved']
                        ];
                        
                        if ($judge_score['is_approved'] && $judge_score['status'] === 'OK' && $judge_score['score_value'] !== null) {
                            $approved_scores[] = floatval($judge_score['score_value']);
                        }
                    }
                    
                    $final_result = null;
                    $approved_count = count($approved_scores);
                    if ($approved_count > 0) {
                        // Apply drop rule if enough scores
                        if ($approved_count >= 3) {
                            sort($approved_scores);
                            array_shift($approved_scores); // Remove lowest
                            array_pop($approved_scores);   // Remove highest
                        }
                        $final_result = $approved_count > 0 ? round(array_sum($approved_scores) / count($approved_scores), 3) : null;
                    }
                    
                    // Calculate final rank position
                    $rank_position = null;
                    if ($final_result !== null) {
                        $ranking_stmt = $pdo->prepare("
                            SELECT ep.bib_number, ep.id as participant_id,
                                   AVG(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN s.score_value END) as avg_score,
                                   COUNT(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN 1 END) as approved_count
                            FROM event_participants ep
                            JOIN runs r ON ep.id = r.event_participant_id
                            JOIN scores s ON r.id = s.run_id
                            WHERE ep.event_id = ? AND ep.heat_number = ? AND r.run_number = ?
                            GROUP BY ep.id, ep.bib_number
                            HAVING approved_count > 0
                            ORDER BY avg_score DESC
                        ");
                        $ranking_stmt->execute([$event_id, $heat_number, $run_number]);
                        $rankings = $ranking_stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        foreach ($rankings as $index => $ranking) {
                            if ($ranking['participant_id'] == $participant_id) {
                                $rank_position = $index + 1;
                                break;
                            }
                        }
                    }
                    
                    // Update notification data with detailed information
                    $notification_data['all_scores'] = $all_scores_info;
                    $notification_data['final_result'] = $final_result;
                    $notification_data['rank_position'] = $rank_position;
                    $notification_data['total_judges'] = count($all_judge_scores);
                    
                    // Enhanced notification message
                    $result_text = $final_result !== null ? " | Final Result: {$final_result}" : "";
                    $rank_text = $rank_position !== null ? " | Final Rank: #{$rank_position}" : "";
                    
                    $api_data = [
                        'type' => 'success',
                        'title' => 'All Scores Approved',
                        'message' => sprintf(
                            "All %d scores approved for Bib #%s (%s) - Heat %s, Run %s%s%s",
                            count($score_ids),
                            $participant_details['bib_number'],
                            $participant_name,
                            $heat_number,
                            $run_number,
                            $result_text,
                            $rank_text
                        ),
                        'category' => 'judge',
                        'event_id' => $event_id,
                        'user_id' => $notification_user_id ?? 'head_judge',
                        'data' => json_encode([
                            'participant_id' => $participant_id,
                            'participant_bib' => $participant_details['bib_number'],
                            'participant_name' => $participant_name,
                            'scores_count' => count($score_ids),
                            'heat' => $heat_number,
                            'run' => $run_number,
                            'score_ids' => $score_ids,
                            'all_judge_scores' => $all_scores_info,
                            'final_result' => $final_result,
                            'rank_position' => $rank_position,
                            'total_judges' => count($all_judge_scores)
                        ])
                    ];
                    
                    // Send to notification API
                    $api_url = '../api/notification_api.php';
                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, $api_url);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($api_data + ['action' => 'send_notification']));
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                    curl_exec($ch);
                    curl_close($ch);
                }
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'All scores approved for participant',
                    'count' => count($score_ids),
                    'notification_data' => $notification_data
                ]);
            } else {
                echo json_encode([
                    'success' => true, 
                    'message' => 'No pending scores to approve',
                    'count' => 0,
                    'notification_data' => $notification_data
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'success' => false, 
                'message' => 'Error approving scores: ' . $e->getMessage()
            ]);
        }
        exit;
    }

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    if (isset($_POST['action']) && $_POST['action'] === 'get_scores') {
        $event_id = $_POST['event_id'] ?? null;
        $heat_number = $_POST['heat_number'] ?? null;
        $run_number = $_POST['run_number'] ?? null;
        
        if ($event_id) {
            try {
                // UPDATED: Get all judges for this event from users table
                $judges_stmt = $pdo->prepare("
                    SELECT DISTINCT u.id, u.username as name, u.judge_id as is_head, cp.control_point_name as control_point_name, cp.id as control_point_id
                    FROM users u
                    JOIN judge_assignments ja ON u.id = ja.judge_id
                    LEFT JOIN event_control_points cp ON ja.control_point_id = cp.id
                    WHERE ja.event_id = ? 
                    ORDER BY cp.control_point_name ASC, u.username ASC
                ");
                $judges_stmt->execute([$event_id]);
                $judges = $judges_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Build WHERE clause based on what filters are provided
                $where_conditions = ["ep.event_id = ?"];
                $params = [$event_id];
                
                if ($heat_number) {
                    $where_conditions[] = "ep.heat_number = ?";
                    $params[] = $heat_number;
                }
                
                $where_clause = implode(" AND ", $where_conditions);
                
                // Get all participants based on filters - UPDATED to get category from event_categories
                $participants_stmt = $pdo->prepare("
                    SELECT ep.id, ep.bib_number, ep.sort_order, ep.category as event_category, ep.heat_number,
                           p.first_name, p.last_name, p.club, p.category as base_category, p.gender,
                           COALESCE(ec.category_name, ep.category, p.category) as category
                    FROM event_participants ep
                    JOIN participants p ON ep.participant_id = p.id
                    LEFT JOIN event_categories ec ON ep.category_id = ec.id
                    WHERE {$where_clause}
                    ORDER BY ep.heat_number ASC, ep.sort_order ASC, ep.bib_number ASC
                ");
                $participants_stmt->execute($params);
                $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Get scores based on filters with approval status AND created_at for sorting
                // Check if figures_json column exists
                $figures_column = '';
                try {
                    $check_column = $pdo->query("SHOW COLUMNS FROM scores LIKE 'figures_json'");
                    if ($check_column->rowCount() > 0) {
                        $figures_column = ', s.figures_json';
                    }
                } catch (Exception $e) {
                    // Column doesn't exist, continue without it
                }
                
                // Build scores WHERE clause
                $scores_where_conditions = ["ep.event_id = ?"];
                $scores_params = [$event_id];
                
                if ($heat_number) {
                    $scores_where_conditions[] = "ep.heat_number = ?";
                    $scores_params[] = $heat_number;
                }
                
                if ($run_number) {
                    $scores_where_conditions[] = "r.run_number = ?";
                    $scores_params[] = $run_number;
                }
                
                $scores_where_clause = implode(" AND ", $scores_where_conditions);
                
                $scores_stmt = $pdo->prepare("
                    SELECT s.judge_id, ep.id as ep_id, s.score_value, s.status, s.id as score_id, 
                           COALESCE(s.is_approved, 0) as is_approved, s.created_at, r.run_number, ep.heat_number{$figures_column}
                    FROM scores s
                    JOIN runs r ON s.run_id = r.id
                    JOIN event_participants ep ON r.event_participant_id = ep.id
                    WHERE {$scores_where_clause}
                ");
                $scores_stmt->execute($scores_params);
                $scores_raw = $scores_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Organize scores by participant, run, and judge
                $scores = [];
                foreach ($scores_raw as $score) {
                    $run_key = $score['run_number'];
                    if (!isset($scores[$score['ep_id']])) {
                        $scores[$score['ep_id']] = [];
                    }
                    if (!isset($scores[$score['ep_id']][$run_key])) {
                        $scores[$score['ep_id']][$run_key] = [];
                    }
                    $scores[$score['ep_id']][$run_key][$score['judge_id']] = $score;
                }
                
                // Get drop_rule configuration with heat override support
                // Priority: 1) Heat-specific drop_rule, 2) Event format drop_rule, 3) Default 'none'
                // Note: Both events.scoring_format and scoring_format_judges.format_id contain the full unique format name
                $format_stmt = $pdo->prepare("
                    SELECT 
                        ehs.drop_rule as heat_drop_rule,
                        sfj.drop_rule as format_drop_rule,
                        COALESCE(ehs.drop_rule, sfj.drop_rule, 'none') as effective_drop_rule
                    FROM events e
                    LEFT JOIN event_heat_settings ehs ON e.id = ehs.event_id AND ehs.heat_number = ?
                    LEFT JOIN scoring_format_judges sfj ON e.scoring_format = sfj.format_id
                    WHERE e.id = ?
                    LIMIT 1
                ");
                $format_stmt->execute([$heat_number, $event_id]);
                $format_config = $format_stmt->fetch(PDO::FETCH_ASSOC);
                $drop_rule = $format_config ? $format_config['effective_drop_rule'] : 'none';
                $heat_drop_rule = $format_config ? $format_config['heat_drop_rule'] : null;
                $format_drop_rule = $format_config ? $format_config['format_drop_rule'] : null;
                
                // Load diversity rules and heat settings
                $diversity_rules = [];
                $diversity_enabled_for_heat = false;
                $scoring_format_id = null;
                
                try {
                    // Get scoring format ID
                    $format_id_stmt = $pdo->prepare("SELECT scoring_format FROM events WHERE id = ?");
                    $format_id_stmt->execute([$event_id]);
                    $scoring_format_id = $format_id_stmt->fetchColumn();
                    
                    if ($scoring_format_id && $heat_number) {
                        // Check if diversity rules are enabled for this specific heat
                        $diversity_check_stmt = $pdo->prepare("
                            SELECT diversity_rules_enabled 
                            FROM event_heat_settings 
                            WHERE event_id = ? AND heat_number = ?
                        ");
                        $diversity_check_stmt->execute([$event_id, $heat_number]);
                        $diversity_enabled_for_heat = (bool)$diversity_check_stmt->fetchColumn();
                        
                        // Load diversity rules if enabled for this heat
                        if ($diversity_enabled_for_heat) {
                            $div_stmt = $pdo->prepare("
                                SELECT rule_type, scope, group_name, min_distinct, is_optional 
                                FROM scoring_format_diversity_rules 
                                WHERE format_id = ? 
                                ORDER BY rule_type ASC, group_name ASC
                            ");
                            $div_stmt->execute([$scoring_format_id]);
                            $diversity_rules = $div_stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
                        }
                    }
                } catch (Exception $e) {
                    // If diversity check fails, continue without diversity rules
                    error_log("Diversity rules loading failed: " . $e->getMessage());
                }
                
                echo json_encode([
                    'success' => true,
                    'judges' => $judges,
                    'participants' => $participants,
                    'scores' => $scores,
                    'heat_number' => $heat_number,
                    'run_number' => $run_number,
                    'drop_rule' => $drop_rule,
                    'drop_rule_source' => $heat_drop_rule ? 'heat' : 'format',
                    'heat_drop_rule' => $heat_drop_rule,
                    'format_drop_rule' => $format_drop_rule,
                    'diversity_rules' => $diversity_rules,
                    'diversity_enabled' => $diversity_enabled_for_heat,
                    'scoring_format_id' => $scoring_format_id
                ]);
                
            } catch (Exception $e) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Database error: ' . $e->getMessage()
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        }
        exit;
    }
    
       if (isset($_POST['action']) && $_POST['action'] === 'approve_score') {
        $score_id = $_POST['score_id'];
        
        try {
            // Get the judge_id from users table for the approved_by field
            $judge_lookup_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ?");
            $judge_lookup_stmt->execute([$judge_id]);
            $approved_by_judge_id = $judge_lookup_stmt->fetchColumn();
            
            // Get score details for notification
            $score_details_stmt = $pdo->prepare("
                SELECT s.score_value, s.status, u.username as judge_name, 
                       p.first_name, p.last_name, ep.bib_number, ep.event_id,
                       ep.heat_number, r.run_number, ep.id as participant_id
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                JOIN participants p ON ep.participant_id = p.id
                JOIN users u ON s.judge_id = u.id
                WHERE s.id = ?
            ");
            $score_details_stmt->execute([$score_id]);
            $score_details = $score_details_stmt->fetch(PDO::FETCH_ASSOC);
            
            $approve_stmt = $pdo->prepare("UPDATE scores SET is_approved = 1, approved_by = ?, approved_at = NOW() WHERE id = ?");
            $approve_stmt->execute([$approved_by_judge_id, $score_id]);
            
            // Prepare notification data for frontend
            $notification_data = null;
            if ($score_details) {
                $participant_name = $score_details['first_name'] . ' ' . $score_details['last_name'];
                $score_display = $score_details['status'] === 'OK' ? $score_details['score_value'] : $score_details['status'];
                
                // Get all judge scores for this participant/heat/run
                $all_scores_stmt = $pdo->prepare("
                    SELECT s.score_value, s.status, u.username as judge_name, s.is_approved,
                           u.id as judge_id
                    FROM scores s
                    JOIN runs r ON s.run_id = r.id
                    JOIN event_participants ep ON r.event_participant_id = ep.id
                    JOIN users u ON s.judge_id = u.id
                    WHERE ep.id = ? AND ep.heat_number = ? AND r.run_number = ?
                    ORDER BY u.username
                ");
                $all_scores_stmt->execute([
                    $score_details['participant_id'], 
                    $score_details['heat_number'], 
                    $score_details['run_number']
                ]);
                $all_judge_scores = $all_scores_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Calculate current result (average of approved scores)
                $approved_scores = [];
                $all_scores_info = [];
                foreach ($all_judge_scores as $judge_score) {
                    $judge_display = $judge_score['status'] === 'OK' ? $judge_score['score_value'] : $judge_score['status'];
                    $all_scores_info[] = [
                        'judge' => $judge_score['judge_name'],
                        'score' => $judge_display,
                        'approved' => (bool)$judge_score['is_approved']
                    ];
                    
                    if ($judge_score['is_approved'] && $judge_score['status'] === 'OK' && $judge_score['score_value'] !== null) {
                        $approved_scores[] = floatval($judge_score['score_value']);
                    }
                }
                
                $current_result = null;
                $approved_count = count($approved_scores);
                if ($approved_count > 0) {
                    // Apply drop rule if enough scores (simplified version)
                    if ($approved_count >= 3) {
                        // Drop highest and lowest
                        sort($approved_scores);
                        array_shift($approved_scores); // Remove lowest
                        array_pop($approved_scores);   // Remove highest
                    }
                    $current_result = $approved_count > 0 ? round(array_sum($approved_scores) / count($approved_scores), 3) : null;
                }
                
                // Calculate current rank position
                $rank_position = null;
                if ($current_result !== null) {
                    // Get all participants' current results for ranking
                    $ranking_stmt = $pdo->prepare("
                        SELECT ep.bib_number, ep.id as participant_id,
                               AVG(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN s.score_value END) as avg_score,
                               COUNT(CASE WHEN s.is_approved = 1 AND s.status = 'OK' THEN 1 END) as approved_count
                        FROM event_participants ep
                        JOIN runs r ON ep.id = r.event_participant_id
                        JOIN scores s ON r.id = s.run_id
                        WHERE ep.event_id = ? AND ep.heat_number = ? AND r.run_number = ?
                        GROUP BY ep.id, ep.bib_number
                        HAVING approved_count > 0
                        ORDER BY avg_score DESC
                    ");
                    $ranking_stmt->execute([
                        $score_details['event_id'],
                        $score_details['heat_number'],
                        $score_details['run_number']
                    ]);
                    $rankings = $ranking_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    foreach ($rankings as $index => $ranking) {
                        if ($ranking['participant_id'] == $score_details['participant_id']) {
                            $rank_position = $index + 1;
                            break;
                        }
                    }
                }
                
                $notification_data = [
                    'bib' => $score_details['bib_number'],
                    'participant_name' => $participant_name,
                    'score' => $score_display,
                    'judge' => $score_details['judge_name'],
                    'heat' => $score_details['heat_number'],
                    'run' => $score_details['run_number'],
                    'all_scores' => $all_scores_info,
                    'current_result' => $current_result,
                    'rank_position' => $rank_position,
                    'approved_count' => $approved_count,
                    'total_judges' => count($all_judge_scores)
                ];
                
                // Enhanced notification message
                $result_text = $current_result !== null ? " | Current Result: {$current_result}" : "";
                $rank_text = $rank_position !== null ? " | Rank: #{$rank_position}" : "";
                $progress_text = " | Scores: {$approved_count}/" . count($all_judge_scores);
                
                // Send to notification API
                $api_data = [
                    'type' => 'success',
                    'title' => 'Score Approved',
                    'message' => sprintf(
                        "Bib #%s (%s) - %s approved: %s (Heat %s, Run %s)%s%s%s",
                        $score_details['bib_number'],
                        $participant_name,
                        $score_details['judge_name'],
                        $score_display,
                        $score_details['heat_number'],
                        $score_details['run_number'],
                        $result_text,
                        $rank_text,
                        $progress_text
                    ),
                    'category' => 'judge',
                    'event_id' => $score_details['event_id'],
                    'user_id' => $notification_user_id ?? 'head_judge',
                    'data' => json_encode([
                        'score_id' => $score_id,
                        'participant_bib' => $score_details['bib_number'],
                        'participant_name' => $participant_name,
                        'judge' => $score_details['judge_name'],
                        'approved_score' => $score_display,
                        'heat' => $score_details['heat_number'],
                        'run' => $score_details['run_number'],
                        'all_judge_scores' => $all_scores_info,
                        'current_result' => $current_result,
                        'rank_position' => $rank_position,
                        'approved_count' => $approved_count,
                        'total_judges' => count($all_judge_scores)
                    ])
                ];
                
                // Send to notification API
                $api_url = '../api/notification_api.php';
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $api_url);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($api_data + ['action' => 'send_notification']));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                curl_exec($ch);
                curl_close($ch);
            }
            
            echo json_encode([
                'success' => true, 
                'message' => 'Score approved successfully',
                'notification_data' => $notification_data
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error approving score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    // Handle unapprove score action
    if (isset($_POST['action']) && $_POST['action'] === 'unapprove_score') {
        $score_id = $_POST['score_id'];
        
        try {
            // Get score details for notification before unapproving
            $score_details_stmt = $pdo->prepare("
                SELECT s.score_value, s.status, u.username as judge_name, 
                       p.first_name, p.last_name, ep.bib_number, ep.event_id,
                       ep.heat_number, r.run_number
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                JOIN participants p ON ep.participant_id = p.id
                JOIN users u ON s.judge_id = u.id
                WHERE s.id = ?
            ");
            $score_details_stmt->execute([$score_id]);
            $score_details = $score_details_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Unapprove the score
            $unapprove_stmt = $pdo->prepare("UPDATE scores SET is_approved = 0, approved_by = NULL, approved_at = NULL WHERE id = ?");
            $unapprove_stmt->execute([$score_id]);
            
            // Prepare notification data
            $notification_data = null;
            if ($score_details) {
                $participant_name = $score_details['first_name'] . ' ' . $score_details['last_name'];
                $score_display = $score_details['status'] === 'OK' ? $score_details['score_value'] : $score_details['status'];
                
                $notification_data = [
                    'bib' => $score_details['bib_number'],
                    'participant_name' => $participant_name,
                    'score' => $score_display,
                    'judge' => $score_details['judge_name'],
                    'heat' => $score_details['heat_number'],
                    'run' => $score_details['run_number']
                ];
            }
            
            echo json_encode([
                'success' => true, 
                'message' => 'Score unapproved successfully',
                'notification_data' => $notification_data
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error unapproving score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    if (isset($_POST['action']) && $_POST['action'] === 'delete_score') {
        $score_id = $_POST['score_id'];
        
        try {
            // Get score details for notification before deleting
            $score_details_stmt = $pdo->prepare("
                SELECT s.score_value, s.status, u.username as judge_name, 
                       p.first_name, p.last_name, ep.bib_number, ep.event_id,
                       ep.heat_number, r.run_number
                FROM scores s
                JOIN runs r ON s.run_id = r.id
                JOIN event_participants ep ON r.event_participant_id = ep.id
                JOIN participants p ON ep.participant_id = p.id
                JOIN users u ON s.judge_id = u.id
                WHERE s.id = ?
            ");
            $score_details_stmt->execute([$score_id]);
            $score_details = $score_details_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Delete the score
            $delete_stmt = $pdo->prepare("DELETE FROM scores WHERE id = ?");
            $delete_stmt->execute([$score_id]);
            
            // Prepare notification data
            $notification_data = null;
            if ($score_details) {
                $participant_name = $score_details['first_name'] . ' ' . $score_details['last_name'];
                $score_display = $score_details['status'] === 'OK' ? $score_details['score_value'] : $score_details['status'];
                
                $notification_data = [
                    'bib' => $score_details['bib_number'],
                    'participant_name' => $participant_name,
                    'score' => $score_display,
                    'judge' => $score_details['judge_name'],
                    'heat' => $score_details['heat_number'],
                    'run' => $score_details['run_number']
                ];
            }
            
            echo json_encode([
                'success' => true, 
                'message' => 'Score deleted successfully',
                'notification_data' => $notification_data
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error deleting score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    if (isset($_POST['action']) && $_POST['action'] === 'update_score') {
        $score_id = $_POST['score_id'];
        $new_score = $_POST['score_value'] !== '' ? floatval($_POST['score_value']) : null;
        $status = $_POST['status'];
        $figures_json = $_POST['figures_json'] ?? null;
        
        try {
            // Check if figures_json column exists
            $figures_column_exists = false;
            try {
                $check_column = $pdo->query("SHOW COLUMNS FROM scores LIKE 'figures_json'");
                $figures_column_exists = $check_column->rowCount() > 0;
            } catch (Exception $e) {
                // Column doesn't exist
            }
            
            if ($figures_column_exists && isset($_POST['figures_json'])) {
                // Handle figures_json data
                $figures_value = null;
                if ($figures_json === 'null' || $figures_json === '') {
                    $figures_value = null;
                } else {
                    // Validate it's proper JSON and not empty object with empty arrays
                    try {
                        $decoded = json_decode($figures_json, true);
                        if ($decoded && is_array($decoded)) {
                            // Check if any category has non-empty data
                            $hasData = false;
                            foreach ($decoded as $category => $items) {
                                if (is_array($items) && count($items) > 0) {
                                    $hasData = true;
                                    break;
                                }
                            }
                            $figures_value = $hasData ? $figures_json : null;
                        } else {
                            $figures_value = null;
                        }
                    } catch (Exception $e) {
                        $figures_value = null;
                    }
                }
                
                // Update with figures_json
                $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, figures_json = ?, updated_at = NOW(), is_approved = 0 WHERE id = ?");
                $update_stmt->execute([$new_score, $status, $figures_value, $score_id]);
            } else {
                // Update without figures_json
                $update_stmt = $pdo->prepare("UPDATE scores SET score_value = ?, status = ?, updated_at = NOW(), is_approved = 0 WHERE id = ?");
                $update_stmt->execute([$new_score, $status, $score_id]);
            }
            
            echo json_encode(['success' => true, 'message' => 'Score updated successfully (requires re-approval)']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error updating score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    // NEW: Handle create empty score action
    if (isset($_POST['action']) && $_POST['action'] === 'create_score') {
        $event_id = $_POST['event_id'];
        $heat_number = $_POST['heat_number'];
        $run_number = $_POST['run_number'];
        $participant_id = $_POST['participant_id'];
        $judge_id_target = $_POST['judge_id'];
        $score_value = $_POST['score_value'] !== '' ? floatval($_POST['score_value']) : null;
        $status = $_POST['status'];
        $figures_json = isset($_POST['figures_json']) ? $_POST['figures_json'] : null;
        
        try {
            // First, find or create the run for this participant/heat/run combination
            $run_stmt = $pdo->prepare("
                SELECT r.id 
                FROM runs r 
                JOIN event_participants ep ON r.event_participant_id = ep.id 
                WHERE ep.id = ? AND r.run_number = ?
            ");
            $run_stmt->execute([$participant_id, $run_number]);
            $run_id = $run_stmt->fetchColumn();
            
            if (!$run_id) {
                // Create the run if it doesn't exist
                $create_run_stmt = $pdo->prepare("INSERT INTO runs (event_participant_id, run_number) VALUES (?, ?)");
                $create_run_stmt->execute([$participant_id, $run_number]);
                $run_id = $pdo->lastInsertId();
            }
            
            // Check if scores table has figures_json column
            $hasColumnsQuery = "SHOW COLUMNS FROM scores LIKE 'figures_json'";
            $hasColumnsStmt = $pdo->query($hasColumnsQuery);
            $hasFiguresColumn = $hasColumnsStmt->rowCount() > 0;
            
            // Process figures_json data
            $figures_value = null;
            if ($hasFiguresColumn && $figures_json) {
                try {
                    $decoded = json_decode($figures_json, true);
                    if ($decoded && is_array($decoded)) {
                        // Check if any category has non-empty data
                        $hasData = false;
                        foreach ($decoded as $category => $items) {
                            if (is_array($items) && count($items) > 0) {
                                $hasData = true;
                                break;
                            }
                        }
                        $figures_value = $hasData ? $figures_json : null;
                    } else {
                        $figures_value = null;
                    }
                } catch (Exception $e) {
                    $figures_value = null;
                }
            }
            
            // Create the score with or without figures_json based on column existence
            if ($hasFiguresColumn) {
                $create_score_stmt = $pdo->prepare("
                    INSERT INTO scores (run_id, judge_id, score_value, status, figures_json, created_at, is_approved) 
                    VALUES (?, ?, ?, ?, ?, NOW(), 0)
                ");
                $create_score_stmt->execute([$run_id, $judge_id_target, $score_value, $status, $figures_value]);
            } else {
                $create_score_stmt = $pdo->prepare("
                    INSERT INTO scores (run_id, judge_id, score_value, status, created_at, is_approved) 
                    VALUES (?, ?, ?, ?, NOW(), 0)
                ");
                $create_score_stmt->execute([$run_id, $judge_id_target, $score_value, $status]);
            }
            
            // Update participant status to 'scoring' when a new score is added
            $update_status_stmt = $pdo->prepare("
                UPDATE event_participants 
                SET participant_status = 'scoring' 
                WHERE id = ?
            ");
            $update_status_stmt->execute([$participant_id]);
            
            echo json_encode(['success' => true, 'message' => 'Score created successfully (requires approval)']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error creating score: ' . $e->getMessage()]);
        }
        exit;
    }
    
    // If no action matched, return error
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    exit;
}

// UPDATED: Get events - admins see all events, head judges see only assigned events
if ($user_role === 'admin') {
    // Admin sees all events
    $events_stmt = $pdo->prepare("
        SELECT DISTINCT e.id, e.name, e.date
        FROM events e
        ORDER BY e.date DESC, e.name
    ");
    $events_stmt->execute();
} else {
    // Head judges see only assigned events
    $events_stmt = $pdo->prepare("
        SELECT DISTINCT e.id, e.name, e.date
        FROM events e
        JOIN judge_assignments ja ON e.id = ja.event_id
        WHERE ja.judge_id = ?
        ORDER BY e.date DESC, e.name
    ");
    $events_stmt->execute([$judge_id]);
}
$events = $events_stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Head Judge Panel</title>
    <!-- style skin holder -->
 <?php include_once '../includes/stylesheets.php'; ?>
    <!-- Lucide Icons and translation utility are now included globally via stylesheets.php and footer.php -->
    
    <style>
        .card-hover:hover {
            transform: translateY(-2px);
            transition: transform 0.2s ease-in-out;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
        }
        .cursor-pointer {
            cursor: pointer;
        }
        
        /* Auto switches styling */
        .form-check-input {
            cursor: pointer;
        }
        .form-check-label {
            cursor: pointer;
        }
        
        /* Results Modal Styles */
        .config-button {
            width: 100%;
            text-align: left;
            padding: 0.75rem;
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
            background-color: white;
            cursor: pointer;
            transition: all 0.2s ease;
            margin-bottom: 0.5rem;
        }
        .config-button:hover {
            background-color: #f8f9fa;
            border-color: #0d6efd;
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        }
        .config-button.active {
            background-color: #0d6efd;
            color: white;
            border-color: #0d6efd;
        }
        .config-button.active .badge {
            background-color: rgba(255, 255, 255, 0.2) !important;
            color: white !important;
        }
        .config-button.active .text-muted {
            color: rgba(255, 255, 255, 0.8) !important;
        }
        #resultsContentDisplay {
            min-height: 500px;
        }
        #savedConfigsList {
            max-height: 70vh;
            overflow-y: auto;
        }
     body {
            padding-top: 0px !important;
     }         
        nav{
            display: none !important;
        }
.heat-loading-indicator {
    position: absolute;
    top: 10px;
    right: 10px;
    width: 20px;
    height: 20px;
    z-index: 10;
    opacity: 0;
    pointer-events: none;
    transition: opacity 0.3s ease;
}
.heat-loading-indicator.is-visible {
    opacity: 1;
    animation: pulse-blink 1s ease-in-out infinite;
}
.heat-loading-indicator i {
    font-size: 20px;
    color: #0d6efd;
}
@keyframes pulse-blink {
    0%, 100% { opacity: 0.3; }
    50% { opacity: 1; }
}
.bg-approved {
    /* background-color: gray;*/
    background: linear-gradient(180deg, #00aa0740 0%, #00ff0a00 10%, #00ff0a00 90%, #00aa0740 100%); 
}
.bg-pending {
    /* background-color: gray; */
    background: linear-gradient(180deg, #67676740 0%, rgba(0, 51, 170, 0) 5%, rgba(0, 51, 170, 0) 95%, #67676740 100%);
}
.bg-empty {
    background: linear-gradient(180deg, #6c757d87 0%, rgba(108, 117, 125, 0) 25%, rgba(108, 117, 125, 0) 75%, #6c757d87 100%)
}

    
/* Sticky thead - requires max-height on table-responsive */
.table-responsive {
    max-height: 90vh;
    overflow-y: auto;
    position: relative;
}
#scoresTable thead {
    position: sticky;
    top: 0;
    z-index: 10;
    background-color: white;
}
#scoresTable thead th {
    background-color: #f8f9fa;
    box-shadow: 0 2px 2px -1px rgba(0, 0, 0, 0.4);
}

#scoresTable th {
    padding: 5px !important;
}

/* Drop score styling */
.score_drop {
    opacity: 0.7;
}

.score_drop .score_value {
        color: #1f2225 !important;
   
}

.score_drop:hover .score_value {
    color: #495057 !important;
}

.figures-display {
    
    border-radius: 0.375rem;
    padding: 0.5rem;
    grid-template-columns: auto auto auto;
    align-self: center !important;
}
.row.score-figure {
    display: flex !important;
    grid-template-columns: auto auto;
    justify-items: center !important;
    min-height: auto;
    flex-wrap: wrap;
    flex-direction: row;
}

.table-responsive td .figures-display, .figure-s {
    transition: transform 0.4s ease-in-out;
    cursor: pointer;
}

.table-responsive td:hover .figures-display, .figure-s:hover{
    transform: scale(1.2);
    z-index: 10;
    position: relative;
}


.bg-empty .row.score-figure {
    display: grid !important;
    grid-template-columns: auto;
    align-items: center !important;
    justify-items: center !important;
}
.scores-col2 {
    align-self: center !important;
    justify-self: center !important;
}
.scores-col1 {
    align-self: center !important;
    justify-self: center !important;
}

/* Center align all table cell content */
.table td {
    vertical-align: middle !important;
    text-align: center !important;
}

/* Ensure the first column (participant info) is left-aligned for readability */
.table td:first-child {
    text-align: left !important;
}

/* Center all content within score cells */
.table td .row.score-figure {
    justify-content: center !important;
    align-items: center !important;
    width: 100% !important;
    margin: 0 !important;
}

/* Center the grid columns */
.scores-col1, .scores-col2 {
    display: flex !important;
    flex-direction: column !important;
    align-items: center !important;
    justify-content: center !important;
}

/* Center figures display content */
.figures-display {
    text-align: center !important;
    display: flex !important;
    flex-direction: row;
    flex-wrap: wrap;
    justify-content: space-evenly;
}


.pendingActions {
    backdrop-filter: blur(11px);
    width: 100px !important;
}
.OK .approval-clickable{
    display: none;
}
.editScoreButton {
    display: none !important;
}
.bg-pending .bg-danger.approval-clickable {
    display: block;
}
.approval-clickable:active,
.approval-clickable.holding {
    transform: scale(1.5);
    transition: transform 0.2s ease-out;
}
div#scoresContainer td {
    padding: 5px !important;
}
#scoresWrapper {
    
    overflow: auto;
    scroll-behavior: smooth;
}
td.participantCol {
    width: 280px;
}

/* Mobile participant row - hidden by default */
.participantColMobileRow {
    display: none !important;
}

/* Hide mobile-specific classes on desktop */
.participantColMobileHide {
    display: table-cell;
}

/* Responsive: Show mobile row and hide regular column on smaller screens */
@media (max-width: 991px) {
    /* Hide the regular participant column */
    td.participantCol {
        display: none !important;
    }
    
    /* Hide desktop average and actions columns (mobile uses rowspan) */
    .participantColMobileHide {
        display: none !important;
    }
    
    /* Show the mobile full-width participant rows */
    .participantColMobileRow {
        display: table-row !important;
        background-color: rgba(13, 110, 253, 0.05);
        border-bottom: 2px solid #dee2e6;
    }
    
    .participantColMobileRow td {
        font-size: 0.9rem;
    }
    
    .participantColMobileRow .badge {
        font-size: 0.75rem;
    }
}

td.resultCom.text-center {
    width: 100px;
}

td.actionsCol.text-center {
    width: 100px;
}

/* Figure chip styles */
.figure-chip {
    transition: all 0.2s ease;
    border: 1px solid #6c757d !important;
    color: #6c757d !important;
}

.figure-chip:hover {
    border-color: #0d6efd !important;
    color: #0d6efd !important;
    background-color: rgba(13, 110, 253, 0.1) !important;
}

.figure-chip.active {
    background-color: #0d6efd !important;
    border-color: #0d6efd !important;
    color: white !important;
}

.figure-chip.active:hover {
    background-color: #0b5ed7 !important;
    border-color: #0b5ed7 !important;
}

/* Active Heat Card Styles */


#activeHeatCardContainer .heat-card {
    transition: all 0.3s ease;
}

#activeHeatCardContainer .heat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

#activeHeatCardContainer .col-md-4.col-lg-3.mb-3 {
    width: 100%;
}

/* Pulse highlight animation for buttons */
@keyframes pulseHighlight {
    0%, 100% {
        box-shadow: 0 0 0 0 rgba(13, 110, 253, 0.7);
        transform: scale(1);
    }
    50% {
        box-shadow: 0 0 0 10px rgba(13, 110, 253, 0);
        transform: scale(1.05);
    }
}

.pulse-highlight {
    animation: pulseHighlight 0.6s ease-in-out 3;
}

/* Validation styling for score inputs */
#editScoreValue.is-invalid,
#createScoreValue.is-invalid {
    border-color: #dc3545 !important;
    background-color: #f8d7da !important;
}

.figure-category h6 {
    font-weight: 600;
}

.figure-chips {
    min-height: 2.5rem;
}

/* Control Point Summary Row Styles */
.control-point-summary {
    background-color: rgba(248, 249, 250, 0.8) !important;
    border-top: 2px solid #dee2e6 !important;
    font-size: 0.85rem;
}

.control-point-summary td {
    padding: 5px 5px !important;
    vertical-align: top !important;
}
.figures-summary{    display: flex
;}
.mb-1.figuresByCategory {
    display: flex;
    flex-wrap: wrap;
    justify-content: center;
}
.control-point-summary .scores-summary .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.75rem);
    padding: 0.3em 0.5em;
}
.figures-summary label {
    display: none !important;
}
.control-point-summary .figures-summary {
    font-size: 0.75rem;
}

.control-point-summary .figures-summary .badge {
   font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.75rem);
    padding: 0.2em 0.4em;
}
.control-point-summary .figures-summary.fig_5 .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.6rem);
    padding: 0.2em 0.4em;
}
.control-point-summary .figures-summary.fig-3_5 .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 0.65rem);
    padding: 0.2em 0.4em;
}
.control-point-summary .figures-summary.fig-2 .badge {
    font-size: clamp(0.6rem, 0.2375rem + 1.1vw, 1rem);
    padding: 0.2em 0.4em;
}
.ss-bg-A {
    background-color: #ede2ff !important;
}
.ss-bg-B {
    background-color: #fcffe2 !important;
}
/* Enhanced figures display under scoreCol */
.scores-col2 .figures-display {
    background-color: rgba(255, 255, 255, 0.9);
    border-radius: 4px;
    margin-top: 4px;
}

/* Edit Score Modal Styles */
#editScoreInfo {
    background: linear-gradient(135deg, #e8f4fd 0%, #f0f8ff 100%);
    border: 1px solid #b3d9f2;
    border-radius: 8px;
}

#editScoreInfo h6 {
    color: #1e3a8a;
    font-weight: 600;
    border-bottom: 1px solid rgba(30, 58, 138, 0.2);
    padding-bottom: 0.5rem;
}

#editScoreInfo .mb-2 {
    margin-bottom: 0.75rem !important;
    padding: 0.25rem 0;
    border-radius: 4px;
    transition: background-color 0.2s ease;
}

#editScoreInfo .mb-2:hover {
    background-color: rgba(59, 130, 246, 0.05);
    padding-left: 0.5rem;
    margin-left: -0.5rem;
    margin-right: -0.5rem;
}

#editScoreInfo .badge {
    font-size: 0.75rem;
    padding: 0.35em 0.6em;
}

#editScoreInfo .text-muted {
    opacity: 0.7;
}

/* Notification System Styles */
.css-bib {
    font-weight: bold !important;
    font-size: 0.9em !important;
    padding: 0.3em 0.6em !important;
    margin: 0 0.2em !important;
}
span.badge.bg-danger.approval-clickable {
    max-width: 90px;
}
.judge_score {
    font-weight: bold !important;
    font-size: 0.85em !important;
    padding: 0.25em 0.5em !important;
    margin: 0 0.1em !important;
}

.judge_score.approved {
    background-color: #28a745 !important;
    border-color: #28a745 !important;
}



/* ===== COMPACT DASHBOARD LAYOUT STYLES ===== */

/* Sticky Top Control Bar */
#headJudgeTopBar {
    position: sticky;
    top: 0;
    z-index: 1020;
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    border-bottom: 2px solid #dee2e6;
    padding: 0.5rem 0;
}

#headJudgeTopBar .control-row {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    flex-wrap: wrap;
}

#headJudgeTopBar select {
    min-width: 120px;
}

#headJudgeTopBar .metrics-pills {
    display: flex;
    gap: 0.5rem;
    align-items: center;
}

#headJudgeTopBar .metric-pill {
    background: white;
    border: 1px solid #dee2e6;
    border-radius: 20px;
    padding: 0.25rem 0.75rem;
    font-size: 0.85rem;
    white-space: nowrap;
}

#headJudgeTopBar .metric-pill .metric-label {
    color: #6c757d;
    margin-right: 0.25rem;
}

#headJudgeTopBar .metric-pill .metric-value {
    font-weight: 600;
    color: #0d6efd;
}

/* Queue Sidebar */
#queueSidebar {
    position: fixed;
    top: 80px;
    right: -380px;
    width: 360px;
    height: calc(100vh - 80px);
    background: white;
    box-shadow: -4px 0 12px rgba(0,0,0,0.15);
    transition: right 0.3s ease;
    z-index: 1010;
    overflow-y: auto;
    border-left: 3px solid #0d6efd;
}

#queueSidebar.open {
    right: 0;
}

#queueSidebar .sidebar-header {
    position: sticky;
    top: 0;
    background: linear-gradient(135deg, #0d6efd 0%, #0b5ed7 100%);
    color: white;
    padding: 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    z-index: 10;
}

#queueSidebar .sidebar-content {
    padding: 1rem;
}

#queueToggleBtn {
    position: fixed;
    bottom: 90px;
    right: 23px;
    z-index: 1015;
    border-radius: 50%;
    width: 48px;
    height: 48px;
    padding: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
    transition: all 0.3s 
ease;
}

#queueToggleBtn:hover {
    transform: scale(1.1);
}

#queueToggleBtn.active {
    right: 370px;
}

/* Performing Queue Styles (from active panel) */
.performing-slot {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-radius: 8px;
    padding: 0.75rem;
    margin-bottom: 0.75rem;
    border-left: 4px solid #6c757d;
}

.performing-slot.current {
    background: linear-gradient(135deg, #d1ecf1 0%, #bee5eb 100%);
    border-left-color: #0d6efd;
}

.performing-slot.next {
    background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
    border-left-color: #ffc107;
}

.performing-slot .slot-label {
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    color: #6c757d;
    margin-bottom: 0.5rem;
    display: block;
}

.performing-slot .slot-content {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.slot-chip {
    background: white;
    border: 1px solid #dee2e6;
    border-radius: 6px;
    padding: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.9rem;
}

.slot-chip .slot-bib {
    background: #0d6efd;
    color: white;
    padding: 0.2rem 0.4rem;
    border-radius: 4px;
    font-weight: 600;
    font-size: 0.75rem;
}

/* Queue Chips List */
.queue-chip {
    background: white;
    border: 1px solid #dee2e6;
    border-radius: 8px;
    padding: 0.75rem;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    transition: all 0.2s ease;
}

.queue-chip:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    transform: translateX(-4px);
}

.queue-chip-rank {
    background: linear-gradient(135deg, #6c757d 0%, #495057 100%);
    color: white;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    font-size: 0.9rem;
    flex-shrink: 0;
}

.queue-chip-current .queue-chip-rank {
    background: linear-gradient(135deg, #0d6efd 0%, #0b5ed7 100%);
}

.queue-chip-next .queue-chip-rank {
    background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);
}

.queue-chip-body {
    flex: 1;
    min-width: 0;
}

.queue-chip-name {
    font-weight: 600;
    font-size: 0.95rem;
    color: #212529;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.queue-chip-meta {
    font-size: 0.75rem;
    color: #6c757d;
    margin-top: 0.25rem;
}

/* Floating Action Button */
#headJudgeFAB {
    position: fixed;
    bottom: 24px;
    right: 8px;
    z-index: 1030;
}

#headJudgeFAB .fab-main {
    width: 56px;
    height: 56px;
    border-radius: 50%;
    background: linear-gradient(135deg, #0d6efd 0%, #0b5ed7 100%);
    color: white;
    border: none;
    box-shadow: 0 6px 20px rgba(13, 110, 253, 0.4);
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.3s ease;
}

#headJudgeFAB .fab-main:hover {
    transform: scale(1.1) rotate(90deg);
    box-shadow: 0 8px 24px rgba(13, 110, 253, 0.6);
}

#headJudgeFAB .fab-menu {
    position: absolute;
    bottom: 70px;
    right: 0;
    display: none;
    flex-direction: column;
    gap: 12px;
    opacity: 0;
    transform: translateY(20px);
    transition: all 0.3s ease;
}

#headJudgeFAB.open .fab-menu {
    display: flex;
    opacity: 1;
    transform: translateY(0);
}

#headJudgeFAB .fab-action {
    background: white;
    border: 2px solid #dee2e6;
    border-radius: 50%;
    width: 48px;
    height: 48px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    transition: all 0.2s ease;
    position: relative;
}

#headJudgeFAB .fab-action:hover {
    transform: scale(1.1);
    border-color: #0d6efd;
    color: #0d6efd;
}

#headJudgeFAB .fab-action::before {
    content: attr(data-tooltip);
    position: absolute;
    right: 60px;
    background: #212529;
    color: white;
    padding: 0.5rem 0.75rem;
    border-radius: 6px;
    white-space: nowrap;
    font-size: 0.85rem;
    opacity: 0;
    pointer-events: none;
    transition: opacity 0.2s ease;
}

#headJudgeFAB .fab-action:hover::before {
    opacity: 1;
}

/* Compact Table Styles */
#scoresTable {
    font-size: 0.9rem;
}

#scoresTable th {
    padding: 0.4rem 0.3rem !important;
    font-size: 0.85rem;
}

#scoresTable td {
    padding: 0.4rem 0.3rem !important;
}

/* Collapsible Filter Panel */
#filterPanel {
    background: #f8f9fa;
    border-radius: 8px;
    padding: 1rem;
   
    max-height: 0;
    overflow: hidden;
    transition: max-height 0.3s ease, padding 0.3s ease;
}

#filterPanel.expanded {
    max-height: 500px;
    padding: 1rem;
}

#filterPanel.collapsed {
    padding: 0 1rem;
}

#filterToggleBtn {
    cursor: pointer;
    transition: transform 0.3s ease;
}

#filterToggleBtn.rotated {
    transform: rotate(180deg);
}
#activeHeatCardContainer .col-md-6.col-lg-4.mb-3 {
    width: 100%;
}
/* Mobile Responsiveness */
@media (max-width: 768px) {
    #queueSidebar {
        width: 100%;
        right: -100%;
    }
    
    #queueToggleBtn.active {
        right: 10px;
    }
    
    #headJudgeFAB {
        bottom: 16px;
        right: 16px;
    }
    #queueToggleBtn {   
    bottom: 70px;
    right: 15px;   
}
    #headJudgeFAB .fab-main {
        width: 48px;
        height: 48px;
    }
    
    #headJudgeTopBar .metrics-pills {
        width: 100%;
        justify-content: space-between;
    }
}

/* Bubble Confetti Effect */
.approval-clickable {
    position: relative;
    overflow: visible;
}

.approval-clickable:before, .approval-clickable:after {
    position: absolute;
    content: '';
    display: block;
    width: 140%;
    height: 100%;
    left: -20%;
    z-index: -1000;
    transition: all ease-in-out 0.5s;
    background-repeat: no-repeat;
    pointer-events: none;
}

.approval-clickable:before {
    display: none;
    top: -75%;
    background-image: radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, transparent 20%, #28a745 20%, transparent 30%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, transparent 10%, #28a745 15%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%);
    background-size: 10% 10%, 20% 20%, 15% 15%, 20% 20%, 18% 18%, 10% 10%, 15% 15%, 10% 10%, 18% 18%;
}

.approval-clickable:after {
    display: none;
    bottom: -75%;
    background-image: radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, transparent 10%, #28a745 15%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%), 
                      radial-gradient(circle, #28a745 20%, transparent 20%);
    background-size: 15% 15%, 20% 20%, 18% 18%, 20% 20%, 15% 15%, 10% 10%, 20% 20%;
}

.approval-clickable.confetti-animate:before {
    display: block;
    animation: topBubbles ease-in-out 0.75s forwards;
}

.approval-clickable.confetti-animate:after {
    display: block;
    animation: bottomBubbles ease-in-out 0.75s forwards;
}

@keyframes topBubbles {
    0% {
        background-position: 5% 90%, 10% 90%, 10% 90%, 15% 90%, 25% 90%, 25% 90%, 40% 90%, 55% 90%, 70% 90%;
    }
    50% {
        background-position: 0% 80%, 0% 20%, 10% 40%, 20% 0%, 30% 30%, 22% 50%, 50% 50%, 65% 20%, 90% 30%;
    }
    100% {
        background-position: 0% 70%, 0% 10%, 10% 30%, 20% -10%, 30% 20%, 22% 40%, 50% 40%, 65% 10%, 90% 20%;
        background-size: 0% 0%, 0% 0%, 0% 0%, 0% 0%, 0% 0%, 0% 0%;
    }
}

@keyframes bottomBubbles {
    0% {
        background-position: 10% -10%, 30% 10%, 55% -10%, 70% -10%, 85% -10%, 70% -10%, 70% 0%;
    }
    50% {
        background-position: 0% 80%, 20% 80%, 45% 60%, 60% 100%, 75% 70%, 95% 60%, 105% 0%;
    }
    100% {
        background-position: 0% 90%, 20% 90%, 45% 70%, 60% 110%, 75% 80%, 95% 70%, 110% 10%;
        background-size: 0% 0%, 0% 0%, 0% 0%, 0% 0%, 0% 0%, 0% 0%;
    }
}

    </style>
</head>

<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>

    <div class="container-fluid container-StyleScore position-sticky sticky-top top-0">
        <!-- Compact Sticky Top Bar -->
            <div id="headJudgeTopBar">
                <div class="container-fluid">
                    <div class="control-row">
                        <!-- Title & Badge -->
                        <div class="d-flex align-items-center gap-2">
                            <h6 class="mb-0 fw-bold">
                                <i class="fas fa-crown me-1"></i>
                                <?php if ($user_role === 'admin'): ?>
                                    Admin Panel
                                <?php else: ?>
                                    Head Judge
                                <?php endif; ?>
                            </h6>
                            <?php if ($user_role === 'admin'): ?>
                                <span class="badge bg-danger small">Admin</span>
                            <?php else: ?>
                                <span class="badge bg-warning small">Judge</span>
                            <?php endif; ?>
                           
                        </div>
                        
                        <!-- Event/Heat/Run Selectors -->
                        <select class="form-select form-select-sm d-none" id="eventSelect" style="max-width: 200px;">
                            <option value="">Select Event</option>
                            <?php foreach ($events as $event): ?>
                                <option value="<?= $event['id'] ?>"><?= htmlspecialchars($event['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                        
                        <select class="form-select form-select-sm" id="heatSelect" disabled style="max-width: 140px;">
                            <option value="">Heat</option>
                        </select>
                        
                        <select class="form-select form-select-sm" id="runSelect" disabled style="max-width: 120px;">
                            <option value="">Run</option>
                        </select>
                        
                        <!-- Quick Actions -->
                        <button class="btn btn-light btn-sm" id="refreshBtn" title="Refresh data">
                            <i class="fas fa-sync-alt"></i>
                        </button>
                        
                        <button class="btn btn-outline-secondary btn-sm" id="clearSelectionsBtn" title="Clear selections">
                            <i class="fas fa-eraser"></i>
                        </button>
                        
                        <?php if ($user_role === 'admin'): ?>
                            <button class="btn btn-outline-primary btn-sm" id="generateRandomScoresBtn" title="Generate random scores for testing" disabled>
                                <i class="fas fa-dice"></i> Random Scores
                            </button>
                        <?php endif; ?>
                        
                        
                        
                        <!-- Auto Switches -->
                        <div class="form-check form-switch mb-0">
                            <input class="form-check-input" type="checkbox" id="autoHeatRun">
                            <label class="form-check-label small" for="autoHeatRun" style="font-size: 0.75rem;">
                                Auto
                            </label>
                        </div>
                        
                        <div class="form-check form-switch mb-0">
                            <input class="form-check-input" type="checkbox" id="autoRefresh" checked>
                            <label class="form-check-label small" for="autoRefresh" style="font-size: 0.75rem;">
                                Refresh
                            </label>
                        </div>

                                    

                    </div>
                </div>
            </div>
        </div>
        
                       
        <!-- Judge Status Card - Always visible, sticky when pending -->
        <div class="card mb-3 mx-2" id="judge-status-card">
            <div class="card-header p-2 d-flex justify-content-between align-items-center d-flex flex-row gap-3" id="judgeStatusHeader"  onclick="toggleJudgeStatusCollapse()">
                <div >
                    <!-- Action buttons will be inserted here -->
                </div>
                <div class="d-flex justify-content-between align-items-center w-100">
                    <div class="text-center flex-grow-1">
                        <i class="fas fa-circle me-2" id="judgeStatusIcon"></i>
                        <span class="fw-bold fs-5" id="judgeStatusText">Status: Ready</span>
                    </div>
                </div>
                
                <div id="judgeStatusActions">
                    <!-- Action buttons will be inserted here -->
                </div>
            </div>
        </div>
        
        <!-- Judge Status Options Collapse -->
        <div class="collapse mb-3" id="judgeStatusCollapse">
            <div class="card shadow-lg mx-2">
                <div class="card-header p-2 d-flex justify-content-between align-items-center bg-secondary-subtle">
                    <span><i class="fas fa-exchange-alt me-2"></i>Change Status</span>
                    <button type="button" class="btn btn-sm btn-outline-secondary" data-bs-toggle="collapse" data-bs-target="#judgeStatusCollapse">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <div class="card-body p-3">
                    <div class="d-grid gap-2">
                        <div class="form-check form-switch ms-2 fs-5">
                            <input class="form-check-input" type="checkbox" id="autoReadyToggle" onchange="handleAutoReadyToggle()">
                            <label class="form-check-label small" for="autoReadyToggle" title="Automatically approve to Ready status">Auto Ready</label>
                        </div>
                        <button type="button" class="btn btn-success btn-lg" onclick="setJudgeStatus('ready')">
                            <i class="fas fa-check-circle me-2"></i>Ready
                        </button>
                        <button type="button" class="btn btn-outline-warning btn-lg" onclick="setJudgeStatus('pending')">
                            <i class="fas fa-clock me-2"></i>Pending
                        </button>
                        <button type="button" class="btn btn-info btn-lg" onclick="setJudgeStatus('info')">
                            <i class="fas fa-info-circle me-2"></i>Info
                        </button>
                        <button type="button" class="btn btn-warning btn-lg" onclick="setJudgeStatus('warning')">
                            <i class="fas fa-exclamation-circle me-2"></i>Warning
                        </button>
                        <button type="button" class="btn btn-danger btn-lg" onclick="setJudgeStatus('danger')">
                            <i class="fas fa-exclamation-triangle me-2"></i>Danger
                        </button>
                        <button type="button" class="btn btn-secondary btn-lg" onclick="setJudgeStatus('off')">
                            <i class="fas fa-power-off me-2"></i>Off
                        </button>
                    </div>
                    <div class="small text-muted mt-2 text-center">
                        Status is visible to head judge and event coordinators
                    </div>
                    
                    <!-- Set All Judges Status Section -->
                    <hr class="my-3">
                    <div class="text-center mb-2">
                        <strong><i class="fas fa-users-cog me-2"></i>Set Status for All Judges</strong>
                    </div>
                    <div class="d-grid gap-2">
                        <button type="button" class="btn btn-outline-success" onclick="setAllJudgesStatus('ready')">
                            <i class="fas fa-check-circle me-2"></i>Set All to Ready
                        </button>
                        <button type="button" class="btn btn-outline-warning" onclick="setAllJudgesStatus('pending')">
                            <i class="fas fa-clock me-2"></i>Set All to Pending
                        </button>
                        <button type="button" class="btn btn-outline-info" onclick="setAllJudgesStatus('info')">
                            <i class="fas fa-info-circle me-2"></i>Set All to Info
                        </button>
                    </div>
                    <div class="small text-muted mt-2 text-center">
                        Updates status for all judges assigned to the current event
                    </div>
                </div>
            </div>
        </div>
        
                
            <!-- Active Heat Card Section -->
            <div class="container-fluid mt-2" id="activeHeatSection" style="display: none;">
                <div class="row">
                    <div class="col-12">
                        <div class="">
                            <div class="card-body p-2">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h6 class="mb-0 fw-bold text-primary">
                                        <i class="fas fa-user-tie me-2"></i>Judges status
                                    </h6>
                                    <div class="d-flex gap-2 align-items-center">
                                        <!-- Heat Card Layout Selector -->
                                        <div class="d-flex flex-column gap-1">
                                            
                                            <div class="btn-group btn-group-sm align-items-center" role="group" aria-label="Heat card layout">
                                                <small class="text-muted fw-bold me-2" style="font-size: 0.7rem;">Layout: </small> 
                                                <button type="button" class="btn btn-outline-secondary active" onclick="setHeadJudgeHeatCardLayout('judges', this)" title="Judges status only">
                                                    <i class="fas fa-user-tie"></i> Judges
                                                </button>
                                                <button type="button" class="btn btn-outline-secondary" onclick="setHeadJudgeHeatCardLayout('monitor-row', this)" title="Monitor row layout">
                                                    <i class="fas fa-align-justify"></i> Row
                                                </button>
                                                <button type="button" class="btn btn-outline-secondary" onclick="setHeadJudgeHeatCardLayout('monitor', this)" title="Monitor layout">
                                                    <i class="fas fa-desktop"></i> Monitor
                                                </button>
                                                <button type="button" class="btn btn-outline-secondary" onclick="setHeadJudgeHeatCardLayout('full', this)" title="Full details">
                                                    <i class="fas fa-expand"></i> Full
                                                </button>
                                            </div>
                                        </div>
                                        <button class="btn btn-sm btn-outline-primary" onclick="loadActiveHeatCardInHeadJudge()" title="Refresh heat card">
                                            <i class="fas fa-sync-alt"></i> Refresh
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Heat Card Display Area -->
                                <div id="activeHeatCardContainer" class="row">
                                    <!-- Active heat card will be loaded here -->
                                    <div class="col-12 text-center text-muted py-3">
                                        <i class="fas fa-fire fa-2x mb-2 opacity-25"></i>
                                        <p class="mb-0 small">Loading active heat...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
         
             <!-- Collapsible Filter Panel -->   
                    <div id="filterPanel" class="collapsed">
                        <!-- Sort Controls -->
                        <div class="sort-controls d-none p-2  bg-white shadow-sm" id="sortControls">
                            <div class="d-flex align-items-center justify-content-between flex-wrap gap-3">
                                
                                <div class="d-flex align-items-center gap-2 d-none" id="sortableMetrics">
                                    <span class="badge bg-info" id="sortableCount">0 complete</span>
                                </div>
                            </div>

                            <!-- Filter Controls -->
                            <div class="">
                                <div class="row g-2 align-items-end">
                                    <div class="col-2 col-md-2">
                                        <label class="form-label small fw-bold text-muted mb-1">
                                            <i class="fas fa-filter me-1"></i>Gender
                                        </label>
                                        <select class="form-select form-select-sm" id="genderFilter">
                                            <option value="">All</option>
                                            <option value="M">Male</option>
                                            <option value="F">Female</option>
                                        </select>
                                    </div>
                                    <div class="col-4 col-md-2">
                                        <label class="form-label small fw-bold text-muted mb-1">Category</label>
                                        <select class="form-select form-select-sm" id="categoryFilter">
                                            <option value="">All</option>
                                        </select>
                                    </div>
                                    <div class="col-4 col-md-2">
                                        <label class="form-label small fw-bold text-muted mb-1">Search</label>
                                        <input type="text" class="form-control form-control-sm" id="bibNameFilter" placeholder="BIB or Name...">
                                    </div>
                                    <div class="col-2 col-md-1">
                                        <button class="btn btn-outline-secondary btn-sm" id="clearFilters">
                                            <i class="fas fa-times me-1"></i>Clear
                                        </button>
                                    </div>
                                    <div class="col-auto gap-2">
                                    <span class="fw-bold text-muted">
                                        <i class="fas fa-sort me-1"></i>Sort:
                                    </span>
                                    <div class="btn-group btn-group-sm w-100" role="group">
                                        <button type="button" class="btn btn-outline-primary active" data-sort="default">
                                            <i class="fas fa-list-ol me-1"></i>Default
                                        </button>
                                        <button type="button" class="btn btn-outline-primary" data-sort="average">
                                            <i class="fas fa-trophy me-1"></i>By Score
                                        </button>
                                        <button type="button" class="btn btn-outline-primary" data-sort="latest">
                                            <i class="fas fa-clock me-1"></i>Latest
                                        </button>
                                    </div>
                                    
                                </div>
                                </div>
                               
                            </div>
                        </div>
                    </div>

                        <!-- Main Content Area -->
                        <div class="container-fluid">
                            <div class="row">
                                <div class="col-12">
                                    <div class="col-md-auto ms-auto">
                                        
                                        <div class="align-items-center d-flex flex-wrap gap-2 justify-content-between">
                                           <div>
                                            <button class="btn btn-sm btn-outline-secondary" id="filterToggleBtn" onclick="toggleFilters()">
                                                <i class="fas fa-filter me-1"></i>
                                                Filters & Sorting
                                                <i class="fas fa-chevron-down ms-1"></i>
                                            </button>
                                           </div>
                                           <div class="d-flex flex-row gap-2">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="showToApprove" checked>
                                                <label class="form-check-label small" for="showToApprove">
                                                    <span class="badge bg-warning">Pending</span>
                                                </label>
                                            </div>
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="showAllApproved" checked>
                                                <label class="form-check-label small" for="showAllApproved">
                                                    <span class="badge bg-success">Approved</span>
                                                </label>
                                            </div>
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="showWaiting" checked>
                                                <label class="form-check-label small" for="showWaiting">
                                                    <span class="badge bg-secondary">Waiting</span>
                                                </label>
                                            </div>
                                        </div>
                                        
                                            <div class="">
                                                <small class="text-muted small">
                                                <span id="filteredCount">0</span> of <span id="totalCount">0</span>
                                                </small>
                                            </div>
                                        </div>
                                    </div>
                                    
                    <!-- Score Grid Card -->
                    <div class="card border-0 shadow-sm">
                         
                        <div class="card-body p-2">
                            <div class="text-center text-muted py-5" id="noDataMessage">
                                <i class="fas fa-clipboard-list fa-3x mb-3 opacity-50"></i>
                                <h5>Select Event, Heat, and Run</h5>
                                <p class="small">Use the controls above to view scores</p>
                            </div>
                           
                            <div id="scoresWrapper">
                                <div id="scoresContainer" style="display: none;">
                                    <!-- Dynamic content loads here -->
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
                                <div class="btn-group bg-light mt-2" role="group">                                   
                                    <button type="button" class="btn" onclick="window.location.href='dashboard.php'">
                                        <i class="fas fa-arrow-left me-1"></i>Go to dashboard
                                    </button>
                            </div>
        <!-- Queue Sidebar Toggle Button -->
        <button class="btn btn-primary" id="queueToggleBtn" onclick="toggleQueueSidebar()" title="Queue Monitor">
            <i class="fas fa-users"></i>
        </button>
        
        <!-- Queue Sidebar -->
        <div id="queueSidebar">
            <div class="sidebar-header">
                <div>
                    <h6 class="mb-0">
                        <i class="fas fa-users me-2"></i>Queue Monitor
                    </h6>
                    <small class="opacity-75" id="queueSummarySubtitle">Waiting for data...</small>
                </div>
                <button class="btn btn-sm btn-light" onclick="toggleQueueSidebar()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="sidebar-content">
                <!-- Queue Summary -->
                <div class="mb-3">
                    <div class="row g-2 text-center">
                        <div class="col-4">
                            <div class="metric-pill">
                                <div class="metric-label">Total</div>
                                <div class="metric-value" id="queueTotalCount">--</div>
                            </div>
                        </div>
                        <div class="col-4">
                            <div class="metric-pill">
                                <div class="metric-label">Performing</div>
                                <div class="metric-value text-primary" id="queuePerformingCount">0</div>
                            </div>
                        </div>
                        <div class="col-4">
                            <div class="metric-pill">
                                <div class="metric-label">Waiting</div>
                                <div class="metric-value text-warning" id="queueWaitingCount">0</div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="metric-pill">
                                <div class="metric-label">Completed</div>
                                <div class="metric-value text-success" id="queueCompletedCount">0</div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="metric-pill">
                                <div class="metric-label">Run</div>
                                <div class="metric-value" id="queueRunLabel">--</div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Performing Slots -->
                <div class="mb-3">
                    <h6 class="fw-bold mb-2">
                        <i class="fas fa-play-circle me-1"></i>Now Performing
                    </h6>
                    <div id="queuePerformingSlots">
                        <div class="performing-slot current">
                            <span class="slot-label">Current</span>
                            <div class="slot-content text-muted small">Waiting for data...</div>
                        </div>
                        <div class="performing-slot next">
                            <span class="slot-label">Up Next</span>
                            <div class="slot-content text-muted small">Queue forming...</div>
                        </div>
                    </div>
                </div>
                
                <!-- Waiting Queue -->
                <div class="mb-3">
                    <h6 class="fw-bold mb-2">
                        <i class="fas fa-hourglass-half me-1"></i>Waiting Queue
                    </h6>
                    <div id="queueChipsList">
                        <div class="text-muted small text-center py-2">No one waiting</div>
                    </div>
                </div>
                
                <!-- Quick Alerts -->
                <div>
                    <h6 class="fw-bold mb-2">
                        <i class="fas fa-bell me-1"></i>Recent Alerts
                        <span class="badge bg-primary ms-1" id="alertsCountBadge">0</span>
                    </h6>
                    <div id="alertsList" style="max-height: 200px; overflow-y: auto;">
                        <div class="text-muted small text-center py-2">No recent alerts</div>
                    </div>
                    <div class="mt-2">
                        <small class="text-muted" id="alertsStatusLabel">Last updated: --</small>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Floating Action Button -->
        <div id="headJudgeFAB">
            <div class="fab-menu">
                <div class="fab-action" data-tooltip="View Results" onclick="showEventResults(); toggleFAB();">
                    <button class="btn btn-danger align-baseline"><i class="fas fa-trophy"></i> Results</button>
                </div>
                <div class="fab-action" data-tooltip="Refresh All" onclick="loadScores(); refreshHeadJudgeQueuePanel(); toggleFAB();">
                    <button class="btn btn-dark align-baseline"><i class="fas fa-sync-alt"></i> Refresh</button>
                </div>
                <div class="fab-action" data-tooltip="Help" onclick="openHeadJudgeHelp(); toggleFAB();">
                    <button class="btn btn-info align-baseline"><i class="fas fa-question-circle"></i> Help</button>
                </div>
            </div>
            <button class="fab-main" onclick="toggleFAB()">
                <i class="fas fa-ellipsis-v"></i>
            </button>
        </div>
    </div>
    
   
    
    <!-- Success Toast -->
    <div class="toast-container position-fixed top-0 end-0 p-3">
        <div id="successToast" class="toast" role="alert">
            <div class="toast-header bg-success text-white">
                <i class="fas fa-check-circle me-2"></i>
                <strong class="me-auto">Success</strong>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                Updated successfully
            </div>
        </div>
    </div>
    
    <!-- Edit Score Modal -->
    <div class="modal fade" id="editScoreModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i>
                        <?php echo $user_role === 'admin' ? 'Admin Override - Edit Score' : 'Head Judge Override - Edit Score'; ?>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <!-- Participant Summary (context moved below figures) -->
                    <div class="alert alert-info mb-3" id="editScoreInfo">
                        <div class="text-center text-muted">
                            <i class="fas fa-spinner fa-spin me-2"></i>Loading participant information...
                        </div>
                    </div>
                    
                    <form id="editScoreForm">
                        <input type="hidden" id="editScoreId">
                         <div class="mb-3">
                            <label class="form-label fw-bold">Status</label>
                            <div class="btn-group w-100 bg-white" role="group">
                                <input type="radio" class="btn-check" name="editStatus" id="editStatusOK" value="OK" checked>
                                <label class="btn btn-outline-success status-btn fs-5" for="editStatusOK">
                                    <i class="fas fa-check me-2"></i>OK
                                </label>

                                <input type="radio" class="btn-check" name="editStatus" id="editStatusDNS" value="DNS">
                                <label class="btn btn-outline-warning status-btn fs-5" for="editStatusDNS">
                                    <i class="fas fa-ban me-2"></i>DNS
                                </label>
                            
                                <input type="radio" class="btn-check" name="editStatus" id="editStatusDNF" value="DNF">
                                <label class="btn btn-outline-danger status-btn fs-5" for="editStatusDNF">
                                    <i class="fas fa-flag me-2"></i>DNF
                                </label>
                                
                                <input type="radio" class="btn-check" name="editStatus" id="editStatusDSQ" value="DSQ">
                                <label class="btn btn-outline-dark status-btn fs-5" for="editStatusDSQ">
                                    <i class="fas fa-times-circle me-2"></i>DSQ
                                </label>
                            </div>
                        </div>
                        <!-- Score Input with Keypad (same as score.php) -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Score (0-99)</label>
                            <input type="text" class="bg-gradient form-control fs-1 score-input ss-f-number bg-light" 
                                id="editScoreValue" 
                                placeholder="Use keypad below" 
                                autocomplete="off" required readonly>
                        </div>
                        <div class="mb-1">
                            <input type="range" class="form-range" id="editScoreRange" min="0" max="99" oninput="document.getElementById('editScoreValue').value=this.value">
                        </div>

                        <!-- Numeric Keypad -->
                        <div class="btn-group-vertical mt-0 rounded-2 w-100 p-3 mb-3 score-input-wrapper" role="group" aria-label="Numeric keypad">
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('1')"><span>1</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('2')"><span>2</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('3')"><span>3</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('4')"><span>4</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('5')"><span>5</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('6')"><span>6</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('7')"><span>7</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('8')"><span>8</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('9')"><span>9</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-1">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="backspaceEditScore()"><span>&lt;</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToEditScore('0')"><span>0</span></button>
                                <button type="button" class="bg-gradient bg-info btn btn-lg btn-primary py-3 key fs-1 border-0" onclick="applyEditScoreValue()"><span>Go</span></button>
                            </div>
                        </div>
                        
                       
                        
                        <div class="mb-3" id="editScoreFigures" style="display:block;">
                            <label class="form-label">Figures</label>
                            <div class="small text-muted mb-2" id="editScoreFiguresContent"></div>
                            <div id="editFiguresEditor" style="display:none;">
                                <div class="border rounded p-3 bg-light">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <small class="fw-bold text-muted">Edit Figures:</small>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="cancelEditFigures()">
                                            <i class="fas fa-times"></i> Cancel
                                        </button>
                                    </div>
                                    <div id="editFigureCategories"></div>
                                    <button type="button" class="btn btn-sm btn-success mt-2" onclick="saveFiguresEdit()">
                                        <i class="fas fa-save"></i> Save Figures
                                    </button>
                                </div>
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" id="editFiguresBtn" onclick="startEditFigures()" style="display:none;">
                                <i class="fas fa-edit"></i> Edit Figures
                            </button>
                            <!-- Score Context inserted below participant summary per request -->
                            <div id="scoreContextInfo" class="mt-3 p-3 border rounded bg-light small"></div>
                        </div>
                        
                        <div class="alert alert-<?php echo $user_role === 'admin' ? 'danger' : 'warning'; ?>">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong><?php echo $user_role === 'admin' ? 'Admin Override:' : 'Head Judge Override:'; ?></strong> 
                            This will permanently modify the judge's score and require re-approval.
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <!-- Approval buttons (hold to trigger) -->
                    <div class="me-auto d-flex gap-2" id="editModalApprovalButtons" style="display: none;">
                        <span class="approval-clickable btn btn-outline-success border-0 w-auto unapprove-btn" data-score-id="" style="cursor: pointer; position: relative; overflow: hidden;" title="Hold to unapprove">
                            <i class="fas fa-check-circle me-1"></i>Approved
                            <span class="hold-progress" style="position: absolute; bottom: 0; left: 0; height: 100%; width: 0%; background: #dc3545; transition: none;"></span>
                        </span>
                        <span class="approval-clickable btn btn-outline-danger w-auto approve-btn" data-score-id="" style="cursor: pointer; position: relative; overflow: hidden; display: none;" title="Hold to approve">
                            <i class="fas fa-times-circle me-1"></i>Not Approved
                            <span class="hold-progress" style="position: absolute; bottom: 0; left: 0; height: 100%; width: 0%; background: #28a745; transition: none;"></span>
                        </span>
                        
                        <!-- Delete Score Button (hold to trigger) -->
                        <span class="btn btn-outline-danger w-auto delete-score-btn" data-score-id="" style="cursor: pointer; position: relative; overflow: hidden;" title="Hold to delete this score permanently">
                            <i class="fas fa-trash me-1"></i>Delete Score
                            <span class="hold-progress" style="position: absolute; bottom: 0; left: 0; height: 100%; width: 0%; background: #dc3545; transition: none;"></span>
                        </span>
                    </div>
                    
                    <button type="button" class="btn btn-primary" id="saveScoreBtn">
                        <i class="fas fa-save me-2"></i>Update Score
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Score Modal -->
    <div class="modal fade" id="createScoreModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-plus me-2"></i>
                        <?php echo $user_role === 'admin' ? 'Admin - Create Score' : 'Head Judge - Create Score'; ?>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <!-- Participant Summary (context moved below figures) -->
                    <div class="alert alert-info mb-3" id="createScoreInfo">
                        <div class="text-center text-muted">
                            <i class="fas fa-spinner fa-spin me-2"></i>Loading participant information...
                        </div>
                    </div>
                    
                    <form id="createScoreForm">
                        <input type="hidden" id="createParticipantId">
                        <input type="hidden" id="createJudgeId">
                         <div class="mb-3">
                            <label class="form-label fw-bold">Status</label>
                            <div class="btn-group w-100 bg-white" role="group">
                                <input type="radio" class="btn-check" name="createStatus" id="createStatusOK" value="OK" checked>
                                <label class="btn btn-outline-success status-btn fs-5" for="createStatusOK">
                                    <i class="fas fa-check me-2"></i>OK
                                </label>

                                <input type="radio" class="btn-check" name="createStatus" id="createStatusDNS" value="DNS">
                                <label class="btn btn-outline-warning status-btn fs-5" for="createStatusDNS">
                                    <i class="fas fa-ban me-2"></i>DNS
                                </label>
                            
                                <input type="radio" class="btn-check" name="createStatus" id="createStatusDNF" value="DNF">
                                <label class="btn btn-outline-danger status-btn fs-5" for="createStatusDNF">
                                    <i class="fas fa-flag me-2"></i>DNF
                                </label>
                                
                                <input type="radio" class="btn-check" name="createStatus" id="createStatusDSQ" value="DSQ">
                                <label class="btn btn-outline-dark status-btn fs-5" for="createStatusDSQ">
                                    <i class="fas fa-times-circle me-2"></i>DSQ
                                </label>
                            </div>
                        </div>
                        <!-- Score Input with Keypad (same as score.php) -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Score (0-99)</label>
                            <input type="text" class="bg-gradient form-control fs-1 score-input ss-f-number bg-light" 
                                id="createScoreValue" 
                                placeholder="Use keypad below" 
                                autocomplete="off" required readonly>
                        </div>
                        <div class="mb-1">
                            <input type="range" class="form-range" id="createScoreRange" min="0" max="99" oninput="document.getElementById('createScoreValue').value=this.value">
                        </div>

                        <!-- Numeric Keypad -->
                        <div class="btn-group-vertical mt-0 rounded-2 w-100 p-3 mb-3" role="group" aria-label="Numeric keypad">
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('1')"><span>1</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('2')"><span>2</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('3')"><span>3</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('4')"><span>4</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('5')"><span>5</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('6')"><span>6</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-2">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('7')"><span>7</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('8')"><span>8</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('9')"><span>9</span></button>
                            </div>
                            <div class="btn-group gap-2 mb-1">
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="backspaceCreateScore()"><span>&lt;</span></button>
                                <button type="button" class="bg-gradient bg-info-subtle btn btn-lg btn-outline-info fs-1 key py-3" onclick="addToCreateScore('0')"><span>0</span></button>
                                <button type="button" class="bg-gradient bg-info btn btn-lg btn-primary py-3 key fs-1 border-0" onclick="applyCreateScoreValue()"><span>Go</span></button>
                            </div>
                        </div>
                        
                       
                        
                        <!-- Figures Section (mirrors Edit Score Modal structure) -->
                        <div class="mb-3" id="createFiguresSection" style="display: block;">
                            <label class="form-label">Figures</label>
                            <div class="small text-muted mb-2" id="createScoreFiguresContent"></div>
                            <div id="createFiguresEditor" style="display:none;">
                                <div class="border rounded p-3 bg-light">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <small class="fw-bold text-muted">Edit Figures:</small>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="cancelCreateFiguresEdit()">
                                            <i class="fas fa-times"></i> Cancel
                                        </button>
                                    </div>
                                    <div id="createFiguresContainer">
                                        <div class="text-muted">Loading figure options...</div>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-success mt-2" onclick="saveCreateFiguresEdit()">
                                        <i class="fas fa-save"></i> Save Figures
                                    </button>
                                </div>
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" id="createFiguresBtn" onclick="startCreateFiguresEdit()" style="display:none;">
                                <i class="fas fa-edit"></i> Edit Figures
                            </button>
                            
                        </div>
                        
                        <div class="alert alert-<?php echo $user_role === 'admin' ? 'danger' : 'warning'; ?>">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong><?php echo $user_role === 'admin' ? 'Admin:' : 'Head Judge:'; ?></strong> 
                            This will create a new score entry that requires approval.
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-success" id="createScoreBtn">
                        <i class="fas fa-plus me-2"></i>Create Score
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Results Modal Integration -->
    <div class="modal fade" id="heatResultsModal" tabindex="-1" aria-labelledby="heatResultsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-fullscreen p-5">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="heatResultsModalLabel">
                        <i class="fas fa-trophy me-2"></i>
                        <span id="resultsEventName">Event Results</span>
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <div class="row g-0">
                        <!-- Sidebar: Saved Configurations -->
                        <div class="col-12 col-lg-3 bg-light border-end">
                            <div class="p-3">
                                <h6 class="fw-bold mb-3">
                                    <i class="fas fa-list me-2"></i>Available Reports
                                </h6>
                                <div id="savedConfigsList" class="d-flex flex-column gap-2">
                                    <!-- Configurations will be loaded here -->
                                    <div class="text-center text-muted py-3">
                                        <i class="fas fa-spinner fa-spin"></i>
                                        <p class="small mb-0 mt-2">Loading configurations...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Main Content: Results Display -->
                        <div class="col-12 col-lg-9">
                            <div class="p-2">
                                <div id="resultsContentDisplay">
                                    <!-- Results content will be loaded here -->
                                    <div class="text-center text-muted py-5">
                                        <i class="fas fa-chart-bar fa-3x mb-3 opacity-25"></i>
                                        <p class="mb-0">Select a report from the sidebar</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Help Modal -->
    <div class="modal fade" id="headJudgeHelpModal" tabindex="-1" aria-labelledby="headJudgeHelpTitle" aria-hidden="true">
        <div class="modal-dialog modal-xl modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="headJudgeHelpTitle">
                        <i class="fas fa-circle-question me-2"></i>Head Judge Panel – Help
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <iframe id="headJudgeHelpIframe" style="width:100%; height:80vh; border:none;" title="Head Judge Help"></iframe>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteScoreConfirmModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-danger">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle me-2"></i>Confirm Delete Score
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-danger mb-3">
                        <i class="fas fa-trash me-2"></i>
                        <strong>Warning:</strong> This action CANNOT be undone!
                    </div>
                    <p class="mb-3">Are you sure you want to permanently delete this score?</p>
                    <div id="deleteScoreDetails" class="bg-light p-3 rounded">
                        <!-- Details will be populated by JavaScript -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-danger" id="confirmDeleteScoreBtn">
                        <i class="fas fa-trash me-1"></i>Delete Score
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Set All Judges Status Confirmation Modal -->
    <div class="modal fade" id="setAllJudgesConfirmModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-warning">
                <div class="modal-header bg-warning text-dark">
                    <h5 class="modal-title">
                        <i class="fas fa-users-cog me-2"></i>Set Status for All Judges
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning mb-3">
                        <i class="fas fa-exclamation-circle me-2"></i>
                        <strong>Attention:</strong> This will update ALL judges in this event!
                    </div>
                    <p class="mb-3" id="setAllJudgesMessage">
                        <!-- Message will be populated by JavaScript -->
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-warning" id="confirmSetAllJudgesBtn">
                        <i class="fas fa-check me-1"></i>Confirm
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Error Alert Modal -->
    <div class="modal fade" id="errorAlertModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-danger">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle me-2"></i>Error
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p id="errorAlertMessage" class="mb-0">
                        <!-- Error message will be populated by JavaScript -->
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Close
                    </button>
                </div>
            </div>
        </div>
    </div>

    
    <script>
        function openHeadJudgeHelp() {
            const modalEl = document.getElementById('headJudgeHelpModal');
            const iframe = document.getElementById('headJudgeHelpIframe');
            if (iframe && !iframe.src) {
                iframe.src = '../help/head_judge_help.html';
            }
            const modal = new bootstrap.Modal(modalEl);
            modal.show();
        }
        
        // ===== COMPACT DASHBOARD FUNCTIONS =====
        
        // Toggle Queue Sidebar
        function toggleQueueSidebar() {
            const sidebar = document.getElementById('queueSidebar');
            const toggleBtn = document.getElementById('queueToggleBtn');
            
            if (sidebar.classList.contains('open')) {
                sidebar.classList.remove('open');
                toggleBtn.classList.remove('active');
            } else {
                sidebar.classList.add('open');
                toggleBtn.classList.add('active');
                // Refresh queue when opening
                refreshHeadJudgeQueuePanel();
            }
        }
        
        // Toggle Floating Action Button Menu
        function toggleFAB() {
            const fab = document.getElementById('headJudgeFAB');
            fab.classList.toggle('open');
        }
        
        // Toggle Filter Panel
        function toggleFilters() {
            const panel = document.getElementById('filterPanel');
            const btn = document.getElementById('filterToggleBtn');
            const icon = btn.querySelector('.fa-chevron-down');
            
            if (panel.classList.contains('collapsed')) {
                panel.classList.remove('collapsed');
                panel.classList.add('expanded');
                icon.style.transform = 'rotate(180deg)';
            } else {
                panel.classList.add('collapsed');
                panel.classList.remove('expanded');
                icon.style.transform = 'rotate(0deg)';
            }
        }
        
        // Update top bar metrics
        function updateTopBarMetrics(data) {
            const metricTotal = document.getElementById('metricTotal');
            const metricPending = document.getElementById('metricPending');
            const metricApproved = document.getElementById('metricApproved');
            
            // Debug logging
            console.log('updateTopBarMetrics called', {
                metricTotal: metricTotal ? 'found' : 'MISSING',
                metricPending: metricPending ? 'found' : 'MISSING', 
                metricApproved: metricApproved ? 'found' : 'MISSING',
                hasData: !!data,
                hasParticipants: !!(data && data.participants)
            });
            
            // If no data, show placeholder dashes
            if (!data || !data.participants) {
                if (metricTotal) metricTotal.textContent = '--';
                if (metricPending) metricPending.textContent = '--';
                if (metricApproved) metricApproved.textContent = '--';
                return;
            }
            
            let pendingCount = 0;
            let approvedCount = 0;
            
            data.participants.forEach(p => {
                const approval = getApprovalStatus(p, data.judges, data.scores, data.run_number);
                if (approval === 'to_approve') pendingCount++;
                else if (approval === 'all_approved') approvedCount++;
            });
            
            console.log('Updating metrics:', {
                total: data.participants.length,
                pending: pendingCount,
                approved: approvedCount
            });
            
            // Update only the digit values - no DOM manipulation of container
            if (metricTotal && metricTotal.textContent !== String(data.participants.length)) {
                metricTotal.textContent = data.participants.length;
            }
            if (metricPending && metricPending.textContent !== String(pendingCount)) {
                metricPending.textContent = pendingCount;
            }
            if (metricApproved && metricApproved.textContent !== String(approvedCount)) {
                metricApproved.textContent = approvedCount;
            }
        }
        
         let autoRefreshInterval;
        let currentData = null;
        let currentSortMode = 'default';
        let currentFilters = {
            gender: '',
            category: '',
            bibName: '',
            showToApprove: true,
            showAllApproved: true,
            showWaiting: true
        };
        let heatCardLayout = 'judges';
        let currentHeatSettings = {};
        let headJudgeQueueLoading = false;
        const HEAD_JUDGE_ALERT_POLL_INTERVAL = 15000;
        let heatNotifications = [];
        let latestNotificationId = 0;
        let heatNotificationsTimer = null;
        let heatNotificationsInitialized = false;
        let notificationsLoading = false;
    let lastHeatsLoadPromise = null;
        
        // Judge Status Variables
        let currentJudgeStatus = 'ready';
        
        // Judge Status Functions
        function setJudgeStatus(status) {
            // Update current status
            currentJudgeStatus = status;
            
            // Update UI
            updateJudgeStatusUI(status);
            
            // Save to localStorage
            saveJudgeStatus(status);
            
            // Send to server
            sendJudgeStatusToServer(status);
            
            // Collapse the status options
            const collapseElement = document.getElementById('judgeStatusCollapse');
            if (collapseElement && collapseElement.classList.contains('show')) {
                const bsCollapse = bootstrap.Collapse.getInstance(collapseElement);
                if (bsCollapse) {
                    bsCollapse.hide();
                }
            }
            
            // Show confirmation
            const statusLabels = {
                'ready': '✓ Ready',
                'pending': '⏱ Pending',
                'danger': '⚠ Danger',
                'warning': '⚡ Warning',
                'info': 'ℹ Info',
                'off': '⏻ Off'
            };
            showSuccessToast('Status changed to: ' + statusLabels[status]);
        }
        
        function handleAutoReadyToggle() {
            const autoReadyToggle = document.getElementById('autoReadyToggle');
            const isEnabled = autoReadyToggle.checked;
            
            // Save to localStorage
            localStorage.setItem('headJudgeAutoReady', isEnabled ? 'true' : 'false');
            
            // If enabled and currently pending, automatically switch to ready
            if (isEnabled && currentJudgeStatus === 'pending') {
                setJudgeStatus('ready');
            }
            
            console.log('Head Judge Auto Ready:', isEnabled ? 'Enabled' : 'Disabled');
        }
        
        function loadAutoReadyPreference() {
            const autoReadyToggle = document.getElementById('autoReadyToggle');
            const savedPreference = localStorage.getItem('headJudgeAutoReady');
            
            if (savedPreference === 'true') {
                autoReadyToggle.checked = true;
            } else {
                autoReadyToggle.checked = false;
            }
        }
        
        function updateJudgeStatusUI(status) {
            const statusCard = document.getElementById('judge-status-card');
            const statusHeader = document.getElementById('judgeStatusHeader');
            const statusIcon = document.getElementById('judgeStatusIcon');
            const statusText = document.getElementById('judgeStatusText');
            const statusActions = document.getElementById('judgeStatusActions');
            
            // Status configurations
            const configs = {
                'ready': { 
                    bg: 'bg-success-subtle', 
                    icon: 'fa-check-circle text-success', 
                    text: 'Status: Ready',
                    action: 'expand'
                },
                'pending': { 
                    bg: 'bg-warning-subtle', 
                    icon: 'fa-clock text-warning', 
                    text: 'Status: Pending',
                    action: 'ready'
                },
                'danger': { 
                    bg: 'bg-danger-subtle', 
                    icon: 'fa-exclamation-triangle text-danger', 
                    text: 'Status: Danger',
                    action: 'expand'
                },
                'warning': { 
                    bg: 'bg-warning-subtle', 
                    icon: 'fa-exclamation-circle text-warning', 
                    text: 'Status: Warning',
                    action: 'expand'
                },
                'info': { 
                    bg: 'bg-info-subtle', 
                    icon: 'fa-info-circle text-info', 
                    text: 'Status: Info',
                    action: 'expand'
                },
                'off': { 
                    bg: 'bg-secondary-subtle', 
                    icon: 'fa-power-off text-secondary', 
                    text: 'Status: Off',
                    action: 'expand'
                }
            };
            
            const config = configs[status];
            
            // Remove all bg classes
            statusHeader.classList.remove('bg-success-subtle', 'bg-warning-subtle', 'bg-danger-subtle', 'bg-info-subtle', 'bg-secondary-subtle');
            statusHeader.classList.add(config.bg);
            
            // Special handling for pending status - bigger, more prominent, sticky
            if (status === 'pending') {
                // Make card sticky
                statusCard.classList.add('sticky-top', 'position-sticky');
                statusCard.style.top = '80px';
                statusCard.style.zIndex = '1000';
                
                // Enhanced pending state with bigger card
                statusCard.classList.add('border-warning', 'shadow');
                statusHeader.classList.remove('p-2');
                statusHeader.classList.add('p-4');
                
                // Update icon and text with larger styling
                statusIcon.className = 'fas fa-clock text-warning me-3';
                statusIcon.style.fontSize = '2rem';
                statusText.innerHTML = '<div style="font-size: 1.5rem; font-weight: bold;">Ready for next heat?</div>';
                statusText.style.fontSize = '';
                
                // Show prominent Ready button and expand option
                statusActions.innerHTML = `
                    <div class="d-flex gap-2">
                        <button type="button" class="btn btn-success btn-lg px-4" onclick="setJudgeStatus('ready')">
                            <i class="fas fa-check me-2"></i>READY
                        </button>
                        <button type="button" class="btn btn-outline-secondary btn-lg" data-bs-toggle="collapse" data-bs-target="#judgeStatusCollapse" title="Other status options">
                            <i class="fas fa-ellipsis-h"></i>
                        </button>
                    </div>
                `;
            } else {
                // Remove sticky positioning
                statusCard.classList.remove('sticky-top', 'position-sticky');
                statusCard.style.top = '';
                statusCard.style.zIndex = '';
                
                // Normal styling for other statuses
                statusCard.classList.remove('border-warning', 'shadow');
                statusHeader.classList.remove('p-4');
                statusHeader.classList.add('p-2');
                
                // Update icon with normal size
                statusIcon.className = 'fas ' + config.icon + ' me-2';
                statusIcon.style.fontSize = '';
                
                // Update text with normal styling
                statusText.textContent = config.text;
                statusText.style.fontSize = '';
                
                // Show expand button for other statuses
                statusActions.innerHTML = '<button type="button" class="btn btn-outline-secondary bg-white btn-sm"><i class="fas fa-ellipsis-v"></i></button>';
            }
        }
        
        function toggleJudgeStatusCollapse() {
            const collapseElement = document.getElementById('judgeStatusCollapse');
            if (collapseElement) {
                const bsCollapse = bootstrap.Collapse.getOrCreateInstance(collapseElement);
                bsCollapse.toggle();
            }
        }
        
        function saveJudgeStatus(status) {
            const eventSelect = document.getElementById('eventSelect');
            const eventId = eventSelect ? eventSelect.value : null;
            const judgeId = <?= json_encode($judge_id) ?>;
            
            if (!eventId) return;
            
            const key = 'head_judge_status_' + eventId + '_' + judgeId;
            localStorage.setItem(key, status);
        }
        
        function loadJudgeStatus() {
            const eventSelect = document.getElementById('eventSelect');
            const eventId = eventSelect ? eventSelect.value : null;
            const judgeId = <?= json_encode($judge_id) ?>;
            
            if (!eventId) return;
            
            const key = 'head_judge_status_' + eventId + '_' + judgeId;
            const savedStatus = localStorage.getItem(key) || 'ready';
            
            currentJudgeStatus = savedStatus;
            updateJudgeStatusUI(savedStatus);
            
            // Load Auto Ready preference
            loadAutoReadyPreference();
        }
        
        function sendJudgeStatusToServer(status) {
            const eventSelect = document.getElementById('eventSelect');
            const eventId = eventSelect ? eventSelect.value : null;
            const judgeId = <?= json_encode($judge_id) ?>;
            
            if (!eventId) return;
            
            // Get current heat and run selection
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            const heatNumber = heatSelect ? parseInt(heatSelect.value) : null;
            const runNumber = runSelect ? parseInt(runSelect.value) : null;
            
            // Send status update to server
            fetch('../api/judge_status_api.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    event_id: eventId,
                    judge_id: judgeId,
                    status: status,
                    heat_number: heatNumber,
                    run_number: runNumber,
                    timestamp: new Date().toISOString()
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Head Judge status updated on server:', status);
                } else {
                    console.error('Failed to update head judge status:', data.error);
                }
            })
            .catch(error => {
                console.error('Error sending head judge status:', error);
            });
        }
        
        // Poll for judge status updates
        let judgeStatusPollInterval = null;
        
        function startJudgeStatusPolling() {
            // Stop any existing polling
            stopJudgeStatusPolling();
            
            // Poll every 3 seconds
            judgeStatusPollInterval = setInterval(() => {
                checkJudgeStatusUpdate();
            }, 3000);
            
            // Also check immediately
            checkJudgeStatusUpdate();
        }
        
        function stopJudgeStatusPolling() {
            if (judgeStatusPollInterval) {
                clearInterval(judgeStatusPollInterval);
                judgeStatusPollInterval = null;
            }
        }
        
        function checkJudgeStatusUpdate() {
            const eventSelect = document.getElementById('eventSelect');
            const eventId = eventSelect ? eventSelect.value : null;
            const judgeId = <?= json_encode($judge_id) ?>;
            
            if (!eventId) return;
            
            // Fetch current status from server
            fetch(`../api/judge_status_api.php?event_id=${eventId}&judge_id=${judgeId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.data) {
                        const serverStatus = data.data.status || 'ready';
                        
                        // Only update if status has changed
                        if (serverStatus !== currentJudgeStatus) {
                            console.log('Status changed from', currentJudgeStatus, 'to', serverStatus);
                            currentJudgeStatus = serverStatus;
                            updateJudgeStatusUI(serverStatus);
                            saveJudgeStatus(serverStatus);
                            
                            // Show notification about status change
                            const statusLabels = {
                                'ready': 'Ready',
                                'pending': 'Pending',
                                'info': 'Info',
                                'warning': 'Warning',
                                'danger': 'Danger',
                                'off': 'Off'
                            };
                            showSuccessToast('Your status was updated to: ' + statusLabels[serverStatus]);
                        }
                    }
                })
                .catch(error => {
                    console.error('Error polling judge status:', error);
                });
        }
        
        function setAllJudgesStatus(status) {
            const eventSelect = document.getElementById('eventSelect');
            const eventId = eventSelect ? eventSelect.value : null;
            
            if (!eventId) {
                showSuccessToast('Please select an event first');
                return;
            }
            
            // Prepare confirmation with user
            const statusLabels = {
                'ready': 'Ready',
                'pending': 'Pending',
                'info': 'Info',
                'warning': 'Warning',
                'danger': 'Danger',
                'off': 'Off'
            };
            
            // Show Bootstrap confirmation modal
            const confirmMessage = `Set status to "<strong>${statusLabels[status]}</strong>" for all judges in this event?`;
            document.getElementById('setAllJudgesMessage').innerHTML = confirmMessage;
            
            const confirmModal = new bootstrap.Modal(document.getElementById('setAllJudgesConfirmModal'));
            confirmModal.show();
            
            // Remove any existing event listeners to prevent duplicates
            const confirmBtn = document.getElementById('confirmSetAllJudgesBtn');
            const newConfirmBtn = confirmBtn.cloneNode(true);
            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
            
            // Add click handler for confirmation
            newConfirmBtn.addEventListener('click', function() {
                // Close the confirmation modal
                confirmModal.hide();
                
                // Get current heat and run selection
                const heatSelect = document.getElementById('heatSelect');
                const runSelect = document.getElementById('runSelect');
                const heatNumber = heatSelect ? parseInt(heatSelect.value) : null;
                const runNumber = runSelect ? parseInt(runSelect.value) : null;
                
                // Send request to server
                fetch('../api/judge_status_api.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'set_all_judges',
                        event_id: parseInt(eventId),
                        status: status,
                        heat_number: heatNumber,
                        run_number: runNumber,
                        timestamp: new Date().toISOString()
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const count = data.updated_count || 0;
                        showSuccessToast(`Status updated for ${count} judge(s) to: ${statusLabels[status]}`);
                        console.log('All judges status updated:', data);
                        
                        // Update head judge's own status display
                        currentJudgeStatus = status;
                        updateJudgeStatusUI(status);
                        saveJudgeStatus(status);
                        
                        // Close the collapse panel
                        toggleJudgeStatusCollapse();
                    } else {
                        showErrorAlert('Failed to update judges status: ' + (data.error || 'Unknown error'));
                        console.error('Failed to update all judges status:', data);
                    }
                })
                .catch(error => {
                    showErrorAlert('Error updating judges status. Please try again.');
                    console.error('Error setting all judges status:', error);
                });
            });
        }
        
        // Helper function to show Bootstrap error alert modal
        function showErrorAlert(message) {
            document.getElementById('errorAlertMessage').textContent = message;
            const errorModal = new bootstrap.Modal(document.getElementById('errorAlertModal'));
            errorModal.show();
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            const eventSelect = document.getElementById('eventSelect');
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            const refreshBtn = document.getElementById('refreshBtn');
            const autoRefreshCheck = document.getElementById('autoRefresh');
            
            // Initialize Event Selection Manager for this page
            if (window.EventSelectionManager) {
                window.EventSelectionManager.init();
            }
            
            // Initialize judge status on page load
            loadJudgeStatus();
            
            // Start polling for judge status updates
            startJudgeStatusPolling();
            
            // Close FAB menu when clicking outside
            document.addEventListener('click', function(e) {
                const fab = document.getElementById('headJudgeFAB');
                if (!fab.contains(e.target) && fab.classList.contains('open')) {
                    fab.classList.remove('open');
                }
            });
            
            // Sort button handlers - using event delegation to persist through DOM updates
            document.addEventListener('click', function(e) {
                const sortBtn = e.target.closest('.btn-outline-primary[data-sort]');
                if (sortBtn && !sortBtn.classList.contains('disabled')) {
                    // Update active state
                    document.querySelectorAll('.btn-outline-primary[data-sort]').forEach(b => b.classList.remove('active'));
                    sortBtn.classList.add('active');
                    
                    // Update sort mode and re-render
                    currentSortMode = sortBtn.getAttribute('data-sort');
                    if (currentData) {
                        renderScoreGrid(currentData);
                    }
                }
            });

            // Filter handlers
            document.getElementById('genderFilter').addEventListener('change', function() {
                currentFilters.gender = this.value;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('categoryFilter').addEventListener('change', function() {
                currentFilters.category = this.value;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('bibNameFilter').addEventListener('input', function() {
                currentFilters.bibName = this.value;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('showToApprove').addEventListener('change', function() {
                currentFilters.showToApprove = this.checked;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('showAllApproved').addEventListener('change', function() {
                currentFilters.showAllApproved = this.checked;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('showWaiting').addEventListener('change', function() {
                currentFilters.showWaiting = this.checked;
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });

            document.getElementById('clearFilters').addEventListener('click', function() {
                // Reset filters
                currentFilters = {
                    gender: '',
                    category: '',
                    bibName: '',
                    showToApprove: true,
                    showAllApproved: true,
                    showWaiting: true
                };
                
                // Reset form controls
                document.getElementById('genderFilter').value = '';
                document.getElementById('categoryFilter').value = '';
                document.getElementById('bibNameFilter').value = '';
                document.getElementById('showToApprove').checked = true;
                document.getElementById('showAllApproved').checked = true;
                document.getElementById('showWaiting').checked = true;
                
                // Re-render
                if (currentData) {
                    renderScoreGrid(currentData);
                }
            });
            
            // Event selection handler - modified to work with localStorage
            eventSelect.addEventListener('change', function() {
                const eventId = this.value;
                heatSelect.disabled = !eventId;
                runSelect.disabled = true;
                runSelect.innerHTML = '<option value="">Select Run</option>';
                
                if (eventId) {
                    lastHeatsLoadPromise = loadHeats(eventId)
                        .catch(error => {
                            lastHeatsLoadPromise = null;
                            throw error;
                        });
                    loadEventCategories(eventId); // Load event-specific categories
                    loadJudgeStatus(); // Load judge status for this event
                    startJudgeStatusPolling(); // Restart polling for new event
                    
                    if (isAutoHeatRunEnabled()) {
                        lastHeatsLoadPromise
                            .then(() => autoSelectActiveHeatRun())
                            .catch(error => console.error('Auto-select failed after loading heats:', error));
                    } else {
                        loadScores(); // Load scores even without heat/run selection
                    }
                } else {
                    lastHeatsLoadPromise = null;
                    heatSelect.innerHTML = '<option value="">Select Heat</option>';
                    runSelect.innerHTML = '<option value="">Select Run</option>';
                    document.getElementById('categoryFilter').innerHTML = '<option value="">All Categories</option>';
                    clearScoreGrid();
                }
            });
            
            // Heat selection handler - modified to work with localStorage
            heatSelect.addEventListener('change', function() {
                const heatNumber = this.value;
                runSelect.disabled = !heatNumber;
                
                if (heatNumber) {
                    loadRuns();
                    loadScores(); // Load scores for selected heat (all runs)
                } else {
                    runSelect.innerHTML = '<option value="">Select Run</option>';
                    loadScores(); // Load all heats/runs when heat is deselected
                }
            });
            
            // Run selection handler - modified to work with localStorage
            runSelect.addEventListener('change', function() {
                loadScores(); // Always load scores, even if run is deselected
                
                // Enable/disable random scores button
                const generateBtn = document.getElementById('generateRandomScoresBtn');
                if (generateBtn) {
                    const eventId = document.getElementById('eventSelect').value;
                    const heatNumber = document.getElementById('heatSelect').value;
                    const runNumber = this.value;
                    generateBtn.disabled = !(eventId && heatNumber && runNumber);
                }
            });
            
            // Refresh button
            refreshBtn.addEventListener('click', function() {
                const eventId = document.getElementById('eventSelect').value;
                if (eventId) {
                    loadScores();
                }
            });
            
            // Clear selections button
            document.getElementById('clearSelectionsBtn').addEventListener('click', function() {
                if (window.EventSelectionManager) {
                    window.EventSelectionManager.clearAllSelections();
                    
                    // Reset form elements
                    eventSelect.value = '';
                    heatSelect.innerHTML = '<option value="">Select Heat</option>';
                    heatSelect.disabled = true;
                    runSelect.innerHTML = '<option value="">Select Run</option>';
                    runSelect.disabled = true;
                    document.getElementById('categoryFilter').innerHTML = '<option value="">All Categories</option>';
                    clearScoreGrid();
                    
                    // Show visual feedback
                    const btn = this;
                    const originalHTML = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-check text-success"></i>';
                    btn.classList.add('btn-success');
                    btn.classList.remove('btn-outline-secondary');
                    
                    // Initialize Lucide icons for the button feedback
                    if (window.initializeLucideIcons) {
                        window.initializeLucideIcons();
                    }
                    
                    setTimeout(() => {
                        btn.innerHTML = originalHTML;
                        btn.classList.remove('btn-success');
                        btn.classList.add('btn-outline-secondary');
                        
                        // Initialize Lucide icons when restoring original content
                        if (window.initializeLucideIcons) {
                            window.initializeLucideIcons();
                        }
                    }, 1500);
                }
            });
            
            // Generate random scores button (Admin only)
            const generateRandomScoresBtn = document.getElementById('generateRandomScoresBtn');
            if (generateRandomScoresBtn) {
                generateRandomScoresBtn.addEventListener('click', function() {
                    const eventId = document.getElementById('eventSelect').value;
                    const heatNumber = document.getElementById('heatSelect').value;
                    const runNumber = document.getElementById('runSelect').value;
                    
                    if (!eventId || !heatNumber || !runNumber) {
                        alert('Please select Event, Heat, and Run first');
                        return;
                    }
                    
                    if (!confirm(`Generate random test scores for Heat ${heatNumber}, Run ${runNumber}?\\n\\nThis will create scores for all participants and all judges. Existing scores will not be overwritten.`)) {
                        return;
                    }
                    
                    const btn = this;
                    const originalHTML = btn.innerHTML;
                    btn.disabled = true;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Generating...';
                    
                    fetch('head_judge.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            action: 'generate_random_scores',
                            event_id: eventId,
                            heat_number: heatNumber,
                            run_number: runNumber
                        })
                    })
                    .then(response => {
                        console.log('Response status:', response.status);
                        console.log('Response headers:', response.headers);
                        return response.text();
                    })
                    .then(text => {
                        console.log('Raw response:', text);
                        console.log('Response length:', text.length);
                        console.log('First 500 chars:', text.substring(0, 500));
                        
                        try {
                            const data = JSON.parse(text);
                            if (data.success) {
                                alert(`Success!\\n\\n${data.message}\\nParticipants: ${data.participants}\\nJudges: ${data.judges}`);
                                loadScores(); // Refresh the score grid
                            } else {
                                alert('Error: ' + data.message);
                            }
                        } catch (e) {
                            console.error('JSON Parse Error:', e);
                            console.error('Failed to parse response as JSON');
                            alert('Server returned invalid response. Check console for details.');
                        }
                    })
                    .catch(error => {
                        console.error('Fetch Error:', error);
                        alert('Failed to generate scores: ' + error.message);
                    })
                    .finally(() => {
                        btn.disabled = false;
                        btn.innerHTML = originalHTML;
                    });
                });
            }
            
           
            // Auto-refresh toggle
            autoRefreshCheck.addEventListener('change', function() {
                if (this.checked) {
                    startAutoRefresh();
                } else {
                    stopAutoRefresh();
                }
            });
            
            // Auto Heat/Run toggle
            const autoHeatRunCheck = document.getElementById('autoHeatRun');
            
            // Load saved state from localStorage
            const savedAutoHeatRun = localStorage.getItem('headJudge_autoHeatRun');
            if (savedAutoHeatRun === 'true') {
                autoHeatRunCheck.checked = true;
            }
            
            // Save state on change
            autoHeatRunCheck.addEventListener('change', function() {
                localStorage.setItem('headJudge_autoHeatRun', this.checked);
                
                // If enabled and event is selected, try to auto-select
                if (this.checked && eventSelect.value) {
                    ensureHeatsLoadedForEvent(eventSelect.value)
                        .then(() => autoSelectActiveHeatRun())
                        .catch(error => console.error('Unable to load heats before auto-select toggle:', error));
                } else if (!this.checked && eventSelect.value) {
                    loadScores();
                }
            });
            
            // Start auto-refresh by default
            startAutoRefresh();
            
            // If Auto Heat/Run is enabled on page load, auto-select after a short delay
            if (autoHeatRunCheck.checked && eventSelect.value) {
                ensureHeatsLoadedForEvent(eventSelect.value)
                    .then(() => autoSelectActiveHeatRun())
                    .catch(error => console.warn('Initial auto-select could not verify heats yet:', error));
            }
        });

        function isAutoHeatRunEnabled() {
            const autoHeatRunCheck = document.getElementById('autoHeatRun');
            return !!(autoHeatRunCheck && autoHeatRunCheck.checked);
        }

        function ensureHeatsLoadedForEvent(eventId) {
            if (!eventId) {
                return Promise.resolve();
            }

            if (!lastHeatsLoadPromise) {
                lastHeatsLoadPromise = loadHeats(eventId);
            }

            return lastHeatsLoadPromise.catch(error => {
                console.error('Failed to load heats for event', eventId, error);
                lastHeatsLoadPromise = null;
                throw error;
            });
        }

        // Function to auto-select active heat and run
        function autoSelectActiveHeatRun() {
            const eventId = document.getElementById('eventSelect').value;
            if (!eventId) {
                return Promise.resolve(false);
            }

            return ensureHeatsLoadedForEvent(eventId)
                .catch(error => {
                    console.warn('Proceeding with auto-select even though heats failed to load.', error);
                })
                .then(() => fetch('get_active_heat_run.php?event_id=' + eventId))
                .then(response => response.json())
                .then(data => {
                    if (!data.success) {
                        return false;
                    }

                    const heatSelect = document.getElementById('heatSelect');
                    const runSelect = document.getElementById('runSelect');

                    if (!heatSelect) {
                        return false;
                    }

                    // Ensure the desired heat exists; if not, log and exit gracefully
                    const matchingHeat = heatSelect.querySelector(`option[value="${data.heat_number}"]`);
                    if (!matchingHeat) {
                        console.warn('Active heat not found in dropdown options.', data.heat_number, data);
                        return false;
                    }

                    heatSelect.value = data.heat_number;
                    heatSelect.dispatchEvent(new Event('change'));
                    if (window.EventSelectionManager) {
                        window.EventSelectionManager.saveSelection('heat', String(data.heat_number));
                    }

                    if (runSelect && data.run_number) {
                        setTimeout(() => {
                            runSelect.value = data.run_number;
                            runSelect.dispatchEvent(new Event('change'));
                            if (window.EventSelectionManager) {
                                window.EventSelectionManager.saveSelection('run', String(data.run_number));
                            }
                        }, 300);
                    }

                    showAutoSelectNotification(data.heat_number, data.run_number);
                    return true;
                })
                .catch(error => {
                    console.error('Error fetching active heat/run:', error);
                    return false;
                });
        }
        
        // Function to show auto-select notification
        function showAutoSelectNotification(heatNumber, runNumber) {
            const toast = document.createElement('div');
            toast.className = 'position-fixed top-0 start-50 translate-middle-x mt-3 alert alert-info alert-dismissible fade show';
            toast.style.zIndex = '9999';
            toast.innerHTML = `
                <i class="fas fa-bolt me-2"></i>
                Auto-selected: Heat ${heatNumber}${runNumber ? ', Run ' + runNumber : ''}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(toast);
            
            // Initialize Lucide icons
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons();
            }
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.remove();
                }
            }, 3000);
        }

         function updateCategoryFilter(participants) {
            const categoryFilter = document.getElementById('categoryFilter');
            const currentValue = categoryFilter.value;
            
            // Get unique categories from participants
            const categories = [...new Set(participants.map(p => p.category))].filter(c => c).sort();
            
            // Rebuild options
            categoryFilter.innerHTML = '<option value="">All Categories</option>';
            categories.forEach(category => {
                const option = document.createElement('option');
                option.value = category;
                option.textContent = category;
                if (category === currentValue) {
                    option.selected = true;
                }
                categoryFilter.appendChild(option);
            });
        }

        // Add function to load event categories
        function loadEventCategories(eventId) {
            if (!eventId) {
                document.getElementById('categoryFilter').innerHTML = '<option value="">All Categories</option>';
                return;
            }
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_event_categories');
            formData.append('event_id', eventId);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const categoryFilter = document.getElementById('categoryFilter');
                    const currentValue = categoryFilter.value;
                    
                    categoryFilter.innerHTML = '<option value="">All Categories</option>';
                    data.categories.forEach(category => {
                        const option = document.createElement('option');
                        option.value = category;
                        option.textContent = category;
                        if (category === currentValue) {
                            option.selected = true;
                        }
                        categoryFilter.appendChild(option);
                    });
                }
            })
            .catch(error => {
                console.error('Error loading categories:', error);
            });
        }

        function getApprovalStatus(participant, judges, scores, targetRunNumber = null) {
            let hasAnyScores = false;
            let hasPendingScores = false;
            let hasAllScores = true;
            
            const participantScores = scores[participant.id] || {};
            
            if (targetRunNumber) {
                // Specific run - check only that run
                const runScores = participantScores[targetRunNumber] || {};
                judges.forEach(judge => {
                    const score = runScores[judge.id];
                    if (score) {
                        hasAnyScores = true;
                        if (score.is_approved == 0) {
                            hasPendingScores = true;
                        }
                    } else {
                        hasAllScores = false;
                    }
                });
            } else {
                // No specific run - check all runs
                Object.keys(participantScores).forEach(runNum => {
                    const runScores = participantScores[runNum] || {};
                    judges.forEach(judge => {
                        const score = runScores[judge.id];
                        if (score) {
                            hasAnyScores = true;
                            if (score.is_approved == 0) {
                                hasPendingScores = true;
                            }
                        }
                    });
                });
                // For multi-run, we don't require all scores
                hasAllScores = hasAnyScores;
            }
            
            if (!hasAnyScores) {
                return 'waiting';
            } else if (hasPendingScores) {
                return 'to_approve';
            } else if (hasAnyScores && !hasPendingScores) {
                return 'all_approved';
            }
            return 'waiting';
        }

        function passesFilters(participant, approvalStatus) {
            // Gender filter
            if (currentFilters.gender && participant.gender !== currentFilters.gender) {
                return false;
            }
            
            // Category filter
            if (currentFilters.category && participant.category !== currentFilters.category) {
                return false;
            }
            
            // BIB/Name filter
            if (currentFilters.bibName) {
                const searchTerm = currentFilters.bibName.toLowerCase();
                const bibMatch = participant.bib_number.toString().includes(searchTerm);
                const nameMatch = (participant.first_name + ' ' + participant.last_name).toLowerCase().includes(searchTerm);
                if (!bibMatch && !nameMatch) {
                    return false;
                }
            }
            
            // Approval status filters
            if (approvalStatus === 'to_approve' && !currentFilters.showToApprove) {
                return false;
            }
            if (approvalStatus === 'all_approved' && !currentFilters.showAllApproved) {
                return false;
            }
            if (approvalStatus === 'waiting' && !currentFilters.showWaiting) {
                return false;
            }
            
            return true;
        }
        
        function loadHeats(eventId) {
            const heatSelect = document.getElementById('heatSelect');
            if (heatSelect) {
                heatSelect.disabled = true;
                heatSelect.innerHTML = '<option value="">Select Heat</option>';
            }

            return fetch('get_event_heats.php?event_id=' + eventId)
                .then(response => response.json())
                .then(data => {
                    if (!heatSelect) {
                        return data;
                    }

                    heatSelect.innerHTML = '<option value="">Select Heat</option>';
                    
                    data.forEach(heat => {
                        const option = document.createElement('option');
                        option.value = heat.heat_number;
                        option.textContent = `Heat ${heat.heat_number}${heat.heat_name ? ' - ' + heat.heat_name : ''}`;
                        heatSelect.appendChild(option);
                    });

                    heatSelect.disabled = false;
                    
                    // Restore heat selection only when auto mode is off
                    if (window.EventSelectionManager && !isAutoHeatRunEnabled()) {
                        setTimeout(() => {
                            window.EventSelectionManager.restoreHeatSelection();
                        }, 100);
                    }

                    return data;
                })
                .catch(error => {
                    console.error('Error loading heats:', error);
                    throw error;
                });
        }
        
        function loadRuns() {
            const runSelect = document.getElementById('runSelect');
            runSelect.innerHTML = '<option value="">Select Run</option>';
            
            for (let i = 1; i <= 3; i++) {
                const option = document.createElement('option');
                option.value = i;
                option.textContent = `Run ${i}`;
                runSelect.appendChild(option);
            }
            
            // Restore run selection if available
            if (window.EventSelectionManager && !isAutoHeatRunEnabled()) {
                setTimeout(() => {
                    window.EventSelectionManager.restoreRunSelection();
                }, 100);
            }
        }
        
        function loadScores() {
            const eventId = document.getElementById('eventSelect').value;
            const heatNumber = document.getElementById('heatSelect').value;
            const runNumber = document.getElementById('runSelect').value;
            const showResultsBtn = document.getElementById('showResultsBtn');
            
            // If no event selected, clear the grid and hide Results button
            if (!eventId) {
                clearScoreGrid();
                hideActiveHeatCard();
                if (showResultsBtn) {
                    showResultsBtn.style.display = 'none';
                }
                return;
            }
            
            // Show Results button when event is selected
            if (showResultsBtn) {
                showResultsBtn.style.display = 'inline-block';
            }
            
            showLoading(true);
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_scores');
            formData.append('event_id', eventId);
            
            // Only append heat/run if they are selected
            if (heatNumber) {
                formData.append('heat_number', heatNumber);
            }
            if (runNumber) {
                formData.append('run_number', runNumber);
            }
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    currentData = data;
                    renderScoreGrid(data);
                    updateTopBarMetrics(data);
                    // Load active heat card if event is selected
                    loadActiveHeatCardInHeadJudge();
                    // Initialize queue panel monitoring
                    refreshHeadJudgeQueuePanel();
                    initializeHeadJudgeNotifications();
                } else {
                    showErrorAlert('Error loading scores: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
            })
            .finally(() => {
                showLoading(false);
            });
        }
        
        // Active Heat Card Functions
        function getActiveHeatPlaceholderHTML() {
            return `
                <div class="col-12 text-center text-muted py-3">
                    <i class="fas fa-fire fa-2x mb-2 opacity-25"></i>
                    <p class="mb-0 small">Loading active heat...</p>
                </div>
            `;
        }

        function toggleHeadJudgeHeatLoadingOverlay(show) {
            const container = document.getElementById('activeHeatCardContainer');
            if (!container || !container.parentElement) return;
            const parent = container.parentElement;
            parent.classList.add('position-relative');
            let indicator = document.getElementById('activeHeatLoadingOverlay');
            if (!indicator) {
                indicator = document.createElement('div');
                indicator.id = 'activeHeatLoadingOverlay';
                indicator.className = 'heat-loading-indicator';
                indicator.innerHTML = '<i class="fas fa-sync-alt"></i>';
                parent.appendChild(indicator);
            }

            requestAnimationFrame(() => {
                if (show) {
                    indicator.classList.add('is-visible');
                } else {
                    indicator.classList.remove('is-visible');
                }
            });
        }

        /**
         * Switch heat card layout mode for head judge panel
         */
        function setHeadJudgeHeatCardLayout(layout, btn) {
            console.log('🎨 Switching heat card layout to:', layout);
            heatCardLayout = layout;
            
            // Update active button state
            document.querySelectorAll('[onclick^="setHeadJudgeHeatCardLayout"]').forEach(b => {
                b.classList.remove('active');
            });
            if (btn) {
                btn.classList.add('active');
            }
            
            // Reload heat card with new layout
            loadActiveHeatCardInHeadJudge();
        }
        
        async function loadActiveHeatCardInHeadJudge() {
            const eventId = document.getElementById('eventSelect').value;
            const heatNumber = document.getElementById('heatSelect').value;
            
            if (!eventId) {
                hideActiveHeatCard();
                return;
            }
            
            const container = document.getElementById('activeHeatCardContainer');
            const section = document.getElementById('activeHeatSection');
            
            // Show the section
            section.style.display = 'block';

            const placeholderHTML = getActiveHeatPlaceholderHTML();
            const hasExistingContent = container.dataset.hasContent === 'true' && container.children.length > 0;

            if (!hasExistingContent) {
                container.innerHTML = placeholderHTML;
            } else {
                toggleHeadJudgeHeatLoadingOverlay(true);
            }
            
            try {
                // Load heat cards from API with specified layout
                const response = await fetch(`../admin/heat_cards_api.php?event_id=${eventId}&layout=${heatCardLayout}`);
                const data = await response.json();
                
                if (!response.ok || !data.success) {
                    throw new Error(data.error || 'Failed to load heat cards');
                }
                
                // Parse the HTML and extract heat cards
                const parser = new DOMParser();
                const doc = parser.parseFromString(data.data, 'text/html');
                // Select all heat card containers (works for all layouts: col-md-4, col-12, etc.)
                const allCards = doc.querySelectorAll('[class*="col-"]');
                
                let activeCard = null;
                let foundHeat = null;
                
                // If specific heat number provided, find that heat
                if (heatNumber) {
                    allCards.forEach(card => {
                        const heatTitle = card.querySelector('.card-title');
                        if (heatTitle && heatTitle.textContent.includes(`Heat ${heatNumber}`)) {
                            activeCard = card;
                            foundHeat = heatNumber;
                        }
                    });
                }
                
                // If no specific heat or not found, look for active heat
                if (!activeCard) {
                    allCards.forEach(card => {
                        // Look for the ACTIVE badge (bg-warning text-dark)
                        const activeStatus = card.querySelector('.badge.bg-warning.text-dark');
                        if (activeStatus && activeStatus.textContent.includes('ACTIVE')) {
                            activeCard = card;
                            const heatTitle = card.querySelector('.card-title, h5');
                            if (heatTitle) {
                                const match = heatTitle.textContent.match(/Heat (\d+)/);
                                if (match) foundHeat = match[1];
                            }
                        }
                        
                        // Alternative: Check if card has active border class
                        if (!activeCard && card.querySelector('.card.border-success')) {
                            activeCard = card;
                            const heatTitle = card.querySelector('.card-title, h5');
                            if (heatTitle) {
                                const match = heatTitle.textContent.match(/Heat (\d+)/);
                                if (match) foundHeat = match[1];
                            }
                        }
                    });
                }
                
                if (activeCard) {
                    // Display the heat card
                    container.innerHTML = activeCard.outerHTML;
                    container.dataset.hasContent = 'true';
                    console.log('Loaded heat card for Heat:', foundHeat);
                } else {
                    container.innerHTML = `
                        <div class="col-12 text-center text-muted py-3">
                            <i class="fas fa-fire fa-2x mb-2 opacity-25"></i>
                            <p class="mb-0">No active heat found</p>
                            <p class="small text-muted">Activate a heat to see details here</p>
                        </div>
                    `;
                    container.dataset.hasContent = 'false';
                }
                
            } catch (error) {
                console.error('Error loading active heat card:', error);
                container.innerHTML = `
                    <div class="col-12">
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Error loading heat data:</strong> ${error.message}
                        </div>
                    </div>
                `;
                container.dataset.hasContent = 'false';
            }
            toggleHeadJudgeHeatLoadingOverlay(false);
        }
        
        function hideActiveHeatCard() {
            const section = document.getElementById('activeHeatSection');
            if (section) {
                section.style.display = 'none';
            }
            const container = document.getElementById('activeHeatCardContainer');
            if (container) {
                container.innerHTML = getActiveHeatPlaceholderHTML();
                container.dataset.hasContent = 'false';
            }
            toggleHeadJudgeHeatLoadingOverlay(false);
        }
        
        // Function to validate participant diversity rules
        function validateParticipantDiversityRules(participant, diversityRules, data) {
            if (!diversityRules || diversityRules.length === 0) {
                return { valid: true, errors: [] };
            }
            
            const errors = [];
            const figuresByCategory = {};
            
            // Collect all figures from all runs for this participant
            const participantScores = data.scores[participant.id] || {};
            Object.keys(participantScores).forEach(runNum => {
                Object.keys(participantScores[runNum]).forEach(judgeId => {
                    const score = participantScores[runNum][judgeId];
                    if (score && score.figures_json) {
                        try {
                            const figuresData = JSON.parse(score.figures_json);
                            const actualFigures = figuresData.figures || figuresData;
                            
                            // Collect figures by category
                            Object.keys(actualFigures).forEach(category => {
                                if (!figuresByCategory[category]) {
                                    figuresByCategory[category] = [];
                                }
                                const items = actualFigures[category];
                                if (Array.isArray(items)) {
                                    figuresByCategory[category].push(...items);
                                }
                            });
                        } catch (e) {
                            // Skip invalid JSON
                        }
                    }
                });
            });
            
            // Validate against each rule
            diversityRules.forEach(rule => {
                const ruleType = rule.rule_type || '';
                const groupName = rule.group_name || '';
                const minDistinct = parseInt(rule.min_distinct || 0);
                const isOptional = rule.is_optional == 1;
                
                // Skip optional rules
                if (isOptional) {
                    return;
                }
                
                // Check for requireDistinct rule type
                if ((ruleType === 'requireDistinct' || ruleType === 'min_distinct') && minDistinct > 0) {
                    const items = figuresByCategory[groupName] || [];
                    const uniqueCount = new Set(items).size;
                    
                    if (uniqueCount < minDistinct) {
                        errors.push('Requires at least ' + minDistinct + ' different ' + groupName + ' figures (has ' + uniqueCount + ')');
                    }
                }
            });
            
            return {
                valid: errors.length === 0,
                errors: errors
            };
        }
        
        // Function to calculate run score with drop rules
        function calculateRunScoreWithDrops(judgeScores, dropRule) {
            if (judgeScores.length === 0) {
                return { score: 0, dropInfo: null };
            }
            
            // Initialize drop info
            let dropInfo = {
                droppedHighest: null,
                droppedLowest: null,
                method: 'none'
            };
            
            // Only apply drop rules if drop_rule is not 'none' and we have enough judges
            if (dropRule !== 'none' && judgeScores.length >= 3) {
                // Apply drop rule based on the format configuration
                const sortedScores = [...judgeScores].sort((a, b) => a.score - b.score);
                
                if (dropRule === 'highest_and_lowest') {
                    const droppedLowest = sortedScores[0];
                    const droppedHighest = sortedScores[sortedScores.length - 1];
                    const usedScores = sortedScores.slice(1, -1);
                    
                    dropInfo = {
                        droppedHighest: droppedHighest,
                        droppedLowest: droppedLowest,
                        method: 'highest_and_lowest'
                    };
                    
                    // Calculate average of remaining scores
                    const runScore = usedScores.reduce((sum, item) => sum + item.score, 0) / usedScores.length;
                    return { score: runScore, dropInfo: dropInfo };
                } else if (dropRule === 'lowest') {
                    const droppedLowest = sortedScores[0];
                    const usedScores = sortedScores.slice(1);
                    
                    dropInfo = {
                        droppedHighest: null,
                        droppedLowest: droppedLowest,
                        method: 'lowest'
                    };
                    
                    const runScore = usedScores.reduce((sum, item) => sum + item.score, 0) / usedScores.length;
                    return { score: runScore, dropInfo: dropInfo };
                } else if (dropRule === 'highest') {
                    const droppedHighest = sortedScores[sortedScores.length - 1];
                    const usedScores = sortedScores.slice(0, -1);
                    
                    dropInfo = {
                        droppedHighest: droppedHighest,
                        droppedLowest: null,
                        method: 'highest'
                    };
                    
                    const runScore = usedScores.reduce((sum, item) => sum + item.score, 0) / usedScores.length;
                    return { score: runScore, dropInfo: dropInfo };
                }
            }
            
            // If drop_rule is 'none' or not enough judges, use all scores
            const runScore = judgeScores.reduce((sum, item) => sum + item.score, 0) / judgeScores.length;
            return { score: runScore, dropInfo: dropInfo };
        }
        
        // Function to check if a judge score should be marked as dropped
        function isJudgeScoreDropped(judgeId, dropInfo) {
            if (!dropInfo || dropInfo.method === 'none') {
                return false;
            }
            
            return (dropInfo.droppedHighest && dropInfo.droppedHighest.judge_id == judgeId) ||
                   (dropInfo.droppedLowest && dropInfo.droppedLowest.judge_id == judgeId);
        }
        
        function renderScoreGrid(data) {
    // Preserve scroll position before updating
    preserveScrollPosition();
    
    const container = document.getElementById('scoresContainer');
    const noDataMessage = document.getElementById('noDataMessage');
    const sortControls = document.getElementById('sortControls');
    
    if (!data.participants.length) {
        container.style.display = 'none';
        sortControls.classList.add('d-none');
        noDataMessage.style.display = 'block';
        
        const heatValue = document.getElementById('heatSelect').value;
        const runValue = document.getElementById('runSelect').value;
        
        let message = '';
        if (!heatValue) {
            message = `
                <i class="fas fa-info-circle fa-3x mb-3 opacity-50"></i>
                <h5>Select a Heat to View Scores</h5>
                <p>Choose a heat from the dropdown above to display participants and their scores</p>
            `;
        } else if (!runValue) {
            message = `
                <i class="fas fa-info-circle fa-3x mb-3 opacity-50"></i>
                <h5>Select a Run to View Scores</h5>
                <p>Choose a run from the dropdown above to display scores for Heat ${heatValue}</p>
            `;
        } else {
            message = `
                <i class="fas fa-users-slash fa-3x mb-3 opacity-50"></i>
                <h5>No participants found</h5>
                <p>No participants are registered for Heat ${heatValue}</p>
            `;
        }
        
        noDataMessage.innerHTML = message;
        return;
    }

    // Update category filter options
    updateCategoryFilter(data.participants);
    
    // Group judges by control points
    const controlPointGroups = {};
    const controlPointColors = ["scale-1",
  "scale-2",
  "scale-3",
  "scale-4",
  "scale-5",
  "scale-6",
  "scale-7",
  "scale-8",
  "scale-9",
  "scale-10",
  "scale-11",
  "scale-12"];
    
    // Assign unique color to each judge
    let judgeColorIndex = 0;
    data.judges.forEach(judge => {
        const cpName = judge.control_point_name || '';
        if (!controlPointGroups[cpName]) {
            controlPointGroups[cpName] = {
                judges: [],
                color: '' // Control point won't have a single color anymore
            };
        }
        // Assign unique color to each judge
        judge.uniqueColor = controlPointColors[judgeColorIndex % controlPointColors.length];
        judgeColorIndex++;
        controlPointGroups[cpName].judges.push(judge);
    });
    
    // Load scoring format configuration for drop rule calculations
    let scoringFormat = null;
    const eventId = document.getElementById('eventSelect').value;
    
    // Prepare participants with calculated data and filter them
    let processedParticipants = data.participants.map(participant => {
        let hasAllScores = true;
        let scoreValues = [];
        let latestScoreTime = null;
        
        // Determine which run to display based on data.run_number
        const targetRunNumber = data.run_number;
        
        // Collect all judge scores for this participant (for the specific run if specified)
        let judgeScores = [];
        
        if (targetRunNumber) {
            // Specific run selected - use old logic
            data.judges.forEach(judge => {
                const score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                    ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                if (!score) {
                    hasAllScores = false;
                } else {
                    // Add to judge scores array with judge info for drop rule processing
                    if (score.score_value !== null && score.status === 'OK') {
                        judgeScores.push({
                            judge_id: judge.id,
                            judge_name: judge.name,
                            score: parseFloat(score.score_value)
                        });
                        scoreValues.push(parseFloat(score.score_value));
                    }
                    
                    // Track latest score time
                    if (score.created_at) {
                        const scoreTime = new Date(score.created_at);
                        if (!latestScoreTime || scoreTime > latestScoreTime) {
                            latestScoreTime = scoreTime;
                        }
                    }
                }
            });
            // Update hasAllScores based on actual judge count
            hasAllScores = judgeScores.length === data.judges.length;
        } else {
            // No specific run - collect scores from all runs for this participant
            const participantScores = data.scores[participant.id] || {};
            Object.keys(participantScores).forEach(runNum => {
                data.judges.forEach(judge => {
                    const score = participantScores[runNum][judge.id];
                    if (score) {
                        if (score.score_value !== null && score.status === 'OK') {
                            judgeScores.push({
                                judge_id: judge.id,
                                judge_name: judge.name,
                                score: parseFloat(score.score_value),
                                run_number: runNum
                            });
                            scoreValues.push(parseFloat(score.score_value));
                        }
                        
                        // Track latest score time
                        if (score.created_at) {
                            const scoreTime = new Date(score.created_at);
                            if (!latestScoreTime || scoreTime > latestScoreTime) {
                                latestScoreTime = scoreTime;
                            }
                        }
                    }
                });
            });
            // For multi-run view, we don't require all scores
            hasAllScores = judgeScores.length > 0;
        }
        
        // Calculate run score with drop rules
        let runScore = 0;
        let dropInfo = null;
        if (judgeScores.length > 0) {
            const result = calculateRunScoreWithDrops(judgeScores, data.drop_rule || 'none');
            runScore = result.score;
            dropInfo = result.dropInfo;
        }
        
        const average = runScore;
        const approvalStatus = getApprovalStatus(participant, data.judges, data.scores, targetRunNumber);
        
        return {
            ...participant,
            hasAllScores,
            average,
            runScore,
            dropInfo,
            judgeScores,
            latestScoreTime,
            canSort: judgeScores.length > 0, // Allow sorting if participant has at least one valid score
            approvalStatus,
            passesFilter: passesFilters(participant, approvalStatus)
        };
    });

    // Filter participants
    const filteredParticipants = processedParticipants.filter(p => p.passesFilter);
    
    // Apply sorting
    const sortableParticipants = filteredParticipants.filter(p => p.canSort);
    const unsortableParticipants = filteredParticipants.filter(p => !p.canSort);
    
    // Sort the sortable participants based on current mode
    switch (currentSortMode) {
        case 'average':
            sortableParticipants.sort((a, b) => b.average - a.average);
            break;
        case 'latest':
            sortableParticipants.sort((a, b) => {
                if (!a.latestScoreTime && !b.latestScoreTime) return 0;
                if (!a.latestScoreTime) return 1;
                if (!b.latestScoreTime) return -1;
                return b.latestScoreTime - a.latestScoreTime;
            });
            break;
        case 'default':
        default:
            sortableParticipants.sort((a, b) => {
                if (a.sort_order !== b.sort_order) {
                    return a.sort_order - b.sort_order;
                }
                return a.sort_order - b.bib_number;
            });
            break;
    }
    
    // Combine sorted and unsorted participants
    const finalParticipants = [...sortableParticipants, ...unsortableParticipants];
    
    // Update sort controls and counters
    sortControls.classList.remove('d-none');
    document.getElementById('sortableCount').textContent = `${sortableParticipants.length} complete participants`;
    document.getElementById('filteredCount').textContent = filteredParticipants.length;
    document.getElementById('totalCount').textContent = data.participants.length;
    
    // Debug logging
    console.log('Sort Controls Debug:', {
        sortableCount: sortableParticipants.length,
        filteredCount: filteredParticipants.length,
        totalCount: data.participants.length,
        totalJudges: data.judges.length,
        hasAnyComplete: sortableParticipants.length > 0,
        currentSortMode: currentSortMode,
        sampleParticipant: processedParticipants[0] ? {
            name: processedParticipants[0].first_name,
            canSort: processedParticipants[0].canSort,
            hasAllScores: processedParticipants[0].hasAllScores,
            judgeScoresCount: processedParticipants[0].judgeScores.length,
            requiredScores: data.judges.length
        } : null
    });
    
    // Enable/disable sort buttons and restore active state
    const hasAnyComplete = sortableParticipants.length > 0;
    document.querySelectorAll('.btn-outline-primary[data-sort]').forEach(btn => {
        const sortType = btn.getAttribute('data-sort');
        
        // Remove disabled class from all first
        btn.classList.remove('disabled');
        
        // Add disabled only to average/latest if no complete participants
        if ((sortType === 'average' || sortType === 'latest') && !hasAnyComplete) {
            btn.classList.add('disabled');
        }
        
        // Restore active state based on current sort mode
        if (sortType === currentSortMode) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
    
    noDataMessage.style.display = 'none';
    container.style.display = 'block';

    // Show "no results" message if all participants are filtered out
    if (filteredParticipants.length === 0) {
        container.innerHTML = `
            <div class="text-center text-muted py-5">
                <i class="fas fa-filter fa-3x mb-3 opacity-50"></i>
                <h5>No participants match current filters</h5>
                <p>Try adjusting your filter settings to see more participants</p>
                <button class="btn btn-outline-primary" onclick="document.getElementById('clearFilters').click()">
                    <i class="fas fa-times me-1"></i>Clear All Filters
                </button>
            </div>
        `;
        return;
    }
    
    // Build the grid HTML using Bootstrap table
    let html = '';
    
    // Add drop rule debug info at the top with source indicator
    const dropRuleSource = data.drop_rule_source || 'format';
    const isHeatOverride = dropRuleSource === 'heat';
        
        html += '<div class="table-responsive"><table class="table table-bordered table-hover table-striped" id="scoresTable">';
    
    // Header row with control point groups and judge names
    html += '<thead style="border-bottom: 2px solid gray;">';
    
    // Control point header row
    html += '<th class="" rowspan="2">' +
                        '</div>#</th>';
    Object.keys(controlPointGroups).forEach(cpName => {
        const group = controlPointGroups[cpName];
        html += `<th colspan="${group.judges.length}" class="text-center">
            <i class="fas fa-map-marker-alt me-2"></i>${cpName}
            <span class="badge bg-light text-dark ms-2">${group.judges.length} judges</span>
        </th>`;
    });
    html += '<th rowspan="2" class="text-center align-middle">Score</th>';
    html += '<th rowspan="2" class="text-center align-middle participantColMobileHide">Actions</th></tr>';
    
    // Judge names row
    html += '';
    Object.keys(controlPointGroups).forEach(cpName => {
        const group = controlPointGroups[cpName];
        group.judges.forEach(judge => {
            const headStar = judge.is_head == 1 ? '<i data-lucide="star" class="text-warning ms-1" title="Head Judge" style="width: 14px; height: 14px;"></i>' : '';
            html += `<th class="text-center ${judge.uniqueColor} bg-opacity-25">
                <div class="small fw-bold">${judge.name}${headStar}</div>
            </th>`;
        });
    });
    html += '</th></thead><tbody>';
    
    // Participant rows
    finalParticipants.forEach((participant, index) => {
        // Validate diversity rules for this participant
        let diversityValidation = { valid: true, errors: [] };
        if (data.diversity_rules && data.diversity_rules.length > 0 && data.diversity_enabled) {
            diversityValidation = validateParticipantDiversityRules(participant, data.diversity_rules, data);
        }
        
        // Apply row class based on approval status and diversity validation
        let rowClass = participant.canSort ? 'bg-approved bg-opacity-25 border' : '';
        if (!diversityValidation.valid) {
            rowClass += ' diversity_rules_not_meet bg-warning bg-opacity-10';
        }
        
        // Position indicator for sorted participants
        const positionBadge = participant.canSort && currentSortMode !== 'default' ? 
            `<span class="badge bg-warning me-1">#${sortableParticipants.indexOf(participant) + 1}</span>` : '';
        
        // Calculate total columns for mobile row
        const totalJudges = data.judges.length;
        const totalColumns = totalJudges; // Only span across judge columns (not Average/Actions)
        
        // Calculate average for mobile rowspan display
        let mobileAverageHTML = '';
        let mobileAverageAndActionsHTML = '';
        if (!diversityValidation.valid && !data.run_number) {
            // Show DNQ if diversity rules not met (only in multi-run view)
            mobileAverageHTML = `
                <span class="badge bg-warning text-white mb-2 fs-2 fw-bold w-auto">DNQ</span>
                <small class="text-danger">${diversityValidation.errors[0]}</small>
            `;
        } else if (participant.judgeScores && participant.judgeScores.length > 0) {
            const displayScore = participant.runScore;
            const formattedScore = displayScore % 1 === 0 ? displayScore.toString() : displayScore.toFixed(2);
            mobileAverageHTML = `<span class="mb-2 text-dark fs-2 fw-bold">${formattedScore}</span>`;
        } else {
            mobileAverageHTML = `
                <h4 class="mb-2 text-muted" title="No valid scores">—</h4>                
            `;
        }
        
        // Calculate pending scores for actions column by checking participant scores
        let hasPendingScores = false;
        let hasAnyScores = false;
        
        const targetRunNumber = data.run_number;
        if (targetRunNumber) {
            // Check scores for specific run
            data.judges.forEach(judge => {
                const score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                    ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                if (score) {
                    hasAnyScores = true;
                    if (score.is_approved != 1) {
                        hasPendingScores = true;
                    }
                }
            });
        } else {
            // Check all runs
            const participantScores = data.scores[participant.id] || {};
            Object.keys(participantScores).forEach(runNum => {
                data.judges.forEach(judge => {
                    const score = participantScores[runNum][judge.id];
                    if (score) {
                        hasAnyScores = true;
                        if (score.is_approved != 1) {
                            hasPendingScores = true;
                        }
                    }
                });
            });
        }
        
        // Build actions HTML for mobile rowspan
        let mobileActionsHTML = '';
        if (hasAnyScores && hasPendingScores) {
            mobileActionsHTML = `
                <button class="approve-all-btn approval-clickable btn btn-primary ms-2 position-relative" data-participant-id="${participant.id}" style="cursor: pointer; overflow: hidden;" title="Hold to approve all scores" title="Hold to approve all scores">
                    <i class="fas fa-check-double"></i>    
                    <div class="hold-progress position-absolute bottom-0 start-0 bg-success" style="height: 100%; width: 0%; transition: none;"></div>
                </button>
            `;
        } else if (hasAnyScores && !hasPendingScores) {
            mobileActionsHTML = `
                <span class="text-success" title="All scores approved">
                    <i class="fas fa-check me-1"></i>
                </span>
            `;
        } else {
            mobileActionsHTML = `
                <span class="text-muted" title="No scores submitted">
                    <i class="fas fa-clock me-1"></i>
                </span>
            `;
        }
        
        // Combine average and actions for mobile view
        mobileAverageAndActionsHTML = mobileAverageHTML + '' + mobileActionsHTML;
        
        // Mobile-only participant info row (full width)
        html += `<tr class="participantColMobileRow d-none ${rowClass}">
            <td class="text-start p-2" rowspan="4">
                ${positionBadge}
                <div class="css-bib"><span class="badge bg-white fs-6 me-2 participant-bib-badge  ss-f-number">${participant.bib_number}</span></div>
            </td>
            <td colspan="${totalColumns}" class="text-start p-2">
                <strong class="me-2">${participant.first_name} ${participant.last_name}</strong>
                <span class="badge bg-${participant.gender === 'M' ? 'info' : 'pink'} bg-opacity-20 me-1">${participant.gender}</span>
                <span class="badge text-muted me-1">${participant.club}</span>
                <span class="badge text-dark">${participant.category}</span>
                ${!diversityValidation.valid && !data.run_number ? `
                    <div class="mt-2">
                        <span class="badge bg-warning text-white w-auto">
                            <i class="fas fa-exclamation-triangle me-1"></i>DNQ
                        </span>
                        <small class="text-danger d-block mt-1">${diversityValidation.errors.join(', ')}</small>
                    </div>
                ` : ''}
            </td>
            <td class="text-center p-2" rowspan="4">
                <div class="d-flex flex-row score-figure">
                    ${mobileAverageAndActionsHTML}
                </div>
            </td>
        </tr>`;
        
        // Second mobile row for scores
        html += `<tr class="participantColMobileRow d-none ${rowClass}">
            <!-- Judge score TDs will be added in the loop below -->
        `;
        
        // Regular desktop row with all columns
        html += `</tr><tr class="${rowClass}">`;
        
        html += `<td rowspan="2" class="participantCol">
            ${positionBadge}
            <div class="css-bib"><span class="badge bg-white fs-6 mb-1 participant-bib-badge ss-f-number">${participant.bib_number}</span></div>
            <strong>${participant.first_name} ${participant.last_name}</strong> 
            <span class="badge bg-${participant.gender === 'M' ? 'info' : 'pink'} bg-opacity-20 ">
                ${participant.gender}
            </span>
            <br><small><span class="badge text-muted">${participant.club}</span><br><span class="badge text-dark">${participant.category}</span></small>
            ${!diversityValidation.valid && !data.run_number ? `
                <div class="mt-2">
                    <span class="badge bg-warning text-white w-auto">
                        <i class="fas fa-exclamation-triangle me-1"></i>DNQ
                    </span>
                    <small class="text-danger d-block mt-1">${diversityValidation.errors.join(', ')}</small>
                </div>
            ` : ''}
        </td>`;
        
        hasPendingScores = false;
        hasAnyScores = false;
        let scoreValues = [];
        
        // Render judge scores grouped by control points
        Object.keys(controlPointGroups).forEach(cpName => {
            const group = controlPointGroups[cpName];
            group.judges.forEach(judge => {
                // Determine which run's score to display
                const targetRunNumber = data.run_number;
                
                // If no specific run, show all runs for this judge
                if (!targetRunNumber) {
                    // Multi-run view - show all runs
                    const participantScores = data.scores[participant.id] || {};
                    const allRunScores = [];
                    
                    // Collect all run scores for this judge
                    Object.keys(participantScores).forEach(runNum => {
                        const runScore = participantScores[runNum][judge.id];
                        if (runScore) {
                            allRunScores.push({
                                run: runNum,
                                ...runScore
                            });
                        }
                    });
                    
                    html += `<td class="scoreCol text-center ${judge.uniqueColor} bg-opacity-10" data-score-value="" data-status=""><div class="row score-figure flex-column">`;
                    
                    if (allRunScores.length > 0) {
                        // Display all run scores vertically
                        allRunScores.forEach(runScore => {
                            hasAnyScores = true;
                            const isApproved = runScore.is_approved == 1;
                            if (!isApproved) hasPendingScores = true;
                            
                            if (runScore.score_value !== null && runScore.status === 'OK') {
                                scoreValues.push(parseFloat(runScore.score_value));
                            }
                            
                            const bgClass = isApproved ? 'bg-success-subtle' : 'bg-warning-subtle';
                            let numericValue = runScore.score_value;
                            let formattedValue = '';
                            if (numericValue !== null && runScore.status === 'OK') {
                                const numValue = parseFloat(numericValue);
                                formattedValue = numValue % 1 === 0 ? numValue.toString() : numValue.toFixed(2);
                            }
                            const badgeClass = getStatusBadgeClass(runScore.status);
                            const displayHtml = (runScore.status !== 'OK')
                                ? `<span class="badge bg-${badgeClass}">${runScore.status}</span>`
                                : `<span class="fw-bold text-${badgeClass === 'success' ? 'primary' : badgeClass === 'danger' ? 'danger' : 'warning'}${isApproved ? '' : ' text-warning-emphasis'}">${formattedValue}</span>`;
                            
                            // Parse figures for this run
                            let figuresHtml = '';
                            if (runScore.figures_json) {
                                try {
                                    const figuresData = JSON.parse(runScore.figures_json);
                                    const actualFigures = figuresData.figures || figuresData;
                                    const figureEntries = Object.entries(actualFigures || {});
                                    
                                    if (figureEntries.length > 0) {
                                        // Count total figures
                                        let totalFigures = 0;
                                        figureEntries.forEach(([category, items]) => {
                                            const itemsArray = Array.isArray(items) ? items : [];
                                            totalFigures += itemsArray.length;
                                        });
                                        
                                        // Determine class based on count
                                        let figClass = '';
                                        if (totalFigures <= 2) {
                                            figClass = 'fig-2';
                                        } else if (totalFigures <= 5) {
                                            figClass = 'fig-3_5';
                                        } else {
                                            figClass = 'fig_5';
                                        }
                                        
                                        figuresHtml = `<div class="figures-summary ${figClass} mt-1">`;
                                        figureEntries.forEach(([category, items]) => {
                                            const itemsArray = Array.isArray(items) ? items : [];
                                            if (itemsArray.length > 0) {
                                                figuresHtml += `<small class="text-muted">${category}:</small> `;
                                                figuresHtml += itemsArray.map(item => `<span class="badge bg-light text-dark border me-1">${item}</span>`).join('');
                                            }
                                        });
                                        figuresHtml += `</div>`;
                                    }
                                } catch (e) {
                                    // Silently fail if figures JSON is invalid
                                }
                            }
                            
                            html += `<div class="mb-1 p-1 rounded ${bgClass} w-auto" onclick="editScore(${runScore.score_id}, ${runScore.score_value || 0}, '${runScore.status}', '${encodeURIComponent(runScore.figures_json || '')}', '${participant.first_name.replace(/'/g, "\\'")} ${participant.last_name.replace(/'/g, "\\'")}', '${participant.bib_number}', '${participant.category.replace(/'/g, "\\'")}', '${judge.name.replace(/'/g, "\\'")}', '${cpName.replace(/'/g, "\\'")}')" style="cursor: pointer;">
                                <small class="text-muted">R${runScore.run}</small>
                                ${displayHtml}
                                ${isApproved ? 
                                    '<small class="badge bg-success bg-opacity-50">✓</small>' : 
                                    '<small class="badge bg-danger bg-opacity-50">!</small>'
                                }
                                ${figuresHtml}
                            </div>`;
                        });
                    } else {
                        html += `<small class="text-muted">No scores</small>`;
                    }
                    
                    html += `</div></td>`;
                    return;
                }
                
                // Single run view - original logic
                const score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                    ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                 
                if (score) {
                    hasAnyScores = true;
                    const isApproved = score.is_approved == 1;
                    if (!isApproved) hasPendingScores = true;
                    
                    // Add to score values for average calculation if it's a valid numeric score
                    if (score.score_value !== null && score.status === 'OK') {
                        scoreValues.push(parseFloat(score.score_value));
                    }
                    
                    // Determine background class based on approval status
                    let bgClass = '';
                    if (isApproved) {
                        bgClass = 'bg-success-subtle';
                    } else {
                        bgClass = 'bg-warning-subtle';
                    }
                    
                    html += `<td class="scoreCol text-center  ${bgClass}" data-score-value="${score.score_value || ''}" data-status="${score.status}"><div class="row score-figure">`;
                    
                    // Check if this score is dropped based on drop rules
                    const isDropped = isJudgeScoreDropped(judge.id, participant.dropInfo);
                    // Format score value to remove unnecessary decimals
                    let numericValue = score.score_value;
                    let formattedValue = '';
                    if (numericValue !== null && score.status === 'OK') {
                        const numValue = parseFloat(numericValue);
                        formattedValue = numValue % 1 === 0 ? numValue.toString() : numValue.toFixed(2);
                    }
                    const badgeClass = getStatusBadgeClass(score.status);
                    const dropClass = isDropped ? 'score_drop fs-4' : 'fs-2';
                    const dropIndicator = isDropped ? '(' : '';
                    const dropIndicatorEnd = isDropped ? ')' : '';
                    const displayContent = (score.status !== 'OK')
                        ? `<span class="badge bg-${badgeClass}">${score.status}</span>`
                        : `<span class="badge bg-white fw-bold score_value m-0 text-${badgeClass === 'success' ? 'primary' : badgeClass === 'danger' ? 'danger' : 'warning'}${isApproved ? '' : ' text-warning-emphasis'}">${dropIndicator}${formattedValue}${dropIndicatorEnd}</span>`;
                    
                    html += `<div class="scores-col1 ${score.status} ${dropClass} w-auto" onclick="editScore(${score.score_id}, ${score.score_value || 0}, '${score.status}', '${encodeURIComponent(score.figures_json || '')}', '${participant.first_name.replace(/'/g, "\\'")} ${participant.last_name.replace(/'/g, "\\'")}', '${participant.bib_number}', '${participant.category.replace(/'/g, "\\'")}', '${judge.name.replace(/'/g, "\\'")}', '${cpName.replace(/'/g, "\\'")}')">
                                ${displayContent}

                       
                                <div class="btn-group btn-group-sm" role="group">
                                    <span class="badge bg-warning editScoreButton"  title="Click to edit score" style="cursor: pointer; pointer-events: auto;" onclick="editScore(${score.score_id}, ${score.score_value || 0}, '${score.status}', '${encodeURIComponent(score.figures_json || '')}', '${participant.first_name.replace(/'/g, "\\'")} ${participant.last_name.replace(/'/g, "\\'")}', '${participant.bib_number}', '${participant.category.replace(/'/g, "\\'")}', '${judge.name.replace(/'/g, "\\'")}', '${cpName.replace(/'/g, "\\'")}')">
                                        <i class="fas fa-edit"></i> Edit score
                                    </span>
                                </div>
                            </div>
                             ${isApproved ? 
                            '<button class="unapprove-btn approval-clickable border-0 btn btn-outline-success w-auto position-relative" data-score-id="' + score.score_id + '" style="cursor: pointer; pointer-events: auto; overflow: hidden;" title="Hold to unapprove">' +
                                '<i class="fas fa-check-circle"></i>' +
                                '<div class="hold-progress position-absolute bottom-0 start-0 bg-danger" style="height: 100%; width: 0%; transition: none;"></div>' +
                                
                            '</button>' : 
                            '<button class="approve-btn approval-clickable btn btn-outline-danger w-auto position-relative pulse-border-danger" data-score-id="' + score.score_id + '" style="cursor: pointer; overflow: hidden;" title="Hold to approve">' +
                                '<i class="fas fa-clock"></i>' +
                                '<div class="hold-progress position-absolute bottom-0 start-0 bg-success" style="height: 100%; width: 0%; transition: none;"></div>' +
                                
                            '</button>'
                        }
                        `;
                    
                    // Display figures information if available
                    if (score.figures_json) {
                        try {
                            const figuresData = JSON.parse(score.figures_json);
                            // Handle nested structure - figures might be in figuresData.figures
                            const actualFigures = figuresData.figures || figuresData;
                            const figuresParts = [];
                            Object.keys(actualFigures || {}).forEach(category => {
                                const items = Array.isArray(actualFigures[category]) ? actualFigures[category] : [];
                                if (items.length > 0) {
                                    figuresParts.push(`${category}: ${items.join(', ')}`);
                                }
                            });
                            if (figuresParts.length > 0) {
                                html += `<div class="scores-col2 w-100" style="display: none !important;">
                                    <div class="figures-display mt-1 border-top p-1 small w-100 hidden">
                                        ${Object.keys(actualFigures).map(cat => {
                                            const items = Array.isArray(actualFigures[cat]) ? actualFigures[cat] : [];
                                            if (!items.length) return '';
                                            return `
                                                <div class="m-0 figure-s w-100 text-center">
                                                    ${items.map(item => `<span class=\"fw-bold badge bg-light text-primary mx-1\"><small class="small m-0 p-0 text-dark-emphasis fig-cat fig-cat-${cat}">${cat.substring(0, 3)}:</small><strong>${item}</strong></span>`).join(' ')}
                                                </div>
                                            `;
                                        }).join('')}
                                    </div></div>
                                `;
                            }
                        } catch (e) {
                            // Ignore JSON parse errors
                        }
                    }
                } else {
                    // Empty score cell - no score exists
                    html += `<td class="scoreCol text-center ${judge.uniqueColor} bg-emptyx" data-score-value="" data-status=""><div class="row score-figure">`;
                    // Empty score cell - clickable for head judges/admins
                    html += `
                        <div class="score-pending" onclick="createScore(${participant.id}, ${judge.id}, '${participant.first_name} ${participant.last_name}', '${judge.name}')" title="Click to add score for this judge">
                                <div class="btn btn-outline-info"><i class="fas fa-plus-circle"></i></div>
                        </div>`;
                }
                
                html += `</div></td>`;
            });
        });
        
        // Average column - only for desktop (mobile uses rowspan above)
        html += `<td rowspan="2" class="resultCom text-center participantColMobileHide"><div class="row score-figure">`;
        if (!diversityValidation.valid && !data.run_number) {
            // Show DNQ if diversity rules not met (only in multi-run view)
            html += `
                <span class="badge bg-warning text-white mb-2 fs-2 fw-bold w-auto">DNQ</span>
                <small class="text-danger">${diversityValidation.errors[0]}</small>
            `;
        } else if (participant.judgeScores && participant.judgeScores.length > 0) {
            // Use the calculated run score with drop rules and format appropriately
            const displayScore = participant.runScore;
            // Format to remove unnecessary decimals
            const formattedScore = displayScore % 1 === 0 ? displayScore.toString() : displayScore.toFixed(2);
            html += `
                <span class="badge bg-white text-dark mb-2 fs-2 fw-bold score_value">${formattedScore}</span>
                <span class="badge bg-white text-dark">${participant.judgeScores.length} scores</span>
            `;
            
            // Show drop rule info if applicable
            if (participant.dropInfo && participant.dropInfo.method !== 'none') {
                //html += `<br><small class="text-muted">Drop: ${participant.dropInfo.method.replace('_', ' ')}</small>`;
            }
        } else {
            html += `
                <h4 class="mb-2 text-muted" title="No valid scores">—</h4>
                
            `;
        }
        html += `</div></td>`;
        
        // Actions column - only for desktop (mobile uses rowspan above)
        html += `<td rowspan="2" class="actionsCol text-center participantColMobileHide"><div class="row score-figure">`;
        if (hasAnyScores && hasPendingScores) {
            html += `
                <button class="approve-all-btn approval-clickable btn btn-primary btn-sm mb-1 position-relative" data-participant-id="${participant.id}" style="cursor: pointer; overflow: hidden;" title="Hold to approve all scores">
                    <i class="fas fa-check-double me-1"></i>Approve&nbsp;All    
                    <div class="hold-progress position-absolute bottom-0 start-0 bg-success" style="height: 100%; width: 0%; transition: none;"></div>
                </button>
            `;
        } else if (hasAnyScores && !hasPendingScores) {
            html += `
                <span class="badge bg-success">
                    <i class="fas fa-check-circle me-1"></i>
                </span>
            `;
        } else {
            html += `
                <span class="text-muted">
                    <i class="fas fa-clock me-1"></i>
                </span>
            `;
        }
        html += `</div></td>`;
        
        html += '</tr>';
        
        // Add merged control point summary row
        html += `<tr class="control-point-summary bg-light border-top-2" style="border-bottom: 2px solid gray;">`;
        html += ``;
        
        // Render individual cells for each judge showing their figures
        Object.keys(controlPointGroups).forEach(cpName => {
            const group = controlPointGroups[cpName];
            
            // Determine which run to display based on data.run_number
            const targetRunNumber = data.run_number;
            
            // For each judge in this control point, create a separate cell
            group.judges.forEach(judge => {
                let score = null;
                
                if (targetRunNumber) {
                    // Specific run selected
                    score = data.scores[participant.id] && data.scores[participant.id][targetRunNumber] && data.scores[participant.id][targetRunNumber][judge.id] 
                        ? data.scores[participant.id][targetRunNumber][judge.id] : null;
                } else {
                    // No specific run - get scores from all runs for this judge
                    const participantScores = data.scores[participant.id] || {};
                    Object.keys(participantScores).forEach(runNum => {
                        const runScore = participantScores[runNum][judge.id];
                        if (runScore) {
                            score = runScore;
                        }
                    });
                }
                
                // Render cell for this specific judge
                html += `<td class="text-center ${judge.uniqueColor} bg-opacity-5 p-0 small">`;
                
                if (score && score.figures_json && targetRunNumber) {
                    // Only show figures in single-run view
                    try {
                        const figuresData = JSON.parse(score.figures_json);
                        const actualFigures = figuresData.figures || figuresData;
                        
                        const figureEntries = Object.entries(actualFigures || {});
                        if (figureEntries.length > 0) {
                            // Count total figures
                            let totalFigures = 0;
                            figureEntries.forEach(([category, items]) => {
                                const itemsArray = Array.isArray(items) ? items : [];
                                totalFigures += itemsArray.length;
                            });
                            
                            // Determine class based on count
                            let figClass = '';
                            if (totalFigures <= 2) {
                                figClass = 'fig-2';
                            } else if (totalFigures <= 5) {
                                figClass = 'fig-3_5';
                            } else {
                                figClass = 'fig_5';
                            }
                            
                            html += `<div class="figures-summary ${figClass} flex-wrap gap-1 d-flex justify-content-center">`;
                            figureEntries.forEach(([category, items]) => {
                                const itemsArray = Array.isArray(items) ? items : [];
                                if (itemsArray.length > 0) {
                                    html += `<div class="mb-1">
                                        <label class="text-muted d-block">${category}:</label>
                                        ${itemsArray.map(item => `<span class="badge bg-light text-dark border ss-bg-${item}">${item}</span>`).join('')}
                                    </div>`;
                                }
                            });
                            html += `</div>`;
                        } else {
                            html += `<span class="text-muted"></span>`;
                        }
                    } catch (e) {
                        html += `<span class="text-muted"></span>`;
                    }
                } else {
                    html += `<span class="text-muted"></span>`;
                }
                
                html += `</td>`;
            });
        });
        
        // Empty cells for average and actions columns
        //html += `<td class="text-muted"></td>`;
        //html += `<td class="text-muted"></td>`;
        html += `</tr>`;
    });
    
    html += '</tbody></table></div>';
    html += `
        <div class="alert alert-info mb-3">
            <strong><i class="fas fa-info-circle me-1"></i>Drop Rule Configuration:</strong>
            <span class="badge bg-primary ms-2">${data.drop_rule || 'none'}</span>
            ${isHeatOverride ? 
                '<span class="badge bg-warning text-dark ms-1"><i class="fas fa-fire me-1"></i>Heat Override</span>' : 
                '<span class="badge bg-secondary ms-1"><i class="fas fa-cog me-1"></i>Format Default</span>'}
            ${data.drop_rule && data.drop_rule !== 'none' ? 
                `<small class="ms-2">→ Removing <strong>${data.drop_rule === 'highest' ? 'highest' : 
                    (data.drop_rule === 'lowest' ? 'lowest' : 
                    (data.drop_rule === 'highest_and_lowest' ? 'highest & lowest' : data.drop_rule))}</strong> scores from average calculation</small>` : 
                '<small class="ms-2">→ All judge scores are used in average calculation</small>'}
        </div>
    `;
    
    // Display diversity rules information if enabled for this heat
    if (data.diversity_enabled && data.diversity_rules && data.diversity_rules.length > 0) {
        html += `
            <div class="alert alert-warning mb-3">
                <strong><i class="fas fa-check-double me-1"></i>Diversity Rules Enabled for This Heat:</strong>
                <div class="mt-2">
        `;
        
        data.diversity_rules.forEach(rule => {
            const isOptional = rule.is_optional == 1;
            if (!isOptional) {
                html += `
                    <div class="mb-1">
                        <span class="badge bg-warning text-dark me-2">${rule.group_name}</span>
                        <small>Requires at least <strong>${rule.min_distinct}</strong> different ${rule.group_name.toLowerCase()} figures</small>
                    </div>
                `;
            }
        });
        
        html += `
                </div>
                <small class="text-muted d-block mt-2">
                    <i class="fas fa-info-circle me-1"></i>
                    Participants not meeting diversity rules are highlighted with a yellow background
                </small>
            </div>
        `;
    }
    
    container.innerHTML = html;
    
    // Initialize Lucide icons for the new dynamic content
    if (window.initializeLucideIcons) {
        window.initializeLucideIcons();
    }
    
    // Restore scroll position after updating
    restoreScrollPosition();
}

function createScore(participantId, judgeId, participantName, judgeName) {
    // Find the judge's control point from current data
    let controlPointName = 'Unknown Control Point';
    if (currentData && currentData.judges) {
        const judge = currentData.judges.find(j => j.id == judgeId);
        if (judge && judge.control_point_name) {
            controlPointName = judge.control_point_name;
        }
    }

    // Enrich participant data from currentData
    let bibNumber = '';
    let category = '';
    let gender = '';
    let club = '';
    if (currentData && Array.isArray(currentData.participants)) {
        const p = currentData.participants.find(pp => pp.id == participantId);
        if (p) {
            bibNumber = p.bib_number || '';
            category = p.category || '';
            gender = p.gender || '';
            club = p.club || '';
            // Prefer canonical name if available
            if (p.first_name && p.last_name) {
                participantName = `${p.first_name} ${p.last_name}`;
            }
        }
    }

    document.getElementById('createParticipantId').value = participantId;
    document.getElementById('createJudgeId').value = judgeId;
    document.getElementById('createScoreValue').value = '';
    
    // Reset radio buttons to OK (default)
    document.getElementById('createStatusOK').checked = true;

    // Participant summary (match Edit modal visual style)
    const genderBadge = gender ? `<span class="badge bg-${gender==='M'?'info':'pink'} bg-opacity-20  ms-1">${gender}</span>` : '';
    const clubLine = club ? `<span class="badge text-muted">${club}</span><br>` : '';
    document.getElementById('createScoreInfo').innerHTML = `
        <div class="participantCol">
            <div class="css-bib"><span class="badge bg-white fs-6 mb-1 participant-bib-badge ss-f-number">${bibNumber}</span></div>
            <strong>${participantName}</strong>
            ${genderBadge}
            <br><small>${clubLine}<span class="badge text-dark">${category}</span></small>
        </div>
    `;

    // Fill score context under figures section (mirror Edit modal)
    const eventSelect = document.getElementById('eventSelect');
    const heatSelect = document.getElementById('heatSelect');
    const runSelect = document.getElementById('runSelect');
    const eventName = eventSelect.options[eventSelect.selectedIndex]?.text || 'Unknown Event';
    const heatNumber = heatSelect.value || 'Unknown';
    const runNumber = runSelect.value || 'Unknown';
    const status = 'OK';
    const scoreContextHtml = `
        <div class="row g-2">
            <div class="col-md-6 small"><i class="fas fa-calendar-alt me-1 text-muted"></i><strong>Event:</strong> ${eventName}</div>
            <div class="col-md-6 small"><i class="fas fa-layer-group me-1 text-muted"></i><strong>Heat:</strong> ${heatNumber} <span class="text-muted">|</span> <strong>Run:</strong> ${runNumber}</div>
            <div class="col-md-6 small"><i class="fas fa-user-tie me-1 text-muted"></i><strong>Judge:</strong> ${judgeName}</div>
            <div class="col-md-6 small"><i class="fas fa-map-marker-alt me-1 text-muted"></i><strong>Control Point:</strong> <span class="badge bg-success ms-1">${controlPointName}</span></div>
            <div class="col-md-6 small"><i class="fas fa-info-circle me-1 text-muted"></i><strong>Status:</strong> <span class="badge bg-${getStatusBadgeClass(status)}">${status}</span></div>
        </div>`;
    const createScoreContextEl = document.getElementById('createScoreContextInfo');
    if (createScoreContextEl) createScoreContextEl.innerHTML = scoreContextHtml;

    // Reset create figures data
    window.createFiguresData = {};

    // Load figure options for create score
    loadCreateFigures();

    const modal = new bootstrap.Modal(document.getElementById('createScoreModal'));
    modal.show();
}

// Keypad functions for create score modal
function addToCreateScore(digit) {
    const input = document.getElementById('createScoreValue');
    const rangeInput = document.getElementById('createScoreRange');
    let currentValue = input.value || '';
    
    // Limit to 2 digits for scores 0-99
    if (currentValue.length < 2) {
        currentValue += digit;
        input.value = currentValue;
        rangeInput.value = currentValue;
    }
}

function backspaceCreateScore() {
    const input = document.getElementById('createScoreValue');
    const rangeInput = document.getElementById('createScoreRange');
    input.value = input.value.slice(0, -1);
    rangeInput.value = input.value || 0;
}

function applyCreateScoreValue() {
    const input = document.getElementById('createScoreValue');
    const createBtn = document.getElementById('createScoreBtn');
    
    // Validate that score value is present
    if (!input.value || input.value.trim() === '') {
        input.classList.add('is-invalid');
        return;
    }
    
    input.classList.remove('is-invalid');
    
    // Scroll modal to bottom to show footer with create button
    const modal = document.getElementById('createScoreModal');
    const modalDialog = modal.querySelector('.modal-dialog');
    
    // Wait a brief moment for any animations to complete
    setTimeout(() => {
        // Scroll the modal footer into view
        const modalFooter = modal.querySelector('.modal-footer');
        if (modalFooter) {
            modalFooter.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    }, 100);
    
    // Highlight create button
    createBtn.classList.add('pulse-highlight');
    setTimeout(() => {
        createBtn.classList.remove('pulse-highlight');
    }, 2000);
    
    // Focus out of input to trigger any validation
    input.blur();
}

function approveScore(scoreId) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'approve_score');
    formData.append('score_id', scoreId);
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessToast('Score approved successfully');
            
            // Send specific notification about score approval
            if (typeof notifyScoreApproval === 'function' && data.notification_data) {
                const nd = data.notification_data;
                // Pass enhanced data including all judge scores, current result, and rank
                const enhancedData = {
                    all_scores: nd.all_scores,
                    current_result: nd.current_result,
                    rank_position: nd.rank_position,
                    approved_count: nd.approved_count,
                    total_judges: nd.total_judges
                };
                notifyScoreApproval(nd.bib, nd.participant_name, nd.score, nd.judge, nd.heat, nd.run, enhancedData);
            } else if (typeof sendNotification === 'function') {
                sendNotification('success', 'Score Approved', 'Head judge approved a score', 'judge');
            }
            
            // Refresh grid to show updated approval status
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
            
            // Update modal buttons if edit modal is open
            updateEditModalApprovalButtons(scoreId, true);
        } else {
            showErrorAlert('Error approving score: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showErrorAlert('Error approving score');
    });
}

function unapproveScore(scoreId) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'unapprove_score');
    formData.append('score_id', scoreId);
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessToast('Score unapproved successfully');
            
            // Send notification about score unapproval
            if (typeof sendNotification === 'function' && data.notification_data) {
                const nd = data.notification_data;
                sendNotification('warning', 'Score Unapproved', 
                    `BIB #${nd.bib} (${nd.participant_name}) - ${nd.judge}: ${nd.score} (Heat ${nd.heat}, Run ${nd.run})`, 
                    'judge');
            }
            
            // Refresh grid to show updated approval status
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
            
            // Update modal buttons if edit modal is open
            updateEditModalApprovalButtons(scoreId, false);
        } else {
            showErrorAlert('Error unapproving score: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showErrorAlert('Error unapproving score');
    });
}

function deleteScore(scoreId) {
    // Get participant info from the edit modal
    const participantInfo = document.getElementById('editScoreInfo');
    const scoreValue = document.getElementById('editScoreValue').value;
    const statusRadios = document.getElementsByName('editStatus');
    let status = 'OK';
    statusRadios.forEach(radio => {
        if (radio.checked) status = radio.value;
    });
    
    // Get participant name from the info div
    const participantText = participantInfo ? participantInfo.textContent.trim() : 'Unknown';
    
    // Get score context info
    const eventSelect = document.getElementById('eventSelect');
    const heatSelect = document.getElementById('heatSelect');
    const runSelect = document.getElementById('runSelect');
    
    const eventName = eventSelect.options[eventSelect.selectedIndex]?.text || 'Unknown Event';
    const heatNumber = heatSelect.value || 'Unknown';
    const runNumber = runSelect.value || 'Unknown';
    
    // Populate the delete confirmation modal
    const deleteDetails = document.getElementById('deleteScoreDetails');
    deleteDetails.innerHTML = `
        <div class="mb-2">
            <strong>Participant:</strong> ${participantText}
        </div>
        <div class="mb-2">
            <strong>Event:</strong> ${eventName}
        </div>
        <div class="mb-2">
            <strong>Heat / Run:</strong> ${heatNumber} / ${runNumber}
        </div>
        <div class="mb-2">
            <strong>Score:</strong> <span class="badge bg-${status === 'OK' ? 'success' : 'warning'}">${status === 'OK' ? scoreValue : status}</span>
        </div>
    `;
    
    // Store scoreId for the confirm button
    document.getElementById('confirmDeleteScoreBtn').setAttribute('data-score-id', scoreId);
    
    // Show the confirmation modal
    const confirmModal = new bootstrap.Modal(document.getElementById('deleteScoreConfirmModal'));
    confirmModal.show();
}

// Handle the actual delete when confirmed
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('confirmDeleteScoreBtn').addEventListener('click', function() {
        const scoreId = this.getAttribute('data-score-id');
        
        // Close confirmation modal
        const confirmModal = bootstrap.Modal.getInstance(document.getElementById('deleteScoreConfirmModal'));
        if (confirmModal) {
            confirmModal.hide();
        }
        
        // Perform the delete
        const formData = new FormData();
        formData.append('ajax', '1');
        formData.append('action', 'delete_score');
        formData.append('score_id', scoreId);
        
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccessToast('Score deleted successfully');
                
                // Send notification about score deletion
                if (typeof sendNotification === 'function' && data.notification_data) {
                    const nd = data.notification_data;
                    sendNotification('danger', 'Score Deleted', 
                        `BIB #${nd.bib} (${nd.participant_name}) - ${nd.judge}: ${nd.score} (Heat ${nd.heat}, Run ${nd.run}) - DELETED by head judge`, 
                        'judge');
                }
                
                // Close the edit modal
                const editModal = bootstrap.Modal.getInstance(document.getElementById('editScoreModal'));
                if (editModal) {
                    editModal.hide();
                }
                
                // Refresh grid to show updated data
                if (document.getElementById('runSelect').value) {
                    loadScores();
                }
            } else {
                showErrorAlert('Error deleting score: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showErrorAlert('Error deleting score');
        });
    });
});

// Add the new approve all function
function approveAllParticipant(participantId) {
    const eventId = document.getElementById('eventSelect').value;
    const heatNumber = document.getElementById('heatSelect').value;
    const runNumber = document.getElementById('runSelect').value;
    
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'approve_all_participant');
    formData.append('event_id', eventId);
    formData.append('heat_number', heatNumber);
    formData.append('run_number', runNumber);
    formData.append('participant_id', participantId);
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessToast(`${data.count} scores approved`);
            
            // Send specific notification about bulk approval
            if (data.notification_data) {
                const nd = data.notification_data;
                // Try to use notification manager if available
                if (window.globalNotificationManager && typeof window.globalNotificationManager.notifyBulkScoreApproval === 'function') {
                    try {
                        window.globalNotificationManager.notifyBulkScoreApproval(nd.bib, nd.participant_name, nd.count, nd.heat, nd.run);
                    } catch (error) {
                        console.warn('Notification manager error:', error);
                        // Fall back to generic notification
                        if (typeof sendNotification === 'function') {
                            const countText = nd.count > 0 ? ` (${nd.count} scores)` : '';
                            sendNotification('success', 'All Scores Approved', `Head judge approved all scores for #${nd.bib} ${nd.participant_name}${countText}`, 'judge');
                        }
                    }
                } else if (typeof sendNotification === 'function') {
                    const countText = nd.count > 0 ? ` (${nd.count} scores)` : '';
                    sendNotification('success', 'All Scores Approved', `Head judge approved all scores for #${nd.bib} ${nd.participant_name}${countText}`, 'judge');
                }
            } else if (typeof sendNotification === 'function') {
                const countText = data.count > 0 ? ` (${data.count} scores)` : '';
                sendNotification('success', 'All Scores Approved', `Head judge approved all scores for participant${countText}`, 'judge');
            }
            
            // Refresh grid
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
        } else {
            showErrorAlert('Error approving scores: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showErrorAlert('Error approving scores');
    });
}

// Update the showSuccessToast function to accept custom messages
function showSuccessToast(message = 'Updated successfully') {
    const toastEl = document.getElementById('successToast');
    const toastBody = toastEl.querySelector('.toast-body');
    toastBody.textContent = message;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
}

// Hold-to-approve/unapprove/delete mechanism
let approvalHoldTimer = null;
let approvalHoldStartTime = null;
const APPROVAL_HOLD_DURATION = 1500; // 1.5 seconds
const DELETE_HOLD_DURATION = 2000; // 2 seconds - longer for destructive action

// Helper function to update Edit Modal approval buttons
function updateEditModalApprovalButtons(scoreId, isApproved) {
    const modal = document.getElementById('editScoreModal');
    const modalInstance = bootstrap.Modal.getInstance(modal);
    
    // Only update if modal is currently shown
    if (!modalInstance || !modal.classList.contains('show')) {
        return;
    }
    
    const approvalButtonsContainer = document.getElementById('editModalApprovalButtons');
    const approveBtn = approvalButtonsContainer.querySelector('.approve-btn');
    const unapproveBtn = approvalButtonsContainer.querySelector('.unapprove-btn');
    
    // Verify this is the same score being edited
    const currentScoreId = approveBtn.getAttribute('data-score-id');
    if (currentScoreId != scoreId) {
        return;
    }
    
    // Show/hide appropriate button based on new approval status
    if (isApproved) {
        unapproveBtn.style.display = 'inline-block';
        approveBtn.style.display = 'none';
    } else {
        unapproveBtn.style.display = 'none';
        approveBtn.style.display = 'inline-block';
    }
}

// Event delegation for approve/unapprove/delete buttons (supporting both mouse and touch)
// Unified handler for starting hold action
function handleHoldStart(e) {
    const unapproveBtn = e.target.closest('.unapprove-btn');
    const approveBtn = e.target.closest('.approve-btn');
    const deleteBtn = e.target.closest('.delete-score-btn');
    const approveAllBtn = e.target.closest('.approve-all-btn');
    const btn = unapproveBtn || approveBtn || deleteBtn || approveAllBtn;
    
    // Only handle and prevent default if we're actually on an approval or delete button
    if (btn) {
        // Prevent default behavior ONLY for these buttons to avoid conflicts
        e.preventDefault();
        
        const scoreId = btn.getAttribute('data-score-id') || document.getElementById('editScoreId').value;
        const participantId = btn.getAttribute('data-participant-id');
        const progressBar = btn.querySelector('.hold-progress');
        const isApprove = btn.classList.contains('approve-btn');
        const isDelete = btn.classList.contains('delete-score-btn');
        const isApproveAll = btn.classList.contains('approve-all-btn');
        const holdDuration = isDelete ? DELETE_HOLD_DURATION : APPROVAL_HOLD_DURATION;
        
        if (!progressBar) return;
        
        // Add holding class for visual feedback
        btn.classList.add('holding');
        
        // Reset progress bar
        progressBar.style.transition = 'none';
        progressBar.style.width = '0%';
        
        // Start hold timer
        approvalHoldStartTime = Date.now();
        
        // Animate progress bar
        requestAnimationFrame(() => {
            progressBar.style.transition = `width ${holdDuration}ms linear`;
            progressBar.style.width = '100%';
        });
        
        // Set timer to trigger approve/unapprove/delete
        approvalHoldTimer = setTimeout(() => {
            // Trigger confetti animation
            btn.classList.add('confetti-animate');
            setTimeout(() => {
                btn.classList.remove('confetti-animate');
            }, 750);
            
            // Remove holding class
            btn.classList.remove('holding');
            
            // Trigger the appropriate action
            if (isDelete) {
                deleteScore(scoreId);
            } else if (isApproveAll) {
                approveAllParticipant(participantId);
            } else if (isApprove) {
                approveScore(scoreId);
            } else {
                unapproveScore(scoreId);
            }
            
            // Visual feedback
            progressBar.style.width = '0%';
            progressBar.style.transition = 'width 200ms';
        }, holdDuration);
    }
}

// Unified handler for ending hold action
function handleHoldEnd(e) {
    const unapproveBtn = e.target.closest('.unapprove-btn');
    const approveBtn = e.target.closest('.approve-btn');
    const deleteBtn = e.target.closest('.delete-score-btn');
    const approveAllBtn = e.target.closest('.approve-all-btn');
    const btn = unapproveBtn || approveBtn || deleteBtn || approveAllBtn;
    
    if (approvalHoldTimer) {
        clearTimeout(approvalHoldTimer);
        approvalHoldTimer = null;
        
        // Remove holding class
        if (btn) {
            btn.classList.remove('holding');
        }
        
        // Reset progress bar if released early
        if (btn) {
            const progressBar = btn.querySelector('.hold-progress');
            if (progressBar) {
                const elapsed = Date.now() - approvalHoldStartTime;
                const isDelete = btn.classList.contains('delete-score-btn');
                const holdDuration = isDelete ? DELETE_HOLD_DURATION : APPROVAL_HOLD_DURATION;
                if (elapsed < holdDuration) {
                    // Released too early - reset with animation
                    progressBar.style.transition = 'width 200ms';
                    progressBar.style.width = '0%';
                }
            }
        }
    }
}

// Unified handler for canceling hold action
function handleHoldCancel(e) {
    // For touchcancel events, e.target might not be available, so check first
    if (!e.target || typeof e.target.closest !== 'function') {
        // Clear timer anyway for safety
        if (approvalHoldTimer) {
            clearTimeout(approvalHoldTimer);
            approvalHoldTimer = null;
        }
        return;
    }
    
    const unapproveBtn = e.target.closest('.unapprove-btn');
    const approveBtn = e.target.closest('.approve-btn');
    const deleteBtn = e.target.closest('.delete-score-btn');
    const approveAllBtn = e.target.closest('.approve-all-btn');
    const btn = unapproveBtn || approveBtn || deleteBtn || approveAllBtn;
    
    if (btn && approvalHoldTimer) {
        clearTimeout(approvalHoldTimer);
        approvalHoldTimer = null;
        
        // Remove holding class
        btn.classList.remove('holding');
        
        // Reset progress bar
        const progressBar = btn.querySelector('.hold-progress');
        if (progressBar) {
            progressBar.style.transition = 'width 200ms';
            progressBar.style.width = '0%';
        }
    }
}

// Mouse events
document.addEventListener('mousedown', handleHoldStart);
document.addEventListener('mouseup', handleHoldEnd);
document.addEventListener('mouseleave', handleHoldCancel);

// Touch events for mobile/tablet support
// Using passive: false only on touchstart to allow preventDefault when needed
document.addEventListener('touchstart', handleHoldStart, { passive: false });
document.addEventListener('touchend', handleHoldEnd, { passive: true });
document.addEventListener('touchcancel', handleHoldCancel, { passive: true });
        
        function getStatusBadgeClass(status) {
            switch(status) {
                case 'OK': return 'success';
                case 'DNS': return 'dark-subtle';
                case 'DNF': return 'warning';
                case 'DSQ': return 'danger';
                default: return 'secondary';
            }
        }
        
        function editScore(scoreId, scoreValue, status, figuresEncoded = '', participantName = '', bibNumber = '', category = '', judgeName = '', controlPoint = '', gender = '', club = '') {
            console.log('editScore called with:', {
                scoreId,
                scoreValue,
                status,
                figuresEncoded,
                participantName,
                bibNumber
            });
            
            document.getElementById('editScoreId').value = scoreId;
            document.getElementById('editScoreValue').value = scoreValue;
            
            // Set the radio button state based on status
            const statusRadios = document.getElementsByName('editStatus');
            statusRadios.forEach(radio => {
                radio.checked = (radio.value === status);
            });
            
            // Update the participant and context information in the modal
            const eventSelect = document.getElementById('eventSelect');
            const heatSelect = document.getElementById('heatSelect');
            const runSelect = document.getElementById('runSelect');
            
            const eventName = eventSelect.options[eventSelect.selectedIndex]?.text || 'Unknown Event';
            const heatNumber = heatSelect.value || 'Unknown';
            const runNumber = runSelect.value || 'Unknown';
            
            // Attempt to enrich gender/club if not passed
            if ((!gender || !club) && window.currentData && Array.isArray(window.currentData.participants)) {
                const found = window.currentData.participants.find(p => {
                    return (p.bib_number == bibNumber) ||
                           ((p.first_name + ' ' + p.last_name).toLowerCase() === participantName.toLowerCase());
                });
                if (found) {
                    if (!gender) gender = found.gender || '';
                    if (!club) club = found.club || '';
                    if (!category) category = found.category || category;
                }
            }

            const genderBadge = gender ? `<span class="badge bg-${gender==='M'?'info':'pink'} bg-opacity-20  ms-1">${gender}</span>` : '';
            const clubLine = club ? `<span class="badge text-muted">${club}</span><br>` : '';
            document.getElementById('editScoreInfo').innerHTML = `
                <div class="participantCol">
                    <div class="css-bib"><span class="badge bg-white fs-6 mb-1 participant-bib-badge ss-f-number">${bibNumber}</span></div>
                    <strong>${participantName}</strong>
                    ${genderBadge}
                    <br><small>${clubLine}<span class="badge text-dark">${category}</span></small>
                </div>
            `;
            // Fill score context under figures section
            const scoreContextHtml = `
                <div class="row g-2">
                    <div class="col-md-6 small"><i class="fas fa-calendar-alt me-1 text-muted"></i><strong>Event:</strong> ${eventName}</div>
                    <div class="col-md-6 small"><i class="fas fa-layer-group me-1 text-muted"></i><strong>Heat:</strong> ${heatNumber} <span class="text-muted">|</span> <strong>Run:</strong> ${runNumber}</div>
                    <div class="col-md-6 small"><i class="fas fa-user-tie me-1 text-muted"></i><strong>Judge:</strong> ${judgeName}</div>
                    <div class="col-md-6 small"><i class="fas fa-map-marker-alt me-1 text-muted"></i><strong>Control Point:</strong> <span class="badge bg-success ms-1">${controlPoint}</span></div>
                    <div class="col-md-6 small"><i class="fas fa-info-circle me-1 text-muted"></i><strong>Status:</strong> <span class="badge bg-${getStatusBadgeClass(status)}">${status}</span></div>
                </div>`;
            const scoreContextEl = document.getElementById('scoreContextInfo');
            if (scoreContextEl) scoreContextEl.innerHTML = scoreContextHtml;
            
            // Store original figures data globally for editing
            window.originalFiguresData = null;
            window.currentFiguresData = null;
            
            // Populate figures if provided
            const figWrap = document.getElementById('editScoreFigures');
            const figContent = document.getElementById('editScoreFiguresContent');
            const editBtn = document.getElementById('editFiguresBtn');
            const editor = document.getElementById('editFiguresEditor');
            
            figContent.innerHTML = '';
            editor.style.display = 'block'; // Show editor immediately
            
            // Always show the figures section, even if no figures exist
            figWrap.style.display = 'block';
            editBtn.style.display = 'none'; // Hide edit button since editor is already open
            
            if (figuresEncoded) {
                try {
                    console.log('Encoded figures:', figuresEncoded);
                    const raw = decodeURIComponent(figuresEncoded);
                    console.log('Decoded figures:', raw);
                    if (raw) {
                        const figuresData = JSON.parse(raw);
                        console.log('Parsed figures data:', figuresData);
                        
                        // Handle nested structure - figures might be in figuresData.figures
                        const actualFigures = figuresData.figures || figuresData;
                        
                        window.originalFiguresData = actualFigures;
                        window.currentFiguresData = JSON.parse(JSON.stringify(actualFigures)); // Deep clone
                        
                        const parts = [];
                        Object.keys(actualFigures || {}).forEach(cat => {
                            const items = Array.isArray(actualFigures[cat]) ? actualFigures[cat] : [];
                            if (items.length) {
                                parts.push(`<span class="badge bg-success-subtle me-1 text-dark fs-6">${cat}: ${items.join(', ')}</span>`);
                            }
                        });
                        if (parts.length) {
                            figContent.innerHTML = parts.join(' ');
                        } else {
                            figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                        }
                    }
                } catch (e) { 
                    console.error('Error parsing figures in editScore:', e);
                    // If JSON parse fails, show empty state
                figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
            }
        } else {
            console.log('No figuresEncoded provided to editScore');
            // No figures data provided, show empty state
            figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
        }
        
        // Update approval buttons in modal footer
        const approvalButtonsContainer = document.getElementById('editModalApprovalButtons');
        const approveBtn = approvalButtonsContainer.querySelector('.approve-btn');
        const unapproveBtn = approvalButtonsContainer.querySelector('.unapprove-btn');
        const deleteBtn = document.querySelector('.delete-score-btn');
        
        // Set score ID on all buttons
        approveBtn.setAttribute('data-score-id', scoreId);
        unapproveBtn.setAttribute('data-score-id', scoreId);
        if (deleteBtn) {
            deleteBtn.setAttribute('data-score-id', scoreId);
        }
        
        // Check current approval status from currentData
        let isApproved = false;
        if (window.currentData && window.currentData.scores) {
            const targetRunNumber = window.currentData.run_number;
            // Find the score in currentData to check approval status
            for (const participantId in window.currentData.scores) {
                const participantScores = window.currentData.scores[participantId];
                if (targetRunNumber && participantScores[targetRunNumber]) {
                    for (const judgeId in participantScores[targetRunNumber]) {
                        const score = participantScores[targetRunNumber][judgeId];
                        if (score.score_id == scoreId) {
                            isApproved = score.is_approved == 1;
                            break;
                        }
                    }
                } else {
                    // Check all runs
                    for (const runNum in participantScores) {
                        for (const judgeId in participantScores[runNum]) {
                            const score = participantScores[runNum][judgeId];
                            if (score.score_id == scoreId) {
                                isApproved = score.is_approved == 1;
                                break;
                            }
                        }
                        if (isApproved !== false) break;
                    }
                }
                if (isApproved !== false) break;
            }
        }
        
        // Show/hide appropriate button based on approval status
        if (isApproved) {
            unapproveBtn.style.display = 'inline-block';
            approveBtn.style.display = 'none';
        } else {
            unapproveBtn.style.display = 'none';
            approveBtn.style.display = 'inline-block';
        }
        
        // Show the approval buttons container
        approvalButtonsContainer.style.display = 'block';
        
        const modal = new bootstrap.Modal(document.getElementById('editScoreModal'));
        modal.show();
        
        // Load figures editor immediately
        startEditFigures();
    }
    
        // Keypad functions for edit score modal
        function addToEditScore(digit) {
            const input = document.getElementById('editScoreValue');
            const rangeInput = document.getElementById('editScoreRange');
            let currentValue = input.value || '';
            
            // Limit to 2 digits for scores 0-99
            if (currentValue.length < 2) {
                currentValue += digit;
                input.value = currentValue;
                rangeInput.value = currentValue;
            }
        }
        
        function backspaceEditScore() {
            const input = document.getElementById('editScoreValue');
            const rangeInput = document.getElementById('editScoreRange');
            input.value = input.value.slice(0, -1);
            rangeInput.value = input.value || 0;
        }
        
        function applyEditScoreValue() {
            const input = document.getElementById('editScoreValue');
            const saveBtn = document.getElementById('saveScoreBtn');
            
            // Validate that score value is present
            if (!input.value || input.value.trim() === '') {
                input.classList.add('is-invalid');
                return;
            }
            
            input.classList.remove('is-invalid');
            
            // Scroll modal to bottom to show footer with save button
            const modal = document.getElementById('editScoreModal');
            const modalDialog = modal.querySelector('.modal-dialog');
            
            // Wait a brief moment for any animations to complete
            setTimeout(() => {
                // Scroll the modal footer into view
                const modalFooter = modal.querySelector('.modal-footer');
                if (modalFooter) {
                    modalFooter.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                }
            }, 100);
            
            // Highlight save button
            saveBtn.classList.add('pulse-highlight');
            setTimeout(() => {
                saveBtn.classList.remove('pulse-highlight');
            }, 2000);
            
            // Focus out of input to trigger any validation
            input.blur();
        }
        
        function startEditFigures() {
            const editor = document.getElementById('editFiguresEditor');
            const editBtn = document.getElementById('editFiguresBtn');
            const categoriesContainer = document.getElementById('editFigureCategories');
            
            editBtn.style.display = 'none';
            categoriesContainer.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading format options...</div>';
            editor.style.display = 'block';
            
            // Get current event ID
            const eventId = document.getElementById('eventSelect').value;
            
            if (!eventId) {
                categoriesContainer.innerHTML = '<div class="alert alert-warning">Please select an event first</div>';
                return;
            }
            
            // Fetch format figures from server
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_format_figures');
            formData.append('event_id', eventId);
            
            // Add heat number if available
            const heatSelect = document.getElementById('currentHeat');
            if (heatSelect && heatSelect.value) {
                formData.append('heat_number', heatSelect.value);
            }
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                console.log('Format figures response:', data); // Debug logging
                
                if (data.success && data.figures && Object.keys(data.figures).length > 0) {
                    buildFiguresEditor(data.figures);
                } else {
                    let message = 'No figure options available for this scoring format';
                    if (data.message) {
                        message = data.message;
                    }
                    if (data.debug) {
                        console.log('Debug info:', data.debug);
                        if (data.debug.format_found === false) {
                            message = 'Event not found or has no scoring format assigned';
                        } else if (data.debug.figures_raw_count === 0) {
                            message = `No figure categories/items configured for format "${data.debug.format_details.name}"`;
                        } else if (!data.has_figures) {
                            message = `Format "${data.debug.format_details.name}" has categories but no figure items configured`;
                        }
                    }
                    categoriesContainer.innerHTML = `<div class="alert alert-info">${message}</div>`;
                }
            })
            .catch(error => {
                console.error('Error loading figures:', error);
                categoriesContainer.innerHTML = '<div class="alert alert-danger">Error loading figure options</div>';
            });
        }
        
        function buildFiguresEditor(availableFigures) {
            const categoriesContainer = document.getElementById('editFigureCategories');
            categoriesContainer.innerHTML = '';
            
            Object.keys(availableFigures).forEach(category => {
                const items = availableFigures[category] || [];
                const currentSelected = window.currentFiguresData ? (window.currentFiguresData[category] || []) : [];
                
                const categoryDiv = document.createElement('div');
                categoryDiv.className = 'figure-category mb-3';
                
                // Build chips for this category
                let chipsHtml = '';
                items.forEach(item => {
                    const isActive = currentSelected.includes(item.name) ? 'active' : '';
                    chipsHtml += `
                        <button type="button" class="btn btn-sm btn-outline-primary figure-chip ${isActive}" 
                                data-category="${category}" data-item="${item.name}" onclick="toggleFigureChip(this)">
                            ${item.name}
                        </button>
                    `;
                });
                
                categoryDiv.innerHTML = `
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h6 class="mb-0 text-primary">
                            <i class="fas fa-tag me-2"></i>${category}
                        </h6>
                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="clearFigureCategory('${category}')">
                            <i class="fas fa-times"></i> Clear
                        </button>
                    </div>
                    <div class="figure-chips d-flex flex-wrap gap-2">
                        ${chipsHtml}
                    </div>
                `;
                categoriesContainer.appendChild(categoryDiv);
            });
        }
        
        // Figure chip interaction functions
        function toggleFigureChip(button) {
            const category = button.getAttribute('data-category');
            const item = button.getAttribute('data-item');
            
            // Toggle active state
            button.classList.toggle('active');
            
            // Initialize currentFiguresData if needed
            if (!window.currentFiguresData) {
                window.currentFiguresData = {};
            }
            
            // Initialize category array if needed
            if (!window.currentFiguresData[category]) {
                window.currentFiguresData[category] = [];
            }
            
            // Add or remove item from the category
            const itemIndex = window.currentFiguresData[category].indexOf(item);
            if (button.classList.contains('active')) {
                // Add item if not already present
                if (itemIndex === -1) {
                    window.currentFiguresData[category].push(item);
                }
            } else {
                // Remove item if present
                if (itemIndex > -1) {
                    window.currentFiguresData[category].splice(itemIndex, 1);
                }
            }
            
            // Clean up empty categories
            if (window.currentFiguresData[category].length === 0) {
                delete window.currentFiguresData[category];
            }
        }
        
        function clearFigureCategory(category) {
            // Remove all active states from chips in this category
            const categoryChips = document.querySelectorAll(`.figure-chip[data-category="${category}"]`);
            categoryChips.forEach(chip => {
                chip.classList.remove('active');
            });
            
            // Remove category from data
            if (window.currentFiguresData && window.currentFiguresData[category]) {
                delete window.currentFiguresData[category];
            }
        }
        
        function cancelEditFigures() {
            const editor = document.getElementById('editFiguresEditor');
            const editBtn = document.getElementById('editFiguresBtn');
            
            editor.style.display = 'none';
            editBtn.style.display = 'inline-block';
            
            // Reset current data to original
            if (window.originalFiguresData) {
                window.currentFiguresData = JSON.parse(JSON.stringify(window.originalFiguresData));
            }
        }
        
        function saveFiguresEdit() {
            // The data is already collected in window.currentFiguresData by the chip interactions
            // No need to collect from select elements anymore since we're using chips
            
            // Check if we have any meaningful data
            const hasAnyData = window.currentFiguresData && Object.keys(window.currentFiguresData).some(category => 
                Array.isArray(window.currentFiguresData[category]) && window.currentFiguresData[category].length > 0
            );
            
            // Update display
            const figContent = document.getElementById('editScoreFiguresContent');
            if (hasAnyData) {
                const parts = [];
                Object.keys(window.currentFiguresData).forEach(cat => {
                    const items = window.currentFiguresData[cat];
                    if (items.length) {
                        parts.push(`<span class="badge bg-light text-dark me-1">${cat}: ${items.join(', ')}</span>`);
                    }
                });
                figContent.innerHTML = parts.join(' ');
            } else {
                figContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                // Set to null if no meaningful data
                window.currentFiguresData = null;
            }
            
            // Hide editor
            const editor = document.getElementById('editFiguresEditor');
            const editBtn = document.getElementById('editFiguresBtn');
            editor.style.display = 'none';
            editBtn.style.display = 'inline-block';
        }
        
        function loadCreateFigures() {
            const eventId = document.getElementById('eventSelect').value;
            const figuresSection = document.getElementById('createFiguresSection');
            const figuresContainer = document.getElementById('createFiguresContainer');
            const figuresContent = document.getElementById('createScoreFiguresContent');
            const editBtn = document.getElementById('createFiguresBtn');
            const editor = document.getElementById('createFiguresEditor');
            
            if (!eventId) {
                figuresSection.style.display = 'none';
                return;
            }
            
            // Show loading
            figuresContainer.innerHTML = '<div class="text-muted"><i class="fas fa-spinner fa-spin me-2"></i>Loading figure options...</div>';
            figuresSection.style.display = 'block';
            // Initialize read-only content and controls similar to Edit modal
            if (figuresContent) figuresContent.innerHTML = '<span class="text-muted">No figures selected</span>';
            if (editor) editor.style.display = 'block'; // Show editor immediately
            if (editBtn) editBtn.style.display = 'none'; // Hide edit button since editor is already open
            
            const formData = new FormData();
            formData.append('action', 'get_format_figures');
            formData.append('event_id', eventId);
            
            // Add heat number if available
            const heatSelect = document.getElementById('currentHeat');
            if (heatSelect && heatSelect.value) {
                formData.append('heat_number', heatSelect.value);
            }
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                console.log('Create figures response:', data);
                
                if (data.success && data.figures && Object.keys(data.figures).length > 0) {
                    buildCreateFiguresEditor(data.figures);
                    // Editor is already visible from initialization
                    if (figuresContent && !figuresContent.innerHTML.trim()) {
                        figuresContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                    }
                } else {
                    let message = 'No figure options available for this scoring format';
                    if (data.message) message = data.message;
                    figuresContainer.innerHTML = `<div class="alert alert-info mb-0">${message}</div>`;
                    // Hide edit button if nothing to edit
                    if (editBtn) editBtn.style.display = 'none';
                }
            })
            .catch(error => {
                console.error('Error loading create figures:', error);
                figuresContainer.innerHTML = '<div class="alert alert-danger mb-0">Error loading figure options</div>';
                if (editBtn) editBtn.style.display = 'none';
            });
        }
        
        function buildCreateFiguresEditor(availableFigures) {
            const container = document.getElementById('createFiguresContainer');
            let html = '';
            
            Object.keys(availableFigures).forEach(category => {
                const items = availableFigures[category];
                if (items && items.length > 0) {
                    // Build chips for this category
                    let chipsHtml = '';
                    items.forEach(item => {
                        chipsHtml += `
                            <button type="button" class="btn btn-sm btn-outline-primary figure-chip" 
                                    data-category="${category}" data-item="${item.name}" onclick="toggleCreateFigureChip(this)">
                                ${item.name}
                            </button>
                        `;
                    });
                    
                    html += `
                        <div class="figure-category mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <h6 class="mb-0 text-primary">
                                    <i class="fas fa-tag me-2"></i>${category}
                                </h6>
                                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="clearCreateFigureCategory('${category}')">
                                    <i class="fas fa-times"></i> Clear
                                </button>
                            </div>
                            <div class="figure-chips d-flex flex-wrap gap-2">
                                ${chipsHtml}
                            </div>
                        </div>
                    `;
                }
            });
            
            if (html) {
                container.innerHTML = html;
            } else {
                container.innerHTML = '<div class="alert alert-info mb-0">No figure categories configured for this format</div>';
            }
            
            // Initialize Lucide icons for the figure display content
            if (window.initializeLucideIcons) {
                window.initializeLucideIcons();
            }
        }

        // Create modal figures editor controls (mirror Edit modal behavior)
        function startCreateFiguresEdit() {
            const editor = document.getElementById('createFiguresEditor');
            const editBtn = document.getElementById('createFiguresBtn');
            if (editBtn) editBtn.style.display = 'none';
            if (editor) editor.style.display = 'block';
        }
        function cancelCreateFiguresEdit() {
            const editor = document.getElementById('createFiguresEditor');
            const editBtn = document.getElementById('createFiguresBtn');
            if (editor) editor.style.display = 'none';
            if (editBtn) editBtn.style.display = 'inline-block';
        }
        function saveCreateFiguresEdit() {
            const figuresContent = document.getElementById('createScoreFiguresContent');
            const hasAnyData = window.createFiguresData && Object.keys(window.createFiguresData).some(category => 
                Array.isArray(window.createFiguresData[category]) && window.createFiguresData[category].length > 0
            );
            if (figuresContent) {
                if (hasAnyData) {
                    const parts = [];
                    Object.keys(window.createFiguresData).forEach(cat => {
                        const items = window.createFiguresData[cat];
                        if (items.length) {
                            parts.push(`<span class="badge bg-light text-dark me-1">${cat}: ${items.join(', ')}</span>`);
                        }
                    });
                    figuresContent.innerHTML = parts.join(' ');
                } else {
                    figuresContent.innerHTML = '<span class="text-muted">No figures selected</span>';
                }
            }
            const editor = document.getElementById('createFiguresEditor');
            const editBtn = document.getElementById('createFiguresBtn');
            if (editor) editor.style.display = 'none';
            if (editBtn) editBtn.style.display = 'inline-block';
        }
        
        // Create figure chip interaction functions
        window.createFiguresData = {};
        
        function toggleCreateFigureChip(button) {
            const category = button.getAttribute('data-category');
            const item = button.getAttribute('data-item');
            
            // Toggle active state
            button.classList.toggle('active');
            
            // Initialize category array if needed
            if (!window.createFiguresData[category]) {
                window.createFiguresData[category] = [];
            }
            
            // Add or remove item from the category
            const itemIndex = window.createFiguresData[category].indexOf(item);
            if (button.classList.contains('active')) {
                // Add item if not already present
                if (itemIndex === -1) {
                    window.createFiguresData[category].push(item);
                }
            } else {
                // Remove item if present
                if (itemIndex > -1) {
                    window.createFiguresData[category].splice(itemIndex, 1);
                }
            }
            
            // Clean up empty categories
            if (window.createFiguresData[category].length === 0) {
                delete window.createFiguresData[category];
            }
        }
        
        function clearCreateFigureCategory(category) {
            // Remove all active states from chips in this category
            const categoryChips = document.querySelectorAll(`#createFiguresContainer .figure-chip[data-category="${category}"]`);
            categoryChips.forEach(chip => {
                chip.classList.remove('active');
            });
            
            // Remove category from data
            if (window.createFiguresData && window.createFiguresData[category]) {
                delete window.createFiguresData[category];
            }
        }
        
        function getCreateFiguresData() {
            // Return the data collected by the chip interactions
            // Check if we have any meaningful data
            const hasAnyData = window.createFiguresData && Object.keys(window.createFiguresData).some(category => 
                Array.isArray(window.createFiguresData[category]) && window.createFiguresData[category].length > 0
            );
            
            // Return null if no meaningful data, otherwise return the data
            return hasAnyData ? window.createFiguresData : null;
        }
        
        function clearScoreGrid() {
            const container = document.getElementById('scoresContainer');
            const sortControls = document.getElementById('sortControls');
            const noDataMessage = document.getElementById('noDataMessage');
            const metricTotal = document.getElementById('metricTotal');
            const metricPending = document.getElementById('metricPending');
            const metricApproved = document.getElementById('metricApproved');
            
            // Only hide if not already hidden to prevent flickering
            if (container && container.style.display !== 'none') {
                container.style.display = 'none';
            }
            if (sortControls && !sortControls.classList.contains('d-none')) {
                sortControls.classList.add('d-none');
            }
            
            // Reset metrics to dashes - don't hide the container
            if (metricTotal) metricTotal.textContent = '--';
            if (metricPending) metricPending.textContent = '--';
            if (metricApproved) metricApproved.textContent = '--';
            
            if (noDataMessage) {
                if (noDataMessage.style.display !== 'block') {
                    noDataMessage.style.display = 'block';
                }
                // Only update innerHTML if it's different to prevent reflow
                const newContent = `
                    <i class="fas fa-clipboard-list fa-3x mb-3 opacity-50"></i>
                    <h5>Select Event, Heat, and Run to view scores</h5>
                    <p>Use the controls above to navigate to specific scoring sessions</p>
                `;
                if (noDataMessage.innerHTML.trim() !== newContent.trim()) {
                    noDataMessage.innerHTML = newContent;
                }
            }
            currentData = null;
        }
        
        function showLoading(show) {
            const refreshBtn = document.getElementById('refreshBtn');
            const metricsDisplay = document.getElementById('metricsDisplay');
            
            if (refreshBtn) {
                if (show) {
                    refreshBtn.disabled = true;
                    refreshBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                } else {
                    refreshBtn.disabled = false;
                    refreshBtn.innerHTML = '<i class="fas fa-sync-alt"></i>';
                }
            }
            
            // Hide metrics while loading
            if (metricsDisplay && show) {
                //metricsDisplay.classList.add('d-none');
            }
        }
        
        // Scroll position preservation functions
        function preserveScrollPosition() {
            const tableContainer = document.querySelector('#scoresContainer .table-responsive');
            if (tableContainer) {
                window.lastScrollTop = tableContainer.scrollTop;
                window.lastScrollLeft = tableContainer.scrollLeft;
            }
        }
        
        function restoreScrollPosition() {
            const tableContainer = document.querySelector('#scoresContainer .table-responsive');
            if (tableContainer && window.lastScrollTop !== undefined) {
                // Use requestAnimationFrame for smoother restoration
                requestAnimationFrame(() => {
                    tableContainer.scrollTop = window.lastScrollTop;
                    tableContainer.scrollLeft = window.lastScrollLeft || 0;
                    
                    // Double-check restoration after a short delay
                    setTimeout(() => {
                        if (tableContainer.scrollTop !== window.lastScrollTop) {
                            tableContainer.scrollTop = window.lastScrollTop;
                            tableContainer.scrollLeft = window.lastScrollLeft || 0;
                        }
                    }, 50);
                });
            }
        }
        
        function showSuccessToast() {
            const toastEl = document.getElementById('successToast');
            const toast = new bootstrap.Toast(toastEl);
            toast.show();
        }
        
        function startAutoRefresh() {
            stopAutoRefresh();
            autoRefreshInterval = setInterval(() => {
                const eventId = document.getElementById('eventSelect').value;
                if (eventId) {
                    loadScores();
                }
            }, 5000);
        }
        
        function stopAutoRefresh() {
            if (autoRefreshInterval) {
                clearInterval(autoRefreshInterval);
                autoRefreshInterval = null;
            }
        }
        
        // Handle score update
        document.getElementById('saveScoreBtn').addEventListener('click', function() {
            const scoreId = document.getElementById('editScoreId').value;
            const scoreValue = document.getElementById('editScoreValue').value;
            
            // Validate score value is present
            if (!scoreValue || scoreValue.trim() === '') {
                const input = document.getElementById('editScoreValue');
                input.classList.add('is-invalid');
                input.focus();
                showErrorAlert('Please enter a score value');
                return;
            }
            
            // Get status from radio buttons
            const statusRadio = document.querySelector('input[name="editStatus"]:checked');
            const status = statusRadio ? statusRadio.value : 'OK';
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'update_score');
            formData.append('score_id', scoreId);
            formData.append('score_value', scoreValue);
            formData.append('status', status);
            
            // Include figures data - send null for empty data
            if (window.currentFiguresData !== undefined) {
                // Send 'null' string for null values, JSON string for actual data
                const figuresValue = window.currentFiguresData === null ? 'null' : JSON.stringify(window.currentFiguresData);
                formData.append('figures_json', figuresValue);
            }
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const modal = bootstrap.Modal.getInstance(document.getElementById('editScoreModal'));
                    modal.hide();
                    
                    showSuccessToast();
                    
                    // Refresh grid
                    if (document.getElementById('runSelect').value) {
                        loadScores();
                    }
                } else {
                    showErrorAlert('Error updating score: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showErrorAlert('Error updating score');
            });
        });

        // Handle score creation
        // Handle score creation
document.getElementById('createScoreBtn').addEventListener('click', function() {
    const participantId = document.getElementById('createParticipantId').value;
    const judgeId = document.getElementById('createJudgeId').value;
    const scoreValue = document.getElementById('createScoreValue').value;
    
    // Validate score value is present
    if (!scoreValue || scoreValue.trim() === '') {
        const input = document.getElementById('createScoreValue');
        input.classList.add('is-invalid');
        input.focus();
        showErrorAlert('Please enter a score value');
        return;
    }
    
    // Get status from radio buttons
    const statusRadio = document.querySelector('input[name="createStatus"]:checked');
    const status = statusRadio ? statusRadio.value : 'OK';
    
    const eventId = document.getElementById('eventSelect').value;
    const heatNumber = document.getElementById('heatSelect').value;
    const runNumber = document.getElementById('runSelect').value;
    
    // Get figures data
    const figuresData = getCreateFiguresData();
    
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'create_score');
    formData.append('event_id', eventId);
    formData.append('heat_number', heatNumber);
    formData.append('run_number', runNumber);
    formData.append('participant_id', participantId);
    formData.append('judge_id', judgeId);
    formData.append('score_value', scoreValue);
    formData.append('status', status);
    
    // Add figures data if available
    if (figuresData) {
        formData.append('figures_json', JSON.stringify(figuresData));
    }
    
    fetch(window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const modal = bootstrap.Modal.getInstance(document.getElementById('createScoreModal'));
            modal.hide();
            
            showSuccessToast('Score created successfully');
            
            // Refresh grid
            if (document.getElementById('runSelect').value) {
                loadScores();
            }
        } else {
            showErrorAlert('Error creating score: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showErrorAlert('Error creating score');
    });
});

// NEW: Handle Enter key press in create score form
document.getElementById('createScoreForm').addEventListener('submit', function(e) {
    e.preventDefault(); // Prevent default form submission
    document.getElementById('createScoreBtn').click(); // Trigger the create button
});

// NEW: Handle Enter key press in score value input
document.getElementById('createScoreValue').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        e.preventDefault(); // Prevent default form submission
        document.getElementById('createScoreBtn').click(); // Trigger the create button
    }
});

// OPTIONAL: Also add the same for edit score form for consistency
document.getElementById('editScoreForm').addEventListener('submit', function(e) {
    e.preventDefault(); // Prevent default form submission
    document.getElementById('saveScoreBtn').click(); // Trigger the save button
});

document.getElementById('editScoreValue').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        e.preventDefault(); // Prevent default form submission
        document.getElementById('saveScoreBtn').click(); // Trigger the save button
    }
});

        // Utility functions for debugging localStorage
        function debugSelections() {
            if (window.EventSelectionManager) {
                console.log('Current Selections:');
                console.log('Event:', window.EventSelectionManager.getSavedSelection('event'));
                console.log('Heat:', window.EventSelectionManager.getSavedSelection('heat'));  
                console.log('Run:', window.EventSelectionManager.getSavedSelection('run'));
            }
        }
        
        // Add debug function to window for console access
        window.debugEventSelections = debugSelections;
        
        // Show visual indicator when selections are restored
        function showRestoredIndicator(type, value) {
            const indicator = document.createElement('div');
            indicator.className = 'position-fixed top-0 start-50 translate-middle-x mt-3 alert alert-info alert-dismissible fade show';
            indicator.style.zIndex = '9999';
            indicator.innerHTML = `
                <i class="fas fa-history me-2"></i>
                Restored ${type}: <strong>${value}</strong>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(indicator);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (indicator.parentNode) {
                    indicator.remove();
                }
            }, 3000);
        }
        
        function clearScoreGrid() {
            const container = document.getElementById('scoresContainer');
            const noDataMessage = document.getElementById('noDataMessage');
            const sortControls = document.getElementById('sortControls');
            const metricTotal = document.getElementById('metricTotal');
            const metricPending = document.getElementById('metricPending');
            const metricApproved = document.getElementById('metricApproved');
            
            container.style.display = 'none';
            sortControls.classList.add('d-none');
            noDataMessage.style.display = 'block';
            
            // Reset metrics to dashes - don't hide the container
            if (metricTotal) metricTotal.textContent = '--';
            if (metricPending) metricPending.textContent = '--';
            if (metricApproved) metricApproved.textContent = '--';
            
            currentData = null;
        }
        
        // Results Modal Functions
        function showEventResults() {
            const eventId = document.getElementById('eventSelect').value;
            if (!eventId) {
                showErrorAlert('Please select an event first');
                return;
            }
            
            const modal = new bootstrap.Modal(document.getElementById('heatResultsModal'));
            modal.show();
            
            // Load saved configurations
            loadHeatConfigurations(eventId);
        }
        
        function loadHeatConfigurations(eventId) {
            const configsList = document.getElementById('savedConfigsList');
            
            // Show loading state
            configsList.innerHTML = `
                <div class="text-center py-3 text-muted">
                    <div class="spinner-border spinner-border-sm" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 small">Loading configurations...</p>
                </div>
            `;
            
            // Fetch all configurations for the event using public_dashboard_api.php
            fetch('../api/public_dashboard_api.php?action=get_configurations&event_id=' + eventId)
                .then(response => response.json())
                .then(data => {
                    if (!data.success || !data.configurations || data.configurations.length === 0) {
                        configsList.innerHTML = `
                            <div class="alert alert-warning mb-0 small">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                No saved reports found for this event.
                            </div>
                        `;
                        document.getElementById('resultsContentDisplay').innerHTML = `
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                No saved reports available for this event.
                            </div>
                        `;
                        return;
                    }
                    
                    // Use all configurations
                    const allConfigs = data.configurations;
                    
                    // Build configuration list
                    let html = '';
                    allConfigs.forEach((config, index) => {
                        const isActive = index === 0 ? 'active' : '';
                        const viewTypeIcon = config.view_type === 'start_list' ? 'fa-list-ol' : 'fa-table';
                        const viewTypeLabel = config.view_type === 'start_list' ? 'Start List' : 'Summary';
                        const heatLabel = config.heat_number ? 'Heat ' + config.heat_number : 'All Heats';
                        
                        html += '<button type="button" class="config-button ' + isActive + '" ';
                        html += 'data-config-id="' + config.id + '" ';
                        html += 'onclick="loadHeatConfiguration(' + config.id + ', \'' + config.view_type + '\')" ';
                        html += 'title="Click to load">';
                        html += '<div class="d-flex flex-column">';
                        html += '<div class="d-flex justify-content-between align-items-start mb-1">';
                        html += '<span class="me-2 text-start flex-grow-1">';
                        html += '<i class="fas ' + viewTypeIcon + ' me-2"></i>' + escapeHtml(config.name);
                        html += '</span>';
                        html += '<span class="badge bg-light text-dark">' + viewTypeLabel + '</span>';
                        html += '</div>';
                        html += '<small class="text-muted">';
                        html += '<i class="fas fa-fire me-1"></i>' + heatLabel;
                        html += '</small>';
                        html += '</div>';
                        html += '</button>';
                    });
                    
                    configsList.innerHTML = html;
                    
                    // Auto-load first configuration
                    if (allConfigs.length > 0) {
                        loadHeatConfiguration(allConfigs[0].id, allConfigs[0].view_type);
                    }
                })
                .catch(error => {
                    console.error('Error loading configurations:', error);
                    configsList.innerHTML = `
                        <div class="alert alert-danger mb-0 small">
                            <i class="fas fa-times-circle me-2"></i>
                            Error loading configurations: ` + escapeHtml(error.message) + `
                        </div>
                    `;
                });
        }
        
        function loadHeatConfiguration(configId, viewType) {
            const contentDisplay = document.getElementById('resultsContentDisplay');
            
            // Update active state
            document.querySelectorAll('#savedConfigsList .config-button').forEach(btn => {
                btn.classList.remove('active');
            });
            const targetBtn = document.querySelector('[data-config-id="' + configId + '"]');
            if (targetBtn) {
                targetBtn.classList.add('active');
            }
            
            // Show loading
            contentDisplay.innerHTML = `
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 text-muted">Loading results...</p>
                </div>
            `;
            
            // Determine API endpoint based on view type
            const apiEndpoint = viewType === 'start_list' 
                ? '../api/start_list_api.php' 
                : '../api/summary_table_api.php';
            
            // Load configuration
            fetch(apiEndpoint + '?config_id=' + configId + '&format=html')
                .then(response => {
                    if (!response.ok) {
                        throw new Error('HTTP ' + response.status);
                    }
                    return response.text();
                })
                .then(html => {
                    contentDisplay.innerHTML = html;
                })
                .catch(error => {
                    console.error('Error loading configuration:', error);
                    contentDisplay.innerHTML = `
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            Error loading results: ` + error.message + `
                        </div>
                    `;
                });
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // ===== QUEUE PANEL FUNCTIONS =====
        
        function updateHeatQueuePanel(queue = [], heatSettings = {}) {
            const subtitleEl = document.getElementById('queueSummarySubtitle');
            const totalEl = document.getElementById('queueTotalCount');
            const performingEl = document.getElementById('queuePerformingCount');
            const waitingEl = document.getElementById('queueWaitingCount');
            const completedEl = document.getElementById('queueCompletedCount');
            const runLabelEl = document.getElementById('queueRunLabel');
            const updatedAtEl = document.getElementById('queueUpdatedAt');

            if (!subtitleEl) return;

            const safeQueue = Array.isArray(queue) ? queue : [];
            const resolvedHeatSettings = heatSettings || currentHeatSettings || {};
            const fallbackHeat = document.getElementById('heatSelect')?.value || '--';
            const fallbackRun = document.getElementById('runSelect')?.value || '1';

            const heatNumber = resolvedHeatSettings.heat_number || fallbackHeat;
            const heatStatus = (resolvedHeatSettings.status || (safeQueue.length ? 'active' : 'pending')).replace(/_/g, ' ');

            subtitleEl.textContent = safeQueue.length
                ? 'Heat ' + heatNumber + ' · ' + heatStatus.toUpperCase()
                : (heatNumber && heatNumber !== '--'
                    ? 'Heat ' + heatNumber + ' · waiting to start'
                    : 'Waiting for participants…');

            if (totalEl) totalEl.textContent = safeQueue.length ? safeQueue.length : '--';

            const performingParticipants = safeQueue.filter(isParticipantPerforming);
            const completedParticipants = safeQueue.filter(isParticipantCompleted);
            const nextParticipants = getNextParticipants(safeQueue);
            const waitingParticipants = safeQueue.filter(participant =>
                !performingParticipants.includes(participant) &&
                !completedParticipants.includes(participant)
            );

            if (performingEl) performingEl.textContent = performingParticipants.length;
            if (waitingEl) waitingEl.textContent = waitingParticipants.length;
            if (completedEl) completedEl.textContent = completedParticipants.length;

            const activeRun = Number(resolvedHeatSettings?.active_run) || Number(safeQueue?.[0]?.current_run) || Number(fallbackRun) || 1;
            const totalRuns = Number(resolvedHeatSettings?.runs_count) || Number(safeQueue?.[0]?.total_runs) || activeRun;
            if (runLabelEl) runLabelEl.textContent = activeRun + '/' + Math.max(activeRun, totalRuns);

            renderQueueSlots(performingParticipants, nextParticipants);
            renderQueueChips(waitingParticipants);
        }

        function isParticipantPerforming(participant) {
            const queueStatus = (participant?.status || '').toLowerCase();
            const participantStatus = (participant?.participant_status || '').toLowerCase();
            return queueStatus === 'current' || participantStatus === 'performing';
        }

        function isParticipantCompleted(participant) {
            const queueStatus = (participant?.status || '').toLowerCase();
            const participantStatus = (participant?.participant_status || '').toLowerCase();
            return queueStatus === 'completed' || queueStatus === 'done' || participantStatus === 'complete' || participant.queue_position === -1;
        }

        function getNextParticipants(queue = []) {
            return queue
                .filter(participant => {
                    const queueStatus = (participant?.status || '').toLowerCase();
                    return queueStatus === 'next' || participant.queue_position === 1;
                })
                .slice(0, 2);
        }

        function renderQueueSlots(performing = [], next = []) {
            const container = document.getElementById('queuePerformingSlots');
            if (!container) return;

            const performingHtml = buildSlotContent(performing, 'No athlete on course');
            const nextHtml = buildSlotContent(next, 'Queue forming soon');

            container.innerHTML = 
                '<div class="performing-slot current">' +
                    '<span class="slot-label">Current</span>' +
                    '<div class="slot-content">' + performingHtml + '</div>' +
                '</div>' +
                '<div class="performing-slot next">' +
                    '<span class="slot-label">Up Next</span>' +
                    '<div class="slot-content">' + nextHtml + '</div>' +
                '</div>';
        }

        function buildSlotContent(list = [], emptyText = '') {
            if (!Array.isArray(list) || list.length === 0) {
                return '<span class="text-muted small">' + escapeHtml(emptyText) + '</span>';
            }
            return list.slice(0, 2).map(participant => {
                const name = formatQueueParticipantName(participant);
                const bib = participant?.bib_number ? '#' + escapeHtml(participant.bib_number) : '';
                return '<span class="slot-chip">' +
                    '<span class="slot-bib">' + bib + '</span>' +
                    '<span>' + name + '</span>' +
                '</span>';
            }).join('');
        }

        function renderQueueChips(participants = []) {
            const container = document.getElementById('queueChipsList');
            if (!container) return;

            if (!participants.length) {
                container.innerHTML = '<div class="text-muted small text-center py-2">All caught up – no one waiting.</div>';
                return;
            }

            container.innerHTML = participants.slice(0, 8).map((participant, index) => {
                const queueStatus = (participant?.status || 'waiting').toLowerCase();
                const statusClass = 'queue-chip-' + queueStatus;
                const name = formatQueueParticipantName(participant);
                const category = escapeHtml(participant?.display_category || participant?.event_category || '');
                const bib = participant?.bib_number ? 'BIB ' + escapeHtml(participant.bib_number) : '';
                return '<div class="queue-chip ' + statusClass + '" data-bib="' + escapeHtml(participant?.bib_number || '') + '">' +
                    '<div class="queue-chip-rank">' + (index + 1) + '</div>' +
                    '<div class="queue-chip-body">' +
                        '<div class="queue-chip-name">' + name + '</div>' +
                        '<div class="queue-chip-meta">' +
                            (bib ? '<span class="badge bg-light text-dark me-1">' + bib + '</span>' : '') +
                            category +
                        '</div>' +
                    '</div>' +
                '</div>';
            }).join('');
        }

        function formatQueueParticipantName(participant) {
            const first = participant?.display_first_name || participant?.first_name || '';
            const last = participant?.display_last_name || participant?.last_name || '';
            const combined = (first + ' ' + last).trim();
            if (combined) return escapeHtml(combined);
            if (participant?.name) return escapeHtml(participant.name);
            if (participant?.bib_number) return 'BIB ' + escapeHtml(participant.bib_number);
            return 'Unknown Athlete';
        }

        function resetHeadJudgeQueuePanel(message = 'Waiting for participants…') {
            const subtitleEl = document.getElementById('queueSummarySubtitle');
            if (subtitleEl) subtitleEl.textContent = message;
            const totalEl = document.getElementById('queueTotalCount');
            if (totalEl) totalEl.textContent = '--';
            ['queuePerformingCount', 'queueWaitingCount', 'queueCompletedCount'].forEach(id => {
                const el = document.getElementById(id);
                if (el) el.textContent = '0';
            });
            const runLabelEl = document.getElementById('queueRunLabel');
            if (runLabelEl) runLabelEl.textContent = '--';
            const slots = document.getElementById('queuePerformingSlots');
            if (slots) {
                slots.innerHTML = 
                    '<div class="performing-slot current">' +
                        '<span class="slot-label">Current</span>' +
                        '<div class="slot-content text-muted small">Waiting for data...</div>' +
                    '</div>' +
                    '<div class="performing-slot next">' +
                        '<span class="slot-label">Up Next</span>' +
                        '<div class="slot-content text-muted small">Queue forming...</div>' +
                    '</div>';
            }
            const chips = document.getElementById('queueChipsList');
            if (chips) {
                chips.innerHTML = '<div class="text-muted small text-center py-2">Queue will appear once participants load.</div>';
            }
        }

        function getSelectedEventId() {
            const select = document.getElementById('eventSelect');
            return select ? select.value : '';
        }

        async function refreshHeadJudgeQueuePanel() {
            const eventId = getSelectedEventId();
            const heatSelect = document.getElementById('heatSelect');
            const heatNumber = heatSelect ? heatSelect.value : '';

            if (!eventId) {
                resetHeadJudgeQueuePanel('Select an event to monitor queue');
                return;
            }
            if (!heatNumber) {
                resetHeadJudgeQueuePanel('Select a heat to load queue');
                return;
            }
            if (headJudgeQueueLoading) {
                return;
            }

            headJudgeQueueLoading = true;
            try {
                const response = await fetch('../admin/heat_management_api.php?action=get_queue&event_id=' + eventId + '&heat_number=' + heatNumber + '&mode=sort_order', {
                    credentials: 'same-origin'
                });
                if (!response.ok) {
                    throw new Error('Failed to query queue API');
                }
                const data = await response.json();
                const queue = data.queue || [];
                currentHeatSettings = data.heat_settings || {
                    heat_number: heatNumber,
                    active_run: data.active_run || currentHeatSettings.active_run || document.getElementById('runSelect')?.value || 1,
                    runs_count: data.heat_settings?.runs_count || currentHeatSettings.runs_count
                };
                window.currentHeatSettings = currentHeatSettings;
                updateHeatQueuePanel(queue, currentHeatSettings);
            } catch (error) {
                console.error('Queue load failed:', error);
                const subtitleEl = document.getElementById('queueSummarySubtitle');
                if (subtitleEl) subtitleEl.textContent = 'Queue unavailable';
                const chips = document.getElementById('queueChipsList');
                if (chips) {
                    chips.innerHTML = '<div class="alert-placeholder text-danger small"><i class="fas fa-exclamation-triangle me-1"></i>' + escapeHtml(error.message) + '</div>';
                }
            } finally {
                headJudgeQueueLoading = false;
            }
        }

        function initializeHeadJudgeNotifications(forceRestart = false) {
            const alertsList = document.getElementById('alertsList');
            if (!alertsList) {
                return;
            }
            const eventId = getSelectedEventId();
            if (!eventId) {
                resetHeadJudgeNotificationsUI();
                return;
            }
            if (heatNotificationsInitialized && !forceRestart) {
                return;
            }
            heatNotificationsInitialized = true;
            latestNotificationId = 0;
            heatNotifications = [];
            refreshHeadJudgeAlerts(true);
            if (heatNotificationsTimer) {
                clearInterval(heatNotificationsTimer);
            }
            heatNotificationsTimer = setInterval(() => {
                if (!document.hidden) {
                    refreshHeadJudgeAlerts();
                }
            }, HEAD_JUDGE_ALERT_POLL_INTERVAL);
        }

        function resetHeadJudgeNotificationsUI(message = 'Select an event to monitor alerts') {
            if (heatNotificationsTimer) {
                clearInterval(heatNotificationsTimer);
                heatNotificationsTimer = null;
            }
            heatNotificationsInitialized = false;
            heatNotifications = [];
            latestNotificationId = 0;
            const alertsList = document.getElementById('alertsList');
            if (alertsList) {
                alertsList.innerHTML = '<div class="text-muted small text-center py-3">' + escapeHtml(message) + '</div>';
            }
            const statusLabel = document.getElementById('alertsStatusLabel');
            if (statusLabel) {
                statusLabel.textContent = message;
                statusLabel.classList.remove('text-danger');
            }
            const countBadge = document.getElementById('alertsCountBadge');
            if (countBadge) {
                countBadge.textContent = '0';
            }
        }

        async function refreshHeadJudgeAlerts(forceFullReload = false) {
            const alertsList = document.getElementById('alertsList');
            const eventId = getSelectedEventId();
            if (!alertsList) {
                return;
            }
            if (!eventId) {
                resetHeadJudgeNotificationsUI();
                return;
            }
            if (notificationsLoading && !forceFullReload) {
                return;
            }

            const statusLabel = document.getElementById('alertsStatusLabel');
            if (statusLabel) {
                statusLabel.textContent = 'Updating…';
                statusLabel.classList.remove('text-danger');
            }

            notificationsLoading = true;
            try {
                const params = new URLSearchParams({
                    action: 'get_notifications',
                    event_id: eventId,
                    category: 'heat',
                    limit: '8'
                });
                if (forceFullReload || !latestNotificationId) {
                    params.append('since', '0');
                } else {
                    params.append('since', String(latestNotificationId));
                }

                const response = await fetch('../api/notification_api.php?' + params.toString(), {
                    credentials: 'same-origin'
                });
                const data = await response.json();
                if (!data.success) {
                    throw new Error(data.error || 'Failed to load alerts');
                }

                latestNotificationId = Math.max(latestNotificationId, data.latest_id || 0);
                
                // Simple render for now - just show count
                const countBadge = document.getElementById('alertsCountBadge');
                if (countBadge) {
                    countBadge.textContent = (data.notifications || []).length;
                }
                
                if (statusLabel) {
                    statusLabel.textContent = 'Updated ' + new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                }
            } catch (error) {
                console.error('Heat alerts error:', error);
                if (statusLabel) {
                    statusLabel.textContent = 'Alerts unavailable';
                    statusLabel.classList.add('text-danger');
                }
            } finally {
                notificationsLoading = false;
            }
        }
    </script>
    
    <?php
    // Render the notification system
    render_notification_system();
    ?>

    <?php include_once '../includes/essential_scripts.php'; ?>
    <?php include '../admin/footer.php'; ?>
</body>
</html>