<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

header('Content-Type: application/json');

$event_id = isset($_GET['event_id']) ? (int)$_GET['event_id'] : null;

if (!$event_id) {
    echo json_encode([
        'success' => false,
        'message' => 'Missing event_id'
    ]);
    exit;
}

try {
    // Step 1: prefer the heat explicitly flagged as active
    $activeHeatStmt = $pdo->prepare(
        "SELECT heat_number, active_run, runs_count, status
         FROM event_heat_settings
         WHERE event_id = ? AND is_active = 1
         ORDER BY heat_number ASC
         LIMIT 1"
    );
    $activeHeatStmt->execute([$event_id]);
    $activeHeat = $activeHeatStmt->fetch(PDO::FETCH_ASSOC);

    if (!$activeHeat) {
        // Step 2: fall back to any heat whose status is explicitly 'active'
        $statusHeatStmt = $pdo->prepare(
            "SELECT heat_number, active_run, runs_count, status
             FROM event_heat_settings
             WHERE event_id = ? AND status = 'active'
             ORDER BY heat_number ASC
             LIMIT 1"
        );
        $statusHeatStmt->execute([$event_id]);
        $activeHeat = $statusHeatStmt->fetch(PDO::FETCH_ASSOC);
    }

    if (!$activeHeat) {
        // Step 3: default to the first configured heat for the event
        $firstHeatStmt = $pdo->prepare(
            "SELECT heat_number, active_run, runs_count, status
             FROM event_heat_settings
             WHERE event_id = ?
             ORDER BY heat_number ASC
             LIMIT 1"
        );
        $firstHeatStmt->execute([$event_id]);
        $activeHeat = $firstHeatStmt->fetch(PDO::FETCH_ASSOC);
    }

    if ($activeHeat) {
        $heatNumber = (int)$activeHeat['heat_number'];
        $runNumber = (int)($activeHeat['active_run'] ?? 0);
        $runsCount = (int)($activeHeat['runs_count'] ?? 0);

        if ($runNumber < 1) {
            $runNumber = 1;
        }
        if ($runsCount > 0 && $runNumber > $runsCount) {
            $runNumber = $runsCount;
        }

        echo json_encode([
            'success' => true,
            'heat_number' => $heatNumber,
            'run_number' => $runNumber,
            'heat_status' => $activeHeat['status'] ?? null,
            'source' => 'event_heat_settings'
        ]);
        exit;
    }

    // Absolute fallback: use the earliest participant assignment
    $participantFallbackStmt = $pdo->prepare(
        "SELECT DISTINCT ep.heat_number
         FROM event_participants ep
         WHERE ep.event_id = ? AND ep.heat_number IS NOT NULL
         ORDER BY ep.heat_number ASC
         LIMIT 1"
    );
    $participantFallbackStmt->execute([$event_id]);
    $participantHeat = $participantFallbackStmt->fetch(PDO::FETCH_ASSOC);

    if ($participantHeat) {
        echo json_encode([
            'success' => true,
            'heat_number' => (int)$participantHeat['heat_number'],
            'run_number' => 1,
            'source' => 'participants'
        ]);
        exit;
    }

    echo json_encode([
        'success' => false,
        'message' => 'No heats configured for this event'
    ]);
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
