<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../includes/auth.php';
include '../includes/db.php';
include '../includes/event_access.php';


// Check if user has permission to access judge dashboard
$scoring_permissions = ['submit_scores', 'view_scores', 'approve_scores'];
if (!hasAnyPermission($scoring_permissions)) {
    header("Location: ../login.php?error=unauthorized");
    exit;
}

$judge_id = $_SESSION['user_id'];
$judge_name = $_SESSION['username'] ?? 'Judge';

// Check if user is a head judge in users table OR is admin
$is_head_stmt = $pdo->prepare("SELECT judge_id FROM users WHERE id = ? AND judge_id = 1");
$is_head_stmt->execute([$judge_id]);
$is_head = $is_head_stmt->fetchColumn();

$is_head_judge = $is_head;

// Get judge's assigned events with access control
$access_filter = getEventAccessSQL('e.id');

$events_query = $pdo->prepare("
    SELECT DISTINCT 
        e.id,
        e.name,
        e.date,
        e.location,
        e.heats_total,
        sf.name as format_name,
        sf.sport as discipline,
        COUNT(DISTINCT ep.id) as total_participants,
        COUNT(DISTINCT s.id) as scores_submitted
    FROM events e
    LEFT JOIN scoring_formats sf ON e.scoring_format = sf.format_id
    LEFT JOIN event_participants ep ON e.id = ep.event_id
    LEFT JOIN runs r ON ep.id = r.event_participant_id
    LEFT JOIN scores s ON r.id = s.run_id AND s.judge_id = ?
    WHERE e.id IN (
        SELECT DISTINCT event_id 
        FROM judge_assignments 
        WHERE judge_id = ?
    )" . $access_filter['sql'] . "
    GROUP BY e.id
    ORDER BY e.date DESC
");
$params = array_merge([$judge_id, $judge_id], $access_filter['params']);
$events_query->execute($params);
$assigned_events = $events_query->fetchAll(PDO::FETCH_ASSOC);

// Get recent scores overview
$recent_scores_query = $pdo->prepare("
    SELECT 
        s.id,
        s.score_value as total_score,
        s.created_at,
        r.run_number,
        ep.event_id,
        ep.heat_number,
        ep.bib_number,
        e.name as event_name,
        p.first_name,
        p.last_name
    FROM scores s
    JOIN runs r ON s.run_id = r.id
    JOIN event_participants ep ON r.event_participant_id = ep.id
    JOIN events e ON ep.event_id = e.id
    JOIN participants p ON ep.participant_id = p.id
    WHERE s.judge_id = ?
    ORDER BY s.created_at DESC
    LIMIT 10
");
$recent_scores_query->execute([$judge_id]);
$recent_scores = $recent_scores_query->fetchAll(PDO::FETCH_ASSOC);

// Get active heat if any
$active_heat_query = $pdo->prepare("
    SELECT 
        ehs.event_id,
        ehs.heat_number,
        ehs.heat_name,
        ehs.active_run,
        e.name as event_name
    FROM event_heat_settings ehs
    JOIN events e ON ehs.event_id = e.id
    WHERE ehs.is_active = 1
    AND e.id IN (
        SELECT DISTINCT event_id 
        FROM judge_assignments 
        WHERE judge_id = ?
    )
    LIMIT 1
");
$active_heat_query->execute([$judge_id]);
$active_heat = $active_heat_query->fetch(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html>
<head>
        <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Judge Dashboard - StyleScore</title>
    <?php include_once 'stylesheets_judge.php'; ?>
</head>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>

    <div class="container-fluid p-4">
        <div class="row justify-content-center">
            <div class="col-12 col-xl-10">
                
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-start flex-column flex-md-row mb-4 panel-header">
                    <h2 class="mb-2 mb-md-0">
                        <i class="fas fa-gavel me-2 text-primary"></i>
                        Judge Dashboard
                    </h2>
                    <div class="d-flex flex-row align-items-start align-items-md-end">
                        <div>
                            <small class="text-muted nowrap">Logged in as</small>
                            <strong><?= htmlspecialchars($judge_name) ?></strong>
                        </div>
                        
                    </div>
                </div>

                <!-- Messages -->
                <?php if (isset($_GET['message'])): ?>
                    <div class="alert alert-dismissible fade show <?= $_GET['message'] === 'no_events_assigned' ? 'alert-warning' : 'alert-info' ?> mb-4">
                        <i class="fas <?= $_GET['message'] === 'no_events_assigned' ? 'fa-exclamation-triangle' : 'fa-info-circle' ?> me-2"></i>
                        <?php
                        switch($_GET['message']) {
                            case 'no_events_assigned':
                                echo '<strong>No Events Assigned</strong><br>You currently have no events assigned to judge. Please contact the administrator to get assigned to an event.';
                                break;
                            default:
                                echo htmlspecialchars($_GET['message']);
                        }
                        ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <!-- Active Heat Alert -->
                <?php if ($active_heat): ?>
                    <div class="alert alert-success mb-4 active-heat-card">
                        <div class="d-flex justify-content-between align-items-center flex-column flex-md-row gap-3">
                            <div class="text-center text-md-start w-100">
                                <h5 class="mb-2">
                                    <i class="fas fa-circle fa-fade me-2"></i>
                                    Active Heat Now!
                                </h5>
                            
                                    <h2 class="fw-bold"><?= htmlspecialchars($active_heat['event_name']) ?></h2>
                                    <span class="badge fs-6 bg-info">
                                    <?php if (!empty($active_heat['heat_name'])): ?>
                                        <span class="fs-5"><?= htmlspecialchars($active_heat['heat_name']) ?> - </span>
                                    <?php endif; ?>  
                                    <span class="fs-6">Heat <?= htmlspecialchars($active_heat['heat_number']) ?> </span>  
                                
                                    <span class="badge fs-6 bg-info-subtle">Run <?= $active_heat['active_run'] ?></span>
                                </span>
                            </div>
                            
                            <div class="w-100 w-md-auto mt-2 mt-md-0">
                                                        <a href="score.php?event_id=<?= $active_heat['event_id'] ?>" 
                            class="btn btn-primary  w-100 w-md-auto fs-4">
                                <i class="fas fa-play me-2"></i> Start Scoring
                                                        </a>
                                                    </div>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Statistics Cards -->
                    <div class="row g-3 mb-4">
                        <div class="col-6 col-md-3">
                            <div class="card  h-100">
                                <div class="card-body text-center p-2 d-flex flex-row justify-content-center align-items-center gap-2 bg-info-subtle rounded">
                                    <i class="fas fa-calendar-check fa-1x text-primary mb-2"></i>
                                    <h4 class="mb-1"><?= count($assigned_events) ?></h4>
                                    <small class="text-muted d-block">Events</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-6 col-md-3">
                            <div class="card  h-100">
                                <div class="card-body text-center p-2 d-flex flex-row justify-content-center align-items-center gap-2 bg-info-subtle rounded">
                                    <i class="fas fa-clipboard-check fa-1x text-success mb-2"></i>
                                    <h4 class="mb-1">
                                        <?php 
                                        $total_scores = array_sum(array_column($assigned_events, 'scores_submitted'));
                                        echo $total_scores;
                                        ?>
                                    </h3>
                                    <small class="text-muted d-block">Scores</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-6 col-md-3">
                            <div class="card  h-100">
                                <div class="card-body text-center p-2 d-flex flex-row justify-content-center align-items-center gap-2 bg-info-subtle rounded">
                                    <i class="fas fa-users fa-1x text-info mb-2"></i>
                                    <h4 class="mb-1">
                                        <?php 
                                        $total_participants = array_sum(array_column($assigned_events, 'total_participants'));
                                        echo $total_participants;
                                        ?>
                                    </h3>
                                    <small class="text-muted d-block">Participants</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-6 col-md-3">
                            <div class="card  h-100">
                                <div class="card-body text-center p-2 d-flex flex-row justify-content-center align-items-center gap-2 bg-info-subtle rounded">
                                    <i class="fas fa-fire fa-1x text-warning mb-2"></i>
                                    <h4 class="mb-1">
                                        <?php 
                                        $total_heats = array_sum(array_column($assigned_events, 'heats_total'));
                                        echo $total_heats;
                                        ?>
                                    </h3>
                                    <small class="text-muted d-block">Heats</small>
                                </div>
                            </div>
                        </div>
                    </div>

                <div class="row">
                    <div class="col-12">
                    <!-- Assigned Events -->
                    <div class="mb-3">
                        <div class="card shadow-sm">
                            <div class="card-header bg-primary">
                                <h5 class="mb-0">
                                    <i class="fas fa-calendar-alt me-2"></i>
                                    My Assigned Events
                                </h5>
                            </div>
                            <div class="card-body p-2 p-md-3">
                                <?php if (empty($assigned_events)): ?>
                                    <div class="text-center text-muted py-5">
                                        <i class="fas fa-info-circle fa-3x mb-3"></i>
                                        <p>No events assigned yet.</p>
                                        <small>Contact the event administrator to be assigned to events.</small>
                                    </div>
                                <?php else: ?>
                                    <div class="list-group list-group-flush">
                                        <?php foreach ($assigned_events as $event): ?>
                                            <div class="list-group-item px-2 px-md-3">
                                                <div class="d-flex justify-content-between align-items-start flex-column flex-md-row gap-2">
                                                    <div class="flex-grow-1 w-100">
                                                        <h4 class="mb-2">
                                                            <i class="fas fa-trophy me-2 text-warning"></i>
                                                            <?= htmlspecialchars($event['name']) ?>
                                                        </h4>
                                                        <div class="small text-muted mb-2">
                                                            <div class="mb-1">
                                                                <i class="fas fa-calendar me-1"></i>
                                                                <?= date('F j, Y', strtotime($event['date'])) ?>
                                                            </div>
                                                            <?php if ($event['location']): ?>
                                                                <div class="mb-1">
                                                                    <i class="fas fa-map-marker-alt me-1"></i>
                                                                    <?= htmlspecialchars($event['location']) ?>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                        <div class="d-flex gap-2 flex-wrap">
                                                            <span class="badge bg-light text-dark fs-6">
                                                                <i class="fas fa-fire me-1"></i>
                                                                <?= $event['heats_total'] ?> Heats
                                                            </span>
                                                            <span class="badge bg-light text-dark fs-6">
                                                                <i class="fas fa-users me-1"></i>
                                                                <?= $event['total_participants'] ?> Participants
                                                            </span>
                                                            <?php if ($event['format_name']): ?>
                                                                <span class="badge bg-info fs-6">
                                                                    <?= htmlspecialchars($event['format_name']) ?>
                                                                </span>
                                                            <?php endif; ?>
                                                            <?php if ($event['scores_submitted'] > 0): ?>
                                                                <span class="badge bg-success fs-6">
                                                                    <i class="fas fa-check me-1"></i>
                                                                    <?= $event['scores_submitted'] ?> Scores
                                                                </span>
                                                            <?php endif; ?>
                                                        </div>
                                                    </div>
                                                    
                                                    <a href="score.php?event_id=<?= $event['id'] ?>" 
                            class="btn btn-primary  w-100 w-md-auto fs-4">
                                <i class="fas fa-play me-2"></i>Start Scoring
                            </a>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Recent Scores Overview -->
                    <div class="mb-3">
                        <div class="card  shadow-sm">
                            <div class="card-header bg-success">
                                <h5 class="mb-0">
                                    <i class="fas fa-history me-2"></i>
                                    My Recent Scores
                                </h5>
                            </div>
                            <div class="card-body p-0 ">
                                <?php if (empty($recent_scores)): ?>
                                    <div class="text-center text-muted py-4">
                                        <i class="fas fa-clipboard fa-2x mb-3"></i>
                                        <p>No scores submitted yet.</p>
                                        <small>Your recent scoring activity will appear here.</small>
                                    </div>
                                <?php else: ?>
                                    <div style="max-height: 500px; overflow-y: auto;">
                                        <table class="table table-sm scores-table mb-0" id="recentScoresTable">
                                            <thead class="small">
                                                <tr>
                                                    <th class="small bg-dark-subtle">BIB</th>
                                                    <th class="small bg-dark-subtle">Name</th>
                                                    <th class="small bg-dark-subtle">H/R</th>
                                                    <th class="small bg-dark-subtle">Score</th>
                                                    <th class="small bg-dark-subtle"></th>
                                                    <th class="small bg-dark-subtle"></th>
                                                    <th class="small bg-dark-subtle"></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($recent_scores as $score): ?>
                                                    <tr>
                                                        <td>
                                                            <span class="badge bg-warning-subtle"><?= htmlspecialchars($score['bib_number']) ?></span>
                                                        </td>
                                                        <td>
                                                            <small><?= htmlspecialchars($score['first_name'] . ' ' . $score['last_name']) ?></small>
                                                        </td>
                                                        <td>
                                                            <small><?= htmlspecialchars($score['heat_number'] . '/' . $score['run_number']) ?></small>
                                                        </td>
                                                        <td>
                                                            <strong class="text-success-emphasis"><?= number_format($score['total_score'], 2) ?></strong>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">&mdash;</small>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-success quick-badge">OK</span>
                                                        </td>
                                                        <td>
                                                            <!-- Edit button intentionally omitted per design -->
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Quick Actions -->
                    <div class="card shadow-sm">
                        <div class="card-header bg-light d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">
                                <i class="fas fa-bolt me-2 text-warning"></i>
                                Quick Actions
                            </h5>
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="openThemeSelector()">
                                <i class="fas fa-palette me-1"></i> Theme
                            </button>
                        </div>
                        <div class="card-body p-2 p-md-3 bg-dark-subtle">
                            <?php if (empty($assigned_events)): ?>
                                <div class="text-center text-muted py-4">
                                    <i class="fas fa-info-circle fa-3x mb-3"></i>
                                    <p class="mb-2"><strong>No Events Assigned</strong></p>
                                    <p class="small">Contact the event administrator to be assigned to events before you can start scoring.</p>
                                    <hr class="my-3">
                                    <a href="<?php echo resolveUrl('/v2/logout.php'); ?>" class="btn btn-outline-danger">
                                        <i class="fas fa-sign-out-alt me-2"></i>Logout
                                    </a>
                                </div>
                            <?php else: ?>
                            <div class="row g-2 g-md-3">
                                <div class="col-<?= $is_head_judge ? '4' : '6' ?> col-md-<?= $is_head_judge ? '4' : '6' ?>">
                                    <a href="score.php" class="btn btn-outline-primary w-100 py-2 py-md-3">
                                        <i class="fas fa-calculator fa-md-2x d-block mb-2"></i>
                                        <strong class="d-block">Score Panel</strong>
                                        <small class="d-none d-md-block">Enter scores</small>
                                    </a>
                                </div>
                                <?php if ($is_head_judge): ?>
                                <div class="col-4 col-md-4">
                                    <a href="head_judge.php" class="btn btn-outline-warning w-100 py-2 py-md-3">
                                        <i class="fas fa-crown fa-md-2x d-block mb-2"></i>
                                        <strong class="d-block">Head judge Panel</strong>
                                        <small class="d-none d-md-block">Enter scores</small>
                                    </a>
                                </div>
                                <?php endif; ?>
                                <div class="col-<?= $is_head_judge ? '4' : '6' ?> col-md-<?= $is_head_judge ? '4' : '6' ?>">
                                    <a href="../public_results.php" class="btn btn-outline-info w-100 py-2 py-md-3">
                                        <i class="fas fa-trophy fa-md-2x d-block mb-2"></i>
                                        <strong class="d-block">Results</strong>
                                        <small class="d-none d-md-block">View results</small>
                                    </a>
                                </div>                           
                                
                            </div>
                            <div class="row g-2 g-md-3">
                                <div class="col-6 col-md-6">
                                <a href="#" class="btn btn-outline-info w-100 py-2 py-md-3" onclick="openJudgeEventDashboard()">
                                
                                        <i class="fas fa-tv  fa-md-2x d-block mb-2"></i>
                                        <strong class="d-block">Dashboard</strong>
                                        <small class="d-none d-md-block">View dashboard</small>
                                </a>
                                </div>
                                <div class="col-6 col-md-6">
                                    <a href="<?php echo resolveUrl('/v2/logout.php'); ?>" class="btn btn-outline-danger w-100 py-2 py-md-3">
                                        <i class="fas fa-sign-out-alt fa-md-2x d-block mb-2"></i>
                                        <strong class="d-block">Logout</strong>
                                        <small class="d-none d-md-block">Sign out</small>
                                    </a>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
        </div>
    </div>
    <!-- Change panel theme -->
    <div class="modal fade" id="themeSelectorModal" tabindex="-1" aria-labelledby="themeSelectorModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="themeSelectorModalLabel">
                        <i class="fas fa-palette me-2"></i>Change Panel Theme
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body bg-dark-subtle">
                    <p class="text-muted mb-3">Select a visual theme for the scoring panel:</p>
                    
                    <div class="row g-3">
                        <!-- Brite Theme -->
                        <div class="col-6">
                            <div class="card theme-option-card" onclick="selectTheme('brite')" style="cursor: pointer;">
                                <div class="card-body text-center p-4" style="background: linear-gradient(135deg, #66ea71ff 0%, #15ae27ff 100%); min-height: 120px;">
                                    <i class="fas fa-sun text-white" style="font-size: 2rem;"></i>
                                    <h6 class="text-white mt-2 mb-0">Brite</h6>
                                    
                                </div>
                                <div class="card-footer text-center">
                                    <div class="form-check form-check-inline">
                                        <input class="form-check-input" type="radio" name="theme_selector" id="theme_brite" value="brite">
                                        <label class="form-check-label" for="theme_brite">Select</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Zephyr Theme -->
                        <div class="col-6">
                            <div class="card theme-option-card" onclick="selectTheme('zephyr')" style="cursor: pointer;">
                                <div class="card-body text-center p-4" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); min-height: 120px;">
                                    <i class="fas fa-wind text-white" style="font-size: 2rem;"></i>
                                    <h6 class="text-white mt-2 mb-0">Zephyr</h6>                                   
                                </div>
                                <div class="card-footer text-center">
                                    <div class="form-check form-check-inline">
                                        <input class="form-check-input" type="radio" name="theme_selector" id="theme_zephyr" value="zephyr">
                                        <label class="form-check-label" for="theme_zephyr">Select</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Zoom Control -->
                    <div class="mt-4">
                        <h6 class="mb-3">
                            <i class="fas fa-search-plus me-2"></i>Interface Zoom
                        </h6>
                        <div class="row align-items-center">
                            <div class="col-12">
                                <label for="zoomRange" class="form-label">
                                    Zoom Level: <span id="zoomValue" class="badge bg-primary">100%</span>
                                </label>
                                <input type="range" class="form-range" id="zoomRange" min="70" max="130" value="100" step="5" oninput="updateZoomDisplay(this.value)">
                                <div class="d-flex justify-content-between">
                                    <small class="text-muted">70%</small>
                                    <small class="text-muted">100%</small>
                                    <small class="text-muted">130%</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-info mt-3 mb-0">
                        <i class="fas fa-info-circle me-1"></i>
                        Your theme and zoom preferences will be saved in your browser.
                    </div>
                </div>
                <div class="modal-footer d-flex justify-content-between">
                    <button type="button" class="btn btn-outline-secondary" onclick="resetZoom()">
                        <i class="fas fa-undo me-1"></i>Reset Zoom
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="applyThemeAndZoom()">
                        <i class="fas fa-check me-1"></i>Apply
                    </button>
                </div>
            </div>
        </div>
    </div>
    <!-- No Event Selected Modal -->
    <div class="modal fade" id="noEventSelectedModal" tabindex="-1" aria-labelledby="noEventSelectedModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header bg-warning text-dark">
                    <h5 class="modal-title" id="noEventSelectedModalLabel">
                        <i class="fas fa-exclamation-triangle me-2"></i>No Event Selected
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info mb-3">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Action Required</strong>
                    </div>
                    
                    <p class="mb-3">You need to select an event before accessing the dashboard.</p>
                    
                    <p class="mb-0"><strong>How to select an event:</strong></p>
                    <ol class="mb-0">
                        <li>Go to the admin panel</li>
                        <li>Select an event from the events list</li>
                        <li>Return to the judge dashboard</li>
                    </ol>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Close
                    </button>
                    <a href="../admin/events.php" class="btn btn-primary">
                        <i class="fas fa-calendar-alt me-1"></i>Go to Events
                    </a>
                </div>
            </div>
        </div>
    </div>

        <!-- Toast Container -->
        <div class="toast-container position-fixed top-0 end-0 p-3" style="z-index: 9999;">
            <div id="liveToast" class="toast" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="toast-header">
                    <i class="fas fa-info-circle me-2 toast-icon"></i>
                    <strong class="me-auto toast-title">Notification</strong>
                    <button type="button" class="btn-close" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
                <div class="toast-body">
                    <!-- Message will be inserted here -->
                </div>
            </div>
        </div>
    </div></div>
<?php include '../admin/footer.php'; ?>

<script>
// Toast notification function
function showToast(message, type = 'info') {
    const toastEl = document.getElementById('liveToast');
    const toastBody = toastEl.querySelector('.toast-body');
    const toastIcon = toastEl.querySelector('.toast-icon');
    const toastTitle = toastEl.querySelector('.toast-title');
    const toastHeader = toastEl.querySelector('.toast-header');
    
    // Set message
    toastBody.textContent = message;
    
    // Set icon and color based on type
    toastHeader.className = 'toast-header';
    if (type === 'success') {
        toastIcon.className = 'fas fa-check-circle me-2 toast-icon text-success';
        toastTitle.textContent = 'Success';
        toastHeader.classList.add('bg-success', 'text-white');
    } else if (type === 'warning') {
        toastIcon.className = 'fas fa-exclamation-triangle me-2 toast-icon text-warning';
        toastTitle.textContent = 'Warning';
        toastHeader.classList.add('bg-warning', 'text-dark');
    } else if (type === 'danger') {
        toastIcon.className = 'fas fa-times-circle me-2 toast-icon text-danger';
        toastTitle.textContent = 'Error';
        toastHeader.classList.add('bg-danger', 'text-white');
    } else {
        toastIcon.className = 'fas fa-info-circle me-2 toast-icon text-info';
        toastTitle.textContent = 'Info';
        toastHeader.classList.add('bg-info', 'text-white');
    }
    
    const toast = new bootstrap.Toast(toastEl, {
        autohide: true,
        delay: 3000
    });
    toast.show();
}

// Get event id from localStorage or cookies (same helper pattern as admin side)
function getStoredEventId() {
    const fromLocal = localStorage.getItem('selectedEventId');
    if (fromLocal) return fromLocal;

    const fromCookieSelected = getCookie('selectedEventId');
    if (fromCookieSelected) return fromCookieSelected;

    const fromCookieEvent = getCookie('event_id');
    if (fromCookieEvent) return fromCookieEvent;

    return null;
}

function getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
    return null;
}

function openJudgeEventDashboard() {
    const eventId = getStoredEventId();
    if (!eventId) {
        const noEventModal = new bootstrap.Modal(document.getElementById('noEventSelectedModal'));
        noEventModal.show();
        return;
    }
    window.location.href = `../admin/event_dashboard.php?event_id=${encodeURIComponent(eventId)}`;
}

// Theme selector functions (shared with score.php)
function selectTheme(themeName) {
    const radio = document.getElementById('theme_' + themeName);
    if (radio) {
        radio.checked = true;
    }
}

function updateZoomDisplay(value) {
    const zoomValueDisplay = document.getElementById('zoomValue');
    if (zoomValueDisplay) {
        zoomValueDisplay.textContent = value + '%';
    }
}

function applyZoom(zoomPercent) {
    // Apply zoom to body element
    document.body.style.zoom = zoomPercent + '%';
    
    // Save to localStorage
    localStorage.setItem('panel_zoom', zoomPercent);
}

function resetZoom() {
    const zoomRange = document.getElementById('zoomRange');
    if (zoomRange) {
        zoomRange.value = 100;
        updateZoomDisplay(100);
    }
    applyZoom(100);
    
    // Show toast notification
    showToast('Zoom reset to 100%', 'success');
}

function applyThemeAndZoom() {
    const selectedTheme = document.querySelector('input[name="theme_selector"]:checked');
    if (!selectedTheme) {
        showToast('Please select a theme', 'warning');
        return;
    }

    const theme = selectedTheme.value;
    const zoomValue = document.getElementById('zoomRange').value;
    
    // Set cookie for theme (365 days)
    const expiryDate = new Date();
    expiryDate.setTime(expiryDate.getTime() + (365 * 24 * 60 * 60 * 1000));
    document.cookie = 'panel_theme=' + theme + '; expires=' + expiryDate.toUTCString() + '; path=/';
    
    // Apply and save zoom
    applyZoom(zoomValue);
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('themeSelectorModal'));
    if (modal) {
        modal.hide();
    }
    
    // Reload only if theme changed
    const cookies = document.cookie.split(';');
    let currentTheme = 'brite';
    for (let cookie of cookies) {
        const [name, value] = cookie.trim().split('=');
        if (name === 'panel_theme') {
            currentTheme = value;
            break;
        }
    }
    
    if (currentTheme !== theme) {
        setTimeout(() => {
            window.location.reload();
        }, 300);
    }
}

function applySelectedTheme() {
    // Kept for backward compatibility - redirects to new function
    applyThemeAndZoom();
}

function openThemeSelector() {
    const cookies = document.cookie.split(';');
    let currentTheme = 'brite';
    for (let cookie of cookies) {
        const [name, value] = cookie.trim().split('=');
        if (name === 'panel_theme') {
            currentTheme = value;
            break;
        }
    }

    const radio = document.getElementById('theme_' + currentTheme);
    if (radio) {
        radio.checked = true;
    }
    
    // Load and set current zoom
    const savedZoom = localStorage.getItem('panel_zoom') || '100';
    const zoomRange = document.getElementById('zoomRange');
    if (zoomRange) {
        zoomRange.value = savedZoom;
        updateZoomDisplay(savedZoom);
    }

    const modal = new bootstrap.Modal(document.getElementById('themeSelectorModal'));
    modal.show();
}

// Apply saved zoom on page load
(function() {
    const savedZoom = localStorage.getItem('panel_zoom');
    if (savedZoom) {
        document.body.style.zoom = savedZoom + '%';
    }
})();
</script>

</body>
</html>
