# SSH2 Extension Installer for PHP 8.2 (ZTS x64)
# StyleScore - SFTP Support Installation Script

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "SSH2 Extension Installer for PHP 8.2" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Configuration
$phpVersion = "8.2"
$phpDir = "C:\xampp\php"
$extDir = "$phpDir\ext"
$phpIni = "$phpDir\php.ini"

# SSH2 DLL URLs (latest compatible versions)
$ssh2Url = "https://windows.php.net/downloads/pecl/releases/ssh2/1.4.1/php_ssh2-1.4.1-8.2-ts-vs16-x64.zip"
$libssh2Url = "https://windows.php.net/downloads/pecl/deps/libssh2-1.11.0-vs16-x64.zip"

Write-Host "Detected Configuration:" -ForegroundColor Yellow
Write-Host "  PHP Version: 8.2.4" -ForegroundColor White
Write-Host "  Thread Safety: ZTS (Thread Safe)" -ForegroundColor White
Write-Host "  Architecture: x64" -ForegroundColor White
Write-Host "  PHP Directory: $phpDir" -ForegroundColor White
Write-Host "  Extensions Directory: $extDir" -ForegroundColor White
Write-Host ""

# Check if running as Administrator
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
if (-not $isAdmin) {
    Write-Host "WARNING: Not running as Administrator" -ForegroundColor Yellow
    Write-Host "Some operations may fail. Consider running PowerShell as Administrator." -ForegroundColor Yellow
    Write-Host ""
}

# Create temp directory
$tempDir = "$env:TEMP\ssh2_install"
if (Test-Path $tempDir) {
    Remove-Item -Path $tempDir -Recurse -Force
}
New-Item -ItemType Directory -Path $tempDir | Out-Null
Write-Host "Created temporary directory: $tempDir" -ForegroundColor Green

try {
    # Download SSH2 extension
    Write-Host ""
    Write-Host "Step 1: Downloading SSH2 extension..." -ForegroundColor Cyan
    $ssh2Zip = "$tempDir\ssh2.zip"
    
    try {
        Invoke-WebRequest -Uri $ssh2Url -OutFile $ssh2Zip -UseBasicParsing
        Write-Host "  Downloaded: php_ssh2-1.4.1-8.2-ts-vs16-x64.zip" -ForegroundColor Green
    }
    catch {
        Write-Host "  ERROR: Could not download SSH2 extension" -ForegroundColor Red
        Write-Host "  URL: $ssh2Url" -ForegroundColor Yellow
        Write-Host "  Please download manually and extract to: $extDir" -ForegroundColor Yellow
        throw
    }

    # Download libssh2 dependency
    Write-Host ""
    Write-Host "Step 2: Downloading libssh2 dependency..." -ForegroundColor Cyan
    $libssh2Zip = "$tempDir\libssh2.zip"
    
    try {
        Invoke-WebRequest -Uri $libssh2Url -OutFile $libssh2Zip -UseBasicParsing
        Write-Host "  Downloaded: libssh2-1.11.0-vs16-x64.zip" -ForegroundColor Green
    }
    catch {
        Write-Host "  ERROR: Could not download libssh2" -ForegroundColor Red
        Write-Host "  URL: $libssh2Url" -ForegroundColor Yellow
        Write-Host "  Please download manually" -ForegroundColor Yellow
        throw
    }

    # Extract SSH2
    Write-Host ""
    Write-Host "Step 3: Extracting SSH2 extension..." -ForegroundColor Cyan
    Expand-Archive -Path $ssh2Zip -DestinationPath "$tempDir\ssh2" -Force
    
    # Find and copy php_ssh2.dll
    $ssh2Dll = Get-ChildItem -Path "$tempDir\ssh2" -Filter "php_ssh2.dll" -Recurse | Select-Object -First 1
    if ($ssh2Dll) {
        Copy-Item -Path $ssh2Dll.FullName -Destination $extDir -Force
        Write-Host "  Copied php_ssh2.dll to: $extDir" -ForegroundColor Green
    }
    else {
        Write-Host "  ERROR: php_ssh2.dll not found in archive" -ForegroundColor Red
        throw
    }

    # Extract libssh2
    Write-Host ""
    Write-Host "Step 4: Extracting libssh2 dependency..." -ForegroundColor Cyan
    Expand-Archive -Path $libssh2Zip -DestinationPath "$tempDir\libssh2" -Force
    
    # Find and copy libssh2.dll to multiple locations
    $libssh2Dll = Get-ChildItem -Path "$tempDir\libssh2" -Filter "libssh2.dll" -Recurse | Select-Object -First 1
    if ($libssh2Dll) {
        # Copy to PHP directory
        Copy-Item -Path $libssh2Dll.FullName -Destination $phpDir -Force
        Write-Host "  Copied libssh2.dll to: $phpDir" -ForegroundColor Green
        
        # Copy to Apache bin directory
        $apacheBin = "C:\xampp\apache\bin"
        if (Test-Path $apacheBin) {
            Copy-Item -Path $libssh2Dll.FullName -Destination $apacheBin -Force
            Write-Host "  Copied libssh2.dll to: $apacheBin" -ForegroundColor Green
        }
    }
    else {
        Write-Host "  ERROR: libssh2.dll not found in archive" -ForegroundColor Red
        throw
    }

    # Update php.ini
    Write-Host ""
    Write-Host "Step 5: Updating php.ini..." -ForegroundColor Cyan
    
    if (Test-Path $phpIni) {
        $iniContent = Get-Content $phpIni -Raw
        
        # Check if extension is already enabled
        if ($iniContent -match "^\s*extension\s*=\s*ssh2\s*$" -or $iniContent -match "^\s*extension\s*=\s*php_ssh2\.dll\s*$") {
            Write-Host "  SSH2 extension already enabled in php.ini" -ForegroundColor Yellow
        }
        else {
            # Find the extensions section
            if ($iniContent -match ";extension=openssl") {
                $iniContent = $iniContent -replace "(;extension=openssl)", "`$1`nextension=ssh2"
                Set-Content -Path $phpIni -Value $iniContent -NoNewline
                Write-Host "  Added 'extension=ssh2' to php.ini" -ForegroundColor Green
            }
            else {
                # Add at the end of file
                Add-Content -Path $phpIni -Value "`nextension=ssh2"
                Write-Host "  Added 'extension=ssh2' to php.ini" -ForegroundColor Green
            }
        }
    }
    else {
        Write-Host "  ERROR: php.ini not found at: $phpIni" -ForegroundColor Red
        throw
    }

    # Clean up temp directory
    Write-Host ""
    Write-Host "Step 6: Cleaning up..." -ForegroundColor Cyan
    Remove-Item -Path $tempDir -Recurse -Force
    Write-Host "  Temporary files removed" -ForegroundColor Green

    # Success message
    Write-Host ""
    Write-Host "========================================" -ForegroundColor Green
    Write-Host "SSH2 Extension Installed Successfully!" -ForegroundColor Green
    Write-Host "========================================" -ForegroundColor Green
    Write-Host ""
    Write-Host "Files installed:" -ForegroundColor Yellow
    Write-Host "  - $extDir\php_ssh2.dll" -ForegroundColor White
    Write-Host "  - $phpDir\libssh2.dll" -ForegroundColor White
    Write-Host "  - C:\xampp\apache\bin\libssh2.dll" -ForegroundColor White
    Write-Host ""
    Write-Host "Next steps:" -ForegroundColor Yellow
    Write-Host "  1. Restart Apache in XAMPP Control Panel" -ForegroundColor White
    Write-Host "  2. Visit: http://localhost/v2/admin/check_ssh2.php" -ForegroundColor White
    Write-Host "  3. You should see a green checkmark!" -ForegroundColor White
    Write-Host ""

}
catch {
    Write-Host ""
    Write-Host "========================================" -ForegroundColor Red
    Write-Host "Installation Failed" -ForegroundColor Red
    Write-Host "========================================" -ForegroundColor Red
    Write-Host "Error: $_" -ForegroundColor Red
    Write-Host ""
    Write-Host "Manual Installation:" -ForegroundColor Yellow
    Write-Host "  1. Download: https://windows.php.net/downloads/pecl/releases/ssh2/1.4.1/php_ssh2-1.4.1-8.2-ts-vs16-x64.zip"
    Write-Host "  2. Extract php_ssh2.dll to: $extDir"
    Write-Host "  3. Download: https://windows.php.net/downloads/pecl/deps/libssh2-1.11.0-vs16-x64.zip"
    Write-Host "  4. Extract libssh2.dll to: $phpDir and C:\xampp\apache\bin\"
    Write-Host "  5. Add 'extension=ssh2' to: $phpIni"
    Write-Host "  6. Restart Apache"
    Write-Host ""
}

Write-Host "Press any key to exit..."
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
