<?php
/**
 * Migration: Create Permissions System Tables
 * Run this once to set up the permissions system
 */

require_once 'includes/db.php';

echo "Creating permissions system tables...\n\n";

try {
    // Read and execute SQL file
    $sql = file_get_contents(__DIR__ . '/create_permissions_tables.sql');
    
    // Split into individual statements
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && strpos($stmt, '--') !== 0;
        }
    );

    $success_count = 0;
    $error_count = 0;

    foreach ($statements as $statement) {
        if (empty(trim($statement))) continue;
        
        try {
            $pdo->exec($statement);
            $success_count++;
            
            // Provide feedback for major operations
            if (stripos($statement, 'CREATE TABLE') !== false) {
                preg_match('/CREATE TABLE.*?`?(\w+)`?/i', $statement, $matches);
                echo "✓ Created table: " . ($matches[1] ?? 'unknown') . "\n";
            } elseif (stripos($statement, 'INSERT INTO permissions') !== false) {
                echo "✓ Inserted default permissions\n";
            } elseif (stripos($statement, 'INSERT INTO role_permissions') !== false) {
                preg_match('/\'(\w+)\'/', $statement, $matches);
                if (isset($matches[1])) {
                    echo "✓ Assigned permissions to role: " . $matches[1] . "\n";
                }
            }
        } catch (PDOException $e) {
            $error_count++;
            echo "✗ Error: " . $e->getMessage() . "\n";
        }
    }

    echo "\n=== Migration Summary ===\n";
    echo "Successful operations: $success_count\n";
    echo "Failed operations: $error_count\n";

    if ($error_count === 0) {
        echo "\n✓ Permissions system installed successfully!\n\n";
        echo "Next steps:\n";
        echo "1. Access the permissions panel: admin/permissions_manager.php\n";
        echo "2. Review and customize permissions as needed\n";
        echo "3. Update your code to use hasPermission() checks\n\n";
        echo "Example usage in code:\n";
        echo "  require_once 'includes/permissions.php';\n";
        echo "  if (hasPermission('create_events')) {\n";
        echo "      // Allow event creation\n";
        echo "  }\n";
    } else {
        echo "\n⚠ Some errors occurred. Please review the output above.\n";
    }

} catch (Exception $e) {
    echo "Fatal error: " . $e->getMessage() . "\n";
    exit(1);
}
