<?php
if(!isset($_SESSION)) { session_start(); }

// db.php includes the getBaseUrl() helper function  
require_once __DIR__ . '/db.php';

$baseUrl = getBaseUrl();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    
    // Validate input
    if (empty($username) || empty($password)) {
        header("Location: {$baseUrl}/v2/login.php?error=missing_fields");
        exit;
    }
    
    // Database connection
    include '../includes/db.php';
    
    try {
        // Query to find user with matching username (role auto-detected)
        $stmt = $pdo->prepare("SELECT id, username, password, role, judge_id FROM users WHERE username = ?");
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Check password - support both hashed (new) and plain text (legacy)
        $password_valid = false;
        if ($user) {
            // Check if password is hashed (bcrypt starts with $2y$ and is 60 chars)
            if (strlen($user['password']) === 60 && strpos($user['password'], '$2y$') === 0) {
                // Use password_verify for hashed passwords
                $password_valid = password_verify($password, $user['password']);
            } else {
                // Legacy plain text comparison
                $password_valid = ($password === $user['password']);
            }
        }
        
        if ($user && $password_valid) {
            // Successful login - set session variables
            $_SESSION['user_logged_in'] = true;
            $_SESSION['user_type'] = $user['role'];
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['user_name'] = $user['username'];
            
            // Store judge_id if user is a judge
            if ($user['role'] === 'judge' && $user['judge_id']) {
                $_SESSION['judge_id'] = $user['judge_id'];
            }
            
            // Backward compatibility session variables
            if (in_array($user['role'], ['admin', 'super_admin'])) {
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['judge_id'] = $user['id'];
            } elseif (in_array($user['role'], ['judge', 'head_judge'])) {
                $_SESSION['judge_logged_in'] = true;
            }
            
            // Store role in session for compatibility
            $_SESSION['role'] = $user['role'];
            
            // Load assigned events for this user
            try {
                $event_stmt = $pdo->prepare("
                    SELECT DISTINCT event_id 
                    FROM user_assigned_events 
                    WHERE user_id = ?
                ");
                $event_stmt->execute([$user['id']]);
                $assigned_events = $event_stmt->fetchAll(PDO::FETCH_COLUMN);
                $_SESSION['assigned_events'] = $assigned_events;
            } catch (Exception $e) {
                // If table doesn't exist or error, set empty array
                $_SESSION['assigned_events'] = [];
            }
            
            // Log successful login (optional)
            error_log("Successful login: User {$user['username']} ({$user['role']}) from IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
            
            // Redirect to appropriate dashboard based on role
            if (in_array($user['role'], ['admin', 'super_admin', 'office'])) {
                header("Location: {$baseUrl}/v2/admin/dashboard.php");
            } elseif (in_array($user['role'], ['judge', 'head_judge'])) {
                header("Location: {$baseUrl}/v2/judge/dashboard.php");
            } else {
                // Default redirect for other roles (media, participant, etc.)
                header("Location: {$baseUrl}/v2/dashboard.php");
            }
            exit;
        } else {
            // Invalid credentials - log the attempt
            error_log("Failed login: Username '{$username}' from IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
            header("Location: {$baseUrl}/v2/login.php?error=invalid_credentials");
            exit;
        }
        
    } catch(PDOException $e) {
        error_log("Database error during login: " . $e->getMessage());
        header("Location: {$baseUrl}/v2/login.php?error=system_error");
        exit;
    }
} else {
    // Not a POST request - redirect to login
    header("Location: {$baseUrl}/v2/login.php");
    exit;
}
?>