<?php
/**
 * Permissions Helper Functions
 * Provides fast permission checking with caching
 */

// Cache for permissions to avoid repeated database queries
$_permissions_cache = [];

/**
 * Check if current user has a specific permission
 * 
 * @param string $permission_name Permission name to check (e.g., 'create_events')
 * @param int|null $user_id Optional user ID (defaults to current session user)
 * @return bool True if user has permission, false otherwise
 */
function hasPermission($permission_name, $user_id = null) {
    global $pdo, $_permissions_cache;
    
    // Get user info
    $user_id = $user_id ?? ($_SESSION['user_id'] ?? null);
    $user_role = $_SESSION['role'] ?? null;
    
    if (!$user_id || !$user_role) {
        return false;
    }
    
    // super_admin always has all permissions
    if ($user_role === 'super_admin') {
        return true;
    }
    
    // Check cache first
    $cache_key = "{$user_role}_{$permission_name}";
    if (isset($_permissions_cache[$cache_key])) {
        return $_permissions_cache[$cache_key];
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as has_permission
            FROM role_permissions rp
            INNER JOIN permissions p ON rp.permission_id = p.id
            WHERE rp.role = ? AND p.name = ?
        ");
        $stmt->execute([$user_role, $permission_name]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $has_perm = ($result['has_permission'] > 0);
        $_permissions_cache[$cache_key] = $has_perm;
        
        return $has_perm;
    } catch (PDOException $e) {
        error_log("Permission check error: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if current user has ANY of the specified permissions
 * 
 * @param array $permission_names Array of permission names
 * @param int|null $user_id Optional user ID
 * @return bool True if user has at least one permission
 */
function hasAnyPermission($permission_names, $user_id = null) {
    foreach ($permission_names as $permission) {
        if (hasPermission($permission, $user_id)) {
            return true;
        }
    }
    return false;
}

/**
 * Check if current user has ALL of the specified permissions
 * 
 * @param array $permission_names Array of permission names
 * @param int|null $user_id Optional user ID
 * @return bool True if user has all permissions
 */
function hasAllPermissions($permission_names, $user_id = null) {
    foreach ($permission_names as $permission) {
        if (!hasPermission($permission, $user_id)) {
            return false;
        }
    }
    return true;
}

/**
 * Require a specific permission or exit with 403
 * 
 * @param string $permission_name Permission name required
 * @param string $message Optional custom error message
 */
function requirePermission($permission_name, $message = null) {
    if (!hasPermission($permission_name)) {
        $msg = $message ?? "You don't have permission to access this resource. Required: {$permission_name}";
        http_response_code(403);
        die(json_encode([
            'success' => false,
            'message' => $msg,
            'required_permission' => $permission_name
        ]));
    }
}

/**
 * Get all permissions for a specific role
 * 
 * @param string $role Role name
 * @return array Array of permission objects
 */
function getRolePermissions($role) {
    global $pdo;
    
    // super_admin gets all permissions
    if ($role === 'super_admin') {
        $stmt = $pdo->query("SELECT * FROM permissions ORDER BY category, display_name");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT p.*
            FROM permissions p
            INNER JOIN role_permissions rp ON p.id = rp.permission_id
            WHERE rp.role = ?
            ORDER BY p.category, p.display_name
        ");
        $stmt->execute([$role]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Get role permissions error: " . $e->getMessage());
        return [];
    }
}

/**
 * Get all permissions grouped by category
 * 
 * @return array Associative array with categories as keys
 */
function getAllPermissionsGrouped() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("SELECT * FROM permissions ORDER BY category, display_name");
        $permissions = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $grouped = [];
        foreach ($permissions as $perm) {
            $category = $perm['category'] ?? 'general';
            if (!isset($grouped[$category])) {
                $grouped[$category] = [];
            }
            $grouped[$category][] = $perm;
        }
        
        return $grouped;
    } catch (PDOException $e) {
        error_log("Get all permissions error: " . $e->getMessage());
        return [];
    }
}

/**
 * Clear permissions cache (call after updating permissions)
 */
function clearPermissionsCache() {
    global $_permissions_cache;
    $_permissions_cache = [];
}
