<?php
/**
 * Event Metadata Helper Functions
 * Handles JSON metadata storage and retrieval for events with flexible field ordering and grouping
 */

/**
 * Get event metadata with default structure
 * @param PDO $pdo
 * @param int $event_id
 * @return array
 */
function getEventMetadata($pdo, $event_id) {
    $stmt = $pdo->prepare("SELECT event_metadata FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $result = $stmt->fetchColumn();
    
    if ($result) {
        $metadata = json_decode($result, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            // Merge with defaults to ensure new fields appear
            return mergeWithDefaults($metadata);
        }
    }
    
    // Return default structure if no metadata exists
    return getDefaultMetadataStructure();
}

/**
 * Merge existing metadata with defaults to add any new system fields
 * @param array $existing
 * @return array
 */
function mergeWithDefaults($existing) {
    $defaults = getDefaultMetadataStructure();
    
    // If old format, migrate it
    if (isset($existing['jury']) || isset($existing['technical_data'])) {
        return migrateOldFormat($existing);
    }
    
    // Get existing field keys
    $existingKeys = [];
    if (isset($existing['metadata_fields'])) {
        foreach ($existing['metadata_fields'] as $field) {
            $existingKeys[] = $field['field_key'];
        }
    }
    
    // Add any missing system fields from defaults
    $merged = $existing;
    if (!isset($merged['metadata_fields'])) {
        $merged['metadata_fields'] = [];
    }
    
    foreach ($defaults['metadata_fields'] as $defaultField) {
        if ($defaultField['is_system'] && !in_array($defaultField['field_key'], $existingKeys)) {
            $merged['metadata_fields'][] = $defaultField;
        }
    }
    
    // Ensure judges array exists
    if (!isset($merged['judges'])) {
        $merged['judges'] = [];
    }
    
    return $merged;
}

/**
 * Migrate old format to new field-based format
 * @param array $oldMetadata
 * @return array
 */
function migrateOldFormat($oldMetadata) {
    $newFormat = ['metadata_fields' => [], 'judges' => []];
    
    // Migrate jury fields
    if (isset($oldMetadata['jury'])) {
        $order = 1;
        foreach ($oldMetadata['jury'] as $key => $value) {
            $newFormat['metadata_fields'][] = [
                'group' => 'Jury Information',
                'field_key' => $key,
                'field_label' => ucwords(str_replace('_', ' ', $key)),
                'field_value' => $value,
                'field_order' => $order++,
                'is_system' => true
            ];
        }
    }
    
    // Migrate technical data
    if (isset($oldMetadata['technical_data'])) {
        $order = 1;
        foreach ($oldMetadata['technical_data'] as $key => $value) {
            $newFormat['metadata_fields'][] = [
                'group' => 'Technical Data',
                'field_key' => $key,
                'field_label' => ucwords(str_replace('_', ' ', $key)),
                'field_value' => $value,
                'field_order' => $order++,
                'is_system' => true
            ];
        }
    }
    
    // Migrate venue details
    if (isset($oldMetadata['venue_details'])) {
        $order = 1;
        foreach ($oldMetadata['venue_details'] as $key => $value) {
            $newFormat['metadata_fields'][] = [
                'group' => 'Venue Details',
                'field_key' => $key,
                'field_label' => ucwords(str_replace('_', ' ', $key)),
                'field_value' => $value,
                'field_order' => $order++,
                'is_system' => true
            ];
        }
    }
    
    // Migrate custom fields
    if (isset($oldMetadata['custom_fields'])) {
        $order = 1;
        foreach ($oldMetadata['custom_fields'] as $key => $value) {
            $newFormat['metadata_fields'][] = [
                'group' => 'Custom Fields',
                'field_key' => $key,
                'field_label' => $key,
                'field_value' => $value,
                'field_order' => $order++,
                'is_system' => false
            ];
        }
    }
    
    // Migrate judges
    if (isset($oldMetadata['judges'])) {
        $newFormat['judges'] = $oldMetadata['judges'];
    }
    
    return $newFormat;
}

/**
 * Save event metadata
 * @param PDO $pdo
 * @param int $event_id
 * @param array $metadata
 * @return bool
 */
function saveEventMetadata($pdo, $event_id, $metadata) {
    $json = json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
    $stmt = $pdo->prepare("UPDATE events SET event_metadata = ? WHERE id = ?");
    return $stmt->execute([$json, $event_id]);
}

/**
 * Get default metadata structure with flexible field ordering
 * @return array
 */
function getDefaultMetadataStructure() {
    return [
        'metadata_fields' => [
            // Jury Information Group
            [
                'group' => 'Jury Information',
                'field_key' => 'technical_delegate_fis',
                'field_label' => 'Technical Delegate FIS',
                'field_value' => '',
                'field_order' => 1,
                'is_system' => true
            ],
            [
                'group' => 'Jury Information',
                'field_key' => 'chief_of_competition',
                'field_label' => 'Chief of Competition',
                'field_value' => '',
                'field_order' => 2,
                'is_system' => true
            ],
            [
                'group' => 'Jury Information',
                'field_key' => 'head_judge',
                'field_label' => 'Head Judge',
                'field_value' => '',
                'field_order' => 3,
                'is_system' => true
            ],
            [
                'group' => 'Jury Information',
                'field_key' => 'chief_of_course',
                'field_label' => 'Chief of Course',
                'field_value' => '',
                'field_order' => 4,
                'is_system' => true
            ],
            
            // Technical Data Group
            [
                'group' => 'Technical Data',
                'field_key' => 'run_length',
                'field_label' => 'Run Length',
                'field_value' => '',
                'field_order' => 1,
                'is_system' => true
            ],
            [
                'group' => 'Technical Data',
                'field_key' => 'pitch',
                'field_label' => 'Pitch',
                'field_value' => '',
                'field_order' => 2,
                'is_system' => true
            ],
            [
                'group' => 'Technical Data',
                'field_key' => 'take_off_to_knoll',
                'field_label' => 'Take-off to Knoll',
                'field_value' => '',
                'field_order' => 3,
                'is_system' => true
            ],
            [
                'group' => 'Technical Data',
                'field_key' => 'jump_height',
                'field_label' => 'Jump Height',
                'field_value' => '',
                'field_order' => 4,
                'is_system' => true
            ],
            [
                'group' => 'Technical Data',
                'field_key' => 'landing_pitch',
                'field_label' => 'Landing Pitch',
                'field_value' => '',
                'field_order' => 5,
                'is_system' => true
            ],
            [
                'group' => 'Technical Data',
                'field_key' => 'codex',
                'field_label' => 'Codex',
                'field_value' => '',
                'field_order' => 6,
                'is_system' => true
            ],
            
            // Venue Details Group
            [
                'group' => 'Venue Details',
                'field_key' => 'location',
                'field_label' => 'Location',
                'field_value' => '',
                'field_order' => 1,
                'is_system' => true
            ],
            [
                'group' => 'Venue Details',
                'field_key' => 'elevation',
                'field_label' => 'Elevation',
                'field_value' => '',
                'field_order' => 2,
                'is_system' => true
            ],
            [
                'group' => 'Venue Details',
                'field_key' => 'weather_condition',
                'field_label' => 'Weather Condition',
                'field_value' => '',
                'field_order' => 3,
                'is_system' => true
            ],
            [
                'group' => 'Venue Details',
                'field_key' => 'temperature',
                'field_label' => 'Temperature',
                'field_value' => '',
                'field_order' => 4,
                'is_system' => true
            ]
        ],
        'judges' => []  // Keep judges array separate as it has different structure
    ];
}

/**
 * Get fields organized by group with proper ordering
 * @param array $metadata
 * @return array [group_name => [fields...]]
 */
function getFieldsByGroup($metadata) {
    $groups = [];
    
    if (empty($metadata['metadata_fields'])) {
        return $groups;
    }
    
    foreach ($metadata['metadata_fields'] as $field) {
        $group = $field['group'] ?? 'Uncategorized';
        if (!isset($groups[$group])) {
            $groups[$group] = [];
        }
        $groups[$group][] = $field;
    }
    
    // Sort fields within each group by field_order
    foreach ($groups as $group => &$fields) {
        usort($fields, function($a, $b) {
            return ($a['field_order'] ?? 999) <=> ($b['field_order'] ?? 999);
        });
    }
    
    return $groups;
}

/**
 * Get specific metadata section (for backward compatibility)
 * @param PDO $pdo
 * @param int $event_id
 * @param string $section (group name)
 * @return array
 */
function getMetadataSection($pdo, $event_id, $section) {
    $metadata = getEventMetadata($pdo, $event_id);
    $groups = getFieldsByGroup($metadata);
    return $groups[$section] ?? [];
}

/**
 * Display metadata by group
 * @param array $metadata
 * @param string $groupName
 * @return string HTML
 */
function displayMetadataGroup($metadata, $groupName) {
    $groups = getFieldsByGroup($metadata);
    
    if (empty($groups[$groupName])) {
        return '<p class="text-muted">No ' . htmlspecialchars($groupName) . ' available</p>';
    }
    
    $html = '<dl class="row mb-0">';
    
    foreach ($groups[$groupName] as $field) {
        if (!empty($field['field_value'])) {
            $html .= '<dt class="col-sm-6">' . htmlspecialchars($field['field_label']) . ':</dt>';
            $html .= '<dd class="col-sm-6">' . htmlspecialchars($field['field_value']) . '</dd>';
        }
    }
    
    $html .= '</dl>';
    return $html;
}

/**
 * Display jury information (backward compatibility wrapper)
 * @param array $metadata
 * @return string HTML
 */
function displayJuryInfo($metadata) {
    return displayMetadataGroup($metadata, 'Jury Information');
}

/**
 * Display technical data (backward compatibility wrapper)
 * @param array $metadata
 * @return string HTML
 */
function displayTechnicalData($metadata) {
    return displayMetadataGroup($metadata, 'Technical Data');
}

/**
 * Display venue details (backward compatibility wrapper)
 * @param array $metadata
 * @return string HTML
 */
function displayVenueDetails($metadata) {
    return displayMetadataGroup($metadata, 'Venue Details');
}

/**
 * Display judges list
 * @param array $metadata
 * @return string HTML
 */
function displayJudgesList($metadata) {
    if (empty($metadata['judges']) || !is_array($metadata['judges'])) {
        return '<p class="text-muted">No judges assigned</p>';
    }
    
    $html = '<ol class="mb-0">';
    foreach ($metadata['judges'] as $judge) {
        $name = htmlspecialchars($judge['name'] ?? 'Unknown');
        $license = !empty($judge['license']) ? ' <span class="badge bg-secondary">' . htmlspecialchars($judge['license']) . '</span>' : '';
        $html .= '<li>' . $name . $license . '</li>';
    }
    $html .= '</ol>';
    return $html;
}

/**
 * Add a custom field to metadata
 * @param array $metadata
 * @param string $group
 * @param string $fieldKey
 * @param string $fieldLabel
 * @param string $fieldValue
 * @param int $fieldOrder
 * @return array Updated metadata
 */
function addCustomField($metadata, $group, $fieldKey, $fieldLabel, $fieldValue = '', $fieldOrder = null) {
    if (!isset($metadata['metadata_fields'])) {
        $metadata['metadata_fields'] = [];
    }
    
    // If no order specified, put it at the end of the group
    if ($fieldOrder === null) {
        $maxOrder = 0;
        foreach ($metadata['metadata_fields'] as $field) {
            if ($field['group'] === $group && isset($field['field_order'])) {
                $maxOrder = max($maxOrder, $field['field_order']);
            }
        }
        $fieldOrder = $maxOrder + 1;
    }
    
    $metadata['metadata_fields'][] = [
        'group' => $group,
        'field_key' => $fieldKey,
        'field_label' => $fieldLabel,
        'field_value' => $fieldValue,
        'field_order' => $fieldOrder,
        'is_system' => false
    ];
    
    return $metadata;
}

/**
 * Update field value
 * @param array $metadata
 * @param string $fieldKey
 * @param string $newValue
 * @return array Updated metadata
 */
function updateFieldValue($metadata, $fieldKey, $newValue) {
    if (!isset($metadata['metadata_fields'])) {
        return $metadata;
    }
    
    foreach ($metadata['metadata_fields'] as &$field) {
        if ($field['field_key'] === $fieldKey) {
            $field['field_value'] = $newValue;
            break;
        }
    }
    
    return $metadata;
}

/**
 * Remove a field (only custom fields can be removed)
 * @param array $metadata
 * @param string $fieldKey
 * @return array Updated metadata
 */
function removeField($metadata, $fieldKey) {
    if (!isset($metadata['metadata_fields'])) {
        return $metadata;
    }
    
    $metadata['metadata_fields'] = array_filter($metadata['metadata_fields'], function($field) use ($fieldKey) {
        // Only allow removing non-system fields
        return $field['field_key'] !== $fieldKey || ($field['is_system'] ?? false);
    });
    
    // Re-index array
    $metadata['metadata_fields'] = array_values($metadata['metadata_fields']);
    
    return $metadata;
}
?>
