<?php
/**
 * Event Access Control Helper Functions
 * Manages user access to events based on assigned_events
 */



/**
 * Check if user has access to a specific event
 * Admins and super_admins have access to all events
 * 
 * @param int $event_id The event ID to check
 * @return bool True if user has access, false otherwise
 */
function hasEventAccess($event_id) {
    // Admin and super_admin have access to all events
    $role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';
    if (in_array($role, ['admin', 'super_admin'])) {
        return true;
    }
    
    // Check if event is in user's assigned events
    $assigned_events = $_SESSION['assigned_events'] ?? [];
    return in_array($event_id, $assigned_events) || empty($assigned_events);
}

/**
 * Get list of event IDs the current user can access
 * 
 * @return array Array of event IDs, or empty array for all events (admin)
 */
function getUserAccessibleEvents() {
    $role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';
    
    // Admin and super_admin can access all events (return empty array as flag)
    if (in_array($role, ['admin', 'super_admin'])) {
        return [];
    }
    
    // Return assigned events for other users
    return $_SESSION['assigned_events'] ?? [];
}

/**
 * Filter SQL WHERE clause to restrict by assigned events
 * 
 * @param string $event_column The column name for event_id (default: 'event_id')
 * @return array ['sql' => string, 'params' => array] SQL fragment and parameters
 */
function getEventAccessSQL($event_column = 'event_id') {
    $accessible_events = getUserAccessibleEvents();
    
    // Empty array means admin - no restriction
    if (empty($accessible_events)) {
        return ['sql' => '', 'params' => []];
    }
    
    // Build IN clause for assigned events
    $placeholders = str_repeat('?,', count($accessible_events) - 1) . '?';
    return [
        'sql' => " AND $event_column IN ($placeholders)",
        'params' => $accessible_events
    ];
}

/**
 * Check event access and exit with error if denied
 * 
 * @param int $event_id The event ID to check
 * @param bool $json_response Return JSON error (true) or redirect (false)
 */
function requireEventAccess($event_id, $json_response = true) {
    if (!hasEventAccess($event_id)) {
        if ($json_response) {
            http_response_code(403);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You do not have permission to access this event',
                'error_code' => 'EVENT_ACCESS_DENIED'
            ]);
        } else {
            header('Location: ../login.php?error=access_denied');
        }
        exit;
    }
}
?>