-- =====================================================================
-- MySQL Strict Mode Compatibility Fixes
-- =====================================================================
-- This migration fixes common issues when moving database to servers
-- with MySQL strict mode enabled (common in newer MySQL versions)
-- =====================================================================

-- Issue 1: BLOB/TEXT/JSON columns cannot have non-NULL default values
-- Error: BLOB, TEXT, GEOMETRY or JSON column 'show_categories' can't have a default value
-- Solution: Change DEFAULT '[]' to DEFAULT NULL

ALTER TABLE `event_notification_settings` 
MODIFY COLUMN `show_categories` longtext CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL CHECK (json_valid(`show_categories`));

-- Update any existing NULL values to '[]' for consistency
UPDATE `event_notification_settings` 
SET `show_categories` = '[]' 
WHERE `show_categories` IS NULL;

-- =====================================================================

-- Issue 2: DATE columns cannot use current_timestamp() as default
-- Error: Invalid default value for 'created_at'
-- Solution: Change column type from DATE to DATETIME

ALTER TABLE `participants` 
MODIFY COLUMN `created_at` datetime NOT NULL DEFAULT current_timestamp();

-- =====================================================================

-- Issue 3: Add missing columns that may have been added via migrations
-- These columns might be missing if using an old version of freestyle_db.sql

-- Check and add photo column if it doesn't exist
SET @col_exists = 0;
SELECT COUNT(*) INTO @col_exists 
FROM information_schema.COLUMNS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'participants' 
  AND COLUMN_NAME = 'photo';

SET @query = IF(@col_exists = 0, 
  'ALTER TABLE `participants` ADD COLUMN `photo` varchar(255) DEFAULT NULL COMMENT ''Path to participant photo file''',
  'SELECT ''Column photo already exists'' AS message');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Check and add motion_video column if it doesn't exist
SET @col_exists = 0;
SELECT COUNT(*) INTO @col_exists 
FROM information_schema.COLUMNS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'participants' 
  AND COLUMN_NAME = 'motion_video';

SET @query = IF(@col_exists = 0, 
  'ALTER TABLE `participants` ADD COLUMN `motion_video` varchar(255) DEFAULT NULL COMMENT ''Path to participant motion/video file''',
  'SELECT ''Column motion_video already exists'' AS message');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================================
-- Verification Queries (Optional - uncomment to verify)
-- =====================================================================

-- SELECT 'Checking event_notification_settings.show_categories...' AS status;
-- SELECT COLUMN_NAME, DATA_TYPE, COLUMN_DEFAULT, IS_NULLABLE 
-- FROM information_schema.COLUMNS 
-- WHERE TABLE_SCHEMA = DATABASE() 
--   AND TABLE_NAME = 'event_notification_settings' 
--   AND COLUMN_NAME = 'show_categories';

-- SELECT 'Checking participants.created_at...' AS status;
-- SELECT COLUMN_NAME, DATA_TYPE, COLUMN_DEFAULT, IS_NULLABLE 
-- FROM information_schema.COLUMNS 
-- WHERE TABLE_SCHEMA = DATABASE() 
--   AND TABLE_NAME = 'participants' 
--   AND COLUMN_NAME = 'created_at';

-- SELECT 'Checking participants columns...' AS status;
-- SELECT COLUMN_NAME, DATA_TYPE, COLUMN_DEFAULT, COLUMN_COMMENT
-- FROM information_schema.COLUMNS 
-- WHERE TABLE_SCHEMA = DATABASE() 
--   AND TABLE_NAME = 'participants' 
--   AND COLUMN_NAME IN ('photo', 'motion_video');

-- =====================================================================
-- DONE
-- =====================================================================


