-- Permissions System Tables

-- Permissions table - stores all available permissions
CREATE TABLE IF NOT EXISTS permissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    display_name VARCHAR(150) NOT NULL,
    description TEXT,
    category VARCHAR(50) DEFAULT 'general',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_name (name),
    INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Role Permissions table - maps roles to permissions
CREATE TABLE IF NOT EXISTS role_permissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    role VARCHAR(50) NOT NULL,
    permission_id INT NOT NULL,
    granted_by INT DEFAULT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE,
    FOREIGN KEY (granted_by) REFERENCES users(id) ON DELETE SET NULL,
    UNIQUE KEY unique_role_permission (role, permission_id),
    INDEX idx_role (role)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default permissions based on existing role hierarchy
INSERT INTO permissions (name, display_name, description, category) VALUES
-- User Management
('create_super_admin', 'Create Super Admin', 'Create super admin users', 'user_management'),
('create_admin', 'Create Admin', 'Create admin users', 'user_management'),
('create_event_organizer', 'Create Event Organizer', 'Create event organizer users', 'user_management'),
('create_office', 'Create Office', 'Create office users', 'user_management'),
('create_media', 'Create Media', 'Create media users', 'user_management'),
('create_judge', 'Create Judge', 'Create judge users', 'user_management'),
('create_head_judge', 'Create Head Judge', 'Create head judge users', 'user_management'),
('create_participant', 'Create Participant', 'Create participant users', 'user_management'),
('view_all_users', 'View All Users', 'View all users in the system', 'user_management'),
('view_created_users', 'View Created Users', 'View only users you created', 'user_management'),
('edit_all_users', 'Edit All Users', 'Edit any user in the system', 'user_management'),
('edit_created_users', 'Edit Created Users', 'Edit only users you created', 'user_management'),
('delete_users', 'Delete Users', 'Delete users from the system', 'user_management'),
('manage_user_groups', 'Manage User Groups', 'Create and manage user groups', 'user_management'),

-- Event Management
('view_all_events', 'View All Events', 'View all events in the system', 'event_management'),
('view_assigned_events', 'View Assigned Events', 'View only assigned events', 'event_management'),
('create_events', 'Create Events', 'Create new events', 'event_management'),
('edit_events', 'Edit Events', 'Edit event details', 'event_management'),
('delete_events', 'Delete Events', 'Delete events', 'event_management'),
('publish_events', 'Publish Events', 'Publish events to static sites', 'event_management'),
('manage_event_categories', 'Manage Categories', 'Manage event categories', 'event_management'),
('manage_heat_settings', 'Manage Heat Settings', 'Configure heat settings', 'event_management'),

-- Judge Management
('view_all_judges', 'View All Judges', 'View all judges in the system', 'judge_management'),
('view_event_judges', 'View Event Judges', 'View judges assigned to events', 'judge_management'),
('assign_judges', 'Assign Judges', 'Assign judges to events', 'judge_management'),
('manage_control_points', 'Manage Control Points', 'Manage judge control points', 'judge_management'),

-- Scoring
('submit_scores', 'Submit Scores', 'Submit participant scores', 'scoring'),
('view_scores', 'View Scores', 'View scoring results', 'scoring'),
('edit_scores', 'Edit Scores', 'Edit submitted scores', 'scoring'),
('approve_scores', 'Approve Scores', 'Approve and finalize scores', 'scoring'),

-- Participant Management
('view_participants', 'View Participants', 'View event participants', 'participant_management'),
('create_participants', 'Create Participants', 'Add new participants', 'participant_management'),
('edit_participants', 'Edit Participants', 'Edit participant details', 'participant_management'),
('delete_participants', 'Delete Participants', 'Delete participants', 'participant_management'),
('assign_bib_numbers', 'Assign BIB Numbers', 'Assign BIB numbers to participants', 'participant_management'),

-- Dashboard & Reports
('view_public_dashboard', 'View Public Dashboard', 'Access public event dashboard', 'dashboard'),
('view_admin_dashboard', 'View Admin Dashboard', 'Access admin dashboard', 'dashboard'),
('generate_reports', 'Generate Reports', 'Generate event reports', 'dashboard'),
('export_data', 'Export Data', 'Export event data', 'dashboard'),

-- System Administration
('manage_permissions', 'Manage Permissions', 'Manage roles and permissions', 'system'),
('view_system_logs', 'View System Logs', 'View system activity logs', 'system'),
('manage_settings', 'Manage Settings', 'Manage system settings', 'system');

-- Assign permissions to roles based on existing hierarchy

-- super_admin: ALL permissions
INSERT INTO role_permissions (role, permission_id)
SELECT 'super_admin', id FROM permissions;

-- admin: All except create_super_admin and manage_permissions
INSERT INTO role_permissions (role, permission_id)
SELECT 'admin', id FROM permissions 
WHERE name NOT IN ('create_super_admin', 'manage_permissions');

-- event_organizer: Event and staff management within assigned events
INSERT INTO role_permissions (role, permission_id)
SELECT 'event_organizer', id FROM permissions 
WHERE name IN (
    'create_office', 'create_media', 'create_judge', 'create_head_judge', 'create_participant',
    'view_created_users', 'edit_created_users',
    'view_assigned_events', 'edit_events', 'manage_event_categories', 'manage_heat_settings',
    'view_event_judges', 'assign_judges', 'manage_control_points',
    'view_scores', 'approve_scores',
    'view_participants', 'create_participants', 'edit_participants', 'assign_bib_numbers',
    'view_admin_dashboard', 'generate_reports', 'export_data'
);

-- office: Limited user and participant management
INSERT INTO role_permissions (role, permission_id)
SELECT 'office', id FROM permissions 
WHERE name IN (
    'create_media', 'create_participant',
    'view_created_users', 'edit_created_users',
    'view_assigned_events',
    'view_participants', 'create_participants', 'edit_participants',
    'view_admin_dashboard'
);

-- head_judge: Scoring and approval
INSERT INTO role_permissions (role, permission_id)
SELECT 'head_judge', id FROM permissions 
WHERE name IN (
    'create_participant',
    'view_assigned_events',
    'view_event_judges',
    'submit_scores', 'view_scores', 'edit_scores', 'approve_scores',
    'view_participants',
    'view_public_dashboard'
);

-- judge: Basic scoring
INSERT INTO role_permissions (role, permission_id)
SELECT 'judge', id FROM permissions 
WHERE name IN (
    'create_participant',
    'view_assigned_events',
    'submit_scores', 'view_scores',
    'view_participants',
    'view_public_dashboard'
);

-- media: View-only access
INSERT INTO role_permissions (role, permission_id)
SELECT 'media', id FROM permissions 
WHERE name IN (
    'create_participant',
    'view_assigned_events',
    'view_scores',
    'view_participants',
    'view_public_dashboard'
);

-- participant: Self-service only
INSERT INTO role_permissions (role, permission_id)
SELECT 'participant', id FROM permissions 
WHERE name IN (
    'view_assigned_events',
    'view_public_dashboard'
);
