<?php
/**
 * Judge Drawings API
 * Handles saving/loading judge drawings and background templates
 */

// Suppress all errors and use error_get_last() to handle them
error_reporting(0);
ini_set('display_errors', '0');

// Start output buffering to catch any errors
ob_start();

require_once '../includes/db.php';
require_once '../includes/auth.php';
requireAuth(['judge', 'head_judge', 'admin']);

// Clean any output that might have occurred
ob_end_clean();

// Restart output buffering for the response
ob_start();

header('Content-Type: application/json');

// Check authentication
$user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? null;
if (!isset($_SESSION['user_id']) || !$user_role || !in_array($user_role, ['judge', 'head_judge', 'admin'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$judge_id = $_SESSION['user_id'];
$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'save_drawing':
            saveDrawing($pdo, $judge_id);
            break;
            
        case 'load_drawing':
            loadDrawing($pdo, $judge_id);
            break;
            
        case 'save_template':
            saveTemplate($pdo, $judge_id);
            break;
            
        case 'list_templates':
            listTemplates($pdo, $judge_id);
            break;
            
        case 'load_template':
            loadTemplate($pdo, $judge_id);
            break;
            
        case 'delete_template':
            deleteTemplate($pdo, $judge_id);
            break;
            
        case 'get_last_background':
            getLastBackground($pdo, $judge_id);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

/**
 * Save a complete drawing (background + foreground)
 */
function saveDrawing($pdo, $judge_id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $event_id = $data['event_id'] ?? null;
    $heat_number = $data['heat_number'] ?? null;
    $run_number = $data['run_number'] ?? null;
    $participant_id = $data['participant_id'] ?? null;
    $canvas_width = $data['canvas_width'] ?? 800;
    $canvas_height = $data['canvas_height'] ?? 600;
    $background = $data['background'] ?? [];
    $foreground = $data['foreground'] ?? [];
    $notes = $data['notes'] ?? '';
    
    if (!$event_id || !$heat_number || !$run_number || !$participant_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        return;
    }
    
    // Generate unique filenames
    $timestamp = time();
    $random = bin2hex(random_bytes(8));
    $background_filename = "bg_{$event_id}_{$heat_number}_{$run_number}_{$participant_id}_{$timestamp}_{$random}.json";
    $drawing_filename = "fg_{$event_id}_{$heat_number}_{$run_number}_{$participant_id}_{$timestamp}_{$random}.json";
    
    // Ensure directories exist
    $bg_dir = __DIR__ . '/../drawings/backgrounds/';
    $fg_dir = __DIR__ . '/../drawings/scores/';
    
    if (!is_dir($bg_dir)) {
        mkdir($bg_dir, 0755, true);
    }
    if (!is_dir($fg_dir)) {
        mkdir($fg_dir, 0755, true);
    }
    
    // Save background JSON
    $background_path = $bg_dir . $background_filename;
    $bg_result = file_put_contents($background_path, json_encode($background, JSON_PRETTY_PRINT));
    if ($bg_result === false) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to save background file']);
        return;
    }
    
    // Save drawing JSON
    $drawing_path = $fg_dir . $drawing_filename;
    $fg_result = file_put_contents($drawing_path, json_encode($foreground, JSON_PRETTY_PRINT));
    if ($fg_result === false) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to save drawing file']);
        return;
    }
    
    // Check if drawing already exists for this score
    $check_stmt = $pdo->prepare("
        SELECT id FROM judge_drawings 
        WHERE judge_id = ? AND event_id = ? AND heat_number = ? 
        AND run_number = ? AND participant_id = ?
    ");
    $check_stmt->execute([$judge_id, $event_id, $heat_number, $run_number, $participant_id]);
    $existing = $check_stmt->fetch();
    
    if ($existing) {
        // Update existing drawing
        $stmt = $pdo->prepare("
            UPDATE judge_drawings SET
                background_json = ?,
                drawing_json = ?,
                canvas_width = ?,
                canvas_height = ?,
                notes = ?,
                status = 'draft',
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $background_filename,
            $drawing_filename,
            $canvas_width,
            $canvas_height,
            $notes,
            $existing['id']
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Drawing updated',
            'drawing_id' => $existing['id']
        ]);
    } else {
        // Insert new drawing
        $stmt = $pdo->prepare("
            INSERT INTO judge_drawings (
                judge_id, event_id, heat_number, run_number, participant_id,
                background_json, drawing_json, canvas_width, canvas_height,
                notes, status, created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', NOW(), NOW())
        ");
        $stmt->execute([
            $judge_id,
            $event_id,
            $heat_number,
            $run_number,
            $participant_id,
            $background_filename,
            $drawing_filename,
            $canvas_width,
            $canvas_height,
            $notes
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Drawing saved',
            'drawing_id' => $pdo->lastInsertId()
        ]);
    }
}

/**
 * Load a drawing for specific participant/heat/run
 */
function loadDrawing($pdo, $judge_id) {
    $event_id = $_GET['event_id'] ?? null;
    $heat_number = $_GET['heat_number'] ?? null;
    $run_number = $_GET['run_number'] ?? null;
    $participant_id = $_GET['participant_id'] ?? null;
    
    if (!$event_id || !$heat_number || !$run_number || !$participant_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing parameters']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT * FROM judge_drawings 
        WHERE judge_id = ? AND event_id = ? AND heat_number = ? 
        AND run_number = ? AND participant_id = ?
        ORDER BY updated_at DESC LIMIT 1
    ");
    $stmt->execute([$judge_id, $event_id, $heat_number, $run_number, $participant_id]);
    $drawing = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$drawing) {
        echo json_encode(['success' => false, 'message' => 'No drawing found']);
        return;
    }
    
    // Load background JSON
    $background_path = __DIR__ . '/../drawings/backgrounds/' . $drawing['background_json'];
    $background = file_exists($background_path) 
        ? json_decode(file_get_contents($background_path), true) 
        : [];
    
    // Load drawing JSON
    $drawing_path = __DIR__ . '/../drawings/scores/' . $drawing['drawing_json'];
    $foreground = file_exists($drawing_path) 
        ? json_decode(file_get_contents($drawing_path), true) 
        : [];
    
    echo json_encode([
        'success' => true,
        'drawing' => [
            'id' => $drawing['id'],
            'background' => $background,
            'foreground' => $foreground,
            'canvas_width' => $drawing['canvas_width'],
            'canvas_height' => $drawing['canvas_height'],
            'notes' => $drawing['notes'],
            'status' => $drawing['status']
        ]
    ]);
}

/**
 * Save current background as reusable template
 */
function saveTemplate($pdo, $judge_id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $event_id = $data['event_id'] ?? null;
    $template_name = $data['template_name'] ?? 'My Template';
    $background_strokes = $data['background'] ?? [];
    $canvas_width = $data['canvas_width'] ?? 800;
    $canvas_height = $data['canvas_height'] ?? 600;
    
    if (!$event_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        return;
    }
    
    // Generate unique filename for template
    $timestamp = time();
    $random = bin2hex(random_bytes(8));
    $template_filename = "template_{$judge_id}_{$timestamp}_{$random}.json";
    
    // Save template JSON
    $template_path = __DIR__ . '/../drawings/backgrounds/' . $template_filename;
    file_put_contents($template_path, json_encode($background_strokes, JSON_PRETTY_PRINT));
    
    // Save template metadata to database
    $stmt = $pdo->prepare("
        INSERT INTO judge_drawings (
            judge_id, event_id, heat_number, run_number, participant_id,
            background_json, drawing_json, canvas_width, canvas_height,
            notes, status, created_at, updated_at
        ) VALUES (?, ?, NULL, NULL, NULL, ?, '', ?, ?, ?, 'template', NOW(), NOW())
    ");
    $stmt->execute([
        $judge_id,
        $event_id,
        $template_filename,
        $canvas_width,
        $canvas_height,
        $template_name
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Template saved: ' . $template_name,
        'template_id' => $pdo->lastInsertId()
    ]);
}

/**
 * List all templates for current judge and event
 */
function listTemplates($pdo, $judge_id) {
    $event_id = $_GET['event_id'] ?? null;
    
    if (!$event_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT id, notes as template_name, canvas_width, canvas_height, created_at
        FROM judge_drawings 
        WHERE judge_id = ? AND event_id = ? AND status = 'template'
        ORDER BY created_at DESC
    ");
    $stmt->execute([$judge_id, $event_id]);
    $templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'templates' => $templates
    ]);
}

/**
 * Load a specific template
 */
function loadTemplate($pdo, $judge_id) {
    $template_id = $_GET['template_id'] ?? null;
    
    if (!$template_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing template_id']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT * FROM judge_drawings 
        WHERE id = ? AND judge_id = ? AND status = 'template'
    ");
    $stmt->execute([$template_id, $judge_id]);
    $template = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$template) {
        echo json_encode(['success' => false, 'message' => 'Template not found']);
        return;
    }
    
    // Load template JSON
    $template_path = __DIR__ . '/../drawings/backgrounds/' . $template['background_json'];
    $background = file_exists($template_path) 
        ? json_decode(file_get_contents($template_path), true) 
        : [];
    
    echo json_encode([
        'success' => true,
        'template' => [
            'id' => $template['id'],
            'name' => $template['notes'],
            'background' => $background,
            'canvas_width' => $template['canvas_width'],
            'canvas_height' => $template['canvas_height']
        ]
    ]);
}

/**
 * Delete a template
 */
function deleteTemplate($pdo, $judge_id) {
    $template_id = $_POST['template_id'] ?? null;
    
    if (!$template_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing template_id']);
        return;
    }
    
    // Get template info
    $stmt = $pdo->prepare("
        SELECT background_json FROM judge_drawings 
        WHERE id = ? AND judge_id = ? AND status = 'template'
    ");
    $stmt->execute([$template_id, $judge_id]);
    $template = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$template) {
        echo json_encode(['success' => false, 'message' => 'Template not found']);
        return;
    }
    
    // Delete file
    $template_path = __DIR__ . '/../drawings/backgrounds/' . $template['background_json'];
    if (file_exists($template_path)) {
        unlink($template_path);
    }
    
    // Delete from database
    $delete_stmt = $pdo->prepare("DELETE FROM judge_drawings WHERE id = ? AND judge_id = ?");
    $delete_stmt->execute([$template_id, $judge_id]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Template deleted'
    ]);
}

/**
 * Get the last used background for this judge/event
 */
function getLastBackground($pdo, $judge_id) {
    $event_id = $_GET['event_id'] ?? null;
    
    if (!$event_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing event_id']);
        return;
    }
    
    // Find most recent drawing with a background for this judge and event
    $stmt = $pdo->prepare("
        SELECT background_json, canvas_width, canvas_height
        FROM judge_drawings 
        WHERE judge_id = ? AND event_id = ? 
        AND background_json IS NOT NULL AND background_json != ''
        ORDER BY updated_at DESC LIMIT 1
    ");
    $stmt->execute([$judge_id, $event_id]);
    $drawing = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$drawing) {
        echo json_encode(['success' => false, 'message' => 'No previous background found']);
        return;
    }
    
    // Load background JSON
    $background_path = __DIR__ . '/../drawings/backgrounds/' . $drawing['background_json'];
    $background = file_exists($background_path) 
        ? json_decode(file_get_contents($background_path), true) 
        : [];
    
    echo json_encode([
        'success' => true,
        'background' => $background,
        'canvas_width' => $drawing['canvas_width'],
        'canvas_height' => $drawing['canvas_height']
    ]);
}
