<?php
/**
 * Generate QR Login Token API
 * Creates a temporary one-time token for QR code authentication
 */

header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/../includes/db.php';

// Check authentication
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$user_id = intval($_POST['user_id'] ?? 0);
$user_type = $_POST['user_type'] ?? 'judge';
$expiry_hours = floatval($_POST['expiry_hours'] ?? 5);
$check_existing = isset($_POST['check_existing']) && $_POST['check_existing'] === 'true';

if ($user_id <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid user_id']);
    exit;
}

if ($expiry_hours <= 0 || $expiry_hours > 24) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Expiry time must be between 0 and 24 hours']);
    exit;
}

try {
    // Verify the user exists and get their role
    $stmt = $pdo->prepare("SELECT id, username, role FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'User not found']);
        exit;
    }
    
    // Check for existing valid token if requested
    if ($check_existing) {
        $existing_stmt = $pdo->prepare("
            SELECT token, expires_at 
            FROM qr_login_tokens 
            WHERE user_id = ? AND expires_at > NOW() 
            ORDER BY expires_at DESC 
            LIMIT 1
        ");
        $existing_stmt->execute([$user_id]);
        $existing_token = $existing_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing_token) {
            $base_url = getBaseUrl();
            $qr_url = "{$base_url}/v2/includes/qr_login.php?token={$existing_token['token']}";
            
            echo json_encode([
                'success' => true,
                'existing' => true,
                'token' => $existing_token['token'],
                'qr_url' => $qr_url,
                'expires_at' => $existing_token['expires_at'],
                'username' => $user['username']
            ]);
            exit;
        }
    }
    
    // Generate a secure random token
    $token = bin2hex(random_bytes(32)); // 64 character hex string
    
    // Token expires based on provided hours (default 5 hours)
    $expires_at = date('Y-m-d H:i:s', time() + ($expiry_hours * 3600));
    
    // Clean up old expired tokens for this user
    $cleanup = $pdo->prepare("DELETE FROM qr_login_tokens WHERE user_id = ? AND expires_at < NOW()");
    $cleanup->execute([$user_id]);
    
    // Insert new token
    $insert = $pdo->prepare("
        INSERT INTO qr_login_tokens (token, user_id, user_type, expires_at, ip_address) 
        VALUES (?, ?, ?, ?, ?)
    ");
    $insert->execute([
        $token,
        $user_id,
        $user_type,
        $expires_at,
        $_SERVER['REMOTE_ADDR'] ?? null
    ]);
    
    // Return token and URL
    $base_url = getBaseUrl();
    $qr_url = "{$base_url}/v2/includes/qr_login.php?token={$token}";
    
    echo json_encode([
        'success' => true,
        'token' => $token,
        'qr_url' => $qr_url,
        'expires_at' => $expires_at,
        'username' => $user['username']
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Server error: ' . $e->getMessage()]);
}
