<?php
header('Content-Type: application/json');
include '../includes/auth.php';
include '../includes/db.php';

$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'get_custom_fields':
            // Get all active custom field definitions
            $stmt = $pdo->query("
                SELECT * FROM participant_custom_field_definitions 
                WHERE is_active = 1 
                ORDER BY display_order ASC, id ASC
            ");
            $fields = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'success' => true,
                'fields' => $fields
            ]);
            break;
            
        case 'update_participant_custom_fields':
            // Update custom fields for a participant
            $participant_id = $_POST['participant_id'] ?? null;
            $custom_fields = $_POST['custom_fields'] ?? null;
            
            if (!$participant_id) {
                throw new Exception('Participant ID is required');
            }
            
            // Validate JSON
            if ($custom_fields && !is_array($custom_fields)) {
                $custom_fields = json_decode($custom_fields, true);
            }
            
            $stmt = $pdo->prepare("
                UPDATE participants 
                SET custom_fields = ? 
                WHERE id = ?
            ");
            
            $stmt->execute([
                $custom_fields ? json_encode($custom_fields) : null,
                $participant_id
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Custom fields updated successfully'
            ]);
            break;
            
        case 'get_participant_with_custom_fields':
            // Get participant data including custom fields
            $participant_id = $_GET['participant_id'] ?? null;
            
            if (!$participant_id) {
                throw new Exception('Participant ID is required');
            }
            
            $stmt = $pdo->prepare("
                SELECT p.*, 
                       p.custom_fields as custom_fields_json
                FROM participants p
                WHERE p.id = ?
            ");
            $stmt->execute([$participant_id]);
            $participant = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$participant) {
                throw new Exception('Participant not found');
            }
            
            // Parse custom fields JSON
            if ($participant['custom_fields_json']) {
                $participant['custom_fields'] = json_decode($participant['custom_fields_json'], true);
            } else {
                $participant['custom_fields'] = [];
            }
            
            unset($participant['custom_fields_json']);
            
            echo json_encode([
                'success' => true,
                'participant' => $participant
            ]);
            break;
            
        case 'search_by_custom_field':
            // Search participants by custom field values
            $field_key = $_GET['field_key'] ?? null;
            $search_value = $_GET['search_value'] ?? '';
            
            if (!$field_key) {
                throw new Exception('Field key is required');
            }
            
            // Use JSON_EXTRACT to search within JSON column
            $stmt = $pdo->prepare("
                SELECT p.id, p.first_name, p.last_name, p.bib, p.custom_fields
                FROM participants p
                WHERE JSON_EXTRACT(p.custom_fields, ?) LIKE ?
            ");
            
            $json_path = '$.' . $field_key;
            $stmt->execute([$json_path, '%' . $search_value . '%']);
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Parse custom_fields JSON for each result
            foreach ($results as &$result) {
                if ($result['custom_fields']) {
                    $result['custom_fields'] = json_decode($result['custom_fields'], true);
                }
            }
            
            echo json_encode([
                'success' => true,
                'results' => $results
            ]);
            break;
            
        case 'bulk_update_custom_field':
            // Bulk update a specific custom field for multiple participants
            $participant_ids = $_POST['participant_ids'] ?? [];
            $field_key = $_POST['field_key'] ?? null;
            $field_value = $_POST['field_value'] ?? '';
            
            if (empty($participant_ids) || !$field_key) {
                throw new Exception('Participant IDs and field key are required');
            }
            
            $pdo->beginTransaction();
            
            $stmt = $pdo->prepare("
                UPDATE participants 
                SET custom_fields = JSON_SET(
                    COALESCE(custom_fields, '{}'), 
                    ?, 
                    ?
                )
                WHERE id = ?
            ");
            
            $json_path = '$.' . $field_key;
            $updated = 0;
            
            foreach ($participant_ids as $id) {
                $stmt->execute([$json_path, $field_value, $id]);
                $updated += $stmt->rowCount();
            }
            
            $pdo->commit();
            
            echo json_encode([
                'success' => true,
                'message' => "Updated $updated participants",
                'updated_count' => $updated
            ]);
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
