<?php
// Advanced User Management System
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/TokenSystem.php';   

// Check if user has permission to access user management
// Require at least one of the user creation permissions to access this page
$view_permissions = ['view_all_users', 'view_created_users'];
if (!hasAnyPermission($view_permissions)) {
    header("Location: ../login.php?error=unauthorized");
    exit;
}

$user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';

$message = '';
$current_user_id = $_SESSION['user_id'] ?? 0;

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add_user':
                $message = handleAddUser($pdo, $_POST, $user_role, $current_user_id);
                break;
            case 'edit_user':
                $message = handleEditUser($pdo, $_POST, $user_role, $current_user_id);
                break;
            case 'delete_user':
                $message = handleDeleteUser($pdo, $_POST, $user_role, $current_user_id);
                break;
            case 'add_group':
                $message = handleAddGroup($pdo, $_POST, $current_user_id);
                break;
            case 'manage_group_membership':
                $message = handleGroupMembership($pdo, $_POST, $current_user_id);
                break;
            case 'edit_group':
                $message = handleEditGroup($pdo, $_POST, $current_user_id);
                break;
            case 'add_member_to_group':
                $message = handleAddMemberToGroup($pdo, $_POST, $current_user_id);
                break;
        }
        
        // Redirect to avoid form resubmission
        if (strpos($message, 'success') !== false) {
            header("Location: " . $_SERVER['PHP_SELF'] . "?msg=" . urlencode($message));
            exit;
        }
    }
}

// Handle GET messages
if (isset($_GET['msg'])) {
    $message = $_GET['msg'];
}

function handleAddUser($pdo, $data, $user_role, $current_user_id) {
    try {
        $username = trim($data['username']);
        $password = $data['password'];
        $role = $data['role'];
        $full_name = trim($data['full_name']);
        $email = trim($data['email']);
        $phone = trim($data['phone'] ?? '');
        $address = trim($data['address'] ?? '');
        $emergency_contact = trim($data['emergency_contact'] ?? '');
        $emergency_phone = trim($data['emergency_phone'] ?? '');
        $notes = trim($data['notes'] ?? '');
        $group_ids = $data['groups'] ?? [];
        
        // Check role permissions using new permission system
        $permission_map = [
            'super_admin' => 'create_super_admin',
            'admin' => 'create_admin',
            'event_organizer' => 'create_event_organizer',
            'office' => 'create_office',
            'media' => 'create_media',
            'judge' => 'create_judge',
            'head_judge' => 'create_head_judge',
            'participant' => 'create_participant'
        ];
        
        $required_permission = $permission_map[$role] ?? 'create_participant';
        if (!hasPermission($required_permission)) {
            return "error: You don't have permission to create users with role: $role (missing: $required_permission)";
        }
        
        // Check if username or email exists
        $check = $pdo->prepare("SELECT id FROM users WHERE username = ? OR EXISTS(SELECT 1 FROM user_profiles WHERE email = ?)");
        $check->execute([$username, $email]);
        if ($check->fetch()) {
            return "error: Username or email already exists";
        }
        
        $pdo->beginTransaction();
        
        // Insert into users table with created_by tracking
        $stmt = $pdo->prepare("INSERT INTO users (username, password, role, created_by, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$username, password_hash($password, PASSWORD_DEFAULT), $role, $current_user_id]);
        $user_id = $pdo->lastInsertId();
        
        // Handle avatar upload
        $avatar = null;
        if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
            $avatar = handleAvatarUpload($_FILES['avatar'], $user_id);
        }
        
        // Insert into user_profiles table
        $profile_stmt = $pdo->prepare("INSERT INTO user_profiles (user_id, full_name, email, phone, address, emergency_contact, emergency_phone, notes, avatar, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        $profile_stmt->execute([$user_id, $full_name, $email, $phone, $address, $emergency_contact, $emergency_phone, $notes, $avatar]);
        
        // Add to groups if selected
        if (!empty($group_ids)) {
            $group_stmt = $pdo->prepare("INSERT INTO user_group_memberships (user_id, group_id, role_in_group) VALUES (?, ?, 'member')");
            foreach ($group_ids as $group_id) {
                $group_stmt->execute([$user_id, $group_id]);
            }
        }
        
        // Add assigned events if selected
        $event_ids = $data['assigned_events'] ?? [];
        if (!empty($event_ids)) {
            $event_stmt = $pdo->prepare("INSERT INTO user_assigned_events (user_id, event_id, assigned_by) VALUES (?, ?, ?)");
            foreach ($event_ids as $event_id) {
                if (is_numeric($event_id) && $event_id > 0) {
                    $event_stmt->execute([$user_id, $event_id, $current_user_id]);
                }
            }
        }
        
        // Add basic login permission
        $perm_stmt = $pdo->prepare("INSERT INTO user_permissions (user_id, permission, granted_by) VALUES (?, 'login', ?)");
        $perm_stmt->execute([$user_id, $current_user_id]);
        
        $pdo->commit();

        $result = TokenSystem::charge('create_user', $_SESSION['user_id'], [
                'user_id' => $user_id,
                'reference_type' => 'user_id'
            ]);   

        return "success: User created successfully with ID: $user_id";
         // Charge with additional context
        


        
    } catch (Exception $e) {
        $pdo->rollback();
        return "error: " . $e->getMessage();
    }
}

function handleEditUser($pdo, $data, $user_role, $current_user_id) {
    try {
        $user_id = intval($data['user_id']);
        
        // Debug: Log incoming data
        error_log("Edit User Data: " . print_r($data, true));
        
        // Check if this is just a status toggle
        if (isset($data['toggle_status'])) {
            $is_active = isset($data['is_active']) ? 1 : 0;
            
            // Try to update user_profiles table first
            try {
                $profile_stmt = $pdo->prepare("UPDATE user_profiles SET is_active = ? WHERE user_id = ?");
                $profile_stmt->execute([$is_active, $user_id]);
            } catch (Exception $e) {
                // If user_profiles doesn't exist, update users table directly
                $stmt = $pdo->prepare("UPDATE users SET is_active = ? WHERE id = ?");
                $stmt->execute([$is_active, $user_id]);
            }
            
            return "success: User status updated successfully";
        }
        
        // Full user edit - check if required fields are present
        if (!isset($data['username']) || !isset($data['role']) || !isset($data['full_name'])) {
            return "error: Missing required fields in form submission";
        }
        
        // Full user edit
        $username = trim($data['username']);
        $role = $data['role'];
        $full_name = trim($data['full_name']);
        $email = trim($data['email'] ?? '');
        $phone = trim($data['phone'] ?? '');
        $address = trim($data['address'] ?? '');
        $emergency_contact = trim($data['emergency_contact'] ?? '');
        $emergency_phone = trim($data['emergency_phone'] ?? '');
        $notes = trim($data['notes'] ?? '');
        $is_active = isset($data['is_active']) ? 1 : 0;
        $new_password = trim($data['new_password'] ?? '');
        
        error_log("Parsed values - Email: $email, Address: $address, Emergency Contact: $emergency_contact");
        
        // Check role permissions using new permission system
        $permission_map = [
            'super_admin' => 'create_super_admin',
            'admin' => 'create_admin',
            'event_organizer' => 'create_event_organizer',
            'office' => 'create_office',
            'media' => 'create_media',
            'judge' => 'create_judge',
            'head_judge' => 'create_head_judge',
            'participant' => 'create_participant'
        ];
        
        $required_permission = $permission_map[$role] ?? 'create_participant';
        if (!hasPermission($required_permission)) {
            return "error: You don't have permission to assign role: $role (missing: $required_permission)";
        }
        
        $pdo->beginTransaction();
        
        // Update users table - include password if provided
        if (!empty($new_password)) {
            $stmt = $pdo->prepare("UPDATE users SET username = ?, password = ?, role = ? WHERE id = ?");
            $stmt->execute([$username, password_hash($new_password, PASSWORD_DEFAULT), $role, $user_id]);
        } else {
            $stmt = $pdo->prepare("UPDATE users SET username = ?, role = ? WHERE id = ?");
            $stmt->execute([$username, $role, $user_id]);
        }
        
        // Handle avatar upload
        $avatar_update = '';
        $avatar_params = [];
        if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
            // Get old avatar to delete
            $old_avatar_stmt = $pdo->prepare("SELECT avatar FROM user_profiles WHERE user_id = ?");
            $old_avatar_stmt->execute([$user_id]);
            $old_avatar = $old_avatar_stmt->fetchColumn();
            
            $new_avatar = handleAvatarUpload($_FILES['avatar'], $user_id);
            if ($new_avatar) {
                $avatar_update = ', avatar = ?';
                $avatar_params[] = $new_avatar;
                
                // Delete old avatar file
                if ($old_avatar && file_exists($old_avatar)) {
                    unlink($old_avatar);
                }
            }
        }
        
        // Update user_profiles table
        try {
            // Check if is_active column exists in user_profiles
            $columns = $pdo->query("SHOW COLUMNS FROM user_profiles")->fetchAll(PDO::FETCH_COLUMN);
            $has_is_active = in_array('is_active', $columns);
            
            if ($has_is_active) {
                $sql = "UPDATE user_profiles SET full_name = ?, email = ?, phone = ?, address = ?, emergency_contact = ?, emergency_phone = ?, notes = ?, is_active = ?" . $avatar_update . ", updated_at = NOW() WHERE user_id = ?";
                $params = array_merge([$full_name, $email, $phone, $address, $emergency_contact, $emergency_phone, $notes, $is_active], $avatar_params, [$user_id]);
            } else {
                $sql = "UPDATE user_profiles SET full_name = ?, email = ?, phone = ?, address = ?, emergency_contact = ?, emergency_phone = ?, notes = ?" . $avatar_update . ", updated_at = NOW() WHERE user_id = ?";
                $params = array_merge([$full_name, $email, $phone, $address, $emergency_contact, $emergency_phone, $notes], $avatar_params, [$user_id]);
            }
            $profile_stmt = $pdo->prepare($sql);
            $profile_stmt->execute($params);
        } catch (Exception $e) {
            // If user_profiles doesn't exist, just update basic info in users table
            $stmt = $pdo->prepare("UPDATE users SET username = ?, role = ? WHERE id = ?");
            $stmt->execute([$username, $role, $user_id]);
        }
        
        // Update assigned events
        $event_ids = $data['assigned_events'] ?? [];
        
        // Delete existing assignments
        $delete_events_stmt = $pdo->prepare("DELETE FROM user_assigned_events WHERE user_id = ?");
        $delete_events_stmt->execute([$user_id]);
        
        // Add new assignments
        if (!empty($event_ids)) {
            $event_stmt = $pdo->prepare("INSERT INTO user_assigned_events (user_id, event_id, assigned_by) VALUES (?, ?, ?)");
            foreach ($event_ids as $event_id) {
                if (is_numeric($event_id) && $event_id > 0) {
                    $event_stmt->execute([$user_id, $event_id, $current_user_id]);
                }
            }
        }
        
        $pdo->commit();
        return "success: User updated successfully";
        
    } catch (Exception $e) {
        $pdo->rollback();
        return "error: " . $e->getMessage();
    }
}

function handleDeleteUser($pdo, $data, $user_role, $current_user_id) {
    try {
        $user_id = intval($data['user_id']);
        
        // Check if user has delete permission
        if (!hasPermission('delete_users')) {
            return "error: You don't have permission to delete users";
        }
        
        // Prevent deleting self
        if ($user_id == $current_user_id) {
            return "error: You cannot delete your own account";
        }
        
        // Check target user role
        $target_user = $pdo->prepare("SELECT role FROM users WHERE id = ?");
        $target_user->execute([$user_id]);
        $target_role = $target_user->fetchColumn();
        
        // Prevent deleting super_admin unless current user is super_admin
        if ($target_role === 'super_admin' && !hasPermission('create_super_admin')) {
            return "error: You cannot delete super admin accounts";
        }
        
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        
        return "success: User deleted successfully";
        
    } catch (Exception $e) {
        return "error: " . $e->getMessage();
    }
}

function handleAddGroup($pdo, $data, $current_user_id) {
    try {
        $name = trim($data['group_name']);
        $description = trim($data['group_description'] ?? '');
        $color = $data['group_color'] ?? '#007bff';
        
        $stmt = $pdo->prepare("INSERT INTO user_groups (name, description, color, created_by) VALUES (?, ?, ?, ?)");
        $stmt->execute([$name, $description, $color, $current_user_id]);
        
        return "success: Group created successfully";
        
    } catch (Exception $e) {
        return "error: " . $e->getMessage();
    }
}

function handleGroupMembership($pdo, $data, $current_user_id) {
    try {
        $user_id = intval($data['user_id']);
        $group_id = intval($data['group_id']);
        $action = $data['membership_action']; // 'add' or 'remove'
        
        if ($action === 'add') {
            $stmt = $pdo->prepare("INSERT IGNORE INTO user_group_memberships (user_id, group_id, role_in_group) VALUES (?, ?, 'member')");
            $stmt->execute([$user_id, $group_id]);
            return "success: User added to group";
        } else {
            $stmt = $pdo->prepare("DELETE FROM user_group_memberships WHERE user_id = ? AND group_id = ?");
            $stmt->execute([$user_id, $group_id]);
            return "success: User removed from group";
        }
        
    } catch (Exception $e) {
        return "error: " . $e->getMessage();
    }
}

function handleEditGroup($pdo, $data, $current_user_id) {
    try {
        $group_id = intval($data['group_id']);
        $name = trim($data['group_name']);
        $description = trim($data['group_description'] ?? '');
        $color = $data['group_color'] ?? '#007bff';
        
        $stmt = $pdo->prepare("UPDATE user_groups SET name = ?, description = ?, color = ? WHERE id = ?");
        $stmt->execute([$name, $description, $color, $group_id]);
        
        return "success: Group updated successfully";
        
    } catch (Exception $e) {
        return "error: " . $e->getMessage();
    }
}

function handleAddMemberToGroup($pdo, $data, $current_user_id) {
    try {
        $user_id = intval($data['user_id']);
        $group_id = intval($data['group_id']);
        $role_in_group = $data['role_in_group'] ?? 'member';
        
        // Check if user is already in group
        $check = $pdo->prepare("SELECT id FROM user_group_memberships WHERE user_id = ? AND group_id = ?");
        $check->execute([$user_id, $group_id]);
        if ($check->fetch()) {
            return "error: User is already a member of this group";
        }
        
        // Check what columns exist in the table
        $columns = $pdo->query("SHOW COLUMNS FROM user_group_memberships")->fetchAll(PDO::FETCH_COLUMN);
        
        // Build query based on available columns
        $insertColumns = "user_id, group_id, role_in_group";
        $insertValues = "?, ?, ?";
        $params = [$user_id, $group_id, $role_in_group];
        
        // Add added_by if column exists
        if (in_array('added_by', $columns)) {
            $insertColumns .= ", added_by";
            $insertValues .= ", ?";
            $params[] = $current_user_id;
        }
        
        // Add timestamp column if any exists
        if (in_array('created_at', $columns)) {
            $insertColumns .= ", created_at";
            $insertValues .= ", NOW()";
        } elseif (in_array('joined_at', $columns)) {
            $insertColumns .= ", joined_at";
            $insertValues .= ", NOW()";
        } elseif (in_array('added_at', $columns)) {
            $insertColumns .= ", added_at";
            $insertValues .= ", NOW()";
        }
        
        $stmt = $pdo->prepare("INSERT INTO user_group_memberships ($insertColumns) VALUES ($insertValues)");
        $result = $stmt->execute($params);
        
        if ($result) {
            return "success: Member added successfully";
        } else {
            return "error: Failed to add member to group";
        }
        
    } catch (Exception $e) {
        // If the table doesn't exist, try to create it with basic structure
        if (strpos($e->getMessage(), "doesn't exist") !== false) {
            try {
                $pdo->exec("
                    CREATE TABLE user_group_memberships (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        user_id INT NOT NULL,
                        group_id INT NOT NULL,
                        role_in_group VARCHAR(50) DEFAULT 'member',
                        added_by INT NULL,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        UNIQUE KEY unique_membership (user_id, group_id),
                        KEY idx_user_id (user_id),
                        KEY idx_group_id (group_id)
                    )
                ");
                
                // Retry the insert
                $stmt = $pdo->prepare("INSERT INTO user_group_memberships (user_id, group_id, role_in_group, added_by, created_at) VALUES (?, ?, ?, ?, NOW())");
                $result = $stmt->execute([$user_id, $group_id, $role_in_group, $current_user_id]);
                
                if ($result) {
                    return "success: Member added successfully (table created)";
                }
            } catch (Exception $e2) {
                return "error: Could not create membership table: " . $e2->getMessage();
            }
        }
        return "error: " . $e->getMessage();
    }
}

// Handle avatar upload
function handleAvatarUpload($file, $user_id) {
    $upload_dir = '../uploads/avatars/';
    
    // Create directory if it doesn't exist
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Validate file type
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($mime_type, $allowed_types)) {
        return null;
    }
    
    // Validate file size (max 5MB)
    if ($file['size'] > 5 * 1024 * 1024) {
        return null;
    }
    
    // Generate unique filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'avatar_' . $user_id . '_' . time() . '.' . $extension;
    $filepath = $upload_dir . $filename;
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return $filepath;
    }
    
    return null;
}

function getRolesUserCanCreate($user_role) {
    // Map roles to their creation permissions
    $role_permission_map = [
        'super_admin' => 'create_super_admin',
        'admin' => 'create_admin',
        'event_organizer' => 'create_event_organizer',
        'office' => 'create_office',
        'media' => 'create_media',
        'judge' => 'create_judge',
        'head_judge' => 'create_head_judge',
        'participant' => 'create_participant'
    ];
    
    $can_create = [];
    foreach ($role_permission_map as $role => $permission) {
        if (hasPermission($permission)) {
            $can_create[] = $role;
        }
    }
    
    return $can_create;
}

// Fetch data for display
try {
    // Build WHERE clause based on user permissions
    $where_conditions = [];
    $where_params = [];
    
    // Filter users based on permissions
    if (hasPermission('view_all_users')) {
        // Can see all users (no filter needed)
    } elseif (hasPermission('view_created_users')) {
        // Can only see users they created
        $where_conditions[] = "u.created_by = ?";
        $where_params[] = $current_user_id;
    } else {
        // No view permission - shouldn't happen due to page-level check, but handle it
        $where_conditions[] = "u.id = ?";
        $where_params[] = $current_user_id; // Only see themselves
    }
    
    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    // Get all users with profiles and creator information
    try {
        $users_query = "
            SELECT u.id, u.username, u.role, u.created_at, u.last_login, u.created_by,
                   p.full_name, p.email, p.phone, 
                   COALESCE(p.is_active, 1) as is_active,
                   creator.username as created_by_name,
                   GROUP_CONCAT(CONCAT(g.name, ':', g.color) SEPARATOR '|') as `groups`
            FROM users u
            LEFT JOIN user_profiles p ON u.id = p.user_id
            LEFT JOIN users creator ON u.created_by = creator.id
            LEFT JOIN user_group_memberships ugm ON u.id = ugm.user_id
            LEFT JOIN user_groups g ON ugm.group_id = g.id
            $where_clause
            GROUP BY u.id
            ORDER BY u.created_at DESC
        ";
        $stmt = $pdo->prepare($users_query);
        $stmt->execute($where_params);
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        // Fallback query without group information if tables don't exist
        $users_query = "
            SELECT u.id, u.username, u.role, u.created_at, u.last_login, u.created_by,
                   p.full_name, p.email, p.phone, 
                   COALESCE(p.is_active, 1) as is_active,
                   creator.username as created_by_name,
                   '' as `groups`
            FROM users u
            LEFT JOIN user_profiles p ON u.id = p.user_id
            LEFT JOIN users creator ON u.created_by = creator.id
            $where_clause
            ORDER BY u.created_at DESC
        ";
        $stmt = $pdo->prepare($users_query);
        $stmt->execute($where_params);
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get all groups
    $groups = $pdo->query("
        SELECT g.*, u.username as created_by_name,
               COUNT(ugm.user_id) as member_count
        FROM user_groups g
        LEFT JOIN users u ON g.created_by = u.id
        LEFT JOIN user_group_memberships ugm ON g.id = ugm.group_id
        GROUP BY g.id
        ORDER BY g.name
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    // Get role statistics
    $role_stats = $pdo->query("
        SELECT role, COUNT(*) as count
        FROM users
        GROUP BY role
        ORDER BY count DESC
    ")->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $users = [];
    $groups = [];
    $role_stats = [];
    if (!$message) {
        $message = "error: Database error: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Style Score Admin</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    <!-- Removed Font Awesome; using Lucide icons -->
    <script src="../assets/js/lucide.min.js"></script>
</head>
<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>
    
    <div class="container-fluid container-StyleScore p-4">
        <?php       
            $use_localStorage = true;
            $show_header = false;
            include 'event_actions_menu.php'; 
            ?>
        <div class="row">
            <div class="col-12">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2 class="mb-1">
                            <i data-lucide="users" class="text-primary" style="width:20px;height:20px;"></i><span class="ms-1">User Management</span>
                        </h2>
                        <p class="text-muted mb-0">Manage users, roles and groups</p>
                    </div>
                    <div class="d-flex align-items-center gap-2">
                        <button class="btn btn-outline-info" data-bs-toggle="modal" data-bs-target="#helpModal" title="Help & Guide">
                            <i data-lucide="help-circle" style="width:16px;height:16px;" class="me-1"></i> Help
                        </button>
                        <span class="badge bg-primary fs-6">Your Role: <?= ucfirst(str_replace('_', ' ', $user_role)) ?></span>
                    </div>
    <!-- Help Modal -->
    <div class="modal fade" id="helpModal" tabindex="-1" aria-labelledby="helpModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="helpModalLabel">
                        <i data-lucide="help-circle" style="width:18px;height:18px;" class="me-2 text-info"></i>User Management Help
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0" style="height:80vh;">
                    <iframe src="../help/user_management_help.html" style="width:100%;height:100%;border:0;" title="User Management Help"></iframe>
                </div>
            </div>
        </div>
    </div>
                </div>

                <!-- Messages -->
                <?php if ($message): ?>
                    <?php
                    $msg_type = strpos($message, 'success:') === 0 ? 'success' : 'danger';
                    $msg_text = substr($message, strpos($message, ':') + 1);
                    ?>
                    <div class="alert alert-<?= $msg_type ?> alert-dismissible fade show">
                        <?php $icon = $msg_type === 'success' ? 'check-circle' : 'alert-triangle'; ?>
                        <i data-lucide="<?= $icon ?>" class="me-2"></i>
                        <?= htmlspecialchars(trim($msg_text)) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Statistics Cards -->
                <div class="row g-3 mb-4 d-none">
                    <div class="col-lg-3 col-md-6">
                        <div class="border-0 shadow-none card border-primary">
                            <div class="border-0 shadow-none card-body text-center">
                                <i data-lucide="users" class="text-primary mb-2" style="width:32px;height:32px;"></i>
                                <h4 class="mb-1"><?= count($users) ?></h4>
                                <p class="text-muted mb-0">Total Users</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6">
                        <div class="border-0 shadow-none card border-success">
                            <div class="border-0 shadow-none card-body text-center">
                                <i data-lucide="layers" class="text-success mb-2" style="width:32px;height:32px;"></i>
                                <h4 class="mb-1"><?= count($groups) ?></h4>
                                <p class="text-muted mb-0">Active Groups</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6">
                        <div class="border-0 shadow-none card border-info">
                            <div class="border-0 shadow-none card-body text-center">
                                <i data-lucide="user-check" class="text-info mb-2" style="width:32px;height:32px;"></i>
                                <h4 class="mb-1"><?= count(array_filter($users, function($u) { return $u['is_active']; })) ?></h4>
                                <p class="text-muted mb-0">Active Users</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6">
                        <div class="border-0 shadow-none card border-warning">
                            <div class="border-0 shadow-none card-body text-center">
                                <i data-lucide="badge-check" class="text-warning mb-2" style="width:32px;height:32px;"></i>
                                <h4 class="mb-1"><?= count($role_stats) ?></h4>
                                <p class="text-muted mb-0">Different Roles</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Main Content Tabs -->
                <div class="border-0 shadow-none card">
                    <div class="border-0 shadow-none card-header p-0">
                        <ul class="nav nav-tabs card-header-tabs" id="managementTabs" role="tablist">
                            <li class="nav-item" role="presentation">
                                <button class="nav-link active" id="users-tab" data-bs-toggle="tab" data-bs-target="#users" type="button" role="tab">
                                    <i data-lucide="users" style="width:16px;height:16px;" class="me-2"></i>Users
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="groups-tab" data-bs-toggle="tab" data-bs-target="#groups" type="button" role="tab">
                                    <i data-lucide="layers" style="width:16px;height:16px;" class="me-2"></i>Groups
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="roles-tab" data-bs-toggle="tab" data-bs-target="#roles" type="button" role="tab">
                                    <i data-lucide="badge-check" style="width:16px;height:16px;" class="me-2"></i>Roles & Permissions
                                </button>
                            </li>
                        </ul>
                    </div>
                    
                    <div class="border-0 shadow-none card-body">
                        <div class="tab-content" id="managementTabsContent">
                        <!-- Users Tab -->
                        <div class="tab-pane fade show active" id="users" role="tabpanel">
                            <div class="d-flex justify-content-between align-items-center mb-4">
                                <h5 class="mb-0">User Management</h5>
                                <div class="d-flex align-items-center gap-2 flex-column ">

                                <?php 
                                $cost = TokenSystem::price('create_user');
                                    if ($cost !== null) {
                                        $cost_msg = "<span class=\"badge bg-info-subtle text-info\">This action costs <i class=\"fa-solid fa-coins\"></i> <span class=\"fw-bold text-black bg-white p-1 rounded\">{$cost}</span> tokens</span>";
                                    }

                                if (!TokenSystem::canAfford('create_user', $_SESSION['user_id'])) {
                                $balance = TokenSystem::getBalance($_SESSION['user_id']);
                                echo "<span class=\"badge bg-warning-subtle text-warning\">Insufficient tokens. Current balance: <i class=\"fa-solid fa-coins\"></i> <span class=\"fw-bold text-black bg-white p-1 rounded\">{$balance}</span></span>";
                                echo $cost_msg;
                                ?>
                                 <button class="btn btn-primary" disabled>
                                    <i data-lucide="plus" style="width:16px;height:16px;" class="me-2"></i>Add New User
                                </button>
                                <?php
                            } else {
                                echo $cost_msg;
                                ?>
                                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addUserModal">
                                    <i data-lucide="plus" style="width:16px;height:16px;" class="me-2"></i>Add New User
                                </button>
                                <?php
                            }
                            ?>                              
                                </div>
                            </div>
                            
                            <div class="table-responsive">
                                <table class="table table-hover align-middle">
                                    <thead class="table-light">
                                        <tr>
                                            <th>ID</th>
                                            <th>User</th>
                                            <th>Contact</th>
                                            <th>Role</th>
                                            <th>Groups</th>
                                            <th>Status</th>
                                            <th>Created</th>
                                            <th>Created By</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($users as $user): ?>
                                        <tr>
                                            <td>
                                                <span class="badge bg-secondary">#<?= $user['id'] ?></span>
                                            </td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="bg-primary text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 40px; height: 40px;">
                                                        <strong><?= strtoupper(substr($user['full_name'] ?: $user['username'], 0, 2)) ?></strong>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?= htmlspecialchars($user['full_name'] ?: $user['username']) ?></h6>
                                                        <small class="text-muted">@<?= htmlspecialchars($user['username']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if ($user['email']): ?>
                                                    <div><i data-lucide="mail" style="width:14px;height:14px;" class="me-1"></i><?= htmlspecialchars($user['email']) ?></div>
                                                <?php endif; ?>
                                                <?php if ($user['phone']): ?>
                                                    <div><i data-lucide="phone" style="width:14px;height:14px;" class="me-1"></i><?= htmlspecialchars($user['phone']) ?></div>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?= getRoleColor($user['role']) ?>">
                                                    <?= ucfirst(str_replace('_', ' ', $user['role'])) ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($user['groups']): ?>
                                                    <?php 
                                                    $groups_data = explode('|', $user['groups']);
                                                    foreach ($groups_data as $group_data):
                                                        if (strpos($group_data, ':') !== false):
                                                            list($group_name, $group_color) = explode(':', $group_data);
                                                    ?>
                                                        <span class="badge me-1" style="background-color: <?= htmlspecialchars($group_color) ?>">
                                                            <?= htmlspecialchars($group_name) ?>
                                                        </span>
                                                    <?php 
                                                        endif;
                                                    endforeach; 
                                                    ?>
                                                <?php else: ?>
                                                    <span class="text-muted">No groups</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <i data-lucide="circle" class="<?= $user['is_active'] ? 'text-success' : 'text-danger' ?>" style="width:10px;height:10px;"></i>
                                                <?= $user['is_active'] ? 'Active' : 'Inactive' ?>
                                            </td>
                                            <td>
                                                <small><?= date('M j, Y', strtotime($user['created_at'])) ?></small>
                                            </td>
                                            <td>
                                                <?php if ($user['created_by_name']): ?>
                                                    <small class="text-muted">@<?= htmlspecialchars($user['created_by_name']) ?></small>
                                                <?php else: ?>
                                                    <small class="badge bg-secondary">System</small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <button class="btn btn-outline-primary" onclick="editUser(<?= $user['id'] ?>)">
                                                        <i data-lucide="edit" style="width:14px;height:14px;"></i>
                                                    </button>
                                                    <button class="btn btn-outline-info" onclick="viewUser(<?= $user['id'] ?>)">
                                                        <i data-lucide="eye" style="width:14px;height:14px;"></i>
                                                    </button>
                                                    <button class="btn btn-outline-<?= $user['is_active'] ? 'warning' : 'success' ?>" onclick="toggleUserStatus(<?= $user['id'] ?>, <?= $user['is_active'] ? 0 : 1 ?>)" title="<?= $user['is_active'] ? 'Deactivate' : 'Activate' ?> User">
                                                        <i data-lucide="<?= $user['is_active'] ? 'user-x' : 'user-check' ?>" style="width:14px;height:14px;"></i>
                                                    </button>
                                                    <?php if ($user['id'] != $current_user_id): ?>
                                                    <button class="btn btn-outline-danger" onclick="deleteUser(<?= $user['id'] ?>)">
                                                        <i data-lucide="trash-2" style="width:14px;height:14px;"></i>
                                                    </button>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        
                        <!-- Groups Tab -->
                        <div class="tab-pane fade" id="groups" role="tabpanel">
                            <div class="d-flex justify-content-between align-items-center mb-4">
                                <h5 class="mb-0">Group Management</h5>
                                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addGroupModal">
                                    <i data-lucide="plus" style="width:16px;height:16px;" class="me-2"></i>Create New Group
                                </button>
                            </div>
                            
                            <div class="row g-4">
                                <?php foreach ($groups as $group): ?>
                                <div class="col-lg-4 col-md-6">
                                    <div class="border-0 shadow-none card h-100 border">
                                        <div class="border-0 shadow-none card-header d-flex justify-content-between align-items-center">
                                            <h6 class="mb-0">
                                                <i data-lucide="layers" style="width:16px;height:16px;" class="me-2 text-primary"></i><?= htmlspecialchars($group['name']) ?>
                                            </h6>
                                            <span class="badge bg-secondary"><?= $group['member_count'] ?> members</span>
                                        </div>
                                        <div class="border-0 shadow-none card-body">
                                            <p class="card-text text-muted"><?= htmlspecialchars($group['description'] ?: 'No description') ?></p>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <small class="text-muted">
                                                    Created by: <?= htmlspecialchars($group['created_by_name'] ?: 'Unknown') ?>
                                                </small>
                                                <div class="btn-group btn-group-sm">
                                                    <button class="btn btn-outline-primary" onclick="editGroup(<?= $group['id'] ?>)">
                                                        <i data-lucide="edit" style="width:14px;height:14px;"></i>
                                                    </button>
                                                    <button class="btn btn-outline-info" onclick="viewGroupMembers(<?= $group['id'] ?>)">
                                                        <i data-lucide="users" style="width:14px;height:14px;"></i>
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <!-- Roles Tab -->
                        <div class="tab-pane fade" id="roles" role="tabpanel">
                            <h5 class="mb-4">Role Statistics & Permissions</h5>
                            
                            <div class="row g-4">
                                <?php foreach ($role_stats as $role_stat): ?>
                                <div class="col-lg-3 col-md-6">
                                    <div class="border-0 shadow-none card text-center">
                                        <div class="border-0 shadow-none card-body">
                                            <i data-lucide="badge-check" class="text-<?= getRoleColor($role_stat['role']) ?> mb-3" style="width:32px;height:32px;"></i>
                                            <h4><?= $role_stat['count'] ?></h4>
                                            <p class="card-text"><?= ucfirst(str_replace('_', ' ', $role_stat['role'])) ?></p>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <div class="mt-5">
                                <h6>Role Hierarchy & Permissions</h6>
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Role</th>
                                                <th>Can Create</th>
                                                <th>Permissions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <tr>
                                                <td><span class="badge bg-danger">Super Admin</span></td>
                                                <td>All roles</td>
                                                <td>Full system access, user management, system settings</td>
                                            </tr>
                                            <tr>
                                                <td><span class="badge bg-warning">Admin</span></td>
                                                <td>Judge, Office, Media, Participant</td>
                                                <td>User management, event management, reports</td>
                                            </tr>
                                            <tr>
                                                <td><span class="badge bg-success">Head Judge</span></td>
                                                <td>-</td>
                                                <td>Judge management, score override, judge reports</td>
                                            </tr>
                                            <tr>
                                                <td><span class="badge bg-primary">Judge</span></td>
                                                <td>-</td>
                                                <td>Score entry, assigned events only</td>
                                            </tr>
                                            <tr>
                                                <td><span class="badge bg-info">Office</span></td>
                                                <td>Media, Participant</td>
                                                <td>Event coordination, participant management</td>
                                            </tr>
                                            <tr>
                                                <td><span class="badge bg-secondary">Media</span></td>
                                                <td>-</td>
                                                <td>Photo/video access, media management</td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add User Modal -->
    <div class="modal fade" id="addUserModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="user-plus" style="width:18px;height:18px;" class="me-2"></i>Add New User
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" id="addUserForm" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="add_user">
                    <div class="modal-body">
                        <div class="row g-3">
                            <!-- Basic Info -->
                            <div class="col-12">
                                <h6 class="border-bottom pb-2 mb-3">Basic Information</h6>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Full Name *</label>
                                <input type="text" class="form-control" name="full_name" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Username *</label>
                                <input type="text" class="form-control" name="username" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Email <span class="email-required-indicator">*</span></label>
                                <input type="email" class="form-control" name="email" id="add_email" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Phone</label>
                                <input type="tel" class="form-control" name="phone">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Password *</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" name="password" id="add_password" required>
                                    <button type="button" class="btn btn-outline-secondary" id="generate_password_btn" style="display:none;" onclick="generateJudgePassword()" title="Generate password">
                                        <i data-lucide="refresh-cw" style="width:16px;height:16px;"></i>
                                    </button>
                                </div>
                                <small class="text-muted" id="password_hint" style="display:none;">Auto-generated 6-character password</small>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Role *</label>
                                <select class="form-select" name="role" required>
                                    <?php foreach (getRolesUserCanCreate($user_role) as $role): ?>
                                    <option value="<?= $role ?>"><?= ucfirst(str_replace('_', ' ', $role)) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <!-- Profile Details -->
                            <div class="col-12">
                                <h6 class="border-bottom pb-2 mb-3 mt-2">Profile Details</h6>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Address</label>
                                <textarea class="form-control" name="address" rows="2"></textarea>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Emergency Contact</label>
                                <input type="text" class="form-control" name="emergency_contact" placeholder="Name">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Emergency Phone</label>
                                <input type="tel" class="form-control" name="emergency_phone" placeholder="Phone number">
                            </div>
                            <div class="col-12">
                                <label class="form-label">Avatar</label>
                                <input type="file" class="form-control" name="avatar" accept="image/*">
                                <small class="text-muted">Allowed: JPG, PNG, GIF, WebP (max 5MB)</small>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Notes</label>
                                <textarea class="form-control" name="notes" rows="2" placeholder="Internal notes about this user"></textarea>
                            </div>
                            
                            <!-- Groups -->
                            <div class="col-12">
                                <h6 class="border-bottom pb-2 mb-3 mt-2">Group Membership</h6>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Groups</label>
                                <div class="row">
                                    <?php foreach ($groups as $group): ?>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" name="groups[]" value="<?= $group['id'] ?>" id="group_<?= $group['id'] ?>">
                                            <label class="form-check-label" for="group_<?= $group['id'] ?>">
                                                <span class="badge" style="background-color: <?= htmlspecialchars($group['color']) ?>">
                                                    <?= htmlspecialchars($group['name']) ?>
                                                </span>
                                            </label>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            
                            <!-- Assigned Events -->
                            <div class="col-12">
                                <h6 class="border-bottom pb-2 mb-3 mt-2">Assigned Events</h6>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Events</label>
                                <select class="form-select" name="assigned_events[]" id="add_assigned_events" multiple size="6">
                                    <option value="" disabled>Loading events...</option>
                                </select>
                                <small class="text-muted">Hold Ctrl (Cmd on Mac) to select multiple events</small>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i data-lucide="plus" style="width:16px;height:16px;" class="me-2"></i>Create User
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Add Group Modal -->
    <div class="modal fade" id="addGroupModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="layers" style="width:18px;height:18px;" class="me-2"></i>Create New Group
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post">
                    <input type="hidden" name="action" value="add_group">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Group Name *</label>
                            <input type="text" class="form-control" name="group_name" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" name="group_description" rows="3"></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Color</label>
                            <input type="color" class="form-control form-control-color" name="group_color" value="#007bff">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i data-lucide="plus" style="width:16px;height:16px;" class="me-2"></i>Create Group
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit User Modal -->
    <div class="modal fade" id="editUserModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="user-pen" style="width:18px;height:18px;" class="me-2"></i>Edit User
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" id="editUserForm" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="edit_user">
                    <input type="hidden" name="user_id" id="edit_user_id">
                    <div class="modal-body">
                        <div class="row g-3">
                            <!-- Basic Information -->
                            <div class="col-12">
                                <h6 class="text-muted mb-3">Basic Information</h6>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Full Name *</label>
                                <input type="text" class="form-control" name="full_name" id="edit_full_name" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Username *</label>
                                <input type="text" class="form-control" name="username" id="edit_username" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Email <span class="email-required-indicator">*</span></label>
                                <input type="email" class="form-control" name="email" id="edit_email" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Phone</label>
                                <input type="tel" class="form-control" name="phone" id="edit_phone">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">New Password <small class="text-muted">(leave blank to keep current)</small></label>
                                <input type="password" class="form-control" name="new_password" id="edit_new_password" placeholder="Enter new password or leave blank">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Role *</label>
                                <select class="form-select" name="role" id="edit_role" required>
                                    <?php foreach (getRolesUserCanCreate($user_role) as $role): ?>
                                    <option value="<?= $role ?>"><?= ucfirst(str_replace('_', ' ', $role)) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <div class="form-check mt-4">
                                    <input class="form-check-input" type="checkbox" name="is_active" id="edit_is_active">
                                    <label class="form-check-label" for="edit_is_active">
                                        Active User
                                    </label>
                                </div>
                            </div>

                            <!-- Profile Details -->
                            <div class="col-12 mt-4">
                                <h6 class="text-muted mb-3">Profile Details</h6>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Address</label>
                                <textarea class="form-control" name="address" id="edit_address" rows="2"></textarea>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Emergency Contact</label>
                                <input type="text" class="form-control" name="emergency_contact" id="edit_emergency_contact">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Emergency Phone</label>
                                <input type="tel" class="form-control" name="emergency_phone" id="edit_emergency_phone">
                            </div>
                            <div class="col-12">
                                <label class="form-label">Avatar</label>
                                <div id="edit_current_avatar" class="mb-2"></div>
                                <input type="file" class="form-control" name="avatar" id="edit_avatar" accept="image/*">
                                <small class="form-text text-muted">Allowed: JPG, PNG, GIF, WebP (max 5MB)</small>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Notes</label>
                                <textarea class="form-control" name="notes" id="edit_notes" rows="2" placeholder="Internal notes about this user..."></textarea>
                            </div>
                            
                            <!-- Assigned Events -->
                            <div class="col-12 mt-4">
                                <h6 class="text-muted mb-3">Assigned Events</h6>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Events</label>
                                <select class="form-select" name="assigned_events[]" id="edit_assigned_events" multiple size="6">
                                    <option value="" disabled>Loading events...</option>
                                </select>
                                <small class="text-muted">Hold Ctrl (Cmd on Mac) to select multiple events</small>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i data-lucide="save" style="width:16px;height:16px;" class="me-2"></i>Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View User Details Modal -->
    <div class="modal fade" id="viewUserModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="user" style="width:18px;height:18px;" class="me-2"></i>User Details
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="userDetailsContent">
                        <div class="text-center">
                            <div class="spinner-border text-primary"></div>
                            <p class="mt-2">Loading user details...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-warning" onclick="changeUserPassword()">
                        <i data-lucide="key" style="width:16px;height:16px;" class="me-2"></i>Change Password
                    </button>
                    <button type="button" class="btn btn-primary" onclick="editCurrentUser()">
                        <i data-lucide="edit" style="width:16px;height:16px;" class="me-2"></i>Edit User
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Group Members Modal -->
    <div class="modal fade" id="groupMembersModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="users" style="width:18px;height:18px;" class="me-2"></i>Group Members
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="groupMembersContent">
                        <div class="text-center">
                            <div class="spinner-border text-primary"></div>
                            <p class="mt-2">Loading group members...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" onclick="showAddMemberForm()">
                        <i data-lucide="user-plus" style="width:16px;height:16px;" class="me-2"></i>Add Member
                    </button>
                </div>
            </div>
        </div>
    </div>

    <    <!-- Edit Group Modal -->
    <div class="modal fade" id="editGroupModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="edit" style="width:18px;height:18px;" class="me-2"></i>Edit Group
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" id="editGroupForm">
                    <input type="hidden" name="action" value="edit_group">
                    <input type="hidden" name="group_id" id="edit_group_id">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Group Name *</label>
                            <input type="text" class="form-control" name="group_name" id="edit_group_name" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" name="group_description" id="edit_group_description" rows="3"></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Color</label>
                            <input type="color" class="form-control form-control-color" name="group_color" id="edit_group_color">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i data-lucide="save" style="width:16px;height:16px;" class="me-2"></i>Update Group
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Add Member to Group Modal -->
    <div class="modal fade" id="addMemberModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i data-lucide="user-plus" style="width:18px;height:18px;" class="me-2"></i>Add Member to Group
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" id="addMemberForm">
                    <input type="hidden" name="action" value="add_member_to_group">
                    <input type="hidden" name="group_id" id="add_member_group_id">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Select User *</label>
                            <select class="form-select" name="user_id" id="add_member_user_id" required>
                                <option value="">Choose user...</option>
                                <?php foreach ($users as $user): ?>
                                <option value="<?= $user['id'] ?>"><?= htmlspecialchars($user['full_name'] ?: $user['username']) ?> (@<?= htmlspecialchars($user['username']) ?>)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Role in Group</label>
                            <select class="form-select" name="role_in_group">
                                <option value="member">Member</option>
                                <option value="admin">Admin</option>
                                <option value="moderator">Moderator</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i data-lucide="plus" style="width:16px;height:16px;" class="me-2"></i>Add Member
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Confirmation Modal -->
    <div class="modal fade" id="confirmActionModal" tabindex="-1" aria-labelledby="confirmActionModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="confirmActionModalLabel">Confirm Action</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="confirmActionMessage">
                    Are you sure you want to proceed?
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirmActionButton">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Change Password Modal -->
    <div class="modal fade" id="changePasswordModal" tabindex="-1" aria-labelledby="changePasswordModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="changePasswordModalLabel">
                        <i data-lucide="key" style="width:18px;height:18px;" class="me-2"></i>Change User Password
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="changePasswordForm">
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i data-lucide="info" style="width:16px;height:16px;" class="me-2"></i>
                            The password will be automatically encrypted using bcrypt hashing.
                        </div>
                        <div class="mb-3">
                            <label for="new_user_password" class="form-label">New Password *</label>
                            <input type="password" class="form-control" id="new_user_password" name="new_user_password" required minlength="4" placeholder="Enter new password">
                            <div class="form-text">Minimum 4 characters</div>
                        </div>
                        <div class="mb-3">
                            <label for="confirm_user_password" class="form-label">Confirm Password *</label>
                            <input type="password" class="form-control" id="confirm_user_password" name="confirm_user_password" required minlength="4" placeholder="Confirm new password">
                        </div>
                        <div id="password_match_error" class="text-danger small" style="display:none;">
                            <i data-lucide="alert-circle" style="width:14px;height:14px;" class="me-1"></i>Passwords do not match
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-warning">
                            <i data-lucide="save" style="width:16px;height:16px;" class="me-2"></i>Change Password
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        let currentUserId = null;
        let currentGroupId = null;
        
        // Confirmation modal helper
        function showConfirmation(message, onConfirm, confirmButtonText = 'Confirm', confirmButtonClass = 'btn-danger') {
            const modal = new bootstrap.Modal(document.getElementById('confirmActionModal'));
            const messageEl = document.getElementById('confirmActionMessage');
            const confirmBtn = document.getElementById('confirmActionButton');
            
            messageEl.textContent = message;
            confirmBtn.textContent = confirmButtonText;
            confirmBtn.className = 'btn ' + confirmButtonClass;
            
            // Remove old event listeners by cloning
            const newConfirmBtn = confirmBtn.cloneNode(true);
            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
            
            newConfirmBtn.addEventListener('click', function() {
                modal.hide();
                onConfirm();
            });
            
            modal.show();
        }
        
        function editUser(userId) {
            currentUserId = userId;
            
            // Fetch user data
            fetch(`api/user_management_api.php?action=get_user&id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const user = data.user;
                        
                        // Populate form fields
                        document.getElementById('edit_user_id').value = user.id;
                        document.getElementById('edit_full_name').value = user.full_name || '';
                        document.getElementById('edit_username').value = user.username;
                        document.getElementById('edit_email').value = user.email || '';
                        document.getElementById('edit_phone').value = user.phone || '';
                        document.getElementById('edit_role').value = user.role;
                        document.getElementById('edit_is_active').checked = user.is_active == 1;
                        document.getElementById('edit_address').value = user.address || '';
                        document.getElementById('edit_emergency_contact').value = user.emergency_contact || '';
                        document.getElementById('edit_emergency_phone').value = user.emergency_phone || '';
                        document.getElementById('edit_notes').value = user.notes || '';
                        
                        // Display current avatar if exists
                        const avatarDisplay = document.getElementById('edit_current_avatar');
                        if (user.avatar) {
                            avatarDisplay.innerHTML = '<div class="d-flex align-items-center gap-2"><img src="' + user.avatar + '" alt="Current Avatar" class="rounded" style="width: 60px; height: 60px; object-fit: cover;"><small class="text-muted">Current avatar (upload new file to replace)</small></div>';
                        } else {
                            avatarDisplay.innerHTML = '<small class="text-muted">No avatar uploaded</small>';
                        }
                        
                        // Ensure events are populated and select assigned events
                        const eventSelect = document.getElementById('edit_assigned_events');
                        if (eventSelect) {
                            // Repopulate dropdown in case it wasn't populated yet
                            if (allEvents.length > 0) {
                                populateEventDropdowns();
                            }
                            
                            // Select assigned events if any
                            if (user.assigned_event_ids) {
                                const assignedIds = user.assigned_event_ids.split(',');
                                // Small delay to ensure options are rendered
                                setTimeout(() => {
                                    Array.from(eventSelect.options).forEach(option => {
                                        option.selected = assignedIds.includes(option.value);
                                    });
                                }, 50);
                            }
                        }
                        
                        // Show modal
                        const modal = new bootstrap.Modal(document.getElementById('editUserModal'));
                        modal.show();
                    } else {
                        showNotification('Error loading user data: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    showNotification('Network error: ' + error.message, 'error');
                });
        }
        
        function viewUser(userId) {
            currentUserId = userId;
            
            // Show modal first
            const modal = new bootstrap.Modal(document.getElementById('viewUserModal'));
            modal.show();
            
            // Fetch user data
            fetch(`api/user_management_api.php?action=get_user&id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const user = data.user;
                        
                        const content = `
                            <div class="row g-4">
                                <div class="col-md-4 text-center">
                                    ${user.avatar ? 
                                        '<img src="' + user.avatar + '" alt="' + user.full_name + '" class="rounded-circle mx-auto mb-3" style="width: 100px; height: 100px; object-fit: cover;">' :
                                        '<div class="bg-primary text-white rounded-circle d-flex align-items-center justify-content-center mx-auto mb-3" style="width: 100px; height: 100px; font-size: 2rem;"><strong>' + (user.full_name || user.username).substring(0, 2).toUpperCase() + '</strong></div>'
                                    }
                                    <h5>${user.full_name || user.username}</h5>
                                    <span class="badge bg-${getRoleColor(user.role)} mb-2">${user.role.replace('_', ' ').toUpperCase()}</span>
                                    <div>
                                        <i data-lucide="circle" class="${user.is_active == 1 ? 'text-success' : 'text-danger'}" style="width:10px;height:10px;"></i>
                                        ${user.is_active == 1 ? 'Active' : 'Inactive'}
                                    </div>
                                </div>
                                <div class="col-md-8">
                                    <table class="table table-borderless">
                                        <tr><th>Username:</th><td>@${user.username}</td></tr>
                                        <tr><th>Email:</th><td>${user.email || 'Not provided'}</td></tr>
                                        <tr><th>Phone:</th><td>${user.phone || 'Not provided'}</td></tr>
                                        ${user.address ? '<tr><th>Address:</th><td>' + user.address + '</td></tr>' : ''}
                                        ${user.emergency_contact ? '<tr><th>Emergency Contact:</th><td>' + user.emergency_contact + '</td></tr>' : ''}
                                        ${user.emergency_phone ? '<tr><th>Emergency Phone:</th><td>' + user.emergency_phone + '</td></tr>' : ''}
                                        <tr><th>Groups:</th><td>${user.group_names ? user.group_names.split(',').map(g => '<span class="badge bg-secondary me-1">' + g + '</span>').join('') : 'No groups'}</td></tr>
                                        ${user.assigned_events && user.assigned_events.length > 0 ? '<tr><th>Assigned Events:</th><td>' + user.assigned_events.map(e => '<span class="badge bg-info me-1">' + e.name + '</span>').join('') + '</td></tr>' : ''}
                                        <tr><th>Created:</th><td>${new Date(user.created_at).toLocaleDateString()}</td></tr>
                                        <tr><th>Last Login:</th><td>${user.last_login ? new Date(user.last_login).toLocaleDateString() : 'Never'}</td></tr>
                                    </table>
                                    ${user.notes ? '<div class="mt-3"><strong>Notes:</strong><p class="text-muted">' + user.notes + '</p></div>' : ''}
                                </div>
                            </div>
                        `;
                        
                        document.getElementById('userDetailsContent').innerHTML = content;
                    } else {
                        document.getElementById('userDetailsContent').innerHTML = 
                            `<div class="alert alert-danger">Error loading user data: ${data.message}</div>`;
                    }
                })
                .catch(error => {
                    document.getElementById('userDetailsContent').innerHTML = 
                        `<div class="alert alert-danger">Network error: ${error.message}</div>`;
                });
        }
        
        function deleteUser(userId) {
            showConfirmation(
                'Are you sure you want to delete this user? This action cannot be undone.',
                function() {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = '<input type="hidden" name="action" value="delete_user">' +
                                   '<input type="hidden" name="user_id" value="' + userId + '">';
                    document.body.appendChild(form);
                    form.submit();
                },
                'Delete User',
                'btn-danger'
            );
        }
        
        function toggleUserStatus(userId, newStatus) {
            const statusText = newStatus ? 'activate' : 'deactivate';
            showConfirmation(
                'Are you sure you want to ' + statusText + ' this user?',
                function() {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = '<input type="hidden" name="action" value="edit_user">' +
                                   '<input type="hidden" name="user_id" value="' + userId + '">' +
                                   '<input type="hidden" name="toggle_status" value="1">' +
                                   (newStatus ? '<input type="hidden" name="is_active" value="1">' : '');
                    document.body.appendChild(form);
                    form.submit();
                },
                statusText.charAt(0).toUpperCase() + statusText.slice(1),
                newStatus ? 'btn-success' : 'btn-warning'
            );
        }
        
        function editGroup(groupId) {
            // Fetch group data and populate the edit form
            fetch(`api/user_management_api.php?action=get_group&id=${groupId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const group = data.group;
                        
                        // Populate form fields
                        document.getElementById('edit_group_id').value = group.id;
                        document.getElementById('edit_group_name').value = group.name;
                        document.getElementById('edit_group_description').value = group.description || '';
                        document.getElementById('edit_group_color').value = group.color || '#007bff';
                        
                        // Show modal
                        const modal = new bootstrap.Modal(document.getElementById('editGroupModal'));
                        modal.show();
                    } else {
                        // Fallback if API doesn't exist - use inline data
                        const groupData = <?= json_encode($groups) ?>;
                        const group = groupData.find(g => g.id == groupId);
                        
                        if (group) {
                            document.getElementById('edit_group_id').value = group.id;
                            document.getElementById('edit_group_name').value = group.name;
                            document.getElementById('edit_group_description').value = group.description || '';
                            document.getElementById('edit_group_color').value = group.color || '#007bff';
                            
                            const modal = new bootstrap.Modal(document.getElementById('editGroupModal'));
                            modal.show();
                        }
                    }
                })
                .catch(error => {
                    // Fallback to inline data
                    const groupData = <?= json_encode($groups) ?>;
                    const group = groupData.find(g => g.id == groupId);
                    
                    if (group) {
                        document.getElementById('edit_group_id').value = group.id;
                        document.getElementById('edit_group_name').value = group.name;
                        document.getElementById('edit_group_description').value = group.description || '';
                        document.getElementById('edit_group_color').value = group.color || '#007bff';
                        
                        const modal = new bootstrap.Modal(document.getElementById('editGroupModal'));
                        modal.show();
                    }
                });
        }
        
        function viewGroupMembers(groupId) {
            currentGroupId = groupId;
            
            // Show modal first
            const modal = new bootstrap.Modal(document.getElementById('groupMembersModal'));
            modal.show();
            
            // Fetch group members
            fetch(`api/user_management_api.php?action=get_group_members&group_id=${groupId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const members = data.members;
                        
                        let content = '<div class="table-responsive"><table class="table table-hover"><thead><tr><th>User</th><th>Role in Group</th><th>Joined</th><th>Actions</th></tr></thead><tbody>';
                        
                        members.forEach(member => {
                            content += `
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="bg-primary text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 32px; height: 32px; font-size: 0.8rem;">
                                                <strong>${(member.full_name || member.username).substring(0, 2).toUpperCase()}</strong>
                                            </div>
                                            <div>
                                                <h6 class="mb-0">${member.full_name || member.username}</h6>
                                                <small class="text-muted">@${member.username}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td><span class="badge bg-info">${member.role_in_group}</span></td>
                                    <td><small>${new Date(member.joined_at).toLocaleDateString()}</small></td>
                                    <td>
                                        <button class="btn btn-sm btn-outline-danger" onclick="removeFromGroup(${member.id}, ${groupId})">
                                            <i data-lucide="x" style="width:14px;height:14px;"></i>
                                        </button>
                                    </td>
                                </tr>
                            `;
                        });
                        
                        content += '</tbody></table></div>';
                        
                        if (members.length === 0) {
                            content = '<div class="text-center text-muted py-4"><i data-lucide="users" class="opacity-25" style="width:56px;height:56px;"></i><h6 class="mt-3">No members in this group</h6></div>';
                        }
                        
                        document.getElementById('groupMembersContent').innerHTML = content;
                    } else {
                        document.getElementById('groupMembersContent').innerHTML = 
                            `<div class="alert alert-danger">Error loading group members: ${data.message}</div>`;
                    }
                })
                .catch(error => {
                    document.getElementById('groupMembersContent').innerHTML = 
                        `<div class="alert alert-danger">Network error: ${error.message}</div>`;
                });
        }
        
        function editCurrentUser() {
            if (currentUserId) {
                const modal = bootstrap.Modal.getInstance(document.getElementById('viewUserModal'));
                modal.hide();
                editUser(currentUserId);
            }
        }
        
        function changeUserPassword() {
            if (!currentUserId) return;
            
            // Reset form
            document.getElementById('changePasswordForm').reset();
            document.getElementById('password_match_error').style.display = 'none';
            
            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('changePasswordModal'));
            modal.show();
        }
        
        function showAddMemberForm() {
            if (currentGroupId) {
                document.getElementById('add_member_group_id').value = currentGroupId;
                
                // Reset the form
                document.getElementById('add_member_user_id').value = '';
                document.querySelector('select[name="role_in_group"]').value = 'member';
                
                // Close group members modal and show add member modal
                const groupModal = bootstrap.Modal.getInstance(document.getElementById('groupMembersModal'));
                if (groupModal) {
                    groupModal.hide();
                }
                
                setTimeout(() => {
                    const addModal = new bootstrap.Modal(document.getElementById('addMemberModal'));
                    addModal.show();
                }, 300);
            }
        }
        
        function removeFromGroup(userId, groupId) {
            showConfirmation(
                'Remove this user from the group?',
                function() {
                    const formData = new FormData();
                    formData.append('action', 'manage_group_membership');
                    formData.append('user_id', userId);
                    formData.append('group_id', groupId);
                    formData.append('membership_action', 'remove');
                    
                    fetch('user_management.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(function() {
                        viewGroupMembers(groupId);
                        showNotification('User removed from group', 'success');
                    })
                    .catch(function(error) {
                        showNotification('Error removing user: ' + error.message, 'error');
                    });
                },
                'Remove User',
                'btn-warning'
            );
        }
        
        function getRoleColor(role) {
            const colors = {
                'super_admin': 'danger',
                'admin': 'warning', 
                'head_judge': 'success',
                'judge': 'primary',
                'office': 'info',
                'media': 'secondary',
                'participant': 'light'
            };
            return colors[role] || 'light';
        }
        
        function showNotification(message, type) {
            const notification = document.createElement('div');
            notification.className = `alert alert-${type === 'success' ? 'success' : 'danger'} alert-dismissible fade show position-fixed`;
            notification.style.top = '20px';
            notification.style.right = '20px';
            notification.style.zIndex = '9999';
            notification.style.minWidth = '300px';
            
            notification.innerHTML = `
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.remove();
                }
            }, 5000);
        }
        
        // Generate username from full name
        document.querySelector('input[name="full_name"]').addEventListener('input', function(e) {
            const fullName = e.target.value;
            const username = fullName.toLowerCase().replace(/[^a-z0-9]/g, '_').replace(/_+/g, '_').replace(/^_|_$/g, '');
            document.querySelector('input[name="username"]').value = username;
        });
        
        // Handle email requirement based on role selection
        function updateEmailRequirement(roleSelect, emailInput, indicator) {
            const selectedRole = roleSelect.value;
            const isJudge = selectedRole === 'judge' || selectedRole === 'head_judge';
            
            if (isJudge) {
                emailInput.removeAttribute('required');
                if (indicator) indicator.style.display = 'none';
            } else {
                emailInput.setAttribute('required', 'required');
                if (indicator) indicator.style.display = 'inline';
            }
        }
        
        // Generate 6-character password for judges (letters and digits only)
        function generateJudgePassword() {
            const chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
            let password = '';
            for (let i = 0; i < 6; i++) {
                password += chars.charAt(Math.floor(Math.random() * chars.length));
            }
            const passwordInput = document.getElementById('add_password');
            if (passwordInput) {
                passwordInput.value = password;
                passwordInput.type = 'text'; // Show the password
            }
            if (window.lucide) lucide.createIcons();
        }
        
        // Update password field based on role selection
        function updatePasswordField(roleSelect) {
            const selectedRole = roleSelect.value;
            const isJudge = selectedRole === 'judge' || selectedRole === 'head_judge';
            const generateBtn = document.getElementById('generate_password_btn');
            const passwordInput = document.getElementById('add_password');
            const passwordHint = document.getElementById('password_hint');
            
            if (isJudge) {
                if (generateBtn) generateBtn.style.display = 'block';
                if (passwordHint) passwordHint.style.display = 'block';
                if (passwordInput) {
                    passwordInput.type = 'text';
                    passwordInput.setAttribute('maxlength', '6');
                    passwordInput.setAttribute('placeholder', 'Click generate or enter 6 chars');
                }
                // Auto-generate password when role is selected
                if (passwordInput && !passwordInput.value) {
                    generateJudgePassword();
                }
            } else {
                if (generateBtn) generateBtn.style.display = 'none';
                if (passwordHint) passwordHint.style.display = 'none';
                if (passwordInput) {
                    passwordInput.type = 'password';
                    passwordInput.removeAttribute('maxlength');
                    passwordInput.setAttribute('placeholder', '');
                    if (passwordInput.value.length === 6) {
                        passwordInput.value = ''; // Clear auto-generated password
                    }
                }
            }
            if (window.lucide) lucide.createIcons();
        }
        
        // Add role change listeners for add user form
        const addRoleSelect = document.querySelector('#addUserModal select[name="role"]');
        const addEmailInput = document.getElementById('add_email');
        const addEmailIndicator = document.querySelector('#addUserModal .email-required-indicator');
        
        if (addRoleSelect && addEmailInput) {
            addRoleSelect.addEventListener('change', function() {
                updateEmailRequirement(this, addEmailInput, addEmailIndicator);
                updatePasswordField(this);
            });
            // Set initial state
            updateEmailRequirement(addRoleSelect, addEmailInput, addEmailIndicator);
            updatePasswordField(addRoleSelect);
        }
        
        // Add role change listeners for edit user form
        const editRoleSelect = document.getElementById('edit_role');
        const editEmailInput = document.getElementById('edit_email');
        const editEmailIndicator = document.querySelector('#editUserModal .email-required-indicator');
        
        if (editRoleSelect && editEmailInput) {
            editRoleSelect.addEventListener('change', function() {
                updateEmailRequirement(this, editEmailInput, editEmailIndicator);
            });
        }
        
        // Handle add member form submission
        document.getElementById('addMemberForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            // Add some debugging
            console.log('Form data:', Object.fromEntries(formData));
            console.log('Current group ID:', currentGroupId);
            
            fetch('user_management.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(data => {
                console.log('Server response:', data);
                
                // Check if response contains success message
                if (data.includes('success:')) {
                    showNotification('Member added successfully', 'success');
                    // Close modal and refresh group members
                    const modal = bootstrap.Modal.getInstance(document.getElementById('addMemberModal'));
                    modal.hide();
                    
                    // Force page reload to ensure we see the changes
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                    
                    // Also try to refresh group members if modal is open
                    if (currentGroupId) {
                        setTimeout(() => {
                            viewGroupMembers(currentGroupId);
                        }, 500);
                    }
                } else if (data.includes('error:')) {
                    const errorMsg = data.split('error: ')[1] || 'Unknown error';
                    showNotification('Error: ' + errorMsg, 'error');
                } else {
                    console.log('Unexpected response format:', data);
                    showNotification('Unexpected response from server', 'error');
                }
            })
            .catch(error => {
                console.error('Network error:', error);
                showNotification('Network error: ' + error.message, 'error');
            });
        });
        
        // Clean URL after showing message
        if (window.location.search.includes('msg=')) {
            const cleanUrl = window.location.pathname;
            window.history.replaceState({}, document.title, cleanUrl);
        }
        
        // Debug form submission
        // Handle password change form submission
        const changePasswordForm = document.getElementById('changePasswordForm');
        if (changePasswordForm) {
            changePasswordForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const newPassword = document.getElementById('new_user_password').value;
                const confirmPassword = document.getElementById('confirm_user_password').value;
                
                // Validate passwords match
                if (newPassword !== confirmPassword) {
                    document.getElementById('password_match_error').style.display = 'block';
                    if (window.lucide) lucide.createIcons();
                    return;
                }
                
                document.getElementById('password_match_error').style.display = 'none';
                
                // Get current user data first
                fetch('api/user_management_api.php?action=get_user&id=' + currentUserId)
                    .then(function(response) { return response.json(); })
                    .then(function(data) {
                        if (data.success) {
                            const user = data.user;
                            
                            // Create form with all required fields
                            const form = document.createElement('form');
                            form.method = 'POST';
                            form.innerHTML = '<input type="hidden" name="action" value="edit_user">' +
                                           '<input type="hidden" name="user_id" value="' + currentUserId + '">' +
                                           '<input type="hidden" name="username" value="' + (user.username || '') + '">' +
                                           '<input type="hidden" name="role" value="' + (user.role || '') + '">' +
                                           '<input type="hidden" name="full_name" value="' + (user.full_name || '') + '">' +
                                           '<input type="hidden" name="email" value="' + (user.email || '') + '">' +
                                           '<input type="hidden" name="phone" value="' + (user.phone || '') + '">' +
                                           '<input type="hidden" name="new_password" value="' + newPassword + '">' +
                                           (user.is_active == 1 ? '<input type="hidden" name="is_active" value="1">' : '');
                            
                            document.body.appendChild(form);
                            form.submit();
                        } else {
                            showNotification('Error loading user data: ' + data.message, 'error');
                        }
                    })
                    .catch(function(error) {
                        showNotification('Error: ' + error.message, 'error');
                    });
            });
        }

        // Initialize / re-initialize Lucide icons after dynamic content updates
        
        // Load events for assignment dropdowns
        let allEvents = [];
        function loadEvents() {
            console.log('Loading events from API...');
            fetch('api/user_management_api.php?action=get_events')
                .then(response => {
                    console.log('Events API response status:', response.status);
                    if (!response.ok) {
                        throw new Error('HTTP error! status: ' + response.status);
                    }
                    return response.json();
                })
                .then(data => {
                    console.log('Events API data:', data);
                    if (data.success) {
                        allEvents = data.events;
                        console.log('Loaded', allEvents.length, 'events');
                        populateEventDropdowns();
                    } else {
                        console.error('Events API returned error:', data.message);
                        populateEventDropdowns(); // Populate with empty to show error message
                    }
                })
                .catch(error => {
                    console.error('Error loading events:', error);
                    populateEventDropdowns(); // Populate with empty to show error message
                });
        }
        
        function populateEventDropdowns() {
            const addSelect = document.getElementById('add_assigned_events');
            const editSelect = document.getElementById('edit_assigned_events');
            
            console.log('Populating event dropdowns with', allEvents.length, 'events');
            
            let options = '';
            if (allEvents.length === 0) {
                options = '<option value="" disabled>No events available</option>';
            } else {
                options = allEvents.map(event => {
                    const date = event.event_date ? new Date(event.event_date).toLocaleDateString() : 'No date';
                    const status = event.status ? ' [' + event.status + ']' : '';
                    return '<option value="' + event.id + '">' + event.name + ' (' + date + ')' + status + '</option>';
                }).join('');
            }
            
            if (addSelect) {
                addSelect.innerHTML = options;
                console.log('Populated add_assigned_events with', addSelect.options.length, 'options');
            }
            if (editSelect) {
                editSelect.innerHTML = options;
                console.log('Populated edit_assigned_events with', editSelect.options.length, 'options');
            }
        }
        
        document.addEventListener('DOMContentLoaded', () => {
            if (window.lucide) lucide.createIcons();
            loadEvents();
        });

        // Monkey patch functions that inject dynamic HTML to refresh icons
        const _viewUser = viewUser;
        viewUser = function(id){ _viewUser(id); setTimeout(()=>{ if(window.lucide) lucide.createIcons(); }, 50); };
        const _viewGroupMembers = viewGroupMembers;
        viewGroupMembers = function(id){ _viewGroupMembers(id); setTimeout(()=>{ if(window.lucide) lucide.createIcons(); }, 100); };
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>

<?php
function getRoleColor($role) {
    switch ($role) {
        case 'super_admin': return 'danger';
        case 'admin': return 'warning';
        case 'head_judge': return 'success';
        case 'judge': return 'primary';
        case 'office': return 'info';
        case 'media': return 'secondary';
        case 'participant': return 'light';
        default: return 'light';
    }
}
?>
