<?php
/**
 * Test HTML Table Generation
 * 
 * This script tests the generation of HTML summary tables for publishing
 */

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/publish_generators.php';

// Set event ID to test
$event_id = 3;

echo "<h1>Testing HTML Table Generation for Event $event_id</h1>\n\n";

// Create temporary directory for testing
$temp_dir = sys_get_temp_dir() . '/stylescore_html_test_' . time();
if (!file_exists($temp_dir)) {
    mkdir($temp_dir, 0755, true);
    echo "<p>✓ Created temp directory: $temp_dir</p>\n";
}

try {
    // Test HTML table generation
    echo "<h2>Generating HTML Tables...</h2>\n";
    $html_files = generateSummaryTableHTMLs($event_id, $temp_dir, $pdo);
    
    echo "<p><strong>Generated " . count($html_files) . " HTML table files:</strong></p>\n";
    echo "<ul>\n";
    
    foreach ($html_files as $file_info) {
        $local_path = $file_info['local'];
        $remote_path = $file_info['remote'];
        $file_size = file_exists($local_path) ? filesize($local_path) : 0;
        
        echo "<li>\n";
        echo "  <strong>Remote:</strong> $remote_path<br>\n";
        echo "  <strong>Local:</strong> $local_path<br>\n";
        echo "  <strong>Size:</strong> " . number_format($file_size) . " bytes<br>\n";
        
        if ($file_size > 0) {
            echo "  <strong>Status:</strong> <span style='color: green;'>✓ Generated successfully</span><br>\n";
            
            // Show preview link
            $basename = basename($local_path);
            echo "  <a href='file:///$local_path' target='_blank'>View File</a> | ";
            echo "  <a href='#' onclick='showPreview(\"$basename\"); return false;'>Show Preview</a>\n";
        } else {
            echo "  <strong>Status:</strong> <span style='color: red;'>✗ File is empty or failed</span>\n";
        }
        
        echo "</li>\n";
    }
    
    echo "</ul>\n";
    
    // Get configurations to show what was generated
    echo "<h2>Summary Table Configurations</h2>\n";
    $configs_stmt = $pdo->prepare("
        SELECT id, name, configuration, status, event_id 
        FROM result_configurations 
        WHERE event_id = ? AND view_type = 'summary_table' AND status = 'public'
        ORDER BY name
    ");
    $configs_stmt->execute([$event_id]);
    $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<p>Found " . count($configs) . " public summary table configurations:</p>\n";
    echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>\n";
    echo "<tr><th>Config ID</th><th>Name</th><th>HTML File</th><th>Test URL</th></tr>\n";
    
    foreach ($configs as $config) {
        $filename = 'summary_table_' . $config['id'] . '.html';
        $test_url = buildSummaryTableHTMLUrl($config);
        
        echo "<tr>\n";
        echo "  <td>" . $config['id'] . "</td>\n";
        echo "  <td>" . htmlspecialchars($config['name']) . "</td>\n";
        echo "  <td><code>tables/$filename</code></td>\n";
        echo "  <td><a href='$test_url' target='_blank'>Test API</a></td>\n";
        echo "</tr>\n";
    }
    
    echo "</table>\n";
    
    echo "<h2>Usage Example</h2>\n";
    echo "<p>After publishing to the public server, access HTML tables at:</p>\n";
    echo "<pre>";
    echo "https://your-public-server.com/tables/summary_table_74.html\n";
    echo "https://your-public-server.com/tables/summary_table_75.html\n";
    echo "</pre>\n";
    
    echo "<p><strong>Temp directory:</strong> $temp_dir</p>\n";
    echo "<p style='color: green;'><strong>✓ HTML table generation test completed successfully!</strong></p>\n";
    
} catch (Exception $e) {
    echo "<p style='color: red;'><strong>✗ Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>\n";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>\n";
}

?>

<script>
function showPreview(filename) {
    alert('Preview functionality would show: ' + filename);
    // In a real implementation, this could load and display the HTML content
}
</script>

<style>
body {
    font-family: Arial, sans-serif;
    margin: 20px;
    background: #f5f5f5;
}
h1 {
    color: #333;
    border-bottom: 2px solid #0066cc;
    padding-bottom: 10px;
}
h2 {
    color: #0066cc;
    margin-top: 30px;
}
ul {
    background: white;
    padding: 20px;
    border-radius: 5px;
}
li {
    margin-bottom: 15px;
    padding: 10px;
    background: #f9f9f9;
    border-left: 3px solid #0066cc;
}
table {
    background: white;
    width: 100%;
}
th {
    background: #0066cc;
    color: white;
    padding: 10px;
}
td {
    padding: 8px;
}
pre {
    background: #f0f0f0;
    padding: 15px;
    border-radius: 5px;
    overflow-x: auto;
}
</style>
