-- Add event_metadata JSON column to events table
-- This stores flexible metadata with field ordering and grouping
-- Created: 2025-11-17 (Updated with field-based structure)

ALTER TABLE events 
ADD COLUMN event_metadata JSON DEFAULT NULL
COMMENT 'Flexible metadata with field ordering: {metadata_fields: [{group, field_key, field_label, field_value, field_order, is_system}], judges: []}';

-- Example JSON structure with flexible field ordering:
-- {
--   "metadata_fields": [
--     {
--       "group": "Jury Information",
--       "field_key": "technical_delegate_fis",
--       "field_label": "Technical Delegate FIS",
--       "field_value": "John Doe",
--       "field_order": 1,
--       "is_system": true
--     },
--     {
--       "group": "Jury Information", 
--       "field_key": "chief_observer",
--       "field_label": "Chief Observer",
--       "field_value": "Jane Smith",
--       "field_order": 2.5,
--       "is_system": false
--     },
--     {
--       "group": "Jury Information",
--       "field_key": "chief_of_competition",
--       "field_label": "Chief of Competition",
--       "field_value": "Bob Johnson",
--       "field_order": 3,
--       "is_system": true
--     },
--     {
--       "group": "Technical Data",
--       "field_key": "run_length",
--       "field_label": "Run Length",
--       "field_value": "120m",
--       "field_order": 1,
--       "is_system": true
--     },
--     {
--       "group": "Safety Information",
--       "field_key": "medical_officer",
--       "field_label": "Medical Officer",
--       "field_value": "Dr. Sarah Lee",
--       "field_order": 1,
--       "is_system": false
--     }
--   ],
--   "judges": [
--     {"position": 1, "name": "Judge Name", "license": "FIS123"}
--   ]
-- }

-- Benefits of this structure:
-- 1. Add custom fields to any group with precise ordering (use 2.5 to insert between 2 and 3)
-- 2. Create new custom groups dynamically (e.g., "Safety Information", "Sponsors")
-- 3. Field labels separate from keys for easy relabeling
-- 4. System fields protected (is_system: true), custom fields can be edited/removed
-- 5. No database migrations needed for new fields
-- 6. Perfect for results display - fields appear in exact order you specify

-- Example use cases:
-- - Insert "Chief Observer" between "Chief of Competition" (order: 2) and "Head Judge" (order: 3)
--   by setting field_order to 2.5
-- - Add custom group "Sponsors" with field "Title Sponsor" at order 1
-- - Reorder any field by changing its field_order value

-- Optional: Create index for better query performance
-- CREATE INDEX idx_event_metadata ON events((CAST(event_metadata AS CHAR(255)) COLLATE utf8mb4_bin));
