<?php
// API endpoint for resetting participant status to 'initial' for a specific heat
error_reporting(E_ALL);
ini_set('display_errors', 0); // Disable display_errors for JSON API

include '../includes/auth.php';
include '../includes/db.php';

header('Content-Type: application/json');

$response = ['success' => false, 'message' => ''];

try {
    // Check if user is authenticated and has proper permissions
    if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'head_judge'])) {
        throw new Exception('Unauthorized access');
    }
    
    $event_id = $_POST['event_id'] ?? null;
    $heat_number = $_POST['heat_number'] ?? null;
    
    if (!$event_id || !$heat_number) {
        throw new Exception('Event ID and Heat Number are required');
    }
    
    // Validate that the event exists
    $event_check = $pdo->prepare("SELECT id FROM events WHERE id = ?");
    $event_check->execute([$event_id]);
    if (!$event_check->fetch()) {
        throw new Exception('Event not found');
    }
    
    // Count participants before reset
    $count_stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM event_participants 
        WHERE event_id = ? AND heat_number = ?
    ");
    $count_stmt->execute([$event_id, $heat_number]);
    $count_data = $count_stmt->fetch(PDO::FETCH_ASSOC);
    $participant_count = $count_data['count'] ?? 0;
    
    if ($participant_count === 0) {
        throw new Exception('No participants found in this heat');
    }
    
    // Reset participant_status to 'initial' for all participants in this heat
    $reset_stmt = $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = 'initial' 
        WHERE event_id = ? AND heat_number = ?
    ");
    $reset_stmt->execute([$event_id, $heat_number]);
    
    $affected_rows = $reset_stmt->rowCount();
    
    // If no rows were affected but participants exist, something went wrong
    if ($affected_rows === 0) {
        // Check if all are already 'initial'
        $check_stmt = $pdo->prepare("
            SELECT COUNT(*) as already_initial 
            FROM event_participants 
            WHERE event_id = ? AND heat_number = ? AND participant_status = 'initial'
        ");
        $check_stmt->execute([$event_id, $heat_number]);
        $check_data = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($check_data['already_initial'] == $participant_count) {
            // All were already 'initial'
            $response['success'] = true;
            $response['message'] = "All {$participant_count} participants in Heat {$heat_number} were already in 'initial' status";
            $response['affected_rows'] = 0;
            $response['total_participants'] = $participant_count;
            $response['already_initial'] = true;
            echo json_encode($response);
            exit;
        }
    }
    
    // Log the action
    $log_stmt = $pdo->prepare("
        INSERT INTO activity_log (user_id, action, details, created_at) 
        VALUES (?, 'reset_participant_status', ?, NOW())
    ");
    $log_details = json_encode([
        'event_id' => $event_id,
        'heat_number' => $heat_number,
        'affected_rows' => $affected_rows,
        'total_participants' => $participant_count
    ]);
    
    try {
        $log_stmt->execute([$_SESSION['user_id'], $log_details]);
    } catch (Exception $e) {
        // Log table might not exist, ignore error
    }
    
    $response['success'] = true;
    $response['message'] = "Successfully reset status for {$affected_rows} participant" . ($affected_rows != 1 ? 's' : '') . " in Heat {$heat_number}";
    $response['affected_rows'] = $affected_rows;
    $response['total_participants'] = $participant_count;
    $response['event_id'] = $event_id;
    $response['heat_number'] = $heat_number;
    
} catch (Exception $e) {
    $response['message'] = $e->getMessage();
    $response['error_details'] = [
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ];
}

echo json_encode($response);
