<?php
// Publish Generators - Generate different file formats for publishing
require_once __DIR__ . '/../includes/db.php';

// Generate Static HTML Dashboard with Full API Compatibility
function generateStaticHTMLDashboard($event_id, $temp_dir, $pdo, $data_types = null) {
    error_log("Starting generateStaticHTMLDashboard for event $event_id in $temp_dir");
    $files = [];
    
    // Get event data
    $event_stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $event_stmt->execute([$event_id]);
    $event = $event_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        error_log("Event not found for ID: $event_id");
        throw new Exception('Event not found');
    }
    
    error_log("Found event: " . $event['name']);
    
    // Create the static HTML dashboard based on public_event_dashboard.php
    $dashboard_content = generateStaticDashboardHTML($event, $event_id);
    
    // Save the main dashboard HTML
    $dashboard_file = $temp_dir . '/index.html';
    error_log("Saving dashboard to: $dashboard_file");
    
    $bytes_written = file_put_contents($dashboard_file, $dashboard_content);
    if ($bytes_written === false) {
        error_log("Failed to write dashboard file");
        throw new Exception('Failed to write dashboard file');
    }
    
    error_log("Dashboard file written successfully, $bytes_written bytes");
    $files[] = ['local' => $dashboard_file, 'remote' => 'index.html'];
    
    // Generate all static API endpoints that the dashboard needs
    error_log("Generating static API endpoints...");
    $api_files = generateStaticAPIFiles($event_id, $temp_dir, $pdo);
    error_log("Generated " . count($api_files) . " API endpoint files");
    $files = array_merge($files, $api_files);
    
    // Generate HTML tables for summary tables
    error_log("Generating HTML tables...");
    $html_table_files = generateSummaryTableHTMLs($event_id, $temp_dir, $pdo);
    error_log("Generated " . count($html_table_files) . " HTML table files");
    $files = array_merge($files, $html_table_files);
    
    // Generate HTML tables for start lists
    error_log("Generating start list HTML tables...");
    $start_list_files = generateStartListHTMLs($event_id, $temp_dir, $pdo);
    error_log("Generated " . count($start_list_files) . " start list HTML files");
    $files = array_merge($files, $start_list_files);
    
    // Include participant photos only if requested
    if ($data_types === null || in_array('participant_photos', $data_types)) {
        error_log("Copying participant photos...");
        $photo_files = copyParticipantPhotos($event_id, $temp_dir, $pdo);
        error_log("Copied " . count($photo_files) . " photo files");
        $files = array_merge($files, $photo_files);
    } else {
        error_log("Skipping participant photos (not in data_types)");
    }
    
    // Include asset files (CSS, JS, logos, etc.)
    error_log("Copying asset files...");
    $asset_files = copyStaticAssetFiles($temp_dir);
    error_log("Copied " . count($asset_files) . " asset files");
    $files = array_merge($files, $asset_files);
    
    error_log("Total files generated: " . count($files));
    return $files;
}

// Generate HTML Dashboard (Legacy function)
function generateHTMLDashboard($event_id, $temp_dir, $pdo, $data_types = null) {
    error_log("Starting generateHTMLDashboard for event $event_id in $temp_dir");
    $files = [];
    
    // Get event data
    $event_stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $event_stmt->execute([$event_id]);
    $event = $event_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        error_log("Event not found for ID: $event_id");
        throw new Exception('Event not found');
    }
    
    error_log("Found event: " . $event['name']);
    
    // Copy the remote dashboard template (prefer new static variant if present)
    $static_template_path = dirname(__FILE__) . '/templates/remote_dashboard_static.html';
    $template_path = file_exists($static_template_path) ? $static_template_path : dirname(__FILE__) . '/templates/remote_dashboard.html';
    error_log("Using template at: $template_path");
    
    if (!file_exists($template_path)) {
        error_log("Remote dashboard template not found at: $template_path");
        throw new Exception('Remote dashboard template not found');
    }
    
    error_log("Template found, reading content...");
    $dashboard_content = file_get_contents($template_path);
    
    if ($dashboard_content === false) {
        error_log("Failed to read template content");
        throw new Exception('Failed to read template content');
    }
    
    error_log("Template content read successfully, length: " . strlen($dashboard_content));
    
    // Customize the template with event-specific data
    $dashboard_content = str_replace(
        '<span id="event-title">StyleScore Competition</span>',
        '<span id="event-title">' . htmlspecialchars($event['name']) . '</span>',
        $dashboard_content
    );
    
    $subtitle = ($event['location'] ? $event['location'] . ' • ' : '') . 
                ($event['sport_discipline'] ? $event['sport_discipline'] : 'Competition');
    $dashboard_content = str_replace(
        '<p class="lead mb-0" id="event-subtitle">Live Results Dashboard</p>',
        '<p class="lead mb-0" id="event-subtitle">' . htmlspecialchars($subtitle) . '</p>',
        $dashboard_content
    );
    
    // Save the main dashboard HTML
    $dashboard_file = $temp_dir . '/index.html';
    error_log("Saving dashboard to: $dashboard_file");
    
    $bytes_written = file_put_contents($dashboard_file, $dashboard_content);
    if ($bytes_written === false) {
        error_log("Failed to write dashboard file");
        throw new Exception('Failed to write dashboard file');
    }
    
    error_log("Dashboard file written successfully, $bytes_written bytes");
    $files[] = ['local' => $dashboard_file, 'remote' => 'index.html'];
    
    // Generate data JSON files that the dashboard will load
    error_log("Generating JSON data files...");
    $data_files = generateJSONData($event_id, $temp_dir, $pdo, true);
    error_log("Generated " . count($data_files) . " JSON data files");
    $files = array_merge($files, $data_files);
    
    // Include participant photos only if requested
    if ($data_types === null || in_array('participant_photos', $data_types)) {
        error_log("Copying participant photos...");
        $photo_files = copyParticipantPhotos($event_id, $temp_dir, $pdo);
        error_log("Copied " . count($photo_files) . " photo files");
        $files = array_merge($files, $photo_files);
    } else {
        error_log("Skipping participant photos (not in data_types)");
    }
    
    // Include asset files (CSS, logos, etc.)
    error_log("Copying asset files...");
    $asset_files = copyAssetFiles($temp_dir);
    error_log("Copied " . count($asset_files) . " asset files");
    $files = array_merge($files, $asset_files);
    
    error_log("Total files generated: " . count($files));
    return $files;
}

/**
 * Generate a set of static JSON files that mimic the responses of the dynamic
 * endpoints used by public_event_dashboard.php so the remote site can run fully
 * static without hitting PHP APIs.
 *
 * Files produced (all at publish root):
 *  - events_overview.json              (GET ?action=get_events_overview)
 *  - heats.json                       (GET ?action=get_event_heats&event_id=...)
 *  - configurations.json              (GET ?action=get_configurations&event_id=...)
 *  - active_heats.json                (GET ?action=get_active_heat_details&event_id=...)
 *  - heat_rows.json                   (GET ?action=get_heat_rows&event_id=...)
 *  - participants_by_bib.json         (GET ?action=get_participant_by_bib&event_id=&bib_number=...)
 *  - latest_updates.json              (GET ?action=get_event_latest_updates&event_id=...)
 *  - summary_table.json               (summary_table_api.php?event_id=&format=json...)
 *  - start_list.json                  (start_list_api.php?event_id=...)
 */
function generateStaticAPIData($event_id, $temp_dir, $pdo) {
    $files = [];
    // 1. events_overview.json (single event list – could be extended for multi-event later)
    $overview_stmt = $pdo->prepare("SELECT id, name, location, start_date, end_date FROM events WHERE id = ?");
    $overview_stmt->execute([$event_id]);
    $event = $overview_stmt->fetch(PDO::FETCH_ASSOC);
    $events_payload = ['success' => true, 'events' => $event ? [ $event ] : []];
    $file = $temp_dir . '/events_overview.json';
    file_put_contents($file, json_encode($events_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'events_overview.json'];

    // 2. heats.json – reuse getHeatsData structure
    try {
        $heats_data = getHeatsData($event_id, $pdo); // returns ['success'=>true,'heats'=>...]
    } catch (Exception $e) {
        $heats_data = ['success'=>false,'message'=>$e->getMessage(),'heats'=>[]];
    }
    $file = $temp_dir . '/heats.json';
    file_put_contents($file, json_encode($heats_data, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'heats.json'];

    // 3. configurations.json – minimal fields
    $config_stmt = $pdo->prepare("SELECT id, name, view_type, heat_number, category, status, created_at FROM result_configurations WHERE event_id = ? AND status = 'active' ORDER BY created_at DESC");
    $config_stmt->execute([$event_id]);
    $configs = $config_stmt->fetchAll(PDO::FETCH_ASSOC);
    $configs_payload = ['success'=>true,'configurations'=>$configs];
    $file = $temp_dir . '/configurations.json';
    file_put_contents($file, json_encode($configs_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'configurations.json'];

    // 4. active_heats.json – derive from event_heat_settings
    $active_stmt = $pdo->prepare("SELECT heat_number, heat_name, scoring_type, runs_count, is_active, bib_on_start, bib_latest_on_run, bib_performing FROM event_heat_settings WHERE event_id = ? ORDER BY heat_number");
    $active_stmt->execute([$event_id]);
    $active_rows = $active_stmt->fetchAll(PDO::FETCH_ASSOC);
    // index by heat_number for parity with dynamic endpoint
    $indexed = [];
    foreach ($active_rows as $row) { $indexed[$row['heat_number']] = $row; }
    $active_payload = ['success'=>true,'heats'=>$indexed];
    $file = $temp_dir . '/active_heats.json';
    file_put_contents($file, json_encode($active_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'active_heats.json'];

    // 5. heat_rows.json – simplified scoreboard rows: latest approved score per participant
    $rows_stmt = $pdo->prepare("SELECT ep.heat_number, ep.bib_number, p.first_name, p.last_name, p.country, p.club, MAX(s.score_value) as best_score FROM event_participants ep LEFT JOIN participants p ON ep.participant_id=p.id LEFT JOIN runs r ON ep.id = r.event_participant_id LEFT JOIN scores s ON r.id = s.run_id AND s.is_approved=1 WHERE ep.event_id=? GROUP BY ep.id ORDER BY ep.heat_number, ep.bib_number");
    $rows_stmt->execute([$event_id]);
    $rows = $rows_stmt->fetchAll(PDO::FETCH_ASSOC);
    $heat_rows_payload = ['success'=>true,'rows'=>$rows];
    $file = $temp_dir . '/heat_rows.json';
    file_put_contents($file, json_encode($heat_rows_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'heat_rows.json'];

    // 6. participants_by_bib.json – map for quick lookup
    $bib_stmt = $pdo->prepare("SELECT ep.bib_number, p.id as participant_id, p.first_name, p.last_name, p.photo, p.country, p.club FROM event_participants ep JOIN participants p ON ep.participant_id=p.id WHERE ep.event_id=?");
    $bib_stmt->execute([$event_id]);
    $map = [];
    while ($r = $bib_stmt->fetch(PDO::FETCH_ASSOC)) {
        if ($r['bib_number'] !== null) { 
            // Update photo filename to match copied format
            if (!empty($r['photo'])) {
                $photo_extension = pathinfo($r['photo'], PATHINFO_EXTENSION);
                $r['photo'] = 'photos/participant_' . $r['participant_id'] . '.' . $photo_extension;
            }
            unset($r['participant_id']); // Remove the temporary ID field
            $map[$r['bib_number']] = $r; 
        }
    }
    $bib_payload = ['success'=>true,'participants'=>$map];
    $file = $temp_dir . '/participants_by_bib.json';
    file_put_contents($file, json_encode($bib_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'participants_by_bib.json'];

    // 7. latest_updates.json – reuse existing helper if present else basic
    try {
        $latest = getLatestUpdatesData($event_id, $pdo);
    } catch (Exception $e) {
        $latest = ['success'=>false,'message'=>$e->getMessage()];
    }
    $file = $temp_dir . '/latest_updates.json';
    file_put_contents($file, json_encode($latest, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'latest_updates.json'];

    // 8. summary_table.json – basic ranking by best approved score (desc)
    $rank_stmt = $pdo->prepare("SELECT ep.bib_number, p.first_name, p.last_name, p.country, MAX(s.score_value) as best_score FROM event_participants ep JOIN participants p ON ep.participant_id = p.id LEFT JOIN runs r ON ep.id = r.event_participant_id LEFT JOIN scores s ON r.id = s.run_id AND s.is_approved=1 WHERE ep.event_id = ? GROUP BY ep.id ORDER BY best_score DESC NULLS LAST");
    $rank_stmt->execute([$event_id]);
    $rank_rows = $rank_stmt->fetchAll(PDO::FETCH_ASSOC);
    $rank = 1;
    foreach ($rank_rows as &$rr) { $rr['rank'] = $rr['best_score'] !== null ? $rank++ : null; }
    $summary_payload = ['success'=>true,'rows'=>$rank_rows];
    $file = $temp_dir . '/summary_table.json';
    file_put_contents($file, json_encode($summary_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'summary_table.json'];

    // 9. start_list.json – order by heat then bib
    $start_stmt = $pdo->prepare("SELECT ep.heat_number, ep.bib_number, p.first_name, p.last_name, p.country, p.club FROM event_participants ep JOIN participants p ON ep.participant_id=p.id WHERE ep.event_id=? ORDER BY ep.heat_number, ep.bib_number");
    $start_stmt->execute([$event_id]);
    $start_rows = $start_stmt->fetchAll(PDO::FETCH_ASSOC);
    $start_payload = ['success'=>true,'participants'=>$start_rows];
    $file = $temp_dir . '/start_list.json';
    file_put_contents($file, json_encode($start_payload, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'start_list.json'];

    return $files;
}

// Generate CSV Summary - wrapper for generateCSVData
function generateCSVSummary($event_id, $temp_dir, $pdo) {
    return generateCSVData($event_id, $temp_dir, $pdo);
}

// Generate CSV Start List
function generateCSVStartList($event_id, $temp_dir, $pdo) {
    $files = [];
    
    try {
        $csv_content = generateParticipantsCSV($event_id, $pdo);
        $csv_file = $temp_dir . '/start_list.csv';
        file_put_contents($csv_file, $csv_content);
        $files[] = ['local' => $csv_file, 'remote' => 'start_list.csv'];
        
        // Also generate start list by category
        $categories_stmt = $pdo->prepare("
            SELECT DISTINCT c.id, c.name 
            FROM categories c
            JOIN event_participants ep ON c.id = ep.category_id 
            WHERE ep.event_id = ?
            ORDER BY c.name
        ");
        $categories_stmt->execute([$event_id]);
        $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($categories as $category) {
            $participants_stmt = $pdo->prepare("
                SELECT 
                    ep.bib_number,
                    p.name,
                    p.club,
                    ep.category_id,
                    c.name as category_name,
                    ep.heat_number,
                    ep.sort_order
                FROM event_participants ep
                LEFT JOIN participants p ON ep.participant_id = p.id
                LEFT JOIN categories c ON ep.category_id = c.id
                WHERE ep.event_id = ? AND ep.category_id = ?
                ORDER BY ep.heat_number, ep.sort_order
            ");
            $participants_stmt->execute([$event_id, $category['id']]);
            $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($participants)) {
                $csv_content = "Bib Number,Name,Club,Category,Heat Number,Sort Order\n";
                foreach ($participants as $participant) {
                    $csv_content .= sprintf(
                        "%s,%s,%s,%s,%s,%s\n",
                        escapeCsvField($participant['bib_number']),
                        escapeCsvField($participant['name']),
                        escapeCsvField($participant['club']),
                        escapeCsvField($participant['category_name']),
                        escapeCsvField($participant['heat_number']),
                        escapeCsvField($participant['sort_order'])
                    );
                }
                
                $filename = 'start_list_' . sanitizeFilename($category['name']) . '.csv';
                $csv_file = $temp_dir . '/' . $filename;
                file_put_contents($csv_file, $csv_content);
                $files[] = ['local' => $csv_file, 'remote' => $filename];
            }
        }
        
    } catch (Exception $e) {
        error_log("Error generating CSV start list: " . $e->getMessage());
    }
    
    return $files;
}

// Generate PDF Results
function generatePDFResults($event_id, $temp_dir, $pdo) {
    $files = [];
    
    // Generate summary table PDFs for each configuration (if table exists)
    $configs = [];
    try {
        $configs_stmt = $pdo->prepare("
            SELECT * FROM result_configurations 
            WHERE event_id = ? AND view_type = 'summary_table'
            ORDER BY name
        ");
        $configs_stmt->execute([$event_id]);
        $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // result_configurations table doesn't exist, skip this data
        error_log("result_configurations table not found in PDF generator: " . $e->getMessage());
    }
    
    foreach ($configs as $config) {
        $pdf_content = generateSummaryPDF($config['id'], $pdo);
        $pdf_file = $temp_dir . '/results_' . sanitizeFilename($config['name']) . '.pdf';
        file_put_contents($pdf_file, $pdf_content);
        $files[] = ['local' => $pdf_file, 'remote' => 'results_' . sanitizeFilename($config['name']) . '.pdf'];
    }
    
    return $files;
}

// Generate CSV Data
function generateCSVData($event_id, $temp_dir, $pdo) {
    $files = [];
    
    // Generate summary table CSVs for each configuration (if table exists)
    $configs = [];
    try {
        $configs_stmt = $pdo->prepare("
            SELECT * FROM result_configurations 
            WHERE event_id = ? AND view_type = 'summary_table'
            ORDER BY name
        ");
        $configs_stmt->execute([$event_id]);
        $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // result_configurations table doesn't exist, skip this data
        error_log("result_configurations table not found in CSV generator: " . $e->getMessage());
    }
    
    foreach ($configs as $config) {
        $csv_content = generateSummaryCSV($config['id'], $pdo);
        $csv_file = $temp_dir . '/results_' . sanitizeFilename($config['name']) . '.csv';
        file_put_contents($csv_file, $csv_content);
        $files[] = ['local' => $csv_file, 'remote' => 'results_' . sanitizeFilename($config['name']) . '.csv'];
    }
    
    // Generate participants CSV
    $participants_csv = generateParticipantsCSV($event_id, $pdo);
    $participants_file = $temp_dir . '/participants.csv';
    file_put_contents($participants_file, $participants_csv);
    $files[] = ['local' => $participants_file, 'remote' => 'participants.csv'];
    
    return $files;
}

// Data retrieval functions
function getEventOverviewData($event_id, $pdo) {
    $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get participant count
    $count_stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM event_participants ep 
        WHERE ep.event_id = ?
    ");
    $count_stmt->execute([$event_id]);
    $participant_count = $count_stmt->fetchColumn();
    
    // Get heat count
    $heat_stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT ep.heat_number) as count 
        FROM event_participants ep 
        WHERE ep.event_id = ?
    ");
    $heat_stmt->execute([$event_id]);
    $heat_count = $heat_stmt->fetchColumn();
    
    // Get latest scores count
    $score_stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM scores s
        JOIN runs r ON s.run_id = r.id
        JOIN event_participants ep ON r.event_participant_id = ep.id
        WHERE ep.event_id = ?
    ");
    $score_stmt->execute([$event_id]);
    $score_count = $score_stmt->fetchColumn();
    
    return [
        'event' => $event,
        'participant_count' => $participant_count,
        'heat_count' => $heat_count,
        'score_count' => $score_count,
        'generated_at' => date('c')
    ];
}

function getLatestUpdatesData($event_id, $pdo) {
    try {
        // Get latest score - simplified query
        $latest_stmt = $pdo->prepare("
            SELECT s.*, p.first_name, p.last_name, p.name, p.club, p.country, 
                   ep.category_id, ep.bib_number, ep.heat_number,
                   j.name as judge_name
            FROM scores s
            JOIN event_participants ep ON s.event_participant_id = ep.id
            JOIN participants p ON ep.participant_id = p.id
            LEFT JOIN judges j ON s.judge_id = j.id
            WHERE ep.event_id = ?
            ORDER BY s.created_at DESC
            LIMIT 1
        ");
        $latest_stmt->execute([$event_id]);
        $latest_score = $latest_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get performing now - find participant with most recent score
        $performing_stmt = $pdo->prepare("
            SELECT p.*, ep.category_id, ep.bib_number, ep.heat_number, 'performing' as status,
                   s.created_at as last_score_time
            FROM participants p
            JOIN event_participants ep ON p.id = ep.participant_id
            JOIN scores s ON ep.id = s.event_participant_id
            WHERE ep.event_id = ? AND s.created_at >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)
            ORDER BY s.created_at DESC
            LIMIT 1
        ");
        $performing_stmt->execute([$event_id]);
        $performing_now = $performing_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get next on start (simplified)
        $next_stmt = $pdo->prepare("
            SELECT p.*, ep.category_id, ep.bib_number, ep.heat_number, 'waiting' as status
            FROM participants p
            JOIN event_participants ep ON p.id = ep.participant_id
            WHERE ep.event_id = ? AND ep.id NOT IN (
                SELECT DISTINCT s.event_participant_id FROM scores s
                JOIN event_participants ep2 ON s.event_participant_id = ep2.id
                WHERE ep2.event_id = ?
            )
            ORDER BY ep.heat_number, ep.sort_order
            LIMIT 1
        ");
        $next_stmt->execute([$event_id, $event_id]);
        $next_on_start = $next_stmt->fetch(PDO::FETCH_ASSOC);
        
        return [
            'latest_score' => $latest_score,
            'performing_now' => $performing_now,
            'next_on_start' => $next_on_start,
            'generated_at' => date('c')
        ];
        
    } catch (Exception $e) {
        error_log("Error in getLatestUpdatesData: " . $e->getMessage());
        return [
            'latest_score' => null,
            'performing_now' => null,
            'next_on_start' => null,
            'generated_at' => date('c'),
            'error' => $e->getMessage()
        ];
    }
}

function getHeatsData($event_id, $pdo) {
    try {
        // Use exact structure from public_dashboard_api.php
        $stmt = $pdo->prepare("
            SELECT 
                hs.heat_number,
                hs.heat_name,
                hs.scoring_type,
                hs.runs_count,
                hs.is_active,
                hs.bib_on_start,
                CONCAT(p_start.first_name, ' ', p_start.last_name) as bib_start_name,
                hs.bib_latest_on_run,
                CONCAT(p_latest.first_name, ' ', p_latest.last_name) as bib_latest_name,
                hs.bib_performing,
                COUNT(DISTINCT ep.id) as participants,
                COUNT(DISTINCT CASE WHEN s.is_approved = 1 THEN s.id END) as total_scores
            FROM event_heat_settings hs
            LEFT JOIN event_participants ep_start ON (hs.event_id = ep_start.event_id AND hs.heat_number = ep_start.heat_number AND ep_start.bib_number = hs.bib_on_start)
            LEFT JOIN participants p_start ON ep_start.participant_id = p_start.id
            LEFT JOIN event_participants ep_latest ON (hs.event_id = ep_latest.event_id AND hs.heat_number = ep_latest.heat_number AND ep_latest.bib_number = hs.bib_latest_on_run)
            LEFT JOIN participants p_latest ON ep_latest.participant_id = p_latest.id
            LEFT JOIN event_participants ep ON hs.event_id = ep.event_id AND hs.heat_number = ep.heat_number
            LEFT JOIN runs r ON ep.id = r.event_participant_id
            LEFT JOIN scores s ON r.id = s.run_id
            WHERE hs.event_id = ?
            GROUP BY hs.heat_number, hs.heat_name, hs.scoring_type, hs.runs_count, hs.is_active, hs.bib_on_start, hs.bib_latest_on_run, hs.bib_performing, p_start.first_name, p_start.last_name, p_latest.first_name, p_latest.last_name
            ORDER BY hs.heat_number
        ");
        $stmt->execute([$event_id]);
        $heats = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format data as key-value by heat_number like the API does
        $heats_data = [];
        foreach ($heats as $heat) {
            $heats_data[$heat['heat_number']] = $heat;
        }
        
        return [
            'success' => true,
            'heats' => $heats_data,
            'generated_at' => date('c')
        ];
        
    } catch (Exception $e) {
        error_log("Error in getHeatsData: " . $e->getMessage());
        // Return empty data structure as fallback
        return [
            'success' => false,
            'heats' => [],
            'generated_at' => date('c'),
            'error' => $e->getMessage()
        ];
    }
}

function getScoresData($event_id, $pdo) {
    try {
        // Use structure similar to public_dashboard_api.php latest score query
        $stmt = $pdo->prepare("
            SELECT 
                s.score_value, 
                r.run_number,
                r.id as run_id,
                p.first_name, 
                p.last_name, 
                p.club, 
                p.country, 
                p.photo,
                ep.bib_number, 
                ep.category_id,
                ep.heat_number,
                ec.category_name as category,
                ehs.heat_name
            FROM scores s
            JOIN runs r ON s.run_id = r.id
            JOIN event_participants ep ON r.event_participant_id = ep.id
            JOIN participants p ON ep.participant_id = p.id
            LEFT JOIN event_categories ec ON ep.category_id = ec.id
            LEFT JOIN event_heat_settings ehs ON ep.event_id = ehs.event_id AND ehs.heat_number = ep.heat_number
            WHERE ep.event_id = ?
            ORDER BY s.id DESC
        ");
        $stmt->execute([$event_id]);
        $scores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Update photo filenames to match copied files
        foreach ($scores as &$score) {
            if (!empty($score['photo'])) {
                // Get the participant ID to match the copied filename format
                $participant_id = null;
                
                // Get participant ID for this participant
                $participant_stmt = $pdo->prepare("
                    SELECT p.id
                    FROM participants p
                    JOIN event_participants ep ON p.id = ep.participant_id
                    JOIN runs r ON ep.id = r.event_participant_id
                    WHERE r.id = ?
                ");
                $participant_stmt->execute([$score['run_id']]);
                $participant_result = $participant_stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($participant_result) {
                    // Update photo filename to match copied format: participant_ID.extension
                    $photo_extension = pathinfo($score['photo'], PATHINFO_EXTENSION);
                    $score['photo'] = 'participant_' . $participant_result['id'] . '.' . $photo_extension;
                    error_log("Updated photo filename for run {$score['run_id']}: participant_{$participant_result['id']}.$photo_extension");
                }
            }
        }
        
        return [
            'success' => true,
            'scores' => $scores,
            'generated_at' => date('c')
        ];
        
    } catch (Exception $e) {
        error_log("Error in getScoresData: " . $e->getMessage());
        
        // Fallback with minimal data structure
        return [
            'success' => false,
            'scores' => [],
            'generated_at' => date('c'),
            'error' => $e->getMessage()
        ];
    }
}

function getSummaryTableData($config_id, $pdo) {
    // This would integrate with your existing summary table API
    $summary_api_url = $_SERVER['REQUEST_SCHEME'] . '://' . $_SERVER['HTTP_HOST'] . 
                      dirname(dirname($_SERVER['REQUEST_URI'])) . '/api/summary_table_api.php?config_id=' . $config_id . '&format=json';
    
    $json_content = @file_get_contents($summary_api_url);
    if ($json_content === false) {
        return ['error' => 'Could not fetch summary table data'];
    }
    
    return json_decode($json_content, true) ?: ['error' => 'Invalid JSON data'];
}

// HTML generation functions
function generateDashboardHTML($event, $pdo) {
    $event_name = htmlspecialchars($event['name']);
    
    return <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Live Results - {$event_name}</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="dashboard.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark sticky-top">
        <div class="container">
            <span class="navbar-brand">
                <i class="fas fa-trophy me-2"></i>
                Live Results - {$event_name}
            </span>
            <span class="navbar-text" id="lastUpdate">
                <i class="fas fa-clock me-1"></i>
                Loading...
            </span>
        </div>
    </nav>
    
    <div class="container-fluid py-4">
        <div id="dashboard-content">
            <div class="text-center py-5">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-3">Loading live results...</p>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="dashboard.js"></script>
</body>
</html>
HTML;
}

function generateDashboardCSS($event, $pdo) {
    return <<<CSS
/* Published Dashboard Styles */
body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
}

.card {
    border: none;
    border-radius: 15px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.card:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 40px rgba(0,0,0,0.15);
}

.participant-photo {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    object-fit: cover;
    border: 3px solid #fff;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.participant-photo-placeholder {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: bold;
    font-size: 16px;
    border: 3px solid #fff;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.facts-icon {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    background: linear-gradient(135deg, #ffd700 0%, #ffed4e 100%);
    color: #333;
    font-size: 12px;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.facts-icon:hover {
    transform: scale(1.1);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.badge {
    font-size: 0.85em;
    padding: 0.5em 0.75em;
}

.table {
    background: white;
    border-radius: 10px;
    overflow: hidden;
}

.table th {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    font-weight: 600;
    text-transform: uppercase;
    font-size: 0.9em;
}

.update-time {
    background: rgba(255,255,255,0.1);
    backdrop-filter: blur(10px);
    border-radius: 20px;
    padding: 0.5rem 1rem;
}
.badge.badge_p {
    display: none;
}
@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.7; }
}

.loading {
    animation: pulse 2s infinite;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .participant-photo,
    .participant-photo-placeholder {
        width: 40px;
        height: 40px;
        font-size: 14px;
    }
    
    .container-fluid {
        padding: 1rem;
    }
}
CSS;
}

function generateDashboardJS($event, $pdo) {
    $event_id = $event['id'];
    
    return <<<JS
// Published Dashboard JavaScript
class LiveDashboard {
    constructor() {
        this.eventId = {$event_id};
        this.refreshInterval = 30000; // 30 seconds
        this.lastUpdate = null;
        
        this.init();
    }
    
    init() {
        this.loadData();
        this.startAutoRefresh();
        this.updateLastUpdateTime();
    }
    
    async loadData() {
        try {
            // Load combined data
            const response = await fetch('data_all.json?t=' + Date.now());
            const data = await response.json();
            
            if (data) {
                this.renderDashboard(data);
                this.lastUpdate = new Date();
                this.updateLastUpdateTime();
            }
        } catch (error) {
            console.error('Error loading data:', error);
            this.showError('Failed to load live data');
        }
    }
    
    renderDashboard(data) {
        const container = document.getElementById('dashboard-content');
        
        let html = `
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card bg-primary text-white">
                        <div class="card-body">
                            <div class="row align-items-center">
                                <div class="col-md-8">
                                    <h2 class="mb-1">\${data.event.event.name}</h2>
                                    <p class="mb-0 opacity-75">Live Results Dashboard</p>
                                </div>
                                <div class="col-md-4 text-end">
                                    <div class="row g-2">
                                        <div class="col-4">
                                            <div class="text-center">
                                                <div class="h4 mb-0">\${data.event.participant_count}</div>
                                                <small>Athletes</small>
                                            </div>
                                        </div>
                                        <div class="col-4">
                                            <div class="text-center">
                                                <div class="h4 mb-0">\${data.event.heat_count}</div>
                                                <small>Heats</small>
                                            </div>
                                        </div>
                                        <div class="col-4">
                                            <div class="text-center">
                                                <div class="h4 mb-0">\${data.event.score_count}</div>
                                                <small>Scores</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Latest updates section
        if (data.latest) {
            html += this.renderLatestUpdates(data.latest);
        }
        
        // Heats section
        if (data.heats) {
            html += this.renderHeats(data.heats);
        }
        
        container.innerHTML = html;
    }
    
    renderLatestUpdates(latest) {
        let html = `
            <div class="row mb-4">
                <div class="col-12">
                    <h3 class="text-white mb-3">
                        <i class="fas fa-bolt me-2"></i>
                        Latest Updates
                    </h3>
                </div>
            </div>
            <div class="row g-3 mb-4">
        `;
        
        // Latest Score
        if (latest.latest_score) {
            const score = latest.latest_score;
            html += `
                <div class="col-md-6">
                    <div class="card bg-success bg-opacity-10">
                        <div class="card-body">
                            <h5 class="text-success mb-3">
                                <i class="fas fa-trophy me-2"></i>
                                Latest Score
                            </h5>
                            <div class="d-flex align-items-center">
                                <div class="me-3">
                                    \${this.createParticipantPhoto(score)}
                                </div>
                                <div class="flex-grow-1">
                                    <h6 class="mb-1">\${score.first_name} \${score.last_name}</h6>
                                    <div class="d-flex gap-2 mb-2">
                                        <span class="badge bg-primary">BIB \${score.bib_number}</span>
                                        \${score.country ? `<span class="badge bg-secondary">\${score.country}</span>` : ''}
                                    </div>
                                    <div class="h4 text-success">\${parseFloat(score.score_value).toFixed(2)}</div>
                                    <small class="text-muted">
                                        <i class="fas fa-clock me-1"></i>
                                        \${new Date(score.created_at).toLocaleTimeString()}
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // Next on Start
        if (latest.next_on_start) {
            const next = latest.next_on_start;
            html += `
                <div class="col-md-6">
                    <div class="card bg-primary bg-opacity-10">
                        <div class="card-body">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-flag me-2"></i>
                                Next on Start
                            </h5>
                            <div class="d-flex align-items-center">
                                <div class="me-3">
                                    \${this.createParticipantPhoto(next)}
                                </div>
                                <div class="flex-grow-1">
                                    <h6 class="mb-1">\${next.first_name} \${next.last_name}</h6>
                                    <div class="d-flex gap-2 mb-2">
                                        <span class="badge bg-primary">BIB \${next.bib_number}</span>
                                        \${next.country ? `<span class="badge bg-secondary">\${next.country}</span>` : ''}
                                    </div>
                                    <small class="text-primary">
                                        <i class="fas fa-arrow-right me-1"></i>
                                        Ready to compete
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }
        
        html += '</div>';
        return html;
    }
    
    renderHeats(heatsData) {
        let html = `
            <div class="row mb-4">
                <div class="col-12">
                    <h3 class="text-white mb-3">
                        <i class="fas fa-users me-2"></i>
                        Heat Progress
                    </h3>
                </div>
            </div>
            <div class="row g-3">
        `;
        
        heatsData.heats.forEach(heat => {
            const completion = heat.participants > 0 ? Math.round((heat.scored_participants / heat.participants) * 100) : 0;
            const progressClass = completion >= 100 ? 'bg-success' : (completion >= 50 ? 'bg-warning' : 'bg-info');
            
            html += `
                <div class="col-md-6 col-lg-4">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h6 class="mb-0">Heat \${heat.heat_number}</h6>
                                <span class="badge \${completion >= 100 ? 'bg-success' : 'bg-secondary'}">
                                    \${completion >= 100 ? 'Complete' : 'In Progress'}
                                </span>
                            </div>
                            <div class="mb-2">
                                <div class="d-flex justify-content-between small text-muted mb-1">
                                    <span>Progress</span>
                                    <span>\${heat.scored_participants}/\${heat.participants}</span>
                                </div>
                                <div class="progress" style="height: 8px;">
                                    <div class="progress-bar \${progressClass}" style="width: \${completion}%"></div>
                                </div>
                            </div>
                            <div class="text-center small text-muted">
                                \${completion}% Complete
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        html += '</div>';
        return html;
    }
    
    createParticipantPhoto(participant) {
        if (participant.photo && participant.photo.trim() !== '') {
            return `<img src="\${participant.photo}" alt="\${participant.first_name} \${participant.last_name}" class="participant-photo" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                    <div class="participant-photo-placeholder" style="display:none;">\${(participant.first_name?.charAt(0) + participant.last_name?.charAt(0)).toUpperCase()}</div>`;
        } else {
            const initials = ((participant.first_name?.charAt(0) || '') + (participant.last_name?.charAt(0) || '')).toUpperCase() || '??';
            return `<div class="participant-photo-placeholder">\${initials}</div>`;
        }
    }
    
    startAutoRefresh() {
        setInterval(() => {
            this.loadData();
        }, this.refreshInterval);
    }
    
    updateLastUpdateTime() {
        const element = document.getElementById('lastUpdate');
        if (element && this.lastUpdate) {
            element.innerHTML = `
                <i class="fas fa-clock me-1"></i>
                Last updated: \${this.lastUpdate.toLocaleTimeString()}
            `;
        }
    }
    
    showError(message) {
        const container = document.getElementById('dashboard-content');
        container.innerHTML = `
            <div class="alert alert-danger">
                <h4><i class="fas fa-exclamation-triangle me-2"></i>Error</h4>
                <p>\${message}</p>
                <button class="btn btn-outline-danger" onclick="location.reload()">
                    <i class="fas fa-refresh me-2"></i>Reload
                </button>
            </div>
        `;
    }
}

// Initialize dashboard when page loads
document.addEventListener('DOMContentLoaded', function() {
    new LiveDashboard();
});
JS;
}

// PDF and CSV generation functions
function generateSummaryPDF($config_id, $pdo) {
    // This would integrate with a PDF library like TCPDF or mPDF
    // For now, return a placeholder
    return "PDF content for config {$config_id} would be generated here";
}

function generateSummaryCSV($config_id, $pdo) {
    $data = getSummaryTableData($config_id, $pdo);
    
    if (isset($data['error'])) {
        return "Error," . $data['error'] . "\n";
    }
    
    $csv = '';
    
    // Add headers if available
    if (isset($data['participants']) && !empty($data['participants'])) {
        $first_participant = $data['participants'][0];
        $headers = array_keys($first_participant);
        $csv .= implode(',', array_map(function($h) { return '"' . str_replace('"', '""', $h) . '"'; }, $headers)) . "\n";
        
        // Add data rows
        foreach ($data['participants'] as $participant) {
            $row = array_map(function($v) { return '"' . str_replace('"', '""', $v) . '"'; }, array_values($participant));
            $csv .= implode(',', $row) . "\n";
        }
    }
    
    return $csv;
}

function generateParticipantsCSV($event_id, $pdo) {
    $stmt = $pdo->prepare("
        SELECT ep.bib_number, p.first_name, p.last_name, p.club, p.country, ep.category, ep.heat_number
        FROM participants p
        JOIN event_participants ep ON p.id = ep.participant_id
        WHERE ep.event_id = ? 
        ORDER BY ep.heat_number, ep.sort_order
    ");
    $stmt->execute([$event_id]);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $csv = "BIB,First Name,Last Name,Club,Country,Category,Heat\n";
    
    foreach ($participants as $participant) {
        $row = [
            $participant['bib_number'],
            $participant['first_name'],
            $participant['last_name'],
            $participant['club'],
            $participant['country'],
            $participant['category'],
            $participant['heat_number']
        ];
        $csv .= implode(',', array_map(function($v) { return '"' . str_replace('"', '""', $v) . '"'; }, $row)) . "\n";
    }
    
    return $csv;
}

function escapeCsvField($field) {
    // Escape CSV field - wrap in quotes if contains comma, quote, or newline
    if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
        return '"' . str_replace('"', '""', $field) . '"';
    }
    return $field;
}

function sanitizeFilename($filename) {
    return preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename);
}

function copyParticipantPhotos($event_id, $temp_dir, $pdo) {
    $files = [];
    
    // Create photos directory in temp
    $photos_dir = $temp_dir . '/photos';
    if (!file_exists($photos_dir)) {
        mkdir($photos_dir, 0755, true);
    }
    
    error_log("=== PHOTO COPY DEBUG START ===");
    error_log("Copying participant photos for event $event_id");
    error_log("Target photos directory: $photos_dir");
    error_log("Photos directory exists: " . (file_exists($photos_dir) ? 'YES' : 'NO'));
    
    // Get all participants for this event
    $stmt = $pdo->prepare("
        SELECT p.id, p.photo, p.first_name, p.last_name, ep.bib_number
        FROM participants p
        INNER JOIN event_participants ep ON p.id = ep.participant_id
        WHERE ep.event_id = ? AND p.photo IS NOT NULL AND p.photo != ''
    ");
    $stmt->execute([$event_id]);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("Found " . count($participants) . " participants with photos");
    
    if (count($participants) === 0) {
        error_log("No participants with photos found for event $event_id");
        error_log("=== PHOTO COPY DEBUG END (NO PHOTOS) ===");
        return $files;
    }
    
    $photos_copied = 0;
    foreach ($participants as $participant) {
        error_log("Processing participant: {$participant['first_name']} {$participant['last_name']} (ID: {$participant['id']})");
        error_log("Photo filename: '{$participant['photo']}'");
        
        // Smart path detection - check if photo already includes path structure
        $photo_filename = $participant['photo'];
        $possible_paths = [];
        
        // Case 1: Photo filename already includes full path (e.g., "uploads/profile_stills/f_7.png")
        if (strpos($photo_filename, 'uploads/') === 0) {
            error_log("Photo includes path structure: $photo_filename");
            $possible_paths[] = dirname(__DIR__) . '/' . $photo_filename;
            $possible_paths[] = 'C:\stylescore2025\htdocs\v2/' . $photo_filename;
            $possible_paths[] = dirname(dirname(__DIR__)) . '/v2/' . $photo_filename;
        } else {
            // Case 2: Photo is just filename (e.g., "f_7.png")
            error_log("Photo is filename only: $photo_filename");
            // Use the confirmed working path structure
            $possible_paths[] = 'C:\stylescore2025\htdocs\v2/uploads/profile_stills/' . $photo_filename;
            $possible_paths[] = dirname(__DIR__) . '/uploads/profile_stills/' . $photo_filename;
            $possible_paths[] = dirname(__DIR__) . '/uploads/' . $photo_filename;
            $possible_paths[] = dirname(__DIR__) . '/uploads/participants/' . $photo_filename;
        }
        
        // Additional fallback paths for edge cases
        $possible_paths[] = dirname(dirname(__DIR__)) . '/v2/uploads/profile_stills/' . $photo_filename;
        $possible_paths[] = '../uploads/profile_stills/' . $photo_filename;
        
        $photo_path = null;
        foreach ($possible_paths as $path) {
            error_log("Trying path: $path");
            $real_path = realpath($path);
            if ($real_path && file_exists($real_path) && is_readable($real_path)) {
                $photo_path = $real_path;
                error_log("SUCCESS: Found photo at: $path (resolved to: $real_path)");
                break;
            } else {
                error_log("Not found: $path" . ($real_path ? " (realpath: $real_path)" : " (realpath failed)"));
            }
        }
        
        if ($photo_path) {
            $photo_extension = pathinfo($participant['photo'], PATHINFO_EXTENSION);
            $safe_filename = 'participant_' . $participant['id'] . '.' . $photo_extension;
            $temp_photo_path = $photos_dir . '/' . $safe_filename;
            
            error_log("Attempting to copy: $photo_path -> $temp_photo_path");
            
            if (copy($photo_path, $temp_photo_path)) {
                $files[] = [
                    'local' => $temp_photo_path,
                    'remote' => 'photos/' . $safe_filename
                ];
                $photos_copied++;
                error_log("SUCCESS: Copied photo: " . $participant['photo'] . " -> photos/" . $safe_filename);
            } else {
                error_log("ERROR: Failed to copy photo: $photo_path -> $temp_photo_path");
                $error = error_get_last();
                if ($error) {
                    error_log("Copy error details: " . $error['message']);
                }
            }
        } else {
            error_log("ERROR: Photo not found for participant " . $participant['id'] . ": " . $participant['photo']);
            error_log("Checked paths:");
            foreach ($possible_paths as $path) {
                error_log("  - $path");
            }
        }
        error_log("--- End participant {$participant['id']} ---");
    }
    
    error_log("SUMMARY: Copied $photos_copied out of " . count($participants) . " participant photos for event $event_id");
    error_log("Total files to upload: " . count($files));
    error_log("=== PHOTO COPY DEBUG END ===");
    return $files;
}

// Copy asset files (CSS, JS, logos, etc.) needed for static dashboard
function copyAssetFiles($temp_dir) {
    $files = [];
    
    error_log("=== ASSET COPY START ===");
    error_log("Copying asset files to: $temp_dir");
    
    // Create assets directory in temp
    $assets_dir = $temp_dir . '/assets';
    if (!file_exists($assets_dir)) {
        mkdir($assets_dir, 0755, true);
    }
    
    // Define assets to copy (if we have custom CSS/JS/logos)
    $source_assets = dirname(__DIR__) . '/assets';
    
    // For now, just create empty directory structure
    // In future, could copy custom stylesheets, logos, etc.
    error_log("Assets directory created: $assets_dir");
    error_log("=== ASSET COPY END ===");
    
    return $files;
}

// Generate static HTML dashboard page that mimics public_results.php design
function generateStaticDashboardHTML($event, $event_id) {
    $event_name = htmlspecialchars($event['name']);
    $event_location = htmlspecialchars($event['location'] ?? '');
    $event_date = $event['date'] ?? '';
    
    return <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Live Results - {$event_name}</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <!-- DataTables CSS -->
    <link href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css" rel="stylesheet">
    
    <style>
        /* Matching public_results.php design */
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .participant-photo {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid #fff;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .participant-photo-placeholder {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 18px;
            border: 3px solid #fff;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .sidebar-filters .card {
            border-radius: 0.5rem;
            border: none;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        
        .card {
            border: none;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        
        .table th {
            
            border: none;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.9em;
        }
        
        .table td {
            vertical-align: middle;
            border-color: #e9ecef;
        }
        
        .badge {
            font-size: 0.85em;
            padding: 0.5em 0.75em;
        }
        span.badge.badge_p {
    display: none;
}
        .heat-card {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .heat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }
        
        .loading {
            animation: pulse 2s infinite;
        }
        
        .btn-group .btn {
            font-size: 0.875rem;
        }
        
        .list-group-item {
            border-color: rgba(0,0,0,0.08);
        }
        
        .list-group-item:hover {
            background-color: #f8f9fa;
        }
        
        .list-group-item.active {
            background-color: #0d6efd;
            border-color: #0d6efd;
        }
        
        .config-button {
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .config-button:hover {
            transform: translateX(4px);
        }
        .bg-ss-gradient {
    background: #B0F761;
    background: linear-gradient(270deg, rgba(176, 247, 97, 1) 0%, rgba(76, 190, 201, 1) 21%, rgba(25, 162, 255, 1) 82%, rgba(52, 89, 230, 1) 100%);
}
    </style>
</head>

<body>
    <!-- Header matching public_results.php -->
    <header class="bg-ss-gradient text-white py-3 mb-4 shadow-sm">
        <div class="container d-flex flex-column flex-md-row align-items-md-center justify-content-between">
            <a class="navbar-brand navbar-brand-logo" href="#">
                <img src="./assets/img/SVG/logo-ss-_2.svg" alt="Logo" class="me-2" style="width: 100%; height: 50px; filter: brightness(2.5);" onerror="this.style.display='none';">
            </a>
            <h1 class="h3 mb-2 mb-md-0">Live Results</h1>
            <span class="badge bg-light text-primary fw-semibold px-3 py-2">
                Event: {$event_name}
            </span>
        </div>
    </header>

    <div class="container-fluid p-4">
        <div class="row">
            <!-- Left Sidebar: Filters -->
            <div class="col-12 col-md-3 order-1 order-md-1 mb-4 mb-md-0">
                <div class="row g-3">
                    <!-- View Type Tabs -->
                    <div class="col-12 col-sm-6 col-md-12">
                        <div class="card shadow-sm">
                            <div class="card-body p-3">
                                <h6 class="card-title mb-3"><i class="fas fa-eye me-2"></i>View Type</h6>
                                <div class="btn-group w-100" role="group" id="viewTypeButtons">
                                    <button class="btn btn-outline-primary" onclick="switchViewType('start_list')">
                                        <i class="fas fa-list-ol me-1"></i>Start Lists
                                    </button>
                                    <button class="btn btn-primary active" onclick="switchViewType('summary_table')">
                                        <i class="fas fa-table me-1"></i>Summary
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Saved Configurations -->
                    <div class="col-12 col-sm-6 col-md-12">
                        <div class="card shadow-sm">
                            <div class="card-body p-3">
                                <h6 class="mb-2 fw-bold text-success"><i class="fas fa-bookmark me-2"></i>Saved Reports</h6>
                                <div id="savedReports" class="list-group list-group-flush">
                                    <div class="text-center text-muted small py-3">
                                        <i class="fas fa-spinner fa-spin"></i> Loading reports...
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Content Area -->
            <div class="col-12 col-md-9 order-2 order-md-2">
                <div class="card shadow-none border-0">
                    <div class="card-body p-3">
                        <div id="contentDisplay">
                            <div class="text-center py-5">
                                <div class="spinner-border text-primary" role="status">
                                    <span class="visually-hidden">Loading...</span>
                                </div>
                                <p class="mt-2 text-muted">Loading results...</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer matching public_results.php -->
    <footer class="border-top p-3 mt-5 d-flex flex-row align-items-center" style="background-color: #B0F761;">
        <div></div>
        <div class="container text-center text-primary fw-medium small">
            <span>&copy; 2025 StyleScore | Freestyle scoring made as dynamic as your tricks.</span>
        </div>
        <div>
            <img src="assets/img/logo-dark.png" style="max-height: 50px; width: auto;" class="img-fluid logo-footer" alt="StyleScore Logo" onerror="this.style.display='none';">
        </div>
    </footer>

    <!-- jQuery (required for DataTables) -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- DataTables JS -->
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
    <script src="https://cdn.datatables.net/responsive/2.5.0/js/responsive.bootstrap5.min.js"></script>
    
    <!-- Dashboard JS -->
    <script src="dashboard.js"></script>
</body>
</html>
HTML;
}

// Generate all static API files that match the public dashboard endpoints
function generateStaticAPIFiles($event_id, $temp_dir, $pdo) {
    $files = [];
    
    // Create api directory
    $api_dir = $temp_dir . '/api';
    if (!file_exists($api_dir)) {
        mkdir($api_dir, 0755, true);
    }
    
    // 1. Events overview endpoint (GET ?action=get_events_overview)
    $events_data = generateEventsOverviewAPI($event_id, $pdo);
    $file = $api_dir . '/events_overview.json';
    file_put_contents($file, json_encode($events_data, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'api/events_overview.json'];
    
    // 2. Event heats endpoint (GET ?action=get_event_heats&event_id=...)
    $heats_data = generateEventHeatsAPI($event_id, $pdo);
    $file = $api_dir . '/event_heats.json';
    file_put_contents($file, json_encode($heats_data, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'api/event_heats.json'];
    
    // 3. Configurations endpoint (GET ?action=get_configurations&event_id=...)
    $configs_data = generateConfigurationsAPI($event_id, $pdo);
    $file = $api_dir . '/configurations.json';
    file_put_contents($file, json_encode($configs_data, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'api/configurations.json'];
    
    // 4. Latest updates endpoint (GET ?action=get_event_latest_updates&event_id=...)
    $latest_data = generateLatestUpdatesAPI($event_id, $pdo);
    $file = $api_dir . '/latest_updates.json';
    file_put_contents($file, json_encode($latest_data, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'api/latest_updates.json'];
    
    // 5. Active heat details endpoint (GET ?action=get_active_heat_details&event_id=...)
    $active_data = generateActiveHeatDetailsAPI($event_id, $pdo);
    $file = $api_dir . '/active_heat_details.json';
    file_put_contents($file, json_encode($active_data, JSON_PRETTY_PRINT));
    $files[] = ['local' => $file, 'remote' => 'api/active_heat_details.json'];
    
    // 6. Summary table endpoints (POST load_summary_table for each config)
    $summary_files = generateSummaryTableAPIs($event_id, $api_dir, $pdo);
    $files = array_merge($files, $summary_files);
    
    // 7. Start list endpoints (POST load_start_list)
    $startlist_files = generateStartListAPIs($event_id, $api_dir, $pdo);
    $files = array_merge($files, $startlist_files);
    
    // 8. Main static API dispatcher
    $dispatcher_js = generateStaticAPIDispatcher($event_id);
    $file = $temp_dir . '/dashboard.js';
    file_put_contents($file, $dispatcher_js);
    $files[] = ['local' => $file, 'remote' => 'dashboard.js'];
    
    return $files;
}

function copyStaticAssetFiles($temp_dir) {
    $files = [];
    
    // Create assets directory structure
    $assets_dir = $temp_dir . '/assets';
    $img_dir = $assets_dir . '/img';
    $svg_dir = $img_dir . '/SVG';
    
    if (!file_exists($svg_dir)) {
        mkdir($svg_dir, 0755, true);
    }
    
    // Copy logo SVG file
    $possible_logo_paths = [
        dirname(__DIR__) . '/assets/img/SVG/logo-ss-_2.svg',
        dirname(dirname(__DIR__)) . '/assets/img/SVG/logo-ss-_2.svg',
        '../assets/img/SVG/logo-ss-_2.svg',
        '../../assets/img/SVG/logo-ss-_2.svg',
        __DIR__ . '/../assets/img/SVG/logo-ss-_2.svg'
    ];
    
    foreach ($possible_logo_paths as $logo_path) {
        $real_path = realpath($logo_path);
        if ($real_path && file_exists($real_path)) {
            $temp_logo_path = $svg_dir . '/logo-ss-_2.svg';
            if (copy($real_path, $temp_logo_path)) {
                $files[] = [
                    'local' => $temp_logo_path,
                    'remote' => 'assets/img/SVG/logo-ss-_2.svg'
                ];
                error_log("Copied SVG logo from: $logo_path");
            }
            break;
        }
    }
    
    // Copy dark logo PNG file
    $possible_dark_logo_paths = [
        dirname(__DIR__) . '/assets/img/logo-dark.png',
        dirname(dirname(__DIR__)) . '/assets/img/logo-dark.png',
        '../assets/img/logo-dark.png',
        '../../assets/img/logo-dark.png',
        __DIR__ . '/../assets/img/logo-dark.png'
    ];
    
    foreach ($possible_dark_logo_paths as $logo_path) {
        $real_path = realpath($logo_path);
        if ($real_path && file_exists($real_path)) {
            $temp_logo_path = $img_dir . '/logo-dark.png';
            if (copy($real_path, $temp_logo_path)) {
                $files[] = [
                    'local' => $temp_logo_path,
                    'remote' => 'assets/img/logo-dark.png'
                ];
                error_log("Copied dark logo from: $logo_path");
            }
            break;
        }
    }
    
    // Copy custom CSS if it exists
    $possible_css_paths = [
        dirname(__DIR__) . '/assets/css/custom-dashboard.css',
        dirname(dirname(__DIR__)) . '/assets/css/custom-dashboard.css',
        '../assets/css/custom-dashboard.css',
        '../../assets/css/custom-dashboard.css'
    ];
    
    foreach ($possible_css_paths as $custom_css_path) {
        $real_path = realpath($custom_css_path);
        if ($real_path && file_exists($real_path)) {
            $css_dir = $assets_dir . '/css';
            if (!file_exists($css_dir)) {
                mkdir($css_dir, 0755, true);
            }
            $temp_css_path = $css_dir . '/custom-dashboard.css';
            if (copy($real_path, $temp_css_path)) {
                $files[] = [
                    'local' => $temp_css_path,
                    'remote' => 'assets/css/custom-dashboard.css'
                ];
                error_log("Copied CSS from: $custom_css_path");
            }
            break;
        }
    }
    
    return $files;
}

// Generate events overview API response
function generateEventsOverviewAPI($event_id, $pdo) {
    try {
        // Use the same query structure as public_dashboard_api.php
        $events_stmt = $pdo->prepare("
            SELECT 
                e.id, e.name, e.date, e.heats_total, e.runs_per_heat, e.status,
                ehs.heat_number as current_heat, ehs.active_run as current_run,
                COUNT(DISTINCT ep.id) as total_participants,
                COUNT(DISTINCT ja.judge_id) as total_judges
            FROM events e
            LEFT JOIN event_participants ep ON e.id = ep.event_id
            LEFT JOIN judge_assignments ja ON e.id = ja.event_id
            LEFT JOIN event_heat_settings ehs ON e.id = ehs.event_id AND ehs.is_active = 1
            WHERE e.id = ?
            GROUP BY e.id, e.name, e.date, e.heats_total, e.runs_per_heat, e.status, ehs.heat_number, ehs.active_run
        ");
        $events_stmt->execute([$event_id]);
        $event = $events_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$event) {
            return ['success' => false, 'message' => 'Event not found', 'events' => []];
        }
        
        // Get latest score data
        $latest_score_stmt = $pdo->prepare("
            SELECT s.score_value, s.created_at, s.run_id,
                   p.first_name, p.last_name, p.club, p.country, p.photo,
                   ep.bib_number, ep.category_id,
                   ec.category_name as category,
                   r.run_number,
                   ehs.heat_name, ehs.heat_number
            FROM scores s
            JOIN runs r ON s.run_id = r.id
            JOIN event_participants ep ON r.event_participant_id = ep.id
            JOIN participants p ON ep.participant_id = p.id
            LEFT JOIN event_categories ec ON ep.category_id = ec.id
            LEFT JOIN event_heat_settings ehs ON ep.event_id = ehs.event_id AND ehs.heat_number = ep.heat_number
            WHERE ep.event_id = ? AND s.is_approved = 1
            ORDER BY s.created_at DESC
            LIMIT 1
        ");
        $latest_score_stmt->execute([$event_id]);
        $latest_score = $latest_score_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Calculate ranking if latest score exists
        if ($latest_score) {
            // Update photo filename to match copied format
            if (!empty($latest_score['photo'])) {
                // Get participant ID for photo filename correction
                $participant_id_stmt = $pdo->prepare("
                    SELECT p.id FROM participants p 
                    JOIN event_participants ep ON p.id = ep.participant_id 
                    JOIN runs r ON ep.id = r.event_participant_id 
                    WHERE r.id = ?
                ");
                $participant_id_stmt->execute([$latest_score['run_id']]);
                $participant_id_result = $participant_id_stmt->fetch(PDO::FETCH_ASSOC);
                if ($participant_id_result) {
                    $photo_extension = pathinfo($latest_score['photo'], PATHINFO_EXTENSION);
                    $latest_score['photo'] = 'photos/participant_' . $participant_id_result['id'] . '.' . $photo_extension;
                }
            }
            
            $rank_stmt = $pdo->prepare("
                SELECT COUNT(*) + 1 as rank
                FROM scores s2 
                JOIN runs r2 ON r2.id = s2.run_id 
                JOIN event_participants ep2 ON ep2.id = r2.event_participant_id 
                WHERE ep2.event_id = ? 
                AND s2.is_approved = 1 
                AND s2.score_value > ?
            ");
            $rank_stmt->execute([$event_id, $latest_score['score_value']]);
            $rank_result = $rank_stmt->fetch(PDO::FETCH_ASSOC);
            $latest_score['current_rank'] = $rank_result['rank'];
        }
        
        $event['latest_score'] = $latest_score ?: null;
        
        // Get next BIB data
        $judge_count_stmt = $pdo->prepare("SELECT COUNT(DISTINCT judge_id) FROM judge_assignments WHERE event_id = ?");
        $judge_count_stmt->execute([$event_id]);
        $judge_count = (int)$judge_count_stmt->fetchColumn();

        $next_bib_stmt = $pdo->prepare("
            SELECT ep.bib_number, ep.category_id, p.id as participant_id,
                   p.first_name, p.last_name, p.club, p.country, p.photo,
                   ec.category_name as category,
                   ehs.heat_name
            FROM event_participants ep
            JOIN participants p ON ep.participant_id = p.id
            LEFT JOIN event_categories ec ON ep.category_id = ec.id
            LEFT JOIN event_heat_settings ehs ON ep.event_id = ehs.event_id AND ehs.heat_number = ep.heat_number
            WHERE ep.event_id = ? AND ep.heat_number = ?
            AND ep.id NOT IN (
                SELECT DISTINCT r.event_participant_id
                FROM runs r
                JOIN scores s ON r.id = s.run_id
                WHERE r.run_number = ?
                GROUP BY r.event_participant_id
                HAVING COUNT(s.id) >= ?
            )
            ORDER BY ep.bib_number
            LIMIT 1
        ");
        $next_bib_stmt->execute([
            $event_id,
            $event['current_heat'] ?? 1,
            $event['current_run'] ?? 1,
            $judge_count
        ]);
        $next_bib_data = $next_bib_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Update photo filename for next_bib data
        if ($next_bib_data && !empty($next_bib_data['photo'])) {
            $photo_extension = pathinfo($next_bib_data['photo'], PATHINFO_EXTENSION);
            $next_bib_data['photo'] = 'photos/participant_' . $next_bib_data['participant_id'] . '.' . $photo_extension;
            unset($next_bib_data['participant_id']); // Remove the temporary ID field
        }
        
        $event['next_bib'] = $next_bib_data ?: null;
        
        return ['success' => true, 'events' => [$event]];
        
    } catch (Exception $e) {
        error_log("Error in generateEventsOverviewAPI: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage(), 'events' => []];
    }
}

// Generate event heats API response
function generateEventHeatsAPI($event_id, $pdo) {
    try {
        // Get category names for all categories in this event
        $categories_stmt = $pdo->prepare("
            SELECT id, category_name FROM event_categories 
            WHERE event_id = ?
        ");
        $categories_stmt->execute([$event_id]);
        $category_map = [];
        while ($cat = $categories_stmt->fetch(PDO::FETCH_ASSOC)) {
            $category_map[$cat['id']] = $cat['category_name'];
        }
        
        // Get judge count
        $judges_count_stmt = $pdo->prepare("SELECT COUNT(*) FROM judge_assignments WHERE event_id = ?");
        $judges_count_stmt->execute([$event_id]);
        $judges_count = (int)$judges_count_stmt->fetchColumn();

        // Get heats data
        $heats_stmt = $pdo->prepare("
            SELECT 
                ehs.id, ehs.event_id, ehs.heat_number, ehs.heat_name, 
                ehs.scoring_type, ehs.runs_count, ehs.categories, ehs.time_start,
                ehs.flow_type, ehs.flow_source_heat, ehs.flow_participants_per_category,
                ehs.is_active, ehs.active_run, ehs.bib_on_start, ehs.bib_latest_on_run,
                COUNT(DISTINCT ep.id) as participants,
                COUNT(DISTINCT CASE WHEN s.is_approved = 1 THEN s.id END) as total_scores,
                (COUNT(DISTINCT ep.id) * ehs.runs_count * ?) as expected_scores
            FROM event_heat_settings ehs
            LEFT JOIN event_participants ep ON ehs.event_id = ep.event_id AND ehs.heat_number = ep.heat_number
            LEFT JOIN runs r ON ep.id = r.event_participant_id
            LEFT JOIN scores s ON r.id = s.run_id
            WHERE ehs.event_id = ?
            GROUP BY ehs.id, ehs.event_id, ehs.heat_number, ehs.heat_name, 
                     ehs.scoring_type, ehs.runs_count, ehs.time_start, ehs.categories,
                     ehs.flow_type, ehs.flow_source_heat, ehs.flow_participants_per_category,
                     ehs.is_active, ehs.active_run, ehs.bib_on_start, ehs.bib_latest_on_run
            ORDER BY ehs.heat_number
        ");
        $heats_stmt->execute([$judges_count, $event_id]);
        $heats = $heats_stmt->fetchAll(PDO::FETCH_ASSOC);

        // Add category names to each heat
        foreach ($heats as &$heat) {
            if (!empty($heat['categories'])) {
                try {
                    $category_ids = json_decode($heat['categories'], true);
                    if (is_array($category_ids)) {
                        $category_names = [];
                        foreach ($category_ids as $id) {
                            if (isset($category_map[$id])) {
                                $category_names[] = $category_map[$id];
                            } else {
                                $category_names[] = "ID: $id";
                            }
                        }
                        $heat['category_names'] = $category_names;
                    }
                } catch (Exception $e) {
                    // Keep original categories if there's an error
                }
            }
        }
        
        // Get configurations
        $configs_stmt = $pdo->prepare("
            SELECT id, name, view_type, heat_number, category, display_order
            FROM result_configurations
            WHERE event_id = ? AND status = 'public'
            ORDER BY heat_number, display_order, name
        ");
        $configs_stmt->execute([$event_id]);
        $configurations = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);

        return ['success' => true, 'heats' => $heats, 'configurations' => $configurations];
        
    } catch (Exception $e) {
        error_log("Error in generateEventHeatsAPI: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage(), 'heats' => [], 'configurations' => []];
    }
}

// Generate configurations API response
function generateConfigurationsAPI($event_id, $pdo) {
    try {
        $configs_stmt = $pdo->prepare("
            SELECT id, name, view_type, heat_number, category, display_order
            FROM result_configurations
            WHERE event_id = ? AND status = 'public'
            ORDER BY display_order, name
        ");
        $configs_stmt->execute([$event_id]);
        $configurations = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);

        return ['success' => true, 'configurations' => $configurations];
        
    } catch (Exception $e) {
        error_log("Error in generateConfigurationsAPI: " . $e->getMessage());
        
        // Provide default configurations if table doesn't exist
        $default_configurations = [
            [
                'id' => 1,
                'name' => 'Event Start List',
                'view_type' => 'start_list',
                'heat_number' => null,
                'category' => null,
                'display_order' => 1
            ],
            [
                'id' => 2,
                'name' => 'Event Results',
                'view_type' => 'summary_table',
                'heat_number' => null,
                'category' => null,
                'display_order' => 2
            ]
        ];
        
        return ['success' => true, 'configurations' => $default_configurations];
    }
}

// Generate latest updates API response
function generateLatestUpdatesAPI($event_id, $pdo) {
    try {
        // Get latest score (most recent approved score)
        $latest_score_stmt = $pdo->prepare("
            SELECT 
                ep.bib_number, 
                p.id as participant_id,
                p.first_name, 
                p.last_name,
                p.club, 
                p.country,
                p.photo,
                s.score_value, 
                s.created_at,
                ec.category_name as category,
                ehs.heat_name,
                ehs.heat_number,
                (
                    SELECT COUNT(*) + 1 
                    FROM scores s2 
                    JOIN runs r2 ON r2.id = s2.run_id 
                    JOIN event_participants ep2 ON ep2.id = r2.event_participant_id 
                    WHERE ep2.event_id = ep.event_id 
                    AND s2.is_approved = 1 
                    AND s2.score_value > s.score_value
                ) as current_rank
            FROM scores s
            JOIN runs r ON r.id = s.run_id
            JOIN event_participants ep ON ep.id = r.event_participant_id
            JOIN participants p ON p.id = ep.participant_id
            LEFT JOIN event_categories ec ON ec.id = ep.category_id
            LEFT JOIN event_heat_settings ehs ON ehs.event_id = ep.event_id AND ehs.heat_number = ep.heat_number
            WHERE ep.event_id = ? AND s.is_approved = 1
            ORDER BY s.created_at DESC
            LIMIT 1
        ");
        $latest_score_stmt->execute([$event_id]);
        $latest_score = $latest_score_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Update photo filename for latest_score
        if ($latest_score && !empty($latest_score['photo'])) {
            $photo_extension = pathinfo($latest_score['photo'], PATHINFO_EXTENSION);
            $latest_score['photo'] = 'photos/participant_' . $latest_score['participant_id'] . '.' . $photo_extension;
            unset($latest_score['participant_id']); // Remove the temporary ID field
        }

        // Get next participant on start
        $next_bib_stmt = $pdo->prepare("
            SELECT 
                ep.bib_number, 
                p.id as participant_id,
                p.first_name, 
                p.last_name,
                p.club, 
                p.country,
                p.photo,
                ec.category_name as category,
                ehs.heat_name,
                ehs.heat_number
            FROM event_participants ep
            JOIN participants p ON p.id = ep.participant_id
            LEFT JOIN event_categories ec ON ec.id = ep.category_id
            LEFT JOIN event_heat_settings ehs ON ehs.event_id = ep.event_id AND ehs.heat_number = ep.heat_number
            LEFT JOIN runs r ON r.event_participant_id = ep.id
            LEFT JOIN scores s ON s.run_id = r.id AND s.is_approved = 1
            WHERE ep.event_id = ? 
            AND ehs.is_active = 1
            AND s.id IS NULL
            ORDER BY ep.sort_order ASC, ep.bib_number ASC
            LIMIT 1
        ");
        $next_bib_stmt->execute([$event_id]);
        $next_bib = $next_bib_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Update photo filename for next_bib
        if ($next_bib && !empty($next_bib['photo'])) {
            $photo_extension = pathinfo($next_bib['photo'], PATHINFO_EXTENSION);
            $next_bib['photo'] = 'photos/participant_' . $next_bib['participant_id'] . '.' . $photo_extension;
            unset($next_bib['participant_id']); // Remove the temporary ID field
        }

        return [
            'success' => true,
            'latest_score' => $latest_score ?: null,
            'next_bib' => $next_bib ?: null,
            'performing_now' => null,
            'heat_status' => null
        ];
        
    } catch (Exception $e) {
        error_log("Error in generateLatestUpdatesAPI: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Generate active heat details API response
function generateActiveHeatDetailsAPI($event_id, $pdo) {
    try {
        $heats_stmt = $pdo->prepare("
            SELECT 
                hs.heat_number, 
                hs.is_active, 
                hs.active_run, 
                hs.bib_on_start,
                CONCAT(p_start.first_name, ' ', p_start.last_name) as bib_start_name,
                hs.bib_latest_on_run,
                CONCAT(p_latest.first_name, ' ', p_latest.last_name) as bib_latest_name
            FROM event_heat_settings hs
            LEFT JOIN event_participants ep_start ON (hs.event_id = ep_start.event_id AND hs.heat_number = ep_start.heat_number AND ep_start.bib_number = hs.bib_on_start)
            LEFT JOIN participants p_start ON ep_start.participant_id = p_start.id
            LEFT JOIN event_participants ep_latest ON (hs.event_id = ep_latest.event_id AND hs.heat_number = ep_latest.heat_number AND ep_latest.bib_number = hs.bib_latest_on_run)
            LEFT JOIN participants p_latest ON ep_latest.participant_id = p_latest.id
            WHERE hs.event_id = ? AND hs.is_active = 1
        ");
        $heats_stmt->execute([$event_id]);
        $active_heats = $heats_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format data as key-value by heat_number
        $heats_data = [];
        foreach ($active_heats as $heat) {
            $heats_data[$heat['heat_number']] = $heat;
        }
        
        return [
            'success' => true,
            'heats' => $heats_data
        ];
        
    } catch (Exception $e) {
        error_log("Error in generateActiveHeatDetailsAPI: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage(), 'heats' => []];
    }
}

// Build API URL from configuration JSON (for summary tables)
function buildSummaryTableAPIUrl($config) {
    $settings = json_decode($config['configuration'], true);
    
    $params = [
        'event_id' => $config['event_id'],
        'format' => 'json',
        'styling' => 'partial'
    ];
    
    // Heat filter - handle both single heat and multiple heats
    if (isset($settings['heat_number']) && $settings['heat_number']) {
        if (is_array($settings['heat_number'])) {
            // Multiple heats - use heat_run_filter format
            $heat_run_filter = [];
            foreach ($settings['heat_number'] as $heat) {
                $heat_run_filter[$heat] = [1, 2, 3, 4, 5]; // All runs
            }
            $params['heat_run_filter'] = json_encode($heat_run_filter);
        } else {
            // Single heat
            $params['heat_number'] = $settings['heat_number'];
        }
    }
    
    // Column visibility
    $params['show_rank'] = isset($settings['showRank']) && $settings['showRank'] ? 'true' : 'false';
    $params['show_bib'] = isset($settings['showBib']) && $settings['showBib'] ? 'true' : 'false';
    $params['show_participant'] = isset($settings['showParticipant']) && $settings['showParticipant'] ? 'true' : 'false';
    $params['show_category'] = isset($settings['showCategory']) && $settings['showCategory'] ? 'true' : 'false';
    $params['show_club'] = isset($settings['showClub']) && $settings['showClub'] ? 'true' : 'false';
    $params['show_gender'] = isset($settings['showGender']) && $settings['showGender'] ? 'true' : 'false';
    $params['show_fis_code'] = isset($settings['showFisCode']) && $settings['showFisCode'] ? 'true' : 'false';
    $params['show_licence_code'] = isset($settings['showLicenceCode']) && $settings['showLicenceCode'] ? 'true' : 'false';
    $params['show_country'] = isset($settings['showCountry']) && $settings['showCountry'] ? 'true' : 'false';
    $params['show_birth_year'] = isset($settings['showBirthYear']) && $settings['showBirthYear'] ? 'true' : 'false';
    $params['show_age'] = isset($settings['showAge']) && $settings['showAge'] ? 'true' : 'false';
    $params['show_runs'] = isset($settings['showRuns']) && $settings['showRuns'] ? 'true' : 'false';
    $params['show_judges'] = isset($settings['showJudges']) && $settings['showJudges'] ? 'true' : 'false';
    $params['show_control_points'] = isset($settings['showControlPoints']) && $settings['showControlPoints'] ? 'true' : 'false';
    $params['show_figures'] = isset($settings['showFigures']) && $settings['showFigures'] ? 'true' : 'false';
    $params['show_event_settings'] = isset($settings['showEventSettings']) && $settings['showEventSettings'] ? 'true' : 'false';
    $params['show_heat_best'] = isset($settings['showHeatBest']) && $settings['showHeatBest'] ? 'true' : 'false';
    $params['show_heat_average'] = isset($settings['showHeatAverage']) && $settings['showHeatAverage'] ? 'true' : 'false';
    $params['show_overall_best'] = isset($settings['showOverallBest']) && $settings['showOverallBest'] ? 'true' : 'false';
    $params['show_highest_average'] = isset($settings['showHighestAverage']) && $settings['showHighestAverage'] ? 'true' : 'false';
    
    // Heat direction
    if (isset($settings['heatDirection'])) {
        $params['heat_direction'] = $settings['heatDirection'];
    }
    
    // Sorting
    if (isset($settings['sortBy'])) {
        $params['sort'] = $settings['sortBy'];
    }
    if (isset($settings['sortDirection'])) {
        $params['sort_direction'] = $settings['sortDirection'];
    }
    
    // Gender filter
    if (isset($settings['gender'])) {
        $params['gender'] = $settings['gender'];
    } elseif (isset($settings['gender_filter_main'])) {
        $params['gender'] = $settings['gender_filter_main'];
    }
    
    // Category filter
    if (isset($settings['category_filter']) && $settings['category_filter'] !== 'all') {
        $params['category'] = $settings['category_filter'];
    }
    
    // Column renames
    if (isset($settings['columnRenames']) && is_array($settings['columnRenames']) && count($settings['columnRenames']) > 0) {
        foreach ($settings['columnRenames'] as $key => $value) {
            $params['rename_' . $key] = $value;
        }
    }
    
    return resolveUrl('/v2/api/summary_table_api.php') . '?' . http_build_query($params);
}

// Build API URL from configuration JSON (for start lists)
function buildStartListAPIUrl($config) {
    $settings = json_decode($config['configuration'], true);
    
    $params = [
        'event_id' => $config['event_id'],
        'format' => 'json'
    ];
    
    // Heat filter
    if (isset($settings['heat_number']) && $settings['heat_number']) {
        $params['heat_number'] = $settings['heat_number'];
    }
    
    // Gender filter
    if (isset($settings['gender_filter_main']) && $settings['gender_filter_main'] !== 'all') {
        $params['gender'] = $settings['gender_filter_main'];
    }
    
    // Category filter
    if (isset($settings['category_filter']) && $settings['category_filter'] !== 'all') {
        $params['category'] = $settings['category_filter'];
    }
    
    // Column visibility
    $params['show_bib'] = isset($settings['showBib']) && $settings['showBib'] ? 'true' : 'false';
    $params['show_category'] = isset($settings['showCategory']) && $settings['showCategory'] ? 'true' : 'false';
    $params['show_club'] = isset($settings['showClub']) && $settings['showClub'] ? 'true' : 'false';
    $params['show_country'] = isset($settings['showCountry']) && $settings['showCountry'] ? 'true' : 'false';
    $params['show_birth_year'] = isset($settings['showBirthYear']) && $settings['showBirthYear'] ? 'true' : 'false';
    $params['show_age'] = isset($settings['showAge']) && $settings['showAge'] ? 'true' : 'false';
    
    return resolveUrl('/v2/ajax_start_list.php') . '?' . http_build_query($params);
}

// Generate summary table API files for all configurations
function generateSummaryTableAPIs($event_id, $api_dir, $pdo) {
    $files = [];
    
    try {
        $configs_stmt = $pdo->prepare("
            SELECT * FROM result_configurations 
            WHERE event_id = ? AND view_type = 'summary_table' AND status = 'public'
            ORDER BY name
        ");
        $configs_stmt->execute([$event_id]);
        $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($configs as $config) {
            error_log("Generating summary table for config ID: " . $config['id'] . " - " . $config['name']);
            
            try {
                // Build full API URL from configuration settings
                $api_url = buildSummaryTableAPIUrl($config);
                
                error_log("Calling API: " . $api_url);
                
                $context = stream_context_create([
                    'http' => [
                        'timeout' => 60,
                        'ignore_errors' => true
                    ]
                ]);
                
                $json_content = @file_get_contents($api_url, false, $context);
                
                if ($json_content === false) {
                    error_log("Failed to fetch summary table data for config " . $config['id']);
                    $summary_data = ['success' => false, 'message' => 'Could not fetch summary table data'];
                } else {
                    // Log response length for debugging
                    error_log("Response length: " . strlen($json_content) . " bytes");
                    
                    // Check if response looks like valid JSON
                    if (empty(trim($json_content))) {
                        error_log("Empty response for config " . $config['id']);
                        $summary_data = ['success' => false, 'message' => 'Empty response from API'];
                    } else {
                        $summary_data = json_decode($json_content, true);
                        if ($summary_data === null) {
                            error_log("Invalid JSON for config " . $config['id'] . ". JSON Error: " . json_last_error_msg());
                            error_log("Response preview: " . substr($json_content, 0, 500));
                            $summary_data = ['success' => false, 'message' => 'Invalid JSON data: ' . json_last_error_msg()];
                        }
                    }
                }
                
                $file = $api_dir . '/summary_table_' . $config['id'] . '.json';
                file_put_contents($file, json_encode($summary_data, JSON_PRETTY_PRINT));
                $files[] = ['local' => $file, 'remote' => 'api/summary_table_' . $config['id'] . '.json'];
                
                error_log("Successfully generated summary_table_" . $config['id'] . ".json");
            } catch (Exception $e) {
                error_log("Error generating summary for config " . $config['id'] . ": " . $e->getMessage());
            }
        }
        
    } catch (Exception $e) {
        error_log("Error generating summary table APIs: " . $e->getMessage());
    }
    
    return $files;
}

// Generate HTML tables for summary tables (for direct public server display)
function generateSummaryTableHTMLs($event_id, $temp_dir, $pdo) {
    $files = [];
    
    // Create tables directory for HTML files
    $tables_dir = $temp_dir . '/tables';
    if (!file_exists($tables_dir)) {
        mkdir($tables_dir, 0755, true);
    }
    
    try {
        $configs_stmt = $pdo->prepare("
            SELECT * FROM result_configurations 
            WHERE event_id = ? AND view_type = 'summary_table' AND status = 'public'
            ORDER BY name
        ");
        $configs_stmt->execute([$event_id]);
        $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($configs as $config) {
            error_log("Generating HTML table for config ID: " . $config['id'] . " - " . $config['name']);
            
            try {
                // Build API URL with HTML format
                $api_url = buildSummaryTableHTMLUrl($config);
                
                error_log("Calling API for HTML: " . $api_url);
                
                $context = stream_context_create([
                    'http' => [
                        'timeout' => 60,
                        'ignore_errors' => true
                    ]
                ]);
                
                $html_content = @file_get_contents($api_url, false, $context);
                
                if ($html_content === false) {
                    error_log("Failed to fetch HTML table for config " . $config['id']);
                    $html_content = '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Error</title></head><body><p>Could not generate table</p></body></html>';
                } else {
                    error_log("HTML table generated successfully, length: " . strlen($html_content) . " bytes");
                }
                
                // Save HTML file with config-specific name
                $filename = 'summary_table_' . $config['id'] . '.html';
                $file = $tables_dir . '/' . $filename;
                file_put_contents($file, $html_content);
                $files[] = ['local' => $file, 'remote' => 'tables/' . $filename];
                
                error_log("Successfully generated " . $filename);
            } catch (Exception $e) {
                error_log("Error generating HTML table for config " . $config['id'] . ": " . $e->getMessage());
            }
        }
        
    } catch (Exception $e) {
        error_log("Error generating summary table HTMLs: " . $e->getMessage());
    }
    
    return $files;
}

// Build HTML summary table API URL from configuration
function buildSummaryTableHTMLUrl($config) {
    $settings = json_decode($config['configuration'], true);
    
    $params = [
        'event_id' => $config['event_id'],
        'format' => 'html',  // Changed to HTML
        'styling' => 'partial'
    ];
    
    // Heat filter - handle both single heat and multiple heats
    if (isset($settings['heat_number']) && $settings['heat_number']) {
        if (is_array($settings['heat_number'])) {
            // Multiple heats - use heat_run_filter format
            $heat_run_filter = [];
            foreach ($settings['heat_number'] as $heat) {
                $heat_run_filter[$heat] = [1, 2, 3, 4, 5]; // All runs
            }
            $params['heat_run_filter'] = json_encode($heat_run_filter);
        } else {
            // Single heat
            $params['heat_number'] = $settings['heat_number'];
        }
    }
    
    // Column visibility
    $params['show_rank'] = isset($settings['showRank']) && $settings['showRank'] ? 'true' : 'false';
    $params['show_bib'] = isset($settings['showBib']) && $settings['showBib'] ? 'true' : 'false';
    $params['show_participant'] = isset($settings['showParticipant']) && $settings['showParticipant'] ? 'true' : 'false';
    $params['show_category'] = isset($settings['showCategory']) && $settings['showCategory'] ? 'true' : 'false';
    $params['show_club'] = isset($settings['showClub']) && $settings['showClub'] ? 'true' : 'false';
    $params['show_gender'] = isset($settings['showGender']) && $settings['showGender'] ? 'true' : 'false';
    $params['show_fis_code'] = isset($settings['showFisCode']) && $settings['showFisCode'] ? 'true' : 'false';
    $params['show_licence_code'] = isset($settings['showLicenceCode']) && $settings['showLicenceCode'] ? 'true' : 'false';
    $params['show_country'] = isset($settings['showCountry']) && $settings['showCountry'] ? 'true' : 'false';
    $params['show_birth_year'] = isset($settings['showBirthYear']) && $settings['showBirthYear'] ? 'true' : 'false';
    $params['show_age'] = isset($settings['showAge']) && $settings['showAge'] ? 'true' : 'false';
    $params['show_runs'] = isset($settings['showRuns']) && $settings['showRuns'] ? 'true' : 'false';
    $params['show_judges'] = isset($settings['showJudges']) && $settings['showJudges'] ? 'true' : 'false';
    $params['show_control_points'] = isset($settings['showControlPoints']) && $settings['showControlPoints'] ? 'true' : 'false';
    $params['show_figures'] = isset($settings['showFigures']) && $settings['showFigures'] ? 'true' : 'false';
    $params['show_event_settings'] = isset($settings['showEventSettings']) && $settings['showEventSettings'] ? 'true' : 'false';
    $params['show_heat_best'] = isset($settings['showHeatBest']) && $settings['showHeatBest'] ? 'true' : 'false';
    $params['show_heat_average'] = isset($settings['showHeatAverage']) && $settings['showHeatAverage'] ? 'true' : 'false';
    $params['show_overall_best'] = isset($settings['showOverallBest']) && $settings['showOverallBest'] ? 'true' : 'false';
    $params['show_highest_average'] = isset($settings['showHighestAverage']) && $settings['showHighestAverage'] ? 'true' : 'false';
    
    // Heat direction
    if (isset($settings['heatDirection'])) {
        $params['heat_direction'] = $settings['heatDirection'];
    }
    
    // Sorting
    if (isset($settings['sortBy'])) {
        $params['sort'] = $settings['sortBy'];
    }
    if (isset($settings['sortDirection'])) {
        $params['sort_direction'] = $settings['sortDirection'];
    }
    
    // Gender filter
    if (isset($settings['gender'])) {
        $params['gender'] = $settings['gender'];
    } elseif (isset($settings['gender_filter_main'])) {
        $params['gender'] = $settings['gender_filter_main'];
    }
    
    // Category filter
    if (isset($settings['category_filter']) && $settings['category_filter'] !== 'all') {
        $params['category'] = $settings['category_filter'];
    }
    
    // Column renames
    if (isset($settings['columnRenames']) && is_array($settings['columnRenames']) && count($settings['columnRenames']) > 0) {
        foreach ($settings['columnRenames'] as $key => $value) {
            $params['rename_' . $key] = $value;
        }
    }
    
    return resolveUrl('/v2/api/summary_table_api.php') . '?' . http_build_query($params);
}

// Generate start list HTML files for publishing
function generateStartListHTMLs($event_id, $temp_dir, $pdo) {
    $files = [];
    
    // Create tables directory for HTML files
    $tables_dir = $temp_dir . '/tables';
    if (!file_exists($tables_dir)) {
        mkdir($tables_dir, 0755, true);
    }
    
    try {
        $configs_stmt = $pdo->prepare("
            SELECT * FROM result_configurations 
            WHERE event_id = ? AND view_type = 'start_list' AND status = 'public'
            ORDER BY name
        ");
        $configs_stmt->execute([$event_id]);
        $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($configs as $config) {
            error_log("Generating start list HTML for config ID: " . $config['id'] . " - " . $config['name']);
            
            try {
                // Build API URL with HTML format
                $api_url = buildStartListHTMLUrl($config);
                
                error_log("Calling API for HTML: " . $api_url);
                
                $context = stream_context_create([
                    'http' => [
                        'timeout' => 60,
                        'ignore_errors' => true
                    ]
                ]);
                
                $html_content = @file_get_contents($api_url, false, $context);
                
                if ($html_content === false) {
                    error_log("Failed to fetch start list HTML for config " . $config['id']);
                    $html_content = '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Error</title></head><body><p>Could not generate start list</p></body></html>';
                } else {
                    error_log("Start list HTML generated successfully, length: " . strlen($html_content) . " bytes");
                }
                
                // Save HTML file with config-specific name
                $filename = 'start_list_' . $config['id'] . '.html';
                $file = $tables_dir . '/' . $filename;
                file_put_contents($file, $html_content);
                $files[] = ['local' => $file, 'remote' => 'tables/' . $filename];
                
                error_log("Successfully generated " . $filename);
            } catch (Exception $e) {
                error_log("Error generating start list HTML for config " . $config['id'] . ": " . $e->getMessage());
            }
        }
        
    } catch (Exception $e) {
        error_log("Error generating start list HTMLs: " . $e->getMessage());
    }
    
    return $files;
}

// Build HTML start list API URL from configuration
function buildStartListHTMLUrl($config) {
    $settings = json_decode($config['configuration'], true);
    
    $params = [
        'event_id' => $config['event_id'],
        'format' => 'html',
        'styling' => 'partial'
    ];
    
    // Heat filter
    if (isset($settings['heat_number']) && $settings['heat_number']) {
        if (is_array($settings['heat_number'])) {
            // Multiple heats
            $params['heat_numbers'] = implode(',', $settings['heat_number']);
        } else {
            // Single heat
            $params['heat_number'] = $settings['heat_number'];
        }
    }
    
    // Column visibility
    $params['show_rank'] = isset($settings['showRank']) && $settings['showRank'] ? 'true' : 'false';
    $params['show_bib'] = isset($settings['showBib']) && $settings['showBib'] ? 'true' : 'false';
    $params['show_participant'] = isset($settings['showParticipant']) && $settings['showParticipant'] ? 'true' : 'false';
    $params['show_category'] = isset($settings['showCategory']) && $settings['showCategory'] ? 'true' : 'false';
    $params['show_club'] = isset($settings['showClub']) && $settings['showClub'] ? 'true' : 'false';
    $params['show_gender'] = isset($settings['showGender']) && $settings['showGender'] ? 'true' : 'false';
    $params['show_fis_code'] = isset($settings['showFisCode']) && $settings['showFisCode'] ? 'true' : 'false';
    $params['show_licence_code'] = isset($settings['showLicenceCode']) && $settings['showLicenceCode'] ? 'true' : 'false';
    $params['show_country'] = isset($settings['showCountry']) && $settings['showCountry'] ? 'true' : 'false';
    $params['show_birth_year'] = isset($settings['showBirthYear']) && $settings['showBirthYear'] ? 'true' : 'false';
    $params['show_age'] = isset($settings['showAge']) && $settings['showAge'] ? 'true' : 'false';
    
    // Gender filter
    if (isset($settings['gender'])) {
        $params['gender'] = $settings['gender'];
    } elseif (isset($settings['gender_filter_main'])) {
        $params['gender'] = $settings['gender_filter_main'];
    }
    
    // Category filter
    if (isset($settings['category_filter']) && $settings['category_filter'] !== 'all') {
        $params['category'] = $settings['category_filter'];
    }
    
    // Sorting
    if (isset($settings['sortBy'])) {
        $params['sort'] = $settings['sortBy'];
    }
    if (isset($settings['sortDirection'])) {
        $params['sort_direction'] = $settings['sortDirection'];
    }
    
    // Column renames
    if (isset($settings['columnRenames']) && is_array($settings['columnRenames']) && count($settings['columnRenames']) > 0) {
        foreach ($settings['columnRenames'] as $key => $value) {
            $params['rename_' . $key] = $value;
        }
    }
    
    return resolveUrl('/v2/api/start_list_api.php') . '?' . http_build_query($params);
}

// Generate start list API files
function generateStartListAPIs($event_id, $api_dir, $pdo) {
    $files = [];
    
    try {
        // First, generate start list JSON for each saved configuration
        $configs_stmt = $pdo->prepare("
            SELECT * FROM result_configurations 
            WHERE event_id = ? AND view_type = 'start_list' AND status = 'public'
            ORDER BY name
        ");
        $configs_stmt->execute([$event_id]);
        $configs = $configs_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($configs as $config) {
            error_log("Generating start list for config ID: " . $config['id'] . " - " . $config['name']);
            
            try {
                // Build full API URL from configuration settings
                $api_url = buildStartListAPIUrl($config);
                
                error_log("Calling API: " . $api_url);
                
                $context = stream_context_create([
                    'http' => [
                        'timeout' => 60,
                        'ignore_errors' => true
                    ]
                ]);
                
                $json_content = @file_get_contents($api_url, false, $context);
                
                if ($json_content === false) {
                    error_log("Failed to fetch start list data for config " . $config['id']);
                    $start_list_data = ['success' => false, 'message' => 'Could not fetch start list data'];
                } else {
                    // Log response length for debugging
                    error_log("Response length: " . strlen($json_content) . " bytes");
                    
                    // Check if response looks like valid JSON
                    if (empty(trim($json_content))) {
                        error_log("Empty response for config " . $config['id']);
                        $start_list_data = ['success' => false, 'message' => 'Empty response from API'];
                    } else {
                        $start_list_data = json_decode($json_content, true);
                        if ($start_list_data === null) {
                            error_log("Invalid JSON for config " . $config['id'] . ". JSON Error: " . json_last_error_msg());
                            error_log("Response preview: " . substr($json_content, 0, 500));
                            $start_list_data = ['success' => false, 'message' => 'Invalid JSON data: ' . json_last_error_msg()];
                        }
                    }
                }
                
                $file = $api_dir . '/start_list_' . $config['id'] . '.json';
                file_put_contents($file, json_encode($start_list_data, JSON_PRETTY_PRINT));
                $files[] = ['local' => $file, 'remote' => 'api/start_list_' . $config['id'] . '.json'];
                
                error_log("Successfully generated start_list_" . $config['id'] . ".json");
            } catch (Exception $e) {
                error_log("Error generating start list for config " . $config['id'] . ": " . $e->getMessage());
            }
        }
        
        // Also generate overall start list
        $participants_stmt = $pdo->prepare("
            SELECT 
                ep.bib_number, 
                p.first_name, 
                p.last_name,
                p.club, 
                p.country,
                p.photo,
                ec.category_name as category,
                ep.heat_number
            FROM event_participants ep
            JOIN participants p ON p.id = ep.participant_id
            LEFT JOIN event_categories ec ON ec.id = ep.category_id
            WHERE ep.event_id = ?
            ORDER BY ep.heat_number, ep.sort_order, ep.bib_number
        ");
        $participants_stmt->execute([$event_id]);
        $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $start_list_data = [
            'success' => true,
            'data' => ['participants' => $participants],
            'heat_number' => null,
            'type' => 'start_list'
        ];
        
        $file = $api_dir . '/start_list_all.json';
        file_put_contents($file, json_encode($start_list_data, JSON_PRETTY_PRINT));
        $files[] = ['local' => $file, 'remote' => 'api/start_list_all.json'];
        
        // Generate start lists by heat
        $heats_stmt = $pdo->prepare("
            SELECT DISTINCT heat_number 
            FROM event_participants 
            WHERE event_id = ? 
            ORDER BY heat_number
        ");
        $heats_stmt->execute([$event_id]);
        $heats = $heats_stmt->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($heats as $heat_number) {
            $heat_participants = array_filter($participants, function($p) use ($heat_number) {
                return $p['heat_number'] == $heat_number;
            });
            
            $heat_start_list_data = [
                'success' => true,
                'data' => ['participants' => array_values($heat_participants)],
                'heat_number' => $heat_number,
                'type' => 'start_list'
            ];
            
            $file = $api_dir . '/start_list_heat_' . $heat_number . '.json';
            file_put_contents($file, json_encode($heat_start_list_data, JSON_PRETTY_PRINT));
            $files[] = ['local' => $file, 'remote' => 'api/start_list_heat_' . $heat_number . '.json'];
        }
        
    } catch (Exception $e) {
        error_log("Error generating start list APIs: " . $e->getMessage());
    }
    
    return $files;
}

// Generate the static API dispatcher JavaScript
function generateStaticAPIDispatcher($event_id) {
    return <<<JS
// Static Dashboard JavaScript for Event ID: {$event_id}
let currentViewType = 'summary_table';
let currentConfigId = null;
let configurations = [];

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    loadConfigurations();
});

// Load saved configurations from API
async function loadConfigurations() {
    try {
        const response = await fetch('api/configurations.json?t=' + Date.now());
        const data = await response.json();
        
        if (data.success && data.configurations) {
            configurations = data.configurations;
            displayConfigurations(configurations);
            
            // Auto-load first configuration if available
            const firstConfig = configurations.find(c => c.view_type === currentViewType);
            if (firstConfig) {
                loadConfiguration(firstConfig.id);
            }
        }
    } catch (error) {
        console.error('Error loading configurations:', error);
        document.getElementById('savedReports').innerHTML = '<div class="text-center text-danger small py-3"><i class="fas fa-exclamation-triangle"></i> Error loading reports</div>';
    }
}

// Display configurations in sidebar
function displayConfigurations(configs) {
    const container = document.getElementById('savedReports');
    
    // Filter by current view type
    const filtered = configs.filter(c => c.view_type === currentViewType);
    
    if (filtered.length === 0) {
        container.innerHTML = '<div class="text-center text-muted small py-3">No saved reports</div>';
        return;
    }
    
    let html = '';
    filtered.forEach(config => {
        const isActive = config.id === currentConfigId ? 'active' : '';
        html += '<button type="button" class="list-group-item list-group-item-action config-button d-flex justify-content-between align-items-center ' + isActive + '" ';
        html += 'data-config-id="' + config.id + '" ';
        html += 'onclick="loadConfiguration(' + config.id + ')" ';
        html += 'title="Click to load: ' + config.name + '">';
        html += '<span class="me-2 text-start"><i class="fas fa-file-alt me-2"></i>' + config.name + '</span>';
        html += '<span class="badge bg-light text-dark">#' + config.id + '</span>';
        html += '</button>';
    });
    
    container.innerHTML = html;
}

// Load a specific configuration
async function loadConfiguration(configId) {
    currentConfigId = configId;
    
    // Find the configuration to check its view type
    const config = configurations.find(c => c.id === configId);
    if (!config) {
        console.error('Configuration not found:', configId);
        return;
    }
    
    // Update active state in sidebar
    document.querySelectorAll('.config-button').forEach(btn => {
        if (parseInt(btn.dataset.configId) === configId) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
    
    // Show loading state
    const contentDisplay = document.getElementById('contentDisplay');
    contentDisplay.innerHTML = '<div class="text-center py-5"><div class="spinner-border text-primary" role="status"></div><p class="mt-2 text-muted">Loading report...</p></div>';
    
    try {
        let htmlUrl;
        
        if (config.view_type === 'summary_table') {
            // Fetch summary table HTML
            htmlUrl = 'tables/summary_table_' + configId + '.html?t=' + Date.now();
            const response = await fetch(htmlUrl);
            
            if (!response.ok) {
                throw new Error('Failed to fetch summary table HTML (HTTP ' + response.status + ')');
            }
            
            const html = await response.text();
            if (!html || html.trim() === '') {
                throw new Error('Empty HTML response');
            }
            
            // Inject the HTML directly and initialize DataTables
            contentDisplay.innerHTML = html;
            
            setTimeout(function() {
                if (jQuery && jQuery.fn.DataTable) {
                    // Destroy existing DataTable instances if present - check each table individually
                    jQuery('.data-table').each(function() {
                        if (jQuery.fn.DataTable.isDataTable(this)) {
                            jQuery(this).DataTable().destroy();
                        }
                    });
                    
                    // Initialize DataTables on all tables with data-table class
                    jQuery('.data-table').each(function() {
                        jQuery(this).DataTable({
                            responsive: true,
                            pageLength: 25,
                            lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                            order: [[0, 'asc']],
                            language: {
                                search: "Search participants:",
                                lengthMenu: "Show _MENU_ participants",
                                info: "Showing _START_ to _END_ of _TOTAL_ participants",
                                infoEmpty: "No participants found",
                                infoFiltered: "(filtered from _MAX_ total participants)"
                            }
                        });
                    });
                }
            }, 100);
            
        } else if (config.view_type === 'start_list') {
            // Fetch start list HTML
            htmlUrl = 'tables/start_list_' + configId + '.html?t=' + Date.now();
            const response = await fetch(htmlUrl);
            
            if (!response.ok) {
                throw new Error('Failed to fetch start list HTML (HTTP ' + response.status + ')');
            }
            
            const html = await response.text();
            if (!html || html.trim() === '') {
                throw new Error('Empty HTML response');
            }
            
            // Inject the HTML directly and initialize DataTables
            contentDisplay.innerHTML = html;
            
            setTimeout(function() {
                if (jQuery && jQuery.fn.DataTable) {
                    // Destroy existing DataTable instances if present - check each table individually
                    jQuery('.data-table').each(function() {
                        if (jQuery.fn.DataTable.isDataTable(this)) {
                            jQuery(this).DataTable().destroy();
                        }
                    });
                    
                    // Initialize DataTables on all tables with data-table class
                    jQuery('.data-table').each(function() {
                        jQuery(this).DataTable({
                            responsive: true,
                            pageLength: 25,
                            lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                            order: [[0, 'asc']],
                            language: {
                                search: "Search participants:",
                                lengthMenu: "Show _MENU_ participants",
                                info: "Showing _START_ to _END_ of _TOTAL_ participants",
                                infoEmpty: "No participants found",
                                infoFiltered: "(filtered from _MAX_ total participants)"
                            }
                        });
                    });
                }
            }, 100);
            
        } else {
            contentDisplay.innerHTML = '<div class="alert alert-info"><i class="fas fa-info-circle me-2"></i>Unsupported view type: ' + config.view_type + '</div>';
        }
    } catch (error) {
        console.error('Error loading configuration:', error);
        contentDisplay.innerHTML = '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>Error loading report HTML. Please try again.</div>';
    }
}

// Display summary table in content area
function displaySummaryTable(data) {
    const contentDisplay = document.getElementById('contentDisplay');
    
    // Use the exact HTML from the API response if available
    if (data.table_badge_colors || data.table_head || data.table_body) {
        let html = '';
        
        // Add badge colors style if present
        if (data.table_badge_colors) {
            html += '<style>' + data.table_badge_colors + '</style>';
        }
        
        // Add stacked runs styles
        html += '<style>';
        html += 'table.data-table td.heat-stacked { vertical-align: top !important; }';
        html += 'table.data-table .stacked-runs { display: flex !important; flex-direction: column !important; gap: 4px !important; min-width: 80px !important; }';
        html += 'table.data-table .stacked-runs .run-item { display: flex !important; align-items: center !important; padding: 2px 4px !important; border-bottom: 1px solid #eee !important; }';
        html += 'table.data-table .stacked-runs .run-item:last-child { border-bottom: none !important; }';
        html += 'table.data-table .stacked-runs .run-label { font-size: 0.75rem !important; font-weight: bold !important; color: #666 !important; min-width: 25px !important; }';
        html += 'table.data-table .stacked-runs .run-score { font-size: 0.85rem !important; }';
        html += 'table.data-table .stacked-runs .judge-breakdown { display: inline-flex !important; gap: 2px !important; flex-wrap: wrap !important; }';
        html += 'tr.diversity_rules_not_meet { background-color: #fff3cd !important; }';
        html += 'tr.diversity_rules_not_meet td { opacity: 0.8; }';
        html += 'tr.diversity_rules_not_meet .css-rank { color: #856404 !important; font-weight: bold; }';
        html += '</style>';
        
        const hasFullTableMarkup = data.table_body && data.table_body.includes('<table');
        
        if (hasFullTableMarkup && !data.table_head) {
            // API already returned fully-rendered table(s) (split heat view). Use as-is.
            html += data.table_body;
        } else {
            // Build single table wrapper
            html += '<div class="heat-table-wrapper mb-4">';
            
            if (data.filter_summary) {
                html += '<div class="heat-table-header bg-white text-dark p-2 rounded-top fw-bold">';
                html += '<h5 class="mb-0"><i class="fas fa-fire me-2"></i>' + data.filter_summary + '</h5>';
                html += '</div>';
            }
            
            html += '<div class="table-responsive">';
            html += '<table class="table table-striped table-hover data-table">';
            
            if (data.table_head) {
                html += '<thead>' + data.table_head + '</thead>';
            }
            
            if (data.table_body) {
                html += '<tbody>' + data.table_body + '</tbody>';
            }
            
            html += '</table>';
            html += '</div>';
            
            if (data.participant_count) {
                html += '<div class="heat-table-footer bg-light p-2 rounded-bottom">';
                html += '<small class="text-muted"><i class="fas fa-users me-1"></i>' + data.participant_count + ' Participants</small>';
                html += '</div>';
            }
            
            html += '</div>';
        }
        
        contentDisplay.innerHTML = html;
        
        // Initialize DataTables on the loaded table
        setTimeout(function() {
            if (jQuery && jQuery.fn.DataTable) {
                jQuery('.data-table').DataTable({
                    responsive: true,
                    pageLength: 25,
                    lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                    order: [[0, 'asc']],
                    language: {
                        search: "Search participants:",
                        lengthMenu: "Show _MENU_ participants",
                        info: "Showing _START_ to _END_ of _TOTAL_ participants",
                        infoEmpty: "No participants found",
                        infoFiltered: "(filtered from _MAX_ total participants)"
                    }
                });
            }
        }, 100);
    } else {
        contentDisplay.innerHTML = '<div class="alert alert-info"><i class="fas fa-info-circle me-2"></i>No table data available</div>';
    }
}

// Display start list in content area
function displayStartList(data, configName) {
    const contentDisplay = document.getElementById('contentDisplay');
    
    if (!data.table_head || !data.table_body) {
        contentDisplay.innerHTML = '<div class="alert alert-info"><i class="fas fa-info-circle me-2"></i>No start list data available</div>';
        return;
    }
    
    let html = '';
    
    // Add badge colors style if present
    if (data.table_badge_colors) {
        html += '<style>' + data.table_badge_colors + '</style>';
    }
    
    // Build the table HTML
    html += '<div class="heat-table-wrapper mb-4">';
    
    html += '<div class="heat-table-header bg-primary text-white p-2 rounded-top fw-bold">';
    html += '<h5 class="mb-0"><i class="fas fa-list-ol me-2"></i>' + (data.filter_summary || configName) + '</h5>';
    html += '</div>';
    
    html += '<div class="table-responsive">';
    html += '<table class="table table-striped table-hover data-table">';
    
    html += '<thead>' + data.table_head + '</thead>';
    html += '<tbody>' + data.table_body + '</tbody>';
    
    html += '</table>';
    html += '</div>';
    
    if (data.participant_count) {
        html += '<div class="heat-table-footer bg-light p-2 rounded-bottom">';
        html += '<small class="text-muted"><i class="fas fa-users me-1"></i>' + data.participant_count + ' Participants</small>';
        html += '</div>';
    }
    
    html += '</div>';
    
    contentDisplay.innerHTML = html;
    
    // Initialize DataTables on the loaded table
    setTimeout(function() {
        if (jQuery && jQuery.fn.DataTable) {
            jQuery('.data-table').DataTable({
                responsive: true,
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                order: [[0, 'asc']],
                language: {
                    search: "Search participants:",
                    lengthMenu: "Show _MENU_ participants",
                    info: "Showing _START_ to _END_ of _TOTAL_ participants",
                    infoEmpty: "No participants found",
                    infoFiltered: "(filtered from _MAX_ total participants)"
                }
            });
        }
    }, 100);
}

// Switch view type
function switchViewType(viewType) {
    currentViewType = viewType;
    currentConfigId = null;
    
    // Update button states
    const buttons = document.querySelectorAll('#viewTypeButtons .btn');
    buttons.forEach(btn => {
        btn.classList.remove('btn-primary', 'active');
        btn.classList.add('btn-outline-primary');
    });
    
    const activeBtn = document.querySelector('#viewTypeButtons .btn[onclick*="' + viewType + '"]');
    if (activeBtn) {
        activeBtn.classList.remove('btn-outline-primary');
        activeBtn.classList.add('btn-primary', 'active');
    }
    
    // Reload configurations for new view type
    displayConfigurations(configurations);
    
    // Auto-load first config of new type
    const firstConfig = configurations.find(c => c.view_type === viewType);
    if (firstConfig) {
        loadConfiguration(firstConfig.id);
    } else {
        document.getElementById('contentDisplay').innerHTML = '<div class="alert alert-info mb-0"><i class="fas fa-info-circle me-2"></i>No saved reports for this view type</div>';
    }
}

// Open configuration in new tab (for middle-click)
function openConfigInNewTab(configId) {
    window.open(window.location.href + '?config=' + configId, '_blank');
}

JS;
}

// Generate Participant Photos - Copy participant photos to publish directory
function generateParticipantPhotos($event_id, $temp_dir, $pdo) {
    error_log("Starting generateParticipantPhotos for event $event_id in $temp_dir");
    $files = [];
    
    try {
        // Get all participants for this event
        $participants_stmt = $pdo->prepare("
            SELECT p.id, p.name, p.photo_filename 
            FROM participants p 
            WHERE p.event_id = ? AND p.photo_filename IS NOT NULL AND p.photo_filename != ''
        ");
        $participants_stmt->execute([$event_id]);
        $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($participants)) {
            error_log("No participants with photos found for event $event_id");
            return $files;
        }
        
        // Create photos directory in temp dir
        $photos_dir = $temp_dir . '/photos';
        if (!file_exists($photos_dir)) {
            mkdir($photos_dir, 0755, true);
        }
        
        $copied_count = 0;
        
        foreach ($participants as $participant) {
            $photo_filename = $participant['photo_filename'];
            
            // Try different possible locations for the photo
            $possible_sources = [
                __DIR__ . '/../uploads/' . $photo_filename,
                __DIR__ . '/../uploads/photos/' . $photo_filename,
                __DIR__ . '/../photos/' . $photo_filename,
                __DIR__ . '/../' . $photo_filename
            ];
            
            $source_found = false;
            foreach ($possible_sources as $source_path) {
                if (file_exists($source_path)) {
                    $destination_path = $photos_dir . '/' . $photo_filename;
                    
                    if (copy($source_path, $destination_path)) {
                        $files[] = [
                            'local' => $destination_path,
                            'remote' => 'photos/' . $photo_filename
                        ];
                        $copied_count++;
                        error_log("Copied photo: $photo_filename for participant: {$participant['name']}");
                    } else {
                        error_log("Failed to copy photo: $photo_filename");
                    }
                    $source_found = true;
                    break;
                }
            }
            
            if (!$source_found) {
                error_log("Photo not found: $photo_filename for participant: {$participant['name']}");
            }
        }
        
        error_log("Copied $copied_count out of " . count($participants) . " participant photos");
        
    } catch (Exception $e) {
        error_log("Error in generateParticipantPhotos: " . $e->getMessage());
        throw $e;
    }
    
    return $files;
}

// Generate JSON Data - Legacy support
function generateJSONData($event_id, $temp_dir, $pdo) {
    $files = [];
    
    try {
        // Generate the same API files as the static dashboard
        $api_files = generateStaticAPIFiles($event_id, $temp_dir, $pdo);
        $files = array_merge($files, $api_files);
        
        error_log("Generated " . count($files) . " JSON data files for event $event_id");
        
    } catch (Exception $e) {
        error_log("Error generating JSON data: " . $e->getMessage());
    }
    
    return $files;
}
