<?php
/**
 * Roles & Permissions Management Panel
 * Simple and effective permission management interface
 */

session_start();
require_once '../includes/db.php';
require_once '../includes/permissions.php';

// Require super_admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'super_admin') {
    header('Location: ../index.php');
    exit;
}

$current_user_id = $_SESSION['user_id'];
$page_title = "Roles & Permissions Manager";

// Get all roles
$all_roles = ['super_admin', 'admin', 'event_organizer', 'office', 'head_judge', 'judge', 'media', 'participant'];

// Get all permissions grouped by category
$permissions_grouped = getAllPermissionsGrouped();

// Get permission matrix (role => [permission_ids])
$permission_matrix = [];
foreach ($all_roles as $role) {
    $role_perms = getRolePermissions($role);
    $permission_matrix[$role] = array_column($role_perms, 'id');
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <?php include_once '../includes/stylesheets.php'; ?>
    
    <style>
        .permission-matrix {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .matrix-header {
            background: #667eea;
            color: white;
            padding: 20px;
        }
        .matrix-table {
            width: 100%;
            margin: 0;
        }
        .matrix-table th {
            background: #f8f9fa;
            font-weight: 600;
            padding: 12px 8px;
            text-align: center;
            font-size: 0.85rem;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        .matrix-table td {
            padding: 10px 8px;
            text-align: center;
            border-bottom: 1px solid #e9ecef;
        }
        .matrix-table tr:hover {
            background: #f8f9fa;
        }
        .permission-name {
            text-align: left !important;
            font-weight: 500;
            min-width: 200px;
        }
        .permission-description {
            font-size: 0.8rem;
            color: #6c757d;
            font-weight: 400;
        }
        .category-header {
            background: #e9ecef;
            font-weight: 700;
            text-align: left !important;
            padding: 12px 16px !important;
            color: #495057;
            font-size: 0.95rem;
        }
        .role-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            white-space: nowrap;
        }
        .role-super_admin { background: #dc3545; color: white; }
        .role-admin { background: #fd7e14; color: white; }
        .role-event_organizer { background: #0dcaf0; color: white; }
        .role-office { background: #20c997; color: white; }
        .role-head_judge { background: #6f42c1; color: white; }
        .role-judge { background: #0d6efd; color: white; }
        .role-media { background: #6c757d; color: white; }
        .role-participant { background: #adb5bd; color: white; }
        .permission-toggle {
            transform: scale(1.2);
            cursor: pointer;
        }
        .permission-toggle:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        .action-buttons {
            position: sticky;
            top: 76px;
            z-index: 100;
            background: white;
            padding: 15px;
            border-bottom: 2px solid #e9ecef;
            margin-bottom: 0;
        }
        .table-wrapper {
            max-height: calc(100vh - 250px);
            overflow-y: auto;
        }
        .save-indicator {
            display: none;
            position: fixed;
            top: 100px;
            right: 20px;
            z-index: 1000;
        }
    </style>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>
     <!-- style skin holder -->
    <?php include_once '../includes/stylesheets.php'; ?>

<div class="container-fluid container-StyleScore p-4">
        <!-- Event Actions Menu -->
    <div class="row mb-3">
        <div class="col-12">
            <?php $use_localStorage = true; $show_header = false; include 'event_actions_menu.php'; ?>
        </div>
    </div>

    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-1">
                <i class="fas fa-shield-check text-primary me-2"></i>Roles & Permissions Manager
            </h1>
            <p class="text-muted mb-0">Manage system-wide permissions for each user role</p>
        </div>
    </div>

    <!-- Save Indicator -->
    <div class="alert alert-success save-indicator" id="saveIndicator">
        <i class="fas fa-check-circle me-2"></i>Changes saved successfully!
    </div>

    <div class="permission-matrix">
        <div class="matrix-header">
            <h5 class="mb-1">
                <i class="fas fa-table-cells me-2"></i>
                Permission Matrix
            </h5>
            <p class="mb-0 opacity-75">Click checkboxes to toggle permissions. Changes save automatically.</p>
        </div>

            <div class="action-buttons">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <button class="btn btn-primary btn-sm me-2" onclick="openAddPermissionModal()">
                        <i class="fas fa-plus me-1"></i>
                        Add New Permission
                    </button>
                    <button class="btn btn-outline-secondary btn-sm" onclick="location.reload()">
                        <i class="fas fa-arrows-rotate me-1"></i>
                        Refresh
                    </button>
                </div>
                <div class="col-md-6 text-end">
                    <small class="text-muted">
                        <i class="fas fa-info-circle me-1"></i>
                        Click checkboxes to toggle permissions. Changes save automatically.
                    </small>
                </div>
            </div>
        </div>

            <div class="table-wrapper">
                <table class="matrix-table">
                    <thead>
                        <tr>
                            <th class="permission-name">Permission</th>
                            <?php foreach ($all_roles as $role): ?>
                                <th>
                                    <span class="role-badge role-<?= $role ?>">
                                        <?= str_replace('_', ' ', $role) ?>
                                    </span>
                                </th>
                            <?php endforeach; ?>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($permissions_grouped as $category => $permissions): ?>
                            <tr>
                                <td colspan="<?= count($all_roles) + 2 ?>" class="category-header">
                                    <i class="fas fa-folder me-2"></i>
                                    <?= ucwords(str_replace('_', ' ', $category)) ?>
                                </td>
                            </tr>
                            <?php foreach ($permissions as $perm): ?>
                                <tr data-permission-id="<?= $perm['id'] ?>">
                                    <td class="permission-name">
                                        <div><?= htmlspecialchars($perm['display_name']) ?></div>
                                        <div class="permission-description">
                                            <?= htmlspecialchars($perm['description']) ?>
                                        </div>
                                        <small class="text-muted">
                                            <code><?= htmlspecialchars($perm['name']) ?></code>
                                        </small>
                                    </td>
                                    <?php foreach ($all_roles as $role): ?>
                                        <td>
                                            <?php 
                                            $has_permission = in_array($perm['id'], $permission_matrix[$role] ?? []);
                                            $is_locked = ($role === 'super_admin'); // super_admin always has all
                                            ?>
                                            <input type="checkbox" 
                                                   class="form-check-input permission-toggle" 
                                                   data-permission-id="<?= $perm['id'] ?>"
                                                   data-role="<?= $role ?>"
                                                   <?= $has_permission ? 'checked' : '' ?>
                                                   <?= $is_locked ? 'disabled' : '' ?>
                                                   onchange="togglePermission(this)">
                                        </td>
                                    <?php endforeach; ?>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-outline-primary" 
                                                    onclick="editPermission(<?= $perm['id'] ?>, '<?= htmlspecialchars($perm['name']) ?>', '<?= htmlspecialchars($perm['display_name']) ?>', '<?= htmlspecialchars($perm['description']) ?>', '<?= htmlspecialchars($perm['category']) ?>')"
                                                    title="Edit">
                                                <i data-lucide="edit" style="width:14px;height:14px;"></i>
                                            </button>
                                            <button class="btn btn-outline-danger" 
                                                    onclick="deletePermission(<?= $perm['id'] ?>, '<?= htmlspecialchars($perm['display_name']) ?>')"
                                                    title="Delete">
                                                <i data-lucide="trash-2" style="width:14px;height:14px;"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mt-4 mb-4">
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <h3 class="mb-0"><?= count($all_roles) ?></h3>
                        <small class="text-muted">Total Roles</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <h3 class="mb-0">
                            <?php
                            $total_perms = 0;
                            foreach ($permissions_grouped as $perms) {
                                $total_perms += count($perms);
                            }
                            echo $total_perms;
                            ?>
                        </h3>
                        <small class="text-muted">Total Permissions</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <h3 class="mb-0"><?= count($permissions_grouped) ?></h3>
                        <small class="text-muted">Categories</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <h3 class="mb-0">
                            <?php
                            $total_assignments = 0;
                            foreach ($permission_matrix as $role_perms) {
                                $total_assignments += count($role_perms);
                            }
                            echo $total_assignments;
                            ?>
                        </h3>
                        <small class="text-muted">Active Assignments</small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Permission Modal -->
    <div class="modal fade" id="permissionModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="permissionModalTitle">Add New Permission</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="permissionForm">
                    <input type="hidden" id="permission_id" name="permission_id">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Permission Name (Code) *</label>
                            <input type="text" class="form-control" id="permission_name" name="name" required 
                                   pattern="[a-z_]+" placeholder="e.g., create_events">
                            <small class="text-muted">Lowercase letters and underscores only</small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Display Name *</label>
                            <input type="text" class="form-control" id="permission_display_name" name="display_name" required
                                   placeholder="e.g., Create Events">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="permission_description" name="description" rows="2"
                                      placeholder="Brief description of what this permission allows"></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Category *</label>
                            <select class="form-select" id="permission_category" name="category" required>
                                <?php foreach (array_keys($permissions_grouped) as $cat): ?>
                                    <option value="<?= $cat ?>"><?= ucwords(str_replace('_', ' ', $cat)) ?></option>
                                <?php endforeach; ?>
                                <option value="custom">Custom Category...</option>
                            </select>
                        </div>
                        <div class="mb-3" id="custom_category_field" style="display: none;">
                            <label class="form-label">Custom Category Name</label>
                            <input type="text" class="form-control" id="custom_category" placeholder="e.g., reports">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save Permission</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle permission for a role
        function togglePermission(checkbox) {
            const permissionId = checkbox.dataset.permissionId;
            const role = checkbox.dataset.role;
            const granted = checkbox.checked;

            fetch('permissions_api.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'toggle_permission',
                    permission_id: permissionId,
                    role: role,
                    granted: granted
                })
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showSaveIndicator();
                } else {
                    alert('Error: ' + data.message);
                    checkbox.checked = !granted; // Revert
                }
            })
            .catch(err => {
                console.error(err);
                alert('Failed to save permission change');
                checkbox.checked = !granted; // Revert
            });
        }

        // Show save indicator
        function showSaveIndicator() {
            const indicator = document.getElementById('saveIndicator');
            indicator.style.display = 'block';
            setTimeout(() => {
                indicator.style.display = 'none';
            }, 2000);
        }

        // Open add permission modal
        function openAddPermissionModal() {
            document.getElementById('permissionModalTitle').textContent = 'Add New Permission';
            document.getElementById('permissionForm').reset();
            document.getElementById('permission_id').value = '';
            new bootstrap.Modal(document.getElementById('permissionModal')).show();
        }

        // Edit permission
        function editPermission(id, name, displayName, description, category) {
            document.getElementById('permissionModalTitle').textContent = 'Edit Permission';
            document.getElementById('permission_id').value = id;
            document.getElementById('permission_name').value = name;
            document.getElementById('permission_display_name').value = displayName;
            document.getElementById('permission_description').value = description;
            document.getElementById('permission_category').value = category;
            new bootstrap.Modal(document.getElementById('permissionModal')).show();
        }

        // Delete permission
        function deletePermission(id, displayName) {
            if (!confirm(`Delete permission "${displayName}"?\n\nThis will remove it from all roles.`)) {
                return;
            }

            fetch('permissions_api.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'delete_permission',
                    permission_id: id
                })
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                }
            });
        }

        // Handle permission form submission
        document.getElementById('permissionForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const formData = new FormData(this);
            const data = {
                action: document.getElementById('permission_id').value ? 'edit_permission' : 'add_permission',
                permission_id: formData.get('permission_id'),
                name: formData.get('name'),
                display_name: formData.get('display_name'),
                description: formData.get('description'),
                category: formData.get('category') === 'custom' ? 
                          document.getElementById('custom_category').value : 
                          formData.get('category')
            };

            fetch('permissions_api.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                }
            });
        });

        // Show/hide custom category field
        document.getElementById('permission_category').addEventListener('change', function() {
            const customField = document.getElementById('custom_category_field');
            customField.style.display = this.value === 'custom' ? 'block' : 'none';
        });
    </script>
</body>
</html>
