<?php
/**
 * Permissions Management API
 * Handles CRUD operations for permissions
 */

session_start();
require_once '../includes/db.php';
require_once '../includes/permissions.php';

header('Content-Type: application/json');

// Require super_admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'super_admin') {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

$current_user_id = $_SESSION['user_id'];

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';

try {
    switch ($action) {
        case 'toggle_permission':
            $permission_id = intval($input['permission_id'] ?? 0);
            $role = $input['role'] ?? '';
            $granted = $input['granted'] ?? false;

            if (!$permission_id || !$role) {
                echo json_encode(['success' => false, 'message' => 'Invalid parameters']);
                exit;
            }

            if ($granted) {
                // Grant permission
                $stmt = $pdo->prepare("
                    INSERT IGNORE INTO role_permissions (role, permission_id, granted_by)
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$role, $permission_id, $current_user_id]);
            } else {
                // Revoke permission
                $stmt = $pdo->prepare("
                    DELETE FROM role_permissions
                    WHERE role = ? AND permission_id = ?
                ");
                $stmt->execute([$role, $permission_id]);
            }

            clearPermissionsCache();

            echo json_encode([
                'success' => true,
                'message' => 'Permission ' . ($granted ? 'granted' : 'revoked')
            ]);
            break;

        case 'add_permission':
            $name = trim($input['name'] ?? '');
            $display_name = trim($input['display_name'] ?? '');
            $description = trim($input['description'] ?? '');
            $category = trim($input['category'] ?? 'general');

            if (!$name || !$display_name) {
                echo json_encode(['success' => false, 'message' => 'Name and display name are required']);
                exit;
            }

            // Validate name format
            if (!preg_match('/^[a-z_]+$/', $name)) {
                echo json_encode(['success' => false, 'message' => 'Name must contain only lowercase letters and underscores']);
                exit;
            }

            $stmt = $pdo->prepare("
                INSERT INTO permissions (name, display_name, description, category)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([$name, $display_name, $description, $category]);

            $new_id = $pdo->lastInsertId();

            // Auto-assign to super_admin
            $stmt = $pdo->prepare("
                INSERT INTO role_permissions (role, permission_id, granted_by)
                VALUES ('super_admin', ?, ?)
            ");
            $stmt->execute([$new_id, $current_user_id]);

            clearPermissionsCache();

            echo json_encode([
                'success' => true,
                'message' => 'Permission created successfully',
                'permission_id' => $new_id
            ]);
            break;

        case 'edit_permission':
            $permission_id = intval($input['permission_id'] ?? 0);
            $name = trim($input['name'] ?? '');
            $display_name = trim($input['display_name'] ?? '');
            $description = trim($input['description'] ?? '');
            $category = trim($input['category'] ?? 'general');

            if (!$permission_id || !$name || !$display_name) {
                echo json_encode(['success' => false, 'message' => 'Invalid parameters']);
                exit;
            }

            // Validate name format
            if (!preg_match('/^[a-z_]+$/', $name)) {
                echo json_encode(['success' => false, 'message' => 'Name must contain only lowercase letters and underscores']);
                exit;
            }

            $stmt = $pdo->prepare("
                UPDATE permissions
                SET name = ?, display_name = ?, description = ?, category = ?
                WHERE id = ?
            ");
            $stmt->execute([$name, $display_name, $description, $category, $permission_id]);

            clearPermissionsCache();

            echo json_encode([
                'success' => true,
                'message' => 'Permission updated successfully'
            ]);
            break;

        case 'delete_permission':
            $permission_id = intval($input['permission_id'] ?? 0);

            if (!$permission_id) {
                echo json_encode(['success' => false, 'message' => 'Invalid permission ID']);
                exit;
            }

            // Delete permission (cascade will remove role_permissions)
            $stmt = $pdo->prepare("DELETE FROM permissions WHERE id = ?");
            $stmt->execute([$permission_id]);

            clearPermissionsCache();

            echo json_encode([
                'success' => true,
                'message' => 'Permission deleted successfully'
            ]);
            break;

        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (PDOException $e) {
    error_log("Permissions API error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
