<?php
include '../includes/auth.php';
include '../includes/db.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!isset($data['participants']) || !is_array($data['participants'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid data format']);
    exit;
}

try {
    $pdo->beginTransaction();
    
    $inserted = 0;
    $updated = 0;
    $errors = [];
    
    foreach ($data['participants'] as $index => $row) {
        // Skip empty rows
        if (empty($row[1]) && empty($row[2]) && empty($row[3])) {
            continue;
        }
        
        // Validate required fields
        if (empty($row[2]) || empty($row[3])) {
            $errors[] = "Row " . ($index + 1) . ": First Name and Last Name are required";
            continue;
        }
        
        // Expected format: [ID, BIB, First Name, Last Name, Gender, Year, Club, Category, Country, FIS, Licence, Licence Type, CustomFields]
        // Validate row has correct number of columns (at least 12, can have 13 with custom fields)
        if (count($row) < 12) {
            $errors[] = "Row " . ($index + 1) . ": Invalid column count (expected at least 12, got " . count($row) . ")";
            error_log("Row $index invalid column count: " . json_encode($row));
            continue;
        }
        
        $id = !empty($row[0]) ? intval($row[0]) : null;
        $bib = $row[1] ?? '';
        $first_name = $row[2] ?? '';
        $last_name = $row[3] ?? '';
        $gender = $row[4] ?? '';
        $year = !empty($row[5]) ? intval($row[5]) : null;
        $club = $row[6] ?? '';
        $category = $row[7] ?? '';
        $country = $row[8] ?? '';
        $fis_code = $row[9] ?? '';
        $licence_code = $row[10] ?? '';
        $licence_type = $row[11] ?? '';
        
        // Extract custom fields if present (index 12)
        $custom_fields_json = null;
        if (isset($row[12]) && is_array($row[12]) && !empty($row[12])) {
            $custom_fields_json = json_encode($row[12]);
        }
        
        // Log first few rows for debugging
        if ($index < 3) {
            error_log("Processing row $index: ID=$id, BIB=$bib, Name=$first_name $last_name, Gender=$gender, Year=$year, Country=$country, Licence=$licence_code, Type=$licence_type");
        }
        
        // Convert year to date_of_birth (use Jan 1 of that year)
        $date_of_birth = $year ? "$year-01-01" : null;
        
        if ($id && $id > 0) {
            // Update existing participant
            $stmt = $pdo->prepare("
                UPDATE participants 
                SET bib = ?,
                    first_name = ?,
                    last_name = ?,
                    gender = ?,
                    date_of_birth = ?,
                    club = ?,
                    category = ?,
                    country = ?,
                    fis_code = ?,
                    licence_code = ?,
                    licence_type = ?,
                    custom_fields = ?
                WHERE id = ?
            ");
            
            $stmt->execute([
                $bib,
                $first_name,
                $last_name,
                $gender,
                $date_of_birth,
                $club,
                $category,
                $country,
                $fis_code,
                $licence_code,
                $licence_type,
                $custom_fields_json,
                $id
            ]);
            
            if ($stmt->rowCount() > 0) {
                $updated++;
            }
        } else {
            // Insert new participant
            $stmt = $pdo->prepare("
                INSERT INTO participants 
                (bib, first_name, last_name, gender, date_of_birth, club, category, country, fis_code, licence_code, licence_type, custom_fields, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $bib,
                $first_name,
                $last_name,
                $gender,
                $date_of_birth,
                $club,
                $category,
                $country,
                $fis_code,
                $licence_code,
                $licence_type,
                $custom_fields_json
            ]);
            
            $inserted++;
        }
    }
    
    $pdo->commit();
    
    $message = [];
    if ($inserted > 0) {
        $message[] = "$inserted participant(s) added";
    }
    if ($updated > 0) {
        $message[] = "$updated participant(s) updated";
    }
    
    $finalMessage = !empty($message) ? implode(', ', $message) : 'No changes made';
    
    if (!empty($errors)) {
        $finalMessage .= ". Errors: " . implode('; ', $errors);
    }
    
    echo json_encode([
        'success' => true,
        'message' => $finalMessage,
        'stats' => [
            'inserted' => $inserted,
            'updated' => $updated,
            'errors' => count($errors)
        ]
    ]);
    
} catch (Exception $e) {
    $pdo->rollBack();
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
