<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/country_codes.php';

$message = '';

// Handle bulk assignment to event
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign_to_event'])) {
    try {
        $event_id = $_POST['event_id'] ?? null;
        $participant_ids = $_POST['participant_ids'] ?? [];
        $assign_to_competition = isset($_POST['assign_to_event_participants']);
        $heat_number = !empty($_POST['heat_number']) ? $_POST['heat_number'] : null;
        $category_override = !empty($_POST['category']) ? trim($_POST['category']) : null;
        
        if (!$event_id) {
            throw new Exception('Event selection is required');
        }
        
        if (empty($participant_ids)) {
            throw new Exception('No athletes selected');
        }
        
        // If participant_ids is a string, convert to array
        if (!is_array($participant_ids)) {
            $participant_ids = array_filter(array_map('trim', explode(',', $participant_ids)));
        }
        
        $assigned_count = 0;
        $skipped_count = 0;
        $errors = [];
        
        foreach ($participant_ids as $participant_id) {
            $participant_id = (int)$participant_id;
            
            if ($participant_id <= 0) continue;
            
            // Get participant details
            $participant_stmt = $pdo->prepare("SELECT * FROM participants WHERE id = ?");
            $participant_stmt->execute([$participant_id]);
            $participant = $participant_stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$participant) {
                $errors[] = "Participant ID {$participant_id} not found";
                continue;
            }
            
            if ($assign_to_competition) {
                // Check if already assigned to event_participants
                $check_stmt = $pdo->prepare("SELECT id FROM event_participants WHERE participant_id = ? AND event_id = ?");
                $check_stmt->execute([$participant_id, $event_id]);
                
                if ($check_stmt->fetch()) {
                    $skipped_count++;
                    continue;
                }
                
                // Determine category to use
                $category = $category_override ?: $participant['category'];
                
                // Insert into event_participants
                $insert_stmt = $pdo->prepare("
                    INSERT INTO event_participants 
                    (participant_id, event_id, heat_number, category, bib) 
                    VALUES (?, ?, ?, ?, ?)
                ");
                
                $insert_stmt->execute([
                    $participant_id,
                    $event_id,
                    $heat_number,
                    $category,
                    $participant['bib']
                ]);
                
                $assigned_count++;
            } else {
                // Just track in participants_event table
                $check_stmt = $pdo->prepare("SELECT id FROM participants_event WHERE participant_id = ? AND event_id = ?");
                $check_stmt->execute([$participant_id, $event_id]);
                
                if ($check_stmt->fetch()) {
                    $skipped_count++;
                    continue;
                }
                
                $insert_stmt = $pdo->prepare("
                    INSERT INTO participants_event (participant_id, event_id) 
                    VALUES (?, ?)
                ");
                
                $insert_stmt->execute([$participant_id, $event_id]);
                $assigned_count++;
            }
        }
        
        // Build success message
        $msg_parts = [];
        if ($assigned_count > 0) {
            $msg_parts[] = "{$assigned_count} athlete(s) assigned successfully";
        }
        if ($skipped_count > 0) {
            $msg_parts[] = "{$skipped_count} already assigned (skipped)";
        }
        if (!empty($errors)) {
            $msg_parts[] = count($errors) . " error(s) occurred";
        }
        
        $message_type = empty($errors) && $assigned_count > 0 ? 'success' : 'warning';
        $icon = $message_type === 'success' ? 'check-circle' : 'exclamation-triangle';
        
        $message = '<div class="alert alert-' . $message_type . ' alert-dismissible fade show" role="alert">
                        <i class="fas fa-' . $icon . ' me-2"></i>' . implode('. ', $msg_parts) . '
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>';
        
        if (!empty($errors)) {
            $message .= '<div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <strong>Errors:</strong>
                            <ul class="mb-0 mt-2">';
            foreach ($errors as $error) {
                $message .= '<li>' . htmlspecialchars($error) . '</li>';
            }
            $message .= '</ul>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>';
        }
        
    } catch (Exception $e) {
        $message = '<div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-triangle me-2"></i>Error: ' . htmlspecialchars($e->getMessage()) . '
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>';
    }
}

// Handle export error messages
if (isset($_GET['error'])) {
    $message = '<div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>' . htmlspecialchars($_GET['error']) . '
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>';
}

// Get events for dropdown
$events = $pdo->query("SELECT id, name, date FROM events ORDER BY date DESC")->fetchAll(PDO::FETCH_ASSOC);

// Get heats for dropdown
$heats = $pdo->query("SELECT id, heat_number, heat_name, event_id FROM event_heat_settings ORDER BY event_id, heat_number")->fetchAll(PDO::FETCH_ASSOC);

// Filter parameters
$filter_gender = $_GET['gender'] ?? '';
$filter_category = $_GET['category'] ?? '';
$filter_event = $_GET['event'] ?? '';
$filter_imported = $_GET['imported'] ?? '';
$filter_licence_type = $_GET['licence_type'] ?? '';
$search = $_GET['search'] ?? '';

// Build query with event filtering
$where_clauses = [];
$params = [];

if ($filter_gender !== '') {
    $where_clauses[] = "p.gender = ?";
    $params[] = $filter_gender;
}

if ($filter_category !== '') {
    $where_clauses[] = "p.category = ?";
    $params[] = $filter_category;
}

if ($filter_licence_type !== '') {
    $where_clauses[] = "p.licence_type = ?";
    $params[] = $filter_licence_type;
}

if ($filter_event !== '') {
    if ($filter_event === 'not_assigned') {
        $where_clauses[] = "ep.event_id IS NULL AND pe.event_id IS NULL";
    } else {
        $where_clauses[] = "(ep.event_id = ? OR pe.event_id = ?)";
        $params[] = $filter_event;
        $params[] = $filter_event;
    }
}

if ($filter_imported !== '') {
    if ($filter_imported === 'yes') {
        $where_clauses[] = "pe.event_id IS NOT NULL";
    } else {
        $where_clauses[] = "pe.event_id IS NULL";
    }
}

if ($search !== '') {
    $where_clauses[] = "(p.first_name LIKE ? OR p.last_name LIKE ? OR p.bib LIKE ? OR p.club LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$where_sql = $where_clauses ? "WHERE " . implode(" AND ", $where_clauses) : "";

// Query to get participants with event assignments
$query = "SELECT p.*, 
                 GROUP_CONCAT(DISTINCT e1.name SEPARATOR ', ') as assigned_event_names,
                 GROUP_CONCAT(DISTINCT e1.id SEPARATOR ',') as assigned_event_ids,
                 GROUP_CONCAT(DISTINCT e2.name SEPARATOR ', ') as imported_event_names,
                 GROUP_CONCAT(DISTINCT e2.id SEPARATOR ',') as imported_event_ids
          FROM participants p
          LEFT JOIN event_participants ep ON p.id = ep.participant_id
          LEFT JOIN events e1 ON ep.event_id = e1.id
          LEFT JOIN participants_event pe ON p.id = pe.participant_id
          LEFT JOIN events e2 ON pe.event_id = e2.id
          $where_sql
          GROUP BY p.id
          ORDER BY p.id DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$participants = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get distinct values for filters
$genders = $pdo->query("SELECT DISTINCT gender FROM participants WHERE gender IS NOT NULL AND gender != '' ORDER BY gender")->fetchAll(PDO::FETCH_COLUMN);
$categories = $pdo->query("SELECT DISTINCT category FROM participants WHERE category IS NOT NULL AND category != '' ORDER BY category")->fetchAll(PDO::FETCH_COLUMN);
$clubs = $pdo->query("SELECT DISTINCT club FROM participants WHERE club IS NOT NULL AND club != '' ORDER BY club")->fetchAll(PDO::FETCH_COLUMN);
$licence_types = $pdo->query("SELECT DISTINCT licence_type FROM participants WHERE licence_type IS NOT NULL AND licence_type != '' ORDER BY licence_type")->fetchAll(PDO::FETCH_COLUMN);

// Get custom field definitions
$custom_field_defs = [];
try {
    $custom_field_defs = $pdo->query("
        SELECT * FROM participant_custom_field_definitions 
        WHERE is_active = 1 
        ORDER BY display_order ASC, id ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Table might not exist yet, silently fail
}

// Get statistics
$total_participants = $pdo->query("SELECT COUNT(*) FROM participants")->fetchColumn();
$participants_with_events = $pdo->query("SELECT COUNT(DISTINCT participant_id) FROM event_participants")->fetchColumn();
$participants_without_events = $total_participants - $participants_with_events;
$imported_participants = $pdo->query("SELECT COUNT(DISTINCT participant_id) FROM participants_event")->fetchColumn();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Participant Management</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    
    <!-- jspreadsheet CSS -->
    <link rel="stylesheet" href="../assets/js/vendor/jspreadsheet/jsuites.min.css">
    <link rel="stylesheet" href="../assets/js/vendor/jspreadsheet/jspreadsheet.min.css">
    
    <style>
        /* Heat Panel Design System */
        .table-hover tbody tr:hover {
            background-color: rgba(0, 123, 255, 0.05);
        }

        tr.selected-row {
            background-color: rgba(0, 123, 255, 0.1) !important;
        }

        /* Jspreadsheet Integration */
        #spreadsheet {
            width: 100%;
            overflow-x: auto;
            position: relative;
        }
        
        /* Mobile Responsive Optimizations */
        @media (max-width: 768px) {
            .container-fluid {
                padding: 0.5rem !important;
            }
            
            /* Sticky first column only on mobile - ID */
            .jexcel tbody tr td:nth-child(1),
            .jexcel thead tr td:nth-child(1) {
                position: sticky !important;
                left: 0 !important;
                z-index: 10 !important;
                background: white !important;
                box-shadow: 2px 0 4px rgba(0,0,0,0.08);
            }
            
            .jexcel thead tr td:nth-child(1) {
                background: linear-gradient(to bottom, #7c5ac2 0%, #6848a6 100%) !important;
            }
            
            .jexcel tbody tr:hover td:nth-child(1) {
                background-color: #f8f9fa !important;
            }
            
            .jexcel tbody tr.jexcel_row_selected td:nth-child(1) {
                background-color: #e8eaf6 !important;
            }
            
            /* Larger touch targets */
            .jexcel tbody tr td {
                padding: 12px 8px !important;
                min-height: 48px;
                font-size: 0.813rem;
            }
            
            .jexcel thead tr td {
                padding: 14px 8px !important;
                font-size: 0.75rem;
            }
            
            /* Larger buttons for touch */
            .jexcel .btn-group-sm .btn {
                padding: 8px 10px !important;
                font-size: 0.875rem !important;
                min-width: 36px;
                min-height: 36px;
            }
            
            /* Larger checkboxes */
            .jexcel > tbody > tr > td input[type="checkbox"],
            .jexcel > thead > tr > td input[type="checkbox"] {
                width: 20px;
                height: 20px;
            }
            
            /* Better scrollbar for mobile */
            .jexcel_content {
                -webkit-overflow-scrolling: touch;
                scroll-behavior: smooth;
            }
            
            /* Compact metrics */
            .metric-card .h4 {
                font-size: 1.25rem !important;
            }
            
            .metric-card .fa-2x {
                font-size: 1.5rem !important;
            }
            
            /* Mobile pagination compact */
            .jexcel_pagination {
                padding: 12px 10px !important;
                flex-wrap: wrap;
                gap: 0.5rem !important;
            }
            
            .jexcel_pagination > div {
                font-size: 0.813rem !important;
            }
            
            .jexcel_pagination button {
                padding: 6px 10px !important;
                font-size: 0.813rem !important;
            }
        }
        
        @media (max-width: 576px) {
            /* Even more compact on very small screens */
            .jexcel tbody tr td {
                padding: 10px 6px !important;
                font-size: 0.75rem;
            }
            
            .jexcel thead tr td {
                padding: 12px 6px !important;
                font-size: 0.688rem;
            }
            
            .metric-card {
                margin-bottom: 0.5rem;
            }
            
            /* Stack metrics 2 per row on small mobile */
            .col-md-3 {
                flex: 0 0 50%;
                max-width: 50%;
            }
            
            /* Smaller buttons */
            .jexcel .btn-group-sm .btn {
                padding: 6px 8px !important;
                font-size: 0.75rem !important;
                min-width: 32px;
                min-height: 32px;
            }
        }

        .jexcel_container {
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }

        .jexcel thead tr td {
            background-color: #495057 !important;
            color: white !important;
            font-weight: 600;
            padding: 8px 4px !important;
            border-right: 1px solid #6c757d !important;
        }

        .jexcel tbody tr:hover {
            background-color: rgba(0, 123, 255, 0.05) !important;
        }

        .jexcel tbody tr.jexcel_row_selected {
            background-color: #fff3cd !important;
        }

        .jexcel tbody tr td {
            padding: 8px !important;
            border-right: 1px solid #dee2e6 !important;
        }

        /* Two-pane layout */
        .sticky-header {
            position: sticky;
            top: 0;
            z-index: 100;
            background: white;
        }

        .scrollable-pane {
            max-height: calc(100vh - 200px);
            overflow-y: auto;
        }

        /* Filter section */
        .vr {
            width: 1px;
            height: 24px;
            background-color: #dee2e6;
        }

        /* Spreadsheet toolbar */
        .spreadsheet-toolbar {
            background: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
            padding: 12px;
        }

        /* Badge styling */
        .metric-card {
            transition: transform 0.2s;
        }

        .metric-card:hover {
            transform: translateY(-2px);
        }

        /* Action buttons */
        .action-btn-group .btn {
            transition: all 0.2s;
        }

        .action-btn-group .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
        }

        /* Scrollbar styling */
        .scrollable-pane::-webkit-scrollbar {
            width: 8px;
        }

        .scrollable-pane::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 4px;
        }

        .scrollable-pane::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #6c757d 0%, #495057 100%);
            border-radius: 4px;
        }

        .scrollable-pane::-webkit-scrollbar-thumb:hover {
            background: linear-gradient(135deg, #495057 0%, #343a40 100%);
        }

        /* Loading state */
        .loading-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.9);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }

        .loading-spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #007bff;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* jspreadsheet Enhanced Styling - Light Clean Layout */
        .jexcel_container {
            border-radius: 0.5rem;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            background: white;
            font-family: 'Lexend', "Twemoji Country Flags", BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol";
        }

        .jexcel > thead > tr > td {
            background: linear-gradient(to bottom, #7c5ac2 0%, #6848a6 100%) !important;
            color: white !important;
            font-weight: 600;
            text-align: center;
            padding: 12px 10px !important;
            border: none !important;
            border-right: 1px solid rgba(255,255,255,0.1) !important;
            font-size: 0.813rem;
            letter-spacing: 0.3px;
            text-transform: uppercase;
            cursor: pointer;
        }

        .jexcel > thead > tr > td:hover {
            background: linear-gradient(to bottom, #8d6bd1 0%, #7959b5 100%) !important;
        }

        /* Sortable column indicator */
        .jexcel > thead > tr > td.arrow-up::after,
        .jexcel > thead > tr > td.arrow-down::after {
            content: '';
            margin-left: 6px;
            border: 4px solid transparent;
            display: inline-block;
            vertical-align: middle;
        }

        .jexcel > thead > tr > td.arrow-up::after {
            border-bottom-color: white;
            border-top: 0;
            margin-bottom: 2px;
        }

        .jexcel > thead > tr > td.arrow-down::after {
            border-top-color: white;
            border-bottom: 0;
            margin-top: 2px;
        }

        .jexcel > tbody > tr > td {
               padding: 5px 15px !important;
            vertical-align: middle !important;
            border: none !important;
            border-bottom: 1px solid #f0f0f0 !important;
            border-right: 1px solid #f5f5f5 !important;
            background: white !important;
            font-size: 0.875rem;
            color: #333;
        }

        .jexcel > tbody > tr:hover > td {
            background-color: #f8f9fa !important;
        }

        .jexcel > tbody > tr.jexcel_row_selected > td {
            background-color: #e8eaf6 !important;
        }

        /* Checkbox styling */
        .jexcel > tbody > tr > td input[type="checkbox"],
        .jexcel > thead > tr > td input[type="checkbox"] {
            width: 18px;
            height: 18px;
            cursor: pointer;
            accent-color: #7c5ac2;
        }

        /* Gender and Country badges in cells */
        .jexcel .badge {
            display: inline-flex;
            align-items: center;
            font-size: 0.75rem;
            padding: 4px 10px;
            font-weight: 500;
            border-radius: 12px;
        }

        .jexcel .badge.bg-primary {
            background-color: #5469d4 !important;
        }

        .jexcel .badge.bg-danger {
            background-color: #f5365c !important;
        }

        .jexcel .country-flag {
            font-size: 1.25rem;
            line-height: 1;
            margin-right: 6px;
            display: inline-block;
            vertical-align: middle;
        }

        .jexcel .country-code {
            font-size: 1.25rem;
            line-height: 1;
            margin-right: 4px;
        }

        /* Action buttons in spreadsheet */
        .jexcel .btn-group-sm {
            display: flex;
            gap: 4px;
        }

        .jexcel .btn-group-sm .btn {
            padding: 6px 10px;
            font-size: 0.813rem;
            border-radius: 0.25rem;
            border-width: 1px;
            transition: all 0.2s ease;
        }

        .jexcel .btn-group-sm .btn i {
            font-size: 0.875rem;
        }

        .jexcel .btn-outline-primary:hover {
            background-color: #5469d4;
            border-color: #5469d4;
            color: white;
        }

        .jexcel .btn-outline-info:hover {
            background-color: #11cdef;
            border-color: #11cdef;
            color: white;
        }

        .jexcel .btn-outline-danger:hover {
            background-color: #f5365c;
            border-color: #f5365c;
            color: white;
        }

        /* Dropdown styling */
        .jexcel_dropdown {
            border-radius: 0.5rem !important;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15) !important;
            border: 1px solid #e0e0e0 !important;
            max-height: 320px !important;
            overflow-y: auto !important;
            background: white !important;
            margin-top: 2px !important;
            font-family: Inter, "Twemoji Country Flags", BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol";
        }

        .jexcel_dropdown > div {
            padding: 10px 14px !important;
            cursor: pointer;
            transition: background-color 0.15s ease;
            border-bottom: 1px solid #f5f5f5;
        }

        .jexcel_dropdown > div:last-child {
            border-bottom: none;
        }

        .jexcel_dropdown > div:hover {
            background-color: #f8f9fa !important;
        }

        .jexcel_dropdown > div.jexcel_dropdown_selected {
            background-color: #e8eaf6 !important;
            color: #7c5ac2;
            font-weight: 500;
        }

        .jexcel_dropdown .badge {
            pointer-events: none;
        }

        /* Pagination styling - Clean white background */
        .jexcel_pagination {
            padding: 16px 20px !important;
            background: white !important;
            border-top: 2px solid #f0f0f0 !important;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 1rem;
            border-radius: 0 0 0.5rem 0.5rem;
        }

        .jexcel_pagination > div {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 0.875rem;
            color: #666;
        }

        .jexcel_pagination select {
            border: 1px solid #e0e0e0;
            border-radius: 0.375rem;
            padding: 6px 12px;
            background: white;
            font-size: 0.875rem;
            cursor: pointer;
        }

        .jexcel_pagination select:focus {
            outline: none;
            border-color: #7c5ac2;
        }

        .jexcel_pagination button {
            background: white !important;
            border: 1px solid #e0e0e0 !important;
            color: #666 !important;
            padding: 6px 14px !important;
            border-radius: 0.375rem !important;
            cursor: pointer !important;
            transition: all 0.2s ease !important;
            font-size: 0.875rem !important;
            font-weight: 500 !important;
        }

        .jexcel_pagination button:hover:not(:disabled) {
            background: #7c5ac2 !important;
            color: white !important;
            border-color: #7c5ac2 !important;
            box-shadow: 0 2px 4px rgba(124, 90, 194, 0.2) !important;
        }

        .jexcel_pagination button:disabled {
            opacity: 0.4;
            cursor: not-allowed !important;
        }

        /* Search box styling */
        .jexcel_search {
            
            background: white !important;
            border-bottom: 1px solid #f0f0f0 !important;
            border-radius: 0.5rem 0.5rem 0 0;
        }

        .jexcel_search input {
            width: 100% !important;
            
            border: 1px solid #e0e0e0 !important;
            border-radius: 0.5rem !important;
            font-size: 0.875rem !important;
            transition: all 0.2s ease !important;
        }

        .jexcel_search input:focus {
            outline: none !important;
            border-color: #7c5ac2 !important;
            box-shadow: 0 0 0 3px rgba(124, 90, 194, 0.1) !important;
        }

        .jexcel_search input::placeholder {
            color: #999;
        }

        /* Input field styling inside cells */
        .jexcel > tbody > tr > td input[type="text"],
        .jexcel > tbody > tr > td input[type="number"] {
            border: 1px solid transparent !important;
            padding: 4px 8px !important;
            width: 100% !important;
            background: transparent !important;
            transition: all 0.2s ease !important;
        }

        .jexcel > tbody > tr > td input[type="text"]:focus,
        .jexcel > tbody > tr > td input[type="number"]:focus {
            border-color: #7c5ac2 !important;
            background: white !important;
            box-shadow: 0 0 0 2px rgba(124, 90, 194, 0.1) !important;
            outline: none !important;
        }

        /* Selected cell highlight */
        .jexcel > tbody > tr > td.highlight {
            background-color: #f0ebf8 !important;
            box-shadow: inset 0 0 0 1px #7c5ac2 !important;
        }

        .jexcel > tbody > tr > td.highlight-selected {
            background-color: #e8eaf6 !important;
            box-shadow: inset 0 0 0 2px #7c5ac2 !important;
        }

        /* Scrollbar styling */
        .jexcel_content::-webkit-scrollbar {
            width: 10px;
            height: 10px;
        }

        .jexcel_content::-webkit-scrollbar-track {
            background: #f5f5f5;
        }

        .jexcel_content::-webkit-scrollbar-thumb {
            background: #ccc;
            border-radius: 5px;
        }

        .jexcel_content::-webkit-scrollbar-thumb:hover {
            background: #999;
        }

        /* Selected Athletes Grid Styling */
        .selected-participants-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 8px;
            max-height: 200px;
            overflow-y: auto;
            padding: 8px;
            background: white;
            border-radius: 6px;
            border: 1px solid #dee2e6;
        }

        .participant-item {
            display: flex;
            align-items: center;
            padding: 4px 8px;
            background: #f8f9fa;
            border-radius: 4px;
            font-size: 0.875rem;
        }

        .participant-name {
            flex-grow: 1;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
    </style>
</head>
<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>
   
    <!-- Help Button -->
    <div class="position-fixed start-0 bottom-0 p-3 help_f_button" style="z-index: 1055;">
        <button type="button" class="btn btn-info btn-lg rounded-circle shadow" title="Help / Guide" 
                data-bs-toggle="modal" data-bs-target="#helpModal" style="width:56px; height:56px;">
            <i class="fas fa-question"></i>
        </button>
    </div>

    <div class="container-fluid p-2 container-StyleScore">
        
        <!-- Event Actions Menu -->
       
           
                <?php       
                $use_localStorage = true;
                $show_header = false;
                include 'event_actions_menu.php'; 
                ?>
          
       
        <div class="d-flex justify-content-between align-items-center p-3">
            <div>
                <h1 class="h3 mb-1">
                    <i class="fas fa-users text-primary me-2"></i>Athletes Manager
                </h1>
                <p class="text-muted mb-0">Manage athletes and distribute to events</p>
            </div>
        
            <!-- Workspace Metrics -->
            <div class="row g-2">
                <div class="col-auto">
                    
                    <div class="card border-0 shadow-sm h-100 metric-card">
                        <div class="card-body p-3">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-users fa-2x text-primary opacity-50"></i>
                                </div>
                                <div class="ms-3">
                                    <div class="text-muted small">Total Athletes</div>
                                    <div class="h4 mb-0 fw-bold">
                                        <?= $total_participants ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-auto">
                    <div class="card border-0 shadow-sm h-100 metric-card">
                        <div class="card-body p-3">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-calendar-check fa-2x text-success opacity-50"></i>
                                </div>
                                <div class="ms-3">
                                    <div class="text-muted small">With Event Assignment</div>
                                    <div class="h4 mb-0 fw-bold text-success">
                                        <?= $participants_with_events ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-auto">
                    <div class="card border-0 shadow-sm h-100 metric-card">
                        <div class="card-body p-3">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-user-times fa-2x text-warning opacity-50"></i>
                                </div>
                                <div class="ms-3">
                                    <div class="text-muted small">Without Assignment</div>
                                    <div class="h4 mb-0 fw-bold text-warning">
                                        <?= $participants_without_events ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-auto">
                    <div class="card border-0 shadow-sm h-100 metric-card">
                        <div class="card-body p-3">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-file-import fa-2x text-info opacity-50"></i>
                                </div>
                                <div class="ms-3">
                                    <div class="text-muted small">Imported</div>
                                    <div class="h4 mb-0 fw-bold text-info">
                                        <?= $imported_participants ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-auto">
                    <button type="button" class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#helpModal">
                        <i class="fas fa-circle-question me-1"></i>Help
                    </button>
                </div>
            </div>
       
    </div>

        <?php echo $message; ?>

        <!-- Two-Pane Layout -->
        <div class="row g-3">
            
            <!-- LEFT PANE: Filters & Actions -->
             <!-- LEFT PANE: Filters & Actions -->
            <div class="col-12 col-lg-2 col-xxl-2">
                <div class="card shadow-sm">
                    

                    <!-- Action Buttons -->
                    <div class="card-body p-3">
                        <div class="action-btn-group gap-2 justify-content-center d-flex flex-wrap">
                            <button type="button" class="w-100 btn btn-success g-col-6 mb-1" onclick="showAddParticipantModal()">
                                <i class="fas fa-user-plus me-2"></i>Add New Participant
                            </button>
                            
                            <button type="button" class="w-100 btn btn-primary g-col-6 mb-1" onclick="bulkAssign()" id="bulkAssignBtn" disabled>
                                <i class="fas fa-calendar-plus me-2"></i>Assign to Event
                                <span class="badge bg-white text-primary ms-2" id="assignCount">0</span>
                            </button>

                            <button type="button" class="w-100 btn btn-info g-col-6 mb-1" onclick="showExcelPasteModal()">
                                <i class="fas fa-table me-2"></i>Excel Paste Import
                            </button>

                            <button type="button" class="w-100 btn btn-outline-info g-col-6 mb-1" data-bs-toggle="modal" data-bs-target="#importModal">
                                <i class="fas fa-file-import me-2"></i>CSV Import
                            </button>

                            <button type="button" class="w-100 btn btn-secondary g-col-6 mb-1 " onclick="exportData()">
                                <i class="fas fa-file-export me-2"></i>Export CSV
                            </button>
                            <button type="button" class="w-100 btn btn-outline-warning g-col-6 mb-1"  onclick="location.href='custom_fields_manager.php'">
                                <i class="fas fa-file-import me-2"></i>Custom Fields
                            </button>
                            <div class="hr"></div>

                            <button type="button" class="w-100 btn btn-outline-success g-col-6 mb-1" onclick="saveChanges()" id="saveBtn" disabled>
                                <i class="fas fa-save me-2"></i>Save Changes
                                <span class="badge bg-success ms-2 d-none" id="changesCount">0</span>
                            </button>

                            <button type="button" class="w-100 btn btn-outline-secondary g-col-6 mb-1" onclick="refreshData()">
                                <i class="fas fa-sync-alt me-2"></i>Refresh Data
                            </button>

                            <div class="hr"></div>

                            <button type="button" class="w-100 btn btn-outline-danger" onclick="bulkDelete()" id="bulkDeleteBtn" disabled>
                                <i class="fas fa-trash me-2"></i>Delete Selected
                                <span class="badge bg-danger ms-2" id="deleteCount">0</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- RIGHT PANE: Spreadsheet Editor -->
            <div class="col-12 col-lg-10 col-xxl-10">
                        <div class="card shadow-sm">
                            <div class="card-header bg-white p-2">
                                <div class="d-flex justify-content-between align-items-center">
                                    
                                    <!-- Status Badge -->
                                    <div class="text-center d-none">
                                        <span class="badge bg-info" id="statusBadge">
                                            <span id="selectedCount">0</span> selected • 
                                            <span id="visibleCount"><?= count($participants) ?></span> visible
                                        </span>
                                    </div>
                                </div>

                                <div class="">
                                  
                                   

                                   

                                    <!-- Filter Dropdowns -->
                                    <div class="row g-2">
                                        <div class="col-auto">
                                             <!-- Search Bar -->
                                            <div class="input-group input-group-sm">
                                                <span class="input-group-text">
                                                    <i class="fas fa-search"></i>
                                                </span>
                                                <input type="text" class="form-control" id="searchInput" 
                                                    placeholder="Search participants..." 
                                                    value="<?= htmlspecialchars($search) ?>">
                                                <button class="btn btn-outline-secondary" type="button" onclick="clearSearch()">
                                                    <i class="fas fa-times"></i>
                                                </button>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <select class="form-select form-select-sm" id="genderFilter">
                                                <option value="">All Genders</option>
                                                <?php foreach ($genders as $g): ?>
                                                    <option value="<?= htmlspecialchars($g) ?>" <?= $filter_gender === $g ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($g) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        <div class="col-auto">
                                            <select class="form-select form-select-sm" id="categoryFilter">
                                                <option value="">All Divisions</option>
                                                <?php foreach ($categories as $c): ?>
                                                    <option value="<?= htmlspecialchars($c) ?>" <?= $filter_category === $c ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($c) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        <div class="col-auto">
                                            <select class="form-select form-select-sm" id="licenceTypeFilter">
                                                <option value="">All Licence Types</option>
                                                <?php foreach ($licence_types as $lt): ?>
                                                    <option value="<?= htmlspecialchars($lt) ?>" <?= $filter_licence_type === $lt ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($lt) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        <div class="col-auto">
                                            <select class="form-select form-select-sm" id="eventFilter">
                                                <option value="">All Events</option>
                                                <option value="not_assigned" <?= $filter_event === 'not_assigned' ? 'selected' : '' ?>>
                                                    Not Assigned
                                                </option>
                                                <?php foreach ($events as $e): ?>
                                                    <option value="<?= $e['id'] ?>" <?= $filter_event == $e['id'] ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($e['name']) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                         <!-- Quick Actions -->
                                    
                                        <div class="col-auto">
                                            <button type="button" class="btn btn-outline-primary btn-sm w-100" onclick="selectAllVisible()">
                                                <i class="fas fa-check-square me-1"></i>Select All
                                            </button>
                                        </div>
                                        <div class="col-auto">
                                            <button type="button" class="btn btn-outline-secondary btn-sm w-100" onclick="clearSelection()">
                                                <i class="fas fa-times me-1"></i>Clear
                                            </button>
                                        </div>
                                        <div class="col-auto">
                                            <button type="button" class="btn btn-outline-danger btn-sm w-100" onclick="clearAllFilters()">
                                                <i class="fas fa-filter-circle-xmark me-1"></i>Reset Filters
                                            </button>
                                        </div>
                                        <div class="col-auto">
                                            <button type="button" class="btn btn-outline-info btn-sm w-100" onclick="showColumnVisibilityModal()">
                                                <i class="fas fa-columns me-1"></i>Columns
                                            </button>
                                        </div>
                                    <div class="ms-auto col-auto btn-group btn-group-sm">
                                        <button type="button" class="btn btn-outline-primary" onclick="addNewRow()">
                                            <i class="fas fa-plus me-1"></i>Add Row
                                        </button>
                                        <button type="button" class="btn btn-outline-secondary" onclick="deleteSelectedRows()">
                                            <i class="fas fa-minus me-1"></i>Delete Rows
                                        </button>
                                    </div>
                                </div>

                                    
                                </div>
                            </div>                    


                        </div>

                    <div class="card-body p-0 position-relative">
                        <div id="loadingOverlay" class="loading-overlay d-none">
                            <div class="loading-spinner"></div>
                        </div>
                        <div id="spreadsheet" class="scrollable-pane"></div>
                    </div>
                </div>
            </div>

        </div>
    </div>

    <!-- Import Modal -->
    <div class="modal fade" id="importModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header border-0">
                    <h5 class="modal-title">
                        <i class="fas fa-file-import text-info me-2"></i>Import CSV
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        Upload a CSV file with athlete data. The file will be imported into the spreadsheet for review before saving.
                    </div>
                    <form id="importForm">
                        <div class="mb-3">
                            <label class="form-label">Select CSV File</label>
                            <input type="file" class="form-control" id="csvFile" accept=".csv" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Assign to Event (Optional)</label>
                            <select class="form-select" id="importEventId">
                                <option value="">Select Event...</option>
                                <?php foreach ($events as $e): ?>
                                    <option value="<?= $e['id'] ?>">
                                        <?= htmlspecialchars($e['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="overwriteDuplicates">
                            <label class="form-check-label" for="overwriteDuplicates">
                                Overwrite existing athletes with same BIB
                            </label>
                        </div>
                    </form>
                </div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-info" onclick="processImport()">
                        <i class="fas fa-upload me-2"></i>Import to Spreadsheet
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Generic Alert Modal -->
    <div class="modal fade" id="genericAlertModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header border-0">
                    <h5 class="modal-title">
                        <i class="fas fa-info-circle text-primary me-2" id="alertIcon"></i>
                        <span id="alertTitle">Notice</span>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="alertMessage"></div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-primary" data-bs-dismiss="modal">OK</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Generic Confirm Modal -->
    <div class="modal fade" id="genericConfirmModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header border-0">
                    <h5 class="modal-title">
                        <i class="fas fa-question-circle text-warning me-2" id="confirmIcon"></i>
                        <span id="confirmTitle">Confirm Action</span>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="confirmMessage"></div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="confirmButton">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Excel Paste Import Modal -->
    <div class="modal fade" id="excelPasteModal" tabindex="-1">
        <div class="modal-dialog modal-fullscreen">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-table me-2"></i>Excel Paste Import Editor
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <!-- Step 1: Paste Area -->
                    <div id="pasteStep" class="step-container">
                        <div class="alert alert-info">
                            <h6><i class="fas fa-clipboard me-2"></i>Step 1: Paste Your Data</h6>
                            <p class="mb-0 small">Copy data from Excel (Ctrl+C) and paste it into the text area below (Ctrl+V)</p>
                        </div>
                        <div class="row">
                            <div class="col-md-8">
                                <textarea id="pasteArea" class="form-control font-monospace" rows="15" 
                                          placeholder="Paste your Excel data here...&#10;&#10;Example (Tab-separated):&#10;BIB&#9;First Name&#9;Last Name&#9;Gender&#9;Country&#9;Year of Birth&#10;101&#9;John&#9;Doe&#9;M&#9;USA&#9;1995&#10;102&#9;Jane&#9;Smith&#9;F&#9;CAN&#9;1998"></textarea>
                                <div class="mt-2">
                                    <button type="button" class="btn btn-primary" onclick="parsePastedData()">
                                        <i class="fas fa-arrow-right me-1"></i>Parse Data
                                    </button>
                                    <button type="button" class="btn btn-outline-secondary" onclick="clearPasteArea()">
                                        <i class="fas fa-eraser me-1"></i>Clear
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card bg-light">
                                    <div class="card-body">
                                        <h6 class="card-title"><i class="fas fa-lightbulb me-1"></i>Tips</h6>
                                        <ul class="small mb-0">
                                            <li>Excel automatically preserves tabs when copying</li>
                                            <li>Include column headers in your first row</li>
                                            <li>You can paste partial data and add more later</li>
                                            <li>Data will be parsed automatically on paste</li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Step 2: Column Mapping -->
                    <div id="mappingStep" class="step-container d-none">
                        <div class="alert alert-success">
                            <h6><i class="fas fa-columns me-2"></i>Step 2: Map Columns</h6>
                            <p class="mb-0 small">Match your Excel columns to athlete fields</p>
                        </div>
                        <div class="row mb-3">
                            <div class="col-12">
                                <div id="columnMappingArea" class="d-flex flex-wrap gap-2"></div>
                            </div>
                        </div>
                        <div class="d-flex gap-2">
                            <button type="button" class="btn btn-secondary" onclick="backToPaste()">
                                <i class="fas fa-arrow-left me-1"></i>Back to Paste
                            </button>
                            <button type="button" class="btn btn-success" onclick="applyMappingExcel()">
                                <i class="fas fa-check me-1"></i>Apply Mapping & Continue
                            </button>
                        </div>
                    </div>

                    <!-- Step 3: Edit Grid -->
                    <div id="editStep" class="step-container d-none">
                        <div class="alert alert-warning">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6><i class="fas fa-edit me-2"></i>Step 3: Edit & Validate Data</h6>
                                    <p class="mb-0 small">Review and edit your data. Click any cell to edit.</p>
                                </div>
                                <div>
                                    <select id="eventSelectImport" class="form-select form-select-sm" style="width: 300px;" required>
                                        <option value="">Select Event for Import *</option>
                                        <?php foreach ($events as $event): ?>
                                            <option value="<?= $event['id'] ?>">
                                                <?= htmlspecialchars($event['name']) ?> (<?= date('M d, Y', strtotime($event['date'])) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="table-responsive" style="max-height: 60vh; overflow: auto;">
                            <table id="editableGrid" class="table table-bordered table-sm table-hover">
                                <thead class="table-dark sticky-top"></thead>
                                <tbody></tbody>
                            </table>
                        </div>
                        <div class="d-flex justify-content-between align-items-center mt-3">
                            <div>
                                <button type="button" class="btn btn-secondary" onclick="backToMapping()">
                                    <i class="fas fa-arrow-left me-1"></i>Back to Mapping
                                </button>
                                <button type="button" class="btn btn-outline-primary" onclick="addRowExcel()">
                                    <i class="fas fa-plus me-1"></i>Add Row
                                </button>
                                <button type="button" class="btn btn-outline-danger" onclick="deleteSelectedRowsExcel()">
                                    <i class="fas fa-trash me-1"></i>Delete Selected
                                </button>
                            </div>
                            <div>
                                <span class="me-3 text-muted">
                                    <i class="fas fa-users me-1"></i>
                                    <span id="rowCount">0</span> athletes ready to import
                                </span>
                                <button type="button" class="btn btn-success btn-lg" onclick="importExcelData()">
                                    <i class="fas fa-upload me-2"></i>Import to Database
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Assignment Modal -->
    <div class="modal fade" id="assignModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary">
                    <h5 class="modal-title text-white">
                        <i class="fas fa-calendar-plus me-2"></i>Assign Athletes to Event
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="assignAlert"></div>
                    
                    <form id="assignForm" method="post">
                        <input type="hidden" name="assign_to_event" value="1">
                        <!-- Participant IDs will be added dynamically by JavaScript -->
                        
                        <!-- Selected Athletes Display -->
                        <div class="mb-4">
                            <label class="form-label fw-semibold">Selected Athletes</label>
                            <div id="selectedParticipantsDisplay" class="p-3 bg-light rounded border">
                                <div>
                                    <i class="fas fa-info-circle me-1"></i>No athletes selected.
                                </div>
                            </div>
                        </div>
                        
                        <!-- Event Selection -->
                        <div class="mb-3">
                            <label for="assignEventSelect" class="form-label fw-semibold">
                                <i class="fas fa-calendar me-1 text-primary"></i>Target Event <span class="text-danger">*</span>
                            </label>
                            <select id="assignEventSelect" name="event_id" class="form-select form-select-lg" required onchange="loadEventHeats(this.value)">
                                <option value="">Select Event</option>
                                <?php foreach ($events as $event): ?>
                                    <option value="<?= $event['id'] ?>">
                                        <?= htmlspecialchars($event['name']) ?> 
                                        (<?= date('M d, Y', strtotime($event['date'])) ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <!-- Assignment Options -->
                        <div class="mb-4">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-cogs me-1"></i>Assignment Options
                            </label>
                            
                            <div class="row g-3">
                                <div class="col-12">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" name="assign_to_event_participants" id="assignToEventParticipants" onchange="toggleHeatOptions()">
                                        <label class="form-check-label fw-semibold" for="assignToEventParticipants">
                                            <i class="fas fa-fire me-1"></i>Assign to Event Competition (Heat Assignment)
                                        </label>
                                    </div>
                                    <div class="form-text">
                                        <small>
                                            <strong>Unchecked:</strong> Only track athletes for this event<br>
                                            <strong>Checked:</strong> Also assign to event competition with heat assignment
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Heat Assignment Options -->
                        <div id="heatOptions" class="mb-3" style="display: none;">
                            <div class="card border-info">
                                <div class="card-header bg-info bg-opacity-10">
                                    <h6 class="mb-0 fw-bold text-info">
                                        <i class="fas fa-fire me-2"></i>Heat Assignment Options
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div class="row g-3">
                                        <div class="col-md-6">
                                            <label for="assignHeatSelect" class="form-label fw-semibold">Heat (Optional)</label>
                                            <select id="assignHeatSelect" name="heat_number" class="form-select" disabled>
                                                <option value="">No specific heat (assign later)</option>
                                            </select>
                                            <div class="form-text">
                                                <small id="heatInfo">Select an event to see available heats</small>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="assignCategory" class="form-label fw-semibold">Division Override</label>
                                            <input type="text" id="assignCategory" name="category" class="form-control" placeholder="Leave empty to use athlete's division">
                                            <div class="form-text">
                                                <small>Override the athlete's default division for this event</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Assignment Summary -->
                        <div class="alert alert-info">
                            <div class="d-flex">
                                <i class="fas fa-info-circle fa-lg me-2 mt-1"></i>
                                <div>
                                    <strong>Assignment Process:</strong>
                                    <ul class="mb-0 mt-1 small">
                                        <li><strong>Step 1:</strong> Track athletes for the selected event (always done)</li>
                                        <li><strong>Step 2:</strong> Optionally assign to event competition with heat details</li>
                                        <li>Duplicate assignments will be detected and skipped</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" onclick="submitAssignment()">
                        <i class="fas fa-calendar-plus me-2"></i>Assign Athletes
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Help Modal -->
    <div class="modal fade" id="helpModal" tabindex="-1">
        <div class="modal-dialog modal-xl modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="fas fa-users me-2"></i>Participant Management Help</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body p-0" style="height:80vh;">
                    <iframe src="../help/participants_help.html" style="width:100%;height:100%;border:0;"></iframe>
                </div>
            </div>
        </div>
    </div>

    <!-- Column Visibility Modal -->
    <div class="modal fade" id="columnVisibilityModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-info text-white">
                    <h5 class="modal-title"><i class="fas fa-columns me-2"></i>Column Visibility</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p class="text-muted small mb-3">Select which columns to display in the table</p>
                    <div id="columnVisibilityOptions" class="d-flex flex-column gap-2">
                        <!-- Column checkboxes will be dynamically generated here -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" onclick="resetColumnVisibility()">
                        <i class="fas fa-undo me-1"></i>Reset to Default
                    </button>
                    <button type="button" class="btn btn-primary" onclick="applyColumnVisibility()">
                        <i class="fas fa-check me-1"></i>Apply Changes
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Participant Modal -->
    <div class="modal fade" id="addParticipantModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title"><i class="fas fa-user-plus me-2"></i>Add New Participant</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="addParticipantAlert"></div>
                    <form id="addParticipantForm" enctype="multipart/form-data">
                        <div class="row g-3">
                            <div class="col-12"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-user me-1"></i>Personal Information</h6></div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">First Name <span class="text-danger">*</span></label>
                                <input type="text" name="first_name" class="form-control" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Last Name <span class="text-danger">*</span></label>
                                <input type="text" name="last_name" class="form-control" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Gender</label>
                                <select name="gender" class="form-select">
                                    <option value="">Select Gender</option>
                                    <option value="M">Male</option>
                                    <option value="F">Female</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Date of Birth</label>
                                <input type="date" name="date_of_birth" class="form-control">
                            </div>
                            <div class="col-12 mt-4"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-id-card me-1"></i>Competition Information</h6></div>
                            <div class="col-md-4">
                                <label class="form-label fw-semibold">BIB Number</label>
                                <input type="text" name="bib" class="form-control">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label fw-semibold">FIS Code</label>
                                <input type="text" name="fis_code" class="form-control">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label fw-semibold">Licence Code</label>
                                <input type="text" name="licence_code" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Licence Type</label>
                                <input type="text" name="licence_type" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Category</label>
                                <input type="text" name="category" class="form-control">
                            </div>
                            <div class="col-12 mt-4"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-flag me-1"></i>Club & Country</h6></div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Club</label>
                                <input type="text" name="club" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Country</label>
                                <select name="country" class="form-control country-code" id="addCountry">
                                    <option value="">Select Country...</option>
                                    <option value="AD">🇦🇩 Andorra</option>
                                    <option value="AE">🇦🇪 United Arab Emirates</option>
                                    <option value="AF">🇦🇫 Afghanistan</option>
                                    <option value="AG">🇦🇬 Antigua and Barbuda</option>
                                    <option value="AI">🇦🇮 Anguilla</option>
                                    <option value="AL">🇦🇱 Albania</option>
                                    <option value="AM">🇦🇲 Armenia</option>
                                    <option value="AO">🇦🇴 Angola</option>
                                    <option value="AQ">🇦🇶 Antarctica</option>
                                    <option value="AR">🇦🇷 Argentina</option>
                                    <option value="AS">🇦🇸 American Samoa</option>
                                    <option value="AT">🇦🇹 Austria</option>
                                    <option value="AU">🇦🇺 Australia</option>
                                    <option value="AW">🇦🇼 Aruba</option>
                                    <option value="AX">🇦🇽 Åland Islands</option>
                                    <option value="AZ">🇦🇿 Azerbaijan</option>
                                    <option value="BA">🇧🇦 Bosnia and Herzegovina</option>
                                    <option value="BB">🇧🇧 Barbados</option>
                                    <option value="BD">🇧🇩 Bangladesh</option>
                                    <option value="BE">🇧🇪 Belgium</option>
                                    <option value="BF">🇧🇫 Burkina Faso</option>
                                    <option value="BG">🇧🇬 Bulgaria</option>
                                    <option value="BH">🇧🇭 Bahrain</option>
                                    <option value="BI">🇧🇮 Burundi</option>
                                    <option value="BJ">🇧🇯 Benin</option>
                                    <option value="BL">🇧🇱 Saint Barthélemy</option>
                                    <option value="BM">🇧🇲 Bermuda</option>
                                    <option value="BN">🇧🇳 Brunei</option>
                                    <option value="BO">🇧🇴 Bolivia</option>
                                    <option value="BQ">🇧🇶 Caribbean Netherlands</option>
                                    <option value="BR">🇧🇷 Brazil</option>
                                    <option value="BS">🇧🇸 Bahamas</option>
                                    <option value="BT">🇧🇹 Bhutan</option>
                                    <option value="BV">🇧🇻 Bouvet Island</option>
                                    <option value="BW">🇧🇼 Botswana</option>
                                    <option value="BY">🇧🇾 Belarus</option>
                                    <option value="BZ">🇧🇿 Belize</option>
                                    <option value="CA">🇨🇦 Canada</option>
                                    <option value="CC">🇨🇨 Cocos Islands</option>
                                    <option value="CD">🇨🇩 Democratic Republic of the Congo</option>
                                    <option value="CF">🇨🇫 Central African Republic</option>
                                    <option value="CG">🇨🇬 Republic of the Congo</option>
                                    <option value="CH">🇨🇭 Switzerland</option>
                                    <option value="CI">🇨🇮 Côte d'Ivoire</option>
                                    <option value="CK">🇨🇰 Cook Islands</option>
                                    <option value="CL">🇨🇱 Chile</option>
                                    <option value="CM">🇨🇲 Cameroon</option>
                                    <option value="CN">🇨🇳 China</option>
                                    <option value="CO">🇨🇴 Colombia</option>
                                    <option value="CR">🇨🇷 Costa Rica</option>
                                    <option value="CU">🇨🇺 Cuba</option>
                                    <option value="CV">🇨🇻 Cape Verde</option>
                                    <option value="CW">🇨🇼 Curaçao</option>
                                    <option value="CX">🇨🇽 Christmas Island</option>
                                    <option value="CY">🇨🇾 Cyprus</option>
                                    <option value="CZ">🇨🇿 Czech Republic</option>
                                    <option value="DE">🇩🇪 Germany</option>
                                    <option value="DJ">🇩🇯 Djibouti</option>
                                    <option value="DK">🇩🇰 Denmark</option>
                                    <option value="DM">🇩🇲 Dominica</option>
                                    <option value="DO">🇩🇴 Dominican Republic</option>
                                    <option value="DZ">🇩🇿 Algeria</option>
                                    <option value="EC">🇪🇨 Ecuador</option>
                                    <option value="EE">🇪🇪 Estonia</option>
                                    <option value="EG">🇪🇬 Egypt</option>
                                    <option value="EH">🇪🇭 Western Sahara</option>
                                    <option value="ER">🇪🇷 Eritrea</option>
                                    <option value="ES">🇪🇸 Spain</option>
                                    <option value="ET">🇪🇹 Ethiopia</option>
                                    <option value="FI">🇫🇮 Finland</option>
                                    <option value="FJ">🇫🇯 Fiji</option>
                                    <option value="FK">🇫🇰 Falkland Islands</option>
                                    <option value="FM">🇫🇲 Micronesia</option>
                                    <option value="FO">🇫🇴 Faroe Islands</option>
                                    <option value="FR">🇫🇷 France</option>
                                    <option value="GA">🇬🇦 Gabon</option>
                                    <option value="GB">🇬🇧 United Kingdom</option>
                                    <option value="GD">🇬🇩 Grenada</option>
                                    <option value="GE">🇬🇪 Georgia</option>
                                    <option value="GF">🇬🇫 French Guiana</option>
                                    <option value="GG">🇬🇬 Guernsey</option>
                                    <option value="GH">🇬🇭 Ghana</option>
                                    <option value="GI">🇬🇮 Gibraltar</option>
                                    <option value="GL">🇬🇱 Greenland</option>
                                    <option value="GM">🇬🇲 Gambia</option>
                                    <option value="GN">🇬🇳 Guinea</option>
                                    <option value="GP">🇬🇵 Guadeloupe</option>
                                    <option value="GQ">🇬🇶 Equatorial Guinea</option>
                                    <option value="GR">🇬🇷 Greece</option>
                                    <option value="GS">🇬🇸 South Georgia and the South Sandwich Islands</option>
                                    <option value="GT">🇬🇹 Guatemala</option>
                                    <option value="GU">🇬🇺 Guam</option>
                                    <option value="GW">🇬🇼 Guinea-Bissau</option>
                                    <option value="GY">🇬🇾 Guyana</option>
                                    <option value="HK">🇭🇰 Hong Kong</option>
                                    <option value="HM">🇭🇲 Heard Island and McDonald Islands</option>
                                    <option value="HN">🇭🇳 Honduras</option>
                                    <option value="HR">🇭🇷 Croatia</option>
                                    <option value="HT">🇭🇹 Haiti</option>
                                    <option value="HU">🇭🇺 Hungary</option>
                                    <option value="ID">🇮🇩 Indonesia</option>
                                    <option value="IE">🇮🇪 Ireland</option>
                                    <option value="IL">🇮🇱 Israel</option>
                                    <option value="IM">🇮🇲 Isle of Man</option>
                                    <option value="IN">🇮🇳 India</option>
                                    <option value="IO">🇮🇴 British Indian Ocean Territory</option>
                                    <option value="IQ">🇮🇶 Iraq</option>
                                    <option value="IR">🇮🇷 Iran</option>
                                    <option value="IS">🇮🇸 Iceland</option>
                                    <option value="IT">🇮🇹 Italy</option>
                                    <option value="JE">🇯🇪 Jersey</option>
                                    <option value="JM">🇯🇲 Jamaica</option>
                                    <option value="JO">🇯🇴 Jordan</option>
                                    <option value="JP">🇯🇵 Japan</option>
                                    <option value="KE">🇰🇪 Kenya</option>
                                    <option value="KG">🇰🇬 Kyrgyzstan</option>
                                    <option value="KH">🇰🇭 Cambodia</option>
                                    <option value="KI">🇰🇮 Kiribati</option>
                                    <option value="KM">🇰🇲 Comoros</option>
                                    <option value="KN">🇰🇳 Saint Kitts and Nevis</option>
                                    <option value="KP">🇰🇵 North Korea</option>
                                    <option value="KR">🇰🇷 South Korea</option>
                                    <option value="KW">🇰🇼 Kuwait</option>
                                    <option value="KY">🇰🇾 Cayman Islands</option>
                                    <option value="KZ">🇰🇿 Kazakhstan</option>
                                    <option value="LA">🇱🇦 Laos</option>
                                    <option value="LB">🇱🇧 Lebanon</option>
                                    <option value="LC">🇱🇨 Saint Lucia</option>
                                    <option value="LI">🇱🇮 Liechtenstein</option>
                                    <option value="LK">🇱🇰 Sri Lanka</option>
                                    <option value="LR">🇱🇷 Liberia</option>
                                    <option value="LS">🇱🇸 Lesotho</option>
                                    <option value="LT">🇱🇹 Lithuania</option>
                                    <option value="LU">🇱🇺 Luxembourg</option>
                                    <option value="LV">🇱🇻 Latvia</option>
                                    <option value="LY">🇱🇾 Libya</option>
                                    <option value="MA">🇲🇦 Morocco</option>
                                    <option value="MC">🇲🇨 Monaco</option>
                                    <option value="MD">🇲🇩 Moldova</option>
                                    <option value="ME">🇲🇪 Montenegro</option>
                                    <option value="MF">🇲🇫 Saint Martin</option>
                                    <option value="MG">🇲🇬 Madagascar</option>
                                    <option value="MH">🇲🇭 Marshall Islands</option>
                                    <option value="MK">🇲🇰 North Macedonia</option>
                                    <option value="ML">🇲🇱 Mali</option>
                                    <option value="MM">🇲🇲 Myanmar</option>
                                    <option value="MN">🇲🇳 Mongolia</option>
                                    <option value="MO">🇲🇴 Macao</option>
                                    <option value="MP">🇲🇵 Northern Mariana Islands</option>
                                    <option value="MQ">🇲🇶 Martinique</option>
                                    <option value="MR">🇲🇷 Mauritania</option>
                                    <option value="MS">🇲🇸 Montserrat</option>
                                    <option value="MT">🇲🇹 Malta</option>
                                    <option value="MU">🇲🇺 Mauritius</option>
                                    <option value="MV">🇲🇻 Maldives</option>
                                    <option value="MW">🇲🇼 Malawi</option>
                                    <option value="MX">🇲🇽 Mexico</option>
                                    <option value="MY">🇲🇾 Malaysia</option>
                                    <option value="MZ">🇲🇿 Mozambique</option>
                                    <option value="NA">🇳🇦 Namibia</option>
                                    <option value="NC">🇳🇨 New Caledonia</option>
                                    <option value="NE">🇳🇪 Niger</option>
                                    <option value="NF">🇳🇫 Norfolk Island</option>
                                    <option value="NG">🇳🇬 Nigeria</option>
                                    <option value="NI">🇳🇮 Nicaragua</option>
                                    <option value="NL">🇳🇱 Netherlands</option>
                                    <option value="NO">🇳🇴 Norway</option>
                                    <option value="NP">🇳🇵 Nepal</option>
                                    <option value="NR">🇳🇷 Nauru</option>
                                    <option value="NU">🇳🇺 Niue</option>
                                    <option value="NZ">🇳🇿 New Zealand</option>
                                    <option value="OM">🇴🇲 Oman</option>
                                    <option value="PA">🇵🇦 Panama</option>
                                    <option value="PE">🇵🇪 Peru</option>
                                    <option value="PF">🇵🇫 French Polynesia</option>
                                    <option value="PG">🇵🇬 Papua New Guinea</option>
                                    <option value="PH">🇵🇭 Philippines</option>
                                    <option value="PK">🇵🇰 Pakistan</option>
                                    <option value="PL">🇵🇱 Poland</option>
                                    <option value="PM">🇵🇲 Saint Pierre and Miquelon</option>
                                    <option value="PN">🇵🇳 Pitcairn</option>
                                    <option value="PR">🇵🇷 Puerto Rico</option>
                                    <option value="PS">🇵🇸 Palestinian Territory</option>
                                    <option value="PT">🇵🇹 Portugal</option>
                                    <option value="PW">🇵🇼 Palau</option>
                                    <option value="PY">🇵🇾 Paraguay</option>
                                    <option value="QA">🇶🇦 Qatar</option>
                                    <option value="RE">🇷🇪 Réunion</option>
                                    <option value="RO">🇷🇴 Romania</option>
                                    <option value="RS">🇷🇸 Serbia</option>
                                    <option value="RU">🇷🇺 Russia</option>
                                    <option value="RW">🇷🇼 Rwanda</option>
                                    <option value="SA">🇸🇦 Saudi Arabia</option>
                                    <option value="SB">🇸🇧 Solomon Islands</option>
                                    <option value="SC">🇸🇨 Seychelles</option>
                                    <option value="SD">🇸🇩 Sudan</option>
                                    <option value="SE">🇸🇪 Sweden</option>
                                    <option value="SG">🇸🇬 Singapore</option>
                                    <option value="SH">🇸🇭 Saint Helena</option>
                                    <option value="SI">🇸🇮 Slovenia</option>
                                    <option value="SJ">🇸🇯 Svalbard and Jan Mayen</option>
                                    <option value="SK">🇸🇰 Slovakia</option>
                                    <option value="SL">🇸🇱 Sierra Leone</option>
                                    <option value="SM">🇸🇲 San Marino</option>
                                    <option value="SN">🇸🇳 Senegal</option>
                                    <option value="SO">🇸🇴 Somalia</option>
                                    <option value="SR">🇸🇷 Suriname</option>
                                    <option value="SS">🇸🇸 South Sudan</option>
                                    <option value="ST">🇸🇹 São Tomé and Príncipe</option>
                                    <option value="SV">🇸🇻 El Salvador</option>
                                    <option value="SX">🇸🇽 Sint Maarten</option>
                                    <option value="SY">🇸🇾 Syria</option>
                                    <option value="SZ">🇸🇿 Eswatini</option>
                                    <option value="TC">🇹🇨 Turks and Caicos Islands</option>
                                    <option value="TD">🇹🇩 Chad</option>
                                    <option value="TF">🇹🇫 French Southern Territories</option>
                                    <option value="TG">🇹🇬 Togo</option>
                                    <option value="TH">🇹🇭 Thailand</option>
                                    <option value="TJ">🇹🇯 Tajikistan</option>
                                    <option value="TK">🇹🇰 Tokelau</option>
                                    <option value="TL">🇹🇱 Timor-Leste</option>
                                    <option value="TM">🇹🇲 Turkmenistan</option>
                                    <option value="TN">🇹🇳 Tunisia</option>
                                    <option value="TO">🇹🇴 Tonga</option>
                                    <option value="TR">🇹🇷 Turkey</option>
                                    <option value="TT">🇹🇹 Trinidad and Tobago</option>
                                    <option value="TV">🇹🇻 Tuvalu</option>
                                    <option value="TW">🇹🇼 Taiwan</option>
                                    <option value="TZ">🇹🇿 Tanzania</option>
                                    <option value="UA">🇺🇦 Ukraine</option>
                                    <option value="UG">🇺🇬 Uganda</option>
                                    <option value="UM">🇺🇲 United States Minor Outlying Islands</option>
                                    <option value="US">🇺🇸 United States</option>
                                    <option value="UY">🇺🇾 Uruguay</option>
                                    <option value="UZ">🇺🇿 Uzbekistan</option>
                                    <option value="VA">🇻🇦 Vatican City</option>
                                    <option value="VC">🇻🇨 Saint Vincent and the Grenadines</option>
                                    <option value="VE">🇻🇪 Venezuela</option>
                                    <option value="VG">🇻🇬 British Virgin Islands</option>
                                    <option value="VI">🇻🇮 U.S. Virgin Islands</option>
                                    <option value="VN">🇻🇳 Vietnam</option>
                                    <option value="VU">🇻🇺 Vanuatu</option>
                                    <option value="WF">🇼🇫 Wallis and Futuna</option>
                                    <option value="WS">🇼🇸 Samoa</option>
                                    <option value="YE">🇾🇪 Yemen</option>
                                    <option value="YT">🇾🇹 Mayotte</option>
                                    <option value="ZA">🇿🇦 South Africa</option>
                                    <option value="ZM">🇿🇲 Zambia</option>
                                    <option value="ZW">🇿🇼 Zimbabwe</option>
                                </select>
                            </div>
                            
                            <?php if (!empty($custom_field_defs)): ?>
                            <div class="col-12 mt-4"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-sliders-h me-1"></i>Custom Fields</h6></div>
                            <?php foreach ($custom_field_defs as $field): ?>
                                <div class="col-md-6">
                                    <label class="form-label fw-semibold">
                                        <?= htmlspecialchars($field['field_label']) ?>
                                        <?php if ($field['is_required']): ?><span class="text-danger">*</span><?php endif; ?>
                                    </label>
                                    <?php if ($field['field_type'] === 'select'): ?>
                                        <select name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" class="form-select" <?= $field['is_required'] ? 'required' : '' ?>>
                                            <option value="">Select...</option>
                                            <?php 
                                            $options = json_decode($field['field_options'], true) ?? [];
                                            foreach ($options as $opt): 
                                            ?>
                                                <option value="<?= htmlspecialchars($opt) ?>"><?= htmlspecialchars($opt) ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php elseif ($field['field_type'] === 'textarea'): ?>
                                        <textarea name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" class="form-control" rows="3" <?= $field['is_required'] ? 'required' : '' ?>></textarea>
                                    <?php elseif ($field['field_type'] === 'date'): ?>
                                        <input type="date" name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" class="form-control" <?= $field['is_required'] ? 'required' : '' ?>>
                                    <?php elseif ($field['field_type'] === 'number'): ?>
                                        <input type="number" name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" class="form-control" <?= $field['is_required'] ? 'required' : '' ?>>
                                    <?php else: ?>
                                        <input type="text" name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" class="form-control" <?= $field['is_required'] ? 'required' : '' ?>>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" onclick="submitAddParticipant()">
                        <i class="fas fa-user-plus me-2"></i>Add Participant
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Participant Modal -->
    <div class="modal fade" id="editParticipantModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title"><i class="fas fa-user-edit me-2"></i>Edit Participant</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="editParticipantAlert"></div>
                    <form id="editParticipantForm" enctype="multipart/form-data">
                        <input type="hidden" name="id" id="editParticipantId">
                        <div class="row g-3">
                            <div class="col-12"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-user me-1"></i>Personal Information</h6></div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">First Name <span class="text-danger">*</span></label>
                                <input type="text" name="first_name" id="editFirstName" class="form-control" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Last Name <span class="text-danger">*</span></label>
                                <input type="text" name="last_name" id="editLastName" class="form-control" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Gender</label>
                                <select name="gender" id="editGender" class="form-select">
                                    <option value="">Select Gender</option>
                                    <option value="M">Male</option>
                                    <option value="F">Female</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Date of Birth</label>
                                <input type="date" name="date_of_birth" id="editDateOfBirth" class="form-control">
                            </div>
                            <div class="col-12 mt-4"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-id-card me-1"></i>Competition Information</h6></div>
                            <div class="col-md-4">
                                <label class="form-label fw-semibold">BIB Number</label>
                                <input type="text" name="bib" id="editBib" class="form-control">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label fw-semibold">FIS Code</label>
                                <input type="text" name="fis_code" id="editFisCode" class="form-control">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label fw-semibold">Licence Code</label>
                                <input type="text" name="licence_code" id="editLicenceCode" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Licence Type</label>
                                <input type="text" name="licence_type" id="editLicenceType" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Category</label>
                                <input type="text" name="category" id="editCategory" class="form-control">
                            </div>
                            <div class="col-12 mt-4"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-flag me-1"></i>Club & Country</h6></div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Club</label>
                                <input type="text" name="club" id="editClub" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">Country</label>
                                <select name="country" class="form-control country-code" id="editCountry">
                                    <option value="">Select Country...</option>
                                    <option value="AD">🇦🇩 Andorra</option>
                                    <option value="AE">🇦🇪 United Arab Emirates</option>
                                    <option value="AF">🇦🇫 Afghanistan</option>
                                    <option value="AG">🇦🇬 Antigua and Barbuda</option>
                                    <option value="AI">🇦🇮 Anguilla</option>
                                    <option value="AL">🇦🇱 Albania</option>
                                    <option value="AM">🇦🇲 Armenia</option>
                                    <option value="AO">🇦🇴 Angola</option>
                                    <option value="AQ">🇦🇶 Antarctica</option>
                                    <option value="AR">🇦🇷 Argentina</option>
                                    <option value="AS">🇦🇸 American Samoa</option>
                                    <option value="AT">🇦🇹 Austria</option>
                                    <option value="AU">🇦🇺 Australia</option>
                                    <option value="AW">🇦🇼 Aruba</option>
                                    <option value="AX">🇦🇽 Åland Islands</option>
                                    <option value="AZ">🇦🇿 Azerbaijan</option>
                                    <option value="BA">🇧🇦 Bosnia and Herzegovina</option>
                                    <option value="BB">🇧🇧 Barbados</option>
                                    <option value="BD">🇧🇩 Bangladesh</option>
                                    <option value="BE">🇧🇪 Belgium</option>
                                    <option value="BF">🇧🇫 Burkina Faso</option>
                                    <option value="BG">🇧🇬 Bulgaria</option>
                                    <option value="BH">🇧🇭 Bahrain</option>
                                    <option value="BI">🇧🇮 Burundi</option>
                                    <option value="BJ">🇧🇯 Benin</option>
                                    <option value="BL">🇧🇱 Saint Barthélemy</option>
                                    <option value="BM">🇧🇲 Bermuda</option>
                                    <option value="BN">🇧🇳 Brunei</option>
                                    <option value="BO">🇧🇴 Bolivia</option>
                                    <option value="BQ">🇧🇶 Caribbean Netherlands</option>
                                    <option value="BR">🇧🇷 Brazil</option>
                                    <option value="BS">🇧🇸 Bahamas</option>
                                    <option value="BT">🇧🇹 Bhutan</option>
                                    <option value="BV">🇧🇻 Bouvet Island</option>
                                    <option value="BW">🇧🇼 Botswana</option>
                                    <option value="BY">🇧🇾 Belarus</option>
                                    <option value="BZ">🇧🇿 Belize</option>
                                    <option value="CA">🇨🇦 Canada</option>
                                    <option value="CC">🇨🇨 Cocos Islands</option>
                                    <option value="CD">🇨🇩 Democratic Republic of the Congo</option>
                                    <option value="CF">🇨🇫 Central African Republic</option>
                                    <option value="CG">🇨🇬 Republic of the Congo</option>
                                    <option value="CH">🇨🇭 Switzerland</option>
                                    <option value="CI">🇨🇮 Côte d'Ivoire</option>
                                    <option value="CK">🇨🇰 Cook Islands</option>
                                    <option value="CL">🇨🇱 Chile</option>
                                    <option value="CM">🇨🇲 Cameroon</option>
                                    <option value="CN">🇨🇳 China</option>
                                    <option value="CO">🇨🇴 Colombia</option>
                                    <option value="CR">🇨🇷 Costa Rica</option>
                                    <option value="CU">🇨🇺 Cuba</option>
                                    <option value="CV">🇨🇻 Cape Verde</option>
                                    <option value="CW">🇨🇼 Curaçao</option>
                                    <option value="CX">🇨🇽 Christmas Island</option>
                                    <option value="CY">🇨🇾 Cyprus</option>
                                    <option value="CZ">🇨🇿 Czech Republic</option>
                                    <option value="DE">🇩🇪 Germany</option>
                                    <option value="DJ">🇩🇯 Djibouti</option>
                                    <option value="DK">🇩🇰 Denmark</option>
                                    <option value="DM">🇩🇲 Dominica</option>
                                    <option value="DO">🇩🇴 Dominican Republic</option>
                                    <option value="DZ">🇩🇿 Algeria</option>
                                    <option value="EC">🇪🇨 Ecuador</option>
                                    <option value="EE">🇪🇪 Estonia</option>
                                    <option value="EG">🇪🇬 Egypt</option>
                                    <option value="EH">🇪🇭 Western Sahara</option>
                                    <option value="ER">🇪🇷 Eritrea</option>
                                    <option value="ES">🇪🇸 Spain</option>
                                    <option value="ET">🇪🇹 Ethiopia</option>
                                    <option value="FI">🇫🇮 Finland</option>
                                    <option value="FJ">🇫🇯 Fiji</option>
                                    <option value="FK">🇫🇰 Falkland Islands</option>
                                    <option value="FM">🇫🇲 Micronesia</option>
                                    <option value="FO">🇫🇴 Faroe Islands</option>
                                    <option value="FR">🇫🇷 France</option>
                                    <option value="GA">🇬🇦 Gabon</option>
                                    <option value="GB">🇬🇧 United Kingdom</option>
                                    <option value="GD">🇬🇩 Grenada</option>
                                    <option value="GE">🇬🇪 Georgia</option>
                                    <option value="GF">🇬🇫 French Guiana</option>
                                    <option value="GG">🇬🇬 Guernsey</option>
                                    <option value="GH">🇬🇭 Ghana</option>
                                    <option value="GI">🇬🇮 Gibraltar</option>
                                    <option value="GL">🇬🇱 Greenland</option>
                                    <option value="GM">🇬🇲 Gambia</option>
                                    <option value="GN">🇬🇳 Guinea</option>
                                    <option value="GP">🇬🇵 Guadeloupe</option>
                                    <option value="GQ">🇬🇶 Equatorial Guinea</option>
                                    <option value="GR">🇬🇷 Greece</option>
                                    <option value="GS">🇬🇸 South Georgia and the South Sandwich Islands</option>
                                    <option value="GT">🇬🇹 Guatemala</option>
                                    <option value="GU">🇬🇺 Guam</option>
                                    <option value="GW">🇬🇼 Guinea-Bissau</option>
                                    <option value="GY">🇬🇾 Guyana</option>
                                    <option value="HK">🇭🇰 Hong Kong</option>
                                    <option value="HM">🇭🇲 Heard Island and McDonald Islands</option>
                                    <option value="HN">🇭🇳 Honduras</option>
                                    <option value="HR">🇭🇷 Croatia</option>
                                    <option value="HT">🇭🇹 Haiti</option>
                                    <option value="HU">🇭🇺 Hungary</option>
                                    <option value="ID">🇮🇩 Indonesia</option>
                                    <option value="IE">🇮🇪 Ireland</option>
                                    <option value="IL">🇮🇱 Israel</option>
                                    <option value="IM">🇮🇲 Isle of Man</option>
                                    <option value="IN">🇮🇳 India</option>
                                    <option value="IO">🇮🇴 British Indian Ocean Territory</option>
                                    <option value="IQ">🇮🇶 Iraq</option>
                                    <option value="IR">🇮🇷 Iran</option>
                                    <option value="IS">🇮🇸 Iceland</option>
                                    <option value="IT">🇮🇹 Italy</option>
                                    <option value="JE">🇯🇪 Jersey</option>
                                    <option value="JM">🇯🇲 Jamaica</option>
                                    <option value="JO">🇯🇴 Jordan</option>
                                    <option value="JP">🇯🇵 Japan</option>
                                    <option value="KE">🇰🇪 Kenya</option>
                                    <option value="KG">🇰🇬 Kyrgyzstan</option>
                                    <option value="KH">🇰🇭 Cambodia</option>
                                    <option value="KI">🇰🇮 Kiribati</option>
                                    <option value="KM">🇰🇲 Comoros</option>
                                    <option value="KN">🇰🇳 Saint Kitts and Nevis</option>
                                    <option value="KP">🇰🇵 North Korea</option>
                                    <option value="KR">🇰🇷 South Korea</option>
                                    <option value="KW">🇰🇼 Kuwait</option>
                                    <option value="KY">🇰🇾 Cayman Islands</option>
                                    <option value="KZ">🇰🇿 Kazakhstan</option>
                                    <option value="LA">🇱🇦 Laos</option>
                                    <option value="LB">🇱🇧 Lebanon</option>
                                    <option value="LC">🇱🇨 Saint Lucia</option>
                                    <option value="LI">🇱🇮 Liechtenstein</option>
                                    <option value="LK">🇱🇰 Sri Lanka</option>
                                    <option value="LR">🇱🇷 Liberia</option>
                                    <option value="LS">🇱🇸 Lesotho</option>
                                    <option value="LT">🇱🇹 Lithuania</option>
                                    <option value="LU">🇱🇺 Luxembourg</option>
                                    <option value="LV">🇱🇻 Latvia</option>
                                    <option value="LY">🇱🇾 Libya</option>
                                    <option value="MA">🇲🇦 Morocco</option>
                                    <option value="MC">🇲🇨 Monaco</option>
                                    <option value="MD">🇲🇩 Moldova</option>
                                    <option value="ME">🇲🇪 Montenegro</option>
                                    <option value="MF">🇲🇫 Saint Martin</option>
                                    <option value="MG">🇲🇬 Madagascar</option>
                                    <option value="MH">🇲🇭 Marshall Islands</option>
                                    <option value="MK">🇲🇰 North Macedonia</option>
                                    <option value="ML">🇲🇱 Mali</option>
                                    <option value="MM">🇲🇲 Myanmar</option>
                                    <option value="MN">🇲🇳 Mongolia</option>
                                    <option value="MO">🇲🇴 Macao</option>
                                    <option value="MP">🇲🇵 Northern Mariana Islands</option>
                                    <option value="MQ">🇲🇶 Martinique</option>
                                    <option value="MR">🇲🇷 Mauritania</option>
                                    <option value="MS">🇲🇸 Montserrat</option>
                                    <option value="MT">🇲🇹 Malta</option>
                                    <option value="MU">🇲🇺 Mauritius</option>
                                    <option value="MV">🇲🇻 Maldives</option>
                                    <option value="MW">🇲🇼 Malawi</option>
                                    <option value="MX">🇲🇽 Mexico</option>
                                    <option value="MY">🇲🇾 Malaysia</option>
                                    <option value="MZ">🇲🇿 Mozambique</option>
                                    <option value="NA">🇳🇦 Namibia</option>
                                    <option value="NC">🇳🇨 New Caledonia</option>
                                    <option value="NE">🇳🇪 Niger</option>
                                    <option value="NF">🇳🇫 Norfolk Island</option>
                                    <option value="NG">🇳🇬 Nigeria</option>
                                    <option value="NI">🇳🇮 Nicaragua</option>
                                    <option value="NL">🇳🇱 Netherlands</option>
                                    <option value="NO">🇳🇴 Norway</option>
                                    <option value="NP">🇳🇵 Nepal</option>
                                    <option value="NR">🇳🇷 Nauru</option>
                                    <option value="NU">🇳🇺 Niue</option>
                                    <option value="NZ">🇳🇿 New Zealand</option>
                                    <option value="OM">🇴🇲 Oman</option>
                                    <option value="PA">🇵🇦 Panama</option>
                                    <option value="PE">🇵🇪 Peru</option>
                                    <option value="PF">🇵🇫 French Polynesia</option>
                                    <option value="PG">🇵🇬 Papua New Guinea</option>
                                    <option value="PH">🇵🇭 Philippines</option>
                                    <option value="PK">🇵🇰 Pakistan</option>
                                    <option value="PL">🇵🇱 Poland</option>
                                    <option value="PM">🇵🇲 Saint Pierre and Miquelon</option>
                                    <option value="PN">🇵🇳 Pitcairn</option>
                                    <option value="PR">🇵🇷 Puerto Rico</option>
                                    <option value="PS">🇵🇸 Palestinian Territory</option>
                                    <option value="PT">🇵🇹 Portugal</option>
                                    <option value="PW">🇵🇼 Palau</option>
                                    <option value="PY">🇵🇾 Paraguay</option>
                                    <option value="QA">🇶🇦 Qatar</option>
                                    <option value="RE">🇷🇪 Réunion</option>
                                    <option value="RO">🇷🇴 Romania</option>
                                    <option value="RS">🇷🇸 Serbia</option>
                                    <option value="RU">🇷🇺 Russia</option>
                                    <option value="RW">🇷🇼 Rwanda</option>
                                    <option value="SA">🇸🇦 Saudi Arabia</option>
                                    <option value="SB">🇸🇧 Solomon Islands</option>
                                    <option value="SC">🇸🇨 Seychelles</option>
                                    <option value="SD">🇸🇩 Sudan</option>
                                    <option value="SE">🇸🇪 Sweden</option>
                                    <option value="SG">🇸🇬 Singapore</option>
                                    <option value="SH">🇸🇭 Saint Helena</option>
                                    <option value="SI">🇸🇮 Slovenia</option>
                                    <option value="SJ">🇸🇯 Svalbard and Jan Mayen</option>
                                    <option value="SK">🇸🇰 Slovakia</option>
                                    <option value="SL">🇸🇱 Sierra Leone</option>
                                    <option value="SM">🇸🇲 San Marino</option>
                                    <option value="SN">🇸🇳 Senegal</option>
                                    <option value="SO">🇸🇴 Somalia</option>
                                    <option value="SR">🇸🇷 Suriname</option>
                                    <option value="SS">🇸🇸 South Sudan</option>
                                    <option value="ST">🇸🇹 São Tomé and Príncipe</option>
                                    <option value="SV">🇸🇻 El Salvador</option>
                                    <option value="SX">🇸🇽 Sint Maarten</option>
                                    <option value="SY">🇸🇾 Syria</option>
                                    <option value="SZ">🇸🇿 Eswatini</option>
                                    <option value="TC">🇹🇨 Turks and Caicos Islands</option>
                                    <option value="TD">🇹🇩 Chad</option>
                                    <option value="TF">🇹🇫 French Southern Territories</option>
                                    <option value="TG">🇹🇬 Togo</option>
                                    <option value="TH">🇹🇭 Thailand</option>
                                    <option value="TJ">🇹🇯 Tajikistan</option>
                                    <option value="TK">🇹🇰 Tokelau</option>
                                    <option value="TL">🇹🇱 Timor-Leste</option>
                                    <option value="TM">🇹🇲 Turkmenistan</option>
                                    <option value="TN">🇹🇳 Tunisia</option>
                                    <option value="TO">🇹🇴 Tonga</option>
                                    <option value="TR">🇹🇷 Turkey</option>
                                    <option value="TT">🇹🇹 Trinidad and Tobago</option>
                                    <option value="TV">🇹🇻 Tuvalu</option>
                                    <option value="TW">🇹🇼 Taiwan</option>
                                    <option value="TZ">🇹🇿 Tanzania</option>
                                    <option value="UA">🇺🇦 Ukraine</option>
                                    <option value="UG">🇺🇬 Uganda</option>
                                    <option value="UM">🇺🇲 United States Minor Outlying Islands</option>
                                    <option value="US">🇺🇸 United States</option>
                                    <option value="UY">🇺🇾 Uruguay</option>
                                    <option value="UZ">🇺🇿 Uzbekistan</option>
                                    <option value="VA">🇻🇦 Vatican City</option>
                                    <option value="VC">🇻🇨 Saint Vincent and the Grenadines</option>
                                    <option value="VE">🇻🇪 Venezuela</option>
                                    <option value="VG">🇻🇬 British Virgin Islands</option>
                                    <option value="VI">🇻🇮 U.S. Virgin Islands</option>
                                    <option value="VN">🇻🇳 Vietnam</option>
                                    <option value="VU">🇻🇺 Vanuatu</option>
                                    <option value="WF">🇼🇫 Wallis and Futuna</option>
                                    <option value="WS">🇼🇸 Samoa</option>
                                    <option value="YE">🇾🇪 Yemen</option>
                                    <option value="YT">🇾🇹 Mayotte</option>
                                    <option value="ZA">🇿🇦 South Africa</option>
                                    <option value="ZM">🇿🇲 Zambia</option>
                                    <option value="ZW">🇿🇼 Zimbabwe</option>
                                </select>
                            </div>
                            
                            <?php if (!empty($custom_field_defs)): ?>
                            <div class="col-12 mt-4"><h6 class="fw-bold border-bottom pb-2"><i class="fas fa-sliders-h me-1"></i>Custom Fields</h6></div>
                            <?php foreach ($custom_field_defs as $field): ?>
                                <div class="col-md-6">
                                    <label class="form-label fw-semibold">
                                        <?= htmlspecialchars($field['field_label']) ?>
                                        <?php if ($field['is_required']): ?><span class="text-danger">*</span><?php endif; ?>
                                    </label>
                                    <?php if ($field['field_type'] === 'select'): ?>
                                        <select name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" id="edit_cf_<?= htmlspecialchars($field['field_key']) ?>" class="form-select" <?= $field['is_required'] ? 'required' : '' ?>>
                                            <option value="">Select...</option>
                                            <?php 
                                            $options = json_decode($field['field_options'], true) ?? [];
                                            foreach ($options as $opt): 
                                            ?>
                                                <option value="<?= htmlspecialchars($opt) ?>"><?= htmlspecialchars($opt) ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php elseif ($field['field_type'] === 'textarea'): ?>
                                        <textarea name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" id="edit_cf_<?= htmlspecialchars($field['field_key']) ?>" class="form-control" rows="3" <?= $field['is_required'] ? 'required' : '' ?>></textarea>
                                    <?php elseif ($field['field_type'] === 'date'): ?>
                                        <input type="date" name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" id="edit_cf_<?= htmlspecialchars($field['field_key']) ?>" class="form-control" <?= $field['is_required'] ? 'required' : '' ?>>
                                    <?php elseif ($field['field_type'] === 'number'): ?>
                                        <input type="number" name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" id="edit_cf_<?= htmlspecialchars($field['field_key']) ?>" class="form-control" <?= $field['is_required'] ? 'required' : '' ?>>
                                    <?php else: ?>
                                        <input type="text" name="custom_fields[<?= htmlspecialchars($field['field_key']) ?>]" id="edit_cf_<?= htmlspecialchars($field['field_key']) ?>" class="form-control" <?= $field['is_required'] ? 'required' : '' ?>>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="submitEditParticipant()">
                        <i class="fas fa-save me-2"></i>Save Changes
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- jspreadsheet JS -->
    <script src="../assets/js/vendor/jspreadsheet/jsuites.min.js"></script>
    <script src="../assets/js/vendor/jspreadsheet/jspreadsheet.min.js"></script>

    <script>
        // Global state
        let spreadsheet = null;
        let participantsData = <?= json_encode($participants) ?>;
        let customFieldDefs = <?= json_encode($custom_field_defs) ?>;
        console.log('Custom Field Definitions loaded:', customFieldDefs);
        let originalData = [];
        let changedRows = new Set();
        let selectedRows = new Set();
        let assignModal = null;
        let currentCellSelection = null; // Track current cell selection range
        
        // Store heats data for JavaScript access
        const heatsData = <?= json_encode($heats) ?>;

        // Bootstrap modal helpers
        function showAlert(message, title = 'Notice', icon = 'info-circle') {
            document.getElementById('alertMessage').textContent = message;
            document.getElementById('alertTitle').textContent = title;
            document.getElementById('alertIcon').className = `fas fa-${icon} text-primary me-2`;
            
            const alertModal = document.getElementById('genericAlertModal');
            // Increase z-index if Excel paste modal is open
            if (document.getElementById('excelPasteModal').classList.contains('show')) {
                alertModal.style.zIndex = '9999';
                const backdrop = document.querySelector('.modal-backdrop:last-of-type');
                if (backdrop) backdrop.style.zIndex = '9998';
            }
            
            new bootstrap.Modal(alertModal).show();
        }

        function showConfirm(message, callback, title = 'Confirm', icon = 'question-circle') {
            document.getElementById('confirmMessage').textContent = message;
            document.getElementById('confirmTitle').textContent = title;
            document.getElementById('confirmIcon').className = `fas fa-${icon} text-warning me-2`;
            
            const confirmBtn = document.getElementById('confirmButton');
            const confirmModal = document.getElementById('genericConfirmModal');
            
            // Increase z-index if Excel paste modal is open
            if (document.getElementById('excelPasteModal').classList.contains('show')) {
                confirmModal.style.zIndex = '9999';
                setTimeout(() => {
                    const backdrop = document.querySelector('.modal-backdrop:last-of-type');
                    if (backdrop) backdrop.style.zIndex = '9998';
                }, 10);
            }
            
            const modal = new bootstrap.Modal(confirmModal);
            
            const newBtn = confirmBtn.cloneNode(true);
            confirmBtn.parentNode.replaceChild(newBtn, confirmBtn);
            
            newBtn.addEventListener('click', function() {
                modal.hide();
                callback();
            });
            
            modal.show();
        }

        // Initialize spreadsheet
        document.addEventListener('DOMContentLoaded', function() {
            initializeSpreadsheet();
            updateStatusBadge();
            
            // Handle window resize for responsive behavior
            let resizeTimer;
            window.addEventListener('resize', function() {
                clearTimeout(resizeTimer);
                resizeTimer = setTimeout(function() {
                    const wasMobile = window.innerWidth <= 768;
                    // You could reinitialize if needed, but it's heavy
                    // For now, just log or handle specific responsive updates
                }, 250);
            });
            
            // Auto-collapse filters on mobile after initial load
            if (window.innerWidth <= 768) {
                const filtersCollapse = document.getElementById('filtersCollapse');
                if (filtersCollapse) {
                    setTimeout(() => {
                        const bsCollapse = new bootstrap.Collapse(filtersCollapse, {
                            toggle: false
                        });
                        bsCollapse.hide();
                    }, 100);
                }
            }
        });

        function initializeSpreadsheet() {
            // Detect mobile device
            const isMobile = window.innerWidth <= 768;
            const isSmallMobile = window.innerWidth <= 576;
            
            // Prepare data for jspreadsheet
            const data = participantsData.map(p => {
                const row = [
                    p.id,
                    false, // Checkbox column
                    p.bib || '',
                    p.first_name || '',
                    p.last_name || '',
                    p.gender || '',
                    p.date_of_birth ? new Date(p.date_of_birth).getFullYear() : '',
                    p.club || '',
                    p.country || '',
                    p.fis_code || '',
                    p.licence_code || '',
                    p.licence_type || ''
                ];
                
                // Add custom field values
                if (customFieldDefs && Array.isArray(customFieldDefs)) {
                    customFieldDefs.forEach(field => {
                        if (field.is_active == 1 || field.is_active === true) {
                            let value = '';
                            if (p.custom_fields) {
                                const customData = typeof p.custom_fields === 'string' ? JSON.parse(p.custom_fields) : p.custom_fields;
                                value = customData[field.field_key] || '';
                            }
                            row.push(value);
                        }
                    });
                }
                
                // Action buttons - icons only on mobile
                row.push('<div class="btn-group btn-group-sm p-0" role="group">' +
                    '<button type="button" class="btn btn-outline-primary btn-sm" title="Edit Participant" onclick="showEditParticipantModal(' + p.id + ')">' +
                        '<i class="fas fa-edit"></i>' + (isMobile ? '' : '') +
                    '</button>' +
                    '<button type="button" class="btn btn-outline-info btn-sm" title="Assign to Event" onclick="showAssignModal(' + p.id + ', \'' + p.first_name + ' ' + p.last_name + '\', \'' + (p.bib || '') + '\')">' +
                        '<i class="fas fa-calendar-plus"></i>' + (isMobile ? '' : '') +
                    '</button>' +
                    '<button type="button" class="btn btn-outline-danger btn-sm" title="Delete" onclick="deleteParticipant(' + p.id + ', \'' + p.first_name + ' ' + p.last_name + '\')">' +
                        '<i class="fas fa-trash"></i>' + (isMobile ? '' : '') +
                    '</button>' +
                '</div>');
                
                return row;
            });

            originalData = JSON.parse(JSON.stringify(data));
            
            // Build columns array with custom fields
            const columns = [
                { type: 'text', title: 'ID', width: isMobile ? 50 : 60, readOnly: true, align: 'center' },
                { type: 'checkbox', title: '', width: isMobile ? 35 : 40, readOnly: false, align: 'center' },
                { type: 'text', title: 'BIB', width: isMobile ? 60 : 80, align: 'center' },
                { type: 'text', title: 'First Name', width: isMobile ? 100 : 150, align: 'left' },
                { type: 'text', title: 'Last Name', width: isMobile ? 100 : 150, align: 'left' },
                { 
                    type: 'dropdown', 
                    title: 'G', 
                    width: isMobile ? 60 : 100,
                    source: ['', 'M', 'F'],
                    autocomplete: true,
                    align: 'center'
                },
                { type: 'numeric', title: 'Year', width: isMobile ? 70 : 100, mask: '0000', align: 'center' },
                { type: 'text', title: 'Club', width: isMobile ? 120 : 200, align: 'left' },
                { 
                    type: 'dropdown', 
                    title: 'Country', 
                    width: isMobile ? 80 : 120,
                    source: ['', 'CZ', 'PL', 'SK', 'DE', 'AT', 'FR', 'IT', 'CH', 'US', 'CA', 'GB', 'NL', 'BE', 'ES', 'SE', 'NO', 'FI', 'RU'],
                    autocomplete: true,
                    align: 'center'
                },
                { type: 'text', title: 'FIS Code', width: isMobile ? 90 : 120, align: 'left' },
                { type: 'text', title: 'Licence', width: isMobile ? 90 : 120, align: 'left' },
                { type: 'text', title: 'Licence Type', width: isMobile ? 90 : 120, align: 'left' }
            ];
            
            // Add custom field columns
            if (customFieldDefs && Array.isArray(customFieldDefs)) {
                customFieldDefs.forEach(field => {
                    if (field.is_active == 1 || field.is_active === true) {
                        const colDef = {
                            type: field.field_type === 'select' ? 'dropdown' : 'text',
                            title: field.field_label,
                            width: 120,
                            align: 'left'
                        };
                        
                        // Add options for select fields
                        if (field.field_type === 'select' && field.field_options) {
                            try {
                                colDef.source = [''].concat(JSON.parse(field.field_options));
                                colDef.autocomplete = true;
                            } catch (e) {
                                colDef.source = [''];
                            }
                        }
                        
                        columns.push(colDef);
                    }
                });
            }
            
            // Add Actions column
            columns.push({ type: 'html', title: 'Actions', width: isMobile ? 100 : 160, readOnly: true, align: 'center' });

            spreadsheet = jspreadsheet(document.getElementById('spreadsheet'), {
                data: data,
                columns: columns,
                tableOverflow: true,
                tableWidth: '100%',
                tableHeight: isMobile ? '500px' : '600px',
                columnSorting: true,
                columnResize: !isMobile, // Disable column resize on mobile
                rowResize: false,
                search: true,
                pagination: isMobile ? 25 : 50, // Smaller page size on mobile
                paginationOptions: [25, 50, 100, 200],
                onchange: handleCellChange,
                onselection: handleSelection,
                freezeColumns: 0, // Don't use jspreadsheet's freeze, use CSS sticky instead
                
                updateTable: function(instance, cell, col, row, val, label, cellName) {
                    // Custom rendering for Gender column (index 5 - was 5, now shifted to 5)
                    if (col == 5 && val) {
                        if (val === 'M') {
                            cell.innerHTML = '<span class="badge bg-primary"><i class="fas fa-male me-1"></i>M</span>';
                        } else if (val === 'F') {
                            cell.innerHTML = '<span class="badge bg-danger"><i class="fas fa-female me-1"></i>F</span>';
                        }
                    }
                    // Custom rendering for Country column (index 8)
                    else if (col == 8 && val) {
                        const countryFlags = {
                            'CZ': '🇨🇿', 'PL': '🇵🇱', 'SK': '🇸🇰', 'DE': '🇩🇪', 
                            'AT': '🇦🇹', 'FR': '🇫🇷', 'IT': '🇮🇹', 'CH': '🇨🇭',
                            'US': '🇺🇸', 'CA': '🇨🇦', 'GB': '🇬🇧', 'NL': '🇳🇱',
                            'BE': '🇧🇪', 'ES': '🇪🇸', 'SE': '🇸🇪', 'NO': '🇳🇴',
                            'FI': '🇫🇮', 'RU': '🇷🇺'
                        };
                        if (countryFlags[val]) {
                            cell.innerHTML = '<span class="country-flag">' + countryFlags[val] + '</span> ' + val;
                        }
                    }
                },
                contextMenu: function(obj, x, y, e) {
                    const items = [];
                    
                    // Add "Check Selected Rows" if there's a cell selection
                    if (currentCellSelection) {
                        items.push({
                            title: 'Check Selected Rows',
                            onclick: function() {
                                checkSelectedCells();
                            }
                        });
                        
                        items.push({
                            title: 'Uncheck Selected Rows',
                            onclick: function() {
                                uncheckSelectedCells();
                            }
                        });
                    }
                    
                    if (selectedRows.size > 0) {
                        items.push({
                            title: 'Delete Checked Rows (' + selectedRows.size + ')',
                            onclick: function() {
                                deleteSelectedRows();
                            }
                        });
                    }
                    
                    items.push({
                        title: 'Add New Row',
                        onclick: function() {
                            addNewRow();
                        }
                    });

                    if (changedRows.size > 0) {
                        items.push({
                            title: 'Save Changes',
                            onclick: function() {
                                saveChanges();
                            }
                        });
                    }
                    
                    return items;
                }
            });
            
            // Apply saved column visibility settings
            setTimeout(() => {
                applyInitialColumnVisibility();
            }, 100);
        }
        
        function checkSelectedCells() {
            if (!currentCellSelection) return;
            
            const { y1, y2 } = currentCellSelection;
            const startRow = Math.min(y1, y2);
            const endRow = Math.max(y1, y2);
            
            // Check the checkbox for each selected row
            for (let y = startRow; y <= endRow; y++) {
                spreadsheet.setValue(1, y, true); // Column 1 is checkbox
            }
        }
        
        function uncheckSelectedCells() {
            if (!currentCellSelection) return;
            
            const { y1, y2 } = currentCellSelection;
            const startRow = Math.min(y1, y2);
            const endRow = Math.max(y1, y2);
            
            // Uncheck the checkbox for each selected row
            for (let y = startRow; y <= endRow; y++) {
                spreadsheet.setValue(1, y, false); // Column 1 is checkbox
            }
        }

        function handleCellChange(instance, cell, x, y, value) {
            // Handle checkbox column (index 1) to track selected rows
            if (x == 1) { // Checkbox column is now at index 1
                if (value === true || value === 'true' || value === 1) {
                    selectedRows.add(y);
                } else {
                    selectedRows.delete(y);
                }
                updateSelectionButtons();
            } else {
                // Track changes for save functionality (non-checkbox columns)
                changedRows.add(y);
                updateSaveButton();
            }
        }

        function handleSelection(instance, x1, y1, x2, y2) {
            // Store the current selection range for context menu use
            currentCellSelection = { x1, y1, x2, y2 };
            
            // Only handle selection if it's NOT the checkbox column
            // Checkbox column (x=1) is handled by handleCellChange
            if (x1 == 1 || x2 == 1) {
                return; // Ignore selection events on checkbox column
            }
            
            // For other columns, don't override the checkbox-based selection
            // Just keep the current selectedRows as is
        }

        function updateSaveButton() {
            const saveBtn = document.getElementById('saveBtn');
            const changesCount = document.getElementById('changesCount');
            
            if (!saveBtn || !changesCount) return;
            
            if (changedRows.size > 0) {
                saveBtn.disabled = false;
                changesCount.textContent = changedRows.size;
                changesCount.classList.remove('d-none');
            } else {
                saveBtn.disabled = true;
                changesCount.classList.add('d-none');
            }
        }

        function updateSelectionButtons() {
            const bulkAssignBtn = document.getElementById('bulkAssignBtn');
            const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
            const assignCount = document.getElementById('assignCount');
            const deleteCount = document.getElementById('deleteCount');
            const selectedCount = document.getElementById('selectedCount');
            
            if (!bulkAssignBtn || !bulkDeleteBtn || !assignCount || !deleteCount || !selectedCount) return;
            
            const count = selectedRows.size;
            
            bulkAssignBtn.disabled = count === 0;
            bulkDeleteBtn.disabled = count === 0;
            assignCount.textContent = count;
            deleteCount.textContent = count;
            selectedCount.textContent = count;
        }

        function updateStatusBadge() {
            const data = spreadsheet.getData();
            document.getElementById('visibleCount').textContent = data.length;
        }

        // Column Visibility Management
        let columnVisibilityModal = null;
        let hiddenColumns = new Set(JSON.parse(localStorage.getItem('hiddenColumns') || '[]'));
        
        // Build column definitions dynamically including custom fields
        function buildColumnDefinitions() {
            const baseColumns = [
                { index: 0, name: 'ID', mandatory: true },
                { index: 1, name: 'Select', mandatory: true },
                { index: 2, name: 'BIB', mandatory: false },
                { index: 3, name: 'First Name', mandatory: false },
                { index: 4, name: 'Last Name', mandatory: false },
                { index: 5, name: 'Gender', mandatory: false },
                { index: 6, name: 'Year of Birth', mandatory: false },
                { index: 7, name: 'Club', mandatory: false },
                { index: 8, name: 'Country', mandatory: false },
                { index: 9, name: 'FIS Code', mandatory: false },
                { index: 10, name: 'Licence', mandatory: false },
                { index: 11, name: 'Licence Type', mandatory: false }
            ];
            
            let currentIndex = 12;
            
            // Add custom field columns
            if (customFieldDefs && Array.isArray(customFieldDefs)) {
                customFieldDefs.forEach(field => {
                    if (field.is_active == 1 || field.is_active === true) {
                        baseColumns.push({
                            index: currentIndex,
                            name: field.field_label + ' (Custom)',
                            mandatory: false,
                            isCustomField: true,
                            fieldKey: field.field_key
                        });
                        currentIndex++;
                    }
                });
            }
            
            // Actions column (always last)
            baseColumns.push({ index: currentIndex, name: 'Actions', mandatory: true });
            
            return baseColumns;
        }
        
        let columnDefinitions = buildColumnDefinitions();
        
        function showColumnVisibilityModal() {
            if (!columnVisibilityModal) {
                columnVisibilityModal = new bootstrap.Modal(document.getElementById('columnVisibilityModal'));
            }
            
            // Generate checkboxes for each column
            const container = document.getElementById('columnVisibilityOptions');
            container.innerHTML = '';
            
            columnDefinitions.forEach(col => {
                const isHidden = hiddenColumns.has(col.index);
                const isVisible = !isHidden;
                
                const div = document.createElement('div');
                div.className = 'form-check';
                div.innerHTML = `
                    <input class="form-check-input" type="checkbox" 
                           id="col_${col.index}" 
                           value="${col.index}" 
                           ${isVisible ? 'checked' : ''}
                           ${col.mandatory ? 'disabled' : ''}>
                    <label class="form-check-label" for="col_${col.index}">
                        ${col.name}
                        ${col.mandatory ? '<span class="badge bg-secondary ms-2">Required</span>' : ''}
                    </label>
                `;
                container.appendChild(div);
            });
            
            columnVisibilityModal.show();
        }
        
        function applyColumnVisibility() {
            hiddenColumns.clear();
            
            columnDefinitions.forEach(col => {
                if (!col.mandatory) {
                    const checkbox = document.getElementById('col_' + col.index);
                    if (checkbox && !checkbox.checked) {
                        hiddenColumns.add(col.index);
                        spreadsheet.hideColumn(col.index);
                    } else {
                        spreadsheet.showColumn(col.index);
                    }
                }
            });
            
            // Save to localStorage
            localStorage.setItem('hiddenColumns', JSON.stringify(Array.from(hiddenColumns)));
            
            if (columnVisibilityModal) {
                columnVisibilityModal.hide();
            }
            
            showAlert('Column visibility updated successfully', 'Success', 'check-circle');
        }
        
        function resetColumnVisibility() {
            hiddenColumns.clear();
            localStorage.removeItem('hiddenColumns');
            
            // Show all columns
            columnDefinitions.forEach(col => {
                spreadsheet.showColumn(col.index);
            });
            
            // Update checkboxes
            columnDefinitions.forEach(col => {
                const checkbox = document.getElementById('col_' + col.index);
                if (checkbox && !col.mandatory) {
                    checkbox.checked = true;
                }
            });
            
            showAlert('Column visibility reset to default', 'Success', 'check-circle');
        }
        
        // Apply hidden columns on initialization
        function applyInitialColumnVisibility() {
            hiddenColumns.forEach(colIndex => {
                if (spreadsheet) {
                    spreadsheet.hideColumn(colIndex);
                }
            });
        }

        function addNewRow() {
            const newRow = ['', '', '', '', '', '', '', '', '', '', '', '', ''];
            spreadsheet.insertRow(newRow, 0);
            showAlert('New row added at the top. Fill in the details and click Save Changes.', 'Row Added', 'check-circle');
        }

        function deleteSelectedRows() {
            if (selectedRows.size === 0) {
                showAlert('Please select rows to delete.', 'Selection Required', 'exclamation-triangle');
                return;
            }

            showConfirm(
                `Delete ${selectedRows.size} selected athlete(s)? This will permanently remove them and all their assignments.`,
                function() {
                    const data = spreadsheet.getData();
                    const rowsArray = Array.from(selectedRows).sort((a, b) => b - a);
                    const idsToDelete = [];
                    
                    console.log('Deleting rows (sorted descending):', rowsArray);
                    console.log('Current spreadsheet data before deletion:', data);
                    
                    // Collect IDs of participants to delete
                    rowsArray.forEach(rowIndex => {
                        const id = data[rowIndex][0]; // Column 0 is ID (first column)
                        console.log(`Row ${rowIndex}: ID = ${id}`);
                        if (id) {
                            idsToDelete.push(parseInt(id));
                        }
                    });
                    
                    if (idsToDelete.length === 0) {
                        showAlert('No valid athletes selected for deletion.', 'Error', 'exclamation-triangle');
                        return;
                    }
                    
                    // Show loading
                    const loadingOverlay = document.getElementById('loadingOverlay');
                    if (loadingOverlay) {
                        loadingOverlay.classList.remove('d-none');
                    }
                    
                    // Delete from database via AJAX
                    fetch('participant_delete_ajax.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({ ids: idsToDelete })
                    })
                    .then(response => {
                        if (!response.ok) {
                            throw new Error('Server responded with status: ' + response.status);
                        }
                        return response.text();
                    })
                    .then(text => {
                        let data;
                        try {
                            data = JSON.parse(text);
                        } catch (e) {
                            console.error('Response was not JSON:', text);
                            throw new Error('Invalid server response');
                        }
                        
                        if (loadingOverlay) {
                            loadingOverlay.classList.add('d-none');
                        }
                        
                        if (data.success) {
                            console.log('Deletion successful. Removing rows from UI...');
                            console.log('Rows to remove:', rowsArray);
                            console.log('IDs deleted:', idsToDelete);
                            
                            // Remove deleted rows from spreadsheet
                            // CRITICAL: rowsArray is already sorted descending (b - a)
                            // This ensures we delete from bottom to top, preventing index shifts
                            rowsArray.forEach((rowIndex, i) => {
                                console.log(`Deleting row ${i + 1}/${rowsArray.length}: index ${rowIndex}`);
                                spreadsheet.deleteRow(rowIndex, 1);
                            });
                            
                            console.log('All rows deleted. Updating global data...');
                            
                            // Update global data arrays to match
                            participantsData = participantsData.filter(p => !idsToDelete.includes(p.id));
                            
                            // Rebuild originalData from current spreadsheet state
                            originalData = JSON.parse(JSON.stringify(spreadsheet.getData()));
                            
                            console.log('Updated participantsData length:', participantsData.length);
                            console.log('Spreadsheet data length:', spreadsheet.getData().length);
                            
                            // Clear changed rows since we just saved/deleted
                            changedRows.clear();
                            updateSaveButton();
                            
                            // Clear selection
                            selectedRows.clear();
                            updateSelectionButtons();
                            updateStatusBadge();
                            
                            // Show success message
                            if (data.errors && data.errors.length > 0) {
                                showAlert(data.message + '\n\nSome errors occurred. Check console for details.', 'Partial Success', 'exclamation-triangle');
                                console.log('Deletion errors:', data.errors);
                            } else {
                                showAlert(data.message, 'Success', 'check-circle');
                            }
                        } else {
                            showAlert(data.message || 'Bulk delete failed', 'Error', 'exclamation-triangle');
                        }
                    })
                    .catch(error => {
                        if (loadingOverlay) {
                            loadingOverlay.classList.add('d-none');
                        }
                        console.error('Error:', error);
                        showAlert('Bulk delete failed: ' + error.message, 'Error', 'exclamation-triangle');
                    });
                },
                'Confirm Bulk Delete',
                'exclamation-triangle'
            );
        }

        function deleteParticipant(id, name) {
            showConfirm(
                `Delete ${name}? This will permanently remove this athlete and all their assignments.`,
                function() {
                    fetch('participants_delete.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({ id: id })
                    })
                    .then(response => {
                        if (!response.ok) {
                            throw new Error('Server responded with status: ' + response.status);
                        }
                        return response.text();
                    })
                    .then(text => {
                        let data;
                        try {
                            data = JSON.parse(text);
                        } catch (e) {
                            console.error('Response was not JSON:', text);
                            throw new Error('Invalid server response');
                        }
                        
                        if (data.success) {
                            showAlert(data.message, 'Success', 'check-circle');
                            setTimeout(() => window.location.reload(), 1000);
                        } else {
                            showAlert(data.message || 'Delete failed', 'Error', 'exclamation-triangle');
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        showAlert('Delete failed: ' + error.message, 'Error', 'exclamation-triangle');
                    });
                },
                'Confirm Delete',
                'exclamation-triangle'
            );
        }

        function saveChanges() {
            if (changedRows.size === 0) {
                showAlert('No changes to save.', 'Notice', 'info-circle');
                return;
            }

            showConfirm(
                `Save changes to ${changedRows.size} athlete(s)?`,
                function() {
                    const data = spreadsheet.getData();
                    
                    console.log('=== SAVE CHANGES DEBUG ===');
                    console.log('Total rows to process:', data.length);
                    console.log('First row sample:', data[0]);
                    
                    // Count active custom fields
                    const customFieldsCount = customFieldDefs ? customFieldDefs.filter(f => f.is_active == 1).length : 0;
                    
                    const dataForBackend = data.map((row, rowIndex) => {
                        // Validate row has minimum required columns
                        if (!row || row.length < 12) {
                            console.warn(`Row ${rowIndex} has insufficient columns:`, row);
                            return null;
                        }
                        
                        // Map standard columns
                        const transformedRow = [
                            row[0],   // ID (keep as is)
                            row[2],   // BIB (skip checkbox at index 1)
                            row[3],   // First Name
                            row[4],   // Last Name
                            row[5],   // Gender
                            row[6],   // Year
                            row[7],   // Club
                            '',       // Category (not in spreadsheet, use empty string)
                            row[8],   // Country
                            row[9],   // FIS Code
                            row[10],  // Licence Code
                            row[11]   // Licence Type
                        ];
                        
                        // Extract custom fields
                        const customFields = {};
                        if (customFieldDefs && Array.isArray(customFieldDefs)) {
                            let customColIndex = 12; // Custom fields start at column 12
                            customFieldDefs.forEach(field => {
                                if (field.is_active == 1 || field.is_active === true) {
                                    const value = row[customColIndex] || '';
                                    if (value !== '') {
                                        customFields[field.field_key] = value;
                                    }
                                    customColIndex++;
                                }
                            });
                        }
                        
                        transformedRow.push(customFields); // Add custom fields as object
                        
                        if (rowIndex < 3) {
                            console.log(`Row ${rowIndex} transformation:`);
                            console.log('  Original:', row);
                            console.log('  Transformed:', transformedRow);
                            console.log('  Custom Fields:', customFields);
                        }
                        
                        return transformedRow;
                    }).filter(row => row !== null); // Remove any invalid rows
                    
                    console.log('Transformed data count:', dataForBackend.length);
                    console.log('Sample transformed row:', dataForBackend[0]);
                    
                    const saveBtn = document.getElementById('saveBtn');
                    
                    if (saveBtn) {
                        saveBtn.disabled = true;
                        saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Saving...';
                    }
                    
                    const loadingOverlay = document.getElementById('loadingOverlay');
                    if (loadingOverlay) {
                        loadingOverlay.classList.remove('d-none');
                    }
                    
                    fetch('participants_save_ajax.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            participants: dataForBackend
                        })
                    })
                    .then(response => response.json())
                    .then(result => {
                        const loadingOverlay = document.getElementById('loadingOverlay');
                        if (loadingOverlay) {
                            loadingOverlay.classList.add('d-none');
                        }
                        
                        if (result.success) {
                            showAlert(result.message, 'Success', 'check-circle');
                            changedRows.clear();
                            updateSaveButton();
                            refreshData();
                        } else {
                            showAlert(result.message || 'Failed to save changes', 'Error', 'exclamation-triangle');
                        }
                    })
                    .catch(error => {
                        const loadingOverlay = document.getElementById('loadingOverlay');
                        if (loadingOverlay) {
                            loadingOverlay.classList.add('d-none');
                        }
                        showAlert('Error saving changes: ' + error.message, 'Error', 'exclamation-triangle');
                    })
                    .finally(() => {
                        const saveBtn = document.getElementById('saveBtn');
                        if (saveBtn) {
                            saveBtn.disabled = false;
                            saveBtn.innerHTML = '<i class="fas fa-save me-2"></i>Save Changes';
                        }
                    });
                },
                'Confirm Save'
            );
        }

        function bulkAssign() {
            if (selectedRows.size === 0) {
                showAlert('Please select athletes to assign.', 'Selection Required', 'exclamation-triangle');
                return;
            }

            // Initialize modal if not already done
            if (!assignModal) {
                assignModal = new bootstrap.Modal(document.getElementById('assignModal'));
            }

            // Convert selectedRows Set to array of participant IDs
            const data = spreadsheet.getData();
            const participantIds = [];
            
            selectedRows.forEach(rowIndex => {
                const id = data[rowIndex][0]; // Column 0 is ID (first column)
                if (id) {
                    participantIds.push(parseInt(id));
                }
            });

            if (participantIds.length === 0) {
                showAlert('No valid athletes selected.', 'Error', 'exclamation-triangle');
                return;
            }

            // Populate selected participants display
            updateSelectedParticipantsDisplay(participantIds, data);
            
            // Clear previous alerts and form
            document.getElementById('assignAlert').innerHTML = '';
            document.getElementById('assignForm').reset();
            
            // Clear existing hidden inputs
            document.getElementById('assignForm').querySelectorAll('input[name="participant_ids[]"]').forEach(input => input.remove());
            
            // Populate hidden inputs with selected participant IDs
            participantIds.forEach(id => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'participant_ids[]';
                input.value = id;
                document.getElementById('assignForm').appendChild(input);
            });
            
            // Reset heat options
            toggleHeatOptions();
            
            assignModal.show();
        }

        function updateSelectedParticipantsDisplay(participantIds, data) {
            const displayDiv = document.getElementById('selectedParticipantsDisplay');
            
            if (!participantIds || participantIds.length === 0) {
                displayDiv.innerHTML = `
                    <div>
                        <i class="fas fa-info-circle me-1"></i>No athletes selected.
                    </div>
                `;
                return;
            }
            
            let html = `
                <div class="d-flex justify-content-between align-items-center mb-2">
                    <span class="fw-bold">
                        <i class="fas fa-users me-1 text-primary"></i>
                        ${participantIds.length} athlete(s) selected
                    </span>
                </div>
                <div class="selected-participants-grid">
            `;
            
            participantIds.forEach(id => {
                // Find participant in data
                const participant = participantsData.find(p => p.id == id);
                if (participant) {
                    html += `
                        <div class="participant-item">
                            <span class="badge bg-primary me-1">${participant.bib || 'N/A'}</span>
                            <span class="participant-name">${participant.first_name} ${participant.last_name}</span>
                        </div>
                    `;
                }
            });
            
            html += '</div>';
            displayDiv.innerHTML = html;
        }

        function loadEventHeats(eventId) {
            const heatSelect = document.getElementById('assignHeatSelect');
            const heatInfo = document.getElementById('heatInfo');
            
            if (!eventId) {
                heatSelect.disabled = true;
                heatSelect.innerHTML = '<option value="">No specific heat (assign later)</option>';
                heatInfo.textContent = 'Select an event to see available heats';
                return;
            }
            
            // Filter heats for selected event
            const eventHeats = heatsData.filter(heat => heat.event_id == eventId);
            
            heatSelect.innerHTML = '<option value="">No specific heat (assign later)</option>';
            
            if (eventHeats.length === 0) {
                heatInfo.innerHTML = '<span class="text-warning">No heats configured for this event</span>';
            } else {
                eventHeats.forEach(heat => {
                    const option = document.createElement('option');
                    option.value = heat.heat_number;
                    option.textContent = `Heat ${heat.heat_number}${heat.heat_name ? ' - ' + heat.heat_name : ''}`;
                    heatSelect.appendChild(option);
                });
                
                heatInfo.innerHTML = `<span class="text-success">${eventHeats.length} heat(s) available</span>`;
            }
            
            // Enable heat select if event_participants is checked
            heatSelect.disabled = !document.getElementById('assignToEventParticipants').checked;
        }

        function toggleHeatOptions() {
            const checkbox = document.getElementById('assignToEventParticipants');
            const heatOptions = document.getElementById('heatOptions');
            const heatSelect = document.getElementById('assignHeatSelect');
            
            if (checkbox.checked) {
                heatOptions.style.display = 'block';
                heatSelect.disabled = false;
                
                // Load heats for currently selected event
                const eventId = document.getElementById('assignEventSelect').value;
                if (eventId) {
                    loadEventHeats(eventId);
                }
            } else {
                heatOptions.style.display = 'none';
                heatSelect.disabled = true;
            }
        }

        function submitAssignment() {
            const form = document.getElementById('assignForm');
            const eventId = document.getElementById('assignEventSelect').value;
            
            if (!eventId) {
                document.getElementById('assignAlert').innerHTML = `
                    <div class="alert alert-warning alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-triangle me-2"></i>Please select an event.
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                `;
                return;
            }
            
            const submitBtn = document.querySelector('[onclick="submitAssignment()"]');
            const originalText = submitBtn.innerHTML;
            
            // Show loading state
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Assigning...';
            
            // Submit the form
            form.submit();
        }

        function showAssignModal(participantId, participantName, bib) {
            // Redirect to participant edit page where event assignment is available
            window.location.href = 'participant_edit.php?id=' + participantId + '&tab=events';
        }

        function bulkDelete() {
            deleteSelectedRows();
        }

        function exportData() {
            spreadsheet.download();
        }

        function refreshData() {
            if (changedRows.size > 0) {
                showConfirm(
                    'You have unsaved changes. Refreshing will discard them. Continue?',
                    function() {
                        window.location.reload();
                    },
                    'Unsaved Changes',
                    'exclamation-triangle'
                );
            } else {
                window.location.reload();
            }
        }

        function selectAllVisible() {
            const data = spreadsheet.getData();
            for (let i = 0; i < data.length; i++) {
                selectedRows.add(i);
            }
            spreadsheet.updateSelectionFromCoords(0, 0, 11, data.length - 1);
            updateSelectionButtons();
        }

        function clearSelection() {
            selectedRows.clear();
            spreadsheet.resetSelection();
            updateSelectionButtons();
        }

        function clearAllFilters() {
            document.getElementById('genderFilter').value = '';
            document.getElementById('categoryFilter').value = '';
            document.getElementById('eventFilter').value = '';
            document.getElementById('searchInput').value = '';
            
            window.location.href = 'participants.php';
        }

        function clearSearch() {
            document.getElementById('searchInput').value = '';
            spreadsheet.search('');
        }

        // Filter handlers
        document.getElementById('searchInput').addEventListener('input', function(e) {
            spreadsheet.search(e.target.value);
        });

        document.getElementById('genderFilter').addEventListener('change', applyFilters);
        document.getElementById('categoryFilter').addEventListener('change', applyFilters);
        document.getElementById('licenceTypeFilter').addEventListener('change', applyFilters);
        document.getElementById('eventFilter').addEventListener('change', applyFilters);

        function applyFilters() {
            const gender = document.getElementById('genderFilter').value;
            const category = document.getElementById('categoryFilter').value;
            const licenceType = document.getElementById('licenceTypeFilter').value;
            const event = document.getElementById('eventFilter').value;
            
            const params = new URLSearchParams();
            if (gender) params.set('gender', gender);
            if (category) params.set('category', category);
            if (licenceType) params.set('licence_type', licenceType);
            if (event) params.set('event', event);
            
            window.location.href = 'participants.php?' + params.toString();
        }

        // Participant Add/Edit Modal Functions
        let addParticipantModal, editParticipantModal;
        
        function showAddParticipantModal() {
            if (!addParticipantModal) {
                addParticipantModal = new bootstrap.Modal(document.getElementById('addParticipantModal'));
            }
            document.getElementById('addParticipantForm').reset();
            document.getElementById('addParticipantAlert').innerHTML = '';
            addParticipantModal.show();
        }
        
        function submitAddParticipant() {
            const form = document.getElementById('addParticipantForm');
            const formData = new FormData(form);
            const submitBtn = event.target;
            const originalText = submitBtn.innerHTML;
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Adding...';
            
            fetch('participant_add_ajax.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('addParticipantAlert').innerHTML = 
                        '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>' + data.message + '</div>';
                    setTimeout(() => {
                        addParticipantModal.hide();
                        refreshData();
                    }, 1500);
                } else {
                    document.getElementById('addParticipantAlert').innerHTML = 
                        '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>' + data.message + '</div>';
                }
            })
            .catch(error => {
                console.error('Error:', error);
                document.getElementById('addParticipantAlert').innerHTML = 
                    '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>An error occurred</div>';
            })
            .finally(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            });
        }
        
        function showEditParticipantModal(participantId) {
            if (!editParticipantModal) {
                editParticipantModal = new bootstrap.Modal(document.getElementById('editParticipantModal'));
            }
            
            document.getElementById('editParticipantAlert').innerHTML = '<div class="text-center py-3"><i class="fas fa-spinner fa-spin fa-2x"></i></div>';
            editParticipantModal.show();
            
            // Fetch participant data
            fetch('participant_edit_ajax.php?id=' + participantId)
            .then(response => response.json())
            .then(data => {
                document.getElementById('editParticipantAlert').innerHTML = '';
                if (data.success) {
                    const p = data.participant;
                    document.getElementById('editParticipantId').value = p.id;
                    document.getElementById('editFirstName').value = p.first_name || '';
                    document.getElementById('editLastName').value = p.last_name || '';
                    document.getElementById('editGender').value = p.gender || '';
                    document.getElementById('editDateOfBirth').value = p.date_of_birth || '';
                    document.getElementById('editBib').value = p.bib || '';
                    document.getElementById('editFisCode').value = p.fis_code || '';
                    document.getElementById('editLicenceCode').value = p.licence_code || '';
                    document.getElementById('editLicenceType').value = p.licence_type || '';
                    document.getElementById('editCategory').value = p.category || '';
                    document.getElementById('editClub').value = p.club || '';
                    document.getElementById('editCountry').value = p.country || '';
                    
                    // Load custom fields if they exist
                    if (p.custom_fields && customFieldDefs.length > 0) {
                        const customFields = typeof p.custom_fields === 'string' ? JSON.parse(p.custom_fields) : p.custom_fields;
                        customFieldDefs.forEach(field => {
                            const elem = document.getElementById('edit_cf_' + field.field_key);
                            if (elem && customFields[field.field_key]) {
                                elem.value = customFields[field.field_key];
                            }
                        });
                    }
                } else {
                    document.getElementById('editParticipantAlert').innerHTML = 
                        '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>' + data.message + '</div>';
                }
            })
            .catch(error => {
                console.error('Error:', error);
                document.getElementById('editParticipantAlert').innerHTML = 
                    '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>Failed to load athlete data</div>';
            });
        }
        
        function submitEditParticipant() {
            const form = document.getElementById('editParticipantForm');
            const formData = new FormData(form);
            const submitBtn = event.target;
            const originalText = submitBtn.innerHTML;
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Saving...';
            
            fetch('participant_edit_ajax.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('editParticipantAlert').innerHTML = 
                        '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>' + data.message + '</div>';
                    setTimeout(() => {
                        editParticipantModal.hide();
                        refreshData();
                    }, 1500);
                } else {
                    document.getElementById('editParticipantAlert').innerHTML = 
                        '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>' + data.message + '</div>';
                }
            })
            .catch(error => {
                console.error('Error:', error);
                document.getElementById('editParticipantAlert').innerHTML = 
                    '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>An error occurred</div>';
            })
            .finally(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            });
        }

        // CSV Import handler
        function processImport() {
            const fileInput = document.getElementById('csvFile');
            const file = fileInput.files[0];
            
            if (!file) {
                showAlert('Please select a CSV file.', 'File Required', 'exclamation-triangle');
                return;
            }

            const reader = new FileReader();
            reader.onload = function(e) {
                const csv = e.target.result;
                const lines = csv.split('\n');
                const headers = lines[0].split(',');
                
                // Parse CSV and add to spreadsheet
                for (let i = 1; i < lines.length; i++) {
                    if (lines[i].trim()) {
                        const values = lines[i].split(',');
                        const newRow = ['', ...values];
                        spreadsheet.insertRow(newRow);
                    }
                }
                
                bootstrap.Modal.getInstance(document.getElementById('importModal')).hide();
                showAlert(`Imported ${lines.length - 1} rows. Review and click Save Changes.`, 'Import Complete', 'check-circle');
                changedRows.add(0);
                updateSaveButton();
            };
            
            reader.readAsText(file);
        }

        // ==================== Excel Paste Import Functionality ====================
        
        let excelPasteModal;
        let parsedData = [];
        let columnMapping = {};
        let editableData = [];
        
        // Field definitions
        const fieldDefinitions = [
            { key: 'bib', label: 'BIB', required: true, type: 'text' },
            { key: 'first_name', label: 'First Name', required: true, type: 'text' },
            { key: 'last_name', label: 'Last Name', required: true, type: 'text' },
            { key: 'gender', label: 'Gender', required: true, type: 'select', options: ['', 'M', 'F'] },
            { key: 'fis_code', label: 'FIS Code', required: false, type: 'text' },
            { key: 'licence_code', label: 'Licence Code', required: false, type: 'text' },
            { key: 'licence_type', label: 'Licence Type', required: false, type: 'text' },
            { key: 'club', label: 'Club', required: false, type: 'text' },
            { key: 'country', label: 'Country', required: false, type: 'text' },
            { key: 'date_of_birth', label: 'Year of Birth', required: false, type: 'year' },
            { key: 'category', label: 'Category', required: false, type: 'text' }
        ];
        
        // Add custom fields to field definitions for Excel import
        if (typeof customFieldDefs !== 'undefined' && customFieldDefs && Array.isArray(customFieldDefs)) {
            console.log('Loading custom fields for import:', customFieldDefs);
            customFieldDefs.forEach(field => {
                if (field.is_active == 1 || field.is_active === true) {
                    const fieldDef = {
                        key: 'cf_' + field.field_key, // Prefix with 'cf_' to distinguish from standard fields
                        label: field.field_label + ' (Custom)',
                        required: field.is_required == 1,
                        type: field.field_type
                    };
                    
                    // Add options for select fields
                    if (field.field_type === 'select' && field.field_options) {
                        try {
                            fieldDef.options = [''].concat(JSON.parse(field.field_options));
                        } catch (e) {
                            fieldDef.options = [''];
                        }
                    }
                    
                    fieldDefinitions.push(fieldDef);
                    console.log('Added custom field to import:', fieldDef);
                }
            });
        } else {
            console.log('No custom fields available or customFieldDefs not defined:', customFieldDefs);
        }
        console.log('Final fieldDefinitions for import:', fieldDefinitions);
        
        function showExcelPasteModal() {
            if (!excelPasteModal) {
                excelPasteModal = new bootstrap.Modal(document.getElementById('excelPasteModal'));
            }
            
            // Reset to step 1
            document.getElementById('pasteStep').classList.remove('d-none');
            document.getElementById('mappingStep').classList.add('d-none');
            document.getElementById('editStep').classList.add('d-none');
            document.getElementById('pasteArea').value = '';
            
            excelPasteModal.show();
        }
        
        function clearPasteArea() {
            document.getElementById('pasteArea').value = '';
            parsedData = [];
        }
        
        function parsePastedData() {
            const pasteArea = document.getElementById('pasteArea');
            const text = pasteArea.value.trim();
            
            if (!text) {
                showAlert('Please paste some data first!', 'Data Required', 'exclamation-triangle');
                return;
            }
            
            // Parse TSV (tab-separated) or CSV
            const lines = text.split('\n').filter(line => line.trim());
            
            if (lines.length < 2) {
                showAlert('Please paste at least a header row and one data row.', 'Invalid Data', 'exclamation-triangle');
                return;
            }
            
            // Detect delimiter (tab or comma)
            const firstLine = lines[0];
            const delimiter = firstLine.includes('\t') ? '\t' : ',';
            
            parsedData = lines.map(line => {
                // Split by delimiter, handling quoted values
                const cells = [];
                let currentCell = '';
                let inQuotes = false;
                
                for (let i = 0; i < line.length; i++) {
                    const char = line[i];
                    
                    if (char === '"') {
                        inQuotes = !inQuotes;
                    } else if (char === delimiter && !inQuotes) {
                        cells.push(currentCell.trim());
                        currentCell = '';
                    } else {
                        currentCell += char;
                    }
                }
                cells.push(currentCell.trim());
                
                return cells;
            });
            
            // Auto-detect column mapping
            autoDetectMapping();
            
            // Move to mapping step
            document.getElementById('pasteStep').classList.add('d-none');
            document.getElementById('mappingStep').classList.remove('d-none');
            
            renderColumnMapping();
        }
        
        function autoDetectMapping() {
            const headers = parsedData[0].map(h => h.toLowerCase().trim());
            columnMapping = {};
            
            headers.forEach((header, index) => {
                // Try to match header to field
                const matchedField = fieldDefinitions.find(field => {
                    const fieldName = field.label.toLowerCase();
                    const fieldKey = field.key.toLowerCase();
                    
                    return header.includes(fieldKey) || 
                           header.includes(fieldName) ||
                           fieldKey.includes(header) ||
                           (field.key === 'bib' && header.match(/^(bib|number|#|no)$/)) ||
                           (field.key === 'first_name' && header.match(/(first|name|given)/)) ||
                           (field.key === 'last_name' && header.match(/(last|surname|family)/)) ||
                           (field.key === 'gender' && header.match(/(gender|sex)/)) ||
                           (field.key === 'fis_code' && header.match(/fis/)) ||
                           (field.key === 'licence_code' && header.match(/licen[cs]e/)) ||
                           (field.key === 'club' && header.match(/club/)) ||
                           (field.key === 'country' && header.match(/(country|nation)/)) ||
                           (field.key === 'date_of_birth' && header.match(/(birth|dob|born|year)/)) ||
                           (field.key === 'category' && header.match(/categor/));
                });
                
                if (matchedField) {
                    columnMapping[index] = matchedField.key;
                }
            });
        }
        
        function renderColumnMapping() {
            const container = document.getElementById('columnMappingArea');
            const headers = parsedData[0];
            
            container.innerHTML = '';
            
            headers.forEach((header, index) => {
                const card = document.createElement('div');
                card.className = 'card column-mapping-card';
                card.style.minWidth = '200px';
                card.innerHTML = `
                    <div class="card-body p-2">
                        <div class="small text-muted mb-1">Column ${index + 1}</div>
                        <div class="fw-bold mb-2">${header || '(No Header)'}</div>
                        <select class="form-select form-select-sm" data-column="${index}">
                            <option value="">-- Skip Column --</option>
                            ${fieldDefinitions.map(field => `
                                <option value="${field.key}" ${columnMapping[index] === field.key ? 'selected' : ''}>
                                    ${field.label}${field.required ? ' *' : ''}
                                </option>
                            `).join('')}
                        </select>
                        <div class="small text-muted mt-1">
                            Sample: ${parsedData[1] ? parsedData[1][index] || '(empty)' : ''}
                        </div>
                    </div>
                `;
                container.appendChild(card);
                
                // Add event listener
                card.querySelector('select').addEventListener('change', function() {
                    const colIndex = parseInt(this.dataset.column);
                    const value = this.value;
                    
                    if (value) {
                        // Remove this field from other mappings
                        Object.keys(columnMapping).forEach(key => {
                            if (columnMapping[key] === value && key != colIndex) {
                                delete columnMapping[key];
                            }
                        });
                        columnMapping[colIndex] = value;
                    } else {
                        delete columnMapping[colIndex];
                    }
                    
                    // Update all selects
                    renderColumnMapping();
                });
            });
        }
        
        // Country code mapping - converts country names (English/Polish) to Alpha-2 codes
        const countryCodeMap = {
            'AFGHANISTAN': 'AF', 'AF': 'AF', 'ALBANIA': 'AL', 'AL': 'AL', 'ALGERIA': 'DZ', 'ALGIERIA': 'DZ', 'DZ': 'DZ',
            'AMERICAN SAMOA': 'AS', 'SAMOA AMERYKAŃSKIE': 'AS', 'AS': 'AS', 'ANDORRA': 'AD', 'ANDORA': 'AD', 'AD': 'AD',
            'ANGOLA': 'AO', 'AO': 'AO', 'ANGUILLA': 'AI', 'AI': 'AI', 'ANTARCTICA': 'AQ', 'ANTARKTYKA': 'AQ', 'AQ': 'AQ',
            'ANTIGUA AND BARBUDA': 'AG', 'ANTIGUA I BARBUDA': 'AG', 'AG': 'AG', 'ARGENTINA': 'AR', 'ARGENTYNA': 'AR', 'AR': 'AR',
            'ARMENIA': 'AM', 'AM': 'AM', 'ARUBA': 'AW', 'AW': 'AW', 'AUSTRALIA': 'AU', 'AU': 'AU',
            'AUSTRIA': 'AT', 'AT': 'AT', 'AZERBAIJAN': 'AZ', 'AZERBEJDŻAN': 'AZ', 'AZ': 'AZ',
            'BAHAMAS': 'BS', 'BAHAMAS (THE)': 'BS', 'BS': 'BS', 'BAHRAIN': 'BH', 'BAHRAJN': 'BH', 'BH': 'BH',
            'BANGLADESH': 'BD', 'BANGLADESZ': 'BD', 'BD': 'BD', 'BARBADOS': 'BB', 'BB': 'BB',
            'BELARUS': 'BY', 'BIAŁORUŚ': 'BY', 'BY': 'BY', 'BELGIUM': 'BE', 'BELGIA': 'BE', 'BE': 'BE',
            'BELIZE': 'BZ', 'BZ': 'BZ', 'BENIN': 'BJ', 'BJ': 'BJ', 'BERMUDA': 'BM', 'BERMUDY': 'BM', 'BM': 'BM',
            'BHUTAN': 'BT', 'BT': 'BT', 'BOLIVIA': 'BO', 'BOLIVIA (PLURINATIONAL STATE OF)': 'BO', 'BOLIWIA': 'BO', 'BO': 'BO',
            'BONAIRE, SINT EUSTATIUS AND SABA': 'BQ', 'BONAIRE, SINT EUSTATIUS I SABA': 'BQ', 'BQ': 'BQ',
            'BOSNIA AND HERZEGOVINA': 'BA', 'BOŚNIA I HERCEGOWINA': 'BA', 'BA': 'BA', 'BOTSWANA': 'BW', 'BW': 'BW',
            'BOUVET ISLAND': 'BV', 'WYSPA BOUVETA': 'BV', 'BV': 'BV', 'BRAZIL': 'BR', 'BRAZYLIA': 'BR', 'BR': 'BR',
            'BRITISH INDIAN OCEAN TERRITORY': 'IO', 'BRITISH INDIAN OCEAN TERRITORY (THE)': 'IO', 'IO': 'IO',
            'BRUNEI': 'BN', 'BRUNEI DARUSSALAM': 'BN', 'BN': 'BN', 'BULGARIA': 'BG', 'BUŁGARIA': 'BG', 'BG': 'BG',
            'BURKINA FASO': 'BF', 'BF': 'BF', 'BURUNDI': 'BI', 'BI': 'BI', 'CABO VERDE': 'CV', 'REPUBLIKA ZIELONEGO PRZYLĄDKA': 'CV', 'CV': 'CV',
            'CAMBODIA': 'KH', 'KAMBODŻA': 'KH', 'KH': 'KH', 'CAMEROON': 'CM', 'KAMERUN': 'CM', 'CM': 'CM',
            'CANADA': 'CA', 'KANADA': 'CA', 'CA': 'CA', 'CAYMAN ISLANDS': 'KY', 'CAYMAN ISLANDS (THE)': 'KY', 'KY': 'KY',
            'CENTRAL AFRICAN REPUBLIC': 'CF', 'CENTRAL AFRICAN REPUBLIC (THE)': 'CF', 'CF': 'CF', 'CHAD': 'TD', 'CZAD': 'TD', 'TD': 'TD',
            'CHILE': 'CL', 'CL': 'CL', 'CHINA': 'CN', 'CHINY': 'CN', 'CN': 'CN',
            'CHRISTMAS ISLAND': 'CX', 'WYSPA BOŻEGO NARODZENIA': 'CX', 'CX': 'CX',
            'COCOS (KEELING) ISLANDS': 'CC', 'COCOS (KEELING) ISLANDS (THE)': 'CC', 'CC': 'CC',
            'COLOMBIA': 'CO', 'KOLUMBIA': 'CO', 'CO': 'CO', 'COMOROS': 'KM', 'COMOROS (THE)': 'KM', 'KM': 'KM',
            'CONGO': 'CG', 'CONGO (THE)': 'CG', 'CG': 'CG', 'CONGO (THE DEMOCRATIC REPUBLIC OF THE)': 'CD', 'CD': 'CD',
            'COOK ISLANDS': 'CK', 'COOK ISLANDS (THE)': 'CK', 'CK': 'CK', 'COSTA RICA': 'CR', 'KOSTARYKA': 'CR', 'CR': 'CR',
            'CROATIA': 'HR', 'CHORWACJA': 'HR', 'HR': 'HR', 'CUBA': 'CU', 'KUBA': 'CU', 'CU': 'CU',
            'CURAÇAO': 'CW', 'CURACAO': 'CW', 'CW': 'CW', 'CYPRUS': 'CY', 'CYPR': 'CY', 'CY': 'CY',
            'CZECHIA': 'CZ', 'CZECHY': 'CZ', 'CZECH REPUBLIC': 'CZ', 'CZ': 'CZ',
            'CÔTE D\'IVOIRE': 'CI', 'COTE D\'IVOIRE': 'CI', 'WYBRZEŻE KOŚCI SŁONIOWEJ': 'CI', 'CI': 'CI',
            'DENMARK': 'DK', 'DANIA': 'DK', 'DK': 'DK', 'DJIBOUTI': 'DJ', 'DŻIBUTI': 'DJ', 'DJ': 'DJ',
            'DOMINICA': 'DM', 'DOMINIKA': 'DM', 'DM': 'DM', 'DOMINICAN REPUBLIC': 'DO', 'DOMINICAN REPUBLIC (THE)': 'DO', 'DO': 'DO',
            'ECUADOR': 'EC', 'EKWADOR': 'EC', 'EC': 'EC', 'EGYPT': 'EG', 'EGIPT': 'EG', 'EG': 'EG',
            'EL SALVADOR': 'SV', 'SALWADOR': 'SV', 'SV': 'SV', 'EQUATORIAL GUINEA': 'GQ', 'GWINEA RÓWNIKOWA': 'GQ', 'GQ': 'GQ',
            'ERITREA': 'ER', 'ERYTREA': 'ER', 'ER': 'ER', 'ESTONIA': 'EE', 'EE': 'EE',
            'ESWATINI': 'SZ', 'SZ': 'SZ', 'ETHIOPIA': 'ET', 'ETIOPIA': 'ET', 'ET': 'ET',
            'FALKLAND ISLANDS': 'FK', 'FALKLAND ISLANDS (THE) [MALVINAS]': 'FK', 'FK': 'FK',
            'FAROE ISLANDS': 'FO', 'FAROE ISLANDS (THE)': 'FO', 'FO': 'FO', 'FIJI': 'FJ', 'FIDŻI': 'FJ', 'FJ': 'FJ',
            'FINLAND': 'FI', 'FINLANDIA': 'FI', 'FI': 'FI', 'FRANCE': 'FR', 'FRANCJA': 'FR', 'FR': 'FR',
            'FRENCH GUIANA': 'GF', 'GUJANA FRANCUSKA': 'GF', 'GF': 'GF', 'FRENCH POLYNESIA': 'PF', 'POLINEZJA FRANCUSKA': 'PF', 'PF': 'PF',
            'FRENCH SOUTHERN TERRITORIES': 'TF', 'FRENCH SOUTHERN TERRITORIES (THE)': 'TF', 'TF': 'TF',
            'GABON': 'GA', 'GA': 'GA', 'GAMBIA': 'GM', 'GAMBIA (THE)': 'GM', 'GM': 'GM',
            'GEORGIA': 'GE', 'GRUZJA': 'GE', 'GE': 'GE', 'GERMANY': 'DE', 'NIEMCY': 'DE', 'DE': 'DE',
            'GHANA': 'GH', 'GH': 'GH', 'GIBRALTAR': 'GI', 'GI': 'GI', 'GREECE': 'GR', 'GRECJA': 'GR', 'GR': 'GR',
            'GREENLAND': 'GL', 'GRENLANDIA': 'GL', 'GL': 'GL', 'GRENADA': 'GD', 'GD': 'GD',
            'GUADELOUPE': 'GP', 'GWADELUPA': 'GP', 'GP': 'GP', 'GUAM': 'GU', 'GU': 'GU',
            'GUATEMALA': 'GT', 'GWATEMALA': 'GT', 'GT': 'GT', 'GUERNSEY': 'GG', 'GG': 'GG',
            'GUINEA': 'GN', 'GWINEA': 'GN', 'GN': 'GN', 'GUINEA-BISSAU': 'GW', 'GWINEA BISSAU': 'GW', 'GW': 'GW',
            'GUYANA': 'GY', 'GUJANA': 'GY', 'GY': 'GY', 'HAITI': 'HT', 'HT': 'HT',
            'HEARD ISLAND AND MCDONALD ISLANDS': 'HM', 'WYSPY HEARD I MCDONALDA': 'HM', 'HM': 'HM',
            'HOLY SEE': 'VA', 'HOLY SEE (THE)': 'VA', 'VA': 'VA', 'HONDURAS': 'HN', 'HN': 'HN',
            'HONG KONG': 'HK', 'HONGKONG': 'HK', 'HK': 'HK', 'HUNGARY': 'HU', 'WĘGRY': 'HU', 'HU': 'HU',
            'ICELAND': 'IS', 'ISLANDIA': 'IS', 'IS': 'IS', 'INDIA': 'IN', 'INDIE': 'IN', 'IN': 'IN',
            'INDONESIA': 'ID', 'INDONEZJA': 'ID', 'ID': 'ID', 'IRAN': 'IR', 'IRAN (ISLAMIC REPUBLIC OF)': 'IR', 'IR': 'IR',
            'IRAQ': 'IQ', 'IRAK': 'IQ', 'IQ': 'IQ', 'IRELAND': 'IE', 'IRLANDIA': 'IE', 'IE': 'IE',
            'ISLE OF MAN': 'IM', 'WYSPA MAN': 'IM', 'IM': 'IM', 'ISRAEL': 'IL', 'IZRAEL': 'IL', 'IL': 'IL',
            'ITALY': 'IT', 'WŁOCHY': 'IT', 'IT': 'IT', 'JAMAICA': 'JM', 'JAMAJKA': 'JM', 'JM': 'JM',
            'JAPAN': 'JP', 'JAPONIA': 'JP', 'JP': 'JP', 'JERSEY': 'JE', 'JE': 'JE',
            'JORDAN': 'JO', 'JORDANIA': 'JO', 'JO': 'JO', 'KAZAKHSTAN': 'KZ', 'KAZACHSTAN': 'KZ', 'KZ': 'KZ',
            'KENYA': 'KE', 'KENIA': 'KE', 'KE': 'KE', 'KIRIBATI': 'KI', 'KI': 'KI',
            'KOREA (THE DEMOCRATIC PEOPLE\'S REPUBLIC OF)': 'KP', 'KP': 'KP', 'KOREA (THE REPUBLIC OF)': 'KR', 'KR': 'KR',
            'KUWAIT': 'KW', 'KUWEJT': 'KW', 'KW': 'KW', 'KYRGYZSTAN': 'KG', 'KIRGISTAN': 'KG', 'KG': 'KG',
            'LAO PEOPLE\'S DEMOCRATIC REPUBLIC': 'LA', 'LAO PEOPLE\'S DEMOCRATIC REPUBLIC (THE)': 'LA', 'LA': 'LA',
            'LATVIA': 'LV', 'ŁOTWA': 'LV', 'LV': 'LV', 'LEBANON': 'LB', 'LIBAN': 'LB', 'LB': 'LB',
            'LESOTHO': 'LS', 'LS': 'LS', 'LIBERIA': 'LR', 'LR': 'LR', 'LIBYA': 'LY', 'LIBIA': 'LY', 'LY': 'LY',
            'LIECHTENSTEIN': 'LI', 'LI': 'LI', 'LITHUANIA': 'LT', 'LITWA': 'LT', 'LT': 'LT',
            'LUXEMBOURG': 'LU', 'LUKSEMBURG': 'LU', 'LU': 'LU', 'MACAO': 'MO', 'MAKAU': 'MO', 'MO': 'MO',
            'MADAGASCAR': 'MG', 'MADAGASKAR': 'MG', 'MG': 'MG', 'MALAWI': 'MW', 'MW': 'MW',
            'MALAYSIA': 'MY', 'MALEZJA': 'MY', 'MY': 'MY', 'MALDIVES': 'MV', 'MALEDIWY': 'MV', 'MV': 'MV',
            'MALI': 'ML', 'ML': 'ML', 'MALTA': 'MT', 'MT': 'MT', 'MARSHALL ISLANDS': 'MH', 'MARSHALL ISLANDS (THE)': 'MH', 'MH': 'MH',
            'MARTINIQUE': 'MQ', 'MARTYNIKA': 'MQ', 'MQ': 'MQ', 'MAURITANIA': 'MR', 'MAURETANIA': 'MR', 'MR': 'MR',
            'MAURITIUS': 'MU', 'MU': 'MU', 'MAYOTTE': 'YT', 'MAJOTTA': 'YT', 'YT': 'YT',
            'MEXICO': 'MX', 'MEKSYK': 'MX', 'MX': 'MX', 'MICRONESIA': 'FM', 'MICRONESIA (FEDERATED STATES OF)': 'FM', 'MIKRONEZJA': 'FM', 'FM': 'FM',
            'MOLDOVA': 'MD', 'MOLDOVA (THE REPUBLIC OF)': 'MD', 'MD': 'MD', 'MONACO': 'MC', 'MONAKO': 'MC', 'MC': 'MC',
            'MONGOLIA': 'MN', 'MN': 'MN', 'MONTENEGRO': 'ME', 'CZARNOGÓRA': 'ME', 'ME': 'ME',
            'MONTSERRAT': 'MS', 'MS': 'MS', 'MOROCCO': 'MA', 'MAROKO': 'MA', 'MA': 'MA',
            'MOZAMBIQUE': 'MZ', 'MOZAMBIK': 'MZ', 'MZ': 'MZ', 'MYANMAR': 'MM', 'MJANMA': 'MM', 'MM': 'MM',
            'NAMIBIA': 'NA', 'NA': 'NA', 'NAURU': 'NR', 'NR': 'NR', 'NEPAL': 'NP', 'NP': 'NP',
            'NETHERLANDS': 'NL', 'NETHERLANDS (THE)': 'NL', 'NL': 'NL', 'NEW CALEDONIA': 'NC', 'NOWA KALEDONIA': 'NC', 'NC': 'NC',
            'NEW ZEALAND': 'NZ', 'NOWA ZELANDIA': 'NZ', 'NZ': 'NZ', 'NICARAGUA': 'NI', 'NIKARAGUA': 'NI', 'NI': 'NI',
            'NIGER': 'NE', 'NIGER (THE)': 'NE', 'NE': 'NE', 'NIGERIA': 'NG', 'NG': 'NG',
            'NIUE': 'NU', 'NU': 'NU', 'NORFOLK ISLAND': 'NF', 'WYSPA NORFOLK': 'NF', 'NF': 'NF',
            'NORTHERN MARIANA ISLANDS': 'MP', 'NORTHERN MARIANA ISLANDS (THE)': 'MP', 'MP': 'MP',
            'NORWAY': 'NO', 'NORWEGIA': 'NO', 'NO': 'NO', 'OMAN': 'OM', 'OM': 'OM',
            'PAKISTAN': 'PK', 'PK': 'PK', 'PALAU': 'PW', 'PW': 'PW', 'PALESTINE': 'PS', 'PALESTINE, STATE OF': 'PS', 'PALESTYNA': 'PS', 'PS': 'PS',
            'PANAMA': 'PA', 'PA': 'PA', 'PAPUA NEW GUINEA': 'PG', 'PAPUA-NOWA GWINEA': 'PG', 'PG': 'PG',
            'PARAGUAY': 'PY', 'PARAGWAJ': 'PY', 'PY': 'PY', 'PERU': 'PE', 'PE': 'PE',
            'PHILIPPINES': 'PH', 'PHILIPPINES (THE)': 'PH', 'PH': 'PH', 'PITCAIRN': 'PN', 'PN': 'PN',
            'POLAND': 'PL', 'POLSKA': 'PL', 'PL': 'PL', 'PORTUGAL': 'PT', 'PORTUGALIA': 'PT', 'PT': 'PT',
            'PUERTO RICO': 'PR', 'PORTORYKO': 'PR', 'PR': 'PR', 'QATAR': 'QA', 'KATAR': 'QA', 'QA': 'QA',
            'REPUBLIC OF NORTH MACEDONIA': 'MK', 'NORTH MACEDONIA': 'MK', 'MACEDONIA': 'MK', 'MK': 'MK',
            'ROMANIA': 'RO', 'RUMUNIA': 'RO', 'RO': 'RO', 'RUSSIA': 'RU', 'RUSSIAN FEDERATION': 'RU', 'RUSSIAN FEDERATION (THE)': 'RU', 'RU': 'RU',
            'RWANDA': 'RW', 'RW': 'RW', 'RÉUNION': 'RE', 'REUNION': 'RE', 'RE': 'RE',
            'SAINT BARTHÉLEMY': 'BL', 'SAINT BARTHELEMY': 'BL', 'SAINT-BARTHÉLEMY': 'BL', 'BL': 'BL',
            'SAINT HELENA, ASCENSION AND TRISTAN DA CUNHA': 'SH', 'WYSPA ŚWIĘTEJ HELENY': 'SH', 'SH': 'SH',
            'SAINT KITTS AND NEVIS': 'KN', 'SAINT KITTS I NEVIS': 'KN', 'KN': 'KN',
            'SAINT LUCIA': 'LC', 'LC': 'LC', 'SAINT MARTIN': 'MF', 'SAINT MARTIN (FRENCH PART)': 'MF', 'MF': 'MF',
            'SAINT PIERRE AND MIQUELON': 'PM', 'SAINT-PIERRE I MIQUELON': 'PM', 'PM': 'PM',
            'SAINT VINCENT AND THE GRENADINES': 'VC', 'SAINT VINCENT I GRENADYNY': 'VC', 'VC': 'VC',
            'SAMOA': 'WS', 'WS': 'WS', 'SAN MARINO': 'SM', 'SM': 'SM',
            'SAO TOME AND PRINCIPE': 'ST', 'WYSPY ŚWIĘTEGO TOMASZA I KSIĄŻĘCA': 'ST', 'ST': 'ST',
            'SAUDI ARABIA': 'SA', 'ARABIA SAUDYJSKA': 'SA', 'SA': 'SA', 'SENEGAL': 'SN', 'SN': 'SN',
            'SERBIA': 'RS', 'RS': 'RS', 'SEYCHELLES': 'SC', 'SESZELE': 'SC', 'SC': 'SC',
            'SIERRA LEONE': 'SL', 'SL': 'SL', 'SINGAPORE': 'SG', 'SINGAPUR': 'SG', 'SG': 'SG',
            'SINT MAARTEN': 'SX', 'SINT MAARTEN (DUTCH PART)': 'SX', 'SX': 'SX',
            'SLOVAKIA': 'SK', 'SŁOWACJA': 'SK', 'SK': 'SK', 'SLOVENIA': 'SI', 'SŁOWENIA': 'SI', 'SI': 'SI',
            'SOLOMON ISLANDS': 'SB', 'WYSPY SALOMONA': 'SB', 'SB': 'SB', 'SOMALIA': 'SO', 'SO': 'SO',
            'SOUTH AFRICA': 'ZA', 'REPUBLIKA POŁUDNIOWEJ AFRYKI': 'ZA', 'ZA': 'ZA',
            'SOUTH GEORGIA AND THE SOUTH SANDWICH ISLANDS': 'GS', 'GEORGIA POŁUDNIOWA I SANDWICH POŁUDNIOWY': 'GS', 'GS': 'GS',
            'SOUTH SUDAN': 'SS', 'SUDAN POŁUDNIOWY': 'SS', 'SS': 'SS', 'SPAIN': 'ES', 'HISZPANIA': 'ES', 'ES': 'ES',
            'SRI LANKA': 'LK', 'LK': 'LK', 'SUDAN': 'SD', 'SUDAN (THE)': 'SD', 'SD': 'SD',
            'SURINAME': 'SR', 'SURINAM': 'SR', 'SR': 'SR', 'SVALBARD AND JAN MAYEN': 'SJ', 'SVALBARD I JAN MAYEN': 'SJ', 'SJ': 'SJ',
            'SWEDEN': 'SE', 'SZWECJA': 'SE', 'SE': 'SE', 'SWITZERLAND': 'CH', 'SZWAJCARIA': 'CH', 'CH': 'CH',
            'SYRIA': 'SY', 'SYRIAN ARAB REPUBLIC': 'SY', 'SY': 'SY', 'TAIWAN': 'TW', 'TAIWAN (PROVINCE OF CHINA)': 'TW', 'TW': 'TW',
            'TAJIKISTAN': 'TJ', 'TADŻYKISTAN': 'TJ', 'TJ': 'TJ', 'TANZANIA': 'TZ', 'TANZANIA, UNITED REPUBLIC OF': 'TZ', 'TZ': 'TZ',
            'THAILAND': 'TH', 'TAJLANDIA': 'TH', 'TH': 'TH', 'TIMOR-LESTE': 'TL', 'TIMOR WSCHODNI': 'TL', 'TL': 'TL',
            'TOGO': 'TG', 'TG': 'TG', 'TOKELAU': 'TK', 'TK': 'TK', 'TONGA': 'TO', 'TO': 'TO',
            'TRINIDAD AND TOBAGO': 'TT', 'TRYNIDAD I TOBAGO': 'TT', 'TT': 'TT', 'TUNISIA': 'TN', 'TUNEZJA': 'TN', 'TN': 'TN',
            'TURKEY': 'TR', 'TURCJA': 'TR', 'TR': 'TR', 'TURKMENISTAN': 'TM', 'TM': 'TM',
            'TURKS AND CAICOS ISLANDS': 'TC', 'TURKS AND CAICOS ISLANDS (THE)': 'TC', 'TC': 'TC',
            'TUVALU': 'TV', 'TV': 'TV', 'UGANDA': 'UG', 'UG': 'UG', 'UKRAINE': 'UA', 'UKRAINA': 'UA', 'UA': 'UA',
            'UNITED ARAB EMIRATES': 'AE', 'UNITED ARAB EMIRATES (THE)': 'AE', 'AE': 'AE',
            'UNITED KINGDOM': 'GB', 'UNITED KINGDOM OF GREAT BRITAIN AND NORTHERN IRELAND': 'GB', 'UNITED KINGDOM OF GREAT BRITAIN AND NORTHERN IRELAND (THE)': 'GB', 'UK': 'GB', 'GB': 'GB',
            'UNITED STATES': 'US', 'UNITED STATES OF AMERICA': 'US', 'UNITED STATES OF AMERICA (THE)': 'US', 'USA': 'US', 'US': 'US',
            'UNITED STATES MINOR OUTLYING ISLANDS': 'UM', 'UNITED STATES MINOR OUTLYING ISLANDS (THE)': 'UM', 'UM': 'UM',
            'URUGUAY': 'UY', 'URUGWAJ': 'UY', 'UY': 'UY', 'UZBEKISTAN': 'UZ', 'UZ': 'UZ',
            'VANUATU': 'VU', 'VU': 'VU', 'VENEZUELA': 'VE', 'VENEZUELA (BOLIVARIAN REPUBLIC OF)': 'VE', 'WENEZUELA': 'VE', 'VE': 'VE',
            'VIET NAM': 'VN', 'VIETNAM': 'VN', 'WIETNAM': 'VN', 'VN': 'VN',
            'VIRGIN ISLANDS (BRITISH)': 'VG', 'WYSPY DZIEWICZE (BRYTYJSKIE)': 'VG', 'VG': 'VG',
            'VIRGIN ISLANDS (U.S.)': 'VI', 'WYSPY DZIEWICZE (USA)': 'VI', 'VI': 'VI',
            'WALLIS AND FUTUNA': 'WF', 'WALLIS I FUTUNA': 'WF', 'WF': 'WF', 'WESTERN SAHARA': 'EH', 'SAHARA ZACHODNIA': 'EH', 'EH': 'EH',
            'YEMEN': 'YE', 'JEMEN': 'YE', 'YE': 'YE', 'ZAMBIA': 'ZM', 'ZM': 'ZM',
            'ZIMBABWE': 'ZW', 'ZW': 'ZW', 'ÅLAND ISLANDS': 'AX', 'ALAND ISLANDS': 'AX', 'AX': 'AX'
        };
        
        function backToPaste() {
            document.getElementById('pasteStep').classList.remove('d-none');
            document.getElementById('mappingStep').classList.add('d-none');
        }
        
        function applyMappingExcel() {
            console.log('=== applyMappingExcel called ===');
            console.log('columnMapping:', columnMapping);
            console.log('parsedData:', parsedData);
            console.log('fieldDefinitions:', fieldDefinitions);
            
            // Validate required fields are mapped
            const requiredFields = fieldDefinitions.filter(f => f.required).map(f => f.key);
            const mappedFields = Object.values(columnMapping);
            
            console.log('requiredFields:', requiredFields);
            console.log('mappedFields:', mappedFields);
            
            const missingFields = requiredFields.filter(field => !mappedFields.includes(field));
            
            console.log('missingFields:', missingFields);
            
            if (missingFields.length > 0) {
                showAlert('Please map the following required fields:\n' + missingFields.join(', '), 'Missing Fields', 'exclamation-triangle');
                return;
            }
            
            // Transform data according to mapping
            editableData = [];
            
            console.log('Starting data transformation...');
            console.log('parsedData.length:', parsedData.length);
            
            for (let i = 1; i < parsedData.length; i++) {
                const row = {};
                
                fieldDefinitions.forEach(field => {
                    row[field.key] = '';
                });
                
                Object.keys(columnMapping).forEach(colIndex => {
                    const fieldKey = columnMapping[colIndex];
                    let value = parsedData[i][colIndex] || '';
                    
                    console.log(`Row ${i}, Col ${colIndex}: ${fieldKey} = "${value}"`);
                    
                    // Special handling for year fields - extract year from dates
                    const field = fieldDefinitions.find(f => f.key === fieldKey);
                    if (field && field.type === 'year' && value) {
                        console.log(`  Processing year field, original value: "${value}"`);
                        // If value looks like a date (contains dash or slash), extract the year
                        if (value.includes('-') || value.includes('/')) {
                            const dateParts = value.split(/[-/]/);
                            // Assume first part is year if 4 digits, otherwise last part
                            value = dateParts[0].length === 4 ? dateParts[0] : dateParts[dateParts.length - 1];
                            console.log(`  Date detected, extracted parts:`, dateParts, 'selected:', value);
                        }
                        // Ensure it's just the numeric year
                        value = value.replace(/[^0-9]/g, '');
                        console.log(`  Final year value: "${value}"`);
                    }
                    
                    // Special handling for country field - convert to Alpha-2 code
                    if (fieldKey === 'country' && value) {
                        console.log(`  Processing country field, original value: "${value}"`);
                        const normalizedValue = value.trim().toUpperCase();
                        const alpha2Code = countryCodeMap[normalizedValue];
                        if (alpha2Code) {
                            value = alpha2Code;
                            console.log(`  Country converted: "${normalizedValue}" -> "${value}"`);
                        } else {
                            console.log(`  Country not found in map, keeping original: "${value}"`);
                        }
                    }
                    
                    row[fieldKey] = value;
                });
                
                // Add internal ID
                row._id = Date.now() + i;
                row._selected = false;
                
                editableData.push(row);
                console.log(`Row ${i} processed:`, row);
            }
            
            console.log('editableData created:', editableData);
            
            // Move to edit step
            console.log('Switching to edit step...');
            document.getElementById('mappingStep').classList.add('d-none');
            document.getElementById('editStep').classList.remove('d-none');
            
            console.log('Calling renderEditableGrid...');
            renderEditableGrid();
            console.log('=== applyMappingExcel complete ===');
        }
        
        function backToMapping() {
            document.getElementById('mappingStep').classList.remove('d-none');
            document.getElementById('editStep').classList.add('d-none');
        }
        
        function renderEditableGrid() {
            const table = document.getElementById('editableGrid');
            const thead = table.querySelector('thead');
            const tbody = table.querySelector('tbody');
            
            // Render header
            thead.innerHTML = `
                <tr>
                    <th style="width: 40px;">
                        <input type="checkbox" class="form-check-input" onchange="toggleSelectAllExcel(this)">
                    </th>
                    ${fieldDefinitions.map(field => `
                        <th>${field.label}${field.required ? ' <span class="text-danger">*</span>' : ''}</th>
                    `).join('')}
                    <th style="width: 80px;">Actions</th>
                </tr>
            `;
            
            // Render rows
            tbody.innerHTML = '';
            editableData.forEach((row, rowIndex) => {
                const tr = document.createElement('tr');
                tr.dataset.rowId = row._id;
                if (row._selected) tr.classList.add('table-primary');
                
                tr.innerHTML = `
                    <td class="text-center">
                        <input type="checkbox" class="form-check-input row-select" ${row._selected ? 'checked' : ''} 
                               onchange="toggleRowSelectExcel(${row._id}, this)">
                    </td>
                    ${fieldDefinitions.map(field => renderCellExcel(row, field, rowIndex)).join('')}
                    <td class="text-center">
                        <button type="button" class="btn btn-sm btn-outline-danger" onclick="deleteRowExcel(${row._id})" title="Delete">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                `;
                
                tbody.appendChild(tr);
            });
            
            // Update count
            document.getElementById('rowCount').textContent = editableData.length;
        }
        
        function renderCellExcel(row, field, rowIndex) {
            const value = row[field.key] || '';
            const cellClass = field.required && !value ? 'table-warning' : '';
            
            if (field.type === 'select') {
                return `
                    <td class="${cellClass}">
                        <select class="form-select form-select-sm" 
                                onchange="updateCellExcel(${row._id}, '${field.key}', this.value)">
                            ${field.options.map(opt => `
                                <option value="${opt}" ${value === opt ? 'selected' : ''}>${opt || '(empty)'}</option>
                            `).join('')}
                        </select>
                    </td>
                `;
            } else if (field.type === 'year') {
                return `
                    <td class="${cellClass}">
                        <input type="number" class="form-control form-control-sm" 
                               value="${value}" 
                               min="1900" 
                               max="2100"
                               step="1"
                               placeholder="YYYY"
                               onchange="updateCellExcel(${row._id}, '${field.key}', this.value)"
                               ${field.required ? 'required' : ''}>
                    </td>
                `;
            } else {
                return `
                    <td class="${cellClass}">
                        <input type="${field.type}" class="form-control form-control-sm" 
                               value="${value}" 
                               onchange="updateCellExcel(${row._id}, '${field.key}', this.value)"
                               ${field.required ? 'required' : ''}>
                    </td>
                `;
            }
        }
        
        function updateCellExcel(rowId, fieldKey, value) {
            const row = editableData.find(r => r._id === rowId);
            if (row) {
                row[fieldKey] = value;
                renderEditableGrid();
            }
        }
        
        function toggleRowSelectExcel(rowId, checkbox) {
            const row = editableData.find(r => r._id === rowId);
            if (row) {
                row._selected = checkbox.checked;
                checkbox.closest('tr').classList.toggle('table-primary', checkbox.checked);
            }
        }
        
        function toggleSelectAllExcel(checkbox) {
            editableData.forEach(row => {
                row._selected = checkbox.checked;
            });
            renderEditableGrid();
        }
        
        function addRowExcel() {
            const newRow = {
                _id: Date.now(),
                _selected: false
            };
            
            fieldDefinitions.forEach(field => {
                newRow[field.key] = '';
            });
            
            editableData.push(newRow);
            renderEditableGrid();
        }
        
        function deleteRowExcel(rowId) {
            showConfirm(
                'Delete this row?',
                function() {
                    editableData = editableData.filter(r => r._id !== rowId);
                    renderEditableGrid();
                },
                'Confirm Delete',
                'exclamation-triangle'
            );
        }
        
        function deleteSelectedRowsExcel() {
            const selectedCount = editableData.filter(r => r._selected).length;
            
            if (selectedCount === 0) {
                showAlert('No rows selected!', 'Selection Required', 'exclamation-triangle');
                return;
            }
            
            showConfirm(
                `Delete ${selectedCount} selected row(s)?`,
                function() {
                    editableData = editableData.filter(r => !r._selected);
                    renderEditableGrid();
                },
                'Confirm Delete',
                'exclamation-triangle'
            );
        }
        
        function importExcelData() {
            const eventId = document.getElementById('eventSelectImport').value;
            
            if (!eventId) {
                showAlert('Please select an event for import!', 'Event Required', 'exclamation-triangle');
                return;
            }
            
            // Validate required fields
            const invalidRows = [];
            editableData.forEach((row, index) => {
                const requiredFields = fieldDefinitions.filter(f => f.required);
                const missingFields = requiredFields.filter(field => !row[field.key] || !row[field.key].trim());
                
                if (missingFields.length > 0) {
                    invalidRows.push({
                        index: index + 1,
                        fields: missingFields.map(f => f.label).join(', ')
                    });
                }
            });
            
            if (invalidRows.length > 0) {
                const message = 'The following rows are missing required fields:\n\n' +
                    invalidRows.slice(0, 5).map(r => `Row ${r.index}: ${r.fields}`).join('\n') +
                    (invalidRows.length > 5 ? `\n... and ${invalidRows.length - 5} more rows` : '');
                
                showAlert(message, 'Validation Error', 'exclamation-triangle');
                return;
            }
            
            showConfirm(
                `Import ${editableData.length} athletes to the database and assign to selected event?`,
                function() {
                    const importBtn = event.target;
                    importBtn.disabled = true;
                    importBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Importing...';
                    
                    // Send to server
                    fetch('participants_excel_import.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            participants: editableData,
                            event_id: eventId,
                            assign_to_event: true
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            showAlert(data.message, 'Success', 'check-circle');
                            setTimeout(() => window.location.reload(), 1500);
                        } else {
                            showAlert(data.message || 'Import failed', 'Error', 'exclamation-triangle');
                            importBtn.disabled = false;
                            importBtn.innerHTML = '<i class="fas fa-upload me-2"></i>Import to Database';
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        showAlert('Import failed: ' + error.message, 'Error', 'exclamation-triangle');
                        importBtn.disabled = false;
                        importBtn.innerHTML = '<i class="fas fa-upload me-2"></i>Import to Database';
                    });
                },
                'Confirm Import'
            );
        }
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>
