<?php
/**
 * Participant Status Control API
 * 
 * Manages participant status workflow in heat management
 * Replaces old bib_performing, bib_on_start, bib_latest_on_run system
 */

include '../includes/auth.php';
include '../includes/db.php';

header('Content-Type: application/json');

try {
    $action = $_POST['action'] ?? $_GET['action'] ?? '';
    $event_id = $_POST['event_id'] ?? $_GET['event_id'] ?? '';
    $heat_number = $_POST['heat_number'] ?? $_GET['heat_number'] ?? '';
    $run_number = $_POST['run_number'] ?? $_GET['run_number'] ?? 1;
    
    if (!$event_id) {
        throw new Exception('Event ID is required');
    }
    
    switch ($action) {
        case 'get_status':
            // Get current status of all participants in heat/run
            echo json_encode(getParticipantStatuses($pdo, $event_id, $heat_number, $run_number));
            break;
            
        case 'update_status':
            // Update single participant status
            $participant_id = $_POST['participant_id'] ?? '';
            $status = $_POST['status'] ?? '';
            echo json_encode(updateParticipantStatus($pdo, $event_id, $heat_number, $run_number, $participant_id, $status));
            break;
            
        case 'advance_next':
            // Advance to next participant (mark current as completed, next as performing)
            echo json_encode(advanceToNext($pdo, $event_id, $heat_number, $run_number));
            break;
            
        case 'mark_performing':
            // Mark specific participant as performing
            $participant_id = $_POST['participant_id'] ?? '';
            echo json_encode(markPerforming($pdo, $event_id, $heat_number, $run_number, $participant_id));
            break;
            
        case 'mark_completed':
            // Mark current performing participant as completed
            echo json_encode(markCompleted($pdo, $event_id, $heat_number, $run_number));
            break;
            
        case 'reset_heat':
            // Reset all participants in heat to initial status
            echo json_encode(resetHeatStatuses($pdo, $event_id, $heat_number, $run_number));
            break;
            
        case 'get_current':
            // Get current performing, start_gate, and next_up participants
            echo json_encode(getCurrentParticipants($pdo, $event_id, $heat_number, $run_number));
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

/**
 * Get all participant statuses for a heat/run
 */
function getParticipantStatuses($pdo, $event_id, $heat_number, $run_number) {
    $stmt = $pdo->prepare("
        SELECT 
            ep.id,
            ep.participant_id,
            ep.bib_number,
            ep.sort_order,
            ep.participant_status,
            ep.heat_number,
            ep.run_number,
            p.first_name,
            p.last_name,
            ep.category
        FROM event_participants ep
        LEFT JOIN participants p ON ep.participant_id = p.id
        WHERE ep.event_id = ? AND ep.heat_number = ?
        ORDER BY ep.sort_order ASC, ep.bib_number ASC
    ");
    $stmt->execute([$event_id, $heat_number]);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'success' => true,
        'participants' => $participants,
        'heat_number' => $heat_number,
        'run_number' => $run_number,
        'total_count' => count($participants)
    ];
}

/**
 * Update single participant status
 */
function updateParticipantStatus($pdo, $event_id, $heat_number, $run_number, $participant_id, $status) {
    $valid_statuses = ['initial', 'next_up', 'start_gate', 'performing', 'completed', 'dns', 'dnf'];
    
    if (!in_array($status, $valid_statuses)) {
        return [
            'success' => false,
            'message' => 'Invalid status. Valid statuses: ' . implode(', ', $valid_statuses)
        ];
    }
    
    $stmt = $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = ?
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_id = ?
    ");
    
    $stmt->execute([$status, $event_id, $heat_number, $participant_id]);
    
    return [
        'success' => true,
        'message' => "Participant status updated to {$status}",
        'participant_id' => $participant_id,
        'status' => $status
    ];
}

/**
 * Advance to next participant in order
 */
function advanceToNext($pdo, $event_id, $heat_number, $run_number) {
    // Get current performing participant
    $current_stmt = $pdo->prepare("
        SELECT id, participant_id, sort_order, bib_number
        FROM event_participants
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_status = 'performing'
        LIMIT 1
    ");
    $current_stmt->execute([$event_id, $heat_number]);
    $current = $current_stmt->fetch(PDO::FETCH_ASSOC);
    
    // Mark current as completed
    if ($current) {
        $pdo->prepare("
            UPDATE event_participants 
            SET participant_status = 'completed'
            WHERE id = ?
        ")->execute([$current['id']]);
    }
    
    // Mark start_gate as performing
    $start_gate_stmt = $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = 'performing'
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_status = 'start_gate'
        LIMIT 1
    ");
    $start_gate_stmt->execute([$event_id, $heat_number]);
    
    // Get next participant in order (by sort_order or bib_number)
    $next_stmt = $pdo->prepare("
        SELECT id, participant_id, bib_number
        FROM event_participants
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_status = 'initial'
        ORDER BY sort_order ASC, bib_number ASC
        LIMIT 2
    ");
    $next_stmt->execute([$event_id, $heat_number]);
    $next_participants = $next_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Set first as start_gate, second as next_up
    if (isset($next_participants[0])) {
        $pdo->prepare("
            UPDATE event_participants 
            SET participant_status = 'start_gate'
            WHERE id = ?
        ")->execute([$next_participants[0]['id']]);
    }
    
    if (isset($next_participants[1])) {
        $pdo->prepare("
            UPDATE event_participants 
            SET participant_status = 'next_up'
            WHERE id = ?
        ")->execute([$next_participants[1]['id']]);
    }
    
    return [
        'success' => true,
        'message' => 'Advanced to next participant',
        'completed' => $current ? $current['bib_number'] : null,
        'performing' => $next_participants[0]['bib_number'] ?? null,
        'start_gate' => $next_participants[1]['bib_number'] ?? null
    ];
}

/**
 * Mark specific participant as performing
 */
function markPerforming($pdo, $event_id, $heat_number, $run_number, $participant_id) {
    // Clear any existing performing status
    $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = 'completed'
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_status = 'performing'
    ")->execute([$event_id, $heat_number]);
    
    // Mark specified participant as performing
    $stmt = $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = 'performing'
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_id = ?
    ");
    $stmt->execute([$event_id, $heat_number, $participant_id]);
    
    return [
        'success' => true,
        'message' => 'Participant marked as performing',
        'participant_id' => $participant_id
    ];
}

/**
 * Mark current performing participant as completed
 */
function markCompleted($pdo, $event_id, $heat_number, $run_number) {
    $stmt = $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = 'completed'
        WHERE event_id = ? 
            AND heat_number = ? 
            AND participant_status = 'performing'
    ");
    $stmt->execute([$event_id, $heat_number]);
    
    return [
        'success' => true,
        'message' => 'Current participant marked as completed',
        'affected_rows' => $stmt->rowCount()
    ];
}

/**
 * Reset all participants in heat to initial status
 */
function resetHeatStatuses($pdo, $event_id, $heat_number, $run_number) {
    $stmt = $pdo->prepare("
        UPDATE event_participants 
        SET participant_status = 'initial'
        WHERE event_id = ? AND heat_number = ?
    ");
    $stmt->execute([$event_id, $heat_number]);
    
    return [
        'success' => true,
        'message' => 'All participant statuses reset to initial',
        'affected_rows' => $stmt->rowCount()
    ];
}

/**
 * Get current performing, start_gate, and next_up participants
 */
function getCurrentParticipants($pdo, $event_id, $heat_number, $run_number) {
    $stmt = $pdo->prepare("
        SELECT 
            ep.id,
            ep.participant_id,
            ep.bib_number,
            ep.participant_status,
            p.first_name,
            p.last_name,
            ep.category
        FROM event_participants ep
        LEFT JOIN participants p ON ep.participant_id = p.id
        WHERE ep.event_id = ? 
            AND ep.heat_number = ? 
            AND ep.participant_status IN ('performing', 'start_gate', 'next_up')
        ORDER BY 
            FIELD(ep.participant_status, 'performing', 'start_gate', 'next_up')
    ");
    $stmt->execute([$event_id, $heat_number]);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $current = [
        'performing' => null,
        'start_gate' => null,
        'next_up' => null
    ];
    
    foreach ($participants as $p) {
        $current[$p['participant_status']] = $p;
    }
    
    return [
        'success' => true,
        'current' => $current,
        'heat_number' => $heat_number,
        'run_number' => $run_number
    ];
}
?>
