<?php
include '../includes/auth.php';
include '../includes/db.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['id'])) {
    // Fetch participant data for editing
    $id = intval($_GET['id']);
    
    $participant = $pdo->prepare("SELECT * FROM participants WHERE id = ?");
    $participant->execute([$id]);
    $p = $participant->fetch(PDO::FETCH_ASSOC);
    
    if (!$p) {
        echo json_encode(['success' => false, 'message' => 'Participant not found']);
        exit;
    }
    
    // Get participant facts
    $facts_query = $pdo->prepare("SELECT * FROM participant_facts WHERE participant_id = ? ORDER BY sort_order ASC");
    $facts_query->execute([$id]);
    $participant_facts = $facts_query->fetchAll(PDO::FETCH_ASSOC);
    
    // Get participant's event assignments
    $assignments = $pdo->prepare("
        SELECT e.id, e.name as event_name, e.date as event_date, ep.heat_number, ep.bib_number
        FROM event_participants ep
        JOIN events e ON ep.event_id = e.id
        WHERE ep.participant_id = ?
        ORDER BY e.date DESC
    ");
    $assignments->execute([$id]);
    $event_assignments = $assignments->fetchAll(PDO::FETCH_ASSOC);
    
    // Parse custom_fields JSON if exists
    if (!empty($p['custom_fields'])) {
        $p['custom_fields'] = json_decode($p['custom_fields'], true);
    }
    
    echo json_encode([
        'success' => true,
        'participant' => $p,
        'facts' => $participant_facts,
        'assignments' => $event_assignments
    ]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

try {
    $id = intval($_POST['id']);
    
    // Get existing participant data
    $participant = $pdo->prepare("SELECT * FROM participants WHERE id = ?");
    $participant->execute([$id]);
    $p = $participant->fetch(PDO::FETCH_ASSOC);
    
    if (!$p) {
        throw new Exception('Participant not found');
    }
    
    // Handle file uploads
    $photo_path = $p['photo']; // Keep existing photo by default
    $motion_video_path = $p['motion_video']; // Keep existing video by default
    
    // Handle photo removal
    if (isset($_POST['remove_photo']) && $_POST['remove_photo'] == '1') {
        if ($p['photo'] && file_exists('../' . $p['photo'])) {
            unlink('../' . $p['photo']);
        }
        $photo_path = null;
    }
    
    // Handle motion video removal
    if (isset($_POST['remove_motion_video']) && $_POST['remove_motion_video'] == '1') {
        if ($p['motion_video'] && file_exists('../' . $p['motion_video'])) {
            unlink('../' . $p['motion_video']);
        }
        $motion_video_path = null;
    }
    
    // Create uploads directory if it doesn't exist
    $upload_dir = '../uploads/participants/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Handle photo upload
    if (isset($_FILES['photo']) && $_FILES['photo']['error'] === UPLOAD_ERR_OK) {
        $photo_ext = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
        $allowed_photo_ext = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        
        if (in_array($photo_ext, $allowed_photo_ext)) {
            // Delete old photo if exists
            if ($p['photo'] && file_exists('../' . $p['photo'])) {
                unlink('../' . $p['photo']);
            }
            
            $photo_filename = 'photo_' . time() . '_' . uniqid() . '.' . $photo_ext;
            $photo_path = $upload_dir . $photo_filename;
            
            if (!move_uploaded_file($_FILES['photo']['tmp_name'], $photo_path)) {
                throw new Exception("Failed to upload photo file.");
            }
            // Store relative path for database
            $photo_path = 'uploads/participants/' . $photo_filename;
        } else {
            throw new Exception("Invalid photo file type. Allowed: JPG, JPEG, PNG, GIF, WEBP");
        }
    }
    
    // Handle motion video upload
    if (isset($_FILES['motion_video']) && $_FILES['motion_video']['error'] === UPLOAD_ERR_OK) {
        $video_ext = strtolower(pathinfo($_FILES['motion_video']['name'], PATHINFO_EXTENSION));
        $allowed_video_ext = ['mp4', 'avi', 'mov', 'wmv', 'webm', 'mkv'];
        
        if (in_array($video_ext, $allowed_video_ext)) {
            // Delete old video if exists
            if ($p['motion_video'] && file_exists('../' . $p['motion_video'])) {
                unlink('../' . $p['motion_video']);
            }
            
            $video_filename = 'motion_' . time() . '_' . uniqid() . '.' . $video_ext;
            $motion_video_path = $upload_dir . $video_filename;
            
            if (!move_uploaded_file($_FILES['motion_video']['tmp_name'], $motion_video_path)) {
                throw new Exception("Failed to upload motion video file.");
            }
            // Store relative path for database
            $motion_video_path = 'uploads/participants/' . $video_filename;
        } else {
            throw new Exception("Invalid video file type. Allowed: MP4, AVI, MOV, WMV, WEBM, MKV");
        }
    }

    // Handle custom fields
    $custom_fields_json = null;
    if (isset($_POST['custom_fields']) && is_array($_POST['custom_fields'])) {
        // Filter out empty values
        $custom_fields = array_filter($_POST['custom_fields'], function($val) {
            return $val !== '' && $val !== null;
        });
        if (!empty($custom_fields)) {
            $custom_fields_json = json_encode($custom_fields);
        }
    }

    $stmt = $pdo->prepare("UPDATE participants SET
        bib=?, first_name=?, last_name=?, gender=?, fis_code=?, licence_code=?, licence_type=?,
        club=?, country=?, date_of_birth=?, category=?, photo=?, motion_video=?, custom_fields=? WHERE id=?");

    $date_of_birth = !empty($_POST['date_of_birth']) ? $_POST['date_of_birth'] : null;

    $stmt->execute([
        $_POST['bib'] ?? '', 
        $_POST['first_name'] ?? '', 
        $_POST['last_name'] ?? '', 
        $_POST['gender'] ?? '',
        $_POST['fis_code'] ?? '', 
        $_POST['licence_code'] ?? '', 
        $_POST['licence_type'] ?? '',
        $_POST['club'] ?? '', 
        $_POST['country'] ?? '', 
        $date_of_birth, 
        $_POST['category'] ?? '', 
        $photo_path, 
        $motion_video_path,
        $custom_fields_json,
        $id
    ]);

    // Handle participant facts (delete old ones and insert new ones)
    $pdo->prepare("DELETE FROM participant_facts WHERE participant_id = ?")->execute([$id]);
    
    if (isset($_POST['fact_titles']) && is_array($_POST['fact_titles'])) {
        $fact_stmt = $pdo->prepare("INSERT INTO participant_facts (participant_id, title, fact, sort_order) VALUES (?, ?, ?, ?)");
        
        foreach ($_POST['fact_titles'] as $index => $title) {
            $fact_text = $_POST['fact_texts'][$index] ?? '';
            $sort_order = $index + 1;
            
            if (!empty(trim($title)) && !empty(trim($fact_text))) {
                $fact_stmt->execute([$id, trim($title), trim($fact_text), $sort_order]);
            }
        }
    }

    echo json_encode([
        'success' => true,
        'message' => 'Participant updated successfully!',
        'participant_id' => $id
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
