<?php
include '../includes/auth.php';
include '../includes/db.php';

header('Content-Type: application/json');

try {
    // Get JSON input
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!isset($data['ids']) || !is_array($data['ids'])) {
        throw new Exception('Invalid request: ids array required');
    }
    
    $ids = array_filter(array_map('intval', $data['ids']));
    
    if (empty($ids)) {
        throw new Exception('No valid participant IDs provided');
    }
    
    $pdo->beginTransaction();
    
    $deleted_count = 0;
    $errors = [];
    
    foreach ($ids as $id) {
        try {
            // Check if participant exists
            $checkStmt = $pdo->prepare("SELECT id, first_name, last_name FROM participants WHERE id = ?");
            $checkStmt->execute([$id]);
            $participant = $checkStmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$participant) {
                $errors[] = "Participant ID {$id} not found in database";
                continue;
            }
            
            // Delete from event_participants
            $stmt = $pdo->prepare("DELETE FROM event_participants WHERE participant_id = ?");
            $stmt->execute([$id]);
            
            // Delete from participants_event
            $stmt = $pdo->prepare("DELETE FROM participants_event WHERE participant_id = ?");
            $stmt->execute([$id]);
            
            // Delete from scoring_results (only if table exists)
            try {
                $stmt = $pdo->prepare("DELETE FROM scoring_results WHERE participant_id = ?");
                $stmt->execute([$id]);
            } catch (PDOException $e) {
                // Table might not exist, continue with deletion
                if ($e->getCode() != '42S02') {
                    throw $e; // Re-throw if it's not a "table doesn't exist" error
                }
            }
            
            // Delete the participant
            $stmt = $pdo->prepare("DELETE FROM participants WHERE id = ?");
            $result = $stmt->execute([$id]);
            
            if ($stmt->rowCount() > 0) {
                $deleted_count++;
            } else {
                $errors[] = "Failed to delete participant ID {$id} ({$participant['first_name']} {$participant['last_name']}) - no rows affected";
            }
        } catch (Exception $e) {
            $errors[] = "Failed to delete participant ID {$id}: " . $e->getMessage();
        }
    }
    
    $pdo->commit();
    
    $message = "{$deleted_count} participant(s) deleted successfully";
    if (!empty($errors)) {
        $message .= ". " . count($errors) . " error(s) occurred.";
    }
    
    echo json_encode([
        'success' => true,
        'message' => $message,
        'deleted_count' => $deleted_count,
        'errors' => $errors
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
