<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/TokenSystem.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

try {
    // Check for duplicate BIB
    if (!empty($_POST['bib'])) {
        $bib_check = $pdo->prepare("SELECT id FROM participants WHERE bib = ?");
        $bib_check->execute([$_POST['bib']]);
        if ($bib_check->fetch()) {
            throw new Exception("BIB number already exists. Please use a different BIB number.");
        }
    }

    // Handle file uploads
    $photo_path = null;
    $motion_video_path = null;
    
    // Create uploads directory if it doesn't exist
    $upload_dir = '../uploads/participants/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Handle photo upload
    if (isset($_FILES['photo']) && $_FILES['photo']['error'] === UPLOAD_ERR_OK) {
        $photo_ext = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
        $allowed_photo_ext = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        
        if (in_array($photo_ext, $allowed_photo_ext)) {
            $photo_filename = 'photo_' . time() . '_' . uniqid() . '.' . $photo_ext;
            $photo_path = $upload_dir . $photo_filename;
            
            if (!move_uploaded_file($_FILES['photo']['tmp_name'], $photo_path)) {
                throw new Exception("Failed to upload photo file.");
            }
            // Store relative path for database
            $photo_path = 'uploads/participants/' . $photo_filename;
        } else {
            throw new Exception("Invalid photo file type. Allowed: JPG, JPEG, PNG, GIF, WEBP");
        }
    }
    
    // Handle motion video upload
    if (isset($_FILES['motion_video']) && $_FILES['motion_video']['error'] === UPLOAD_ERR_OK) {
        $video_ext = strtolower(pathinfo($_FILES['motion_video']['name'], PATHINFO_EXTENSION));
        $allowed_video_ext = ['mp4', 'avi', 'mov', 'wmv', 'webm', 'mkv'];
        
        if (in_array($video_ext, $allowed_video_ext)) {
            $video_filename = 'motion_' . time() . '_' . uniqid() . '.' . $video_ext;
            $motion_video_path = $upload_dir . $video_filename;
            
            if (!move_uploaded_file($_FILES['motion_video']['tmp_name'], $motion_video_path)) {
                throw new Exception("Failed to upload motion video file.");
            }
            // Store relative path for database
            $motion_video_path = 'uploads/participants/' . $video_filename;
        } else {
            throw new Exception("Invalid video file type. Allowed: MP4, AVI, MOV, WMV, WEBM, MKV");
        }
    }

    // Handle custom fields
    $custom_fields_json = null;
    if (isset($_POST['custom_fields']) && is_array($_POST['custom_fields'])) {
        // Filter out empty values
        $custom_fields = array_filter($_POST['custom_fields'], function($val) {
            return $val !== '' && $val !== null;
        });
        if (!empty($custom_fields)) {
            $custom_fields_json = json_encode($custom_fields);
        }
    }

    // Insert new participant
    $stmt = $pdo->prepare("INSERT INTO participants (
        bib, first_name, last_name, gender, fis_code, licence_code, licence_type,
        club, country, date_of_birth, category, photo, motion_video, custom_fields
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

    $date_of_birth = !empty($_POST['date_of_birth']) ? $_POST['date_of_birth'] : null;

    $stmt->execute([
        $_POST['bib'] ?? '', 
        $_POST['first_name'] ?? '', 
        $_POST['last_name'] ?? '', 
        $_POST['gender'] ?? '',
        $_POST['fis_code'] ?? '', 
        $_POST['licence_code'] ?? '', 
        $_POST['licence_type'] ?? '',
        $_POST['club'] ?? '', 
        $_POST['country'] ?? '', 
        $date_of_birth, 
        $_POST['category'] ?? '',
        $photo_path, 
        $motion_video_path,
        $custom_fields_json
    ]);

    $participant_id = $pdo->lastInsertId();
    
    // Charge with additional context
    $result = TokenSystem::charge('add_participant', $_SESSION['user_id'], [
        'participant_id' => $participant_id,
        'event_id' => $_COOKIE['event_id'] ?? null,
        'reference_type' => 'event'
    ]);
    if (!$result['success']) {
        throw new Exception($result['message']);
    }

    // Handle participant facts (repeater field)
    if (isset($_POST['fact_titles']) && is_array($_POST['fact_titles'])) {
        $fact_stmt = $pdo->prepare("INSERT INTO participant_facts (participant_id, title, fact, sort_order) VALUES (?, ?, ?, ?)");
        
        foreach ($_POST['fact_titles'] as $index => $title) {
            $fact_text = $_POST['fact_texts'][$index] ?? '';
            $sort_order = $index + 1;
            
            if (!empty(trim($title)) && !empty(trim($fact_text))) {
                $fact_stmt->execute([$participant_id, trim($title), trim($fact_text), $sort_order]);
            }
        }
    }

    // If assign to event is selected
    if (!empty($_POST['assign_event_id'])) {
        $event_id = $_POST['assign_event_id'];
        
        // Add to participants_event (import tracking)
        $stmt = $pdo->prepare("INSERT INTO participants_event (participant_id, event_id) VALUES (?, ?)");
        $stmt->execute([$participant_id, $event_id]);

        // If heat is selected, also add to event_participants
        if (!empty($_POST['heat_number'])) {
            $heat_number = $_POST['heat_number'];
            
            // Check for duplicate BIB in same heat
            $bib_check = $pdo->prepare("
                SELECT COUNT(*) FROM event_participants ep
                JOIN participants p ON ep.participant_id = p.id
                WHERE ep.event_id = ? AND ep.heat_number = ? AND p.bib = ?
            ");
            $bib_check->execute([$event_id, $heat_number, $_POST['bib'] ?? '']);
            
            if ($bib_check->fetchColumn() == 0) {
                $stmt = $pdo->prepare("INSERT INTO event_participants (event_id, participant_id, heat_number) VALUES (?, ?, ?)");
                $stmt->execute([$event_id, $participant_id, $heat_number]);
            }
        }
    }

    echo json_encode([
        'success' => true,
        'message' => 'Participant added successfully!',
        'participant_id' => $participant_id
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
