<?php
/**
 * Fixed Navbar Component - Reusable navigation bar for StyleScore platform
 * Based on event_actions_menu.php structure with enhanced features
 * 
 * Required variables:
 * - $current_user: User information (optional)
 * - $event_id: The ID of the current event (optional)
 * 
 * Optional variables:
 * - $show_token_balance: Whether to show token balance (default: true)
 * - $show_event_selector: Whether to show event dropdown (default: true)
 * - $use_localStorage: Whether to use localStorage for event switching (default: true)
 * - $navbar_theme: 'light' or 'dark' (default: 'dark')
 */

// Get the base URL for all links
$baseUrl = getBaseUrl();

// Get event_id from URL, then cookies, then null
$event_id = $_GET['event_id'] ?? $_COOKIE['selectedEventId'] ?? $_COOKIE['event_id'] ?? null;

// Set default values for optional variables
$show_token_balance = $show_token_balance ?? true;
$show_event_selector = $show_event_selector ?? true;
$use_localStorage = $use_localStorage ?? true;
$navbar_theme = $navbar_theme ?? 'dark';

// Database connection for dynamic data
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/event_access.php';

// Fetch user token balance if available using existing TokenSystem
$token_balance = 0;
if (isset($_SESSION['user_id']) && $_SESSION['user_id'] > 0 && $show_token_balance) {
    try {
        if (file_exists(__DIR__ . '/../includes/TokenSystem.php')) {
            include_once __DIR__ . '/../includes/TokenSystem.php';
            $token_balance = TokenSystem::getBalance($_SESSION['user_id']);
        }
    } catch (Exception $e) {
        error_log("Token balance error: " . $e->getMessage());
        $token_balance = 0;
    }
}

// Fetch recent events for dropdown with access control
$recent_events = [];
if ($show_event_selector) {
    $access_filter = getEventAccessSQL('id');
    $sql = "SELECT id, name, date, status FROM events WHERE 1=1" . $access_filter['sql'] . " ORDER BY date DESC LIMIT 10";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($access_filter['params']);
    $recent_events = $stmt->fetchAll();
}

// Get current event info - try from URL/cookie first, then localStorage will update it via JS
$current_event = null;
$invalid_event_id = false;
if (isset($event_id) && $event_id) {
    // Verify user has access to this event
    if (hasEventAccess($event_id)) {
        $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
        $stmt->execute([$event_id]);
        $current_event = $stmt->fetch();
    } else {
        // User doesn't have access - mark for JS cleanup
        $invalid_event_id = true;
        $event_id = null;
    }
}

// Generate unique ID for this navbar instance
$navbar_id = 'stylescore-navbar-' . uniqid();
?>

<style>
.navbar-fixed-top {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    z-index: 1050;
    backdrop-filter: blur(10px);
    box-shadow: 0 2px 20px rgba(0,0,0,0.1);
}

.navbar-brand-logo {
    font-size: 1.5rem;
    font-weight: bold;
    background: linear-gradient(45deg, #007bff, #6f42c1);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.token-balance {
    background: linear-gradient(45deg, #28a745, #20c997);
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.875rem;
    font-weight: 600;
    animation: pulse-subtle 2s infinite;
}

@keyframes pulse-subtle {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.8; }
}

.event-status-indicator {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    display: inline-block;
    margin-right: 0.5rem;
}

.event-status-active { background-color: #28a745; }
.event-status-setup { background-color: #ffc107; }
.event-status-completed { background-color: #6c757d; }

.navbar-nav .nav-link {
    display: flex;
    align-items: center;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    transition: all 0.2s ease;
}

.navbar-nav .nav-link:hover {
    background-color: rgba(255,255,255,0.1);
    transform: translateY(-1px);
}

.user-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: linear-gradient(45deg, #007bff, #6f42c1);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: bold;
    font-size: 0.875rem;
}

.notification-badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #dc3545;
    color: white;
    border-radius: 50%;
    width: 18px;
    height: 18px;
    font-size: 0.75rem;
    display: flex;
    align-items: center;
    justify-content: center;
    animation: bounce 2s infinite;
}

@keyframes bounce {
    0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
    40% { transform: translateY(-10px); }
    60% { transform: translateY(-5px); }
}

.dropdown-menu {
    border: none;
    box-shadow: 0 10px 30px rgba(0,0,0,0.15);
    border-radius: 10px;
    margin-top: 0.5rem;
}

.dropdown-item {
    padding: 0.75rem 1rem;
    transition: all 0.2s ease;
}

.dropdown-item:hover {
    background-color: #f8f9fa;
    transform: translateX(5px);
}
.navbar-brand-logo img {
    height: 35px;
}

/* Quick Search Styles */
#quickSearch {
    background: rgba(255,255,255,0.9);
    border: 1px solid rgba(255,255,255,0.3);
    transition: all 0.3s ease;
}

#quickSearch:focus {
    background: rgba(255,255,255,1);
    border-color: #007bff;
    box-shadow: 0 0 0 0.2rem rgba(0,123,255,0.25);
}

.search-result-item:hover {
    background-color: #f8f9fa !important;
    transform: translateX(3px);
}

#quickSearchResults {
    border: 1px solid rgba(0,0,0,0.1);
    border-radius: 8px;
    max-width: 400px;
}

.search-result-item mark {
    background-color: #fff3cd;
    padding: 1px 2px;
    border-radius: 2px;
}
.navbar-fixed-top .nav-link {
    color: white;
}
.nav-link:focus, .nav-link:hover {
    color: var(--bs-nav-link-hover-color);
}
/* Mobile responsiveness */
@media (max-width: 768px) {
            .navbar-nav {
                text-align: center;
                padding: 1rem 0;
            }
            .navbar-nav .nav-link {
            justify-content: center !important;
            }
            ul.dropdown-menu {
                text-align: center !important;
            }
            .token-balance {
                margin: 0.5rem 0;
            }

            /* Make first-level mobile menu items ~30% larger for easier tapping */
            .navbar-nav > .nav-item > .nav-link {
                font-size: 1.1rem;
                padding-top: 0.75rem;
                padding-bottom: 0.75rem;
            }

            .navbar-nav > .nav-item > .nav-link i {
                width: 1.5rem;
                height: 1.5rem;
            }

            /* Fix mobile menu scrolling when content exceeds viewport */
            .navbar-collapse {
                max-height: calc(100vh - 80px);
                overflow-y: auto;
                overflow-x: hidden;
                -webkit-overflow-scrolling: touch;
            }

            .navbar-collapse.show,
            .navbar-collapse.collapsing {
                position: absolute;
                top: 100%;
                left: 0;
                right: 0;
                background: inherit;
                backdrop-filter: inherit;
                box-shadow: 0 4px 20px rgba(0,0,0,0.15);
                z-index: 1040;
            }
            #navbarContent{
                                    background: linear-gradient(199deg, rgba(25, 162, 255, 1) 0%, rgba(52, 89, 230, 1) 30%);
             }
            }
   

</style>

<nav class="navbar navbar-expand-lg navbar-<?= $navbar_theme ?> bg-ss-gradient navbar-fixed-top" id="<?= $navbar_id ?>">
    <div class="container-fluid">
        <!-- Brand -->
        
        <a class="navbar-brand navbar-brand-logo" href="<?= $baseUrl ?>/v2/dashboard.php">
            <img src="<?= $baseUrl ?>/v2/assets/img/SVG/logo-ss-_neo.svg" alt="Logo" class="me-2">
        </a>

        <!-- Mobile toggle -->
        <div class="navbar-toggler border-0 text-white" type="button" data-bs-toggle="collapse" data-bs-target="#navbarContent">
            <i data-lucide="menu" class=""s tyle="
    width: 30px;
    height: 30px;
"></i>
</div>

        <div class="collapse navbar-collapse" id="navbarContent">
            <!-- Event Selector -->
            <?php if ($show_event_selector): ?>
            <div class="navbar-nav me-auto">
                <div class="nav-item dropdown">
                    <a id="eventSelectorButton" class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                        <i data-lucide="calendar" class="me-1"></i>
                        <span id="eventSelectorText">
                        <?php if ($current_event): ?>
                            <span class="event-status-indicator event-status-<?= $current_event['status'] ?>"></span>
                            <?= htmlspecialchars(substr($current_event['name'], 0, 20)) ?><?= strlen($current_event['name']) > 20 ? '...' : '' ?>
                        <?php else: ?>
                            Select Event
                        <?php endif; ?>
                        </span>
                    </a>
                    <ul class="dropdown-menu" id="eventSelectorDropdown">
                        <li><h6 class="dropdown-header">Recent Events</h6></li>
                        <?php foreach ($recent_events as $recent_event): ?>
                        <li>
                            <a class="dropdown-item event-selector-item <?= ($current_event && $current_event['id'] == $recent_event['id']) ? 'active' : '' ?>" 
                               href="<?= $baseUrl ?>/v2/admin/event_dashboard.php?event_id=<?= $recent_event['id'] ?>" 
                               data-event-id="<?= $recent_event['id'] ?>"
                               data-event-name="<?= htmlspecialchars($recent_event['name']) ?>"
                               data-event-status="<?= $recent_event['status'] ?>">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <span class="event-status-indicator event-status-<?= $recent_event['status'] ?>"></span>
                                         <i data-lucide="calendar-cog" class="me-2"></i> <?= htmlspecialchars($recent_event['name']) ?>
                                    </div>
                                    <small class="text-muted"><?= date('M j', strtotime($recent_event['date'])) ?></small>
                                </div>
                            </a>
                        </li>
                        <?php endforeach; ?>
                        <li><hr class="dropdown-divider"></li>
                          <li class="nav-item dropdown">
                            <a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/event_edit.php?id=<?= $event_id ?>" role="button">
                                <i data-lucide="settings-2" class="me-2"></i>Edit Event
                            </a>                    
                        </li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/events.php">
                            <i data-lucide="calendar-days" class="me-2"></i>Manage Events
                        </a></li>
                    </ul>
                </div>
            </div>
            <?php endif; ?>
            
            
            <!-- Main Navigation Dropdowns -->
            <ul class="navbar-nav me-auto">


            <?php if ($event_id): ?>
                      
              
                <?php endif; ?>
                <li class="nav-item dropdown">
                    <a class="nav-link" href="#"  data-action="event-dashboard" role="button">
                        <i data-lucide="layout-dashboard" class="me-2"></i>Event Dashboard
                    </a>                    
                </li>

                <!-- Athletes & Scoring -->
                <li class="nav-item dropdown">
                    
                    <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                        <i data-lucide="users" class="me-1"></i>Athletes
                    </a>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item event-link" href="#" data-action="categories">
                            <i data-lucide="tags" class="me-2"></i>Divisions
                        </a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><h6 class="dropdown-header">Athletes & BIBs</h6></li>
                                        
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/bib_manager.php">
                            <i data-lucide="hash" class="me-2"></i>BIB Management
                        </a></li>
                        
                         <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/participants.php">
                            <i data-lucide="users" class="me-2"></i>Athletes
                        </a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><h6 class="dropdown-header">Heat Configuration</h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/heats_configure.php">
                            <i data-lucide="user-check" class="me-2"></i>Assign to Heats
                        </a></li>
                        
                    </ul>
                </li>

                <li class="nav-item dropdown">
                    <a class="nav-link"  href="<?= $baseUrl ?>/v2/admin/admin_heat_config.php" role="button">
                        <i data-lucide="clipboard-clock" class="me-2"></i>Heats control
                    </a>                    
                </li>
                

                

                <!-- Judges & Scoring -->
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                        <i data-lucide="scale" class="me-1"></i>Judges
                    </a>
                    <ul class="dropdown-menu">
                        <li><h6 class="dropdown-header">Management</h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/judge_manager.php">
                            <i data-lucide="user-star" class="me-2"></i>Judges Management
                        </a></li>
                        <li><h6 class="dropdown-header">Scoring</h6></li>                        
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/judge/dashboard.php" title="Alt+S">
                            <i data-lucide="calculator" class="me-2"></i>Score Input
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/judge/head_judge.php" title="Alt+H">
                            <i data-lucide="crown" class="me-2"></i>Head Judge
                        </a></li>
                        <li><hr class="dropdown-divider"></li>
                         <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/head_judge_override.php">
                            <i data-lucide="edit-3" class="me-2"></i>Override Scores
                        </a></li>
                         <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/active_heat_panel.php">
                            <i data-lucide="activity" class="me-2"></i>Active Heat Panel
                        </a></li>
                    </ul>
                </li>

                <!-- Results & Reports -->
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                        <i data-lucide="file-text" class="me-1"></i>Results
                    </a>
                    <ul class="dropdown-menu">
                        <li><h6 class="dropdown-header">Setup</h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/event_start_list.php">
                            <i data-lucide="clipboard-list" class="me-2"></i>Start Lists
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/event_start_list.php?view_type=summary_table">
                            <i data-lucide="trophy" class="me-2"></i>Results
                        </a></li>
                         <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/results_styling.php">
                            <i data-lucide="palette" class="me-2"></i>Results Styling
                        </a></li>
                       <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/public_results.php">
                            <i data-lucide="globe" class="me-2"></i>Results Public
                        </a></li>
                       <li><hr class="dropdown-divider"></li>
                         <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/publish_servers.php">
                            <i data-lucide="server" class="me-2"></i>Publish Servers
                        </a></li>                       
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/judge/event_reports.php">
                            <i data-lucide="line-chart" class="me-2"></i>Event Reports
                        </a></li>
                        
                    </ul>
                </li>

                <!-- Live Features -->
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                        <i data-lucide="monitor" class="me-1"></i>Live
                    </a>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/live_scoreboard_tabs.php">
                            <i data-lucide="monitor" class="me-2"></i>Live Board
                            <span class="badge bg-danger ms-1">LIVE</span>
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/public_event_dashboard.php" target="_blank">
                            <i data-lucide="globe" class="me-2"></i>Public View
                            <span class="badge bg-danger ms-1">LIVE</span>
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/tv-overlay/" target="_blank">
                            <i data-lucide="tv" class="me-2"></i>TV Overlay
                        </a></li>
                    </ul>
                </li>

                <!-- System -->
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                        <i data-lucide="settings" class="me-1"></i>System
                    </a>
                    <ul class="dropdown-menu">
                        <li><h6 class="dropdown-header">Format Management</h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/scoring_format_edit.php">
                            <i data-lucide="settings" class="me-2"></i>Scoring Format Editor
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/figures_viewer.php">
                            <i data-lucide="book-open" class="me-2"></i>Figures Dictionary
                        </a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><h6 class="dropdown-header">Advanced</h6></li>
                       
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/scoring_analytics.php">
                            <i data-lucide="bar-chart-3" class="me-2"></i>Analytics
                        </a></li>
                       
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/notifications.php">
                            <i data-lucide="bell" class="me-2"></i>Notifications
                            <span class="notification-badge" id="navbar-notification-badge" style="display: none;">0</span>
                        </a></li>
                         <li><hr class="dropdown-divider"></li>
                        <li><h6 class="dropdown-header">Legacy  panels</h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/participants_v2.php">
                            <i data-lucide="users" class="me-2"></i>Athletes
                        </a></li>       
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/bib_assign.php">
                            <i data-lucide="hash" class="me-2"></i>BIB Series
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/bib_distribute.php">
                            <i data-lucide="target" class="me-2"></i>BIB Distribution
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/judge_add.php">
                            <i data-lucide="user-plus" class="me-2"></i>Judges Management
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/judge_assign.php">
                            <i data-lucide="link-2" class="me-2"></i>Assign Judge
                        </a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><h6 class="dropdown-header">Administration</h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/user_management.php">
                            <i data-lucide="users" class="me-2"></i>User Management
                            <span class="badge bg-primary ms-1">Admin</span>
                        </a></li>
                        <?php if (hasPermission('manage_permissions')): ?>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/permissions_manager.php">
                            <i data-lucide="users" class="me-2"></i>Permissions
                            <span class="badge bg-primary ms-1">Super Admin</span>
                        </a></li>
                        <?php endif; ?>
                         <?php if (hasPermission('manage_token_system')): ?>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/admin/token_management.php">
                            <i data-lucide="coins" class="me-2"></i>Token System
                            <span class="badge bg-warning ms-1">Credits</span>
                        </a></li>
                        <?php endif; ?>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/faq.php">
                            <i data-lucide="help-circle" class="me-2"></i>Help
                        </a></li>
                    </ul>
                </li>
            </ul>

            <!-- Right Side - Token Balance & User Menu -->
            <ul class="navbar-nav">
                <!-- Token Balance -->
                <?php if ($show_token_balance && isset($_SESSION['user_id']) && $_SESSION['user_id'] > 0): ?>
                <li class="nav-item">
                    <a class="nav-link" href="<?= $baseUrl ?>/v2/admin/token_management.php" title="Manage Tokens" data-ss-tooltip="Current token balance - click to manage tokens" data-ss-placement="bottom">
                        <span class="token-balance" data-stat="token-balance">
                            <i data-lucide="coins" class="me-1"></i>
                            <?= number_format($token_balance, 2) ?>
                        </span>
                    </a>
                </li>
                <?php endif; ?>

                <!-- User Menu -->
                <?php if (isset($_SESSION['user_id']) && $_SESSION['user_id'] > 0): ?>
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown" data-ss-tooltip="User menu and profile options" data-ss-placement="bottom-end">
                        <div class="user-avatar me-2">
                            <?= strtoupper(substr($_SESSION['username'] ?? 'U', 0, 1)) ?>
                        </div>
                        <span class="d-none d-lg-inline"><?= htmlspecialchars($_SESSION['username'] ?? 'User') ?></span>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><h6 class="dropdown-header">
                            <?= htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username'] ?? 'User') ?>
                        </h6></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/profile.php">
                            <i data-lucide="user" class="me-2"></i>Profile
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/settings.php">
                            <i data-lucide="settings" class="me-2"></i>Settings
                        </a></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/help.html">
                            <i data-lucide="help-circle" class="me-2"></i>Help
                        </a></li>
                        <li><a class="dropdown-item" data-bs-toggle="modal" data-bs-target="#sessionModal">
              <i data-lucide="info" class="dropdown-icon"></i>Session Info
            </a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="<?= $baseUrl ?>/v2/logout.php">
                            <i data-lucide="log-out" class="me-2"></i>Logout
                        </a></li>
                    </ul>
                </li>
                <?php else: ?>
                <li class="nav-item">
                    <a class="nav-link" href="<?= $baseUrl ?>/v2/login.php">
                        <i data-lucide="log-in" class="me-1"></i>Login
                    </a>
                </li>
                <?php endif; ?>
            </ul>
        </div>
    </div>
</nav>

<!-- Event Selection Required Modal -->
<div class="modal fade" id="eventSelectionModal" tabindex="-1" aria-labelledby="eventSelectionModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title" id="eventSelectionModalLabel">
                    <i data-lucide="alert-triangle" class="me-2"></i>Event Selection Required
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p class="mb-3">
                    <i data-lucide="info" class="me-2"></i>
                    Please select an event to continue. This action requires an active event context.
                </p>
                
                <div class="mb-3">
                    <label for="modalEventSelect" class="form-label fw-bold">Select Event:</label>
                    <select class="form-select" id="modalEventSelect">
                        <option value="">Choose an event...</option>
                        <?php foreach ($recent_events as $recent_event): ?>
                        <option value="<?= $recent_event['id'] ?>" 
                                data-event-name="<?= htmlspecialchars($recent_event['name']) ?>"
                                data-event-status="<?= $recent_event['status'] ?>">
                            <?= htmlspecialchars($recent_event['name']) ?> 
                            (<?= date('M j, Y', strtotime($recent_event['date'])) ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="alert alert-info mb-0">
                    <i data-lucide="lightbulb" class="me-2"></i>
                    <small>You can also select an event from the top navigation bar at any time.</small>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmEventSelection">
                    <i data-lucide="check" class="me-1"></i>Select Event
                </button>
            </div>
        </div>
    </div>
</div>


<script>
// Base URL for all JavaScript fetch calls - preserves port number
const BASE_URL = '<?= $baseUrl ?>';

<?php if ($invalid_event_id): ?>
// Clear invalid event_id from cookies and localStorage
(function() {
    // Clear localStorage
    localStorage.removeItem('selectedEventId');
    localStorage.removeItem('event_id');
    
    // Clear cookies
    document.cookie = 'event_id=;expires=Thu, 01 Jan 1970 00:00:00 UTC;path=/;SameSite=Lax';
    document.cookie = 'selectedEventId=;expires=Thu, 01 Jan 1970 00:00:00 UTC;path=/;SameSite=Lax';
    
    console.warn('⚠️ Access denied to selected event. Cookies and localStorage cleared.');
})();
<?php endif; ?>

// Global helper functions for event_id management (available across admin)
window.saveEventIdGlobal = function(eventId) {
    if (!eventId) return;
    
    localStorage.setItem('selectedEventId', eventId);
    localStorage.setItem('event_id', eventId);
    
    // Set cookie for 30 days
    const expires = new Date();
    expires.setTime(expires.getTime() + (30 * 24 * 60 * 60 * 1000));
    document.cookie = `event_id=${eventId};expires=${expires.toUTCString()};path=/;SameSite=Lax`;
    document.cookie = `selectedEventId=${eventId};expires=${expires.toUTCString()};path=/;SameSite=Lax`;
    
    window.currentEventId = eventId;
    console.log('Event ID saved to localStorage and cookie:', eventId);
};

window.getEventIdGlobal = function() {
    // Priority: explicit selectedEventId, then event_id, then cookies
    const fromStorage = localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
    if (fromStorage) return fromStorage;

    const fromCookie = getCookieGlobal('selectedEventId') || getCookieGlobal('event_id');
    return fromCookie || null;
};

// Hard guard: ensure an event is selected before using backend tools
window.requireEventIdGlobal = function(showAlert = true) {
    const eventId = window.getEventIdGlobal();
    if (!eventId) {
        if (showAlert) {
            showEventSelectionModal();
        }
        return null;
    }
    return eventId;
};

// Show event selection modal
window.showEventSelectionModal = function() {
    const modal = new bootstrap.Modal(document.getElementById('eventSelectionModal'));
    modal.show();
};

window.getCookieGlobal = function(name) {
    const nameEQ = name + "=";
    const ca = document.cookie.split(';');
    for(let i = 0; i < ca.length; i++) {
        let c = ca[i];
        while (c.charAt(0) === ' ') c = c.substring(1, c.length);
        if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
    }
    return null;
};

document.addEventListener('DOMContentLoaded', function() {
    // Initialize Lucide icons
    if (typeof lucide !== 'undefined') {
        lucide.createIcons();
    }
    
    // Event Selection Modal handlers
    const confirmEventBtn = document.getElementById('confirmEventSelection');
    const modalEventSelect = document.getElementById('modalEventSelect');
    const eventSelectionModal = document.getElementById('eventSelectionModal');
    
    if (confirmEventBtn && modalEventSelect) {
        confirmEventBtn.addEventListener('click', function() {
            const selectedEventId = modalEventSelect.value;
            
            if (!selectedEventId) {
                // Highlight the select field
                modalEventSelect.classList.add('is-invalid');
                return;
            }
            
            // Remove invalid class if present
            modalEventSelect.classList.remove('is-invalid');
            
            // Save the event ID
            saveEventId(selectedEventId);
            window.saveEventIdGlobal(selectedEventId);
            
            // Close modal
            const modal = bootstrap.Modal.getInstance(eventSelectionModal);
            if (modal) {
                modal.hide();
            }
            
            // Redirect to event dashboard
            window.location.href = `${BASE_URL}/v2/admin/event_dashboard.php?event_id=${selectedEventId}`;
        });
        
        // Remove invalid class when user selects an option
        modalEventSelect.addEventListener('change', function() {
            this.classList.remove('is-invalid');
        });
    }
    
    // Re-initialize Lucide icons when modal is shown
    if (eventSelectionModal) {
        eventSelectionModal.addEventListener('shown.bs.modal', function() {
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }
        });
    }
    
    
    
    // Quick Search functionality with enhanced Popper.js
    const quickSearch = document.getElementById('quickSearch');
    const quickSearchBtn = document.getElementById('quickSearchBtn');
    const quickSearchResults = document.getElementById('quickSearchResults');
    
    if (quickSearch && quickSearchResults) {
        let searchTimeout;
        let searchDropdown = null;
        
        // Initialize search dropdown with Popper.js
        if (window.StyleScorePopper) {
            searchDropdown = StyleScorePopper.createSearchDropdown(quickSearch, quickSearchResults, {
                placement: 'bottom-start',
                modifiers: [
                    {
                        name: 'offset',
                        options: { offset: [0, 4] }
                    }
                ]
            });
        }
        
        quickSearch.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            const query = this.value.trim();
            
            if (query.length >= 2) {
                searchTimeout = setTimeout(() => {
                    performQuickSearch(query);
                }, 300);
            } else {
                hideSearchResults();
            }
        });
        
        quickSearch.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                const query = this.value.trim();
                if (query.length >= 2) {
                    performQuickSearch(query);
                }
            } else if (e.key === 'Escape') {
                hideSearchResults();
                this.blur();
            }
        });
        
        quickSearchBtn.addEventListener('click', function() {
            const query = quickSearch.value.trim();
            if (query.length >= 2) {
                performQuickSearch(query);
            }
        });
        
        // Hide results when clicking outside
        document.addEventListener('click', function(e) {
            if (!quickSearch.contains(e.target) && !quickSearchResults.contains(e.target)) {
                hideSearchResults();
            }
        });
        
        function showSearchResults() {
            if (searchDropdown) {
                searchDropdown.show();
            } else {
                quickSearchResults.style.display = 'block';
            }
        }
        
        function hideSearchResults() {
            if (searchDropdown) {
                searchDropdown.hide();
            } else {
                quickSearchResults.style.display = 'none';
            }
        }
    }
    
    function performQuickSearch(query) {
        // Show loading state
        quickSearchResults.innerHTML = '<div class="p-3 text-center"><i data-lucide="loader" class="me-2"></i>Searching...</div>';
        showSearchResults();
        
        // Simple search implementation - can be enhanced to use existing search APIs
        const searchData = [
            { type: 'events', name: 'Sample Event 1', id: 1, url: '/v2/admin/event_dashboard.php?event_id=1' },
            { type: 'events', name: 'Championship 2025', id: 2, url: '/v2/admin/event_dashboard.php?event_id=2' },
            { type: 'participants', name: 'John Doe', id: 1, url: '/v2/admin/participants.php' },
            { type: 'judges', name: 'Jane Smith', id: 1, url: '/v2/admin/judges.php' }
        ];
        
        const results = searchData.filter(item => 
            item.name.toLowerCase().includes(query.toLowerCase())
        );
        
        setTimeout(() => displayQuickSearchResults(results, query), 200);
    }
    
    function displayQuickSearchResults(results, query) {
        if (results.length === 0) {
            quickSearchResults.innerHTML = `
                <div class="p-3 text-center text-muted">
                    <i data-lucide="search-x" class="me-2"></i>No results found for "${query}"
                </div>`;
        } else {
            let html = '';
            results.slice(0, 8).forEach(item => {
                const icon = getTypeIcon(item.type);
                html += `
                    <a href="${item.url}" class="d-block p-2 text-decoration-none text-dark border-bottom search-result-item">
                        <div class="d-flex align-items-center">
                            <i data-lucide="${icon}" class="me-2" style="width: 16px; height: 16px;"></i>
                            <div>
                                <div class="fw-medium">${highlightMatch(item.name, query)}</div>
                                <small class="text-muted text-capitalize">${item.type}</small>
                            </div>
                        </div>
                    </a>`;
            });
            
            quickSearchResults.innerHTML = html;
        }
        
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }
    
    function getTypeIcon(type) {
        const icons = {
            events: 'calendar',
            participants: 'user',
            judges: 'gavel'
        };
        return icons[type] || 'search';
    }
    
    function highlightMatch(text, query) {
        const regex = new RegExp(`(${query})`, 'gi');
        return text.replace(regex, '<mark>$1</mark>');
    }
    
    // Keyboard shortcut for quick search (Ctrl+K)
    document.addEventListener('keydown', function(e) {
        if (e.ctrlKey && e.key === 'k') {
            e.preventDefault();
            if (quickSearch) quickSearch.focus();
        }
    });
    
    // Event selector functionality
    <?php if ($use_localStorage): ?>
    
    // Helper function to set cookie
    function setCookie(name, value, days = 30) {
        const expires = new Date();
        expires.setTime(expires.getTime() + (days * 24 * 60 * 60 * 1000));
        document.cookie = `${name}=${value};expires=${expires.toUTCString()};path=/;SameSite=Lax`;
    }
    
    // Helper function to get cookie
    function getCookie(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for(let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
        }
        return null;
    }
    
    // Helper function to save event_id to both localStorage and cookie
    function saveEventId(eventId) {
        localStorage.setItem('selectedEventId', eventId);
        localStorage.setItem('event_id', eventId);
        setCookie('event_id', eventId, 30);
        setCookie('selectedEventId', eventId, 30);
        window.currentEventId = eventId;
        console.log('Event ID saved to localStorage and cookie:', eventId);
    }
    
    // Initialize event from localStorage/cookie on page load
    (function() {
        const urlParams = new URLSearchParams(window.location.search);
        const urlEventId = urlParams.get('event_id');
        
        // If URL has event_id, save it to localStorage and cookie
        if (urlEventId) {
            saveEventId(urlEventId);
        } else {
            // Try to load from localStorage or cookie
            const storedEventId = localStorage.getItem('selectedEventId') || 
                                  localStorage.getItem('event_id') ||
                                  getCookie('event_id') ||
                                  getCookie('selectedEventId');
            
            if (storedEventId) {
                // Sync to both storage mechanisms
                saveEventId(storedEventId);
            }
        }
        
        // Set global currentEventId for use throughout the page
        window.currentEventId = localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
        
        // Update event selector display to reflect stored event
        updateEventSelectorDisplay();
    })();
    
    // Function to update the event selector button text and active state in dropdown
    function updateEventSelectorDisplay() {
        const currentEventId = window.currentEventId || localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
        
        if (!currentEventId) {
            return; // No event selected
        }
        
        // Find the event in the dropdown by ID
        const eventItem = document.querySelector(`.event-selector-item[data-event-id="${currentEventId}"]`);
        
        if (eventItem) {
            // Remove active class from all items
            document.querySelectorAll('.event-selector-item').forEach(item => {
                item.classList.remove('active');
            });
            
            // Add active class to current event
            eventItem.classList.add('active');
            
            // Update the button text with event name and status
            const eventName = eventItem.getAttribute('data-event-name');
            const eventStatus = eventItem.getAttribute('data-event-status');
            const eventSelectorText = document.getElementById('eventSelectorText');
            
            if (eventSelectorText && eventName) {
                const truncatedName = eventName.length > 20 ? eventName.substring(0, 20) + '...' : eventName;
                eventSelectorText.innerHTML = `
                    <span class="event-status-indicator event-status-${eventStatus}"></span>
                    ${truncatedName}
                `;
            }
            
            console.log('✅ Event selector updated to show:', eventName);
        }
    }
    
    document.querySelectorAll('[data-event-id]').forEach(item => {
        item.addEventListener('click', function(e) {
            const eventId = this.getAttribute('data-event-id');
            saveEventId(eventId);
            
            // Update event selector display immediately
            updateEventSelectorDisplay();
            
            // If the link has an href, let it navigate naturally
            // Otherwise, redirect to event dashboard
            if (!this.href || this.href === '#') {
                e.preventDefault();
                window.location.href = `${BASE_URL}/v2/admin/event_dashboard.php?event_id=${eventId}`;
            }
            // If href exists, the browser will handle navigation automatically
        });
    });
    
    // Load event from localStorage if available
    const savedEventId = localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
    if (savedEventId && !<?= json_encode(isset($event_id) ? $event_id : null) ?>) {
        //window.location.href = window.location.pathname + '?event_id=' + savedEventId;
    }
    
    // Event Actions functionality for localStorage mode
    function handleEventAction(action) {
        const eventId = localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
        
        if (!eventId) {
            showEventSelectionModal();
            return;
        }
        
        let url = '';
        switch(action) {
            case 'event-dashboard':
                url = `${BASE_URL}/v2/admin/event_dashboard.php?event_id=${eventId}`;
                break;
            case 'edit':
                url = `${BASE_URL}/v2/admin/event_edit.php?id=${eventId}`;
                break;
            case 'participants':
                url = `${BASE_URL}/v2/admin/heats_configure.php?event_id=${eventId}`;
                break;
            case 'categories':
                url = `${BASE_URL}/v2/admin/event_categories.php?event_id=${eventId}`;
                break;
            case 'judges':
                url = `${BASE_URL}/v2/admin/judge_assign.php?event_id=${eventId}`;
                break;
            case 'head-judge':
                url = `${BASE_URL}/v2/judge/head_judge.php?event_id=${eventId}`;
                break;
            case 'score':
                url = `${BASE_URL}/v2/judge/score.php?event_id=${eventId}`;
                break;
            case 'scoreboard':
                url = `${BASE_URL}/v2/admin/live_scoreboard_3.php?event_id=${eventId}`;
                break;
            case 'results':
                url = `${BASE_URL}/v2/admin/event_start_list.php?view_type=summary_table&event_id=${eventId}`;
                break;
            case 'reports':
                url = `${BASE_URL}/v2/judge/event_reports.php?event_id=${eventId}`;
                break;
            case 'heat-flow':
                url = `${BASE_URL}/v2/admin/heat_flow_preview.php?event_id=${eventId}`;
                break;
            case 'start-list':
                url = `${BASE_URL}/v2/admin/event_start_list.php?view_type=start_list&event_id=${eventId}`;
                break;
            case 'public-view':
                url = `${BASE_URL}/v2/public_event_dashboard.php?event_id=${eventId}`;
                window.open(url, '_blank');
                return;
            case 'publish_servers':
                url = `${BASE_URL}/v2/admin/publish_servers.php?event_id=${eventId}`;
                window.open(url, '_blank');
                return;
            default:
                console.warn('Unknown action:', action);
                return;
        }
        
        if (url) {
            window.location.href = url;
        }
    }
    
    // Add event listeners for localStorage-based event actions
    document.querySelectorAll('[data-action]').forEach(item => {
        item.addEventListener('click', function(e) {
            e.preventDefault();
            const action = this.getAttribute('data-action');
            handleEventAction(action);
        });
    });
    <?php endif; ?>
    
    // Auto-refresh token balance every 30 seconds using existing TokenSystem
    const tokenElement = document.querySelector('[data-stat="token-balance"]');
    if (tokenElement) {
        setInterval(function() {
            // Use existing token refresh mechanism if available
            if (typeof refreshTokenBalance === 'function') {
                refreshTokenBalance();
            } else {
                // Fallback to simple page reload for token balance
                // This will be replaced by existing token refresh system
                fetch(window.location.href)
                    .then(response => response.text())
                    .then(html => {
                        const parser = new DOMParser();
                        const doc = parser.parseFromString(html, 'text/html');
                        const newTokenElement = doc.querySelector('[data-stat="token-balance"]');
                        if (newTokenElement && tokenElement) {
                            tokenElement.innerHTML = newTokenElement.innerHTML;
                            if (typeof lucide !== 'undefined') {
                                lucide.createIcons();
                            }
                        }
                    })
                    .catch(console.error);
            }
        }, 30000);
    }
    
    
    // Initialize notification badge polling
    initializeNotificationBadge();
    
    // Use existing notification system if available
    if (typeof updateNotificationBadge === 'function') {
        // Integrate with existing notification system
        const notificationBadge = document.getElementById('navbar-notification-badge');
        if (notificationBadge) {
            // Hook into existing notification updates
            document.addEventListener('notificationUpdate', function(e) {
                const count = e.detail.unreadCount || 0;
                updateNotificationBadgeDisplay(count);
            });
        }
    }
    
    function initializeNotificationBadge() {
        const currentEventId = getCurrentEventId();
        const currentUserId = getCurrentUserId();
        
        if (!currentEventId || !currentUserId) {
            console.log('Notification badge: Missing event ID or user ID');
            return;
        }
        
        let lastNotificationId = 0;
        
        function pollNotificationCount() {
            fetch(`${BASE_URL}/v2/api/get_notification_count.php?event_id=${currentEventId}&user_id=${currentUserId}&since=${lastNotificationId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        updateNotificationBadgeDisplay(data.count);
                        lastNotificationId = data.latest_id || lastNotificationId;
                    }
                })
                .catch(error => {
                    console.warn('Error polling notification count:', error);
                });
        }
        
        // Poll every 10 seconds
        pollNotificationCount();
        setInterval(pollNotificationCount, 10000);
    }
    
    function updateNotificationBadgeDisplay(count) {
        const notificationBadge = document.getElementById('navbar-notification-badge');
        if (notificationBadge) {
            if (count > 0) {
                notificationBadge.style.display = 'flex';
                notificationBadge.textContent = count > 99 ? '99+' : count;
                notificationBadge.classList.add('animate__animated', 'animate__bounce');
            } else {
                notificationBadge.style.display = 'none';
                notificationBadge.classList.remove('animate__animated', 'animate__bounce');
            }
        }
    }
    
    function getCurrentEventId() {
        // Try to get event ID from various sources
        const urlParams = new URLSearchParams(window.location.search);
        const urlEventId = urlParams.get('event_id');
        
        // Priority: URL parameter > localStorage
        if (urlEventId) {
            // Save to localStorage for future use
            localStorage.setItem('selectedEventId', urlEventId);
            localStorage.setItem('event_id', urlEventId);
            return urlEventId;
        }
        
        // Try localStorage
        const storedEventId = localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
        if (storedEventId) {
            return storedEventId;
        }
        
        // Fallback to other sources
        return document.querySelector('[data-event-id]')?.getAttribute('data-event-id') ||
               window.currentEventId ||
               null;
    }
    
    function getCurrentUserId() {
        // Get user ID from session or page data
        return window.currentUserId || 
               document.querySelector('[data-user-id]')?.getAttribute('data-user-id') ||
               '<?php echo $_SESSION['user_id'] ?? 'admin'; ?>' ||
               'admin';
    }
    
    // Update all event-related links with current event_id from localStorage
    function updateEventLinks() {
        const eventId = localStorage.getItem('selectedEventId') || localStorage.getItem('event_id');
        
        if (eventId) {
            // Update all links with .event-link class
            document.querySelectorAll('.event-link').forEach(link => {
                const baseHref = link.getAttribute('href');
                if (baseHref && baseHref !== '#') {
                    // Only update if href doesn't already have event_id
                    if (!baseHref.includes('event_id=')) {
                        const separator = baseHref.includes('?') ? '&' : '?';
                        link.setAttribute('href', `${baseHref}${separator}event_id=${eventId}`);
                    }
                }
            });
        }
    }
    
    // Update links on page load
    updateEventLinks();
    
    // Update links whenever event selection changes
    window.addEventListener('storage', function(e) {
        if (e.key === 'selectedEventId' || e.key === 'event_id') {
            updateEventLinks();
        }
    });
    
    // Add body padding to account for fixed navbar - only if navbar is visible
    const navbar = document.getElementById('<?= $navbar_id ?>');
    if (navbar && getComputedStyle(navbar).display !== 'none') {
        const navbarHeight = navbar.offsetHeight || 80;
        document.body.style.paddingTop = navbarHeight + 'px';
    }
});
</script>

<!-- Add body padding to prevent content from being hidden behind fixed navbar -->


<!-- StyleScore JavaScript includes -->
<?php include_once __DIR__ . '/../includes/scripts.php'; ?>

<script>
// Integrate with EventSelectionManager to sync cookies
document.addEventListener('DOMContentLoaded', function() {
    // Hook into EventSelectionManager if it exists
    if (window.EventSelectionManager) {
        // Override saveSelection to also set cookie
        const originalSaveSelection = window.EventSelectionManager.saveSelection;
        
        window.EventSelectionManager.saveSelection = function(type, value) {
            // Call original function
            originalSaveSelection.call(this, type, value);
            
            // Also set cookie for event type
            if (type === 'event' && value) {
                window.saveEventIdGlobal(value);
            }
        };
        
        console.log('EventSelectionManager integrated with cookie sync');
    }
});
</script>

<?php
// Include existing notification system
if (function_exists('render_notification_system')) {
    render_notification_system();
}
?>
