<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/validate_event_access.php';

$message = '';
$selected_event_id = $_GET['event_id'] ?? $_COOKIE['event_id'] ?? $_COOKIE['selectedEventId'] ?? null;
$selected_control_point_id = $_GET['control_point_id'] ?? null;

// ========== JUDGE CRUD OPERATIONS (from judge_add.php) ==========

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Delete judge
    if (isset($_POST['delete_id'])) {
        $delete_id = intval($_POST['delete_id']);
        
        // First check the user's role
        $check_stmt = $pdo->prepare("SELECT role, username FROM users WHERE id = ?");
        $check_stmt->execute([$delete_id]);
        $user = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            header("Location: judge_manager.php?msg=user_not_found");
            exit;
        }
        
        if ($user['role'] !== 'judge') {
            header("Location: judge_manager.php?msg=cannot_delete_role&username=" . urlencode($user['username']) . "&role=" . urlencode($user['role']));
            exit;
        }
        
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ? AND role = 'judge'");
        if ($stmt->execute([$delete_id])) {
            header("Location: judge_manager.php?msg=deleted");
            exit;
        } else {
            header("Location: judge_manager.php?msg=delete_error");
            exit;
        }
    }
    
    // Edit judge
    elseif (isset($_POST['edit_id'])) {
        $edit_id = intval($_POST['edit_id']);
        $edit_username = $_POST['edit_username'];
        $edit_is_head = isset($_POST['edit_is_head']) ? 1 : 0;
        $edit_password = trim($_POST['edit_password'] ?? '');
        
        // Determine role based on is_head checkbox
        $role = $edit_is_head ? 'head_judge' : 'judge';
        
        if (!empty($edit_password)) {
            // Hash the password before storing
            $hashed_password = password_hash($edit_password, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE users SET username = ?, password = ?, role = ?, judge_id = ? WHERE id = ?");
            $success = $stmt->execute([$edit_username, $hashed_password, $role, $edit_is_head, $edit_id]);
        } else {
            $stmt = $pdo->prepare("UPDATE users SET username = ?, role = ?, judge_id = ? WHERE id = ?");
            $success = $stmt->execute([$edit_username, $role, $edit_is_head, $edit_id]);
        }
        
        if ($success) {
            if (!empty($edit_password)) {
                header("Location: judge_manager.php?msg=updated_with_password&username=" . urlencode($edit_username) . "&password=" . urlencode($edit_password));
            } else {
                header("Location: judge_manager.php?msg=updated");
            }
            exit;
        } else {
            header("Location: judge_manager.php?msg=update_error");
            exit;
        }
    }
    
    // Add new judge
    elseif (isset($_POST['name'])) {
        $name = trim($_POST['name']);
        $is_head = isset($_POST['is_head']) ? 1 : 0;

        if (!empty($name)) {
            $username = strtolower(str_replace(' ', '_', $name));
            
            // Generate 6-character password for judges (letters and digits only)
            $chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
            $password = '';
            for ($i = 0; $i < 6; $i++) {
                $password .= $chars[rand(0, strlen($chars) - 1)];
            }
            
            // Check if username exists
            $check_stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
            $check_stmt->execute([$username]);
            
            if ($check_stmt->fetch()) {
                $username = $username . rand(10, 99);
            }
            
            try {
                $pdo->beginTransaction();
                
                // Determine role based on is_head checkbox
                $role = $is_head ? 'head_judge' : 'judge';
                
                // Insert into users table with hashed password and created_by tracking
                $stmt = $pdo->prepare("INSERT INTO users (username, password, role, judge_id, created_by, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                if ($stmt->execute([$username, $hashed_password, $role, $is_head, $current_user_id])) {
                    $user_id = $pdo->lastInsertId();
                    
                    // Create user profile with full name
                    try {
                        $profile_stmt = $pdo->prepare("INSERT INTO user_profiles (user_id, full_name, created_at) VALUES (?, ?, NOW())");
                        $profile_stmt->execute([$user_id, $name]);
                    } catch (Exception $e) {
                        // user_profiles table might not exist, continue without it
                    }
                    
                    // Auto-assign to selected event if one is selected
                    if ($selected_event_id) {
                        try {
                            $assign_stmt = $pdo->prepare("INSERT INTO user_assigned_events (user_id, event_id, assigned_at) VALUES (?, ?, NOW())");
                            $assign_stmt->execute([$user_id, $selected_event_id]);
                        } catch (Exception $e) {
                            // user_assigned_events table might not exist or duplicate, continue without it
                        }
                    }
                    
                    $pdo->commit();
                    
                    // Build redirect URL with event_id if available
                    $redirect_url = "judge_manager.php?msg=success&name=" . urlencode($name) . "&username=" . urlencode($username) . "&password=" . urlencode($password) . "&user_id=" . $user_id . "&is_head=" . $is_head;
                    if ($selected_event_id) {
                        $redirect_url .= "&event_id=" . $selected_event_id;
                    }
                    
                    header("Location: " . $redirect_url);
                    exit;
                }
            } catch (Exception $e) {
                $pdo->rollback();
                header("Location: judge_manager.php?msg=db_error&error=" . urlencode($e->getMessage()));
                exit;
            }
        }
    }
    
    // ========== CONTROL POINT OPERATIONS (from judge_assign.php) ==========
    
    // Create control point
    elseif (isset($_POST['create_control_point'])) {
        $event_id = $_POST['event_id'];
        $control_point_name = trim($_POST['control_point_name']);
        $description = trim($_POST['description'] ?? '');
        
        try {
            $sort_stmt = $pdo->prepare("SELECT COALESCE(MAX(sort_order), 0) + 1 FROM event_control_points WHERE event_id = ?");
            $sort_stmt->execute([$event_id]);
            $sort_order = $sort_stmt->fetchColumn();
            
            $stmt = $pdo->prepare("INSERT INTO event_control_points (event_id, control_point_name, description, sort_order, status, weight) VALUES (?, ?, ?, ?, 1, 1.0)");
            $stmt->execute([$event_id, $control_point_name, $description, $sort_order]);
            
            header("Location: judge_manager.php?event_id=" . $event_id . "&msg=control_point_created");
            exit;
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger">Error creating control point: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    // Edit control point
    elseif (isset($_POST['edit_control_point'])) {
        $event_id = $_POST['event_id'];
        $control_point_id = $_POST['control_point_id'];
        $control_point_name = trim($_POST['control_point_name']);
        $description = trim($_POST['description'] ?? '');
        $status = (int)$_POST['status'];
        $weight = (float)$_POST['weight'];
        
        try {
            if ($weight < 0 || $weight > 2) {
                throw new Exception("Weight must be between 0 and 2");
            }
            
            $verify_stmt = $pdo->prepare("SELECT id FROM event_control_points WHERE id = ? AND event_id = ?");
            $verify_stmt->execute([$control_point_id, $event_id]);
            if (!$verify_stmt->fetchColumn()) {
                throw new Exception("Invalid control point or access denied");
            }
            
            $stmt = $pdo->prepare("UPDATE event_control_points SET control_point_name = ?, description = ?, status = ?, weight = ? WHERE id = ? AND event_id = ?");
            $stmt->execute([$control_point_name, $description, $status, $weight, $control_point_id, $event_id]);
            
            header("Location: judge_manager.php?event_id=" . $event_id . "&msg=control_point_updated");
            exit;
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger">Error updating control point: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    // Delete control point
    elseif (isset($_POST['delete_control_point'])) {
        $event_id = $_POST['event_id'];
        $control_point_id = $_POST['control_point_id'];
        
        try {
            $verify_stmt = $pdo->prepare("SELECT id FROM event_control_points WHERE id = ? AND event_id = ?");
            $verify_stmt->execute([$control_point_id, $event_id]);
            if (!$verify_stmt->fetchColumn()) {
                throw new Exception("Invalid control point or access denied");
            }
            
            $judge_check = $pdo->prepare("SELECT COUNT(*) FROM judge_assignments WHERE control_point_id = ?");
            $judge_check->execute([$control_point_id]);
            $judge_count = $judge_check->fetchColumn();
            
            if ($judge_count > 0) {
                throw new Exception("Cannot delete control point with assigned judges. Please unassign judges first.");
            }
            
            $stmt = $pdo->prepare("DELETE FROM event_control_points WHERE id = ? AND event_id = ?");
            $stmt->execute([$control_point_id, $event_id]);
            
            header("Location: judge_manager.php?event_id=" . $event_id . "&msg=control_point_deleted");
            exit;
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger">Error deleting control point: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    // ========== JUDGE ASSIGNMENT OPERATIONS ==========
    
    // Assign judges to event/control point
    elseif (isset($_POST['judge_ids']) && isset($_POST['event_id'])) {
        $event_id = $_POST['event_id'];
        $control_point_id = $_POST['control_point_id'] ?? null;
        $judge_ids = $_POST['judge_ids'];

        if ($control_point_id === '' || $control_point_id === '0') {
            $control_point_id = null;
        }

        try {
            if ($control_point_id !== null) {
                $cp_check = $pdo->prepare("SELECT id FROM event_control_points WHERE id = ? AND event_id = ?");
                $cp_check->execute([$control_point_id, $event_id]);
                if (!$cp_check->fetchColumn()) {
                    throw new Exception("Invalid control point selected.");
                }
            }

            $conflicts = [];
            $other_event_conflicts = [];
            
            if (!empty($judge_ids)) {
                $in = implode(',', array_fill(0, count($judge_ids), '?'));
                $params = array_merge([$event_id], $judge_ids);
                
                // Check for conflicts within the same event
                if ($control_point_id !== null) {
                    $conflict_sql = "
                        SELECT ja.judge_id, u.username, 
                               CASE 
                                   WHEN ja.control_point_id IS NULL THEN 'General Judging'
                                   ELSE ecp.control_point_name
                               END as control_point_name
                        FROM judge_assignments ja
                        JOIN users u ON ja.judge_id = u.id
                        LEFT JOIN event_control_points ecp ON ja.control_point_id = ecp.id
                        WHERE ja.event_id = ? AND ja.judge_id IN ($in) AND (ja.control_point_id != ? OR ja.control_point_id IS NULL)
                    ";
                    $params[] = $control_point_id;
                } else {
                    $conflict_sql = "
                        SELECT ja.judge_id, u.username, 
                               CASE 
                                   WHEN ja.control_point_id IS NULL THEN 'General Judging'
                                   ELSE ecp.control_point_name
                               END as control_point_name
                        FROM judge_assignments ja
                        JOIN users u ON ja.judge_id = u.id
                        LEFT JOIN event_control_points ecp ON ja.control_point_id = ecp.id
                        WHERE ja.event_id = ? AND ja.judge_id IN ($in) AND ja.control_point_id IS NOT NULL
                    ";
                }
                
                $conflict_stmt = $pdo->prepare($conflict_sql);
                $conflict_stmt->execute($params);
                $conflicts = $conflict_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Check for assignments to other events
                $other_event_sql = "
                    SELECT ja.judge_id, u.username, e.name as event_name, e.id as other_event_id,
                           CASE 
                               WHEN ja.control_point_id IS NULL THEN 'General Judging'
                               ELSE ecp.control_point_name
                           END as control_point_name
                    FROM judge_assignments ja
                    JOIN users u ON ja.judge_id = u.id
                    JOIN events e ON ja.event_id = e.id
                    LEFT JOIN event_control_points ecp ON ja.control_point_id = ecp.id
                    WHERE ja.event_id != ? AND ja.judge_id IN ($in)
                ";
                
                $other_event_stmt = $pdo->prepare($other_event_sql);
                $other_event_stmt->execute($params);
                $other_event_conflicts = $other_event_stmt->fetchAll(PDO::FETCH_ASSOC);
            }

            if (!empty($conflicts) && !isset($_POST['force_assign'])) {
                $conflict_list = [];
                foreach ($conflicts as $conflict) {
                    $conflict_list[] = $conflict['username'] . ' (assigned to ' . $conflict['control_point_name'] . ')';
                }
                
                $message = '<div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Warning: Some judges are already assigned:</strong>
                    <ul class="mb-2">' . implode('', array_map(fn($item) => '<li>' . $item . '</li>', $conflict_list)) . '</ul>
                    <form method="post" class="d-inline">
                        ' . implode('', array_map(fn($id) => '<input type="hidden" name="judge_ids[]" value="' . $id . '">', $judge_ids)) . '
                        <input type="hidden" name="event_id" value="' . $event_id . '">
                        <input type="hidden" name="control_point_id" value="' . ($control_point_id ?? '') . '">
                        <input type="hidden" name="force_assign" value="1">
                        <button type="submit" class="btn btn-warning btn-sm me-2">
                            <i class="fas fa-check-circle me-1"></i>Assign Anyway
                        </button>
                    </form>
                    <button type="button" class="btn btn-secondary btn-sm" onclick="this.closest(\'.alert\').remove()">Cancel</button>
                </div>';
            } elseif (!empty($other_event_conflicts) && !isset($_POST['unassign_from_other_events'])) {
                $conflict_list = [];
                foreach ($other_event_conflicts as $conflict) {
                    $conflict_list[] = '<strong>' . $conflict['username'] . '</strong> is assigned to <strong>' . 
                                       $conflict['event_name'] . '</strong> (' . $conflict['control_point_name'] . ')';
                }
                
                $message = '<div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Warning: Some judges are assigned to other events:</strong>
                    <ul class="mb-3">' . implode('', array_map(fn($item) => '<li>' . $item . '</li>', $conflict_list)) . '</ul>
                    <p class="mb-2">Do you want to unassign them from other events and assign to this event?</p>
                    <form method="post" class="d-inline">
                        ' . implode('', array_map(fn($id) => '<input type="hidden" name="judge_ids[]" value="' . $id . '">', $judge_ids)) . '
                        <input type="hidden" name="event_id" value="' . $event_id . '">
                        <input type="hidden" name="control_point_id" value="' . ($control_point_id ?? '') . '">
                        <input type="hidden" name="unassign_from_other_events" value="1">
                        <button type="submit" class="btn btn-warning btn-sm me-2">
                            <i class="fas fa-exchange-alt me-1"></i>Unassign & Assign Here
                        </button>
                    </form>
                    <button type="button" class="btn btn-secondary btn-sm" onclick="this.closest(\'.alert\').remove()">Cancel</button>
                </div>';
            } else {
                // Unassign from other events if requested
                if (isset($_POST['unassign_from_other_events']) && !empty($judge_ids)) {
                    $in = implode(',', array_fill(0, count($judge_ids), '?'));
                    $unassign_params = array_merge([$event_id], $judge_ids);
                    $unassign_stmt = $pdo->prepare("DELETE FROM judge_assignments WHERE event_id != ? AND judge_id IN ($in)");
                    $unassign_stmt->execute($unassign_params);
                }
                
                if ($control_point_id !== null) {
                    $pdo->prepare("DELETE FROM judge_assignments WHERE event_id = ? AND control_point_id = ?")->execute([$event_id, $control_point_id]);
                } else {
                    $pdo->prepare("DELETE FROM judge_assignments WHERE event_id = ? AND control_point_id IS NULL")->execute([$event_id]);
                }

                if (!empty($judge_ids)) {
                    $insert = $pdo->prepare("INSERT INTO judge_assignments (event_id, judge_id, control_point_id) VALUES (?, ?, ?)");
                    foreach ($judge_ids as $jid) {
                        $insert->execute([$event_id, $jid, $control_point_id]);
                    }
                }

                $redirect_url = "judge_manager.php?event_id=" . $event_id;
                if ($control_point_id !== null) {
                    $redirect_url .= "&control_point_id=" . $control_point_id;
                }
                $redirect_url .= "&msg=assigned_success";
                
                header("Location: " . $redirect_url);
                exit;
            }
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger">Error assigning judges: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    // Unassign judge
    elseif (isset($_POST['unassign_judge_id'])) {
        $event_id = $_POST['event_id'];
        $judge_id = $_POST['unassign_judge_id'];
        $control_point_id = $_POST['control_point_id'] ?? null;
        
        if ($control_point_id === '' || $control_point_id === '0') {
            $control_point_id = null;
        }
        
        try {
            if ($control_point_id !== null) {
                $stmt = $pdo->prepare("DELETE FROM judge_assignments WHERE event_id = ? AND judge_id = ? AND control_point_id = ?");
                $stmt->execute([$event_id, $judge_id, $control_point_id]);
            } else {
                $stmt = $pdo->prepare("DELETE FROM judge_assignments WHERE event_id = ? AND judge_id = ? AND control_point_id IS NULL");
                $stmt->execute([$event_id, $judge_id]);
            }
            
            $redirect_url = "judge_manager.php?event_id=" . $event_id;
            if ($control_point_id !== null) {
                $redirect_url .= "&control_point_id=" . $control_point_id;
            }
            $redirect_url .= "&msg=unassigned";
            
            header("Location: " . $redirect_url);
            exit;
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger">Error unassigning judge: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// ========== HANDLE SUCCESS MESSAGES ==========

if (isset($_GET['msg'])) {
    switch ($_GET['msg']) {
        case 'success':
            $name = $_GET['name'] ?? '';
            $username = $_GET['username'] ?? '';
            $password = $_GET['password'] ?? '';
            $user_id = $_GET['user_id'] ?? '';
            $is_head = $_GET['is_head'] ?? 0;
            
            $message = "<div class='alert alert-success alert-dismissible fade show'>
                <h6><i class='fas fa-check-circle me-2'></i>Judge Added Successfully!</h6>
                <hr>
                <strong>Name:</strong> " . htmlspecialchars($name) . "<br>
                <strong>Username:</strong> <code>" . htmlspecialchars($username) . "</code><br>
                <strong>Password:</strong> <code>" . htmlspecialchars($password) . "</code><br>
                <strong>User ID:</strong> <span class='badge bg-primary'>" . htmlspecialchars($user_id) . "</span><br>
                <strong>Head Judge:</strong> " . ($is_head ? '<span class="badge bg-warning">Yes</span>' : '<span class="badge bg-secondary">No</span>') . "
                <button type='button' class='btn-close' data-bs-dismiss='alert'></button>
            </div>";
            break;
        case 'updated':
            $message = '<div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i>Judge updated successfully.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'updated_with_password':
            $username = $_GET['username'] ?? '';
            $password = $_GET['password'] ?? '';
            $message = "<div class='alert alert-success alert-dismissible fade show'>
                <h6><i class='fas fa-check-circle me-2'></i>Judge Updated!</h6>
                <hr>
                <strong>Username:</strong> <code>" . htmlspecialchars($username) . "</code><br>
                <strong>New Password:</strong> <code>" . htmlspecialchars($password) . "</code>
                <button type='button' class='btn-close' data-bs-dismiss='alert'></button>
            </div>";
            break;
        case 'deleted':
            $message = '<div class="alert alert-warning alert-dismissible fade show">
                <i class="fas fa-trash me-2"></i>Judge deleted successfully.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'cannot_delete_role':
            $username = $_GET['username'] ?? '';
            $role = $_GET['role'] ?? '';
            $message = '<div class="alert alert-danger alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Cannot delete user:</strong> ' . htmlspecialchars($username) . ' has role "' . htmlspecialchars($role) . '". Only users with role "judge" can be deleted from this interface.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'user_not_found':
            $message = '<div class="alert alert-danger alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>User not found.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'assigned_success':
            $message = '<div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i>Judges assigned successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'unassigned':
            $message = '<div class="alert alert-warning alert-dismissible fade show">
                <i class="fas fa-user-minus me-2"></i>Judge unassigned successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'control_point_created':
            $message = '<div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-map-marker-alt me-2"></i>Control point created successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'control_point_updated':
            $message = '<div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-edit me-2"></i>Control point updated successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'control_point_deleted':
            $message = '<div class="alert alert-warning alert-dismissible fade show">
                <i class="fas fa-trash me-2"></i>Control point deleted successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
    }
}

// ========== LOAD DATA ==========

// Get current user's role for filtering
$current_user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';
$current_user_id = $_SESSION['user_id'] ?? 0;

// Define role hierarchy
$role_hierarchy = [
    'super_admin' => ['admin', 'event_organizer', 'head_judge', 'judge', 'office', 'media', 'participant'],
    'admin' => ['event_organizer', 'head_judge', 'judge', 'office', 'media', 'participant'],
    'event_organizer' => ['head_judge', 'judge', 'office', 'media', 'participant'],
    'office' => ['participant'],
    'head_judge' => ['judge'],
    'judge' => [],
    'media' => [],
    'participant' => []
];

$allowed_roles = $role_hierarchy[$current_user_role] ?? [];

// Load all judges with filtering based on:
// 1. Admin/super_admin: See all judges assigned to any event + all judges they can manage (roles below them)
// 2. Event_organizer/office: See judges assigned to events they own + judges they created (roles below them)
try {
    $where_conditions = [];
    $params = [];
    
    // Role-based filtering
    if (in_array($current_user_role, ['admin', 'super_admin'])) {
        // Admin and super_admin see all users with roles below them + super_admins (for super_admin only)
        if (!empty($allowed_roles)) {
            // For super_admin, include super_admin role in the list
            if ($current_user_role === 'super_admin') {
                $roles_to_show = array_merge(['super_admin'], $allowed_roles);
            } else {
                $roles_to_show = $allowed_roles;
            }
            $role_placeholders = implode(',', array_fill(0, count($roles_to_show), '?'));
            $where_conditions[] = "u.role IN ($role_placeholders)";
            $params = array_merge($params, $roles_to_show);
        } else {
            // Super admin sees everyone
            $where_conditions[] = "1=1";
        }
        
        // If event selected, show all judges assigned to that event (via user_assigned_events OR judge_assignments)
        if ($selected_event_id) {
            $where_conditions[] = "(
                EXISTS (SELECT 1 FROM user_assigned_events uae WHERE uae.user_id = u.id AND uae.event_id = ?) 
                OR EXISTS (SELECT 1 FROM judge_assignments ja WHERE ja.judge_id = u.id AND ja.event_id = ?) 
                OR u.created_by = ?
            )";
            $params[] = $selected_event_id;
            $params[] = $selected_event_id;
            $params[] = $current_user_id;
        }
    } else {
        // Event organizer, office, etc: See only judges they created OR assigned to events they own
        if (!empty($allowed_roles)) {
            $role_placeholders = implode(',', array_fill(0, count($allowed_roles), '?'));
            $where_conditions[] = "u.role IN ($role_placeholders)";
            $params = array_merge($params, $allowed_roles);
        } else {
            // No roles below them - can't see any judges
            $where_conditions[] = "0=1";
        }
        
        if ($selected_event_id) {
            // Show judges assigned to this event (via user_assigned_events) OR created by current user
            // For event_organizer, also check if they're assigned to manage this event
            $where_conditions[] = "(
                (EXISTS (
                    SELECT 1 FROM user_assigned_events uae 
                    WHERE uae.user_id = u.id 
                    AND uae.event_id = ?
                    AND EXISTS (
                        SELECT 1 FROM user_assigned_events manager_check
                        WHERE manager_check.event_id = uae.event_id 
                        AND manager_check.user_id = ?
                    )
                )) 
                OR (EXISTS (
                    SELECT 1 FROM judge_assignments ja 
                    WHERE ja.judge_id = u.id 
                    AND ja.event_id = ? 
                    AND EXISTS (
                        SELECT 1 FROM user_assigned_events uae 
                        WHERE uae.event_id = ja.event_id 
                        AND uae.user_id = ?
                    )
                )) 
                OR u.created_by = ?
            )";
            $params[] = $selected_event_id;
            $params[] = $current_user_id;
            $params[] = $selected_event_id;
            $params[] = $current_user_id;
            $params[] = $current_user_id;
        } else {
            // No event selected - show only judges they created
            $where_conditions[] = "u.created_by = ?";
            $params[] = $current_user_id;
        }
    }
    
    $where_clause = implode(" AND ", $where_conditions);
    
    $judges_query = "
        SELECT u.id, u.username, u.password, u.judge_id, u.role, u.created_at, u.created_by,
               creator.username as created_by_name,
               COUNT(DISTINCT ja.event_id) as event_count
        FROM users u
        LEFT JOIN users creator ON u.created_by = creator.id
        LEFT JOIN judge_assignments ja ON u.id = ja.judge_id
        WHERE $where_clause
        GROUP BY u.id, u.username, u.password, u.judge_id, u.role, u.created_at, u.created_by, creator.username
        ORDER BY u.id DESC
    ";
    
    $stmt = $pdo->prepare($judges_query);
    $stmt->execute($params);
    $judges = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $judges = [];
    if (!$message) {
        $message = '<div class="alert alert-danger">Error fetching judges: ' . htmlspecialchars($e->getMessage()) . '</div>';
    }
}

// Load events
$events = $pdo->query("SELECT * FROM events ORDER BY date DESC")->fetchAll(PDO::FETCH_ASSOC);

// Load control points for selected event
$control_points = [];
if ($selected_event_id) {
    $stmt = $pdo->prepare("SELECT id, control_point_name, description, sort_order, status, weight FROM event_control_points WHERE event_id = ? ORDER BY sort_order, control_point_name");
    $stmt->execute([$selected_event_id]);
    $control_points = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Load assigned judges for selected event/control point
$assigned_judges = [];
if ($selected_event_id) {
    if ($selected_control_point_id) {
        $stmt = $pdo->prepare("SELECT judge_id FROM judge_assignments WHERE event_id = ? AND control_point_id = ?");
        $stmt->execute([$selected_event_id, $selected_control_point_id]);
    } else {
        $stmt = $pdo->prepare("SELECT judge_id FROM judge_assignments WHERE event_id = ? AND control_point_id IS NULL");
        $stmt->execute([$selected_event_id]);
    }
    $assigned_judges = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'judge_id');
}

// Load all assignments for overview
$all_assignments = [];
$judge_assignments_map = [];
if ($selected_event_id) {
    $stmt = $pdo->prepare("
        SELECT ja.judge_id, ja.control_point_id, u.username, u.judge_id as is_head,
               CASE 
                   WHEN ja.control_point_id IS NULL THEN 'General Judging'
                   ELSE ecp.control_point_name
               END as control_point_name,
               COALESCE(ecp.sort_order, 0) as sort_order
        FROM judge_assignments ja
        JOIN users u ON ja.judge_id = u.id
        LEFT JOIN event_control_points ecp ON ja.control_point_id = ecp.id
        WHERE ja.event_id = ?
        ORDER BY sort_order, control_point_name, u.username
    ");
    $stmt->execute([$selected_event_id]);
    $all_assignments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($all_assignments as $assignment) {
        if (!isset($judge_assignments_map[$assignment['judge_id']])) {
            $judge_assignments_map[$assignment['judge_id']] = [];
        }
        $judge_assignments_map[$assignment['judge_id']][] = $assignment;
    }
}

// Get current control point name
$current_control_point_name = 'General Judging';
if ($selected_control_point_id) {
    $cp_stmt = $pdo->prepare("SELECT control_point_name FROM event_control_points WHERE id = ?");
    $cp_stmt->execute([$selected_control_point_id]);
    $current_control_point_name = $cp_stmt->fetchColumn() ?: 'Unknown Control Point';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Judge Manager - StyleScore</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrcodejs/1.0.0/qrcode.min.js"></script>
    <style>
        .sticky-top { position: sticky; top: 0; z-index: 100; }
        .scrollable-pane { max-height: calc(100vh - 200px); overflow-y: auto; }
        .judge-card { transition: all 0.2s ease; border-left: 3px solid transparent; }
        .judge-card:hover { transform: translateY(-2px); box-shadow: 0 4px 12px rgba(0,0,0,0.15); }
        .cp-card { cursor: pointer; transition: all 0.2s ease; border-left: 4px solid transparent; }
        .cp-card:hover { box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .cp-card.selected { border-left-color: #0d6efd; background-color: rgba(13, 110, 253, 0.05); }
        .cp-card.selected-general { border-left-color: #198754; background-color: rgba(25, 135, 84, 0.05); }
    </style>
</head>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>

<div class="container-fluid container-StyleScore p-4">
    <!-- Event Actions Menu -->
    <div class="row mb-3">
        <div class="col-12">
            <?php 
            $use_localStorage = true;
            $show_header = false;
            include 'event_actions_menu.php'; 
            ?>
        </div>
    </div>

    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-3">
        <div>
            <h1 class="h4 mb-1">
                <i class="fas fa-users text-primary me-2"></i>Judge Manager
            </h1>
            <p class="text-muted mb-0 small">Manage judges, control points, and event assignments</p>
        </div>
        <button type="button" class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#helpModal">
            <i class="fas fa-question-circle me-1"></i>Help
        </button>
    </div>

    <!-- Messages -->
    <?php if ($message): ?>
        <?php echo $message; ?>
    <?php endif; ?>

    <!-- Metrics Cards -->
    <div class="row g-2 mb-3 d-none">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body py-2">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <div class="h5 mb-0"><?= count($judges) ?></div>
                            <small>Total Judges</small>
                        </div>
                        <i class="fas fa-users fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-warning text-dark">
                <div class="card-body py-2">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <div class="h5 mb-0"><?= count(array_filter($judges, function($j) { return $j['judge_id'] == 1; })) ?></div>
                            <small>Head Judges</small>
                        </div>
                        <i class="fas fa-star fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body py-2">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <div class="h5 mb-0"><?= $selected_event_id ? count($control_points) : 0 ?></div>
                            <small>Control Points</small>
                        </div>
                        <i class="fas fa-map-marker-alt fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-info text-white">
                <div class="card-body py-2">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <div class="h5 mb-0"><?= count($all_assignments) ?></div>
                            <small>Assignments</small>
                        </div>
                        <i class="fas fa-clipboard-check fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Two-Pane Layout -->
    <div class="row g-3">
        
        <!-- LEFT PANE: Judge Management -->
        <div class="col-lg-5 col-xxl-4">
            
            <!-- Add Judge Card -->
            <div class="card shadow-sm mb-3">
                <div class="card-header bg-white sticky-top">
                    <h6 class="mb-0"><i class="fas fa-user-plus text-success me-2"></i>Add New Judge</h6>
                </div>
                <div class="card-body p-3">
                    <form method="post" class="needs-validation" novalidate>
                        <div class="mb-3">
                            <label class="form-label small fw-medium">Judge Name</label>
                            <input type="text" class="form-control form-control-sm" name="name" required 
                                   placeholder="Enter judge's full name">
                        </div>
                        
                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" name="is_head" id="is_head">
                            <label class="form-check-label" for="is_head">
                                <i class="fas fa-star text-warning me-1"></i>Head Judge
                            </label>
                        </div>
                        
                        <div class="d-grid">
                            <button type="submit" class="btn btn-success btn-sm">
                                <i class="fas fa-plus-circle me-1"></i>Add Judge
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <!-- Judges List -->
            <div class="card shadow-sm">
                <div class="card-header bg-white sticky-top">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6 class="mb-0"><i class="fas fa-list text-primary me-2"></i>All Judges</h6>
                        <span class="badge bg-secondary"><?= count($judges) ?></span>
                    </div>
                </div>
                <div class="card-body p-0 scrollable-pane">
                    <?php if (empty($judges)): ?>
                        <div class="text-center py-5 text-muted">
                            <i class="fas fa-users fa-3x opacity-25"></i>
                            <p class="mt-3 mb-0 small">No judges yet</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($judges as $j): ?>
                            <div class="judge-card border-bottom p-3">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <h6 class="mb-1">
                                            <i class="fas fa-user-circle text-muted me-1"></i>
                                            <?= htmlspecialchars($j['username']) ?>
                                            <?php if ($j['judge_id'] == 1): ?>
                                                <span class="badge bg-warning text-dark ms-1">
                                                    <i class="fas fa-star"></i>
                                                </span>
                                            <?php endif; ?>
                                            <?php 
                                            // Show role badge
                                            $role_badges = [
                                                'head_judge' => '<span class="badge bg-danger ms-1">Head Judge</span>',
                                                'admin' => '<span class="badge bg-primary ms-1">Admin</span>',
                                                'super_admin' => '<span class="badge bg-dark ms-1">Super Admin</span>',
                                                'judge' => '<span class="badge bg-secondary ms-1">Judge</span>'
                                            ];
                                            echo $role_badges[$j['role']] ?? '';
                                            ?>
                                        </h6>
                                        
                                        <?php if ($current_user_role === 'super_admin'): ?>
                                            <!-- Show login credentials for super_admin -->
                                            <div class="alert alert-light border py-1 px-2 mb-2 small">
                                                <i class="fas fa-key text-primary me-1"></i>
                                                <strong>Login as:</strong> 
                                                <code class="text-primary"><?= htmlspecialchars($j['username']) ?></code>
                                                <?php if (strpos($j['password'], '$2y$') === 0): ?>
                                                    / <code class="text-muted">••••••••</code>
                                                    <small class="text-warning ms-1">(encrypted - reset to view)</small>
                                                <?php else: ?>
                                                    / <code class="text-danger"><?= htmlspecialchars($j['password']) ?></code>
                                                    <button class="btn btn-sm btn-outline-secondary py-0 px-1 ms-2" 
                                                            onclick='copyToClipboard(<?= json_encode($j["username"]) ?>, <?= json_encode($j["password"]) ?>)'
                                                            title="Copy credentials">
                                                        <i class="fas fa-copy"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <small class="text-muted">
                                            <i class="fas fa-calendar me-1"></i><?= date('M d, Y', strtotime($j['created_at'])) ?>
                                            <?php if ($j['event_count'] > 0): ?>
                                                <span class="badge bg-success ms-2"><?= $j['event_count'] ?> events</span>
                                            <?php endif; ?>
                                            <?php if (!empty($j['created_by_name'])): ?>
                                                <span class="text-muted ms-2">• Created by @<?= htmlspecialchars($j['created_by_name']) ?></span>
                                            <?php endif; ?>
                                        </small>
                                        
                                        <?php if ($current_user_role === 'super_admin' && isset($judge_assignments_map[$j['id']]) && !empty($judge_assignments_map[$j['id']])): ?>
                                            <!-- Show current assignments for super_admin -->
                                            <div class="mt-2">
                                                <small class="text-muted d-block mb-1"><i class="fas fa-clipboard-list me-1"></i>Assigned to:</small>
                                                <?php foreach ($judge_assignments_map[$j['id']] as $assignment): ?>
                                                    <span class="badge bg-info me-1 mb-1">
                                                        <i class="fas fa-map-marker-alt me-1"></i><?= htmlspecialchars($assignment['control_point_name']) ?>
                                                    </span>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($current_user_role === 'super_admin' && $selected_event_id): ?>
                                            <!-- Quick assign button for super_admin -->
                                            <div class="mt-2">
                                                <button class="btn btn-sm btn-outline-success" 
                                                        onclick='quickAssignJudge(<?= $j["id"] ?>, <?= json_encode($j["username"]) ?>)'>
                                                    <i class="fas fa-user-plus me-1"></i>Quick Assign to Current Event
                                                </button>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ($j['role'] === 'judge' || $j['role'] === 'head_judge'): ?>
                                    <div class="btn-group btn-group-sm judge-actions" role="group" aria-label="Judge Actions">
                                        <button class="btn btn-outline-info btn-sm" 
                                                onclick='viewJudgeInfo(<?= $j["id"] ?>, <?= json_encode($j["username"]) ?>, <?= json_encode($j["password"]) ?>, <?= $j["judge_id"] ?? 0 ?>, <?= json_encode($j["created_at"]) ?>)'
                                                title="View Info">
                                            <i class="fas fa-info-circle"></i>
                                        </button>
                                        <button class="btn btn-outline-primary btn-sm" 
                                                onclick='editJudge(<?= $j["id"] ?>, <?= json_encode($j["username"]) ?>, <?= $j["judge_id"] ?? 0 ?>, <?= json_encode($j["password"]) ?>)'
                                                title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-outline-danger btn-sm" 
                                                onclick='confirmDeleteJudge(<?= $j["id"] ?>, <?= json_encode($j["username"]) ?>, <?= json_encode($j["role"]) ?>)'
                                                title="Delete">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- RIGHT PANE: Event Assignments -->
        <div class="col-lg-7 col-xxl-8">
            
            <!-- Control Points & Assignment -->
            <?php if ($selected_event_id): ?>
                
                <!-- Control Points Selection -->
                <div class="card shadow-sm mb-3">
                    <div class="card-header bg-white sticky-top d-flex justify-content-between align-items-center">
                        <h6 class="mb-0"><i class="fas fa-map-marker-alt text-info me-2"></i>Control Points</h6>
                        <button class="btn btn-info btn-sm" data-bs-toggle="modal" data-bs-target="#createControlPointModal">
                            <i class="fas fa-plus me-1"></i>Add
                        </button>
                    </div>
                    <div class="card-body p-2 scrollable-pane">
                        <!-- General Judging -->
                        <div class="cp-card p-3 mb-2 border rounded <?= !$selected_control_point_id ? 'selected-general' : '' ?>" 
                             onclick="selectControlPoint(null)">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="mb-1">
                                        <i class="fas fa-star text-warning me-1"></i>General Judging
                                        <?= !$selected_control_point_id ? '<i class="fas fa-check-circle text-success ms-2"></i>' : '' ?>
                                    </h6>
                                    <small class="text-muted">Overall event judging</small>
                                </div>
                                <?php if (!$selected_control_point_id): ?>
                                    <span class="badge bg-success">SELECTED</span>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Specific Control Points -->
                        <?php foreach ($control_points as $cp): ?>
                            <div class="cp-card p-3 mb-2 border rounded <?= $selected_control_point_id == $cp['id'] ? 'selected' : '' ?>" 
                                 onclick="selectControlPoint(<?= $cp['id'] ?>)">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <h6 class="mb-1">
                                            <i class="fas fa-map-marker-alt text-primary me-1"></i>
                                            <?= htmlspecialchars($cp['control_point_name']) ?>
                                            <?= $selected_control_point_id == $cp['id'] ? '<i class="fas fa-check-circle text-primary ms-2"></i>' : '' ?>
                                        </h6>
                                        <?php if ($cp['description']): ?>
                                            <small class="text-muted d-block"><?= htmlspecialchars($cp['description']) ?></small>
                                        <?php endif; ?>
                                        <small>
                                            <span class="badge bg-secondary me-1">Order: <?= $cp['sort_order'] ?></span>
                                            <span class="badge bg-info">Weight: <?= number_format($cp['weight'], 1) ?></span>
                                        </small>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-outline-primary" 
                                            onclick="event.stopPropagation(); editControlPoint(<?= $cp['id'] ?>, '<?= htmlspecialchars($cp['control_point_name'], ENT_QUOTES) ?>', '<?= htmlspecialchars($cp['description'], ENT_QUOTES) ?>', <?= $cp['status'] ?>, <?= $cp['weight'] ?>)">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <!-- Judge Assignment -->
                <div class="card shadow-sm">
                    <div class="card-header bg-white sticky-top">
                        <h6 class="mb-0"><i class="fas fa-user-check text-success me-2"></i>Assign Judges</h6>
                        <small class="text-muted">
                            Event: <strong><?= htmlspecialchars($events[array_search($selected_event_id, array_column($events, 'id'))]['name']) ?></strong>
                            | Control Point: <span class="badge bg-primary"><?= htmlspecialchars($current_control_point_name) ?></span>
                        </small>
                    </div>
                    <div class="card-body p-3">
                        <form method="post">
                            <input type="hidden" name="event_id" value="<?= $selected_event_id ?>">
                            <input type="hidden" name="control_point_id" value="<?= $selected_control_point_id ?>">
                            
                            <?php if (empty($judges)): ?>
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle me-2"></i>
                                    No judges available. Add judges first.
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-sm table-hover">
                                        <thead class="table-light">
                                            <tr>
                                                <th width="40">
                                                    <input type="checkbox" class="form-check-input" id="selectAll" 
                                                           onchange="toggleAll(this)">
                                                </th>
                                                <th>Judge</th>
                                                <th width="100">Role</th>
                                                <th width="150">Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($judges as $j): ?>
                                                <?php
                                                $is_assigned_here = in_array($j['id'], $assigned_judges);
                                                $other_assignments = [];
                                                if (isset($judge_assignments_map[$j['id']])) {
                                                    foreach ($judge_assignments_map[$j['id']] as $assignment) {
                                                        if (($selected_control_point_id && $assignment['control_point_id'] == $selected_control_point_id) || 
                                                            (!$selected_control_point_id && $assignment['control_point_id'] === null)) {
                                                            continue;
                                                        }
                                                        $other_assignments[] = $assignment['control_point_name'];
                                                    }
                                                }
                                                ?>
                                                <tr class="<?= $is_assigned_here ? 'table-success' : '' ?>">
                                                    <td>
                                                        <input type="checkbox" class="form-check-input" 
                                                               name="judge_ids[]" value="<?= $j['id'] ?>" 
                                                               <?= $is_assigned_here ? 'checked' : '' ?>>
                                                    </td>
                                                    <td><?= htmlspecialchars($j['username']) ?></td>
                                                    <td>
                                                        <?php 
                                                        // Display role badge
                                                        $role_display = [
                                                            'head_judge' => '<span class="badge bg-danger"><i class="fas fa-star"></i> Head Judge</span>',
                                                            'admin' => '<span class="badge bg-primary">Admin</span>',
                                                            'super_admin' => '<span class="badge bg-dark">Super Admin</span>',
                                                            'judge' => '<span class="badge bg-secondary">Judge</span>'
                                                        ];
                                                        echo $role_display[$j['role']] ?? '<span class="badge bg-secondary">Judge</span>';
                                                        
                                                        // Also show head judge indicator if judge_id == 1
                                                        if ($j['judge_id'] == 1 && $j['role'] != 'head_judge'): ?>
                                                            <span class="badge bg-warning ms-1"><i class="fas fa-star"></i></span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <?php if ($is_assigned_here): ?>
                                                            <span class="badge bg-success"><i class="fas fa-check"></i> Assigned</span>
                                                        <?php elseif (!empty($other_assignments)): ?>
                                                            <small class="text-muted"><?= implode(', ', $other_assignments) ?></small>
                                                        <?php else: ?>
                                                            <span class="text-muted">Available</span>
                                                        <?php endif; ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                                
                                <div class="d-grid gap-2 mt-3">
                                    <button type="submit" class="btn btn-success">
                                        <i class="fas fa-save me-2"></i>Save Judge Assignments
                                    </button>
                                </div>
                            <?php endif; ?>
                        </form>
                    </div>
                </div>
                
            <?php else: ?>
                <div class="card shadow-sm">
                    <div class="card-body text-center py-5">
                        <i class="fas fa-calendar-alt fa-3x text-muted opacity-25 mb-3"></i>
                        <h5 class="text-muted">No Event Selected</h5>
                        <p class="text-muted">Select an event to manage control points and assign judges</p>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- View Judge Info Modal -->
<div class="modal fade" id="viewJudgeModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-info text-white">
                <h6 class="modal-title"><i class="fas fa-info-circle me-2"></i>Judge Information</h6>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="row g-3">
                    <div class="col-md-7">
                        <div id="judgeInfoContent"></div>
                    </div>
                    <div class="col-md-5 border-start">
                        <h6 class="mb-2"><i class="fas fa-qrcode me-2"></i>QR Login</h6>
                        <p class="small text-muted mb-2">Scan this code on the judge device to login directly.</p>
                        <div id="judgeQrCode" class="d-flex justify-content-center flex-column align-items-center py-2"></div>
                        <div id="qrTokenControls" class="mt-2" style="display:none;">
                            <div class="row g-2 mb-2">
                                <div class="col-8">
                                    <select id="tokenExpirySelect" class="form-select form-select-sm">
                                        <option value="0.5">30 minutes</option>
                                        <option value="1">1 hour</option>
                                        <option value="2">2 hours</option>
                                        <option value="5" selected>5 hours</option>
                                        <option value="8">8 hours</option>
                                        <option value="12">12 hours</option>
                                        <option value="24">24 hours</option>
                                    </select>
                                </div>
                                <div class="col-4">
                                    <button class="btn btn-primary btn-sm w-100" onclick="regenerateToken()">
                                        <i class="fas fa-sync me-1"></i>New
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Edit Judge Modal -->
<div class="modal fade" id="editJudgeModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h6 class="modal-title"><i class="fas fa-edit me-2"></i>Edit Judge</h6>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="post">
                <div class="modal-body">
                    <input type="hidden" name="edit_id" id="edit_id">
                    
                    <div class="mb-3">
                        <label class="form-label">Username</label>
                        <input type="text" class="form-control" name="edit_username" id="edit_username" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Password <small class="text-muted">(leave empty to keep current)</small></label>
                        <input type="password" class="form-control" name="edit_password" id="edit_password">
                        <small class="text-muted">Current: <code id="currentPassword">••••••</code></small>
                    </div>
                    
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="edit_is_head" id="edit_is_head">
                        <label class="form-check-label" for="edit_is_head">
                            <i class="fas fa-star text-warning me-1"></i>Head Judge
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary btn-sm">
                        <i class="fas fa-check me-1"></i>Update
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Create Control Point Modal -->
<div class="modal fade" id="createControlPointModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h6 class="modal-title"><i class="fas fa-map-marker-alt me-2"></i>Create Control Point</h6>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="post">
                <div class="modal-body">
                    <input type="hidden" name="event_id" value="<?= $selected_event_id ?>">
                    <input type="hidden" name="create_control_point" value="1">
                    
                    <div class="mb-3">
                        <label class="form-label">Control Point Name</label>
                        <input type="text" class="form-control" name="control_point_name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" name="description" rows="2"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary btn-sm">
                        <i class="fas fa-plus me-1"></i>Create
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Control Point Modal -->
<div class="modal fade" id="editControlPointModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h6 class="modal-title"><i class="fas fa-edit me-2"></i>Edit Control Point</h6>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="post">
                <div class="modal-body">
                    <input type="hidden" name="event_id" value="<?= $selected_event_id ?>">
                    <input type="hidden" name="control_point_id" id="edit_control_point_id">
                    <input type="hidden" name="edit_control_point" value="1">
                    
                    <div class="mb-3">
                        <label class="form-label">Name</label>
                        <input type="text" class="form-control" name="control_point_name" id="edit_control_point_name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" name="description" id="edit_description" rows="2"></textarea>
                    </div>
                    
                    <div class="row">
                        <div class="col-6">
                            <label class="form-label">Status</label>
                            <select class="form-select" name="status" id="edit_status">
                                <option value="1">Active</option>
                                <option value="0">Inactive</option>
                            </select>
                        </div>
                        <div class="col-6">
                            <label class="form-label">Weight</label>
                            <input type="number" class="form-control" name="weight" id="edit_weight" 
                                   step="0.1" min="0" max="2" value="1.0">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-danger btn-sm" onclick="deleteControlPoint()">
                        <i class="fas fa-trash me-1"></i>Delete
                    </button>
                    <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary btn-sm">
                        <i class="fas fa-check me-1"></i>Update
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Help Modal -->
<div class="modal fade" id="helpModal" tabindex="-1">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-question-circle me-2"></i>Judge Manager Help</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body p-0" style="height:80vh;">
                <iframe src="../help/judge_add_help.html" style="border:0;width:100%;height:100%;"></iframe>
            </div>
        </div>
    </div>
</div>

<!-- Delete Judge Confirmation Modal -->
<div class="modal fade" id="deleteJudgeModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h6 class="modal-title"><i class="fas fa-exclamation-triangle me-2"></i>Delete Judge</h6>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="mb-2">Are you sure you want to delete <strong id="deleteJudgeName"></strong>?</p>
                <div id="deleteJudgeRoleWarning" class="alert alert-warning mb-2" style="display:none;">
                    <i class="fas fa-exclamation-triangle me-1"></i>
                    <strong>Warning:</strong> This user has role <span id="deleteJudgeRole" class="badge"></span>. Only users with role "judge" can be deleted.
                </div>
                <p class="text-danger small mb-0"><i class="fas fa-warning me-1"></i>This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancel</button>
                <form method="post" id="deleteJudgeForm" class="d-inline">
                    <input type="hidden" name="delete_id" id="delete_judge_id">
                    <button type="submit" class="btn btn-danger btn-sm">
                        <i class="fas fa-trash me-1"></i>Delete Judge
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>

<script>
// Auto-resolve event from localStorage
document.addEventListener('DOMContentLoaded', function() {
    <?php if (empty($selected_event_id)): ?>
    const storedEventId = localStorage.getItem('selectedEventId') || getCookie('selectedEventId') || getCookie('event_id');
    if (storedEventId) {
        window.location.href = 'judge_manager.php?event_id=' + storedEventId;
    }
    <?php endif; ?>
});

function getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
    return null;
}

let currentJudgeId = null;
let currentJudgeUsername = null;

function viewJudgeInfo(id, username, password, isHead, createdAt) {
    console.log('viewJudgeInfo called', {id, username, isHead, isEncrypted: password.startsWith('$2y$')});
    
    currentJudgeId = id;
    currentJudgeUsername = username;
    
    // Check if password is encrypted
    const isEncrypted = password.startsWith('$2y$');
    const passwordDisplay = isEncrypted 
        ? '<code class="text-muted">••••••••</code> <small class="text-warning">(encrypted - reset to view)</small>'
        : '<code>' + password + '</code>';
    
    const content = '<div class="alert alert-info">' +
        '<strong>User ID:</strong> <span class="badge bg-primary">' + id + '</span><br>' +
        '<strong>Username:</strong> <code>' + username + '</code><br>' +
        '<strong>Password:</strong> ' + passwordDisplay + '<br>' +
        '<strong>Head Judge:</strong> ' + (isHead == 1 ? '<span class="badge bg-warning">Yes</span>' : '<span class="badge bg-secondary">No</span>') + '<br>' +
        '<strong>Created:</strong> ' + new Date(createdAt).toLocaleString() +
    '</div>';
    
    document.getElementById('judgeInfoContent').innerHTML = content;

    // Generate QR code
    const qrContainer = document.getElementById('judgeQrCode');
    console.log('QR Container found:', qrContainer);
    console.log('QRCode library loaded:', typeof QRCode !== 'undefined');
    
    qrContainer.innerHTML = '<div class="text-center"><div class="spinner-border spinner-border-sm text-primary" role="status"></div><p class="small text-muted mt-2">Generating secure QR code...</p></div>';

    if (isEncrypted) {
        // Use token-based authentication for encrypted passwords
        console.log('Using token-based QR for encrypted password');
        document.getElementById('qrTokenControls').style.display = 'block';
        generateQRToken(id, true);
    } else {
        // Use legacy plain-text password method
        console.log('Using legacy QR for plain-text password');
        document.getElementById('qrTokenControls').style.display = 'none';
        const baseUrl = '<?= getBaseUrl(); ?>';
        const qrUrl = baseUrl + '/v2/includes/qr_login.php?user_type=judge&username=' + encodeURIComponent(username) + '&password=' + encodeURIComponent(password);
        console.log('QR URL:', qrUrl);
        
        qrContainer.innerHTML = '';
        if (typeof QRCode !== 'undefined') {
            const qr = new QRCode(qrContainer, {
                text: qrUrl,
                width: 180,
                height: 180,
            });
            console.log('QR code created:', qr);
            const warning = document.createElement('p');
            warning.className = 'small text-warning mt-2';
            warning.innerHTML = '<i class="fas fa-exclamation-triangle me-1"></i>Contains plain-text password';
            qrContainer.appendChild(warning);
        } else {
            console.error('QRCode library not found');
            qrContainer.innerHTML = '<span class="text-danger small">QR library not loaded.</span>';
        }
    }

    new bootstrap.Modal(document.getElementById('viewJudgeModal')).show();
}

function generateQRToken(userId, checkExisting = true) {
    const qrContainer = document.getElementById('judgeQrCode');
    const expiryHours = parseFloat(document.getElementById('tokenExpirySelect').value);
    
    qrContainer.innerHTML = '<div class="text-center"><div class="spinner-border spinner-border-sm text-primary" role="status"></div><p class="small text-muted mt-2">Generating QR code...</p></div>';
    
    const formData = new URLSearchParams();
    formData.append('user_id', userId);
    formData.append('user_type', 'judge');
    formData.append('expiry_hours', expiryHours);
    if (checkExisting) {
        formData.append('check_existing', 'true');
    }
    
    fetch('<?= getBaseUrl(); ?>/v2/api/generate_qr_token.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: formData.toString()
    })
    .then(response => response.json())
    .then(data => {
        console.log('Token API data:', data);
        if (data.success) {
            qrContainer.innerHTML = '';
            if (typeof QRCode !== 'undefined') {
                const qr = new QRCode(qrContainer, {
                    text: data.qr_url,
                    width: 180,
                    height: 180,
                });
                
                // Calculate time remaining
                const expiresAt = new Date(data.expires_at);
                const now = new Date();
                const hoursRemaining = ((expiresAt - now) / (1000 * 60 * 60)).toFixed(1);
                
                const notice = document.createElement('p');
                notice.className = 'small text-muted mt-2 mb-0';
                notice.innerHTML = '<i class="fas fa-clock me-1"></i>Expires in ' + hoursRemaining + 'h';
                if (data.existing) {
                    notice.innerHTML += ' <span class="badge bg-info">Existing</span>';
                }
                qrContainer.appendChild(notice);
            } else {
                qrContainer.innerHTML = '<span class="text-danger small">QR library not loaded.</span>';
            }
        } else {
            qrContainer.innerHTML = '<div class="alert alert-danger small mb-0"><i class="fas fa-exclamation-triangle me-1"></i>' + data.message + '</div>';
        }
    })
    .catch(error => {
        console.error('Token API error:', error);
        qrContainer.innerHTML = '<div class="alert alert-danger small mb-0"><i class="fas fa-exclamation-triangle me-1"></i>Error: ' + error.message + '</div>';
    });
}

function regenerateToken() {
    if (currentJudgeId) {
        generateQRToken(currentJudgeId, false);
    }
}

function editJudge(id, username, isHead, password) {
    document.getElementById('edit_id').value = id;
    document.getElementById('edit_username').value = username;
    document.getElementById('edit_is_head').checked = isHead == 1;
    document.getElementById('edit_password').value = '';
    
    // Check if password is encrypted
    const isEncrypted = password.startsWith('$2y$');
    const passwordDisplay = isEncrypted 
        ? '•••••••• (encrypted - enter new password to change)'
        : password;
    
    document.getElementById('currentPassword').textContent = passwordDisplay;
    new bootstrap.Modal(document.getElementById('editJudgeModal')).show();
}

function confirmDeleteJudge(id, username, role) {
    document.getElementById('deleteJudgeName').textContent = username;
    document.getElementById('delete_judge_id').value = id;
    
    // Show warning if not a judge role
    const warningDiv = document.getElementById('deleteJudgeRoleWarning');
    const submitBtn = document.querySelector('#deleteJudgeForm button[type="submit"]');
    
    if (role !== 'judge') {
        warningDiv.style.display = 'block';
        const roleSpan = document.getElementById('deleteJudgeRole');
        roleSpan.textContent = role;
        roleSpan.className = 'badge bg-' + (role === 'head_judge' ? 'danger' : role === 'admin' ? 'primary' : 'dark');
        submitBtn.disabled = true;
        submitBtn.classList.add('disabled');
    } else {
        warningDiv.style.display = 'none';
        submitBtn.disabled = false;
        submitBtn.classList.remove('disabled');
    }
    
    new bootstrap.Modal(document.getElementById('deleteJudgeModal')).show();
}

function selectControlPoint(controlPointId) {
    const url = new URL(window.location);
    if (controlPointId) {
        url.searchParams.set('control_point_id', controlPointId);
    } else {
        url.searchParams.delete('control_point_id');
    }
    window.location.href = url.toString();
}

function editControlPoint(id, name, description, status, weight) {
    document.getElementById('edit_control_point_id').value = id;
    document.getElementById('edit_control_point_name').value = name;
    document.getElementById('edit_description').value = description;
    document.getElementById('edit_status').value = status;
    document.getElementById('edit_weight').value = weight;
    new bootstrap.Modal(document.getElementById('editControlPointModal')).show();
}

function deleteControlPoint() {
    const controlPointId = document.getElementById('edit_control_point_id').value;
    const controlPointName = document.getElementById('edit_control_point_name').value;
    
    if (confirm(`Delete control point "${controlPointName}"? This cannot be undone.`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="event_id" value="<?= $selected_event_id ?>">
            <input type="hidden" name="control_point_id" value="${controlPointId}">
            <input type="hidden" name="delete_control_point" value="1">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}

function toggleAll(checkbox) {
    document.querySelectorAll('input[name="judge_ids[]"]').forEach(cb => {
        cb.checked = checkbox.checked;
    });
}

function copyToClipboard(username, password) {
    const text = `Username: ${username}\nPassword: ${password}`;
    navigator.clipboard.writeText(text).then(() => {
        // Show temporary success message
        const btn = event.target.closest('button');
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-check text-success"></i>';
        setTimeout(() => {
            btn.innerHTML = originalHTML;
        }, 1500);
    }).catch(err => {
        console.error('Failed to copy:', err);
        alert('Failed to copy credentials');
    });
}

function quickAssignJudge(judgeId, judgeName) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    const controlPointId = <?= json_encode($selected_control_point_id) ?>;
    
    if (!eventId) {
        alert('Please select an event first');
        return;
    }
    
    // Get control point name
    const controlPointName = <?= json_encode($current_control_point_name) ?>;
    
    if (confirm(`Assign ${judgeName} to "${controlPointName}"?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="event_id" value="${eventId}">
            <input type="hidden" name="control_point_id" value="${controlPointId || ''}">
            <input type="hidden" name="judge_ids[]" value="${judgeId}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}

// Clean URL
if (window.location.search.includes('msg=')) {
    const url = new URL(window.location);
    url.searchParams.delete('msg');
    window.history.replaceState({}, document.title, url.toString());
}
</script>

</body>
</html>
